static char rcsid[] = "@(#)$Id: mlist.c,v 2.11 2017/11/18 16:49:26 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.11 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> (was hurtta+elm@posti.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *
 *****************************************************************************
 *  This code is reimplemented. Original code on on Elm 2.5 lib/mlist.c
 *  was following copyright:
 *
 *  The Elm Mail System  -  $Revision: 2.11 $
 *
 * This file and all associated files and documentation:
 *                      Copyright (c) 1988-1995 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

#include "def_misc.h"
#include "s_me.h"
#include "mlist_imp.h"
#include "rc_imp.h"

DEBUG_VAR(Debug,__FILE__,"config");


#define TO_ME_TOKEN     "[to-me]"
#define TO_MANY_TOKEN   "[to-many]"
#define CC_ME_TOKEN     "[cc-me]"

#define LIST_TOKEN      "[list]"


static struct mlist_conf * malloc_mlist_conf P_((void));
static struct mlist_conf * malloc_mlist_conf()
{
    int i;
    struct mlist_conf *ret = safe_zero_alloc(sizeof (*ret));

    ret->magic   = MLIST_CONF_magic;

    for (i = 0; i < mlist_TOKEN_COUNT; i++)
	ret->mlist_tokens[i] = NULL;

    ret->nlist  = 0;
    ret->lists  = NULL;

    return ret;
}

void free_mlist_conf(conf)
     struct mlist_conf ** conf;
{
    int i;

    if (MLIST_CONF_magic != (*conf)->magic)
	panic("MLIST PANIC",__FILE__,__LINE__,"free_mlist_conf",
	      "Bad magic number",0);
    
    for (i = 0;  i < mlist_TOKEN_COUNT; i++) {
	if ((*conf)->mlist_tokens[i]) {
	    free_string(& ((*conf)->mlist_tokens[i]));
	}
    }

    if ((*conf)->lists) {
	for (i = 0; i < (*conf)->nlist; i++) {
	    if ((*conf)->lists[i].to_pattern) {
		free((*conf)->lists[i].to_pattern);
		(*conf)->lists[i].to_pattern = NULL;
	    }

	    if ((*conf)->lists[i].list_id) {
		free((*conf)->lists[i].list_id);
		(*conf)->lists[i].list_id = NULL;
	    }

	    if ((*conf)->lists[i].display_name) {
		free_string(& ((*conf)->lists[i].display_name));
	    }
	}

	free((*conf)->lists);
	(*conf)->lists = NULL;	
    }
    (*conf)->nlist = 0;

    (*conf)->magic = 0;   /* Invalidate */
    free(*conf);
    *conf = NULL;    
}

static int match_item P_((struct mlist_info *new, struct mlist_info *old));
static int match_item(new,old)
     struct mlist_info *new;
     struct mlist_info *old;
{
    if (old->to_pattern &&
	new->to_pattern) {
	if (0 != strcmp(old->to_pattern,
			new->to_pattern))
	    return 0;
    } else if (old->to_pattern ||
	       new->to_pattern)
	return 0;
	

    if (old->list_id &&
	new->list_id) {
	if (0 != strcmp(old->list_id,
			new->list_id))
	    return 0;
    } else if (old->list_id ||
	       new->list_id)
	return 0;


    return 1;
}

void change_mlist_conf(conf,new)
     struct mlist_conf ** conf;
     struct mlist_conf * new; 
{
    int count;
    int i;

    if (!*conf) 
	*conf = malloc_mlist_conf();
    else if (MLIST_CONF_magic != (*conf)->magic)
	panic("MLIST PANIC",__FILE__,__LINE__,"change_mlist_conf",
	      "Bad magic number (conf)",0);

    if (!new)
	return;

    if (MLIST_CONF_magic != new->magic)
	panic("MLIST PANIC",__FILE__,__LINE__,"change_mlist_conf",
	      "Bad magic number (new)",0);

    for (i = 0;  i < mlist_TOKEN_COUNT; i++) {
	if (new->mlist_tokens[i]) {
	    if ((*conf)->mlist_tokens[i]) 
		free_string(& ((*conf)->mlist_tokens[i]));
	    
	    (*conf)->mlist_tokens[i] = dup_string(new->mlist_tokens[i]);
	}
    }

    if (new->nlist < 1)
	return;

    count = (*conf)->nlist + new->nlist;

    (*conf)->lists = safe_array_realloc((*conf)->lists,
					count,sizeof ((*conf)->lists[0]));

    
    for (i = 0; i < new->nlist; i++) {
	int j;

	for (j = 0; j < (*conf)->nlist; j++) {
	    if (match_item(& (new->lists[i]),& ((*conf)->lists[j]))) {
		goto set_it;
	    }
	}
	
	if (j >= count)
	    panic("MLIST PANIC",__FILE__,__LINE__,"change_mlist_conf",
		  "Overflow",0);
	    
	if (j != (*conf)->nlist)
	    panic("MLIST PANIC",__FILE__,__LINE__,"change_mlist_conf",
		  "Bad index",0);
	
	(*conf)->lists[j].display_name = NULL;
	(*conf)->lists[j].to_pattern   = NULL;
	(*conf)->lists[j].list_id      = NULL;

	(*conf)->nlist++;

	if (new->lists[i].to_pattern)
	    (*conf)->lists[j].to_pattern = strdup(new->lists[i].to_pattern);
	if (new->lists[i].list_id)
	    (*conf)->lists[j].list_id = strdup(new->lists[i].list_id);


    set_it:
	if ((*conf)->lists[j].display_name)
	    free_string(& ((*conf)->lists[j].display_name));

	(*conf)->lists[j].display_name = 
	    dup_string(new->lists[i].display_name);

    }
}


/*  Format:
 *
 * redefining delimiters:
 * 
 * tag         default
 * [to-me]     ------------
 * [to-many]   ======------
 * [cc-me]     ============
 *
 * Defining list:
 *
 * List name   <to@match>
 * .. other lines ...
 *
 * or
 *
 * [list]      List name
 * List-id:    <list-id>
 * 
 *
 * Inside of [ ] are literal keywords
 */


struct mlist_conf * load_mlist_conf(filename,errors,fileset,propline)
     const char *filename;
     int *errors;
     charset_t *fileset;
     struct editor_propline **propline;  
{
    struct mlist_conf * ret = NULL;
    struct mlist_info  * L;

    charset_t  cs  = system_charset;    /* Charset of file */
    int lineno = 0;
    int last_c = 0;
    int max_result = 0;
    int result_len = 0;
    FILE * f;

    int err = can_open(filename,"r");
    int c;
    char * buf = NULL;
    struct mlist_info * current_list = NULL;

    struct string  * mlist_tokens[mlist_TOKEN_COUNT];  
    int i;
   
    if (err) {
	DPRINT(Debug,2,(&Debug,"load_mlist_conf: %s: %s (can_open)\n",
			filename,strerror(err)));
	return NULL;
    }

    for (i = 0; i < mlist_TOKEN_COUNT; i++)
	mlist_tokens[i] = NULL;

    f = fopen(filename,"r");
    if (!f) {
 	int err UNUSED_VAROK = errno;
	DPRINT(Debug,2,(&Debug,"load_mlist_conf: %s: %s\n",
			filename,strerror(err)));
	return NULL;
    }

    if (fileset && *fileset) {
	const char * cs_name = get_charset_MIME_name(*fileset);
	
	cs = *fileset;
	
	if (cs_name) {
	    DPRINT(Debug,2,(&Debug, 
			    "load_mlist_conf: %s: default charset: %s\n",
			    filename,cs_name));
	}
    }

    /* Look for editor property line */

    if (propline)  {
	enum editor_propline_v propline_mode = 
	    give_dt_enumerate_as_int(&editor_ml_propline);
	
	/* Nothing detected yet */
	
	if (*propline)
	    free_editor_propline(propline);

	if (propline_mode != editor_propline_ignore) {
	    int i;
	    
	    for (i = 0; i < 3 && !*propline; i++) {
		int l1;

		int c = fgetc(f);

		if (EOF == c)
		    goto propline_failure;
		if ('\n' == c)
		    continue;

		ungetc(c,f);

		if ('#' != c)
		    break;

		l1 = malloc_gets(&buf,LONG_STRING,f);
		if (l1 > 1) {

		    if ('#' != buf[0])
			goto propline_failure;
		    
		    *propline =
			detect_editor_propline(propline_mode,
					       buf,l1,filename,i,&cs);
					       
		} else { /* something wrong? */
		propline_failure:

		    rewind(f);
		    break;
		}
	    }
	}	
    }


    while(EOF != (c = fgetc(f))) {
	if ('\n' == c)
	    max_result++;
	last_c = c;
    }

    if (last_c && '\n' != last_c) {
	DPRINT(Debug,9,(&Debug, 
			"load_mlist_conf: %s, no newline on end of file\n",
			filename));
	max_result++;	
    }

    DPRINT(Debug,10,(&Debug,"load_mlist_conf: %s, max_result=%d\n",
		     filename,max_result));
    
    if (!max_result) {
	fclose(f);
	return NULL;
    }
    rewind(f);

    L = safe_calloc(max_result, sizeof (L[0]));

 
    /* malloc_gets reallocates buf and do not return \n.
       It returns -1 if line is longer than given limit (32000)
       and -2 is returned on error  (buffer may still be alloced)
    */

    while (result_len < max_result &&
           !feof(f) && !ferror(f)) {
        int l1 = malloc_gets(&buf,32000,f);
	char * c = buf;
	char *x1;

	if (-1 == l1) {
            lib_error(CATGETS(elm_msg_cat, MeSet, MeTooLongLine,
                              "%.30s: Too long line: %.30s..."),
                      filename,buf ? buf : "???");
            (*errors) ++;
            goto OUT;
	    
        } else if (-2 == l1) {
            int err = errno;
	    
            lib_error(CATGETS(elm_msg_cat, MeSet, MeReadError,
			      "%.30s: Reading error: %s"),
                      filename,strerror(err));
            (*errors) ++;
	    
            goto OUT;
        } 

	lineno++;

	if (0 == l1) {
	    /* Empty line start new list specification */
	    current_list = NULL;  
            continue;
	}

	l1 = trim_whitespace_from_end(buf,l1);

	if (read_charset_tag(buf,l1,filename,lineno,errors,&cs,NULL))
	    continue;

        while (*c && whitespace ( *c)) /* skip leading whitespace */
            c++;
        if (*c == '#') /* Skip comments */
            continue;
        if (! *c) {
	    /* Empty line start new list specification */
	    current_list = NULL;  
            continue;
	}

	/* Tag line */
	if ('[' == buf[0]) {
	    int i = -1;
	    char * x = NULL;
		
	    current_list = NULL;  
 
	    if (0 == strncmp(TO_ME_TOKEN,buf,strlen(TO_ME_TOKEN))) {
		i = mlist_to_me_token;
		x = buf + strlen(TO_ME_TOKEN);
	    } else if (0 == strncmp(TO_MANY_TOKEN,buf,strlen(TO_MANY_TOKEN))) {
		i = mlist_to_many_token;
		x = buf + strlen(TO_MANY_TOKEN);
	    } else if (0 == strncmp(CC_ME_TOKEN,buf,strlen(CC_ME_TOKEN))) {
		i = mlist_cc_me_token;
		x = buf + strlen(CC_ME_TOKEN);
	    } 
		       
	    if (i >= 0) {

		if (mlist_tokens[i]) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeMlistDupToken,
				      "%.30s: Duplicate token on line: %.30s..."),
			      filename,buf ? buf : "???");
		    free_string(& mlist_tokens[i]);
		    (*errors) ++;
		}

		while (*x && whitespace ( *x)) /* skip leading whitespace */
		    x++;
		
		if (!*x) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLine,
				      "%s: Bad line: %.30s..."),
			      filename,buf);
		    (*errors) ++;
		    continue;
		}
		    
		/* Allow mime encoding used on here .... */

		mlist_tokens[i] = hdr_to_string(HDR_PHRASE,x,cs,
						1);

	    } else if (0 == strncmp(LIST_TOKEN,buf,strlen(LIST_TOKEN))) {
		char * x = buf + strlen(LIST_TOKEN);

		while (*x && whitespace ( *x)) /* skip leading whitespace */
		    x++;
		
		if (!*x) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLine,
				      "%s: Bad line: %.30s..."),
			      filename,buf);
		    (*errors) ++;
		    continue;
		}
		
		current_list = &L[result_len++];

		current_list->to_pattern = NULL;
		current_list->list_id = NULL;

		/* Allow mime encoding used on here .... */
		current_list->display_name = hdr_to_string(HDR_PHRASE,x,cs,
						1);


	    } else {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeMlistBadToken,
                              "%.30s: Bad token on line: %.30s..."),
                      filename,buf ? buf : "???");

		(*errors) ++;
	    }

	    continue;
	}

	x1 = qstrpbrk(buf,":<");

	if (!x1) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLine,
			      "%s: Bad line: %.30s..."),
		      filename,buf);
	    (*errors) ++;
	    continue;
	}

	if (*x1 == '<') {
	    char ** tokens = rfc822_tokenize(buf);
	    char * phrase = NULL;
	    char * pattern = NULL;
	    int i;

	    /* 1) read phrase */

	    for (i = 0; tokens[i]; i++) {
		
		if ('\n' == tokens[i][0] &&
		    !tokens[i+1])
		    break;
		
		if ('<' == tokens[i][0])
		    break;
		
		if ('(' == tokens[i][0])
		    goto error;
		
		if ('\n' == tokens[i][0] ||
		    '\t' == tokens[i][0])
		    phrase = strmcat(phrase," ");
		else
		    phrase = strmcat(phrase,tokens[i]);
	    }


	    if (!tokens[i] || '<' != tokens[i][0])
		goto error;


	    /* 2) read pattern */

	     for (i++; tokens[i]; i++) {

		 if ('\n' == tokens[i][0] &&
		     !tokens[i+1])
		     break;
		 
		 if ('>' == tokens[i][0])
		     break;
		 
		 if ('(' == tokens[i][0])
		     goto error;
		 if (' ' == tokens[i][0])
		     goto error;
		 if ('\n' == tokens[i][0])
		     goto error;
		 
		 pattern = strmcat(pattern,tokens[i]);
	     }

	     if (!tokens[i] || '>' != tokens[i][0])
		 goto error;

	     if (!phrase || !pattern)
		 goto error;

	     current_list = &L[result_len++];
	     
	     current_list->to_pattern = pattern; pattern = NULL;
	     current_list->list_id = NULL;

	     /* Allow mime encoding used on here .... */
	     current_list->display_name = hdr_to_string(HDR_PHRASE,
							phrase,cs,
							1);
	     if (0) {
	     error:
		 lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLine,
				   "%s: Bad line: %.30s..."),
			   filename,buf);
		 (*errors) ++;
		 
	     }
	     
	     if (phrase)
		 free(phrase);
	     
	     if (pattern)
		 free(pattern);
	     
	     
	     free_rfc822tokenized(tokens);
   
	} else {

	    if (!current_list) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeListNameExpected,
				  "%.30s: List name expected: %.30s..."),
			  filename,buf);
		(*errors) ++;
		continue;
	    }

	    if (strincmp(buf,"List-id:",8) == 0) {
		char * x1 = buf+8;

		char ** tokens = rfc822_tokenize(x1);
		char * id = NULL;
		int i = 0;

		while (tokens[i] && (' ' == tokens[i][0] ||
				     '\t' == tokens[i][0]))
		    i++;

		if (!tokens[i] || '<' != tokens[i][0])
		    goto error1;
		
		for (i++; tokens[i]; i++) {
		    
		    if ('\n' == tokens[i][0] &&
			!tokens[i+1])
		     break;
		    
		    if ('>' == tokens[i][0])
			break;
		    
		    if ('(' == tokens[i][0])
			goto error1;
		    if (' ' == tokens[i][0])
			goto error1;
		    if ('\n' == tokens[i][0])
			goto error1;
		    
		    id = strmcat(id,tokens[i]);
		}

		if (!tokens[i] || '>' != tokens[i][0])
		    goto error1;

		if (!id)
		    goto error1;

		if (current_list->list_id) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeMlistDupToken,
				      "%.30s: Duplicate token on line: %.30s..."),
			      filename,buf ? buf : "???");
		    
		    free(current_list->list_id);
		    (*errors) ++;
		}

		current_list->list_id = id; id = NULL;
		
		if (0) {
		error1:
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLine,
				      "%s: Bad line: %.30s..."),
			      filename,buf);
		    (*errors) ++;
		    
		}


		if (id)
		    free(id);

		free_rfc822tokenized(tokens);

	    } else {
		 lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLine,
				   "%s: Bad line: %.30s..."),
			   filename,buf);
		 (*errors) ++;
	    }
	}

    }

    if (ferror(f)) {
        int err = errno;

        lib_error(CATGETS(elm_msg_cat, MeSet, MeReadError,
                          "%s: Reading error: %s"),
                  filename,strerror(err));
        (*errors) ++;
    }

 OUT:
    if (buf) {
        free(buf);
    }

    if (!feof(f)) {
        DPRINT(Debug,3,(&Debug,
                        "load_mlist_conf: %s, All results not readed.\n",
                        filename));
    }

    fclose(f);

    if (fileset)
	*fileset = cs;

    DPRINT(Debug,10,(&Debug,
                     "load_mlist_conf: %s, result_len=%d\n",
                     filename,result_len));


    ret = malloc_mlist_conf();

    ret->nlist = result_len;
    ret->lists = L;

    for (i = 0; i < mlist_TOKEN_COUNT; i++)
	ret->mlist_tokens[i] = mlist_tokens[i];

    return ret;
}

static int need_mime_encoding P_((const struct string *S, charset_t fileset));
static int need_mime_encoding(S,fileset)
     const struct string *S; 
     charset_t fileset;
{
    charset_t cs = get_string_type(S);
    int ret = 0;

    if (fileset != cs) 
	ret = 1;
    else {
	uint16 prev = 0x0000;
	int len = string_len(S);
	int x;

	for (x = 0; x < len; x++) {
            uint16 u = give_unicode_from_string(S,x);

	    if ((0x003D /* = */ == prev &&
		 0x003F /* ? */ == u)    ||
		(0x003F /* ? */ == prev &&
		 0x003D /* = */ == u)) {
		ret = 1;
                break;
	    }
	}
    }
    
    return ret;
}

void dump_mlist_conf(f,conf,commentfile,actor,version_buff,
		     fileset,propline)
     FILE *f;
     struct mlist_conf *conf;
     FILE *commentfile;
     const char *actor;
     char *version_buff;
     charset_t fileset;
     const struct editor_propline *propline;    
{
    int i;
     enum editor_propline_v propline_mode = 
	 give_dt_enumerate_as_int(&editor_ml_propline);
    const char * sysnam = NULL;

    if (!conf)
	return;

    if (MLIST_CONF_magic != conf->magic)
	panic("MLIST PANIC",__FILE__,__LINE__,"dump_mlist_conf",
	      "Bad magic number",0);

    if (!fileset) {
	DPRINT(Debug,10,(&Debug,
			 "dump_mlist_conf: File charset is not set, using system charset\n"));
	fileset = system_charset;    
    }

    sysnam = get_charset_MIME_name(fileset);

    /* Must have first line */
    if (propline_mode != editor_propline_ignore)
	write_editor_propline(propline,propline_mode,
			      f,"# ","\n",fileset);

    if (sysnam) {
	/* Store used character set */

	elm_fprintf(f, 
		    CATGETS(elm_msg_cat,MeSet, MeCharsetExplain,
			    "# Character set used on this file\n"));

	fprintf(f,"@charset = %s\n",sysnam);
    }

    insert_commentfile(f,ELMMAILINGLISTS_INFO,
		       commentfile,actor,version_buff);

    for (i = 0; i < mlist_TOKEN_COUNT; i++) {
	if (conf->mlist_tokens[i]) {
	    char * s1 = NULL;

	    s1 = string_to_hdr(HDR_PHRASE,conf->mlist_tokens[i], 
			       fileset, 
			       need_mime_encoding(conf->mlist_tokens[i],
						  fileset),
			       NULL,1);

	    switch(i) {
	    case mlist_to_me_token:
		fprintf(f,"%s\t%s\n",TO_ME_TOKEN,s1); 
		break;

	    case mlist_to_many_token:
		fprintf(f,"%s\t%s\n",TO_MANY_TOKEN,s1); 
		break;
		
	    case mlist_cc_me_token:
		fprintf(f,"%s\t%s\n",CC_ME_TOKEN,s1); 
		break;		
	    }
	    free(s1);

	}
    }

    for (i = 0; i < conf->nlist; i++) {
	putc('\n',f);
	
	if (conf->lists[i].to_pattern) {
	    char * s1 = NULL;

	    s1 = string_to_hdr(HDR_PHRASE,conf->lists[i].display_name,
			       fileset, 
			       need_mime_encoding(conf->lists[i].display_name,
						  fileset),
			       NULL,0);

	    
	    fprintf(f,"%s<%s>\n",
		    s1,conf->lists[i].to_pattern);

	    free(s1);

	} else {
	    char * s1 = NULL;

	    s1 = string_to_hdr(HDR_PHRASE,conf->lists[i].display_name,
			       fileset, 
			       need_mime_encoding(conf->lists[i].display_name,
						  fileset),
			       NULL,1);

	    fprintf(f,"%s\t%s\n",
		    LIST_TOKEN,s1);

	    free(s1);
	}

	if (conf->lists[i].list_id) {
	    fprintf(f,"List-id:\t<%s>\n",conf->lists[i].list_id);
	}
    }
}


/* ------------------------------------------------------------------- */



const struct string * get_mlist_token(list,idx)
     struct mlist_conf *list;
     enum mlist_tokens idx;
{

    if (MLIST_CONF_magic != list->magic)
	panic("MLIST PANIC",__FILE__,__LINE__,"get_mlist_token",
	      "Bad magic number",0);

    if (idx < 0 || idx >= mlist_TOKEN_COUNT)
	panic("MLIST PANIC",__FILE__,__LINE__,"get_mlist_token",
	      "Bad index",0);

    return list->mlist_tokens[idx];

}


const struct string * get_mlist_display_name(info)
     const struct mlist_info *info;
{
    return info->display_name;
}



/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
