static char rcsid[] = "@(#)$Id: terminal.c,v 2.15 2023/12/13 16:55:32 hurtta Exp $";

/*************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.15 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *          (was hurtta+elm@posti.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 ************************************************************************/

#include "def_misc.h"

#include "s_me.h"

#include "cs_imp.h"
#include "cs_terminal.h"

DEBUG_VAR(Debug,__FILE__,"charset");

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif


static char *us2s P_((unsigned char *str));
static char *us2s(str)
     unsigned char *str;
{
    return (char *)str;
}

static unsigned char * s2us P_((char *str));
static unsigned char * s2us(str)
     char *str;
{
    return (unsigned char *)str;
}



enum terminal_flags { terminal_bad_flag    = -1,  
		      terminal_xterm_title = 1,
		      terminal_xwsh_title  = 2
};

enum terminal_map_keyword { terminal_bad = -1, terminal_iso2022         = 0,
                            terminal_private = 1, terminal_iso2022_mb   = 2,
			    terminal_flag    = 3, terminal_iso2022_like = 4,
			    terminal_dw,          terminal_ansi_sgr
};


struct terminal_map_item {
    char                    * match;
    enum terminal_map_keyword keyword;
    union {
	struct cset {
	    struct term_cond  * condition;
	    charset_t  charset;  
	    int        is_def;
	} iso2022;
        struct privset {
	    struct term_cond  * condition;
            charset_t           charset;
            unsigned char *     bytes;
	} private;
	enum terminal_flags flag;
	struct cset2 {
	    charset_t           charset;
	    struct setlist    * iso2022_info;   
	} iso2022_like;

	struct dw {
	    struct term_cond  * condition;
	    char              * charset;
	} dw;

	struct ansi_sgr {
	    struct term_cond  * condition;
	    size_t              ANSI_sgr_index; /* Index to ANSI_sgr_values[] */
	} ansi_sgr;
	
    } value;
};

static void strXcat P_((char **ret,char *val, 
			int static_buffer, char *buffer, int size));
static void strXcat(ret,val,static_buffer,buffer,size)
     char **ret;
     char *val; 
     int static_buffer; 
     char *buffer; 
     int size;
{
    
    if (static_buffer) {
	if (!*ret)
	    *ret = strfcpy(buffer,val,size);
	else
	    *ret = strfcat(buffer,val,size);
    } else
	*ret = strmcat(*ret,val);
}


static enum terminal_map_keyword get_keyword P_((const char *word));

static enum terminal_map_keyword get_keyword(word)
     const char *word;
{
    if (0 == istrcmp(word,"ISO-2022"))
	return terminal_iso2022;
    if (0 == istrcmp(word,"ISO-2022-LIKE"))
	return terminal_iso2022_like;
    if (0 == istrcmp(word,"ISO-2022/DW"))
	return terminal_iso2022_mb;
    if (0 == istrcmp(word,"PRIVATE"))
	return terminal_private;
    if (0 == istrcmp(word,"FLAG"))
	return terminal_flag;
    if (0 == istrcmp(word,"DW"))
	return terminal_dw;
    if (0 == istrcmp(word,"ANSI-SGR"))
	return terminal_ansi_sgr;

    return terminal_bad;
}

static char * sStr P_((unsigned char *p));
static char * sStr (p)
     unsigned char *p;
{
    return (char *)p;
}


#define TERM_cond_magic      0xE901

struct term_cond {
    unsigned short          magic;   /* TERM_cond_magic */
    enum tc_type { tt_value, tt_simple_var, tt_equal, tt_match } X;


    struct term_cond        * p1;
    struct term_cond        * p2;

    char                    * value;

};

static void zero_term_cond P_((struct term_cond *X));
static void zero_term_cond(X)
     struct term_cond *X;
{
    X->magic  = TERM_cond_magic;
    X->X      = tt_value;
    X->p1     = NULL;
    X->p2     = NULL;
    X->value  = NULL;
}

static struct term_cond * copy_term_cond P_((struct term_cond *ptr));
static struct term_cond * copy_term_cond(ptr)
     struct term_cond *ptr;
{
    struct term_cond * ret = safe_malloc(sizeof (*ret));

    zero_term_cond(ret);

    ret->X = ptr->X;

    if (ptr->p1)
	ret->p1 = copy_term_cond(ptr->p1);
    if (ptr->p2)
	ret->p2 = copy_term_cond(ptr->p2);

    if (ptr->value)
	ret->value = safe_strdup(ptr->value);

    return ret;
}

static void free_term_cond P_((struct term_cond **X));
static void free_term_cond(X)
     struct term_cond **X;
{
    struct term_cond *A = *X; 

    if (TERM_cond_magic != A->magic)
	panic("CHARSET PANIC",__FILE__,__LINE__,"free_term_cond",
	      "Bad magic number",0);

    if (A->p1)
	free_term_cond(&(A->p1));
    if (A->p2)
	free_term_cond(&(A->p2));

    if (A->value) {
	free(A->value);
	A->value = NULL;
    }

    free(A);
    A = NULL;
	
    *X = A;
}

static int match_term_cond P_((struct term_cond *A, struct term_cond *B));
static int match_term_cond(A,B)
     struct term_cond *A;
     struct term_cond *B;
{  
    if (TERM_cond_magic != A->magic ||
	TERM_cond_magic != B->magic)
	panic("CHARSET PANIC",__FILE__,__LINE__,"match_term_cond",
	      "Bad magic number",0);

    if (A->X != B->X)
	return 0;

    if (A->p1 && B->p1) {
	if (! match_term_cond(A->p1,B->p1))
	    return 0;
    } else if (A->p1 || B->p1) 
	return 0;

    if (A->p2 && B->p2) {
	if (! match_term_cond(A->p2,B->p2))
	    return 0;
    } else if (A->p2 || B->p2) 
	return 0;


    if (A->value && B->value) {
	if (0 != strcmp(A->value,B->value))
	    return 0;
    } else if (A->value || B->value)
	return 0;

    return 1;
}



enum tc_token_type { tc_left = '[', tc_rigth = ']', tc_equal = '=', 
                     tc_match = '~',		     
		     tc_quoted = '"', tc_plain = 'a', tc_simple_var = '$',
		     tc_none = 0 };

static enum tc_token_type tc_get_token P_((char **value, char **ptr));
static enum tc_token_type tc_get_token (value,ptr)
     char **value;
     char **ptr;
{
    enum tc_token_type r = tc_none;
    char *C = *ptr;

    *value = NULL;

    while (*C && whitespace (*C)) /* skip leading whitespace */
	C++;

    switch (*C) {
	int L;
    case '\0':    goto fail;
    case '[':     r = tc_left; C++;  break;
    case ']':     r = tc_rigth; C++; break;
    case '=':     
	r =  tc_equal; C++; 

	if ('~' == *C) {
	    r = tc_match; C++;
	}
	break;
    case '"':
	L = len_next_part(C);
	*value = dequote_opt(C,L);
	C += L;
	r = tc_quoted;
	break;
    case '$':	
	L = strcspn(C+1," []=\"$/");
	if (1 == L) {
	    r = tc_none;
	    goto fail;
	}
	*value = safe_malloc(L+1);
	memcpy(*value,C+1,L);
	(*value)[L] = '\0';
	C += L +1;
	r = tc_simple_var;
	break;
    default:
	L = strcspn(C," []=\"$");
	*value = safe_malloc(L+1);
	memcpy(*value,C,L);
	(*value)[L] = '\0';
	C += L;
	r = tc_plain;
	break;
    }

 fail:
    *ptr = C;
    return r;
}

static struct term_cond * tc_get_expression P_((char              **ptr));
static struct term_cond * tc_get_expression(ptr)     
     char              **ptr;
{
    struct term_cond *r  = NULL;
    char *value = NULL;

    switch (tc_get_token(&value,ptr)) {
    case tc_plain:
	lib_error(CATGETS(elm_msg_cat, MeSet, MeTreatedAsQuoted,
			  "Unregonized token %s treated as quoted value"),
		  value); 
	/* FALLTHRU */
    case tc_quoted:
	r = safe_malloc(sizeof (*r));
	zero_term_cond(r);
	r->value = value;
	r->X     = tt_value;
	value = NULL;
	break;
    case tc_simple_var:
	r = safe_malloc(sizeof (*r));
	zero_term_cond(r);
	r->value = value;
	r->X     = tt_simple_var;
	value = NULL;
	break;
    default:
	r = NULL;
	break;
    }

    return r;
}

static struct term_cond * parse_condition P_((char              **ptr));
static struct term_cond * parse_condition(ptr)     
     char              **ptr;
{
    struct term_cond *r  = NULL;
    struct term_cond *r1 = NULL;
    struct term_cond *r2 = NULL;

    enum tc_token_type t1;

    char *value = NULL;

    if (tc_left != tc_get_token(&value,ptr))
	goto err;
    r1 = tc_get_expression(ptr);
    if (!r1)
	goto err;

    t1 = tc_get_token(&value,ptr);

    if (tc_equal != t1 && tc_match != t1)
	goto err;
    r2 = tc_get_expression(ptr);
    if (!r2)
	goto err;
    if (tc_rigth != tc_get_token(&value,ptr))
	goto err;

    r = safe_malloc(sizeof (*r));
    zero_term_cond(r);

    if (tc_match == t1)
	r->X = tt_match;
    else
	r->X = tt_equal;
    r->p1 = r1;
    r->p2 = r2;

    r1 = NULL; r2 = NULL;

 err:
    if (value)
	free(value);
    if (r1) 
	free_term_cond(&r1);
    if (r2) 
	free_term_cond(&r2);

    return r;
}

static void tc_dump_expression P_((FILE *f, struct term_cond * C));

static char * tc_eval_expression P_((struct term_cond * C));
static char * tc_eval_expression(C)
     struct term_cond * C;
{
    char * r = NULL;

    if (TERM_cond_magic != C->magic)
	panic("CHARSET PANIC",__FILE__,__LINE__,"tc_eval_expression",
	      "Bad magic number",0);

    switch (C->X) {
	char buffer[1024];
	int v;
    case tt_value:
	/* expand_path converts $VAR between / characters */
	v = expand_path(buffer,C->value,sizeof buffer);
	if (v > 0) {
	    DPRINT(Debug,9,(&Debug,"tc_eval_expression : \"%s\" expanded\n",
			    C->value));
	    r = safe_strdup(buffer);
	} else if (v == 0) {
	    DPRINT(Debug,9,(&Debug,"tc_eval_expression : \"%s\" copied\n",
			    C->value));
	    r = safe_strdup(C->value);       
	} else {
	    DPRINT(Debug,9,(&Debug,"tc_eval_expression : \"%s\" failed\n",
			    C->value));
	}

	DPRINT(Debug,9,(&Debug,"tc_eval_expression \"%s\" = %s\n",
			C->value,
			r ? r : "NULL"));

	break;
    case tt_simple_var:
	r = getenv(C->value);
	if (r)
	    r = safe_strdup(r);

	DPRINT(Debug,9,(&Debug,"tc_eval_expression $%s = %s\n",
			C->value,
			r ? r : "NULL"));

	break;
    default:
	DPRINT(Debug,8,(&Debug,"tc_eval_expression %d (bad)\n",C->X));
    }

    return r;
}

static void dump_condition P_((FILE *f, struct term_cond * C));

static int eval_condition P_((struct term_cond * C));
static int eval_condition(C)
     struct term_cond * C;
{
    int r = 0;
    char * v1 = NULL;
    char * v2 = NULL;

    if (TERM_cond_magic != C->magic)
	panic("CHARSET PANIC",__FILE__,__LINE__,"eval_condition",
	      "Bad magic number",0);

#ifdef DEBUG
    if (Debug.active >= 8) {

	FILE *F;

	DPRINT(Debug,8,(&Debug,"eval_condition   "));
	F = debug_to_FILE(&Debug);
	if (F) {
	    dump_condition(F,C);

	    fputc('\n',F);
	    fclose(F);
	}	
    } 
#endif

    if (C->p1)
	v1 = tc_eval_expression(C->p1);
    if (C->p2)
	v2 = tc_eval_expression(C->p2);


    switch (C->X) {
    case tt_equal:
	
	if (!v1 || !v2)
	    break;
	r = 0 == strcmp(v1,v2);

	DPRINT(Debug,8,(&Debug,"eval_condition  \"%s\" = \"%s\"\n",v1,v2));

	break;

    case tt_match:
	if (!v1 || !v2)
	    break;	

	DPRINT(Debug,8,(&Debug,"eval_condition  \"%s\" =~ \"%s\"\n",v1,v2));

	r = compare_helper(v1,v2,0);
	break;

    default:
	break;
    }

    if (v1)
	free(v1);
    if (v2)
	free(v2);
    DPRINT(Debug,8,(&Debug,"eval_condition=%d\n",r));
    return r;
}

static void tc_dump_expression(f,C)
     FILE *f;
     struct term_cond * C;
{

    if (TERM_cond_magic != C->magic)
	panic("CHARSET PANIC",__FILE__,__LINE__,"tc_dump_expression",
	      "Bad magic number",0);

    switch (C->X) {
    case tt_value:
	elm_fprintf(f,FRM("%Q"),C->value);
	break;
    case tt_simple_var:
	fprintf(f,"$%s",C->value);
	break;
    default:
	fputc('?',f);
    }
}

static void dump_condition(f,C)
     FILE *f;
     struct term_cond * C;
{

    if (TERM_cond_magic != C->magic)
	panic("CHARSET PANIC",__FILE__,__LINE__,"dump_condition",
	      "Bad magic number",0);

    fputc('[',f);
    fputc(' ',f);
    
    if (C->p1)
	tc_dump_expression(f,C->p1);

    if (tt_equal == C->X)
	fputc('=',f);
    else if (tt_match == C->X) {
	fputc('=',f);
	fputc('~',f);
    } else
	fputc('?',f);

    if (C->p2)
	tc_dump_expression(f,C->p2);

    fputc(' ',f);
    fputc(']',f);
}

static struct {
    char                *flag;
    enum terminal_flags  flag_value;
} terminal_flag_list[] = {
    { "xterm-title" , terminal_xterm_title },
    { "xwsh-title"  , terminal_xwsh_title }

};
    
struct ansi_sgr_attribute ANSI_sgr_values[] = {
    { "bold",          pg_BOLD,          1,22, pg_DIM              },
    { "dim",           pg_DIM,           2,22, pg_BOLD|pg_STANDOUT },
    { "italic",        pg_ITALIC,        3,23, 0 },
    { "underline",     pg_UNDERLINE,     4,24, 0 },
    { "blinking",      pg_BLINKING,      5,25, 0 },
    { "reverse",       pg_REVERSE,       7,27, 0 },
    { "strikethrough", pg_STRIKETHROUGH, 9,29, 0 },
    { "superscript",   pg_SUPERSCRIPT,   73,75, pg_SUBSCRIPT   },
    { "subscript",     pg_SUBSCRIPT,     74,75, pg_SUPERSCRIPT },
};
size_t NUM_ANSI_sgr_values =
    (sizeof ANSI_sgr_values) / sizeof (ANSI_sgr_values[0]);


#if DEBUG
/* Returns pointer to static storage */

char * give_pg_flags(p)
     int p;
{
    static char buffer[15*NUM_pg_attr_bits];
    enum pg_attr_bit i;
    
    if (!p)
	return "none";

    if (~0 == p)
	return "all (~0)";
    
    buffer[0] = '\0';
    buffer[1] = '\0'; /* In case there is unknown bits set */

    for (i = 0; i < NUM_pg_attr_bits; i++) {

	if (ison(p,(1 << i))) {
	    switch (i) {
	    case pg_bit_BOLD:
		strfcat(buffer," pg_BOLD",sizeof buffer);
		break;
	    case pg_bit_UNDERLINE:
		strfcat(buffer," pg_UNDERLINE",sizeof buffer);
		break;
	    case pg_bit_STANDOUT:
		strfcat(buffer," pg_STANDOUT",sizeof buffer);
		break;
	    case pg_bit_BLINKING:
		strfcat(buffer," pg_BLINKING",sizeof buffer);
		break;
	    case pg_bit_REVERSE:
		strfcat(buffer," pg_REVERSE", sizeof buffer);
		break;
	    case pg_bit_ITALIC:
		strfcat(buffer," pg_ITALIC", sizeof buffer);
		break;
	    case pg_bit_DIM:
		strfcat(buffer," pg_DIM", sizeof buffer);
		break;
	    case pg_bit_SUBSCRIPT:
		strfcat(buffer," pg_SUBSCRIPT", sizeof buffer);
		break;
	    case pg_bit_SUPERSCRIPT:
		strfcat(buffer," pg_SUPERSCRIPT", sizeof buffer);
		break;
	    case pg_bit_STRIKETHROUGH:
		strfcat(buffer," pg_STRIKETHROUGH", sizeof buffer);
		break;
	    case NUM_pg_attr_bits:
		break;
	    }
	}
    }
    
    return buffer+1;
}
#endif

int pg_set_or_disable_flags(res_pg_flag,pg_flag)
     int * res_pg_flag /* maybe NULL */;
     int pg_flag;
{
    size_t i;
    int disable_pg_flags = 0;
    
    if (!pg_flag)  /* no changes */
	return 0;

    /* pg_STANDOUT is not on ANSI_sgr_values[] */
    if (ison(pg_flag,pg_STANDOUT))
	setit(disable_pg_flags,pg_DIM);

    for (i = 0; i < NUM_ANSI_sgr_values; i++) {
	if (ison(pg_flag,ANSI_sgr_values[i].pg_flag))
	    setit(disable_pg_flags,ANSI_sgr_values[i].disable_pg_flags);
    }
    
    if (res_pg_flag) {
	int val = *res_pg_flag;

	clearit(val,disable_pg_flags);
	setit(val,pg_flag);

	*res_pg_flag = val;
    }

    return disable_pg_flags;
}




struct terminal_map_item * load_terminal_map(filename,errors)
     const char *filename;
     int *errors;
{
    struct terminal_map_item *result;
    int result_len = 0;
    FILE * f;
    int max_result = 0;
    int c;
    char * buf = NULL;
    int lineno = 0;

    int last_c = 0;
    int err = can_open(filename,"r");

    if (err) {
	DPRINT(Debug,2,(&Debug,
			"load_terminal_map: %s: %s (can_open)\n",
			filename,strerror(err)));
	return NULL;
    }

    f = fopen(filename,"r");
    if (!f) {
 	err = errno;
	
	DPRINT(Debug,2,(&Debug,
			"load_terminal_map: %s: %s\n",
			filename,strerror(err)));
	return NULL;
    }

    while(EOF != (c = fgetc(f))) {
	if ('\n' == c)
	    max_result++;
	last_c = c;
    }

    if (last_c && '\n' != last_c) {
	DPRINT(Debug,9,(&Debug, 
			"load_terminal_map: %s, no newline on end of file\n",
			filename));
	max_result++;	
    }

    DPRINT(Debug,10,(&Debug,"load_terminal_map: %s, max_result=%d\n",
		     filename,max_result));
    
    if (!max_result) {
	fclose(f);
	return NULL;
    }
    rewind(f);
    
    result = safe_calloc((max_result +1), sizeof (result[0]));
    
    /* malloc_gets reallocates buf and do not return \n.
       It returns -1 if line is longer than given limit (32000)
       and -2 is returned on error  (buffer may still be alloced)
    */

    while (result_len < max_result &&
	   !feof(f) && !ferror(f)) {
	int l1 = malloc_gets(&buf,32000,f);
	char * c = buf,*c1,*c2;
	
	enum terminal_map_keyword kw;
	struct terminal_map_item  item;

	int item_ok = 0;
	
	bzero((void *)&item,sizeof item);
	
	if (-1 == l1) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeTooLongLineNo,
			      "%s: %d: Too long line: %.30s..."),
		      filename,lineno+1,buf);

	    (*errors) ++;
	    goto OUT;

	} else if (-2 == l1) {
	    err = errno;
	    
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeReadError,
			  "%.30s: Reading error: %s"),
		      filename,strerror(err));
	    (*errors) ++;	    

	}

	lineno++;
	
	if (0 == l1)
	    continue;

	while (l1-- > 0 && whitespace(buf[l1]))
	    buf[l1] = '\0';
	
	c = buf;
	while (*c && whitespace (*c)) /* skip leading whitespace */
	    c++;
	if ('#' == *c)
	    continue;
	if (!*c)
	    continue;

	c1 = strpbrk(c," \t");

	if (!c1) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
			      "%s: %d: Bad line: %.30s..."),
		      filename,lineno,buf);
	    
	    (*errors) ++;
	    break;	    
	}
	*c1 = '\0';
	
	c1++;

	while (*c1 && whitespace (*c1)) /* skip leading whitespace */
	    c1++;
	if (!*c1) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
			      "%s: %d: Bad line: %.30s..."),
		      filename,lineno,buf);

	    (*errors) ++;
	    break;	    
	}
	    
	c2 = strpbrk(c1," \t");

	if (!c2) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
			      "%s: %d: Bad line: %.30s..."),
		      filename,lineno,buf);

	    (*errors) ++;
	    break;	    
	}
	*c2 = '\0';
	
	c2++;

	while (*c2 && whitespace (*c2)) /* skip leading whitespace */
	    c2++;
	if (!*c2) {
           lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
			     "%s: %d: Bad line: %.30s..."),
		     filename,lineno,buf);
	   
	    (*errors) ++;
	    break;	    
	}
	    
	kw = get_keyword(c1);

	if (terminal_bad == kw) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
			      "%s: %d: Bad line: %.30s..."),
		      filename,lineno,buf);
	    
	    (*errors) ++;
	    break;	    
	}
	
	switch(kw) {
	    char * c3;
	    int size;
	    int i;
	    size_t x;
	    
	case terminal_dw:
 
	    item.value.dw.condition = NULL;
	    
	    c3 = qstrpbrk(c2,"[");  
	    if (c3 && '[' == *c3) {
		char * ptr = c3;
		
		item.value.dw.condition =
		    parse_condition(&c3);
		if (!item.value.dw.condition)
		    goto err_5;

		while (*c3 && whitespace (*c3)) /* skip leading whitespace */
		    c3++;
		if (*c3) {

		    free_term_cond(& (item.value.dw.condition));
		    
		err_5:
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
				      "%s: %d: Bad line: %.30s..."),
			      filename,lineno,buf);

		    goto fail_item;
		}


		/* Remove trailing whitespace */
		
		while (ptr > c2 && whitespace(*(ptr-1)))
		    ptr--;
		
		*ptr = '\0';
	    }

	    item.value.dw.charset = safe_strdup(c2);
	    item_ok = 1;
	    break;

	case terminal_iso2022:
	case terminal_iso2022_mb:	    
	    c3 = qstrpbrk(c2,"[");  
	    
	    item.value.iso2022.condition = NULL;
	    if (c3 && '[' == *c3) {
		char * ptr = c3;
		
		item.value.iso2022.condition = parse_condition(&c3);
		if (!item.value.iso2022.condition)
		    goto err_4;
		
		while (*c3 && whitespace (*c3)) /* skip leading whitespace */
		    c3++;
		if (*c3) {
		err_4:
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
				      "%s: %d: Bad line: %.30s..."),
			      filename,lineno,buf);
		    		    
		    goto error_iso2022;
		}

		/* Remove trailing whitespace */

		while (ptr > c2 && whitespace(*(ptr-1)))
		       ptr--;
	    
		*ptr = '\0';
	    }

	    if (load_charset_map_info(&item.
				      value.iso2022.charset,c2)) {
		item.value.iso2022.is_def = 
		    ison(item.value.iso2022.charset->flags,
			 SET_nodata);
	    } else {
		item.value.iso2022.charset = 
		    MIME_name_to_charset(c2,0); 
		item.value.iso2022.is_def = 0;
	    }
	    
	    if (!item.value.iso2022.charset) {
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeUnknownTerminalCharset,
				  "Unknown terminal charset %s (file %s)"),
			  c2,filename);
		
		goto error_iso2022;
	    }

	    if (!item.value.iso2022.charset->map_info) {
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeCharsetNeedsMap,
				  "Charset %s needs map= defination (file %s)"),
			  item.value.iso2022.charset->MIME_name ?
			  item.value.iso2022.charset->MIME_name :
			  "<no MIME name>",
			  filename);

		goto error_iso2022;
	    }

	    if (!item.value.iso2022.charset->iso2022_info) {
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeCharsetNeedsIso2022,
				  "Charset %s needs iso2022 defination (file %s)"),
			  item.value.iso2022.charset->MIME_name ?
			  item.value.iso2022.charset->MIME_name :
			  "<no MIME name>",
			  filename);

		goto error_iso2022;
	    }

	    if (terminal_iso2022_mb == item.keyword &&
		isoff(charset_properties(item.value.iso2022.
					 charset),
		      CS_printable_len)) {
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeCharsetNotMB,
				  "Charset %s does not support /DW flag (file %s)"),
			  item.value.iso2022.charset->MIME_name ?
			  item.value.iso2022.charset->MIME_name :
			  "<no MIME name>",
			  filename);
		
	    error_iso2022:
	    
		if (item.value.iso2022.condition)
		    free_term_cond(& (item.value.iso2022.condition));
		
		goto fail_item;		
	    }

	    item_ok = 1;
	    break;
	case terminal_iso2022_like: {
	    struct setlist     new_setlist;
	    int         setcount          = 0;

	    char * opt;
	    char * WALK;
	
	    iso2022_clear_setlist(&new_setlist);

	    c3 = strpbrk(c2," \t");

	    if (!c3) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
				  "%s: %d: Bad line: %.30s..."),
			  filename,lineno,buf);

		goto fail_item;
	    }
	    *c3 = '\0';
	
	    c3++;

	    while (*c3 && whitespace (*c3)) /* skip leading whitespace */
		c3++;
	    if (!*c3) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
				  "%s: %d: Bad line: %.30s..."),
			  filename,lineno,buf);

		goto fail_item;
	    }

	    item.value.iso2022_like.charset = 
		MIME_name_to_charset(c2,0); 
	    if (!item.value.iso2022_like.charset) {
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeUnknownTerminalCharset,
				  "Unknown terminal charset %s (file %s)"),
			  c2,filename);
		goto fail_item;
	    }

	    if (';' == *c3)
		c3++;

	    for (opt = mime_parse_content_opts(c3, &WALK); 
		 opt; 
		 opt = mime_parse_content_opts(NULL, &WALK)) {
		
		char * q = strchr(opt,'=');
		char *val; 

		if (!q) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
				      "%s: %d: Bad line: %.30s..."),
			      filename,lineno,buf);

		    goto fail_item;
		}
		*q++ = '\0';
		val = dequote_opt(q,strlen(q));
		
		if (! parse_iso2022_specification(opt,val,
						  &setcount,
						  &new_setlist)) {
		    
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
				      "%s: %d: Bad line: %.30s..."),
			      filename,lineno,buf);
		    
		    free(val); val = NULL;
		    goto fail_item;
		}

		free(val);
	    }
	    
	    if (0 == setcount) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
				  "%s: %d: Bad line: %.30s..."),
			  filename,lineno,buf);
		
		goto fail_item;
	    }
		
	    item.value.iso2022_like.iso2022_info =
		loc_setlist(new_setlist);
	    item_ok = 1;

	}
	    break;
	case terminal_private:

	    c3 = strpbrk(c2," \t");

	    if (!c3) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
				  "%s: %d: Bad line: %.30s..."),
			  filename,lineno,buf);

		goto fail_item;
	    }
	    *c3 = '\0';
	
	    c3++;

	    while (*c3 && whitespace (*c3)) /* skip leading whitespace */
		c3++;
	    if (!*c3) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
				  "%s: %d: Bad line: %.30s..."),
			  filename,lineno,buf);

		goto fail_item;
	    }
	    item.value.private.condition         = NULL;

	    if ('[' == *c3) {

		item.value.private.condition = parse_condition(&c3);
		if (!item.value.private.condition)
		    goto err_3;
		
		/* skip leading whitespace */
		while (*c3 && whitespace (*c3)) 
		    c3++;
		if (!*c3) {
		err_3:
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
				      "%s: %d: Bad line: %.30s..."),
			      filename,lineno,buf);
		    
		    goto  error_private;
		}
	    }

	    item.value.private.charset = 
		MIME_name_to_charset(c2,0); 
	    if (!item.value.private.charset) {
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeUnknownTerminalCharset,
				  "Unknown terminal charset %s (file %s)"),
			  c2,filename);

		goto  error_private;
	    }
	    size = strlen(c3);
	    item.value.private.bytes = safe_malloc(size+1);

	    if (!parse_gen_spec(c3,item.
				value.private.bytes,
				size)) {

		free(item.value.private.bytes);
		item.value.private.bytes = NULL;
		
	    error_private:
		if (item.value.private.condition)
		    free_term_cond(& (item.value.private.condition));
		
		goto fail_item;
	    }
	    item.value.private.bytes[size] = '\0';
	    item_ok = 1;
	    break;
	case terminal_flag:
	    
	    item.value.flag    = terminal_bad_flag;
	    
	    for (i = 0; 
		 i < sizeof terminal_flag_list / sizeof (terminal_flag_list[0]);
		 i++) {
		if (0 == strcmp(terminal_flag_list[i].flag,c2)) {
		    item.value.flag   = 
			terminal_flag_list[i].flag_value;
		    goto ok1;
		}
	    }

	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeUnknownTerminalFlag,
			      "Unknown terminal flag %s (file %s)"),
		      c2,filename);
	    goto fail_item;
	    
	ok1:
	    item_ok = 1;
	    break;

	case terminal_ansi_sgr:
	    item.value.ansi_sgr.condition         = NULL;

	    c3 = qstrpbrk(c2,"[");  
	    if (c3 && '[' == *c3) {
		char * ptr = c3;

		item.value.ansi_sgr.condition = parse_condition(&c3);
		if (!item.value.ansi_sgr.condition)
		    goto err_6;
		
		while (*c3 && whitespace (*c3)) /* skip leading whitespace */
		    c3++;
		if (*c3) {
		err_6:
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
				      "%s: %d: Bad line: %.30s..."),
			      filename,lineno,buf);

		    goto error_ansi_sgr;
		}

		/* Remove trailing whitespace */
		
		while (ptr > c2 && whitespace(*(ptr-1)))
		    ptr--;
		
		*ptr = '\0';
	    }

	    
	    for (x = 0; x < NUM_ANSI_sgr_values; x++) {
		if (0 == strcmp(ANSI_sgr_values[x].name,c2)) {
		    
		    item.value.ansi_sgr.ANSI_sgr_index = x;
		    goto ok2;
		}
	    }

	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeUnknownTerminalAttribute,
			      "Unknown terminal attribute %s (file %s)"),
		      c2,filename);

	error_ansi_sgr:
	    if (item.value.ansi_sgr.condition)
		free_term_cond(& (item.value.ansi_sgr.condition));
	    
	    goto fail_item;
	    
	ok2:
	    item_ok = 1;
	    break;
	    
	case terminal_bad:
	    /* Not used */
	    break;
	}
	
	if (item_ok) {
	    item.keyword = kw;
	    item.match = safe_strdup(c);
	    
	    result[result_len] = item;
	    result_len++;
	} else {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
			      "%s: %d: Bad line: %.30s..."),
		      filename,lineno,buf);
	    
	fail_item:
	    (*errors) ++;
	}
    }

    if (ferror(f)) {
	err = errno;

	lib_error(CATGETS(elm_msg_cat, MeSet, MeReadError,
			  "%.30s: Reading error: %s"),
		  filename,strerror(err));
	(*errors) ++;
    }

 OUT:
    if (buf) {
	free(buf);
    }

    result[result_len].match   = NULL;
    result[result_len].keyword = terminal_bad;


    if (!feof(f)) {
	DPRINT(Debug,3,(&Debug,
			"load_terminal_map: %s, All results not readed\n",
			filename));
    }

    fclose(f);

    

    DPRINT(Debug,10,(&Debug,
		     "load_terminal_map: %s, result_len=%d\n",
		     filename,result_len));

    return result;
}

static int name_ok P_((const char *name));
static int name_ok(name)
     const char *name;
{
    if ('\0' == *name)
	return 0;

    return strlen(name) == strspn(name,"ABCDEFGHIJKLMNOPQRSTUVXYZ-0123456789");
}

void dump_terminal_map(f,map,commentfile,actor,version_buff)
     FILE *f; 
     struct terminal_map_item *map;
     FILE *commentfile;
     const char *actor;
     char *version_buff;
{
    struct terminal_map_item *ptr;
    
    if (!map)
	return;

    insert_commentfile(f,ELMTERMINALINFO_INFO,
		       commentfile,actor,version_buff);

    
    for (ptr = map; ptr && ptr->match; ptr++) {
	switch(ptr->keyword) {
	    char *s;
	    int i;

	case terminal_dw:
	    fputs(ptr->match,f);
	    fputc('\t',f);
	    fputs("DW",f);
	    fputc('\t',f);
	    fputs(ptr->value.dw.charset,f);

	    if (ptr->value.dw.condition) {
		fputc('\t',f);
		dump_condition(f,ptr->value.dw.condition);
	    }

	    break;

	case terminal_iso2022:
	case terminal_iso2022_mb:
	    fputs(ptr->match,f);
	    fputc('\t',f);
	    if (terminal_iso2022_mb == ptr->keyword)
		fputs("ISO-2022/DW",f);
	    else
		fputs("ISO-2022",f);
	    fputc('\t',f);
	    if (ptr->value.iso2022.is_def)
		dump_map_info(f,ptr->value.iso2022.charset);
	    else {
		if (!ptr->value.iso2022.charset->MIME_name) {
		    fputs("# <No MIME name> ",f);
		} else if (name_ok(ptr->value.iso2022.charset->MIME_name)) {
		    fputs(ptr->value.iso2022.charset->MIME_name,f);
		} else {
		    elm_fprintf(f,FRM("%Q;!"),
				ptr->value.iso2022.charset->MIME_name);
		}
	    }

	    if (ptr->value.iso2022.condition) {
		fputc('\t',f);
		dump_condition(f,ptr->value.iso2022.condition);
	    }

	    break;

	case terminal_iso2022_like:
	    fputs(ptr->match,f);
	    fputc('\t',f);
	    fputs("ISO-2022-LIKE",f);
	    fputc('\t',f);
	    fputs(ptr->value.iso2022_like.charset->MIME_name,f);
	    fputc('\t',f);
	    print_setlist(f,ptr->value.iso2022_like.iso2022_info);
	    break;

	case terminal_private:
	    fputs(ptr->match,f);
	    fputc('\t',f);
	    fputs("PRIVATE",f);
	    fputc('\t',f);
	    fputs(ptr->value.private.charset->MIME_name,f);
	    if (ptr->value.private.condition) {
		fputc('\t',f);
		dump_condition(f,ptr->value.private.condition);
	    }
	    fputc('\t',f);
	    s = iso2022_codestr(ptr->value.private.bytes,
				strlen(sStr(ptr->value.private.bytes)));
	    if (s) {
		fputs(s,f);
		free(s);
	    }
	    break;
	case terminal_flag:
	    fputs(ptr->match,f);
	    fputc('\t',f);
	    fputs("FLAG",f);
	    fputc('\t',f);
	    
	    for (i = 0; 
		 i < sizeof terminal_flag_list / sizeof (terminal_flag_list[0]);
		 i++) {
		if (ptr->value.flag ==  terminal_flag_list[i].flag_value) {
		    fputs(terminal_flag_list[i].flag,f);
		    break;
		}
	    }
	    break;

	case terminal_ansi_sgr:
	    fputs(ptr->match,f);
	    fputc('\t',f);
	    fputs("ANSI-SGR",f);
	    fputc('\t',f);

	    if (ptr->value.ansi_sgr.ANSI_sgr_index >= 0 &&
		ptr->value.ansi_sgr.ANSI_sgr_index <= NUM_ANSI_sgr_values)
		fputs(ANSI_sgr_values[ptr->value.ansi_sgr.ANSI_sgr_index].name,f);

	    if (ptr->value.ansi_sgr.condition) {
		fputc('\t',f);
		dump_condition(f,ptr->value.ansi_sgr.condition);
	    }
	    break;
	    
	case terminal_bad:
	    /* Not used */
	    break;
	}
	fputc('\n',f);
    }
}

static int match_item P_((struct terminal_map_item *a, 
			  struct terminal_map_item *b));
static int match_item(a,b)
     struct terminal_map_item *a;
     struct terminal_map_item*b;
{
    if (a->keyword != b->keyword)
	return 0;
    if (0 != strcmp(a->match,b->match))
	return 0;

    switch(a->keyword) {

    case terminal_dw:
	if (a->value.dw.condition &&
	    b->value.dw.condition) {
	    if (! match_term_cond(a->value.dw.condition,
				  b->value.dw.condition))
		return 0;
	} else if (a->value.dw.condition ||
		   b->value.dw.condition)
	    return 0;



	if (0 != strcmp(a->value.dw.charset,
			a->value.dw.charset))
	    return 0;
	return 1;

    case terminal_iso2022:
    case terminal_iso2022_mb:
	
	if (a->value.iso2022.is_def !=
	    b->value.iso2022.is_def)
	    return 0;

	if (a->value.iso2022.condition &&
	    b->value.iso2022.condition) {
	    if (! match_term_cond(a->value.iso2022.condition,
				  b->value.iso2022.condition))
		return 0;
	} else if (a->value.iso2022.condition ||
		   b->value.iso2022.condition)
	    return 0;

	if (a->value.iso2022.charset ==
	    b->value.iso2022.charset)
	    return 1;

	if (a->value.iso2022.charset->MIME_name &&
	    b->value.iso2022.charset->MIME_name) {
	    if (0 == strcmp(a->value.iso2022.charset->MIME_name,
			    b->value.iso2022.charset->MIME_name))
		return 1;
	    else
		return 0;
	}

	if (a->value.iso2022.charset->codeset &&
	    b->value.iso2022.charset->codeset &&
	    0 == strcmp(a->value.iso2022.charset->codeset,
			b->value.iso2022.charset->codeset))
	    return 1;
	return 0;

    case terminal_iso2022_like:

	if (a->value.iso2022_like.charset !=
	    b->value.iso2022_like.charset)
	    return 0;

	if (a->value.iso2022_like.iso2022_info !=
	    b->value.iso2022_like.iso2022_info)
	    return 0;
	return 1;

    case terminal_private:
	if (a->value.private.charset !=
	    b->value.private.charset)
	    return 0;

	if (a->value.private.condition &&
	    b->value.private.condition) {
	    if (! match_term_cond(a->value.private.condition,
				  b->value.private.condition))
		return 0;
	} else if (a->value.private.condition ||
		   b->value.private.condition)
	    return 0;
	
	return 1;

    case terminal_flag:	
	if (a->value.flag != b->value.flag)
	    return 0;
	return 1;       

    case terminal_ansi_sgr:
	if (a->value.ansi_sgr.ANSI_sgr_index !=
	    b->value.ansi_sgr.ANSI_sgr_index)
	    return 0;

	if (a->value.ansi_sgr.condition &&
	    b->value.ansi_sgr.condition) {
	    if (! match_term_cond(a->value.ansi_sgr.condition,
				  b->value.ansi_sgr.condition))
		return 0;
	} else if (a->value.ansi_sgr.condition ||
		   b->value.ansi_sgr.condition)
	    return 0;
		
	return 1;
	
    case terminal_bad:
	/* Not used */
	break;
    }

    panic("CHARSET PANIC",__FILE__,__LINE__,"match_item",
	  "Unsupported type",0);


    return 0;
}

void change_terminal_map(map,new) 
     struct terminal_map_item **map;
     struct terminal_map_item *new;
{
    int count = 0;

    struct terminal_map_item *ptr;

    for (ptr = *map; ptr && ptr->match; ptr++)
	count++;

    for (ptr = new; ptr && ptr->match; ptr++)
	count++;

    if (!count)
	return;

    if (!*map) {
	*map = safe_calloc((count+1), sizeof ((*map)[0]));
	(*map)->match   = NULL;
	
    } else
	*map = safe_array_realloc(*map, (count+1), sizeof ((*map)[0]));

   for (ptr = new; ptr && ptr->match; ptr++) {
	struct terminal_map_item *ptr2;

	for (ptr2 = *map; ptr2->match; ptr2++) {
	    if (match_item(ptr,ptr2)) {
		goto set_it;
	    }
	}

	if (ptr2 > (*map) + count)
	    panic("CHARSET PANIC",__FILE__,__LINE__,"change_terminal_map",
		  "Overflow",0);
	
	ptr2->match    = safe_strdup(ptr->match);
	ptr2->keyword  = ptr->keyword;

	bzero((void *)(& ptr2->value), sizeof (ptr2->value));

	(ptr2+1)->match   = NULL;
	


    set_it:
	
	switch(ptr2->keyword) {
	case terminal_dw:
	    ptr2->value.dw.charset = 
		strmcpy(ptr2->value.dw.charset,
			ptr->value.dw.charset);
	    
	    if (ptr2->value.dw.condition)
		free_term_cond(& (ptr2->value.dw.condition));
	    
	    if (ptr->value.dw.condition)
		ptr2->value.dw.condition =
		    copy_term_cond(ptr->value.dw.condition);	   
	    break;
	    
	case terminal_iso2022:
	case terminal_iso2022_mb:

	    ptr2->value.iso2022.charset = ptr->value.iso2022.charset;
	    ptr2->value.iso2022.is_def  = ptr->value.iso2022.is_def;

	    if (ptr2->value.iso2022.condition)
		free_term_cond(& (ptr2->value.iso2022.condition));
	    
	    if (ptr->value.iso2022.condition)
		ptr2->value.iso2022.condition =
		    copy_term_cond(ptr->value.iso2022.condition);	   
	    break;

	case terminal_iso2022_like:
	    ptr2->value.iso2022_like = ptr->value.iso2022_like;
	    break;

	case terminal_private:
	    ptr2->value.private.charset = ptr->value.private.charset;
	    
	    ptr2->value.private.bytes =
		s2us(strmcpy(us2s(ptr2->value.private.bytes),
			     us2s(ptr->value.private.bytes)));
	    if (ptr2->value.private.condition)
		free_term_cond(& (ptr2->value.private.condition));
	    
	    if (ptr->value.private.condition)
		ptr2->value.private.condition =
		    copy_term_cond(ptr->value.private.condition);
	    
	    break;

	case terminal_flag:
	    ptr2->value.flag = ptr->value.flag;
	    break;

	case terminal_ansi_sgr:
	     ptr2->value.ansi_sgr.ANSI_sgr_index =
		 ptr->value.ansi_sgr.ANSI_sgr_index;

	     if (ptr2->value.ansi_sgr.condition)
		 free_term_cond(& (ptr2->value.ansi_sgr.condition));

	     if (ptr->value.ansi_sgr.condition)
		 ptr2->value.ansi_sgr.condition =
		     copy_term_cond(ptr->value.ansi_sgr.condition);
	     break;
	    
	default:
	    panic("CHARSET PANIC",__FILE__,__LINE__,"change_terminal_map",
		  "unsupported type",0);
		

	}	
    }

}

void free_terminal_map(map) 
     struct terminal_map_item **map;
{

    struct terminal_map_item *ptr;

    for (ptr = *map; ptr && ptr->match; ptr++) {

	if (ptr->match)
	    free(ptr->match);
	ptr->match = NULL;

	switch(ptr->keyword) {
	case terminal_dw:
	    if (ptr->value.dw.charset)
		free(ptr->value.dw.charset);

	    if (ptr->value.dw.condition)
		free_term_cond(& (ptr->value.dw.condition));


	    break;

	case terminal_iso2022:
	case terminal_iso2022_mb:
	    break;

	case terminal_iso2022_like:
	    break;
	    
	case terminal_private:
	    if (ptr->value.private.bytes)
		free(ptr->value.private.bytes);
	    ptr->value.private.bytes = NULL;

	    if (ptr->value.private.condition)
		free_term_cond(& (ptr->value.private.condition));
	    
	    break;

	case terminal_flag:
	    break;

	case terminal_ansi_sgr:
	    if (ptr->value.ansi_sgr.condition)
		free_term_cond(& (ptr->value.ansi_sgr.condition));
	    break;
	    
	default:
	    panic("CHARSET PANIC",__FILE__,__LINE__,"free_terminal_map",
		  "unsupported type",0);
	}
    }
    
    free(*map);
    *map = NULL;
}


/* ----------------------------------------------------------------- */

/* Hopefully quite safe to be called from signal handler */
static struct terminal_map_item * loc_info P_((const char *terminal, 
					       charset_t set,
					       int only_switch,
					       int maybe_signal));

static struct terminal_map_item * loc_info(terminal,set, only_switch, maybe_signal)
     const char *terminal; 
     charset_t set;
     int only_switch;
     int maybe_signal;
{
    int i;
    
    SIGDPRINT(Debug,9,(&Debug,"** Looking terminal %s (set %s)%s%s\n",
		       terminal,
		       set->MIME_name ? set->MIME_name : "<no MIME name>",
		       only_switch ? " charset switches only" : "",
		       maybe_signal ? " maybe called from signal handler" : ""
		       ));

    for (i =  0; i < 2; i++) {
	struct terminal_map_item *ptr = user_terminal_map;

	if (i)
	    ptr = system_terminal_map;

	while (ptr && ptr->match) {
	    char * c = NULL;

	    if (0 == strcmp(ptr->match,terminal) ||
		( (c = strchr(ptr->match,'*')) &&
		  *(c+1) == '\0' &&
		  0 == strncmp(ptr->match,terminal,c - ptr->match))) {


		if (ptr->keyword == terminal_iso2022 &&
		    match_charset_name(ptr->value.iso2022.charset,set)) {
		    
		    if (ptr->value.iso2022.condition) {
			if (maybe_signal) {
			    
			    SIGDPRINT(Debug,9,(&Debug,"-- Ignoring ISO2022 match %s, can't eval condition\n",
				       ptr->match));
			    goto no_this;
			}
			    
			/* eval_condition can not called from signal handler */
			
			if (!eval_condition(ptr->value.iso2022.condition)) {
			    DPRINT(Debug,9,(&Debug,"-- Ignoring ISO2022 match %s, condition does not match\n",
					    ptr->match));

			    goto no_this;
			}
		    }

		    SIGDPRINT(Debug,9,(&Debug,"-- Found ISO2022 match %s\n",
				    ptr->match));

		    return ptr;	    
		}

		if (ptr->keyword == terminal_iso2022_mb &&
		    match_charset_name(ptr->value.iso2022.charset,set)) {
		    
		    if (ptr->value.iso2022.condition) {
			if (maybe_signal) {
			    SIGDPRINT(Debug,9,(&Debug,"-- Ignoring ISO2022 MB match %s, can't eval condition\n",
					       ptr->match));

			    goto no_this;
			}
			    
			/* eval_condition can not called from signal handler */
			
			if (!eval_condition(ptr->value.iso2022.condition)) {
			    DPRINT(Debug,9,(&Debug,"-- Ignoring ISO2022 MB match %s, condition does not match\n",
					    ptr->match));
			    
			    goto no_this;
			}
		    }

		    SIGDPRINT(Debug,9,(&Debug,"-- Found ISO2022 MB match %s\n",
				       ptr->match));

		    return ptr;	    
		}

		if (ptr->keyword == terminal_iso2022_like &&
		    match_charset_name(ptr->value.iso2022_like.charset,set)) {
		    
		    SIGDPRINT(Debug,9,(&Debug,"-- Found ISO2022-LIKE match %s\n",
				       ptr->match));

		    return ptr;	    
		}

		if (ptr->keyword == terminal_private &&
		    match_charset_name(ptr->value.private.charset, set)) {
		    if (ptr->value.private.condition) {
			if (maybe_signal) {
			    SIGDPRINT(Debug,9,(&Debug,"-- Ignoring private match %s, can't eval condition\n",
					       ptr->match));

			    goto no_this;
			}
			/* eval_condition can not called from signal handler */

			
			if (!eval_condition(ptr->value.private.condition)) {
			    DPRINT(Debug,9,(&Debug,"-- Ignoring private match %s, condition does not match\n",
					    ptr->match));

			    goto no_this;
			}
		    }

		    SIGDPRINT(Debug,9,(&Debug,"-- Found private match %s\n",
				       ptr->match));

		    return ptr;	    
		}

		if (ptr->keyword == terminal_dw && set->MIME_name &&
		    0 == istrcmp(set->MIME_name,ptr->value.dw.charset)) {

		    if (ptr->value.dw.condition) {
			if (maybe_signal) {
			    SIGDPRINT(Debug,9,(&Debug,"-- Ignoring DW match %s, can't eval condition\n",
					       ptr->match));

			    goto no_this;
			}
			/* eval_condition can not called from signal handler */

			if (!eval_condition(ptr->value.dw.condition)) {
			    DPRINT(Debug,9,(&Debug,"-- Ignoring DW match %s, condition does not match\n",
					    ptr->match));

			    goto no_this;
			}
		    }

		    if (only_switch) {

			SIGDPRINT(Debug,9,(&Debug,"-- Found DW match %s (ignored)\n",
					   ptr->match));
			
			goto no_this;

		    }
		    SIGDPRINT(Debug,9,(&Debug,"-- Found DW match %s\n",
				       ptr->match));
			
		    return ptr;
		}

	    }
	no_this:
	    ptr++;
	}
    }
    return NULL;
}

void terminal_can_switch(terminal,storage,n,max,maybe_signal)
     const char *terminal;
     charset_t *storage;
     int *n;
     int max;
     int maybe_signal;
{

    int i;
    struct terminal_map_item * back;

    /* Can not switch to other sets if can not
       switch back to display_charset
    */
    back = loc_info(terminal,display_charset,1,maybe_signal);
    if (!back)
	return;


    for (i =  0; i < 2; i++) {
	struct terminal_map_item *ptr = user_terminal_map;
	
	if (i)
	    ptr = system_terminal_map;
	
	while (ptr && ptr->match && *n < max) {
	    char * c = NULL;
	    int j;

	    SIGDPRINT(Debug,9,(&Debug,"-- Looking %s ...\n",
			       ptr->match));

	    if (0 == strcmp(ptr->match,terminal) ||
		( (c = strchr(ptr->match,'*')) &&
		  *(c+1) == '\0' &&
		  0 == strncmp(ptr->match,terminal,c - ptr->match))) {
	
		switch (ptr->keyword) {
		case terminal_iso2022:
		case terminal_iso2022_mb:

		    if (ptr->value.iso2022.condition) {
			if (maybe_signal) {
			    SIGDPRINT(Debug,8,(&Debug,
					       "-- NOT adding %s (%p) to possible sets (can't eval condition)\n",
					       ptr->value.iso2022.charset->MIME_name ?
					       ptr->value.iso2022.charset->MIME_name :
					       "<no MIME name>",
					       ptr->value.iso2022.charset
					       ));

			    
			    goto no_this;
			}
			/* eval_condition can not called from signal handler */

			if (!eval_condition(ptr->value.iso2022.condition)) {
			    SIGDPRINT(Debug,8,(&Debug,
					       "-- NOT adding %s (%p) to possible sets (condition fail)\n",
					       ptr->value.iso2022.charset->MIME_name ?
					       ptr->value.iso2022.charset->MIME_name :
					       "<no MIME name>",
					       ptr->value.iso2022.charset
					       ));

			    goto no_this;
			}
		    }

		    SIGDPRINT(Debug,8,(&Debug,
				       "-- Adding %s (%p) to possible sets\n",
				       ptr->value.iso2022.charset->MIME_name ?
				       ptr->value.iso2022.charset->MIME_name :
				       "<no MIME name>",
				       ptr->value.iso2022.charset
				       ));
		    storage[(*n)++] = ptr->value.iso2022.charset;
		    break;

		case terminal_iso2022_like:
		    SIGDPRINT(Debug,8,(&Debug,
				       "-- Adding %s (%p) to possible sets\n",
				       ptr->value.iso2022_like.charset->MIME_name ?
				       ptr->value.iso2022_like.charset->MIME_name :
				       "<no MIME name>",
				       ptr->value.iso2022_like.charset
				       ));
		    storage[(*n)++] = ptr->value.iso2022_like.charset;
		    break;

		case terminal_private:
		    if (ptr->value.private.condition) {
			if (maybe_signal) {
			    SIGDPRINT(Debug,8,(&Debug,
					       "-- NOT adding %s (%p) to possible sets (can't eval condition)\n",
					       ptr->value.private.charset->MIME_name ?
					       ptr->value.private.charset->MIME_name :
					       "<no MIME name>",
					       ptr->value.private.charset
					       ));

			    goto no_this;
			}
			/* eval_condition can not called from signal handler */

			if (!eval_condition(ptr->value.private.condition)) {
			    SIGDPRINT(Debug,8,(&Debug,
					       "-- NOT adding %s (%p) to possible sets (condition fail)\n",
					       ptr->value.private.charset->MIME_name ?
					       ptr->value.private.charset->MIME_name :
					       "<no MIME name>",
					       ptr->value.private.charset
					       ));

			    goto no_this;
			}
		    }

		    SIGDPRINT(Debug,8,(&Debug,
				       "-- Adding %s (%p) to possible sets\n",
				       ptr->value.private.charset->MIME_name ?
				       ptr->value.private.charset->MIME_name :
				       "<no MIME name>",
				       ptr->value.private.charset
				       ));

		    storage[(*n)++] = ptr->value.private.charset;
		    break;

		case terminal_bad:
		default:
		    break;
		}

		for (j = 0; j < (*n) -1; j++)
		    if (storage[j] == storage[(*n)-1]) {
			SIGDPRINT(Debug,8,(&Debug,
					   "-- Removing duplicate %s from possible sets\n",
					   storage[(*n)-1]->MIME_name ?
					   storage[(*n)-1]->MIME_name :
					   "<no MIME name>"));
			(*n)--;  /* Remove duplicate */  
			break;
		    }
	    }
	no_this:
	    ptr++;
	}
    }
}


int terminal_can_switch_to(terminal,set,silent,maybe_signal)
     const char *terminal; 
     charset_t set;
     int silent;
     int maybe_signal;
{

    struct terminal_map_item * a = loc_info(terminal,set,1,
					    maybe_signal);

    if (!a) {
	if (!silent && !maybe_signal)
	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeNoInformationToSwitchTerminal,
			      "No information to switch %s terminal to %s charset!"), 
		      terminal,
		      set->MIME_name ? set->MIME_name : "<no MIME name>");
		      
	return 0;
    }
    
    a = loc_info(terminal,display_charset,1,
		 maybe_signal);

    if (!a) {
	if (!silent && !maybe_signal)
	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeNoInformationToSwitchBack,
			      "No information to switch %s terminal back to %s charset!"), 
		      terminal,
		      display_charset->MIME_name ? display_charset->MIME_name :
		      "<no MIME name>");
	return 0;
    }
    
    return 1;
}


/* Returns available pg_flags */

int ansi_sgr_attributes(terminal,vector,vector_len,result_len)
     char *terminal;
     const struct ansi_sgr_attribute **vector;
     size_t                      vector_len;
     size_t * result_len;
{
    int pg_flags = 0;

    int i;

    size_t idx = 0;

    DPRINT(Debug,9,(&Debug,"** Looking terminal %s (ansi sgr vector_len=%zu)\n",
		    terminal,vector_len));
    
    for (i =  0; i < 2; i++) {
	struct terminal_map_item *ptr = user_terminal_map;
	
	if (i)
	    ptr = system_terminal_map;
	
	while (ptr && ptr->match) {
	    char * c = NULL;

	    DPRINT(Debug,9,(&Debug,"-- Looking %s ...\n",
			    ptr->match));
	    
	    if (0 == strcmp(ptr->match,terminal) ||
		( (c = strchr(ptr->match,'*')) &&
		  *(c+1) == '\0' &&
		  0 == strncmp(ptr->match,terminal,c - ptr->match))) {

		if (ptr->keyword == terminal_ansi_sgr) {

		    if (ptr->value.ansi_sgr.condition) {

			if (!eval_condition(ptr->value.ansi_sgr.
					    condition)) {
			    DPRINT(Debug,10,(&Debug,
					     "-- Looking terminal %s - condition does not match\n",
					     terminal));
			    goto no_this;
			}
		    }

		    if (ptr->value.ansi_sgr.ANSI_sgr_index >= 0 &&
			ptr->value.ansi_sgr.ANSI_sgr_index <=
			NUM_ANSI_sgr_values)  {
					    
			if (isoff(pg_flags,
				  ANSI_sgr_values[ptr->value.ansi_sgr.ANSI_sgr_index].pg_flag)
			    ) {

			    if (idx < vector_len && vector)			    
				vector[idx++] =
				    &(ANSI_sgr_values[ptr->value.ansi_sgr.ANSI_sgr_index]);

			    setit(pg_flags,
				  ANSI_sgr_values[ptr->value.ansi_sgr.ANSI_sgr_index].pg_flag);

			    DPRINT(Debug,9,(&Debug,"-- Looking terminal %s - found %s\n",
					    terminal,
					    ANSI_sgr_values[ptr->value.ansi_sgr.ANSI_sgr_index].name));
			    
			} else {
			    DPRINT(Debug,9,(&Debug,"-- Looking terminal %s - found %s (duplicate)\n",
					    terminal,
					    ANSI_sgr_values[ptr->value.ansi_sgr.ANSI_sgr_index].name));

			}
		    }
		    
		}
	    }
	no_this:
	    ptr++;
	}
    }

    if (result_len)
	*result_len = idx;

    DPRINT(Debug,9,(&Debug,
		    "ansi_sgr_attributes=%d (%s)",
		    pg_flags,give_pg_flags(pg_flags)));
	   
    if (result_len) {
	DPRINT(Debug,9,(&Debug,
			", *result_len=%zu",
			*result_len));
    }
    DPRINT(Debug,9,(&Debug," -- terminal=%s\n",
		    terminal));
    	   
    return pg_flags;
}


/* Returns first flags found for terminal from list or terminal_bad_flag, 
   flags[] is terminated with terminal_bad_flag

   Hopefully quite safe to be called from signal handler 
*/
static enum terminal_flags find_terminal_flag P_((const char *terminal,
						  enum terminal_flags flags[]));

static enum terminal_flags find_terminal_flag(terminal,flags)
     const char *terminal;
     enum terminal_flags flags[];
{

    int i,j;

    SIGDPRINT(Debug,9,(&Debug,"** Looking terminal %s (flags",
		       terminal));
    for (j = 0; terminal_bad_flag != flags[j]; j++) {
	SIGDPRINT(Debug,9,(&Debug," %d",flags[j]));
    }
    SIGDPRINT(Debug,9,(&Debug,")\n"));
    
    for (i =  0; i < 2; i++) {
	struct terminal_map_item *ptr = user_terminal_map;

	if (i)
	    ptr = system_terminal_map;

	while (ptr && ptr->match) {
	    char * c = NULL;
	    
	    SIGDPRINT(Debug,9,(&Debug,"-- Looking %s ...\n",
			       ptr->match));
	    
	    if (0 == strcmp(ptr->match,terminal) ||
		( (c = strchr(ptr->match,'*')) &&
		  *(c+1) == '\0' &&
		  0 == strncmp(ptr->match,terminal,c - ptr->match))) {


		if (ptr->keyword == terminal_flag) {
		    int j;

		    for (j = 0; terminal_bad_flag != flags[j]; j++) {

			if (ptr->value.flag == flags[j]) {
			    SIGDPRINT(Debug,9,(&Debug,
					       "-- Found flag match %s (flag %d)\n",
					       ptr->match,flags[j]));

			    return flags[j];
			}

		    }		   
		}

	    }
	    ptr++;
	}
    }

    return terminal_bad_flag; 
}


/* May be called from signal handler -- on that situation use
   buffer specified as argument -- otherwise result is malloced
*/
char * terminal_set_title(terminal,window_title, icon_title,buffer,size)
     char *terminal; 
     char *window_title;
     char *icon_title;
     char * buffer; 
     int size;
{
    /* window_title and icon_title are assumed to be US-ASCII ... */

    int maybe_signal = buffer != NULL;

    char *ret = NULL;

    static enum terminal_flags methods[] = {
	terminal_xterm_title,
	terminal_xwsh_title,
	terminal_bad_flag
    };

    enum terminal_flags method = find_terminal_flag(terminal,methods);

    if (maybe_signal) {
	SIGDPRINT(Debug,8,(&Debug,
			   "terminal_set_title: buffer=%p size=%d\n",
			   buffer,size));
    }


    switch (method) {
    case terminal_xterm_title:
	if (window_title == icon_title) {  /* Optimize */

	    if (window_title) {
		strXcat(&ret,"\033]0;",
			maybe_signal,buffer,size);
		strXcat(&ret,window_title,
			maybe_signal,buffer,size);
		strXcat(&ret,"\007",
			maybe_signal,buffer,size);
	    }
	} else {
	    if (icon_title) {
		strXcat(&ret,"\033]1;",
			maybe_signal,buffer,size);
		strXcat(&ret,icon_title,
			maybe_signal,buffer,size);
		strXcat(&ret,"\007",
			maybe_signal,buffer,size);
	    }

	    if (window_title) {
		strXcat(&ret,"\033]2;",
			maybe_signal,buffer,size);
		strXcat(&ret,window_title,
			maybe_signal,buffer,size);
		strXcat(&ret,"\007",
			maybe_signal,buffer,size);
	    }

	}
	break;
    case terminal_xwsh_title:

	if (icon_title) {
	    strXcat(&ret,"\033P3.y",
		    maybe_signal,buffer,size);
	    strXcat(&ret,icon_title,
		    maybe_signal,buffer,size);
	    strXcat(&ret,"\033\\",
		    maybe_signal,buffer,size);
	}

	if (window_title) {
	    strXcat(&ret,"\033P1.y",
		    maybe_signal,buffer,size);
	    strXcat(&ret,window_title,
		    maybe_signal,buffer,size);
	    strXcat(&ret,"\033\\",
		    maybe_signal,buffer,size);
	}

    case terminal_bad_flag:
	SIGDPRINT(Debug,8,(&Debug,
			   "terminal_set_title: No way to set title for %s\n",
			   terminal));
	break;
    }


    if (maybe_signal && ret && ret != buffer)
	panic("STRING PANIC",__FILE__,__LINE__,
	      "terminal_set_title",
	      "buffer != ret when called from signal",1);
    
    return ret;
}


void terminal_change_system_charset(terminal,have_DW_characters)
     char *terminal; 
     int have_DW_characters;
{
#if defined(WCHAR) && defined(__STDC_ISO_10646__)
    struct terminal_map_item *    a = loc_info(terminal,system_charset,0,
					       0 /* Not from signal handler */);
#endif

    if (system_charset->charset_type == &cs_unknown) {
	DPRINT(Debug,9,(&Debug,"terminal_change_system_charset: System charset is already type unknown\n"));
	return;
    }

#if defined(WCHAR) && defined(__STDC_ISO_10646__) && defined(WCWIDTH) 
    if ((a && a->keyword == terminal_dw) || 
	have_DW_characters ) {
	change_system_charset_1();
	return;
    }
#endif
}

static void change_helper_2 P_((char **ret,struct iso2022_setid ID,
				screen_info_p terminal_info,   
				char * buffer,
				int size, 
				int static_buffer,
				int setpos
				));

static void change_helper_2(ret,ID,terminal_info,buffer,size,static_buffer,setpos)
     char **ret;
     struct iso2022_setid ID;
     screen_info_p terminal_info;   
     char * buffer; 
     int size;
     int static_buffer;
     int setpos;
{   
    if (static_buffer) { 
	if (!*ret)
	    *ret = 
		iso2022_change_helper_1(terminal_info,ID,
					setpos, 
					buffer,size);
	else {
	    char buffer1[100];
	    char *c = 
		iso2022_change_helper_1(terminal_info,ID,
					setpos, 
					buffer1,sizeof buffer1);
	    if (c) 
		*ret = strfcat(buffer,c,size);
	}

    } else {
	char *c = 
	    iso2022_change_helper_1(terminal_info,ID,
				    setpos, 
				    NULL,0);
	if (c) {
	    *ret = strmcat(*ret,c);
	    free(c);
	}
    }
}


static void change_helper_3 P_((char **RET,struct setlist * iso2022_info,
				screen_info_p terminal_info,   
				char * buffer,
				int size, 
				int static_buffer,
				int MB));

static void change_helper_3(RET,iso2022_info,terminal_info,buffer,size,
			    static_buffer,MB)
     char **RET;
     struct setlist * iso2022_info;
     screen_info_p terminal_info;   
     char * buffer;
     int size; 
     int static_buffer;
     int MB;
{
    char * ret = *RET;
    int i;    
	    
    if (bank_unspecified != iso2022_info->initial_L) {
	terminal_info->current_L = iso2022_info->initial_L;

	if (static_buffer) {
	    if (!ret)
		ret = lock_shift(0,terminal_info->current_L,buffer,
				 size);
	    else {
		char buffer1[10];
		char * c = lock_shift(0,terminal_info->current_L,
				      buffer1,sizeof buffer1);
		ret = strfcat(buffer,c,size);
	    }
		    
	} else {
	    char *c = lock_shift(0,terminal_info->current_L,NULL,0);
	    ret = strmcat(ret,c);
	    free(c);
	}	       
    }
    
    if (bank_unspecified != iso2022_info->initial_R) {
	terminal_info->current_R = iso2022_info->initial_R;
		
	if (static_buffer) {
	    if (!ret)
		ret = lock_shift(1,terminal_info->current_R,
				 buffer,size);
	    else {
		char buffer1[10];
		char *c =  lock_shift(1,terminal_info->current_R,
				      buffer1,sizeof buffer1);
		ret = strfcpy(buffer,c,size);
	    }
	} else {
	    char *c = lock_shift(1,terminal_info->current_R,NULL,0);
	    ret = strmcat(ret,c);
	    free(c);
	}
    }
	    
    for (i = 0; 
	 i < sizeof (iso2022_info->initial_bank) / 
	     sizeof (iso2022_info->initial_bank[0]);
	 i++) {
	int x = iso2022_info->initial_bank[i];
		
	if (-1 != x) {
	    int n;
	    struct iso2022_setid ID;

	    if (x < 0 ||
		x >= sizeof (iso2022_info->sets)  /
			sizeof (iso2022_info->sets[0]) ||
			! iso2022_info->sets[x]) 
		panic("STRING PANIC",__FILE__,__LINE__,
		      "change_helper_3",
		      "Bad initial_bank (set)",
		      static_buffer);
	       
	    ID = * (iso2022_info->sets[x]);
		    
	    if (ID.bank != i)
		panic("STRING PANIC",__FILE__,__LINE__,
		      "change_helper_3",
		      "Bad initial_bank (bank number)",
		      static_buffer);
		    
	    n = set_initial_bank(&ret,ID,terminal_info,buffer,size,
				 static_buffer);

	    /* Do not support variable width inside of bank */
	    if (MB &&
		(iso2022_94x94 == ID.type ||
		 iso2022_96x96 == ID.type))
		terminal_info->width[n]  = 2;
	    else
		terminal_info->width[n]  = 1;	  
	    
	}
    }

#ifdef WCHAR
    terminal_info->wcwidth       = MB;
#endif

    for (i = 0; 
	 i < sizeof (iso2022_info->sets)  / 
	     sizeof (iso2022_info->sets[0]) &&
	     iso2022_info->sets[i];
	 i++) {
	int setpos;
	struct iso2022_setid ID = * (iso2022_info->sets[i]);

	/* iso2022_give_setpos does iso2022_setid_select_bank() */
	setpos = iso2022_give_setpos(&ID,terminal_info);

	if (-1 == setpos) {
	    if ((terminal_info)->set_count >=
			sizeof ((terminal_info)->sets) /
			sizeof ((terminal_info)->sets[0]))
			panic("STRING PANIC",__FILE__,__LINE__,
			      "change_helper_3",
			      "Too many iso 2022 sets on display charset",
			      static_buffer);

		    setpos = terminal_info->set_count++;
		    terminal_info->sets[setpos] = ID;
	}

	change_helper_2(&ret,ID,terminal_info,buffer,size,
			static_buffer,setpos);
		
	/* Do not support variable width inside of bank */
	if (MB &&
	    (iso2022_94x94 == ID.type ||
	     iso2022_96x96 == ID.type))
	    terminal_info->width[setpos]  = 2;
	else
	    terminal_info->width[setpos]  = 1;
    }

    debug_display_settings(terminal_info,static_buffer,0);
	   
    *RET = ret;
}



char * terminal_set_info (terminal,set,terminal_info,have_DW_characters)
     char *terminal; 
     charset_t set;
     screen_info_p terminal_info;   
     int have_DW_characters;
{
    char * ret = NULL;
    struct terminal_map_item * a;
    int dw_set = 0;

    static int warned = 0;

    if (!terminal_info) {
	panic("STRING PANIC",__FILE__,__LINE__,
	      "terminal_set_info",
	      "terminal_info not set",0);
    } 
    
    if (DISPLAY_STATE_magic != terminal_info->magic)
	panic("STRING PANIC",__FILE__,__LINE__,
	      "terminal_set_info",
	      "Bad terminal_info (bad magic)",
	      0);

    /* We do not reset bank settings on here */
#ifdef WCHAR
    terminal_info->wcwidth             = 0;
#endif

    a = loc_info(terminal,set,0,
		 0 /* Not from signal handler */);

    if (a) {
	DPRINT(Debug,8,(&Debug,
			"terminal_set_info: Info found for terminal %s\n",
			terminal));

	switch (a->keyword) {
	case terminal_dw: {
	    
#ifdef WCHAR
	    if (&cs_unknown  != set->charset_type ||
		0 == (CS_printable_len & charset_properties(set))) {
		
		if (!warned)
		    lib_error(CATGETS(elm_msg_cat, MeSet, 
				      MeUnsupportedDW,
				      "Double wide characters are not supported"));
		
		warned = 1;

		DPRINT(Debug,8,(&Debug,
				"terminal_set_info: Failure\n"));
		return NULL;
	    }
	    
	    terminal_info->wcwidth             = 1;
	    dw_set = 1;
	    
#else
	    
	    if (!warned)
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeUnsupportedDW,
				  "Double wide characters are not supported"));
	    
	    warned = 1;	    
#endif	   
	}
	    break;
	    
	case terminal_iso2022:
	
	    if (have_DW_characters) {
		DPRINT(Debug,8,(&Debug,
				"terminal_set_info: ISO-2022, --have-double-wide-characters set -- treating like ISO-2022/DW\n"));
	    } else 
		break;
	    
	    /* FALLTHRU */
	
	case terminal_iso2022_mb: {
	    static int warned = 0;
	    
	    if (0 == (CS_printable_len & charset_properties(set))) {
		
		if (!warned)
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeUnsupportedDW,
				  "Double wide characters are not supported"));
	    
	    warned = 1;

	    DPRINT(Debug,8,(&Debug,
			    "terminal_set_info: Failure\n"));
	    return ret;
	}
	    
	    if (set->iso2022_info) {
		change_helper_3(&ret,set->iso2022_info,terminal_info,NULL,0,0,1);
		
		dw_set = 1;
	    }
	    
#ifdef WCHAR
	    terminal_info->wcwidth             = 1;
#endif
	}
	    break;
	    
	default:
	    break;
	    
	}
    }

#ifdef WCHAR
    if (&cs_unknown  == set->charset_type &&
	0 != (CS_printable_len & charset_properties(set)) &&
	have_DW_characters) {

	DPRINT(Debug,8,(&Debug,
			"terminal_set_info: --have-double-wide-characters set -- OK\n"));
	
	terminal_info->wcwidth             = 1;
	dw_set = 1;
    }


    DPRINT(Debug,8,(&Debug,
		       "terminal_set_info: wcwidth=%d dw_set=%d\n",
		    terminal_info->wcwidth,dw_set));
#endif    

    if (0 != (CS_printable_len & charset_properties(set)) && dw_set) {
	DPRINT(Debug,8,(&Debug,
			"terminal_set_info: Double wide characters supported\n"));
    } else if (have_DW_characters) {
	DPRINT(Debug,8,(&Debug,
			"terminal_set_info: Double wide characters not supported and --have-double-wide-characters set\n"));

	if (!warned)
	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeUnsupportedDW,
			      "Double wide characters are not supported"));
	
	warned = 1;
    }

    if (!ret) {
	DPRINT(Debug,8,(&Debug,
			"terminal_set_info: No initialization string\n"));
    }


    return ret;
}

/* May be called from signal handler -- on that situation use
   buffer specified as argument -- otherwise result is malloced
*/
char * terminal_switch_to(terminal,set,terminal_info,buffer,size,silent,maybe_signal)
     char *terminal; 
     charset_t set;
     screen_info_p terminal_info;   
     char * buffer; 
     int size;
     int silent;
     int maybe_signal;
{
    int static_buffer = buffer != NULL;

    char *ret = NULL;
    struct terminal_map_item * a = loc_info(terminal,set,1,maybe_signal);

    if (!terminal_info) {
	panic("STRING PANIC",__FILE__,__LINE__,
	      "terminal_switch_to",
	      "terminal_info not set",maybe_signal);
    } 
    
    if (DISPLAY_STATE_magic != (terminal_info)->magic)
	panic("STRING PANIC",__FILE__,__LINE__,
	      "terminal_switch_to",
	      "Bad terminal_info (bad magic)",
	      maybe_signal);
       
    if (!a) {
	return NULL;
    }

    if (maybe_signal) {

	if (!buffer)
	    panic("STRING PANIC",__FILE__,__LINE__,
		  "terminal_switch_to",
		  "Buffer not set",
		  maybe_signal);
	
	SIGDPRINT(Debug,8,(&Debug,
			   "terminal_switch_to: buffer=%p size=%d\n",
			   buffer,size));
    }

    if ((terminal_info)->set_count > 0 &&     
	(terminal_info)->sets[0].type == iso2022_other) {
	SIGDPRINT(Debug,8,(&Debug,
			   "Terminal was on no ISO2022 mode ... returning\n"));
	if (static_buffer)
	    ret = iso2022_setid_stream(return_to_iso2022,buffer,size);
	else
	    ret = iso2022_setid_stream(return_to_iso2022,NULL,0);
    }
    
    reset_display_settings(terminal_info);

    switch(a->keyword) {
	int mb;
    case terminal_iso2022:
    case terminal_iso2022_mb:
	if (set->iso2022_info) {
	    SIGDPRINT(Debug,8,(&Debug,
			       "Using ISO 2022 code for assignment\n"));
	    
	    mb = 0;
	    
	    if (a->keyword == terminal_iso2022_mb) {	    
		if (0 != (CS_printable_len & charset_properties(set)))
		    mb = 1;
		else if (!silent && !maybe_signal) {		    
		    static int warned = 0;
		    
		    if (!warned)
			lib_error(CATGETS(elm_msg_cat, MeSet, 
					  MeUnsupportedDW,
					  "Double wide characters are not supported"));
		    
		    warned = 1;
		} 
	    }
	    
	    change_helper_3(&ret,set->iso2022_info,terminal_info,
			    buffer,size,static_buffer,
			    mb);
	} else {
	    SIGDPRINT(Debug,1,(&Debug,
			       "No ISO 2022 code for changing charset!\n"));

	}
	break;
    
    case terminal_iso2022_like:

	SIGDPRINT(Debug,8,(&Debug,
			   "Using ISO 2022 LIKE code for assignment\n"));
	
	change_helper_3(&ret,a->value.iso2022_like.iso2022_info,
			terminal_info,
			buffer,size,static_buffer,0);
	
	break;

    case terminal_private:
	SIGDPRINT(Debug,8,(&Debug,
			   "Using terminal private code for assignment\n"));
	
	
	strXcat(&ret,sStr(a->value.private.bytes),
		static_buffer,buffer,size);

	break;
	
    case terminal_bad:
    default:
	break;
    }

#ifdef WCHAR
    SIGDPRINT(Debug,8,(&Debug,
		       "terminal_switch_to: wcwidth=%d\n",
		       (terminal_info)->wcwidth));
#endif

    if (maybe_signal && ret && ret != buffer)
	panic("STRING PANIC",__FILE__,__LINE__,
	      "terminal_switch_to",
	      "buffer != ret when called from signal",1);
    

    if (!ret) {
	SIGDPRINT(Debug,8,(&Debug,
			   "terminal_switch_to: No switch sequence returned.\n"));
    }

    return ret;
}



/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */

