static char rcsid[] = "@(#)$Id: rc_delay.c,v 2.12 2018/12/21 11:21:42 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.12 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> (was hurtta+elm@posti.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "elm_defs.h"
#include "rc_imp.h"
#include "save_opts.h"
#include "s_elmrc.h"

DEBUG_VAR(Debug,__FILE__,"config");

static unsigned char * s2us P_((char *str)) UNUSED_FUNCOK;
static unsigned char * s2us(str)
     char *str;
{
    return (unsigned char *)str;
}

struct rec1 {
    enum record_mode         lcl;
    struct string          * value; 
    char                   * get_value_buffer;
    int                      lineno; 
    char                   * filename;
    enum rec_line_mode       negate_continue;
};

static void replay_rec P_((struct rec1 *rec,
			   struct rc_save_info_rec * real,
			   int read_flags /* READ_FLAG_IGNORE_MISSING */));
static void replay_rec(rec,real,read_flags)
     struct rec1 *rec;
     struct rc_save_info_rec * real;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    rc_eval_tail_line(real,rec->lcl,rec->value,rec->lineno,rec->filename,
		      rec->negate_continue,
		      read_flags /* READ_FLAG_IGNORE_MISSING */);
}

#define RC_SAVE_magic		0xF408

static struct rc_save {
    unsigned short magic;     /* RC_SAVE_magic */
    
    struct rc_save_info_rec * real;        
    struct rc_save_info_rec * delayed;  

    struct rec1  * lines;
    size_t         line_count;

    struct rc_save  * next;
} * root = NULL;

static void add_line_rec P_((struct rc_save   * p,
			     /* shares pointers */
			     enum record_mode   lcl,
			     struct string    * value,
			     int                lineno, 
			     char             * filename,
			     enum rec_line_mode negate_continue));
static void add_line_rec(p,lcl,value,lineno,filename,negate_continue)
     struct rc_save      * p;
     /* shares pointers, caller must alloc copy */
     enum record_mode      lcl;
     struct string       * value;
     int                   lineno;
     char                * filename;
     enum rec_line_mode    negate_continue;
{
    if (RC_SAVE_magic != p->magic)
	panic("RC PANIC",__FILE__,__LINE__,"add_line_rec",
	      "Bad magic number (rc_save)",0);

    p->lines = safe_array_realloc(p->lines,
				  (p->line_count + 1),
				  sizeof (p->lines[0]));
    
    bzero((void *)& (p->lines[p->line_count]), 
	      sizeof (p->lines[p->line_count]) );

    p->lines[p->line_count].lcl              = lcl;
    p->lines[p->line_count].get_value_buffer = NULL;
    p->lines[p->line_count].value            = value;
    p->lines[p->line_count].lineno           = lineno;
    p->lines[p->line_count].filename         = filename;
    p->lines[p->line_count].negate_continue  = negate_continue;
    p->line_count++;
}


void  mark_delayed_changed(A, no_history_record)
     void *A;
     int no_history_record;
{
    struct rc_save *x;

    for (x = root; x; x = x->next) {

	if (RC_SAVE_magic != x->magic)
	    panic("RC PANIC",__FILE__,__LINE__,"mark_delayed_changed",
		  "Bad magic number (rc_save)",0);
	
	if (x->real && x->real->val.dummy == A) {

	    DPRINT(Debug,10,(&Debug,
			     "mark_delayed_changed: %p: %p\n",
			     x,A));

	    mark_XX(x->real,no_history_record);

	    if (x->delayed)
		mark_XX(x->delayed, no_history_record);
	    else {
		DPRINT(Debug,10,(&Debug,
				 "mark_delayed_changed: %p: %p -- no delayed registered\n",
				 x,A));
		
	    }
	}
    }
}

void  mark_fdelayed_changed(A,no_history_record)
     option_func *A;
     int no_history_record;
{
    struct rc_save *x;

    for (x = root; x; x = x->next) {

	if (RC_SAVE_magic != x->magic)
	    panic("RC PANIC",__FILE__,__LINE__,"mark_fdelayed_changed",
		  "Bad magic number (rc_save)",0);
	
	if (x->real && x->real->func_val == A) {

	    DPRINT(Debug,10,(&Debug,
			     "mark_fdelayed_changed: %p: %p\n",
			     x,A));

	    mark_XX(x->real,no_history_record);

	    if (x->delayed)
		mark_XX(x->delayed, no_history_record);
	    else {
		DPRINT(Debug,10,(&Debug,
				 "mark_fdelayed_changed: %p: %p -- no delayed registered\n",
				 x,A));		
	    }
	}
    }

}


static struct rc_save * new_record P_((void));
static struct rc_save * new_record()
{
    struct rc_save * x = safe_zero_alloc(sizeof (*x));

    x->magic    = RC_SAVE_magic;
    x->real     = NULL;
    x->delayed  = NULL;

    x->next     = root;
    root        = x;

    return x;
}

static void free_saved_lines  P_((struct rc_save *p));
static void free_saved_lines(p) 
     struct rc_save *p;
{
    int i;

    if (RC_SAVE_magic != p->magic)
	panic("RC PANIC",__FILE__,__LINE__,"free_saved_lines",
	      "Bad magic number (rc_save)",0);
    
    for (i = 0; i < p->line_count; i++) {
	if (p->lines[i].value)
	    free_string(& (p->lines[i].value));
	p->lines[i].value = NULL;

	if (p->lines[i].get_value_buffer)	
	    free(p->lines[i].get_value_buffer);
	p->lines[i].get_value_buffer = NULL;	

	if (p->lines[i].filename)
	    free(p->lines[i].filename);
	p->lines[i].filename = NULL;       
    }
    
    if (p->lines)
	free(p->lines);
    p->lines      = NULL;
    p->line_count = 0;
}

static void replay_saved P_((struct rc_save *p,
			     int read_flags /* READ_FLAG_IGNORE_MISSING */));
static void replay_saved(p,read_flags)
     struct rc_save *p;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    int i;

    if (RC_SAVE_magic != p->magic)
	panic("RC PANIC",__FILE__,__LINE__,"replay_saved",
	      "Bad magic number (rc_save)",0);
    
    for (i = 0; i < p->line_count; i++) {
	replay_rec(& (p->lines[i]), p->real,
		   read_flags);
    }

    free_saved_lines(p);
}

static void add_delayed P_((struct rc_save *p));
static void add_delayed(p)
     struct rc_save *p;
{
    int x;

    if (RC_SAVE_magic != p->magic)
	panic("RC PANIC",__FILE__,__LINE__,"add_delayed",
	      "Bad magic number (rc_save)",0);
    
    for (x = 0; x < NUMBER_OF_SAVEABLE_OPTIONS; x++) {
	if (0 == strcmp(save_info[x].name,p->real->name)) {

	    if (save_info[x].dt_type != &rc_DT_DELAY &&
		save_info[x].dt_type != &rc_DT_DELAYSHARED)
		panic("RC PANIC",__FILE__,__LINE__,"add_delayed",
		      "Option is not delayed",0);

	    p->delayed = & (save_info[x]);
	    return;
	}

    }
    
    panic("RC PANIC",__FILE__,__LINE__,"add_delayed",
	  "Deleayed option not found",0);
}

static void register_delayed1 P_((struct rc_save_info_rec *option_real,
				  int read_flags /* READ_FLAG_IGNORE_MISSING */
				  ));
static void register_delayed1(option_real,read_flags)
     struct rc_save_info_rec *option_real;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    struct rc_save *p;

    for (p = root; p; p = p->next) {

	if (RC_SAVE_magic != p->magic)
	    panic("RC PANIC",__FILE__,__LINE__,"register_delayed1",
		  "Bad magic number (rc_save)",0);

	if (option_real == p->real)
	    panic("RC PANIC",__FILE__,__LINE__,"register_delayed1",
		  "Duplicate registeration",0);
	if (p->delayed &&
	    0 == strcmp(option_real->name,p->delayed->name)) {
	    
	    if (p->real)
		panic("RC PANIC",__FILE__,__LINE__,"register_delayed1",
		      "Duplicate name",0);
	    p->real = option_real;	    

	    replay_saved(p,read_flags);
	    
	    return;
	}
    }
    p = new_record();
    p->real = option_real;

    add_delayed(p);
}


void register_delayed(rc_options,rc_option_count,read_flags)
     struct rc_save_info_rec * rc_options;
     int rc_option_count;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    int i;

    for (i = 0; i < rc_option_count; i++) {
	register_delayed1(& (rc_options[i]) ,
			  read_flags
			  );
    }
}

static struct rc_save * register_delayed2 P_((struct rc_save_info_rec *option_proxy));
static struct rc_save * register_delayed2(option_proxy)
     struct rc_save_info_rec *option_proxy;
{
    struct rc_save *p;
    
    for (p = root; p; p = p->next) {

	if (RC_SAVE_magic != p->magic)
	    panic("RC PANIC",__FILE__,__LINE__,"register_delayed2",
		  "Bad magic number (rc_save)",0);
	
	if (option_proxy == p->delayed)
	    return p;

	if (p->real &&
	    0 == strcmp(option_proxy->name,p->real->name)) {
	    
	    if (p->delayed)
		panic("RC PANIC",__FILE__,__LINE__,"register_delayed2",
		      "Duplicate name",0);
	    p->delayed = option_proxy;
	    
	    return p;
	}
    }

    p = new_record();
    p->delayed = option_proxy;

    return p;
}

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

S_(rc_parse_line dt_DELAY_parse_line)
static int dt_DELAY_parse_line P_((struct rc_save_info_rec *r,
				   enum record_mode          lcl,
				   struct string *value, int lineno, 
				   const char *filename,
				   int negate,
				   int read_flags /* READ_FLAG_IGNORE_MISSING */));
static int dt_DELAY_parse_line(r,lcl,value,lineno,filename, negate,read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int negate;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    int ret = 1;
    struct rc_save *p = register_delayed2(r);
    
    if (p->real && p->line_count)
	panic("RC PANIC",__FILE__,__LINE__,"dt_DELAY_parse_line",
	      "Bad state",0);
    
    if (p->real) 	
	ret = rc_eval_tail(p->real,lcl,value,lineno,filename,negate,
			   read_flags);		
     else 
	 add_line_rec(p,
		      /* shares pointers, caller must alloc copy */
		      lcl,
		      dup_string(value),
		      lineno,
		      safe_strdup(filename),
		      negate ? line_negate : line_normal);
    
    DPRINT(Debug,20,(&Debug, 
		     "dt_DELAY_parse_line [%s:%d] = %d\n",
		     filename,lineno,ret));
    return ret;
}

S_(rc_parse_cline dt_DELAY_parse_cline)
static int dt_DELAY_parse_cline P_((struct rc_save_info_rec *r,
				    enum record_mode          lcl,
				    struct string *value, int lineno, 
				    const char *filename,
				    int read_flags /* READ_FLAG_IGNORE_MISSING */));
static int dt_DELAY_parse_cline(r,lcl,value,lineno,filename,
				read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{     
    int ret = 1;
    struct rc_save *p = register_delayed2(r);
    
    if (p->real && p->line_count)
	panic("RC PANIC",__FILE__,__LINE__,"dt_DELAY_parse_cline",
	      "Bad state",0);
    
    if (p->real) 
	ret = rc_eval_tail_cline(p->real,lcl,value,lineno,filename,
				 read_flags);
     else {
	
	 if (!p->line_count)
	    panic("RC PANIC",__FILE__,__LINE__,"dt_DELAY_parse_cline",
		  "Bad state",0);
    
	add_line_rec(p,
		     /* shares pointers, caller must alloc copy */
		     lcl,
		     dup_string(value),
		     lineno,
		     safe_strdup(filename),
		     line_continue)	;
    }

    DPRINT(Debug,20,(&Debug, 
		     "dt_DELAY_parse_cline [%s:%d] = %d\n",
		     filename,lineno,ret));

    return ret;    
}

S_(rc_print_value dt_DELAY_print_value)
static void dt_DELAY_print_value P_((FILE *F,
				     struct rc_save_info_rec *r,
				     int comment,
				     charset_t rcset));
static void dt_DELAY_print_value(F,r,comment,rcset)
     FILE *F;
     struct rc_save_info_rec *r;
     int comment;
     charset_t rcset;
{
    struct rc_save *p = register_delayed2(r);

    if (p->real && p->line_count)
	panic("RC PANIC",__FILE__,__LINE__,"dt_DELAY_print_value",
	      "Bad state",0);
    
    if (p->real) {
     
	if (RCTYPE_magic != p->real->dt_type->magic)
	    panic("RC PANIC",__FILE__,__LINE__,"dt_DELAY_print_value",
		  "Bad config item type",0);

	
	p->real->dt_type->print_value(F,p->real,comment,rcset);

    } else if (p->line_count) {
	int i,start = 0;
	int fail = 0;
	int unsupported = 0;
	
	for (i = 0; i < p->line_count; i++) {
	    if (line_normal == p->lines[i].negate_continue)
		start = i;
	}

	DPRINT(Debug,9,(&Debug, 
			" option \"%s\", delayed values lines %d .. %d\n",
			r->name,start,p->line_count-1));

	for (i = start; i < p->line_count; i++) {

	    char * buffer = NULL;
	    int   buffer_len = 0;

	    int ok = sconvert_to_rcset(p->lines[i].value,
				       r->name,rcset,&buffer,&buffer_len);
	    if (!ok)
		fail = 1;
	    
	    if (comment || !ok || !buffer)
		fprintf(F, "### ");

	    if (!buffer) {
		fprintf(F, "(not set)\n");
		continue;
	    }
	    

	    switch (p->lines[i].negate_continue) {
	    case line_normal:
		fprintf(F, "%s = %s\n", r->name, buffer);

		break;

	    case line_continue:
		if (0 == i)
		    panic("RC PANIC",__FILE__,__LINE__,"dt_DELAY_print_value",
			  "Bad state",0);

		if (unsupported)
		    putc('#',F);
		
		fprintf(F, "\t%s\n", buffer);

		break;

	    case line_negate:
		/* unsupported */
		fprintf(F, "# !%s = %s\n", r->name, buffer);
		unsupported = 1;
		break;
	    }
	    
	    free(buffer);
	    buffer = NULL;
	}

	if (fail)
	    elm_fprintf(F, 
			CATGETS(elm_msg_cat, ElmrcSet, ElmrcFailConvCharset,
				"# Failed to convert charset of option %s\n"),
				r->name);
	
    } else {
	fprintf(F, "### %s (not set)\n", r->name);

	DPRINT(Debug,9,(&Debug, 
			" option \"%s\", no value\n",
			r->name));
	
    }
}

S_(rc_get_value dt_DELAY_get_value)
static char * dt_DELAY_get_value P_((struct rc_save_info_rec *r));
static char * dt_DELAY_get_value(r)
     struct rc_save_info_rec *r;
{
    struct rc_save *p = register_delayed2(r);

    if (p->real && p->line_count)
	panic("RC PANIC",__FILE__,__LINE__,"dt_DELAY_get_value",
	      "Bad state",0);
    
    if (p->real) {
     
	if (RCTYPE_magic != p->real->dt_type->magic)
	    panic("RC PANIC",__FILE__,__LINE__,"dt_DELAY_get_value",
		  "Bad config item type",0);

	return p->real->dt_type->get_value(p->real);
    } else if (p->line_count) {

	int i,start = 0;

	for (i = 0; i < p->line_count; i++) {
	    if (line_normal == p->lines[i].negate_continue)
		start = i;
	}

	if (start == p->line_count-1 &&
	    line_normal == p->lines[start].negate_continue) {

	    int buffer_len;
	    int ok UNUSED_VAROK = 
		sconvert_to_rcset(p->lines[start].value,
				  r->name,system_charset,
				  & (p->lines[p->line_count].get_value_buffer),
				  &buffer_len);

	    if (p->lines[p->line_count].get_value_buffer)
		return p->lines[p->line_count].get_value_buffer;
	}
    }

    return "*bad*";
}

S_(rc_free_option NO_free_option)
static void NO_free_option  P_((struct rc_save_info_rec *r));
static void NO_free_option(r)
     struct rc_save_info_rec *r;
{
    /* Empty */
}


struct rc_type rc_DT_DELAY = { RCTYPE_magic,
			       dt_DELAY_parse_line, dt_DELAY_parse_cline,
			       dt_DELAY_print_value, dt_DELAY_get_value,
			       NO_free_option };

#ifdef USE_DLOPEN
static int call_delay_lib_reg P_((struct rc_save_info_rec *r,
				  struct string *value,
				  int lineno, 
				  const char *filename,
				  struct string **V,
				  int read_flags /* READ_FLAG_IGNORE_MISSING */));
static int call_delay_lib_reg(r,value,lineno,filename,
			      V,read_flags)
     struct rc_save_info_rec *r;
     struct string *value;
     int lineno;
     const char *filename;
     struct string **V;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    int ret;
    char * buffer = NULL;
    int buffer_len = 0;
    
    ret = convert_to_system_charset(value,lineno,filename,
				    &buffer,&buffer_len);
    
    if (!delay_lib_reg(r,&buffer,lineno,filename,
		       read_flags))
	ret = 0;

    if (*V)
	free_string(V);
    
    if (buffer) {
	*V = new_string2(system_charset,
			s2us(buffer));
	
	DPRINT(Debug,12,(&Debug,
			 "call_delay_lib_reg: Rewriting %S => %S\n",
			 value,*V));
	
	free(buffer);
    }
    
    return ret;
}
#endif


S_(rc_parse_line dt_DELAYSH_parse_line)
static int dt_DELAYSH_parse_line P_((struct rc_save_info_rec *r,
				     enum record_mode          lcl,
				     struct string *value, int lineno, 
				     const char *filename,
				     int negate,
				     int read_flags /* READ_FLAG_IGNORE_MISSING */
				     ));
static int dt_DELAYSH_parse_line(r,lcl,value,lineno,filename, negate,read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int negate;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    int ret = 1;
    struct rc_save *p = register_delayed2(r);
    
    if (p->real && p->line_count)
	panic("RC PANIC",__FILE__,__LINE__,"dt_DELAYSH_parse_line",
	      "Bad state",0);

     
    if (p->real) {
	
	ret = rc_eval_tail(p->real,lcl,value,lineno,filename,negate,
			   read_flags);
		
    } else {
	struct string * V = NULL;
#ifdef USE_DLOPEN
	ret = call_delay_lib_reg(r,value,lineno,filename,&V,
				 read_flags);
       	
#endif
	if (!V)
	    V =  dup_string(value);

	
	add_line_rec(p,
		     /* shares pointers, caller must alloc copy */
		     lcl,
		     V,
		     lineno,
		     safe_strdup(filename),
		     negate ? 
		     line_negate : line_normal);	
    }

    return ret;
}

S_(rc_parse_cline dt_DELAYSH_parse_cline)
static int dt_DELAYSH_parse_cline P_((struct rc_save_info_rec *r,
				      enum record_mode          lcl,
				      struct string *value, int lineno, 
				      const char *filename,
				      int read_flags /* READ_FLAG_IGNORE_MISSING */
				      ));
static int dt_DELAYSH_parse_cline(r,lcl,value,lineno,filename,read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    int ret = 1;
    struct rc_save *p = register_delayed2(r);
    
    if (p->real && p->line_count)
	panic("RC PANIC",__FILE__,__LINE__,"dt_DELAYSH_parse_cline",
	      "Bad state",0);

    if (p->real) {

	ret =  rc_eval_tail_cline(p->real,lcl,value,lineno,filename,
				  read_flags);
				  		
    } else {
	struct string * V = NULL;
	
	if (!p->line_count)
	    panic("RC PANIC",__FILE__,__LINE__,"dt_DELAYSH_parse_cline",
		  "Bad state",0);

#ifdef USE_DLOPEN
	ret = call_delay_lib_reg(r,value,lineno,filename,&V,
				 read_flags);
#endif
	if (!V)
	    V =  dup_string(value);

	add_line_rec(p,
		     /* shares pointers, caller must alloc copy */
		     lcl,
		     V,
		     lineno,
		     safe_strdup(filename),
		     line_continue);
    }
    
    return ret;
}

struct rc_type rc_DT_DELAYSHARED = { RCTYPE_magic,
				     dt_DELAYSH_parse_line,
				     dt_DELAYSH_parse_cline,
				     dt_DELAY_print_value,
				     dt_DELAY_get_value,
				     NO_free_option };




void free_delayed_rc_options()
{
    struct rc_save *p, *n = NULL;
    
    for (p = root; p; p = n) {
	n = p -> next;

	if (p->real) {
	    if (RCTYPE_magic != p->real->dt_type->magic)
		panic("RC PANIC",__FILE__,__LINE__,"free_delayed_rc_options",
		      "Bad config item type",0);
	    DPRINT(Debug,35,(&Debug,
			     "free_delayed_rc_options: Freeing delayed %p %p %s\n",
			     p, p->real, p->real->name));

	    p->real->dt_type->free_option(p->real);
	}

	if (p->lines) {

	    DPRINT(Debug,35,(&Debug,
			     "free_delayed_rc_options: Freeing delayed %p %d lines\n",
			     p,p->line_count));

	    free_saved_lines(p);
	}
	
	free(p);
	p = NULL;
	root = n;
    }
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
