static char rcsid[] = "@(#)$Id: schedule_time.c,v 2.9 2018/11/18 12:18:38 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.9 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "elm_defs.h"
#include "schedule_time.h"

DEBUG_VAR(Debug,__FILE__,"net");

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif

/* For INT_MAX */
#include <limits.h>


const long MAX_tv_nsec     = 999999999;   /* nanoseconds  [0 .. 999999999] */
#ifdef USE_GETTIMEOFDAY
#ifdef SUSECONDS
const SUSECONDS MAX_tv_usec = 999999;   /* microseconds [0 ..    999999] */ 
#else
#define MAX_tv_usec 999999
#endif
#endif


const struct schedule_timelimit NO_schedule_timelimit = {
    0,
#ifdef USE_GETTIMEOFDAY
    { 0 , 0 },
#endif
    
#ifdef USE_CLOCK_MONOTONIC
    { 0 , 0 },
#endif
};



int schedule_set_next_timeout(X,timeout_sec)
     struct schedule_timelimit *X;
     int                        timeout_sec;
{
    struct schedule_timelimit current = NO_schedule_timelimit;
    int r = 0;
    
    schedule_set_current_time(&current);

    r = schedule_set_next_timeout0(X,&current,timeout_sec,
				   se_use_time_MAX);
    
    return r;
}

int same_sec_ms(a,b)
     const struct sec_ms *a;
     const struct sec_ms *b;
{
    return (a->timeout_sec == b->timeout_sec &&
	    a->timeout_ms  == b->timeout_ms);
}

int schedule_time_passed(start,end,result)
     const struct schedule_timelimit * start;
     const struct schedule_timelimit * end;
     struct sec_ms                   * result;
{
    int r = 0;

    result->timeout_sec = INT_MAX;
    result->timeout_ms  = 0;
    
#ifdef USE_CLOCK_MONOTONIC
    if ((time_t)-1 != start->monotonic.tv_sec &&
	start->monotonic.tv_sec > 0 &&
	start->monotonic.tv_nsec >= 0 &&
	start->monotonic.tv_nsec <= MAX_tv_nsec &&

	(time_t)-1 != end->monotonic.tv_sec &&
	end->monotonic.tv_sec > 0 &&
	end->monotonic.tv_nsec >= 0 &&
	end->monotonic.tv_nsec <= MAX_tv_nsec) {

	DPRINT(Debug,21,(&Debug,"schedule_time_passed: CLOCK_MONOTONIC start %ld.%09ld, end %ld.%09ld\n",
			 (long)(start->monotonic.tv_sec),
			 start->monotonic.tv_nsec,
			 (long)(end->monotonic.tv_sec),
			 end->monotonic.tv_nsec));

	if (start->monotonic.tv_sec < end->monotonic.tv_sec) {
	    long diff = end->monotonic.tv_sec - start->monotonic.tv_sec;

	    if (diff > INT_MAX) {
		DPRINT(Debug,21,(&Debug,"schedule_time_passed: CLOCK_MONOTONIC overflow, diff=%ld\n",
				 diff));
	    } else {
		int diff_ms = (end->monotonic.tv_nsec - start->monotonic.tv_nsec) / 1000000;
		
		result->timeout_sec = diff;
	       		
		if (diff_ms >= 0) {
		    result->timeout_ms = diff_ms;
		    r =  SCHEDULE_TP_HAVE_TIME | SCHEDULE_TP_HAVE_MSEC;
		} else {
		    result->timeout_sec --;

		    result->timeout_ms = diff_ms + 1000;
		    r =  SCHEDULE_TP_HAVE_TIME | SCHEDULE_TP_HAVE_MSEC;
		}		
	    }
	    
	} else if (start->monotonic.tv_sec == end->monotonic.tv_sec) {

	    if (start->monotonic.tv_nsec <= end->monotonic.tv_nsec) {
		int diff_ms = (end->monotonic.tv_nsec - start->monotonic.tv_nsec) / 1000000;
	    
		result->timeout_sec = 0;
		result->timeout_ms = diff_ms;

		r =  SCHEDULE_TP_HAVE_TIME | SCHEDULE_TP_HAVE_MSEC;

	    } else {
		DPRINT(Debug,21,(&Debug,"schedule_time_passed: CLOCK_MONOTONIC end before start!\n"));
		
		result->timeout_sec = 0;
		result->timeout_ms  = 0;
	    }
		
	} else {
	    DPRINT(Debug,21,(&Debug,"schedule_time_passed: CLOCK_MONOTONIC end before start!\n"));

	    result->timeout_sec = 0;
	    result->timeout_ms  = 0;
	}

	/* Even when there is failure, use this result! */
	goto done;
    }
#endif

#ifdef USE_GETTIMEOFDAY
    if ((time_t)-1 != start->timeofday.tv_sec &&
	start->timeofday.tv_sec > 0 &&
	start->timeofday.tv_usec >= 0 &&
	start->timeofday.tv_usec <=  MAX_tv_usec &&

	(time_t)-1 != end->timeofday.tv_sec &&
	end->timeofday.tv_sec > 0 &&
	end->timeofday.tv_usec >= 0 &&
	end->timeofday.tv_usec <=  MAX_tv_usec) {

	DPRINT(Debug,21,(&Debug,"schedule_time_passed:: time of day start %ld.%06ld, end %ld.%06ld\n",
			 (long)(start->timeofday.tv_sec),
			 (long)(start->timeofday.tv_usec),
			 (long)(end->timeofday.tv_sec),
			 (long)(end->timeofday.tv_usec)));

	if (start->timeofday.tv_sec < end->timeofday.tv_sec) {
	    long diff = end->timeofday.tv_sec - start->timeofday.tv_sec;

	     if (diff > INT_MAX) {
		 DPRINT(Debug,21,(&Debug,"schedule_time_passed: time of day overflow, diff=%ld\n",
				  diff));

	     } else {
		 int diff_ms = (end->timeofday.tv_usec - start->timeofday.tv_usec) / 1000;

		 result->timeout_sec = diff;
		 
		 if (diff_ms >= 0) {
		    result->timeout_ms = diff_ms;
		    r =  SCHEDULE_TP_HAVE_TIME | SCHEDULE_TP_HAVE_MSEC;
		} else {
		    result->timeout_sec --;

		    result->timeout_ms = diff_ms + 1000;
		    r =  SCHEDULE_TP_HAVE_TIME | SCHEDULE_TP_HAVE_MSEC;
		}		
	     }
	     

	} else if (start->timeofday.tv_sec == end->timeofday.tv_sec) {

	    if (start->timeofday.tv_usec <= end->timeofday.tv_usec) {
		int diff_ms = (end->timeofday.tv_usec - start->timeofday.tv_usec) / 1000;

		result->timeout_sec = 0;
		result->timeout_ms = diff_ms;

		r =  SCHEDULE_TP_HAVE_TIME | SCHEDULE_TP_HAVE_MSEC;
	    } else {
		DPRINT(Debug,21,(&Debug,"schedule_time_passed: time of day end before start!\n"));
		
		result->timeout_sec = 0;
		result->timeout_ms  = 0;
	    }
	    
	} else {
	    DPRINT(Debug,21,(&Debug,"schedule_time_passed: time of day end before start!\n"));
	    
	    result->timeout_sec = 0;
	    result->timeout_ms  = 0;
	}
	
	/* Even when there is failure, use this result! */
	goto done;
    }
#endif

    if ((time_t)-1 != start->wallclock &&
	start->wallclock > 0 &&

	(time_t)-1 != end->wallclock &&
	end->wallclock > 0) {

	DPRINT(Debug,21,(&Debug,"schedule_time_passed: wallclock start %ld, end %ld\n",
			 (long)(start->wallclock),
			 (long)(end->wallclock)));
	
	if ( start->wallclock <= end->wallclock ) {

	    long diff =  end->wallclock -start->wallclock;

	    if (diff > INT_MAX) {
		 DPRINT(Debug,21,(&Debug,"schedule_time_passed: wallclock overflow, diff=%ld\n",
				  diff));

	     } else {
		result->timeout_sec = diff;
		result->timeout_ms  = 0;

		r =  SCHEDULE_TP_HAVE_TIME;
	    }

	} else {
	    DPRINT(Debug,21,(&Debug,"schedule_time_passed: wallclock end before start!\n"));
	    
	    result->timeout_sec = 0;
	    result->timeout_ms  = 0;
	}
    }

#if defined(USE_CLOCK_MONOTONIC) || defined(USE_GETTIMEOFDAY)
 done:
#endif

    DPRINT(Debug,21,(&Debug,"schedule_time_passed=%d%s%s; result=%d.%03d\n",
		     r,
		     (r & SCHEDULE_TP_HAVE_TIME) ? " SCHEDULE_TP_HAVE_TIME" : "",
		     (r & SCHEDULE_TP_HAVE_MSEC) ? " SCHEDULE_TP_HAVE_MSEC" : "",
		     result->timeout_sec,
		     result->timeout_ms));
    
    return r;
}


int schedule_set_next_timeout2(X,timeout)
     struct schedule_timelimit * X;
     const struct sec_ms       * timeout;
{
    struct schedule_timelimit current = NO_schedule_timelimit;
    int r = 0;
    
    schedule_set_current_time(&current);

    r = schedule_set_next_timeout2a(X,&current,timeout,
				    se_use_time_MAX);

    return r;
}

int schedule_set_next_timeout2a(X,current,timeout,errmode)
     struct schedule_timelimit       * X;
     const struct schedule_timelimit * current;
     const struct sec_ms             * timeout;
     enum schedule_errmode             errmode;
{
    int r = 0;
    struct sec_ms temp;
    
    if (timeout->timeout_sec < 0) {
	DPRINT(Debug,21,(&Debug,
			 "schedule_set_next_timeout2a: resetting timeout %d seconds\n",
			 timeout->timeout_sec));

	temp    = *timeout;
	timeout = &temp;
	
	temp.timeout_sec = 0;
    }
    
    if (timeout->timeout_ms > 1000) {
	int add = timeout->timeout_ms / 1000;

	DPRINT(Debug,21,(&Debug,
			"schedule_set_next_timeout2a: timeout %d sec %d ms => ",
			timeout->timeout_sec, timeout->timeout_ms));

	if (timeout != &temp) {
	    temp    = *timeout;
	    timeout = &temp;
	}
	
	temp.timeout_ms -= add * 1000;

	if (temp.timeout_sec < INT_MAX - add)
	    temp.timeout_sec += add;
	else {
	    temp.timeout_sec = INT_MAX;

	    DPRINT(Debug,21,(&Debug,"(overflow) "));
	}
	DPRINT(Debug,21,(&Debug,"%d.%03d\n",
			 timeout->timeout_sec,timeout->timeout_ms));
	
    } else {
	DPRINT(Debug,21,(&Debug,
			 "schedule_set_next_timeout2a: timeout %d.%03d\n",
			 timeout->timeout_sec,timeout->timeout_ms));
    }
      
    if ((time_t)-1 == current->wallclock ||
	current->wallclock <= 0) {
	
	DPRINT(Debug,9,(&Debug,
			"schedule_set_next_timeout2a: no current time\n"));

	switch (errmode) {
	case se_use_time_MAX:         X->wallclock = time_MAX;   break;
	case se_use_before_time_MAX:  X->wallclock = time_MAX-1; break;
	case se_use_zero:             X->wallclock = 0;          break;
	default:                      X->wallclock = (time_t)-1; break;
	}
	
    } else if (current->wallclock < time_MAX  /* May fail at Tue Jan 19 03:14:07 2038 */
	       - timeout->timeout_sec) {

	X->wallclock =
	    current->wallclock + timeout->timeout_sec;
	r = 1;

	if (X->wallclock < time_MAX &&
	    0 == timeout->timeout_sec &&
	    timeout->timeout_ms > 0) {
	    X->wallclock++;
	    
	    DPRINT(Debug,21,(&Debug,
			     "schedule_set_next_timeout2a: Adjusting %d ms timeout to next second (on wallclock)\n",
			     timeout->timeout_ms));
	}
	
	DPRINT(Debug,21,(&Debug,"schedule_set_next_timeout2a: next timeout wallclock %ld\n",
			 (long)(X->wallclock)));
	
    } else {
	char * s UNUSED_VAROK = "";
	
	switch (errmode) {
	case se_use_time_MAX:
	    X->wallclock = time_MAX;
	    s = " (time_MAX)";
	    break;
	case se_use_before_time_MAX:
	    X->wallclock = time_MAX-1;
	    s = " (time_MAX-1)";
	    break;
	case se_use_zero:
	    X->wallclock = 0;
	    s = "";
	    break;
	default:
	    X->wallclock = (time_t)-1;
	    s = " ((time_t)-1, bad errmode)";
	    break;
	}

	DPRINT(Debug,9,(&Debug,
			"schedule_set_next_timeout2a: current time, timeout overflow, using %ld%s\n",
			(long)(X->wallclock),
			s));
    }

#ifdef USE_GETTIMEOFDAY
    if ((time_t)-1 == current->timeofday.tv_sec ||
	current->timeofday.tv_sec <= 0 ||
	current->timeofday.tv_usec < 0 ||
	current->timeofday.tv_usec >  MAX_tv_usec) {

	DPRINT(Debug,9,(&Debug,
			"schedule_set_next_timeout2a: no time of day\n"));

	X->timeofday.tv_sec = 0;   /* Use X->wallclock */
	X->timeofday.tv_usec = 0;	
    } else if (current->timeofday.tv_sec < time_MAX  /* May fail at Tue Jan 19 03:14:07 2038 */
	       - timeout->timeout_sec) {
#ifdef SUSECONDS
       SUSECONDS usec = (SUSECONDS)
#else
       long usec = 
#endif
	   (timeout->timeout_ms) * 1000;
	
	X->timeofday.tv_sec  = current->timeofday.tv_sec +  timeout->timeout_sec;
	X->timeofday.tv_usec = current->timeofday.tv_usec;

	if (current->timeofday.tv_usec < MAX_tv_usec - usec)
	    X->timeofday.tv_usec += usec;
	else if (X->timeofday.tv_sec <  time_MAX) {
	    X->timeofday.tv_sec ++;
	    X->timeofday.tv_usec -= ( 1000000 - usec);
	} else
	    goto overflow_timeofday;

	r = 1;
	
	DPRINT(Debug,21,(&Debug,"schedule_set_next_timeout2a: next timeout time of day %ld.%06ld\n",
			 (long)(X->timeofday.tv_sec),
			 (long)(X->timeofday.tv_usec)));
    } else {
    overflow_timeofday:
	if ((time_t)-1 == current->wallclock ||
	    current->wallclock <= 0 ||
	    current->timeofday.tv_sec <= current->wallclock) {
	    char * s UNUSED_VAROK = "";
	    switch (errmode) {
	    case se_use_time_MAX:
		X->timeofday.tv_sec = time_MAX;
		s = " (time_MAX)";
		break;
	    case se_use_before_time_MAX:
		X->timeofday.tv_sec = time_MAX-1;
		s = " (time_MAX-1)";
		break;
	    case se_use_zero:
		X->timeofday.tv_sec = 0;
		s = "";
		break;
	    default:
		X->timeofday.tv_sec = (time_t)-1;
		s = " ((time_t)-1, bad errmode)";
		break;
	    }
	    X->timeofday.tv_usec = 0;
	
	    DPRINT(Debug,9,(&Debug,"schedule_set_next_timeout2a: next timeout day, timeout overflow, using %ld.%06ld%s\n",
			    (long)(X->timeofday.tv_sec),
			    (long)(X->timeofday.tv_usec),
			    s));
	    
	} else {   /* Should not happen */
	    DPRINT(Debug,9,(&Debug,"schedule_set_next_timeout2a: next timeout day, timeout overflow\n"));
	
	    X->timeofday.tv_sec = 0;   /* Use X->wallclock */
	    X->timeofday.tv_usec = 0;
	}
    }
#endif
 
#ifdef USE_CLOCK_MONOTONIC   
    if ((time_t)-1 == current->monotonic.tv_sec ||
	current->monotonic.tv_sec <= 0  ||
	current->monotonic.tv_nsec < 0 ||
	current->monotonic.tv_nsec > MAX_tv_nsec) {
	
	DPRINT(Debug,9,(&Debug,
			"schedule_set_next_timeout2a: no CLOCK_MONOTONIC time\n"));

	X->monotonic.tv_sec  = 0;  /* Use X->wallclock or current->timeofday */
	X->monotonic.tv_nsec = 0;
	
    } else if (current->monotonic.tv_sec < time_MAX - timeout->timeout_sec 
	       ) {
	long nsec = (long)(timeout->timeout_ms) * 1000000;
	
	X->monotonic.tv_sec  = current->monotonic.tv_sec + timeout->timeout_sec;
	X->monotonic.tv_nsec = current->monotonic.tv_nsec;

	if (X->monotonic.tv_nsec < MAX_tv_nsec - nsec)
	    X->monotonic.tv_nsec += nsec;
	else if (X->monotonic.tv_sec < time_MAX) {
	    X->monotonic.tv_sec ++;
	    X->monotonic.tv_nsec -= ( 1000000000 - nsec);
	} else
	    goto overflow_monotonic;
	
	r = 1;

	DPRINT(Debug,21,(&Debug,"schedule_set_next_timeout2a: next timeout CLOCK_MONOTONIC %ld.%09ld\n",
			 (long)(X->monotonic.tv_sec),
			 X->monotonic.tv_nsec));

    } else {
    overflow_monotonic:
	if ((time_t)-1 == current->wallclock ||
	    current->wallclock <= 0 ||
	    current->monotonic.tv_sec <= current->wallclock) {
	    char * s UNUSED_VAROK = "";
	    
	    switch (errmode) {
	    case se_use_time_MAX:
		X->monotonic.tv_sec = time_MAX;
		s = " (time_MAX)";
		break;
	    case se_use_before_time_MAX:
		X->monotonic.tv_sec = time_MAX-1;
		s = " (time_MAX-1)";
		break;
	    case se_use_zero:
		X->monotonic.tv_sec = 0;
		s = "";
		break;
	    default:
		X->monotonic.tv_sec = (time_t)-1;
		s = " ((time_t)-1, bad errmode)";
	    break;
	    }
	    
	    X->monotonic.tv_nsec = 0;
	    
	    DPRINT(Debug,9,(&Debug,
			    "schedule_set_next_timeout2a: CLOCK_MONOTONIC time, timeout overflow, using %ld%s\n",
			    (long)(X->monotonic.tv_sec),
			    s));
	    
	} else {
	    DPRINT(Debug,9,(&Debug,
			    "schedule_set_next_timeout2a: CLOCK_MONOTONIC time, timeout overflow\n"));
	    X->monotonic.tv_sec  = 0;  /* Use X->wallclock */
	    X->monotonic.tv_nsec = 0;
	}
    }
#endif
   
    return r;
}

int schedule_set_next_timeout0(X,current,timeout_sec,errmode)
     struct schedule_timelimit       * X;
     const struct schedule_timelimit * current;
     int                               timeout_sec;
     enum schedule_errmode             errmode;
{
    int r = 0;
    struct sec_ms timeout;

    timeout.timeout_sec = timeout_sec;
    timeout.timeout_ms  = 0;

    r = schedule_set_next_timeout2a(X,current,&timeout,errmode);
        
    return r;
}

int schedule_shorten_next_timeout(next_timeout,candinate)
     struct schedule_timelimit *next_timeout;
     const struct schedule_timelimit * candinate;
{
    int r = -1;
    
#ifdef USE_CLOCK_MONOTONIC
    if ((time_t)-1 == next_timeout->monotonic.tv_sec
	||
	(0 == next_timeout->monotonic.tv_sec &&
	 0 == next_timeout->monotonic.tv_nsec))
	r = 1;
    else if ((time_t)-1 != next_timeout->monotonic.tv_sec &&
	(time_t)-1 != candinate->monotonic.tv_sec &&
	next_timeout->monotonic.tv_sec > 0 &&
	candinate->monotonic.tv_sec > 0 &&
	next_timeout->monotonic.tv_nsec >= 0 &&
	next_timeout->monotonic.tv_nsec <= MAX_tv_nsec &&	
	candinate->monotonic.tv_nsec >= 0 &&
	candinate->monotonic.tv_nsec <= MAX_tv_nsec) {

	r = next_timeout->monotonic.tv_sec > candinate->monotonic.tv_sec
	    ||
	    (next_timeout->monotonic.tv_sec == candinate->monotonic.tv_sec &&
	     next_timeout->monotonic.tv_nsec > candinate->monotonic.tv_nsec);
    }
#endif

#ifdef USE_GETTIMEOFDAY
    if (-1 == r) {
	if ((time_t)-1 == next_timeout->timeofday.tv_sec 
	    ||
	    (0 ==  next_timeout->timeofday.tv_sec &&
	     0 == next_timeout->timeofday.tv_usec))
	    r = 1;
	else if ((time_t)-1 != next_timeout->timeofday.tv_sec &&
		 (time_t)-1 != candinate->timeofday.tv_sec &&
		 next_timeout->timeofday.tv_sec > 0 &&
		 candinate->timeofday.tv_sec > 0 &&
		 next_timeout->timeofday.tv_usec >= 0 &&
		 candinate->timeofday.tv_usec >= 0 &&
		 next_timeout->timeofday.tv_usec <=  MAX_tv_usec &&
		 candinate->timeofday.tv_usec <=  MAX_tv_usec) {

	    r = next_timeout->timeofday.tv_sec >  candinate->timeofday.tv_sec
		||
		(next_timeout->timeofday.tv_sec == candinate->timeofday.tv_sec &&
		 next_timeout->timeofday.tv_usec > candinate->timeofday.tv_usec);
	       
	}
    }
#endif
    
    if (-1 == r) {
	if ((time_t)-1 == next_timeout->wallclock ||
	    0 ==  next_timeout->wallclock)
	    r = 1;
	else if ((time_t)-1 != next_timeout->wallclock &&
		 (time_t)-1 != candinate->wallclock &&
		 next_timeout->wallclock > 0 &&
		 candinate->wallclock > 0) {

	    r = next_timeout->wallclock > candinate->wallclock;
	}
    }

    if (1 == r) {
	*next_timeout = *candinate;

	DPRINT(Debug,21,(&Debug,"schedule_shorten_next_timeout: next timeout wallclock %ld\n",
			 (long)(next_timeout->wallclock)));
#ifdef USE_GETTIMEOFDAY
	DPRINT(Debug,21,(&Debug,"schedule_shorten_next_timeout: next timeout time of day %ld.%06ld\n",
			 (long)(next_timeout->timeofday.tv_sec),
			 (long)(next_timeout->timeofday.tv_usec)));
#endif
#ifdef USE_CLOCK_MONOTONIC
	DPRINT(Debug,21,(&Debug,"schedule_shorten_next_timeout: next timeout CLOCK_MONOTONIC %ld.%09ld\n",
			 (long)(next_timeout->monotonic.tv_sec),
			 next_timeout->monotonic.tv_nsec));
#endif
	
    } else
	r = 0;
	
    return r;
}

int schedule_extend_valid_until(valid_until,candinate)
     struct schedule_timelimit *valid_until;
     const struct schedule_timelimit * candinate;
{
    int r = -1;

#ifdef USE_CLOCK_MONOTONIC
    if ((time_t)-1 == valid_until->monotonic.tv_sec
	||
	(0 == valid_until->monotonic.tv_sec &&
	 0 == valid_until->monotonic.tv_nsec))
	r = 1;
    else if ((time_t)-1 != valid_until->monotonic.tv_sec &&
	(time_t)-1 != candinate->monotonic.tv_sec &&
	valid_until->monotonic.tv_sec > 0 &&
	candinate->monotonic.tv_sec > 0 &&
	valid_until->monotonic.tv_nsec >= 0 &&
	valid_until->monotonic.tv_nsec <= MAX_tv_nsec &&	
	candinate->monotonic.tv_nsec >= 0 &&
	candinate->monotonic.tv_nsec <= MAX_tv_nsec) {

	r = valid_until->monotonic.tv_sec < candinate->monotonic.tv_sec
	    ||
	    (valid_until->monotonic.tv_sec == candinate->monotonic.tv_sec &&
	     valid_until->monotonic.tv_nsec < candinate->monotonic.tv_nsec);
    }
#endif

#ifdef USE_GETTIMEOFDAY
    if (-1 == r) {
	if ((time_t)-1 == valid_until->timeofday.tv_sec 
	    ||
	    (0 ==  valid_until->timeofday.tv_sec &&
	     0 == valid_until->timeofday.tv_usec))
	    r = 1;
	else if ((time_t)-1 != valid_until->timeofday.tv_sec &&
		 (time_t)-1 != candinate->timeofday.tv_sec &&
		 valid_until->timeofday.tv_sec > 0 &&
		 candinate->timeofday.tv_sec > 0 &&
		 valid_until->timeofday.tv_usec >= 0 &&
		 candinate->timeofday.tv_usec >= 0 &&
		 valid_until->timeofday.tv_usec <=  MAX_tv_usec &&
		 candinate->timeofday.tv_usec <=  MAX_tv_usec) {

	    r = valid_until->timeofday.tv_sec < candinate->timeofday.tv_sec
		||
		(valid_until->timeofday.tv_sec == candinate->timeofday.tv_sec &&
		 valid_until->timeofday.tv_usec < candinate->timeofday.tv_usec);
	}
    }
#endif
    
    if (-1 == r) {
	if ((time_t)-1 == valid_until->wallclock ||
	    0 ==  valid_until->wallclock)
	    r = 1;
	else if ((time_t)-1 != valid_until->wallclock &&
		 (time_t)-1 != candinate->wallclock &&
		 valid_until->wallclock > 0 &&
		 candinate->wallclock > 0) {

	    r = valid_until->wallclock < candinate->wallclock;
	}
    }

    if (1 == r) {
	*valid_until = *candinate;

	DPRINT(Debug,21,(&Debug,"schedule_extend_valid_until: valid until wallclock %ld\n",
			 (long)(valid_until->wallclock)));
#ifdef USE_GETTIMEOFDAY
	DPRINT(Debug,21,(&Debug,"schedule_extend_valid_until: valid until time of day %ld.%06ld\n",
			 (long)(valid_until->timeofday.tv_sec),
			 (long)(valid_until->timeofday.tv_usec)));
#endif	
#ifdef USE_CLOCK_MONOTONIC
	DPRINT(Debug,21,(&Debug,"schedule_extend_valid_until: valid until CLOCK_MONOTONIC %ld.%09ld\n",
			 (long)(valid_until->monotonic.tv_sec),
			 valid_until->monotonic.tv_nsec));
#endif	       
    } else
	r = 0;
	
    return r;
}

void schedule_set_current_time(X)
  struct schedule_timelimit *X;
{

  /* Gives (time_t)-1 on failure */
  X->wallclock = time(NULL);
  
  if ( (time_t)-1 ==  X->wallclock) {
    int err UNUSED_VAROK = errno;
    DPRINT(Debug,9,(&Debug,
		    "schedule_set_current_time: Failed to get current time: time: %s (errno=%d)\n",
		    strerror(err),err));
  } else {
      DPRINT(Debug,21,(&Debug,"schedule_set_current_time: current wallclock %ld\n",
		       (long)(X->wallclock)));
  }

#ifdef USE_GETTIMEOFDAY
  if (-1 == gettimeofday(&(X->timeofday), NULL)) {
      int err UNUSED_VAROK = errno;
      DPRINT(Debug,9,(&Debug,
		      "schedule_set_current_time: Failed to get time of day: (errno=%d)\n",
		    strerror(err),err));

      X->timeofday.tv_sec  = (time_t)-1;  /* Simulate error */
      X->timeofday.tv_usec = 0;
  } else {
	DPRINT(Debug,21,(&Debug,"schedule_set_current_time: current time of day %ld.%06ld\n",
			 (long)(X->timeofday.tv_sec),
			 (long)(X->timeofday.tv_usec)));
  }
#endif
  
#ifdef USE_CLOCK_MONOTONIC
  if (-1 == clock_gettime(CLOCK_MONOTONIC,
			  & (X->monotonic))) {
    
    int err UNUSED_VAROK = errno;
    DPRINT(Debug,9,(&Debug,
		    "schedule_set_current_time: Failed to get CLOCK_MONOTONIC:  clock_gettime: %s (errno=%d)\n",
		    strerror(err),err));

    X->monotonic.tv_sec  = (time_t)-1;  /* Simulate error */
    X->monotonic.tv_nsec = 0;
  } else {
      DPRINT(Debug,21,(&Debug,"schedule_set_current_time: current CLOCK_MONOTONIC %ld.%09ld\n",
		       (long)(X->monotonic.tv_sec),
		       X->monotonic.tv_nsec));
  }
#endif
}

int schedule_valid_until_is_MAX(valid_until)
     const struct schedule_timelimit * valid_until;
{
    int r = -1;

#ifdef USE_CLOCK_MONOTONIC
    if ((time_t)-1 != valid_until->monotonic.tv_sec &&
	valid_until->monotonic.tv_sec > 0 &&
	valid_until->monotonic.tv_nsec >= 0 &&
	valid_until->monotonic.tv_nsec <= MAX_tv_nsec) {

	/* Just ignore tv_nsec */

	r = time_MAX == valid_until->monotonic.tv_sec;	
    }
#endif

#ifdef USE_GETTIMEOFDAY
    if (-1 == r &&
	(time_t)-1 != valid_until->timeofday.tv_sec &&
	valid_until->timeofday.tv_sec > 0 &&
	valid_until->timeofday.tv_usec >= 0 &&
	valid_until->timeofday.tv_usec <=  MAX_tv_usec) {

	/* Just ignore tv_usec */

	r = time_MAX == valid_until->timeofday.tv_sec;
    }
#endif
    
    if (-1 == r &&
	(time_t)-1 != valid_until->wallclock &&
	valid_until->wallclock > 0) {

	/* Permanent  -- May fail at Tue Jan 19 03:14:07 2038 UTC */
	
	r = time_MAX == valid_until->wallclock;
    }

    if (-1 == r)
	r = 0;
   
    return r;
}

void  set_MAX_schedule_timelimit(max)
     struct schedule_timelimit *max;
{
#ifdef USE_CLOCK_MONOTONIC
    max->monotonic.tv_sec  = time_MAX;
    max->monotonic.tv_nsec = 0;
#endif

#ifdef USE_GETTIMEOFDAY
     /*  May fail at Tue Jan 19 03:14:07 2038 UTC */
    max->timeofday.tv_sec  = time_MAX;
    max->timeofday.tv_usec = 0;
#endif

    /*  May fail at Tue Jan 19 03:14:07 2038 UTC */
    max->wallclock =  time_MAX;
}

const struct schedule_timelimit * give_MAX_schedule_timelimit()
{
    static struct schedule_timelimit max;

    set_MAX_schedule_timelimit(&max);

    return &max;
}



int schedule_timeout_reached(next_timeout,current)
     const struct schedule_timelimit * next_timeout;
     const struct schedule_timelimit * current;
{
    int r = -1;

#ifdef USE_CLOCK_MONOTONIC
    if ((time_t)-1 != next_timeout->monotonic.tv_sec &&
	(time_t)-1 != current->monotonic.tv_sec &&
	next_timeout->monotonic.tv_sec > 0 &&
	current->monotonic.tv_sec > 0 &&
	next_timeout->monotonic.tv_nsec >= 0 &&
	next_timeout->monotonic.tv_nsec <= MAX_tv_nsec &&	
	current->monotonic.tv_nsec >= 0 &&
	current->monotonic.tv_nsec <= MAX_tv_nsec) {

	r = next_timeout->monotonic.tv_sec < current->monotonic.tv_sec
	    ||
	    (next_timeout->monotonic.tv_sec == current->monotonic.tv_sec &&
	     next_timeout->monotonic.tv_nsec <= current->monotonic.tv_nsec);
		
    }    	
#endif

#ifdef USE_GETTIMEOFDAY
    if (-1 == r &&
	(time_t)-1 != next_timeout->timeofday.tv_sec &&
	(time_t)-1 != current->timeofday.tv_sec &&
	next_timeout->timeofday.tv_sec > 0 &&
	current->timeofday.tv_sec > 0 &&
	next_timeout->timeofday.tv_usec >= 0 &&
	current->timeofday.tv_usec >= 0 &&
	next_timeout->timeofday.tv_usec <=  MAX_tv_usec &&
	current->timeofday.tv_usec <=  MAX_tv_usec) {

	r = next_timeout->timeofday.tv_sec < current->timeofday.tv_sec
	    ||
	    (next_timeout->timeofday.tv_sec == current->timeofday.tv_sec &&
	     next_timeout->timeofday.tv_usec <= current->timeofday.tv_usec);
	}
#endif
  
    if (-1 == r &&
	(time_t)-1 != next_timeout->wallclock &&
	(time_t)-1 != current->wallclock &&
	next_timeout->wallclock > 0 &&
	current->wallclock > 0) {

	r = next_timeout->wallclock <= current->wallclock;
	    
    }

    if (-1 == r)
	r = 0;

    return r;
}

/* Almost same than schedule_timeout_reached */
int schedule_valid_until_expired(valid_until,current)
     const struct schedule_timelimit * valid_until;
     const struct schedule_timelimit * current;
{
     int r = -1;

#ifdef USE_CLOCK_MONOTONIC
    if ((time_t)-1 != valid_until->monotonic.tv_sec &&
	(time_t)-1 != current->monotonic.tv_sec &&
	valid_until->monotonic.tv_sec > 0 &&
	current->monotonic.tv_sec > 0 &&
	valid_until->monotonic.tv_nsec >= 0 &&
	valid_until->monotonic.tv_nsec <= MAX_tv_nsec &&	
	current->monotonic.tv_nsec >= 0 &&
	current->monotonic.tv_nsec <= MAX_tv_nsec) {

	r = valid_until->monotonic.tv_sec < current->monotonic.tv_sec
	    ||
	    (valid_until->monotonic.tv_sec == current->monotonic.tv_sec &&
	     valid_until->monotonic.tv_nsec < current->monotonic.tv_nsec);
		
    }    	
#endif

#ifdef USE_GETTIMEOFDAY
     if (-1 == r &&
	 (time_t)-1 != valid_until->timeofday.tv_sec &&
	 (time_t)-1 != current->timeofday.tv_sec &&
	 valid_until->timeofday.tv_sec > 0 &&
	 current->timeofday.tv_sec > 0 &&
	 valid_until->timeofday.tv_usec >= 0 &&
	 current->timeofday.tv_usec >= 0 &&
	 valid_until->timeofday.tv_usec <=  MAX_tv_usec &&
	 current->timeofday.tv_usec <=  MAX_tv_usec) {

	 r = valid_until->timeofday.tv_sec < current->timeofday.tv_sec
	     ||
	     (valid_until->timeofday.tv_sec == current->timeofday.tv_sec &&
	      valid_until->timeofday.tv_usec < current->timeofday.tv_usec);
     }
#endif

    
    if (-1 == r &&
	(time_t)-1 != valid_until->wallclock &&
	(time_t)-1 != current->wallclock &&
	valid_until->wallclock > 0 &&
	current->wallclock > 0) {

	r = valid_until->wallclock < current->wallclock;
	    
    }

    if (-1 == r)
	r = 0;

    return r;
}

/* Rougly reverse of schedule_valid_until_expired */
int schedule_valid_until_ok(valid_until,current)
     const struct schedule_timelimit * valid_until;
     const struct schedule_timelimit * current;
{
    int r = -1;

#ifdef USE_CLOCK_MONOTONIC   
    if ((time_t)-1 == current->monotonic.tv_sec) {
	DPRINT(Debug,9,(&Debug,
			"schedule_valid_until_ok: current CLOCK_MONOTONIC time failed\n"));
	r = -2;
    } else if ((time_t)-1 != valid_until->monotonic.tv_sec &&
	       valid_until->monotonic.tv_sec > 0 &&
	       current->monotonic.tv_sec > 0 &&
	       valid_until->monotonic.tv_nsec >= 0 &&
	       valid_until->monotonic.tv_nsec <= MAX_tv_nsec &&	
	       current->monotonic.tv_nsec >= 0 &&
	       current->monotonic.tv_nsec <= MAX_tv_nsec) {

	r = valid_until->monotonic.tv_sec > current->monotonic.tv_sec
	    ||
	    (valid_until->monotonic.tv_sec == current->monotonic.tv_sec &&
	     valid_until->monotonic.tv_nsec >= current->monotonic.tv_nsec);
    }
#endif

#ifdef USE_GETTIMEOFDAY
     if (r < 0) {
	 if ((time_t)-1 == current->timeofday.tv_sec) {
	     DPRINT(Debug,9,(&Debug,
			     "schedule_valid_until_ok: current time of day failed\n"));
	     r = -2;
	 } else if ((time_t)-1 != valid_until->timeofday.tv_sec &&
		    valid_until->timeofday.tv_sec > 0 &&
		    current->timeofday.tv_sec > 0 &&
		    valid_until->timeofday.tv_usec >= 0 &&
		    current->timeofday.tv_usec >= 0 &&
		    valid_until->timeofday.tv_usec <=  MAX_tv_usec &&
		    current->timeofday.tv_usec <=  MAX_tv_usec) {

	     r = valid_until->timeofday.tv_sec >  current->timeofday.tv_sec
		 ||  
		 (valid_until->timeofday.tv_sec ==  current->timeofday.tv_sec &&
		  valid_until->timeofday.tv_usec >= current->timeofday.tv_usec);
	 }
     }
#endif
    
    if (r < 0) {
	if ((time_t)-1 == current->wallclock) {
	    DPRINT(Debug,9,(&Debug,
			    "schedule_valid_until_ok: current wallclock time failed\n"));
	    r = -2;
	} else if ((time_t)-1 != valid_until->wallclock &&
		   valid_until->wallclock > 0 &&
		   current->wallclock > 0) {

	    r = valid_until->wallclock >= current->wallclock;

	}
    }

    if (-2 == r) {
	r = 1;  /* No current time, assume valid entry */
    } else if (-1 == r) {

	if (0 == current->wallclock
#ifdef USE_GETTIMEOFDAY
	    &&
	    0 == current->timeofday.tv_sec &&
	    0 == current->timeofday.tv_usec
#endif
	    
#ifdef USE_CLOCK_MONOTONIC
	    &&
	    0 == current->monotonic.tv_sec &&
	    0 == current->monotonic.tv_nsec
#endif
	    )
	    r= 1;  /*  No current time, assume valid entry */
	else
	    r = 0; 
    }

    return r;
}

int  schedule_invalid_next_timeout(next_timeout)
     const struct schedule_timelimit * next_timeout;
{
    int r = -1;

#ifdef USE_CLOCK_MONOTONIC
    r = (time_t)-1 == next_timeout->monotonic.tv_sec ||
	next_timeout->monotonic.tv_sec < 0 ||
	next_timeout->monotonic.tv_nsec < 0 ||
	next_timeout->monotonic.tv_nsec > MAX_tv_nsec;
#endif

#ifdef USE_GETTIMEOFDAY
     if (-1 == r)
	 r = (time_t)-1 == next_timeout->timeofday.tv_sec ||
	     next_timeout->timeofday.tv_sec < 0 ||
	     next_timeout->timeofday.tv_usec < 0 ||
	     next_timeout->timeofday.tv_usec > MAX_tv_usec;
#endif
    
    if (-1 == r)
	r = (time_t)-1 == next_timeout->wallclock ||
	    next_timeout->wallclock < 0;

    return r;
}

int  schedule_have_timelimit(timelimit)
     const struct schedule_timelimit * timelimit;
{
    int r = 0;

#ifdef USE_CLOCK_MONOTONIC
    if ((time_t)-1 != timelimit->monotonic.tv_sec &&
	timelimit->monotonic.tv_sec > 0 &&
	timelimit->monotonic.tv_nsec >= 0 &&
	timelimit->monotonic.tv_nsec <= MAX_tv_nsec)
	r = 1;
#endif

#ifdef USE_GETTIMEOFDAY
     if (!r &&
	 (time_t)-1 != timelimit->timeofday.tv_sec &&
	 timelimit->timeofday.tv_sec > 0 &&
	 timelimit->timeofday.tv_usec >= 0 &&
	 timelimit->timeofday.tv_usec <=  MAX_tv_usec)
	 r = 1;
#endif

    if (!r &&
	(time_t)-1 != timelimit->wallclock &&
	timelimit->wallclock > 0)
	r = 1;

    return r;
}

/* Result malloced, caller must free */
char * schedule_timeout_string(next_timeout)
     const struct schedule_timelimit * next_timeout;
{
    char * result = NULL;
    char * A = NULL;
    char * B = NULL;

    if ((time_t)-1 == next_timeout->wallclock) {
	result = safe_strdup("wallclock time failed");
    } else if (next_timeout->wallclock > 0) {
	char *X = ctime(& (next_timeout->wallclock));
	int a;
	
	if (X && (a = strlen(X)))
	    result = elm_message(FRM("wallclock time %ld (%.*s)"),
				 (long)(next_timeout->wallclock),
				 a-1,X);
	else
	    result = elm_message(FRM("wallclock time %ld"),
				 (long)(next_timeout->wallclock));	
    }

#ifdef USE_GETTIMEOFDAY
    if ((time_t)-1 == next_timeout->timeofday.tv_sec)
	B = safe_strdup("time of day failed");
    else if (next_timeout->timeofday.tv_sec > 0 &&
	     next_timeout->timeofday.tv_usec >= 0 &&
	     next_timeout->timeofday.tv_usec <=  MAX_tv_usec) {
	
	if (next_timeout->wallclock != next_timeout->timeofday.tv_sec) {
	    char *X = ctime(& (next_timeout->timeofday.tv_sec));
	    int a;
	
	    if (X && (a = strlen(X)))
		B = elm_message(FRM("time of day %ld.%06ld (about %.*s)"),
				(long)(next_timeout->timeofday.tv_sec),
				(long)(next_timeout->timeofday.tv_usec),
				a-1,X);				
	    else
		goto msg1;
	} else {
	msg1:
	    B = elm_message(FRM("time of day %ld.%06ld"),
			    (long)(next_timeout->timeofday.tv_sec),
			    (long)(next_timeout->timeofday.tv_usec));
	}
    }
#endif

    if (B) {
	if (!result) {
	    result = B;
	} else {
	    result = strmcat(result," ~~ ");
	    result = strmcat(result,B);
	    free(B);
	}
	B      = NULL;
    }
    
#ifdef USE_CLOCK_MONOTONIC
    if ((time_t)-1 == next_timeout->monotonic.tv_sec) {
	A = safe_strdup("CLOCK_MONOTONIC time failed");
    } else if (next_timeout->monotonic.tv_sec > 0 &&
	       next_timeout->monotonic.tv_nsec >= 0 &&
	       next_timeout->monotonic.tv_nsec <= MAX_tv_nsec) {

	A = elm_message(FRM("CLOCK_MONOTONIC time %ld.%09ld"),
			(long)(next_timeout->monotonic.tv_sec),
			next_timeout->monotonic.tv_nsec);
    }    
#endif

    if (A) {
	if (!result) {
	    result = A;
	} else {
	    result = strmcat(result," ~~ ");
	    result = strmcat(result,A);
	    free(A);
	}
	A      = NULL;
    }
    
    if (!result)
	result = safe_strdup("none");
    
    return result;
}

#if defined(SELECT) || defined(I_SYS_SELECT) || defined(S_TIMEVAL) || defined(USE_GETTIMEOFDAY)

int  schedule_timeout_to_timeval(next_timeout,timeout)
     const struct schedule_timelimit * next_timeout;
     struct timeval * timeout;
{
    struct schedule_timelimit current = NO_schedule_timelimit;
    int r = 0;

    schedule_set_current_time(&current);

#ifdef USE_CLOCK_MONOTONIC
    if ((time_t)-1 != next_timeout->monotonic.tv_sec &&
	(time_t)-1 != current.monotonic.tv_sec &&
	next_timeout->monotonic.tv_sec > 0 &&
	current.monotonic.tv_sec > 0 &&
	next_timeout->monotonic.tv_nsec >= 0 &&
	next_timeout->monotonic.tv_nsec <= MAX_tv_nsec &&	
	current.monotonic.tv_nsec >= 0 &&
	current.monotonic.tv_nsec <= MAX_tv_nsec) {

	DPRINT(Debug,21,(&Debug,"schedule_timeout_to_timeval: next timeout CLOCK_MONOTONIC %ld.%09ld\n",
			 (long)(next_timeout->monotonic.tv_sec),
			 next_timeout->monotonic.tv_nsec));
	
	if (next_timeout->monotonic.tv_sec > current.monotonic.tv_sec) {

	    timeout->tv_sec =
		next_timeout->monotonic.tv_sec -
		current.monotonic.tv_sec;

	    if (next_timeout->monotonic.tv_nsec >= current.monotonic.tv_nsec) {
		timeout->tv_usec =
		    ( next_timeout->monotonic.tv_nsec - current.monotonic.tv_nsec)
		    / 1000;
	    } else {
		timeout->tv_usec =
		    1000000 +
		    ( next_timeout->monotonic.tv_nsec - current.monotonic.tv_nsec)
		    / 1000;
		timeout->tv_sec--;
	    }
	    
	} else if (next_timeout->monotonic.tv_sec == current.monotonic.tv_sec &&
		   next_timeout->monotonic.tv_nsec > current.monotonic.tv_nsec) {
	    
	    timeout->tv_sec = 0;
	    timeout->tv_usec =
		( next_timeout->monotonic.tv_nsec - current.monotonic.tv_nsec)
		/ 1000;

	} else {
	     timeout->tv_sec  = 0;
	     timeout->tv_usec = 0;
	}

	DPRINT(Debug,21,(&Debug,"schedule_timeout_to_timeval:  timeout %ld.%06ld\n",
			 (long)(timeout->tv_sec),
			 timeout->tv_usec));
	
	r = 1;
    }
#endif

#ifdef USE_GETTIMEOFDAY
    if (!r &&
	(time_t)-1 != next_timeout->timeofday.tv_sec &&
	(time_t)-1 != current.timeofday.tv_sec &&
	next_timeout->timeofday.tv_sec > 0 &&
	current.timeofday.tv_sec > 0 &&
	next_timeout->timeofday.tv_usec >= 0 &&
	current.timeofday.tv_usec >= 0 &&
	next_timeout->timeofday.tv_usec <=  MAX_tv_usec &&
	current.timeofday.tv_usec <=  MAX_tv_usec) {
	
	DPRINT(Debug,21,(&Debug,
			 "schedule_timeout_to_timeval: next timeout time of day  %ld.%06ld\n",
			 (long)(next_timeout->timeofday.tv_sec),
			 (long)(next_timeout->timeofday.tv_usec)));

	if (next_timeout->timeofday.tv_sec > current.timeofday.tv_sec) {
	    timeout->tv_sec =
		next_timeout->timeofday.tv_sec -
		current.timeofday.tv_sec;
	    
	    if (next_timeout->timeofday.tv_usec >= current.timeofday.tv_usec) {
		timeout->tv_usec =
		    next_timeout->timeofday.tv_usec - current.timeofday.tv_usec;
	    } else {
		timeout->tv_usec =
		    1000000 +
		    next_timeout->timeofday.tv_usec - current.timeofday.tv_usec;
		timeout->tv_sec--;
	    }

	} else if (next_timeout->timeofday.tv_sec == current.timeofday.tv_sec &&
		   next_timeout->timeofday.tv_usec > current.timeofday.tv_usec) {
	    
	     timeout->tv_sec = 0;
	     timeout->tv_usec =
		 next_timeout->timeofday.tv_usec - current.timeofday.tv_usec;
	}  else {
	     timeout->tv_sec  = 0;
	     timeout->tv_usec = 0;
	}

	DPRINT(Debug,21,(&Debug,"schedule_timeout_to_timeval:  timeout %ld.%06ld\n",
			 (long)(timeout->tv_sec),
			 timeout->tv_usec));
	
	r = 1;
    }
#endif

#ifdef USE_GETTIMEOFDAY
    if (!r &&
	(time_t)-1 != next_timeout->timeofday.tv_sec &&
	(time_t)-1 != current.timeofday.tv_sec &&
	next_timeout->timeofday.tv_sec > 0 &&
	current.timeofday.tv_sec > 0 &&
	next_timeout->timeofday.tv_usec >= 0 &&
	current.timeofday.tv_usec >= 0 &&
	next_timeout->timeofday.tv_usec <=  MAX_tv_usec &&
	current.timeofday.tv_usec <=  MAX_tv_usec) {
	
	DPRINT(Debug,21,(&Debug,
			 "schedule_timeout_to_timeval: next timeout time of day  %ld.%06ld\n",
			 (long)(next_timeout->timeofday.tv_sec),
			 (long)(next_timeout->timeofday.tv_usec)));

	if (next_timeout->timeofday.tv_sec > current.timeofday.tv_sec) {
	    timeout->tv_sec =
		next_timeout->timeofday.tv_sec -
		current.timeofday.tv_sec;
	    
	    if (next_timeout->timeofday.tv_usec >= current.timeofday.tv_usec) {
		timeout->tv_usec =
		    next_timeout->timeofday.tv_usec - current.timeofday.tv_usec;
	    } else {
		timeout->tv_usec =
		    1000000 +
		    next_timeout->timeofday.tv_usec - current.timeofday.tv_usec;
		timeout->tv_sec--;
	    }

	} else if (next_timeout->timeofday.tv_sec == current.timeofday.tv_sec &&
		   next_timeout->timeofday.tv_usec > current.timeofday.tv_usec) {
	    
	    timeout->tv_sec = 0;
	    timeout->tv_usec =
		 next_timeout->timeofday.tv_usec - current.timeofday.tv_usec;

	}  else {
	     timeout->tv_sec  = 0;
	     timeout->tv_usec = 0;
	}

	DPRINT(Debug,21,(&Debug,"schedule_timeout_to_timeval:  timeout %ld.%06ld\n",
			 (long)(timeout->tv_sec),
			 timeout->tv_usec));
	
	r = 1;	
    }
#endif

    if (!r &&
	(time_t)-1 != next_timeout->wallclock &&
	(time_t)-1 != current.wallclock &&
	next_timeout->wallclock > 0 &&
	current.wallclock > 0) {

	DPRINT(Debug,21,(&Debug,"schedule_timeout_to_timeval: next timeout wallclock %ld\n",
			 (long)(next_timeout->wallclock)));
	       	
	if (next_timeout->wallclock >  current.wallclock) {
	    timeout->tv_sec =
		next_timeout->wallclock - current.wallclock;
	    timeout->tv_usec    = 0;
	} else {
	    timeout->tv_sec  = 0;
	    timeout->tv_usec = 0;
	}

	DPRINT(Debug,21,(&Debug,"schedule_timeout_to_timeval:  timeout %ld.%06ld\n",
			 (long)(timeout->tv_sec),
			 timeout->tv_usec));

	r = 1;
    }

    
    
    return r;
}
#endif

int  schedule_timeout_to_timeout_msec(next_timeout,timeout_msec)
     const struct schedule_timelimit * next_timeout;
     int                             * timeout_msec;
{
    int r = 0;

    struct schedule_timelimit  current = NO_schedule_timelimit;

    schedule_set_current_time(&current);

#ifdef USE_CLOCK_MONOTONIC
    if ((time_t)-1 != next_timeout->monotonic.tv_sec &&
	(time_t)-1 != current.monotonic.tv_sec &&
	next_timeout->monotonic.tv_sec > 0 &&
	current.monotonic.tv_sec > 0 &&
	next_timeout->monotonic.tv_nsec >= 0 &&
	next_timeout->monotonic.tv_nsec <= MAX_tv_nsec &&	
	current.monotonic.tv_nsec >= 0 &&
	current.monotonic.tv_nsec <= MAX_tv_nsec) {

	DPRINT(Debug,21,(&Debug,"schedule_timeout_to_timeout_msec: next timeout CLOCK_MONOTONIC %ld.%09ld\n",
			 (long)(next_timeout->monotonic.tv_sec),
			 next_timeout->monotonic.tv_nsec));

	if (next_timeout->monotonic.tv_sec > current.monotonic.tv_sec) {

	    long diff =   next_timeout->monotonic.tv_sec -
		current.monotonic.tv_sec;

	    if (diff < (long)INT_MAX / 1000) {
		int add;
		
		*timeout_msec = diff * 1000;

		/* may be negative */
		add = ( next_timeout->monotonic.tv_nsec - current.monotonic.tv_nsec)
		    / 1000000;

		if (add < 0)
		    *timeout_msec += add;
		else if (*timeout_msec <  INT_MAX  - add)
		    *timeout_msec += add;
		else
		    *timeout_msec = INT_MAX; 
		 	      
	    } else
		*timeout_msec = INT_MAX;

	}  else if (next_timeout->monotonic.tv_sec == current.monotonic.tv_sec &&
		   next_timeout->monotonic.tv_nsec > current.monotonic.tv_nsec) {

	    *timeout_msec =  ( next_timeout->monotonic.tv_nsec - current.monotonic.tv_nsec)
		/ 1000000;
		
	   
	} else
	    *timeout_msec = 0;
	
	r = 1;

	DPRINT(Debug,21,(&Debug,"schedule_timeout_to_timeout_msec:  timeout %d.%03d\n",
			 (*timeout_msec) / 1000,
			 (*timeout_msec) % 1000));
    }
#endif

#ifdef USE_GETTIMEOFDAY
    if (!r &&
	(time_t)-1 != next_timeout->timeofday.tv_sec &&
	(time_t)-1 != current.timeofday.tv_sec &&
	next_timeout->timeofday.tv_sec > 0 &&
	current.timeofday.tv_sec > 0 &&
	next_timeout->timeofday.tv_usec >= 0 &&
	current.timeofday.tv_usec >= 0 &&
	next_timeout->timeofday.tv_usec <=  MAX_tv_usec &&
	current.timeofday.tv_usec <=  MAX_tv_usec) {

	DPRINT(Debug,21,(&Debug,
			 "schedule_timeout_to_timeout_msec: next timeout time of day %ld.%06ld\n",
			 (long)(next_timeout->timeofday.tv_sec),
			 (long)(next_timeout->timeofday.tv_usec)));

	if (next_timeout->timeofday.tv_sec > current.timeofday.tv_sec) {
	    
	    long diff =  next_timeout->timeofday.tv_sec -
		current.timeofday.tv_sec;

	    if (diff < (long)INT_MAX / 1000) {
		int add;
		
		*timeout_msec = diff * 1000;

		/* may be negative */
		add = ( next_timeout->timeofday.tv_usec - current.timeofday.tv_usec )
		    / 1000;

		if (add < 0)
		    *timeout_msec += add;
		else if (*timeout_msec <  INT_MAX  - add)
		    *timeout_msec += add;
		else
		    *timeout_msec = INT_MAX;
	    } else
		*timeout_msec = INT_MAX;

	} else if (next_timeout->timeofday.tv_sec == current.timeofday.tv_sec &&
		   next_timeout->timeofday.tv_usec > current.timeofday.tv_usec) {

	    *timeout_msec = ( next_timeout->timeofday.tv_usec - current.timeofday.tv_usec )
		/  1000;

	} else
	    *timeout_msec = 0;
	
	r = 1;

	DPRINT(Debug,21,(&Debug,"schedule_timeout_to_timeout_msec:  timeout %d.%03d\n",
			 (*timeout_msec) / 1000,
			 (*timeout_msec) % 1000));
    }
#endif
    
    if (!r &&
	(time_t)-1 != next_timeout->wallclock &&
	(time_t)-1 != current.wallclock &&
	next_timeout->wallclock > 0 &&
	current.wallclock > 0) {

	DPRINT(Debug,21,(&Debug,"schedule_timeout_to_timeout_msec: next timeout wallclock %ld\n",
			 (long)(next_timeout->wallclock)));
	
	if (next_timeout->wallclock >  current.wallclock) {

	    long diff =  next_timeout->wallclock - current.wallclock;
	    
	    if (diff < (long)INT_MAX / 1000)
		*timeout_msec = diff * 1000;
	    else
		*timeout_msec = INT_MAX;
	} else
	     *timeout_msec = 0;

	r = 1;

	DPRINT(Debug,21,(&Debug,"schedule_timeout_to_timeout_msec:  timeout %d.%03d\n",
			 (*timeout_msec) / 1000,
			 (*timeout_msec) % 1000));
    }
	
    return r;
}

int  schedule_timeout_to_timeout_sec(next_timeout,timeout_sec)
     const struct schedule_timelimit * next_timeout;
     unsigned int                    * timeout_sec;
{
    int r = 0;

    struct schedule_timelimit  current = NO_schedule_timelimit;

    schedule_set_current_time(&current);

#ifdef USE_CLOCK_MONOTONIC
    if ((time_t)-1 != next_timeout->monotonic.tv_sec &&
	(time_t)-1 != current.monotonic.tv_sec &&
	next_timeout->monotonic.tv_sec > 0 &&
	current.monotonic.tv_sec > 0 &&
	next_timeout->monotonic.tv_nsec >= 0 &&
	next_timeout->monotonic.tv_nsec <= MAX_tv_nsec &&	
	current.monotonic.tv_nsec >= 0 &&
	current.monotonic.tv_nsec <= MAX_tv_nsec) {

	DPRINT(Debug,21,(&Debug,
			 "schedule_timeout_to_timeout_sec: next timeout CLOCK_MONOTONIC %ld.%09ld\n",
			 (long)(next_timeout->monotonic.tv_sec),
			 next_timeout->monotonic.tv_nsec));

	
	if (next_timeout->monotonic.tv_sec > current.monotonic.tv_sec) {
	    
	    long diff =  next_timeout->monotonic.tv_sec - current.monotonic.tv_sec;
	    
	    if (diff < (long)INT_MAX)
		*timeout_sec = diff;
	    else
		*timeout_sec = INT_MAX;
	} else if (next_timeout->monotonic.tv_sec == current.monotonic.tv_sec &&
		   next_timeout->monotonic.tv_nsec > current.monotonic.tv_nsec) 
	    *timeout_sec = 1;
	else
	    *timeout_sec = 0;
   	
	r  = 1;

	DPRINT(Debug,21,(&Debug,"schedule_timeout_to_timeout_sec:  timeout %u\n",
			 *timeout_sec));
    }
#endif

#ifdef USE_GETTIMEOFDAY
    if (!r &&
	(time_t)-1 != next_timeout->timeofday.tv_sec &&
	(time_t)-1 != current.timeofday.tv_sec &&
	next_timeout->timeofday.tv_sec > 0 &&
	current.timeofday.tv_sec > 0 &&
	next_timeout->timeofday.tv_usec >= 0 &&
	current.timeofday.tv_usec >= 0 &&
	next_timeout->timeofday.tv_usec <=  MAX_tv_usec &&
	current.timeofday.tv_usec <=  MAX_tv_usec) {

	DPRINT(Debug,21,(&Debug,
			 "schedule_timeout_to_timeout_sec: next timeout time of day %ld.%06ld\n",
			 (long)(next_timeout->timeofday.tv_sec),
			 (long)(next_timeout->timeofday.tv_usec)));

	if (next_timeout->timeofday.tv_sec > current.timeofday.tv_sec) {
	    
	    long diff =  next_timeout->timeofday.tv_sec -
		current.timeofday.tv_sec;
	    
	    if (diff < (long)INT_MAX)
		*timeout_sec = diff;
	    else
		*timeout_sec = INT_MAX;
	} else if (next_timeout->timeofday.tv_sec == current.timeofday.tv_sec &&
		   next_timeout->timeofday.tv_usec > current.timeofday.tv_usec)
	    *timeout_sec = 1;
	else
	    *timeout_sec = 0;
   	
	r  = 1;

	DPRINT(Debug,21,(&Debug,"schedule_timeout_to_timeout_sec:  timeout %u\n",
			 *timeout_sec));
    }
#endif
    
    if (!r &&
	(time_t)-1 != next_timeout->wallclock &&
	(time_t)-1 != current.wallclock &&
	next_timeout->wallclock > 0 &&
	current.wallclock > 0) {

	DPRINT(Debug,21,(&Debug,"schedule_timeout_to_timeout_msec: next timeout wallclock %ld\n",
			 (long)(next_timeout->wallclock)));
	
	if (next_timeout->wallclock >  current.wallclock) {
	    
	    long diff =  next_timeout->wallclock - current.wallclock;
	    
	    if (diff < (long)INT_MAX)
		*timeout_sec = diff;
	    else
		*timeout_sec = INT_MAX;
	} else
	    *timeout_sec = 0;
	
	r  = 1;

	DPRINT(Debug,21,(&Debug,"schedule_timeout_to_timeout_sec:  timeout %u\n",
			 *timeout_sec));
    }
    
    return r;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
