static char rcsid[] = "@(#)$Id: service_list.c,v 2.55 2025/05/29 05:42:57 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.55 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "elm_defs.h"
#include "ss_imp.h"
#include "schedule_time.h"
#include "connection_imp.h"
#ifdef USE_DLOPEN
#include "shared_imp.h"
#endif
#include "s_me.h"
#include "s_elm.h"
#include "rc_imp.h"
#include "service_imp.h"

DEBUG_VAR(Debug,__FILE__,"net");

/* For INT_MAX and LONG_MAX */
#include <limits.h>

/* What is maximum value of time_t ?
   LONG_MAX = 2147483647   fails at Tue Jan 19 03:14:07 2038 UTC

   Also INT_MAX = 2147483647   fails at Tue Jan 19 03:14:07 2038 UTC
*/

/* If time_t is smaller that long, this may produce error ?
   
   it is not possible to test sizeof on #if directive

   Also this does not work if time_t is unsigned type

   limits.h does not tell range of time_t !
*/

const time_t time_MAX = 
    (time_t)((sizeof (time_t) < sizeof (long)) ? INT_MAX : LONG_MAX);

static char *us2s P_((unsigned char *str));
static char *us2s(str) 
     unsigned char *str;
{
    return (char *)str;
}

static const unsigned char *cs2us P_((const char *str));
static const unsigned char *cs2us(str) 
     const char *str;
{
    return (const unsigned char *)str;
}


/* Returns new len */
int trim_whitespace_from_end(buf,l1)
     char *buf;
     int l1;
{
    int x = l1;
    int old = l1;


    if (l1 < 1) 
	return 0;
    
    while (x-- > 0 && whitespace(buf[x])) {
	buf[x] = '\0';
	l1 = x;
    }
    
    if (old != l1) {
	DPRINT(Debug,12,(&Debug,
			 "trim_whitespace_from_end=%d (original len %d), buf=\"%.*s\"\n",
			 l1,old,l1,buf));
    }

    return l1;
}


/* Return 1 if @charset seen */
int read_charset_tag(buf,l1,filename,lineno, errors,cs,csn) 
     char *buf; 
     int l1; 
     const char * filename;
     const int lineno;
     int *errors;
     charset_t *cs;
     const char ** csn;
{
    if (l1 > 8 && 0 == strncmp(buf,"@charset",8)) {
	charset_t A;
	char * c = buf +8;
	
	while (*c && whitespace (*c)) /* skip whitespace */
	    c++;
	
	if ('=' != *c) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadCharsetLine,
			      "%s: %d: Bad @charset line: %s"),
		      filename,lineno, buf);
	    if (errors)
		(*errors) ++;
	    
	    return 1;
	}
	c++;

	while (*c && whitespace (*c)) /* skip whitespace */
	    c++;
	
	A = MIME_name_to_charset(c,0);
	
	if (A) {
	    const char * csn1 = get_charset_MIME_name(A);
	    
	    if (!(CS_mapping & charset_properties(A))) {
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeUnsupportedCharsetLine,
				  "%s: %d: Unsupported @charset '%s' on line: %s"),
			  filename,lineno,csn1 ? csn1 : c,buf);
		if (errors)
		    (*errors) ++;
		
		return 1;
	    }
	    
	    if (csn)
		*csn = csn1;

	    if (cs) {
		*cs = A;
		
		if (csn1) {
		    DPRINT(Debug,2,(&Debug, 
				    "read_charset_tag: %s: charset: %s (%s)\n",
				    filename,c,csn1));
		    
		} else {
		    DPRINT(Debug,2,(&Debug, 
				    "read_charset_tag: %s: charset: %s\n",
				    filename,c));
		}
	    }
	    
	} else {
	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeUnsupportedCharsetLine,
			      "%s: %d: Unsupported @charset '%s' on line: %s"),
		      filename,lineno,c,buf);
	    if (errors)
		(*errors) ++;
	}
	return 1;
    }

    return 0;
}

/* Mallocs to result */
char * convert_to_fileset2(value,fileset,err)
     const struct string *value;
     charset_t fileset; 
     int *err;
{
    int ERRORS;
    struct string * S2 = NULL;
    unsigned char * res = NULL;	

    S2 = convert_string2(fileset,value,&ERRORS);
    (*err) += ERRORS;

    /* result is malloced */
    res = stream_from_string(S2,0,NULL);
    free_string(&S2);
    
    return us2s(res);
}

/* Mallocs to result, if conversion, otherwise returns value */
char * convert_to_fileset(value,fileset,err)
     char *value; 
     charset_t fileset;
     int *err;
{
    if (system_charset != fileset) {
	struct string  * S = new_string(system_charset);

	int ERRORS;
	int len = strlen(value);
	int rs = add_streambytes_to_string(S,len,cs2us(value),&ERRORS);
	char * ret = NULL;

	(*err) += ERRORS;
		
	if (rs < len) {
	    
	    free_string(&S);
	    return NULL;
	}
	
	ret = convert_to_fileset2(S,fileset,err);
	free_string(&S);
	return ret;
    } else
	return value;

}

#ifdef REMOTE_MBX

/* Seems that h_errno is macro on AIX */
#ifndef h_errno
extern int h_errno;
#endif

#include <errno.h>

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
extern int errno;
#endif

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}


/* Hard coded */
static struct service_type {
    const  char * name;
    int           flags;
    PORTS         defport;
    const char  * have_init_tls; /* message text for 
				    remote_account_init_tls */
} SERVICE_TYPES[] = {
    { "*",       
      STFLAG_browser|STFLAG_mbox|
      STFLAG_is_imap|STFLAG_is_pop |
      STFLAG_smtp_like|
      STFLAG_is_submission|STFLAG_is_smtp,
      PORT_end, NULL},

    /* plain text (+ possible STARTTLS) ------------------------------------ */
    
    { "imap",    STFLAG_browser|STFLAG_mbox|STFLAG_is_imap,    PORT_imap4, 
      NULL
    },
    { "pop",     STFLAG_mbox|STFLAG_is_pop,                    PORT_pop3,
      NULL
    },

    { "submission", STFLAG_smtp_like|STFLAG_is_submission,     PORT_submission, 
      NULL
    },
    { "smtp",    STFLAG_smtp_like|STFLAG_is_smtp,              PORT_smtp,
      NULL
    },


    /* Implict TLS   -- notice 'have_init_tls' field ----------------------- */
    
    { "smtps",    STFLAG_smtp_like|STFLAG_require_port, 
      PORT_end  /* no port given */,
      "SMTPs"   /* == have_init_tls */
    },


    { "imaps",   STFLAG_browser|STFLAG_mbox|STFLAG_is_imap,    PORT_imaps,
      "IMAPs"   /* == have_init_tls */
    },
    { "pops",    STFLAG_mbox|STFLAG_is_pop,                    PORT_pop3s,
      "POPs"    /* == have_init_tls */
    },

    { "submissions", STFLAG_smtp_like|STFLAG_is_submission,    PORT_submissions,
      "SUBMISSIONs"  /* == have_init_tls */
    },
      
    { NULL,      0,                                            PORT_end, 
      NULL
    },
};

static const int SERVICE_TYPES_count = (sizeof SERVICE_TYPES) / sizeof (SERVICE_TYPES[0]);

struct service_type * const any_service_type = &(SERVICE_TYPES[0]);

						 
int service_type_flags(st,mask)
     const struct service_type *st;
     int mask;
{
    return st->flags & mask;
}

const char * service_type_name(st)
     const struct service_type *st;
{
    return st->name;
}

PORTS service_type_defport(st)
     const struct service_type *st;
{
    return st->defport;
}


static const char * STFLAG_debug_info P_((int flag, char buffer[], size_t buffer_size));
static const char * STFLAG_debug_info(flag,buffer,buffer_size)
     int flag;
     char buffer[];
     size_t buffer_size;
{
    if (buffer_size < 4)
	return "**";

    if (!flag)
	return "none";
    
    buffer[0] = '\0';
    buffer[2] = '\0';   /* In case there is unknown bits set */
    
    if (ison(flag,STFLAG_browser))
	strfcat(buffer,", STFLAG_browser", buffer_size);
    if (ison(flag,STFLAG_mbox))
	strfcat(buffer,", STFLAG_mbox",buffer_size);	
    if (ison(flag,STFLAG_smtp_like))
	strfcat(buffer,", STFLAG_smtp_like",buffer_size);
    if (ison(flag,STFLAG_require_port))
	strfcat(buffer,", STFLAG_require_port",buffer_size);

    if (ison(flag,STFLAG_is_imap))
	strfcat(buffer,", STFLAG_is_imap",buffer_size);
    if (ison(flag,STFLAG_is_pop))
	strfcat(buffer,", STFLAG_is_pop",buffer_size);
    if (ison(flag,STFLAG_is_submission))
	strfcat(buffer,", STFLAG_is_submission",buffer_size);
    if (ison(flag,STFLAG_is_smtp))
	strfcat(buffer,", STFLAG_is_smtp",buffer_size);
    
    return buffer+2;
}

void init_enum_service_type(est,start_tls_idx,any_mask,require_mask)
     struct enum_service_type *est;
     enum init_tls_pass        start_tls_idx;
     int                       any_mask;
     int                       require_mask;
{
    bzero (est, sizeof (*est));

    if (start_tls_idx < no_init_tls || start_tls_idx >= NUM_init_tls_pass)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "init_enum_service_type",
	      "Bad start_tls_idx",0);

    
    est->magic        = ENUM_SERVICE_TYPE_magic;
    est->idx          = 0;
    est->tls_idx      = start_tls_idx;
    est->any_mask     = any_mask;
    est->mask_idx     = -1;
    est->mask_list    = NULL;
    est->require_mask = require_mask;
}

void init_enum_service_list(est,start_tls_idx,mask_list,require_mask)
     struct enum_service_type * est;
     enum init_tls_pass        start_tls_idx;
     const int                * mask_list;
     int                       require_mask;
{
    bzero (est, sizeof (*est));

    if (start_tls_idx < no_init_tls || start_tls_idx >= NUM_init_tls_pass)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "init_enum_service_list",
	      "Bad start_tls_idx",0);
    
    est->magic        = ENUM_SERVICE_TYPE_magic;
    est->idx          = 0;
    est->tls_idx      = start_tls_idx;
    est->any_mask     = 0;
    est->mask_idx     = 0;
    est->mask_list    = mask_list;
    est->require_mask = require_mask;

}


/* Passing copy of this value produces empty enumeration */
const struct enum_service_type NULL_enum_service_type = {
    ENUM_SERVICE_TYPE_magic, 0, init_tls_default, -1, 0 , NULL, 0
};

const struct service_type * enumerate_service_type(est)
     struct enum_service_type *est;
{
    const struct service_type * res = NULL;
    
    int idx;
    enum init_tls_pass tls_idx;
    int any_mask,require_mask,mask_idx;

    if (ENUM_SERVICE_TYPE_magic != est->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "enumerate_service_type",
	      "Bad magic (enum_service_type)",0);

    idx          = est->idx;
    tls_idx      = est->tls_idx;
    any_mask     = est->any_mask;
    require_mask = est->require_mask;
    mask_idx     = est->mask_idx;
    
    if (idx < 0 || idx >= SERVICE_TYPES_count)
	goto done;
    if (tls_idx < no_init_tls || tls_idx >= NUM_init_tls_pass)
	goto done;
    if (mask_idx < -1 ||
	(mask_idx >= 0 && ! est->mask_list) ||
	(mask_idx >= 0 && ! est->mask_list[mask_idx]))
	goto done;
    
    DPRINT(Debug,14,(&Debug,
		     "enumerate_service_type:  starting idx %d ",
		     idx));
    if (SERVICE_TYPES[idx].name) {
	DPRINT(Debug,14,(&Debug," (name %s)",SERVICE_TYPES[idx].name));
    }    
    DPRINT(Debug,14,(&Debug," tls_idx %d",
		     tls_idx));
    switch (tls_idx) {
    case no_init_tls:        DPRINT(Debug,14,(&Debug," no_init_tls"));        break;
    case init_tls_default:   DPRINT(Debug,14,(&Debug," init_tls_default"));   break;
    case try_Implicit_TLS:   DPRINT(Debug,14,(&Debug," try_Implicit_TLS"));   break;
    case try_no_init_tls:    DPRINT(Debug,14,(&Debug," try_no_init_tls")); break;
    case NUM_init_tls_pass:  DPRINT(Debug,14,(&Debug," NUM_init_tls_pass"));  break;
    }
    if (any_mask) {
	char debug_buffer[80];
	DPRINT(Debug,14,(&Debug," any_mask %d (%s)",
			 any_mask,STFLAG_debug_info(any_mask,
						    debug_buffer, sizeof debug_buffer)));
    }    
    DPRINT(Debug,14,(&Debug,", mask_idx %d",mask_idx));

    if (mask_idx >= 0) {
	any_mask = est->mask_list[mask_idx];

	if (any_mask) {
	    char debug_buffer4[80];
	    DPRINT(Debug,14,(&Debug," => mask %d (%s)",
			     any_mask,STFLAG_debug_info(require_mask,
							debug_buffer4, sizeof debug_buffer4)));
	}
    }
    DPRINT(Debug,14,(&Debug,"\n"));

    
    do {

	if (0 == any_mask) {
	    DPRINT(Debug,14,(&Debug,
			     "enumerate_service_type: idx %d -- empty mask -- skipping scan with mask_idx %d\n",
			     idx,mask_idx));
	    goto next_mask;
	}
	
	if (any_service_type == &(SERVICE_TYPES[idx])) {
	    idx++;
	    goto next_service;
	}
	    
	if (idx >= SERVICE_TYPES_count || NULL == SERVICE_TYPES[idx].name) {
	    
	next_mask:	    
	    if (mask_idx < -1 ||
		(mask_idx >= 0 && ! est->mask_list) ||
		(mask_idx >= 0 && ! est->mask_list[mask_idx])) {
		
		DPRINT(Debug,14,(&Debug,
				 "enumerate_service_type: bad mask_idx %d\n",
				 mask_idx));
		
	    } else if (est->mask_list) {
		char debug_buffer5[80];

		idx      = 0;
		mask_idx++;

		DPRINT(Debug,14,(&Debug,
				 "enumerate_service_type: idx %d mask_idx %d",
				 idx, mask_idx));

		any_mask = est->mask_list[mask_idx];

		DPRINT(Debug,14,(&Debug,
				 " mask %d (%s)\n",
				 any_mask,STFLAG_debug_info(any_mask,
							    debug_buffer5, sizeof debug_buffer5)));
		if (any_mask) {
		    goto next_service;
		}
	    }
	    
	    
	    /* Next tls_idx */
	    
	    if (tls_idx > init_tls_default &&
		tls_idx < NUM_init_tls_pass) { 
		char debug_buffer3[80];
		
		tls_idx++;
		idx      = 0;
		
		mask_idx     = -1; 
		any_mask     = est->any_mask;
		
		DPRINT(Debug,14,(&Debug,
				 "enumerate_service_type: idx %d tls_idx %d mask_idx %d mask %d (%s)\n",
				 idx,tls_idx,mask_idx,
				 any_mask,STFLAG_debug_info(any_mask,
							    debug_buffer3, sizeof debug_buffer3)));
				 
		if (tls_idx >= NUM_init_tls_pass)
		    break;
	    
		goto next_service;
	    }
	    	
	    break;
	}
	
	if (PORT_end == SERVICE_TYPES[idx].defport) {
	    idx++;
	    goto next_service;
	}

	if (isoff(SERVICE_TYPES[idx].flags,any_mask)) {
	    idx++;
	    goto next_service;
	}
	
	if (require_mask != (SERVICE_TYPES[idx].flags & require_mask)) {
	    idx++;
	    goto next_service;
	}
    
	switch (tls_idx) {
	    char debug_buffer6[80];
	case no_init_tls:
	case try_no_init_tls:
	    
	    if (NULL == SERVICE_TYPES[idx].have_init_tls) {
		res = &( SERVICE_TYPES[idx] );

		idx++;   /* Advance to next */		
		goto done;
	    }
	    break;
	case init_tls_default:

	    if (dt_flag_is_set(&use_tls,use_tls_implicit)) {
		DPRINT(Debug,14,(&Debug,
				 "enumerate_service_type: Implicit_TLS enabled\n"));

		tls_idx = try_Implicit_TLS;

	    } else {
		DPRINT(Debug,14,(&Debug,
				 "enumerate_service_type: No Implicit_TLS\n"));
		tls_idx = no_init_tls;
	    }
	    
	    idx          = 0;
	    mask_idx     = -1; 
	    any_mask     = est->any_mask;
	    
	    DPRINT(Debug,14,(&Debug,
			     "enumerate_service_type: idx %d tls_idx %d mask_idx %d mask %d (%s)\n",
			     idx,tls_idx,mask_idx,
			     any_mask,STFLAG_debug_info(any_mask,
							debug_buffer6, sizeof debug_buffer6)));
	    
	    goto next_service;

	case try_Implicit_TLS:
	    if (NULL != SERVICE_TYPES[idx].have_init_tls) {
		res = &( SERVICE_TYPES[idx] );

		idx++;   /* Advance to next */		
		goto done;
	    }
	    break;
	case NUM_init_tls_pass:  /* Not used */   break;
	}

	idx++;
	
    next_service:
	DPRINT(Debug,14,(&Debug,
			 "enumerate_service_type:  trying idx %d mask_idx %d tls_idx %d",
			 idx,mask_idx,tls_idx));
	switch (tls_idx) {
	case no_init_tls:        DPRINT(Debug,14,(&Debug," no_init_tls"));        break;
	case init_tls_default:   DPRINT(Debug,14,(&Debug," init_tls_default"));   break;
	case try_Implicit_TLS:   DPRINT(Debug,14,(&Debug," try_Implicit_TLS"));   break;
	case try_no_init_tls:    DPRINT(Debug,14,(&Debug," try_no_init_tls")); break;
	case NUM_init_tls_pass:  DPRINT(Debug,14,(&Debug," NUM_init_tls_pass"));  break;
	}
	DPRINT(Debug,14,(&Debug,"\n"));
        
    } while (idx < SERVICE_TYPES_count ||
	     mask_idx < 0 || (mask_idx >= 0 && est->mask_list && est->mask_list[mask_idx]) ||
	     tls_idx < NUM_init_tls_pass);
	
    DPRINT(Debug,14,(&Debug,
		     "enumerate_service_type:  failed at idx %d mask_idx %d tls_idx %d",
		     idx,mask_idx,tls_idx));
    switch (tls_idx) {
    case no_init_tls:        DPRINT(Debug,14,(&Debug," no_init_tls"));        break;
    case init_tls_default:   DPRINT(Debug,14,(&Debug," init_tls_default"));   break;
    case try_Implicit_TLS:   DPRINT(Debug,14,(&Debug," try_Implicit_TLS"));   break;
    case try_no_init_tls:    DPRINT(Debug,14,(&Debug," try_no_init_tls")); break;
    case NUM_init_tls_pass:  DPRINT(Debug,14,(&Debug," NUM_init_tls_pass"));  break;
    }
    DPRINT(Debug,14,(&Debug,"\n"));
    
 done:

    DPRINT(Debug,14,(&Debug,
		     "enumerate_service_type="));
    if (res) {
	char debug_buffer3[80];
	
	DPRINT(Debug,14,(&Debug,"%p name=%s flags=%d (%s) defport=%d",
			 res,res->name,
			 res->flags,
			 STFLAG_debug_info(res->flags,
					   debug_buffer3, sizeof debug_buffer3),
			 res->defport));

	if (res->have_init_tls) {
	    DPRINT(Debug,14,(&Debug," have_init_tls=%s",res->have_init_tls));
	}
	
    } else {
	DPRINT(Debug,14,(&Debug,"NULL"));
    }

    if (idx != est->idx) {
	DPRINT(Debug,14,(&Debug," ->idx=%d => %d",
			 est->idx,idx));
	est->idx = idx;
    }
    if (tls_idx != est->tls_idx) {
	DPRINT(Debug,14,(&Debug," ->tls_idx=%d => %d",
			 est->tls_idx,tls_idx));
	est->tls_idx = tls_idx;

	switch (tls_idx) {
	case no_init_tls:        DPRINT(Debug,14,(&Debug," no_init_tls"));        break;
	case init_tls_default:   DPRINT(Debug,14,(&Debug," init_tls_default"));   break;
	case try_Implicit_TLS:   DPRINT(Debug,14,(&Debug," try_Implicit_TLS"));   break;
	case try_no_init_tls:    DPRINT(Debug,14,(&Debug," try_no_init_tls")); break;
	case NUM_init_tls_pass:  DPRINT(Debug,14,(&Debug," NUM_init_tls_pass"));  break;
	}	
    }
    
    if (mask_idx != est->mask_idx) {
	DPRINT(Debug,14,(&Debug," ->mask_idx=%d => %d",
			 est->mask_idx,mask_idx));
	mask_idx = est->mask_idx;
    }
    DPRINT(Debug,14,(&Debug,"\n"));

    return res;
}

     


struct service_entry * malloc_service_entry(name,st,flags)
     const char *name;
     const struct service_type *st;
     int flags;
{
    struct service_entry * entry = safe_zero_alloc(sizeof (*entry));

    entry->refcount       = 1;
    
    entry->flags          = flags;           
    entry->official_name  = safe_strdup(name);
    entry->ip_literal_address.dummy = NULL;
    entry->ip_literal_addrsize = 0;

    entry->aliases_list   = 0;
    entry->aliases_count  = 0;
    entry->addr_list      = NULL;    
    entry->addr_count     = 0;
    
    entry->addr_name_list  = NULL;
    entry->addr_name_count = 0;

    entry->port_name_list  = NULL;
    entry->port_name_count = 0;

    entry->service        = st;
    /* ??? */
    entry->port_list      = NULL;
    entry->port_count     = 0;

    entry->option_list    = NULL;
    entry->option_count   = 0;

    entry->req_tls_peername_list    = NULL;
    entry->req_tls_peername_count   = 0;

    entry->magic = SERVICE_ENTRY_magic;
    return entry;
}

int  STFLAG_service_type(se)
     struct service_entry *se;
{
    if (SERVICE_ENTRY_magic != se->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "STFLAG_service_type",
	      "Bad magic (service_entry)",0);
    
    return se->service->flags & STFLAG_type_mask;
}

const char * have_init_tls(se)
     struct service_entry *se;
{
    if (SERVICE_ENTRY_magic != se->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "have_init_tls",
	      "Bad magic (service_entry)",0);

    return se->service->have_init_tls;
}

void add_addr_name_to_entry(entry,name,malloc_it)
     struct service_entry *entry;
     char *name;
     int malloc_it;
{
    if (SERVICE_ENTRY_magic != entry->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "add_addr_name_to_entry",
		      "Bad magic (service_entry)",0);

    entry->addr_name_list = 
	safe_array_realloc(entry->addr_name_list,
			   (entry->addr_name_count+1),
			   sizeof (entry->addr_name_list[0]));
    entry->addr_name_list[entry->addr_name_count++]
	= malloc_it ? safe_strdup(name) : name;
}

void add_port_name_to_entry(entry,name,malloc_it)
     struct service_entry *entry;
     char *name;
     int malloc_it;
{
    if (SERVICE_ENTRY_magic != entry->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "add_port_name_to_entry",
		      "Bad magic (service_entry)",0);

    entry->port_name_list = 
	safe_array_realloc(entry->port_name_list,
			   (entry->port_name_count+1),
			   sizeof (entry->port_name_list[0]));
    entry->port_name_list[entry->port_name_count++]
	= malloc_it ? safe_strdup(name) : name;
}

void l_append_name_list(aliases_list,aliases_count,alias,malloc_it)
     struct service_entry_name  ** aliases_list;
     int                         * aliases_count;
     struct service_entry_name   * alias;
     int malloc_it;
{
    struct service_entry_name  * ALIASES_LIST  = * aliases_list;
    int                          ALIASES_COUNT = * aliases_count;

    struct service_entry_name X = *alias;
    
    if (malloc_it && alias->aliasname) 
	X.aliasname = safe_strdup(alias->aliasname);

    ALIASES_LIST = 
	safe_array_realloc(ALIASES_LIST,
			   (ALIASES_COUNT+1),
			   sizeof (ALIASES_LIST[0]));
    ALIASES_LIST[ALIASES_COUNT++] = X;

    * aliases_list  = ALIASES_LIST;
    * aliases_count = ALIASES_COUNT;
}

static int is_same_se_name P_(( struct service_entry_name *A, 
				struct service_entry_name *B));
static int is_same_se_name(A,B)
     struct service_entry_name *A;
     struct service_entry_name *B;
{
    if (! A->aliasname &&
	! B->aliasname)
	return 1;
    if (! A->aliasname ||
	! B->aliasname)
	return 0;

    return 0 == istrcmp(A->aliasname,B->aliasname);
}

static void update_next_mailserv_cleanup P_((const struct schedule_timelimit * valid_until)) ;


void add_se_alias_to_entry(entry,name,malloc_it)
     struct service_entry      * entry;
     struct service_entry_name * name;
     int malloc_it;
{
    if (SERVICE_ENTRY_magic != entry->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "add_se_alias_to_entry",
	      "Bad magic (service_entry)",0);

   if (entry->aliases_list) {
       /* possible refilling expired aliases */

       int j;

       for (j = 0; j < entry->aliases_count; j++) {
	   if (is_same_se_name(name,
			       &(entry->aliases_list[j]))) {
	       DPRINT(Debug,10,(&Debug, 
				"add_se_alias_to_entry: %s alias %ss already on list",
				entry->official_name,
				name->aliasname ? name->aliasname : "<NULL>"));

	       if (schedule_extend_valid_until(& (entry->aliases_list[j].valid_until),
					       & (name->valid_until))) {
		   DPRINT(Debug,10,(&Debug, ", updating expiry time"));
	       }

	       DPRINT(Debug,10,(&Debug, "\n"));
	       return;
	   }
       }
   }

   update_next_mailserv_cleanup(& (name->valid_until));
   
   l_append_name_list(& (entry->aliases_list),
		      & (entry->aliases_count),
		      name,
		      malloc_it);
}

void add_alias_to_entry(entry,name, valid_until)
     struct service_entry *entry;
     const char *name;   
     const struct schedule_timelimit * valid_until; 
{
    struct service_entry_name EN;

    if (SERVICE_ENTRY_magic != entry->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "add_alias_to_entry",
	      "Bad magic (service_entry)",0);

    EN.aliasname   = safe_strdup(name);
    EN.valid_until = *valid_until;

    add_se_alias_to_entry(entry,&EN,0 /* do not malloc */);
}

static int is_same_se_address P_((struct service_entry_addr *A,
				  struct service_entry_addr *B)); 

void l_append_addr_list(addr_list,addr_count,addr,malloc_it)
     struct service_entry_addr  ** addr_list;
     int                         * addr_count;
     struct service_entry_addr *addr;
     int malloc_it;
{
    struct service_entry_addr  * ADDR_LIST  = *addr_list;
    int                         ADDR_COUNT = *addr_count;
    
    struct service_entry_addr X = *addr;
    
    ADDR_LIST = safe_array_realloc(ADDR_LIST, (ADDR_COUNT+1),
				   sizeof (ADDR_LIST[0]));
    
    if (malloc_it) {
	if (addr->address.dummy && addr->addrsize) {

	    X.address.dummy = safe_malloc(addr->addrsize);
	    
	    memcpy(X.address.dummy,addr->address.dummy,
		   addr->addrsize);
	    X.addrsize = addr->addrsize;
		   
	} else {
	    X.address.dummy    = NULL;
	    X.addrsize = 0;
	}

	if (addr->hostname)
	    X.hostname = safe_strdup(addr->hostname);
    }

    ADDR_LIST[ADDR_COUNT++] = X;

    *addr_list  = ADDR_LIST;
    *addr_count = ADDR_COUNT;
}


void add_se_addr_to_entry(entry,addr,malloc_it)
     struct service_entry *entry;
     struct service_entry_addr *addr;
     int malloc_it;
{

    if (SERVICE_ENTRY_magic != entry->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "add_se_addr_to_entry",
	      "Bad magic (service_entry)",0);

    if (entry->addr_list) {
	/* possible refilling expired addresses */
	int j;
	
	for (j = 0; j < entry->addr_count; j++) {
	    if (is_same_se_address(addr,& (entry->addr_list[j]))) {
		DPRINT(Debug,10,(&Debug, 
				 "add_se_addr_to_entry: %s address already on list",
				 entry->official_name));

		if (schedule_extend_valid_until(& (entry->addr_list[j].valid_until),
						& (addr->valid_until)))  {
		    DPRINT(Debug,10,(&Debug, ", updating expiry time"));
		}

		DPRINT(Debug,10,(&Debug, "\n"));
		return;
	    }				   
	}
    }

    update_next_mailserv_cleanup(& (addr->valid_until));
    
    l_append_addr_list(& (entry->addr_list),
		       & (entry->addr_count),
		       addr,malloc_it);
}

static void add_addr2_to_entry P_((struct service_entry   * entry,
				   size_t                   addr_size,
				   const union SOCKADDR_ptr addr,
				   const struct schedule_timelimit *   valid_until,
				   const char             * hostname));
static void add_addr2_to_entry(entry,addr_size,addr,valid_until,hostname)
     struct service_entry   * entry;
     size_t                   addr_size;
     const union SOCKADDR_ptr addr;
     const struct schedule_timelimit * valid_until;
     const char             * hostname;
{
    struct service_entry_addr EA;

    if (SERVICE_ENTRY_magic != entry->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "add_addr2_to_entry",
	      "Bad magic (service_entry)",0);

    /* bzero is defined hdrs/elm_defs.h */
    bzero ((void *)&EA, sizeof (EA));

    if (! addr.dummy || ! addr_size) {
	DPRINT(Debug,12,(&Debug,"add_addr2_to_entry: %s: No address, size %d\n",
			 hostname ? hostname : "<NULL>",
			 addr_size));
	return;	
    }

    EA.addrsize = addr_size;
    EA.address  = addr;
    EA.valid_until   = *valid_until;
    EA.hostname      = hostname ? safe_strdup(hostname) : NULL;

    add_se_addr_to_entry(entry,&EA,1 /* malloc copy */);
    if (EA.hostname)
	free(EA.hostname);
}

void add_addr_to_entry(entry,result,valid_until,hostname)
     struct service_entry            * entry;
     const SOCKADDR                  * result;
     const struct schedule_timelimit * valid_until;
     const char                      * hostname;
{
    struct service_entry_addr EA;

    if (SERVICE_ENTRY_magic != entry->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "add_addr_to_entry",
	      "Bad magic (service_entry)",0);

    /* bzero is defined hdrs/elm_defs.h */
    bzero ((void *)&EA, sizeof (EA));

    EA.addrsize      = 0;
    EA.address.dummy = 0;
 
    /* This malloces copy */
    set_address_from_gen_helper(& (EA.addrsize),
				& (EA.address),
				result);
    EA.valid_until   = *valid_until;
    EA.hostname      = hostname ? safe_strdup(hostname) : NULL;

    add_se_addr_to_entry(entry,&EA,0 /* do not malloc */);
}

void add_port_to_entry(entry,port)
     struct service_entry *entry;
     int port;    
{
    if (SERVICE_ENTRY_magic != entry->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "add_port_to_entry",
	      "Bad magic (service_entry)",0);

    entry->port_list = 
	safe_array_realloc(entry->port_list,
			   (entry->port_count+2),
			   sizeof (entry->port_list[0]));

    entry->port_list[entry->port_count++] = port;
    entry->port_list[entry->port_count]   = PORT_end;
}

int add_option_type_to_entry(entry,Y,prefix)
     struct service_entry *entry;
     struct SE_option_type * Y;
     const char * prefix;
{
    if (SERVICE_ENTRY_magic != entry->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "add_option_type_to_entry",
	      "Bad magic (service_entry)",0);

    entry->option_list =
	safe_array_realloc(entry->option_list,
			   (entry->option_count+1),
			   sizeof (entry->option_list[0]));
    bzero((void *) &(entry->option_list[entry->option_count]),
	   sizeof (entry->option_list[0]));

    entry->option_list[entry->option_count].type   = Y;
    entry->option_list[entry->option_count].prefix = safe_strdup(prefix);
    entry->option_list[entry->option_count].value = NULL;

    Y->zero_options(& (entry->option_list[entry->option_count]));
    return entry->option_count++;
}

void add_req_peername_to_entry(entry,req_peername)
     struct service_entry *entry;
     const struct string * req_peername;
{
    int i;

    if (SERVICE_ENTRY_magic != entry->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "add_req_peername_to_entry",
	      "Bad magic (service_entry)",0);

    for (i = 0; i < entry->req_tls_peername_count; i++) {
	if (0 == string_cmp(entry->req_tls_peername_list[i],
			    req_peername,-99 /* Unknown value*/))
	    break;

    }

    if (i >= entry->req_tls_peername_count) {
	entry->req_tls_peername_list
	    = safe_array_realloc(entry->req_tls_peername_list,
				 (entry->req_tls_peername_count +1),	 
				 sizeof (entry->req_tls_peername_list[0]));

	entry->req_tls_peername_list[entry->req_tls_peername_count++]
	    = dup_string(req_peername);
    }
}

static int is_same_se_address(A,B)
     struct service_entry_addr *A;
     struct service_entry_addr *B; 
{
    if (! A->address.dummy &&
	! B->address.dummy)
	return 1;
    if (! A->address.dummy ||
	! B->address.dummy)
	return 0;
    
    if (A->addrsize != B->addrsize)
	return 0;
    if (! A->addrsize)
	return 1;

    return 0 == memcmp(A->address.dummy,
		       B->address.dummy,
		       A->addrsize);
}
/* Returns number of expired entries */

static int free_expired_aliases P_((struct service_entry            * se,
				    const struct schedule_timelimit * now,
				    struct schedule_timelimit       * next_cleanup));
static int free_expired_aliases(se,now,next_cleanup) 
     struct service_entry            * se;
     const struct schedule_timelimit * now;
     struct schedule_timelimit       * next_cleanup;
{
    int ret = 0;

    if (SERVICE_ENTRY_magic != se->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_expired_aliases",
	      "Bad magic (service_entry)",0);
    
    if (se->aliases_list) {
	int t = 0;
	int i;

	/*  now == time_MAX if all cached addresses are cleared */

	for (i = 0; i < se->aliases_count; i++) {
	    if (schedule_valid_until_expired(& (se->aliases_list[i].valid_until),
					     now)) {
		if (se->aliases_list[i].aliasname) {
		    free(se->aliases_list[i].aliasname);
		    se->aliases_list[i].aliasname = NULL;
		}

		ret++;
	    } else {

		if (next_cleanup) {
		    if (schedule_shorten_next_timeout(next_cleanup,
						      (& (se->aliases_list[i].valid_until)))) {

			char * X = schedule_timeout_string(next_cleanup);
			
			if (X) {
			    DPRINT(Debug,14,(&Debug,
					     "free_expired_aliases: setting or shortened next_cleanup to %s\n",
					     X));			
			    free(X);
			}
			
		    }
		}	
		
		if (t < i) {
		    se->aliases_list[t] = se->aliases_list[i];
		    se->aliases_list[i].aliasname = NULL;
		}
		t++;
	    }
	}

	se->aliases_count = t;
    }

    return ret;
}
    
/* Returns number of expired entries */
static int free_expired_addrs P_((struct service_entry            * se,
				  const struct schedule_timelimit * now,
				  struct schedule_timelimit       * next_cleanup));
static int free_expired_addrs(se,now,next_cleanup) 
     struct service_entry            * se;
     const struct schedule_timelimit * now;
     struct schedule_timelimit       * next_cleanup;
{
    int ret = 0;

    if (SERVICE_ENTRY_magic != se->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_expired_addrs",
	      "Bad magic (service_entry)",0);
    
    if (se->addr_list) {
	int t = 0;
	int i;

	/*  now == time_MAX if all cached addresses are cleared */

	for (i = 0; i < se->addr_count; i++) {
	    if (schedule_valid_until_expired(& (se->addr_list[i].valid_until),
					     now)) {
		if (se->addr_list[i].address.dummy) {
		    free(se->addr_list[i].address.dummy);
		    se->addr_list[i].address.dummy = NULL;
		    se->addr_list[i].addrsize = 0;
		}

		ret++;
	    } else {

		if (next_cleanup) {
		    if (schedule_shorten_next_timeout(next_cleanup,
						      & (se->addr_list[i].valid_until))) {

			char * X = schedule_timeout_string(next_cleanup);
			
			if (X) {
			    DPRINT(Debug,14,(&Debug,
					     "free_expired_addrs: setting or shortened next_cleanup to %s\n",
					     X));			
			    free(X);
			}			
		    }
		}	
		
		if (t < i) {
		    se->addr_list[t] = se->addr_list[i];
		    se->addr_list[i].address.dummy = NULL;
		    se->addr_list[i].addrsize = 0;
		}
		t++;
	    }
	}

	se->addr_count = t;
    }

    return ret;
}

/* Returns number of cleared entries */
int free_expired_on_service_entry(se,now,next_cleanup)
     struct service_entry            * se;
     const struct schedule_timelimit * now;
     struct schedule_timelimit       * next_cleanup;
{
    int count = 0;
    
    if (SERVICE_ENTRY_magic != se->magic)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "free_expired_mailsrv_conf",
		  "Bad magic (service_entry)",0);

    count += free_expired_addrs(se,now,
				next_cleanup);
    
    count +=  free_expired_aliases(se,now,
				   next_cleanup);

    if (count > 0) {
	DPRINT(Debug,16,(&Debug,"free_expired_on_service_entry=%d: %s",
			 count,se->official_name));

	if (next_cleanup) {
	    char * X = schedule_timeout_string(next_cleanup);
	    
	    if (X) {
		DPRINT(Debug,14,(&Debug,
				 ", next cleanup %s",
				 X));			
		free(X);
	    }	
	}        
	
	DPRINT(Debug,16,(&Debug,"\n"));
    }
    
    return count;
    
}

/* Returns number of cleared entries */
static int free_expired_mailsrv_conf P_((struct mail_services_conf        * conf,
					 const struct schedule_timelimit * now,
					 struct schedule_timelimit       * next_cleanup));
static int free_expired_mailsrv_conf(conf,now,next_cleanup)
     struct mail_services_conf       * conf;
     const struct schedule_timelimit * now;
     struct schedule_timelimit       * next_cleanup;
{
    int count = 0;
    int i;

    if (MAIL_SERVICES_magic != conf->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_expired_mailsrv_conf",
	      "Bad magic (mail_services_conf) (conf)",0);

    for (i = 0; i <  conf->service_count; i++) {

	count += free_expired_on_service_entry(conf->service_list[i],
					       now,
					       next_cleanup);	
    }

    DPRINT(Debug,14,(&Debug,"free_expired_mailsrv_conf=%d, service_count=%d",
		     count, conf->service_count));
    
    if (next_cleanup) {
	char * X = schedule_timeout_string(next_cleanup);
	
	if (X) {
	    DPRINT(Debug,14,(&Debug,
			    ", next cleanup %s",
			    X));			
	    free(X);
	}	
    }        
    DPRINT(Debug,14,(&Debug,"\n"));
    
    return count;
}

static struct timed_task_handle  * cleanup_mailserv_handle = NULL;
struct schedule_timelimit          next_mailserv_cleanup;

S_(timed_task_action_f cleanup_mailserv_action)

static enum timed_task_r cleanup_mailserv_action P_((struct timed_task *task,
						     const struct schedule_timelimit *now));
static enum timed_task_r cleanup_mailserv_action(task,now)
     struct timed_task *task;
     const struct schedule_timelimit *now;
{
    int count = 0;
    char * A;
    int is_max;

    next_mailserv_cleanup = NO_schedule_timelimit;

    if (user_mail_services_conf) {
	int A = free_expired_mailsrv_conf(user_mail_services_conf,now,
					  &next_mailserv_cleanup);

	 DPRINT(Debug,9,(&Debug,
			 "cleanup_mailserv_action: %d cache items cleaned from user_mail_services_conf\n",
			 A));
	count += A;
    }

    if (system_mail_services_conf) {
	int B = free_expired_mailsrv_conf(system_mail_services_conf,now,
					  &next_mailserv_cleanup);
	
	DPRINT(Debug,9,(&Debug,
			"cleanup_mailserv_action: %d cache items cleaned from system_mail_services_conf\n",
			B));
	
	count += B;
    }

    if (static_hosts_opened) {
	int C = free_expired_static_hosts(now,&next_mailserv_cleanup);

	DPRINT(Debug,9,(&Debug,
			"cleanup_mailserv_action: %d cache items cleaned from static hosts\n",
			C));
	
	count += C;
    }
    
    DPRINT(Debug,9,(&Debug, "cleanup_mailserv_action: %d cache items cleaned",
		    count));

    A = schedule_timeout_string(&next_mailserv_cleanup);
    if (A) {
	DPRINT(Debug,9,(&Debug, ", next mail service cleanup %s",
			A));
	free(A);
    }

    is_max = schedule_valid_until_is_MAX(&next_mailserv_cleanup);
    if (is_max) {
	DPRINT(Debug,9,(&Debug, ", all entries are permanent"));
    }
    
    DPRINT(Debug,9,(&Debug, "\n"));

    if (schedule_have_timelimit(&next_mailserv_cleanup)
	&& !is_max) {

	if (alter_timed_task(&cleanup_mailserv_handle,NULL,
			     NO_timed_task_free,
			     NO_inc_timed_task_refcount,
			     cleanup_mailserv_action,
			     &next_mailserv_cleanup)) {
	    
	    DPRINT(Debug,9,(&Debug, "cleanup_mailserv_action: Updated mail service cleanup, handle %p\n",
			    cleanup_mailserv_handle));
	}
    } else {
	DPRINT(Debug,9,(&Debug, "cleanup_mailserv_action: freeing mail service cleanup handle %p\n",
			cleanup_mailserv_handle));

	alter_timed_task(&cleanup_mailserv_handle,NULL,
			 NO_timed_task_free,
			 NO_inc_timed_task_refcount,
			 NO_timed_task_action,
			 &next_mailserv_cleanup);
    }

    return timed_task_default;
}
    
static void update_next_mailserv_cleanup(valid_until)
     const struct schedule_timelimit * valid_until;
{

    if (schedule_shorten_next_timeout(&next_mailserv_cleanup,
				      valid_until)) {


	char * X = schedule_timeout_string(&next_mailserv_cleanup);
			
	if (X) {
	    DPRINT(Debug,9,(&Debug,
			    "update_next_mailserv_cleanup: setting or shortened next mail service cleanup to %s\n",
			    X));			
	    free(X);
	}

	if (schedule_valid_until_is_MAX(&next_mailserv_cleanup)) {
	    
	    DPRINT(Debug,9,(&Debug, "update_next_mailserv_cleanup: permanent entry\n"));
	    
	} else if (alter_timed_task(&cleanup_mailserv_handle,NULL,
				    NO_timed_task_free,
				    NO_inc_timed_task_refcount,
				    cleanup_mailserv_action,
				    &next_mailserv_cleanup)) {
	    
	    DPRINT(Debug,9,(&Debug, "update_next_mailserv_cleanup: Updated mail service cleanup, handle %p\n",
			    cleanup_mailserv_handle));
	}
    }

}

void free_service_list()
{
    if (cleanup_mailserv_handle) {
	DPRINT(Debug,9,(&Debug, "free_service_list: freeing mail service cleanup handle %p\n",
			cleanup_mailserv_handle));

	alter_timed_task(&cleanup_mailserv_handle,NULL,
			 NO_timed_task_free,
			 NO_inc_timed_task_refcount,
			 NO_timed_task_action,
			 &next_mailserv_cleanup);
    }
}

static void copy_service_data P_((struct service_entry *target,
				  struct service_entry *value));
static void copy_service_data(target,value)
     struct service_entry *target;
     struct service_entry *value;
{
    if (0 != istrcmp(target->official_name,value->official_name)) {
	int j;

	for (j = 0; j < target->aliases_count; j++) {
	    if (target->aliases_list[j].aliasname &&		
		/* Permanent  --
		   May fail at Tue Jan 19 03:14:07 2038 UTC
		*/
		schedule_valid_until_is_MAX(& (target->aliases_list[j].valid_until)) &&		
		0 == istrcmp(value->official_name,
			     target->aliases_list[j].aliasname))
		break;
	}
	if (j >= target->aliases_count) 
	    add_alias_to_entry(target,value->official_name,			       
			       /* Permanent  --
				  May fail at Tue Jan 19 03:14:07 2038 UTC
			       */
			       give_MAX_schedule_timelimit());
	
    }

    /* Add missing aliases */
    if ((value->flags & SE_given_alias) &&
	value->aliases_count > 0) {
	int i;

	target->flags |= SE_given_alias;

	/* Forget cached aliases */
	free_expired_aliases(target,			     
			     /* Permanent  --
				May fail at Tue Jan 19 03:14:07 2038 UTC
			     */
			     give_MAX_schedule_timelimit(),
			     NULL);

	for (i = 0; i < value->aliases_count; i++) {
	    int j;

	    for (j = 0; j < target->aliases_count; j++) {
		if (is_same_se_name(& (value->aliases_list[i]),
				    & (target->aliases_list[j])))
		    break;
	    }
	    if (j >= target->aliases_count) 
		add_se_alias_to_entry(target,& (value->aliases_list[i]),
				      1 /* malloc copy */);
	}
    }
	
    if ((value->flags & SE_given_name_addr) &&
	value->addr_name_count > 0) {
	int i;
	target->flags |= SE_given_name_addr;

	for (i = 0; i < value->addr_name_count; i++) {
	    int j;

	    for (j = 0; j < target->addr_name_count; j++) {
		if (0 == istrcmp(value->addr_name_list[i],
				 target->addr_name_list[j]))
		    break;		
	    }
	    if (j >= target->addr_name_count)
		add_addr_name_to_entry(target,
				       value->addr_name_list[i],
				       1 /* strdup it */);
	}
    }

    if ((value->flags & SE_given_name_port) &&
	value->port_name_count > 0) {
	int i;
	target->flags |= SE_given_name_port;

	for (i = 0; i < value->port_name_count; i++) {
	    int j;

	    for (j = 0; j < target->port_name_count; j++) {
		if (0 == istrcmp(value->port_name_list[i],
				 target->port_name_list[j]))
		    break;		
	    }
	    if (j >= target->port_name_count)
		add_port_name_to_entry(target,
				       value->port_name_list[i],
				       1 /* strdup it */);
	}
    }

    if ((value->flags & SE_given_addr) &&
	value->addr_count > 0) {
	int i;
	
	/* Forget cached addresses */
	free_expired_addrs(target,
			   /* Permanent  --
			      May fail at Tue Jan 19 03:14:07 2038
			   */
			   give_MAX_schedule_timelimit(),
			   NULL);

	target->flags |= SE_given_addr;

	for (i = 0; i < value->addr_count; i++) {

	    /* Ignore cached entries */
	    if (schedule_valid_until_ok(& (value->addr_list[i].valid_until),
					/* Permanent  --
					   May fail at Tue Jan 19 03:14:07 2038 UTC
					*/
					give_MAX_schedule_timelimit())) {
					
		/* add_se_addr_to_entry() checks for duplicates */
		
		add_se_addr_to_entry(target,& (value->addr_list[i]),
				     1 /* malloc copy */);
	    }		
	}       
    }
    
    if ((value->flags & SE_given_port) &&
	value->port_count > 0) {
	int i;
	
	if (! (target->flags & SE_given_port))
	    free_se_port_list(target);   /* Forget default port */

	target->flags |= SE_given_port;

	for (i = 0; i < value->port_count; i++) {
	    int j;

	    for (j = 0; j < target->port_count; j++) {	    
		if (value->port_list[i] ==
		    target->port_list[j])
		    break;
	    }

	    if (j >= target->port_count)
		add_port_to_entry(target,value->port_list[i]);	    
	}
    }

    if (value->option_count > 0) {
	int i;

	for (i = 0; i < value->option_count; i++) {
	    int j;
	    int idx = -1;
	    
	    for (j = 0; j < target->option_count; j++) {
		if (0 == strcmp(value->option_list[i].prefix,
				target->option_list[j].prefix))
		    idx = j;
	    }
	    if (-1 == idx) {
		struct SE_option_type * Y = 
#ifdef USE_DLOPEN
		    get_option_type(value->option_list[i].prefix)
#else
		    NULL
#endif
		    ;

		if (!Y) {
		    panic("CONNECTION PANIC",__FILE__,__LINE__,
			  "copy_service_data",
			  "option type disappeared",0);
		    continue;
		}
		
		idx = add_option_type_to_entry(target,Y,
					       value->option_list[i].prefix);
					       
	    }
	    if (SE_option_t_magic != target->option_list[idx].type->magic)
		panic("CONNECTION PANIC",__FILE__,__LINE__,
		      "copy_service_data",
		      "Bad option type magic on service list",0);

	    if (target->option_list[idx].type != value->option_list[i].type)
		panic("CONNECTION PANIC",__FILE__,__LINE__,
                      "copy_service_data",
                      "option type mismatch",0);

	    target->option_list[idx].type->
		merge_options(& (target->option_list[idx]),
			      & (value->option_list[i]) );
	}
    }

    if (ison(value->flags,SE_nouse_tls_checks)) {
	target->flags |= SE_nouse_tls_checks;

	/* This is allowed same time with
	   SE_ver_tls_cert and SE_require_tls_name
	*/
	
    }
	
    if (ison(value->flags,SE_ver_tls_cert)) {
	target->flags |= SE_ver_tls_cert;

	if (ison(target->flags,SE_nover_tls_cert)) {
	    target->flags &= ~SE_nover_tls_cert;   
	    DPRINT(Debug,10,(&Debug,  
			     "copy_service_data: conflicting verify-tls-certificate settings, verify-tls-certificate=off ignored\n"));
	}
    }

    if (ison(value->flags,SE_nover_tls_cert)) {
	if (isoff(target->flags,SE_ver_tls_cert))
	    target->flags |= SE_nover_tls_cert;
	else {
	    DPRINT(Debug,10,(&Debug,  
			     "copy_service_data: conflicting verify-tls-certificate settings, verify-tls-certificate=off ignored\n"));
	}
    }

    if ((value->flags & SE_require_tls_name)) {
	if (0 == strcmp(target->official_name,value->official_name))	    
	    target->flags |= SE_require_tls_name;
	else {
	    int i;

	    for (i = 0; i < target->req_tls_peername_count; i++) {
		if (string_matches_ascii(target->req_tls_peername_list[i],
					 s2us(value->official_name),
					 0,SMA_op_normal))
		    break;
	    }

	    if (i >= target->req_tls_peername_count) {
		struct string * req = new_string2(system_charset,
						  s2us(value->official_name));
		add_req_peername_to_entry(target,req);
		free_string(&req);
	    }
	}
	
    } else if (value->req_tls_peername_list) {
	int i;

	for (i = 0; i < value->req_tls_peername_count; i++) {
	    add_req_peername_to_entry(target,
				      value->req_tls_peername_list[i]);
	}
    }

}

/* Increments refcount */
struct service_entry *convert_service_entry(source,st)
     struct service_entry *source;
     const struct service_type * st;
{
    struct service_entry *res = NULL;

    if (SERVICE_ENTRY_magic != source->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "merge1_entries",
	      "Bad magic (service_entry) (target)",0);

    DPRINT(Debug,10,(&Debug,
		     "convert_service_entry: source=%p type=%p %s official_name=%s\n",
		     source,source->service,source->service->name,
		     source->official_name));
    DPRINT(Debug,10,(&Debug,
		     "convert_service_entry: target st=%p %s\n",
		     st,st->name));
    
    if (st == source->service) {
	DPRINT(Debug,10,(&Debug,
			 "convert_service_entry: Correct type - no conversion needed\n"));
	res = source;
	inc_service_entry_refcount(res);
    } else if (&(SERVICE_TYPES[0]) == source->service) {
	DPRINT(Debug,10,(&Debug,
			 "convert_service_entry: Source can be converted\n"));
	
	res = malloc_service_entry(source->official_name,
				   st, SE_temporary);

	if (source->ip_literal_addrsize && source->ip_literal_address.dummy) {
	    if (!  set_SOCKADDR_ptr_helper(& (res->ip_literal_addrsize),
					   & (res->ip_literal_address),
					   source->ip_literal_addrsize,
					   source->ip_literal_address)) {
		DPRINT(Debug,10,(&Debug,  "convert_service_entry: failed to copy literal ip address!\n"));
	    }
	}

	copy_service_data(res,source);

    } else {
	DPRINT(Debug,10,(&Debug,
			 "convert_service_entry: Source is not wildcard type -- no conversion\n"));
	res = source;
	inc_service_entry_refcount(res);	
    }
    
    return res;
}

void merge1_entries(target,value)
     struct service_entry *target;
     struct service_entry *value;
{
    if (SERVICE_ENTRY_magic != target->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "merge1_entries",
	      "Bad magic (service_entry) (target)",0);
    if (SERVICE_ENTRY_magic != value->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "merge1_entries",
	      "Bad magic (service_entry) (value)",0);

    if (target->ip_literal_addrsize != value->ip_literal_addrsize) {
	DPRINT(Debug,10,(&Debug,
			 "... literal ip address size differ. Merge ignored\n"));
	return;
    }

    if (target->ip_literal_address.dummy &&
	value->ip_literal_address.dummy &&
	target->ip_literal_addrsize) {
	
	if (0 != memcmp(target->ip_literal_address.dummy,
			value->ip_literal_address.dummy,
			target->ip_literal_addrsize)) {
	    
	    DPRINT(Debug,10,(&Debug,
			     "... literal ip address differ. Merge ignored\n"));
	    return;
	}

    } else if (target->ip_literal_address.dummy ||
	       value->ip_literal_address.dummy) {

	DPRINT(Debug,10,(&Debug,
			 "... literal ip address differ. Merge ignored\n"));
	return;
    }

    if (target->service == &(SERVICE_TYPES[0])) {
	target->service = value->service;
	if (target->service != &(SERVICE_TYPES[0])) {
	    DPRINT(Debug,10,(&Debug,  "... setting temporary entry for %s to be type %s\n",
			     target->official_name,
			     target->service->name));
	}
    } else if (target->service != value->service &&
	       value->service != &(SERVICE_TYPES[0])) {
	DPRINT(Debug,1,(&Debug,  
			 "... conflicting service entry types. Type %s used, %s ignored\n",
			target->service->name,
			value->service->name
			));
	return;
    }

    copy_service_data(target,value);   
}

static void merge_entries P_((struct service_entry **ret,
			      struct service_entry *value));
static void merge_entries(ret,value)
     struct service_entry **ret;
     struct service_entry *value;
{
    if (SERVICE_ENTRY_magic != value->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "merge_entries",
	      "Bad magic (service_entry) (value)",0);

    DPRINT(Debug,12,(&Debug,
		     "merge_entries: value=%p (%s), refcount=%d\n",
		     value,value->official_name,value->refcount));
  
    if (value == *ret)
	return;

    if (!*ret) {
	*ret = value;
	inc_service_entry_refcount(*ret);
    } else { 
	if (SERVICE_ENTRY_magic != (*ret)->magic)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "merge_entries",
		  "Bad magic (service_entry) (*ret)",0);

	DPRINT(Debug,12,(&Debug,
			 "merge_entries: (*ret)=%p (%s), refcount=%d\n",
			 (*ret),(*ret)->official_name,(*ret)->refcount));


	if ((*ret)->flags & SE_temporary) {
	    merge1_entries(*ret,value);

	} else {
	    struct service_entry * A = 
		malloc_service_entry((*ret)->official_name,
				     (*ret)->service, SE_temporary);

	    if ((*ret)->ip_literal_addrsize && (*ret)->ip_literal_address.dummy) {
		if (!  set_SOCKADDR_ptr_helper(& (A->ip_literal_addrsize),
					       & (A->ip_literal_address),
					       (*ret)->ip_literal_addrsize,
					       (*ret)->ip_literal_address)) {
		    DPRINT(Debug,10,(&Debug,  "... failed to copy literal ip address!\n"));
		}
	    }

	    merge1_entries(A,*ret);
	    merge1_entries(A,value);

	    free_service_entry(ret);

	    *ret = A;
	    DPRINT(Debug,10,(&Debug,  "... making temporary entry (%p) for %s from permanent\n",
			     (*ret),(*ret)->official_name));
	}
    }

    DPRINT(Debug,12,(&Debug,
		     "merge_entries: result (*ret)=%p (%s), refcount=%d%s%s\n",
		     (*ret),(*ret)->official_name,(*ret)->refcount,
		     ((*ret)->flags & SE_temporary) ? " (temporary)" : "",
		     ((*ret)->flags & SE_static)    ? " (static)"    : ""));
}


int host_name_ok(name)
     const char *name;
{
    size_t x;
    if ('\0' == *name)
	return 0;

    x = strspn(name,"abcdefghijklmnopqrstuvwxyz.-");
    
    return '\0' == name[x];
}

static int check_ascii P_((const char *c));
static int check_ascii(c)
     const char *c;
{
    const unsigned char *x;
    
    for (x = cs2us(c); *x; x++) {
	if (!isascii(*x))
	    return 0;
    }

    return 1;
}


/* Return 1 if succeed, 0 if too small, or does not fit */
int set_SOCKADDR_from_ptr_helper(target,addrsize,source)
     SOCKADDR               * target;
     const size_t             addrsize;
     const union SOCKADDR_ptr source;
{
    /* bzero is defined hdrs/elm_defs.h */
    bzero ((void *)target, sizeof (*target));

    /* sa_family is assumed to be first field !! */
    if (addrsize < sizeof (source.sa->sa_family)) {
	DPRINT(Debug,11,(&Debug,
			 "set_SOCKADDR_from_ptr_helper=0; size=%d\n",
			 addrsize));
	return 0;
    }

    switch (source.sa->sa_family) {
#if defined(I_NETINET_IN) && defined(I_ARPA_INET)
    case AF_INET:
	if (addrsize < sizeof (source.sin)) {
	    DPRINT(Debug,11,(&Debug,
			     "set_SOCKADDR_from_ptr_helper=0; size=%d\n",
			     addrsize));
	    return 0;
	}
	target->sin = *(source.sin);
	break;
#ifdef HAVE_IN6
    case AF_INET6: 
	if (addrsize < sizeof (source.sin6)) {
	   DPRINT(Debug,11,(&Debug,
			     "set_SOCKADDR_from_ptr_helper=0; size=%d\n",
			     addrsize));
	    return 0;
	}
	target->sin6 = *(source.sin6);
	break;
#endif
#endif
    default:
	DPRINT(Debug,11,(&Debug,
			 "set_SOCKADDR_from_ptr_helper: Unknown address family %d, size %d\n",
			 source.sa->sa_family,addrsize));
	
	if (addrsize > sizeof (*target)) {
	    DPRINT(Debug,11,(&Debug,
			     "set_SOCKADDR_from_ptr_helper=0; size=%d (too big)\n",
			     addrsize));
	    return 0;
	}
	memcpy((void *)target,source.dummy,addrsize);
	break;
    }

    return 1;
}


int dump_service_entry(f,entry,fileset)
     FILE *f; 
     struct service_entry *entry;
     charset_t fileset;
{
    char * sep = "\t";
    int idx;
    int iserr = 0;

    if (SERVICE_ENTRY_magic != entry->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "dump_service_entry",
	      "Bad magic (service_entry)",0);

    if (!check_ascii(entry->official_name) ||
	!check_ascii(entry->service->name))
	return 0;

    fputs(entry->official_name,f);
    putc('\t',f);
    fputs(entry->service->name,f);
	
    if (entry->flags & SE_given_alias) {
	int y;
	for (y = 0; y < entry->aliases_count; y++) {

	    if (entry->aliases_list[y].aliasname) {
		
		int ERR = 0;
		
		char *val = convert_to_fileset(entry->aliases_list[y].aliasname,
					       fileset,&ERR);
		iserr += ERR;
		
		if (!val)
		    continue;
	    
		if (host_name_ok(val))
		    elm_fprintf(f,FRM("%salias=%s"),
				sep,val);
		else
		    elm_fprintf(f,FRM("%salias=%Q"),
				sep,val);
		sep = "; ";
		
		if (val != entry->aliases_list[y].aliasname)
		    free(val);
	    }
	}
    }

    if (entry->flags & SE_given_name_addr) {
	int y;
	for (y = 0; y < entry->addr_name_count; y++) {
	    if (!check_ascii(entry->addr_name_list[y])) {
		iserr += 1;
		continue;
	    }

	    if (host_name_ok(entry->addr_name_list[y]))
		elm_fprintf(f,FRM("%saddr=%s"),
			    sep,entry->addr_name_list[y]);
	    else
		elm_fprintf(f,FRM("%saddr=%Q"),
			    sep,entry->addr_name_list[y]);
	    sep = "; ";
	}
    }

    if (entry->flags & SE_given_name_port) {
	int y;
	for (y = 0; y < entry->port_name_count; y++) {
	    if (!check_ascii(entry->port_name_list[y])) {
		iserr += 1;
		continue;
	    }

	    if (host_name_ok(entry->port_name_list[y]) &&
		! isdigit(entry->port_name_list[y][0]))
		elm_fprintf(f,FRM("%sport=%s"),
			    sep,entry->port_name_list[y]);
	    else
		elm_fprintf(f,FRM("%sport=%Q"),
			    sep,entry->port_name_list[y]);
	    sep = "; ";
	}
    }
	
    if (entry->flags & SE_given_addr) {
	int y;
	for (y = 0; y < entry->addr_count; y++) {

	    /* Ignore cached entries */
	    if (schedule_valid_until_expired(& (entry->addr_list[y].valid_until),
					     /* Permanent  --
						May fail at Tue Jan 19 03:14:07 2038 UTC
					     */
					     give_MAX_schedule_timelimit())) {
		DPRINT(Debug,8,(&Debug,"dump_service_entry: %d: cached address skipped\n",
				y));
		continue;
	    }

	    if (entry->addr_list[y].address.dummy) {
		
		SOCKADDR X;
		char mybuffer[256];
		const char * z;

		if (! set_SOCKADDR_from_ptr_helper(& X,
						   entry->addr_list[y].addrsize,
						   entry->addr_list[y].address)) {
		    
		    DPRINT(Debug,8,(&Debug,"dump_service_entry: %d: Bad address skipped (size %d)\n",
				    y,entry->addr_list[y].addrsize));
		    continue;
		}

		z = 
		    give_addr_as_string(&X,mybuffer,sizeof mybuffer);

		if (z) {
		    elm_fprintf(f,FRM("%saddr=ip:%Q"),
				sep,z);

		    switch (X.sa.sa_family) {
			int port UNUSED_VAROK;
#if defined(I_NETINET_IN) && defined(I_ARPA_INET)
		    case AF_INET:		    
#ifdef AF_INET6
		    case AF_INET6:
#endif
			port = give_addr_port(&X);
			if (port != PORT_end)
			    elm_fprintf(f,FRM(":%d"),
					port);
		    
			break;
#endif
		    default:
			/* none */
			break;
		    }		
		    sep = "; ";
		}
	    }
	}
    }

    if (ison(entry->flags,SE_given_port)) {
	int y;
	for (y = 0; y < entry->port_count; y++) {
	    elm_fprintf(f,FRM("%sport=%d"),
			sep,entry->port_list[y]);
	    sep = "; ";
	}
    }

    if (ison(entry->flags,SE_ver_tls_cert)) {
	elm_fprintf(f,FRM("%sverify-tls-certificate"),
		    sep);
	sep = "; ";
    }

    if (ison(entry->flags,SE_nover_tls_cert)) {
	elm_fprintf(f,FRM("%sverify-tls-certificate=OFF"),
		    sep);
	sep = "; ";
    }

    if (ison(entry->flags,SE_nouse_tls_checks)) {
	elm_fprintf(f,FRM("%suse-tls-checks=no"),
		    sep);
	sep = "; ";
    }
    
    if (ison(entry->flags,SE_require_tls_name)) {
	elm_fprintf(f,FRM("%srequire-tls-peer-name"),
		    sep);
	sep = "; ";
    }

    for (idx = 0; idx < entry->req_tls_peername_count; idx++) {
	int ERR = 0;

	char *val = convert_to_fileset2(entry->req_tls_peername_list[idx],
					   fileset,&ERR);
	iserr += ERR;

	if (!val)
	    continue;
	
	if (host_name_ok(val))
	    elm_fprintf(f,FRM("%srequire-tls-peer-name=%s"),
			sep,val);
	else
	    elm_fprintf(f,FRM("%srequire-tls-peer-name=%Q"),
			sep,val);
	sep = "; ";
	
	free(val);
    }

    for (idx = 0; idx < entry->option_count; idx++) {
	char * val;
	
	if (SE_option_t_magic != entry->option_list[idx].type->magic)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "dump_service_entry",
		  "Bad option type magic on service list",0);
	
	val = entry->option_list[idx].type->
	    give_values(& (entry->option_list[idx]),
			entry->option_list[idx].prefix);
	
	if (val) {
	    int ERR = 0;
	    char *val1 = convert_to_fileset(val,fileset,&ERR);
	    iserr += ERR; 
	    
	    if (val1) {
		fputs(sep,f);
		fputs(val1,f);
		sep = "; ";

		if (val1 != val)
		    free(val1);
	    }
	    free(val); val = NULL;
	}
    }
    putc('\n',f);

    if (iserr) {
	const char * sysnam = get_charset_MIME_name(fileset);

	elm_fprintf(f, 
		    CATGETS(elm_msg_cat,MeSet, MeFailedConvertSome,
			    "# Failed to convert some values of %s to %s -charset\n"),
		    entry->official_name,
		    sysnam ? sysnam : "<no MIME name>");
		    
    }

    return !iserr;
}				
 
static int initial_tls_match P_((const struct service_type *st,
				 enum initial_tls_flag initial_tls));
static int initial_tls_match(st,initial_tls)
     const struct service_type *st;
     enum initial_tls_flag initial_tls;
{
    switch (initial_tls) {
    case itls_ignore:
	return 1;
    case itls_disable:
	return NULL == st->have_init_tls;
    case itls_require:
	return NULL != st->have_init_tls;
    }

    return 0; /* Bad value */
}

struct service_entry *scan_mail_services_conf(conf,hostname,flag,
					      initial_tls,now,
					      have_ip_literal,
					      st)
     struct mail_services_conf *conf;
     const char *hostname;
     int flag;
     enum initial_tls_flag initial_tls;
     const struct schedule_timelimit * now;
     SOCKADDR * have_ip_literal;
     const struct service_type *st;
{
    struct service_entry *ret = NULL;
    int i;

    if (MAIL_SERVICES_magic != conf->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "scan_mail_services_conf",
	      "Bad magic (mail_services_conf) (conf)",0);
    
    for (i = conf->service_count-1; i >= 0; i--) {
	int tls_match = 0;

	if (SERVICE_ENTRY_magic != conf->service_list[i]->magic)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "scan_mail_services_conf",
		  "Bad magic (service_entry)",0);

	if (st && st != &(SERVICE_TYPES[0]) &&
	    conf->service_list[i]->service != &(SERVICE_TYPES[0]) &&
	    st != conf->service_list[i]->service) {

	    DPRINT(Debug,8,(&Debug,
			    "\nscan_mail_services_conf(%s,%d): [%d]  official_name=%s %s -- skipping, service type %s requested\n",
			    hostname,flag,i,
			    conf->service_list[i]->official_name,
			    conf->service_list[i]->service->name,
			    st->name));
	    continue;

	}


	tls_match = initial_tls_match(conf->service_list[i]->service,
				      initial_tls);
	if (0 != (flag & conf->service_list[i]->service->flags) &&
	    tls_match) {
	    int j;

	    if (have_ip_literal) {

		if (! conf->service_list[i]->ip_literal_addrsize ||
		    ! conf->service_list[i]->ip_literal_address.dummy) {

		    DPRINT(Debug,8,(&Debug,
				    "\nscan_mail_services_conf(%s,%d): [%d] official_name=%s  %s flag=%d -- skipping, literal ip address requested\n",
				hostname,flag,i,
				conf->service_list[i]->official_name,
				conf->service_list[i]->service->name,
				conf->service_list[i]->service->flags));
		    continue;
		}
		
		if (! same_SOCKADDR_ptr_generic(conf->service_list[i]->
						ip_literal_address,
						conf->service_list[i]->
						ip_literal_addrsize,
						have_ip_literal)) {

		    DPRINT(Debug,8,(&Debug,
				    "\nscan_mail_services_conf(%s,%d): [%d] official_name=%s  %s flag=%d -- literal ip address mismatch\n",
				    hostname,flag,i,
				    conf->service_list[i]->official_name,
				    conf->service_list[i]->service->name,
				    conf->service_list[i]->service->flags));
		    continue;
		}

	    } else if (conf->service_list[i]->ip_literal_addrsize) {
		/* skip ip literal address entries */
		
		DPRINT(Debug,8,(&Debug,
				"\nscan_mail_services_conf(%s,%d): [%d] official_name=%s  %s flag=%d skipping literal ip address, addrsize=%d\n",
				hostname,flag,i,
				conf->service_list[i]->official_name,
				conf->service_list[i]->service->name,
				conf->service_list[i]->service->flags,
				conf->service_list[i]->ip_literal_addrsize));

		continue;
	    }

	    if (0 == istrcmp(hostname,
			     conf->service_list[i]->official_name)) {

#ifdef DEBUG
		if (Debug.active >= 8) {
		    FILE *F;
		    DPRINT(Debug,8,(&Debug,"\n-- found [%d] ",i));
		    F = debug_to_FILE(&Debug);
		    if (F) {
			dump_service_entry(F,conf->service_list[i],
					   system_charset);
			fclose(F);
		    }
		}
#endif

		/* ret = service_list[i]	     */
		merge_entries(&ret,conf->service_list[i]);
	    }
	    
	    for (j = 0; j < conf->service_list[i]->aliases_count; j++) {
		if (conf->service_list[i]->aliases_list[j].aliasname &&
		    schedule_valid_until_ok(& (conf->service_list[i]->
					       aliases_list[j].valid_until),
					    now) &&
		    0 == istrcmp(hostname,
				 conf->service_list[i]->aliases_list[j].
				 aliasname)) {
#ifdef DEBUG
		    if (Debug.active >= 8) {
			FILE *F;
			DPRINT(Debug,8,(&Debug,"\n-- found [%d] ",i));
			F = debug_to_FILE(&Debug);
			if (F) {
			    dump_service_entry(F,conf->service_list[i], 
					       system_charset);
			    fclose(F);
			}
		    }
#endif

		    /* ret = service_list[i] */
		    merge_entries(&ret,conf->service_list[i]);
		}
	    }
	    
	} else {
	    if (0 == istrcmp(hostname,
			     conf->service_list[i]->official_name)) {
		DPRINT(Debug,8,(&Debug,
				"\nscan_mail_services_conf(%s,%d): [%d] official_name=%s  %s flag=%d MISMATCH%s\n",
				hostname,flag,i,
				conf->service_list[i]->official_name,
				conf->service_list[i]->service->name,
				conf->service_list[i]->service->flags,
				tls_match ? ", tls ok" : " on tls"));
	    }
	}	
    }

    if (ret && st && st != &(SERVICE_TYPES[0]) && st != ret->service) {

	struct service_entry * A;

	if (ret->service != &(SERVICE_TYPES[0])) 
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "scan_mail_services_conf",
		  "Returnted type mismatch",0);

	DPRINT(Debug,10,(&Debug,  "... changing type fron %s to %s\n",
			 ret->service->name,st->name));

	A = malloc_service_entry(ret->official_name,st,SE_temporary);

	if (ret->ip_literal_addrsize && ret->ip_literal_address.dummy) {
	    if (!  set_SOCKADDR_ptr_helper(& (A->ip_literal_addrsize),
					   & (A->ip_literal_address),
					   ret->ip_literal_addrsize,
					   ret->ip_literal_address)) {
		DPRINT(Debug,10,(&Debug,  "... failed to copy literal ip address!\n"));
	    }
	}

	merge1_entries(A,ret);

	DPRINT(Debug,10,(&Debug,  "... making temporary entry (%p) for %s from %p\n",
			 A,A->official_name,ret));

	free_service_entry(&ret);
	
	ret = A;
    }


    return ret;
}
  

/* Increments refcount */
static struct service_entry *scan_list P_((const char *hostname,
					   int flag,
					   enum initial_tls_flag initial_tls,
					    const struct schedule_timelimit * now,
					   SOCKADDR * have_ip_literal,
					   const struct service_type  * 
					   servicetype));
static struct service_entry *scan_list(hostname,flag,initial_tls,now,
				       have_ip_literal,
				       servicetype)
     const char *hostname;
     int flag;
     enum initial_tls_flag initial_tls;
     const struct schedule_timelimit * now;
     SOCKADDR * have_ip_literal;
     const struct service_type  * servicetype;
{
    struct service_entry *ret = NULL;

    if (user_mail_services_conf) {
	ret = scan_mail_services_conf(user_mail_services_conf,hostname,flag,
				      initial_tls,now,
				      have_ip_literal,
				      servicetype);
	
	if (ret) {
	    DPRINT(Debug,12,(&Debug,
			     "scan_list(%s,%d): %p -- Found from user_mail_services_conf\n",
			     hostname,flag,ret));
	}


	if (system_mail_services_conf) {
	    struct service_entry *ret1 = 
		scan_mail_services_conf(system_mail_services_conf,hostname,
					flag,initial_tls,now,
					have_ip_literal,
					servicetype);

	    if (ret1) {

		DPRINT(Debug,12,(&Debug,
				 "scan_list(%s,%d): %p -- Found from system_mail_services_conf\n",
				 hostname,flag,ret1));


		merge_entries(&ret,ret1);
		free_service_entry(&ret1);
	    }
	}

    } else if (system_mail_services_conf) {
	ret = scan_mail_services_conf(system_mail_services_conf,hostname,
				      flag,initial_tls,now,
				      have_ip_literal,
				      servicetype);

	if (ret) {
	    DPRINT(Debug,12,(&Debug,
			     "scan_list(%s,%d): %p -- Found from system_mail_services_conf\n",
			     hostname,flag,ret));
	}
	
    } else {
	DPRINT(Debug,8,(&Debug,
			"scan_list: No user_mail_services_conf or system_mail_services_conf\n"));
    }

#ifdef DEBUG
    if (Debug.active >= 8 && ret) {
	FILE *F;
	if (ret->flags & SE_temporary) {
	    DPRINT(Debug,8,(&Debug,"\nMERGED        "));
	} else {
	    DPRINT(Debug,8,(&Debug,"\nRESULT        "));
	}
	F = debug_to_FILE(&Debug);
	if (F) {
	    dump_service_entry(F,ret, system_charset);
	    fclose(F);
	}
	DPRINT(Debug,8,(&Debug,"\n"));
    }
#endif

    if (ret) {
    	DPRINT(Debug,12,(&Debug,"scan_list(%s,%d,...)=%p, %s: refcount=%d\n",
			 hostname,flag,ret,ret->official_name,ret->refcount));
    } else {
	DPRINT(Debug,12,(&Debug,"scan_list(%s,%d,...)=NULL\n",
			 hostname,flag));
    }

    return ret;
}

static void looking_up_msg_helper P_((struct cancel_data    ** cancel_p
				    /* May be NULL, used if dns lookup was cancelable */,
				      enum lookup_show_message show_message,
				      const char             * s));


#ifdef HAVE_ADDRINFO
static struct addrinfo * lookup_namei P_((const char          * hostname,
					  int                 * was_error_p,			 
					  struct cancel_data ** cancel_p,					  
					  const char ** reserved_name_p));
static struct addrinfo * lookup_namei(hostname,was_error_p,cancel_p,reserved_name_p)
     const char          * hostname;
     int                 * was_error_p;			 
     struct cancel_data ** cancel_p 
     /* May be NULL, Used if dns lookup was cancelable */;
     const char         ** reserved_name_p;
{
    struct addrinfo * ai = NULL;
    struct addrinfo hints;
    int status;
    const char * reserved_name = is_special_use_domain(hostname);
    const int    * ipv_option_array = NULL;
    size_t     ipv_option_array_len = get_ipv_option(&ipv_option_array,NULL);

    bzero((void *)&hints, sizeof hints);
    
    hints.ai_flags      = AI_CANONNAME
#ifdef AI_NUMERICSERV 
	| AI_NUMERICSERV
#endif
	;

    if (ipv_option_array && ipv_option_array_len > 1) {
	/* Assuming that ipv_option_array includes
	   AF_INET and AF_INET6 in some order
	*/
	hints.ai_family  = AF_UNSPEC ;
    } else if (ipv_option_array && 1 == ipv_option_array_len) {
	hints.ai_family  = ipv_option_array[0];
    } else {

#if defined(AI_ADDRCONFIG) && defined(USABLE_ADDRCONFIG)
	if (0 == istrcmp(hostname,"localhost")) {
	    DPRINT(Debug,9,(&Debug, 
			    "lookup_namei: %s: do not use AI_ADDRCONFIG\n",
			    hostname));
	} else {
	    hints.ai_flags |= AI_ADDRCONFIG;   /* Only if system have address of given type */
	}
#endif
	
	hints.ai_family     = 
#ifdef  HAVE_IN6
	    AF_UNSPEC    /* Allow IPv4 or IPv6 */
#else
	    AF_INET
#endif
	    ;
    }
    
    hints.ai_socktype   = SOCK_STREAM;    
    hints.ai_protocol   = IPPROTO_TCP;
    hints.ai_canonname  = NULL;
    hints.ai_addr       = NULL;
    hints.ai_next       = NULL;

    if (reserved_name) {

	if (reserved_name_p)
	    *reserved_name_p = reserved_name;
	else if (was_error_p) {
	    lib_error(CATGETS(elm_msg_cat, MeSet,
			      MeReservedDomainUnsupported,
			      "Reserved domain %s is unsupported: %s"),
		      reserved_name,hostname);
	    *was_error_p = 1;
	}
	    
	DPRINT(Debug,9,(&Debug, 
			"lookup_namei: %s: special use domain %s\n",
			hostname,reserved_name));

	return NULL;
    }

    looking_up_msg_helper(cancel_p,show_lookup_message,hostname);
        
    DPRINT(Debug,9,(&Debug, 
		    "lookup_namei: %s: using getaddrinfo()\n",
		    hostname));

    errno = 0;
    
    /* PORT_end = 0 == port not given , so use "0" as service */
   
    status =  getaddrinfo(hostname,"0",&hints,&ai);

    if (0 != status) {
	int err = errno;

	DPRINT(Debug,9,(&Debug, 
			"lookup_namei: %s: status %d (%s)",
			hostname,status,gai_strerror(status)));
	if (err) {
	    DPRINT(Debug,9,(&Debug, " errno=%d (%s)",
			    err,strerror(err)));
	}
	DPRINT(Debug,9,(&Debug, "\n"));

	/* We are silent about errors if cached
	   data exists
	*/

	if (EAI_SYSTEM == status &&
	    EINTR == err &&  cancel_p && *cancel_p &&
	    is_canceled(*cancel_p)) {
	    DPRINT(Debug,9,(&Debug, 
			    "lookup_namei: %s: query canceled\n",
			    hostname));
	    goto fail;

	} else if (was_error_p) {
	    switch (status) {
	    case EAI_NONAME:
		lib_error(CATGETS(elm_msg_cat, MeSet,MeNoRemoteHost,
				  "No remote server host: %s"),
			  hostname);
		*was_error_p = 1;
		break;
		
#ifdef EAI_NODATA
	    case EAI_NODATA:
		lib_error(CATGETS(elm_msg_cat, MeSet,MeNoAddress,
				  "Name %s have not IP-address"),
			  hostname);
		*was_error_p = 1;
		break;
#endif
		
	    case EAI_AGAIN:
		lib_error(CATGETS(elm_msg_cat, MeSet,MeTemporary,
				  "Address for %s is not yet found..."),
			  hostname);
		*was_error_p = 1;
		break;

	    case EAI_SYSTEM:

		if (err) {		    
		    lib_error(CATGETS(elm_msg_cat, MeSet,MeNameError1,
				      "Failed to get address for %s: %s"),
			      hostname,strerror(err));
		} else {
		    lib_error(CATGETS(elm_msg_cat, MeSet,MeNameError,
				      "Failed to get address for %s"),
			      hostname);
		}
		*was_error_p = 1;
		break;
	    default:
		lib_error(CATGETS(elm_msg_cat, MeSet,MeNameError1,
				  "Failed to get address for %s: %s"),
			  hostname,gai_strerror(status));
		*was_error_p = 1;
		break;		
	    }
	    
	} 
    }

 fail:
    
    return ai;
}
#endif

static struct hostent * lookup_name P_((const char           * hostname,
					int                  * was_error_p,			 
					struct cancel_data  ** cancel_p,
					const char ** reserved_name_p));
static struct hostent * lookup_name(hostname,was_error_p,cancel_p,reserved_name_p)
     const char          * hostname;
     int                 * was_error_p;			 
     struct cancel_data ** cancel_p 
     /* May be NULL, Used if dns lookup was cancelable */;
     const char         ** reserved_name_p;
{
    struct hostent *he = NULL;

    const char * reserved_name = is_special_use_domain(hostname);
    
    if (reserved_name) {

	if (reserved_name_p)
	    *reserved_name_p = reserved_name;
	else if (was_error_p) {
	    lib_error(CATGETS(elm_msg_cat, MeSet,
			      MeReservedDomainUnsupported,
			      "Reserved domain %s is unsupported: %s"),
		      reserved_name,hostname);
	    *was_error_p = 1;
	}
	    
	DPRINT(Debug,9,(&Debug, 
			"lookup_name: %s: special use domain %s\n",
			hostname,reserved_name));

	return NULL;
    }


    looking_up_msg_helper(cancel_p,show_lookup_message,hostname);

    DPRINT(Debug,9,(&Debug, 
		    "lookup_name: %s: using gethostbyname()\n",
		    hostname));

    errno = 0;
    
    he = gethostbyname(hostname);
	
    if (!he) {
	int err = errno; 

	DPRINT(Debug,9,(&Debug, 
			"lookup_name: %s: h_errno=%d",
			hostname,h_errno));
	if (err) {
	    DPRINT(Debug,9,(&Debug, " errno=%d (%s)",
			    err,strerror(err)));
	}
	DPRINT(Debug,9,(&Debug, "\n"));
	
	/* We are silent about errors if cached
	   data exists
	*/

	if (HOST_NOT_FOUND != h_errno &&
	    NO_DATA        != h_errno &&
	    NO_RECOVERY    != h_errno &&
	    EINTR == err &&  cancel_p && *cancel_p &&
	    is_canceled(*cancel_p)) {
	    DPRINT(Debug,9,(&Debug, 
			    "lookup_name: %s: query canceled\n",
			    hostname));
	    goto fail;
	    
	} else if (was_error_p) {

	    switch(h_errno) {
	    case HOST_NOT_FOUND:
		lib_error(CATGETS(elm_msg_cat, MeSet,MeNoRemoteHost,
				  "No remote server host: %s"),
			  hostname);
		*was_error_p = 1;
		break;
	    case NO_DATA:
		lib_error(CATGETS(elm_msg_cat, MeSet,MeNoAddress,
				  "Name %s have not IP-address"),
			  hostname);
		*was_error_p = 1;
		break;
	    case TRY_AGAIN:
		
		if (err)
		    lib_error(CATGETS(elm_msg_cat, 
				      MeSet,
				      MeLookErrno,
				      "Looking up %s: %s"),
			      hostname,strerror(err));
		else	       
		    lib_error(CATGETS(elm_msg_cat, MeSet,MeTemporary,
				      "Address for %s is not yet found..."),
			      hostname);
		*was_error_p = 1;
		break;

	    case NO_RECOVERY:
		lib_error(CATGETS(elm_msg_cat, MeSet,MeNoRecoveryBadName,
				  "Looking up %s: No recovery; Bad name or refused query?"),
			  hostname);
		*was_error_p = 1;
		break;
		
	    default:
		if (err)
		    lib_error(CATGETS(elm_msg_cat, 
				      MeSet,
				      MeLookErrno,
				      "Looking up %s: %s"),
			      hostname,strerror(err));
		else 
		    lib_error(CATGETS(elm_msg_cat, MeSet,MeNameError,
				      "Failed to get address for %s"),
			      hostname);
		*was_error_p = 1;
		break;
	    }
	} 
		   
    } else {
	const int             * ipv_option_array = NULL;
	const enum ipv_option * ipv_option_copy  = NULL;
	size_t              ipv_option_array_len = get_ipv_option(&ipv_option_array,
								  &ipv_option_copy);

	
	if (ipv_option_array && ipv_option_array_len > 0) {
	    int have_type = 0;

	    char option[NUM_protocols+1];
	    size_t i;
	    size_t a = 0;
	    
	    for (i = 0; i < NUM_protocols && i < ipv_option_array_len; i++) {
		if (he->h_addrtype == ipv_option_array[i])
		    have_type = 1;

		if (ipv_option_copy &&
		    isascii(ipv_option_copy[i]) && isprint(ipv_option_copy[i])) {
		    option[a++] = ipv_option_copy[i];
		}
	    }			
	    option[a] = '\0';

	    if (have_type) {
		lib_transient(CATGETS(elm_msg_cat, MeSet,MeLookingUpOK,
				      "Looking up %s ... OK"),
			      hostname);
	    } else if (a > 0) {
		lib_error(CATGETS(elm_msg_cat, MeSet,MeLookingUpUnsupOpt,
				  "Looking up %s ... Option -%s not supported"),
			  hostname,option);
	    }
	    
		    
	    
	} else {       	
	    lib_transient(CATGETS(elm_msg_cat, MeSet,MeLookingUpOK,
				  "Looking up %s ... OK"),
			  hostname);
	}
	
    }

 fail:
    return he;
}

static void fill_port P_((struct service_entry *ret,
			  struct servent  *se));
static void fill_port(ret,se)
     struct service_entry *ret;
     struct servent  *se;
{   
    if (SERVICE_ENTRY_magic != ret->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "fill_port",
	      "Bad magic (service_entry)",0);

    add_port_to_entry(ret,ntohs(se->s_port));
}

#ifdef HAVE_ADDRINFO
static void fill_addresses_from_info P_((struct service_entry *ret,
					 struct addrinfo * ai,
					 const struct schedule_timelimit * valid_until));
static void fill_addresses_from_info(ret,ai,valid_until)
     struct service_entry * ret;
     struct addrinfo      * ai;
     const struct schedule_timelimit * valid_until;
{
    int count UNUSED_VAROK = 0;
    struct addrinfo * walk;
    char * hostname = NULL;

    if (SERVICE_ENTRY_magic != ret->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "fill_addresses_from_info",
	      "Bad magic (service_entry)",0);

    for (walk = ai; walk; walk = walk->ai_next) {

	struct service_entry_addr addr;

	bzero((void *)&addr, sizeof addr);
	
	addr.address.sa    = walk->ai_addr;
	addr.addrsize      = walk->ai_addrlen;

	/* First entry includes canonical name ? */
	if (!hostname && walk->ai_canonname)
	    hostname = walk->ai_canonname;

	addr.valid_until = *valid_until;
	addr.hostname    = walk->ai_canonname ? 
	    walk->ai_canonname : hostname;

	add_se_addr_to_entry(ret,&addr,1 /* malloc copy */);
	count++;
    }

    DPRINT(Debug,10,(&Debug, 
		     "fill_addresses_from_info: %s: %d addresses added\n",
		     ret->official_name,count));

}
#endif

static void fill_address P_((struct service_entry *ret,
			     struct hostent *he,
			      const struct schedule_timelimit * valid_until));
static void fill_address(ret,he,valid_until)
     struct service_entry *ret;
     struct hostent *he;
     const struct schedule_timelimit * valid_until;
{   
    int count UNUSED_VAROK = 0;

    if (SERVICE_ENTRY_magic != ret->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "fill_address",
	      "Bad magic (service_entry)",0);

    if (he->h_addr_list) {
	int x;
	for (x = 0; he->h_addr_list[x]; x++) {

	    size_t             addrsize = 0;
	    union SOCKADDR_ptr addr;
	    addr.dummy = NULL;

	    if (set_SOCKADDR_from_data(&addrsize,
				       &addr,
				       he->h_addrtype,
				       he->h_length,
				       he->h_addr_list[x])) {

		add_addr2_to_entry(ret,addrsize,addr,
				   valid_until, he->h_name);
		
		count++;
		
	    } else {
		lib_error(CATGETS(elm_msg_cat, MeSet,MeUnsupAddrType2,
				  "Name %s have odd type address (family %d, length %d)"),
			  he->h_name,he->h_addrtype,he->h_length);
		
		/* NOTE: We do not know on what components arbitary
		   socket address is constructed so we can't
		   fill it
		*/

		if (addr.dummy)
		    free(addr.dummy);

		return;
	    }

	    if (addr.dummy)
		free(addr.dummy);
	}
    }

    DPRINT(Debug,10,(&Debug, 
		     "fill_addresses: %s (%s): %d addresses added (address family %d, length %d)\n",
		     ret->official_name,
		     he->h_name,
		     count,
		     he->h_addrtype,he->h_length));
}

static int  addrs_have_port P_((struct service_entry *se));
static int  addrs_have_port(se)
     struct service_entry *se;
{
    int have_port = 0;

    if (SERVICE_ENTRY_magic != se->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "addrs_have_port",
	      "Bad magic (service_entry)",0);


    if (se->addr_count > 0) {
	int x;

	have_port = 1;

	for (x = 0; x < se->addr_count; x++) {
	    if (se->addr_list[x].address.dummy) {
		SOCKADDR A;

		if (! set_SOCKADDR_from_ptr_helper(& A,
						   se->addr_list[x].addrsize,
						   se->addr_list[x].address)) {
		    DPRINT(Debug,14,(&Debug,"addrs_have_port:  %d: Bad address skipped (size %d)\n",
				     x,se->addr_list[x].addrsize));
		    have_port = 0;
		    continue;
		}
		    
		switch(A.sa.sa_family) {
		    int port UNUSED_VAROK;
#if defined(I_NETINET_IN) && defined(I_ARPA_INET)
		case AF_INET:
#ifdef AF_INET6
		case AF_INET6:
#endif
		    port = give_addr_port(& A);
		    if (PORT_end == port)
			have_port = 0;
		    break;
#endif
		default:
		    have_port = 0;
		    break;
		}
	    }
	}
    }

    DPRINT(Debug,14,(&Debug,
		     "addrs_have_port=%d, se=%p; %d addresses\n",
		     have_port,se,se->addr_count));

    return have_port;
}

/* Increments refcount */
struct service_entry * give_service_entry2(hostname,flag,initial_tls)
     const char          * hostname;
     int                   flag;
     enum initial_tls_flag initial_tls;
{
    struct schedule_timelimit now = NO_schedule_timelimit;
    char * current_time_s  UNUSED_VAROK = NULL;
    
    /* Should succeed -- error == ((time_t) -1) ignored */
    schedule_set_current_time(&now);

    current_time_s =  schedule_timeout_string(&now);
    if (current_time_s) {
	DPRINT(Debug,9,(&Debug, "give_service_entry2: got current time %s\n",
			current_time_s));
	
	free(current_time_s);
    }
    
    return give_service_entry3(hostname,flag,initial_tls,&now);
}

static enum lookup_hook_mode {
    mserv_reserved        = -4,    
    mserv_failed          = -1,  
    mserv_not_found       =  0,
    mserv_hook_done       =  1, 
    mserv_bad_syntax      =  3,   /* name have bad syntax */
    mserv_do_gethostname  = 10,
    mserv_do_gethostbyname,
    mserv_do_getaddrinfo

} service_entry_hook P_((struct service_entry **ret,
			 const char *hostname,
			 char     **rewrite,
			 const struct service_type *st,
			 int flag,
			 enum initial_tls_flag initial_tls,
			 const struct schedule_timelimit * now,		       
			 const struct schedule_timelimit * valid_until
			 /* default */,
			 int                             * was_error_p,
			 struct cancel_data             ** cancel_p,
			 const char                     ** reserved_name_p			 
			 ));   

static enum lookup_hook_mode service_entry_hook(ret,hostname,rewrite,
						st,flag,initial_tls,now,
						valid_until,was_error_p,
						cancel_p,
						reserved_name_p)
     struct service_entry **ret;
     const char           * hostname;
     char                ** rewrite;
     const struct service_type *st;
     int flag;
     enum initial_tls_flag initial_tls;
     const struct schedule_timelimit * now;
     const struct schedule_timelimit * valid_until /* default */;
     int                             * was_error_p;
     struct cancel_data             ** cancel_p    
     /* May be NULL, Used if dns lookup was cancelable */;
     const char         ** reserved_name_p;
{
    enum lookup_hook_mode result =  mserv_failed; 
    const char *tag = NULL;
    int         translated_value = 0;
    struct service_entry_addr  * addr_list  = NULL;
    int                          addr_count = 0;
    struct service_entry_name  * aliases_list = NULL;
    int                          aliases_count = 0;


    const struct dt_enum_shared * is_shared_value
	= give_dt_shared_value(&mail_services_lookup,
			       &tag,&translated_value);

    const char * reserved_name = is_special_use_domain(hostname);

    if (reserved_name) {

	if (reserved_name_p)
	    *reserved_name_p = reserved_name;
	else if (was_error_p) {
	    lib_error(CATGETS(elm_msg_cat, MeSet,
			      MeReservedDomainUnsupported,
			      "Reserved domain %s is unsupported: %s"),
		      reserved_name,hostname);
	    *was_error_p = 1;
	}

	result = mserv_reserved;	
	DPRINT(Debug,9,(&Debug, 
			"service_entry_hook: %s: special use domain %s => %d mserv_reserved\n",
			hostname,reserved_name,result));   

    } else if (is_shared_value) {

	DPRINT(Debug,9,(&Debug, 
			"service_entry_hook: %s: using shared routine for tag %s\n",
			hostname,tag));

#ifdef USE_DLOPEN
	{
	    
	    enum address_lookup_result al = 
		shared_address_lookup(hostname,rewrite,tag,translated_value,
				      is_shared_value,
				      &addr_list,&addr_count,
				      &aliases_list,&aliases_count,
				      st,now,valid_until,
				      was_error_p,cancel_p);
	    switch(al) {
	    case address_lookup_no_routine:
		result = mserv_do_gethostname; 
		DPRINT(Debug,9,(&Debug, 
				"service_entry_hook: %s: %d address_lookup_no_routine => %d mserv_do_gethostname\n",
				hostname,al,result));
		break;
	    case address_lookup_fallback: 
		result = mserv_do_gethostname; 
		DPRINT(Debug,9,(&Debug, 
				"service_entry_hook: %s: %d address_lookup_no_fallback => %d mserv_do_gethostname\n",
				hostname,al,result));
		break;		
	    case address_lookup_failed:		
		result = mserv_failed;
		DPRINT(Debug,9,(&Debug, 
				"service_entry_hook: %s: %d address_lookup_no_failed => %d mserv_failed\n",
				hostname,al,result));
		
		if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {
		    DPRINT(Debug,12,(&Debug,
				     "service_entry_hook: %s: DNS lookup canceled\n",
				     hostname));
		    goto fail;
		}		
		break;
	    case address_lookup_not_found:
		result = mserv_not_found;		
		DPRINT(Debug,9,(&Debug, 
				"service_entry_hook: %s: %d address_lookup_not_found => %d mserv_not_found\n",
				hostname,al,result));
		break;
	    case address_lookup_done:
		result = mserv_hook_done;
		DPRINT(Debug,9,(&Debug, 
				"service_entry_hook: %s: %d address_lookup_done => %d mserv_hook_done\n",
				hostname,al,result));
		break;
	    case address_lookup_bad_syntax:
		result = mserv_bad_syntax;
		DPRINT(Debug,9,(&Debug, 
				"service_entry_hook: %s: %d address_lookup_bad_syntax => %d mserv_bad_syntax\n",
				hostname,al,result));
		break;
	    }
	}
#else
	result = mserv_do_gethostname; 
	DPRINT(Debug,9,(&Debug, 
			"service_entry_hook: no shared routines available => %d mserv_do_gethostname\n",
			result));  
#endif
    } else if (!tag) { 

	switch(translated_value) {
	case mserv_lookup_shared:
	    
	    DPRINT(Debug,9,(&Debug, 
			    "service_entry_hook: %s: using shared routine\n",
			    hostname));
	    
#ifdef USE_DLOPEN
	    {
		/* Search any lookup module */
		enum address_lookup_result al = 
		    shared_address_lookup(hostname,rewrite,NULL,
					  translated_value,NULL,
					  &addr_list,&addr_count,
					  &aliases_list,&aliases_count,
					  st,now,valid_until,
					  was_error_p,cancel_p); 
		switch(al) {
		case address_lookup_no_routine:
		    result = mserv_do_gethostname; 
		    DPRINT(Debug,9,(&Debug, 
				    "service_entry_hook: %s: %d address_lookup_no_routine => %d mserv_do_gethostname\n",
				    hostname,al,result));
		break;		    
		case address_lookup_fallback: 
		    result = mserv_do_gethostname; 
		    DPRINT(Debug,9,(&Debug, 
				    "service_entry_hook: %s: %d address_lookup_fallback => %d mserv_do_gethostname\n",
				    hostname,al,result));
		    break;		
		case address_lookup_failed:
		    result = mserv_failed;
		    DPRINT(Debug,9,(&Debug, 
				    "service_entry_hook: %s: %d address_lookup_failed => %d mserv_failed\n",
				    hostname,al,result));

		    if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {
			DPRINT(Debug,12,(&Debug,
					 "service_entry_hook: %s: DNS lookup canceled\n",
					 hostname));
			goto fail;
		    }		    
		    break;
		case address_lookup_not_found:
		    result = mserv_not_found;
		    DPRINT(Debug,9,(&Debug, 
				    "service_entry_hook: %s: %d address_not_found => %d mserv_not_found\n",
				    hostname,al,result));				
		    break;
		case address_lookup_done:		    
		    result = mserv_hook_done;
		    DPRINT(Debug,9,(&Debug, 
				    "service_entry_hook: %s: %d address_lookup_done => %d mserv_hook_done\n",
				    hostname,al,result));
		    break;
		case address_lookup_bad_syntax:
		    result = mserv_bad_syntax;
		    DPRINT(Debug,9,(&Debug, 
				    "service_entry_hook: %s: %d address_lookup_bad_synatx => %d mserv_bad_syntax\n",
				    hostname,al,result));
		    break;
		}
	    }
#else
	    result = mserv_do_gethostname; 
	    DPRINT(Debug,9,(&Debug, 
			    "service_entry_hook: %s: no shared routines available  => %d mserv_do_gethostname\n",
			    hostname,result));  
#endif
	break;

	case mserv_lookup_hostname:
	    DPRINT(Debug,9,(&Debug, 
			    "service_entry_hook: %s: using gethostname\n",
			    hostname));
	    result = mserv_do_gethostname; 
	    break;

	case mserv_lookup_gethostbyname:
	    DPRINT(Debug,9,(&Debug, 
			    "service_entry_hook: %s: using gethostbyname\n",
			    hostname));
	    result = mserv_do_gethostbyname;
	    break;

	case mserv_lookup_getaddrinfo:
	    DPRINT(Debug,9,(&Debug, 
			    "service_entry_hook: %s: using getaddrinfo\n",
			    hostname));
	    result = mserv_do_getaddrinfo;
	    break;
	}
    }    

    if (mserv_hook_done == result) {
	
	DPRINT(Debug,9,(&Debug, 
			"service_entry_hook: %s: Filling shared result value%s, %d addresses%s, %d aliases",
			hostname,
			addr_list ? "" : ", no result found",
			addr_count,
			aliases_list ? "" : ", no aliases found",
			aliases_count));

	if (rewrite && *rewrite) {
	    DPRINT(Debug,9,(&Debug, 
			    ", rewrite: %s",
			    *rewrite));
	}
	DPRINT(Debug,9,(&Debug, "\n"));
	       
	if (!*ret && rewrite && *rewrite) {
	    *ret =  scan_list(*rewrite,flag,initial_tls,now,NULL,st);
	    if (*ret)
		add_alias_to_entry(*ret,hostname,valid_until);
	}
	if (!*ret) 
	    *ret =  scan_list(hostname,flag,initial_tls,now,NULL,st);

	if (!*ret && aliases_list) {
	    struct schedule_timelimit X = *valid_until;

	    int i;
	    
	    for (i = 0; i < aliases_count && !ret; i++) {
		if (aliases_list[i].aliasname) {
		    *ret =  scan_list(aliases_list[i].aliasname,
				      flag,initial_tls,now,NULL,st); 	    
		    if (*ret) {
			schedule_shorten_next_timeout(&X,
						      & aliases_list[i].valid_until);
		    }
		}
	    }

	    if (*ret) {
		add_alias_to_entry(*ret,hostname,&X);
		if (rewrite && *rewrite) 
		    add_alias_to_entry(*ret,*rewrite,&X);
	    }
	}

	if (!*ret) {
	    if ( rewrite && *rewrite) {
		*ret = malloc_service_entry(*rewrite,st,SE_temporary);
		add_alias_to_entry(*ret,hostname,valid_until);
	    } else 
		*ret = malloc_service_entry(hostname,st,SE_temporary);
	}
	
	if (addr_list) {
	    int i;
	    
	    for (i = 0; i < addr_count; i++) 
		add_se_addr_to_entry(*ret,& (addr_list[i]),1);
	}	    	    
	
	if (aliases_list) {
	     int i;
	    
	     for (i = 0; i < aliases_count; i++) {
		 add_se_alias_to_entry(*ret, &(aliases_list[i]),1);
	     }
	}
    }

#ifdef USE_DLOPEN
 fail:
#endif
    
    l_free_name_list(&aliases_list,&aliases_count);
    l_free_addr_list(&addr_list,&addr_count);

    return result;
}

static const struct service_type *give_servicetype_from_flags P_((int  flag,
								  enum initial_tls_flag  initial_tls));

static const struct service_type *give_servicetype_from_flags(flag,initial_tls)
     int  flag;
     enum initial_tls_flag  initial_tls;
{
    const struct service_type *st;

    for (st = &SERVICE_TYPES[0]; st->name; st++) {
	if (flag == (flag &st->flags) &&
	    initial_tls_match(st,initial_tls)) {

	    DPRINT(Debug,10,(&Debug, 
			    "give_servicetype_from_flags(flag=%d, initial_tls=%d)=%p; name=%s\n",
			     flag,initial_tls,st,st->name));

	    return st;
	}
    }

    DPRINT(Debug,10,(&Debug, 
		     "give_servicetype_from_flags(flag=%d, initial_tls=%d)=NULL\n",
		     flag,initial_tls));

    return NULL;
}

static int gse_rescan_name_addr P_((struct service_entry           ** ret, 
				    const struct schedule_timelimit * now,
				    const struct schedule_timelimit * valid_until,
				    int                               flag,
				    enum initial_tls_flag             initial_tls,
				    int                             * was_error_p,
				    struct cancel_data             ** cancel_p
				   ));
static int gse_rescan_name_addr(ret,now,valid_until,flag,initial_tls,was_error_p,
				cancel_p)
     struct service_entry           ** ret;
     const struct schedule_timelimit * now;
     const struct schedule_timelimit * valid_until;
     int                               flag;
     enum initial_tls_flag             initial_tls;
     int                             * was_error_p;
     struct cancel_data             ** cancel_p     
     /* May be NULL, Used if dns lookup was cancelable */;
{
    int r = 0;
    int expired = 0;
    int was_error = *was_error_p;
    struct service_entry  * value = NULL;

    if (SERVICE_ENTRY_magic != (*ret)->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "gse_rescan_name_addr",
	      "Bad magic (service_entry)",0);	

    value = *ret;
    inc_service_entry_refcount(value);

    if (0 != (value->flags & SE_static)) {
	DPRINT(Debug,9,(&Debug, 
			"gse_rescan_name_addr: %s: rescanning skipped for static entries\n",
			value->official_name));
			    
    } else if (0 != (value->flags & SE_rescan_name_addr) ||		
	( 0 != (value->flags & SE_given_name_addr) &&
	  ( 0 == value->addr_count ||
	    (schedule_have_timelimit(now) && 
	     (expired = free_expired_addrs(value,now,NULL)) > 0)
	    )
	  )
	) {

	int x;
	
	r = 1;

	if (expired > 0) {
	    DPRINT(Debug,9,(&Debug, 
			    "gse_rescan_name_addr: %s: %d addresses expired\n",
			    value->official_name,expired));
	} 
	
	if (0 == value->addr_count) {
	    DPRINT(Debug,9,(&Debug, 
			    "gse_rescan_name_addr: %s: filling named addresses, %d names\n",
			    value->official_name,value->addr_name_count));
	}
	
	if (0 != (value->flags & SE_rescan_name_addr)) {
	    /* Clear data on case of rescan 	       
	       Forget cached addresses */
	    int count UNUSED_VAROK = 
		free_expired_addrs(value,
				   /* Permanent  --
				      May fail at Tue Jan 19 03:14:07 2038
				   */
				   give_MAX_schedule_timelimit(),
				   NULL);

	    DPRINT(Debug,9,(&Debug, 
			    "gse_rescan_name_addr: %s: rescanning named addressed",
			    value->official_name));
	    
	    if (count) {
		DPRINT(Debug,9,(&Debug, ", %d address cleared",
				count));
	    }
	    
	    DPRINT(Debug,9,(&Debug,".\n"));
	}
		    	    
	value->flags   &= ~SE_rescan_name_addr;
	
	for (x = 0; x < value->addr_name_count; x++) {

	    /* Assumes that value can be changed */
	    enum lookup_hook_mode hookret = 

		service_entry_hook(&value,value->addr_name_list[x],
				   NULL /* no rewrite */,
				   value->service,flag,initial_tls,now,
				   valid_until,&was_error,cancel_p,
				   NULL);
	    
	    switch (hookret) {
	    case mserv_reserved:
		/* do not rescan because of reserved names! */
		break;
		
	    case mserv_not_found:
		/* Should we rescan ? */
		break;
		
	    case mserv_bad_syntax:
		break;
		
	    case mserv_failed:

		if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {
		    DPRINT(Debug,12,(&Debug,
				     "gse_rescan_name_addr: %s: DNS lookup canceled\n",
				     value->official_name
				     ));
		    goto fail;
		}
		
		value->flags |= SE_rescan_name_addr;
		break;
		    
	    case mserv_do_gethostname: 
#ifdef HAVE_ADDRINFO
		goto use_addrinfo;
#endif
	    case mserv_do_gethostbyname: {
		struct hostent *he  = 
		    lookup_name(value->addr_name_list[x],
				&was_error,
				cancel_p,
				NULL
				);
		
		if (he)
		    fill_address(value,he,valid_until);
		else if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {
		    DPRINT(Debug,12,(&Debug,
				     "gse_rescan_name_addr: %s: DNS lookup canceled\n",
				     value->official_name
				     ));
		    goto fail;
		} else {
		    value->flags |= SE_rescan_name_addr;
		    was_error = 1;
		}
		
	    }
		break;
		    	    
#ifdef HAVE_ADDRINFO
	    use_addrinfo:
#endif
	    case mserv_do_getaddrinfo: {
#ifdef HAVE_ADDRINFO
		struct addrinfo * ai =
		    lookup_namei(value->addr_name_list[x],
				 &was_error,
				 cancel_p,
				 NULL
				 );
		
		if (ai) {
		    fill_addresses_from_info(value,ai,valid_until);
		    freeaddrinfo(ai);  /* Addresses copied */
		} else if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {
		    DPRINT(Debug,12,(&Debug,
				     "gse_rescan_name_addr: %s: DNS lookup canceled\n",
				     value->official_name
				     ));
		    goto fail;
		} else {
		    value->flags |= SE_rescan_name_addr;
		    was_error = 1;
		}				
#endif		
	    }
		break;
	    case mserv_hook_done:                
		break;
	    }
	}
    } else {
	 DPRINT(Debug,9,(&Debug, 
			 "gse_rescan_name_addr: %s: no rescanning\n",
			 value->official_name));
    }

 fail:
    
    *was_error_p = was_error;

    free_service_entry(ret);
    *ret = value;

    return r;
}

static int gse_scan_hostname P_((struct service_entry           ** ret,
				 const char                      * hostname,
				 const struct schedule_timelimit * now,
				 const struct schedule_timelimit * valid_until,
				 SOCKADDR                        * have_ip_literal,
				 const struct service_type       * servicetype,
				 int                               flag,
				 enum initial_tls_flag             initial_tls,
				 int                             * was_error_p,
				 struct cancel_data             ** cancel_p));
				 
static int gse_scan_hostname(ret,hostname,now,valid_until,
			     have_ip_literal,servicetype,flag,
			     initial_tls, was_error_p,
			     cancel_p)
     struct service_entry           ** ret;
     const char                      * hostname;
     const struct schedule_timelimit * now;
     const struct schedule_timelimit * valid_until;
     SOCKADDR                        * have_ip_literal;
     const struct service_type       * servicetype;
     int                               flag;
     enum initial_tls_flag             initial_tls;
     int                             * was_error_p;
     struct cancel_data             ** cancel_p    
     /* May be NULL, Used if dns lookup was cancelable */;

{
    int r = 0;
    int expired = 0;
    int was_error = *was_error_p;
    struct service_entry  * value = NULL;

    if (*ret) {
	if (SERVICE_ENTRY_magic != (*ret)->magic)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "gse_scan_hostname",
		  "Bad magic (service_entry)",0);	

	value = *ret;
	inc_service_entry_refcount(value);

    }

    if (value &&
	0 != (value->flags & SE_static)) {
	int exp = 0;
	DPRINT(Debug,9,(&Debug, 
			"gse_rescan_name_addr: %s: scanning skipped for static entries\n",
			value->official_name));

	if ( schedule_invalid_next_timeout (now) ||
	     ! schedule_have_timelimit(now) ||
	     (exp = expired_service_entry(value,now))) {

	    struct service_entry * newret = NULL;
	    enum etc_hosts_status stat1 = etc_hosts_maybe;
	    
	    DPRINT(Debug,9,(&Debug, 
			    "gse_rescan_name_addr: %s: will realod static entry%s\n",
			    value->official_name,
			    exp ? " (have expired data)" : ""));
	    
	    newret =  lookup_static_host_sentry(hostname,
						now,valid_until,
						cancel_p,lc_default,
						&was_error,
						&stat1);

	    if (stat1 >=  etc_hosts_not_found)
		r = 1;

	    if (newret) {
		r = 1;
				
		free_service_entry(&value);
		value = newret;

		DPRINT(Debug,9,(&Debug, 
				"gse_rescan_name_addr: %s: static entry replaced\n",
				value->official_name));
		
	    } else if (etc_hosts_failed == stat1 &&
		cancel_p && *cancel_p && is_canceled(*cancel_p)) {
		DPRINT(Debug,12,(&Debug,
				 "gse_rescan_name_addr: %s: lookup canceled\n",
				 value ? value->official_name : hostname
				 ));
		goto fail;
	    }
	}
		 	    
    } else if (!value || (
		   0 == (value->flags & SE_given_addr) &&
		   0 == (value->flags & SE_given_name_addr)
		   && ( schedule_invalid_next_timeout (now) ||
			! schedule_have_timelimit(now) ||
			0 == value->addr_count ||
			(expired = free_expired_addrs(value,now,
						      NULL)) > 0 
		      )
		   )
	) {
	
	enum lookup_hook_mode mode = mserv_failed;
	char * rewrite = NULL;
	const char * reserved_name = NULL;
		
	r = 1;
	
	if (expired > 0) {
	    DPRINT(Debug,9,(&Debug, 
			    "gse_scan_hostname: %s: %d addresses expired\n",
			    value ? value->official_name : hostname,
			    expired));
	} 
	    
	if (value && 0 == value->addr_count) {
	    DPRINT(Debug,9,(&Debug, 
			    "gse_scan_hostname: %s: need address\n",
			    value->official_name));
	}

	if (!value) {
	    const struct service_type *st =
		servicetype ? servicetype : 
		give_servicetype_from_flags(flag,initial_tls);
		
	    
	    if (!st || !st->name)
		panic("CONNECTION PANIC",__FILE__,__LINE__,
		      "gse_scan_hostname",
		      "Service type do not found", 0);
	    
	    mode = service_entry_hook(&value,hostname,&rewrite,
				      st,flag,
				      initial_tls,now,				      
				      valid_until,
				      &was_error,
				      cancel_p,
				      dt_flag_is_set(&special_domains_lookup,
						     special_dom_static_hosts_flag)
				      ? &reserved_name : NULL);
	} else {       		
	    mode = service_entry_hook(&value,hostname,&rewrite,
				      value->service,flag,
				      initial_tls,now,				      
				      valid_until,
				      value->addr_count > 0 ? NULL : &was_error,
				      cancel_p,
				      dt_flag_is_set(&special_domains_lookup,
						     special_dom_static_hosts_flag)
				      ? &reserved_name : NULL);
	}

	switch(mode) {
	case mserv_reserved: 
	    break;
	    
	case mserv_not_found:
	    break;
	    
	case mserv_bad_syntax:
	    break;
	    
	case mserv_failed:

	    if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {
		DPRINT(Debug,12,(&Debug,
				 "gse_rescan_name_addr: %s: DNS lookup canceled\n",
				 value ? value->official_name : hostname
				 ));
		goto fail;
	    }
	    
	    break;
	    
	case mserv_do_gethostname: 
#ifdef HAVE_ADDRINFO
	    goto use_addrinfo2;
#endif

	case mserv_do_gethostbyname: {
	    struct hostent *he  = 
		lookup_name(hostname,
			    value && value->addr_count > 0 ?
			    NULL : &was_error,
			    cancel_p,
			    dt_flag_is_set(&special_domains_lookup,
					   special_dom_static_hosts_flag)
			    ? &reserved_name : NULL);
	    
	    if (he) {  	  
		/* Now try found entry again ... */
		if (!value) { 
		    value = scan_list(he->h_name,flag,initial_tls,now,
				      have_ip_literal,
				      servicetype);
		    
		    if (!value && he->h_aliases) {
			int j;
			for (j = 0; he->h_aliases[j] && !value; j++)
			    value = scan_list(he->h_aliases[j],flag,
					      initial_tls,now,have_ip_literal,
					      servicetype);
		    }
			
		    /* we need to add hostname to aliases because
		       it was not on there (otherwise entry
		       should have found earlier)
		    */
		    if (value) 
			add_alias_to_entry(value,hostname,valid_until);
		}
	
		if (!value) {
		    const struct service_type *st =
			servicetype ? servicetype : 
			give_servicetype_from_flags(flag,initial_tls);
		    		    
		    if (!st || !st->name)
			panic("CONNECTION PANIC",__FILE__,__LINE__,
			      "gse_scan_hostname",
			      "Service type do not found", 0);

		    value = malloc_service_entry(hostname,st,SE_temporary);
		}

		if (he->h_aliases) {  /* Add missing aliases */
		    int x;
		    for (x = 0; he->h_aliases[x]; x++) {
			    
			/* add_alias_to_entry or add_se_alias_to_entry
			   checks for duplicates */
			add_alias_to_entry(value,he->h_aliases[x],
					   valid_until);
		    }
		}
		
		if (he->h_addr_list) {
		    if (0 == expired) {
			DPRINT(Debug,9,(&Debug, 
					"gse_scan_hostname: %s: clearing old addresses\n",
					value->official_name));
			/* Clear old data */
			free_se_addr_list(value);
		    }
		    
		    fill_address(value,he,valid_until);
		}		
	    } else if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {
		DPRINT(Debug,12,(&Debug,
				 "gse_rescan_name_addr: %s: DNS lookup canceled\n",
				 value ? value->official_name : hostname
				 ));
		goto fail;
	    }
	       
	}
	    break;
	    
#ifdef HAVE_ADDRINFO
	use_addrinfo2:
#endif
	case mserv_do_getaddrinfo: {
#ifdef HAVE_ADDRINFO
	    struct addrinfo * ai =
		lookup_namei(hostname,
			     value && value->addr_count > 0  ?
			     NULL : &was_error,
			     cancel_p,
			     dt_flag_is_set(&special_domains_lookup,
					    special_dom_static_hosts_flag)
			     ? &reserved_name : NULL);
	    
	    if (ai) {  	  
		struct addrinfo * walk;
		
		/* Now try found entry again ... */
		if (!value) {
		    struct addrinfo * walk;
		    
		    for (walk = ai; walk && !value; walk = walk->ai_next) {
			if (walk->ai_canonname)
			    value = scan_list(walk->ai_canonname,flag,
					      initial_tls,now,have_ip_literal,
					      servicetype);
			
		    }
		    
		    /* we need to add hostname to aliases because
		       it was not on there (otherwise entry
		       should have found earlier)
		    */
		    if (value) 
			add_alias_to_entry(value,hostname,valid_until);
		}
		
		if (!value) {
		    const struct service_type *st =
			servicetype ? servicetype : 
			give_servicetype_from_flags(flag,initial_tls);
		    
		    
		    if (!st || !st->name)
			panic("CONNECTION PANIC",__FILE__,__LINE__,
			      "gse_scan_hostname",
			      "Service type do not found", 0);
		    
		    value = malloc_service_entry(hostname,st,SE_temporary);
		}
		
		/* Add missing aliases */
		for (walk = ai; walk ; walk = walk->ai_next) {
		    
		    /* add_alias_to_entry or add_se_alias_to_entry
		       checks for duplicates */
		    if (walk->ai_canonname)
			add_alias_to_entry(value,walk->ai_canonname,
					   valid_until);
		    
		} 

		if (0 == expired) {
		    DPRINT(Debug,9,(&Debug, 
				    "gse_scan_hostname: %s: clearing old addresses\n",
				    value->official_name));
		    /* Clear old data */
		    free_se_addr_list(value);
		}
		
		fill_addresses_from_info(value,ai,valid_until);
		freeaddrinfo(ai);  /* Addresses copied */
	    } else if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {
		DPRINT(Debug,12,(&Debug,
				 "gse_rescan_name_addr: %s: DNS lookup canceled\n",
				 value ? value->official_name : hostname
				 ));
		goto fail;
	    }
#endif
	}
	    break;
	    
	case mserv_hook_done:                
	    
	    if (rewrite && (!value || 0 != strcmp(value->official_name,rewrite))) {
		struct service_entry * newret;
		
		DPRINT(Debug,9,(&Debug, 
				"gse_scan_hostname: %s: Generating entry for rewritten name %s\n",
				hostname,rewrite));
		
		if (!value) {
		    const struct service_type *st =
			servicetype ? servicetype : 
			give_servicetype_from_flags(flag,initial_tls);
		    
		    
		    if (!st || !st->name)
			panic("CONNECTION PANIC",__FILE__,__LINE__,
			      "gse_scan_hostname",
			      "Service type do not found", 0);
		    
		    newret = malloc_service_entry(rewrite,st,SE_temporary);
		    
		} else
		    newret =  malloc_service_entry(rewrite,value->service,
						   value->flags | SE_temporary);
		
		if (0 != istrcmp(hostname,rewrite))
		    add_alias_to_entry(newret,hostname,valid_until);
		
		if (value) {
		    merge1_entries(newret,value);
		    free_service_entry(&value);
		}
		value = newret;
		
		DPRINT(Debug,9,(&Debug, 
				"gse_scan_hostname: New temporary service entry %p\n",
				value));
	    }
	    
	    break;
	}

	if (reserved_name) {

	    DPRINT(Debug,9,(&Debug, 
			    "gse_rescan_name_addr: %s: reserved domain %s\n",
			    hostname,reserved_name));
	    
	    if (!value &&
		dt_flag_is_set(&special_domains_lookup,
			       special_dom_static_hosts_flag)) {

		enum etc_hosts_status stat2 = etc_hosts_maybe;

		DPRINT(Debug,9,(&Debug, 
				"gse_rescan_name_addr: %s: will lookup static entry\n",
				hostname));

		value =  lookup_static_host_sentry(hostname,
						   now,valid_until,
						   cancel_p,lc_default,
						   &was_error,
						   &stat2);
		if (!value) {

		    if (etc_hosts_failed == stat2) {
			
			if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {
			    DPRINT(Debug,12,(&Debug,
					     "gse_rescan_name_addr: %s: lookup canceled\n",
					     hostname
					     ));
			    goto fail;
			}

		    } else
			goto reserved_error;
		}
		
	    } else {
		int silent;
	    reserved_error:
		silent = value && value->addr_count > 0 ;
		if (!silent) {
		    lib_error(CATGETS(elm_msg_cat, MeSet,
				      MeReservedDomainUnsupported,
				      "Reserved domain %s is unsupported: %s"),
			      reserved_name,hostname);
		    was_error = 1;
		}
	    }	    	    
	}
	
	
	if (rewrite) {
	    free(rewrite);
	    rewrite = NULL;
	}
	
    }

 fail:
    *was_error_p = was_error;

    if (*ret)
	free_service_entry(ret);
    *ret = value;
    
    return r;
}

int gse_rescan_name_port P_((struct service_entry * ret,
			     int                  * was_error_p));
int gse_rescan_name_port(ret,was_error_p)
     struct service_entry  * ret;
     int                   * was_error_p;
{
    int r = 0;
    int was_error = *was_error_p;
    int have_port = 0;   /* set if addresses have port set */

    if (SERVICE_ENTRY_magic != ret->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "gse_rescan_name_port",
	      "Bad magic (service_entry)",0);
    
    if (ret->addr_count > 0)
	have_port = addrs_have_port(ret);
    
    if (!have_port && (
		       ( 0 != (ret->flags & SE_given_name_port) &&
			 ret->port_count == 0 ) 
		       ||
		       0 != (ret->flags & SE_rescan_name_port)
		       )
	) { 
	
	int x;
	int was_rescan = 0 != (ret->flags & SE_rescan_name_port);
	
	r = 1;

	if (0 == ret->port_count) {
	    DPRINT(Debug,9,(&Debug, 
			    "gse_rescan_name_port: %s: filling named ports, %d names\n",
			    ret->official_name,ret->port_name_count));
	}

	free_se_port_list(ret);
	ret->flags   &= ~SE_rescan_name_port;
	
	for (x = 0; x < ret->port_name_count; x++) {
	    struct servent * se =
		getservbyname(ret->port_name_list[x],
			      "tcp");
	    
	    if (!se && !was_rescan) {
		lib_error(CATGETS(elm_msg_cat, MeSet,
				  MeNoTcpPortNotFound,
				  "%s: TCP port name %s not found"),
			  ret->official_name,
			  ret->port_name_list[x]);
		was_error = 1;
	    }

	    if (se)
		fill_port(ret,se);
	    else
		ret->flags |= SE_rescan_name_port;
	}
    }
    
    if (ret->port_count == 0 && !have_port &&
	0 == (ret->flags & SE_rescan_name_port)) {
	PORTS r = PORT_end;
	
	/* builtin definations here */
	if ( PORT_end == ret->service->defport) {
	    
	    /* Caller will define portlist */
	    if (ret->port_list)
	 	free(ret->port_list);
	    ret->port_list = NULL;
	    
	} else 
	    r = ret->service->defport;
	
	if (r != PORT_end) {
	    ret->port_list = safe_array_realloc(ret->port_list,
						2,sizeof (ret->port_list[0]));
	    ret->port_list[0] = r;
	    ret->port_list[1] = PORT_end;
	    ret->port_count  = 1;

	    DPRINT(Debug,9,(&Debug, 
			    "gse_rescan_name_port: %s: filling named ports, using service default port %d\n",
			    ret->official_name,r));
	}
    }

    *was_error_p = was_error;

    return r;
}
			


/* Increments refcount */
struct service_entry * give_service_entryS(value,flag,initial_tls,
					   now,cancel_p)
     struct service_entry      * value;
     int                         flag;
     enum initial_tls_flag       initial_tls;
     const struct schedule_timelimit * now;
     struct cancel_data         ** cancel_p   
     /* May be NULL, Used if dns lookup was cancelable */;
{
    struct service_entry * ret = NULL;

    int was_error = 0;

    if (SERVICE_ENTRY_magic != value->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "give_service_entryS",
	      "Bad magic (service_entry) (value)",0);

    if (! initial_tls_match(value->service,initial_tls)) {
	 DPRINT(Debug,9,(&Debug, 
			 "give_service_entryS: %s: service %s do not match to inital tls %d\n",
			 value->official_name,value->service->name,initial_tls));
	 goto fail;
    }

    if (flag != (flag & value->service->flags)) {
	DPRINT(Debug,9,(&Debug, 
			"give_service_entryS: %s: service %s flags %d do not match to flag %d\n",
			value->official_name,value->service->name,
			value->service->flags,flag));
	goto fail;
    }

    if (cancel_p && !*cancel_p) {

	enum name_lookup_cancel_v cancel_mode = give_dt_enumerate_as_int(&name_lookup_cancel);

	DPRINT(Debug,9,(&Debug, 
			"give_service_entryS: %s: name-lookup-cancel = %d\n",
			value->official_name,
			cancel_mode));
	
	switch (cancel_mode) {
	case name_lookup_cancel_enabled:
	    *cancel_p =
		new_schedule_cancel(100  /* 100 ms,
					    do not print this now, this text is
					    reset before (some) hostname lookups */,
				    CATGETS(elm_msg_cat, MeSet,MeLookingUp,
					    "Looking up %s ..."),
				    value->official_name);	
	    
	    break;
	case name_lookup_cancel_disabled:
	case name_lookup_cancel_auto:
	case NUM_name_lookup_cancel:
	    break;
	}
    }

    
    ret = malloc_service_entry(value->official_name,
			       value->service, SE_temporary);
	
    if (value->ip_literal_addrsize && value->ip_literal_address.dummy) {
	char mybuffer[256];
	const char * ip_literal_string UNUSED_VAROK =
	    give_SOCKADDR_ptr_as_string(value->ip_literal_address,
					value->ip_literal_addrsize,
					mybuffer,sizeof mybuffer);

	DPRINT(Debug,9,(&Debug, 
			"give_service_entryS: %s: is literal ip address %s\n",
			value->official_name,
			ip_literal_string ? ip_literal_string : "<NULL>"));

	if (!  set_SOCKADDR_ptr_helper(& (ret->ip_literal_addrsize),
				       & (ret->ip_literal_address),
				       value->ip_literal_addrsize,
				       value->ip_literal_address)) {
	    DPRINT(Debug,9,(&Debug,  "... failed to copy literal ip address!\n"));
	}

	merge1_entries(ret,value);
    } else {
	struct schedule_timelimit  valid_until =   NO_schedule_timelimit;

	merge1_entries(ret,value);

	if (! schedule_have_timelimit(now)) {  /* no time check */
	    valid_until = NO_schedule_timelimit;
	    now         = &NO_schedule_timelimit;
	    DPRINT(Debug,9,(&Debug, 
			    "give_service_entryS: %s: No time checks\n",
			    hostname));

	    } else if (/* May fail at Tue Jan 19 03:14:07 2038 */
		   schedule_set_next_timeout0(&valid_until,now,
					      name_lookup_cache_time < MIN_NAME_CACHE_TIME ?
					      MIN_NAME_CACHE_TIME : name_lookup_cache_time,
					      /* Use time_MAX -1 on failure
						 May fail at Tue Jan 19 03:14:06 2038 */
					      se_use_before_time_MAX)) {

	    char * valid_until_s  UNUSED_VAROK = schedule_timeout_string(&valid_until);

	    if (valid_until_s) {

		DPRINT(Debug,9,(&Debug, 
				"give_service_entryS: %s: default valid_until %s\n",
				hostname,valid_until_s));
		free(valid_until_s);
	    }

	} else {
	    DPRINT(Debug,9,(&Debug, 
			    "give_service_entryS: %s: time overflow\n",
			    hostname));
	}

	if (ret->ip_literal_addrsize) {
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "give_service_entryS","Got literal address", 0);
	}

	if (gse_rescan_name_addr(&ret,now,&valid_until,flag,initial_tls,
				 &was_error,cancel_p)) {
	   
	    DPRINT(Debug,9,(&Debug, 
			    "give_service_entryS: %s: rescanned named addresses\n",
			    ret->official_name));

	}

	if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {
	    DPRINT(Debug,12,(&Debug,
			     "give_service_entryS: %s: DNS lookup canceled\n",
			     ret->official_name
			     ));
	    goto fail;
	}
		
	if (gse_scan_hostname(&ret,value->official_name,
			      now,&valid_until,
			      value->ip_literal_address.generic  /* NULL */,
			      value->service,flag,initial_tls,
			      &was_error,cancel_p)) {

	    DPRINT(Debug,9,(&Debug, 
			    "give_service_entryS: %s: scanned hostname\n",
			    ret->official_name));
	}

	if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {
	    DPRINT(Debug,12,(&Debug,
			     "give_service_entryS: %s: DNS lookup canceled\n",
			     ret->official_name
			     ));
	    goto fail;
	}
    }

    if (gse_rescan_name_port(ret,&was_error)) {

	DPRINT(Debug,9,(&Debug, 
			"give_service_entryS: %s: rescanned named ports\n",
			ret->official_name));
	
    }

    if (was_error) {
	DPRINT(Debug,1,(&Debug, 
			"give_service_entryS: %s: Was error\n",
			ret->official_name));

	if (sleepmsg > 0) {  /* HACK */
	    if (POLL_method)
		wait_for_timeout(sleepmsg);
	    else
		sleep(sleepmsg);
	}
    }

 fail:
    if (ret) {
	DPRINT(Debug,9,(&Debug, 
			"give_service_entryS=%p: type=%p (%s)%s%s, refcount=%d, addr_count=%d\n",
			ret,ret->service,ret->service->name,
			ret->flags & SE_temporary ? ", temporary" : "",
			ret->flags & SE_static    ? ", static"    : "",
			ret->refcount,
			ret->addr_count));

    } else {
	DPRINT(Debug,9,(&Debug, 
			"give_service_entryS=NULL\n"));
    }

    return ret;
}

/* Increments refcount */
struct service_entry * give_service_entry4(hostname,have_ip_literal,
					   servicetype,
					   flag,initial_tls,now)
     const char                      * hostname;
     SOCKADDR                        * have_ip_literal;
     const struct service_type       * servicetype;
     int                               flag;
     enum initial_tls_flag             initial_tls;
     const struct schedule_timelimit * now;
{

    return give_service_entry5(hostname,have_ip_literal,
			       servicetype,
			       flag,initial_tls,now,
			       NULL);
}


/* Increments refcount */
struct service_entry * give_service_entry5(hostname,have_ip_literal,
					       servicetype,
					       flag,initial_tls,now,cancel_p)
     const char                      * hostname;
     SOCKADDR                        * have_ip_literal;
     const struct service_type       * servicetype;
     int                               flag;
     enum initial_tls_flag             initial_tls;
     const struct schedule_timelimit * now;
     struct cancel_data         ** cancel_p
     /* May be NULL, Used if dns lookup was cancelable */;
{
   struct service_entry *ret;       

   int was_error = 0;
   
   struct schedule_timelimit valid_until = NO_schedule_timelimit;
   
   static int report_time_MAX = 1;

    char mybuffer[256];
    const char * ip_literal_string = NULL;

    if (report_time_MAX) {
	char * X = ctime(&time_MAX);
	
	report_time_MAX = 0;

	if (X) {
	    DPRINT(Debug,1,(&Debug,
			    "Using time_MAX = %ld  = %s",
			    (long)time_MAX,X));
	} else {
	    DPRINT(Debug,1,(&Debug,
			    "Using time_MAX = %ld\n",
			    (long)time_MAX));
	}
	DPRINT(Debug,1,(&Debug,"\ntime_t size is %ld bytes\n",
			(long)sizeof(time_t)));
    }

    if (have_ip_literal) {

	ip_literal_string = 
	    give_addr_as_string(have_ip_literal,mybuffer,sizeof mybuffer);

    }

    DPRINT(Debug,9,(&Debug, 
		    "give_service_entry5(name=\"%s\",have_ip_literal=%p (%s), servicetype=%p (%s),flag=%d, initial_tls=%d)\n",
		    hostname,
		    have_ip_literal, 
		    ip_literal_string ? ip_literal_string : "<none>",
		    servicetype,
		    servicetype && servicetype->name ? 
		    servicetype->name : "<none>",
		    flag,initial_tls));

    if (! schedule_have_timelimit(now)) {  /* no time check */
	valid_until = NO_schedule_timelimit;
	now         = &NO_schedule_timelimit;
	DPRINT(Debug,9,(&Debug, 
			"give_service_entry5: %s: No time checks\n",
			hostname));
    } else if (/* May fail at Tue Jan 19 03:14:07 2038 */
	       schedule_set_next_timeout0(&valid_until,now,
					  name_lookup_cache_time < MIN_NAME_CACHE_TIME ?
					  MIN_NAME_CACHE_TIME : name_lookup_cache_time,					      
					  /* Use time_MAX -1 on failure
					     May fail at Tue Jan 19 03:14:06 2038 */
					  se_use_before_time_MAX)) {

	char * valid_until_s  = schedule_timeout_string(&valid_until);

	if (valid_until_s) {

	    DPRINT(Debug,9,(&Debug, 
			    "give_service_entry5: %s: default valid_until %s\n",
			    hostname,valid_until_s));
	    free(valid_until_s);
	}

    } else {
	DPRINT(Debug,9,(&Debug, 
			"give_service_entry5: %s: time overflow\n",
			hostname));
    }

    if (servicetype) {

	if (flag == (flag & servicetype->flags) &&
	    initial_tls_match(servicetype,initial_tls)) {
	    DPRINT(Debug,9,(&Debug, 
			    "give_service_entry5: service type %s matches to flags=%d, initial_tls=%d\n",
			    servicetype->name ? servicetype->name : "<none>",
			    flag,initial_tls));
			    
	} else {
	    DPRINT(Debug,9,(&Debug, 
			    "give_service_entry5=NULL: bad service type %s; flags=%d, initial_tls=%d\n",
			    servicetype->name ? servicetype->name : "<none>",
			    flag,initial_tls));
	    return NULL;
 
	}
    }

    if (cancel_p && !*cancel_p) {

	enum name_lookup_cancel_v cancel_mode = give_dt_enumerate_as_int(&name_lookup_cancel);

	DPRINT(Debug,9,(&Debug, 
			"give_service_entry5: %s: name-lookup-cancel = %d\n",
			hostname,
			cancel_mode));
	
	switch (cancel_mode) {
	case name_lookup_cancel_enabled:
	    *cancel_p =
		new_schedule_cancel(100  /* 100 ms,
					    do not print this now, this text is
					    reset before (some) hostname lookups */,
				    CATGETS(elm_msg_cat, MeSet,MeLookingUp,
					    "Looking up %s ..."),
				    hostname);
	    
	    break;
	case name_lookup_cancel_disabled:
	case name_lookup_cancel_auto:
	case NUM_name_lookup_cancel:
	    break;
	}
    }
    
    ret = scan_list(hostname,flag,initial_tls,now,have_ip_literal,
		    servicetype);
    /* If on service entry do not have ip addresses
       they needed to be retrieved. Notice that we
       retrieve new entries every time (alternatively
       we should keep track of expiry times).
       However all other data is cached (also addresses if
       them is generated from names given on configuration data)
    */

    if (ret)
	hostname = ret->official_name;

    if (have_ip_literal) {

	if (!ret) {
	    const struct service_type *st = 
		servicetype ? servicetype : 
		give_servicetype_from_flags(flag,initial_tls);
	    struct service_entry_addr SE;

	    bzero((void *)&SE, sizeof SE);
	    
	    if (!st || !st->name)
		panic("CONNECTION PANIC",__FILE__,__LINE__,
		      "give_service_entry5",
		      "Service type do not found", 0);

	    ret = malloc_service_entry(hostname,st,SE_temporary);
	    
	    set_address_from_gen_helper(& (ret->ip_literal_addrsize),
					& (ret->ip_literal_address),
					have_ip_literal);
	    
	    
	    set_address_from_gen_helper(& (SE.addrsize),
					& (SE.address),
					have_ip_literal);
	    
	    SE.hostname = safe_strdup(ip_literal_string ? ip_literal_string : hostname);

	    /* Permanent  --
	       May fail at Tue Jan 19 03:14:07 2038 UTC
	    */
	    set_MAX_schedule_timelimit(&(SE.valid_until));
	    
	    add_se_addr_to_entry(ret,&SE,1 /* malloc copy */);
	    
	    free(SE.address.dummy);
	    free(SE.hostname);
	}

    } else {

	if (ret && ret->ip_literal_addrsize) {
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "give_service_entry5","Got literal address", 0);
	}

	if (ret && 
	    gse_rescan_name_addr(&ret,now,&valid_until,flag,initial_tls,
				 &was_error,cancel_p)) {
	   
	    DPRINT(Debug,9,(&Debug, 
			    "give_service_entry5: %s: rescanned named addresses\n",
			    ret->official_name));

	}

	if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {
	    DPRINT(Debug,12,(&Debug,
			     "give_service_entry5: %s: DNS lookup canceled\n",
			     ret->official_name
			     ));
	    goto fail;
	}

	
	if (gse_scan_hostname(&ret,hostname,now,&valid_until,
			      have_ip_literal,servicetype,flag,initial_tls,
			      &was_error,cancel_p)) {

	    DPRINT(Debug,9,(&Debug, 
			    "give_service_entry5: %s: scanned hostname\n",
			    hostname));

	}

	if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {
	    DPRINT(Debug,12,(&Debug,
			     "give_service_entry5: %s: DNS lookup canceled\n",
			     hostname));
	    goto fail;
	}
	
    } /* ! have_ip_literal */

    if (ret &&  gse_rescan_name_port(ret,&was_error)) {

	DPRINT(Debug,9,(&Debug, 
			"give_service_entry5: %s: rescanned named ports\n",
			hostname));
	
    }

    if (was_error) {
	DPRINT(Debug,1,(&Debug, 
			"give_service_entry5: %s: Was error\n",
			hostname));

	if (sleepmsg > 0) {  /* HACK */
	    if (POLL_method)
		wait_for_timeout(sleepmsg);
	    else
		sleep(sleepmsg);
	}
    }

 fail:
    if (ret) {
	DPRINT(Debug,9,(&Debug, 
			"give_service_entry5=%p: type=%p (%s)%s%s, refcount=%d, addr_count=%d\n",
			ret,ret->service,ret->service->name,
			ret->flags & SE_temporary ? ", temporary" : "",
			ret->flags & SE_static    ? ", static"    : "",
			ret->refcount,
			ret->addr_count));

    } else {
	DPRINT(Debug,9,(&Debug, 
			"give_service_entry5=NULL\n"));
    }

    return ret;
}

static void looking_up_msg_helper(cancel_p,show_message,s)
     struct cancel_data    ** cancel_p
     /* May be NULL, used if dns lookup was cancelable */;
     enum lookup_show_message show_message;
     const char             * s;
{
    if ( cancel_p && *cancel_p) { /* Interrupting of 
				     gethostbyaddr(), getnameinfo(),
				     getaddrinfo() or gethostbyname()
				     is not really
				     supported, result is uncertain */
	    
	    	    
	if (is_schedule_cancel(*cancel_p,NULL)) {
	    set_cancel_message(*cancel_p,
			       -1 /* also show it now */,
			       CATGETS(elm_msg_cat, MeSet,MeLookingUp,
				       "Looking up %s ..."),
			       s ? s : "(addr)");
	    
	} else {
	    DPRINT(Debug,12,(&Debug,
			     "looking_up_msg_helper: %s: 'Looking up' already printed.\n",
			     s ? s : "(addr)"
			     ));
	}
	
    } else if (s) {
	switch (show_message) {
	case no_lookup_message:
	    break;
	case show_lookup_message:
	    lib_transient(CATGETS(elm_msg_cat, MeSet,MeLookingUp,
				  "Looking up %s ..."),
			  s);
	    break;
	}
    }
}


#ifdef HAVE_GETNAMEINFO
static int lookupi_SOCKADDR_ptr P_((const union SOCKADDR_ptr sockaddr_ptr,
				    const size_t             sockaddr_size,
				    char                 *** res_p,
				    size_t                 * reslen_p,
				    struct cancel_data    ** cancel_p
				    /* May be NULL, used if dns lookup was cancelable */,
				    enum lookup_show_message show_message,
				    int                    * was_error_p
				    /* Error message is shown if was_error_p != NULL */,
				    const char             * s));

static int lookupi_SOCKADDR_ptr(sockaddr_ptr,sockaddr_size,res_p,reslen_p,cancel_p,
				show_message,was_error_p,s)
     const union SOCKADDR_ptr sockaddr_ptr;
     const size_t             sockaddr_size;
     char                 *** res_p;
     size_t                 * reslen_p;
     struct cancel_data    ** cancel_p         /* May be NULL, used if dns lookup was cancelable */;
     enum lookup_show_message show_message;
     int                    * was_error_p      /* Error message is shown if was_error_p != NULL */;
     const char             * s;
{
    char hostname[SLEN];
    int ret = 0;
    int status;
    char           ** res     = *res_p;
    size_t            reslen  = *reslen_p;
    
    looking_up_msg_helper(cancel_p,show_message,s);

    errno = 0;

    status = getnameinfo(sockaddr_ptr.sa,
			 sockaddr_size,hostname,sizeof hostname,
			 NULL,0,
			 NI_NAMEREQD);

    if (0 == status) {
	size_t idx = 0;
	size_t count = 1;
	    	    
	if (res) {
	    /* CLear previous data */
	    
	    size_t i;
	    
	    for (i = 0; i < reslen; i++) {
		if (res[i]) {
		    free(res[i]);
		    res[i] = NULL;
		}
	    }
	}

	res = safe_array_realloc(res,count+1,sizeof (res[0]));

#define ADD(s) do { if (idx >= count) { goto failx; } res[idx++] =  safe_strdup(s); } while(0)
	
	ADD(hostname);

    failx:
	DPRINT(Debug,12,(&Debug,
			 "lookupi_SOCKADDR_ptr: %zu items\n",
			 idx));
	
	res = safe_array_realloc(res,idx+1,sizeof (res[0]));
	res[idx] = NULL;
	
	reslen = idx;
	ret = 1;

    } else {  /* Failure */
	int err = errno;

	DPRINT(Debug,9,(&Debug, 
			"lookupi_SOCKADDR_ptr: %s: status %d (%s)",
			hostname,status,gai_strerror(status)));
	if (err) {
	    DPRINT(Debug,9,(&Debug, " errno=%d (%s)",
			    err,strerror(err)));
	}
	DPRINT(Debug,9,(&Debug, "\n"));

	if (EAI_SYSTEM == status &&
	    EINTR == err &&  cancel_p && *cancel_p &&
	    is_canceled(*cancel_p)) {
	    DPRINT(Debug,9,(&Debug, 
			    "lookupi_SOCKADDR_ptri: %s: query canceled\n",
			    hostname));
	    goto fail;

	} else if (was_error_p) {
	    switch (status) {
	    case EAI_AGAIN:
		if (s) {
		    lib_error(CATGETS(elm_msg_cat, MeSet,MeTempNoNameAddress,
				      "Name for address %s is not yet found..."),
			      s);
		    *was_error_p = 1;
		}
		break;
	    case EAI_NONAME:
		if (s) {
		    lib_error(CATGETS(elm_msg_cat, MeSet,MeNoNameAddress,
				      "No name for address: %s"),
			      s);
		    *was_error_p = 1;
		}
		break;
	    case EAI_SYSTEM:
		if (s) {
		    if (err)
			lib_error(CATGETS(elm_msg_cat, 
					  MeSet,
					  MeAddrNameError1,
					  "Failed to get name for address %s: %s"),
				  s,strerror(err));
		    else 
			lib_error(CATGETS(elm_msg_cat, MeSet,MeAddrNameError,
					  "Failed to get name for address %s"),
				  s);
		    *was_error_p = 1;
		}
		break;
	    default:
		if (s) {
		    lib_error(CATGETS(elm_msg_cat, 
				      MeSet,
				      MeAddrNameError1,
				      "Failed to get name for address %s: %s"),
			      s,gai_strerror(status));
		    *was_error_p = 1;
		}
		break;
	    }
	}
    }
    
    if (!ret) {
 fail:
	if (res) {
	    size_t i;
	    
	    for (i = 0; i < reslen; i++) {
		if (res[i]) {
		    free(res[i]);
		    res[i] = NULL;
		}
	    }

	    if (res) {
		free(res);
		res = NULL;
	    }
	}
	reslen = 0;
    }
    
    *res_p    =  res;
    *reslen_p =  reslen  ;
    
    DPRINT(Debug,12,(&Debug,
		     "lookupi_SOCKADDR_ptr=%d length %zu",ret,reslen));

    if (res) {
	size_t i;
	
	DPRINT(Debug,12,(&Debug," result %p",res));

	for (i = 0; i < reslen; i++) {
	    DPRINT(Debug,12,(&Debug,
			     "%c %Q",
			     i ? ',' : ':',
			     res[i]));
	    
	}
    } 
    
    DPRINT(Debug,12,(&Debug,"\n"));
    

    return ret;
}

#endif


static int lookup_SOCKADDR_ptr P_((const union SOCKADDR_ptr sockaddr_ptr,
				   const size_t             sockaddr_size,
				   char                 *** res_p,
				   size_t                 * reslen_p,
				   struct cancel_data    ** cancel_p
				   /* May be NULL, used if dns lookup was cancelable */,
				   enum lookup_show_message show_message,
				   int                    * was_error_p
				   /* Error message is shown if was_error_p != NULL */,
				   const char             * s));

static int lookup_SOCKADDR_ptr(sockaddr_ptr,sockaddr_size,res_p,reslen_p,cancel_p,
			       show_message,was_error_p,s)
     const union SOCKADDR_ptr sockaddr_ptr;
     const size_t             sockaddr_size;
     char                 *** res_p;
     size_t                 * reslen_p;
     struct cancel_data    ** cancel_p         /* May be NULL, used if dns lookup was cancelable */;
     enum lookup_show_message show_message;
     int                    * was_error_p      /* Error message is shown if was_error_p != NULL */;
     const char             * s;
     
{
    int ret = 0;

    char           ** res     = *res_p;
    size_t            reslen  = *reslen_p;
	
    struct hostent * he = NULL;
    int check_error = 0;
    
    errno = 0;
    
    if (sockaddr_size < sizeof (sockaddr_ptr.sa->sa_family)) {
	DPRINT(Debug,12,(&Debug,
			 "lookup_SOCKADDR_ptr: sockaddr size = %zu < sa_family size %zu\n",
			 sockaddr_size,sizeof (sockaddr_ptr.sa->sa_family)));
	goto fail;
	    
    } else {

	looking_up_msg_helper(cancel_p,show_message,s);
			
	switch (sockaddr_ptr.sa->sa_family) {
#if defined(I_NETINET_IN) && defined(I_ARPA_INET)
	case AF_INET:
	    
	    if (sockaddr_size < sizeof (* (sockaddr_ptr.sin))) {
		DPRINT(Debug,12,(&Debug,
				 "lookup_SOCKADDR_ptr: sockaddr size = %zu < sockaddr_sin size %zu\n",
				 sockaddr_size,sizeof (* (sockaddr_ptr.sin))));
		goto fail;
	    }

	    he = gethostbyaddr((void *)&(sockaddr_ptr.sin->sin_addr),
			       sizeof (sockaddr_ptr.sin->sin_addr),
			       sockaddr_ptr.sin->sin_family);

	    if (!he)
		check_error = 1;
	    break;
		
#ifdef HAVE_IN6
	case AF_INET6:

	    if (sockaddr_size < sizeof (* (sockaddr_ptr.sin6))) {
		DPRINT(Debug,12,(&Debug,
				 "lookup_SOCKADDR_ptr: sockaddr size = %zu < sockaddr_sin6 size %zu\n",
				 sockaddr_size,sizeof (* (sockaddr_ptr.sin6))));
		goto fail;
	    }
	    
	    he =  gethostbyaddr((void *)&(sockaddr_ptr.sin6->sin6_addr),
				sizeof (sockaddr_ptr.sin6->sin6_addr),
				sockaddr_ptr.sin6->sin6_family);

	    if (!he)
		check_error = 1;

	    break;
#endif		
#endif
	default:
	    DPRINT(Debug,12,(&Debug,
			     "lookup_SOCKADDR_ptr: Unsupported address family %d\n",
			     sockaddr_ptr.sa->sa_family));
	    break;
	}


	if (check_error) {
	    int err = errno; 
	    
	    DPRINT(Debug,9,(&Debug, 
			    "lookup_SOCKADDR_ptr: %s: h_errno=%d",
			    s ? s : "(addr)",h_errno));
	    if (err) {
		DPRINT(Debug,9,(&Debug, " errno=%d (%s)",
				err,strerror(err)));
	    }
	    DPRINT(Debug,9,(&Debug, "\n"));

	    if (HOST_NOT_FOUND != h_errno &&
		NO_DATA        != h_errno &&
		NO_RECOVERY    != h_errno &&
		EINTR == err &&  cancel_p && *cancel_p &&
		is_canceled(*cancel_p)) {
		DPRINT(Debug,9,(&Debug, 
				"lookup_SOCKADDR_ptr: %s: query canceled\n",
				s ? s : "(addr)"));
		goto fail;

	    } else if (was_error_p) {

		switch(h_errno) {
		case HOST_NOT_FOUND:
		case NO_DATA:
		    if (s) {
			lib_error(CATGETS(elm_msg_cat, MeSet,MeNoNameAddress,
					  "No name for address: %s"),
				  s);
			*was_error_p = 1;
		    }
		    break;
		case TRY_AGAIN:
		    if (s) {
			if (err)
			    lib_error(CATGETS(elm_msg_cat, 
					      MeSet,
					      MeLookErrno,
					      "Looking up %s: %s"),
				      s,strerror(err));
			else	       
			    lib_error(CATGETS(elm_msg_cat, MeSet,MeTempNoNameAddress,
					      "Name for address %s is not yet found..."),
				      s);
			*was_error_p = 1;
			
		    }
		    break;
		case NO_RECOVERY:
		    if (s) {
			lib_error(CATGETS(elm_msg_cat, MeSet,MeNoRecoveryBadAddress,
					  "Looking up %s: No recovery; Bad address or refused query?"),
				  hostname);
			*was_error_p = 1;
		    }
		    break;
		default:
		    if (s) {
			if (err)
			    lib_error(CATGETS(elm_msg_cat, 
					      MeSet,
					      MeAddrNameError1,
					      "Failed to get name for address %s: %s"),
				      s,strerror(err));
			else 
			    lib_error(CATGETS(elm_msg_cat, MeSet,MeAddrNameError,
					      "Failed to get name for address %s"),
				      s);
			*was_error_p = 1;
		    }
		    break;
		}
	    }

	} else if (s) {
	    switch (show_message) {
	    case no_lookup_message:
		break;
	    case show_lookup_message:		
		lib_transient(CATGETS(elm_msg_cat, MeSet,MeLookingUpOK,
				      "Looking up %s ... OK"),
			      s);
		break;
	    }
	}
    }

    if (he) {
	size_t count = 0;
	size_t idx = 0;
	
	if (he->h_name)
	    count++;
	if (he->h_aliases) {
	    size_t i;
	    
	    for (i = 0; he->h_aliases[i]; i++)
		count++;
	}


	if (res) {
	    /* CLear previous data */
	    
	    size_t i;
	    
	    for (i = 0; i < reslen; i++) {
		if (res[i]) {
		    free(res[i]);
		    res[i] = NULL;
		}
	    }
	}

	
	res = safe_array_realloc(res,count+1,sizeof (res[0]));

	if (reslen < count+1) {
	    /* Clear rest */
	    
	    size_t i;
	    
	    for (i = reslen; i < count+1; i++) {
		res[i] = NULL;	    
	    }
	}

#define ADD(s) do { if (idx >= count) { goto failx; } res[idx++] =  safe_strdup(s); } while(0)

	if (he->h_name)
	    ADD(he->h_name);

	if (he->h_aliases) {
	    size_t i;
	    
	    for (i = 0; he->h_aliases[i]; i++) {
		ADD(he->h_aliases[i]);
	    }
	}

#undef ADD
	    
    failx:
	DPRINT(Debug,12,(&Debug,
			 "lookup_SOCKADDR_ptr: %zu items\n",
			 idx));
	
	res = safe_array_realloc(res,idx+1,sizeof (res[0]));
	res[idx] = NULL;
	
	reslen = idx;
	ret = 1;
	
    }
    
    if (!ret) {
 fail:
	if (res) {
	    size_t i;
	    
	    for (i = 0; i < reslen; i++) {
		if (res[i]) {
		    free(res[i]);
		    res[i] = NULL;
		}
	    }

	    if (res) {
		free(res);
		res = NULL;
	    }
	}
	reslen = 0;
    }
    
	
    *res_p    =  res;
    *reslen_p =  reslen  ;
    
    DPRINT(Debug,12,(&Debug,
		     "lookup_SOCKADDR_ptr=%d length %zu",ret,reslen));

    if (res) {
	size_t i;
	
	DPRINT(Debug,12,(&Debug," result %p",res));

	for (i = 0; i < reslen; i++) {
	    DPRINT(Debug,12,(&Debug,
			     "%c %Q",
			     i ? ',' : ':',
			     res[i]));
	    
	}
    } 
    
    DPRINT(Debug,12,(&Debug,"\n"));
    
    return ret;
}

				   
/* Return 1 if succedd, 0 on failure 

   Check with 		
      if (cancel_p && *cancel_p && is_canceled(*cancel_p))
   if failed for cancelation

   (re)allocates (*res_p) array
   
   Caller must free (*res_p) array
*/


int give_name_from_sockaddr(sockaddr_ptr,sockaddr_size,res_p,reslen_p,cancel_p,
			    show_message,was_error_p)
     const union SOCKADDR_ptr sockaddr_ptr;
     const size_t             sockaddr_size;
     char                 *** res_p;
     size_t                 * reslen_p;
     struct cancel_data    ** cancel_p         /* May be NULL, used if dns lookup was cancelable */;
     enum lookup_show_message show_message;
     int                    * was_error_p      /* Error message is shown if was_error_p != NULL */;
{

    int ret = 0;
    char           ** res     = *res_p;
    size_t            reslen  = *reslen_p;


    
    enum addr_lookup_v alookup =
	give_dt_enumerate_as_int(&address_lookup);

    char mybuffer[256];
    const char * s = give_SOCKADDR_ptr_as_string(sockaddr_ptr,sockaddr_size,
						     mybuffer,sizeof mybuffer);

    

    DPRINT(Debug,12,(&Debug,
		     "give_name_from_sockaddr: %s: address-lookup=%d",
		     s ? s : "(addr)", alookup));        
    switch (alookup) {
    case addr_lookup_normal:
	DPRINT(Debug,12,(&Debug," addr_lookup_normal"));
	break;
    case addr_lookup_shared:  /* RESERVED */
	DPRINT(Debug,12,(&Debug," addr_lookup_shared"));
	break;
    case addr_lookup_gethostbyaddr:
	DPRINT(Debug,12,(&Debug," addr_lookup_gethostbyaddr"));
	break;
    case addr_lookup_getnameinfo:
	DPRINT(Debug,12,(&Debug," addr_lookup_getnameinfo"));
	break;
    case NUM_addr_lookup:
	DPRINT(Debug,12,(&Debug,"  NUM_addr_lookup"));
	break;
    }
    DPRINT(Debug,12,(&Debug,"\n"));

    switch (alookup) {
    case addr_lookup_normal:
    case addr_lookup_gethostbyaddr:
	ret = lookup_SOCKADDR_ptr(sockaddr_ptr,sockaddr_size,&res,&reslen,cancel_p,
				  show_message,was_error_p,s);
	break;
    case addr_lookup_getnameinfo:
#ifdef HAVE_GETNAMEINFO
	ret = lookupi_SOCKADDR_ptr(sockaddr_ptr,sockaddr_size,&res,&reslen,cancel_p,
				   show_message,was_error_p,s);

#else
    DPRINT(Debug,12,(&Debug,
		     "give_name_from_sockaddr: %s: getaddrinfo() not available.\n",
		     s ? s : "(addr)"));
#endif
	break;
    case addr_lookup_shared:
    case NUM_addr_lookup: /* Not used */
	break;
    }

    *res_p    =  res;
    *reslen_p =  reslen  ;
    
    DPRINT(Debug,12,(&Debug,
		     "give_name_from_sockaddr=%d %s length %zu",ret,s ? s : "(addr)",reslen));

    if (res) {
	size_t i;
	
	DPRINT(Debug,12,(&Debug," result %p",res));

	for (i = 0; i < reslen; i++) {
	    DPRINT(Debug,12,(&Debug,
			     "%c %Q",
			     i ? ',' : ':',
			     res[i]));
	    
	}
    } 
    
    DPRINT(Debug,12,(&Debug,"\n"));
    
    return ret;
}

/* Increments refcount */
struct service_entry * give_service_entry_can(hostname,flag,initial_tls,now,
					      cancel_p)
     const char                      * hostname;
     int                               flag;
     enum initial_tls_flag 	       initial_tls;
     const struct schedule_timelimit * now  /* May be NULL */;
     struct cancel_data             ** cancel_p
     /* May be NULL, Used if dns lookup was cancelable */;
{
    struct service_entry *ret = NULL;
    struct schedule_timelimit current_time = NO_schedule_timelimit;

    SOCKADDR ip_literal;
    SOCKADDR * have_ip_literal = NULL;

    /* Make comparision easier */
    bzero((void *)&ip_literal, sizeof ip_literal);

    
    if (!now) {
	char * current_time_s  UNUSED_VAROK = NULL;
	/* Should succeed -- error == ((time_t) -1) ignored */
	schedule_set_current_time(&current_time);

	current_time_s =  schedule_timeout_string(&current_time);
	if (current_time_s) {
	    DPRINT(Debug,9,(&Debug, "give_service_entry_can: got current time %s\n",
			    current_time_s));
	    
	    free(current_time_s);
	}
	
	now = &current_time;
    }

#if defined(I_NETINET_IN) && defined(I_ARPA_INET)
    if (get_ip(&ip_literal,hostname)) {

	have_ip_literal = &ip_literal;

	DPRINT(Debug,9,(&Debug, 
			"give_service_entry_can: %s: is IP-literal\n",
			hostname));
   }
#endif
    
    ret = give_service_entry5(hostname,have_ip_literal,
			      NULL /* service type */,
			      flag, initial_tls,now,
			      cancel_p);
    
    return ret;
}

/* Increments refcount */
struct service_entry * give_service_entry3(hostname,flag,initial_tls,now)
     const char                      * hostname;
     int                               flag;
     enum initial_tls_flag             initial_tls;
     const struct schedule_timelimit * now;
{
    struct service_entry *ret;       

    SOCKADDR ip_literal;
    SOCKADDR * have_ip_literal = NULL;

    /* Make comparision easier */
    bzero((void *)&ip_literal, sizeof ip_literal);
	
    DPRINT(Debug,9,(&Debug, 
		    "give_service_entry3(name=\"%s\",flags=%d, initial_tls=%d)\n",
		    hostname,flag,initial_tls));


#if defined(I_NETINET_IN) && defined(I_ARPA_INET)
    if (get_ip(&ip_literal,hostname)) {

	have_ip_literal = &ip_literal;

	DPRINT(Debug,9,(&Debug, 
			"give_service_entry3: %s: is IP-literal\n",
			hostname));
   }
#endif

    ret = give_service_entry4(hostname,have_ip_literal,
			      NULL /* service type */,
			      flag,initial_tls,now);

    if (ret) {
	DPRINT(Debug,9,(&Debug, 
			"give_service_entry3=%p: type=%p (%s)%s%s, refcount=%d, addr_count=%d\n",
			ret,ret->service,ret->service->name,
			ret->flags & SE_temporary ? ", temporary" : "",
			ret->flags & SE_static    ? ", static"    : "",
			ret->refcount,
			ret->addr_count));

    } else {
	DPRINT(Debug,9,(&Debug, 
			"give_service_entry3=NULL\n"));
    }

    return ret;
}

/* Increments refcount */
struct service_entry * give_service_entry(hostname,flag)
     const char *hostname;
     int flag;
{
    return give_service_entry2(hostname,flag,itls_ignore);
}

void l_free_name_list(aliases_list,aliases_count)
     struct service_entry_name  ** aliases_list;
     int                         * aliases_count;
{
    struct service_entry_name  * ALIASES_LIST  = * aliases_list;
    int                          ALIASES_COUNT = * aliases_count;

    if (ALIASES_LIST) {
	int x;
		
	for (x = 0; x < ALIASES_COUNT; x++) {
	    if (ALIASES_LIST[x].aliasname) {
		free(ALIASES_LIST[x].aliasname);
		ALIASES_LIST[x].aliasname = NULL;
	    }
	}
	
	free(ALIASES_LIST);
	ALIASES_LIST  = NULL;
    }
    ALIASES_COUNT = 0;    
   
    * aliases_list  = ALIASES_LIST;
    * aliases_count = ALIASES_COUNT;
}

void free_se_aliases_list(se) 
     struct service_entry * se;
{
    if (SERVICE_ENTRY_magic != se->magic)
	 panic("CONNECTION PANIC",__FILE__,__LINE__,
	       "free_se_aliases_list",
	       "Bad magic (service_entry) (*ret)",0);

    l_free_name_list(& (se->aliases_list),
		     & (se->aliases_count));
}

void l_free_addr_list(addr_list,addr_count)
     struct service_entry_addr  ** addr_list;
     int                         * addr_count;
{
    struct service_entry_addr  * ADDR_LIST  = *addr_list;
    int                          ADDR_COUNT = *addr_count;
    

    if (ADDR_LIST) {
	int i;

	for (i = 0; i < ADDR_COUNT; i++) {
	    if (ADDR_LIST[i].address.dummy) {
		free(ADDR_LIST[i].address.dummy);
		ADDR_LIST[i].address.dummy = NULL;
	    }
	    ADDR_LIST[i].addrsize = 0;
	    if (ADDR_LIST[i].hostname) {
		free(ADDR_LIST[i].hostname);
		ADDR_LIST[i].hostname = NULL;		
	    }
	}

	free(ADDR_LIST);
	ADDR_LIST = NULL;
    }
    ADDR_COUNT = 0;

    *addr_list  = ADDR_LIST;
    *addr_count = ADDR_COUNT;
}

void free_se_addr_list(se) 
     struct service_entry * se;
{
    if (SERVICE_ENTRY_magic != se->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_se_addr_list",
	      "Bad magic (service_entry)",0);
     
    l_free_addr_list(& (se->addr_list),
		     & (se->addr_count));   
}

void free_se_port_name_list(se) 
     struct service_entry * se;
{
    if (SERVICE_ENTRY_magic != se->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_se_port_name_list",
	      "Bad magic (service_entry) (*ret)",0);

    if (se->port_name_list) {
	int x;
	
	for (x = 0; x < se->port_name_count; x++) {
	    if (se->port_name_list[x])
		free(se->port_name_list[x]);
	    se->port_name_list[x] = NULL;
	}
	
	free(se->port_name_list);
	se->port_name_list = NULL;
    }
    se->port_name_count = 0;
}
    
void free_se_addr_name_list(se) 
     struct service_entry * se;
{
    if (SERVICE_ENTRY_magic != se->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_se_addr_name_list",
	      "Bad magic (service_entry) (*ret)",0);

    if (se->addr_name_list) {
	int x;

	for (x = 0; x < se->addr_name_count; x++) {
	    if (se->addr_name_list[x])
		free(se->addr_name_list[x]);
	    se->addr_name_list[x] = NULL;
	}
	
	free(se->addr_name_list);
	se->addr_name_list = NULL;
    }
    se->addr_name_count = 0;
}

void free_se_port_list(se) 
     struct service_entry * se;
{
    if (SERVICE_ENTRY_magic != se->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_se_addr_name_list",
	      "Bad magic (service_entry) (*ret)",0);

    if (se->port_list)
	free(se->port_list);
    se->port_list  = NULL;
    se->port_count = 0;
}

static void free_se_option_list P_((struct service_entry * se));
static void free_se_option_list(se) 
     struct service_entry * se;
{
    if (SERVICE_ENTRY_magic != se->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_se_option_list",
	      "Bad magic (service_entry) (*ret)",0);

    if (se->option_list) {
	int x;
	
	for (x = 0; x < se->option_count; x++) {
	    if (SE_option_t_magic != se->option_list[x].type->magic)
		panic("CONNECTION PANIC",__FILE__,__LINE__,
		      "free_se_option_list",
		      "Bad option type magic on service list",0);
	    
	    se->option_list[x].type->
		free_options(& (se->option_list[x]));	    
	    
	    if (se->option_list[x].prefix)
		free(se->option_list[x].prefix);
	    se->option_list[x].prefix = NULL;
	}

	free(se->option_list);
	se->option_list = NULL;
    }
    se->option_count = 0;
}

void free_se_rqtls_pername_list(se) 
     struct service_entry * se;
{
    if (SERVICE_ENTRY_magic != se->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_se_rqtls_pername_list",
	      "Bad magic (service_entry) (*ret)",0);

    if (se->req_tls_peername_list) {
	int x;

	for (x = 0; x < se->req_tls_peername_count; x++)
	    if (se->req_tls_peername_list[x])
		free_string(& (se->req_tls_peername_list[x]));
	free(se->req_tls_peername_list);
	se->req_tls_peername_list = NULL;
    }
    se->req_tls_peername_count = 0;
}

void inc_service_entry_refcount(Y)
      struct service_entry *Y;
 {
     if (SERVICE_ENTRY_magic != Y->magic)
	 panic("CONNECTION PANIC",__FILE__,__LINE__,
	       "inc_service_entry_refcount",
	       "Bad magic (service_entry) (*ret)",0);

     Y->refcount++;
 }

void free_service_entry(Y)
     struct service_entry **Y;
{
    if (!(*Y))
	return;

    if (SERVICE_ENTRY_magic != (*Y)->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_service_entry",
	      "Bad magic (service_entry)",0);
    
    DPRINT(Debug,12,(&Debug, 
		     "free_service_entry: entry %p: %s%s%s\n",
		     (*Y), (*Y)->official_name,  
		     ((*Y)->flags & SE_temporary) ? ", (temporary)" : "",
		     ((*Y)->flags & SE_static)    ? ", (static)"    : ""));

    if ((*Y)->refcount < 1)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_service_entry",
	      "Bad refcount",0);
    
    (*Y)->refcount--;

    DPRINT(Debug,12,(&Debug, 
		     "free_service_entry: entry %p => refcount=%d\n",
		     *Y,(*Y)->refcount));
    
    if ((*Y)->refcount > 0) {

	*Y = NULL;
	return;
    }

    DPRINT(Debug,12,(&Debug, 
		     "free_service_entry: Freeing entry %p\n",
		     *Y));

    if ((*Y)->flags  & SE_temporary) {
	DPRINT(Debug,12,(&Debug, 
			 "free_service_entry: Is temporary\n"));
    }

    if ((*Y)->official_name)
	free((*Y)->official_name);
    (*Y)->official_name = NULL;

    if ((*Y)->ip_literal_address.dummy)
	free((*Y)->ip_literal_address.dummy);
    (*Y)->ip_literal_address.dummy = NULL;
    (*Y)->ip_literal_addrsize = 0;

    free_se_aliases_list(*Y);
    free_se_addr_list(*Y);

    free_se_addr_name_list(*Y);
    free_se_port_name_list(*Y);
    free_se_port_list(*Y);

    (*Y)->service = NULL;

    free_se_option_list(*Y);
    free_se_rqtls_pername_list(*Y);

    (*Y)->magic = 0;  /* Invalidate */

    free(*Y);
    (*Y) = NULL;
}

int is_similar_service_entry(se1,se2)
     struct service_entry * se1;
     struct service_entry * se2;
{
    int ret = 0;

    if (SERVICE_ENTRY_magic != se1->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "is_similar_service_entry",
	      "Bad magic (service_entry) (se1)",0);

    if (SERVICE_ENTRY_magic != se2->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "is_similar_service_entry",
	      "Bad magic (service_entry) (se2)",0);

    if (se1->service != se2->service) {
	DPRINT(Debug,12,(&Debug,
			 "is_similar_service_entry: service differ\n"));
    }

    if (0 != strcmp(se1->official_name,se2->official_name)) {
	DPRINT(Debug,12,(&Debug,
			 "is_similar_service_entry: name differ\n"));
	goto fail;
    }

    if (se1->ip_literal_addrsize != se2->ip_literal_addrsize) {
	DPRINT(Debug,12,(&Debug,
			 "is_similar_service_entry: literal ip address size differ\n"));
	goto fail;
    }

    if (se1->ip_literal_address.dummy && se2->ip_literal_address.dummy &&
	se1->ip_literal_addrsize) {

	if (0 != memcmp(se1->ip_literal_address.dummy,
			se2->ip_literal_address.dummy,
			se1->ip_literal_addrsize)) {

	    DPRINT(Debug,12,(&Debug,
			     "is_similar_service_entry: literal ip addresses differ\n"));
	    goto fail;	    
	}
	
    } else if (se1->ip_literal_address.dummy || se2->ip_literal_address.dummy) {
	DPRINT(Debug,12,(&Debug,
			 "is_similar_service_entry: bad ip address data\n"));
	goto fail;
    }

    if (se1->flags != se2->flags) {
	DPRINT(Debug,12,(&Debug,
			 "is_similar_service_entry: flags differ\n"));
	goto fail;	

    } else if (se1->flags) {
	
	/* Compare only flagged data, this does not compare
	   generated (hidden) data
	*/
	
	if (0 != (se1->flags & SE_given_alias)) {
	    int y;
	    
	    if (se1->aliases_count != se2->aliases_count) {
		DPRINT(Debug,12,(&Debug,
				 "is_similar_service_entry: aliases differ\n"));
		goto fail;	
	    }
	    
	    for (y = 0; y < se1->aliases_count; y++) {
		if (! se1->aliases_list[y].aliasname &&
		    ! se2->aliases_list[y].aliasname)
		    continue;

		if (! se1->aliases_list[y].aliasname ||
		    ! se2->aliases_list[y].aliasname ||
		    0 != strcmp(se1->aliases_list[y].aliasname,
				se2->aliases_list[y].aliasname)) {
		    DPRINT(Debug,12,(&Debug,
				     "is_similar_service_entry: aliases differ\n"));
		    goto fail;
		}
	    }
	    
	    DPRINT(Debug,13,(&Debug,
			     "is_similar_service_entry: aliases (%d) are same\n",
			     y));

	} 
	
	if (0 != (se1->flags & SE_given_name_addr)) {
	    int y;
	    
	    if (se1->addr_name_count != se2->addr_name_count) {
		DPRINT(Debug,12,(&Debug,
				 "is_similar_service_entry: named addr differ\n"));
		goto fail;
	    }
	    
	    for (y = 0; y < se1->addr_name_count; y++) {
		if (0 != strcmp(se1->addr_name_list[y],
				se2->addr_name_list[y])) {
		    DPRINT(Debug,12,(&Debug,
				     "is_similar_service_entry: named addr differ\n"));
		    goto fail;
		}
	    }
	    
	    DPRINT(Debug,13,(&Debug,
			     "is_similar_service_entry: named addrs (%d) are same\n",
			     y));
	    
	} 
	
	if (0 != (se1->flags & SE_given_name_port)) {
	    int y;
	    
	    if (se1->port_name_count != se2->port_name_count) {
		DPRINT(Debug,12,(&Debug,
				 "is_similar_service_entry: named port differ\n"));
		goto fail;
	    }
	    
	    for (y = 0; y < se1->port_name_count; y++) {
		if (0 != strcmp(se1->port_name_list[y],
				se2->port_name_list[y])) {
		    DPRINT(Debug,12,(&Debug,
				     "is_similar_service_entry: named port differ\n"));
		    goto fail;
		}
	}
	    
	DPRINT(Debug,13,(&Debug,
			 "is_similar_service_entry: named ports (%d) are same\n",
			 y));
	
	} 
	
	if (0 != (se1->flags & SE_given_addr)) {
	    int y;
	    
	    if (se1->addr_count != se2->addr_count) {
		DPRINT(Debug,12,(&Debug,
				 "is_similar_service_entry: addr differ\n"));
		goto fail;
	    }
	    
	    for (y = 0; y < se1->addr_count; y++) {
		
		if (! is_same_se_address(& (se1->addr_list[y]),
					 & (se1->addr_list[y]))) {
		    DPRINT(Debug,12,(&Debug,
				     "is_similar_service_entry: addr differ\n"));
		    goto fail;
		}
	    }
	    
	    DPRINT(Debug,13,(&Debug,
			     "is_similar_service_entry: addrs (%d) are same\n",
			     y));
	    
	} 

	if (0 != (se1->flags & SE_given_port)) {
	    int y;
	    
	    if (se1->port_count != se2->port_count) {
		DPRINT(Debug,12,(&Debug,
				 "is_similar_service_entry: port differ\n"));
		goto fail;
	    }
	    
	    for (y = 0; y < se1->port_count; y++) {
		if (se1->port_list[y] != se2->port_list[y]) {
		    DPRINT(Debug,12,(&Debug,
				     "is_similar_service_entry: port differ\n"));
		    goto fail;
		}
	    }	
	    
	    DPRINT(Debug,13,(&Debug,
			     "is_similar_service_entry: ports (%d) are same\n",
			     y));
	    
	} 
    }	

    if (se1->req_tls_peername_count != se2->req_tls_peername_count) {
	DPRINT(Debug,12,(&Debug,
			 "is_similar_service_entry: req tls peername differ\n"));
	goto fail;
    } else if (se1->req_tls_peername_count) {
	int y;

	for (y = 0; y < se1->req_tls_peername_count; y++) {
	    if (0 != string_cmp(se1->req_tls_peername_list[y],
				se2->req_tls_peername_list[y],
				-99 /* Unknown value*/)) {
		DPRINT(Debug,12,(&Debug,
				 "is_similar_service_entry: req tls peername differ\n"));
		goto fail;
	    } 
	}

	DPRINT(Debug,13,(&Debug,
			 "is_similar_service_entry: req tls peernames (%d) are same\n",
			 y));
	
    }
    
    if (se1->option_count != se2->option_count) {
	DPRINT(Debug,12,(&Debug,
			 "is_similar_service_entry: options differ\n"));
	goto fail;
    } else if (se1->option_count) {
	int idx;

	for (idx = 0; idx < se1->option_count; idx++) {
	    char * val1, * val2;;

	    if (SE_option_t_magic != se1->option_list[idx].type->magic)
		panic("CONNECTION PANIC",__FILE__,__LINE__,
		      "is_similar_service_entry",
		      "Bad option type magic on service list",0);

	    if (se1->option_list[idx].type !=
		se2->option_list[idx].type) {
		DPRINT(Debug,12,(&Debug,
				 "is_similar_service_entry: options differ\n"));
		goto fail;
	    }

	    val1 = se1->option_list[idx].type->
		give_values(& (se1->option_list[idx]),
			    se1->option_list[idx].prefix);
	    val2 = se2->option_list[idx].type->
		give_values(& (se2->option_list[idx]),
			    se2->option_list[idx].prefix);

	    if (val1 && val2) {
		if (0 != strcmp(val1,val2)) {

		    free(val1);
		    free(val2);

		    DPRINT(Debug,12,(&Debug,
				     "is_similar_service_entry: options differ\n"));
		    goto fail;

		}
	    } else if (val1 || val2) {

		if (val1)
		    free(val1);
		if (val2)
		    free(val2);
		
		DPRINT(Debug,12,(&Debug,
				 "is_similar_service_entry: options differ\n"));
		goto fail;
	    }

	    if (val1)
		free(val1);
	    if (val2)
		free(val2);
	    
	}

	DPRINT(Debug,13,(&Debug,
			 "is_similar_service_entry: options (%d extensions) are same\n",
			 idx));
	
    }

    ret = 1;

 fail:
#ifdef DEBUG
    DPRINT(Debug,12,(&Debug,"is_similar_service_entry=%d",
		     ret));
    
    if (Debug.active >= 13) {
	FILE *F;
		
	DPRINT(Debug,13,(&Debug,"\n  -- se1=%p : ",se1));

	F = debug_to_FILE(&Debug);
	dump_service_entry(F,se1,system_charset);
	fclose(F);

	DPRINT(Debug,13,(&Debug,"\n  -- se2=%p : ",se2));

	F = debug_to_FILE(&Debug);
	dump_service_entry(F,se2,system_charset);
	fclose(F);
    }
    DPRINT(Debug,12,(&Debug,"\n"));
#endif

    return ret;
}

static struct string *convert_se_value P_((const char *filename,
					   int lineno,
					   int *errors,
					   const char *opt,
					   charset_t cs,
					   const char *val,
					   int vallen));
static struct string *convert_se_value(filename,lineno,errors,opt,cs,val,vallen)
     const char *filename;
     int lineno;
     int *errors;
     const char *opt;
     charset_t cs;
     const char *val;
     int vallen;
{
    struct string * buffer = new_string(cs);
    const char * csn  = get_charset_MIME_name(cs);

    int ERRORS;
    int rs = add_streambytes_to_string(buffer,vallen,cs2us(val),&ERRORS);

    if (rs < vallen) {
	lib_error(CATGETS(elm_msg_cat, MeSet, MeOptionValueCharset,
			  "%s: %d: Failed to parse option \"%s\" value as charset %s\n"),
		  filename, lineno, opt, csn ? csn :  "<no MIME name>");
	(*errors) ++;

	free_string(&buffer);
	goto fail;
    }

    if (ERRORS) {
	lib_error(CATGETS(elm_msg_cat, MeSet, MeOptionValueCSErrors,
			  "%s: %d: There is %d errors when parsing option \"%s\" value as charset %s"),	  
		  filename, lineno,ERRORS,opt, csn ? csn :  "<no MIME name>");

	(*errors) ++;
    }

 fail:
    return buffer;
}

static int is_ascii_value P_((const char *filename,
			      int lineno,
			      int *errors,
			      const char *opt,
			      const char *val));
static int is_ascii_value(filename,lineno,errors,opt,val)
     const char *filename;
     int lineno;
     int *errors;
     const char *opt;
     const char *val;
{

    const unsigned char *x;

    for (x = cs2us(val); *x; x++) {
	 if (!isascii(*x)) {
	     if (opt)
		 lib_error(CATGETS(elm_msg_cat, MeSet, MeAsciiOptionValue,
				   "%s: %d: Only ASCII characters (not 0x%02x) are allowed on  option \"%s\" value"),	  
			   filename, lineno, *x,opt);
	     else
		 lib_error(CATGETS(elm_msg_cat, MeSet, MeAsciiFieldValue,
				   "%s: %d: Only ASCII characters (not 0x%02x) are allowed on field"),
			   filename, lineno, *x);

	     (*errors) ++;
	     return 0;
	 }

    }

    return 1;
}

static int rc_value_to_syscs P_((const char *filename,
				 int lineno,
				 int *errors,
				 const char *opt,
				 const struct string * buffer,
				 char **result));
static int rc_value_to_syscs(filename,lineno,errors,opt,buffer,result)
     const char *filename;
     int lineno;
     int *errors;
     const char *opt;
     const struct string * buffer;
     char **result;
{
    const char *csn = get_string_MIME_name(buffer);
    const char * rcsetnam = get_charset_MIME_name(system_charset);
    int ret = 0;

    int ERRORS;
    struct string * res_S = convert_string2(system_charset,buffer,&ERRORS);

    if (*result) {
	free(*result);
	*result = NULL;
    }

    if (ERRORS) {
	lib_error(CATGETS(elm_msg_cat, MeSet, MeConvOptionValErrors,
			  "%s: %d: There is %d errors when converting option \"%s\" value  from charset %s to charset %s"),
		  filename,lineno,ERRORS,opt,
		  csn ? csn : "<no MIME name>",
		  rcsetnam ? rcsetnam : "<no MIME name>");

	(*errors) ++;
    }

     /* result is malloced */
    bytestream_from_string(res_S,result,&ret);

    return ret;
}
				 
void free_mail_services_conf(conf)
     struct mail_services_conf **conf;
{
    if (MAIL_SERVICES_magic != (*conf)->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_mail_services_conf",
	      "Bad magic ( mail_services_conf)",0);
 
    if ((*conf)->service_list) {
	int i; 

	for (i = 0; i < (*conf)->service_count; i++) {
	    free_service_entry(& ((*conf)->service_list[i]));
	}

	free((*conf)->service_list);
	(*conf)->service_list = NULL;
    }
    (*conf)->service_count = 0;

    (*conf)->magic = 0;   /* Invalidate */

    free(*conf);
    *conf = NULL;
}

struct mail_services_conf * malloc_mail_services_conf()
{
    struct mail_services_conf *ret =  safe_zero_alloc(sizeof (*ret));

    ret->service_count  = 0;
    ret->service_list   = NULL;
    ret->magic          = MAIL_SERVICES_magic;

    return ret;
}

#ifdef I_NET_IF
#include <net/if.h>
#endif

/* Retuns 1 if IP address can be parsed */

#if defined(I_NETINET_IN) && defined(I_ARPA_INET)
int get_ip(result,val)
     SOCKADDR * result; 
     const char * val;
{
#ifdef USE_INET_XTOX_PN
    char * string_temp;
    int ret;
    
    const char * body;
    const char * tail UNUSED_VAROK;
    unsigned scope_as_if_index  UNUSED_VAROK = 0;
    const char * pos;
    uint32 scope;    /* uint32 is defined on hdrs/elm_defs.h */

    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)result, sizeof (*result));
	
    /* inet_pton() and inet_ntop() are available */

    /* 1: parsing succeed
       0: bad string
      -1: errno set (address family not supported)
    */

    if (inet_pton(AF_INET,val,
		  &(result->sin.sin_addr)) > 0) {

	result->sin.sin_family = AF_INET;

	DPRINT(Debug,12,(&Debug,  
			 "get_ip=1: Parsed %s as IPv4 address (inet_pton).\n",
			 val));

	return 1;
    } else {
	DPRINT(Debug,12,(&Debug,  
			 "get_ip: Failed parse %s as IPv4 address (inet_pton)\n",
			 val));

    }

#ifdef HAVE_IN6
    string_temp = NULL;
    body = val;
    tail = NULL;
    scope = 0;
    ret = 0;

    /* if_nametoindex may do mapping from name to index */

    pos = strchr(val,'%');   /* Have scope id */

    if (pos) {
#ifdef HAVE_SCOPE

	char * end;
	long l = strtol(pos+1,&end,10);
	int len;

	scope = l;
	if ('\0' != *end || scope != l) {

#ifdef HAVE_NAMEINDEX
	    if (dt_flag_is_set(&ipv6_scope_id,
			       scope_link_local_if_index_flag)) {
		
		/* 0 == not found */
		scope_as_if_index = if_nametoindex(pos+1);

		if (scope_as_if_index > 0) {
		    DPRINT(Debug,12,(&Debug,  
				     "get_ip: Parsed %s as possible interface name (index %u) on IP-address %s\n",
				     pos+1,scope_as_if_index,val));
		    goto try_scope;
		}
	    }
#endif /* HAVE_NAMEINDEX */

	    DPRINT(Debug,12,(&Debug,  
			     "get_ip: Unsuported scope %s on IP-address %s\n",
			     pos+1,val));
	    goto fail_scope;
	}

	DPRINT(Debug,12,(&Debug,  
			 "get_ip: Parsed %s as scope %u from %s\n",
			 pos+1,(unsigned)scope,val));

    try_scope:

	len = pos - val;
	string_temp = safe_malloc(len+1);
	memcpy(string_temp,val,len);
	string_temp[len] = '\0';
	body = string_temp;

	DPRINT(Debug,12,(&Debug,  
			 "get_ip: address is %s (len %d)\n",
			 body,len));

#else /* HAVE_SCOPE */
	DPRINT(Debug,12,(&Debug,  
			 "get_ip: Unsuported scope %s on IP-address %s\n",
			 pos+1,val));
	goto fail_scope;
#endif /* not HAVE_SCOPE */
    }

    if (inet_pton(AF_INET6,body,
		  &(result->sin6.sin6_addr)) > 0) {

	result->sin6.sin6_family   = AF_INET6;

	if (scope_as_if_index) {
	    DPRINT(Debug,12,(&Debug,  
			     "get_ip: Parsed %s as IPv6 address (inet_pton) from %s with possible scope id %u\n",
			     body,val,
			     scope_as_if_index));

	    if (IN6_IS_ADDR_LINKLOCAL(&(result->sin6.sin6_addr))) {
		DPRINT(Debug,12,(&Debug,  
				 "get_ip: is link local address, scope %u OK\n",
				 scope_as_if_index));
		scope = scope_as_if_index;
	    } else {
		DPRINT(Debug,12,(&Debug,  
				 "get_ip: is not link local address, not valid scope\n"));
		goto fail_scope;
	    }
	}

#ifdef HAVE_SCOPE
	result->sin6.sin6_scope_id = scope;
#endif /* HAVE_SCOPE */

	DPRINT(Debug,12,(&Debug,  
			 "get_ip: Parsed %s as IPv6 address (inet_pton) from %s\n",
			 body,val));



	ret = 1;
    } else {
	DPRINT(Debug,12,(&Debug,  
			 "get_ip: Failed parse %s as IPv6 address (inet_pton)\n",
			 body));
    }

 fail_scope:
    if (string_temp)
	free(string_temp);

    if (ret) {
	DPRINT(Debug,12,(&Debug,  
			 "get_ip=%d: Parsed %s as IPv6 address (inet_pton).\n",
			 ret,val));

	return ret;
    }
    
#endif /* HAVE_IN6 */

#else  /* USE_INET_XTOX_PN */
    
    /* inet_aton() is better but not available on
       some systems so we need use inet_addr()
    */

#ifndef USE_INET_ATON
    long tmp;
#endif /* USE_INET_ATON */

    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)result, sizeof (*result));
   
    if (
#ifdef USE_INET_ATON
	inet_aton(val,&(result->sin.sin_addr))
#else /* USE_INET_ATON */
	-1 != (tmp = inet_addr(val)) 
#endif /* not USE_INET_ATON */
	) {

#ifndef USE_INET_ATON
	memcpy(&(result->sin.sin_addr.s_addr),
	       &tmp,
	       sizeof (result->sin.sin_addr.s_addr));
#endif /* USE_INET_ATON */

	result->sin.sin_family = AF_INET;

	DPRINT(Debug,12,(&Debug,  
			 "get_ip=1: Parsed %s as IPv4 address.\n",
			 val));


	return 1;
    } 
#endif /* USE_INET_XTOX_PN */

    DPRINT(Debug,12,(&Debug,  
		     "get_ip=0; Failed to parse %s as address.\n",
		     val));
    return 0;
}
#endif

const struct service_type * service_type_from_name(ascii_name)
     const char *ascii_name;
{
    const struct service_type *st;

    for (st = &SERVICE_TYPES[0]; st->name; st++) {
	if (0 == istrcmp(ascii_name,st->name)) {

	    DPRINT(Debug,10,(&Debug, "service_type_from_name(\"%s\")=%p; name=\"%s\"\n",
			     ascii_name,
			     st,
			     st->name));
	    
	    return st;
	}
    }
    
    DPRINT(Debug,10,(&Debug, "service_type_from_name(\"%s\")=NULL\n",
		     ascii_name));


    return NULL;
}



struct mail_services_conf * parse_service_entries (filename,errors, 
						   fileset, propline)
     const char *filename; 
     int *errors;
     charset_t *fileset;
     struct editor_propline **propline;
{
    struct mail_services_conf *ret = NULL;

    /*   hostname   service        options               */
    int max_count = 0;
    FILE * f;
    char * buf = NULL;
    int c;
    charset_t cs = system_charset;
    int lineno = 0;
    int last_c = 0;
    int err = can_open(filename,"r");

    if (err) {
	DPRINT(Debug,2,(&Debug, 
			"parse_service_entries=NULL: %s: %s (can_open)\n",
			filename,strerror(err)));
	return NULL;
    }

    f = fopen(filename,"r");
    if (!f) {
	int err UNUSED_VAROK = errno;
	DPRINT(Debug,2,(&Debug, 
			"parse_service_entries=NULL: %s: %s\n",
			filename,strerror(err)));
	return NULL;
    }

    if (fileset && *fileset) {
	const char * cs_name = get_charset_MIME_name(*fileset);
	
	cs = *fileset;

	if (cs_name) {
	    DPRINT(Debug,2,(&Debug, 
			    "parse_service_entries: %s: default charset: %s\n",
			    filename,cs_name));
	}
    }

    /* Look for editor property line */

    if (propline)  {
	enum editor_propline_v propline_mode = 
	    give_dt_enumerate_as_int(&editor_ms_propline);

	/* Nothing detected yet */

	if (*propline)
	    free_editor_propline(propline);

	if (propline_mode != editor_propline_ignore) {
	    int i;
	    
	    for (i = 0; i < 3 && !*propline; i++) {
		int l1;

		c = fgetc(f);

		if (EOF == c)
		    goto propline_failure;
		if ('\n' == c)
		    continue;

		ungetc(c,f);

		if ('#' != c)
		    break;

		l1 = malloc_gets(&buf,LONG_STRING,f);
		if (l1 > 1) {

		    if ('#' != buf[0])
			goto propline_failure;
		    
		    *propline =
			detect_editor_propline(propline_mode,
					       buf,l1,filename,i,&cs);
					       
		} else { /* something wrong? */
		propline_failure:

		    rewind(f);
		    break;
		}
	    }
	}	
    }

    while(EOF != (c = fgetc(f))) {
	if ('\n' == c)
	    max_count++;
	last_c = c;
    }

    if (last_c && '\n' != last_c) {
	DPRINT(Debug,9,(&Debug, 
			"parse_service_entries: %s, no newline on end of file\n",
			filename));
	max_count++;	
    }

    DPRINT(Debug,9,(&Debug, 
		    "parse_service_entries: %s, max_count=%d\n",
		    filename,max_count));

    ret = malloc_mail_services_conf();
    
    if (!max_count) {
	fclose(f);
	return ret;
    }
    rewind(f);

    ret->service_list =
	safe_calloc(max_count,sizeof (ret->service_list[0]));

    while (ret->service_count < max_count) {
	
	int l1 = malloc_gets(&buf,LONG_STRING,f);

	char *c,*c1, *options;
	const struct service_type *st = NULL;
	struct service_entry * se = NULL;

	SOCKADDR ip_literal;
	/* Make comparision easier */
	bzero((void *)&ip_literal, sizeof ip_literal);

	if (-1 == l1) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeTooLongLineNo,
			      "%s: %d: Too long line: %.30s..."),
		      filename,lineno+1,buf);
	    (*errors) ++;
	    break;
	} else if (l1 < 0) {
	    DPRINT(Debug,9,(&Debug, 
			    "parse_service_entries: %s: %d: read error or EOF\n",
			    filename,lineno+1));
	    break;
	}

	lineno++;

	if (l1 == 0)
	    continue;

	l1 = trim_whitespace_from_end(buf,l1);
	
	if (read_charset_tag(buf,l1,filename,lineno,errors,&cs,NULL))
	    continue;
	    
	c = buf;
	while (*c && whitespace (*c)) /* skip leading whitespace */
	    c++;
	if ('#' == *c)
	    continue;
	if (!*c)
	    continue;
	
	c1 = strpbrk(c," \t");
	if (!c1) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
			      "%s: %d: Bad line: %.30s..."),
		      filename,lineno,buf);
	    (*errors) ++;
	    continue;
	}
	*c1 = '\0';
	c1++;

	if (!is_ascii_value(filename,lineno,errors,NULL,c))
	    continue; /* Non-ascii hostname */

	while (*c1 && whitespace (*c1)) /* skip leading whitespace */
	    c1++;
	if (!*c1) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
			      "%s: %d: Bad line: %.30s..."),
		      filename,lineno,buf);
	    (*errors) ++;
	    continue;
	}
	
	options = strpbrk(c1," \t;");
	if (options) {
	    *options = '\0';
	    options++;
	}

	if (!is_ascii_value(filename,lineno,errors,NULL,c1))
	    continue; /* Non-ascii keyword */

	st = service_type_from_name(c1);

	if (!st) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeServiceTypeNo,
			      "%s: %d: Bad service type %s"),
		      filename,lineno,c1);
	    (*errors) ++;
	    continue;
	}

	se =  malloc_service_entry(c  /* hostname */,
				   st /* service */,
				   0  /* This is NOT temporary entry */);		

#if defined(I_NETINET_IN) && defined(I_ARPA_INET)			
	if (get_ip(&ip_literal,c)) {

	    char mybuffer[256];
	    const char * ip_literal_string = 
		give_addr_as_string(&ip_literal,mybuffer,sizeof mybuffer);

	    struct service_entry_addr SE;

	    bzero((void *)&SE, sizeof SE);


	    set_address_from_gen_helper(& (se->ip_literal_addrsize),
					& (se->ip_literal_address),
					&ip_literal);
	    
	    DPRINT(Debug,9,(&Debug, 
			    "parse_service_entries: %s: %d: %s: is IP-literal %s\n",
			    filename, lineno, c, ip_literal_string ? ip_literal_string : "???"));

	    set_address_from_gen_helper(& (SE.addrsize),
					& (SE.address),
					&ip_literal);
	    
	    SE.hostname = safe_strdup(ip_literal_string ? ip_literal_string : c);
	    
	    /* Permanent  --
	       May fail at Tue Jan 19 03:14:07 2038 UTC
				       */
	    set_MAX_schedule_timelimit(& (SE.valid_until));
	    
	    add_se_addr_to_entry(se,&SE,1 /* malloc copy */);
	    
	    free(SE.address.dummy);
	    free(SE.hostname);	    	    
	} 
#endif /* defined(I_NETINET_IN) && defined(I_ARPA_INET) */

	if (options) {
	    char *opt;
	    char * WALK = NULL;
	    
	    for (opt = mime_parse_content_opts(options, &WALK); 
		 opt; 
		 opt = mime_parse_content_opts(NULL, &WALK)) {
		char * q = strchr(opt,'=');
#ifdef USE_DLOPEN
		char * zopt = NULL;
#endif /* USE_DLOPEN */
		char * val = NULL;
		
		DPRINT(Debug,65,(&Debug, 
				 "mime_parse_content_opts gives: %s\n",
				 opt));
		
		if (q) 
		    *q++ = '\0';		    

		if (!is_ascii_value(filename,lineno,errors,NULL,opt))
		    continue; /* Non-ascii keyword */


		if (0 == strcmp(opt,"alias")) {
		    char * conv = NULL;
		    int convlen = 0;
		    if (!q || !q[0]) {
			lib_error(CATGETS(elm_msg_cat, MeSet, MeOptionReqValueNo,
					  "%s: %d: Option %s requires value"),
				  filename,lineno,opt);
			(*errors) ++;
			continue;
		    }

		    if (cs != system_charset) {
			struct string * buffer = convert_se_value(filename,lineno,errors,
								  opt,cs,q,strlen(q));
			if (!buffer)
			    continue;

			convlen =  rc_value_to_syscs(filename,lineno,errors,
						     opt,buffer,&conv);
			if (!conv)
			    continue;

		    } else {
			conv = q;
			convlen = strlen(q);
		    }

		    val = dequote_opt(conv,convlen);		
		    add_alias_to_entry(se,
				       val,
				       /* Permanent  --
					  May fail at Tue Jan 19 03:14:07 2038
				       */
				       give_MAX_schedule_timelimit());
		    free(val);  /* add_alias_to_entry strdup's value */
		    
		    if (conv != q)
			free(conv);
		    se->flags |= SE_given_alias;

		} else if (0 == strcmp(opt,"addr")) {
		    char *q1 = NULL;
		    if (!q || !q[0]) {
			lib_error(CATGETS(elm_msg_cat, MeSet, MeOptionReqValueNo,
					  "%s: %d: Option %s requires value"),
				  filename,lineno,opt);
			(*errors) ++;
			continue;
		    }

		    if (! is_ascii_value(filename,lineno,errors,opt,q))
			continue;

		    if (se->ip_literal_address.dummy ||
			se->ip_literal_addrsize) {

			lib_error(CATGETS(elm_msg_cat, MeSet, MeOptionNotAllowedIsIP,
					  "%s: %d: Option %s not allowed. Name %s is IP-address"),
				  filename,lineno,opt,se->official_name);

			(*errors) ++;
			continue;		       					  
		    }

		    if ('"' != q[0])
			q1 = strchr(q,':');

		    if (!q1) { /* name given */
			add_addr_name_to_entry(se,
					       /* Malloces result */
					       dequote_opt(q,strlen(q)),
					       0);
			se->flags |= 
			    SE_given_name_addr;
		    } else {  /* Addr? given */
			SOCKADDR result;
			char * port, *tail = "";
			char tail1[2];
			
			tail1[0] = '\0';

			/* Make comparision easier */
			bzero((void *)&result, sizeof result);

			*q1++ = '\0';
			port = qstrpbrk(q1,":/");
			if (port) {
			    tail1[0] = *port;
			    tail1[1] = '\0';
			    *port++ = '\0';
			    tail = port;
			}
			val = dequote_opt(q1,strlen(q1));		


#if defined(I_NETINET_IN) && defined(I_ARPA_INET)			

			/* Note: get_ip zeroes result first */

			if (0 == strcmp(q,"ip") &&
			    get_ip(&result,val)) {
			    int p = PORT_end;   /* No port given */

			    if (port) {
				char *p2;
				long l;
				
				p = (l = strtol(port,&p2,10));
				if (*p2 != '\0' || p != l) {
				    DPRINT(Debug,10,(&Debug,  "... bad port: %s\n",port));
				    goto addr_bad_port;
				}
			    }
			    
			    set_addr_port(&result,p);
			} else
#endif /* defined(I_NETINET_IN) && defined(I_ARPA_INET) */
			    {
			    addr_bad_port:

				lib_error(CATGETS(elm_msg_cat, MeSet,
						  MeBadAddrSpecNo,
						  "%s: %d: Bad address specification %s:%s%s%s"),
					  filename,lineno,q,q1,tail1,tail);	
				(*errors) ++;

				DPRINT(Debug,10,(&Debug,  "... dequoted address: %s\n",val));

				free(val); val= NULL;
				continue;			    
			    }	

			add_addr_to_entry(se,&result,
					  /* Permanent  --
					     May fail at Tue Jan 19 03:14:07 2038
					  */
					  give_MAX_schedule_timelimit(),
					  NULL);
			free(val); val = NULL;
			se->flags |= 
			    SE_given_addr;
		    }
		} else if (0 == strcmp(opt,"port")) {

		    if (!q || !q[0]) {
			lib_error(CATGETS(elm_msg_cat, MeSet, MeOptionReqValueNo,
					  "%s: %d: Option %s requires value"),
				  filename,lineno,opt);
			(*errors) ++;
			continue;
		    }

		    if (! is_ascii_value(filename,lineno,errors,opt,q))
			continue;

		    if (!isdigit(*q)) {

			add_port_name_to_entry(se,
					       /* Malloces result */
					       dequote_opt(q,strlen(q)),
					       0);
			se->flags |= 
			    SE_given_name_port;

		    } else {

			int port;
			char *p2 = NULL;
			long l;


			port = (l = strtol(q,&p2,10));
			if (*p2 != '\0' || port != l) {
			    lib_error(CATGETS(elm_msg_cat, MeSet, MeOptionBad,
					      "%s: %d: Option %s=%s unsupported"),
				      filename,lineno,opt,q);
			    (*errors) ++;
			    continue;
			}
			
			if (port == PORT_end) {
			    lib_error(CATGETS(elm_msg_cat, MeSet, MeOptionPortBadNo,
					      "%s: %d: Option port=%d unsupported"),
				      filename,lineno,port);
			    (*errors) ++;
			    continue;
			}

			add_port_to_entry(se,
					  port);

			se->flags |= SE_given_port;
		    }

		}  else if (0 == strcmp(opt,"use-tls-checks")) {

		    if (q) {
			
			if (! is_ascii_value(filename,lineno,errors,opt,q))
			    continue;

			if (0 == istrcmp(q,"no") || 0 == istrcmp(q,"off")) {
			    se->flags |= SE_nouse_tls_checks;
			} else {
			    lib_error(CATGETS(elm_msg_cat, MeSet, MeOptionBad,
					      "%s: %d: Option %s=%s unsupported"),
				      filename,lineno,opt,q);
			    (*errors) ++;
			}   
		    } /* use-tls-checks without parameter is ignored */
					    
		}  else if (0 == strcmp(opt,"verify-tls-certificate")) {

			if (q) {

			    if (! is_ascii_value(filename,lineno,errors,opt,q))
				continue;

			    if (0 == istrcmp(q,"on") || 0 == istrcmp(q,"yes")) {
				if (!(se->flags & 
				      SE_nover_tls_cert))
				    se->flags |= SE_ver_tls_cert;
				else {
				    lib_error(CATGETS(elm_msg_cat, MeSet,MeConfOpt2,
						  "%s: %d: Conflicting option %s=%s"),
					      filename,lineno,opt,q);
				    (*errors) ++;
				}
			    } else  if (0 == istrcmp(q,"no") || 0 == istrcmp(q,"off")) {

				if (!(se->flags & 
				      SE_ver_tls_cert))
				    se->flags 
					|= SE_nover_tls_cert;
				else {
				    lib_error(CATGETS(elm_msg_cat, MeSet,MeConfOpt2,
						  "%s: %d: Conflicting option %s=%s"),
					      filename,lineno,opt,q);
				    (*errors) ++;
				}
			    } else {
				lib_error(CATGETS(elm_msg_cat, MeSet, MeOptionBad,
					  "%s: %d: Option %s=%s unsupported"),
					  filename,lineno,opt,q);
				(*errors) ++;
			    }

			} else {
			    if (!(se->flags & 
				  SE_nover_tls_cert))
				se->flags |= SE_ver_tls_cert;
			    else {
				lib_error(CATGETS(elm_msg_cat, MeSet,MeConfOpt,
						  "%s: %d: Conflicting option %s"),
					  filename,lineno,opt);
				(*errors) ++;
			    }
			}

		}  else if (0 == strcmp(opt,"require-tls-peer-name")) {

		    if (q) {
			char * val0 = dequote_opt(q,strlen(q));
			struct string * val =
			    convert_se_value(filename,lineno,errors,
					     opt,cs,val0,strlen(val0));

			free(val0); val0 = NULL;
			if (!val) 
			    continue;

			add_req_peername_to_entry(se,val);

			free_string(&val);

		    } else
			se->flags |= SE_require_tls_name;

#ifdef USE_DLOPEN
		} else if (NULL != (zopt = strchr(opt,':'))) {
		    int idx = -1;
		    int x;

		    char * conv = NULL;
		    int convlen UNUSED_VAROK = 0;

		    *zopt++ = '\0';

		    for (x = 0; x < se->option_count; x++) {
			if (0 == strcmp(opt,se->
					option_list[x].prefix))
			    idx = x;
		    }
		    if (-1 == idx) {
			struct SE_option_type * Y = get_option_type(opt);
			if (!Y)
			    continue;

			idx = 
			    add_option_type_to_entry(se,
						     Y,opt);
		    }

		    if (SE_option_t_magic != 
			se->option_list[idx].type->magic)
			panic("CONNECTION PANIC",__FILE__,__LINE__,
			      "parse_service_entries",
			      "Bad option type on service list",0);

		    if (cs != system_charset && q) {
			struct string * buffer = convert_se_value(filename,lineno,errors,
								  opt,cs,q,strlen(q));
			if (!buffer)
			    continue;
			
			convlen =  rc_value_to_syscs(filename,lineno,errors,
						     opt,buffer,&conv);
			if (!conv)
			    continue;
		    } else {
			conv = q;
			convlen = strlen(q);
		    }

		    if (!se->option_list[idx].type ->
			parse_on_option(& (se->
					   option_list[idx]),
					zopt,conv)) {
			(*errors) ++;

			if (conv && conv != q)
			    free(conv);

			continue;
		    }					   
		    
		    if (conv && conv != q)
			free(conv);
		    
#endif /* USE_DLOPEN */
		} else {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeOptionUnspportedNo,
				      "%s: %d: Option %s unsupported"),
			      filename,lineno,opt);
		    (*errors) ++;
		    continue;
		}
	    }

	}

	if (0 != (st->flags & STFLAG_require_port) &&
	    0 == (se->flags & SE_given_name_port) &&
	    0 == (se->flags & SE_given_port)) {
	    
	    int have_port = 0;

	    if (0 != (se->flags & SE_given_addr)) {	       
		have_port = addrs_have_port(se);
	    }

	    if (!have_port) {
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeServiceTypeReqPort,
				  "%s: %d: Service type %s requires port= or addr= with port definition"),
			  filename,lineno, st->name);
		(*errors) ++;
		goto free_it;
	    }
	}
	
	ret->service_list[ret->service_count] = se;
	inc_service_entry_refcount(ret->service_list[ret->service_count++]);

    free_it:
	free_service_entry(&se);
    }

    if (fileset)
	*fileset = cs;

    DPRINT(Debug,9,(&Debug, 		    
		    "parse_service_entries: %s, %d parsed\n",
		    filename,ret->service_count));
    fclose(f);

    if (buf)
	free(buf);

    return ret;
}

#if defined(I_NETINET_IN) && defined(I_ARPA_INET)

int give_addr_port(sa)
     SOCKADDR *sa;
{
    int ret = -1;

    /* Need know sa_family, so that port can be accessed! */
    switch (sa->sa.sa_family) {
    case AF_INET:
	ret = ntohs(sa->sin.sin_port);
	break;
#ifdef HAVE_IN6
    case AF_INET6: 
	ret = ntohs(sa->sin6.sin6_port);
	break;
#endif
    default:
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "give_addr_port",
	      "Unsupported address family",0);
    }	

    DPRINT(Debug,12,(&Debug,"give_addr_port=%d\n",ret));	
    return ret;
}

void set_addr_port(sa,port)
     SOCKADDR *sa;
     int port;
{
    /* Need know sa_family, so that port can be accessed! */
    switch (sa->sa.sa_family) {
    case AF_INET:
	sa->sin.sin_port = htons(port);
	break;
#ifdef HAVE_IN6
    case AF_INET6:
	sa->sin6.sin6_port =  htons(port);
	break;
#endif /* HAVE_IN6 */

    default:
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "set_addr_port",
	      "Unsupported address family",0);
    }		
}

#endif /* defined(I_NETINET_IN) && defined(I_ARPA_INET) */

const char * give_SOCKADDR_ptr_as_string(addr,addrsize,buffer,buffer_size)
     const union SOCKADDR_ptr addr;
     const size_t             addrsize;
     char                   * buffer;
     size_t                   buffer_size;
{
    const char * s = NULL;
    int unsupported UNUSED_VAROK = 0;
    int scope_done UNUSED_VAROK = 0;

    if (addrsize < sizeof (addr.sa->sa_family)) {
	DPRINT(Debug,12,(&Debug,
			 "give_SOCKADDR_ptr_as_string: addr size = %zu < sa_family size %zu\n",
			 addrsize,sizeof (addr.sa->sa_family)));
	goto fail;
	
    } else {

	/* Need know sa_family, so that address can be accessed! */
	switch (addr.sa->sa_family) {

#if defined(I_NETINET_IN) && defined(I_ARPA_INET)

	case AF_INET:
	    
	    if (addrsize < sizeof (* (addr.sin))) {
		DPRINT(Debug,12,(&Debug,
				 "give_SOCKADDR_ptr_as_string: addr size = %zu < sockaddr_sin size %zu\n",
				 addrsize,sizeof (* (addr.sin))));
		goto fail;
	    }

#ifdef USE_INET_XTOX_PN
	    s = inet_ntop(AF_INET,
			  &(addr.sin->sin_addr),
			  buffer,buffer_size);

	    if (!s) {
		int err UNUSED_VAROK = errno;
		
		DPRINT(Debug,12,(&Debug,
				 "give_SOCKADDR_ptr_as_string: inet_ntop failed (errno=%d) %s\n",
				 err,strerror(err)));
		goto fail;

	    }
#else /* USE_INET_XTOX_PN */
	    s = inet_ntoa(addr.sin->sin_addr);
#endif
	    break;

#ifdef HAVE_IN6
	case AF_INET6:

	    if (addrsize < sizeof (* (addr.sin6))) {
		DPRINT(Debug,12,(&Debug,
				 "give_SOCKADDR_ptr_as_string: addr size = %zu < sockaddr_sin6 size %zu\n",
				 addrsize,sizeof (* (addr.sin6))));
		goto fail;
	    }

#ifdef USE_INET_XTOX_PN
	     s = inet_ntop(AF_INET6,
			   &(addr.sin6->sin6_addr),
			   buffer,buffer_size);

	    if (!s) {
		int err UNUSED_VAROK = errno;
		
		DPRINT(Debug,12,(&Debug,
				 "give_SOCKADDR_ptr_as_string: inet_ntop failed (errno=%d) %s\n",
				 err,strerror(err)));
		goto fail;
	    }

#ifdef HAVE_SCOPE
	    /* ??? Should sa->sin6.sin6_scope_id included? */
	    if (s == buffer && addr.sin6->sin6_scope_id) {
		int l = strlen(buffer);

#ifdef HAVE_NAMEINDEX
		if (dt_flag_is_set(&ipv6_scope_id,
				   scope_link_local_if_index_flag)
		    &&
		    (IN6_IS_ADDR_LINKLOCAL(&(addr.sin6->sin6_addr)))) {

		    char buffer2[IF_NAMESIZE];
		
		    char *s2 = if_indextoname(addr.sin6->sin6_scope_id,
					      buffer2);
		    
		    if (s2 && l + strlen(s2) < buffer_size -3) {
			elm_sfprintf(buffer+l,
				     buffer_size-l,
				     FRM("%%%s"),s2);
			scope_done = 1;
		    }
		}
#endif /* HAVE_NAMEINDEX */

		if (l < buffer_size -3 && !scope_done) {
		    
		    elm_sfprintf(buffer+l,
				 buffer_size-l,
				 FRM("%%%u"),
				 (unsigned)(addr.sin6->sin6_scope_id));
		}	   	    
	    }
#endif /* HAVE SCOPE */
	    
#endif /* USE_INET_XTOX_PN */

	    break;
#endif /* HAVE_IN6 */
	    
#endif /* defined(I_NETINET_IN) && defined(I_ARPA_INET) */
	default:
		DPRINT(Debug,12,(&Debug,
				 "give_SOCKADDR_ptr_as_string: Unsupported address family %d\n",
				 addr.sa->sa_family));
		break;
	}	    
    }
	
 fail:
    if (s) {
	DPRINT(Debug,12,(&Debug,"give_SOCKADDR_ptr_as_string=%s\n",s));	
    } 

    return s;    
}

const char * give_addr_as_string(sa,mybuffer,buffer_size)
     SOCKADDR *sa;
     char * mybuffer UNUSED_VAROK;
     size_t buffer_size UNUSED_VAROK;
{
    const char * s = NULL;
    int unsupported UNUSED_VAROK = 0;
    int scope_done UNUSED_VAROK = 0;

#if defined(I_NETINET_IN) && defined(I_ARPA_INET)

#ifdef USE_INET_XTOX_PN

    /* Need know sa_family, so that address can be accessed! */
    switch (sa->sa.sa_family) {
    case AF_INET:  s = inet_ntop(AF_INET,
				 &(sa->sin.sin_addr),
				 mybuffer,buffer_size);
	break;

#ifdef HAVE_IN6
    case AF_INET6: s = inet_ntop(AF_INET6,
				&(sa->sin6.sin6_addr),
				mybuffer,buffer_size);

#ifdef HAVE_SCOPE
	/* ??? Should sa->sin6.sin6_scope_id included? */
	if (s == mybuffer && sa->sin6.sin6_scope_id) {
	    int l = strlen(mybuffer);
	    
#ifdef HAVE_NAMEINDEX
	    if (dt_flag_is_set(&ipv6_scope_id,
			       scope_link_local_if_index_flag)
		&&
		(IN6_IS_ADDR_LINKLOCAL(&(sa->sin6.sin6_addr)))) {

		char buffer2[IF_NAMESIZE];
		
		char *s2 = if_indextoname(sa->sin6.sin6_scope_id,
					  buffer2);

		if (s2 && l + strlen(s2) < buffer_size -3) {
		    elm_sfprintf(mybuffer+l,
				 buffer_size-l,
				 FRM("%%%s"),s2);
		    scope_done = 1;
		}
	    }
#endif /* HAVE_NAMEINDEX */

	    if (l < buffer_size -3 && !scope_done) {

		elm_sfprintf(mybuffer+l,
			     buffer_size-l,
			     FRM("%%%u"),
			     (unsigned)(sa->sin6.sin6_scope_id));
	    }	   
	}
#endif /* HAVE_SCOPE */

	break;
#endif  /* HAVE_IN6 */

    default: unsupported = 1;
	break;
    }

    if (!s && !unsupported) {
	int err UNUSED_VAROK = errno;

	DPRINT(Debug,12,(&Debug,
			 "give_addr_as_string: inet_ntop failed (errno=%d) %s\n",
			 err,strerror(err)));
    }

#else /* USE_INET_XTOX_PN */
    switch (sa->sa.sa_family) {
    case AF_INET:  s = inet_ntoa(sa->sin.sin_addr);
	break;
    default: unsupported = 1;
	break;
    }
#endif  /* not USE_INET_XTOX_PN */
    
    if (unsupported) {
	DPRINT(Debug,12,(&Debug,
			 "give_addr_as_string: Address family %d unsupported\n",
			 sa->sa.sa_family));
    }

#endif /* defined(I_NETINET_IN) && defined(I_ARPA_INET) */

    if (s) {
	DPRINT(Debug,12,(&Debug,"give_addr_as_string=%s\n",s));	
    } 

    return s;

}
#endif


/* Returns domain name if matches to special-use-domains-blacklist */
const char * is_special_use_domain(host_domain_name)
     const char * host_domain_name;
{
    return on_domain_blacklist(&special_bl_domains,
			       "special-use-domains-blacklist",
			       host_domain_name);

}

const char * on_domain_blacklist(ptr,fieldname,host_domain_name)
     struct dt_path_info *ptr;
     const char * fieldname;
     const char * host_domain_name;
{

    /* Pointer to internal list, do NOT free */
    const char ** blacklist = 
	give_dt_path_as_elems(ptr,fieldname);

    /* If domain name is given as ".domain" then it blacklists names under 
       that domain. If name is given as "domain" then also "domain" itself 
       (and names under it) are blacklisted. 
    */

    const char * p;
    int i;

    if (!blacklist)
	return NULL;

    /* These domains fail on normal processing */
    for (i = 0; blacklist[i]; i++)
	if (0 == istrcmp(host_domain_name,blacklist[i]))
	    return blacklist[i];

    /* look names under these domains */

    for (p = host_domain_name; *p; p++) {
	if ('.' == *p) {
	    for (i = 0; blacklist[i]; i++) {
		if (0 == istrcmp(p+1,blacklist[i]))
		    return blacklist[i];
		if (0 == istrcmp(p,blacklist[i]))
		    return blacklist[i];
	    }
	}
    }
    
    return NULL;    
}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */

