static char rcsid[] = "@(#)$Id: shared.c,v 2.25 2022/11/02 16:26:29 hurtta Exp $";

/******************************************************************************
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

/*
 * This file is compiled only if dlopen() is available, so
 * that file does not need to be guarded with #ifdef 
 */

#include "elm_defs.h"

DEBUG_VAR(Debug,__FILE__,"dl");

#include "shared_imp.h"
#include "schedule_time.h"
#include "connection_imp.h"
#include "rc_imp.h"
#include "save_opts.h"
#include "s_me.h"
#include "cs_imp.h"
#include "ss_imp.h"

#include "s_elm.h"

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

/* FIXME -- is shared_SE_option_types really needed */
struct shared_SEOT  * shared_SE_option_types      = NULL;
int                   shared_SE_option_type_count = 0;

struct shared_CST * shared_CS_types             = NULL;
int    shared_CS_type_count                     = 0;

struct shared_MCF  * shared_MCF_types           = NULL;
int                  shared_MCF_type_count      = 0;

int use_sharedfunc P_((char **value, int enter,
		       int lineno, char *filename));

struct ImpInfo **library_list            = NULL;
int              library_list_count      = 0;

static char * give_shname P_((const char *tag)); 
static char * give_shname(tag)
     const char *tag;
{
    char * ret = elm_message(FRM("%s/%s%s%s"),
			     SHARED_DIR,
			     SHARED_LIBPREFIX,
			     tag,
			     SHARED_LIBSUFFIX);

    DPRINT(Debug,4,(&Debug,"use-*-library %s => %s\n",tag,ret));

    return ret;				 
}

S_(RC_post_init_f no_rc_post_init)
static void no_rc_post_init P_((int *errors, int flags,
				const char * tag));
static void no_rc_post_init(errors,flags,tag)
     int *errors;
     int flags;
     const char * tag;
{
    /* Nothing */
}

S_(wants_rand_bits_f no_wants_rand_bits)
static void no_wants_rand_bits P_((const char *buf,
				   int size,
				   int entropy_bits));
static void no_wants_rand_bits (buf,size,entropy_bits)
     const char *buf; 
     int size;
     int entropy_bits;
{
    /* Nothing */
}

S_(RC_fill_enum_values_f no_fill_enum_values)
static struct dt_enum_shared * no_fill_enum_values 
   P_((struct dt_enumerate_info *ptr,
       const char *tag,
       int used_start_value,
       size_t dt_enumerate_info_size,
       size_t dt_enum_shared_size));
static struct dt_enum_shared * no_fill_enum_values(ptr,tag,used_start_value,
						   dt_enumerate_info_size,
						   dt_enum_shared_size)
     struct dt_enumerate_info *ptr;
     const char *tag;
     int used_start_value;
     size_t dt_enumerate_info_size;
     size_t dt_enum_shared_size;
{
    /* Nothing */

    return NULL;
}

S_(free_shared_cache_f no_free_shared_cache)
static void no_free_shared_cache  P_((void));
static void no_free_shared_cache()  
{
    /* Nothing */

}


struct ImpInfo * give_impinfo(tag)
     const char * tag;
{
    int i;

    for (i = 0; i < library_list_count; i++) {

	if (IMPINFO_magic != library_list[i]->magic)
	    panic("SHARED PANIC",__FILE__,__LINE__,
		  "give_impinfo",
		  "Bad magic number (ImpInfo)",0);
	
	if (0 == strcmp(library_list[i]->tag,tag))
	    return library_list[i];
    }

    library_list = safe_array_realloc(library_list, 
				      (i+1), sizeof(library_list[0]));

    library_list[i] = safe_zero_alloc(sizeof (* library_list[i]) );

    library_list[i]->magic  = IMPINFO_magic;
    library_list[i]->tag    = safe_strdup(tag);
    library_list[i]->shname = give_shname(tag);
    
    library_list[i]->handle = NULL;
    

    library_list[i]->regs       = NULL;
    library_list[i]->regs_count = 0;

    library_list[i]->rc_options      = NULL;
    library_list[i]->rc_option_count = 0;

    library_list[i]->rc_post_init    = no_rc_post_init;
    library_list[i]->wants_rand_bits = no_wants_rand_bits;
    library_list[i]->fill_enum_values = no_fill_enum_values;
    library_list[i]->free_shared_cache = no_free_shared_cache;

    library_list[i]->valid              = 0;	
    library_list[i]->base_lists_updated = 0;
    library_list[i]->post_init_done     = 0;
    library_list[i]->delayed            = 0;
   
    library_list_count = i+1;
    return library_list[i];
}

static void mirror_parent_regs P_((struct dt_shared_info *parent,
				   struct dt_shared_info *lib));

static void mirror_parent_regs(parent,lib)
     struct dt_shared_info *parent;
     struct dt_shared_info *lib;
{
    int i;

    if (parent->loader->magic != SHARED_LOADER_magic)
	panic("SHARED PANIC",__FILE__,__LINE__,
	      "mirror_parent_regs",
	      "Bad magic number (parent)",0);
    if (lib->loader->magic != SHARED_LOADER_magic)
	panic("SHARED PANIC",__FILE__,__LINE__,
	      "mirror_parent_regs",
	      "Bad magic number (lib)",0);

    for (i = 0; i < parent->shared_list_len; i++) {
	int found_parent = -1;
	int found_lib = -1;

	int j;

	struct ImpInfo * I = parent->shared_list[i];

	if (IMPINFO_magic != I->magic)
	    panic("SHARED PANIC",__FILE__,__LINE__,
		  "mirror_parent_regs",
		  "Bad magic number (ImpInfo)",0);
	
	for (j = 0; j < I->regs_count; j++) {
	    if (I->regs[j].var == parent)
		found_parent = j;	
	    if (I->regs[j].var == lib)
		found_lib = j;	
	}

	if (found_parent >= 0 && found_lib >= 0) {

	    DPRINT(Debug,7,(&Debug,
			    "mirror_parent_regs: %s: found parent [var %p] ~ [%p]->regs[%d] and found lib ~ [%p]->regs[%d]\n",
			    I->tag,
			    parent,I,found_parent,
			    lib,I,found_lib));
	} else if (found_parent >= 0) {
	    int rnum;

	    DPRINT(Debug,7,(&Debug,
			    "mirror_parent_regs: %s: found parent [var %p] ~ [%p]->regs[%d]\n",
			    I->tag,
			    parent,I,found_parent));

	    rnum = give_rnum(I,lib);
	    
	    I->regs[rnum].valid = 1;

	    DPRINT(Debug,10,(&Debug,
			     "mirror_parent_regs: [%p]->regs[%d].valid=%d (hidden), shname=%s\n",
			     I,rnum,I->regs[rnum].valid,
			     I->shname));

	    if (I->regs[found_parent].updated)
		reg_code1(I,rnum);
	}
    }
}

int give_rnum(I,var)
     struct ImpInfo * I; 
     struct dt_shared_info *var;
{
    int i;
    int k;

    for (i = 0; i < I->regs_count; i++) 
	if (I->regs[i].var == var)
	    return i;

    DPRINT(Debug,7,(&Debug,
		    "give_rnum: Adding var %p to %s => rnum=%d\n",
		    var,I->shname,i));

    I->regs = safe_array_realloc(I->regs,
				 (I->regs_count+1), sizeof (I->regs[0]));
    
    I->regs[i].var   = var;
    I->regs[i].valid   = 0;
    I->regs[i].updated = 0;
    I->regs[i].r.dummy = NULL;

    DPRINT(Debug,10,(&Debug,
		    "give_rnum: [%p]->regs[%d].valid=%d\n",
		     I,i,I->regs[i].valid));


    if (SHARED_LOADER_magic != var->loader->magic)
	panic("SHARED PANIC",__FILE__,__LINE__,
	      "give_rnum",
	      "Bad magic bumber",0);

    var->loader->func_zero(var,& (I->regs[i]) );

    I->regs_count = i+1;


    for (k = 0; k < var->shared_list_len; k++)
	if (var->shared_list[k] == I)
	    goto found;

    DPRINT(Debug,7,(&Debug,
		    "give_rnum: Adding %s to var %p shared_list (index %d)\n",
		    I->shname,var,k));

    var->shared_list = safe_array_realloc(var->shared_list,
					  (k+1), sizeof (var->shared_list[0]));

    var->shared_list[k] = I;
    var->shared_list_len = k+1;


 found:

    DPRINT(Debug,7,(&Debug,
		    "give_rnum=%d [var %p]\n",i,var));

    return i;


}

void load_code0(I)
     struct ImpInfo *I;
{
    if (! I->handle) {
	DPRINT(Debug,4,(&Debug," ... library loading %s\n",
			I->shname));
	
	I->handle = dlopen(I->shname,RTLD_NOW);
	
	if (! I->handle) {
	    lib_error(CATGETS(elm_msg_cat, MeSet,
			      MeDLopen0Error,
			      "library %s: %s: %s"),
		      I->tag, I->shname,
		      dlerror());
	    I->valid = 0;
	    return;
	}
    }

    if (! I->base_lists_updated) {

	/* union hack to avoid warnings about casting
         * between pointer-to-object and pointer-to-function
	 */
       
	union F8 {
	    void *                       ptr;
	    provides_RC_options_f      * f8;
	} f8;

	union F9 {
	    void *                       ptr;
	    RC_post_init_f             * f9;
	} f9;

	union F10 {
	    void *                       ptr;
	    wants_rand_bits_f          * f10;
	} f10;

	union F11 {
	    void *                       ptr;
	    init_library_REG_f         * f11;
	} f11;

	union F17 {
	    void *                       ptr;
	    RC_fill_enum_values_f      * f17;
	} f17;

	union F19 {
	    void *                       ptr;
	    free_shared_cache_f        * f19;
	} f19;

	f8.ptr = dlsym( I->handle,  "provides_RC_options2");
	f9.ptr  = dlsym( I->handle,  "RC_post_init2");
	f10.ptr = dlsym( I->handle,  "wants_rand_bits");
	f11.ptr = dlsym( I->handle,  "init_library_REG");
	f17.ptr = dlsym( I->handle,  "RC_fill_enum_values");
	f19.ptr = dlsym( I->handle,  "free_shared_cache");

	if (!f8.f8) {
	    DPRINT(Debug,7,(&Debug, " ... NO provides_RC_options2\n"));
	}
	if (!f9.f9) {
	   DPRINT(Debug,7,(&Debug, " ... NO RC_post_init2\n"));
	}
	if (!f10.f10) {
	    DPRINT(Debug,7,(&Debug, " ... NO wants_rand_bits\n"));
	}

	if (!f11.f11) {
	    DPRINT(Debug,7,(&Debug, " ... NO init_library_REG\n"));
	}

	if (!f17.f17) {
	    DPRINT(Debug,7,(&Debug, " ... NO RC_fill_enum_values\n"));
	}

	if (!f19.f19) {
	    DPRINT(Debug,7,(&Debug, " ... NO free_shared_cache\n"));
	}

	if (f8.f8) {
	    size_t   count;
	    size_t   s_res8;
	    struct rc_save_info_rec * res8 = f8.f8(&count, &s_res8);

	    if (s_res8 != sizeof (*res8)) {
		DPRINT(Debug,1,(&Debug,
				"... struct rc_save_info_rec mismatch: %d should be %d\n",
				s_res8,sizeof (*res8)));
	    } else {
		DPRINT(Debug,7,(&Debug,
				" ... provides_RC_options: count %d\n",
				count));

		 I->rc_options      = res8;
		 I->rc_option_count = count;
	    }
	}

	if (f9.f9) {
	     I->rc_post_init  = f9.f9;

	    DPRINT(Debug,7,(&Debug," ... RC_post_init2\n"));

	}

	if (f10.f10) {
	     I->wants_rand_bits = f10.f10;
	     DPRINT(Debug,7,(&Debug," ... wants_rand_bits\n"));

	}


	if (f17.f17) {
	    I->fill_enum_values = f17.f17;
	    DPRINT(Debug,7,(&Debug," ... RC_fill_enum_values\n"));
	}

	if (f19.f19) {
	    I->free_shared_cache = f19.f19;
	    DPRINT(Debug,7,(&Debug," ... free_shared_cache\n"));
	}

	if (f11.f11) {

	    DPRINT(Debug,7,(&Debug," ... init_library_REG\n"));

	    f11.f11();	   
	}

	I->base_lists_updated = 1;
    }

    /* TODO: Other bookkeeping */


}

int reg_code1(I,r)
     struct ImpInfo *I; 
     int r;
{
    int res;

    if (SHARED_LOADER_magic != I->regs[r].var->loader->magic)
	panic("SHARED PANIC",__FILE__,__LINE__,
	      "reg_code1",
	      "Bad magic bumber",0);
    
    res = I->regs[r].valid;

    if ( I->regs[r].valid &&
	 ! I->regs[r].updated ) {

	DPRINT(Debug,7,(&Debug,"reg_code1: Checking library %s ... [%d var %p]\n",
			I->shname,r,I->regs[r].var));

	res = I->regs[r].var->loader->func_reg(I, r);
	I->regs[r].updated = 1;
	DPRINT(Debug,7,(&Debug,"reg_code1: ... %s result = %d   [%d var %p]  valid = %d\n",
			I->shname,res,r,I->regs[r].var,
			I->regs[r].valid));
    } else {
	DPRINT(Debug,7,(&Debug,
			"reg_code1: Skipping library %s ... [%d var %p] valid=%d updated=%d\n",
			I->shname,r,I->regs[r].var,
			I->regs[r].valid,
			I->regs[r].updated));
    }

    return res;
}

static void load_code P_((int idx));
static void load_code(idx)
     int idx;
{
   if (idx < 0 || idx >= library_list_count)
	panic("RC PANIC",__FILE__,__LINE__,
	      "load_code",
	      "Bad index",0);
 
    if (IMPINFO_magic != library_list[idx]->magic)
	panic("SHARED PANIC",__FILE__,__LINE__,
	      "load_code",
	      "Bad magic number (ImpInfo)",0);
    
    DPRINT(Debug,4,(&Debug,"library [%d] - processing %s ... \n",
		    idx, library_list[idx]->shname));

    load_code0(library_list[idx]);
}

static void unreg1 P_((struct ImpInfo * I, int r));
static void unreg1(I,r)
     struct ImpInfo * I; 
     int r;
{
    if (IMPINFO_magic != I->magic)
	panic("SHARED PANIC",__FILE__,__LINE__,
	      "unreg1",
	      "Bad magic number (ImpInfo)",0);
    
    if (SHARED_LOADER_magic != I->regs[r].var->loader->magic)
	panic("SHARED PANIC",__FILE__,__LINE__,
	      "unreg1",
	      "Bad magic bumber",0);

    I->regs[r].var->loader->
	func_unreg(I, r);

    I->regs[r].valid   = 0;
    I->regs[r].updated = 0;

    DPRINT(Debug,10,(&Debug,
		     "unreg1: [%p]->regs[%d].valid=%d\n",
		     I,r,I->regs[r].valid));


}

static void free_shared_rc_opt1 P_((int i)); 
static void free_shared_cache1 P_((int i)); 

static void unload_code P_((int idx));
static void unload_code(idx)
     int idx;
{
    if (idx < 0 || idx >= library_list_count)
	panic("RC PANIC",__FILE__,__LINE__,
	      "unload_code",
	      "Bad index",0);

    if (IMPINFO_magic != library_list[idx]->magic)
	panic("SHARED PANIC",__FILE__,__LINE__,
	      "unload_code",
	      "Bad magic number (ImpInfo)",0);
    
    DPRINT(Debug,35,(&Debug,
		     "unload_code: Unloading %s\n",
		     library_list[idx]->tag));

    if (library_list[idx]->regs_count > 0) {
	int i;

	for (i = 0; i < library_list[idx]->regs_count; i++) {
	    	    
	    if (library_list[idx]->regs[i].valid &&
		library_list[idx]->regs[i].updated) {

		unreg1(library_list[idx],i);

	    }	    
	}
    }

    /* TODO: Other bookkeeping */

    if (library_list[idx]->handle) {

	free_shared_cache1(idx);
	free_shared_rc_opt1(idx);

	DPRINT(Debug,4,(&Debug,"library  [%d] - unloading %s\n",
			idx,library_list[idx]->shname));


	dlclose(library_list[idx]->handle);
	library_list[idx]->handle = NULL;
    }
}

int tag_ok(name)
     const char *name;
{
    if ('\0' == *name)
	return 0;

    if (strspn(name,"abcdefghijklmnopqrstuvwxyz") < 3)
	return 0;

    return strlen(name) == 
	strspn(name,
	       "abcdefghijklmnopqrstuvwxyz0123456789-");
}

static int libraries_loaded = 0;


static int locate_and_load_library P_((const char *prefix));
static int locate_and_load_library(prefix)
     const char *prefix;
{
    int i;

    for (i = 0; i < library_list_count; i++) {

	if (IMPINFO_magic != library_list[i]->magic)
	    panic("SHARED PANIC",__FILE__,__LINE__,
		  "locate_and_load_library",
		  "Bad magic number (ImpInfo)",0);
	
	if (0 == strcmp(library_list[i]->tag,prefix)) {
	    
	    /* Do not update libraries_loaded
	       because this loads only one library 
	    */
	    if (library_list[i]->valid &&
		!libraries_loaded)
		load_code(i);

	    if (library_list[i]->valid) {
		int r;

		for (r = 0; r < library_list[i]->regs_count; r++) 
		    reg_code1(library_list[i],r);
	    }

	    return i;
	}
    }
    return -1;
}

void load_shared_libs() 
{
    int i;

    if (libraries_loaded)
	return;

    for (i = 0; i < library_list_count; i++) {
	
	if (IMPINFO_magic != library_list[i]->magic)
	    panic("SHARED PANIC",__FILE__,__LINE__,
		  "load_shared_libs",
		  "Bad magic number (ImpInfo)",0);

	if (library_list[i]->valid)
	    load_code(i);

	/* NOTE: This do not load variable specific
	         code ...
	*/
    }
    libraries_loaded = 1;
}

enum verlib_status verify_library_prefix(prefix,var,read_flags)
     const char *prefix;
     struct dt_shared_info *var;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    enum verlib_status ret = verlib_not_loaded;
    int i;
    int found = 0;

    
    DPRINT(Debug,9,(&Debug,
		    "verify_library_prefix: Checking [var %p] for prefix %s ...\n",
		    var,prefix)); 

    for (i = 0; i < var->shared_list_len; i++) {
	
	if (IMPINFO_magic != var->shared_list[i]->magic)
	    panic("SHARED PANIC",__FILE__,__LINE__,
		  "verify_library_prefix",
		  "Bad magic number (ImpInfo)",0);
	
	if (0 == strcmp(var->shared_list[i]->tag,prefix)) {

	    found = 1;
	    
	    if (var->shared_list[i]->valid) {
		ret = verlib_found;
		DPRINT(Debug,9,(&Debug,
				"verify_library_prefix: %s found and valid\n",
				prefix));
	    } else {
		DPRINT(Debug,9,(&Debug,
				"verify_library_prefix: %s found, but not valid\n",
				prefix));
		
	    }
	}
    }

    if (!found) {
	if (tag_ok(prefix)) {
	    struct ImpInfo * I = give_impinfo(prefix);

	    if (IMPINFO_magic != I->magic)
		panic("SHARED PANIC",__FILE__,__LINE__,
		      "verify_library_prefix",
		      "Bad magic number (ImpInfo)",0);

	    if (I->valid) {
		DPRINT(Debug,9,(&Debug,
				"verify_library_prefix: %s not found, but valid\n",
				prefix));
		ret = verlib_delayed;
	    } else {
		int rcode = access(I->shname,ACCESS_EXISTS);

		if (0 == rcode) {
		    DPRINT(Debug,9,(&Debug,
				    "verify_library_prefix: %s not found, but %s exists\n",
				    prefix,I->shname));

		    ret = verlib_delayed;
		    
		} else if (-1 == rcode) {
		    int err = errno;

		    if (ENOENT == err &&
			0 != (read_flags & READ_FLAG_IGNORE_MISSING)) {

			DPRINT(Debug,9,(&Debug,
					"verify_library_prefix: %s not found, but %s missing, but ignored\n",
					prefix,I->shname));
			
			ret = verlib_delayed;
			
		    } else {
		    
			DPRINT(Debug,9,(&Debug,
					"verify_library_prefix: %s not found, but %s: %s\n",
					prefix,I->shname, strerror(err)));
			
			ret = verlib_bad;
		    }
		}		
	    }
	    
	} else {
	    DPRINT(Debug,9,(&Debug,
			    "verify_library_prefix: %s not found, invalid tag\n",
			    prefix));
	    
	    ret = verlib_bad;
	}
    }

    DPRINT(Debug,9,(&Debug,
		    "verify_library_prefix=%d",ret));
    switch(ret) {
    case verlib_bad:         DPRINT(Debug,9,(&Debug," verlib_bad"));        break;
    case verlib_not_loaded:  DPRINT(Debug,9,(&Debug," verlib_not_loaded")); break;
    case verlib_found:       DPRINT(Debug,9,(&Debug," verlib_found"));      break;
    case verlib_delayed:     DPRINT(Debug,9,(&Debug," verlib_delayed"));    break;
    }
    DPRINT(Debug,9,(&Debug,"\n"));
    
    return ret;
}

int locate_and_load_library1(prefix,var)
     const char *prefix;
     struct dt_shared_info *var;
{
    int i;

    DPRINT(Debug,7,(&Debug,"Loading [var %p] for prefix %s ...\n",
		    var,prefix)); 
   
    for (i = 0; i < var->shared_list_len; i++) {

	if (IMPINFO_magic != var->shared_list[i]->magic)
	    panic("SHARED PANIC",__FILE__,__LINE__,
		  "locate_and_load_library1",
		  "Bad magic number (ImpInfo)",0);
	
	if (0 == strcmp(var->shared_list[i]->tag,prefix)) {

	    /* Do not update libraries_loaded
	       because this loads only one library 
	    */

	    if (var->shared_list[i]->valid &&
		! var->libraries_loaded)
		load_code0(var->shared_list[i]);

	    if (var->shared_list[i]->valid) {
		int r = give_rnum(var->shared_list[i],var);

		DPRINT(Debug,7,(&Debug,
			    "Prosessing [var %p] %d, %s : valid=%d => r=%d\n",
			    var, i, 
			    var->shared_list[i]->shname,
			    var->shared_list[i]->valid,
			    r));

		reg_code1(var->shared_list[i],r);
		
		return i;
		
	    } else {
	    
		DPRINT(Debug,7,(&Debug,
				"Skipping [var %p] %d, %s : valid=%d\n",
				var, i, 
				var->shared_list[i]->shname,
				var->shared_list[i]->valid));
    

	    }
	}

    } 

    return -1;
}

void load_shared_libs1(var)
     struct dt_shared_info *var;
{
    int i;
    
    if (var->libraries_loaded)
	return;

    DPRINT(Debug,7,(&Debug,"Loading [var %p] ...\n",var));

    for (i = 0; i < var->shared_list_len; i++) {

	if (IMPINFO_magic != var->shared_list[i]->magic)
	    panic("SHARED PANIC",__FILE__,__LINE__,
		  "load_shared_libs1",
		  "Bad magic number (ImpInfo)",0);
	
	if (var->shared_list[i]->valid)
	    load_code0(var->shared_list[i]);

	if (var->shared_list[i]->valid) {
	    int r = give_rnum(var->shared_list[i],var);

	    DPRINT(Debug,7,(&Debug,
			    "Prosessing [var %p] %d, %s : valid=%d => r=%d\n",
			    var, i, 
			    var->shared_list[i]->shname,
			    var->shared_list[i]->valid,
			    r));

	    reg_code1(var->shared_list[i],r);

	} else {
	    
	    DPRINT(Debug,7,(&Debug,
			    "Skipping [var %p] %d, %s : valid=%d\n",
			    var, i, 
			    var->shared_list[i]->shname,
			    var->shared_list[i]->valid));
    

	}
    }

    DPRINT(Debug,7,(&Debug,"Loading [var %p] done.\n",var));
    var->libraries_loaded = 1;
}

void register_hidden_library(parent,lib)
     struct dt_shared_info *parent;
     struct dt_shared_info *lib;
{
    if (lib->hidden_var)
	panic("SHARED PANIC",__FILE__,__LINE__,
	      "register_hidden_library",
	      "register_hidden_library have children or already registered",0);

    DPRINT(Debug,7,(&Debug,
		    "register_hidden_library: [var %p] have hidden [var %p]",
		    parent,lib));

    lib->hidden_var = parent->hidden_var;
    lib->libraries_loaded = 0;

    if (parent->hidden_var) {
	DPRINT(Debug,7,(&Debug," and [var %p]",
			lib->hidden_var));
    }
    DPRINT(Debug,7,(&Debug,"\n"));
    
    parent->hidden_var = lib;

    mirror_parent_regs(parent,lib);
}

int delay_lib_reg(r,value,lineno,filename,read_flags)
     struct rc_save_info_rec *r;
     char **value;  /* Changed to valid, must be malloced */
     int lineno;
     const char *filename;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    /* partially duplicate  process_lib_reg */
    
    int ret = 1;
    char * temp = safe_strdup(*value);
    char *p;
    
    (*value)[0] = '\0';
    
    for (p = strtok(temp," "); p; p = strtok(NULL," ")) {
	
	if (!tag_ok(p)) {
	    
	    if (filename)
		lib_error(CATGETS(elm_msg_cat, MeSet,
				  MeInvalidUseXLibRC,
				  "%s:%d: %s: %s: Invalid name"),
			  filename,lineno,
			  r->name,p);
	    else
		lib_error(CATGETS(elm_msg_cat, MeSet,
				  MeInvalidUseXLib,
				  "%s: %s: Invalid name"),
			  r->name,p);
	    
	    ret = 0;
	    
	} else {
	    
	    struct ImpInfo * I = give_impinfo(p);
	    int rcode;

	    if (IMPINFO_magic != I->magic)
		panic("SHARED PANIC",__FILE__,__LINE__,
		      "delay_lib_reg",
		      "Bad magic number (ImpInfo)",0);
	    	    
	    rcode = access(I->shname,ACCESS_EXISTS);

	    if (0 == rcode) {

		if ((*value)[0])
		    *value = strmcat(*value," ");

		*value = strmcat(*value,I->tag);
		
		I->delayed = 1;   /* This is OK but not
				   * available yet 
				   */
		
	    } else if (-1 == rcode) {
		int err = errno;

		if (ENOENT == err &&
		    0 != (read_flags & READ_FLAG_IGNORE_MISSING)) {
		    
		    DPRINT(Debug,2,(&Debug,
				    "Ignoring missing %s\n",
				    I->shname));
		    
		} else {
		
		    if (filename)
			lib_error(CATGETS(elm_msg_cat, MeSet,
					  MeNonExistUseXLibRC,
					  "%s:%d: %s: %s: %s: %s"),
				  filename,lineno,
				  r->name,I->tag,I->shname,
				  strerror(err));
		    else
			lib_error(CATGETS(elm_msg_cat, MeSet,
					  MeNonExistUseXLib,
					  "%s: %s: %s: %s"),
				  r->name,I->tag,I->shname,
				  strerror(err));
		
		    ret = 0;
		}
	    }   
	}
    }
    
    free(temp);
    
    return ret;
}

static int process_lib_reg P_((struct rc_save_info_rec *r,
			       char *value, int lineno,
			       const char *filename,
			       int read_flags /* READ_FLAG_IGNORE_MISSING */));
static int process_lib_reg(r,value,lineno,filename,read_flags)
     struct rc_save_info_rec *r;
     char *value; 
     int lineno;
     const char *filename;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    int ret = 1;
    struct dt_shared_info *S = r->val.shared;
    char * temp = safe_strdup(value);
    char *p;


    if (S->loader->magic != SHARED_LOADER_magic)
	panic("SHARED PANIC",__FILE__,__LINE__,
	      "process_lib_reg",
	      "Bad magic number (1)",0);

    for (p = strtok(temp," "); p; p = strtok(NULL," ")) {
	if (!tag_ok(p)) {
	    if (filename)
		lib_error(CATGETS(elm_msg_cat, MeSet,
				  MeInvalidUseXLibRC,
				  "%s:%d: %s: %s: Invalid name"),
			  filename,lineno,
			  r->name,p);
	    else
		lib_error(CATGETS(elm_msg_cat, MeSet,
				  MeInvalidUseXLib,
				  "%s: %s: Invalid name"),
			  r->name,p);

	    ret = 0;
	    
	} else {
	    
	    struct ImpInfo * I = give_impinfo(p);
	    
	    int rnum;

	    /* We do not load code here with dlopen, because we
	       cant load code only when it is needed.
	    */
	    int rcode;

	    if (IMPINFO_magic != I->magic)
		panic("SHARED PANIC",__FILE__,__LINE__,
		      "process_lib_reg",
		      "Bad magic number (ImpInfo)",0);

	    rnum = give_rnum(I,S);

	    rcode = access(I->shname,ACCESS_EXISTS);

	    if (0 == rcode) {

		struct dt_shared_info *S1;

		I->valid         = 1;
		I->delayed       = 0;
		libraries_loaded = 0;     /* Rescan... */

		I->regs[rnum].valid = 1;
		S->libraries_loaded = 0;

		DPRINT(Debug,10,(&Debug,
				 "process_lib_reg: [%p]->regs[%d].valid=%d, shname=%s\n",
				 I,rnum,I->regs[rnum].valid,
				 I->shname));

		for (S1 = S->hidden_var; S1;  S1 = S1->hidden_var) {
		    int rnum = give_rnum(I,S1);

		    I->regs[rnum].valid = 1;
		    S1->libraries_loaded = 0;

		    DPRINT(Debug,10,(&Debug,
				     "process_lib_reg: [%p]->regs[%d].valid=%d (hidden), shname=%s\n",
				     I,rnum,I->regs[rnum].valid,
				     I->shname));
		}
		

	    } else if (-1 == rcode) {
		int err = errno;
		
		if (ENOENT == err &&
		    0 != (read_flags & READ_FLAG_IGNORE_MISSING)) {
		    
		    DPRINT(Debug,2,(&Debug,
				    "Ignoring missing %s\n",
				    I->shname));
		    
		} else {
		    
		    if (filename)
			lib_error(CATGETS(elm_msg_cat, MeSet,
					  MeNonExistUseXLibRC,
					  "%s:%d: %s: %s: %s: %s"),
				  filename,lineno,
				  r->name,I->tag,I->shname,
				  strerror(err));
		    else
			lib_error(CATGETS(elm_msg_cat, MeSet,
					  MeNonExistUseXLib,
					  "%s: %s: %s: %s"),
				  r->name,I->tag,I->shname,
				  strerror(err));
		    
		    ret = 0;
		}
	    }   
	}
    
    }
    
    free(temp);
    return ret;
}

S_(rc_parse_line dt_SHARED_parse_line)
static int dt_SHARED_parse_line(r,lcl,value,lineno,filename,
				negate,read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int negate;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{   
    int i;

    int ret;
    char * buffer = NULL;
    int buffer_len = 0;

    if (negate) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadNegate,
			  "!%s is not supported in line %d in \"%s\" file"),
		  r->name,lineno,filename);
	return 0;
    }

    if (r->val.shared->loader->magic != SHARED_LOADER_magic)
	panic("SHARED PANIC",__FILE__,__LINE__,
	      "dt_SHARED_parse_line",
	      "Bad magic bumber (1)",0);

    

    for (i = 0; i < r->val.shared->shared_list_len; i++) {
	int j;

	for (j = 0; j < r->val.shared->shared_list[i]->regs_count; j++) {
	    if (r->val.shared->shared_list[i]->regs[j].var->loader->magic !=
		SHARED_LOADER_magic)
		panic("SHARED PANIC",__FILE__,__LINE__,
		      "dt_SHARED_parse_line",
		      "Bad magic bumber (2)",0);

	    if (r->val.shared == r->val.shared->shared_list[i]->regs[j].var) {
		r->val.shared->shared_list[i]->regs[j].valid = 0;


		DPRINT(Debug,10,(&Debug,
				 "dt_SHARED_parse_line: [%p]->regs[%d].valid=%d\n",
				 r->val.shared->shared_list[i],
				 j,
				 r->val.shared->shared_list[i]->regs[j].valid));


		goto found1;
	    }

	}

	panic("SHARED PANIC",__FILE__,__LINE__,
	      "dt_SHARED_parse_line",
	      "No info on list",0);


    found1:;

    }
    
    ret = convert_to_system_charset(value,lineno,filename,&buffer,&buffer_len);

    if (!process_lib_reg(r,buffer,lineno,filename,read_flags))
	ret = 0;
    
    free(buffer);
    
    DPRINT(Debug,20,(&Debug, 
		     "dt_SHARED_parse_line [%s:%d] = %d\n",
		     filename,lineno,ret));
    
    return ret;
}

S_(rc_parse_cline dt_SHARED_parse_cline)
static int dt_SHARED_parse_cline(r,lcl,value,lineno,filename,
				 read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    int ret;
    char * buffer = NULL;
    int buffer_len = 0;
    
    ret = convert_to_system_charset(value,lineno,filename,&buffer,&buffer_len);
    
    if (!process_lib_reg(r,buffer,lineno,filename,read_flags))
	ret = 0;
    
    free(buffer);
    
    DPRINT(Debug,20,(&Debug, 
		     "dt_SHARED_parse_cline [%s:%d] = %d\n",
		     filename,lineno,ret));
    
     return ret;
}

S_(rc_print_value dt_SHARED_print_value)
static void dt_SHARED_print_value(F,r,comment,rcset)
     FILE *F;
     struct rc_save_info_rec *r;
     int comment;
     charset_t rcset;
{
    int i;
    int len=0;
    char * buffer = NULL;
    int   buffer_len = 0;

    if (r->val.shared->loader->magic != SHARED_LOADER_magic)
	panic("SHARED PANIC",__FILE__,__LINE__,
	      "dt_SHARED_print_value",
	      "Bad magic bumber (1)",0);

    if (comment)
	fprintf(F, "### ");
    fprintf(F, "%s =", r->name);


    for (i = 0; i < r->val.shared->shared_list_len; i++) {
	int j;

	for (j = 0; j < r->val.shared->shared_list[i]->regs_count; j++) {
	    if (r->val.shared->shared_list[i]->regs[j].var->loader->magic !=
		SHARED_LOADER_magic)
		panic("SHARED PANIC",__FILE__,__LINE__,
		      "dt_SHARED_print_value",
		      "Bad magic bumber (2)",0);

	    if (r->val.shared == r->val.shared->shared_list[i]->regs[j].var) {
		goto found1;
	    }

	}

	panic("SHARED PANIC",__FILE__,__LINE__,
	      "dt_SHARED_print_value",
	      "No info on list",0);


    found1:
	
	if (r->val.shared->shared_list[i]->regs[j].valid) {
	    char * s0 = r->val.shared->shared_list[i]->tag;	    
	    char * s = s0;
	    int ok UNUSED_VAROK = 1;
	    
	    if (rcset != system_charset) {
		ok = convert_to_rcset(s,-1,r->name,rcset,&buffer,&buffer_len);	
		if (buffer)
		    s = buffer;
	    }
	    
	    if (buffer_len + len > 72) {
		if (!comment)
		    fprintf(F, "\n\t");
		else
		    fprintf(F, "\n###\t");
		len = 8;
	    }
	    else {
		fprintf(F, " ");
		++len;
	    }
	    fprintf(F, "%s", s);
	    len += strlen(s);
	}
    }
    fprintf(F,"\n");

    if (buffer)
	free(buffer);
    buffer = NULL;
}

S_(rc_get_value dt_SHARED_get_value)
static char * dt_SHARED_get_value(r)
     struct rc_save_info_rec *r;
{
    /* static pointer to buffer accross invocations */
    static char * return_buffer = NULL;
    int i;

    return_buffer = strmcpy(return_buffer,"");    /* Reset result */

    if (r->val.shared->loader->magic != SHARED_LOADER_magic)
	panic("SHARED PANIC",__FILE__,__LINE__,
	      "dt_SHARED_get_value",
	      "Bad magic bumber (1)",0);

    for (i = 0; i < r->val.shared->shared_list_len; i++) {
	int j;

	for (j = 0; j < r->val.shared->shared_list[i]->regs_count; j++) {
	    if (r->val.shared->shared_list[i]->regs[j].var->loader->magic !=
		SHARED_LOADER_magic)
		panic("SHARED PANIC",__FILE__,__LINE__,
		      "dt_SHARED_get_value",
		      "Bad magic bumber (2)",0);

	    if (r->val.shared == r->val.shared->shared_list[i]->regs[j].var) {
		goto found1;
	    }

	}

	panic("SHARED PANIC",__FILE__,__LINE__,
	      "dt_SHARED_get_value",
	      "No info on list",0);

    found1:

	if (r->val.shared->shared_list[i]->regs[j].valid) {
	    char * s = r->val.shared->shared_list[i]->tag;

	    if (return_buffer[0])
		return_buffer = strmcat(return_buffer," ");
	    
	    return_buffer = strmcat(return_buffer,s);
	}
    }

    return return_buffer;
}

S_(rc_free_option dt_SHARED_free_option)
static void dt_SHARED_free_option  P_((struct rc_save_info_rec *r));
static void dt_SHARED_free_option(r)
     struct rc_save_info_rec *r;
{
    if (r->val.shared->loader->magic != SHARED_LOADER_magic)
	panic("SHARED PANIC",__FILE__,__LINE__,
	      "dt_SHARED_free_option",
	      "Bad magic bumber",0);

    /* XXXX this is too complex to completely free */

    /* give_rnum() adds to list if needed */
    if (r->val.shared->shared_list) {  
	free(r->val.shared->shared_list);
	r->val.shared->shared_list = NULL;	
    }
    r->val.shared->shared_list_len = 0;
    
}

struct rc_type rc_DT_SHARED = { RCTYPE_magic,
				dt_SHARED_parse_line, dt_SHARED_parse_cline,
				dt_SHARED_print_value, dt_SHARED_get_value,
				dt_SHARED_free_option};


int verify_shared_index(i)
     struct ImpInfo *i;
{
    if (IMPINFO_magic != i->magic)
	panic("SHARED PANIC",__FILE__,__LINE__,
	      "verify_shared_index",
	      "Bad magic number (ImpInfo)",0);

    if (i->handle &&
	i->valid)
	return 1;

    return 0;
}

struct SE_option_type * get_option_type(prefix)
     const char *prefix;
{
    int idx1 = locate_and_load_library(prefix);

    /* locate_and_load_library() may return -1 */

    if (idx1 >= 0) {
	int i;

	for (i = 0; i < shared_SE_option_type_count; i++) {
	    struct ImpInfo *I = shared_SE_option_types[i].imp_idx;

	    if (IMPINFO_magic != I->magic)
		panic("SHARED PANIC",__FILE__,__LINE__,
		      "get_option_type",
		      "Bad magic number (ImpInfo)",0);
	    
	    if (library_list[idx1] == I) {
		
		if (! I->handle) {
		    lib_error(CATGETS(elm_msg_cat, MeSet,
				      MeInvalidOptionTag,
				      "Options %s:* ignored -- code not loaded"),
			      prefix);
		    return NULL;
		}
		return shared_SE_option_types[i].T;
	    }
	}
    }

    lib_error(CATGETS(elm_msg_cat, MeSet,
		      MeInvalidOptionTag,
		      "Options %s:* ignored -- code not loaded"),
	      prefix);
    return NULL;
}

void mark_shared_changed(A,no_history_record)
     void *A;
     int no_history_record;
{
    int i;

    for (i = 0; i < library_list_count; i++) {
	size_t x;

	if (IMPINFO_magic != library_list[i]->magic)
	    panic("SHARED PANIC",__FILE__,__LINE__,
		  "mark_shared_changed",
		  "Bad magic number (ImpInfo)",0);

	if (!library_list[i]->handle) 
	    continue;

	for (x = 0; x < library_list[i]->rc_option_count; x++) {
	
	    if (library_list[i]->rc_options[x].val.dummy == A) {
		mark_XX(& (library_list[i]->rc_options[x]),
			no_history_record);
	    }
	}
    }
}


void mark_fshared_changed(A, no_history_record)
     option_func *A;
     int no_history_record;
{
    int i;

    for (i = 0; i < library_list_count; i++) {
	size_t x;

	if (IMPINFO_magic != library_list[i]->magic)
	    panic("SHARED PANIC",__FILE__,__LINE__,
		  " mark_fshared_changed",
		  "Bad magic number (ImpInfo)",0);
	
	if (!library_list[i]->handle) 
	    continue;

	for (x = 0; x < library_list[i]->rc_option_count; x++) {
	
	    if (library_list[i]->rc_options[x].func_val == A) {
		mark_XX(& (library_list[i]->rc_options[x]),
			no_history_record);
	    }

	}
    }
}

int give_options(tag,rc_options,rc_option_count,have_code)
     char *tag;
     struct rc_save_info_rec ** rc_options;
     size_t                   * rc_option_count;
     enum shared_have_code    * have_code;
{
    int idx1 = locate_and_load_library(tag);
    
    *rc_options           = NULL;
    *rc_option_count      = 0;

    DPRINT(Debug,20,(&Debug,"give_options: tag=%s, idx1=%d",
		     tag,idx1));
		     
    if (idx1 < 0) {
	DPRINT(Debug,20,(&Debug,"\n"));
	
	if (tag_ok(tag)) {
	    DPRINT(Debug,20,(&Debug,"give_options: tag=%s ok\n",
			     tag));
	    if (have_code)
		*have_code = have_code_not;
	} else {
	    DPRINT(Debug,20,(&Debug,"give_options: tag=%s invalid\n",
			     tag));

	    if (have_code)
		*have_code = have_code_invalid;
	}
	
	DPRINT(Debug,20,(&Debug,
			 "give_options=0 (library not specified)\n"));
	return 0;
    }

    /* give_impinfo() is only called if tag_ok()
       returns true, therefore if idx1 >= 0,
       tag is OK
    */
    
    if (IMPINFO_magic != library_list[idx1]->magic)
	panic("SHARED PANIC",__FILE__,__LINE__,
	      "give_options",
	      "Bad magic number (ImpInfo)",0);

    if (!library_list[idx1]->handle) {
	DPRINT(Debug,20,(&Debug,"; handle=NULL"));

	if (library_list[idx1]->valid) {
	    DPRINT(Debug,20,(&Debug,"; valid\n"));

	    if (have_code)
		*have_code = have_code_delayed;
	    
	    DPRINT(Debug,20,(&Debug,
			     "give_options=0 (library valid, not loaded)\n"));
	    
	    return 0;
	}

	if (library_list[idx1]->delayed) {
	    DPRINT(Debug,20,(&Debug,"; delayed\n"));

	    if (have_code)
		*have_code = have_code_delayed;
	    
	    DPRINT(Debug,20,(&Debug,
			     "give_options=0 (library not loaded, setting delayed)\n"));
	    
	    return 0;
	}
	
	DPRINT(Debug,20,(&Debug,"; tag ok (code not loaded)\n"));
		
	if (have_code)
	    *have_code = have_code_not;
		
	DPRINT(Debug,20,(&Debug,
			 "give_options=0 (library not loaded)\n"));

	return 0;
    }

    DPRINT(Debug,20,(&Debug,"; tag ok (code loaded)"));
	        
    *rc_options       = library_list[idx1]->rc_options;
    *rc_option_count  = library_list[idx1]->rc_option_count;
    
    DPRINT(Debug,20,(&Debug,"; %lu options\n",
		     (unsigned long)(library_list[idx1]->rc_option_count)));
    DPRINT(Debug,20,(&Debug,
		     "give_options=1\n"));
    
    return 1;
}

static void seed_rand_bits1(const char *buf, int size,
			    int entropy_bits);

static void seed_rand_bits1(buf,size,entropy_bits)
     const char *buf; 
     int size;
     int entropy_bits;
{
    int i;

    DPRINT(Debug,10,(&Debug, 
		    "Seeding random generator %d bytes (%d entropy bits)\n",
		    size,entropy_bits));

    if (entropy_bits > size*8)
	panic("SHARED PANIC",__FILE__,__LINE__,"seed_rand_bits1",
	      "Impossible entropy",0);


    /* Only seed libraries which are loaded */

    for (i = 0; i < library_list_count; i++) {
	if (library_list[i]->handle) {
	    /* We should perhaps mangle data so same bits are
	       not given to different modules
	    */
	    library_list[i]->wants_rand_bits(buf,size,entropy_bits);	    
	}
    }
}

void seed_rand_bits(buf,size,entropy_bits)
     const char *buf; 
     int size;
     int entropy_bits;
{
    
    static char buffer[1024];
    static int  byte_count    = 0;
    static int  entropy_count  = 0;
    
    DPRINT(Debug,10,(&Debug, 
		    "Got random generator %d bytes (%d entropy bits)\n",
		    size,entropy_bits));

    if (entropy_bits > size*8)
	panic("SHARED PANIC",__FILE__,__LINE__,"seed_rand_bits",
	      "Impossible entropy",0);

    if (size > sizeof buffer)
	seed_rand_bits1(buf,size,entropy_bits);
    else {

	int i;

	if (size + byte_count > sizeof buffer) {
	    seed_rand_bits1(buffer,byte_count,entropy_count);
	    byte_count = 0;
	    entropy_count = 0;
	}

	/* We can just XOR bits ... */
	for (i = 0; i < size; i++) 
	    buffer[byte_count++] ^= buf[i];

	if (byte_count > sizeof buffer)
	    panic("SHARED PANIC",__FILE__,__LINE__,"seed_rand_bits",
		  "Overflow",0);
	entropy_count += entropy_bits;

	if (entropy_count > 16 && library_list_count) {
	    seed_rand_bits1(buffer,byte_count,entropy_count);
	    byte_count = 0;
	    entropy_count = 0;
	}
    }
}


static void post_init_shop_one P_((struct ImpInfo * imp_idx,
				   int *errors,
			      int read_flags /* READ_FLAG_IGNORE_MISSING */));
static void post_init_shop_one(imp_idx,errors,read_flags)
     struct ImpInfo * imp_idx;
     int *errors;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    if (IMPINFO_magic != imp_idx->magic)
	panic("SHARED PANIC",__FILE__,__LINE__,
	      "post_init_shop_one",
	      "Bad magic number (ImpInfo)",0);
    
    DPRINT(Debug,8,(&Debug,"rc_post_init library %s\n",
		    imp_idx->tag));
    
    imp_idx->rc_post_init(errors,read_flags,
			  imp_idx->tag);
    imp_idx->post_init_done = 1;

    DPRINT(Debug,8,(&Debug,
		    "processing delayed enumerated values for library %s\n",
		    imp_idx->tag));
	    
    if (process_delayed_enumerated(imp_idx->tag)) {
	
	DPRINT(Debug,8,(&Debug,
			"enumerated values for library %s OK (or not used)\n",
			imp_idx->tag));
	
    } else if (errors) {
	*errors += 1;
    }
    
    process_delayed_blocks(imp_idx->tag,
			   imp_idx->rc_options,
			   imp_idx->rc_option_count,
			   errors,read_flags);        
}

static int stored_read_flags = 0;

void post_init_shared_opt1(var,idx)
     struct dt_shared_info *var;	  
     int idx;
{
    DPRINT(Debug,8,(&Debug,
		    "post_init_shared_opt1 [var %p] idx %d\n",
		    var,idx));
    
    if (idx < 0 || idx >= var->shared_list_len)
	 panic("SHARED PANIC",__FILE__,__LINE__,
		  "post_init_shared_opt1",
		  "Bad idx",0);

    if (IMPINFO_magic != var->shared_list[idx]->magic)
	panic("SHARED PANIC",__FILE__,__LINE__,
	      "post_init_shared_opt1",
	      "Bad magic number (ImpInfo)",0);
    
    if (var->shared_list[idx]->handle &&
	var->shared_list[idx]->rc_option_count > 0) {

	if (var->shared_list[idx]->post_init_done) {
	    
	    DPRINT(Debug,8,(&Debug,
			    "post_init_shared_opt1: [var %p] #%d %s done earlier\n",
			    var,idx,var->shared_list[idx]->tag));
	} else {
	    DPRINT(Debug,8,(&Debug,
			   "post_init_shared_opt1: [var %p] #%d %s processing\n",
			    var,idx,var->shared_list[idx]->tag));

	    post_init_shop_one(var->shared_list[idx],NULL,
			       stored_read_flags);
	}	
    }    
}


void post_init_shared_options1(var)
     struct dt_shared_info *var;
{
    int i;
    int count = 0;
    DPRINT(Debug,8,(&Debug,"post_init_shared_options1 [var %p] ...\n",var));
    
    for (i = 0; i < var->shared_list_len; i++) {
	
	if (IMPINFO_magic != var->shared_list[i]->magic)
	    panic("SHARED PANIC",__FILE__,__LINE__,
		  "post_init_shared_options1",
		  "Bad magic number (ImpInfo)",0);

	if (var->shared_list[i]->handle &&
	    var->shared_list[i]->rc_option_count > 0) {

	    if (var->shared_list[i]->post_init_done) {
		DPRINT(Debug,8,(&Debug,
				"rc_post_init library %s done earlier\n",
				var->shared_list[i]->tag));
	    } else {

		post_init_shop_one(var->shared_list[i],NULL,
				   stored_read_flags);

		count++;
	    }
	    
	}	   	
    }

    DPRINT(Debug,8,(&Debug,"post_init_shared_options1 [var %p] done %d libraries\n",
		    var,count));
}

void post_init_shared_options(errors, read_flags)
     int *errors;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    int i;
    int count = 0;
    
    DPRINT(Debug,8,(&Debug,"post_init_shared_options ... read_flags=%d\n",
		    read_flags));

    /* Automatically move libraries out from
       use-library ... 
    */

    /* load generic ... */
    load_shared_libs1(&use_shared_all);

    /* unreg possible registered but no longer valid... */

    for (i = 0; i < library_list_count; i++) {
	int j;
	for (j = 0; j < library_list[i]->regs_count; j++) {
	    
	    if (! library_list[i]->regs[j].valid &&
		library_list[i]->regs[j].updated) {
		
		DPRINT(Debug,4,(&Debug,"Unregistering no longer valid library %s\n",
				library_list[i]->tag));
		
		unreg1(library_list[i],j);
	    }
	}
    }

    /* Unload possible loaded but no longer valid... */
    for (i = 0; i < library_list_count; i++) {
	if (!library_list[i]->valid) {
	    DPRINT(Debug,4,(&Debug,"Unloading no longer valid library %s\n",
			    library_list[i]->tag));

	    unload_code(i);
	}
    }

    /* Only post_init libraries which are loaded
       (other do not have options)
    */

    for (i = 0; i < library_list_count; i++) {
	if (library_list[i]->handle &&
	    library_list[i]->rc_option_count > 0) {


	    post_init_shop_one(library_list[i],
			       errors,read_flags);
	    
	    count++;
	}
    }

    stored_read_flags = read_flags;
    
    DPRINT(Debug,8,(&Debug,"post_init_shared_options - done %d libraries\n",
		    count));
}

/* Returns commentfile */
static FILE * print_library_header P_((FILE *F,const char *tag,
				       int default_commentfile  /* use default commentfile */));
static FILE * print_library_header(F,tag,default_commentfile)
     FILE *F;
     const char *tag;
     int default_commentfile  /* use default commentfile */;
{

    FILE * commentfile = NULL;
    
    if (default_commentfile) {
	char * filename = 
	    elm_message(FRM("%s/elmrc-%s-info"),
			INFO_DIR,tag);
	
	commentfile = fopen(filename,"r"); 
	if (!commentfile) {
	    int err = errno;
	    
	    if (ENOENT != err) 
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeElmrcCommentFileError,
				  "Comment file %s for %s library not readable: %s\n"),
			  filename, tag,
			  strerror(err));
	    
	    DPRINT(Debug,8,(&Debug,
			    "library %s: no comment file %s: %s\n",
			    tag,
			    filename,
			    strerror(err)));
	    
	} 
	
	free(filename);
    }
    
    fprintf(F,"\n");
    
    elm_fprintf(F, 
		CATGETS(elm_msg_cat, MeSet, MeElmrcSection,
			"# Options for %s section (shared library)\n#\n\n"), 
		tag);
    
    
    return commentfile;
}


void print_local_shared_options(F,global,rcset,default_commentfile)
     FILE *F;
     int global;
     charset_t rcset;
     int default_commentfile;  /* use default commentfile */
{
    int i;

    /* We want print options of all shared libs */
    load_shared_libs();

    for (i = 0; i < library_list_count; i++) {
	
	if (library_list[i]->handle &&
	    library_list[i]->rc_option_count > 0) {

	     FILE * commentfile =
		 print_library_header(F,
				      library_list[i]->tag,
				      default_commentfile);
	     

	    if (!library_list[i]->valid) {

		/* Is this needed? */

		DPRINT(Debug,8,(&Debug,"library %s not valid\n",
				library_list[i]->tag));

		elm_fprintf(F, 
			    CATGETS(elm_msg_cat, MeSet, MeElmrcLibraryNotSelected,
				    "# Library of %s section no longer selected to use\n"),
			    library_list[i]->tag);


	    } else if (! library_list[i]->post_init_done) {
		int errors = 0;

		DPRINT(Debug,8,(&Debug,"rc_post_init library %s\n",
				library_list[i]->tag));
		
		library_list[i]->rc_post_init(&errors,0,
					      library_list[i]->tag);
		library_list[i]->post_init_done = 1;

		if (errors) {
		    elm_fprintf(F, 
				CATGETS(elm_msg_cat, MeSet, MeElmrcLibraryErrors,
					"# Failed to initialize options of %s section\n"),
				library_list[i]->tag);
		}
	    }

	    fprintf(F,"%s:\n",
		    library_list[i]->tag);

	    DPRINT(Debug,8,(&Debug, " -- library %s  -- %lu options\n",
			    library_list[i]->tag,
			    (unsigned long)(library_list[i]->rc_option_count)));

	    write_rc_part(F,library_list[i]->tag,commentfile,
			  global,rcset,
			  library_list[i]->rc_options,
			  library_list[i]->rc_option_count);

	    if (commentfile)
		fclose(commentfile);
			  
	} else {
	    struct delayed_block * walk         = 0;
	    enum record_mode       lcl          = RC_MODE_COUNT    /* Invalid value */;
	    charset_t              in_rcset     = system_charset   /* Default value */;
	    char                 * in_filename  = NULL;
	    struct cached_line   * cached_lines = NULL;
	    size_t                 cached_lines_len = 0;

	    struct collected_keywords * keywords = NULL;
	    size_t                      keywords_len = 0;
	    
	    while (give_cached_lines(library_list[i]->tag,
				     &walk,&lcl,
				     &in_rcset,
				     &in_filename,
				     &cached_lines,
				     &cached_lines_len)) {
		
		const char * in_csn = get_charset_MIME_name(in_rcset);
		    
		/* Just shared reference, do not free! */
		if (cached_lines) {
		    size_t x;
		    struct collected_keywords * last = NULL;
		    
		    for (x = 0; x < cached_lines_len; x++) {

			struct string * rest_S = 
			    cs_option_value(cached_lines[x].rest,
					    cached_lines[x].restlen,
					    cached_lines[x].lineno,
					    in_filename,
					    in_rcset);

			if (cached_lines[x].keyword) {
			    /* New keyword */
			    
			    keywords =
				safe_array_realloc(keywords,
						   keywords_len+1,
						   sizeof(keywords[0]));
			    last = &keywords[keywords_len++];


			    /* bzero is defined hdrs/elm_defs.h */			    
			    bzero((void *)last,sizeof (*last));

			    last->keyword    = safe_strdup(cached_lines[x].keyword);
			    last->lcl        = lcl;
			    last->rest       = NULL;
			    last->rest_count = 0;
			    last->seen       = 0;
			    last->failed     = 0;
			}

			if (last) {
			    last->rest =
				safe_array_realloc(last->rest,
						   last->rest_count+1,
						   sizeof (last->rest[0]));
			    
			    if (rest_S) {
				
				last->rest[last->rest_count++] = rest_S;
				rest_S = NULL;
			    } else {
				last->rest[last->rest_count++] =
				    format_string(CATGETS(elm_msg_cat, 
							  MeSet,
							  MeFailedParseAsHash,
							  "# Failed to parse as charset %s"),
						  in_csn ?  in_csn : "<no MIME name>");
				last->failed = 1;
			    }
			} else
			    rc_line_ignored_message(cached_lines[x].rest,
						    cached_lines[x].restlen,
						    cached_lines[x].lineno,
						    in_filename);

			if (rest_S)
			    free_string(& (rest_S));
		    }
		}
	    }
	    
	    if (keywords) {
		size_t kw;

		FILE * commentfile =
		    print_library_header(F,
					 library_list[i]->tag,
					 default_commentfile);

		fprintf(F,"%s:\n",
			library_list[i]->tag);
		
		DPRINT(Debug,8,(&Debug, " -- library %s  -- %lu option settings (may include duplicates and invalid options)\n",
				library_list[i]->tag,
				(unsigned long)keywords_len));

		write_rc_part_cached(F,
				     library_list[i]->tag,
				     commentfile,
				     global,
				     rcset,
				     keywords,
				     keywords_len);
		
		if (commentfile)
		    fclose(commentfile);
		
		for (kw = 0; kw < keywords_len; kw++) {
		    if (keywords[kw].keyword) {
			free(keywords[kw].keyword);
			keywords[kw].keyword = NULL;
		    }

		    if (keywords[kw].rest) {
			size_t j;

			for (j = 0; j < keywords[kw].rest_count; j++)
			    if (keywords[kw].rest[j])
				free_string(& (keywords[kw].rest[j]));

			free(keywords[kw].rest);
			keywords[kw].rest = NULL;
		    }
		    keywords[kw].rest_count = 0;
		}
		
		free(keywords);
		keywords = NULL;		
	    }
	    keywords_len = 0;
	    
	}
    }
}

static void free_shared_cache1(i)
     int i;
{
    if (i < 0 || i >= library_list_count)
	panic("RC PANIC",__FILE__,__LINE__,
	      "free_shared_cache1",
	      "Bad index",0);
    
    DPRINT(Debug,35,(&Debug,
		     "free_shared_cache11: Freeing %s shared cache\n",
		     library_list[i]->tag));
    
    library_list[i]->free_shared_cache();
    
}

static void free_shared_rc_opt1(i)
     int i;
{
    if (i < 0 || i >= library_list_count)
	panic("RC PANIC",__FILE__,__LINE__,
	      "free_shared_rc_opt1",
	      "Bad index",0);

    if (library_list[i]->rc_option_count > 0) {
	int x;
	
	for (x = 0; x < library_list[i]->rc_option_count; x++) {
	    
	    if (RCTYPE_magic != library_list[i]->rc_options[x].dt_type->magic) {
		DPRINT(Debug,1,(&Debug,
				"shlib/option %d -- \"%s\" BAD, flags=%0x (library=%s)\n",
				x,library_list[i]->rc_options[x].name, 
				save_info[x].flags,
				library_list[i]->tag));
		
		panic("RC PANIC",__FILE__,__LINE__,
		      "free_shared_rc_opt1",
		      "Bad config item type",0);
	    }
	    
	    DPRINT(Debug,35,(&Debug,
			     "free_shared_rc_opt1: Freeing %s rc_options[%d]  %p %s\n",
			     library_list[i]->tag,x,
			     & library_list[i]->rc_options[x],
			     library_list[i]->rc_options[x].name));
	    
	    library_list[i]->
		rc_options[x].dt_type->free_option(& library_list[i]->rc_options[x]);
	}	
    }   
}

void free_shared_rc_options()
{
    int i;

    for (i = 0; i < library_list_count; i++) {
	if (library_list[i]->handle)
	    free_shared_rc_opt1(i);
    }
}

void free_shared_caches() 
{
    int i;

    for (i = 0; i < library_list_count; i++) {
	if (library_list[i]->handle)
	    free_shared_cache1(i);
    }
 }

struct dt_enum_shared * shared_fill_dt_enumerate_values(ptr,tag,have_code,fieldname)
     struct dt_enumerate_info *ptr;
     const char *tag;
     enum shared_have_code *have_code;
     const char * fieldname;
{
    int idx1 = locate_and_load_library(tag);
    int used_start_value = 0;   /* smallest used start_value */
    struct dt_enum_shared * have_already;
    struct dt_enum_shared * result = NULL;

    /* locate_and_load_library() may return -1 */

    DPRINT(Debug,20,(&Debug,"shared_fill_dt_enumerate_values: tag=%s, idx1=%d",
		     tag,idx1));
    
    if (idx1 < 0) {
	DPRINT(Debug,20,(&Debug,"\n"));

	if (tag_ok(tag)) {

	    DPRINT(Debug,20,(&Debug,"shared_fill_dt_enumerate_values: tag=%s ok\n",
			     tag));
	    
	    if (have_code) {
		*have_code = have_code_not;
		return NULL;
		
	    } else
		goto print_error;
	} else {
	    DPRINT(Debug,20,(&Debug,"shared_fill_dt_enumerate_values:  tag=%s invalid\n",
			     tag));

	    if (have_code)
		*have_code = have_code_invalid;

	}

	goto print_error;
    }

    /* give_impinfo() is only called if tag_ok()
       returns true, therefore if idx1 >= 0,
       tag is OK
    */
    
    if (IMPINFO_magic != library_list[idx1]->magic)
	panic("SHARED PANIC",__FILE__,__LINE__,
	      "shared_fill_dt_enumerate_values",
	      "Bad magic number (ImpInfo)",0);       
 
    if (!library_list[idx1]->handle) {
	DPRINT(Debug,20,(&Debug,"; handle=NULL"));

	if (library_list[idx1]->valid) {
	    DPRINT(Debug,20,(&Debug,"; valid\n"));

	    if (have_code) {
		*have_code = have_code_delayed;
		return NULL;
	    }	    

	} else if (library_list[idx1]->delayed) {
	    
	    DPRINT(Debug,20,(&Debug,"; delayed\n"));

	    if (have_code) {
		*have_code = have_code_delayed;
		return NULL;
	    }	    
	} else  {
	    DPRINT(Debug,20,(&Debug,"; tag ok (code not loaded)\n"));
	    
	    if (have_code) {
		*have_code = have_code_not;
		return NULL;
	    }
	}


    print_error:
	    
	if (fieldname)
	    lib_error(CATGETS(elm_msg_cat, MeSet,
			      MeInvalidEnumeratedTag1,
			      "Enumerated values %s:* ignored on key \"%s\" -- code not loaded"),
		      tag,fieldname);
	else
	    lib_error(CATGETS(elm_msg_cat, MeSet,
			      MeInvalidEnumeratedTag,
			      "Enumerated values %s:* ignored -- code not loaded"),
		      tag);

	return NULL;	
    }
    
    DPRINT(Debug,20,(&Debug,"; tag ok (code loaded)\n"));
    
    if (have_code)
	*have_code = have_code_loaded;
    
    for (have_already = ptr->first_shared;
	 have_already;
	 have_already = have_already->next_shared) {

       if (DT_ENUM_SHARED_magic != have_already->magic)
	   panic("RC PANIC",__FILE__,__LINE__,
                 "shared_fill_dt_enumerate_values",
                 "Bad magic number (dt_enum_shared)",0);

       if (have_already->start_value < used_start_value)
	   used_start_value = have_already->start_value;

       if (0 == strcmp(have_already->tag,tag))
	   panic("RC PANIC",__FILE__,__LINE__,
                 "shared_fill_dt_enumerate_values",
                 "Tag already added (dt_enum_shared)",0);

    }

    result = 
	library_list[idx1]->fill_enum_values(ptr,tag,
					     used_start_value,
					     sizeof (*ptr),
					     sizeof (*result));

    if (result) {
	
	if (DT_ENUM_SHARED_magic != result->magic)
	    panic("RC PANIC",__FILE__,__LINE__,
		  "shared_fill_dt_enumerate_values",
		  "Bad magic number (dt_enum_shared)",0);

	if (0 != strcmp(result->tag,tag))
	   panic("RC PANIC",__FILE__,__LINE__,
                 "shared_fill_dt_enumerate_values",
                 "Bad returned tag (dt_enum_shared)",0);

	if (result->start_value + result->nlen > used_start_value)
	   panic("RC PANIC",__FILE__,__LINE__,
                 "shared_fill_dt_enumerate_values",
                 "Bad returned start_value (dt_enum_shared)",0);	
    }
    
    return result;
}
/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
