static char rcsid[] = "@(#)$Id: sortlist.c,v 2.6 2021/07/15 17:30:00 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.6 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "elm_defs.h"

DEBUG_VAR(Debug,__FILE__,"core");

#define SORTLIST_magic	     0xE601

struct sortlist {
    unsigned short magic; 

    const struct sort_operation * op;

    struct sortlist_item {
	union sort_item  item;
	struct string  * debug_name;
    }                * list;
    size_t             list_len;
    size_t             alloced;
    
} * alloc_sort_list(op,prealloc)
   const struct sort_operation * op;
   size_t  prealloc;
{
    struct sortlist * sortlist
	= safe_zero_alloc(sizeof (*sortlist));

    if (SORT_OPERATION_magic != op->magic)
	panic("SORTLIST PANIC",__FILE__,__LINE__,
	      "alloc_sort_list",
              "Bad magic number (sort_operation)",0);

    sortlist->op = op;

    sortlist->list = NULL;
    sortlist->list_len = 0;
    sortlist->alloced = 0;

    if (prealloc > 0) {
	sortlist->list =
	    calloc(prealloc,sizeof (sortlist->list[0]));

	if (sortlist->list) {
	    size_t i;

	    for (i = 0; i < prealloc; i++) {
		sortlist->list[i].item.dummy = 0;
		sortlist->list[i].debug_name = NULL;
	    }
	    sortlist->alloced = prealloc;
	    
	} else {
	    int err UNUSED_VAROK = errno;
	    DPRINT(Debug,14,(&Debug,
			     "alloc_sort_list [%p]: Failed to prealloc %zu elems: %s\n",
			     sortlist,prealloc,strerror(err)));
	}
	
    }
    
    sortlist->magic = SORTLIST_magic;

    DPRINT(Debug,16,(&Debug,"alloc_sort_list=%p\n",sortlist));
    
    return sortlist;
}

static void free_sort_list_item P_((struct sortlist * sortlist,
				    size_t idx));
static void free_sort_list_item(sortlist,idx)
     struct sortlist * sortlist;
     size_t idx;
{
    if (SORTLIST_magic != sortlist->magic)
	panic("SORTLIST PANIC",__FILE__,__LINE__,
	      "free_sort_list_item",
	      "Bad magic number (sortlist)",0);

    if (SORT_OPERATION_magic !=  sortlist->op->magic)
	panic("SORTLIST PANIC",__FILE__,__LINE__,
	      "free_sort_list_item",
	      "Bad magic number (sort_operation)",0);

    if (idx >= sortlist->list_len)
	panic("SORTLIST PANIC",__FILE__,__LINE__,
	      "free_sort_list_item",
	      "Bad idx",0);

    if (sortlist->list[idx].item.dummy) {
	sortlist->op->free_item(& (sortlist->list[idx].item));
    }
    
    if (sortlist->list[idx].item.dummy) {
	DPRINT(Debug,1,(&Debug,"free_sort_list_item [%p]: Item %zu not freed",
			(*sortlist),idx));
	if (sortlist->list[idx].debug_name) {
	    DPRINT(Debug,1,(&Debug,"; debug_name=%S",
			    sortlist->list[idx].debug_name));
	}
	DPRINT(Debug,1,(&Debug,"\n"));

	free(sortlist->list[idx].item.dummy);
	sortlist->list[idx].item.dummy = NULL;
    }

    if (sortlist->list[idx].debug_name)
	free_string(& (sortlist->list[idx].debug_name));    
}


void free_sort_list(sortlist)
     struct sortlist ** sortlist;
{
    if (SORTLIST_magic != (*sortlist)->magic)
	panic("SORTLIST PANIC",__FILE__,__LINE__,
	      "free_sort_list",
	      "Bad magic number (sortlist)",0);

    if ((*sortlist)->list) {
	size_t i;

	if (SORT_OPERATION_magic !=  (*sortlist)->op->magic)
	    panic("SORTLIST PANIC",__FILE__,__LINE__,
		  "free_sort_list",
		  "Bad magic number (sort_operation)",0);
	
	for (i = 0; i < (*sortlist)->list_len; i++) {
	    free_sort_list_item(*sortlist,i);
	}

	free((*sortlist)->list);
	(*sortlist)->list = NULL;
    }
    (*sortlist)->list_len = 0;
    (*sortlist)->alloced = 0;

    (*sortlist)->magic = 0;  /* Invalidate */
    free(*sortlist);
    (*sortlist) = NULL;    
}

void prealloc_sort_list(sortlist,prealloc)
     struct sortlist * sortlist;
     size_t  prealloc;
{
    if (SORTLIST_magic != sortlist->magic)
	panic("SORTLIST PANIC",__FILE__,__LINE__,
	      "prealloc_sort_list",
	      "Bad magic number (sortlist)",0);

    if (0 == prealloc &&
	0 == sortlist->list_len) {

	if (sortlist->list) {
	    free(sortlist->list);
	    sortlist->list = 0;
	}
	sortlist->alloced = 0;
    } else {
	
	if (prealloc < sortlist->list_len)
	    prealloc = sortlist->list_len;

	if (prealloc > 0) {
	    size_t newsize;
	
	    if (safe_array_size(&newsize,
				prealloc,sizeof (sortlist->list[0]))) {
		
		struct sortlist_item * tmp =
		    realloc(sortlist->list,newsize);

		if (tmp) {
		    
		    sortlist->list = tmp;
		    if (sortlist->alloced < prealloc) {
			size_t i;
		    
			for (i = sortlist->alloced; i < prealloc; i++) {
			    sortlist->list[i].item.dummy = 0;
			    sortlist->list[i].debug_name = NULL;
			}
		    }
		    
		    sortlist->alloced = prealloc;
		    
		} else {
		    int err UNUSED_VAROK = errno;
		    DPRINT(Debug,14,(&Debug,
				     "prealloc_sort_list [%p]: Failed to prealloc %zu elems: %s\n",
				     sortlist,prealloc,strerror(err)));
		}		
	    }
	}
    }
    
}

size_t sort_list_len(sortlist)
     struct sortlist * sortlist;
{
    if (SORTLIST_magic != sortlist->magic)
	panic("SORTLIST PANIC",__FILE__,__LINE__,
	      "sort_list_len",
	      "Bad magic number (sortlist)",0);

    return  sortlist->list_len;
}

/* Moves indexes */

static void sort_list_remove_item P_((struct sortlist     * sortlist,
				      size_t                idx));
static void sort_list_remove_item(sortlist,idx)
     struct sortlist     * sortlist;
     size_t                idx;
{

    if (SORTLIST_magic != sortlist->magic)
	panic("SORTLIST PANIC",__FILE__,__LINE__,
	      "sort_list_remove_item",
	      "Bad magic number (sortlist)",0);

    if (idx >=  sortlist->list_len)
	panic("SORTLIST PANIC",__FILE__,__LINE__,
	      "sort_list_remove_item",
	      "Bad index",0);

    if (SORT_OPERATION_magic != sortlist->op->magic)
	panic("SORTLIST PANIC",__FILE__,__LINE__,
	      "sort_list_remove_item",
	      "Bad magic number (sort_operation)",0);


    free_sort_list_item(sortlist,idx);
	
    if (sortlist->list[idx].debug_name) 
	free_string(& (sortlist->list[idx].debug_name));

    if (idx+1 < sortlist->list_len) {
        size_t i;
	
	for (i = idx+1; i <  sortlist->list_len; i++) {
	    sortlist->list[i-1] = sortlist->list[i];
	    
	    bzero(& (sortlist->list[i]), sizeof sortlist->list[i]);
	    
	    sortlist->list[i].item.dummy = NULL;
	    sortlist->list[i].debug_name = NULL;
	}
    }
    
    sortlist->list_len--;
}

/* Moves indexes */

static void sort_list_insert_item P_((struct sortlist     * sortlist,
				      size_t                idx,
				      union sort_item       item /* Increments refcount */
				                                 /* must not be NULL */
				      ));
static void sort_list_insert_item(sortlist,idx,item)
     struct sortlist     * sortlist;
     size_t                idx;
     union sort_item       item /* Increments refcount */;
{
    size_t newlen;
    
    if (SORTLIST_magic != sortlist->magic)
	panic("SORTLIST PANIC",__FILE__,__LINE__,
	      "sort_list_insert_item",
	      "Bad magic number (sortlist)",0);

    /* Allow adding to end of list, when
       idx ==  sortlist->list_len
    */

    if (idx >  sortlist->list_len)
	panic("SORTLIST PANIC",__FILE__,__LINE__,
	      "sort_list_insert_item",
	      "Bad index",0);

    if (SORT_OPERATION_magic != sortlist->op->magic)
	panic("SORTLIST PANIC",__FILE__,__LINE__,
	      "sort_list_insert_item",
	      "Bad magic number (sort_operation)",0);

    newlen = sortlist->list_len+1;
    
    if (sortlist->alloced < newlen) {
	size_t newalloc = newlen;
	size_t i;
	
	sortlist->list =
	    safe_array_realloc(sortlist->list,newalloc,
			       sizeof (sortlist->list[0]));

	
	for (i = sortlist->alloced; i < newalloc; i++) {
	    sortlist->list[i].item.dummy = NULL;
	    sortlist->list[i].debug_name = NULL;
	}

	sortlist->alloced = newalloc;	
    }

    if (idx < sortlist->list_len) {
	size_t x;
	for (x = newlen-1; x > idx; x--) {
	    sortlist->list[x] = sortlist->list[x-1];
	    
	    bzero(& (sortlist->list[x-1]), sizeof sortlist->list[x-1]);
	    
	    sortlist->list[x-1].item.dummy = NULL;
	    sortlist->list[x-1].debug_name = NULL;
	}
    }

    if (idx >=  sortlist->alloced)
	panic("SORTLIST PANIC",__FILE__,__LINE__,
	      "sort_list_insert_item",
	      "idx not alloced",0);
        
    if (sortlist->list[idx].item.dummy ||
	sortlist->list[idx].debug_name)
	panic("SORTLIST PANIC",__FILE__,__LINE__,
	      "sort_list_insert_item",
	      "Slot not empty",0);
	
    sortlist->list[idx].item = item;
    sortlist->op->inc_item_refcount(sortlist->list[idx].item);
    
    sortlist->list_len = newlen;
}

/* increments refcount */
void get_sort_list_item(sortlist,op,idx,res)
     struct sortlist     * sortlist;
     enum sort_list_get_op op;
     size_t                idx;
     union sort_item     * res;
{
    int remove_it = 0;
    
    if (SORTLIST_magic != sortlist->magic)
	panic("SORTLIST PANIC",__FILE__,__LINE__,
	      "get_sort_list_item",
	      "Bad magic number (sortlist)",0);

    if (idx >=  sortlist->list_len)
	panic("SORTLIST PANIC",__FILE__,__LINE__,
	      "get_sort_list_item",
	      "Bad index",0);

    if (SORT_OPERATION_magic != sortlist->op->magic)
	panic("SORTLIST PANIC",__FILE__,__LINE__,
	      "free_sort_list",
	      "Bad magic number (sort_operation)",0);
    
    if (res && res->dummy) {
	sortlist->op->free_item(res);

	if (res->dummy) {
	    DPRINT(Debug,1,(&Debug,
			    "get_sort_list_item [%p]: item [%p: %p] not freed, or res not initalized\n",
			    sortlist,res,res->dummy));
	    
	    free(res->dummy);
	    res->dummy = NULL;
	}	
    }

	
    switch (op) {
    case sort_list_get_remove:
	remove_it = 1;

	/* FALLTHRU */

    case sort_list_get_normal:

	if (res) {
	    *res = sortlist->list[idx].item;
	    if (res->dummy) {
		sortlist->op->inc_item_refcount(*res);
	    }
	}		
    }

    if (remove_it) {
	/* Moves indexes */
	
	sort_list_remove_item(sortlist,idx);
	
	
    }        
}

/* caller must free string */
struct string  * get_sort_list_debug_name(sortlist,idx)
     struct sortlist     * sortlist;
     size_t                idx;
{
    if (SORTLIST_magic != sortlist->magic)
	panic("SORTLIST PANIC",__FILE__,__LINE__,
	      "get_sort_list_debug_name",
	      "Bad magic number (sortlist)",0);

    if (idx >= sortlist->list_len)
	panic("SORTLIST PANIC",__FILE__,__LINE__,
	      "get_sort_list_debug_name",
	      "Bad index",0);

    if (SORT_OPERATION_magic != sortlist->op->magic)
	panic("SORTLIST PANIC",__FILE__,__LINE__,
	      "get_sort_list_debug_name",
	      "Bad magic number (sort_operation)",0);

    if (sortlist->list[idx].debug_name) {
	return dup_string(sortlist->list[idx].debug_name);
    }

    if (sortlist->list[idx].item.dummy) {

	sortlist->list[idx].debug_name =
	    sortlist->op->item_name(sortlist->list[idx].item);

	if (sortlist->list[idx].debug_name) {
	    return dup_string(sortlist->list[idx].debug_name);
	}	
    }

    return NULL;
}

int search_sort_list_item(sortlist,op,key,def,res,res_idx,
			  append_need_rewrite)
     struct sortlist             * sortlist;
     enum sort_list_search_op      op;
     const union sort_key          key;
     const union sort_item_default def;
     union sort_item             * res;
     size_t                      * res_idx;
     int                         * append_need_rewrite;
{
    int             ok             = 0;
    int             r              = 0;
    struct string * key_debug_name = NULL;
    size_t          mid_index;
    int start_hint = 0;
    
    if (SORTLIST_magic != sortlist->magic)
	panic("SORTLIST PANIC",__FILE__,__LINE__,
	      "search_sort_list_item",
	      "Bad magic number (sortlist)",0);

    if (SORT_OPERATION_magic != sortlist->op->magic)
	panic("SORTLIST PANIC",__FILE__,__LINE__,
	      "search_sort_list_item",
	      "Bad magic number (sort_operation)",0);

    if (key.dummy) {
	key_debug_name =
	    sortlist->op->key_name(key);
	
	if (key_debug_name) {
	    DPRINT(Debug,14,(&Debug,
			     "search_sort_list_item [%p]: key=\"%S\"\n",
			     sortlist,key_debug_name));
	} else {
	    DPRINT(Debug,14,(&Debug,
			     "search_sort_list_item [%p]: key=[%p]\n",
			     sortlist,key.dummy));
	}
	
    } else {
	DPRINT(Debug,14,(&Debug,
			 "search_sort_list_item [%p]: key is NULL\n",sortlist));
    }


    if (res && res->dummy) {
	sortlist->op->free_item(res);

	if (res->dummy) {
	    DPRINT(Debug,1,(&Debug,
			    "search_sort_list_item [%p]: item [%p: %p] not freed, or res not initalized\n",
			    sortlist,res,res->dummy));
	    
	    free(res->dummy);
	    res->dummy = NULL;
	}	
    }

    if (sort_list_insert_hint == op && res_idx) {
	DPRINT(Debug,14,(&Debug,
			 "search_sort_list_item [%p]: Given insert hint #%zu\n",
			 sortlist,*res_idx));

	if (*res_idx < sortlist->list_len) {
	    start_hint = 1;
	} else if (sortlist->list_len > 0) {
	    mid_index = sortlist->list_len-1;
	    DPRINT(Debug,14,(&Debug,
			     "search_sort_list_item [%p]: Testing append mode\n",
			     sortlist));
	    
	    if (!sortlist->list[mid_index].item.dummy) {		
		DPRINT(Debug,1,(&Debug,
				"search_sort_list_item [%p]: No item on %zu",
				sortlist,mid_index));
		if (sortlist->list[mid_index].debug_name) {
		    DPRINT(Debug,1,(&Debug,";  name=\"%S\"\n",
				    sortlist->list[mid_index].debug_name));
		}
		DPRINT(Debug,1,(&Debug,"\n"));
		goto append_fail;
	    }

	    if (! sortlist->list[mid_index].debug_name)
		sortlist->list[mid_index].debug_name =
		    sortlist->op->item_name(sortlist->list[mid_index].item);
	    if (sortlist->list[mid_index].debug_name) {
		DPRINT(Debug,14,(&Debug,
				 "search_sort_list_item [%p]: Item #%zu = [%p] \"%S\"\n",
				 sortlist,
				 mid_index,
				 sortlist->list[mid_index].item.dummy,
				 sortlist->list[mid_index].debug_name));
	    }
	    	    
	    r = sortlist->op->cmp_key_item(key,sortlist->list[mid_index].item);

	    if (r < 0) {
		DPRINT(Debug,14,(&Debug,
				 "search_sort_list_item [%p]: append mode failed\n",
				 sortlist));
		goto append_fail;
	    } else if (r > 0) {
		DPRINT(Debug,14,(&Debug,
				 "search_sort_list_item [%p]: append mode -- after last item\n",
				 sortlist));
		goto append_ok;
	    } else {
		DPRINT(Debug,14,(&Debug,
				 "search_sort_list_item [%p]: append mode -- is last item\n",
				 sortlist));
		*res_idx = mid_index;
		start_hint = 1;
	    }
	    
	} else {
	    DPRINT(Debug,14,(&Debug,
			     "search_sort_list_item [%p]: append mode -- no items\n",
			     sortlist));

	    mid_index = 0;
	    goto append_ok;
	}
		
    } else  {
    append_fail:
	
	if (res_idx) 
	    *res_idx = sortlist->list_len;   /* Initialize to invalid value */

	if (append_need_rewrite) {
	    * append_need_rewrite = 1;
	     DPRINT(Debug,14,(&Debug,
			      "search_sort_list_item [%p]: append mode need rewrite\n",
			      sortlist));
	}
	
    }
    mid_index = sortlist->list_len;
    
	
    if (sortlist->list_len > 0) {
	/* From give_resolv_cache() */
	size_t min_index = 0;
	size_t max_index = sortlist->list_len-1;
	size_t          loop = 0;

	DPRINT(Debug,14,(&Debug,
			 "search_sort_list_item [%p]: %zu items\n",
			 sortlist,sortlist->list_len));
	
	while (min_index <= max_index) {
	    int quit = 0;

	    if (start_hint) {
		mid_index = *res_idx;
		
		DPRINT(Debug,14,(&Debug,
				 "search_sort_list_item [%p]: Using start hint %zu\n",
				 sortlist,mid_index));

		start_hint = 0;
	    } else {	    
		mid_index = (min_index + max_index) / 2;
		loop++;
	    }
		
	    DPRINT(Debug,14,(&Debug,
			     "search_sort_list_item [%p]: loop #%zu: min_index = %zu max_index=%zu => mid_index = %zu\n",
			     sortlist,loop,min_index,max_index,mid_index));
	    
	    if (mid_index >= sortlist->list_len)
		panic("SORTLIST PANIC",__FILE__,__LINE__,
		      "search_sort_list_item",
		      "Bad mid_index",0);

	    if (!sortlist->list[mid_index].item.dummy) {		
		DPRINT(Debug,1,(&Debug,
				 "search_sort_list_item [%p]: No item on %zu",
				sortlist,mid_index));
		if (sortlist->list[mid_index].debug_name) {
		    DPRINT(Debug,1,(&Debug,";  name=\"%S\"\n",
				    sortlist->list[mid_index].debug_name));
		}
		DPRINT(Debug,1,(&Debug,"\n"));
		
		break;
	    }

	    if (! sortlist->list[mid_index].debug_name)
		sortlist->list[mid_index].debug_name =
		    sortlist->op->item_name(sortlist->list[mid_index].item);
	    if (sortlist->list[mid_index].debug_name) {
		 DPRINT(Debug,14,(&Debug,
				  "search_sort_list_item [%p]: Item #%zu = [%p] \"%S\"\n",
				  sortlist,
				  mid_index,
				  sortlist->list[mid_index].item.dummy,
				  sortlist->list[mid_index].debug_name));
	    }
	    	    
	    r = sortlist->op->cmp_key_item(key,sortlist->list[mid_index].item);
	    
	    if (key_debug_name) {
		DPRINT(Debug,14,(&Debug,
				 "search_sort_list_item [%p]: key \"%S\"",
				 sortlist,key_debug_name));
	    } else {
		DPRINT(Debug,14,(&Debug,
				 "search_sort_list_item [%p]: key [%p]",
				 sortlist,key.dummy));
	    }

	    if (r < 0) {
		DPRINT(Debug,14,(&Debug," < "));

		if (mid_index > 0)
		    max_index = mid_index - 1;
		else
		    quit = 1;
	    } else if (r > 0) {
		DPRINT(Debug,14,(&Debug," > "));

		if (mid_index < sortlist->list_len)
		    min_index = mid_index + 1;
		else
		    quit = 1;
	    } else {
		DPRINT(Debug,14,(&Debug," = "));
	    }

	    DPRINT(Debug,14,(&Debug,"[%zu] ",mid_index));
	    if (sortlist->list[mid_index].debug_name) {
		DPRINT(Debug,14,(&Debug,"item \"%S\"\n",
				 sortlist->list[mid_index].debug_name));
	    } else {
		DPRINT(Debug,14,(&Debug,"item [%p]\n",sortlist->list[mid_index].item.dummy));
	    }

	    if (quit) {		
		DPRINT(Debug,14,(&Debug,
				 "search_sort_list_item [%p]: quit search at %zu\n",
				 sortlist,mid_index));
		break;
	    } else if (0 == r) {
		DPRINT(Debug,14,(&Debug,
				 "search_sort_list_item [%p]: succeed search at %zu\n",
				 sortlist,mid_index));
		
		ok = 1;
		break;
	    }	    
	}
	
	if (ok && mid_index < sortlist->list_len && sortlist->list[mid_index].item.dummy) {
	    if (res_idx)
		*res_idx = mid_index;

	    if (res)  {
		*res = sortlist->list[mid_index].item;
		if (res->dummy) {
		    sortlist->op->inc_item_refcount(*res);
		}
	    }	
	    
	    DPRINT(Debug,14,(&Debug,
			     "search_sort_list_item [%p]: Found index %zu\n",
			     sortlist,mid_index));
	} 


    } else {
	DPRINT(Debug,14,(&Debug,
			 "search_sort_list_item [%p]: No items\n",
			 sortlist));
	mid_index = 0;
    }
    
append_ok:
    switch (op) {
	union sort_item item;
    case sort_list_search_normal:
	/* Do not insert if not found */

	DPRINT(Debug,14,(&Debug,
			 "search_sort_list_item [%p]: %s\n",
			 sortlist,
			 ok ? " FOUND" : "NOT FOUND"));
	
	break;
    case sort_list_search_create:
    case sort_list_insert_hint:
	
	if (ok) {
	    DPRINT(Debug,14,(&Debug,
			     "search_sort_list_item [%p]: Found\n",
			     sortlist));
	    break;
	}

	if (0 == mid_index && 0 == sortlist->list_len) {

	    DPRINT(Debug,14,(&Debug,
			     "search_sort_list_item [%p]: Adding",
			     sortlist));
	    if (key_debug_name) {
		DPRINT(Debug,14,(&Debug,
				 " key \"%S\"",
				 key_debug_name));
	    } else {
		DPRINT(Debug,14,(&Debug,
				 " key [%p]\n",
				 key.dummy));
	    }
	    DPRINT(Debug,14,(&Debug," to beginning of empty list\n"));
	    
	} else if (r > 0 && mid_index < sortlist->list_len &&
		   sortlist->list[mid_index].item.dummy) {

	    DPRINT(Debug,14,(&Debug,
			     "search_sort_list_item [%p]: Adding",
			     sortlist));
	    if (key_debug_name) {
		DPRINT(Debug,14,(&Debug,
				 " key \"%S\"",
				 key_debug_name));
	    } else {
		DPRINT(Debug,14,(&Debug,
				 " key [%p]",
				 key.dummy));
	    }
	    DPRINT(Debug,14,(&Debug," after [%zu] ",
			     mid_index));
	    if (sortlist->list[mid_index].debug_name) {
		DPRINT(Debug,14,(&Debug,"item \"%S\"\n",
				 sortlist->list[mid_index].debug_name));
	    } else {
		DPRINT(Debug,14,(&Debug,"item [%p]\n",sortlist->list[mid_index].item.dummy));
	    }
	 
	    mid_index++;
	    
	} else if (r < 0  && mid_index < sortlist->list_len &&
		   sortlist->list[mid_index].item.dummy) {
	    
	    DPRINT(Debug,14,(&Debug,
			     "search_sort_list_item [%p]: Adding",
			     sortlist));
	    if (key_debug_name) {
		DPRINT(Debug,14,(&Debug,
				 " key \"%S\"",
				 key_debug_name));
	    } else {
		DPRINT(Debug,14,(&Debug,
				 " key [%p]",
				 key.dummy));
	    }
	    DPRINT(Debug,14,(&Debug," before [%zu] ",
			     mid_index));

	    if (sortlist->list[mid_index].debug_name) {
		DPRINT(Debug,14,(&Debug,"item \"%S\"\n",
				 sortlist->list[mid_index].debug_name));
	    } else {
		DPRINT(Debug,14,(&Debug,"item [%p]\n",sortlist->list[mid_index].item.dummy));
	    }
	    
	} else {
	    panic("SORTLIST PANIC",__FILE__,__LINE__,
		  "search_sort_list_item",
		  "Bad result on create operation",0);
	}

	if (res && res->dummy)
	    panic("SORTLIST PANIC",__FILE__,__LINE__,
		  "search_sort_list_item",
		  "res already set",0);

	ok = 1;

	item.dummy = NULL;
	
	sortlist->op->alloc_item(&item,key,def);

	sort_list_insert_item(sortlist,mid_index,item); /* Increments refcount */

	if (res) {
	    *res = item;
	    item.dummy = NULL;
	} else
	    sortlist->op->free_item(&item);
	
	break;
    case sort_list_search_remove:
	if (!ok) {
	    DPRINT(Debug,14,(&Debug,
			     "search_sort_list_item [%p]: Not found\n",
			     sortlist));
	    break;
	}

	if (mid_index < sortlist->list_len) {
	    if (sortlist->list[mid_index].item.dummy) {
		sortlist->op->free_item(& (sortlist->list[mid_index].item));
	    }
	    
	    /* Moves indexes */
	    
	    sort_list_remove_item(sortlist,mid_index);
	    
	}
	
	break;	
    }

    
    if (key_debug_name)
	free_string(&key_debug_name);

    DPRINT(Debug,14,(&Debug,
		     "search_sort_list_item [%p]=%d",
		     sortlist,ok));

    if (res) {
	if (res->dummy) {

	    DPRINT(Debug,14,(&Debug,"; res ~ [%p]",
			     res->dummy));
	    
	    /* DPRINT does equivalent of DEBUG_CHECK_INIT */
    
	    if (DEBUG_CHECK(Debug,14)) {
		struct string * S = sortlist->op->item_name(*res);

		if (S) {
		    DPRINT(Debug,14,(&Debug," \"%S\"",
				     S));
		    
		    free_string(&S);
		}		
	    }
	} else {
	    DPRINT(Debug,14,(&Debug,"; res ~ NULL"));
	}
    }
    if (res_idx) {
	DPRINT(Debug,14,(&Debug,"; res_idx ~ %zu",
			 *res_idx));
    }
    
    DPRINT(Debug,14,(&Debug,"\n"));
    
    return ok; 
}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
