static char rcsid[] = "@(#)$Id: state_in.c,v 2.7 2017/08/02 04:46:01 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.7 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI) 
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 *
 ******************************************************************************
 *  Some code copied from lib/state.c. It have following note:
 *
 *     Initially written by: Michael Elkins <elkins@aero.org>, 1995
 *****************************************************************************/

#include "elm_defs.h"
#include "state_imp.h"

DEBUG_VAR(DebugIO,__FILE__,"stateio");





/* ------------------------------------------------------ */

#define IN_STATE_file_magic  0xEA03

struct in_state_file {
    unsigned short magic;     /* IN_STATE_file_magic */

    FILE         * fpin;
    int          errno_val;
};


struct in_state * new_in_state(m)
     struct in_state_type * m;
{
    struct in_state * res = safe_zero_alloc(sizeof (*res));
        
    if (IN_STATE_type_magic != m->magic)
	state_panic(__FILE__,__LINE__,"new_in_state","Bad magic number");

    res->state_type = m;
    res->error_message = NULL;

    res->magic = STATE_in_magic;  
    
    res->refcount   = 1;

    m->init_si_type(res);

    DPRINT(DebugIO,15,(&DebugIO,
		       "new_in_state: %p => refcount=%d\n",
		       res,res->refcount));

    return res;
}

int inc_in_state_refcount(ptr)
     struct in_state *ptr;
{
    if (STATE_in_magic != ptr->magic)
	state_panic(__FILE__,__LINE__,"inc_in_state_refcount",
		    "Bad state magic number");

    ptr->refcount++;

    DPRINT(DebugIO,15,(&DebugIO,
		       "inc_in_state_refcount: %p => refcount=%d\n",
		       ptr,ptr->refcount));

    return ptr->refcount;

}

void free_in_state(ptr) 
     struct in_state **ptr;
{

    if (STATE_in_magic != (*ptr)->magic)
	state_panic(__FILE__,__LINE__,"free_in_state",
		    "Bad state magic number");

    if (IN_STATE_type_magic != (*ptr)->state_type->magic)
	state_panic(__FILE__,__LINE__,"free_in_state",
		    "Bad state type magic number");

    if ((*ptr)->refcount < 1) 
	state_panic(__FILE__,__LINE__,"free_in_state",
		    "Bad refcount");
    
    (*ptr)->refcount--;

    DPRINT(DebugIO,15,(&DebugIO,
		       "free_in_state: %p => refcount=%d\n",
		       (*ptr),(*ptr)->refcount));

    if ((*ptr)->refcount > 0) {
	/* Just remove this reference */
	
	*ptr = NULL;

	return;
    }
    
    (*ptr)->state_type->dest_si_type(*ptr);

    if ((*ptr)->error_message)
	free_string(& (*ptr)->error_message);


    (*ptr)->state_type = NULL;
    (*ptr)->magic = 0;   /* Invalidate */

    free(*ptr);
    *ptr = NULL;
}


int in_state_seekable (s)
     struct in_state *s;
{

    if (STATE_in_magic != s->magic)
	state_panic(__FILE__,__LINE__,"in_state_seekable",
		    "Bad state magic number");

    if (IN_STATE_type_magic != s->state_type->magic)
	state_panic(__FILE__,__LINE__,"in_state_seekable",
		    "Bad state type magic number");

    return s->state_type->seekable_si_type(s);
}

int state_getc (s)
     struct in_state *s;
{
    int r = EOF;

    DPRINT(DebugIO,30,(&DebugIO,
		       "state_getc(s=%p)\n",s));

    if (STATE_in_magic != s->magic)
	state_panic(__FILE__,__LINE__,"state_getc",
		    "Bad state magic number");

    if (IN_STATE_type_magic != s->state_type->magic)
	state_panic(__FILE__,__LINE__,"state_getc",
		    "Bad state type magic number");

    r = s->state_type->getc_si_type(s);
        
    DPRINT(DebugIO,30,(&DebugIO,
		       "state_getc=%d\n",
		       r));
    
    return r;
}

int state_ungetc (c,s)
     int c;
     struct in_state *s;
{
    int r = EOF;

    DPRINT(DebugIO,30,(&DebugIO,
		       "state_ungetc(s=%p)\n",s));


    if (STATE_in_magic != s->magic)
	state_panic(__FILE__,__LINE__,"state_ungetc",
		    "Bad state magic number");

    if (IN_STATE_type_magic != s->state_type->magic)
	state_panic(__FILE__,__LINE__,"state_ungetc",
		    "Bad state type magic number");


    r = s->state_type->ungetc_si_type(c,s);

    DPRINT(DebugIO,30,(&DebugIO,
		       "state_ungetc=%d\n",
		       r));
    
    return r;
}


const struct string * in_state_error_message P_((struct in_state *s,
						 int clearerr /* pop message from stack*/ ));
const struct string * in_state_error_message(s,clearerr)
     struct in_state *s;
     int clearerr /* pop message from stack*/ ;
{
    if (STATE_in_magic != s->magic)
	state_panic(__FILE__,__LINE__,"in_state_error_message",
		    "Bad state magic number");

    if (IN_STATE_type_magic != s->state_type->magic)
	state_panic(__FILE__,__LINE__,"in_state_error_message",
		    "Bad state type magic number");

    if (s->error_message)
	free_string(& (s->error_message));

    s->error_message = s->state_type->errmsg_si_type(s,clearerr);

    return s->error_message;
}


char *state_gets (dest, length, s)
     char *dest;
     int length;
     struct in_state *s;
{
    char * r = NULL;
    int gets_via_getc;

    DPRINT(DebugIO,30,(&DebugIO,
		       "state_gets(s=%p)\n",s));

    if (STATE_in_magic != s->magic)
	state_panic(__FILE__,__LINE__,"state_gets",
		    "Bad state magic number");

    if (IN_STATE_type_magic != s->state_type->magic)
	state_panic(__FILE__,__LINE__,"state_gets",
		    "Bad state type magic number");

    
    gets_via_getc = s->state_type->policy_si_type(s,
						  state_policy_gets_via_getc);

    if (! gets_via_getc) {

	r = s->state_type->gets_si_type(dest,length,s);
    
    } else {
	r = state_gets_helper(dest,length,s);
    }


    DPRINT(DebugIO,30,(&DebugIO,
		       "state_gets=%p\n",
		       r));
    
    return r;
}

int state_getl (dest, length, s)
     char *dest;
     int length;
     struct in_state *s;
{
    int r = 0;
    int gets_via_getc;

    DPRINT(DebugIO,30,(&DebugIO,
		       "state_getl(s=%p) -- magic=%d\n",s));
  
    if (STATE_in_magic != s->magic)
	state_panic(__FILE__,__LINE__,"state_getl",
		    "Bad state magic number");

    if (IN_STATE_type_magic != s->state_type->magic)
	state_panic(__FILE__,__LINE__,"state_getl",
		    "Bad state type magic number");

    gets_via_getc = s->state_type->policy_si_type(s,
						  state_policy_gets_via_getc);

    if (! gets_via_getc) {

	r = s->state_type->getl_si_type(dest,length,s);

    } else {
	r = state_getl_helper(dest, length, s);

    }


    DPRINT(DebugIO,30,(&DebugIO,
		       "state_getl=%d\n",
		       r));
    
    return r;
}


long in_state_ftell (s)
     struct in_state *s;
{
    if (STATE_in_magic != s->magic)
	state_panic(__FILE__,__LINE__,"in_state_ftell",
		    "Bad state magic number");

    if (IN_STATE_type_magic != s->state_type->magic)
	state_panic(__FILE__,__LINE__,"in_state_ftell",
		    "Bad state type magic number");
    
    return s->state_type->ftell_si_type(s);
}


int in_state_fseek (s,pos)
     struct in_state *s;
     long pos;
{
    if (STATE_in_magic != s->magic)
        state_panic(__FILE__,__LINE__,"in_state_fseek",
                    "Bad state magic number");

    if (IN_STATE_type_magic != s->state_type->magic)
        state_panic(__FILE__,__LINE__,"in_state_fseek",
                    "Bad state type magic number");

    return s->state_type->fseek_si_type(s,pos);
}

int in_state_ferror(s)
     struct in_state *s;
{
    if (STATE_in_magic != s->magic)
        state_panic(__FILE__,__LINE__,"in_state_ferror",
                    "Bad state magic number");

    if (IN_STATE_type_magic != s->state_type->magic)
        state_panic(__FILE__,__LINE__,"in_state_ferror",
                    "Bad state type magic number");

    return s->state_type->ferror_si_type(s);
}

int in_state_feof(s)
     struct in_state *s;
{
    if (STATE_in_magic != s->magic)
        state_panic(__FILE__,__LINE__,"in_state_feof",
                    "Bad state magic number");

    if (IN_STATE_type_magic != s->state_type->magic)
        state_panic(__FILE__,__LINE__,"in_state_feof",
                    "Bad state type magic number");

    return s->state_type->feof_si_type(s);
}


/* ----------------------------------------------------------------- */

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif


S_(init_si_type_f init_si_file)
static void init_si_file P_((struct in_state *)); /* Prototype */
static void init_si_file(s)
     struct in_state *s;
{
    s->u.file = safe_zero_alloc(sizeof (* s->u.file));

    s->u.file->magic = IN_STATE_file_magic;
    s->u.file->fpin  = NULL;
    s->u.file->errno_val  = 0;
}

S_(dest_si_type_f dest_si_file)
static void dest_si_file P_((struct in_state *)); /* Prototype */
static void dest_si_file(s)
     struct in_state *s;
{
    if (IN_STATE_file_magic != s->u.file->magic)
	state_panic(__FILE__,__LINE__,"dest_si_file",
		    "Bad state type magic number");

    s->u.file->fpin = NULL;

    s->u.file->magic = 0;  /* Invalidate */
    free(s->u.file);
    s->u.file = NULL;
}

S_(seekable_si_type_f seekable_si_file)
static int  seekable_si_file P_((struct in_state *s));
static int  seekable_si_file(s)
   struct in_state *s;
{
    return 1;
}

void set_in_state_file (F, s)
     FILE *F;
     struct in_state *s;
{
    if (STATE_in_magic != s->magic)
	state_panic(__FILE__,__LINE__,"set_in_state_file","Bad magic number");
    
    if (STATE_in_file != s->state_type)
	state_panic(__FILE__,__LINE__,"set_in_state_file","Bad state type");

    if (IN_STATE_file_magic != s->u.file->magic)
	state_panic(__FILE__,__LINE__,"set_in_state_file",
		    "Bad subtype magic number");	

    if (s->u.file->fpin)
	state_panic(__FILE__,__LINE__,"set_in_state_file","Already called");
    
    s->u.file->fpin = F;
}

S_(getc_si_type_f getc_si_file)
static int getc_si_file P_((struct in_state *)); /* Prototype */
static int getc_si_file(s)
     struct in_state *s;
{    
    int r;

    if (IN_STATE_file_magic != s->u.file->magic)
	state_panic(__FILE__,__LINE__,"getc_si_file",
		    "Bad subtype magic number");	

    if (s->u.file->fpin == NULL)
	state_panic(__FILE__,__LINE__,"getc_si_file","NULL filepointer");

    r = fgetc (s->u.file->fpin);

    if (EOF == r && ferror(s->u.file->fpin))  {
	int err = errno;

	DPRINT(DebugIO,10,(&DebugIO,
			   "getc_si_file(%p): errno = %d : %s\n",
			   s,err,strerror(err)));
	s->u.file->errno_val = err;
    }


    return r;
}

S_(ungetc_si_type_f ungetc_si_file)
static int ungetc_si_file P_((int,struct in_state *)); /* Prototype */
static int ungetc_si_file (c,s)
     int c;
     struct in_state *s;
{
    int r;

    if (IN_STATE_file_magic != s->u.file->magic)
	state_panic(__FILE__,__LINE__,"ungetc_si_file",
		    "Bad subtype magic number");	

    if (s->u.file->fpin == NULL)
	state_panic(__FILE__,__LINE__,"ungetc_si_file","NULL filepointer");

    r = ungetc(c,s->u.file->fpin);

    if (EOF == r) {
	int err = errno;

	DPRINT(DebugIO,10,(&DebugIO,
			   "ungetc_si_file(%p): errno = %d : %s\n",
			   s,err,strerror(err)));
	s->u.file->errno_val = err;
    }

    return r;
}


S_(gets_si_type_f gets_si_file)
static char *gets_si_file P_((char *, int, struct in_state *));
static char *gets_si_file (dest, length, s)
     char *dest;
     int length;
     struct in_state *s;
{
    char * r = NULL;

    if (IN_STATE_file_magic != s->u.file->magic)
	state_panic(__FILE__,__LINE__,"gets_si_file",
		    "Bad subtype magic number");	

    if (s->u.file->fpin == NULL)
	state_panic(__FILE__,__LINE__,"gets_si_file","NULL filepointer");

    r = fgets (dest, length, s->u.file->fpin);

    if (!r && ferror(s->u.file->fpin))  {
	int err = errno;

	DPRINT(DebugIO,10,(&DebugIO,
			   "gets_si_file(%p): errno = %d : %s\n",
			   s,err,strerror(err)));
	s->u.file->errno_val = err;
    }

    return r;
}

S_(getl_si_type_f getl_si_file)
static int getl_si_file P_((char *, int, struct in_state *));
static int getl_si_file (dest, length, s)
     char *dest;
     int length;
     struct in_state *s;
{
    int r;

    if (IN_STATE_file_magic != s->u.file->magic)
	state_panic(__FILE__,__LINE__,"getl_si_file",
		    "Bad subtype magic number");

    if (s->u.file->fpin == NULL)
	state_panic(__FILE__,__LINE__,"getl_si_file","NULL filepointer");
    
    r = mail_gets (dest, length, s->u.file->fpin);

    if (r <= 0 && ferror(s->u.file->fpin))  {
	int err = errno;

	DPRINT(DebugIO,10,(&DebugIO,
			   "getl_si_file(%p): errno = %d : %s\n",
			   s,err,strerror(err)));
	s->u.file->errno_val = err;
    }

    return r;
}

S_(ftell_si_type_f ftell_si_file)
static long ftell_si_file P_((struct in_state *s));
static long ftell_si_file(s)
     struct in_state *s;
{
    long r;

    if (IN_STATE_file_magic != s->u.file->magic)
	state_panic(__FILE__,__LINE__,"ftell_si_file",
		    "Bad subtype magic number");	
    
    if (s->u.file->fpin == NULL)
	state_panic(__FILE__,__LINE__,"ftell_si_file","NULL file pointer");
    
    r = ftell(s->u.file->fpin);    

    if (-1 == r && ferror(s->u.file->fpin))  {
	int err = errno;

	DPRINT(DebugIO,10,(&DebugIO,
			   "ftell_si_file(%p): errno = %d : %s\n",
			   s,err,strerror(err)));
	s->u.file->errno_val = err;
    }

    return r;
}

S_(fseek_si_type_f fseek_si_file)
static int fseek_si_file P_((struct in_state *s, long pos));
static int fseek_si_file(s,pos)
     struct in_state *s; 
     long pos;
{
    int r;

    if (IN_STATE_file_magic != s->u.file->magic)
	state_panic(__FILE__,__LINE__,"fseek_si_file",
		    "Bad subtype magic number");	

    if (s->u.file->fpin == NULL)
        state_panic(__FILE__,__LINE__,"fseek_si_file",
		    "NULL file pointer");

    r = fseek(s->u.file->fpin,pos,SEEK_SET);
    if (-1 == r)  {
	int err = errno;

	DPRINT(DebugIO,10,(&DebugIO,
			   "fseek_si_file(%p): errno = %d : %s\n",
			   s,err,strerror(err)));
	s->u.file->errno_val = err;
    }

    return r;

}

/* Caller frees result, NULL if no error */
S_(errmsg_si_type_f errmsg_si_file)
struct string * errmsg_si_file P_((struct in_state *s,
				   int clearerr /* pop message from stack*/ ));
struct string * errmsg_si_file(s,clearerr)
     struct in_state *s;
     int clearerr /* pop message from stack*/;
{
    struct string * S = NULL;

    if (IN_STATE_file_magic != s->u.file->magic)
	state_panic(__FILE__,__LINE__,"errmsg_si_file",
		    "Bad subtype magic number");	

    if (s->u.file->fpin &&
	! ferror(s->u.file->fpin)) {
	
	s->u.file->errno_val = 0;
    }

    if (! s->u.file->errno_val)
	return NULL;

    S = format_string(FRM("%s"),
		      strerror(s->u.file->errno_val));

    s->u.file->errno_val = 0;

    return S;
}

S_(policy_si_type_f policy_si_file)
static int policy_si_file P_((struct in_state *s, 
			      enum state_policy question));
static int policy_si_file(s,question)
     struct in_state *s;
     enum state_policy question;
{
    switch (question) {
    case state_policy_gets_via_getc: return 0;

    default:
	panic("STATE PANIC",__FILE__,__LINE__,"policy_si_file",
	      "Bad question",0);
    }

    return 0;
}

S_(ferror_si_type_f ferror_si_file)
static int ferror_si_file P_((struct in_state *s));
static int ferror_si_file(s)
     struct in_state *s;
{
    if (IN_STATE_file_magic != s->u.file->magic)
	state_panic(__FILE__,__LINE__,"ferror_si_file",
		    "Bad subtype magic number");

    if (s->u.file->fpin)
	return ferror(s->u.file->fpin);

    return 0;
}


S_(feof_si_type_f feof_si_file)
static int feof_si_file P_((struct in_state *s));
static int feof_si_file(s)
     struct in_state *s;
{
    if (IN_STATE_file_magic != s->u.file->magic)
	state_panic(__FILE__,__LINE__,"feof_si_file",
		    "Bad subtype magic number");

    if (s->u.file->fpin)
	return feof(s->u.file->fpin);

    return 1;
}



struct in_state_type STATE_in_file_type = {
    IN_STATE_type_magic,

    init_si_file,
    dest_si_file,
    seekable_si_file,
    getc_si_file,
    ungetc_si_file,
    gets_si_file,
    getl_si_file,
    ftell_si_file,
    fseek_si_file,
    errmsg_si_file,
    policy_si_file,
    ferror_si_file,
    feof_si_file
};


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
