static char rcsid[] = "@(#)$Id: state_out.c,v 2.18 2023/12/13 16:55:32 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.18 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                       (was hurtta+elm@posti.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 *
 *
 *  
 *  Some code copied from lib/state.c. It have following note:
 *
 *     Initially written by: Michael Elkins <elkins@aero.org>, 1995
 *****************************************************************************/

#include "elm_defs.h"
#include "state_imp.h"
#include "pg_range_imp.h"

/* Needs STRING_magic */
#include "elmme-magic.h"
#include "pg_params_imp.h"

static char *us2s P_((unsigned char *str));
static char *us2s(str)
     unsigned char *str;
{
    return (char *)str;
}

static const unsigned char *cs2us P_((const char *str));
static const unsigned char *cs2us(str)
     const char *str;
{
    return (const unsigned char *)str;
}


DEBUG_VAR(Debug,__FILE__,"misc");
DEBUG_VAR(DebugIO,__FILE__,"stateio");

#define OUT_STATE_file_magic 0xEB06

struct out_state_file {
    unsigned short magic;     /* OUT_STATE_file_magic */

    FILE *fpout;
};   

#define MAX_FILTER_line   4000
#define MIN_FILTER_line   1040

struct out_state * new_out_state(m)
     struct out_state_type *m;
{
    struct out_state * res = safe_zero_alloc(sizeof (*res));
    static charset_t default_vector[2];
    
    default_vector[0] = display_charset;
    default_vector[1] = NULL;

    if (OUT_STATE_type_magic != m->magic)
	state_panic(__FILE__,__LINE__,"new_out_state","Bad magic number");

    res->state_type = m;
    res->magic = STATE_out_magic;  

    res->EOLN_is_CRLF    = 0;         /* EOLN is \n */
    res->filter          = NULL;
    res->filter_on_BOL   = 1;         /* Beginning of line */
    res->filter_on_BOR   = 0;         /* no beginning of range */
    
    res->pg_flags        = 0;
    res->pager_range     = NULL;
    res->display_charset = default_vector;
    res->filter_line     = NULL;
    res->u.dummy         = NULL;

    res->state_type->init_so_it(res);

    return res;
}

static void flush_filter P_((struct out_state *s, int munge_eoln,
			     int force_eoln));


void free_out_state(ptr) 
     struct out_state **ptr;
{
    if (STATE_out_magic != (*ptr)->magic)
	state_panic(__FILE__,__LINE__,"free_out_state","Bad magic number");

    if (OUT_STATE_type_magic != (*ptr)->state_type->magic)
	state_panic(__FILE__,__LINE__,"free_out_state",
		    "Bad type magic number");

    if ((*ptr)->filter_line)
	flush_filter(*ptr,0,0);
    
    
    (*ptr)->state_type->dest_so_it(*ptr);


    (*ptr)->filter          = NULL;
    (*ptr)->display_charset = NULL;

    if ((*ptr)->filter_line)
	free_string(&((*ptr)->filter_line));

    if ((*ptr)->pager_range)
	free_pager_range(& ((*ptr)->pager_range));

    (*ptr)->magic      = 0; /* No longer initialized */

    free(*ptr);
    *ptr = NULL;
}


int  out_state_seekable(s)
     struct out_state *s;
{
    int r = 0;

    DPRINT(DebugIO,30,(&DebugIO,
		       "out_state_seekable(s=%p)\n",s));

    if (STATE_out_magic != s->magic)
	state_panic(__FILE__,__LINE__,"out_state_seekable","Bad magic number");

    if (OUT_STATE_type_magic != s->state_type->magic)
	state_panic(__FILE__,__LINE__,"out_state_seekable",
		    "Bad type magic number");

    r = s->state_type->seekable_so_it(s);
    
    DPRINT(DebugIO,30,(&DebugIO,
		       "out_state_seekable=%d\n",r));
    
    return r;
}

/* Returns NULL if no available ... */
FILE *out_state_FILE(s)
     struct out_state *s;
{
    FILE * r = NULL;

    DPRINT(DebugIO,30,(&DebugIO,
		       "out_state_FILE(s=%p)\n",s));

    if (STATE_out_magic != s->magic)
	state_panic(__FILE__,__LINE__,"out_state_FILE","Bad magic number");

    if (OUT_STATE_type_magic != s->state_type->magic)
	state_panic(__FILE__,__LINE__,"out_state_FILE",
		    "Bad type magic number");

    if (s->filter_line)
	flush_filter(s,0,0);

    r = s->state_type->FILE_so_it(s);

    DPRINT(DebugIO,30,(&DebugIO,
		       "out_state_FILE=%p\n",
		       r));
    
    return r;
}


int  out_state_fseek(s,pos)
     struct out_state *s;
     long pos;
{ 
    int r = -1;

    DPRINT(DebugIO,30,(&DebugIO,
		       "out_state_fseek(s=%p,pos=%ld\n",s,pos));

    if (STATE_out_magic != s->magic)
	state_panic(__FILE__,__LINE__,"out_state_fseek","Bad magic number");

    if (OUT_STATE_type_magic != s->state_type->magic)
	state_panic(__FILE__,__LINE__,"out_state_fseek",
		    "Bad type magic number");

    if (s->filter_line)
	flush_filter(s,0,0);
   
    r = s->state_type->seek_so_it(s,pos);

    if (r != -1)
	s->filter_on_BOL = 1;   /* Assume beginning of line */

    DPRINT(DebugIO,30,(&DebugIO,
		       "out_state_fseek=%p\n",
		       r));
    return r;
}

long out_state_ftell (s)
     struct out_state *s;
{ 
    long r = -1;

    DPRINT(DebugIO,30,(&DebugIO,
		       "out_state_ftell(s=%p)\n",s));


    if (STATE_out_magic != s->magic)
	state_panic(__FILE__,__LINE__,"out_state_ftell","Bad magic number");

    if (OUT_STATE_type_magic != s->state_type->magic)
	state_panic(__FILE__,__LINE__,"out_state_ftell",
		    "Bad type magic number");

    if (s->filter_line)
	flush_filter(s,0,0);

    r = s->state_type->ftell_so_it(s);

    DPRINT(DebugIO,30,(&DebugIO,
		       "out_state_ftell=%ld\n",
		       r));
    return r;
}

static void flush_filter(s, munge_eoln,force_eoln)
     struct out_state *s;
     int munge_eoln;
     int force_eoln;   
{
    int len UNUSED_VAROK = string_len(s->filter_line);
    int flush_via_put;
    
    DPRINT(DebugIO,30,(&DebugIO,
		       "flush_filter: munge_eoln=%d, len=%d%s%s\n",
		       munge_eoln,len,
		       s->filter_on_BOL ? "; filter on BOL" : "",
		       s->filter_on_BOR ? "; filter on beginning of range" : ""
		       ));

    if (STATE_out_magic != s->magic)
	state_panic(__FILE__,__LINE__,"flush_filter",
		    "Bad magic number (out_state)");

    if (OUT_STATE_type_magic != s->state_type->magic)
	state_panic(__FILE__,__LINE__,"flush_filter",
		    "Bad type magic number");
    
    flush_via_put = s->state_type->policy_so_it(s,
						state_policy_flush_via_put);
    
    if (!flush_via_put) {

	s->state_type->flush_filter_so_it(s,munge_eoln,
					  s->pg_flags,
					  s->pager_range,
				          force_eoln);

    } else {
	char *s1;
	int l;
	/* s->display_charset[0] is either display_charset (internal pager)
	 * or system_charset (external pager)
	 */
	struct string * tmp;	
	struct string * res = s->filter_line;
	int eoln = 0, len;

	DPRINT(DebugIO,30,(&DebugIO,
			   "flush_filter: Flushing via put\n"));
	
	if (s->pager_range)    /* Does reply quotation  and [ escaping */
	    res = pager_range_change_line(s->pager_range,s->filter_line,
					  s->filter_on_BOL,
					  s->filter_on_BOR);
	/* changes res, does not free original pointer */
	
	if ((len = string_len(res)) > 0) {   /* Check LF from end */
	    uint16 c = give_unicode_from_string(res,len-1);
	    
	    if (0x000A == c)
		eoln = 1;
	}

	if (force_eoln && !eoln && (len > 0 || !s->filter_on_BOL)) {
	    int x = 0;
	    uint16 newline[2];

	    if (s->EOLN_is_CRLF)
		newline[x++] = 0x00D /*   CR  '\r' */;
	    newline[x++] = 0x000A /*   LF  '\n' */;

	    DPRINT(DebugIO,10,(&DebugIO,
			       "flush_filter: Adding newline to filter\n"));

	    add_unicode_to_string(res,x,newline);
	    len += x;
	}

	tmp = convert_string(s->display_charset[0],
			     res,0);
	
	if (res != s->filter_line)
	    free_string(&res);

	/* Can't ask just printable characters because we need
	   NLs also ...
	*/
	s1 = us2s(stream_from_string(tmp,0,NULL));
	l = strlen(s1);

	if (munge_eoln) {
	    int size = l + 4;

	    s1 = safe_realloc(s1,size);
		    
	    /* Take care of CRLF => LF conversion or
	       LF -> CRLF conversion 
	    */
	    state_convert_EOLN(s1,&l,size,s);	    
	}
	
	s->state_type->put_so_it(s,s1,l);

	if (len > 0)
	    s->filter_on_BOL = eoln;

	free(s1);
	free_string(&tmp);
    }

    free_string(&(s->filter_line));	

    if (s->filter_on_BOL) {
	s->pg_flags = 0;   /* Reset flags */
	if (s->pager_range)
	    free_pager_range(& (s->pager_range));
    } else {
	DPRINT(DebugIO,10,(&DebugIO,
			   "flush_filter: Not beginning of line after flush\n"));
    }

    s->filter_on_BOR = 0;
}

int state_puts (string, state)
     const char *string;
     struct out_state *state;
{
    return state_put(string,strlen(string),state);
}


void state_nlputs(string,state)
     char *string;
     struct out_state *state;
{
    int i;
    int last_start = 0;

    for (i = 0; string[i]; i++) {
	switch (string[i]) {
	case '\n':
	    if (last_start < i) {
		int x = i;
		
		if ('\r' == string[x-1])
		    x--;
		if (last_start < x) {
		    state_put(string + last_start, x - last_start, state);
		}
	    }
	    if (state->EOLN_is_CRLF)
		state_puts("\r\n",state);
	    else
		state_puts("\n",state);
	    last_start = i+1;
	    break;
	}
    }

    /* Print rest of data in case there is no \n on end ... */
    if (last_start < i) {
	state_put(string + last_start, i - last_start, state);
    }
}

static void flush_filter_charset1 P_((struct out_state *s, int munge_eoln,
				      charset_t cs));
static void flush_filter_charset1(s,munge_eoln,cs)
     struct out_state *s; 
     int munge_eoln;
     charset_t cs;
{
    if (s->filter_line &&
	cs != get_string_type(s->filter_line)) {

	const char * filter_line_MIME_name  UNUSED_VAROK = 
	    get_string_MIME_name(s->filter_line);
	const char * filter_MIME_name       UNUSED_VAROK = 
	    get_charset_MIME_name(cs);

	int len  UNUSED_VAROK = string_len(s->filter_line);
	

	DPRINT(Debug,10,(&Debug,
			 "Changing charset on current line from charset %s to charset %s, flushing line (%d characters)\n",
			 filter_line_MIME_name ? 
			 filter_line_MIME_name :
			 "<not set>",
			 filter_MIME_name ? filter_MIME_name : "<not set>",
			 len));

	flush_filter(s, munge_eoln, 0);

	if (s->filter_line) 
	    state_panic(__FILE__,__LINE__,"flush_filter_charset1",
			"filter_line not reset");

	s->filter_line = new_string(cs);
    }
}

static int state_putstring0 P_((const struct string *S,
				struct out_state *st));

static int state_putstring0(S,st)
     const struct string *S; 
     struct out_state *st; 
{
    int len, i, first = 0,l=0;
    
    if (STATE_out_magic != st->magic)
	state_panic(__FILE__,__LINE__,"state_putstring0","Bad magic number");

    if (OUT_STATE_type_magic != st->state_type->magic)
	state_panic(__FILE__,__LINE__,"state_putstring0",
		    "Bad type magic number");

    if (st->filter_line &&
	(l = string_len(st->filter_line)) >= MAX_FILTER_line) {

	DPRINT(DebugIO,10,(&DebugIO,
			   "state_putstring0: filter line %d chars (max %d), flushing\n",
			   l,MAX_FILTER_line));
	
	flush_filter(st, 0, 0);	    
    } else
	flush_filter_charset1(st, 0, get_string_type(S));
       
    len = string_len(S);
	
    for (i = 0; i < len; i++) {
	uint16 u = give_unicode_from_string(S,i);

	 if (0x000A /* LF  '\n' */ == u ||
	     (0x0020 /* SPACE */    == u &&
	      l + i-first+1 >= MIN_FILTER_line)) {
	     struct string * S1 = clip_from_string(S,&first,i-first+1);

	     append_string(& (st->filter_line),S1,0);

	     free_string(&S1);

	     flush_filter(st, 0,0); 
	     first = i+1;
	     l = 0;
	 }
    }

    if (first < len) {
	struct string * S1 = clip_from_string(S,&first,len-first);

	append_string(& (st->filter_line),S1,0);

	free_string(&S1);
    }

    return len;
}


int state_putstring(S,st)
     const struct string *S; 
     struct out_state *st; 
{    
    int len;


    
    DPRINT(DebugIO,30,(&DebugIO,
		       "state_putstring(S=%p,st=%p)\n",
		       S,st));

    len =  state_putstring0(S,st);
    

    DPRINT(DebugIO,30,(&DebugIO,
		       "state_putstring=%d (filter)\n",len));

    return len;
}

static void state_force_filter_line0 P_((struct out_state *st));
static void state_force_filter_line0(st)
     struct out_state *st;
{
    DPRINT(DebugIO,30,(&DebugIO,
		       "state_force_filter_line0(st=%p): adding filter_line\n",
		       st));
    
    if (!st->filter) {
	/* STATE_out_dir also requires s->display_charset[0]
	   -- value RAW_BUFFER indicates raw copying ... 
	*/
	st->filter_line = new_string(st->display_charset[0]);
    } else
	st->filter_line = new_string(st->filter);
}

static void state_force_filter_line P_((struct out_state *st));
static void state_force_filter_line(st)
     struct out_state *st;
{
    if (!st->filter_line) 
	state_force_filter_line0(st);	      
}

static void state_putunicode0 P_((int len,
				  const uint16 *vector,
				  struct out_state *st));

static void state_putunicode0(len,vector,st)
     int len;
     const uint16 *vector; 
     struct out_state *st; 
{
    int i, first = 0;

    if (STATE_out_magic != st->magic)
	state_panic(__FILE__,__LINE__,"state_putunicode0",
		    "Bad magic number");

    
    for (i = 0; i < len; i++) {
	if (0x000A /* LF  '\n' */ == vector[i]) {
	    state_force_filter_line(st);

	    /* Does not return status indicator */
	    add_unicode_to_string(st->filter_line,i-first+1,vector+first);
	    flush_filter(st, 0,0);
	    first = i+1;
	}
    }

    if (first < len) {
	state_force_filter_line(st);
	
	/* Does not return status indicator */
	add_unicode_to_string(st->filter_line, len-first,vector+first);
	
    }    
}

int state_putunicode(len,vector,st)
     int len;
     const uint16 *vector; 
     struct out_state *st; 
{
    DPRINT(DebugIO,30,(&DebugIO,
		       "state_putunicode(len=%d,vector=%p,st=%p)\n",
		       len,vector,st));

    state_putunicode0(len,vector,st);
    

    DPRINT(DebugIO,30,(&DebugIO,
		       "state_putunicode=%d (filter)\n",len));

    return len;
}

static void flush_filter_charset P_((struct out_state *s, int munge_eoln));
static void flush_filter_charset(s,munge_eoln)
     struct out_state *s; 
     int munge_eoln;
{
    if (s->filter)
	flush_filter_charset1(s,munge_eoln,s->filter);
}

int state_putentity(C,st)
     struct out_entity *C;
     struct out_state *st;
{
    int ret = 0;
    int convert_entity;

    
    DPRINT(DebugIO,30,(&DebugIO,
		       "state_putentity(C=%p,st=%p)\n",
		       C,st));
    
    if (STATE_out_magic != st->magic)
	state_panic(__FILE__,__LINE__,
		    "state_putentity",
		    "Bad magic number (out_state)");

    if (OUT_STATE_type_magic != st->state_type->magic)
	state_panic(__FILE__,__LINE__,"state_putentity",
		    "Bad type magic number");

    if (OUT_ENTITY_magic != C->magic)
	state_panic(__FILE__,__LINE__,
		    "state_putentity",
		    "Bad magic number (out_entity)");

    convert_entity =  st->state_type->policy_so_it(st,
						   state_policy_convert_entity);

    if (!convert_entity) {
	enum sout_entity_result r1;

	if (st->filter_line)
	    flush_filter(st,0,0);


	r1 = st->state_type->putentity_so_it(st,C);

	DPRINT(DebugIO,30,(&DebugIO,"state_putentity: ->putentity_so_it=%d",
			   r1));

	switch (r1) {
	case sout_entity_reference_key:
	    DPRINT(DebugIO,30,(&DebugIO," sout_entity_reference_key\n"));
	    goto print_reference_key;
	case sout_entity_fallback:
	    DPRINT(DebugIO,30,(&DebugIO," sout_entity_fallback\n"));

	    goto fallback;
	case sout_entity_failed:
	    DPRINT(DebugIO,30,(&DebugIO," sout_entity_failed"));

	    if (0) {
	    case sout_entity_succeed:
		DPRINT(DebugIO,30,(&DebugIO," sout_entity_succeed"));
	    }
	    DPRINT(DebugIO,30,(&DebugIO,"\n"));
	    
	    ret = r1;
	    goto done;
	}
	
	       
	if (r1 > 0)
	    ret = r1;
	
    } else {
        
    fallback:
	DPRINT(DebugIO,30,(&DebugIO,"state_putentity: Using fallback\n"));
	
	if (C->entity_value) {

	    if (st->filter) {
		int len = string_len(C->entity_value);
		int i;

		for (i = 0; i < len; i++) {
		    uint16 c = give_unicode_from_string(C->entity_value,i);
		    enum charset_unicode_stat x;
		    
		    if (UNICODE_BAD_CHAR == c)
			goto print_reference_key;

		    x = string_type_have_unicode(st->filter,
						 c);

		    switch (x) {
		    case charset_unicode_bad:
		    case charset_unicode_unknown:
		    case charset_missing_unicode:
			goto print_reference_key;
		    case charset_have_unicode:
			break;			
		    }		    
		}

		flush_filter_charset(st,0);

		ret = state_putstring0(C->entity_value,st);
		goto done;
	    } else {
		/* XXXX is this correct? */
		flush_filter_charset1(st, 0,
				      get_string_type(C->entity_value));

		ret = state_putstring0(C->entity_value,st);
		goto done;
	    }	    	    	    
	}

	if (UNICODE_BAD_CHAR != C->unicode_value) {

	    if (st->filter) {
		enum charset_unicode_stat x =
		    string_type_have_unicode(st->filter,C->unicode_value);
		
		switch (x) {
		case charset_unicode_bad:
		case charset_unicode_unknown:
		case charset_missing_unicode:
		    goto print_reference_key;
		case charset_have_unicode:
		    break;			
		}		    

		flush_filter_charset(st,0);
		state_putunicode0(1,&( C->unicode_value),st);
		ret = 1;
		goto done;
	    } else {
		enum charset_unicode_stat x;

		state_force_filter_line(st);

		x = string_type_have_unicode(get_string_type(st->filter_line),
					     C->unicode_value);

		switch (x) {
		case charset_unicode_bad:
		case charset_unicode_unknown:
		case charset_missing_unicode:
		    goto print_reference_key;
		case charset_have_unicode:
		    break;			
		}
		
		state_putunicode0(1,&( C->unicode_value),st);
		ret = 1;
		goto done;   
	    }
	}
	
    print_reference_key:
	if (C->reference_key) {
	    int support_pg_flags 
		=  st->state_type->policy_so_it(st,
						state_policy_support_pg_flags);
	    int pg_flags = st->pg_flags;

	    if (st->filter_line) {
		DPRINT(DebugIO,30,(&DebugIO,
				   "state_putentity: Flushing line\n"));
		flush_filter(st,0,0);
	    }

	    if (support_pg_flags)		
		st->pg_flags = pg_flags | C->reference_key_pg_flags;

	    

	    ret = state_putstring0(C->reference_key,st);

	    if (st->filter_line) {
		DPRINT(DebugIO,30,(&DebugIO,
				   "state_putentity: Flushing line\n"));
		flush_filter(st,0,0);
	    }
	    st->pg_flags = pg_flags;
	}
    }

 done:
    DPRINT(DebugIO,30,(&DebugIO,
			"state_putentity=%d\n",ret));
    
	
    return ret;
}




int state_putstchar(C,st)
     struct charset_state *C; 
     struct out_state *st;
{
    int use_filter = 0;
    int ch;

    DPRINT(DebugIO,30,(&DebugIO,
		       "state_putstchar(C=%p,st=%p)\n",
		       C,st));

    if (STATE_out_magic != st->magic)
	state_panic(__FILE__,__LINE__,"state_putstchar","Bad magic number");

    if (OUT_STATE_type_magic != st->state_type->magic)
	state_panic(__FILE__,__LINE__,"state_putstchar",
		    "Bad type magic number");


    use_filter = st->state_type->policy_so_it(st,
					      state_policy_putc_use_filter);

    ch = state_is_onebyte(C);

    if (ch && !st->filter && st->filter_line &&
	!st->pg_flags && !st->pager_range) {
	int flush_via_put = st->state_type->policy_so_it(st,
							 state_policy_flush_via_put);

	if (flush_via_put)
	    flush_filter(st, 0,0);
    } 

    if (!use_filter && ch && !st->filter_line && !st->filter &&
	!st->pg_flags && !st->pager_range) {
	int r = st->state_type->putc_so_it(st,ch);

	DPRINT(DebugIO,30,(&DebugIO,
			   "state_putstchar=%d\n",
			   r));

	return r;
    } else {
	int l;

	if (st->filter_line &&
	    (l = string_len(st->filter_line)) >= MAX_FILTER_line) {

	    DPRINT(DebugIO,10,(&DebugIO,
			       "state_putstchar: filter line %d chars (max %d), flushing\n",
			       l,MAX_FILTER_line));

	    flush_filter(st,0,0);	    
	} else
	    flush_filter_charset1(st, 0, get_state_charset(C));
      
	if (!st->filter_line) {
	    DPRINT(DebugIO,15,(&DebugIO,"state_putstchar: New filter line\n"));
	    st->filter_line = new_string(get_state_charset(C));
	}

	/* add_state_to_string() does not return status */

	add_state_to_string(st->filter_line,C);

	/* SPACE is assumed to be on same character position */
	if (' ' == ch &&  
	    (l = string_len(st->filter_line)) >= MIN_FILTER_line) {

	    DPRINT(DebugIO,10,(&DebugIO,
			       "state_putstchar: filter line %d chars (min %d), flushing\n",
			       l,MIN_FILTER_line));

	    flush_filter(st,0,0);	    
	}

	DPRINT(DebugIO,30,(&DebugIO,
			   "state_putstchar=1 (status not available)\n"));

	return 1;
    }
} 

/* convert one line -- *len is assumed to point end of line */
void state_convert_EOLN(buffer,len,buffer_size,st)
     char *buffer; 
     int *len; 
     int buffer_size;
     struct out_state *st;
{
    int buf_len = *len;

    if (st->EOLN_is_CRLF) {

	if (buf_len > 0 && buffer[buf_len - 1] == '\n' &&
	    (buf_len < 2 || buffer[buf_len - 2] != '\r')) {

	    if (buf_len < buffer_size -2) {
		buffer[buf_len - 1] = '\r';
		buffer[buf_len] = '\n';
		buffer[buf_len+1] = '\0';
		buf_len++;

		DPRINT(Debug,25,(&Debug,
				 "state_convert_EOLN: '%.10s'... LF -> CRLF\n",
				 buffer));

	    } else {
		DPRINT(Debug,2,(&Debug,
				"state_convert_EOLN: '%.10s'... No space (size=%d, len=%d) for LF -> CRLF conversion\n",
				buffer,buffer_size,buf_len));
	    }
	}

    } else {

	/* Take care of CRLF => LF conversion */
	if (buf_len > 1 &&
	    buffer[buf_len - 1] == '\n' &&
	    buffer[buf_len - 2] == '\r') {
	    buffer[buf_len - 2] = '\n';
	    buffer[buf_len - 1] = '\0';
	    buf_len--;

	    DPRINT(Debug,25,(&Debug,
			     "state_convert_EOLN: '%.10s'... CRLF -> LF\n",
			     buffer));

	}

    }

    if (*len != buf_len) {
	DPRINT(Debug,25,(&Debug,
			 "state_convert_EOLN- len=%d -> %d\n",
			 *len,buf_len));
    }

    *len = buf_len;
}

int state_putc (ch, s)
     int ch;                        /* unsigned char assumed */
     struct out_state *s;
{    
    int use_filter = 0;

    DPRINT(DebugIO,30,(&DebugIO,
		       "state_putc(ch=%d,s=%p)\n",ch,s));


    if (STATE_out_magic != s->magic)
	state_panic(__FILE__,__LINE__,"state_putc","Bad magic number");

    if (OUT_STATE_type_magic != s->state_type->magic)
	state_panic(__FILE__,__LINE__,"state_putc",
		    "Bad type magic number");

    use_filter = s->state_type->policy_so_it(s,
					     state_policy_putc_use_filter);

    if (!s->filter && s->filter_line &&
	!s->pg_flags && !s->pager_range) {
	int flush_via_put = s->state_type->policy_so_it(s,
							state_policy_flush_via_put);

	if (flush_via_put)
	    flush_filter(s,0,0);
    } 

    if (s->filter_line || s->filter || use_filter || s->pg_flags ||
	s->pager_range) {

	int r,l;

	if (s->filter_line &&
	    (l = string_len(s->filter_line)) >= MAX_FILTER_line) {

	    DPRINT(DebugIO,10,(&DebugIO,
			       "state_putc: filter line %d chars (max %d), flushing\n",
			       l,MAX_FILTER_line));

	    flush_filter(s,0,0);	    
	}

	/* ch is assumed to be with charset s->filter */

	if (!s->filter_line) {
	    state_force_filter_line0(s);	    
	} else
	    flush_filter_charset(s,0);

	r = add_streambyte_to_string(s->filter_line,ch);
	if (!r) {
	    uint16 BAD = UNICODE_BAD_CHAR;
	    const char * filter_line_MIME_name  UNUSED_VAROK = 
		get_string_MIME_name(s->filter_line);

	    DPRINT(Debug,10,(&Debug,
			     "state_putc: failed to add byte %d to charset %s\n",
			     ch,
			     filter_line_MIME_name ? 
			     filter_line_MIME_name :
			     "<not set>"));

	    /* Indicate bad character */
	    add_unicode_to_string(s->filter_line,1,&BAD);
	}

	if ('\n' == ch) {
	    flush_filter(s,0,0); 
	} else if (' ' == ch &&  
		   (l = string_len(s->filter_line)) >= MIN_FILTER_line) {

	    DPRINT(DebugIO,10,(&DebugIO,
			       "state_putc: filter line %d chars (min %d), flushing\n",
			       l,MIN_FILTER_line));

	    flush_filter(s, 0,0);	    
	}
	
	DPRINT(DebugIO,30,(&DebugIO,
			   "state_putc=%d (filter)\n",
			   r ? 1 : EOF));

	return r ? 1 : EOF;

    } else {
	
	int r = EOF;

	r = s->state_type->putc_so_it(s,ch);
	
	if (r != -1)
	    s->filter_on_BOL = ('\n' == ch);

	DPRINT(DebugIO,30,(&DebugIO,
			   "state_putc=%d\n",
			   r));

	return r;
    }
}

int state_put (string, len, s)
     const char *string;
     int len;
     struct out_state *s;
{
    int use_filter = 0;

    DPRINT(DebugIO,30,(&DebugIO,
		       "state_put(string=%p,len=%d,s=%p)\n",
		       string,len,s));

    if (STATE_out_magic != s->magic)
	state_panic(__FILE__,__LINE__,"state_putc","Bad magic number");

    if (OUT_STATE_type_magic != s->state_type->magic)
	state_panic(__FILE__,__LINE__,"state_putc",
		    "Bad type magic number");

    use_filter = s->state_type->policy_so_it(s,
					     state_policy_putc_use_filter);

    if (!s->filter && s->filter_line &&
	!s->pg_flags && !s->pager_range) {
	int flush_via_put = s->state_type->policy_so_it(s,
							state_policy_flush_via_put);
	
	if (flush_via_put)
	    flush_filter(s,0,0);
    } 

    if (s->filter_line || s->filter ||
	use_filter || s->pg_flags ||
	s->pager_range) {         
	int i, first = 0,l = 0;
	
	if (s->filter_line &&
	    (l = string_len(s->filter_line)) >= MAX_FILTER_line) {

	    DPRINT(DebugIO,10,(&DebugIO,
			       "state_put: filter line %d chars (max %d), flushing\n",
			       l,MAX_FILTER_line));

	    flush_filter(s,0,0);	    
	}

	for (i = 0; i < len; i++) {
	    if ('\n' == string[i] ||
		(' '  == string[i] && l + i-first+1 >= MIN_FILTER_line)) {
		int r;
		int errors = 0;
		
		if (!s->filter_line) {
		    state_force_filter_line0(s);	
		} else
		    flush_filter_charset(s,0);
		
		r = add_streambytes_to_string(s->filter_line,
					      i-first+1,cs2us(string+first),
					      &errors);
		if (r < i-first+1) {
		    const char * filter_line_MIME_name  UNUSED_VAROK = 
			get_string_MIME_name(s->filter_line);
		    
		    
		    DPRINT(Debug,10,(&Debug,
				     "state_put: failed to add all streambytes to string (r=%d != %d=len), ret=%d, charset %s\n",
				     r,i-first+1,first + r,
				     filter_line_MIME_name ?
				     filter_line_MIME_name :
				     "<not set>"));
		    
		    flush_filter(s,0,0); 
		    return first + r;  /* FAILURE */
		}
		
		if (errors) {
		    DPRINT(Debug,10,(&Debug,
				     "state_put: %d errors\n",
				     errors));
		}
		
		flush_filter(s,0,0); 
		first = i+1;
		l = 0;
	    }
	}
	if (first < len) {
	    int r;
	    int errors = 0;

	    if (!s->filter_line) {
		state_force_filter_line0(s);
	    } else
		flush_filter_charset(s,0);

	    r = add_streambytes_to_string(s->filter_line,
					  len-first,cs2us(string+first),
					  &errors);
	    if (r < len-first) {
		DPRINT(Debug,10,(&Debug,
				 "state_put: failed to add all streambytes to string (r=%d != %d=len), ret=%d\n",
				 r,len-first,first + r));


		DPRINT(DebugIO,30,(&DebugIO,
				   "state_put=%d (filter -- FAILURE)\n",
				   first + r));


		return first + r;  /* FAILURE */
	    }


	    if (errors) {
		DPRINT(Debug,10,(&Debug,
				 "state_put: %d errors\n",
				 errors));
	    }
	}

	DPRINT(DebugIO,30,(&DebugIO,
			   "state_put=%d (filter)\n",
			   len));

	return len;
    } else {
	int r = EOF;

	r = s->state_type->put_so_it(s,string,len);

	if (r > 0 && r <= len)
	    s->filter_on_BOL = ('\n' == string[r-1]);
		
	DPRINT(DebugIO,30,(&DebugIO,
			   "state_put=%d\n",
			   r));

	return r;
    }
}


int state_printf P_((struct out_state *s,
		     const char * format, const char *msg, ...));

int state_printf (
#if ANSI_C
		  struct out_state *s,
		  const char * format, 
		  const char *msg, ...
#else
		  s, format, msg, va_alist
#endif
		  )
#if !ANSI_C
     struct out_state *s;
     const char * format;
     const char *msg;
     va_dcl
#endif
{
    int ret = 0;
    va_list vl;
    int  t     = strlen(format);
    char * t1  = strchr(format,'\n');

    /* elm_vmessage uses display_charset -- if buffers
       'display_charset' is different we can not use 
       elm_vmessage 
    */
    int need_conversion = s->display_charset[0] != display_charset;
    int use_filter = 0;

    DPRINT(DebugIO,30,(&DebugIO,
		       "state_printf(s=%p,...)\n",
		       s));

    if (need_conversion) {
	const char * MIME_name_0  UNUSED_VAROK = 
	    get_charset_MIME_name(s->display_charset[0]);
	const char * MIME_name_d  UNUSED_VAROK = 
	    get_charset_MIME_name(display_charset);

	DPRINT(DebugIO,10,(&DebugIO,
			   "state_printf: Conversion needed: display charset (buffer)=%s (real)=%s\n",
			   MIME_name_0 ? MIME_name_0 : "<not set>",
			   MIME_name_d ? MIME_name_d : "<not set>"));
    }

    if (t1 && t1 != &(format[t-1])) 
	state_panic(__FILE__,__LINE__,"state_printf",
		    "Embedded newlines are not supported");

    if (STATE_out_magic != s->magic)
	state_panic(__FILE__,__LINE__,"state_printf","Bad magic number");

    if (OUT_STATE_type_magic != s->state_type->magic)
	state_panic(__FILE__,__LINE__,"state_printf",
		    "Bad type magic number");

    use_filter = s->state_type->policy_so_it(s,
					     state_policy_putc_use_filter);


    if (!s->filter && s->filter_line &&
	!s->pg_flags && !s->pager_range && !need_conversion) {
	int flush_via_put = s->state_type->policy_so_it(s,
							state_policy_flush_via_put);
	
	if (flush_via_put)
	    flush_filter(s,1 /* do LF <=> CRLF conversion */,0);
    } 


    
    if (need_conversion || s->filter_line || s->filter || use_filter ||
	s->pg_flags || s->pager_range) {
	struct string * temp;
	int l;

	Va_start(vl, msg);           /* defined in hdrs/elm_defs.h */
	
	temp = elm_smessage(-1,format,msg,vl);

	va_end(vl);

	if (s->filter_line &&
	    (l = string_len(s->filter_line)) >= MAX_FILTER_line) {

	    DPRINT(DebugIO,10,(&DebugIO,
			       "state_printf: filter line %d chars (max %d), flushing\n",
			       l,MAX_FILTER_line));

	    flush_filter(s,1 /* do LF <=> CRLF conversion */,0);	    
	}
	
	ret = string_len(temp);
	if (!s->filter_line) {	   
	    if (!s->filter) {
		if (use_filter /* is required? */ || need_conversion) {
		    /* STATE_out_dir also requires s->display_charset[0]
		       -- value RAW_BUFFER indicates raw copying ...
		    */
		    s->filter_line = convert_string(s->display_charset[0],
						    temp,0);
		    free_string(&temp);
		} else 
		    s->filter_line = temp;
	    } else {
		s->filter_line = convert_string(s->filter,temp,0);
		free_string(&temp);
	    }
	} else {
	    struct string * temp2 = cat_strings(s->filter_line,
						temp,0);
	    free_string(&(s->filter_line));
	    if (!s->filter) {
		if (use_filter /* is required? */) {
		    /* STATE_out_dir also requires s->display_charset[0]
		       -- value RAW_BUFFER indicates raw copying ...
		    */		    
		    s->filter_line = convert_string(s->display_charset[0],
						    temp2,0);
		    free_string(&temp2);
		} else 
		    s->filter_line = temp2;
	    } else {
		s->filter_line = convert_string(s->filter,temp2,0);
		free_string(&temp2);
	    }
	    free_string(&temp);
	}
	if (t1)
	    flush_filter(s, 1 /* do LF <=> CRLF conversion */,0);

    } else {
	/* This uses display_charset */

	char * str;
	int len;

	Va_start(vl, msg);           /* defined in hdrs/elm_defs.h */
	
	str = elm_vmessage(-1,format,msg,vl);
	len = strlen(str);

	va_end(vl);
	
	if (t1) {
	    int size = len + 4;
	    
	    str = safe_realloc(str,size);

	    /* Take care of CRLF => LF conversion or
	       LF -> CRLF conversion 
	    */
	    state_convert_EOLN(str,&len,size,s);
	}

	ret = s->state_type->put_so_it(s,str,len);

	if (ret > 0 && ret <= len)
	    s->filter_on_BOL = ('\n' == str[ret-1]);

	free(str);
    }

    DPRINT(DebugIO,30,(&DebugIO,
		       "state_printf=%d\n",
		       ret));

    return ret;
}

int out_state_EOLN_is_CRLF(ptr)
     struct out_state *ptr;
{
    return ptr->EOLN_is_CRLF;
}

charset_t get_out_state_filter(ptr)
     struct out_state *ptr;
{
    return ptr->filter;
}

const char *get_out_state_f_MIME_name(ptr)
     struct out_state *ptr;
{

    if (!ptr->filter)
	return NULL;

    return get_charset_MIME_name(ptr->filter);
}

const charset_t * get_out_state_charset_vector(ptr)
     struct out_state *ptr;
{
    return ptr->display_charset;
}

const char * get_out_state_cv_MIME_name(ptr,idx)
     struct out_state *ptr;
     int idx;
{
    int i;

    for (i = 0; i < idx; i++) {

	if ( ! ptr->display_charset[i]) 
	    state_panic(__FILE__,__LINE__,"get_out_state_cv_MIME_name",
			"Bad index (index point past of vector)");
    }

    if (i != idx)
	state_panic(__FILE__,__LINE__,"get_out_state_cv_MIME_name",
		    "Bad index (impossible?)");

    if (! ptr->display_charset[idx])
	return NULL;

    return get_charset_MIME_name(ptr->display_charset[idx]);
}

void set_out_state_filter(ptr,cs)
     struct out_state *ptr;
     charset_t cs;
{
    ptr->filter = cs;
}


void set_out_state_cs_from_state(ptr,src)
     struct out_state *ptr;
     const struct out_state *src;
{
    ptr->filter           = src->filter;
    ptr->display_charset  = src->display_charset;
}

void set_out_state_cs(ptr,filter,vector)
     struct out_state *ptr;
     charset_t filter;
     charset_t * vector;
{
    ptr->filter          = filter;
    ptr->display_charset = vector;
}

void set_out_state_EOLN(ptr,is_CRLF)
     struct out_state *ptr;
     int is_CRLF;
{
    ptr->EOLN_is_CRLF = is_CRLF;
}



int out_state_ferror(s)
     struct out_state *s;
{
    int ret = 0;

    DPRINT(DebugIO,30,(&DebugIO,
		       "out_state_ferror(s=%p)\n",s));


    if (STATE_out_magic != s->magic)
	state_panic(__FILE__,__LINE__,"out_state_ferror","Bad magic number");

    if (OUT_STATE_type_magic != s->state_type->magic)
	state_panic(__FILE__,__LINE__,"out_state_ferror",
		    "Bad type magic number");
    
    if (s->filter_line)
	flush_filter(s,0,0);

    ret = s->state_type->ferror_so_it(s);

    DPRINT(DebugIO,30,(&DebugIO,
		       "out_state_ferror=%d\n",
		       ret));

    return ret;
}


int set_out_state_line_pg_flags(ptr,pg_flags)
     struct out_state *ptr;
     int pg_flags;
{
    int support_pg_flags;

    DPRINT(DebugIO,30,(&DebugIO,
		       "set_out_state_line_pg_flags(ptr=%p,pg_flags=%d)\n",
		       ptr,pg_flags));

    if (STATE_out_magic != ptr->magic)
	state_panic(__FILE__,__LINE__,"set_out_state_line_pg_flags",
		    "Bad magic number");

    if (OUT_STATE_type_magic != ptr->state_type->magic)
	state_panic(__FILE__,__LINE__,"set_out_state_line_pg_flags",
		    "Bad type magic number");

    /* Should be called on beginning of line so that
       line is empty */
    
    if (ptr->filter_line) {
	DPRINT(DebugIO,30,(&DebugIO,
			   "set_out_state_line_pg_flags: Flushing line\n"));

	flush_filter(ptr,0,0);
    }

    support_pg_flags = ptr->state_type->policy_so_it(ptr,
						     state_policy_support_pg_flags);

    if (support_pg_flags)
	ptr->pg_flags = pg_flags;

    DPRINT(DebugIO,30,(&DebugIO,
		       "set_out_state_line_pg_flags=%d\n",
		       support_pg_flags));

    return support_pg_flags;
}

/* Allocates new struct pager_range */
struct pager_range * state_add_opcode_pager_range(outfile,opcode,inherit,
						  range_flags,quote_level,
						  indent)
     struct out_state       * outfile;
     enum pager_range_opcode  opcode;
     struct pager_range *inherit;
     int range_flags;
     int quote_level;
     int indent;
{
    struct pager_range * ret = NULL;

    if (STATE_out_magic != outfile->magic)
	state_panic(__FILE__,__LINE__,"state_add_opcode_pager_range",
		    "Bad magic number");

   ret = malloc_simple_pager_range(inherit,range_flags,
				    quote_level,indent);

   if (OUT_STATE_type_magic != outfile->state_type->magic)
       state_panic(__FILE__,__LINE__,"state_add_opcode_pager_range",
		   "Bad type magic number");

   set_pager_range_opcode(ret,opcode);
   
   outfile->state_type->add_pager_range_it(outfile,ret);
   
   return ret;
}

void state_pager_add_param_text(outfile,range,opcode,text)
     struct out_state      * outfile;
     struct pager_range    * range;
     enum pager_param_opcode opcode;
     struct string         * text;   
{
    struct pager_param_value * param_value = NULL;
    union pager_param_uvalue value;
    
    if (STATE_out_magic != outfile->magic)
	state_panic(__FILE__,__LINE__,"state_pager_add_param_text",
		    "Bad magic number");

    value.text = text;

    param_value = malloc_pager_param_value(opcode,value);

   if (OUT_STATE_type_magic != outfile->state_type->magic)
       state_panic(__FILE__,__LINE__,"state_pager_add_param_text",
		   "Bad type magic number");

   outfile->state_type->add_pager_param_it(outfile,range,opcode,
					   param_value);

   free_pager_param_value(&param_value);
}

/* Allocates new struct pager_range */
struct pager_range * state_add_simple_pager_range(outfile,inherit,
						  range_flags,quote_level,
						  indent)
     struct out_state *outfile;
     struct pager_range *inherit;
     int range_flags;
     int quote_level;
     int indent;
{
    struct pager_range * ret = NULL;

    if (STATE_out_magic != outfile->magic)
	state_panic(__FILE__,__LINE__,"state_add_simple_pager_range",
		    "Bad magic number");

    DPRINT(DebugIO,30,(&DebugIO,
		       "state_add_simple_pager_range(outfile=%p,%s,range_flags=%d,quote_level=%d,indent=%d)\n",
		       outfile,inherit ? "(parent)" : "...",range_flags,
		       quote_level,indent));


    ret = malloc_simple_pager_range(inherit,range_flags,
				    quote_level,indent);

    if (OUT_STATE_type_magic != outfile->state_type->magic)
	state_panic(__FILE__,__LINE__,"state_add_simple_pager_range",
		    "Bad type magic number");

    outfile->state_type->add_pager_range_it(outfile,ret);

    return ret;
}

/* should called on beginning of line, may flush line
 */
void set_out_state_line_pager_range(ptr,range)
     struct out_state *ptr;
     struct pager_range *range;
{
    DPRINT(DebugIO,30,(&DebugIO,
		       "set_out_state_line_pager_range(ptr=%p,...)\n",
		       ptr));

    if (STATE_out_magic != ptr->magic)
	state_panic(__FILE__,__LINE__,"set_out_state_line_pager_range",
		    "Bad magic number");

    if (OUT_STATE_type_magic != ptr->state_type->magic)
	state_panic(__FILE__,__LINE__,"set_out_state_line_pager_range",
		    "Bad type magic number");

    /* Should be called on beginning of line so that
       line is empty */
    
    if (ptr->filter_line) {
	DPRINT(DebugIO,30,(&DebugIO,
			   "set_out_state_line_pager_ranges: Flushing line\n"));
	flush_filter(ptr,0,0);
    }
    
    if (ptr->pager_range) {
	free_pager_range(& (ptr->pager_range));

	/* flush_filter() resets pager range only on end of line */

	DPRINT(DebugIO,10,(&DebugIO,
			   "set_out_state_line_pager_ranges: Was already pager range on line\n"));

    }

    ptr->pager_range = range;
    if (ptr->pager_range) {
	inc_pager_range_refcount(ptr->pager_range);
	ptr->filter_on_BOR = 1;
    } else {
	DPRINT(DebugIO,10,(&DebugIO,
			   "set_out_state_line_pager_ranges: resetting page range\n"));
    }
}
 
/* Returns 0 if pg_ line flags are not supported,
   should called on beginning of line, may flush line 

   if range is not set, keeps current range
 */
int set_out_state_line_mode(ptr,pg_flags,range,force_eoln)
     struct out_state *ptr;
     int pg_flags;
     struct pager_range *range;
     int force_eoln;
{
    int support_pg_flags;

    DPRINT(DebugIO,30,(&DebugIO,
		       "set_out_state_line_mode(ptr=%p,pg_flags=%d,...)\n",
		       ptr,pg_flags));

    if (STATE_out_magic != ptr->magic)
	state_panic(__FILE__,__LINE__,"set_out_state_line_mode",
		    "Bad magic number");

    if (OUT_STATE_type_magic != ptr->state_type->magic)
	state_panic(__FILE__,__LINE__,"set_out_state_line_mode",
		    "Bad type magic number");

    /* Should be called on beginning of line so that
       line is empty */
    
    if (ptr->filter_line) {
	DPRINT(DebugIO,30,(&DebugIO,
			   "set_out_state_line_mode: Flushing line\n"));

	flush_filter(ptr,0,force_eoln);
    } else if (! ptr->filter_on_BOL && force_eoln) {
	DPRINT(DebugIO,30,(&DebugIO,
			   "set_out_state_line_mode: Flushing when no line?\n"));
	state_force_filter_line0(ptr);
	flush_filter(ptr,0,force_eoln);
    }

    support_pg_flags = ptr->state_type->policy_so_it(ptr,
						     state_policy_support_pg_flags);

    if (support_pg_flags)
	ptr->pg_flags = pg_flags;

    if (range) {
	/* if range is not set, keeps current range */

	if (ptr->pager_range) {
	    
	    free_pager_range(& (ptr->pager_range));
	    
	    /* flush_filter() resets pager range only on end of line */
	    
	    DPRINT(DebugIO,10,(&DebugIO,
			       "set_out_state_line_mode: Was already pager range on line\n"));
	}
	
	ptr->pager_range = range;
	inc_pager_range_refcount(ptr->pager_range);
	ptr->filter_on_BOR = 1;
    }

    DPRINT(DebugIO,30,(&DebugIO,
		       "set_out_state_line_mode=%d\n",
		       support_pg_flags));

    return support_pg_flags;
}


/* ---------------------------------------------------------- */

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

S_(init_so_func init_so_file)
static void init_so_file P_((struct out_state *)); /* Prototype */
static void init_so_file(s)
     struct out_state *s;
{
    s->u.file = safe_zero_alloc(sizeof (* s->u.file));

    s->u.file->magic = OUT_STATE_file_magic;
    s->u.file->fpout = NULL;
}

S_(dest_so_func dest_so_file)
static void dest_so_file P_((struct out_state *)); /* Prototype */
static void dest_so_file(s)
     struct out_state *s;
{
    if (OUT_STATE_file_magic != s->u.file->magic)
	state_panic(__FILE__,__LINE__,"dest_so_file","Bad magic number");

    s->u.file->fpout = NULL;

    s->u.file->magic = 0;  /* Invalidate */
    free (s->u.file);
    s->u.file = NULL;
}

void set_out_state_file (file,s)
     FILE * file;
     struct out_state *s;
{
    if (STATE_out_magic != s->magic)
	state_panic(__FILE__,__LINE__,"set_out_state_file","Bad magic number");

    if (STATE_out_file != s->state_type)
	state_panic(__FILE__,__LINE__,"set_out_state_file","Bad type");

    if (OUT_STATE_file_magic != s->u.file->magic)
	state_panic(__FILE__,__LINE__,"dest_so_file","Bad sub magic number");

    if (s->u.file->fpout != NULL)
	state_panic(__FILE__,__LINE__,"set_out_state_file","Already called");

    s->u.file->fpout = file;
}


S_(seekable_so_func seekable_so_file)
static int seekable_so_file P_((struct out_state *)); /* Prototype */
static int seekable_so_file(s)
     struct out_state *s;
{
    return 1;
}

S_(FILE_so_func FILE_so_file)
static  FILE * FILE_so_file P_((struct out_state *s));
static  FILE * FILE_so_file(s)
     struct out_state *s;
{
    if (OUT_STATE_file_magic != s->u.file->magic)
	state_panic(__FILE__,__LINE__,"FILE_so_file","Bad magic number");
    
    return s->u.file->fpout;
}

S_(seek_so_func seek_so_file)
int  seek_so_file P_((struct out_state *s, long pos));
int  seek_so_file(s,pos)
     struct out_state *s;
     long pos;
{
    if (OUT_STATE_file_magic != s->u.file->magic)
	state_panic(__FILE__,__LINE__,"seek_so_file","Bad magic number");
    
    if (!s->u.file->fpout)
	state_panic(__FILE__,__LINE__,"out_state_fseek",
		    "NULL file pointer");
    
    return fseek(s->u.file->fpout,pos,SEEK_SET);
}

S_(ftell_so_func ftell_so_file)
static long ftell_so_file P_((struct out_state *s));
static long ftell_so_file(s)
     struct out_state *s;
{
    if (OUT_STATE_file_magic != s->u.file->magic)
	state_panic(__FILE__,__LINE__,"ftell_so_file","Bad magic number");
    
    if (!s->u.file->fpout)
	state_panic(__FILE__,__LINE__,"ftell_so_file",
		    "NULL file pointer");

    return ftell(s->u.file->fpout);
}

S_(flush_filter_so_func flush_filter_so_file)
static void flush_filter_so_file P_((struct out_state *s, int munge_eoln,
				     int pg_flags, struct pager_range *range,
				     int force_eoln));
static void flush_filter_so_file(s,munge_eoln,pg_flags, range, force_eoln)
     struct out_state *s; 
     int munge_eoln;
     int pg_flags;
     struct pager_range *range;
     int force_eoln;
{
    state_panic(__FILE__,__LINE__,"flush_filter_so_file",
		"flush_filter_so_file called");
}

S_(policy_so_func policy_so_file)
static int policy_so_file P_((struct out_state *s, 
			      enum state_policy question));
static int policy_so_file(s,question)
     struct out_state *s;
     enum state_policy question;
{
    switch (question) {
    case state_policy_putc_use_filter:   return 0;
    case state_policy_flush_via_put:     return 1;
    case state_policy_support_pg_flags:  return 0;
    case state_policy_convert_entity:    return 1;
	
    default:
	panic("STATE PANIC",__FILE__,__LINE__,"policy_so_file",
	      "Bad question",0);

    }

    return 0;
}

S_(putc_so_func putc_so_file)
static int putc_so_file P_((struct out_state *, int)); /* Prototype */
static int putc_so_file(s,ch)
     struct out_state *s;
     int ch;
{
    if (OUT_STATE_file_magic != s->u.file->magic)
	state_panic(__FILE__,__LINE__,"putc_so_file","Bad magic number");

    if (s->u.file->fpout == NULL)
	state_panic(__FILE__,__LINE__,"putc_so_file","NULL file pointer");
    
    return (fputc (ch, s->u.file->fpout));
}

S_(put_so_func put_so_file)
static int put_so_file P_((struct out_state *, const char *,int)); /* Prototype */
static int put_so_file(s,string,len)
     struct out_state *s;
     const char *string;
     int len;
{
    if (OUT_STATE_file_magic != s->u.file->magic)
	state_panic(__FILE__,__LINE__,"put_so_file","Bad magic number");
    
    if (s->u.file->fpout == NULL)
	state_panic(__FILE__,__LINE__,"put_so_file","NULL file pointer");
    
    return fwrite (string, 1, len, s->u.file->fpout);
}

S_(ferror_so_func  ferror_so_file)
static int ferror_so_file P_((struct out_state *s));
static int ferror_so_file(s)
     struct out_state *s;
{
    if (OUT_STATE_file_magic != s->u.file->magic)
	state_panic(__FILE__,__LINE__,"ferror_so_file","Bad magic number");
    
    if (s->u.file->fpout == NULL)
	state_panic(__FILE__,__LINE__,"ferror_so_file","NULL file pointer");
    
    return ferror(s->u.file->fpout);
}

S_(add_pager_range_func add_pager_range_file)
static void add_pager_range_file P_((struct out_state *s, 
				     struct pager_range *range));
static void add_pager_range_file(s,range) 
     struct out_state *s;
     struct pager_range *range;
{
    /* Nothing to do */
}

S_(putentity_so_func putentity_so_file)
static enum sout_entity_result putentity_so_file P_((struct out_state *s,
						     struct out_entity *oe));
static enum sout_entity_result putentity_so_file(s,oe)
     struct out_state *s;
     struct out_entity *oe;
{
    /* Should not be called */

    return sout_entity_fallback;
}

S_(add_pager_param_func add_pager_param_file)
static void add_pager_param_file P_((struct out_state *s, struct pager_range *range,
				     enum pager_param_opcode opcode,
				     struct pager_param_value * param_value));
static void add_pager_param_file(s,range,opcode,param_value)
     struct out_state *s;
     struct pager_range *range;
     enum pager_param_opcode opcode;
     struct pager_param_value * param_value;
{

    /* Only supported on builtin pager */
}


struct out_state_type STATE_out_file_type = {
    OUT_STATE_type_magic,

    init_so_file,
    dest_so_file,
    seekable_so_file,
    FILE_so_file,
    seek_so_file,
    ftell_so_file,
    flush_filter_so_file,
    policy_so_file,
    putc_so_file,
    put_so_file,
    ferror_so_file,
    add_pager_range_file,
    putentity_so_file,
    add_pager_param_file
};

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
