static char rcsid[] = "@(#)$Id: static-hosts.c,v 2.4 2024/06/26 16:54:27 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.4 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "elm_defs.h"
#include "schedule_time.h"
#include "connection_imp.h"
#include "service_imp.h"
#include "s_me.h"

/* Parses /etc/hosts file without using gethostent() function  */

DEBUG_VAR(Debug,__FILE__,"net");

#ifdef REMOTE_MBX

#define FILE_HOSTENTRY_magic	0xF920

static struct file_hostentry {
    unsigned short magic;     /* FILE_HOSTENTRY_magic */

    char * lowercase_name;    /* ASCII lowercased */
    
    struct service_entry    * combined_one_name;    /* If line have only one name */;

    struct service_entry   ** file_line;
    size_t                    file_line_count;

    unsigned int             valid:1;    /* Used during rescanning */
} * malloc_file_hostentry P_((const char *name));

static struct file_hostentry * malloc_file_hostentry(name)
     const char *name;
{
    struct file_hostentry  * ret =  safe_zero_alloc(sizeof (* ret));
    
    char *c ;

    ret->lowercase_name  = safe_strdup(name);

    /* lowercase ascii */    
    for (c = ret->lowercase_name; *c; c++) {
	if (isascii(*c))
	    *c = tolower(*c);
    }

    ret->combined_one_name = NULL;
    ret->file_line         = NULL;
    ret->file_line_count   = 0;

    ret->valid             = 0;
    ret->magic             = FILE_HOSTENTRY_magic;
    
    return ret;
}

static void clear_hostentry_addrs P_((struct file_hostentry *entry));
static void clear_hostentry_addrs(entry)
     struct file_hostentry *entry;
{
    if (FILE_HOSTENTRY_magic != entry->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "clear_hostentry_addrs",
	      "Bad magic (file_hostentry)",0);

    /* entry->combined_one_name can be NULL
       decrements refcount */
    free_service_entry(& (entry->combined_one_name));

    if (entry->file_line) {
	size_t idx;

	for (idx = 0; idx < entry->file_line_count; idx++) {
	    if (entry->file_line[idx])
		free_service_entry(& (entry->file_line[idx]));
	}
	    
	free (entry->file_line);
	entry->file_line = NULL;
    }
}

static void free_file_hostentry P_((struct file_hostentry **entry));
static void free_file_hostentry(entry)
     struct file_hostentry **entry;
{
    if (FILE_HOSTENTRY_magic != (*entry)->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_file_hostentry",
	      "Bad magic (file_hostentry)",0);

    clear_hostentry_addrs(*entry);

    (*entry)->magic = 0;   /* Invalidate */
    free(*entry);
    *entry = NULL;    
}

static void free_file_hostentry_cache
     P_((struct file_hostentry *** names_p,
	 size_t                  * names_count_p));
static void free_file_hostentry_cache(names_p,
				      names_count_p)
     struct file_hostentry *** names_p;
     size_t                  * names_count_p;
{
    struct file_hostentry ** names       = * names_p;
    size_t                   names_count = * names_count_p;

    if (names) {
	size_t i;

	for (i = 0; i < names_count; i++) {
	    if (names[i]) 
		free_file_hostentry(& (names[i]));
	}

	free(names);
	names = NULL;
   }
   names_count = 0;
   
    * names_p        = names;
    * names_count_p  = names_count;
}

/* Returns number of cleared entries */
static int free_expired_file_hostentry_cache
    P_((struct file_hostentry           ** names,
	 size_t                            names_count,
	const struct schedule_timelimit * now,
	struct schedule_timelimit       * next_cleanup,
	int                             * have_expired));
static int free_expired_file_hostentry_cache(names,names_count,
					     now,next_cleanup,
					     have_expired)
     struct file_hostentry         ** names;
     size_t                           names_count;
     const struct schedule_timelimit * now;
     struct schedule_timelimit        * next_cleanup;
     int                             * have_expired;
{
    int count = 0;  /* Overflow ? */

    if (names) {
	size_t i;

	for (i = 0; i < names_count; i++) {
	    if (names[i]) {
		int c1 = 0;
		
		if (FILE_HOSTENTRY_magic != names[i]->magic)
		    panic("CONNECTION PANIC",__FILE__,__LINE__,
			  "free_expired_file_hostentry_cache",
			  "Bad magic (file_hostentry)",0);


		if (names[i]->combined_one_name)
		    c1 += free_expired_on_service_entry(names[i]->combined_one_name,
							now,
							next_cleanup);

		if (names[i]->file_line) {
		    size_t idx;
		    
		    for (idx = 0; idx < names[i]->file_line_count; idx++) {

			if (names[i]->file_line[idx])
			    c1 = free_expired_on_service_entry(names[i]->file_line[idx],
							       now,
							       next_cleanup);
			
		    }
		}
		
		if (c1) {

		    DPRINT(Debug,16,(&Debug,
				     "free_expired_file_hostentry_cache: %s: %d entries expired",
				     names[i]->lowercase_name,c1));

		    
		    DPRINT(Debug,16,(&Debug,"\n"));

		    /* Not work very well, because service entry have several pointters / names */		    
		    names[i]->valid = 0;

		    if (have_expired)
			*have_expired = 1;
		}
		    
		count += c1;
		
	    }
	}
    }
	
    DPRINT(Debug,14,(&Debug,"free_expired_file_hostentry_cache=%d, names_count=%lu",
		     count,
		     (unsigned long)names_count));

    if (next_cleanup) {
	char * X = schedule_timeout_string(next_cleanup);
	
	if (X) {
	    DPRINT(Debug,14,(&Debug,
			    ", next cleanup %s",
			    X));			
	    free(X);
	}	
    }        
    DPRINT(Debug,14,(&Debug,"\n"));
    
    return count;
}

static struct file_hostentry * give_file_hostentry_cache
     P_((const char              * name,
	 int                       create,
	 struct file_hostentry *** names_p,
	 size_t                  * names_count_p));
static struct file_hostentry * give_file_hostentry_cache(name,create,
							 names_p,
							 names_count_p)
     const char * name;
     int          create;
     struct file_hostentry *** names_p;
     size_t                  * names_count_p;
{
    struct file_hostentry ** names       = * names_p;
    size_t                   names_count = * names_count_p;
    int                      r           = 0;
    size_t                   mid_index   = 0;
    
    DPRINT(Debug,14,(&Debug,
		     "give_file_hostentry_cache: name=\"%s\"\n",name));
    
    if (names_count > 0) {
	size_t min_index  = 0;
	size_t max_index  = names_count -1;

	while (min_index <= max_index) {
	    size_t add        = (max_index - min_index) / 2;
	    
	    mid_index  = min_index + add;

	    
	    if (mid_index >= names_count ||
		! names[mid_index])
		panic("CONNECTION PANIC",__FILE__,__LINE__,
		      "give_file_hostentry_cache",
		      "Bad index",0);
	    
	    if (FILE_HOSTENTRY_magic != names[mid_index]->magic)
		panic("CONNECTION PANIC",__FILE__,__LINE__,
		      "give_file_hostentry_cache",
		      "Bad magic (file_hostentry)",0);

	    r = istrcmp(name,names[mid_index]->lowercase_name);

	    if (r < 0) {
		DPRINT(Debug,16,(&Debug,
				 "give_file_hostentry_cache: name=\"%s\" < [%lu] name=\"%s\"\n",
				 name,
				 (unsigned long)mid_index,
				 names[mid_index]->lowercase_name));
		if (mid_index > 0)
		    max_index = mid_index - 1;
		else {
		    DPRINT(Debug,16,(&Debug,
				     "give_file_hostentry_cache: quiting  [%lu] = min value\n",
				     (unsigned long)mid_index));
		    break;
		}
	    } else if (r > 0) {
		DPRINT(Debug,16,(&Debug,
				 "give_file_hostentry_cache: name=\"%s\" > [%lu] name=\"%s\"\n",
				 name,
				 (unsigned long)mid_index,
				 names[mid_index]->lowercase_name));
		if (mid_index+1 < names_count) 
		    min_index = mid_index + 1;
		else {
		    DPRINT(Debug,16,(&Debug,
				     "give_file_hostentry_cache: quiting  [%lu] = max value\n",
				     (unsigned long)mid_index));
		    break;
		}
	    } else {
		DPRINT(Debug,16,(&Debug,
				 "give_file_hostentry_cache: name=\"%s\" = [%lu] name=\"%s\"\n",
				 name,
				 (unsigned long)mid_index,
				 names[mid_index]->lowercase_name));

		DPRINT(Debug,14,(&Debug,
				 "give_file_hostentry_cache=%p FOUND\n",
				 names[mid_index]));
		return names[mid_index];
	    }
	}       
    }

    if (!create) {
	DPRINT(Debug,14,(&Debug,
			 "give_file_hostentry_cache=NULL   -- NOT FOUND\n"));

	return NULL;
    }

    if (0 ==  names_count) {
	mid_index = 0;
    } else {     
	if (r > 0) {
	    DPRINT(Debug,16,(&Debug,
			     "give_file_hostentry_cache: Adding name=\"%s\" after [%lu] name=\"%s\"\n",
			     name,
			     (unsigned long)mid_index,
			     names[mid_index]->lowercase_name));
	    mid_index++;
	} else if (r < 0) {
	    DPRINT(Debug,16,(&Debug,
			     "give_file_hostentry_cache: Adding name=\"%s\" before [%lu] name=\"%s\"\n",
			     name,
			     (unsigned long)mid_index,
			     names[mid_index]->lowercase_name));
	} else {
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "give_file_hostentry_cache",
		  "Bad result",0);
	}
    }

    names = safe_array_realloc(names,
			       (names_count+1), sizeof(names[0]));
    
    if (mid_index < names_count) {
	size_t i;  
	DPRINT(Debug,16,(&Debug,
			 "give_file_hostentry_cache: Moving items %lu .. %lu upwards\n",
			 (unsigned long)mid_index,
			 (unsigned long)(names_count-1)));

	for (i =  names_count-1; i >= mid_index; i--) {
	    if (! names[i]) 
		panic("CONNECTION PANIC",__FILE__,__LINE__,"give_file_hostentry_cache",
		      "Bad index",0);

	    if (FILE_HOSTENTRY_magic != names[i]->magic)
		panic("CONNECTION PANIC",__FILE__,__LINE__,"give_file_hostentry_cache",
		      "Bad magic number",0);


	    names[i+1] =  names[i];
	    names[i] = NULL;

	    if (i == mid_index)
		break;
	}

    } else 
	names[names_count] = NULL;

    names_count++;

    if (names[mid_index])
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "give_file_hostentry_cache",
	      "Bad index",0);

    names[mid_index] =  malloc_file_hostentry(name);
        
    DPRINT(Debug,16,(&Debug,
		     "give_file_hostentry_cache: name=\"%s\" = [%lu] name=\"%s\"\n",
		     name,
		     (unsigned long)mid_index,
		     names[mid_index]->lowercase_name));
    
    * names_p        = names;
    * names_count_p  = names_count;

    DPRINT(Debug,14,(&Debug,
		     "give_file_hostentry_cache=%p CREATED\n",
		     names[mid_index]));
    return names[mid_index];
}

static void add_to_file_line P_((const char              * name,
				 struct service_entry    * se /* Increments refcount */,
				 struct file_hostentry *** names_p,
				 size_t                  * names_count_p,
				 int                     * too_many_names_p));

static void add_to_file_line(name,se,names_p,names_count_p,too_many_names_p)
     const char                      * name;
     struct service_entry            * se /* Increments refcount */;
     struct file_hostentry         *** names_p;
     size_t                          * names_count_p;
     int                             * too_many_names_p;
{
    struct file_hostentry ** names          = * names_p;
    size_t                   names_count    = * names_count_p;
    int                      too_many_names = * too_many_names_p;

    int too_many              = names_count >= static_hosts_max_names;
    struct file_hostentry * X =
	give_file_hostentry_cache(name,
				  !too_many,
				  &names,&names_count);

    if (SERVICE_ENTRY_magic != se->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "add_to_file_line",
	      "Bad magic (service_entry)",0);

    
    if (too_many)
	too_many_names = 1;
	
    if (X) {
	if (FILE_HOSTENTRY_magic != X->magic)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "add_to_file_line",
		  "Bad magic (file_hostentry)",0);

	X->file_line = safe_array_realloc(X->file_line,
					  X->file_line_count+1,
					  sizeof(X->file_line[0]));
	X->file_line[X->file_line_count] = se;
	inc_service_entry_refcount(X->file_line[X->file_line_count]);

	X->file_line_count++;

    } else if (!too_many)
    	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "add_to_file_line",
	      "Not created",0);       
    * names_p          = names;
    * names_count_p    = names_count;
    * too_many_names_p = too_many_names;

}

static void add_to_combined P_((const char                      * name,
				const SOCKADDR                  * ip_address_p,
				struct file_hostentry         *** names_p,
				size_t                          * names_count_p,
				int                             * too_many_names_p,
				const struct schedule_timelimit * valid_until /* default        */));

static void add_to_combined(name,ip_address_p,names_p,names_count_p,too_many_names_p,valid_until)
     const char                      * name;
     const SOCKADDR                  * ip_address_p;
     struct file_hostentry         *** names_p;
     size_t                          * names_count_p;
     int                             * too_many_names_p;
     const struct schedule_timelimit * valid_until /* default        */;
{
    struct file_hostentry ** names          = * names_p;
    size_t                   names_count    = * names_count_p;
    int                      too_many_names = * too_many_names_p;

    int too_many              = names_count >= static_hosts_max_names;
    struct file_hostentry * X =
	give_file_hostentry_cache(name,
				  !too_many,
				  &names,&names_count);

    if (too_many)
	too_many_names = 1;
	
    if (X) {
	if (FILE_HOSTENTRY_magic != X->magic)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "add_to_combined",
		  "Bad magic (file_hostentry)",0);
	
	if (!X->combined_one_name)
	    X->combined_one_name =
		malloc_service_entry(name,any_service_type,
				     SE_static   /* Is this needed */ |
				     0 /* This is NOT temporary entry ???
					*/);	

	add_addr_to_entry(X->combined_one_name,ip_address_p,valid_until,
			  name);

    } else if (!too_many)
    	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "add_to_combined",
	      "Not created",0);       
    * names_p          = names;
    * names_count_p    = names_count;
    * too_many_names_p = too_many_names;

}

static void invalidate_file_hostentry_cache
     P_((struct file_hostentry ** names,
	 size_t                   names_count));
static void invalidate_file_hostentry_cache(names,names_count)
     struct file_hostentry ** names;
     size_t                   names_count;
{
    if (names) {
	size_t i;

	for (i = 0; i < names_count; i++) {
	    if (names[i]) {
		if (FILE_HOSTENTRY_magic != names[i]->magic)
		    panic("CONNECTION PANIC",__FILE__,__LINE__,
			  "invalidate_file_hostentry_cache",
			  "Bad magic (file_hostentry)",0);

		names[i]->valid = 0;       
		
		clear_hostentry_addrs(names[i]);
	    }
	}
    }
}

static void trim_file_hostentry_cache
     P_((struct file_hostentry *** names_p,
	 size_t                  * names_count_p));
static void trim_file_hostentry_cache(names_p,names_count_p)
     struct file_hostentry *** names_p;
     size_t                  * names_count_p;
{
    struct file_hostentry ** names       = * names_p;
    size_t                   names_count = * names_count_p;
    size_t count = 0;
    
    if (names) {
	size_t i,k;

	for (i = 0, k = 0; i < names_count; i++) {
	    if (names[i]) {
		if (FILE_HOSTENTRY_magic != names[i]->magic)
		    panic("CONNECTION PANIC",__FILE__,__LINE__,
			  "trim_file_hostentry_cache",
			  "Bad magic (file_hostentry)",0);
				
		if (names[i]->valid ||
		    names[i]->combined_one_name ||
		    names[i]->file_line ||
		    names[i]->file_line_count) {

		    if (k < i) {
		    
			if (names[k]) {
			    panic("CONNECTION PANIC",__FILE__,__LINE__,
				      "trim_file_hostentry_cache",
				      "Not free'ed",0);
			}
			names[k] = names[i];
			names[i] = NULL;
		    }
		    k++;
		    
		} else {
		    free_file_hostentry(& (names[i]));
		    count++;
		}
	    }	   	   
	}

	names_count = k;
    }

    if (count > 0) {
	DPRINT(Debug,10,(&Debug,"trim_file_hostentry_cache: %lu entires trimmed; count %lu => %lu\n",
			 (unsigned long)count,
			 (unsigned long)* names_count_p,
			 (unsigned long) names_count));
    }
    
    * names_p        = names;
    * names_count_p  = names_count;    
}

/* Return 1 if there is some expired data */
int expired_service_entry(se,now)
     struct service_entry            * se;
     const struct schedule_timelimit * now;
{
    int ret = 0;

    char * now_t = schedule_timeout_string(now);
    
    if (SERVICE_ENTRY_magic != se->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "expired_service_entry",
	      "Bad magic (service_entry)",0);

    if (se->aliases_list) {
	int i;

	for (i = 0; i < se->aliases_count; i++) {
	    if (schedule_valid_until_expired(& (se->aliases_list[i].valid_until),
					     now)) {
		
		char * X = schedule_timeout_string(& (se->aliases_list[i].valid_until));
		
		ret = 1;
		
		if (X) {
		    DPRINT(Debug,14,(&Debug,
				     "expired_service_entry: %s: alias #%d %s expired on %s\n",
				     se->official_name,
				     i,
				     se->aliases_list[i].aliasname,
				     X));
		    free(X);
		    X = NULL;
		}
	    }
	}
    }

    if (se->addr_list) {
	int i;

	for (i = 0; i < se->addr_count; i++) {
	    if (schedule_valid_until_expired(& (se->addr_list[i].valid_until),
					     now)) {

		char * X = schedule_timeout_string(& (se->addr_list[i].valid_until));

		if (X) {
		    DPRINT(Debug,14,(&Debug,
				     "expired_service_entry: %s: addr #%d expired on %s\n",
				     se->official_name,
				     i,
				     X));
		    free(X);
		    X = NULL;
		}
	    }
	}
    }

    if (now_t) {
	 DPRINT(Debug,14,(&Debug,
			  "expired_service_entry=%d: %s: now %s\n",
			  ret,se->official_name,now_t));
	 free(now_t);
	 now_t = NULL;
    }

    return ret;    
}


				     

static const char * etc_hosts     = "/etc/hosts";

static int eat_token P_((FILE    * F,
			 unsigned  lineno,
			 size_t    token_alloced,
			 char    * token,
			 size_t  * token_len_p,
			 int     * ch_p));
static int eat_token(F,lineno,token_alloced,token,token_len_p,ch_p)
     FILE    * F;
     unsigned  lineno;
     size_t    token_alloced;
     char    * token;
     size_t  * token_len_p;
     int     * ch_p;
{
    int ret = 1;
    size_t  token_len = 0;
    int   ch = * ch_p;

    do {

	if (token_len < token_alloced) 
	    token[token_len] = ch;	    
	else 
	    ret = 0;

	token_len++;
	ch       = getc(F);

    } while (ch != EOF && ch != '#' &&
	     (! whitespace(ch)) &&
	     isascii(ch) && isprint(ch));

    if (token_len < token_alloced) 
	token[token_len] = 0;	    
    else 
	ret = 0;
        
    if (EOF == ch && ferror(F)) {
	ret = 0;

	DPRINT(Debug,10,(&Debug,
			 "eat_token: %s: %u: Read error\n",
			 etc_hosts,lineno));
    } else if (!ret) {
	DPRINT(Debug,10,(&Debug,
			 "eat_token: %s: %u: Too long token (%lu), max %lu\n",
			 etc_hosts,
			 lineno,
			 (unsigned long)token_len,
			 (unsigned long)token_alloced));
    } else if ('.' == token[token_len-1]) {
	int d = token_len < 80 ? token_len : 83;

	if (d > token_alloced)
	    d = token_alloced;
	
	ret = 0;
	
	DPRINT(Debug,10,(&Debug,
			 "parse_etc_hosts: %s: %u: Bad token %.*s%s\n",
			 etc_hosts,
			 lineno,
			 d,token,
			 d < token_len ? "..." : ""));	
    }
	
    * ch_p        = ch;
    * token_len_p = token_len;

    return ret;
}

struct name_token {
    char * token;
    size_t  token_len;    
};

static int eat_line P_((FILE               * F,
			unsigned             lineno,
			SOCKADDR           * ip_address_p,
			struct name_token ** token_list_p,
			size_t             * token_list_len_p,
			int                * ch_p,
			const char         * lookup_name /* May be NULL    */,
			int                * is_lookup_name_p,
			int                * too_many_names_p));
static int eat_line(F,lineno,ip_address_p,token_list_p,token_list_len_p,ch_p,
		    lookup_name,is_lookup_name_p,too_many_names_p)
     FILE               * F;
     unsigned             lineno;
     SOCKADDR           * ip_address_p;
     struct name_token ** token_list_p;
     size_t             * token_list_len_p;
     int                * ch_p;
     const char         * lookup_name /* May be NULL    */;
     int                * is_lookup_name_p;
     int                * too_many_names_p;
{
    int ret = 1;

    int max_names = static_hosts_max_names > 2 ? static_hosts_max_names : 2;
    
    SOCKADDR            ip_address     = * ip_address_p;
    struct name_token * token_list     = * token_list_p;
    size_t              token_list_len = 0;
    int                 ch             = * ch_p;
    int                 is_lookup_name = 0;
    int                 too_many_names = * too_many_names_p;
    
    size_t token_alloced = 254;     /* Maximum name of host.domain +1 */
    char * token         = safe_malloc(token_alloced);
    size_t token_len     = 0;
    
    while (ch != EOF && whitespace(ch)) 	    
	ch       = getc(F);
    
    switch (ch) {
    case EOF:
    case '\n':
	ret = 0;
	goto out;
    case '#':
    case '.':
	ret = 0;
	goto skip_line;	
    }

    if (isascii(ch) && isprint(ch)) {
    
	/* Read IP address */
	if (eat_token(F,lineno,token_alloced,token,&token_len,&ch)) {
	    
	    if (token_len > 0 && token_len < token_alloced &&
		'\0' == token[token_len] &&
		get_ip(&ip_address,token)) {
		
		int d = token_len < 80 ? token_len : 83;
	    
		if (d > token_alloced)
		    d = token_alloced;
		
		DPRINT(Debug,16,(&Debug,
				 "eat_line: %s: %u: Got IP address %.*s%s\n",
				 etc_hosts,
				 lineno,
				 d,token,
				 d < token_len ? "..." : ""));
		
	    } else {	    
		int d = token_len < 80 ? token_len : 83;
		
		if (d > token_alloced)
		    d = token_alloced;
		
		ret = 0;
		
		DPRINT(Debug,12,(&Debug,
				 "eat_line: %s: %u: Bad IP address %.*s%s\n",
				 etc_hosts,
				 lineno,
				 d,token,
				 d < token_len ? "..." : ""));
		
		goto skip_line;
	    }
	    
	    /* Eat names */
	    while (1) {
		
		while (ch != EOF && whitespace(ch)) {
		    
		    ch       = getc(F);
		}
		
		switch (ch) {
		case EOF:
		case '\n':
		    goto out;
		    
		case '#':
		    goto skip_line;
		case '.':
		    ret = 0;
		    goto skip_line;
		    
		}

		if (isascii(ch) && isprint(ch)) {
		    
		    if (eat_token(F,lineno,token_alloced,token,&token_len,&ch)) {
			int matches = 0;
			
			if (token_len > 0 && token_len < token_alloced &&
			    '\0' == token[token_len]) {
			    
			    if (lookup_name &&
				0 == istrcmp(token,lookup_name)) {
				
				int d = token_len < 80 ? token_len : 83;
				
				if (d > token_alloced)
				    d = token_alloced;
				
				DPRINT(Debug,12,(&Debug,
						 "eat_line: %s: %u: Name found %.*s%s%s\n",
						 etc_hosts,
						 lineno,
						 d,token,
						 d < token_len ? "..." : "",
						 is_lookup_name ? " (again) ": ""));
				
				is_lookup_name = 1;
				matches = 1;
			    }
			    
			    if (token_list_len > max_names)
				too_many_names = 1;
			    
			    else if (!too_many_names || matches) {
				size_t i;
				
				for (i = 0; i < token_list_len; i++) {
				    if (token_list[i].token_len == token_len &&
					0 == strcmp(token,token_list[token_list_len].token)) {
					int d = token_len < 80 ? token_len : 83;
					
					if (d > token_alloced)
					    d = token_alloced;
					
					DPRINT(Debug,12,(&Debug,
							 "eat_line: %s: %u: Dublicate name %.*s%s\n",
							 etc_hosts,
							 lineno,
							 d,token,
							 d < token_len ? "..." : ""));
					
					goto skip_it;
				    }
				}
			    
				token_list = safe_array_realloc(token_list,token_list_len+1,
								sizeof (token_list[0]));
				
				token_list[token_list_len].token     = safe_realloc(token,token_len+1);
				token_list[token_list_len].token_len = token_len;
				
				token_list_len++;
				token         = safe_malloc(token_alloced);
			    
			    skip_it:
				token_len     = 0;
			    }
			    
			} else {
			    int d = token_len < 80 ? token_len : 83;
			    
			    if (d > token_alloced)
				d = token_alloced;
			    
			    DPRINT(Debug,12,(&Debug,
					     "eat_line: %s: %u: Bad name result %.*s%s\n",
					     etc_hosts,
					     lineno,
					     d,token,
					     d < token_len ? "..." : ""));
			    
			    goto skip_line;
			}
			
		    } else {
			ret = 0;
			goto skip_line;
		    }
		} else {
		    ret = 0;
		    goto skip_line;
		}		    
	    }
	    
	} else {
	    ret = 0;
	    goto skip_line;
	}
	
    } else {

	ret = 0;
	
    skip_line:

	if (ch != EOF && ch != '\n') {
	    size_t count = 0;
	    DPRINT(Debug,16,(&Debug,
			     "eat_line: %s: %u: Skipping%s%s: ",
			     etc_hosts,
			     lineno,
			     ret ? " (ok)" : " on failure",
			     '#' == ch ? " (on comment)" : ""
			     ));
	    
	    while (ch != EOF && ch != '\n') {
		
		if (count++ < 80) {
		    if (isascii(ch) && isprint(ch)) {
			DPRINT(Debug,16,(&Debug,"%c",ch));
		    } else {
			DPRINT(Debug,16,(&Debug,"."));
		    }
		}
				
		ch       = getc(F);	    
	    }
	
	    if (count > 80) {
		DPRINT(Debug,16,(&Debug,"..."));
	    }
	
	    DPRINT(Debug,16,(&Debug,"\n"));
	}
    }

out:

    if (EOF == ch && ferror(F)) {
	ret = 0;

	DPRINT(Debug,10,(&Debug,
			 "eat_line: %s: %u: Read error\n",
			 etc_hosts,lineno));
    }

    * ip_address_p     = ip_address;
    * ch_p             = ch;
    * token_list_p     = token_list;
    * token_list_len_p = token_list_len;
    * is_lookup_name_p = is_lookup_name;
    * too_many_names_p = too_many_names;

    if (token) {
	if (token_len > 0) {
	    int d UNUSED_VAROK = token_len < 80 ? token_len : 83;
	    
	    DPRINT(Debug,10,(&Debug,
			     "eat_line: %s: %u: token not consumed: %.*s%s\n",
			     etc_hosts,
			     lineno,
			     d,token,
			     d < token_len ? "..." : ""));	
	}
	
	free(token);
	token = NULL;
    }
    
    return ret;
}

static void add_to_service_entry P_((struct service_entry            * target,
				     const SOCKADDR                  * ip_address_p,
				     const struct name_token         * token_list,
				     const size_t                      token_list_len,
				     const struct schedule_timelimit * valid_until /* default        */
				     ));
static void add_to_service_entry(target,ip_address_p,
				 token_list,token_list_len,
				 valid_until)
     struct service_entry            * target;
     const SOCKADDR                  * ip_address_p;
     const struct name_token         * token_list;
     const size_t                      token_list_len;
     const struct schedule_timelimit * valid_until /* default        */;
{
    size_t i;

    if (SERVICE_ENTRY_magic != target->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "add_to_service_entry",
	      "Bad magic (service_entry) (target)",0);
    
    add_addr_to_entry(target,ip_address_p,valid_until,
		      token_list[0].token);
				
    for (i = 0; i < token_list_len; i++) {
	if (0 != strcmp(target->official_name,
			token_list[i].token))
	    add_alias_to_entry(target,
			       token_list[i].token,
			       valid_until);
    }				   				
}

static FILE       * etc_hosts_F   = NULL;

static void close_etc_hosts P_((void));
static void close_etc_hosts()
{
    if (etc_hosts_F) {
	int r = fclose(etc_hosts_F);

	if (0 == r) {
	    DPRINT(Debug,10,(&Debug,
			     "close_etc_hosts: %s closed\n",
			     etc_hosts));	    
	} else if (EOF == r) {
	    int err UNUSED_VAROK = errno;

	     DPRINT(Debug,10,(&Debug,
			      "close_etc_hosts: %s: %s (errno %d)\n",
			      etc_hosts,strerror(err),err));

	}
	etc_hosts_F = NULL;
    }

}

static struct schedule_timelimit etc_hosts_valid_until;

int static_hosts_opened;
static int static_hosts_expired;   /* Some entries are expired */
struct stat etc_hosts_stat;
int static_hosts_cached;

/* Short names */
static struct file_hostentry ** dotless_names        = NULL;
size_t                          dotless_names_count  = 0;

/* qualified names, names with domain */

static struct file_hostentry ** qualified_names       = NULL;
size_t                          qualified_names_count = 0;



/* Increments refcount, service type is always SERVICE_TYPES[0] ,
   if lookup_name is NULL returns always NULL
*/
static enum etc_hosts_status parse_etc_hosts
    P_((struct service_entry           ** ret_p
	/* Increments refcount, service type is always SERVICE_TYPES[0] */,
	const char                      * lookup_name /* May be NULL    */,
	const struct schedule_timelimit * now,
	const struct schedule_timelimit * valid_until /* default        */,
	struct cancel_data             ** cancel_p,
	int                             * was_error_p  /* Print errors only if
							  was_error_p != NULL */
	));
static enum etc_hosts_status parse_etc_hosts(ret_p,lookup_name,now,valid_until,cancel_p,
					     was_error_p)
     struct service_entry           ** ret_p
     /* Increments refcount, service type is always SERVICE_TYPES[0] */;
     const char                      * lookup_name /* May be NULL    */;		    
     const struct schedule_timelimit * now;
     const struct schedule_timelimit * valid_until /* default        */;
     struct cancel_data             ** cancel_p;
     int                             * was_error_p  /* Print errors only if
						       was_error_p != NULL */;
	
{
    enum etc_hosts_status    status    =  etc_hosts_maybe;
    enum open_etc_status {
	etc_hosts_open_fail     = -1,
	etc_hosts_opened        =  0,
	etc_hosts_need_open     =  1	
    } reopen_etc_hosts  = etc_hosts_open_fail;

    /* *ret_p can be NULL */
    if (ret_p)
	free_service_entry(ret_p);
    
    if (etc_hosts_F && static_hosts_opened) {

	if (schedule_valid_until_expired(&etc_hosts_valid_until,
					 now)) {

	    struct stat temp_stat;
	    
	    int r = stat(etc_hosts,&temp_stat);

	    if (0 == r) {

		if (temp_stat.st_dev == etc_hosts_stat.st_dev &&
		    temp_stat.st_ino == etc_hosts_stat.st_dev) {
		    
		    reopen_etc_hosts = etc_hosts_opened;    /* Use current  etc_hosts_F,
							       file not changed           
							    */

		    DPRINT(Debug,10,(&Debug,
				     "parse_etc_hosts: %s: uses current etc_hosts_F, same file\n",
				     etc_hosts));

		} else {		    
		    reopen_etc_hosts = etc_hosts_need_open;   /* file changed */

		    DPRINT(Debug,10,(&Debug,
				     "parse_etc_hosts: %s: need reopen, file changed\n",
				     etc_hosts));
		}
		
	    } else if (-1 == r) {
		int err = errno;
		
		DPRINT(Debug,10,(&Debug,
				 "parse_etc_hosts: %s: %s (errno %d)\n",
				 etc_hosts,strerror(err),err));

		reopen_etc_hosts = etc_hosts_opened;  /* Use current  etc_hosts_F,
							 current file is not accessible
						      */

		if (was_error_p) {
		    if (lookup_name)
			lib_error(CATGETS(elm_msg_cat, MeSet,MeLookingUpFileErr,
					  "Looking up %s: File %s: %s"),
				  lookup_name,etc_hosts,strerror(err));
		    else
			lib_error(CATGETS(elm_msg_cat, MeSet,MeFileErr,
					  "File %s: %s"),
				  etc_hosts,strerror(err));
		    *was_error_p = 1;
		}		
	    }
	    
	} else {
	    reopen_etc_hosts = etc_hosts_opened;     /* Use current  etc_hosts_F */

	    DPRINT(Debug,10,(&Debug,
			     "parse_etc_hosts: %s: uses current etc_hosts_F, file not checked\n",
			     etc_hosts));
	}
	    
    } else {

	DPRINT(Debug,10,(&Debug,
			 "parse_etc_hosts: %s: need open%s\n",
			 etc_hosts,
			 etc_hosts_F ? " (have etc_hosts_F)" : ""));
	reopen_etc_hosts = etc_hosts_need_open;

    }

 restart_on_open:
    if (reopen_etc_hosts > etc_hosts_opened) {
	int fd = open(etc_hosts,O_RDONLY);

	if (-1 == fd) {
	    int err = errno;

	    DPRINT(Debug,10,(&Debug,
			     "parse_etc_hosts: %s: %s (errno %d)\n",
			     etc_hosts,strerror(err),err));

	    if (EINTR == err &&  cancel_p && *cancel_p &&
		is_canceled(*cancel_p)) {
		
		DPRINT(Debug,10,(&Debug,
				 "parse_etc_hosts: Query canceled\n"));
		status = etc_hosts_failed;
		goto fail;
	    }

	    reopen_etc_hosts = etc_hosts_open_fail;

	    if (was_error_p) {
		if (lookup_name)
		    lib_error(CATGETS(elm_msg_cat, MeSet,MeLookingUpFileErr,
				      "Looking up %s: File %s: %s"),
			      lookup_name,etc_hosts,strerror(err));
		else
		    lib_error(CATGETS(elm_msg_cat, MeSet,MeFileErr,
				      "File %s: %s"),
			      etc_hosts,strerror(err));
		
		*was_error_p = 1;
	    }
	    
	} else {
	    /* Does nothing if etc_hosts_F is NULL */
	    close_etc_hosts();

	    etc_hosts_F = fdopen(fd,"r");

	    if (!etc_hosts_F) {
		int err = errno;

		DPRINT(Debug,10,(&Debug,
				 "parse_etc_hosts: %s (fd=%d): %s (errno %d)\n",
				 etc_hosts,fd,strerror(err),err));

		close(fd);

		if (was_error_p) {
		    if (lookup_name)
			lib_error(CATGETS(elm_msg_cat, MeSet,MeLookingUpFileErr,
					  "Looking up %s: File %s: %s"),
				  lookup_name,etc_hosts,strerror(err));
		    else
			lib_error(CATGETS(elm_msg_cat, MeSet,MeFileErr,
					  "File %s: %s"),
				  etc_hosts,strerror(err));
		    *was_error_p = 1;
		}		
	    }	    
	}	   
    }

    if ( reopen_etc_hosts < etc_hosts_need_open && etc_hosts_F) {

	rewind(etc_hosts_F);
    }

    if (etc_hosts_F) {
	int      too_many_names = 0;
	unsigned lineno   = 0;
	int      ch       = getc(etc_hosts_F);
	int      new_line = 1;
	
	if (EOF != ch) {
	    
	    static_hosts_opened  = 1;
	    etc_hosts_valid_until = *valid_until;
	    static_hosts_cached = 0;
	    
	    invalidate_file_hostentry_cache(dotless_names,dotless_names_count);
	    invalidate_file_hostentry_cache(qualified_names,qualified_names_count);
	    
	    while (EOF != ch) {
		SOCKADDR            ip_address;
		struct name_token * token_list     = NULL;
		size_t              token_list_len = 0;
		
		if (new_line) {
		    int is_lookup_name = 0;

		    lineno++;
		    new_line = 0;

		    if (eat_line(etc_hosts_F,lineno,
				 &ip_address,&token_list,&token_list_len,
				 &ch,lookup_name,&is_lookup_name,
				 &too_many_names)) {

			if (is_lookup_name) {
			    if (token_list_len < 1)
				panic("CONNECTION PANIC",__FILE__,__LINE__,
				      "parse_etc_hosts",
				      "token_list_len < 1",0);

			    status = etc_hosts_found;

			    if (ret_p) {
				/* Is this correct ?? */
				
				if (!*ret_p)
				   *ret_p =
				       malloc_service_entry(token_list[0].token,
							    any_service_type,
							    SE_static   /* Is this needed */ |
							    SE_temporary );

				add_to_service_entry(*ret_p,&ip_address,
						     token_list,token_list_len,
						     valid_until);
			    }
			}

			if (1 == token_list_len) {
			    size_t dotc = 0;
			    size_t k;
			    
			    for (k = 0 ; k < token_list[0].token_len; k++)
				if ('.' == token_list[0].token[k])
				    dotc++;

			    if (dotc)
				add_to_combined(token_list[0].token,&ip_address,
						&qualified_names,&qualified_names_count,
						&too_many_names,valid_until);
			    else
				add_to_combined(token_list[0].token,&ip_address,
						&dotless_names,&dotless_names_count,
						&too_many_names,valid_until);
				    
			} else if (token_list_len > 1) {

			    struct service_entry * se =
				malloc_service_entry(token_list[0].token,
						     any_service_type,
						     SE_static   /* Is this needed */ |
						     0 /* This is NOT temporary entry ???
							*/);	
			    size_t i;

			    add_to_service_entry(se,&ip_address,
						 token_list,token_list_len,
						 valid_until);


				
			    for (i = 0; i < token_list_len; i++) {

				size_t dotc = 0;
				size_t k;
				
				for (k = 0 ; k < token_list[i].token_len; k++)
				    if ('.' == token_list[i].token[k])
					dotc++;

				/* Increments 'se' refcount */
				if (dotc)
				    add_to_file_line(token_list[i].token,se,
						     &qualified_names,&qualified_names_count,
						     &too_many_names);
				else
				    add_to_file_line(token_list[i].token,se,
						     &dotless_names,&dotless_names_count,
						     &too_many_names);
			    }
			    		       	
			    free_service_entry(&se);
			    
			} else {
			    DPRINT(Debug,20,(&Debug,
					     "parse_etc_hosts: %s: %u: No names\n",
					     etc_hosts,
					     lineno));
			}
		    } else 
			goto skip_line;
		    		    
		} else {
		skip_line:
		    if (EOF != ch &&
			'\n' != ch) {
			size_t count = 0;
			
			DPRINT(Debug,16,(&Debug,
					 "parse_etc_hosts: %s: %u: Ignoring: ",
					 etc_hosts,
					 lineno));
			
			while (EOF != ch &&
			       '\n' != ch) {
			    
			    if (count++ < 80) {
				if (isascii(ch) && isprint(ch)) {
				    DPRINT(Debug,16,(&Debug,"%c",ch));
				} else {
				    DPRINT(Debug,16,(&Debug,"."));
				}
			    }
			    
			    ch       = getc(etc_hosts_F);	    
			}
			
			if (count > 80) {
			    DPRINT(Debug,16,(&Debug,"..."));
			}
		     	
			DPRINT(Debug,16,(&Debug,"\n"));				     
		    }
		}
		
		if ('\n' == ch) {
		    ch       = getc(etc_hosts_F);
		    new_line = 1;
		}

		if (token_list) {
		    size_t i;

		    for (i = 0; i < token_list_len; i++) {
			if (token_list[i].token) {
			    free(token_list[i].token);
			    token_list[i].token = NULL;
			}
			token_list[i].token_len = 0;
		    }
		    free(token_list);
		    token_list = 0;
		}
		token_list_len = 0;
		
	    }

	     DPRINT(Debug,16,(&Debug,
			      "parse_etc_hosts: %s: %u lines\n",
			      etc_hosts,lineno));
	}
	
	if (ferror(etc_hosts_F)) {
	    int err = errno;
	    
	    DPRINT(Debug,10,(&Debug,
			     "parse_etc_hosts: %u: %s: %s (errno %d)\n",
			     etc_hosts,lineno,strerror(err),err));
	    
	    if (EINTR == err &&  cancel_p && *cancel_p &&
		is_canceled(*cancel_p)) {
		
		DPRINT(Debug,10,(&Debug,
				 "parse_etc_hosts: Query canceled\n"));
		
	    } else if (etc_hosts_opened == reopen_etc_hosts) {
		DPRINT(Debug,10,(&Debug,
				 "parse_etc_hosts: %s: reopening after error\n",
				 etc_hosts));

		close_etc_hosts();
		
		reopen_etc_hosts =  reopen_etc_hosts;
		goto restart_on_open;
	    }
	    status = etc_hosts_failed;

	    if (was_error_p) {
		if (lookup_name)
		    lib_error(CATGETS(elm_msg_cat, MeSet,MeLookingUpFileErr,
				      "Looking up %s: File %s: %s"),
			      lookup_name,etc_hosts,strerror(err));
		else
		    lib_error(CATGETS(elm_msg_cat, MeSet,MeFileErr,
				      "File %s: %s"),
			      etc_hosts,strerror(err));
		*was_error_p = 1;
	    }	    
	}
	
	if (feof(etc_hosts_F)) {
	    if (!new_line) {
		DPRINT(Debug,10,(&Debug,
				 "parse_etc_hosts: %s: %u: Unexpected EOF\n",
				 etc_hosts,lineno));
	    } else if (etc_hosts_maybe == status) {
		
		if (lookup_name)
		    status = etc_hosts_not_found;
		else
		    status = etc_hosts_parsed;

		static_hosts_cached = ! too_many_names;
	    }
	}

	trim_file_hostentry_cache(&dotless_names,&dotless_names_count);
	trim_file_hostentry_cache(&qualified_names,&qualified_names_count);
	
    } else {
	DPRINT(Debug,14,(&Debug,
			 "parse_etc_hosts: %s: file not open\n",
			 etc_hosts));
	status = etc_hosts_failed;
    }

 fail:
    if (etc_hosts_maybe == status) {
	DPRINT(Debug,10,(&Debug,
			 "parse_etc_hosts: %s: no status -- resetting to failed\n",
			 etc_hosts));
	status = etc_hosts_failed;
    }
	
    DPRINT(Debug,10,(&Debug,
		     "parse_etc_hosts=%d",
		     status));
    switch (status) {
    case etc_hosts_maybe:
	DPRINT(Debug,10,(&Debug," (maybe) ERROR"));
	break;
    case etc_hosts_failed:	
	DPRINT(Debug,10,(&Debug," (failed, can't read %s or canceled)",etc_hosts));
	break;
    case etc_hosts_not_found:
	DPRINT(Debug,10,(&Debug," (%s opened, %s not found)",
			 etc_hosts,
			 lookup_name ? lookup_name : "<none>"));
	break;
    case etc_hosts_found:
	DPRINT(Debug,10,(&Debug," (%s opened, %s found)",
			 etc_hosts,
			 lookup_name ? lookup_name : "<none>"));
	break;
    case etc_hosts_parsed:
	DPRINT(Debug,10,(&Debug," (%s parsed)",
			 etc_hosts));
	break;
    case etc_hosts_found_cached:
	DPRINT(Debug,10,(&Debug," (found cached) ERROR"));
	break;

    }

    if (static_hosts_opened) {
	DPRINT(Debug,10,(&Debug,", static hosts opened"));
    }

    if (static_hosts_expired) {
	DPRINT(Debug,10,(&Debug,", static hosts expired"));
    }

    if (static_hosts_cached) {
	DPRINT(Debug,10,(&Debug,", static hosts cached"));
    }
    
    DPRINT(Debug,10,(&Debug,"\n"));
    
    return status;
}

struct service_entry * lookup_static_host_sentry(lookup_name,now,valid_until,cancel_p,
						 lc,was_error_p,static_status_p)
     const char * lookup_name;
     const struct schedule_timelimit * now;
     const struct schedule_timelimit * valid_until /* default        */;
     struct cancel_data             ** cancel_p;
     enum lookup_class                 lc;
     int                             * was_error_p /* Print errors only if
						      was_error_p != NULL */;
     enum etc_hosts_status           * static_status_p;
{
    struct service_entry *ret   = NULL;
    int cache_tried             = 0;
    int etc_hosts_tried         = 0;
    
    if (static_hosts_opened && !static_hosts_expired &&
	schedule_valid_until_ok(&etc_hosts_valid_until,
				now)) {
    
	struct file_hostentry * fhe;
	char * etc_valid_until_t;
	char * now_t;

	if (static_status_p)
	    *static_status_p = etc_hosts_maybe;   /* Not queried */
	
    retry_cache:
	fhe = NULL;
	etc_valid_until_t = schedule_timeout_string(&etc_hosts_valid_until);
	now_t             = schedule_timeout_string(now);
	
	DPRINT(Debug,14,(&Debug,"lookup_static_host_sentry: %s",
			 lookup_name));
	if (etc_valid_until_t) {
	    DPRINT(Debug,14,(&Debug,", %s valid until %s",
			     etc_hosts,etc_valid_until_t));
	    free(etc_valid_until_t);
	    etc_valid_until_t = NULL;
	}
	if (now_t) {
	    DPRINT(Debug,14,(&Debug,", now %s",
			     now_t));
	    free(now_t);
	    now_t = NULL;
	}	
	DPRINT(Debug,14,(&Debug,"\n"));
			 
	switch (lc) {
	    size_t dotc,k;
	    
	case lc_default:
	    dotc =  0;
	    for (k = 0; lookup_name[k]; k++)
		if ('.' == lookup_name[k])
		    dotc ++;
	    
	    if (dotc) {
		lc = lc_qualified;
		goto qualified;
	    }
	    
	    lc = lc_dotless;
	    /* FALLTHROUGH */
	case lc_dotless:
	    fhe = give_file_hostentry_cache(lookup_name,0,
					    &dotless_names,&dotless_names_count);
	    break;
	case lc_qualified:
	qualified:
	    fhe = give_file_hostentry_cache(lookup_name,0,
					    &qualified_names,&qualified_names_count);
	    break;	  
	}
	cache_tried = 1;
	
	if (fhe) {
	    const char * name = lookup_name;
	    
	    if (FILE_HOSTENTRY_magic != fhe->magic)
		panic("CONNECTION PANIC",__FILE__,__LINE__,
		      "lookup_static_host_sentry",
		      "Bad magic (file_hostentry)",0);

	    if (ret)
		free_service_entry(& ret);
	    	    
	    DPRINT(Debug,14,(&Debug,"lookup_static_host_sentry: %s: Got entry %s\n",
			     lookup_name,fhe->lowercase_name));

	    if (fhe->combined_one_name) {

		if (SERVICE_ENTRY_magic != fhe->combined_one_name->magic)
		    panic("CONNECTION PANIC",__FILE__,__LINE__,
			  "lookup_static_host_sentry",
			  "Bad magic (service_entry)",0);
		
		if (expired_service_entry(fhe->combined_one_name,now)) {

		    if (etc_hosts_tried) {
			DPRINT(Debug,14,(&Debug,"lookup_static_host_sentry: %s: Will return expired data\n",
					 lookup_name));
		    } else		    
			goto reload_etc_hosts;
		}
		
		if (fhe->file_line) {
		    size_t idx;

		    for (idx = 0; idx < fhe->file_line_count; idx++) {
			if (fhe->file_line[idx]) {
			    if (expired_service_entry(fhe->file_line[idx],now)) {				
				if (etc_hosts_tried) {
				    DPRINT(Debug,14,(&Debug,"lookup_static_host_sentry: %s: Will return expired data\n",
						     lookup_name));
				} else		    				    
				    goto reload_etc_hosts;
			    }
			}
		    }

		    ret = malloc_service_entry(fhe->combined_one_name->official_name,
					       fhe->combined_one_name->service,
					       SE_static   /* Is this needed */ |
					       SE_temporary);

		    merge1_entries(ret,fhe->combined_one_name);
		    
		    goto copy_file_line;
		} else {
		    ret = fhe->combined_one_name;
		    inc_service_entry_refcount(ret);		    
		}
		
	    } else if (fhe->file_line) {
		size_t idx;
		char * candinate = NULL;
		int is_same = 1;
		
		for (idx = 0; idx < fhe->file_line_count; idx++) {
		    if (fhe->file_line[idx]) {

			if (SERVICE_ENTRY_magic != fhe->file_line[idx]->magic)
			    panic("CONNECTION PANIC",__FILE__,__LINE__,
				  "lookup_static_host_sentry",
				  "Bad magic (service_entry)",0);

			if (expired_service_entry(fhe->file_line[idx],now)) {
			    if (etc_hosts_tried) {
				DPRINT(Debug,14,(&Debug,"lookup_static_host_sentry: %s: Will return expired data\n",
						 lookup_name));
			    } else		    
				goto reload_etc_hosts;
			}
			
			if (!candinate)
			    candinate = fhe->file_line[idx]->official_name;
			else if (0 != istrcmp(candinate,
					      fhe->file_line[idx]->official_name))
			    is_same = 0;
		    }
		}

		if (candinate && is_same)
		    name = candinate;
		
	    copy_file_line:

		for (idx = 0; idx < fhe->file_line_count; idx++) {
		    if (fhe->file_line[idx]) {

			if (!ret)
			    ret = malloc_service_entry(name,
						       any_service_type,
						       SE_static   /* Is this needed */ |
						       SE_temporary);

			merge1_entries(ret,fhe->file_line[idx]);				       
		    }
		}
	    }

	    if (ret) {
		DPRINT(Debug,14,(&Debug,"lookup_static_host_sentry: %s: found (cache)\n",
				 lookup_name));

		if (static_status_p)
		    *static_status_p = etc_hosts_found_cached;
		       
	    } else {

		DPRINT(Debug,14,(&Debug,"lookup_static_host_sentry: %s: entry %s do not have address(es)\n",
				 lookup_name,fhe->lowercase_name));

		goto not_found;
	    }
	    
	} else {   /* Not found */
	    DPRINT(Debug,14,(&Debug,"lookup_static_host_sentry: %s: not found from cache (mode %d",
			     lookup_name,lc));
	    switch (lc) {
	    case lc_default:   DPRINT(Debug,14,(&Debug," default -- ERROR"));  break;
	    case lc_dotless:   DPRINT(Debug,14,(&Debug," dotless"));           break;
	    case lc_qualified: DPRINT(Debug,14,(&Debug," qualified"));         break;
	    }
	    DPRINT(Debug,14,(&Debug,")\n"));
	    
	not_found:
	    if (static_hosts_cached) {
		DPRINT(Debug,14,(&Debug,"lookup_static_host_sentry: %s: %s is cached, returing\n",
				 lookup_name,etc_hosts));
		goto done;
	    }
	}
	
    } else {
	enum etc_hosts_status  st   =  etc_hosts_maybe;
	
    reload_etc_hosts:
	if (etc_hosts_tried) {
	    DPRINT(Debug,14,(&Debug,
			     "lookup_static_host_sentry: %s: %s already tried, returning\n",
			     lookup_name,etc_hosts));
	    goto done;
	}
	
	/* parse_etc_hosts() free's previous result from ret */
	st = parse_etc_hosts(&ret,
			     lookup_name,
			     now,valid_until,
			     cancel_p,
			     was_error_p);
	etc_hosts_tried = 1;
	if (static_status_p)
	    *static_status_p = st;

	if (ret) {
	    DPRINT(Debug,14,(&Debug,"lookup_static_host_sentry: %s: found (file)\n",
			     lookup_name));

	} else {
	    if (etc_hosts_failed == st &&
		cancel_p && *cancel_p &&
		is_canceled(*cancel_p)) {
		DPRINT(Debug,14,(&Debug, 
				 "lookup_static_host_sentry: %s: query canceled\n",
				 lookup_name));
		goto fail;
	    }
	    
	    if (st <  etc_hosts_not_found && !cache_tried &&
		static_hosts_opened) {
		
		DPRINT(Debug,14,(&Debug,
				 "lookup_static_host_sentry: %s: status %d",
				 lookup_name,st));
		switch (st) {
		case etc_hosts_maybe:        DPRINT(Debug,14,(&Debug," etc_hosts_maybe"));
		    break;
		case etc_hosts_failed:       DPRINT(Debug,14,(&Debug," etc_hosts_failed"));
		    break;
		case etc_hosts_not_found:    DPRINT(Debug,14,(&Debug," etc_hosts_not_found"));
		    break;
		case etc_hosts_found:        DPRINT(Debug,14,(&Debug," etc_hosts_found"));
		    break;
		case etc_hosts_parsed:       DPRINT(Debug,14,(&Debug," etc_hosts_parsed"));
		    break;
		case etc_hosts_found_cached: DPRINT(Debug,14,(&Debug," etc_hosts_found_cached"));
		    break;
		}
		DPRINT(Debug,14,(&Debug," -- trying (possible expired) cache instead.\n"));
		goto retry_cache;
	    }
	}
    }

 done:
 fail:
    
     if (ret) {

	 if (SERVICE_ENTRY_magic != ret->magic)
	     panic("CONNECTION PANIC",__FILE__,__LINE__,
		   "lookup_static_host_sentry",
		   "Bad magic (service_entry)",0);
	 
	 DPRINT(Debug,14,(&Debug,"lookup_static_host_sentry=%p: type=%p (%s)%s%s, refcount=%d, addr_count=%d\n",
			  ret,
			  ret->service,
			  service_type_name(ret->service),
			  ret->flags & SE_temporary ? ", temporary" : "",
			  ret->flags & SE_static    ? ", static"    : "",
			  ret->refcount,
			  ret->addr_count));

     } else {	 
	 DPRINT(Debug,14,(&Debug,"lookup_static_host_sentry=NULL"));
     }

     if (static_status_p) {
	 DPRINT(Debug,14,(&Debug,", *static_status_p=%d",*static_status_p));
	 switch (*static_status_p) {
	 case etc_hosts_maybe:        DPRINT(Debug,14,(&Debug," etc_hosts_maybe"));     break;
	 case etc_hosts_failed:       DPRINT(Debug,14,(&Debug," etc_hosts_failed"));    break;
	 case etc_hosts_not_found:    DPRINT(Debug,14,(&Debug," etc_hosts_not_found")); break;
	 case etc_hosts_found:        DPRINT(Debug,14,(&Debug," etc_hosts_found"));     break;
	 case etc_hosts_parsed:       DPRINT(Debug,14,(&Debug," etc_hosts_parsed"));    break;
	 case etc_hosts_found_cached: DPRINT(Debug,14,(&Debug," etc_hosts_found_cached")); break;
	 }
     }
     
     DPRINT(Debug,14,(&Debug,"\n"));

     
    return ret;
}

enum etc_hosts_status lookup_static_host(lookup_name,now,valid_until,
					  lc,rewrite_p,
					  addr_list_p,addr_count_p,
					  aliases_list_p,aliases_count_p,
					  was_error_p,cancel_p)
     const char                      * lookup_name;
     const struct schedule_timelimit * now;
     const struct schedule_timelimit * valid_until /* default        */;
      enum lookup_class                 lc           /* Optimization */;
     char                           ** rewrite_p;
     struct service_entry_addr      ** addr_list_p;
     int                             * addr_count_p;
     struct service_entry_name      ** aliases_list_p;
     int                             * aliases_count_p;

     int                             * was_error_p  /* Print errors only if
						       was_error_p != NULL */;
     struct cancel_data             ** cancel_p;
{
    enum etc_hosts_status ret = etc_hosts_maybe;
    
    struct service_entry_addr  * ADDR_LIST     = * addr_list_p;
    int                          ADDR_COUNT    = * addr_count_p;
    
    struct service_entry_name  * ALIASES_LIST  = * aliases_list_p;
    int                          ALIASES_COUNT = * aliases_count_p;
    
    struct service_entry * se =
	lookup_static_host_sentry(lookup_name,now,valid_until,
				  cancel_p,lc,was_error_p,
				  &ret);

    if (se) {
	if (SERVICE_ENTRY_magic != se->magic)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "lookup_static_host",
		  "Bad magic (service_entry)",0);

	DPRINT(Debug,16,(&Debug,
			 "lookup_static_host: %s: Got result %s\n",
			 lookup_name,se->official_name));
	
	if (rewrite_p)
	    *rewrite_p =
		strmcpy(*rewrite_p,
			se->official_name);

	if (se->aliases_list &&
	    se->aliases_count > 0) {
	    int newcount = ALIASES_COUNT + se->aliases_count   /* Overflow ? */;
	    int i;
	    
	    ALIASES_LIST = 
		safe_array_realloc(ALIASES_LIST,
				   newcount,
				   sizeof (ALIASES_LIST[0]));
	    
	    for (i = 0;
		 i < se->aliases_count && ALIASES_COUNT < newcount;
		 i++) {
		struct service_entry_name X = se->aliases_list[i];

		X.aliasname = safe_strdup(se->aliases_list[i].aliasname);

		DPRINT(Debug,16,(&Debug,
				 "lookup_static_host: %s: aliasname #%d: %s\n",
				 se->official_name,
				 i,
				 X.aliasname));

		ALIASES_LIST[ALIASES_COUNT++] = X;
	    }
	}

	if (se->addr_list &&
	    se->addr_count > 0) {
	    int newcount = ADDR_COUNT + se->addr_count  /* Overflow? */;
	    int i;

	     ADDR_LIST = safe_array_realloc(ADDR_LIST,
					    newcount,
					    sizeof (ADDR_LIST[0]));

	     for (i = 0; i < se->addr_count && ADDR_COUNT < newcount;i++) {

		 struct service_entry_addr X = se->addr_list[i];
		 char mybuffer[256];
		 const char * ip_string = NULL;
		 
		 if (se->addr_list[i].address.dummy && se->addr_list[i].addrsize) {
		     X.address.dummy = safe_malloc(se->addr_list[i].addrsize);
		     
		     memcpy(X.address.dummy,se->addr_list[i].address.dummy,
			    se->addr_list[i].addrsize);
		     X.addrsize = se->addr_list[i].addrsize;

		     ip_string = give_SOCKADDR_ptr_as_string(X.address,
							     X.addrsize,
							     mybuffer,sizeof mybuffer);
		 } else {
		     X.address.dummy    = NULL;
		     X.addrsize = 0;
		 }

		 if (se->addr_list[i].hostname)
		     X.hostname = safe_strdup(se->addr_list[i].hostname);
		 
		 DPRINT(Debug,16,(&Debug,
				  "lookup_static_host: %s: addr #%d:",
				  se->official_name,
				  i));
		 if (X.hostname) {
		     DPRINT(Debug,16,(&Debug," %s:",
				      X.hostname));
		 }
		 if (ip_string) {
		     DPRINT(Debug,16,(&Debug," %s",
				      ip_string));

		 }
		 DPRINT(Debug,16,(&Debug,"\n"));

		 ADDR_LIST[ADDR_COUNT++] = X;				  
	     }
	}
	
	free_service_entry(& se);
    }

    DPRINT(Debug,14,(&Debug,
		     "lookup_static_host=%d",
		     ret));
    switch (ret) {
    case etc_hosts_maybe:        DPRINT(Debug,14,(&Debug," etc_hosts_maybe"));
	break;
    case etc_hosts_failed:       DPRINT(Debug,14,(&Debug," etc_hosts_failed"));
	break;
    case etc_hosts_not_found:    DPRINT(Debug,14,(&Debug," etc_hosts_not_found"));
	break;
    case etc_hosts_found:        DPRINT(Debug,14,(&Debug," etc_hosts_found"));
	break;
    case etc_hosts_parsed:       DPRINT(Debug,14,(&Debug," etc_hosts_parsed"));
	break;
    case etc_hosts_found_cached: DPRINT(Debug,14,(&Debug," etc_hosts_found_cached"));
	break;
    }
    
    DPRINT(Debug,14,(&Debug,": %s, aliases_count: %d => %d, addr_count: %d => %d\n",
		     lookup_name,
		     * aliases_count_p,ALIASES_COUNT,
		     * addr_count_p,ADDR_COUNT));

    DPRINT(Debug,14,(&Debug,"\n"));
    
    * aliases_list_p  = ALIASES_LIST;
    * aliases_count_p = ALIASES_COUNT;

    * addr_list_p     = ADDR_LIST;
    * addr_count_p    = ADDR_COUNT;

    return ret;
}


/* Returns number of expired entries */

int free_expired_static_hosts(now,next_cleanup)
     const struct schedule_timelimit * now;
     struct schedule_timelimit       * next_cleanup;
{
    int count = 0;
    

    count +=
	free_expired_file_hostentry_cache(dotless_names,dotless_names_count,
					  now,next_cleanup,
					  &static_hosts_expired);


    count +=
	free_expired_file_hostentry_cache(qualified_names,qualified_names_count,
					  now,next_cleanup,
					  &static_hosts_expired);

    if (count > 0)
	static_hosts_expired = 1;

    if (static_hosts_expired) {
	DPRINT(Debug,14,(&Debug,"free_expired_static_hosts=%d",
			 count));

	if (next_cleanup) {
	    char * X = schedule_timeout_string(next_cleanup);
	    
	    if (X) {
		DPRINT(Debug,14,(&Debug,
				 ", next cleanup %s",
				 X));			
		free(X);
	    }	
	}   
	
	DPRINT(Debug,14,(&Debug,"\n"));
    }
    
    return count;

}

void free_static_hosts()
{

    close_etc_hosts();
    static_hosts_opened = 0;
    
    free_file_hostentry_cache(&dotless_names,&dotless_names_count);
    free_file_hostentry_cache(&qualified_names,&qualified_names_count);
    
}

#endif

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
