static char rcsid[] = "@(#)$Id: syscall.c,v 2.17 2020/03/20 19:01:15 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.17 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *
 * Some code based on Elm 2.4 src/syscall.c. It have following copyright:
 *
 *  The Elm Mail System 
 *
 *			Copyright (c) 1988-1992 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

#include "elm_defs.h"
#include "s_elm.h"

#if ANSI_C
#define E_(x) extern x;
#else
#define E_(x)
#endif


DEBUG_VAR(Debug,__FILE__,"system");

#include <errno.h>

static void no_SR_print_status P_((struct run_state *rs,int sig,
				   int exit_code));
static void no_SR_print_status(rs,sig,exit_code)
     struct run_state *rs;
     int sig;
     int exit_code;
{
    DPRINT(Debug,5,(&Debug, "No print_status_hook ...\n"));
}

static enum rawstate  no_SR_RawState     P_((void));
static enum rawstate  no_SR_RawState()
{
    DPRINT(Debug,5,(&Debug, "No RawState_hook ...\n"));
    return OFF;
}

static void no_SR_Raw          P_((int state));
static void no_SR_Raw(state)
     int state;
{
    DPRINT(Debug,5,(&Debug, "No Raw_hook ...\n"));
}

static void no_SR_tty_init     P_((void));
static void no_SR_tty_init()
{
    DPRINT(Debug,5,(&Debug, "No run_tty_init_hook ...\n"));
}

static void no_SR_ClearScreen  P_((void));
static void no_SR_ClearScreen()
{
    DPRINT(Debug,5,(&Debug, "No ClearScreen_hook ...\n"));
}

static char *us2s P_((unsigned char *str));
static char *us2s(str) 
     unsigned char *str;
{
    return (char *)str;
}

static void no_SR_PutLineS P_((struct string *S));
static void no_SR_PutLineS(S)
     struct string *S;
{
    struct string * R1 = convert_string(display_charset,S,1);
    char * store1;

    DPRINT(Debug,5,(&Debug, "No PutLineS_hook ...\n"));
       	
    store1 = us2s(stream_from_string(R1,0,NULL));

    fprintf(stderr,"%s",store1);
    
    free(store1);
    free_string(&R1);
}

static SR_print_status * print_status_hook = no_SR_print_status;
static SR_RawState     * RawState_hook     = no_SR_RawState;
static SR_Raw          * Raw_hook          = no_SR_Raw;
static SR_tty_init     * run_tty_init_hook = no_SR_tty_init;
static SR_ClearScreen  * ClearScreen_hook  = no_SR_ClearScreen;
static SR_PutLineS     * PutLineS_hook     = no_SR_PutLineS;
static SR_print_status * print_status_cooked_hook = no_SR_print_status;

void set_start_run_hooks(print_status_func,raw_query_func,raw_set_func,
			 run_tty_init,clear_screen_func,put_line_function,
			 print_status_cooked_func)
     SR_print_status * print_status_func;
     SR_RawState     * raw_query_func;
     SR_Raw          * raw_set_func;
     SR_tty_init     * run_tty_init;
     SR_ClearScreen  * clear_screen_func;
     SR_PutLineS     * put_line_function;
     SR_print_status * print_status_cooked_func;
{
    print_status_hook = print_status_func;
    RawState_hook     = raw_query_func;
    Raw_hook          = raw_set_func;
    run_tty_init_hook = run_tty_init;
    ClearScreen_hook  = clear_screen_func;
    PutLineS_hook     = put_line_function;
    print_status_cooked_hook = print_status_cooked_func;
}

int set_child_signals(options)
     int options;
{
    /*
     * Program to exec may or may not be able to handle
     * interrupt, quit, hangup and stop signals.
     */
    if (options&SY_ENAB_SIGINT)
	options |= SY_ENAB_SIGHUP;
    (void) signal(SIGHUP,  (options&SY_ENAB_SIGHUP) ? SIG_DFL : SIG_IGN);
    (void) signal(SIGINT,  (options&SY_ENAB_SIGINT) ? SIG_DFL : SIG_IGN);
    (void) signal(SIGQUIT, (options&SY_ENAB_SIGINT) ? SIG_DFL : SIG_IGN);
#ifdef SIGTSTP
    (void) signal(SIGTSTP, (options&SY_ENAB_SIGHUP) ? SIG_DFL : SIG_IGN);
    (void) signal(SIGCONT, (options&SY_ENAB_SIGHUP) ? SIG_DFL : SIG_IGN);
#endif
    return 0;
}

int set_child_env(options)
     int options;
{
    /* Optionally override the MM_CHARSET environment variable. */
    if (options&SY_ENV_METAMAIL) {
	const char * A = get_charset_MIME_name (display_charset);
	char mm[] = "MM_CHARSET=";
	char mm1[] = "MAILCAPS=";

	if (A) {
	    int size = sizeof(mm) + strlen(A);
	    
	    /* \0 character is included in size returned by sizeof */
	    char *p = malloc(size);
	    if (p) {
		elm_sfprintf(p, size,
			     FRM("%s%s"), mm , A );
		if (0 == putenv(p)) {
		    DPRINT(Debug,5,(&Debug, "Child: added %s\n",p));
		}
	    }
	}

	A = give_dt_path_as_str(&metamail_mailcaps,"metamail-mailcaps");
	if (A) {
	    int size = sizeof(mm1) + strlen(A);
	    
	    /* \0 character is included in size returned by sizeof */
	    char *p = malloc(size);
	    if (p) {
		elm_sfprintf(p, size,
			     FRM("%s%s"), mm1 , A );
		if (0 == putenv(p)) {
		    DPRINT(Debug,5,(&Debug, "Child: added %s\n",p));
		}
	    }
	}

	/* Used 'h' command */
	if (!elm_filter) {
	    static char mm2[] = "KEYHEADS=*";	
	    static char mm3[] = "KEYIGNHEADS=";	

	    if (0 == putenv(mm2)) {
		DPRINT(Debug,5,(&Debug, "Child: added %s\n",mm2));
	    }
	    if (0 == putenv(mm3)) {
		DPRINT(Debug,5,(&Debug, "Child: added %s\n",mm3));
	    }
	}
    }
    
    /* Optionally override the SHELL environment variable. */
    if (options&SY_ENV_SHELL) {
	static char sheq[] = "SHELL=";
	const char * sh =  "/bin/sh";
	int size;
	char *p;

	if (0 != (options & SY_USER_SHELL)) {
	    const char * s = give_dt_estr_as_str(&shell_e,"shell",NULL,NULL);
	    if (s)
		sh = s;
	}

	size = sizeof(sheq) + strlen(sh);

	p = malloc(size);
	if (p) {
	    elm_sfprintf(p, size,
			 FRM("%s%s"), sheq, sh);
	    if (0 == putenv(p)) {
		DPRINT(Debug,5,(&Debug, "Child: added %s\n",p));
	    }
	}
    }
    return 0;
}

E_(free_any_fd_f NO_free_any_fd)
void NO_free_any_fd(fd)
   union any_fd *fd;
{
   
}


E_(inc_any_fd_refcount_f NO_inc_any_fd_refcount)
void NO_inc_any_fd_refcount(fd)
    union any_fd fd;
{

}

#ifdef BACKGROUD_PROCESSES       /* We assume POSIX in here */

volatile int handle_sigchld = 0;       /* got SIGCHLD */

#define PROCESS_LIST_magic	0xF209

static struct process_list {
    unsigned short magic;        /* PROCESS_LIST_magic */
    
    union any_fd fd;
    char * message;
    struct run_state state_information;
    end_handler           * handler;
    free_any_fd_f         * free_any_fd;
    inc_any_fd_refcount_f * inc_any_fd_refcount;
    struct process_list * next;
} * my_processes = NULL;

static void got_sigchld P_((int sig));
static void got_sigchld (sig) 
     int sig;
{
    SIGDPRINT(Debug,2,(&Debug,
			"got_sigchld: Setting handle_sigchld\n"));

    handle_sigchld = 1;
}

/* < 0 signal, 0 not found,  removes process from list 
   1 exited normally
   2 no process (removed from list)
*/

int wait_background_end(pid,exit_code)
     int pid;
     int *exit_code;
{
    int r = 0;
    struct process_list *tmp = my_processes, *prev = NULL;
    int w;
    S__ status;
    
    DPRINT(Debug,10,(&Debug,
		     "wait_background_end: pid %d\n",pid));

    *exit_code = -1;
    
    while (tmp) {
	struct process_list * this = tmp;
	
	if (PROCESS_LIST_magic != this->magic)
	    panic("PROCESS PANIC",
		      __FILE__,__LINE__,"wait_background_end",
		      "Bad magic number (process_list)",0);

	if (pid == tmp->state_information.pid) {
	    int ret;
	    DPRINT(Debug,10,(&Debug,
			     "wait_background_end: Found process from process list\n"));
	    
	    ret =  wait_end(&(tmp->state_information),exit_code);

	    if (!r &&
		ECHILD == tmp->state_information.save_errno) {

		DPRINT(Debug,10,(&Debug,
				 "wait_background_end: No process\n"));
		r = 2;
		goto remove;
	    } else if (ret != 0) {
		
		tmp->handler(tmp->fd,tmp->message,&(tmp->state_information),
			     ret,*exit_code);
		
		r = ret;

	    remove:
		if (prev)
		    prev -> next = tmp -> next;
		else
		    my_processes = tmp -> next;
		
		DPRINT(Debug,2,(&Debug,
				"wait_background_end: deleting %d from list: %s\n",
				tmp->state_information.pid, tmp->message));

		tmp->free_any_fd(& (tmp->fd));	
		free(tmp->message);
		
		tmp = tmp -> next;

		this->magic = 0;  /* Invalidate */
		free(this);
	    }

	    goto found;
	}
	prev = tmp;
	tmp  = tmp -> next;
    }

    DPRINT(Debug,10,(&Debug,
		     "wait_background_end: Pid %d not on list\n",pid));

    while ((w = my_wait(pid,&status)) != pid) {
	if (w == -1 && errno != EINTR) 
	    break;

	DPRINT(Debug,10,(&Debug,
			 "wait_background_end: got my_wait=%d (!= %d) .. looping\n",
			 w,pid));
    }

    if (w == -1) {
	int err = errno;

	DPRINT(Debug,10,(&Debug,
			 "wait_background_end: errno %d\n",
			 err));

	if (ECHILD == err) {
	    DPRINT(Debug,10,(&Debug,
			     "wait_background_end: No process\n"));
	    goto found;
	}
    }

    if (w == pid) {
	int sig = convert_status(status,exit_code);
	
	DPRINT(Debug,2,(&Debug,
			"wait_background_end: exit_code=%d, sig=%d\n", 
			*exit_code,sig));
	
	if (sig) {
	    r = -sig;
	} else {
	    r = 1;
	}
    }
 found:
    DPRINT(Debug,10,(&Debug,
		     "wait_background_end=5d; *exit_code=%d\n",
		     r,exit_code));
    return r;
}

void sigchld_handler() {
    
    DPRINT(Debug,2,(&Debug,
		    "sigchld_handler --> ENTER (not on signal)\n"));
    
    do {
	struct process_list *tmp = my_processes, *prev = NULL;
	handle_sigchld = 0;
	
	while (tmp) {
	    struct process_list * this = tmp;
	    int exit_code;
	    int ret;

	    if (PROCESS_LIST_magic != this->magic)
		panic("PROCESS PANIC",
		      __FILE__,__LINE__,"sigchld_handler",
		      "Bad magic number (process_list)",0);


	    ret = run_already_done(&(tmp->state_information),&exit_code);
	    /* < 0 got signal */
	    
	    if (ret != 0) {
		
		tmp->handler(tmp->fd,tmp->message,&(tmp->state_information),
			     ret,exit_code);
		
		if (prev)
		    prev -> next = tmp -> next;
		else
		    my_processes = tmp -> next;
		
		DPRINT(Debug,2,(&Debug,
				"sigchld_handler: deleting %d from list: %s\n",
				tmp->state_information.pid, tmp->message));
		
		tmp->free_any_fd(& (tmp->fd));		
		free(tmp->message);
		tmp = tmp -> next;

		this->magic = 0;  /* Invalidate */
		free(this);
		continue;
	    }
	    
	    prev = tmp;
	    tmp  = tmp -> next;
	}
	
    } while (handle_sigchld);
    
    DPRINT(Debug,2,(&Debug,
		    "sigchld_handler --> LEAVE\n"));
}

void init_backgroud_handling () {
    /* We use POSIX sigaction here,
     * so that we not depend different semantic
     * between SYSV and BSD signal(SIGCHLD, ...)
     */
    
    struct sigaction new_child;
    
    new_child.sa_flags    = 0;
#ifdef SA_INTERRUPT
    new_child.sa_flags |= SA_INTERRUPT;           /* SunOS? */
#endif
    new_child.sa_handler = got_sigchld;
    sigemptyset(& (new_child.sa_mask));
    
    if (-1 == sigaction(SIGCHLD,&new_child,NULL)) {
	int err = errno;
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSigaction,"sigaction: %s"),
		  strerror(err));
	exit (1);
    }
}

static void add_to_my_processes P_((struct run_state *rs,
				    union any_fd fd,
				    const char *title, 
				    end_handler *func,
				    free_any_fd_f         * free_any_fd,
				    inc_any_fd_refcount_f * inc_any_fd_refcount
				    ));
static void add_to_my_processes(rs,fd,title,func,
				free_any_fd,
				inc_any_fd_refcount
				)
     struct run_state * rs;
     union any_fd       fd;
     const char       * title;
     end_handler      * func;
     free_any_fd_f         * free_any_fd;
     inc_any_fd_refcount_f * inc_any_fd_refcount;
{
    struct process_list * listptr = safe_zero_alloc(sizeof (*listptr));
    
    rs -> listptr      = listptr;
    listptr -> magic   = PROCESS_LIST_magic;
    listptr -> message = safe_strdup(title);
    listptr -> state_information = *rs;
    listptr -> handler = func;
    listptr -> free_any_fd = free_any_fd;
    listptr ->  inc_any_fd_refcount =  inc_any_fd_refcount;

    listptr -> fd      = fd;
    listptr -> inc_any_fd_refcount(listptr -> fd);
    
    listptr -> next = my_processes;
    
    my_processes = listptr;
}


static  void no_ext_init P_((struct run_state *rs));
static  void no_ext_init(rs)
     struct run_state *rs;
{

}

void add_foreign_background(pid,fd,title,func,
			    free_any_fd,
			    inc_any_fd_refcount)
     int pid;
     union any_fd fd;
     const char *title; 
     end_handler *func;
     free_any_fd_f         * free_any_fd;
     inc_any_fd_refcount_f * inc_any_fd_refcount;
{
    struct run_state RS;
    
    /* bzero is defined hdrs/elm_defs.h */
    
    bzero((void *)&RS,sizeof RS);
    RS.pid        = pid;
    RS.save_errno = 0;
    RS.raw        = 0;   /* so that raw_exit() does nothing */
    RS.options    = 0;
    RS.listptr    = NULL;
    RS.ext_init   =  no_ext_init;  /* Only used on start_run() */
    RS.ext_init_data = NULL;
    RS.pfd           = NULL;

    add_to_my_processes(&RS,fd,title,func,
			free_any_fd,
			inc_any_fd_refcount);
}

int maybe_background (rs,exit_code,fd,title,func,
		      free_any_fd,
		      inc_any_fd_refcount) 
     struct run_state *rs;
     int *exit_code;
     union any_fd fd;
     const char *title; 
     end_handler *func;
     free_any_fd_f         * free_any_fd;
     inc_any_fd_refcount_f * inc_any_fd_refcount;
{
    int ret;
    
    if (!title)
	title = "NO NAME";

    DPRINT(Debug,10,(&Debug,"maybe_background: %s: rs { pid %d } raw = %d",
		     title,rs->pid,rs->raw));
    switch (rs->raw) {
    case OFF:    DPRINT(Debug,10,(&Debug," (OFF)"));    break;
    case ON:     DPRINT(Debug,10,(&Debug," (ON)"));    break;
    }  
    DPRINT(Debug,10,(&Debug,"\n"));

    
    if (rs->raw == ON && RawState_hook() == OFF) {
	
	ret = wait_end (rs,exit_code);
	if (ret != 0) {
	    DPRINT(Debug,10,(&Debug,"maybe_background=%d",ret));
	    
	    if (exit_code) {
		DPRINT(Debug,10,(&Debug,"; *exit_code=%d",
				 *exit_code));	 
	    }
	    
	    DPRINT(Debug,10,(&Debug,"\n"));
	    return ret;
	}
    }

    add_to_my_processes(rs,fd,title,func,
			free_any_fd,
			inc_any_fd_refcount);

    DPRINT(Debug,10,(&Debug,"maybe_background=0\n"));
    return 0;
} 
#endif

static void raw_exit P_((struct run_state *rs));
static void raw_exit(rs)
     struct run_state *rs;
{
    DPRINT(Debug,10,(&Debug,"raw_exit: rs { pid %d } raw = %d",
		     rs->pid,rs->raw));
    switch (rs->raw) {
    case OFF:    DPRINT(Debug,10,(&Debug," (OFF)"));    break;
    case ON:     DPRINT(Debug,10,(&Debug," (ON)"));    break;
    }  
    DPRINT(Debug,10,(&Debug,"\n"));

    
    if (rs->raw == ON && RawState_hook() == OFF) {
	DPRINT(Debug,4,(&Debug,
			"raw_exit: setting RAW on\n"));
	Raw_hook(ON|NO_TITE);
    }  else {
	DPRINT(Debug,4,(&Debug,
			"raw_exit: no state change\n"));
    }
}


int run_already_done (rs,exit_code)
     struct run_state *rs;
     int *exit_code;
{
    S__ status;
    int w UNUSED_VAROK;

    DPRINT(Debug,10,(&Debug,"run_already_done: rs { pid %d } raw = %d",
		     rs->pid,rs->raw));
    switch (rs->raw) {
    case OFF:    DPRINT(Debug,10,(&Debug," (OFF)"));    break;
    case ON:     DPRINT(Debug,10,(&Debug," (ON)"));    break;
    }  
    DPRINT(Debug,10,(&Debug,"\n"));

#ifdef HASWAITPID

    /* waitpid is on POSIX */
    
    *exit_code = -1;
    
    w = waitpid(rs->pid,&status,WNOHANG);
    
    if (w == 0) {
	DPRINT(Debug,2,(&Debug,
			"run_already_done=%d (w=%d)\n",0,w));
	return 0;
    }

    if (w == -1) {
	rs->save_errno = errno;
	DPRINT(Debug,2,(&Debug,
			"run_already_done: errno=%d\n", rs->save_errno));
	
	if (rs->save_errno == EINTR) {
	    DPRINT(Debug,2,(&Debug,
			    "run_already_done=%d (w=%d) [EINTR]\n",0,w));
	    return 0;
	}
    }
    
    raw_exit(rs);

    if(rs->pfd) {
	DPRINT(Debug,4,(&Debug,
			"run_already_done: closing rs->pfd=%p\n",
			rs->pfd));
	fclose(rs->pfd);
	rs->pfd = NULL;
    }

    if (w == rs->pid) {
	int sig = convert_status(status,exit_code);
	
	DPRINT(Debug,2,(&Debug,
			"run_already_done: exit_code=%d, sig=%d\n", 
			*exit_code,sig));
	
	print_status_hook(rs,sig,*exit_code) ;
	
	if (sig) {
	    DPRINT(Debug,2,(&Debug,
			    "run_already_done=%d\n",-sig));
	    return -sig;
	}
    }
    DPRINT(Debug,2,(&Debug,
		    "run_already_done=%d\n",w != -1));
    return w != -1;
#else
    DPRINT(Debug,2,(&Debug,
	       "run_already_done=%d (no HASWAITPID)\n",0));
    return 0;      
#endif
}

int wait_end  (rs,exit_code)
     struct run_state *rs;
     int *exit_code;
{
    int w;
    S__ status;

    DPRINT(Debug,10,(&Debug,"wait_end: rs { pid %d } raw = %d",
		     rs->pid,rs->raw));
    switch (rs->raw) {
    case OFF:    DPRINT(Debug,10,(&Debug," (OFF)"));    break;
    case ON:     DPRINT(Debug,10,(&Debug," (ON)"));    break;
    }  
    DPRINT(Debug,10,(&Debug,"\n"));
    
    *exit_code = -1;
    
    DPRINT(Debug,9,(&Debug,
		     "wait_end:  Waiting for pid %d\n",rs->pid));

    while ((w = my_wait(rs->pid,&status)) != rs->pid) {
	if (w == -1 && errno != EINTR) 
	    break;

	DPRINT(Debug,10,(&Debug,
			 "wait_end: got my_wait=%d (!= %d) .. looping\n",
			 w,rs->pid));
    }

    if (w == -1) {
	rs->save_errno = errno;
	DPRINT(Debug,2,(&Debug,
			"wait_end: errno=%d\n", rs->save_errno));    
    }
    
    raw_exit(rs);
    
    if(rs->pfd) {
	DPRINT(Debug,4,(&Debug,
			"wait_end: closing rs->pfd=%p\n",
			rs->pfd));
	fclose(rs->pfd);
	rs->pfd = NULL;
    }

    if (w == rs->pid) {
	int sig = convert_status(status,exit_code);
	
	DPRINT(Debug,2,(&Debug,
			"wait_end: exit_code=%d, sig=%d\n", 
			*exit_code,sig));
	
	print_status_hook(rs,sig,*exit_code);
	
	if (sig) {
	    DPRINT(Debug,2,(&Debug,
			    "wait_end=%d\n",-sig));
	    return -sig;
	}
    }
      
    DPRINT(Debug,2,(&Debug,
		    "wait_end=%d\n",w != -1));
    return w != -1;
}

const char ** join_argv(argv1,argv2)
     char * argv1[];
     char * argv2[];
{
    int count1;
    int count2;
    const char ** res;
    int i;
    for (count1 = 0; argv1[count1]; count1++);
    for (count2 = 0; argv2[count2]; count2++);
    
    res = safe_calloc((count1 + count2 + 1), sizeof (char *));
    
    for (i = 0; i < count1; i++)
	res[i] = argv1[i];
    for (i = 0; i < count2; i++)
	res[i + count1] = argv2[i];
    res[count1 + count2] = NULL;
    
    return res;
}

const char ** join_argvc1(argv1,argv2)
     const char * argv1[];
     char * argv2[];
{
    int count1;
    int count2;
    const char ** res;
    int i;

    for (count1 = 0; argv1[count1]; count1++);
    for (count2 = 0; argv2[count2]; count2++);
    
    res = safe_calloc((count1 + count2 + 1), sizeof (char *));
    
    for (i = 0; i < count1; i++)
	res[i] = argv1[i];
    for (i = 0; i < count2; i++)
	res[i + count1] = argv2[i];
    res[count1 + count2] = NULL;
    
    return res;
}

void sr_call_ClearScreen() 
{
    ClearScreen_hook();
}


void sr_call_Raw(x)
     int x;
{
    Raw_hook(x);
}

int sr_call_RawState()
{
    int x = RawState_hook();
    return x;
}

/* copied from curses.c */
void sr_call_Write_to_screen (
#if ANSI_C
		      const char *format, const char *msg, ...
#else
		      format, msg, va_alist
#endif
		      )
#if !ANSI_C
     const char *format; 
     const char *msg;
     va_dcl
#endif
{
    va_list vl;
    struct string *text;
    
    /** This routine writes to the screen at the current location.
	when done, it increments lines & columns accordingly by
	looking for "\n" sequences... **/

    Va_start(vl, msg);           /* defined in hdrs/elm_defs.h */
    text = elm_smessage(0,format,msg,vl);
    va_end(vl);
    
    PutLineS_hook(text);
    free_string(&text);
        
}

void call_print_status_cooked(rs,sig,exit_code)
     struct run_state *rs;
     int sig;
     int exit_code;
{
    DPRINT(Debug,10,(&Debug,
		    "call_print_status_cooked: exit_code=%d, sig=%d\n", 
		    exit_code,sig));   

    print_status_cooked_hook(rs,sig,exit_code);
}

int start_run(rs, options, argv, infd, outfd)
     struct run_state *rs;
     int options;
     const char * argv[];
     int infd, outfd;
{
    int count;
    int pfd[2], opipe[2];
    static int fd = -1;
    int notty    = (options & SY_NOTTY) != 0;
    int outpipe  = (options & SY_RUN_STATE_OPIPE) != 0;
    int errpipe  = (options & SY_RUN_STATE_EPIPE) != 0;
    
    DPRINT(Debug,2,(&Debug,
	       "start_run: [0] %s\n", argv[0]));
    for (count = 1; argv[count]; count++) {
	DPRINT(Debug,2,(&Debug,
			"           [%d] %s\n", 
			count,argv[count]));
    }
    DPRINT(Debug,2,(&Debug,
		    "    infd=%d, outfd=%d\n",infd,outfd));
    
    if (fd == -1)
	fd = open("/dev/null",O_RDWR);
    DPRINT(Debug,2,(&Debug,
		    "    fd=%d \n", fd));
    
    /* flush any pending output */
    fflush(stdout);
    rs->save_errno = 0;
    rs->raw     = RawState_hook();
    rs->options = options;
#ifdef BACKGROUD_PROCESSES      
    rs->listptr = NULL; 
#endif
    rs->pfd     = NULL;

    if (!notty) {
	run_tty_init_hook();

	if (rs->raw == ON) {
	    if (options & SY_CLRWAIT) {
		ClearScreen_hook();
	    }
	    DPRINT(Debug,4,(&Debug,
			    "start_run: setting RAW off\n"));
	    Raw_hook(OFF|NO_TITE);
	}
	if (options & SY_CLRWAIT) {
	    printf("Executing: %s ...\n\n",argv[0]);
	}
    }
    
    if (outpipe || errpipe) {
	if (-1 == pipe(opipe)) {
	    rs->save_errno = errno;
	    
	    raw_exit(rs);
	    return 0;
	}
    }

    if (pipe(pfd) == -1) {
	rs->save_errno = errno;
	
	if (outpipe || errpipe) {
	    close(opipe[0]);
	    close(opipe[1]);
	}

	raw_exit(rs);
	return 0;
    }

#ifdef FD_CLOEXEC
    fcntl(pfd[0], F_SETFD, FD_CLOEXEC);
    fcntl(pfd[1], F_SETFD, FD_CLOEXEC);
#else
    fcntl(pfd[0], F_SETFD, 1);
    fcntl(pfd[1], F_SETFD, 1);
#endif    

    rs->pid = fork();
    
    if (rs->pid == -1) {
	rs->save_errno = errno;

	if (outpipe || errpipe) {
	    close(opipe[0]);
	    close(opipe[1]);
	}
	close(pfd[0]);
	close(pfd[1]);
	
	raw_exit(rs);
	return 0;
    }
    else if (rs->pid == 0) {
	int r UNUSED_VAROK;

	close(pfd[0]);
	if (outpipe || errpipe) 
	    close(opipe[0]);

	/*
	 * Set group and user back to their original values.
	 * Note that group must be set first.
	 */
	if (-1 == setgid(groupid)) {
	    int err = errno;
	    int r UNUSED_VAROK;

	    fprintf(stderr,"start_run: setgid(%d) FAILED: %s\n",
		    groupid,strerror(err));
	    fflush(stderr);
	    r = write(pfd[1],(void *)&err,sizeof err); 
	    _exit(127); 
	}
	if (-1 == setuid(userid)) {
	    int err = errno;
	    int r UNUSED_VAROK;

	    fprintf(stderr,"start_run: setruid(%d) FAILED: %s\n",
		    userid,strerror(err));
	    fflush(stderr);
	    r = write(pfd[1],(void *)&err,sizeof err); 
	    _exit(127); 
	}
	
	set_child_signals(options);
	
	set_child_env(options);
	
	if (options & SY_RUN_STATE_ENV) {
	    int i;
	    for (i = 0; rs->ext_env[i]; i++) {
		if (0 == putenv(rs->ext_env[i])) {
		    DPRINT(Debug,5,(&Debug, 
				    "Child: added %s\n",
				    rs->ext_env[i]));
		}
	    }
	}

	if (options & SY_RUN_STATE_INIT) {
	    rs->ext_init(rs);
	}

	if (outpipe || errpipe) {
	    if (outpipe) {
		if (opipe[1] != 1) {
		    if (-1 == dup2(opipe[1],1)) {
			int r UNUSED_VAROK = write(pfd[1],(void *)&errno,
						   sizeof errno); 
			_exit(127); 
		    }	    
		}
	    }
	    if (errpipe) {
		if (opipe[1] != 2) {
		    if (-1 == dup2(opipe[1],2)) {
			int r UNUSED_VAROK = write(pfd[1],(void *)&errno,
						   sizeof errno); 
			_exit(127); 
		    }	    
		}
	    }
	    if (opipe[1] != 1 && opipe[1] != 2) {
		close(opipe[1]);
	    }
	}

	if (notty) {
	    if ((infd != 0  && -1 == dup2(fd,0)) ||
		(!outpipe && outfd != 1 && -1 == dup2(fd,1)) ||
		(!errpipe && outfd != 2 && -1 == dup2(fd,2))) { 
		int r UNUSED_VAROK = write(pfd[1],(void *)&errno,
					   sizeof errno); 
		_exit(127); 
	    }

#ifdef SIGTTSTP
	    signal(SIGTTIN, SIG_IGN); 
	    signal(SIGTTOU, SIG_IGN);
	    signal(SIGTSTP, SIG_IGN);
#endif
	}
	
	if (infd >= 0) {
	    if (infd != 0  && -1 == dup2(infd,0)) {
		int r UNUSED_VAROK = write(pfd[1],(void *)&errno,
					   sizeof errno); 
		_exit(127); 
	    }
	}
	if (outfd >= 0) {
	    if (outfd != 1  && -1 == dup2(outfd,1)) {
		int r UNUSED_VAROK = write(pfd[1],(void *)&errno,
					   sizeof errno); 
		_exit(127); 
	    }
	}



	/*              ... lose const */
	execvp(argv[0],(char **)argv);
	r = write(pfd[1],(void *)&errno,sizeof errno); 
	_exit(127); 
    }
    else {
	int code, rd;
	close(pfd[1]);
	
	DPRINT(Debug,4,(&Debug,
			"start_run: child pid=%d\n", rs->pid));
	rd = read(pfd[0],(void *)&code, sizeof code);
	close(pfd[0]);
	
	if (outpipe || errpipe) 
	    close(opipe[1]);


	if (rd > 0) {
	    int exitcode;
	    
	    if (outpipe || errpipe) 
		close(opipe[0]);

	    wait_end(rs,&exitcode);
	    if (rd == sizeof code)
		rs->save_errno = code;
	    else
		rs->save_errno = 0;
	    return 0;
	}

	if (outpipe || errpipe) {
	    rs->pfd = fdopen(opipe[0],"r");
	    if (!rs->pfd) {
		DPRINT(Debug,4,(&Debug,
				"start_run: Failed open rs->pfd (fd=%d)\n",
				opipe[0]));
		close(opipe[0]);
	    } else {
		DPRINT(Debug,4,(&Debug,
				"start_run: opened rs->pfd=%p (fd=%d)\n",
				rs->pfd,opipe[0]));
	    }
	}
	return 1;
    }
    raw_exit(rs);

    return 0;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */

