static char rcsid[] = "@(#)$Id: flowed.c,v 2.8 2023/12/13 16:55:32 hurtta Exp $";

/************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.8 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 ************************************************************************/


#include "def_melib.h"
#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"mime");

static const unsigned char *cs2us P_((const char *str));
static const unsigned char *cs2us(str)
     const char *str;
{
    return (const unsigned char *)str;
}

/* RFC 2646:   The Text/Plain Format and DelSp Parameters  */

struct flowed_line {
    int quote_level;

    enum fl_line_class {
	fl_special,
	fl_fixed,
	fl_flowed,
	fl_too_long
    }             line_class;
    unsigned int   have_nl :1;   /* newline read (and not stored) */
    unsigned int   eof :1;       /* got EOF (or error)              */
    unsigned int   error :1;     /* error on input line */

    /* On UTF-7 character set  > -character is on 
       "optional direct characters" -set, therefore 
       > -character may be UTF-7 encoded. Must operate
       with characters when charset is taken account!
    */

    struct string        * sbuffer;

    /* Process one character on time */
    struct charset_state * rchar;
    charset_t              text_charset;
    
};

#if DEBUG
static const char * line_class_string P_((enum fl_line_class lc));
static const char * line_class_string(lc)
     enum fl_line_class lc;
{
    switch (lc) {
    case fl_special: return "special";
    case fl_fixed:   return "fixed";
    case fl_flowed:  return "flowed";
    case fl_too_long: return "too long";
    }

    return "<unknown>";
}
#endif

static void flowed_line_init P_((struct flowed_line *line,charset_t text_charset));
static void flowed_line_init(line,text_charset)
     struct flowed_line *line;
     charset_t text_charset;
{
    /* bzero is defined on hdrs/defs.h */
    bzero((void *)line, sizeof (*line));
    
    line->quote_level = 0;
    line->line_class  = fl_special;
    line->have_nl     = 0;
    line->eof         = 0;
    line->error       = 0;

    line->sbuffer = NULL;
    line->rchar  = NULL;
    line->text_charset = text_charset;
}

static void flowed_line_dest P_((struct flowed_line *line));
static void flowed_line_dest(line)
     struct flowed_line *line;
{
    line->quote_level = 0;
    line->line_class  = fl_special;
    line->have_nl     = 0;
    line->eof         = 0;
    line->error       = 0;

    if (line->sbuffer) 
	free_string(& (line->sbuffer));
    if (line->rchar)
	free_state( & (line->rchar));
}


/* flowed lines should be under 80 characters */
#define MAX_FLOWED_LINE  1002

/* Returns -1 if EOF (or error seen) before any characters */
static int get_new_flowed_line P_((struct flowed_line *line, 
			    struct in_state   *state_in,
			    int delsp));
static int get_new_flowed_line(line,state_in,delsp)
     struct flowed_line *line; 
     struct in_state   *state_in;
     int delsp;
{
    int ret = 0;
    int gotbyte = 0;
    int l;

    if (!line->rchar)
	line->rchar = new_state(line->text_charset);
    else
	reset_state(line->rchar,0);  

    if (line->sbuffer)
	free_string(& (line->sbuffer));
    line->sbuffer = new_string(line->text_charset);

    line->quote_level = 0;
    line->error       = 0;
    line->line_class  = fl_fixed;
    line->have_nl     = 0;
    line->eof         = 0;

    do {

	if (state_ready(line->rchar)) {
	    uint16 u = give_unicode_from_state(line->rchar);
	    
	    if (0x003E /* > */ != u)
		break;
	    
	    /* Count quote level */
	    line->quote_level++;
	    reset_state(line->rchar,0);  /* Get next character */

	} else {
	    int ch = state_getc(state_in);
	
	    if (EOF == ch) {

		if (!gotbyte)  {
		    ret = -1;
		    line->line_class  = fl_special;
		    
		    if (line->sbuffer)
			free_string(& (line->sbuffer));
		}
		line->eof         = 1;
		
		goto out;
	    }
	    gotbyte++;

	    if (! add_streambyte_to_state(line->rchar,ch)) {
		uint16 bad_char = UNICODE_BAD_CHAR;
		
		DPRINT(Debug,20,(&Debug, 
				 "get_new_flowed_line: Failed to add byte %02x to character when scanning quote prefix\n"));
		line->error       = 1;
		reset_state(line->rchar,0);  
		
		add_unicode_to_string(line->sbuffer,1,&bad_char);
		goto on_line;
	    }
	    
	}
    } while(1);

    do {
	
	if (state_ready(line->rchar)) {
	    uint16 u = give_unicode_from_state(line->rchar);
	    
	    /* Remove space-stuffing */
	    if (0x0020 /* SPACE */ == u)
		reset_state(line->rchar,0);  /* Get next character */
	    
	    break;

	} else {
	    int ch = state_getc(state_in);
	
	    if (EOF == ch) {
		line->eof         = 1;
		
		goto out;
	    }
	    gotbyte++;

	    if (! add_streambyte_to_state(line->rchar,ch)) {
		uint16 bad_char = UNICODE_BAD_CHAR;
		
		DPRINT(Debug,20,(&Debug, 
				 "get_new_flowed_line: Failed to add byte %02x to character when scanning space-stuffing\n"));
		line->error       = 1;
		reset_state(line->rchar,0);  
		
		add_unicode_to_string(line->sbuffer,1,&bad_char);
		goto on_line;
	    }
	}
    } while(1);

 on_line:

    do {  /* NOTE:  It is assumed that CRLF are same than on ascii */
	int ch = EOF;

	if (state_ready(line->rchar)) {
	    uint16 u = give_unicode_from_state(line->rchar);
	    
	    if (0x000D /* CR  '\r' */ == u) {
		int peekch = state_getc(state_in);

		if ('\n' == peekch) {   /* Got newline */
		    ch = peekch;

		    reset_state(line->rchar,0);
		    goto process_char;
		} else if (EOF != peekch)
		    state_ungetc(peekch,state_in);
	    } else if (0x000A /* LF  '\n' */ == u) {
		line->have_nl = 1;
		
		reset_state(line->rchar,0);
		break;   /* Line done */
	    }
	    
	    /* Returns no status */
	    add_state_to_string(line->sbuffer,line->rchar);
	    reset_state(line->rchar,0);  /* Get next character */

	    l = string_len(line->sbuffer);
	    if (l >= MAX_FLOWED_LINE) {
		DPRINT(Debug,20,(&Debug, 
				 "get_new_flowed_line: Read %d characters\n",
				 l));
		break;
	    }
	} else {
	    ch = state_getc(state_in);
	    
	process_char:
	    if (EOF == ch) {
		line->eof         = 1;
		
		goto out;
	    }
	    gotbyte++;

	    if (! add_streambyte_to_state(line->rchar,ch)) {
		uint16 bad_char = UNICODE_BAD_CHAR;
		
		DPRINT(Debug,20,(&Debug, 
				 "get_new_flowed_line: Failed to add byte %02x to character when reading line\n"));
		line->error       = 1;
		reset_state(line->rchar,0);  
		
		add_unicode_to_string(line->sbuffer,1,&bad_char);
	    }
	}

    } while(1);

    l = string_len(line->sbuffer);
    if (l > 0 && l < MAX_FLOWED_LINE
	&& 0x0020 /* SPACE */ == give_unicode_from_string(line->sbuffer,l-1)) {

	/* Usenet Signature Convention */
	if (3 == l && string_matches_ascii(line->sbuffer,
					   cs2us("-- "),0,SMA_op_normal))
	    line->line_class  = fl_special;
	else {
	    line->line_class  = fl_flowed;
	    
	    if (delsp) {
		int X = 0;
		struct string * S = clip_from_string(line->sbuffer,&X,l-1);

		free_string(&(line->sbuffer));
		line->sbuffer = S;
	    }
	}
    } else if (l >= MAX_FLOWED_LINE)
	line->line_class  = fl_too_long;

    ret = l;

 out:

    DPRINT(Debug,20,(&Debug, 
		     "get_new_flowed_line=%d (%d bytes), quote level=%d, line class=%s%s%s",
		     ret,gotbyte,line->quote_level,
		     line_class_string(line->line_class),
		     line->eof     ? ", got EOF" : "",
		     line->error   ? ", got bad sequence of bytes" : ""));

    if (line->sbuffer) {
	DPRINT(Debug,25,(&Debug,", buffer=%S",
			 line->sbuffer));
    }

    DPRINT(Debug,20,(&Debug, "\n"));
		     		   
    return ret;
}


static int collect_special P_((mime_t *body, struct in_state   *state_in, 
			       struct out_state  *state_out,
			       const struct decode_opts *decode_opt,
			       struct flowed_line *current_line,
			       int delsp, int *error));
static int collect_special(body,state_in,state_out,decode_opt,current_line,
			   delsp,error)
     mime_t *body; 
     struct in_state   *state_in; 
     struct out_state  *state_out;
     const struct decode_opts *decode_opt;
     struct flowed_line *current_line;
     int delsp;
     int *error;
{
    int r = 0;
    int last_level = current_line->quote_level;
    struct decode_opts  DECODE_OPT = *decode_opt;

    struct pager_range * decode_data = NULL;


    if (last_level > 0) {
	decode_data =  state_add_simple_pager_range(state_out,
						    decode_opt->range,0,
						    last_level,0);

	DPRINT(Debug,15,(&Debug,
			 "collect_special: decode_data quote_level=%d range_flags=%d\n",
			 get_pager_range_quote_level(decode_data),
			 get_pager_range_flags(decode_data)));

	DECODE_OPT.range  =  decode_data;
    }

    while (fl_special == current_line->line_class &&
	   last_level == current_line->quote_level &&
	   r >= 0) {

	if (current_line->error && error)
	    *error = 1;

	state_add_prefix(state_out,&DECODE_OPT);

	if (current_line->sbuffer)
	    state_putstring(current_line->sbuffer,state_out); 

	if (current_line->have_nl)
	    state_putc('\n',state_out);

	r = get_new_flowed_line(current_line,state_in,delsp);
    }

    if (decode_data)
	free_pager_range(&decode_data);

    return r;
}

static int collect_fixed P_((mime_t *body, struct in_state   *state_in, 
			     struct out_state  *state_out,
			     const struct decode_opts *decode_opt,
			     struct flowed_line *current_line,
			     int delsp, int *error));
static int collect_fixed(body,state_in,state_out,decode_opt,current_line,delsp,error)
     mime_t *body; 
     struct in_state   *state_in; 
     struct out_state  *state_out;
     const struct decode_opts *decode_opt;
     struct flowed_line *current_line;
     int delsp;
     int *error;
{
    /* Almost as collect_special */

    int r = 0;
    int last_level = current_line->quote_level;
    struct decode_opts  DECODE_OPT = *decode_opt;

    struct pager_range * decode_data =  
	state_add_simple_pager_range(state_out,decode_opt->range,PR_PREFORMAT,
				     last_level,0);

    DPRINT(Debug,15,(&Debug,
		     "collect_fixed: decode_data quote_level=%d range_flags=%d\n",
		     get_pager_range_quote_level(decode_data),
		     get_pager_range_flags(decode_data)));
    
    DECODE_OPT.range  =  decode_data;

    while (fl_fixed == current_line->line_class &&
	   last_level == current_line->quote_level &&
	   r >= 0) {

	if (current_line->error && error)
	    *error = 1;

	state_add_prefix(state_out,&DECODE_OPT);

	if (current_line->sbuffer)
	    state_putstring(current_line->sbuffer,state_out); 

	if (current_line->have_nl)
	    state_putc('\n',state_out);

	r = get_new_flowed_line(current_line,state_in,delsp);
    }

    free_pager_range(&decode_data);

    return r;

}

static int collect_flowed P_((mime_t *body, struct in_state   *state_in, 
			      struct out_state  *state_out,
			      const struct decode_opts *decode_opt,
			      struct flowed_line *current_line,
			      int delsp, int *error));
static int collect_flowed(body,state_in,state_out,decode_opt,current_line,delsp,error)
     mime_t *body; 
     struct in_state   *state_in; 
     struct out_state  *state_out;
     const struct decode_opts *decode_opt;
     struct flowed_line *current_line;
     int delsp;
     int *error;
{
    /* flowed paragraph   consists
       current_line->line_class  == fl_flowed lines followed by one
       current_line->line_class  == fl_fixed line */

    int r = 0;
    int last_level = current_line->quote_level;
    struct decode_opts  DECODE_OPT = *decode_opt;

    struct pager_range * decode_data =  
	state_add_simple_pager_range(state_out,decode_opt->range,
				     PR_WORD_WRAP | PR_JOIN_LINES, 
				     last_level,0);

    DPRINT(Debug,15,(&Debug,
		     "collect_flowed: decode_data quote_level=%d range_flags=%d\n",
		     get_pager_range_quote_level(decode_data),
		     get_pager_range_flags(decode_data)));
    
    DECODE_OPT.range  =  decode_data;

    /* 1) Flowed lines */

    while (fl_flowed == current_line->line_class &&
	   last_level == current_line->quote_level &&
	   r >= 0) {

	if (current_line->error && error)
	    *error = 1;

	state_add_prefix(state_out,&DECODE_OPT);

	if (current_line->sbuffer)
	    state_putstring(current_line->sbuffer,state_out); 

	if (current_line->have_nl)
	    state_putc('\n',state_out);

	r = get_new_flowed_line(current_line,state_in,delsp);
    }

    /* 2) and fixed line */

    if (fl_fixed == current_line->line_class &&
	last_level == current_line->quote_level &&
	r >= 0) {
	
	if (current_line->error && error)
	    *error = 1;

	state_add_prefix(state_out,&DECODE_OPT);
	
	if (current_line->sbuffer)
	    state_putstring(current_line->sbuffer,state_out); 
	
	if (current_line->have_nl)
	    state_putc('\n',state_out);
	
	r = get_new_flowed_line(current_line,state_in,delsp);
    }
    

    free_pager_range(&decode_data);

    return r;
}

static int collect_overflow P_((mime_t *body, struct in_state   *state_in, 
				struct out_state  *state_out,
				const struct decode_opts *decode_opt,
				struct flowed_line *current_line,
				int delsp, int *error));
static int collect_overflow(body,state_in,state_out,decode_opt,
			    current_line,delsp,error)
     mime_t *body; 
     struct in_state   *state_in; 
     struct out_state  *state_out;
     const struct decode_opts *decode_opt;
     struct flowed_line *current_line;
     int delsp;
     int *error;
{
    /* Almost as collect_special */

    int r = 0;
    int last_level = current_line->quote_level;
    struct decode_opts  DECODE_OPT = *decode_opt;

    struct pager_range * decode_data = NULL;


    if (last_level > 0) {
	decode_data =  state_add_simple_pager_range(state_out,
						    decode_opt->range,0,
						    last_level,0);

	DPRINT(Debug,15,(&Debug,
			 "collect_overflow: decode_data quote_level=%d range_flags=%d\n",
			 get_pager_range_quote_level(decode_data),
			 get_pager_range_flags(decode_data)));

	DECODE_OPT.range  =  decode_data;
    }

    while (fl_too_long == current_line->line_class &&
	   last_level == current_line->quote_level &&
	   r >= 0) {

	if (current_line->error && error)
	    *error = 1;

	state_add_prefix(state_out,&DECODE_OPT);


	if (current_line->sbuffer)
	    state_putstring(current_line->sbuffer,state_out); 

	if (current_line->have_nl)   /* Should not happen */
	    state_putc('\n',state_out);
	else {

	    /* Copy rest of line, which was not read because of overflow 
	     * This does not check is line flowed or fixed
	     */

	    do {  /* NOTE:  It is assumed that CRLF are same than on ascii */
		int ch = EOF;

		if (state_ready(current_line->rchar)) {
		    uint16 u = give_unicode_from_state(current_line->rchar);
		    
		    if (0x000D /* CR  '\r' */ == u) {
			int peekch = state_getc(state_in);
			
			if ('\n' == peekch) {   /* Got newline */
			    ch = peekch;
			    
			    reset_state(current_line->rchar,0);
			    goto process_char;
			} else if (EOF != peekch)
			    state_ungetc(peekch,state_in);
			
		    } else if (0x000A /* LF  '\n' */ == u) {
		
			reset_state(current_line->rchar,0);
			break;   /* Line done */
		    }

		    state_putstchar(current_line->rchar,state_out);
		    reset_state(current_line->rchar,0);  /* Get next character */
		} else {
		    ch = state_getc(state_in);
		    
		process_char:
		    if (EOF == ch) {
			current_line->eof         = 1;
			
			goto out;
		    }
		    
		    if (! add_streambyte_to_state(current_line->rchar,ch)) {
			uint16 bad_char = UNICODE_BAD_CHAR;
			
			DPRINT(Debug,20,(&Debug, 
					 "collect_overflow: Failed to add byte %02x to character when reading line\n"));
			if (error)
			    *error = 1;
			reset_state(current_line->rchar,0);  

			state_putunicode(1,&bad_char,state_out);
		    }

		}
	    } while(1);

	}

	r = get_new_flowed_line(current_line,state_in,delsp);
    }

 out:

    if (decode_data)
	free_pager_range(&decode_data);

    return r;
}

void flowed_text_decoder(body,state_in,state_out,decode_opt,text_charset)
     mime_t *body;
     struct in_state   *state_in;
     struct out_state  *state_out;
     const struct decode_opts *decode_opt;
     charset_t text_charset;  
{
    const char *pv;
    int delsp = 0;
    int r;
    int errors2 UNUSED_VAROK;
    int overflow = 0;
    int error = 0;

    struct flowed_line current_line;

    if (body->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"flowed_text_decoder",
		   "Bad magic number");

    pv = get_mime_param_compat(body->TYPE_opts,"delsp");
    
    DPRINT(Debug,15,(&Debug,"flowed_text_decoder: "));
    if (pv) {
	DPRINT(Debug,15,(&Debug,"delsp=%s",pv));

	delsp = 0 == istrcmp(pv,"yes");
    }
    DPRINT(Debug,15,(&Debug,"\n"));

    flowed_line_init(&current_line,text_charset);

    r = get_new_flowed_line(&current_line,state_in,delsp);

    while (r >= 0) {
	int errors;
	int errflag = 0;
	enum fl_line_class prev UNUSED_VAROK = 
	    current_line.line_class;

	switch (current_line.line_class) {
	    
	case fl_special:
	    
	    r = collect_special(body,state_in,state_out,decode_opt,
				&current_line,delsp,&errflag);
	    
	    break;
	    
	case fl_fixed:
	    
	    r = collect_fixed(body,state_in,state_out,decode_opt,
			      &current_line,delsp,&errflag);
	    
	    break;
	    
	case fl_flowed:
	    
	    r = collect_flowed(body,state_in,state_out,decode_opt,
			       &current_line,delsp,&errflag);
	    
	    break;
	    
	case fl_too_long:
	    
	    if (!overflow) {
		struct pager_range *title_range = 
		    state_add_simple_pager_range(state_out,NULL,PR_MAX_WIDTH,
						 0,0);

		
		/* \n resets this */
		set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */); 
		state_printf(state_out,
			     CATGETS(elm_msg_cat, MeSet, 
				     MeDecodeTooLongLines,
				     "[ text/plain format with too long lines, treating like fixed format. ]\n"));

		free_pager_range(&title_range);

		overflow = 1;
	    }

	    r = collect_overflow(body,state_in,state_out,decode_opt,
				 &current_line,delsp,&errflag);
	    
	    break;
	    
	default:
	    mime_panic(__FILE__,__LINE__,"flowed_text_decoder",
		       "Bad current_line.line_class");
	    
	}

	if (errflag && !error) {
	    const char * text_charset_MIME_name = get_charset_MIME_name(text_charset);

	    struct pager_range *title_range = 
		state_add_simple_pager_range(state_out,NULL,PR_MAX_WIDTH,0,
					     0);
	    /* \n resets this */
	    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */); 

	    if (text_charset_MIME_name)
		state_printf(state_out,
			     CATGETS(elm_msg_cat, MeSet, 
				     MeDecodeBadCharacters,
				     "[ text/plain format have some bytes, which can't be interpreted as characters of charset %s. ]\n"),
			     text_charset_MIME_name);
	    else
		state_printf(state_out,
			     CATGETS(elm_msg_cat, MeSet, 
				     MeDecodeBadCharacters1,
				     "[ text/plain format have some  bytes, which can't be interpreted as characters. ]\n"));
				     

	    free_pager_range(&title_range);
	    error = 1;
	}
	

	errors = print_in_errors(state_in,state_out,decode_opt);

	if (errors > 0) {
	    DPRINT(Debug,10,(&Debug, 
			     "flowed_text_decoder: %d errors when reading %s, block return %d; next block %s\n",
			     errors,line_class_string(prev),r, line_class_string(current_line.line_class)));
	    
	    /* Retry after errors */
	    if ((!current_line.sbuffer || string_len(current_line.sbuffer) < 1) 
		&& r < 0)
		r = get_new_flowed_line(&current_line,state_in,delsp);
	}

    }

    flowed_line_dest(&current_line);

    errors2 = print_in_errors(state_in,state_out,decode_opt);
    
    DPRINT(Debug,10,(&Debug, 
		     "flowed_text_decoder: %d errors on end\n",errors2));
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
