static char rcsid[] = "@(#)$Id: state_decode.c,v 2.18 2022/11/30 18:01:38 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.18 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                       (was hurtta+elm@posti.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 *
 ******************************************************************************
 *  Some code copied from lib/mime_decode.c. It have following note:
 *
 *  Initially written by: Michael Elkins <elkins@aero.org>, 1995
 *****************************************************************************/

#include "def_melib.h"
#include "state_imp.h"
#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"mime");
DEBUG_VAR(DebugIO,__FILE__,"stateio");


static char *us2s P_((unsigned char *str));
static char *us2s(str) 
     unsigned char *str;
{
    return (char *)str;
}

/* ----------------------------------------------- */

#define STATE_decode_type_magic	0xEA05

/* Return 1 on succeed, 0 on error */
typedef int decode_initial_scan_f P_((struct in_state *s));

/* Return EOF when length is done or all is decoded */
typedef int decode_getc_f         P_((struct in_state *s)); 

/* Return 1 on succeed, 0 on error */
typedef int decode_finalize_f     P_((struct in_state *s)); 

typedef void decode_init_f        P_((struct in_state *s)); 
typedef void decode_dest_f        P_((struct in_state *s));

 
static struct state_decode_type {
    unsigned short magic;     /* STATE_decode_type_magic */

    decode_initial_scan_f   * decode_initial_scan;
    decode_getc_f           * decode_getc;
    decode_finalize_f       * decode_finalize;

    
    decode_init_f           * decode_init;
    decode_dest_f           * decode_dest;
} * decode_types [ ENCODING_count ] = {
    NULL        /* ENCODING_NONE   */,
    NULL        /* ENCODING_7BIT   */,
    NULL        /* ENCODING_8BIT   */,
    NULL        /* ENCODING_BINARY */,
    NULL        /* ENCODING_QUOTED */,
    NULL        /* ENCODING_BASE64 */,
    NULL        /* ENCODING_EXPERIMENTAL */,
    NULL        /* ENCODING_UUENCODED */   
};





#define IN_STATE_decode_magic	0xEA04

struct in_state_decode {
    unsigned short magic;     /* IN_STATE_decode_magic */

    struct in_state  * source;
    int                length;

    unsigned int       is_text : 1;         /* treate input as text 
					       (ie includes newlines) */
    unsigned int       EOLN_is_CRLF : 1;    /* Output newline as CRLF */
    unsigned int       from_buffer : 1;     /* Reading buffered output */
    unsigned int       unlink_name : 1;

    unsigned int       initial_scan_done : 1;
    unsigned int       end_decoded  : 1;
    unsigned int       finalize_done : 1;

    unsigned int       corrupted : 1;           /* Mark for corruption */

#define STORE_CH_LEN 2
    unsigned int       store_ch_len : 2;        /* store_ch buffer len - 
						   2 bits is enough
						*/
						       
    unsigned char store_ch[STORE_CH_LEN];             /* stack for buffered \r\n */


    
    struct state_decode_type   * decode_type;

    struct string    * decode_error;

    int                input_count;         /* number of bytes readed from source */
    int                decoded_count;       /* number of bytes decoded    note EOLN */

    int                lone_cr_convert_count; /* number of lone CR converted */
    struct delay_unlink * timer_unlink;
    
    char             * buffer_name;
    FILE             * buffer_F;

    int                buffer_pos;          /* position when reading from buffer, 
					       buffer_pos <= decoded_count */

    /* ungetc does not change position or counters .... */

    unsigned char    * unget_buffer;
    int                unget_buffer_len;
    
    union {
	void                 * dummy;
	struct base64_decode * base64;
	struct uudecode      * uuencoded;
	struct quoted_printable_decode * quoted_printable;

    }                 decode_data;

};

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

/* limit range ------------------------------------------------------------- */

/* Returns EOF when limit reached */
static int limit_getc P_((struct in_state_decode *st));
static int limit_getc(st)
     struct in_state_decode *st;
{
    int ch = EOF;
    
    if (st->input_count >= st->length) {	
	DPRINT(DebugIO,19,(&DebugIO,
			   "limit_getc: input-count=%d >= length=%d\n",
			   st->input_count,
			   st->length));
	goto handle_EOF;
    }  

    ch = state_getc(st->source);
    if (EOF == ch) {
	if (in_state_ferror(st->source)) {
	    DPRINT(DebugIO,19,(&DebugIO,
			       "limit_getc: [%p]  Error on source\n",
			       st->source));
	}
	
	DPRINT(DebugIO,17,(&DebugIO,
			   "limit_getc: [%p] got EOF from source, input-count=%d\n",
			   st->source,st->input_count));
	goto handle_EOF;
    } 
    
    st->input_count++;
    
 handle_EOF:
	
    if (EOF == ch) {
	DPRINT(DebugIO,29,(&DebugIO,
			   "limit_getc=EOF [%p]\n",
			   st->source));
    } else if (isascii(ch) && isprint(ch)) {
	DPRINT(DebugIO,29,(&DebugIO,
			   "limit_getc=%d '%c' [%p]\n",
			   ch,ch,st->source));
    } else {
	DPRINT(DebugIO,29,(&DebugIO,
			   "limit_getc=%d [%p]\n",
			   ch,st->source));
    }
    
    return ch;
}



/* store_ch handling ------------------------------------------------------- */

static int store_c_enter_getc P_((struct in_state *s, int *res_ch));
static int store_c_enter_getc(s,res_ch)
     struct in_state *s; 
     int *res_ch;
{
    int x;
    
    if ((x = s->u.decode->store_ch_len) > 0 &&
	'\r' != s->u.decode->store_ch[x-1]) {

	*res_ch = s->u.decode->store_ch[x-1];
	s->u.decode->store_ch[x-1] = '\0';

	DPRINT(Debug,47,(&Debug,
			 "store_c_enter_getc: returning cached #%d = %d\n",
			 x-1,			 
			 *res_ch));

	s->u.decode->store_ch_len--;
	

	return 1;
    }

    return 0;    
}

static int store_c_middle_getc P_((struct in_state *s, int *res_ch));
static int store_c_middle_getc(s,res_ch)
     struct in_state *s; 
     int *res_ch;
{
    int x;

    enum convert_cr_v convert_CR_to_newline_hack =
	give_dt_enumerate_as_int(&convert_cr_to_newline_hack);
    
    if ((x = s->u.decode->store_ch_len) > 0) {

	int pop_ch;
	
	/* Now s->u.decode->store_ch == '\r' 

	   store_c_enter_getc() handles case where
	   s->u.decode->store_ch != '\r'

	   Also  s->u.decode->is_text should be
	   text, or s->u.decode->store_ch should
	   not be used at all.

	 */
	
	if ('\r' != s->u.decode->store_ch[x-1])
	    panic("STATE PANIC",__FILE__,__LINE__,"store_c_middle_getc",
		  "bad store_ch",0);

	pop_ch = s->u.decode->store_ch[x-1];
	
	if (! s->u.decode->is_text)
	    panic("STATE PANIC",__FILE__,__LINE__,"store_c_middle_getc",
		  "Not in text mode",0);
	    	
	if ('\n' != *res_ch) {

	    /* Replacing character */

	    s->u.decode->store_ch[x-1] = *res_ch;

	    switch (convert_CR_to_newline_hack) {
	    case convert_cr_off:
		break;
	    case convert_cr_error:
	    case convert_cr_silent:
		if (s->u.decode->EOLN_is_CRLF) {
		    
		    if (s->u.decode->store_ch_len >= sizeof (s->u.decode->store_ch))  {
			
			 DPRINT(Debug,47,(&Debug,			      
					  "store_c_middle_getc: Cache full?\n"));
		    } else {
			s->u.decode->store_ch_len++;

			/* This works as stack */
			
			s->u.decode->store_ch[x] =  '\n';
			*res_ch = '\r';

			DPRINT(Debug,47,(&Debug,			      
					 "store_c_middle_getc: caching #%d %d #%d %d, returning CR %d\n",
					 x-1,s->u.decode->store_ch[x-1],
					 x,s->u.decode->store_ch[x],
					 *res_ch));

			s->u.decode->lone_cr_convert_count++;
			
			return 1;
		    }
		
		} else {
		    *res_ch = '\n';
		    
		    DPRINT(Debug,47,(&Debug,			      
				     "store_c_middle_getc: caching #%d %d, returning newline %d\n",
				     x-1,s->u.decode->store_ch[x-1], *res_ch));
		    
		    s->u.decode->lone_cr_convert_count++;
		 
		    return 1;
		}
		break;
	    case NUM_convert_cr_hack:
		break;
	    }
			    
	    *res_ch = '\r';

	    DPRINT(Debug,47,(&Debug,
			     "store_c_middle_getc: caching #%d %d, returning cached %d CR\n",
			     x-1,s->u.decode->store_ch[x-1], *res_ch));

	    return 1;
	}

	DPRINT(Debug,47,(&Debug,
			 "store_c_middle_getc: Popping (discarding) cached #%d %d CR\n",
			 x-1,pop_ch));
	
	s->u.decode->store_ch_len--;
    }

    /* Previous lines should have popped one character from cache */
    
    if (s->u.decode->store_ch_len >= sizeof (s->u.decode->store_ch)) 
	panic("STATE PANIC",__FILE__,__LINE__,"store_c_middle_getc",
	      "Cache full",0);
    
    if (s->u.decode->is_text && '\r' == *res_ch) {

	x = s->u.decode->store_ch_len++;
	
	s->u.decode->store_ch[x] = '\r';

	*res_ch = EOF;    /* Not character readed yet */

	DPRINT(Debug,47,(&Debug,
			 "store_c_middle_getc: caching #%d %d CR, returning continue\n",
			 x,s->u.decode->store_ch[x]));
	


	return -1;   /* translate to continue */
    }

    if (s->u.decode->is_text && s->u.decode->EOLN_is_CRLF && '\n' == *res_ch) {

	x = s->u.decode->store_ch_len++;
	
	s->u.decode->store_ch[x] = '\n';

	*res_ch = '\r';

	DPRINT(Debug,47,(&Debug,
			 "store_c_middle_getc: caching #%d %d, returning new %d CR\n",
			 x,s->u.decode->store_ch[x], *res_ch));

	return 1;
    }

    return 0;
}

static int store_c_eof_getc P_((struct in_state *s, int *res_ch));
static int store_c_eof_getc(s,res_ch)
     struct in_state *s; 
     int *res_ch;
{
    int x;
    
    /* Make sure to flush anything left in the internal buffer. */

    if ((x = s->u.decode->store_ch_len) > 0) {
	*res_ch = s->u.decode->store_ch[x-1];
        s->u.decode->store_ch[x-1] = '\0';

	DPRINT(Debug,47,(&Debug,
			 "store_c_eof_getc: returning cached #%d %d\n",
			 x-1,*res_ch));

	s->u.decode->store_ch_len--;
	
	return 1;
    }

    return 0;

}

/* ------------------------------------------------------------------------- */
/* Xbit_decode */

S_(decode_initial_scan_f Xbit_initial_scan)
static int Xbit_initial_scan P_((struct in_state *s));
static int Xbit_initial_scan(s)
     struct in_state *s;
{
    /* Nothing to do */

    return 1;
}


/* Return EOF when length is done or all is decoded */
S_(decode_getc_f Xbit_getc)
static int Xbit_getc         P_((struct in_state *s)); 
static int Xbit_getc(s)
     struct in_state *s;
{

    /* \r\n and \n    both are translated either \r\n or \n
       if s->u.decode->is_text  is set
    */ 

    int ch = EOF;

    if (store_c_enter_getc(s,&ch)) {

	DPRINT(Debug,40,(&Debug, 
			 "Xbit_getc=%d (cached)\n",ch));

	return ch;
    }

    while (EOF == ch) {
	int r UNUSED_VAROK;
	
	ch = limit_getc(s->u.decode);

	if (EOF == ch) {
	    	    
	    DPRINT(Debug,10,(&Debug, "Xbit_getc: got EOF\n"));

	    goto handle_EOF;
	}
	
	r = store_c_middle_getc(s,&ch);

	DPRINT(Debug,45,(&Debug, 
			 "Xbit_getc: r = %d, ch = %d %s\n",r,ch, (ch == EOF) ? "EOF" : ""));
    }


    DPRINT(Debug,40,(&Debug, 
		     "Xbit_getc=%d",ch));
    return ch;


 handle_EOF:    
    /* Make sure to flush anything left in the internal buffer. */

    if (store_c_eof_getc(s,&ch)) {
	DPRINT(Debug,40,(&Debug, 
			 "Xbit_getc=%d (at EOF)\n",ch));
	
	return ch;
    }

    if (s->u.decode->input_count >= s->u.decode->length ||
	in_state_feof(s->u.decode->source)) {	    
	s->u.decode->end_decoded = 1;

	DPRINT(Debug,15,(&Debug, 
			 "Xbit_getc: 'end decoded'\n"));
    }


    DPRINT(Debug,40,(&Debug, 
		     "Xbit_getc=EOF\n"));
    return EOF;
}

/* Return 1 on succeed, 0 on error */
S_(decode_finalize_f Xbit_finalize)
static int Xbit_finalize     P_((struct in_state *s)); 
static int Xbit_finalize(s)    
     struct in_state *s;
{

    if (s->u.decode->store_ch_len > 0)
	return 0;
    
    if (! s->u.decode->end_decoded)
	return 0;
    
    return 1;
}

S_(decode_init_f Xbit_init)
static void Xbit_init        P_((struct in_state *s)); 
static void Xbit_init(s)
     struct in_state *s;
{
    int x;
    
    s->u.decode->decode_data.dummy = NULL;
    s->u.decode->store_ch_len = 0;

    for (x = 0; x < sizeof (s->u.decode->store_ch); x++)
	s->u.decode->store_ch[x] = 0;

} 

S_(decode_dest_f   Xbit_dest)
static void Xbit_dest        P_((struct in_state *s));
static void Xbit_dest(s)
     struct in_state *s;
{
    s->u.decode->decode_data.dummy = NULL;
}



static struct state_decode_type Xbit_DECODE = {  
    STATE_decode_type_magic,

    Xbit_initial_scan,
    Xbit_getc,
    Xbit_finalize,
    Xbit_init,
    Xbit_dest
}; 

/* radix64 -- base64 and uuencode ------------------------------------------ */

/* based on from base64_decode() on mime_decode.c. It have comment
 *
 * Partially based on base64 decode on kehlist
 * (first occured on kehlist 1.1.2   9 Jun 1999
 *  Author: Kari E. Hurtta <Kari.Hurtta@Fmi.FI>)
 *
 * Well, basically same decoding routine is on
 * cs_add_streambyte_to_s_utf7() on lib/cs_utf.c
 *
 * And also found from kehpager (UTF7_convert()
 * on charset.c,  kehpager 1.2   1994
 * Author: Kari Hurtta <Kari.hurtta@helsinki.fi> )
 */

struct radix64 {

    int res;                 /* Result of decoding    */
    int bit;                 /* Number of bits on res */
    
};

static void radix64_init P_((struct radix64 *p));
static void radix64_init(p)
     struct radix64 *p;
{
    p->bit = 0;
    p->res = 0;
}

/* Returns EOF if value is not available */

static int radix64_step P_((struct radix64 *p, int code));
static int radix64_step(p,code)
     struct radix64 *p;
     int code;
{
    int ch = EOF;

    if (code < 0 || code >= 1<<6 ) {

	DPRINT(Debug,1,(&Debug,
			 "radix64_step(%p,code=%d)\n",
			 p,code));

	state_panic(__FILE__,__LINE__,"radix64_step",
		    "Code not in range (0-63)!");
	
    }


    /* store code value to res */
    p->res <<= 6;
    p->res |= code;
    p->bit += 6;

    if (p->bit >= 8) {
	/* extract characer value from res */

	ch       = p->res >> (p->bit-8);
	p->res  -= ch     << (p->bit-8);
	p->bit -= 8; 
    }

    DPRINT(Debug,46,(&Debug,
		     "radix64_step(%p,code=%d)=%d%s\n",
		     p,code,ch, ch == EOF ? " (EOF)" : ""));

    return ch;
}

/* -------------------------------------------------------------------------- */
/* base64_decode */


#define IN_STATE_base64_magic	0xEA07

struct base64_decode {
    unsigned short magic;     /* IN_STATE_base64_magic */
    
    struct radix64 r;

};


S_(decode_initial_scan_f base64_initial_scan)
static int base64_initial_scan P_((struct in_state *s));
static int base64_initial_scan(s)
     struct in_state *s;
{
    /* Nothing to do */

    return 1;
}



/* Return EOF when length is done or all is decoded */
S_(decode_getc_f base64_getc)
static int base64_getc         P_((struct in_state *s)); 
static int base64_getc(s)
     struct in_state *s;
{

    /* \r\n and \n    both are translated either \r\n or \n
       if s->u.decode->is_text
    */ 

    int ch = EOF;
    int res_ch = EOF;
    struct base64_decode * ptr;

    
    if (IN_STATE_base64_magic != s->u.decode->decode_data.base64->magic)
	state_panic(__FILE__,__LINE__,"base64_getc","Bad magic number");
    ptr = s->u.decode->decode_data.base64;

    
    if (store_c_enter_getc(s,&res_ch)) {

	DPRINT(Debug,40,(&Debug, 
			 "base64_getc=%d (cached)\n",res_ch));


	return res_ch;
    }

    while (EOF == res_ch) {
	int code;

	ch = limit_getc(s->u.decode);
	if (EOF == ch) {
	    DPRINT(Debug,10,(&Debug, "base64_getc: got EOF\n"));

	    goto handle_EOF;
	}
	
	if (' ' == ch || '\t' == ch || '\r' == ch || '\n' == ch)
	    continue;

	if ('=' == ch) {
	    DPRINT(Debug,10,(&Debug, 
			     "base64_getc: '=' seen  ('end decoded') input_count=%d\n",
			     s->u.decode->input_count));

	    s->u.decode->end_decoded = 1;
	    goto handle_EOF;
	}

	code = base64(ch);
	if (-1 == code) {
	    DPRINT(Debug,10,(&Debug, 
			     "base64_getc: char=%d bad  input_count=%d\n",
			     ch,s->u.decode->input_count));
	    s->u.decode->corrupted = 1;                                   
	    continue;  /* Bad characters are supposed to be skipped */
	}

	DPRINT(Debug,46,(&Debug, 
			 "base64_getc: %c => code =%d\n",
			 ch,code));

	res_ch = radix64_step(& (ptr->r),code);
	
	if (EOF != res_ch) {
	    int r UNUSED_VAROK = 
		store_c_middle_getc(s,&res_ch);

	    DPRINT(Debug,45,(&Debug, 
			     "base64_getc: r = %d, res_ch = %d %s\n",r, res_ch, (res_ch == EOF) ? "EOF" : ""));


	} else {

	    DPRINT(Debug,45,(&Debug, 
			     "base64_getc: res_ch = %d %s\n",res_ch, (res_ch == EOF) ? "EOF" : ""));
	}
	
    }

    DPRINT(Debug,40,(&Debug, 
		 "base64_getc=%d\n",res_ch));


    return res_ch;


 handle_EOF:
  
    if (s->u.decode->input_count >= s->u.decode->length ||
	in_state_feof(s->u.decode->source)) {	    
	s->u.decode->end_decoded = 1;
	
	DPRINT(Debug,15,(&Debug, 
			 "base64_getc: 'end decoded'\n"));

    } else {

	while ('=' == ch) {   
	    ch = limit_getc(s->u.decode);
	}
	
	if (EOF != ch) {
	    int ok = (' ' == ch || '\t' == ch || '\r' == ch || '\n' == ch);
	    
	    DPRINT(Debug,15,(&Debug, 
			     "base64_getc: char=%d %s  input_count=%d  [EOF]\n",
			     ch,
			     ok ? "OK" : "???",
			     s->u.decode->input_count));
	    
	    if (!ok) {	    
		if (EOF != state_ungetc(ch,s->u.decode->source)) {
		    s->u.decode->input_count--;
		}
	    }
	}
	
	if (s->u.decode->input_count >= s->u.decode->length ||
	    in_state_feof(s->u.decode->source)) {	    
	    s->u.decode->end_decoded = 1;
	    
	    DPRINT(Debug,15,(&Debug, 
			     "base64_getc: 'end decoded'\n"));
	}
    }

    /* Make sure to flush anything left in the internal buffer. */
    if (store_c_eof_getc(s,&res_ch)) {

	DPRINT(Debug,40,(&Debug, 
			 "base64_getc=%d (at EOF)\n",res_ch));
	
	return res_ch;
    }

    DPRINT(Debug,40,(&Debug, 
		     "base64_getc=EOF\n"));
    return EOF;
}

/* Return 1 on succeed, 0 on error */
S_(decode_finalize_f base64_finalize);
static int base64_finalize     P_((struct in_state *s)); 
static int base64_finalize(s)
     struct in_state *s;
{
    struct base64_decode * ptr UNUSED_VAROK;

    if (IN_STATE_base64_magic != s->u.decode->decode_data.base64->magic)
	state_panic(__FILE__,__LINE__,"base64_finalize","Bad magic number");
    ptr = s->u.decode->decode_data.base64;

    if (s->u.decode->store_ch_len > 0)
	return 0;

    while (1) {

	int ch = limit_getc(s->u.decode);
	
	if (EOF == ch) {
	    DPRINT(Debug,15,(&Debug, 
			     "base64_finalize: EOF  input_count=%d\n",
			     s->u.decode->input_count));
	    break;
	}

	if (' ' == ch || '\t' == ch || '\r' == ch || '\n' == ch) {
	    DPRINT(Debug,15,(&Debug, 
			     "base64_finalize: char=%d OK  input_count=%d\n",
			     ch,s->u.decode->input_count));
	    continue;
	} else {
	    DPRINT(Debug,15,(&Debug, 
			     "base64_finalize: char=%d ??? input_count=%d\n",
			     ch,s->u.decode->input_count));

	    s->u.decode->corrupted = 1;
	}
    }

    if (s->u.decode->corrupted) {
	
	if (! s->u.decode->decode_error)
		s->u.decode->decode_error = 
		    format_string(CATGETS(elm_msg_cat, MeSet, MeDecodeBASE64Corrupt,
					  "BASE64 data was corrupt!"));

	return 0;
    }

    if (s->u.decode->end_decoded)
	return 1;

    return 0;
} 

S_(decode_init_f base64_init)
static void base64_init        P_((struct in_state *s)); 
static void base64_init(s) 
     struct in_state *s;
{
    s->u.decode->decode_data.base64 =
	safe_malloc(sizeof (* s->u.decode->decode_data.base64));

    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)s->u.decode->decode_data.base64,
	  sizeof (* s->u.decode->decode_data.base64));

    
    radix64_init(& (s->u.decode->decode_data.base64->r));

    s->u.decode->decode_data.base64->magic = IN_STATE_base64_magic;    
}

S_(decode_dest_f base64_dest)
static void base64_dest        P_((struct in_state *s)); 
static void base64_dest(s) 
     struct in_state *s;
{

    if (IN_STATE_base64_magic != s->u.decode->decode_data.base64->magic)
	state_panic(__FILE__,__LINE__,"base64_dest","Bad magic number");
    
    s->u.decode->decode_data.base64->magic = 0; /* Invalidate */
    free(s->u.decode->decode_data.base64);
    s->u.decode->decode_data.base64 = NULL;
}

static struct state_decode_type base64_DECODE = {  
    STATE_decode_type_magic,

    base64_initial_scan,
    base64_getc,
    base64_finalize,
    base64_init,
    base64_dest
}; 

/* ------------------------------------------------------------------------- */

struct line_buffer {

    unsigned char     *buffer;
    int      line_len;
    int      ptr;

};

static void line_buffer_init P_((struct line_buffer *line));
static void line_buffer_init(line)
     struct line_buffer *line;
{
    line->buffer = NULL;
    line->line_len = 0;
    line->ptr    = 0;
}

static void line_buffer_dest P_((struct line_buffer *line));
static void line_buffer_dest(line)
     struct line_buffer *line;
{
    if (line->buffer) {
	free(line->buffer);
	line->buffer = NULL;
    }
    line->line_len = 0;
}


#define INITIAL_LINE_LEN 256
#define LINE_LEN_STEP    512

/* Returns 0 if EOF (or error seen) before any characters */
static int line_buffer_get_new_line P_((struct in_state_decode *st,
					struct line_buffer *line,
					int max_len));
static int line_buffer_get_new_line(st,line,max_len)
     struct in_state_decode *st;
     struct line_buffer *line;
     int max_len;
{
    int ch = EOF;
    int alloced = max_len;

    if (alloced > INITIAL_LINE_LEN)
	alloced = INITIAL_LINE_LEN;

    line->buffer = safe_realloc(line->buffer,alloced);

    line->line_len = 0;

    while (line->line_len < max_len) {
	ch = limit_getc(st);

	if (EOF == ch) {
	    
	    DPRINT(Debug,10,(&Debug, "line_buffer_get_new_line: got EOF\n"));
	    goto handle_EOF;
	}

	if (line->line_len >= alloced) {
	    alloced = ( line->line_len / LINE_LEN_STEP + 1) * LINE_LEN_STEP;

	    if (alloced > max_len+1)
		alloced = max_len+1;

	    line->buffer = safe_realloc(line->buffer,alloced);
	}
	
	if (line->line_len >= alloced)
	    state_panic(__FILE__,__LINE__,"line_buffer_get_new_line",
		    "Bad len");

	line->buffer[line->line_len++] = ch;

	if ('\n' == ch)
	    break;
    }
 
 handle_EOF:
   
    if (st->input_count >= st->length) {	
	DPRINT(Debug,15,(&Debug, 
			 "line_buffer_get_new_line: input-count=%d >= length=%d\n",
			 st->input_count,
			 st->length));
    }

    if (line->line_len >= max_len) {
	DPRINT(Debug,15,(&Debug, 
			 "line_buffer_get_new_line: line_len=%d >= max_len=%d\n",
			 line->line_len,max_len));			 

    }

    if (line->line_len > 0) {

	if ('\n' != ch) {
	    DPRINT(Debug,15,(&Debug, 
			     "line_buffer_get_new_line: Incomplete line\n"));
	}

	alloced = line->line_len+1;

	line->buffer = safe_realloc(line->buffer,alloced);
	line->buffer[line->line_len] = '\0';

    } else {
	DPRINT(Debug,15,(&Debug, 
			 "line_buffer_get_new_line: No line\n"));
    }
    
    DPRINT(Debug,45,(&Debug, 
		     "line_buffer_get_new_line=%d",
		     line->line_len));

    if (line->buffer) {
	DPRINT(Debug,45,(&Debug,", buffer="));
	DEBUG_PRINT_BUFFER(Debug,45,line->line_len,line->buffer);
    }

    if (! line->buffer || line->line_len < 1 ||
	line->buffer[line->line_len-1] != '\n') {
	DPRINT(Debug,45,(&Debug,", no newline\n"));
    }

    return line->line_len;
}


/* ------------------------------------------------------------------------- */
/* uudecode */


/* uuencoded lines should be under 80 characters */
# define MAX_UUENCODE_LINE          1002      

#define IN_STATE_uudecode_magic	0xEA08

struct uudecode {
    unsigned short magic;     /* IN_STATE_uudecode_magic */
    
    struct line_buffer  line;

    int      coded; /* count number of bytes decoded from line */
    int      expected;

    struct radix64   r;

};

/* Return 1 on succeed, 0 on error */
S_(decode_initial_scan_f uudecode_initial_scan)
static int uudecode_initial_scan P_((struct in_state *s));
static int uudecode_initial_scan(s)
     struct in_state *s;
{
    struct uudecode *ptr;

    if (IN_STATE_uudecode_magic != s->u.decode->decode_data.uuencoded->magic)
	state_panic(__FILE__,__LINE__,"uudecode_initial_scan",
		    "Bad magic number");
    ptr = s->u.decode->decode_data.uuencoded;



    do {

	if (! line_buffer_get_new_line(s->u.decode,& (ptr->line),MAX_UUENCODE_LINE)) {
	    
	    if (in_state_ferror(s->u.decode->source)) {
		DPRINT(Debug,15,(&Debug, 
				 "uudecode_initial_scan: Error on source\n"));
	    }
	    
	    DPRINT(Debug,10,(&Debug, 
			     "uudecode_initial_scan: Nothing read\n"));
	    
	    goto handle_EOF;
	}

	if (s->u.decode->input_count >= s->u.decode->length)
	    break;

    } while ( (ptr->line.line_len == 1 && 0 == strcmp(us2s(ptr->line.buffer),
						      "\n")) ||
	      (ptr->line.line_len == 2 && 0 == strcmp(us2s(ptr->line.buffer),"\r\n")) );

    
    if (ptr->line.line_len < 6 || strncmp(us2s(ptr->line.buffer),"begin ",
					  6) != 0) {

    handle_EOF:

	DPRINT(Debug,10,(&Debug, 
			 "uudecode_initial_scan: 'begin' was expected\n"));

	s->u.decode->corrupted = 1;
	return 0;

    }

    return 1;
}


/* Assumes ASCII */
#define UUDECODE(x)  (((x) - ' ') & 077)
/*  Both space ' ' and '`'  character need to be treated
 *  as 0 ... therefore & 077 is needed
 */


S_(decode_getc_f uudecode_getc)
static int uudecode_getc         P_((struct in_state *s)); 
static int uudecode_getc(s)
     struct in_state *s;
{
    /* \r\n and \n    both are translated either \r\n or \n
       if s->u.decode->is_text  is set
    */ 

    int res_ch = EOF;
    struct uudecode *ptr;

    if (IN_STATE_uudecode_magic != s->u.decode->decode_data.uuencoded->magic)
	state_panic(__FILE__,__LINE__,"uudecode_getc","Bad magic number");
    ptr = s->u.decode->decode_data.uuencoded;
	

    if (store_c_enter_getc(s,&res_ch)) {

	DPRINT(Debug,40,(&Debug, 
			 "uudecode_getc=%d (cached)\n",res_ch));

	return res_ch;
    }

    while (EOF == res_ch) {

	int code;
	
	/* Get a next line of uuencoded data */
	if (ptr->coded >= ptr->expected) {

	    if (! line_buffer_get_new_line(s->u.decode,& (ptr->line),MAX_UUENCODE_LINE)) {
		
		DPRINT(Debug,10,(&Debug, 
				 "uudecode_getc: Nothing read\n"));
		
		goto handle_EOF;
	    }


	    /* 'Remove' newline */
	    if (ptr->line.line_len > 0 && '\n' == ptr->line.buffer[ptr->line.line_len-1]) {
		ptr->line.line_len--;
		
		if (ptr->line.line_len > 0 && '\r' == ptr->line.buffer[ptr->line.line_len-1])
		    ptr->line.line_len--;
	    } else {
		DPRINT(Debug,10,(&Debug, 
				 "uudecode_getc: newline expected -- too long line?\n"));
		s->u.decode->corrupted = 1;
	    }

	    if (ptr->line.line_len < 1) {
		DPRINT(Debug,10,(&Debug, 
				 "uudecode_getc: empty line!\n"));
		s->u.decode->corrupted = 1;
		
		/* short lines are padded with space and 
		   and space indicates value 0
		*/
		ptr->expected  = 0;
	    } else      
		ptr->expected  =  UUDECODE(ptr->line.buffer[0]);

	    ptr->line.ptr      = 1;  
	    ptr->coded         = 0;  /* New line */

	    /* If last line before 'end' ? */
	    if (!ptr->expected) {
		DPRINT(Debug,15,(&Debug, 
				 "uudecode_getc: expected line len 0 -- EOF\n"));
		goto handle_EOF;
	    }	    
	}

	if (ptr->line.ptr < ptr->line.line_len) {

	    code = UUDECODE(ptr->line.buffer[ptr->line.ptr]);

	    DPRINT(Debug,46,(&Debug, 
			     "uudecode_getc: [%d] %c => code =%d\n",
			     ptr->line.ptr,ptr->line.buffer[ptr->line.ptr],
			     code));


	} else
	    code = 0; /* If line is short, it is badded with space
		       * which means value 0
		       */
	ptr->line.ptr++;
	
	
	res_ch = radix64_step(& (ptr->r),code);
	if (EOF != res_ch) {
	    int r UNUSED_VAROK = 
		store_c_middle_getc(s,&res_ch);

	    ptr->coded++; /* count number of bytes decoded from line */

	    DPRINT(Debug,45,(&Debug, 
			     "uudecode_getc: r = %d, res_ch = %d %s\n",r, 
			     res_ch, (res_ch == EOF) ? "EOF" : ""));
	    
	} else {
	    
	    DPRINT(Debug,45,(&Debug, 
			     "uudecode_getc: res_ch = %d %s\n",res_ch, (res_ch == EOF) ? "EOF" : ""));

	}
    }

    DPRINT(Debug,40,(&Debug, 
		     "uudecode_getc=%d\n",res_ch));

    return res_ch;

 handle_EOF:    
    /* Make sure to flush anything left in the internal buffer. */

    if (store_c_eof_getc(s,&res_ch)) {
	DPRINT(Debug,40,(&Debug, 
			 "uudecode_getc=%d (at EOF)\n",res_ch));

	return res_ch;
    }
    
    DPRINT(Debug,40,(&Debug, 
		     "uudecode_getc=EOF\n"));
    return EOF;
}

/* Return 1 on succeed, 0 on error */
S_(decode_finalize_f uudecode_finalize)
/* Return 1 on succeed, 0 on error */
static int uudecode_finalize     P_((struct in_state *s)); 
static int uudecode_finalize(s) 
     struct in_state *s; 
{
    struct uudecode *ptr;

    if (IN_STATE_uudecode_magic != s->u.decode->decode_data.uuencoded->magic)
	state_panic(__FILE__,__LINE__,"uudecode_finalize","Bad magic number");
    ptr = s->u.decode->decode_data.uuencoded;

    if (s->u.decode->store_ch_len > 0)
	return 0;

    if (ptr->line.ptr < 
	ptr->expected) {
	DPRINT(Debug,10,(&Debug, 
			 "uudecode_finalize: Not on end of line: ptr=%d expected = %d!\n",
			 ptr->line.ptr,
			 ptr->expected));

	s->u.decode->corrupted = 1;	

    } else if (s->u.decode->input_count >= s->u.decode->length) {	
	DPRINT(Debug,15,(&Debug, 
			 "uudecode_finalize: input-count=%d >= length=%d\n",
			 s->u.decode->input_count,
			 s->u.decode->length));
	
	s->u.decode->corrupted = 1;	
	

    } else {

	if (ptr->expected) {
	    
	    DPRINT(Debug,10,(&Debug, 
			     "uudecode_finalize: last line was not empty: expected = %d!\n",
			     ptr->expected));	
	    
	    s->u.decode->corrupted = 1;
	}


	line_buffer_get_new_line(s->u.decode,& (ptr->line), MAX_UUENCODE_LINE);

	if (s->u.decode->input_count > s->u.decode->length) {
	    DPRINT(Debug,15,(&Debug, 
			     "uudecode_finalize: input-count=%d > length=%d\n",
			     s->u.decode->input_count,
			     s->u.decode->length));
	    
	    s->u.decode->corrupted = 1;	 
	}
	
	
	if (ptr->line.line_len < 3 || strncmp(us2s(ptr->line.buffer),
					      "end",3) != 0) {
	    DPRINT(Debug,10,(&Debug, 
			     "uudecode_finalize: 'end' was expected\n"));
	    s->u.decode->corrupted = 1;	
	} else {
	    s->u.decode->end_decoded = 1;
	    
	    DPRINT(Debug,15,(&Debug, 
			     "uudecode_finalize: 'end decoded'\n"));
	    
	}
    }
    
    
    while (1) {
	
	int ch = limit_getc(s->u.decode);
	
	if (EOF == ch) {
	    DPRINT(Debug,15,(&Debug, 
			     "uudecode_finalize: EOF  input_count=%d\n",
			     s->u.decode->input_count));
	    break;
	}
	
	if (' ' == ch || '\t' == ch || '\r' == ch || '\n' == ch) {
	    DPRINT(Debug,15,(&Debug, 
			     "uudecode_finalize: char=%d OK  input_count=%d\n",
			     ch,s->u.decode->input_count));
	    continue;
	} else {
	    DPRINT(Debug,15,(&Debug, 
			     "uudecode_finalize: char=%d ??? input_count=%d\n",
			     ch,s->u.decode->input_count));
	    
	    s->u.decode->corrupted = 1;
	}
    }
    
    if (s->u.decode->corrupted) {
	
	if (! s->u.decode->decode_error)
	    s->u.decode->decode_error = 
		format_string(CATGETS(elm_msg_cat, MeSet, MeUUDECODECorrupt,
				      "UUENCODED data was corrupt!"));

	return 0;
    }

    if (s->u.decode->end_decoded)
	return 1;


    return 0;
}

S_(decode_init_f  uudecode_init)
static void uudecode_init        P_((struct in_state *s)); 
static void uudecode_init(s)
     struct in_state *s;
{
    s->u.decode->decode_data.uuencoded =
	safe_malloc(sizeof (* s->u.decode->decode_data.uuencoded));

    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)s->u.decode->decode_data.uuencoded,
	  sizeof (* s->u.decode->decode_data.uuencoded));

    line_buffer_init(& (s->u.decode->decode_data.uuencoded->line));
    s->u.decode->decode_data.uuencoded->expected = 0;
    s->u.decode->decode_data.uuencoded->coded    = 0;

    radix64_init(& (s->u.decode->decode_data.uuencoded->r));

    s->u.decode->decode_data.uuencoded->magic = IN_STATE_uudecode_magic;
}


S_(decode_dest_f uudecode_dest)
static void uudecode_dest        P_((struct in_state *s));
static void uudecode_dest(s)
     struct in_state *s;
{
    if (IN_STATE_uudecode_magic != s->u.decode->decode_data.uuencoded->magic)
	state_panic(__FILE__,__LINE__,"uudecode_dest","Bad magic number");


    line_buffer_dest(& (s->u.decode->decode_data.uuencoded->line));

    s->u.decode->decode_data.uuencoded->magic = 0;  /* Invalidate */
    free(s->u.decode->decode_data.uuencoded);
    s->u.decode->decode_data.uuencoded = NULL;
}

static struct state_decode_type uuDECODE = {  
    STATE_decode_type_magic,

    uudecode_initial_scan,
    uudecode_getc,
    uudecode_finalize,

    uudecode_init,
    uudecode_dest

}; 

/* ------------------------------------------------------------------------- */
/* quoted_printable_decode */


#define IN_STATE_quoted_printable_magic	0xEA09

struct quoted_printable_decode {
    unsigned short magic;  /* IN_STATE_quoted_printable_magic  */

    struct line_buffer line;

};

/* Return 1 on succeed, 0 on error */
S_(decode_initial_scan_f quoted_printable_initial_scan)
static int quoted_printable_initial_scan P_((struct in_state *s));
static int quoted_printable_initial_scan(s)
     struct in_state *s;
{
    /* Nothing to do */

    return 1;   
}

/* quoted-printed lines should be under 80 characters */
#define MAX_QUOTED_PRINTABLE_LINE   1002 

/* Return EOF when length is done or all is decoded */
S_(decode_getc_f quoted_printable_getc)
static int quoted_printable_getc         P_((struct in_state *s)); 
static int quoted_printable_getc(s)
     struct in_state *s; 
{
    /* \r\n and \n    both are translated either \r\n or \n
       if s->u.decode->is_text
    */ 

    int res_ch = EOF;
    struct quoted_printable_decode *ptr;


    if (IN_STATE_quoted_printable_magic != 
	s->u.decode->decode_data.quoted_printable->magic)
	state_panic(__FILE__,__LINE__,"quoted_printable_getc","Bad magic number");
    ptr = s->u.decode->decode_data.quoted_printable;


    if (store_c_enter_getc(s,&res_ch)) {
	DPRINT(Debug,40,(&Debug, 
			 "quoted_printable_getc=%d (cached)\n",res_ch));

	return res_ch;
    }

    while (EOF == res_ch) {
	int r UNUSED_VAROK;

	if (ptr->line.ptr >= ptr->line.line_len) {

	    if (! line_buffer_get_new_line(s->u.decode,& (ptr->line),
					   MAX_QUOTED_PRINTABLE_LINE)) {
		
		DPRINT(Debug,10,(&Debug, 
				 "quoted_printable_getc: Nothing read\n"));
		
		goto handle_EOF;
	    }
	    
	    ptr->line.ptr      = 0;
	        
	}
   	
	if ('=' == ptr->line.buffer[ptr->line.ptr]) { 
	    int x;

	    ptr->line.ptr++;

	    /* Ignore spaces in end of line   -- see MIME */
	    for (x = ptr->line.ptr; x < ptr->line.line_len; x++)
		if ('\r' != ptr->line.buffer[x] &&
		    ' ' != ptr->line.buffer[x] &&
		    '\t' != ptr->line.buffer[x])
		    break;

	    if (x >= ptr->line.line_len) {

		if (s->u.decode->input_count >= s->u.decode->length) {
		    DPRINT(Debug,10,(&Debug, 
				     "quoted_printable_getc: newline expected -- input-count=%d >= length=%d\n",
				     s->u.decode->input_count,
				     s->u.decode->length));

		    ptr->line.ptr = x+1;
		    goto handle_EOF;  /* Skip soft line break */
		} else if ( in_state_feof(s->u.decode->source) ) {
		    DPRINT(Debug,10,(&Debug, 
				     "quoted_printable_getc: newline expected -- EOF on source\n"));

		    ptr->line.ptr = x+1;
		    goto handle_EOF;  /* Skip soft line break */
		} else {

		    if (in_state_ferror(s->u.decode->source) ) {
			DPRINT(Debug,10,(&Debug, 
					 "quoted_printable_getc: newline expected -- Error on source\n"));
			
		    } else {
			DPRINT(Debug,10,(&Debug, 
					 "quoted_printable_getc: newline expected -- too long line?\n"));
		    }
		    s->u.decode->corrupted = 1;
		    
		    res_ch = '=';
		}

	    } else if ('\n' == ptr->line.buffer[x]) {
		ptr->line.ptr = x+1;
		continue;  /* Skip soft line break */
	
	    } else if (ptr->line.ptr < ptr->line.line_len-1) {
		int c1 = hex(ptr->line.buffer[ptr->line.ptr]);
		int c2 = hex(ptr->line.buffer[ptr->line.ptr+1]);

		if (c1 != -1 && c2 != -1) {  /* Valid quote printable encoded character */
		    res_ch =  (c1 << 4) | c2;

		    DPRINT(Debug,46,(&Debug, 				     
				     "quoted_printable_getc: [%d] %c%c => res_ch = %d\n",
				     ptr->line.ptr,
				     ptr->line.buffer[ptr->line.ptr],
				     ptr->line.buffer[ptr->line.ptr+1],
				     res_ch));

		    ptr->line.ptr += 2;



		} else {
		   DPRINT(Debug,10,(&Debug, 
				    "quoted_printable_getc: Bad encoding character =%c%c\n",
				    ptr->line.buffer[ptr->line.ptr],
				    ptr->line.buffer[ptr->line.ptr+1]));
			  
		   s->u.decode->corrupted = 1;
		   res_ch = '=';		
		}
	    } else {
		DPRINT(Debug,10,(&Debug, 
				 "quoted_printable_getc: Too short encoded character -- too long line?\n"));
		s->u.decode->corrupted = 1;
		res_ch = '=';
	    }

	} else {
	    res_ch = ptr->line.buffer[ptr->line.ptr];
	    ptr->line.ptr++;
	}
	    
	r = store_c_middle_getc(s,&res_ch);

	DPRINT(Debug,45,(&Debug, 
			 "quoted_printable_getc: r = %d, res_ch = %d %s\n",r, res_ch, 
			 (res_ch == EOF) ? "EOF" : ""));
    }

    DPRINT(Debug,40,(&Debug, 
		     "quoted_printable_getc=%d\n",res_ch));
    return res_ch;

 handle_EOF:
    if (s->u.decode->input_count >= s->u.decode->length ||
	in_state_feof(s->u.decode->source)) {	    
	s->u.decode->end_decoded = 1;
	
	DPRINT(Debug,15,(&Debug, 
			 "quoted_printable_getc: 'end decoded'\n"));
    }

    /* Make sure to flush anything left in the internal buffer. */
    if (store_c_eof_getc(s,&res_ch)) {
	DPRINT(Debug,40,(&Debug, 
			 "quoted_printable_getc=%d (at EOF)\n",res_ch));

	return res_ch;
    }   

    DPRINT(Debug,40,(&Debug, 
		     "quoted_printable_getc=EOF\n"));
    return EOF;
}

/* Return 1 on succeed, 0 on error */
S_(decode_finalize_f quoted_printable_finalize)
static int quoted_printable_finalize     P_((struct in_state *s)); 
static int quoted_printable_finalize(s)
     struct in_state *s;
{
    struct quoted_printable_decode *ptr;

    if (IN_STATE_quoted_printable_magic != 
	s->u.decode->decode_data.quoted_printable->magic)
	state_panic(__FILE__,__LINE__,"quoted_printable_finalize","Bad magic number");
    ptr = s->u.decode->decode_data.quoted_printable;

    if (s->u.decode->store_ch_len > 0)
	return 0;
    
    if (ptr->line.ptr < ptr->line.line_len 
	||
	(s->u.decode->input_count < s->u.decode->length && 
	 !in_state_feof(s->u.decode->source))
	) {
	
	s->u.decode->corrupted = 1;
    }
	

    if (s->u.decode->corrupted) {
	if (! s->u.decode->decode_error)
		s->u.decode->decode_error =
		    format_string(CATGETS(elm_msg_cat, MeSet, MeDecodeQPCorrupt,
					  "Seems that QUOTED-PRINTABLE data was corrupted."));

	return 0;
    }

    if (s->u.decode->end_decoded)
	return 1;

    return 0;
} 

S_(decode_init_f quoted_printable_init)
static void quoted_printable_init        P_((struct in_state *s));
static void quoted_printable_init(s) 
     struct in_state *s;
{
    s->u.decode->decode_data.quoted_printable =
	safe_malloc(sizeof (* s->u.decode->decode_data.quoted_printable));

    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)s->u.decode->decode_data.quoted_printable,
	  sizeof (* s->u.decode->decode_data.quoted_printable));

    line_buffer_init(& (s->u.decode->decode_data.quoted_printable->line));

    s->u.decode->decode_data.quoted_printable->magic = IN_STATE_quoted_printable_magic;
}


S_(decode_dest_f quoted_printable_dest)
static void quoted_printable_dest        P_((struct in_state *s));
static void quoted_printable_dest(s)        
     struct in_state *s;
{
    if (IN_STATE_quoted_printable_magic != 
	s->u.decode->decode_data.quoted_printable->magic)
	state_panic(__FILE__,__LINE__,"quoted_printable_dest","Bad magic number");
    line_buffer_dest(& (s->u.decode->decode_data.quoted_printable->line));
    
    s->u.decode->decode_data.quoted_printable->magic = 0;   /* Invalidate */
    free(s->u.decode->decode_data.quoted_printable);
    s->u.decode->decode_data.quoted_printable = NULL;

}

static struct state_decode_type quoted_printable_DECODE = {  
    STATE_decode_type_magic,

    quoted_printable_initial_scan,
    quoted_printable_getc,
    quoted_printable_finalize,
    quoted_printable_init,
    quoted_printable_dest
}; 





/* ------------------------------------------------------------------------- */

static int decode_types_initialized = 0;


static void initialize_decode_types P_((void));
static void initialize_decode_types()
{
    
    /* Fill array */
    
    decode_types[ENCODING_NONE]   =  &Xbit_DECODE;
    decode_types[ENCODING_7BIT]   =  &Xbit_DECODE;
    decode_types[ENCODING_8BIT]   =  &Xbit_DECODE;
    decode_types[ENCODING_BINARY] =  &Xbit_DECODE;

    decode_types[ENCODING_QUOTED] =  &quoted_printable_DECODE;
    decode_types[ENCODING_BASE64] =  &base64_DECODE;

    decode_types[ENCODING_UUENCODED] = &uuDECODE;

    decode_types_initialized =  1;
}


/* ----------------------------------------------------------------------- */



S_(init_si_type_f init_si_decode)
static void init_si_decode P_((struct in_state *s)); 
static void init_si_decode(s) 
     struct in_state *s;
{
    int x;
    s->u.decode = safe_malloc(sizeof (* s->u.decode));

    /* defined in hdrs/defs.h */
    bzero((void *)s->u.decode, sizeof (* s->u.decode));

    s->u.decode->magic = IN_STATE_decode_magic;

    s->u.decode->source  = NULL;
    s->u.decode->length  = 0;

    s->u.decode->is_text = 0;
    s->u.decode->EOLN_is_CRLF = 0;
    s->u.decode->from_buffer  = 0;

    s->u.decode->decode_type  = NULL;
    s->u.decode->decode_error = NULL;

    s->u.decode->input_count           = 0;
    s->u.decode->decoded_count         = 0;
    s->u.decode->lone_cr_convert_count = 0;
    
    s->u.decode->timer_unlink = NULL;
    
    s->u.decode->buffer_name = NULL;
    s->u.decode->buffer_F    = NULL;

    s->u.decode->buffer_pos  = 0;
    
    s->u.decode->unget_buffer     = NULL;
    s->u.decode->unget_buffer_len = 0;

    s->u.decode->store_ch_len = 0;

    for (x = 0; x < sizeof (s->u.decode->store_ch); x++)
	s->u.decode->store_ch[x] = 0;

    s->u.decode->decode_data.dummy = NULL;

    s->u.decode->initial_scan_done = 0;
    s->u.decode->end_decoded       = 0;
    s->u.decode->finalize_done     = 0;

    s->u.decode->corrupted         = 0;
}


S_(dest_si_type_f dest_si_decode)
static void dest_si_decode P_((struct in_state *s));
static void dest_si_decode(s)
     struct in_state *s;
{
    if (IN_STATE_decode_magic != s->u.decode->magic)
	state_panic(__FILE__,__LINE__,"dest_si_decode",
		    "Bad state type magic number");

    if (s->u.decode->decode_type) {
	
	if (STATE_decode_type_magic != s->u.decode->decode_type->magic)
	    state_panic(__FILE__,__LINE__,"dest_si_decode",
			"Bad decode type magic number");
	    
	
	s->u.decode->decode_type->decode_dest(s);

    }

    if (s->u.decode->decode_data.dummy) {
	DPRINT(DebugIO,1,(&DebugIO,"dest_si_decode: Error: decode_data not freed?\n"));
    }
    
    if (s->u.decode->decode_error)
	free_string(& (s->u.decode->decode_error));
    
    if (s->u.decode->buffer_F) {
	fclose(s->u.decode->buffer_F);
	s->u.decode->buffer_F = NULL;
    }

    if (s->u.decode->buffer_name) {

	if (s->u.decode->unlink_name) {

	    if (s->u.decode->timer_unlink &&
		delay_unlink_add_name(s->u.decode->timer_unlink,
				      s->u.decode->buffer_name)) {
		
		DPRINT(DebugIO,10,(&DebugIO,"dest_si_decode: %s: Handled by timer unlink, rec %p\n",
				   s->u.decode->buffer_name,
				   s->u.decode->timer_unlink));
				
	    } else if (0 == unlink(s->u.decode->buffer_name)) {
		DPRINT(DebugIO,10,(&DebugIO,"dest_si_decode: %s: Unlinked\n",
				  s->u.decode->buffer_name));
	    } else {
		int err UNUSED_VAROK = errno;
		
		DPRINT(DebugIO,1,(&DebugIO,
				  "dest_si_decode: %s: %s (errno=%d)\n",
				  s->u.decode->buffer_name,strerror(err),err));		
	    }	   
	}


	free(s->u.decode->buffer_name);
	s->u.decode->buffer_name = NULL;
    }
    
    if (s->u.decode->timer_unlink) {
	if (s->u.decode->unlink_name) {
	    delay_unlink_mark_done(s->u.decode->timer_unlink);
	}
	free_delay_unlink(& (s->u.decode->timer_unlink));
    }
    
    if (s->u.decode->unget_buffer) {
	free(s->u.decode->unget_buffer);
	s->u.decode->unget_buffer = NULL;
    }
    s->u.decode->unget_buffer_len = 0;

    /* That just decrements refcount */
    if (s->u.decode->source)
	free_in_state(& (s->u.decode->source));

    s->u.decode->magic = 0;   /* Invalidate */
    free(s->u.decode);
    s->u.decode = NULL;
}

static int check_initial_scan_done P_((struct in_state *state));
static int check_initial_scan_done(state)
     struct in_state *state;
{

    if (STATE_decode_type_magic !=	state->u.decode->decode_type->magic)
	state_panic(__FILE__,__LINE__,"check_initial_scan_done",
		    "Bad decode type magic number");
    
    if (! state->u.decode->initial_scan_done) {
	if (state->u.decode->decode_type->decode_initial_scan(state)) {
	    state->u.decode->initial_scan_done = 1;
	    
	} else {
	    
	    if (! state->u.decode->decode_error)
		state->u.decode->decode_error = 
		    format_string(CATGETS(elm_msg_cat, MeSet, MeDecodeFailedInitialize,
					  "Failed to initialize decoding"));
	    
	    return 0;
	    
	}
    }

    return 1;
}


static void check_finalize_done P_((struct in_state *state));
static void check_finalize_done(state)
     struct in_state *state;
{
    if (STATE_decode_type_magic !=	state->u.decode->decode_type->magic)
	state_panic(__FILE__,__LINE__,"",
		    "Bad decode type magic number");

    if (! state->u.decode->finalize_done) {
	if (state->u.decode->decode_type->decode_finalize(state)) {
	    state->u.decode->finalize_done = 1;
	} else {	   
	    if (! state->u.decode->decode_error)
		state->u.decode->decode_error = 
		    format_string(CATGETS(elm_msg_cat, MeSet, MeDecodeFailedFinalize,
					  "Failed to finalize decoding"));
	}
    }
}

static int state_decode_handle_one P_((struct in_state *s, int c));
static int state_decode_handle_one(s,c) 
     struct in_state *s; 
     int c;
{

    if (EOF != c) {
	s->u.decode->decoded_count ++;

	if (s->u.decode->buffer_F) {
	    
	    if (EOF != putc(c,s->u.decode->buffer_F)) {
		s->u.decode->buffer_pos++;
		
	    } else {
		int err = errno;
		
		if (s->u.decode->decode_error)
		    free_string(& s->u.decode->decode_error);
		
		s->u.decode->decode_error =
		    format_string(CATGETS(elm_msg_cat, MeSet, MeDecodeBufferError,
					  "Decode buffer error: %s"),
				  strerror(err));

		return 0;
	    }
	} else
	    return 0;
    }

    return 1;
}

static int state_decode_buffer_it P_((struct in_state *s));
static int state_decode_buffer_it(s)
     struct in_state *s;
{
    if (! check_initial_scan_done(s)) {

	return 0;
    }

    do {
	int c = s->u.decode->decode_type->decode_getc(s);

	if (EOF == c)
	    break;

	if (! state_decode_handle_one(s,c)) {
	    return 0;
	}

    } while (! s->u.decode->end_decoded);

    DPRINT(DebugIO,11,(&DebugIO,
		       "state_decode_buffer_it: %d characters buffered\n",
		       s->u.decode->buffer_pos));

    check_finalize_done(s);

    return 1;
}

/* Returns 1 if decoding is possible */
int set_in_state_decode_helper(ptr,src,state,flags,name)
     mime_t *ptr;
     struct in_state *src;
     struct in_state *state; 
     int flags;
     const char **name       /* Only if STATE_DECODE_buffer */;
{
    int ret;
    int is_text = decoder_is_textual(ptr);

    DPRINT(Debug,15,(&Debug, "set_in_state_decode_helper: is_text=%d\n",
		     is_text));

    if (is_text > 0) {
	flags |=  STATE_DECODE_is_text;
	
	DPRINT(Debug,15,(&Debug, "     Adding STATE_DECODE_is_text to flags\n"));
    }

    ret = set_in_state_decode(src,state,ptr->length,flags,ptr->encoding,name);

    DPRINT(Debug,15,(&Debug, "set_in_state_decode_helper=%d\n",ret));

    return ret;
}

int is_valid_encoding(encoding)
     enum encoding encoding;
{
    if (!decode_types_initialized)
	initialize_decode_types();

    if (encoding < 0 || encoding >= ENCODING_count ||
	! decode_types[ encoding ]) {

	DPRINT(Debug,15,(&Debug,
			 "is_valid_encoding(%d)=0: unsupported encoding (%s)\n",
			 encoding,ENCODING(encoding)));

	return 0;
    }

    DPRINT(Debug,15,(&Debug,
		     "is_valid_encoding(%d)=1: encoding (%s)\n",
		     encoding,ENCODING(encoding)));
    return 1;
}



/* Returns 1 if decoding is possible */
int set_in_state_decode(src,state,length,state_decode_flags,encoding,name)
     struct in_state *src;
     struct in_state *state;
     int length;
     int state_decode_flags;
     enum encoding encoding;   
     const char **name;       /* Only if STATE_DECODE_buffer */
{
    int keep_file = 0;

    if (STATE_in_magic != state->magic)
	state_panic(__FILE__,__LINE__,"set_in_state_decode","Bad magic number");

    if (STATE_in_decode != state->state_type)
	state_panic(__FILE__,__LINE__,"set_in_state_decode","Bad state type");

    if (IN_STATE_decode_magic != state->u.decode->magic)
	state_panic(__FILE__,__LINE__,"set_in_state_decode",
		    "Bad state type magic number");

    if (STATE_in_magic != src->magic)
	state_panic(__FILE__,__LINE__,"set_in_state_decode","Bad source magic number");
 
    
    if (state->u.decode->source || state->u.decode->decode_type ||
	state->u.decode->buffer_F   || state->u.decode->buffer_name ||
	state->u.decode->timer_unlink)
	state_panic(__FILE__,__LINE__,"set_in_state_decode","Already called");

    if (!decode_types_initialized)
	initialize_decode_types();

    if (encoding < 0 || encoding >= ENCODING_count ||
	! decode_types[ encoding ]) {

	DPRINT(DebugIO,10,(&DebugIO,
			   "set_in_state_decode(%p,%p, ... encoding=%d)=0: unsupported encoding (%s)\n",
			   src,state,encoding,ENCODING(encoding)));
	
	
	return 0;
    }

    if (STATE_decode_type_magic != decode_types[ encoding ]->magic) 
	state_panic(__FILE__,__LINE__,"set_in_state_decode","Bad decode type magic");
    

    state->u.decode->source = src;

    /* Take reference */
    if (inc_in_state_refcount(state->u.decode->source) > 2 &&
	0 == (STATE_DECODE_buffer & state_decode_flags)) {

	DPRINT(DebugIO,11,(&DebugIO,
			   "set_in_state_decode: Several STATE_in_decode from same source does not work correctly without buffering\n"));
	
    }


    state->u.decode->length = length;
    state->u.decode->decode_type = decode_types[ encoding ];

    DPRINT(Debug,11,(&Debug,"set_in_state_decode: length=%d\n",state->u.decode->length));
    if (in_state_seekable(src)) {
	DPRINT(Debug,11,(&Debug,
			 "              (file): ftell=%ld\n",
			 in_state_ftell(src)));

    }
    
    DPRINT(Debug,11,(&Debug,"    state_decode_flags=%d%s%s%s%s\n",
		     state_decode_flags,
		     STATE_DECODE_is_text & state_decode_flags ? " STATE_DECODE_is_text" : "",
		     STATE_DECODE_EOLN_is_CRLF & state_decode_flags ? " STATE_DECODE_EOLN_is_CRLF" : 
		     "",
		     STATE_DECODE_buffer & state_decode_flags ? " STATE_DECODE_buffer" : "",
		     STATE_DECODE_keep_file & state_decode_flags ? "STATE_DECODE_keep_file" : ""
		     ));

    state->u.decode->is_text      = 0 != (STATE_DECODE_is_text & state_decode_flags);
    state->u.decode->EOLN_is_CRLF = 0 != (STATE_DECODE_EOLN_is_CRLF & state_decode_flags);
    state->u.decode->from_buffer  = 0;

    keep_file = 0 != (STATE_DECODE_keep_file & state_decode_flags);

    if (name)
	*name = NULL;
    else if (keep_file)
	state_panic(__FILE__,__LINE__,"set_in_state_decode",
		    "STATE_DECODE_keep_file requires name argument");

    if (0 != (STATE_DECODE_buffer & state_decode_flags)) {
	char *fname = NULL;
	const char * tmp = give_dt_estr_as_str(&temp_dir_e,"tmpdir",NULL,NULL);
	FILE * tmpfp = NULL;

	static int counter = 0;
	int err = 0;

	if (!tmp) 
	    goto FAIL;

	fname = elm_message(FRM("%selmdecode.%d.%d"), 
			    tmp, getpid (), counter++);	

	tmpfp = safeopen_rdwr(fname,&err);
	if (!tmpfp) {
	    DPRINT(DebugIO,1,(&DebugIO,
			      "set_in_state_decode: %s: %s (errno=%d)\n",
			      fname,strerror(err),err));

	    lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeFailedCreate,
			      "Failed to create file for decoding."));

	    if (! state->u.decode->decode_error)
		state->u.decode->decode_error = 
		    format_string(CATGETS(elm_msg_cat, MeSet, MeDecodeFailedCreate,
					  "Failed to create file for decoding."));

	    free(fname);
	    fname = NULL;
	    goto FAIL;
	    
	}

	DPRINT(DebugIO,10,(&DebugIO,
			   "set_in_state_decode: %s: created temporary buffer\n",
			   fname));

	state->u.decode->buffer_name = fname;
	state->u.decode->buffer_F    = tmpfp;

	if (name) {
	    *name                    = fname;
	    state->u.decode->unlink_name = !keep_file;
	    
	} else {
	    
	    if (0 == unlink(state->u.decode->buffer_name)) {
		DPRINT(DebugIO,10,(&DebugIO,"set_in_state_decode: %s: Unlinked\n",
				   state->u.decode->buffer_name));
	    } else {
		int err UNUSED_VAROK = errno;
		
		DPRINT(DebugIO,1,(&DebugIO,
				  "dest_si_decode: %s: %s (errno=%d)\n",
				  state->u.decode->buffer_name,strerror(err),err));		
	    }	   	   
	}

    } else if (keep_file)
	state_panic(__FILE__,__LINE__,"set_in_state_decode",
		    "STATE_DECODE_keep_file requires also STATE_DECODE_buffer flag");


    state->u.decode->decode_type->decode_init(state);
    
    if (name && *name) {

	/* If filename is returned, we need buffer result now */
	if (state_decode_buffer_it(state)) {
	    
	    rewind(state->u.decode->buffer_F);
	    state->u.decode->from_buffer = 1;
	    state->u.decode->buffer_pos = 0;
	    
	} else
	    goto FAIL;
    }

    DPRINT(DebugIO,10,(&DebugIO,
		       "set_in_state_decode(%p,%p, ...)=1\n",
		       src,state));

    return 1;


 FAIL:

    DPRINT(DebugIO,10,(&DebugIO,
			   "set_in_state_decode(%p,%p, ...)=0\n",
			   src,state));

    return 0;
}


/* Increments refcount, possibly also creates struct delay_unlink */ 

struct delay_unlink * delay_unlink_from_state_decode(state,timer_secs)
     struct in_state *state;
     int timer_secs;
{    
    if (STATE_in_magic != state->magic)
	state_panic(__FILE__,__LINE__,"delay_unlink_from_state_decode","Bad magic number");

    if (STATE_in_decode != state->state_type)
	state_panic(__FILE__,__LINE__,"delay_unlink_from_state_decode","Bad state type");

    if (! state->u.decode->buffer_F)
	state_panic(__FILE__,__LINE__,"delay_unlink_from_state_decode","No buffer");

    if (! state->u.decode->buffer_name)
	state_panic(__FILE__,__LINE__,"delay_unlink_from_state_decode","No buffer name");
   
    if (! state->u.decode->timer_unlink) {
	int fd = fileno(state->u.decode->buffer_F);

	DPRINT(DebugIO,10,(&DebugIO,"delay_unlink_from_state_decode: fd: %d timer_secs: %d name: %s\n",
			   fd,timer_secs,
			   state->u.decode->buffer_name));
	
	state->u.decode->timer_unlink = alloc_delay_unlink(state->u.decode->buffer_name,fd,
							   timer_secs);
    }

    if (state->u.decode->timer_unlink) {
	inc_delay_unlink_recount(state->u.decode->timer_unlink);

	DPRINT(DebugIO,10,(&DebugIO,"delay_unlink_from_state_decode=%p name: %s\n",
			   state->u.decode->timer_unlink,state->u.decode->buffer_name));

	
	return state->u.decode->timer_unlink;
    }

    DPRINT(DebugIO,10,(&DebugIO,"delay_unlink_from_state_decode=NULL name: %s\n",
		       state->u.decode->buffer_name));
    
    return NULL;
}

/* Opens new handle FILE *, original handle is dup()ed, caller must fclose() */
FILE * file_handle_from_state_decode(state) 
     struct in_state *state;
{
    int fd;
    int fd2;
    FILE * F;

    if (STATE_in_magic != state->magic)
	state_panic(__FILE__,__LINE__,"file_handle_from_state_decode","Bad magic number");

    if (STATE_in_decode != state->state_type)
	state_panic(__FILE__,__LINE__,"file_handle_from_state_decode","Bad state type");

    if (! state->u.decode->buffer_F)
	state_panic(__FILE__,__LINE__,"file_handle_from_state_decode","No buffer");

    fd = fileno(state->u.decode->buffer_F);

    if (-1 == fd) {
	int err UNUSED_VAROK = errno;

	DPRINT(DebugIO,1,(&DebugIO,
			  "file_handle_from_state_decode=NULL: fileno: %s (errno=%d)\n",
			  strerror(err),err));	
	
	return NULL;			  
    }

    fd2 = dup(fd);
    if (-1 == fd2) {
	int err UNUSED_VAROK = errno;
	DPRINT(DebugIO,1,(&DebugIO,
			  "file_handle_from_state_decode=NULL: dup: %s (errno=%d)\n",
			  strerror(err),err));	
	
	return NULL;			  
    }

    F = fdopen(fd2,"r+");

    if (!F) {
	int err UNUSED_VAROK = errno;

	DPRINT(DebugIO,1,(&DebugIO,
			  "file_handle_from_state_decode=NULL: fdopen: %s (errno=%d)\n",
			  strerror(err),err));	
	
	close(fd2);

	return NULL;
    }

    return F;
}


/* --------------------------------------------------------------------------------------- */

S_(seekable_si_type_f seekable_si_decode)
static int  seekable_si_decode P_((struct in_state *s));
static int  seekable_si_decode(s)
     struct in_state *s;
{

    if (IN_STATE_decode_magic != s->u.decode->magic)
	state_panic(__FILE__,__LINE__,"seekable_si_decode",
		    "Bad state type magic number");

    return s->u.decode->buffer_F != NULL;
}

S_(getc_si_type_f getc_si_decode)
static int  getc_si_decode P_((struct in_state *s));
static int  getc_si_decode(s)
     struct in_state *s;
{
    int c = EOF;

    if (IN_STATE_decode_magic != s->u.decode->magic)
	state_panic(__FILE__,__LINE__,"getc_si_decode",
		    "Bad state type magic number");

    if (s->u.decode->decode_error) {
	c = EOF;
	goto FAIL;
    }

    /* ungetc does not change position or counters .... */

    if (s->u.decode->unget_buffer_len > 0) {
	s->u.decode->unget_buffer_len--;

	c = s->u.decode->unget_buffer[s->u.decode->unget_buffer_len];

	if (! s->u.decode->unget_buffer_len) {
	    free(s->u.decode->unget_buffer);
	    s->u.decode->unget_buffer = NULL;
	}

	goto DONE;
    }

    if (s->u.decode->from_buffer) {

	if (! s->u.decode->buffer_F)
	    state_panic(__FILE__,__LINE__,"getc_si_decode",
			"No buffer FILE handle");

	c = fgetc(s->u.decode->buffer_F);
	
	if (EOF == c) {
	    int err = errno;

	    if (ferror(s->u.decode->buffer_F) && err) {
		if (s->u.decode->decode_error)
		    free_string(& s->u.decode->decode_error);
       
		s->u.decode->decode_error =
		    format_string(CATGETS(elm_msg_cat, MeSet, MeDecodeBufferError,
					  "Decode buffer error: %s"),
				  strerror(err));
	    }
	    

	    if (feof(s->u.decode->buffer_F)) {
		if (s->u.decode->decode_error)
		    free_string(& s->u.decode->decode_error);
       
		s->u.decode->decode_error =
		    format_string(CATGETS(elm_msg_cat, MeSet, MeDecodeBufferEOF,
					  "Unexpected EOF on buffer"));

	    }

	} else {
	    s->u.decode->buffer_pos++;

	    /* Next byte is not from buffer */
	    if (s->u.decode->buffer_pos >= s->u.decode->decoded_count) {
		s->u.decode->from_buffer = 0;
	    
		/* Need change buffer from reading to writing ??? */
		if (fseek(s->u.decode->buffer_F,s->u.decode->buffer_pos,SEEK_SET) <0) {
		    int err = errno;

		    s->u.decode->decode_error =
			format_string(CATGETS(elm_msg_cat, MeSet, MeDecodeBufferModeSeek,
					      "Failed to change buffer mode: %s"),
				      strerror(err));

		} 

	    }
	    

	}
	
    } else { /* NOT from_buffer */

	if (STATE_decode_type_magic !=	s->u.decode->decode_type->magic)
	    state_panic(__FILE__,__LINE__,"getc_si_decode",
			"Bad decode type magic number");

	if (! check_initial_scan_done(s)) {
	    c = EOF;
	    goto FAIL;
	}

	if (s->u.decode->end_decoded) 
	    c = EOF;    /* Not failure */
	else
	    c = s->u.decode->decode_type->decode_getc(s);
	
	
	if (s->u.decode->end_decoded && c == EOF) 
	    check_finalize_done(s);
	

	state_decode_handle_one(s,c);
    }


 DONE:
 FAIL:
    return c;
}

S_(ungetc_si_type_f ungetc_si_decode)
static int  ungetc_si_decode P_((int c,struct in_state *s));
static int  ungetc_si_decode(c,s)
     int c;
     struct in_state *s;
{
    if (IN_STATE_decode_magic != s->u.decode->magic)
	state_panic(__FILE__,__LINE__,"getc_si_decode",
		    "Bad state type magic number");

    /* ungetc does not change position or counters .... */

    s->u.decode->unget_buffer =
	safe_realloc(s->u.decode->unget_buffer,
		     s->u.decode->unget_buffer_len+1);
    
    s->u.decode->unget_buffer[s->u.decode->unget_buffer_len] = c;

    s->u.decode->unget_buffer_len++;

    return c;
}

S_(gets_si_type_f gets_si_decode)
static char *gets_si_decode P_((char *buffer, int l, struct in_state *s)); 
static char *gets_si_decode(buffer,l,s)
     char *buffer;
     int l;
     struct in_state *s; 
{
    panic("STATE PANIC",__FILE__,__LINE__,"gets_si_decode",
	  "gets_si_decode called",0);

    return NULL;
}

S_(getl_si_type_f getl_si_decode)
static int getl_si_decode P_((char *buffer, int l, struct in_state *s));
static int getl_si_decode(buffer,l,s)
     char *buffer;
     int l;
     struct in_state *s;
{
    panic("STATE PANIC",__FILE__,__LINE__,"getl_si_decode",
	  "getl_si_decode called",0);

    return -1;
}

S_(ftell_si_type_f ftell_si_decode)
static long ftell_si_decode P_((struct in_state *s));
static long ftell_si_decode(s)
     struct in_state *s;
{
    if (IN_STATE_decode_magic != s->u.decode->magic)
	state_panic(__FILE__,__LINE__,"ftell_si_decode",
		    "Bad state type magic number");

    return  s->u.decode->buffer_pos;
}

S_(fseek_si_type_f fseek_si_decode)
static int fseek_si_decode P_((struct in_state *s, long pos));
static int fseek_si_decode(s,pos)
     struct in_state *s;
     long pos;
{
    int ret = 0;

    if (IN_STATE_decode_magic != s->u.decode->magic)
	state_panic(__FILE__,__LINE__,"fseek_si_decode",
		    "Bad state type magic number");


    if (! s->u.decode->buffer_F)
	state_panic(__FILE__,__LINE__,"fseek_si_decode",
		    "Not buffered");

    if (! s->u.decode->from_buffer) {
	
	if (0 != fflush(s->u.decode->buffer_F)) {

	    int err = errno;
	    
	    if (s->u.decode->decode_error)
		free_string(& s->u.decode->decode_error);
	    
	    s->u.decode->decode_error =
		format_string(CATGETS(elm_msg_cat, MeSet, MeDecodeBufferFlushError,
				      "Decode buffer flush error: %s"),
			      strerror(err));	   	    
	}
    }


    /* Forget unget data */

    s->u.decode->unget_buffer_len = 0;

    if (pos < 0 || pos > s->u.decode->decoded_count)
	state_panic(__FILE__,__LINE__,"fseek_si_decode",
		    "Bad position");

    s->u.decode->buffer_pos = pos;

    ret = fseek(s->u.decode->buffer_F,pos,SEEK_SET);
    if (-1 == ret) {
	int err = errno;

	if (s->u.decode->decode_error)
	    free_string(& s->u.decode->decode_error);
       
	s->u.decode->decode_error =
	    format_string(CATGETS(elm_msg_cat, MeSet, MeDecodeBufferSeekError,
				  "Decode buffer seek error: %s"),
			  strerror(err));
    } else {
    
	s->u.decode->from_buffer = 1;
	
	/* Next byte is not from buffer */
	if (s->u.decode->buffer_pos >= s->u.decode->decoded_count) 
	    s->u.decode->from_buffer = 0;
    }

    return ret;
}

/* Caller frees result, NULL if no error */
S_(errmsg_si_type_f errmsg_si_decode)
static struct string * errmsg_si_decode P_((struct in_state *s,int clearerr /* pop message from stack*/));
static struct string * errmsg_si_decode(s,clearerr)
     struct in_state *s;
     int clearerr /* pop message from stack*/;
{
    if (IN_STATE_decode_magic != s->u.decode->magic)
	state_panic(__FILE__,__LINE__,"errmsg_si_decode",
		    "Bad state type magic number");

    if (s->u.decode->decode_error) {
	struct string *S = dup_string(s->u.decode->decode_error);

	if (clearerr)
	    free_string(& s->u.decode->decode_error);

	return S;
    }

    if (s->u.decode->lone_cr_convert_count > 0) {
	enum convert_cr_v convert_CR_to_newline_hack =
	    give_dt_enumerate_as_int(&convert_cr_to_newline_hack);
	struct string *S = NULL;
	switch (convert_CR_to_newline_hack) {
	case convert_cr_off:
	case convert_cr_error:
	    if (s->u.decode->lone_cr_convert_count > 1)
		S = format_string(CATGETS(elm_msg_cat, MeSet, MeDecodeCRsConverted,
					 "Text format: %d lone CRs converted to newline."),
				 s->u.decode->lone_cr_convert_count);
	    else
		S = format_string(CATGETS(elm_msg_cat, MeSet, MeDecodeCRConverted,
					  "Text format: %d lone CR converted to newline."),
				  s->u.decode->lone_cr_convert_count);

	    if (clearerr)
		s->u.decode->lone_cr_convert_count = 0;
	    
	    return S;
	case convert_cr_silent:
	case NUM_convert_cr_hack:
	    break;
	}

	if (clearerr)
	    s->u.decode->lone_cr_convert_count = 0;	
    }
    
    if (s->u.decode->source) {
	const struct string  * errmsg = in_state_error_message(s->u.decode->source,clearerr);

	if (errmsg)
	    return dup_string(errmsg);   
    }

    return NULL; 
}


S_(policy_si_type_f policy_si_decode)
static int policy_si_decode P_((struct in_state *s, 
			      enum state_policy question));
static int policy_si_decode(s,question)
     struct in_state *s;
     enum state_policy question;
{
    switch (question) {
    case state_policy_gets_via_getc: return 1;

    default:
	panic("STATE PANIC",__FILE__,__LINE__,"policy_si_decode",
	      "Bad question",0);
    }

    return 0;
}

S_(ferror_si_type_f ferror_si_decode)
static int ferror_si_decode P_((struct in_state *s));
static int ferror_si_decode(s)
     struct in_state *s;
{

    if (IN_STATE_decode_magic != s->u.decode->magic)
	state_panic(__FILE__,__LINE__,"ferror_si_decode",
		    "Bad state type magic number");

    if (s->u.decode->decode_error)
	return 1;

    if (s->u.decode->source)
	return in_state_ferror(s->u.decode->source);

    return 0;
}


S_(feof_si_type_f feof_si_decode)
static int feof_si_decode P_((struct in_state *s));
static int feof_si_decode(s)
     struct in_state *s;
{

    if (IN_STATE_decode_magic != s->u.decode->magic)
	state_panic(__FILE__,__LINE__,"feof_si_decode",
		    "Bad state type magic number");

    return s->u.decode->end_decoded && !s->u.decode->from_buffer;
}



struct in_state_type STATE_in_decode_type = {
    IN_STATE_type_magic,

    init_si_decode,
    dest_si_decode,
    seekable_si_decode,
    getc_si_decode,
    ungetc_si_decode,
    gets_si_decode,
    getl_si_decode,
    ftell_si_decode,
    fseek_si_decode,
    errmsg_si_decode,
    policy_si_decode,
    ferror_si_decode,
    feof_si_decode,
};


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
