static char rcsid[] = "@(#)$Id: tagfilter_entity.c,v 2.1 2023/12/13 16:55:32 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.1 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_melib.h"
#include "tagfilter_imp.h"
#include "rc_imp.h"
#include "s_me.h"
#include "s_elm.h"

#include "commentfile.h"


#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

DEBUG_VAR(Debug,__FILE__,"mime");

static const unsigned char *cs2us P_((const char *str));
static const unsigned char *cs2us(str) 
     const char *str;
{
    return (const unsigned char *)str;
}

/* --------------------------------------------------------------------------- */

#define NAME_ENTITY_WALK_magic		0xFD0E

static struct name_entity_walk {
    unsigned short               magic;    /*  NAME_ENTITY_WALK_magic */

    int refcount;
    
    struct tagfilter_entities * ents;    /*  tagfilter_entities is static, 
					     no dynamically alloced, so no 
					     refcounted 
					     - backlink
					 */

    uint16 code;                         /* character code for entity
					    parse tree

					    0x0026 AMPERSAND (&) on
					    root of tree
					 */

    /* next node alternatives for tree parsing */
    
    struct name_entity_walk ** alternatives;
    size_t                     alternatives_count;

    /* possible results on this point */

    struct name_entity_match ** results;
    size_t                      results_count;
    
} * malloc_name_entity_walk P_((struct tagfilter_entities * ents,
				uint16 code));

/* --------------------------------------------------------------------------- */

enum ne_match_bits {
    ne_match_bit_SEMICOLON = 0,
    ne_match_bit_STATIC,                  /* static, not malloced */

    ne_match_bit_IGNORE, 
    
    NUM_ne_match_bits
};


#define ne_match_SEMICOLON 	(1 << ne_match_bit_SEMICOLON)
#define ne_match_STATIC		(1 << ne_match_bit_STATIC)
#define ne_match_IGNORE		(1 << ne_match_bit_IGNORE)

#define NAME_ENTITY_MATCH_magic		0xFD0F

static struct name_entity_match {
    unsigned short               magic;       /* NAME_ENTITY_MATCH_magic */

    int                              refcount;
    
    struct tagfilter_entities_conf * conf;   /* Backlink - no refcount --
						NULL if builtin 
					     */
    int                              lineno; /* Corresponding lineno */

    struct name_entity_walk        * linked_walk; /* Backlink - no refcount */
    
    struct string * entity_value;
    uint16          unicode_value;          /* Numeric reference     */

    unsigned    flags : NUM_ne_match_bits;
    
} * malloc_name_entity_match P_((struct string * entity_value,
				 uint16          unicode_value,
				 struct tagfilter_entities_conf * conf,
				 int             lineno,
				 unsigned        ne_match_flags
				 ));

/* --------------------------------------------------------------------------- */

#define ENTITY_SORT_ITEM_magic		0xFD11

struct entity_sort_item {
    unsigned short      magic;		/* ENTITY_SORT_ITEM_magic */

    int   refcount;
    
    struct string_sort * named_reference;      /* Named reference including & ; */
    struct string      * reference_key;        /* copy of named_reference       */
    
    struct entity_item {
	struct tagfilter_entities * entity_type;          /* "html" */
	struct name_entity_match  * result;               /* Entity value */

	unsigned int old          :1;              /* read by  parse_tagfilter_entities() */
	unsigned int need_on_disk :1;              /* need seen on disk during merge */
	unsigned int delete       :1;              /* delete after dump */


    } *     entities;
    size_t  entities_count;

};




S_(alloc_sort_item_f alloc_entity_sort_item)
static void alloc_entity_sort_item P_((union sort_item      * res,
				       const union sort_key   key,
				       const union sort_item_default def
				       ));
static void alloc_entity_sort_item(res,key,def)
     union sort_item      * res;
     const union sort_key   key;
     const union sort_item_default def;  
{
    struct entity_sort_item * entity_item =
	safe_zero_alloc(sizeof(*entity_item));

    entity_item->refcount = 1;
    
    entity_item->named_reference = key.string_sort;    
    inc_string_sort_refcount(entity_item->named_reference);

    /* Increments refcount */
    entity_item->reference_key =
	give_string_from_string_sort(entity_item->named_reference);
    
    entity_item->entities = NULL;
    entity_item->entities_count = 0;
    
    entity_item->magic =  ENTITY_SORT_ITEM_magic;

    res->entity = entity_item;
}

static void free_entity_item_data P_((struct entity_item *I));
static void free_entity_item_data(I)
     struct entity_item *I;
{
    I->entity_type = NULL;   /* This is static data */
	   	    
    if (I->result) {
	if (NAME_ENTITY_MATCH_magic != I->result->magic)
	    mime_panic(__FILE__,__LINE__,"free_entity_item_data",
		       "Bad magic number (name_entity_match)");
		    
	I->result->conf = NULL;
		
	free_name_entity_match(& (I->result));
    }
}



S_(free_sort_item_f free_entity_sort_item)
static void free_entity_sort_item P_((union sort_item      * ptr)); /* Decrements refcount */
static void free_entity_sort_item(ptr)
     union sort_item      * ptr; /* Decrements refcount */
{    
    struct entity_sort_item * entity_item = ptr->entity;
	
    if (ENTITY_SORT_ITEM_magic != entity_item->magic)
	mime_panic(__FILE__,__LINE__,"free_entity_sort_item",
		   "Bad magic number (entity_sort_item)");
    
    if (entity_item->refcount < 1)
	mime_panic(__FILE__,__LINE__,"free_entity_sort_item",
		   "Bad recount");

    entity_item->refcount--;

    if (entity_item->refcount > 0) {
	ptr->entity = NULL;
	return;
    }

    /* Decrement refcount */
    if (entity_item->named_reference)
	free_string_sort(& (entity_item->named_reference));

    if (entity_item->reference_key)
	free_string(& (entity_item->reference_key));
    
    if (entity_item->entities) {
	size_t i;

	for (i = 0; i < entity_item->entities_count; i++) {

	    free_entity_item_data(& (entity_item->entities[i]));
	    
	}

	free(entity_item->entities);
	entity_item->entities = NULL;
    }
    entity_item->entities_count  = 0;

    entity_item->magic = 0; /* Invalidate */
    free(entity_item);
    
    ptr->entity = NULL;
}

S_(inc_sort_item_refcount_f inc_entity_sort_item_refcount)
static void inc_entity_sort_item_refcount P_((union sort_item item));
static void inc_entity_sort_item_refcount(item)
     union sort_item item;
{
    struct entity_sort_item * entity_item = item.entity;
    
    if (ENTITY_SORT_ITEM_magic != entity_item->magic)
	mime_panic(__FILE__,__LINE__,"inc_entity_sort_item_refcount",
		   "Bad magic number (entity_sort_item)");

    entity_item->refcount++;
}

S_(sort_item_debug_name_f entity_sort_item_name)
static struct string * entity_sort_item_name   P_((const union sort_item item));
static struct string * entity_sort_item_name (item)
     const union sort_item item;
{
    struct entity_sort_item * entity_item = item.entity;
    
    if (ENTITY_SORT_ITEM_magic != entity_item->magic)
	mime_panic(__FILE__,__LINE__,"free_entity_sort_item",
		   "Bad magic number (entity_sort_item)");

    /* give_string_from_string_sort() returns original string with 
       incremented refcount         
     */

    return give_string_from_string_sort(entity_item->named_reference);
}

S_(compare_sort_key_to_item_f entity_key_cmp_item)
static int entity_key_cmp_item P_((const union sort_key key,
				   const union sort_item item));
static int entity_key_cmp_item(key,item)
     const union sort_key key;
     const union sort_item item;
{
    struct entity_sort_item * entity_item = item.entity;
    
    if (ENTITY_SORT_ITEM_magic != entity_item->magic)
	mime_panic(__FILE__,__LINE__,"free_entity_sort_item",
		   "Bad magic number (entity_sort_item)");
    
    return string_sort_cmp(key.string_sort,entity_item->named_reference);
}

S_(sort_key_debug_name_f entity_key_name)
static struct string * entity_key_name P_((const union sort_key key));
static struct string * entity_key_name(key)
     const union sort_key key;
{
    /* give_string_from_string_sort() returns original string with 
       incremented refcount */
    
    return give_string_from_string_sort(key.string_sort);
}

static struct sort_operation entity_map_operation = {
    SORT_OPERATION_magic,
    alloc_entity_sort_item,
    free_entity_sort_item,
    inc_entity_sort_item_refcount,
    entity_sort_item_name,
    
    entity_key_cmp_item,

    /* KEY operations */
    entity_key_name
};


#define TAGFILTER_ENTITIES_CONF_magic	0xFD10

static struct tagfilter_entities_conf {
    unsigned short      magic;		/* TAGFILTER_ENTITIES_CONF_magic */

    enum record_mode    rc;
    char              * filename;

    struct sortlist    * root;

    unsigned int  linked:1;
    
} * malloc_tagfilter_entities_conf P_((enum record_mode rc,const char *filename,
				       size_t alloc_size));

/* ---------------------------------------------------- */


static struct name_entity_match amp_match = {
    NAME_ENTITY_MATCH_magic,
    1,
    NULL /*  struct tagfilter_entities_conf * */,
    1,
    NULL /* struct name_entity_walk        *  */,
    NULL /* struct string *                   */,
    0x0026 /* &  */,
    ne_match_bit_SEMICOLON|ne_match_bit_STATIC
};

static struct name_entity_match  nbsp_match = {
    NAME_ENTITY_MATCH_magic,
    1,
    NULL /*  struct tagfilter_entities_conf * */,
    2,
    NULL /* struct name_entity_walk        *  */,
    NULL /* struct string *                   */,
    0x00A0 /* NO-BREAK SPACE */,
    ne_match_bit_SEMICOLON|ne_match_bit_STATIC
};

static struct name_entity_match quot_match = {
    NAME_ENTITY_MATCH_magic,
    1,
    NULL /*  struct tagfilter_entities_conf * */,
    3,
    NULL /* struct name_entity_walk        * */,
    NULL /* struct string *                   */,
    0x0022 /* "  */,
    ne_match_bit_SEMICOLON|ne_match_bit_STATIC
};

static struct name_entity_match lt_match = {
    NAME_ENTITY_MATCH_magic,
    1,
    NULL /*  struct tagfilter_entities_conf * */,
    4,
    NULL /* struct name_entity_walk        * */,
    NULL /* struct string *                   */,
    0x003C /* < */,
    ne_match_bit_SEMICOLON|ne_match_bit_STATIC
};

static struct name_entity_match gt_match = {
    NAME_ENTITY_MATCH_magic,
    1,
    NULL /*  struct tagfilter_entities_conf * */,
    5,
    NULL /* struct name_entity_walk        * */,
    NULL /* struct string *                   */,
    0x003E /* > */,
    ne_match_bit_SEMICOLON|ne_match_bit_STATIC
};

struct builtin_entity {
    const char * ascii_entity;
    
    struct name_entity_match * static_result;
    
} html_default_entities[] = {
    { "&amp;",  & amp_match  },
    { "&nbsp;", & nbsp_match },
    { "&quot;", & quot_match },
    { "&lt;",   & lt_match   },
    { "&gt;",   & gt_match   }
};
    

#define TAGFILTER_ENTITIES_magic	0xFD0C

struct tagfilter_entities  { 
    unsigned short               magic;    /* TAGFILTER_ENTITIES_magic */

    const char * type_tag;
    /* Now not supported? */


    struct name_entity_walk * ampersand;   /* root of tree - entities start
					      with 0026 AMPERSAND (&)
					   */

    unsigned int  builtin_generated : 1;

    struct builtin_entity * default_entities;
    size_t                  default_entities_count;

    
} HTML_entities  = {
    TAGFILTER_ENTITIES_magic,
    "html",
    NULL, 
    0  /* need generate tree from builtin */,
    &(html_default_entities[0]),
    (sizeof html_default_entities) / sizeof (html_default_entities[0])    
};


static const int entity_character_count =
    ( 0x0039 - 0x0030 + 1 ) +
    ( 0x005A - 0x0041 + 1 ) +
    ( 0x007A - 0x0061 + 1 );
    
int tagfilter_entity_character(u,res)
     uint16 u;
     int * res;
{
    int v = 0;
    int b = 0;
    int r = 0;
    
    if (0x0030 /* 0 */ <= u && u <= 0x0039 /* 9 */) {
	v = b + u - 0x0030;
	goto done;
    }
    b += 0x0039 - 0x0030 + 1;
    
    if (0x0041 /* A */ <= u && u <= 0x005A /* Z */) {
	v = b + u - 0x0041;
	goto done;
    }
    b += 0x005A - 0x0041 + 1;
    
    if (0x0061 /* a */ <= u && u <= 0x007A /* z */) {
	v = b + u - 0x0061;
	goto done;
    }
    b += 0x007A - 0x0061 + 1;

    if (b != entity_character_count)
	mime_panic(__FILE__,__LINE__,"tagfilter_entity_character",
		   "Bad b (base) or entity_character_count");
    
    if (r) {
    done:
	r = 1;
	if (res)
	    *res = v;
    } else {
	if (res)
	    *res = 0;
    }

    return r;
}

static void inc_name_entity_walk_refcount P_((struct name_entity_walk *ptr));

/* Increment refcount */
enum walk_mode { walk_search = 0, walk_create };

static struct name_entity_walk * advance_walk P_((struct name_entity_walk * walk,  uint16 u,
						  enum walk_mode create_name_entity_walk,
						  struct tagfilter_entities * ents));
static struct name_entity_walk * advance_walk(walk,u,create_name_entity_walk,ents)
     struct name_entity_walk * walk;
     uint16 u;
     enum walk_mode create_name_entity_walk;
     struct tagfilter_entities * ents;
{
    struct name_entity_walk * res = NULL;
    int                       vidx = 0;
    
    if (NAME_ENTITY_WALK_magic != walk->magic)
	mime_panic(__FILE__,__LINE__,"advance_walk",
                   "Bad magic number");

    if (tagfilter_entity_character(u,&vidx)) {

	/* Ideally rel index tells directly correct index */
	
	size_t rel = walk->alternatives_count * vidx / entity_character_count;
	size_t min = 0;
	size_t max = walk->alternatives_count;
	int ok = 0;
	size_t mid_idx;
	
	DPRINT(Debug,14,(&Debug,
			 "advance_walk: u=x%04x vidx=%d rel=%zu (alternatives_count=%zu)\n",
			 u,vidx,rel,walk->alternatives_count));

	if (walk->alternatives_count > 0) {
	    size_t loopidx;
	    
	    for (loopidx = 0;
		 min <= max && loopidx < walk->alternatives_count;
		 loopidx++) {
		
		int done = 0;
		uint16 mid_code;
		
		DPRINT(Debug,14,(&Debug,"advance_walk: #%zu:",
				 loopidx));
		
		if (!loopidx) {		    
		    mid_idx = rel;
		    DPRINT(Debug,14,(&Debug," u=x%04x (#%d) relative idx #%zu",
				     u,vidx,rel));
		} else {
		    mid_idx = (min+max)/2;
		    DPRINT(Debug,14,(&Debug," min=%zu max=%zu",
				     min,max));
		}
		DPRINT(Debug,14,(&Debug," => mid_idx=%zu\n",mid_idx));
		

		if (mid_idx >=  walk->alternatives_count)
		    mime_panic(__FILE__,__LINE__,"advance_walk",
			       "Bad mid_idx");
		
		if (!  walk->alternatives[mid_idx]) {
		    DPRINT(Debug,14,(&Debug,"advance_walk: #%zu: No item for idx %zu\n",
				     loopidx,mid_idx));
		    break;
		}

		if (NAME_ENTITY_WALK_magic != walk->alternatives[mid_idx]->magic)
		    mime_panic(__FILE__,__LINE__,"advance_walk",
			       "Bad magic number (on mid_idx)");

		if (ents != walk->alternatives[mid_idx]->ents)
		     mime_panic(__FILE__,__LINE__,"advance_walk",
				"Bad ents (on mid_idx)");
		
		mid_code = walk->alternatives[mid_idx]->code;
		DPRINT(Debug,14,(&Debug,
				 "advance_walk: #%zu: Item #%zu x%04x ",
				 loopidx,mid_idx,mid_code));
		
		if (mid_code < u) {
		    DPRINT(Debug,14,(&Debug,"<"));

		    if (mid_idx < walk->alternatives_count-1)
			min = mid_idx + 1;
		    else
			done = 1;
		    
		} else if (mid_code > u) {
		    DPRINT(Debug,14,(&Debug,">"));

		    if (mid_idx > 0)
			max = mid_idx - 1;
		    else
			done = 1;
		    		    
		} else {
		    DPRINT(Debug,14,(&Debug,"="));
		}

		DPRINT(Debug,14,(&Debug," u=x%04x\n",u));

		if (done) {
		    DPRINT(Debug,14,(&Debug,
				     "advance_walk: #%zu: quit search at %zu\n",
				     loopidx,mid_idx));
		    break;
		} else if (mid_code == u) {
		    DPRINT(Debug,14,(&Debug,
				     "advance_walk: #%zu: succeed search at %zu\n",
				     loopidx,mid_idx));
		    ok = 1;
		    break;
		} else {
		    DPRINT(Debug,14,(&Debug,
				     "advance_walk: #%zu: mid_idx=%zu -> min=%zu max=%zu\n",
				     loopidx, mid_idx,min,max));
		}
	    }


	    if (ok && mid_idx <  walk->alternatives_count && walk->alternatives[mid_idx]) {
		
		if (NAME_ENTITY_WALK_magic != walk->alternatives[mid_idx]->magic)
		    mime_panic(__FILE__,__LINE__,"advance_walk",
			       "Bad magic number (on mid_idx)");

		if (ents != walk->alternatives[mid_idx]->ents)
		    mime_panic(__FILE__,__LINE__,"advance_walk",
			       "Bad ents (on mid_idx)");
		
		if (u !=  walk->alternatives[mid_idx]->code)
		    mime_panic(__FILE__,__LINE__,"advance_walk",
			       "Bad code (on mid_idx)");

		res = walk->alternatives[mid_idx];

		goto found;
	    }

	    DPRINT(Debug,14,(&Debug,
			     "advance_walk: u=x%04x Search ended mid_idx=%zu\n",
			     u,mid_idx));

	} else {
	    DPRINT(Debug,14,(&Debug,
			     "advance_walk: u=x%04x no alternatives\n"));
	    mid_idx = 0;
	}

	switch (create_name_entity_walk) {
	    size_t pos;
	    size_t newcount;
	    size_t z;
	    
	case walk_search: DPRINT(Debug,14,(&Debug,
					   "advance_walk:  u=x%04x not found\n"));
	    break;
	    
	case walk_create: DPRINT(Debug,14,(&Debug,
					   "advance_walk:  u=x%04x creating "));
	    pos = 0;
	    
	    if (mid_idx <  walk->alternatives_count) {
		
		if (NAME_ENTITY_WALK_magic != walk->alternatives[mid_idx]->magic)
		    mime_panic(__FILE__,__LINE__,"advance_walk",
			       "Bad magic number (on mid_idx)");
		
		if (walk->alternatives[mid_idx]->code > u) {
		    DPRINT(Debug,14,(&Debug,
				     " before [%zu] code=x%04x ",
				     mid_idx,walk->alternatives[mid_idx]->code));
		    
		    pos = mid_idx;
		    
		} else if (u > walk->alternatives[mid_idx]->code) {
		    DPRINT(Debug,14,(&Debug,
				     " after  [%zu] code=x%04x ",
				     mid_idx,walk->alternatives[mid_idx]->code));
		    
		    pos = mid_idx+1;
		    
		} else {
		    mime_panic(__FILE__,__LINE__,"advance_walk",
			       "Bad mid_idx / compare result");
		}
		
	    } else {
		DPRINT(Debug,14,(&Debug,
				 " - at end  (alternatives_count=%zu)",
				 walk->alternatives_count));
		
		pos = walk->alternatives_count;
	    }
	    
	    DPRINT(Debug,14,(&Debug," pos=%zu\n",pos));
	    
	    
	    newcount = walk->alternatives_count+1;
	    walk->alternatives = safe_array_realloc(walk->alternatives,
						    newcount,
						    sizeof (walk->alternatives[0]));
	    walk->alternatives[walk->alternatives_count] = NULL;
	    
	    for (z = walk->alternatives_count; z > pos; z--) {
		walk->alternatives[z] = walk->alternatives[z-1];
		walk->alternatives[z-1] = NULL;
	    }
	    
	    if (z != pos)
		mime_panic(__FILE__,__LINE__,"advance_walk",
			   "Loop error");
	    
	    if (walk->alternatives[pos])
		mime_panic(__FILE__,__LINE__,"advance_walk",
			   "pos not empty");
	    
	    walk->alternatives[pos] =
		malloc_name_entity_walk(ents,u);
	    
	    walk->alternatives_count = newcount;
	    res = walk->alternatives[pos];
	    break;	    
	}
		       	
    } else {
	DPRINT(Debug,14,(&Debug,"advance_walk: u=x%04x not entity_character\n",u));

    }

    if (res) {
	
    found:	
	DPRINT(Debug,14,(&Debug,"advance_walk=%p (found or created)\n",res));
	
	inc_name_entity_walk_refcount(res);
    } else {
	DPRINT(Debug,14,(&Debug,"advance_walk=NULL (not found or created)\n"));
    }
	
    return res;
}
						  
/* Increment refcount */
static struct name_entity_walk * walk_reference_key P_((struct tagfilter_entities * entity_type,
							struct string             * reference_key,
							int have_semicolon));
static struct name_entity_walk * walk_reference_key(entity_type,
						    reference_key,
						    have_semicolon)
     struct tagfilter_entities * entity_type;
     struct string             * reference_key;
     int                         have_semicolon;
{
    struct name_entity_walk *walk = NULL;
    int len = string_len(reference_key);
    int idx;

    for (idx = 0; idx < len; idx++) {
	uint16 code = give_unicode_from_string(reference_key,idx);
	
	if (0 == idx && 0x0026 /* & */ == code) {
	    if (TAGFILTER_ENTITIES_magic != entity_type->magic)
		mime_panic(__FILE__,__LINE__,"walk_reference_key",
			   "Bad magic number (tagfilter_entities)");

	    if (! entity_type->ampersand)
		entity_type->ampersand = malloc_name_entity_walk(entity_type,code);

	    walk = entity_type->ampersand;
	    
	    inc_name_entity_walk_refcount(walk);

	} else if (idx == len-1 && 0x003B /* ; */ == code && have_semicolon) {
	    goto found;
	} else if (walk) {
	    struct name_entity_walk *tmp =  advance_walk(walk,code,walk_create,
							 entity_type);

	    free_name_entity_walk(&walk);
	    
	    walk = tmp;
	}

	if (!walk) {
	    DPRINT(Debug,14,(&Debug,
			     "walk_reference_key: \"%S\" failed at %d code=x%04x\n",
			     reference_key,idx,code));
	    break;
	}	
    }
    
 found:
    if (!walk) {
	DPRINT(Debug,14,(&Debug,
			 "walk_reference_key: Failed to create node for \"%S\"",
			 reference_key));
    }
	
    return walk;
}

/* Increment refcount */
static struct name_entity_walk * walk_reference_ascii P_((struct tagfilter_entities * entity_type,
							  const unsigned char * ascii_key,
							  int have_semicolon));
static struct name_entity_walk * walk_reference_ascii(entity_type,
						      ascii_key,
						      have_semicolon)
     struct tagfilter_entities * entity_type;
     const unsigned char * ascii_key;
     int                         have_semicolon;
{
    struct name_entity_walk *walk = NULL;
    int idx;

    DPRINT(Debug,14,(&Debug,
		    "walk_reference_ascii: ascii_key=%Q - %s \n",
		    ascii_key,
		    have_semicolon ? " have semicolon" : " no semicolon"));

    
    for (idx = 0; ascii_key[idx]; idx++) {
	DPRINT(Debug,14,(&Debug,
			 "walk_reference_ascii: #%d '%c' ",
			 idx,ascii_key[idx]));
	if (walk) {
	    if (NAME_ENTITY_WALK_magic != walk->magic)
		mime_panic(__FILE__,__LINE__,"name_entity_walk",
			   "Bad magic number (tagfilter_entities)");

	    DPRINT(Debug,14,(&Debug,"walk => x%04x",walk->code));
	}		
	DPRINT(Debug,14,(&Debug,"\n"));
	
	if (0 == idx && '&' == ascii_key[idx]) {
	    if (TAGFILTER_ENTITIES_magic != entity_type->magic)
		mime_panic(__FILE__,__LINE__,"walk_reference_ascii",
			   "Bad magic number (tagfilter_entities)");

	    if (! entity_type->ampersand)
		entity_type->ampersand = malloc_name_entity_walk(entity_type,ascii_key[idx]);

	    walk = entity_type->ampersand;
	    
	    inc_name_entity_walk_refcount(walk);

	} else if (!ascii_key[idx+1] && ';' == ascii_key[idx]  && have_semicolon) {
	    goto found;
	} else if (walk) {
	    struct name_entity_walk *tmp =  advance_walk(walk,
                                                         ascii_key[idx],walk_create,
							 entity_type);
	    free_name_entity_walk(&walk);
	    
	    walk = tmp;
	}

	if (!walk) {
	    DPRINT(Debug,1,(&Debug,
			     "walk_reference_ascii: %Q failed at %d char=%d\n",
			     ascii_key,idx,ascii_key[idx]));
	    break;
	}	     
    }

 found:
    if (!walk) {
	DPRINT(Debug,1,(&Debug,
			"walk_reference_ascii: Failed to create node for %Q",
			 ascii_key));
    }
	
    return walk;
}

static void inc_name_entity_match_refcount P_((struct name_entity_match *ptr));

static void link_name_entity_walk P_((struct name_entity_walk  * walk,
				      struct name_entity_match * result,
				      int *errors));
static void link_name_entity_walk(walk,result,errors)
     struct name_entity_walk  * walk;
     struct name_entity_match * result;
     int                      * errors;
{
    int ok = 0;
    struct tagfilter_entities_conf *conf = NULL;
    char * filename = "<builtin>";
    
    if (NAME_ENTITY_MATCH_magic != result->magic)
	mime_panic(__FILE__,__LINE__,"link_name_entity_walk",
		   "Bad magic number (name_entity_match)");

    if (NAME_ENTITY_WALK_magic != walk->magic)
	mime_panic(__FILE__,__LINE__,"link_name_entity_walk",
                   "Bad magic number");

    if (result-> linked_walk &&
	result-> linked_walk != walk)
	mime_panic(__FILE__,__LINE__,"link_name_entity_walk",
		   "Result already used for elsewhere");

    if (result->conf) {
	struct tagfilter_entities_conf *conf = result->conf;

	if (TAGFILTER_ENTITIES_CONF_magic != conf->magic)
	    mime_panic(__FILE__,__LINE__,"link_name_entity_match",
		       "Bad magic number");

	filename = conf->filename;	
    }

    if (walk->results) {
	size_t i;

	for (i = 0; i < walk->results_count; i++) {
	    
	    if (walk->results[i] == result) {
		
		DPRINT(Debug,14,(&Debug,
				 "link_name_entity_walk: walk(%p)->results[%zu] == result(%p) -- already done\n",
				 walk,i,result));
		goto found;
	    }
	}

	if (ison(result->flags,ne_match_IGNORE)) {
	    DPRINT(Debug,14,(&Debug,
			     "link_name_entity_walk: walk(%p) skipping result(%p)\n",
			     walk,result));
	    goto skipping;
	}
	
	for (i = 0; i < walk->results_count; i++) {
	    
	    if (walk->results[i]) {
		if (NAME_ENTITY_MATCH_magic != walk->results[i]->magic)
		    mime_panic(__FILE__,__LINE__,"link_name_entity_walk",
			       "Bad magic number (name_entity_match)");
		

		if (conf == walk->results[i]->conf &&
		    result->flags == walk->results[i]->flags) {

		    if (result->unicode_value == walk->results[i]->unicode_value &&
			! result->entity_value &&
			! walk->results[i]->entity_value) {

			DPRINT(Debug,14,(&Debug,
					 "link_name_entity_walk: walk(%p) skipping duplicate result(%p)\n",
					 walk,result));

			setit(result->flags,ne_match_IGNORE);
			
			goto skipping;
		    }


		    if (errors) {
			lib_error(CATGETS(elm_msg_cat, MeSet, 
					  MeEntityDuplicate2,
					  "%s: %d: Duplicate entity - already given on line %d"),
				  filename,result->lineno,
				  walk->results[i]->lineno);
			(*errors)++;
		    }
			
		    DPRINT(Debug,14,(&Debug,
				     "link_name_entity_walk: walk(%p) skipping duplicate result(%p)\n",
				     walk,result));
		    
		    setit(result->flags,ne_match_IGNORE);

		    
		    goto skipping;		    
		}				
	    }	    
	}	
    }

    ok = 1;

    walk->results =  safe_array_realloc(walk->results,
					walk->results_count+1,
					sizeof (walk->results[0]));

    walk->results[walk->results_count] = result;
    inc_name_entity_match_refcount(walk->results[walk->results_count]);
    
    DPRINT(Debug,14,(&Debug,
		     "link_name_entity_walk: walk(%p)->results[%zu] == result(%p) -- added\n",
		     walk,walk->results_count,result));

    walk->results_count++;

    
 skipping:
    if (ok) {
	
    found:
	result-> linked_walk =  walk;

	DPRINT(Debug,14,(&Debug,
			 "link_name_entity_walk: walk(%p) linked with result(%p)\n",
			 walk,result));
			 	
    }	
}

static void tagfilter_generate_defaults P_((struct tagfilter_entities *ents));
static void tagfilter_generate_defaults(ents)
     struct tagfilter_entities *ents;
{
    if (TAGFILTER_ENTITIES_magic != ents->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_generate_defaults",
                   "Bad magic number");

    ents->builtin_generated = 1;

    if (ents->default_entities) {
	size_t i;

	DPRINT(Debug,14,(&Debug,
			 "tagfilter_generate_defaults: %zu default entities\n",
			 ents->default_entities_count));

	for (i = 0; i < ents->default_entities_count; i++) {

	    struct name_entity_walk * walk;
	    int have_semicolon = 0;

	    if (NAME_ENTITY_MATCH_magic != ents->default_entities[i].static_result->magic)
		mime_panic(__FILE__,__LINE__,"tagfilter_generate_defaults",
			   "Bad magic number (name_entity_match)");

	    have_semicolon = ison(ents->default_entities[i].static_result->flags,
				  ne_match_SEMICOLON);

	    DPRINT(Debug,14,(&Debug,
			     "tagfilter_generate_defaults: #%zu: %Q %s\n",
			     i,ents->default_entities[i].ascii_entity,
			     have_semicolon ? "(have semicolon)" : "(no semicolon)"));
	    
	    walk = walk_reference_ascii(ents,
					cs2us(ents->default_entities[i].ascii_entity),
					have_semicolon);
	    if (!walk)
		mime_panic(__FILE__,__LINE__,"tagfilter_generate_defaults",
			   "Linking failed");

	    if (! ents->default_entities[i].static_result->linked_walk) {

		DPRINT(Debug,14,(&Debug,
				 "tagfilter_generate_defaults: #%zu %s linking\n",
				 i,ents->default_entities[i].ascii_entity));
		
		link_name_entity_walk(walk,ents->default_entities[i].static_result,NULL);

	    } else if (ents->default_entities[i].static_result->linked_walk == walk) {

		DPRINT(Debug,14,(&Debug,
				 "tagfilter_generate_defaults: #%zu %s already linked\n",
				 i,ents->default_entities[i].ascii_entity));
	    } else {
		mime_panic(__FILE__,__LINE__,"tagfilter_generate_defaults",
			   "Bad linking");
	    }
		
	    free_name_entity_walk(&walk);	    
	}

    }
}


static void link_name_entity_match P_((struct name_entity_match  * result,
				       struct tagfilter_entities * entity_type,
				       struct string             * reference_key,
				       int                       * errors));
static void link_name_entity_match(result,entity_type,reference_key,errors)
     struct name_entity_match  * result;
     struct tagfilter_entities * entity_type;  /* Root of linked tree */
     struct string             * reference_key;
     int                       * errors;
{
    int have_semicolon;
    
    struct name_entity_walk * walk;
    
    if (NAME_ENTITY_MATCH_magic != result->magic)
	mime_panic(__FILE__,__LINE__,"link_name_entity_match",
		   "Bad magic number (name_entity_match)");

    if (TAGFILTER_ENTITIES_magic != entity_type->magic)
	mime_panic(__FILE__,__LINE__,"link_name_entity_match",
		   "Bad magic number (tagfilter_entities)");
    
    
    have_semicolon = ison(result->flags,ne_match_SEMICOLON);

    /* Increment refcount */
    walk =  walk_reference_key(entity_type,
			       reference_key,
			       have_semicolon);


    if (walk) {
    
	if (! result->linked_walk) {

	    link_name_entity_walk(walk,result,errors);

	} else {
	    /* just verify */

	    if (walk != result->linked_walk)
		mime_panic(__FILE__,__LINE__,"link_name_entity_match",
			   "Bad linking");
	}
	
	free_name_entity_walk(&walk);
	
    } else {

	if (result->linked_walk)
	    mime_panic(__FILE__,__LINE__,"link_name_entity_match",
		       "Bad linking / reference_key");

	mime_panic(__FILE__,__LINE__,"link_name_entity_match",
		   "Can not link given  reference_key");

    }        
}

void link_tagfilter_entities(conf,errors)
     struct tagfilter_entities_conf *conf;
     int * errors;
{
    if (TAGFILTER_ENTITIES_CONF_magic != conf->magic)
	mime_panic(__FILE__,__LINE__,"link_tagfilter_entities",
		   "Bad magic number");

    conf->linked = 1;
    
    if (conf->root) {
	size_t map_len = sort_list_len(conf->root);
	size_t i;

	for (i = 0; i < map_len; i++) {
	    
	    union sort_item res;
	    
	    res.entity = NULL;
	    
	    /* Increments refcount */
	    get_sort_list_item(conf->root,sort_list_get_normal,i,&res);
	    
	    if (res.entity) {
		struct entity_sort_item * entity_item = res.entity;
		
		if (ENTITY_SORT_ITEM_magic != entity_item->magic)
		    mime_panic(__FILE__,__LINE__,"link_tagfilter_entities",
			       "Bad magic number (entity_sort_item)");

		if (entity_item->entities) {
		    size_t j;
		    
		    for (j = 0; j < entity_item->entities_count; j++) {

			if (entity_item->entities[j].entity_type &&
			    entity_item->entities[j].result) {

			    if (TAGFILTER_ENTITIES_magic !=
				entity_item->entities[j].entity_type->magic)
				mime_panic(__FILE__,__LINE__,"link_tagfilter_entities",
					   "Bad magic number (tagfilter_entities)");

			    if (NAME_ENTITY_MATCH_magic !=
				entity_item->entities[j].result->magic)
				mime_panic(__FILE__,__LINE__,"link_tagfilter_entities",
					   "Bad magic number (name_entity_match)");
			    
			    if (conf != entity_item->entities[j].result->conf)
				mime_panic(__FILE__,__LINE__,"link_tagfilter_entities",
					   "Bad conf (name_entity_match)");

			    link_name_entity_match(entity_item->entities[j].result,
						   entity_item->entities[j].entity_type,
						   entity_item->reference_key,
						   errors
						   );
			    
			}
			
		    }
		}
		
		free_entity_sort_item(&res);
	    }
	}
    }
}


static void unlink_name_entity_walk P_((struct name_entity_walk **walk,
					struct tagfilter_entities * ents));

static void unlink_name_entity_walk(walk,ents)
     struct name_entity_walk **walk;
     struct tagfilter_entities * ents;
{
    struct name_entity_walk ** worklist = NULL;
    size_t worklist_len = 0;
    size_t idx= 0;
    
    if (NAME_ENTITY_WALK_magic != (*walk)->magic)
	mime_panic(__FILE__,__LINE__,"unlink_name_entity_walk",
                   "Bad magic number");
    
    if (TAGFILTER_ENTITIES_magic != ents->magic)
	mime_panic(__FILE__,__LINE__,"unlink_name_entity_walk",
                   "Bad magic number");

    worklist    = safe_zero_alloc(sizeof (worklist[0]));
    worklist[0] = *walk;
    worklist_len = 1;
    *walk = NULL;
    
    /* Flatten */

    while (idx < worklist_len) {
	if (NAME_ENTITY_WALK_magic != worklist[idx]->magic)
	    mime_panic(__FILE__,__LINE__,"unlink_name_entity_walk",
		       "Bad magic number");

	if (ents != worklist[idx]->ents)
	    mime_panic(__FILE__,__LINE__,"unlink_name_entity_walk",
		       "Bad backlink (struct tagfilter_entities * ents)");

	/* Remove only tree part */
	
	if (worklist[idx]->alternatives) {
	    size_t newlen = worklist_len + worklist[idx]->alternatives_count;
	    size_t i;

	    worklist = safe_array_realloc(worklist,newlen,sizeof (worklist[0]));

	    for (i = 0; i < worklist[idx]->alternatives_count && worklist_len < newlen; i++) {
		if (worklist[idx]->alternatives[i]) {
		    worklist[worklist_len++]       = worklist[idx]->alternatives[i];
		    worklist[idx]->alternatives[i] = NULL;
		}
	    }


	    free(worklist[idx]->alternatives);
	    worklist[idx]->alternatives = NULL;
	}
	worklist[idx]->alternatives_count = 0;
		    
	worklist[idx]->ents = NULL;
	idx++;
    }

    /* Release */
    for (idx = 0; idx < worklist_len; idx++) {
	free_name_entity_walk(& worklist[idx]);
    }

    free(worklist);
    worklist = NULL;    
}


static struct tagfilter_entities * entity_types[] = {
    &HTML_entities
};
static const size_t entity_types_count = (sizeof entity_types) / sizeof (entity_types[0]);


void unlink_tagfilter_entities()
{
    size_t y;

    for (y = 0; y < entity_types_count; y++) {
	if (TAGFILTER_ENTITIES_magic != entity_types[y]->magic)
	    mime_panic(__FILE__,__LINE__,"unlink_tagfilter_entities",
		       "Bad magic number (tagfilter_entities)");


	DPRINT(Debug,14,(&Debug,
			 "unlink_tagfilter_entities: %zu: unlinking %s\n",
			 y,entity_types[y]->type_tag));

	if (entity_types[y]->ampersand)
	    unlink_name_entity_walk(& (entity_types[y]->ampersand),
				    entity_types[y]);

	entity_types[y]->builtin_generated = 0;

	
    }
}


struct tagfilter_entities_conf * new_tagfilter_entities_conf(rc,filename)
     enum record_mode rc;
     const char *filename;
{
    /* Just dummy */
    
    return malloc_tagfilter_entities_conf(rc,filename,1);
}

int dump_tagfilter_entities(f,conf,commentfile,actor,version_buff,fileset,
			    propline,rc,errno_res)
     FILE *f;
     struct tagfilter_entities_conf *conf;
     FILE *commentfile;
     const char *actor;
     char *version_buff;
     charset_t fileset;
     const struct editor_propline *propline;
     enum record_mode rc  /* SYSTEM_RC = 0, LOCAL_RC = 1 */;
     int * errno_res;
{
    int ret = 0;
    const char * sysnam = NULL;

    enum editor_propline_v propline_mode = 
	give_dt_enumerate_as_int(&editor_tfent_propline);

    int r;
    
    if (errno_res)
	*errno_res = 0;
    
    if (!conf)
	return 0;

    if (TAGFILTER_ENTITIES_CONF_magic != conf->magic)
	mime_panic(__FILE__,__LINE__,"dump_tagfilter_entities",
		   "Bad magic number (tagfilter_entities_conf)");

    if (!fileset) {
	DPRINT(Debug,10,(&Debug,
			 "dump_tagfilter_entities: File charset is not set, using system charset\n"));
	fileset = system_charset;    
    }
    
    sysnam = get_charset_MIME_name(fileset);
 
    /* Must have first line */
    if (propline_mode != editor_propline_ignore)
	write_editor_propline(propline,propline_mode,
			      f,"# ","\n",fileset);

    if (sysnam) {
	/* Store used character set */

	elm_fprintf(f, 
		    CATGETS(elm_msg_cat,MeSet, MeCharsetExplain,
			    "# Character set used on this file\n"));

	fprintf(f,"@charset = %s\n",sysnam);
    }

     insert_commentfile(f,ELMTAGFILENT_INFO,commentfile,
		       actor,version_buff);

     if (conf->root) {
	 size_t map_len = sort_list_len(conf->root);
	 size_t i;
	 int need_delete = 0;

	 DPRINT(Debug,14,(&Debug,
			  "dump_tagfilter_entities: %zu names\n",
			  map_len));


	 for (i = 0; i < map_len; i++) {
		
	     union sort_item res;
	     
	     res.entity = NULL;
	     
	     /* Increments refcount */
	     get_sort_list_item(conf->root,sort_list_get_normal,i,&res);
	     
	     if (res.entity) {
		 struct entity_sort_item * entity_item = res.entity;
		 int failcount = 0;
		 struct string * S1 = NULL;

		 char * S1_res = NULL;
		 int    S1_len = 0;
		 
		 if (ENTITY_SORT_ITEM_magic != entity_item->magic)
		     mime_panic(__FILE__,__LINE__,"dump_tagfilter_entities",
				"Bad magic number (entity_sort_item)");

		 S1 = convert_string2(fileset,entity_item->reference_key,&failcount);
		 
		 
		 DPRINT(Debug,14,(&Debug,"dump_tagfilter_entities: entity %zu = \"%S\"",
				  i,entity_item->reference_key));

		 if (failcount > 0 || ! S1) {
		     DPRINT(Debug,14,(&Debug,", can't convert to %s\n",
				      sysnam ? sysnam : "<no MIME name>"));
		     goto failentity;
		 }




		 bytestream_from_string(S1,&S1_res,&S1_len);
		 
		 if (entity_item->entities) {
		     size_t j;
		     size_t width = 8;
		     
		     DPRINT(Debug,14,(&Debug," -- count %zu\n",
				      entity_item->entities_count));
		     
		     for (j = 0; j < entity_item->entities_count; j++) {
			 const char * type = "???";
			 
			 struct string * S2 = NULL;
			 int failS2count = 0;
			 char * S2_res = NULL;
			 int    S2_len = 0;
			 
			 if (entity_item->entities[j].entity_type) {
			     if (TAGFILTER_ENTITIES_magic !=
				 entity_item->entities[j].entity_type->magic)
				 mime_panic(__FILE__,__LINE__,"dump_tagfilter_entities",
					    "Bad magic number (tagfilter_entities)");
			     type = entity_item->entities[j].entity_type->type_tag;
			     
			 } else {
			     DPRINT(Debug,14,(&Debug,"                       : %zu -- skipping\n",
					      j));
			     continue;
			 }
			 
			 DPRINT(Debug,14,(&Debug,"                       : %zu %s", 
					  j,type));
			 
			 if (entity_item->entities[j].delete) {
			     DPRINT(Debug,14,(&Debug,", marked for delete -- skipping\n"));
			     
			     need_delete++;
			     continue;
			 }
			 
			 if (entity_item->entities[j].result) {
			     
			     if (NAME_ENTITY_MATCH_magic != entity_item->entities[j].result->magic)
				 mime_panic(__FILE__,__LINE__,"dump_tagfilter_entities",
					    "Bad magic number (name_entity_match)");
			     
			     
			     if (ison(entity_item->entities[j].result->flags,ne_match_IGNORE)) {
				 DPRINT(Debug,14,(&Debug,", marked for ignore -- skipping\n"));
				 
				 entity_item->entities[j].delete = 1;
				 need_delete++;
				 continue;
			     }
			     			     
			     if (entity_item->entities[j].result->entity_value) {
				 
				 S2 = convert_string2(fileset,
						      entity_item->entities[j].result->entity_value,
						      &failS2count);

				 if (!S2 || failS2count > 0) {
				     DPRINT(Debug,14,(&Debug,
						      ", failed to convert value \"%S\" to %s\n",
						      entity_item->entities[j].result->entity_value,
						      sysnam ? sysnam : "<no MIME name>"));
				     goto failedresult;
				 }

				 bytestream_from_string(S2,&S2_res,&S2_len);
				 
			     }			     
			     
			     entity_item->entities[j].old = 1;
			     
			     fwrite(S1_res,1,S1_len,f);

			     if (S1_len < width) {
				 size_t zz = S1_len;

				 /* Pad with space */
				 while (zz < width) {
				     putc(' ',f);
				     zz++;  /* Ascii compatible assumed */
				 }
			     } else if (S1_len < 20 && S1_len/8 > width/8) {
				 width = S1_len;
			     }
				 			     
			     putc('\t',f);    /* Ascii compatible assumed */
			     fputs(type,f);   /* Ascii compatible assumed */
			     putc('\t',f);    /* Ascii compatible assumed */
			     
			     if (S2) {
				 putc('"',f);    /* Ascii compatible assumed */
				 fwrite(S2_res,1,S2_len,f);
				 putc('"',f);    /* Ascii compatible assumed */
			     } else {
				 fprintf(f,"x%04x",
					 entity_item->entities[j].result->unicode_value);
				 /* Ascii compatible assumed */
			     }
			     
			     putc('\n',f);
			     
			 } else {
			     DPRINT(Debug,14,(&Debug," -- skipping\n"));
			 }			 

			 if (S2_res) {
			     free(S2_res);
			     S2_res = NULL;
			 }
			 failedresult:
			 if (S2)
			     free_string(&S2);
			 
			 
		     }
		 } else {
		     need_delete ++;

		     DPRINT(Debug,14,(&Debug," -- empty (need delete)\n"));		     
		 }

		 if (S1_res) {
		     free(S1_res);
		     S1_res = NULL;
		 }
		 
	     failentity:
		 if (S1)
		     free_string(&S1);
		 
		 free_entity_sort_item(&res);
	     }
	 }

	 if (need_delete) {
	     DPRINT(Debug,14,(&Debug,
			      "dump_tagfilter_entities: %d entities need to be delete (total %zu names)\n",
			      need_delete,map_len));

	     if (map_len > 0) {

		 size_t i = map_len -1;
		size_t deleted = 0;
		
		while (1) {
		    union sort_item res;
		    
		    res.aliases = NULL;
		    
		    /* Increments refcount */
		    get_sort_list_item(conf->root,sort_list_get_normal,i,&res);
		    
		    if (res.entity) {
			struct entity_sort_item * entity_item = res.entity;
			int have_names = 0;

			if (ENTITY_SORT_ITEM_magic != entity_item->magic)
			    mime_panic(__FILE__,__LINE__,"dump_tagfilter_entities",
				       "Bad magic number (entity_sort_item)");
			
			DPRINT(Debug,14,(&Debug,"dump_tagfilter_entities: entity %zu = \"%S\"",
					 i,entity_item->reference_key));
			
			if (entity_item->entities) {
			    size_t j;
			    size_t t;

			    DPRINT(Debug,14,(&Debug," -- count %zu\n",
					     entity_item->entities_count));
					     			    
			    for (j = 0, t = 0; j < entity_item->entities_count; j++) {
				const char * type UNUSED_VAROK = "???";


				if (entity_item->entities[j].entity_type) {
				    if (TAGFILTER_ENTITIES_magic !=
					entity_item->entities[j].entity_type->magic)
					mime_panic(__FILE__,__LINE__,"dumo_tagfilter_entities",
						   "Bad magic number (tagfilter_entities)");
				    type = entity_item->entities[j].entity_type->type_tag;
				}
			    
				DPRINT(Debug,14,(&Debug,"                       : %zu %s", 
						 j,type));

				if (entity_item->entities[j].delete) {
				    DPRINT(Debug,14,(&Debug,", marked for delete\n"));

				    free_entity_item_data(&  (entity_item->entities[j]));
				    
				} else {
				    
				    if (t != i) {
					DPRINT(Debug,14,(&Debug,", => %zu (move)"));

					entity_item->entities[t] = entity_item->entities[j];
					t++;

					bzero(&(entity_item->entities[j]),
					      sizeof (entity_item->entities[j]));
				    }

				    DPRINT(Debug,14,(&Debug,"\n"));
				    have_names = 1;
				}
			    }

			    if (t < entity_item->entities_count) {
				DPRINT(Debug,14,(&Debug,
						 "dump_tagfilter_entities: entity %zu = \"%S\" - count=%zu => %zu\n",
						 i,entity_item->reference_key,entity_item->entities_count,t));

				entity_item->entities_count = t;
			    }
			    
			} else {
			    DPRINT(Debug,14,(&Debug," -- empty, deleting\n"));
			}

			if (!have_names) {
			    entity_item = NULL;
			    
			    DPRINT(Debug,14,(&Debug,
					     "dump_tagfilter_entities: entity %zu = \"%S\" -- deleting\n",
					     i,entity_item->reference_key));

			    free_entity_sort_item(&res);
			    
			    
			    /* Increments refcount */
			    get_sort_list_item(conf->root,sort_list_get_remove,i,&res); 

			    if (res.entity) {
				entity_item = res.entity;
				
				if (ENTITY_SORT_ITEM_magic != entity_item->magic)
				    mime_panic(__FILE__,__LINE__,"dump_tagfilter_entities",
					       "Bad magic number (entity_sort_item)");
				
				DPRINT(Debug,14,(&Debug,"dump_tagfilter_entities: entity %zu = \"%S\" deleted\n",
						 i,entity_item->reference_key));

			    }
			}

			free_entity_sort_item(&res);
		    }
		    
		    if (i > 0)
			i--;
		    else
			break;
		    
		}

		if (deleted) {
		    map_len = sort_list_len(conf->root);

		    
		    DPRINT(Debug,14,(&Debug,
				     "dump_tagfilter_entities: %zu enitites deleted, %zu names left\n",
				     deleted,map_len));

		}
		
	     }
	 }
     }
	 
     r = fflush(f);

    switch (r) {
	int err;

    case 0:
	ret = 1;

	break;
    case EOF:
	err = errno;
	
	DPRINT(Debug,14,(&Debug,
			 "dump_tagfilter_entities: fflush: %s\n",
			 strerror(err)));

	if (errno_res)
	    *errno_res = err;

	break;	
    }

    DPRINT(Debug,14,(&Debug,
		     "dump_tagfilter_entities=%d",ret));
    if (errno_res && *errno_res) {
	DPRINT(Debug,14,(&Debug,"; error %s",
			 strerror(*errno_res)));
    }
    DPRINT(Debug,14,(&Debug,"\n"));


    return ret;
}

static int read_tagfilter_entities P_((const char                      * filename,
				       FILE                            * f,
				       struct tagfilter_entities_conf ** conf,
				       int                             * errors,
                                       charset_t                       * fileset,
				       struct editor_propline         ** propline,
				       enum record_mode rc  /* SYSTEM_RC = 0, 
							       LOCAL_RC = 1 */,
				       int                             * need_rewrite));



/* XXXXX same as  hashmark_is_printableln() */
static int tagfilter_is_printableln P_((const char *filename,
					int lineno,
					const struct string * rest_S));

static int tagfilter_is_printableln(filename,lineno,rest_S)
     const char *filename;
     int lineno;
     const struct string * rest_S;
{
    int L = string_len(rest_S);
    int idx;
    
    for (idx = 0; idx < L; idx++) {
	uint16        ucode = 0;
	unsigned char bcode = 0;
	int gchar_flags = give_character_from_string(rest_S,idx,
						     &ucode,&bcode);

	if (0 != (gchar_flags & GCHAR_unicode) &&
	    0x0009 /* HT  '\t' (horizontal tab) */ == ucode) {
	     
	    /* Allow tabulator */

	} else if (0 == (gchar_flags & GCHAR_unicode ) ||
		   0 == unicode_ch(ucode,UOP_printable)) {
	    char ubuffer[8], * ub = "";
	    char bbuffer[5], * bb = "";
	    
	    if (0 != (gchar_flags & GCHAR_unicode)) {
		elm_sfprintf(ubuffer,sizeof ubuffer,
			     FRM(" u%04X"),
			     ucode);
		ub = ubuffer;
	    }
	    if (0 !=  (gchar_flags & GCHAR_bytecode)) {
		elm_sfprintf(bbuffer,sizeof bbuffer,
			     FRM(" x%02X"),
			     bcode);
		bb = bbuffer;
	    }

	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeNonPrintableCharOnLine,					  
			      "%s: %d: Non-printable character%s%s on line"),
		      filename,lineno,ub,bb);
	    return 0;
	}
    }

    return 1;
}

enum change_mode {
    read_entities = 0,
    merge_entities
};


static int set_tagfilter_entity P_((const char                     * filename,
				    struct tagfilter_entities_conf * conf,
				    struct string                  * named_reference,
				    struct tagfilter_entities      * enttyp,
				    struct string                  * entity_value,
				    uint16                           unicode_value,
				    int                              lineno,
				    unsigned                         ne_match_flags,
				    enum change_mode                 merged_entity,
				    size_t                         * res_idx_p /* append mode */,
				    int                            * append_need_rewrite_p,
				    int                            * errors,
				    struct string                  * rest_S,
				    struct string                  * entity_type,
				    int                              set_old
				    ));
static int set_tagfilter_entity(filename,conf,named_reference,
				enttyp,entity_value, unicode_value,
				lineno,ne_match_flags, merged_entity,
				res_idx_p,
				append_need_rewrite_p,
				errors,rest_S, entity_type,set_old)
     const char                     * filename;
     struct tagfilter_entities_conf * conf;
     struct string                  * named_reference;
     struct tagfilter_entities      * enttyp;
     struct string                  * entity_value;
     uint16                           unicode_value;
     int                              lineno;
     unsigned                         ne_match_flags;
     enum change_mode                 merged_entity;
     size_t                         * res_idx_p /* append mode */;
     int                            * append_need_rewrite_p;
     int                            * errors;
     struct string                  * rest_S;
     struct string                  * entity_type;
     int                              set_old;
{
    int r = 1;
    
    union sort_key          key;
    union sort_item_default def;
    union sort_item         res;

    struct string_sort        * search_key = NULL;

    if (TAGFILTER_ENTITIES_CONF_magic != conf->magic)
	mime_panic(__FILE__,__LINE__," set_tagfilter_entity",
		   "Bad magic number (tagfilter_entities_conf)");

    
    /* Shares named_reference and increments refcount */
    search_key = new_string_sort(named_reference);
    
    key.string_sort = search_key;
    def.dummy       = NULL;
    res.dummy       = NULL;
  
    /* Increments refcount, returns 1 if found or created */
    if (search_sort_list_item(conf->root,
			      sort_list_insert_hint  /* sequential append mode,
							uses res_idx
						     */
			      /* sort_list_search_create  <-- when random order */
			      ,key,def,
			      &res,res_idx_p,append_need_rewrite_p)) {

	DPRINT(Debug,14,(&Debug,
			 "set_tagfilter_entity: search_sort_list_item found or created item #%zu\n",
			 *res_idx_p));

	if (res.entity) {
	    struct entity_sort_item * entity_item = res.entity;
	    size_t  newpos = 0;
	    int merged = 0;
	    
	    if (ENTITY_SORT_ITEM_magic != entity_item->magic)
		mime_panic(__FILE__,__LINE__,"set_tagfilter_entity",
			   "Bad magic number (entity_sort_item)");

	    if (entity_item->entities) {
		size_t a;
		
		for (a = 0; a < entity_item->entities_count; a++) {
		    
		    if (enttyp == entity_item->entities[a].entity_type) {
			
			if (! entity_item->entities[a].result) {
			    
			    newpos = a;
			    goto found1;
			}
			
			if (NAME_ENTITY_MATCH_magic !=
			    entity_item->entities[a].result->magic)
			    mime_panic(__FILE__,__LINE__,"set_tagfilter_entity",
				       "Bad magic number (name_entity_match)");

			if (conf != entity_item->entities[a].result->conf)
			    mime_panic(__FILE__,__LINE__,"set_tagfilter_entities",
				       "Bad conf (name_entity_match)");


			switch (merged_entity) {
			case read_entities:
			    /* Is this merge */
			    if (entity_item->entities[a].need_on_disk) {
				
				newpos = a;
				merged = 1;
				
				goto found1;
			    }
			    break;
			case merge_entities:
			    newpos = a;
			    merged = 1;
				
			    goto found1;
			}

			if (rest_S && entity_type) {
			    if (tagfilter_is_printableln(filename,lineno,rest_S))
				lib_error(CATGETS(elm_msg_cat, MeSet, 
						  MeEntityDuplicate,
						  "%s: %d: Duplicate entity %S with type %S - already given on line %d: %S"),
					  filename,lineno,
					  named_reference,entity_type,
					  entity_item->entities[a].result->lineno,
					  rest_S);
			}
			
			(*errors) ++;
			goto fail2;
			
		    }
		}
	    }

	    entity_item->entities =
		safe_array_realloc(entity_item->entities,
				   entity_item->entities_count+1,
				   sizeof (entity_item->entities[0]));
	    
	    newpos = entity_item->entities_count++;
	    
	    /* bzero is defined on hdrs/defs.h */
	    bzero((void *)&(entity_item->entities[newpos]),
		  sizeof ((entity_item->entities[newpos])));
	    
	    entity_item->entities[newpos].entity_type = enttyp;
	    entity_item->entities[newpos].result      = NULL;
	    entity_item->entities[newpos].old         = set_old;

	    
	found1:
	    
	    if (newpos >= entity_item->entities_count)
		mime_panic(__FILE__,__LINE__,"set_tagfilter_entity",
			   "Bad newpos");
	    
	    if (entity_item->entities[newpos].entity_type != enttyp)
		mime_panic(__FILE__,__LINE__,"set_tagfilter_entity",
			   "Bad type on newpos");
	    
	    entity_item->entities[newpos].need_on_disk = 0;
	    entity_item->entities[newpos].delete       = 0;

	    if (entity_item->entities[newpos].result) {
		if (NAME_ENTITY_MATCH_magic !=
		    entity_item->entities[newpos].result->magic)
		    mime_panic(__FILE__,__LINE__,"set_tagfilter_entity",
			       "Bad magic number (name_entity_match)");
		
		if (conf != entity_item->entities[newpos].result->conf)
		    mime_panic(__FILE__,__LINE__,"set_tagfilter_entity",
			       "Bad conf ((name_entity_match)");
		
		if (!merged) {
		    if (rest_S && entity_type) {
			if (tagfilter_is_printableln(filename,lineno,rest_S))
			    lib_error(CATGETS(elm_msg_cat, MeSet, 
					      MeEntityDuplicate,
					      "%s: %d: Duplicate entity %S with type %S - already given on line %d: %S"),
				      filename,lineno,
				      named_reference,entity_type,
				      entity_item->entities[newpos].result->lineno,
				      rest_S);
		    }
		    (*errors) ++;
		}

		entity_item->entities[newpos].result->lineno = lineno;

		if (entity_item->entities[newpos].result->entity_value)
		    free_string(& (entity_item->entities[newpos].result->
				   entity_value));
		if (entity_value)
		    entity_item->entities[newpos].result->entity_value =
			dup_string(entity_value);
		
		entity_item->entities[newpos].result->unicode_value = unicode_value;
		entity_item->entities[newpos].result->flags = ne_match_flags;
		
	    } else
		entity_item->entities[newpos].result =
		    malloc_name_entity_match(entity_value,unicode_value,
					     conf,lineno,ne_match_flags);

	    if (conf->linked)
		link_name_entity_match(entity_item->entities[newpos].result,
				       entity_item->entities[newpos].entity_type,
				       entity_item->reference_key,
				       errors);
	    
	    free_entity_sort_item(&res);  /* Decrement refcount */
	} else {
	    DPRINT(Debug,14,(&Debug,
			     "set_tagfilter_entity: search_sort_list_item did not set entity\n"));
	    r = 0;
	}

    } else {
	DPRINT(Debug,14,(&Debug,
			 "set_tagfilter_entity: search_sort_list_item  failed to create entry\n"));
	
	r = 0;
    }
		    
 fail2:
    if (search_key)
	free_string_sort(&search_key);	    

		
    return r;
}




int  change_tagfilter_entities(conf,new,rc,filename,errors)
     struct tagfilter_entities_conf **conf;
     struct tagfilter_entities_conf * new;
     enum record_mode rc  /* SYSTEM_RC = 0, 
			     LOCAL_RC = 1 */;
     const char        * filename;
     int               * errors;
{
    int ret = 1;
    
    if (*conf) {

	if (TAGFILTER_ENTITIES_CONF_magic != (*conf)->magic)
	    mime_panic(__FILE__,__LINE__,"change_tagfilter_entities",
		       "Bad magic number (tagfilter_entities_conf, *conf)");

    }

    if (TAGFILTER_ENTITIES_CONF_magic != new->magic)
	mime_panic(__FILE__,__LINE__,"change_tagfilter_entities",
		   "Bad magic number (tagfilter_entities_conf, new");

    if (new->root) {
	size_t count = sort_list_len(new->root);
	size_t i;
	size_t new_idx = 0;
	int append_need_rewrite = 0;
	
	if (!*conf) {

	    DPRINT(Debug,14,(&Debug,
			     "change_tagfilter_entities: Creating config,  count=%zu entities for %s\n",
			     count,filename));

	    *conf = malloc_tagfilter_entities_conf(rc,filename,count);
	} else {
	    
	    if ((*conf)->root) {
		size_t old_count = sort_list_len((*conf)->root);

		DPRINT(Debug,14,(&Debug,
				 "change_tagfilter_entities: %zu entires already on %s\n",
				 old_count,(*conf)->filename));
	    } else {
		(*conf)->root = alloc_sort_list(&entity_map_operation,count);
	    }
	    
	    (*conf)->rc = rc;
	    if ((*conf)->filename != filename) {
		DPRINT(Debug,14,(&Debug,
				 "change_tagfilter_entities: filename %s => %s\n",
				 (*conf)->filename,filename));
		(*conf)->filename = strmcpy((*conf)->filename,filename);
	    }
	}
	              	
	/* merge request from elmtagfilterents */

	for (i = 0; i < count; i++) {
		
	    union sort_item res;
	    
	    res.entity = NULL;
	    
	    /* Increments refcount */
	    get_sort_list_item(new->root,sort_list_get_normal,i,&res);
	    
	    if (res.entity) {
		struct entity_sort_item * entity_item = res.entity;
		
		if (ENTITY_SORT_ITEM_magic != entity_item->magic)
		    mime_panic(__FILE__,__LINE__,"change_tagfilter_entities",
			       "Bad magic number (entity_sort_item)");

		DPRINT(Debug,14,(&Debug,"change_tagfilter_entities: entity %zu = \"%S\"\n",
				 i,entity_item->reference_key));


		if (entity_item->entities) {
		    size_t j;
		    
		    for (j = 0; j < entity_item->entities_count; j++) {
			const char * type = "???";
			    
			if (! entity_item->entities[j].entity_type ||
			    ! entity_item->entities[j].result)
			    continue;
			
			if (TAGFILTER_ENTITIES_magic !=
			    entity_item->entities[j].entity_type->magic)
			    mime_panic(__FILE__,__LINE__,"change_tagfilter_entities",
				       "Bad magic number (tagfilter_entities)");
			type = entity_item->entities[j].entity_type->type_tag;

			DPRINT(Debug,14,(&Debug,"                       : %zu %s\n", 
					 j,type));
			
			if (NAME_ENTITY_MATCH_magic != entity_item->entities[j].result->magic)
			    mime_panic(__FILE__,__LINE__,"change_tagfilter_entities",
				       "Bad magic number (name_entity_match)");
			
			


			if (set_tagfilter_entity(filename,*conf,entity_item->reference_key,
						 entity_item->entities[j].entity_type,
						 entity_item->entities[j].result->entity_value,
						 entity_item->entities[j].result->unicode_value,

						 /* Not really correct */
						 entity_item->entities[j].result->lineno,

						 entity_item->entities[j].result->flags,
						 merge_entities,
						 &new_idx,
						 &append_need_rewrite,
						 errors,NULL,NULL,0
						 )) {
			    
			    DPRINT(Debug,14,(&Debug,"                       - %zu %s => %zu\n",
					     j,type,new_idx));

			} else {
			    DPRINT(Debug,14,(&Debug,"                       - %zu %s => failed\n",
					     j,type));
			    
			    ret = 0;
			}			
		    }
		}
			    		
		free_entity_sort_item(&res);
	    }
	}
    }

    return ret;
}

int merge_tagfilter_entities(filename,f,conf,errors,fileset,propline,rc,need_rewrite)
     const char                     * filename;
     FILE                           * f;
     struct tagfilter_entities_conf * conf;
     int *errors;
     charset_t *fileset;
     struct editor_propline **propline;
     enum record_mode rc  /* SYSTEM_RC = 0, 
			     LOCAL_RC = 1 */;
     int *need_rewrite;
{
    int r;

    r = read_tagfilter_entities(filename,f,&conf,errors,fileset,propline,
				rc, need_rewrite);
    
    return r;
}

static int read_tagfilter_entities(filename,f,conf,errors,fileset,propline,rc,need_rewrite)
     const char                      * filename;
     FILE                            * f;
     struct tagfilter_entities_conf ** conf;
     int                             * errors;
     charset_t                       * fileset;
     struct editor_propline         ** propline;
     enum record_mode                  rc  /* SYSTEM_RC = 0, 
					      LOCAL_RC = 1 */;     
     int                             * need_rewrite;
{
    charset_t     cs = system_charset;
    const char * csn = get_charset_MIME_name(cs);

    size_t res_idx          = 0;   /* Append mode position */
    char * buf = NULL;
    int c;
    size_t count = 0;
    int last_c = '\n';
    int lineno = 0;

    int append_need_rewrite = 0;

    int fail_it             = 0;
    size_t pos = 0;
    enum syscall_status r;

    int ret_status = 0;
    int set_old = 0;  

    if (fileset && *fileset) {
	    csn = get_charset_MIME_name(*fileset);
	
	cs = *fileset;
	
	if (csn) {
	    DPRINT(Debug,2,(&Debug, 
			    "read_tagfilter_entities: %s: default charset: %s\n",
			    filename,csn));
	}
    }

    /* Look for editor property line */

    if (propline)  {
	enum editor_propline_v propline_mode = 
	    give_dt_enumerate_as_int(&editor_tfent_propline);
	
	/* Nothing detected yet */
	
	if (*propline)
	    free_editor_propline(propline);

	if (propline_mode != editor_propline_ignore) {
	    int i;
	    
	    for (i = 0; i < 3 && !*propline; i++) {
		int l1;

		c = fgetc(f);

		if (EOF == c)
		    goto propline_failure;
		if ('\n' == c)
		    continue;

		ungetc(c,f);

		if ('#' != c)
		    break;

		l1 = malloc_gets(&buf,LONG_STRING,f);
		if (l1 > 1) {

		    if ('#' != buf[0])
			goto propline_failure;
		    
		    *propline =
			detect_editor_propline(propline_mode,
					       buf,l1,filename,i,&cs);
	
		    csn = get_charset_MIME_name(cs);
				       
		} else { /* something wrong? */
		propline_failure:

		    rewind(f);
		    break;
		}
	    }
	}

	pos = ftell(f);
    }


    if (*conf) {

	if (TAGFILTER_ENTITIES_CONF_magic != (*conf)->magic)
	    mime_panic(__FILE__,__LINE__,"read_tagfilter_entities",
		       "Bad magic number (tagfilter_entities_conf)");
	
	if ((*conf)->root) {
	    size_t map_len = sort_list_len((*conf)->root);
	    size_t i;
	    size_t old_count = 0;
		
	    DPRINT(Debug,14,(&Debug,
			     "read_tagfilter_entities: %zu items on tagfilter_entities_conf already\n",
			     map_len));

	    for (i = 0; i < map_len; i++) {
		
		union sort_item res;
		
		res.entity = NULL;
		
		/* Increments refcount */
		get_sort_list_item((*conf)->root,sort_list_get_normal,i,&res);
		
		if (res.entity) {
		    struct entity_sort_item * entity_item = res.entity;

		    if (ENTITY_SORT_ITEM_magic != entity_item->magic)
			mime_panic(__FILE__,__LINE__,"read_tagfilter_entities",
				   "Bad magic number (entity_sort_item)");
		    
		    DPRINT(Debug,14,(&Debug,"read_tagfilter_entities: entity %zu = \"%S\"\n",
				     i,entity_item->reference_key));
		    
		    if (entity_item->entities) {
			size_t j;
			
			for (j = 0; j < entity_item->entities_count; j++) {
			    const char * type = "???";
			    
			    if (entity_item->entities[j].entity_type) {
				if (TAGFILTER_ENTITIES_magic !=
				    entity_item->entities[j].entity_type->magic)
				    mime_panic(__FILE__,__LINE__,"read_tagfilter_entities",
					       "Bad magic number (tagfilter_entities)");
				type = entity_item->entities[j].entity_type->type_tag;
			    }
			    
			    DPRINT(Debug,14,(&Debug,"                       : %zu %s", 
					     j,type));
			    if (entity_item->entities[j].delete) {
				DPRINT(Debug,14,(&Debug,", marked for delete"));
			    } else if (entity_item->entities[j].old) {
				entity_item->entities[j].need_on_disk = 1;
				DPRINT(Debug,14,(&Debug,", old -- marked need_on_disk"));
				old_count++;				    
			    } else if (entity_item->entities[j].need_on_disk) {
				entity_item->entities[j].need_on_disk = 0;
				DPRINT(Debug,14,(&Debug,", clearing need_on_disk"));
			    }				
			    DPRINT(Debug,14,(&Debug,"\n"));
			    
			}			
		    }

		    free_entity_sort_item(&res);
		}
		
		if (old_count) {
		    DPRINT(Debug,14,(&Debug,
				     "read_tagfilter_entities:  %zu old entities marked need_on_disk\n",
				     old_count));
		}		
	    }	    
	}
    } else {
	set_old = 1;    /* When later merged, need found from disk */
    }

    if (!*conf || ! ((*conf)->root)) {
    	    
	DPRINT(Debug,14,(&Debug,
			 "read_tagfilter_entities: counting entity lines from offset %ld at %s\n",
			 pos,filename));
	
	while(EOF != (c = fgetc(f))) {
	    /* Rough guess about &entity lines */
	    
	    if (last_c == '\n' &&
		c == '&') 
		count++;
	    last_c = c;
	}

	r = fseek(f,pos,SEEK_SET);
	switch (r) {
	    int err;
	case syscall_success:
	    DPRINT(Debug,14,(&Debug,
			     "read_tagfilter_entities: Reading again from offset %ld at %s\n",
			     pos,filename));
	    break;
	case syscall_error:
	    err = errno;
	    DPRINT(Debug,14,(&Debug,
			     "read_tagfilter_entities: Failed to seek to offset %ld at %s: %s\n",
			     pos,filename,strerror(err)));
	    	    
	    return 0;
	}
	

	if (!*conf) {

	    DPRINT(Debug,14,(&Debug,
			     "read_tagfilter_entities: Creating config,  count=%zu entity lines on %s\n",
			     count,filename));
	    
	    *conf = malloc_tagfilter_entities_conf(rc,filename,count);
	} else if (! ((*conf)->root)) {
	    (*conf)->root = alloc_sort_list(&entity_map_operation,count);
	}
    }

    while(!feof(f) && !ferror(f) && !fail_it) {
	int l1 = malloc_gets(&buf,LONG_STRING,f);
	struct string * rest_S = NULL;
	char * cx = buf;
	int restlen = 0;
	int ERRORS = 0;
	int rs;
	int L;

	if (-1 == l1) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeTooLongLineNo,
			      "%s: %d: Too long line: %.30s..."),
		      filename,lineno+1,buf);
	    (*errors) ++;
	    
	    DPRINT(Debug,9,(&Debug, 
			    "read_tagfilter_entities: %s: %d: failed to parse line, too long line\n",
			    filename,lineno+1));


	    break;
	} else if (l1 < 0) {
	    DPRINT(Debug,9,(&Debug, 
			    "read_tagfilter_entities: %s: %d: read error or EOF\n",
			    filename,lineno+1));
	    break;
	}

	lineno++;

	if (l1 == 0)
	    continue;

	l1 = trim_whitespace_from_end(buf,l1);

	if (read_charset_tag(buf,l1,
			     filename,lineno,
			     errors,&cs,&csn))
	     continue;
			    
	cx = buf;
	while (*cx && whitespace (*cx)) /* skip leading whitespace */
	    cx++;
	if ('#' == *cx)
	    continue;

	if (!*cx) {
		    
	    if (cx < buf + l1) {
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeNULcharacterOnLine,
				  "%s: %d: NUL character on line."),
			  filename,lineno);
		(*errors) ++;

		DPRINT(Debug,9,(&Debug, 
				"read_tagfilter_entities: %s: %d: failed to parse line (NUL character)\n",
				filename,lineno));
		
		
	    }

	}

	rest_S = new_string(cs);
	restlen = (buf + l1) - cx;
	rs =  add_streambytes_to_string(rest_S,restlen,
					cs2us(cx),&ERRORS);
	if (rs < restlen) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailedAsCharset,
			      "Failed to parse line %d as charset %s in \"%s\" file"),
		      lineno, csn ? csn : "<no MIME name>",filename);

	    (*errors) ++;

	    DPRINT(Debug,9,(&Debug, 
			    "read_tagfilter_entities: %s: %d: failed to parse line\n",
			    filename,lineno));

	    goto fail;
	}

	if (ERRORS) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFileAsCharsetErrors,
			      "There is %d errors when parsing line %d as charset %s in \"%s\" file"),
		      ERRORS,lineno,
		      csn ? csn : "<no MIME name>",filename);

	    (*errors) ++;

	    DPRINT(Debug,9,(&Debug, 
			    "read_tagfilter_entities: %s: %d: failed to parse line (%d character errors)\n",
			    filename,lineno, ERRORS));


	    goto fail;
	}

	L = string_len(rest_S);
	if (L > 0) {
	    uint16 code = give_unicode_from_string(rest_S,0);
	    struct string             * named_reference = NULL;
	    struct string             * entity_type     = NULL;
	    struct string             * entity_value    = NULL;
	    	    
	    if (0x0026 /* &  */ == code) {

		int x;
		int seen_semicolon = 0;
		int X = 0;
		int rs;
		uint16 found_delim = 0;
				
		enum value_mode {
		    null_value = 0,
		    quoted_string,
		    octal_value      = 8,
		    decimal_value    = 10,
		    hexdecimal_value = 16,

		} mode = null_value;   
		uint32 value = 0;

		int start_x = 0;
		int end_x   = 0;

		size_t y;

		struct tagfilter_entities * enttyp = NULL;
		int ne_match_flags;
		

		
		for (x = 1; x < L; x++) {
		    code = give_unicode_from_string(rest_S,x);

		    
		    if (0x003B /* ; */ == code) {
			seen_semicolon = 1;
			x++;
			break;
		    } else if (0x0009 /* HT  '\t' (horizontal tab) */ == code ||
			       0x0020 /* SPACE */                     == code) {
			break;
		    } else  if (tagfilter_entity_character(code,NULL)) {
			/* OK */
		    } else {
			goto badline;
		    }
		}

		if (x < 2 || ( seen_semicolon && x < 3)) {
		    if (tagfilter_is_printableln(filename,lineno,rest_S))
			lib_error(CATGETS(elm_msg_cat, MeSet, 
					  MeEntityNotGiven,
					  "%s: %d: Entity name not given: %S"),
				  filename,lineno,rest_S);
		    (*errors) ++;
		    goto fail2;
		}

		named_reference = clip_from_string(rest_S,&X,x);

		rs = get_word_from_string(rest_S,&entity_type,&X,
					  GWF_lowercase,cs2us(" \t"),
					  &found_delim);

		if (rs < 1 || !entity_type || string_len(entity_type) < 1) {
		    if (tagfilter_is_printableln(filename,lineno,rest_S))
			lib_error(CATGETS(elm_msg_cat, MeSet, 
					  MeEntityTypeNotGiven,
					  "%s: %d: Entity type not given: %S"),
				  filename,lineno,rest_S);
		    (*errors) ++;
		    goto fail2;
		}

		for (x = X; x < L; x++) {
		    code = give_unicode_from_string(rest_S,x);

		    if (0x0031 /* 1 */ <= code && code <= 0x0039 /* 9 */) {
			mode = decimal_value;
			break;
		    } else if (0x0078 /* x */ == code ||
			       0x0058 /* X */ == code) {
			x++;
			mode = hexdecimal_value;
			break;
		    } else if (0x0022 /* " */ == code) {
			x++;
			mode = quoted_string;
			break;
		    } else if (0x0030 /* 0 */ == code) {
			mode = octal_value;
			break;
		    } else if (0x0009 /* HT  '\t' (horizontal tab) */ == code ||
			       0x0020 /* SPACE */                     == code) {
			/* Skip */
		    } else {
			if (tagfilter_is_printableln(filename,lineno,rest_S))
			    lib_error(CATGETS(elm_msg_cat, MeSet, 
					      MeEntityValNotOK,
					      "%s: %d: Entity value is not number or quoted string: %S"),
				      filename,lineno,rest_S);
			(*errors) ++;
			goto fail2;

		    }
		}

		switch (mode) {
		case null_value: goto badline;
		case quoted_string:
		    start_x = x;
		    
		    for (x = start_x; x < L; x++) {
			code = give_unicode_from_string(rest_S,x);

			if (0x0022 /* " */ == code) {
			    end_x = x;
			    x++;
			    break;
			} else if (0x005C  /*  \  */ == code) {
			    if (tagfilter_is_printableln(filename,lineno,rest_S))
				lib_error(CATGETS(elm_msg_cat, MeSet, 
						  MeEntityValNotBackSlash,
						  "%s: %d: Entity value does not allow backslash (\\): %S"),
					  filename,lineno,rest_S);
			    (*errors) ++;
			    goto fail2;		  				       
			}
		    }

		    if (end_x < start_x) {
			if (tagfilter_is_printableln(filename,lineno,rest_S))
			    lib_error(CATGETS(elm_msg_cat, MeSet, 
					      MeEntityValNotQuoteEnd,
					      "%s: %d: Entity value does not does not end with quote (\"): %S"),
				      filename,lineno,rest_S);
			(*errors) ++;
			goto fail2;		  				       
		    }	      

		    X = start_x;
		    entity_value = clip_from_string(rest_S,&X,end_x-start_x);
		    X++;
		    
		    break;
		case octal_value:
		case decimal_value:
		case hexdecimal_value:
		    start_x = x;
		    
		    for (x = start_x; x < L; x++) {
			unsigned char v = 0;
			
			code = give_unicode_from_string(rest_S,x);

	

			if (0x0030 /* 0 */ <= code && code <= 0x0039 /* 9 */) {
			    v = code - 0x0030 /* 0 */;
			} else if (0x0041 /* A */ <= code && code <= 0x005A /* Z */) {
			    v = code - 0x0041 /* A */ + 10;
			} else if (0x0061 /* a */ <= code && code <= 0x007A /* z */) {
			    v = code - 0x0061 /* a */ + 10;

			} else if (0x0009 /* HT  '\t' (horizontal tab) */ == code ||
				   0x0020 /* SPACE */                     == code) {

			    break;
			    
			} else {
			    if (tagfilter_is_printableln(filename,lineno,rest_S))
				lib_error(CATGETS(elm_msg_cat, MeSet, 
						  MeEntityValNotValidNumber,
						  "%s: %d: Entity value is not valid number: %S"),
					  filename,lineno,rest_S);
			    (*errors) ++;
			    goto fail2;		  				       
			}	      
			
			if (v >= mode) {
			    if (tagfilter_is_printableln(filename,lineno,rest_S))
				lib_error(CATGETS(elm_msg_cat, MeSet, 
						  MeEntityValNotValidNumber,
						  "%s: %d: Entity value is not valid number: %S"),
					  filename,lineno,rest_S);
			    (*errors) ++;
			    goto fail2;
			}
			
			value *= mode;
			value += v;

			if (value > 0xFFFF) {
			    lib_error(CATGETS(elm_msg_cat, MeSet, 
					      MeEntityValTooBig,
					      "%s: %d: Entity value is too big: %S"),
				      filename,lineno,rest_S);
			    (*errors) ++;
			    goto fail2;
			    
			}
		    }

		    X = x;
		    break;
		    
		}

		for (x = X; x < L; x++) {
		    code = give_unicode_from_string(rest_S,x);

		    if (0x0009 /* HT  '\t' (horizontal tab) */ == code ||
			0x0020 /* SPACE */                     == code) {
			
			/* OK */
		    } else if (0x0023 /* #  */ == code) {
			break;
		    } else {
			if (tagfilter_is_printableln(filename,lineno,rest_S))
			    lib_error(CATGETS(elm_msg_cat, MeSet, 
					      MeEntityValTarilinData,
					      "%s: %d: Entity value have trailing data: %S"),
				      filename,lineno,rest_S);
			(*errors) ++;
			goto fail2;
			
		    }
		}

		for (y = 0; y < entity_types_count; y++) {
		    if (TAGFILTER_ENTITIES_magic != entity_types[y]->magic)
			mime_panic(__FILE__,__LINE__,"read_tagfilter_entities",
				   "Bad magic number (tagfilter_entities)");
		    
		    if (string_matches_ascii(entity_type,
					     cs2us(entity_types[y]->type_tag),0,
					     SMA_op_normal)) {
			enttyp = entity_types[y];
			break;
		    }
		}

		if (!enttyp) {
		    if (tagfilter_is_printableln(filename,lineno,rest_S))
			    lib_error(CATGETS(elm_msg_cat, MeSet, 
					      MeEntityTypeNotSup,
					      "%s: %d: Entity type %S not supported: %S"),
				      filename,lineno,entity_type,rest_S);
		    (*errors) ++;
		    goto fail2;		    
		}

		if (mode < octal_value && !value)
		    value = UNICODE_BAD_CHAR;  /* REPLACEMENT CHARACTER */

		
		ne_match_flags =
		    seen_semicolon ?  ne_match_SEMICOLON : 0;
					
		if (set_tagfilter_entity(filename,*conf,
					 named_reference,enttyp, entity_value, value,
					 lineno,ne_match_flags,read_entities,
					 &res_idx,&append_need_rewrite,
					 errors,rest_S,entity_type,set_old
					 )) {

		    /* Nothing */

		} else {
		    DPRINT(Debug,14,(&Debug,
				     "read_tagfilter_entities: set_tagfilter_entity failed\n"));
		    fail_it = 1;
		}

		if (append_need_rewrite && need_rewrite && !*need_rewrite) {
		    DPRINT(Debug,14,(&Debug,
				     "read_tagfilter_entities: Setting *need_rewrite\n"));
		    *need_rewrite = 1;
		}

	    } else {
	    badline:

		
		if (tagfilter_is_printableln(filename,lineno,rest_S))
		    lib_error(CATGETS(elm_msg_cat, MeSet, 
				      MeUnsupportedLineS,
				      "%s: %d: Unsupported line: %S"),
			      filename,lineno,rest_S);
		(*errors) ++;
		
		DPRINT(Debug,9,(&Debug, 
				"read_tagfilter_entities: %s: %d: failed to parse line\n",
				filename,lineno));
	    }

	fail2:
	    
	    if (entity_value)
		free_string(&entity_value);
	    if (named_reference)
		free_string(&named_reference);
	    if (entity_type)
		free_string(&entity_type);	    
	}

    fail:

	free_string(&rest_S);
    }

    if (buf)
	free(buf);

    if (fail_it)
	ret_status = 0;
    else if (feof(f) && (*conf)->root) {
	size_t i;
	size_t map_len =   sort_list_len((*conf)->root);


	for (i = 0; i < map_len; i++) {
	    
	    union sort_item res;
	    
	    res.entity = NULL;
	    
	    /* Increments refcount */
	    get_sort_list_item((*conf)->root,sort_list_get_normal,i,&res);
	    
	    if (res.entity) {
		struct entity_sort_item * entity_item = res.entity;
		
		if (ENTITY_SORT_ITEM_magic != entity_item->magic)
		    mime_panic(__FILE__,__LINE__,"read_tagfilter_entities",
			       "Bad magic number (entity_sort_item)");

		DPRINT(Debug,14,(&Debug,"read_tagfilter_entities: entity %zu = \"%S\"\n",
				 i,res.entity->reference_key));

		
		if (res.entity->entities) {
		    size_t j;
		    
		    for (j = 0; j < res.entity->entities_count; j++) {
			const char * type = "???";
			
			if (res.entity->entities[j].entity_type) {
			    if (TAGFILTER_ENTITIES_magic !=
				res.entity->entities[j].entity_type->magic)
				mime_panic(__FILE__,__LINE__,"read_tagfilter_entities",
					   "Bad magic number (tagfilter_entities)");
			    type = res.entity->entities[j].entity_type->type_tag;
			}
			
			DPRINT(Debug,14,(&Debug,"                       : %zu %s", 
					 j,type));
			
			if (res.entity->entities[j].old) {
			    DPRINT(Debug,14,(&Debug,", old"));
			}
			
			if (res.entity->entities[j].delete) {
			    DPRINT(Debug,14,(&Debug,", marked for delete"));
			}
			
			if (res.entity->entities[j].need_on_disk) {
			    DPRINT(Debug,14,(&Debug,", not seen on disk"));
			    
			    if (! res.entity->entities[j].need_on_disk) {
				DPRINT(Debug,14,(&Debug," (marking for delete)"));
				
				res.entity->entities[j].delete = 1;
			    }
			}
			
			DPRINT(Debug,14,(&Debug,"\n"));
		    }
		}

		free_entity_sort_item(&res);		
	    }
	}
	    
	ret_status = 1;
    }
    
    return ret_status;
}


struct tagfilter_entities_conf * parse_tagfilter_entities(filename,errors,fileset,propline,rc,
							  need_rewrite
							  )
     const char *filename;
     int *errors;
     charset_t *fileset;
     struct editor_propline **propline;
     enum record_mode rc  /* SYSTEM_RC = 0, 
			     LOCAL_RC = 1 */;
     int *need_rewrite;
{
    struct tagfilter_entities_conf * ret = NULL;

    int err = can_open(filename,"r");
    FILE * f;
   
    if (err) {
	DPRINT(Debug,2,(&Debug, 
			"parse_tagfilter_entities=NULL: %s: %s (can_open)\n",
			filename,strerror(err)));
	return NULL;
    }

    f = fopen(filename,"r");
    if (!f) {
	err = errno;
	DPRINT(Debug,2,(&Debug, 
			"parse_tagfilter_entities=NULL: %s: %s\n",
			filename,strerror(err)));
	return NULL;
    }

    if (read_tagfilter_entities(filename,f,&ret,
				errors,fileset,propline,rc,need_rewrite)) {
	fclose(f);

	return ret;
    }

    if (ret)
	free_tagfilter_entities_conf(&ret);
        
    return ret;
}

static  struct tagfilter_entities_conf * malloc_tagfilter_entities_conf(rc,filename,
									alloc_size)
    enum record_mode    rc;
    const char        * filename;
    size_t alloc_size;
{
    struct tagfilter_entities_conf * ret = safe_zero_alloc(sizeof(*ret));

    ret->magic = TAGFILTER_ENTITIES_CONF_magic;
    ret->rc = rc;

    ret->filename = NULL;
    if (filename)
	ret->filename = safe_strdup(filename);
    
    ret->root = alloc_sort_list(&entity_map_operation,alloc_size);

    ret->linked  = 0;
    
    return ret;
}

void free_tagfilter_entities_conf(ptr)
     struct tagfilter_entities_conf **ptr;
{
    if (TAGFILTER_ENTITIES_CONF_magic != (*ptr)->magic)
	mime_panic(__FILE__,__LINE__,"free_tagfilter_entities_conf",
		   "Bad magic number (tagfilter_entities_conf)");

    if ((*ptr)->filename) {
	free((*ptr)->filename);
	(*ptr)->filename = NULL;
    }

    if ((*ptr)->root)
	free_sort_list(& ((*ptr)->root));
    
    (*ptr)->magic = 0; /* Invalidate */
    free(*ptr);
    *ptr = NULL;
}


struct name_entity_walk * tagfilter_start_reference(ents)
    struct tagfilter_entities *ents;
{
    struct name_entity_walk * ret;
    
    if (TAGFILTER_ENTITIES_magic != ents->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_start_reference",
                   "Bad magic number");

    if (! ents->builtin_generated) {
	DPRINT(Debug,14,(&Debug,
			 "tagfilter_start_reference: Generating builtin defaults for %s\n",
			 ents->type_tag));

	tagfilter_generate_defaults(ents);
    }
    
    ret = ents->ampersand;   /* Parsing ampersand */
    if (ret)
	inc_name_entity_walk_refcount(ret);
    else {
	/* Make dummy if not tree */
	ret = malloc_name_entity_walk(ents,0x0026 /* & */);
	
	DPRINT(Debug,14,(&Debug,
			 "tagfilter_start_reference: No root record for %s - generated dummy\n",
			 ents->type_tag));
    }
	
    return ret;
}

static struct name_entity_walk * malloc_name_entity_walk(ents,code)
     struct tagfilter_entities * ents;
     uint16 code;
{
    struct name_entity_walk * ret;
    
    if (TAGFILTER_ENTITIES_magic != ents->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_start_reference",
                   "Bad magic number");

    ret = safe_zero_alloc(sizeof(*ret));

    ret->refcount = 1;
    ret->magic    = NAME_ENTITY_WALK_magic;
    ret->ents     = ents;

    ret->code = code;
    ret->alternatives       = NULL;
    ret->alternatives_count = 0;

    ret->results            = NULL;
    ret->results_count      = 0;

    return ret;
}

static void inc_name_entity_walk_refcount(ptr) 
     struct name_entity_walk *ptr;
{
    if (NAME_ENTITY_WALK_magic != ptr->magic)
	mime_panic(__FILE__,__LINE__,"inc_name_entity_walk_refcount",
                   "Bad magic number");

    ptr->refcount++;
}

void free_name_entity_walk(ptr) 
    struct name_entity_walk **ptr;
{
    if (NAME_ENTITY_WALK_magic != (*ptr)->magic)
	mime_panic(__FILE__,__LINE__,"free_name_entity_walk",
                   "Bad magic number");

    if ((*ptr)->refcount < 1)
	mime_panic(__FILE__,__LINE__,"free_name_entity_walk",
		   "Bad refcount");

    (*ptr)->refcount--;
    if ((*ptr)->refcount > 0) {
	*ptr = NULL;
	return;
    }
    
    (*ptr)->ents = NULL;   /* struct tagfilter_entities is static, not dynamically
			      alloced
			   */

    if ((*ptr)->alternatives) {
	size_t x;

	/* Remove recursively */
	
	for (x = 0; x < (*ptr)->alternatives_count; x++) {

	    if ((*ptr)->alternatives[x])
		free_name_entity_walk(& ((*ptr)->alternatives[x]));	    
	}

	free((*ptr)->alternatives);
	(*ptr)->alternatives = NULL;
    }
    (*ptr)->alternatives_count = 0;


    if ((*ptr)->results) {
	size_t x;

	for (x = 0; x < (*ptr)->results_count; x++) {

	    if ((*ptr)->results[x]) {
		if (NAME_ENTITY_MATCH_magic != (*ptr)->results[x]->magic)
		    mime_panic(__FILE__,__LINE__,"free_name_entity_walk",
			       "Bad magic number (name_entity_match)");

		if ((*ptr)->results[x]->linked_walk != (*ptr))
		    mime_panic(__FILE__,__LINE__,"free_name_entity_walk",
			       "Bad linked walk (name_entity_match)");

		(*ptr)->results[x]->linked_walk = NULL;

		free_name_entity_match(& ((*ptr)->results[x]));
	    }
	}
	
	free((*ptr)->results);
	(*ptr)->results = NULL;
    }
    (*ptr)->results_count = 0;


    (*ptr)->magic = 0; /* Invalidate */
    free (*ptr);
    *ptr = NULL;
 }

/*  free'es struct name_entity_walk, if no match */
void advance_entity_walk(walk,u)
     struct name_entity_walk **walk;
     uint16 u;
{
    struct name_entity_walk * res = NULL;
    
    if (NAME_ENTITY_WALK_magic != (*walk)->magic)
	mime_panic(__FILE__,__LINE__,"advance_entity_walk",
                   "Bad magic number");

    res = advance_walk(*walk,u,walk_search,(*walk)->ents);
    
    free_name_entity_walk(walk);
    
    *walk = res;
}

static struct name_entity_match  * malloc_name_entity_match(entity_value,
							    unicode_value,
							    conf,lineno,
							    ne_match_flags)
     struct string                  * entity_value;
     uint16                           unicode_value;
     struct tagfilter_entities_conf * conf;
     int                              lineno;
     unsigned                         ne_match_flags;
{
     struct name_entity_match  * ret = safe_zero_alloc(sizeof(*ret));

     ret->magic = NAME_ENTITY_MATCH_magic;
     ret->refcount = 1;

     ret->conf   = conf;
     ret->lineno = lineno;

     if (entity_value)
	 ret->entity_value = dup_string(entity_value);
     else
	 ret->entity_value = NULL;
     ret->unicode_value = unicode_value;
          
     ret->linked_walk = NULL;

     ret->flags = ne_match_flags;
     
     return ret;     
}

/* increment refcount */
static struct name_entity_match * tagfilter_match_ref_conf P_((struct name_entity_walk *walk,
							       enum tagfilter_match have_semicolon,
							       struct tagfilter_entities_conf * conf));

static struct name_entity_match * tagfilter_match_ref_conf(walk,have_semicolon,conf)
     struct name_entity_walk *walk;
     enum tagfilter_match have_semicolon;
     struct tagfilter_entities_conf * conf; /* 	NULL if builtin  */
{
    struct name_entity_match *ret = NULL;
    
     if (NAME_ENTITY_WALK_magic != walk->magic)
	 mime_panic(__FILE__,__LINE__,"tagfilter_match_ref_conf",
		    "Bad magic number");


    if (walk->results) {
	size_t i;

	for (i = 0; i < walk->results_count; i++) {

	    if (NAME_ENTITY_MATCH_magic != walk->results[i]->magic)
		mime_panic(__FILE__,__LINE__," tagfilter_match_reference",
			   "Bad magic number (name_entity_match)");

	    if (conf != walk->results[i]->conf)
		continue;
	    
	    if (ison(walk->results[i]->flags,ne_match_IGNORE)) {
		continue;
	    }
	    
	    switch (have_semicolon) {
	    case tagfilter_match_prefix:
		if (isoff(walk->results[i]->flags,ne_match_SEMICOLON)) {
		    ret = walk->results[i];
		    goto found;
		}
		break;
	    case  tagfilter_match_semicolon:
		if (ison(walk->results[i]->flags,ne_match_SEMICOLON)) {
		    ret = walk->results[i];
		    goto found;
		}
		break;
	    }
	}
    }


    if (ret) {
    found:
	inc_name_entity_match_refcount(ret);

    }

    return ret;
}


/* increment refcount */
struct name_entity_match * tagfilter_match_reference(walk,have_semicolon)
     struct name_entity_walk *walk;
     enum tagfilter_match have_semicolon;
{
    struct name_entity_match *ret = NULL;
    
     if (NAME_ENTITY_WALK_magic != walk->magic)
	 mime_panic(__FILE__,__LINE__,"tagfilter_match_reference",
		    "Bad magic number");

     if (!ret && user_tagfilter_entities_conf) {
	 if (TAGFILTER_ENTITIES_CONF_magic != user_tagfilter_entities_conf->magic)
	    mime_panic(__FILE__,__LINE__,"tagfilter_match_reference",
		       "Bad magic number (user_tagfilter_entities_conf)");

	 ret = tagfilter_match_ref_conf(walk, have_semicolon,user_tagfilter_entities_conf);

	 
	 if (ret) {
	    DPRINT(Debug,14,(&Debug,
			     "tagfilter_match_reference: Found user conf %s match\n",
			     user_tagfilter_entities_conf->filename));
	}
     }

     if (!ret && system_tagfilter_entities_conf) {
	 if (TAGFILTER_ENTITIES_CONF_magic != system_tagfilter_entities_conf->magic)
	    mime_panic(__FILE__,__LINE__,"tagfilter_match_reference",
		       "Bad magic number (system_tagfilter_entities_conf)");

	ret = tagfilter_match_ref_conf(walk, have_semicolon,system_tagfilter_entities_conf);

	if (ret) {
	    DPRINT(Debug,14,(&Debug,
			     "tagfilter_match_reference: Found system conf %s match\n",
			     system_tagfilter_entities_conf->filename));
	}	
     }
     
     if (!ret) {

	 ret = tagfilter_match_ref_conf(walk, have_semicolon,NULL);

	 if (ret) {
	     DPRINT(Debug,14,(&Debug,"tagfilter_match_reference: Found builtin match\n"));
	 }
	 
     }
     
     return ret;
}

static void inc_name_entity_match_refcount(ptr)
     struct name_entity_match *ptr;
{
    if (NAME_ENTITY_MATCH_magic != ptr->magic)
	mime_panic(__FILE__,__LINE__,"inc_name_entity_match_refcount",
		   "Bad magic number");

    ptr->refcount++;
}

void free_name_entity_match(ptr)
     struct name_entity_match **ptr;
{
    if (NAME_ENTITY_MATCH_magic != (*ptr)->magic)
	mime_panic(__FILE__,__LINE__,"free_name_entity_match",
		   "Bad magic number");
    
    if ((*ptr)->refcount < 1)
	mime_panic(__FILE__,__LINE__,"free_name_entity_match",
		   "Bad refcount");
    
    (*ptr)->refcount--;
    
    if ((*ptr)->refcount > 0) {
	*ptr = NULL;
	return;
    }
    
    if ((*ptr)->conf) {
	mime_panic(__FILE__,__LINE__,"free_name_entity_match",
		   "Backlink (conf) not cleared");
    }

    if ((*ptr)->linked_walk) {
	mime_panic(__FILE__,__LINE__,"free_name_entity_match",
		   "Backlink (linked_walk) not cleared");
    }
    
    if (ison((*ptr)->flags,ne_match_STATIC)) {
	/* Not malloced */
	
	*ptr = NULL;
	return;
    }
       
    if ((*ptr)->entity_value)
	free_string(&  ((*ptr)->entity_value));

    (*ptr)->unicode_value = 0;
    (*ptr)->flags         = 0;
    
    (*ptr)->magic = 0; /* Invalidate */
    free(*ptr);
    *ptr = NULL;
}

struct out_entity * out_entity_from_match(reference_key,
					  match,
					  reference_key_pg_flags)
     struct string            * reference_key;
     struct name_entity_match * match;
     int                       reference_key_pg_flags;
{
    if (NAME_ENTITY_MATCH_magic != match->magic)
	mime_panic(__FILE__,__LINE__,"out_entity_from_match",
		   "Bad magic number (name_entity_match)");

    DPRINT(Debug,22,(&Debug,
		     "out_entity_from_match: reference_key=%S",
		     reference_key));
    if (match->entity_value) {
	DPRINT(Debug,22,(&Debug,", entity_value %S",
			 match->entity_value));	
    }
    DPRINT(Debug,22,(&Debug,", unicode value x%04x",
		     match->unicode_value));
    switch(match->unicode_value) {
    case UNICODE_NO_BREAK_SPACE: DPRINT(Debug,22,(&Debug,
						   " UNICODE_NO_BREAK_SPACE"));
	break;
    case UNICODE_SOFT_HYPHEN: DPRINT(Debug,22,(&Debug,
					       " UNICODE_SOFT_HYPHEN"));
	break;
    case UNICODE_BAD_CHAR: DPRINT(Debug,22,(&Debug,
					    " UNICODE_BAD_CHAR"));
	break;	
    }
    DPRINT(Debug,22,(&Debug,"\n"));
    
    return  new_out_entity(reference_key,
			   match->entity_value,
			   match->unicode_value,
			   reference_key_pg_flags);
}



/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */

