static char rcsid[] = "@(#)$Id: tagfilter_tag.c,v 2.4 2024/10/13 10:57:10 hurtta Exp $";

/******************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.4 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************/

#include "def_melib.h"
#include "tagfilter_imp.h"

/* For INT_MAX */
#include <limits.h>


DEBUG_VAR(Debug,__FILE__,"mime");

#if ANSI_C
#define E_(x) extern x;
#define S_(x) static x;
#else
#define E_(x)
#define S_(x)
#endif

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

static const unsigned char *cs2us P_((const char *str));
static const unsigned char *cs2us(str)
     const char *str;
{
    return (const unsigned char *)str;
}



#include "elmme-magic.h"

#define TAGFILTER_TAGS_magic		0xFD0D

struct tagfilter_tags {
    unsigned short          magic;  /* TAGFILTER_TAGS_magic */

    const char            * context; /* for debug output */
    struct tagfilter_cmd ** tags;    /* NULL terminated      */
    struct sortlist       * sorted_tags;         
};

static struct tagfilter_stack_item * tagfilter_new_stack_item
P_((struct pager_range       * range     /* INC refcount */,
    int                        pg_flags,
    struct tagfilter_tags    * nested_tags  /* static - copies pointer */
                                            /* If NULL, current_tag->nesting is used */,
    struct tagfilter_cmd    * current_tag /* INC refcount */,
    union tagfilter_stack_uitem uitem
    ));
static struct tagfilter_stack_item * tagfilter_new_stack_item(range,
							      pg_flags,
							      nested_tags,
							      current_tag,
							      uitem)
     struct pager_range       * range     /* INC refcount */;
     int                        pg_flags;
     struct tagfilter_tags    * nested_tags      /* static - copies pointer */;
     struct tagfilter_cmd     * current_tag /* INC refcount */;
     union tagfilter_stack_uitem uitem;
{
     struct tagfilter_stack_item * ret = 
	 safe_zero_alloc(sizeof (*ret));
     
     ret->range = range;
     if (ret->range)
	 inc_pager_range_refcount(ret->range);
     ret->pg_flags = pg_flags;

     if (nested_tags) {
	 if (TAGFILTER_TAGS_magic != nested_tags->magic)
	     mime_panic(__FILE__,__LINE__,"tagfilter_new_stack_item",
			"Bad magic number (tagfilter_tags)");

	 
	 ret->nested_tags = nested_tags;    /* Assumed to be static */

     } else if (current_tag) {
	 if (TAGFILTER_CMD_magic != current_tag->magic)
	     mime_panic(__FILE__,__LINE__,"tagfilter_new_stack_item",
			"Bad magic number (tagfilter_cmd)");

	 ret->nested_tags = current_tag->nesting;
     } else
	 ret->nested_tags = NULL;

     
     ret-> current_tag =  current_tag;
     if (ret-> current_tag) 
	 tagfilter_inc_cmd_refcount(ret-> current_tag);
     /* refcount is incremented also for static commands */

     
     ret->reset_pg_flags = 0;
     ret->force_eoln     = 0;
     ret->text_visible   = 1;

     ret->u              = uitem;
     
     ret->magic   =  TAGFILTER_STACK_ITEM_magic;
     return ret;
}

enum tf_tag_sort_type {
    t_tagfilter_cmd   = TAGFILTER_CMD_magic,
    t_string          = STRING_magic,
    t_string_sort     = STRING_SORT_magic
    
};

#define TF_TAG_SORT_magic		0xFD15

struct tf_tag_sort {     /* union sort_key */
    unsigned short               magic;    /* TF_TAG_SORT_magic */

    struct tagfilter_tags      * backlink;
    struct tagfilter_selection * sel_backlink;
    
    union {

	/* Use unsigned short               magic
	   field as type indicator
	*/
	unsigned short        * type;  
	struct tagfilter_cmd  * u_tagfilter_cmd;
	struct string         * u_string;
	struct string_sort    * u_string_sort;
    } u;    
};

#define TF_CMD_DEF_magic		0xFD16

struct tf_cmd_def {
    unsigned short             magic; /* TF_CMD_DEF_magic */

    int                        pg_flags;
    enum  tagfilter_cmd_base   code;
    struct tagfilter_tags    * nesting; /* static always */

    unsigned  int   cmd_mode  : TAGFLT_CMD_bit_count;
} NULL_tf_cmd_def = {
    TF_CMD_DEF_magic,
    0,
    tfc_none,
    NULL,
    0
};

struct tagfilter_cmd * tagfilter_cmd_from_string(text,base_code,cmd_mode_flags)
     struct string * text;
     enum  tagfilter_cmd_base base_code;
     int cmd_mode_flags;
{
    struct tagfilter_cmd * ret =  safe_zero_alloc(sizeof (*ret));

    ret->magic = TAGFILTER_CMD_magic;
    ret->refcount = 1;

    ret->ascii_command = NULL;

    ret->pg_flags  = 0;
    ret->code.base = base_code;
    ret->nesting   = NULL;
    
    ret->cmd_mode = cmd_mode_flags;

    /* Increments refcount and shares string! */
    ret->string_command = new_string_sort(text);

    return ret;
}

S_(alloc_sort_item_f alloc_tf_cmd_sort_item)
static void alloc_tf_cmd_sort_item P_((union sort_item      * res,
                                       const union sort_key   key,
                                       const union sort_item_default def
                                       ));
static void alloc_tf_cmd_sort_item(res,key,def)
     union sort_item      * res;
     const union sort_key   key;
     const union sort_item_default def;  
{
    struct tf_tag_sort       * tf_key = key.tf_tag_sort;
    enum tf_tag_sort_type      tf_key_type;
    const struct tf_cmd_def  * tf_cmd_def = def.tf_cmd;
    
    if (TF_TAG_SORT_magic != tf_key->magic)
	mime_panic(__FILE__,__LINE__,"alloc_tf_cmd_sort_item",
		   "Bad magic number (tf_tag_sort)");

    if (tf_cmd_def) {

	if (TF_CMD_DEF_magic != tf_cmd_def->magic)
	    mime_panic(__FILE__,__LINE__,"alloc_tf_cmd_sort_item",
		       "Bad magic number (tf_cmd_def)");

    }

    
    tf_key_type = * (tf_key->u.type);
    res->dummy  = NULL;
    
    switch (tf_key_type) {
    case t_tagfilter_cmd:
	res->tf_cmd = tf_key->u.u_tagfilter_cmd;
	tagfilter_inc_cmd_refcount(res->tf_cmd);
	break;
    case t_string:

	res->tf_cmd = tagfilter_cmd_from_string(tf_key->u.u_string,
						tfc_html_none,0);
		
	if (tf_cmd_def) {

	    res->tf_cmd->pg_flags   = tf_cmd_def->pg_flags;
	    res->tf_cmd->code.base  = tf_cmd_def->code;
	    res->tf_cmd->nesting    = tf_cmd_def->nesting;
	    res->tf_cmd->cmd_mode   = tf_cmd_def->cmd_mode;
	} 
	
	break;
    case t_string_sort:
	res->tf_cmd = safe_zero_alloc(sizeof (* (res->tf_cmd)));

	res->tf_cmd->magic = TAGFILTER_CMD_magic;
	res->tf_cmd->refcount = 1;

	res->tf_cmd->ascii_command = NULL;

	res->tf_cmd->pg_flags   = 0;
	res->tf_cmd->code.base  = tfc_none;
	res->tf_cmd->nesting    = NULL;
	res->tf_cmd->cmd_mode   = 0;
	
	if (tf_cmd_def) {

	    res->tf_cmd->pg_flags   = tf_cmd_def->pg_flags;
	    res->tf_cmd->code.base  = tf_cmd_def->code;
	    res->tf_cmd->nesting    = tf_cmd_def->nesting;
	    res->tf_cmd->cmd_mode   = tf_cmd_def->cmd_mode;	    
	} 
	
	res->tf_cmd->string_command = tf_key->u.u_string_sort;
	inc_string_sort_refcount(res->tf_cmd->string_command);

	break;
    }
    
}

S_(free_sort_item_f free_tf_cmd_sort_item)
static void free_tf_cmd_sort_item P_((union sort_item * ptr));
/* Decrements refcount */
static void free_tf_cmd_sort_item(ptr)
     union sort_item      * ptr; /* Decrements refcount */
{    
    tagfilter_release_cmd(&(ptr->tf_cmd));
}
    
S_(inc_sort_item_refcount_f inc_tf_cmd_sort_item_refcount)
static void inc_tf_cmd_sort_item_refcount P_((union sort_item item));
static void inc_tf_cmd_sort_item_refcount(item)
     union sort_item item;
{
    tagfilter_inc_cmd_refcount(item.tf_cmd);
}
    
S_(sort_item_debug_name_f tf_cmd_sort_item_name)
static struct string * tf_cmd_sort_item_name P_((const union sort_item item));
static struct string * tf_cmd_sort_item_name (item)
     const union sort_item item;
{
    struct tagfilter_cmd    * tf_cmd = item.tf_cmd;
    struct string           * ret = NULL;

    if (TAGFILTER_CMD_magic != tf_cmd->magic)
	mime_panic(__FILE__,__LINE__,"tf_cmd_sort_item_name",
		   "Bad magic number (tagfilter_cmd)");

    if (tf_cmd->string_command) {
	ret =
	    give_string_from_string_sort(tf_cmd->
					 string_command);
    } else if (tf_cmd->ascii_command) {
	ret = new_string2(ASCII_SET,s2us(tf_cmd->ascii_command));

	/* Cache it */

	tf_cmd->string_command =
	    new_string_sort(ret);
    }
    
    return ret;
}

/* Return < 0 if key < item 
   Return > 0 if key > item 
*/
S_(compare_sort_key_to_item_f tf_cmd_key_cmp_item)
static int tf_cmd_key_cmp_item P_((const union sort_key key,
                                   const union sort_item item));
static int tf_cmd_key_cmp_item(key,item)
     const union sort_key key;
     const union sort_item item;
{

    struct tagfilter_cmd    * tf_cmd = item.tf_cmd;
    struct tf_tag_sort      * tf_key = key.tf_tag_sort;
    enum tf_tag_sort_type     tf_key_type;
    struct tagfilter_tags      * backlink = NULL;
    struct tagfilter_selection * sel_backlink = NULL;

    
    int ret = 0;
    int ignore_case = 0;
    
    if (TAGFILTER_CMD_magic != tf_cmd->magic)
	mime_panic(__FILE__,__LINE__,"tf_cmd_key_cmp_item",
		   "Bad magic number (tagfilter_cmd)");

    if (TF_TAG_SORT_magic != tf_key->magic)
	mime_panic(__FILE__,__LINE__,"tf_cmd_key_cmp_item",
		   "Bad magic number (tf_tag_sort)");
    
    backlink = tf_key->backlink;
    if (backlink) {
	if (TAGFILTER_TAGS_magic != backlink->magic)
	    	mime_panic(__FILE__,__LINE__,"tf_cmd_key_cmp_item",
			   "Bad magic number (tagfilter_tags)");

    }
    
    sel_backlink = tf_key->sel_backlink;
    if (sel_backlink) {
	if (TAGFILTER_SELECTION_magic != sel_backlink->magic) 
	    mime_panic(__FILE__,__LINE__,"tf_cmd_key_cmp_item",
		       "Bad magic number (tagfilter_selection)");
	
	ignore_case = ison(sel_backlink->tagflt_mode,TAGFLT_MODE_lowerascii);
    }
    
    tf_key_type = * (tf_key->u.type);

    switch (tf_key_type) {
	struct string         * u_string;
	struct tagfilter_cmd  * u_tagfilter_cmd;
	struct string_sort    * u_string_sort;
    case t_tagfilter_cmd:

	u_tagfilter_cmd = tf_key->u.u_tagfilter_cmd;

	if (TAGFILTER_CMD_magic != u_tagfilter_cmd->magic)
	    mime_panic(__FILE__,__LINE__,"tf_cmd_key_cmp_item",
		       "Bad magic number (tagfilter_cmd)");

	/* Return < 0 if key (tf_key, u_tagfilter_cmd) < item (tf_cmd)
	   Return > 0 if key (tf_key, u_tagfilter_cmd) > item (tf_cmd)
	*/
	
	if (tf_cmd->ascii_command && u_tagfilter_cmd->ascii_command) {
	    ret = ignore_case ?
		istrcmp(u_tagfilter_cmd->ascii_command,
			tf_cmd->ascii_command) :
		strcmp(u_tagfilter_cmd->ascii_command,
		       tf_cmd->ascii_command);

	    DPRINT(Debug,21,(&Debug,
			     "tf_cmd_key_cmp_item ret=%d %s <> %s (ascii compare)%s\n",
			     ret,u_tagfilter_cmd->ascii_command,tf_cmd->ascii_command,
			     ignore_case ? ", ignore case" : ", case sensitive"));
	    
	} else if (tf_cmd->ascii_command && u_tagfilter_cmd->string_command) {
	    struct string * key_string =
		give_string_from_string_sort(u_tagfilter_cmd->string_command);
	    
	    ret = string_matches_ascii(key_string,
				       s2us(tf_cmd->ascii_command),
				       ignore_case ? SMA_ignore_case : 0,
				       SMA_op_return_order);


	    DPRINT(Debug,21,(&Debug,
			     "tf_cmd_key_cmp_item ret=%d %S <> %s%s\n",
			     ret,key_string,tf_cmd->ascii_command,
			     ignore_case ? ", ignore case" : ", case sensitive"));			    

	   	    
	    free_string(& key_string);	    
	} else if (tf_cmd->string_command && u_tagfilter_cmd->ascii_command) {

	    struct string * item_string =
		give_string_from_string_sort(tf_cmd->string_command);

	    /* Reverse order */

	    ret = -string_matches_ascii(item_string,
					s2us(u_tagfilter_cmd->ascii_command),
					ignore_case ? SMA_ignore_case : 0,
					SMA_op_return_order);

	    DPRINT(Debug,21,(&Debug,
			     "tf_cmd_key_cmp_item ret=%d %s <> %S%s\n",
			     ret,u_tagfilter_cmd->ascii_command,
			     item_string,
			     ignore_case ? ", ignore case" : ", case sensitive"));
	    
	    free_string(& item_string);	    
	} else if (tf_cmd->string_command && u_tagfilter_cmd->string_command) {
	    /* NOTE:
	       Only ascii_command can be case insensitive 
	    */

	    ret = string_sort_cmp(u_tagfilter_cmd->string_command,
				  tf_cmd->string_command);

	    
	    DPRINT(Debug,21,(&Debug,
			     "tf_cmd_key_cmp_item ret=%d\n",ret));

	}

	break;
    case t_string:

	u_string = tf_key->u.u_string;
	
	/* Return < 0 if key (tf_key, u_string) < item (tf_cmd)
	   Return > 0 if key (tf_key, u_string) > item (tf_cmd)
	*/

	if  (tf_cmd->ascii_command) {

	    ret = string_matches_ascii(u_string,
				       s2us(tf_cmd->ascii_command),
				       ignore_case ? SMA_ignore_case : 0,
				       SMA_op_return_order);
	    
	    DPRINT(Debug,21,(&Debug,
			     "tf_cmd_key_cmp_item ret=%d %S <> %s%s\n",
			     ret,u_string,tf_cmd->ascii_command,
			     ignore_case ? ", ignore case" : ", case sensitive"));
	    				      
	} else if (tf_cmd->string_command) {
	    /* NOTE:
	       Only ascii_command can be case insensitive 
	    */
	    
	    struct string_sort * temp =
		new_string_sort(u_string);

	    ret = string_sort_cmp(temp,tf_cmd->string_command);

	    DPRINT(Debug,21,(&Debug,
			     "tf_cmd_key_cmp_item ret=%d\n",ret));
	    
	    free_string_sort(& temp);		
	}
	
	break;
    case t_string_sort:

	u_string_sort = tf_key->u.u_string_sort;

	/* Return < 0 if key (tf_key, u_string_sort) < item (tf_cmd)
	   Return > 0 if key (tf_key, u_string_sort) > item (tf_cmd)
	*/

	if  (tf_cmd->ascii_command) {
	    struct string * key_string =
		give_string_from_string_sort(u_string_sort);

	    ret = string_matches_ascii(key_string,
				       s2us(tf_cmd->ascii_command),
				       ignore_case ? SMA_ignore_case : 0,
				       SMA_op_return_order);

	    DPRINT(Debug,21,(&Debug,
			     "tf_cmd_key_cmp_item ret=%d %S <> %s%s\n",
			     ret,key_string,tf_cmd->ascii_command,
			     ignore_case ? ", ignore case" : ", case sensitive"));
	    
	    free_string(& key_string);	 
	} else if (tf_cmd->string_command) {
	    /* NOTE:
	       Only ascii_command can be case insensitive 
	    */

	    ret = string_sort_cmp(u_string_sort,
				  tf_cmd->string_command);

	    DPRINT(Debug,21,(&Debug,
			     "tf_cmd_key_cmp_item ret=%d\n",ret));
	    
	}
	
	break;
    }    

    return ret;
}
    
S_(sort_key_debug_name_f tf_cmd_key_name)
static struct string * tf_cmd_key_name P_((const union sort_key key));
static struct string * tf_cmd_key_name(key)
     const union sort_key key;
{
    struct tf_tag_sort      * tf_key = key.tf_tag_sort;
    enum tf_tag_sort_type     tf_key_type;
    struct string * ret = NULL;
    
    if (TF_TAG_SORT_magic != tf_key->magic)
	mime_panic(__FILE__,__LINE__,"tf_cmd_key_name",
		   "Bad magic number (tf_tag_sort)");

    tf_key_type = * (tf_key->u.type);

    switch (tf_key_type) {
	struct string         * u_string;
	struct tagfilter_cmd  * u_tagfilter_cmd;
	struct string_sort    * u_string_sort;
    case t_tagfilter_cmd:

	u_tagfilter_cmd = tf_key->u.u_tagfilter_cmd;

	if (TAGFILTER_CMD_magic != u_tagfilter_cmd->magic)
	    mime_panic(__FILE__,__LINE__,"tf_cmd_key_name",
		       "Bad magic number (tagfilter_cmd)");

	if (u_tagfilter_cmd->string_command) {
	    ret =
		give_string_from_string_sort(u_tagfilter_cmd->
					     string_command);
	} else if (u_tagfilter_cmd->ascii_command) {
	    ret = new_string2(ASCII_SET,
			      s2us(u_tagfilter_cmd->ascii_command));
	    
	    /* Cache it */
	    
	    u_tagfilter_cmd->string_command =
		new_string_sort(ret);
	}
	break;
    case t_string:

	u_string = tf_key->u.u_string;

	ret = dup_string(u_string);
	break;

    case t_string_sort:

	u_string_sort = tf_key->u.u_string_sort;

	ret = give_string_from_string_sort(u_string_sort);
	break;
    }

    return ret;
}

static struct sort_operation tagfilter_cmd_operation = {
    SORT_OPERATION_magic,
    alloc_tf_cmd_sort_item,
    free_tf_cmd_sort_item,
    inc_tf_cmd_sort_item_refcount,
    tf_cmd_sort_item_name,
    tf_cmd_key_cmp_item,
    tf_cmd_key_name
};

static const int max_unknown_tags = 10;  /* Remember 10 tags on report */


struct string **tagfilter_list_tags(unsupported_tags)
     struct sortlist * unsupported_tags;
{
    size_t count =  sort_list_len(unsupported_tags);

    struct string ** ret = safe_calloc(count+1,sizeof (ret[0]));
    size_t i = 0;

    for (i = 0; i < count; i++) {
	union sort_item  res;

	res.tf_cmd = NULL;
	
	get_sort_list_item(unsupported_tags,sort_list_get_normal,i,&res);

	if (res.tf_cmd) {
	    
	    if (TAGFILTER_CMD_magic != res.tf_cmd->magic)
		mime_panic(__FILE__,__LINE__,"tagfilter_list_tags",
			   "Bad magic number (tagfilter_cmd)");

	    if (res.tf_cmd->string_command)
		ret[i] =
		    give_string_from_string_sort(res.tf_cmd->
						 string_command);
	    else if (res.tf_cmd->ascii_command) 
		ret[i] = new_string2(ASCII_SET,
				     s2us(res.tf_cmd->ascii_command));
	    else
		ret[i] = format_string(FRM("( error on item %zu )"),i);
	    
	} else {

	    DPRINT(Debug,15,(&Debug,
			     "tagfilter_list_tags: tag #%zu -  no res.tf_cmd\n",
			     i));

	    ret[i] = format_string(FRM("( error on item %zu )"),i);
	}

	free_tf_cmd_sort_item(& res);	
	
    }

    ret[i] = NULL;

    return ret;
}


static void tagfilter_add_unknown_tag P_((struct tagfilter_global * counter,
					  struct tagfilter_cmd * tag,
					  struct tagfilter_selection * sel_backlink));
static void tagfilter_add_unknown_tag(counter,tag,sel_backlink)
     struct tagfilter_global * counter;
     struct tagfilter_cmd * tag;
     struct tagfilter_selection * sel_backlink;
{
    size_t count;

    struct tf_cmd_def tf_cmd_def;
    
    union sort_key            key;
    union sort_item_default   def;
    union sort_item           res;
    struct tf_tag_sort  tf_tag_sort;
    size_t res_idx    =  0;
    
     /* bzero is defined on hdrs/elm_defs.h */
    bzero(&tf_cmd_def,sizeof tf_cmd_def);
    tf_cmd_def = NULL_tf_cmd_def;
    
    if (TAGFILTER_GLOBAL_magic != counter->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_add_unknown_tag",
                   "Bad magic number (tagfilter_global)");


    if (TAGFILTER_CMD_magic != tag->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_add_unknown_tag",
		   "Bad magic number (tagfilter_cmd)");
    
    if (!counter->unsupported_tags)
	counter->unsupported_tags =
	    alloc_sort_list(&tagfilter_cmd_operation,1);

    count = sort_list_len(counter->unsupported_tags);

    tf_tag_sort.magic             = TF_TAG_SORT_magic;
    tf_tag_sort.backlink          = NULL;
    tf_tag_sort.sel_backlink      = sel_backlink;
    tf_tag_sort.u.u_tagfilter_cmd = tag;

    res.tf_cmd = NULL;

    key.tf_tag_sort = &tf_tag_sort;
    def.tf_cmd      = &tf_cmd_def;

    if (search_sort_list_item(counter->unsupported_tags,
			      count >= max_unknown_tags ?
			      sort_list_search_normal :
			      sort_list_search_create,
			      key,def,&res,&res_idx,NULL)) {
	
	if (res.tf_cmd) {
	    
	    if (TAGFILTER_CMD_magic != res.tf_cmd->magic)
		mime_panic(__FILE__,__LINE__,"tagfilter_add_unknown_tag",
			   "Bad magic number (tagfilter_cmd)");

	     DPRINT(Debug,15,(&Debug,
			     "tagfilter_add_unknown_tag: tag found from #%zu or created.\n",
			      res_idx));

	     free_tf_cmd_sort_item(& res);
	     return;
	}
	
	DPRINT(Debug,15,(&Debug,
			 "tagfilter_add_unknown_tag: tag found from #%zu -  no res.tf_cmd\n",
			 
			 res_idx));

	free_tf_cmd_sort_item(& res);
    } else {
	DPRINT(Debug,15,(&Debug,
			 "tagfilter_add_unknown_tag: tag not found\n"));	
    }

    counter->have_other = 1;
}


static struct tagfilter_cmd * tagfilter_give_unknown_tag
     P_((struct tagfilter_global * counter,
	 struct string * tag_name,
	 struct tagfilter_selection * sel_backlink,
	 enum  tagfilter_cmd_base   base_code,
	 int cmd_mode_flags));
static struct tagfilter_cmd * tagfilter_give_unknown_tag(counter,tag_name,sel_backlink,
							 base_code,
							 cmd_mode_flags)
     struct tagfilter_global * counter;
     struct string       * tag_name;
     struct tagfilter_selection * sel_backlink;
     enum  tagfilter_cmd_base   base_code;
     int cmd_mode_flags;
{
    size_t count;

    struct tf_cmd_def tf_cmd_def;
    
    union sort_key            key;
    union sort_item_default   def;
    union sort_item           res;
    struct tf_tag_sort  tf_tag_sort;
    size_t res_idx    =  0;
    
     /* bzero is defined on hdrs/elm_defs.h */
    bzero(&tf_cmd_def,sizeof tf_cmd_def);
    tf_cmd_def = NULL_tf_cmd_def;
    tf_cmd_def.code     = base_code;
    tf_cmd_def.cmd_mode = cmd_mode_flags;
    
    if (TAGFILTER_GLOBAL_magic != counter->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_give_unknown_tag",
                   "Bad magic number (tagfilter_global)");

    if (!counter->unsupported_tags)
	counter->unsupported_tags = alloc_sort_list(&tagfilter_cmd_operation,1);

    count = sort_list_len(counter->unsupported_tags);

    tf_tag_sort.magic             = TF_TAG_SORT_magic;
    tf_tag_sort.backlink          = NULL;
    tf_tag_sort.sel_backlink      = sel_backlink;
    tf_tag_sort.u.u_string        = tag_name;

    res.tf_cmd = NULL;

    key.tf_tag_sort = &tf_tag_sort;
    def.tf_cmd      = &tf_cmd_def;

    if (search_sort_list_item(counter->unsupported_tags,
			      count >= max_unknown_tags ?
			      sort_list_search_normal :
			      sort_list_search_create,
			      key,def,&res,&res_idx,NULL)) {
	
	if (res.tf_cmd) {
	    
	    if (TAGFILTER_CMD_magic != res.tf_cmd->magic)
		mime_panic(__FILE__,__LINE__,"tagfilter_give_unknown_tag",
			   "Bad magic number (tagfilter_cmd)");

	    DPRINT(Debug,15,(&Debug,
			     "tagfilter_give_unknown_tag: tag %S found from #%zu or created.\n",
			     tag_name,res_idx));
	    
	    
	    return res.tf_cmd;
	}
	
	DPRINT(Debug,15,(&Debug,
			 "tagfilter_give_unknown_tag: tag %S found from #%zu -  no res.tf_cmd\n",
			 
			 tag_name,res_idx));

	free_tf_cmd_sort_item(& res);	
    } else {
	DPRINT(Debug,15,(&Debug,
			 "tagfilter_give_unknown_tag: tag %S not found\n",
			 tag_name));	
    }
    
    counter->have_other = 1;

    return NULL;
}

static struct sortlist *  generate_sorted_tags
    P_((struct tagfilter_tags * tags,
	struct tagfilter_selection * sel_backlink));
static struct sortlist *  generate_sorted_tags(tags,sel_backlink)
     struct tagfilter_tags * tags;
     struct tagfilter_selection * sel_backlink;
{
    struct sortlist *  ret = NULL;
    size_t count = 0;
    size_t i;

    struct tf_cmd_def tf_cmd_def;  /* Not really needed
				      on this case
				   */

    size_t                res_idx = 0;
    
    if (TAGFILTER_TAGS_magic != tags->magic)
	mime_panic(__FILE__,__LINE__,"generate_sorted_tags",
		   "Bad magic number (tagfilter_tags)");

    if (TAGFILTER_SELECTION_magic != sel_backlink->magic) 
	mime_panic(__FILE__,__LINE__,"generate_sorted_tags",
		   "Bad magic number (tagfilter_selection)");

    /* bzero is defined on hdrs/elm_defs.h */
    bzero(&tf_cmd_def,sizeof tf_cmd_def);
    tf_cmd_def = NULL_tf_cmd_def;   

       
    if (tags->tags) {
	while (tags->tags[count]) {
	    
	    if (TAGFILTER_CMD_magic != tags->tags[count]->magic)
		mime_panic(__FILE__,__LINE__,"generate_sorted_tags",
			   "Bad magic number (tagfilter_cmd)");

	    count++;
	}
    }

    DPRINT(Debug,20,(&Debug,
		     "generate_sorted_tags: Adding %zu tags for %s\n",
		     count,tags->context));
    
    ret = alloc_sort_list(&tagfilter_cmd_operation,count);


    for (i = 0; i < count; i++) {

	union sort_key            key;
	union sort_item_default   def;
	union sort_item           res;
	
	struct tf_tag_sort  tf_tag_sort;
	
	if (TAGFILTER_CMD_magic != tags->tags[i]->magic)
	    mime_panic(__FILE__,__LINE__,"generate_sorted_tags",
		       "Bad magic number (tagfilter_cmd)");

	/* bzero is defined on hdrs/elm_defs.h */
	bzero(&tf_tag_sort,sizeof tf_tag_sort);

	tf_tag_sort.magic             = TF_TAG_SORT_magic;
	tf_tag_sort.backlink          = tags;
	tf_tag_sort.sel_backlink      = sel_backlink;
	tf_tag_sort.u.u_tagfilter_cmd = tags->tags[i];
	
	res.tf_cmd = NULL;
	
	key.tf_tag_sort = &tf_tag_sort;
	def.tf_cmd      = &tf_cmd_def;

	if (search_sort_list_item(ret,sort_list_insert_hint,
				  key,def,&res,&res_idx,NULL)) {
	    
	    if (res.tf_cmd) {

		if (TAGFILTER_CMD_magic != res.tf_cmd->magic)
		    mime_panic(__FILE__,__LINE__,"generate_sorted_tags",
			       "Bad magic number (tagfilter_cmd)");
		
		if (res.tf_cmd != tags->tags[i]) {
		    DPRINT(Debug,1,(&Debug,
				    "generate_sorted_tags: failed to add %zu %Q - res_idx #%zu - got different entry",
				    i,tags->tags[i]->ascii_command,
				    res_idx));
		    if (res.tf_cmd->ascii_command) {
			DPRINT(Debug,1,(&Debug," %Q",
					res.tf_cmd->ascii_command));
		    }
		    DPRINT(Debug,1,(&Debug,"\n"));
		}
		

	    } else {
		DPRINT(Debug,1,(&Debug,
				"generate_sorted_tags: failed to add %zu %Q - res_idx #%zu - no res.tf_cmd\n",
				i,tags->tags[i]->ascii_command,
				res_idx));

	    }

	    free_tf_cmd_sort_item(& res);
	    
	} else {
	    DPRINT(Debug,1,(&Debug,
			    "generate_sorted_tags: failed to add %zu %Q - res_idx #%zu\n",
			    i,tags->tags[i]->ascii_command,
			    res_idx));
	}
    }
    
    return ret;
}

struct tagfilter_global NULL_counter = {
    TAGFILTER_GLOBAL_magic,
    NULL,
    0
};

void tagfilter_global_dest(counter)
     struct tagfilter_global *counter;
{
    if (TAGFILTER_GLOBAL_magic != counter->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_global_dest",
                   "Bad magic number (tagfilter_global)");

    if (counter->unsupported_tags)
	free_sort_list(& (counter->unsupported_tags));

    counter->have_other = 0;

}

/* Increments refount ! 
   does not recurse
*/

static struct tagfilter_cmd * locate_command
          P_((struct tagfilter_tags * tags,  /* assumed static */
	      struct string   * tag_name,
	      struct tagfilter_selection * sel_backlink));
static struct tagfilter_cmd * locate_command(tags,tag_name,
					     sel_backlink)
     struct tagfilter_tags * tags;
     struct string         * tag_name;
     struct tagfilter_selection * sel_backlink;
{

    struct tf_cmd_def tf_cmd_def;  /* Not really needed
				      on this case
				   */
    
    if (TAGFILTER_TAGS_magic != tags->magic)
	mime_panic(__FILE__,__LINE__,"locate_command",
		   "Bad magic number (tagfilter_tags)");
    
    /* bzero is defined on hdrs/elm_defs.h */
    bzero(&tf_cmd_def,sizeof tf_cmd_def);
    tf_cmd_def = NULL_tf_cmd_def;
    
    tf_cmd_def.nesting = tags;   /* By default same tags 
				    assumed static
				 */    
     
     if (! tags->sorted_tags)
	 tags->sorted_tags = generate_sorted_tags(tags,sel_backlink);
     
     if (tags->sorted_tags) {
	 union sort_key            key;
	 union sort_item_default   def;
	 union sort_item           res;
	 struct tf_tag_sort  tf_tag_sort;
	 size_t res_idx    =  0;

	 DPRINT(Debug,20,(&Debug,
			  "locate_command: Searching %S tag from %s\n",
			  tag_name,tags->context));
	 
	 tf_tag_sort.magic             = TF_TAG_SORT_magic;
	 tf_tag_sort.backlink          = tags;
	 tf_tag_sort.sel_backlink      = sel_backlink;
	 tf_tag_sort.u.u_string        = tag_name;

	 res.tf_cmd = NULL;


	 key.tf_tag_sort = &tf_tag_sort;
	 def.tf_cmd      = &tf_cmd_def;

	 if (search_sort_list_item(tags->sorted_tags,
				   sort_list_search_normal,
				   key,def,&res,&res_idx,NULL)) {

	     if (res.tf_cmd) {
		 
		 if (TAGFILTER_CMD_magic != res.tf_cmd->magic)
		     mime_panic(__FILE__,__LINE__,"locate_command",
				"Bad magic number (tagfilter_cmd)");
		 
		 DPRINT(Debug,15,(&Debug,
				  "locate_command: tag %S found from #%zu (context %s)",
				  tag_name,res_idx,
				  tags->context));

		 
		 DPRINT(Debug,15,(&Debug,", code=%d",
				  res.tf_cmd->code.code));
		 
		 switch (res.tf_cmd->code.base) {
		 case tfc_unsupported:
		     DPRINT(Debug,15,(&Debug," tfc_unsupported")); break;
		 case tfc_none:
		     DPRINT(Debug,15,(&Debug," tfc_none"));break;
		 case tfc_enriched_cmd:
		 case tfc_html_cmd:
		     break;
		 }


		 DPRINT(Debug,15,(&Debug,"\n"));
		 
		 return res.tf_cmd;
	     }

	     DPRINT(Debug,15,(&Debug,
			      "locate_command: tag %S found from #%zu (context %s) -  no res.tf_cmd\n",
			      
			      tag_name,res_idx,tags->context));

	     free_tf_cmd_sort_item(& res);

	 } else {
	     DPRINT(Debug,15,(&Debug,
			      "locate_command: tag %S not found from %s\n",
			      tag_name,tags->context));
	 }
	 	 
     } else {
	 DPRINT(Debug,9,(&Debug,
			 "locate_command: no sorted_tags when searching %S from %s\n",
			 tag_name,tags->context));
     }
     
    return NULL;
}

/* non-common parts need aleady be free'ed */



static void tagfilter_free_tag_state_common  P_((struct tagfilter_tag_state **tag_state));
static void tagfilter_free_tag_state_common(tag_state)
     struct tagfilter_tag_state **tag_state;
{
    if (TAGFILTER_TAG_STATE_magic != (*tag_state)->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_free_tag_state_common",
		   "Bad magic number (tagfilter_tag_state)");


    if ((*tag_state)->u.dummy) {
	DPRINT(Debug,1,(&Debug,
			"tagfilter_free_tag_state_common: (*tag_state)->u.dummy not NULL\n"));

	/* leaks data */
	
	free ((*tag_state)->u.dummy);
	(*tag_state)->u.dummy = NULL;
    }
    
    if ((*tag_state)-> current_tag)
	tagfilter_release_cmd(& ((*tag_state)-> current_tag));

    
    (*tag_state)->magic = 0; /* Invalidate */

    free(*tag_state);
    *tag_state = NULL;
}


struct tagfilter_tag_state * tagfilter_new_tag_state(current_tag)
     struct tagfilter_cmd    * current_tag;
{
    struct tagfilter_tag_state * ret = 
	safe_zero_alloc(sizeof (*ret));
    
    ret->magic = TAGFILTER_TAG_STATE_magic;
    
    ret->current_tag = current_tag;
    if (ret->current_tag)
	tagfilter_inc_cmd_refcount(ret->current_tag);
	    
    ret->u.dummy     = NULL;

    ret->no_context  = 0;

    if (ret->current_tag) {
	if (TAGFILTER_CMD_magic != ret->current_tag->magic)
	    mime_panic(__FILE__,__LINE__,"tagfilter_new_tag_state",
		       "Bad magic number (tagfilter_cmd)");

	ret->no_context = ison(ret->current_tag->cmd_mode,
			       TAGFLT_CMD_no_context);
    }
    
    return ret;
}

/* enriched ------------------------------ */

struct tagfilter_tags text_enriched_tags = {
    TAGFILTER_TAGS_magic,

    "text/enriched tags",
    
    &(tagfilter_enriched_cmds[0]),
    NULL
};

E_(give_top_stack_item_f give_top_text_enriched)
struct tagfilter_stack_item * give_top_text_enriched
  P_((mime_t                     * body,
      struct in_state            * state_in,
      struct out_state           * state_out,
      const struct decode_opts   * decode_opt,
      charset_t                    text_charset,
      struct tagfilter_selection * tagfilter));
struct tagfilter_stack_item * give_top_text_enriched(body,state_in,state_out,
						     decode_opt,text_charset,tagfilter)
     mime_t                     * body;
     struct in_state            * state_in;
     struct out_state           * state_out;
     const struct decode_opts   * decode_opt;
     charset_t                    text_charset;
     struct tagfilter_selection * tagfilter;
{
    struct tagfilter_stack_item * ret = NULL;
    struct pager_range          * enriched_body_range = NULL;
    union tagfilter_stack_uitem uitem;

    uitem.magic = NULL;
    
    if (MIME_magic != body->magic)
	mime_panic(__FILE__,__LINE__,"give_top_text_enriched",
		   "Bad magic number (mime_t)");
	
    if (decode_opt)
	enriched_body_range =
	    state_add_simple_pager_range(state_out,
					 /* Handles quote prefix */
					 decode_opt->range,
					 PR_WORD_WRAP|PR_JOIN_LINES|
					 PR_ADD_SPACE,0,0);
    else
	enriched_body_range =
	    state_add_simple_pager_range(state_out,
					 NULL,
					 PR_WORD_WRAP|PR_JOIN_LINES|
					 PR_ADD_SPACE,0,0);

    ret =  tagfilter_new_stack_item(enriched_body_range,
				    0,
				    &text_enriched_tags,
				    NULL,uitem);

    if (TAGFILTER_STACK_ITEM_magic != ret->magic)
	mime_panic(__FILE__,__LINE__,"give_top_text_enriched",
		   "Bad magic number (tagfilter_stack_item)");
    
    free_pager_range(&enriched_body_range);
    
    return ret;
}

static  struct pager_range * give_prev_range
     P_((struct tagfilter_stack_item **stack,
	 size_t                        stack_len,
	 int                          * inherit_pg_flags_p,
	 int                          * text_visible_p
	 ));
static  struct pager_range * give_prev_range(stack,stack_len,
					     inherit_pg_flags_p,
					     text_visible_p)
     struct tagfilter_stack_item ** stack;
     size_t                         stack_len;
     int                          * inherit_pg_flags_p;
     int                          * text_visible_p;
{
    struct pager_range * prev_range = NULL;

    if (stack && stack_len > 0) {
	size_t i = stack_len;

	int is_reset = 0;
	
	while (i > 0) {
	    i--;
	    
	    if (stack[i]) {
		if (TAGFILTER_STACK_ITEM_magic != stack[i]->magic)
		    mime_panic(__FILE__,__LINE__,
			       "give_prev_range",
			       "Bad magic number (tagfilter_stack_item)");
		if (! prev_range)
		    prev_range = stack[i]->range;

		if (! is_reset && inherit_pg_flags_p)
		    *inherit_pg_flags_p |= stack[i]->pg_flags;

		if (stack[i]->reset_pg_flags)
		    is_reset = 1;
		
		if (prev_range && is_reset)
		    break;
		if (prev_range && ! inherit_pg_flags_p)
		    break;
	    }
	}

	if (TAGFILTER_STACK_ITEM_magic != stack[stack_len-1]->magic)
	    mime_panic(__FILE__,__LINE__,
		       "give_prev_range",
		       "Bad magic number (tagfilter_stack_item)");

	if (text_visible_p)
	    *text_visible_p = stack[stack_len-1]->text_visible;

    }
    
    return prev_range;
}



E_(give_stack_item_f give_item_text_enriched)
struct tagfilter_tag_state *
  give_item_text_enriched P_((struct tagfilter_stack_item **new_item,
			      struct tagfilter_tags      * tags,
			      struct string              * tag_name,
			      struct tagfilter_stack_item **stack,
			      size_t                        stack_len,
			      size_t                     * stack_pos,
			      mime_t                     * body,
			      struct in_state            * state_in,
			      struct out_state           * state_out,
			      const struct decode_opts   * decode_opt,
			      charset_t                    text_charset,
			      struct tagfilter_selection * tagfilter,
			      struct tagfilter_global    * counter));
struct tagfilter_tag_state *give_item_text_enriched(new_item,tags,
						    tag_name,
						    stack,stack_len,
						    stack_pos,
						    body,state_in,
						    state_out,
						    decode_opt,
						    text_charset,
						    tagfilter,
						    counter)
     struct tagfilter_stack_item **new_item;
     struct tagfilter_tags      * tags;
     struct string              * tag_name;
     struct tagfilter_stack_item **stack;
     size_t                        stack_len;
     size_t                     * stack_pos;
     mime_t                     * body;
     struct in_state            * state_in;
     struct out_state           * state_out;
     const struct decode_opts   * decode_opt;
     charset_t                    text_charset;
     struct tagfilter_selection * tagfilter;
     struct tagfilter_global    * counter;   
{

    struct tagfilter_cmd       * current_tag = NULL;
    struct tagfilter_tag_state * ret = NULL;
    union tagfilter_stack_uitem uitem;
    
    int text_visible = 1;

    uitem.magic = NULL;
    
    /* Allocates new  tagfilter_tag_state 
       (also NULL return possible) -- 
       gives struct tagfilter_stack_item if
       push_stack_item() needs not be used */
           
    if (TAGFILTER_SELECTION_magic != tagfilter->magic) 
	mime_panic(__FILE__,__LINE__,"give_item_text_enriched",
		   "Bad magic number (tagfilter_selection)");

    if (new_item && *new_item)
	tagfilter_free_stack_item(new_item,tagfilter);

    * stack_pos = stack_len;

    if (tags) {
	struct pager_range * prev_range =
	    give_prev_range(stack,
			    stack_len,
			    NULL,
			    & text_visible
			    );
	struct pager_range * new_range = NULL;
		       	    
	if (TAGFILTER_TAGS_magic != tags->magic)
	    mime_panic(__FILE__,__LINE__,"give_item_text_enriched",
		       "Bad magic number (tagfilter_tags)");

	if (tag_name)
	    current_tag = locate_command(tags,tag_name,tagfilter);

	/* <ParaIndent> command requires <param>
	   and is not now supported
	*/
		
	if (current_tag) {

	    if (TAGFILTER_CMD_magic != current_tag->magic)
		mime_panic(__FILE__,__LINE__,"give_item_text_enriched",
			   "Bad magic number (tagfilter_cmd)");

	    if (text_visible && ison(current_tag->cmd_mode,
				     TAGFLT_CMD_hide_text))
		text_visible = 0;
	    else if (ison(current_tag->cmd_mode,
			  TAGFLT_CMD_show_text))
		text_visible = 1;
	    
	    switch (current_tag->code.enriched) {
		
	    case tfc_enriched_unsupported:

		
		tagfilter_add_unknown_tag(counter,current_tag,tagfilter);

		break;
		
	    case tfc_enriched_none:      break;
		
	    case tfc_center:

		/* Print new pager range 

		   quote level, range flags, indent are automatically
		   incremented with  prev_range

		 */
		
		new_range = state_add_simple_pager_range(state_out,
							 prev_range,
							 /* Not work very well */
							 PR_CENTER_THIS,
							 0 /* quote level */,
							 0 /* indent */);
		
		break;

	    case tfc_nojoin: {

		/* Calculate absolute range flags */
		int range_flags = PR_FLAGS_ABSOLUTE;
		
		if (prev_range)
		    setit(range_flags,get_pager_range_flags(prev_range));
		clearit(range_flags,PR_JOIN_LINES);
		clearit(range_flags,PR_ADD_SPACE);
		clearit(range_flags,PR_SAME_LINE);


		new_range = state_add_simple_pager_range(state_out,
							 prev_range,
							 range_flags,
							 0,
							 0);		    
	    }
		break;
		
	    case tfc_quote:

		/* Print new pager range 

		   quote level, range flags, indent are automatically
		   incremented with  prev_range

		 */
		
		new_range = state_add_simple_pager_range(state_out,
							 prev_range,
							 0 /* range flags */,
							 1 /* quote level */,
							 0 /* indent */);
		break;
	    }


	    if (!new_range &&
		ison(current_tag->cmd_mode,TAGFLT_CMD_force_newline)) {

		DPRINT(Debug,21,(&Debug,
				 "give_item_text_enriched: %S -- TAGFLT_CMD_force_newline set, forcing new range\n",
				 tag_name));
		
		new_range = state_add_simple_pager_range(state_out,
							 prev_range,
							 0,0,0);

	    }
	    
	    if (new_item) {

		DPRINT(Debug,21,(&Debug,
				 "give_item_text_enriched: %S pg_flags %d %s\n",
				 tag_name,current_tag->pg_flags,
				 give_pg_flags(current_tag->pg_flags)));
		
		*new_item =
		    tagfilter_new_stack_item(new_range,
					     current_tag->pg_flags,
					     tags /* PASSTRU for enriched */,
					     current_tag,uitem);

		if (TAGFILTER_STACK_ITEM_magic != (*new_item)->magic)
		    mime_panic(__FILE__,__LINE__,"give_item_text_enriched",
			       "Bad magic number (tagfilter_stack_item)");

		(*new_item)->text_visible = text_visible;
	    }
	    
	    switch (current_tag->code.enriched) {
	    case tfc_enriched_unsupported:
	    case tfc_enriched_none:        break;

		
	    case tfc_center:
		break;

	    case tfc_nojoin:
		break;
		
	    case tfc_quote:
		break;
	    }
	    
	    if (new_range)
		free_pager_range(& new_range);

	}		
    } else {
	if (tag_name) {
	    DPRINT(Debug,21,(&Debug,
			     "give_item_text_enriched: %S - no tags\n",
			     tag_name));
	}
    }

    if (!current_tag && tag_name) {
	struct pager_range * prev_range  UNUSED_VAROK =
	    give_prev_range(stack,
			    stack_len,
			    NULL,
			    & text_visible
			    );
	if (counter) 	   
	    current_tag = tagfilter_give_unknown_tag(counter,tag_name,
						     tagfilter,
						     tfc_unsupported,0
						     );

	if (!current_tag)
	    current_tag = tagfilter_cmd_from_string(tag_name,
						    tfc_unsupported,0);


	if (TAGFILTER_CMD_magic != current_tag->magic)
	    mime_panic(__FILE__,__LINE__,"give_item_text_enriched",
		       "Bad magic number (tagfilter_cmd)");
	
	if (new_item) {

	    DPRINT(Debug,21,(&Debug,
			     "give_item_text_enriched: %S pg_flags %d %s\n",
			     tag_name,current_tag->pg_flags,
			     give_pg_flags(current_tag->pg_flags)));
	    
	    *new_item =
		tagfilter_new_stack_item(NULL,
					 current_tag->pg_flags,
					 tags /* PASHTRU for enriched */,
					 current_tag,uitem);
	    
	    if (TAGFILTER_STACK_ITEM_magic != (*new_item)->magic)
		mime_panic(__FILE__,__LINE__,"give_item_text_enriched",
			   "Bad magic number (tagfilter_stack_item)");
	    
	    (*new_item)->text_visible = text_visible;
	}
    }

    if (current_tag)
	ret = tagfilter_new_tag_state(current_tag);

    if (new_item && *new_item && current_tag) {
	if (TAGFILTER_STACK_ITEM_magic != (*new_item)->magic)
	    mime_panic(__FILE__,__LINE__,"give_item_text_enriched",
		       "Bad magic number ( tagfilter_stack_item)");
	if (TAGFILTER_CMD_magic != current_tag->magic)
	    mime_panic(__FILE__,__LINE__,"give_item_text_enriched",
		       "Bad magic number (tagfilter_cmd)");

	(*new_item)->force_eoln =
	    ison(current_tag->cmd_mode,TAGFLT_CMD_force_newline);
	
    }

    if (current_tag)
	tagfilter_release_cmd(& current_tag);
    
    return ret;

}

static struct tagfilter_tag_state * tagfilter_locate_stack_item
   P_((struct tagfilter_selection * tagfilter,
       struct string_sort        ** res_tag_name_X,
       struct string               * tag_name,
       struct tagfilter_stack_item **stack,
       size_t                        stack_len,
       size_t                      * stack_pos));
static struct tagfilter_tag_state *
    tagfilter_locate_stack_item(tagfilter,res_tag_name_X,
				tag_name,stack,stack_len,stack_pos)
     struct tagfilter_selection   * tagfilter;
     struct string_sort          ** res_tag_name_X;
     struct string                * tag_name;
     struct tagfilter_stack_item ** stack;
     size_t                         stack_len;
     size_t                      * stack_pos;
{
    struct tagfilter_tag_state * ret        = NULL;
    struct string_sort         * tag_name_X = * res_tag_name_X;
    int ignore_case = 0;
    
    if (TAGFILTER_SELECTION_magic != tagfilter->magic) 
	mime_panic(__FILE__,__LINE__,"tagfilter_locate_stack_item",
		   "Bad magic number (tagfilter_selection)");

    ignore_case = ison(tagfilter->tagflt_mode,TAGFLT_MODE_lowerascii);

    * stack_pos = stack_len;
    
    if (tag_name && stack && stack_len > 0) {
	size_t i = stack_len;
	
	while (i > 0) {
	    i--;
	    
	    if (stack[i]) {
		if (TAGFILTER_STACK_ITEM_magic != stack[i]->magic)
		    mime_panic(__FILE__,__LINE__,
			       "tagfilter_locate_stack_item",
			       "Bad magic number (tagfilter_stack_item)");
		
		if (stack[i]->current_tag) {
		    struct tagfilter_cmd *cmd;
		    
		    if (TAGFILTER_CMD_magic !=
			stack[i]->current_tag->magic)
			mime_panic(__FILE__,__LINE__,
				   "tagfilter_locate_stack_item",
				   "Bad magic number (tagfilter_cmd)");
		    
		    cmd = stack[i]->current_tag;
		    
		    if (cmd->ascii_command) { 
			if (string_matches_ascii(tag_name,
						 s2us(cmd->ascii_command),
						 ignore_case ? SMA_ignore_case : 0,
						 SMA_op_normal)) {
			    
			    * stack_pos = i;
			    ret = tagfilter_new_tag_state(cmd);
			    
			    break;			
			}

		    } else if (cmd->string_command) {

			/* Increments refcount and shares string! */
			if (!tag_name_X)
			    tag_name_X = new_string_sort(tag_name);
			
			if (0 == string_sort_cmp(cmd->string_command,
						 tag_name_X)) {
			    
			    * stack_pos = i;
			    ret = tagfilter_new_tag_state(cmd);
			    
			    break;			    
			}			

		    }
		}
	    }
	}
    }
    

    * res_tag_name_X = tag_name_X ;
    
    return ret;
}


E_(locate_stack_item_f locate_item_text_enriched)
struct tagfilter_tag_state *
    locate_item_text_enriched P_((struct string               * tag_name,
				  struct tagfilter_stack_item **stack,
				  size_t                        stack_len,
				  size_t                      * stack_pos
				  /* or stack_len if not found */,
				  mime_t                     * body,
				  struct in_state            * state_in,
				  struct out_state           * state_out,
				  charset_t                    text_charset,
				  struct tagfilter_selection * tagfilter));
struct tagfilter_tag_state * locate_item_text_enriched(tag_name,
						       stack,stack_len,
						       stack_pos,body,
						       state_in,state_out,
						       text_charset,tagfilter)
     struct string               * tag_name;
     struct tagfilter_stack_item **stack;
     size_t                        stack_len;
     size_t                      * stack_pos
     /* or stack_len if not found */;
     mime_t                     * body;
     struct in_state            * state_in;
     struct out_state           * state_out;
     charset_t                    text_charset;
     struct tagfilter_selection * tagfilter;
{
    /* Allocates new  tagfilter_tag_state 
       (also NULL return possible) */
    
    struct tagfilter_tag_state * ret = NULL;

    struct string_sort  * tag_name_X = NULL;  /* Possible lowercased
						 already 
					      */
    
    if (TAGFILTER_SELECTION_magic != tagfilter->magic) 
	mime_panic(__FILE__,__LINE__,"locate_item_text_enriched",
		   "Bad magic number (tagfilter_selection)");

    ret = tagfilter_locate_stack_item(tagfilter,
				      &tag_name_X,
				      tag_name,
				      stack,stack_len,stack_pos);
    
    if (tag_name_X)
	free_string_sort(& tag_name_X);
    
    
    return ret;

}

E_(end_tag_state_f end_tag_text_enriched)
void end_tag_text_enriched P_((struct tagfilter_tag_state **tag_state,
			       mime_t                     * body,
			       struct in_state            * state_in,
			       struct out_state           * state_out,
			       charset_t                    text_charset,
			       struct tagfilter_selection * tagfilter,
			       struct tagfilter_global    * counter));
void end_tag_text_enriched(tag_state,body,state_in,state_out,text_charset,tagfilter,
			   counter)
     struct tagfilter_tag_state **tag_state;
     mime_t                     * body;
     struct in_state            * state_in;
     struct out_state           * state_out;
     charset_t                    text_charset;
     struct tagfilter_selection * tagfilter;
     struct tagfilter_global    * counter;
{
    
    if (TAGFILTER_SELECTION_magic != tagfilter->magic) 
	mime_panic(__FILE__,__LINE__,"end_tag_text_enriched",
		   "Bad magic number (tagfilter_selection)");
    
    
    if (TAGFILTER_TAG_STATE_magic != (*tag_state)->magic)
	mime_panic(__FILE__,__LINE__,"end_tag_text_enriched",
		   "Bad magic number (tagfilter_tag_state)");
    
    /* Free enriched specific parts */
    if ((*tag_state)->u.dummy) {

	DPRINT(Debug,1,(&Debug,
			"end_tag_text_enriched: (*tag_state)->u.dummy is not NULL\n"));
    }
    tagfilter_free_tag_state_common (tag_state);
}

E_(push_stack_item_f push_tag_text_enriched)
struct tagfilter_stack_item *
   push_tag_text_enriched P_((struct tagfilter_tag_state *tag_state,
			      struct tagfilter_stack_item **stack,
			      size_t                        stack_len,
			      mime_t                     * body,
			      struct in_state            * state_in,
			      struct out_state           * state_out,
			      const struct decode_opts   * decode_opt,
			      charset_t                    text_charset,
			      struct tagfilter_selection * tagfilter));
struct tagfilter_stack_item *push_tag_text_enriched(tag_state,stack, stack_len,
						    body,state_in,state_out,
						    decode_opt,
						    text_charset,tagfilter)
     struct tagfilter_tag_state *tag_state;
     struct tagfilter_stack_item **stack;
     size_t                        stack_len;
     mime_t                     * body;
     struct in_state            * state_in;
     struct out_state           * state_out;
     const struct decode_opts   * decode_opt;
     charset_t                    text_charset;
     struct tagfilter_selection * tagfilter;
{
    if (TAGFILTER_SELECTION_magic != tagfilter->magic) 
	mime_panic(__FILE__,__LINE__,"push_tag_text_enriched",
		   "Bad magic number (tagfilter_selection)");
    
    
    if (TAGFILTER_TAG_STATE_magic !=  tag_state->magic)
	mime_panic(__FILE__,__LINE__,"push_tag_text_enriched",
		   "Bad magic number (tagfilter_tag_state)");


    /* No implicit stack items now */

    return NULL;
}

E_(close_stack_item_f close_tag_text_enriched)
int close_tag_text_enriched P_((struct tagfilter_tag_state   * tag_state,
				struct tagfilter_stack_item ** stack,
				size_t                         stack_len,
				size_t                         idx,
				enum stack_close_mode          close_pos,
				mime_t                       * body,
				struct in_state              * state_in,
				struct out_state             * state_out,
				charset_t                      text_charset,
				struct tagfilter_selection   * tagfilter,
				struct tagfilter_global      * counter));
int close_tag_text_enriched(tag_state,stack,stack_len,idx,close_pos,
			    body,state_in,state_out,text_charset,
			    tagfilter,counter)
     struct tagfilter_tag_state *tag_state;
     struct tagfilter_stack_item **stack;
     size_t                        stack_len;
     size_t                        idx;
     enum stack_close_mode         close_pos;
     mime_t                     * body;
     struct in_state            * state_in;
     struct out_state           * state_out;
     charset_t                    text_charset;
     struct tagfilter_selection * tagfilter;
     struct tagfilter_global    * counter;

{

    /* tagfilter_free_stack_item() is called by caller -
       return 1 (or true) if item need to be free'ed
    */

    if (TAGFILTER_SELECTION_magic != tagfilter->magic) 
	mime_panic(__FILE__,__LINE__,"close_tag_text_enriched",
		   "Bad magic number (tagfilter_selection)");
    
    
    if (TAGFILTER_TAG_STATE_magic !=  tag_state->magic)
	mime_panic(__FILE__,__LINE__,"close_tag_text_enriched",
		   "Bad magic number (tagfilter_tag_state)");


    /* Nothing to do */
    
    return 1;
}

E_(start_attribute_f start_atr_text_enriched)
struct  tagfilter_atr_state *
   start_atr_text_enriched P_((struct string              * atr_name,
			       struct tagfilter_tag_state * tag_state,
			       mime_t                     * body,
			       struct in_state            * state_in,
			       struct out_state           * state_out,
			       charset_t                    text_charset,
			       struct tagfilter_selection * tagfilter));
struct  tagfilter_atr_state *start_atr_text_enriched(atr_name,tag_state,
						     body,state_in,
						     state_out,
						     text_charset,
						     tagfilter)
     struct string              * atr_name;
     struct tagfilter_tag_state * tag_state;
     mime_t                     * body;
     struct in_state            * state_in;
     struct out_state           * state_out;
     charset_t                    text_charset;
     struct tagfilter_selection * tagfilter;
{
    
    if (TAGFILTER_SELECTION_magic != tagfilter->magic) 
	mime_panic(__FILE__,__LINE__,"start_atr_text_enriched",
		   "Bad magic number (tagfilter_selection)");
        
    if (TAGFILTER_TAG_STATE_magic !=  tag_state->magic)
	mime_panic(__FILE__,__LINE__,"start_atr_text_enriched",
		   "Bad magic number (tagfilter_tag_state)");

    
    /* text/enriched do not have attributes */

    return NULL;
}

E_(end_attribute_f end_atr_text_enriched)
void end_atr_text_enriched P_((struct tagfilter_atr_state **atr_state,
			       struct tagfilter_tag_state * tag_state,
			       mime_t                     * body,
			       struct in_state            * state_in,
			       struct out_state           * state_out,
			       charset_t                    text_charset,
			       struct tagfilter_selection * tagfilter));
void end_atr_text_enriched(atr_state,tag_state,body,state_in,state_out,
			   text_charset,tagfilter)
     struct tagfilter_atr_state **atr_state;
     struct tagfilter_tag_state * tag_state;
     mime_t                     * body;
     struct in_state            * state_in;
     struct out_state           * state_out;
     charset_t                    text_charset;
     struct tagfilter_selection * tagfilter;
{

    if (TAGFILTER_SELECTION_magic != tagfilter->magic) 
	mime_panic(__FILE__,__LINE__,"end_atr_text_enriched",
		   "Bad magic number (tagfilter_selection)");
        
    if (TAGFILTER_TAG_STATE_magic !=  tag_state->magic)
	mime_panic(__FILE__,__LINE__,"end_atr_text_enriched",
		   "Bad magic number (tagfilter_tag_state)");

    
    /* text/enriched do not have attributes */


    free(atr_state);
    *atr_state = NULL;
}

E_(start_end_atrvalue_f start_end_atrval_text_enriched)
void start_end_atrval_text_enriched
         P_((struct tagfilter_atr_state * atr_state,
	     enum atrvalue_start_end      start_end,
	     struct tagfilter_tag_state * tag_state,
	     mime_t                     * body,
	     struct in_state            * state_in,
	     struct out_state           * state_out,
	     charset_t                    text_charset,
	     struct tagfilter_selection * tagfilter));
void start_end_atrval_text_enriched(atr_state,start_end,tag_state,
				    body,state_in,state_out,
				    text_charset,tagfilter)
     struct tagfilter_atr_state * atr_state;
     enum atrvalue_start_end      start_end;
     struct tagfilter_tag_state * tag_state;
     mime_t                     * body;
     struct in_state            * state_in;
     struct out_state           * state_out;
     charset_t                    text_charset;
     struct tagfilter_selection * tagfilter;
{

    if (TAGFILTER_SELECTION_magic != tagfilter->magic) 
	mime_panic(__FILE__,__LINE__,"start_end_atrval_text_enriched",
		   "Bad magic number (tagfilter_selection)");
        
    if (TAGFILTER_TAG_STATE_magic !=  tag_state->magic)
	mime_panic(__FILE__,__LINE__,"start_end_atrval_text_enriched",
		   "Bad magic number (tagfilter_tag_state)");

    
    /* text/enriched do not have attributes */

}

E_(atrvalue_seg_string_f atrvalue_seg_str_text_enriched)
void atrvalue_seg_str_text_enriched
   P_((struct tagfilter_atr_state * atr_state,
       struct string              * atr_value_segment,
       struct tagfilter_tag_state * tag_state,
       mime_t                     * body,
       struct in_state            * state_in,
       struct out_state           * state_out,
       charset_t                    text_charset,
       struct tagfilter_selection * tagfilter));
void atrvalue_seg_str_text_enriched(atr_state,atr_value_segment,
				    tag_state,body,state_in,
				    state_out,text_charset,tagfilter)
     struct tagfilter_atr_state * atr_state;
     struct string              * atr_value_segment;
     struct tagfilter_tag_state * tag_state;
     mime_t                     * body;
     struct in_state            * state_in;
     struct out_state           * state_out;
     charset_t                    text_charset;
     struct tagfilter_selection * tagfilter;
{

    if (TAGFILTER_SELECTION_magic != tagfilter->magic) 
	mime_panic(__FILE__,__LINE__,"atrvalue_seg_str_text_enriched",
		   "Bad magic number (tagfilter_selection)");
        
    if (TAGFILTER_TAG_STATE_magic !=  tag_state->magic)
	mime_panic(__FILE__,__LINE__,"atrvalue_seg_str_text_enriched",
		   "Bad magic number (tagfilter_tag_state)");

    /* text/enriched do not have attributes */
}

E_(atrvalue_seg_ent_f atrvalue_seg_ent_text_enriched)
void atrvalue_seg_ent_text_enriched
       P_((struct tagfilter_atr_state * atr_state,
	   struct out_entity          * atr_value_segment,
	   struct tagfilter_tag_state * tag_state,
	   mime_t                     * body,
	   struct in_state            * state_in,
	   struct out_state           * state_out,
	   charset_t                    text_charset,
	   struct tagfilter_selection * tagfilter));
void atrvalue_seg_ent_text_enriched(atr_state,atr_value_segment,
				    tag_state,body,state_in,state_out,
				    text_charset,tagfilter)
     struct tagfilter_atr_state * atr_state;
     struct out_entity          * atr_value_segment;
     struct tagfilter_tag_state * tag_state;
     mime_t                     * body;
     struct in_state            * state_in;
     struct out_state           * state_out;
     charset_t                    text_charset;
     struct tagfilter_selection * tagfilter;
{
    if (TAGFILTER_SELECTION_magic != tagfilter->magic) 
	mime_panic(__FILE__,__LINE__,"atrvalue_seg_ent_text_enriched",
		   "Bad magic number (tagfilter_selection)");
        
    if (TAGFILTER_TAG_STATE_magic !=  tag_state->magic)
	mime_panic(__FILE__,__LINE__,"atrvalue_seg_ent_text_enriched",
		   "Bad magic number (tagfilter_tag_state)");

    /* text/enriched do not have attributes */
}

E_(stack_item_clear_f stack_item_clear_text_enriched)
void stack_item_clear_text_enriched
  P_((struct tagfilter_stack_item * item,
      struct tagfilter_selection  * tagfilter));
void stack_item_clear_text_enriched(item,tagfilter)
     struct tagfilter_stack_item * item;
     struct tagfilter_selection  * tagfilter;
{
    if (TAGFILTER_STACK_ITEM_magic != item->magic)
	mime_panic(__FILE__,__LINE__,
		   "stack_item_clear_text_enriched",
		   "Bad magic number (tagfilter_stack_item)");

    if (TAGFILTER_SELECTION_magic != tagfilter->magic) 
	mime_panic(__FILE__,__LINE__,"stack_item_clear_text_enriched",
		   "Bad magic number (tagfilter_selection)");

    /* Nothing to clear */
        
}


/* html ------------------------------------ */

#define TAGFILTER_HTML_STATE_magic	0xFD19

static struct tagfilter_html_state {
    unsigned short            magic; 

    struct tagfilter_push_tag {
	struct tagfilter_cmd   * cmd;
	struct tagfilter_tags  * used_tags; /* static */
    } * push_tag;
    
    int                        push_tag_len;

    struct tagfilter_stack_html_item * current_stack;
    
    int     ol_start_value;

    struct tagfilter_stack_html_item * ol_stack;
    int     li_value;  

    enum ol_list_type {
	atr_type_none    = 0,   /* no value given */
	atr_type_decimal = '1', /* decimal */
	atr_type_lalpha  = 'a', /* lower-alpha */
	atr_type_ualpha  = 'A',  /* upper-alpha */
	atr_type_lroman  = 'i',  /* lower-roman */
	atr_type_uroman  = 'I'   /* upper-roman */
    }  ol_type_value;
    
    unsigned int   have_ol_start      :1;
    unsigned int   duplicate_ol_start :1;

    unsigned int   have_li_value      :1;
    unsigned int   duplicate_li_value :1;
    unsigned int   default_li_value   :1;
    unsigned int   bad_li_value       :1;

    unsigned int   duplicate_ol_type  :1;
    unsigned int   seen_img_alt       :1;
    
} * tagfilter_new_html_state P_((void));
static struct tagfilter_html_state * tagfilter_new_html_state()
{
    struct tagfilter_html_state * ret =
	safe_zero_alloc(sizeof (*ret));

    ret->push_tag = NULL;
    ret->push_tag_len = 0;

    ret->current_stack = NULL;
    
    ret->ol_start_value = 0;
    ret->have_ol_start  = 0;
    ret->duplicate_ol_start = 0;

    ret->ol_stack      = NULL;
    ret->li_value      = 0;
    ret->ol_type_value = atr_type_none;
    ret->have_li_value = 0;
    ret->duplicate_li_value = 0;
    ret->default_li_value = 0;
    ret->bad_li_value     = 0;
    ret->duplicate_ol_type = 0;
    ret->seen_img_alt      = 0;
        
    ret->magic = TAGFILTER_HTML_STATE_magic;
    return ret;
}

static void tagfilter_free_stack_html_item P_((struct tagfilter_stack_html_item **ptr));

static void tagfilter_free_html_state P_((struct tagfilter_html_state **ptr));
static void tagfilter_free_html_state(ptr)
     struct tagfilter_html_state **ptr;
{
    if (TAGFILTER_HTML_STATE_magic != (*ptr)->magic) 
	mime_panic(__FILE__,__LINE__,"tagfilter_free_html_state",
		   "Bad magic number (tagfilter_selection)");

    if ((*ptr)->push_tag) {
	int i;

	for (i = 0; i < (*ptr)->push_tag_len; i++) {
	    if ((*ptr)->push_tag[i].cmd)
		tagfilter_release_cmd(& ((*ptr)->push_tag[i].cmd));
	    (*ptr)->push_tag[i].used_tags = 0;
	}

	free((*ptr)->push_tag);
	(*ptr)->push_tag = NULL;
    }
    (*ptr)->push_tag_len = 0;

    if ((*ptr)->current_stack)
	tagfilter_free_stack_html_item(& ((*ptr)->current_stack));

    if ((*ptr)->ol_stack)
	tagfilter_free_stack_html_item(& ((*ptr)->ol_stack));

    free(*ptr);
    *ptr = NULL;
}

static void tagfilter_html_state_push_command
  P_((struct tagfilter_html_state * html_state,
      struct tagfilter_cmd        * cmd,
      struct tagfilter_tags       * used_tags
      ));
static void tagfilter_html_state_push_command(html_state,cmd,used_tags)
     struct tagfilter_html_state * html_state;
     struct tagfilter_cmd        * cmd;
     struct tagfilter_tags       * used_tags;
{
    if (TAGFILTER_HTML_STATE_magic != html_state->magic) 
	mime_panic(__FILE__,__LINE__,
		   "tagfilter_html_state_push_command",
		   "Bad magic number (tagfilter_selection)");

    if (TAGFILTER_CMD_magic != cmd->magic)
	mime_panic(__FILE__,__LINE__,
		   "tagfilter_html_state_push_command",
		   "Bad magic number (tagfilter_cmd)");

    html_state->push_tag =
	safe_array_realloc(html_state->push_tag,
			   html_state->push_tag_len+1,
			   sizeof (html_state->push_tag[0]));

    html_state->push_tag[html_state->push_tag_len].cmd = cmd;
    tagfilter_inc_cmd_refcount(html_state->
			       push_tag[html_state->push_tag_len].cmd);
    html_state->push_tag[html_state->push_tag_len].used_tags =
	used_tags;
    
    html_state->push_tag_len++;
}

struct tagfilter_tags text_html_top = {    /* <html> .... </html>  */
    TAGFILTER_TAGS_magic,

    "text/html document root",

    &(tagfilter_html_top[0]),
    NULL
};
struct tagfilter_tags text_html_head_body = {
    TAGFILTER_TAGS_magic,

    "text/html head and body",
    
    &(tagfilter_html_head_body[0]),
    NULL
};
struct tagfilter_tags text_html_metadata = {
    TAGFILTER_TAGS_magic,

    "text/html metadata",
    
    &(tagfilter_html_metadata[0]),
    NULL
};
struct tagfilter_tags text_html_flow = {
    TAGFILTER_TAGS_magic,

    "text/html flow content",
    
    &(tagfilter_html_flow[0]),
    NULL
};

struct tagfilter_tags text_html_phrasing = {
    TAGFILTER_TAGS_magic,

    "text/html phrasing content",
    
    &(tagfilter_html_phrasing[0]),
    NULL
};


struct tagfilter_tags text_html_close_head = {
    /* Not really working */
    TAGFILTER_TAGS_magic,

    "text/html closing tags for <head>",
    
    &(tagfilter_html_close_head[0]),
    NULL
};

struct tagfilter_tags text_html_noscript_head = {
    TAGFILTER_TAGS_magic,

    "text/html content of <noscript> on <head>",
    
    &(tagfilter_html_noscript_head[0]),
    NULL
};

struct tagfilter_tags text_html_datalist = {
    TAGFILTER_TAGS_magic,

    "text/html content of <datalist>",
    &(tagfilter_html_datalist[0]),
    NULL
};

struct tagfilter_tags text_html_details = {
    TAGFILTER_TAGS_magic,
    "text/html content of <details>",
    &(tagfilter_html_details[0]),
    NULL
};

struct tagfilter_tags text_html_dl = {
    TAGFILTER_TAGS_magic,
    "text/html content of <dl>",
    &(tagfilter_html_dl[0]),
    NULL
};

struct tagfilter_tags text_html_fieldset = {
    TAGFILTER_TAGS_magic,
    "text/html content of <fieldset>",
    &(tagfilter_html_fieldset[0]),
    NULL
};

struct tagfilter_tags text_html_figure = {
    TAGFILTER_TAGS_magic,
    "text/html content of <figure>",
    &(tagfilter_html_figure[0]),
    NULL
};

struct tagfilter_tags text_html_footer = {
    TAGFILTER_TAGS_magic,
    "text/html content of <footer>",
    &(tagfilter_html_footer[0]),
    NULL
};

struct tagfilter_tags text_html_form = {
    TAGFILTER_TAGS_magic,
    "text/html content of <form>",
    &(tagfilter_html_form[0]),
    NULL
};

struct tagfilter_tags text_html_hgroup = {
    TAGFILTER_TAGS_magic,
    "text/html content of <hgroup>",
    &(tagfilter_html_hgroup[0]),
    NULL
};

struct tagfilter_tags text_html_label = {
    TAGFILTER_TAGS_magic,
    "text/html content of <label>",
    &(tagfilter_html_label[0]),
    NULL
};

struct tagfilter_tags text_html_list_items = {
    TAGFILTER_TAGS_magic,
    "text/html list items",
    &(tagfilter_html_list_items[0]),
    NULL
};

struct tagfilter_tags text_html_close_li = {
    TAGFILTER_TAGS_magic,
    "text/html closing tags for <li>",
    &(tagfilter_html_close_li[0]),
    NULL
};

struct tagfilter_tags text_html_close_p = {
    TAGFILTER_TAGS_magic,
    "text/html closing tags for <p>",
    &(tagfilter_html_close_p[0]),
    NULL
};

struct tagfilter_tags text_html_address = {
    TAGFILTER_TAGS_magic,
    "text/html content of <address>",
    &(tagfilter_html_address[0]),
    NULL
};

struct tagfilter_tags text_html_meter = {
    TAGFILTER_TAGS_magic,
    "text/html content of <meter>",
    &(tagfilter_html_meter[0]),
    NULL
};

struct tagfilter_tags text_html_picture = {
    TAGFILTER_TAGS_magic,
    "text/html content of <picture>",
    &(tagfilter_html_picture[0]),
    NULL
};

struct tagfilter_tags text_html_progress = {
    TAGFILTER_TAGS_magic,
    "text/html content of <progress>",
    &(tagfilter_html_progress[0]),
    NULL
};

struct tagfilter_tags text_html_ruby = {
    TAGFILTER_TAGS_magic,
    "text/html content of <ruby>",
    &(tagfilter_html_ruby[0]),
    NULL
};

struct tagfilter_tags text_html_table = {
    TAGFILTER_TAGS_magic,
    "text/html content of <table>",
    &(tagfilter_html_table[0]),
    NULL
};

struct tagfilter_tags text_html_div_dl = {
    TAGFILTER_TAGS_magic,
    "text/html content of <div> on <dl>",
    &(tagfilter_html_div_dl[0]),
    NULL
};

struct tagfilter_tags text_html_caption = {
    TAGFILTER_TAGS_magic,
    "text/html content of <caption>",
    &(tagfilter_html_caption[0]),
    NULL
};

struct tagfilter_tags text_html_colgroup = {
    TAGFILTER_TAGS_magic,
    "text/html content of <colgroup>",
    &(tagfilter_html_colgroup[0]),
    NULL
};

struct tagfilter_tags text_html_tr_list = {
    TAGFILTER_TAGS_magic,
    "text/html <tr> items",
    &(tagfilter_html_tr_list[0]),
    NULL
};

struct tagfilter_tags text_html_tr = {
    TAGFILTER_TAGS_magic,
    "text/html content of <tr>",
    &(tagfilter_html_tr[0]),
    NULL
};

struct tagfilter_tags text_html_close_thead = {
    TAGFILTER_TAGS_magic,
    "text/html closing tags for <thead>",
    &(tagfilter_html_close_thead[0]),
    NULL
};

struct tagfilter_tags text_html_close_tbody = {
    TAGFILTER_TAGS_magic,
    "text/html closing tags for <tbody>",
    &(tagfilter_html_close_tbody[0]),
    NULL
};

struct tagfilter_tags text_html_close_tr = {
    TAGFILTER_TAGS_magic,
    "text/html closing tags for <tr>",
    &(tagfilter_html_close_tr[0]),
    NULL
};

struct tagfilter_tags text_html_close_td = {
    TAGFILTER_TAGS_magic,
    "text/html closing tags for <td>",
    &(tagfilter_html_close_td[0]),
    NULL
};

struct tagfilter_tags text_html_th = {
    TAGFILTER_TAGS_magic,
    "text/html content of <th>",
    &(tagfilter_html_th[0]),
    NULL
};

struct tagfilter_tags text_html_close_th = {
    TAGFILTER_TAGS_magic,
    "text/html closing tags for <th>",
    &(tagfilter_html_close_th[0]),
    NULL
};

struct tagfilter_tags text_html_close_dtdd = {
    TAGFILTER_TAGS_magic,
    "text/html closing tags for <dt> and <dd>",
    &(tagfilter_html_close_dtdd[0]),
    NULL
};

struct tagfilter_tags text_html_close_rtrp = {
    TAGFILTER_TAGS_magic,
    "text/html closing tags for <rt> and <rp>",
    &(tagfilter_html_close_rtrp[0]),
    NULL
};

E_(give_top_stack_item_f give_top_text_html)
struct tagfilter_stack_item * give_top_text_html
  P_((mime_t                     * body,
      struct in_state            * state_in,
      struct out_state           * state_out,
      const struct decode_opts   * decode_opt,
      charset_t                    text_charset,
      struct tagfilter_selection * tagfilter));
struct tagfilter_stack_item * give_top_text_html(body,state_in,state_out,
						 decode_opt,text_charset,
						 tagfilter)
     mime_t                     * body;
     struct in_state            * state_in;
     struct out_state           * state_out;
     const struct decode_opts   * decode_opt;
     charset_t                    text_charset;
     struct tagfilter_selection * tagfilter;
{
    struct tagfilter_stack_item * ret = NULL;
    struct pager_range          * html_body_range = NULL;

    union tagfilter_stack_uitem uitem;
    
    /* What is good settings ? */

    uitem.magic = NULL;
    
    if (MIME_magic != body->magic)
	mime_panic(__FILE__,__LINE__,"give_top_text_html",
		   "Bad magic number (mime_t)");

    if (decode_opt)
	html_body_range =
	    state_add_simple_pager_range(state_out,
					 /* Handles quote prefix */
					 decode_opt->range,
					 PR_WORD_WRAP|PR_JOIN_LINES|
					 PR_ADD_SPACE,0,0);
    else
	html_body_range =
	    state_add_simple_pager_range(state_out,
					 NULL,
					 PR_WORD_WRAP|PR_JOIN_LINES|
					 PR_ADD_SPACE,0,0);

    ret =  tagfilter_new_stack_item(html_body_range,
				    0,
				    &text_html_top,
				    NULL,uitem);

    if (TAGFILTER_STACK_ITEM_magic != ret->magic)
	mime_panic(__FILE__,__LINE__,"give_top_text_html",
		   "Bad magic number (tagfilter_stack_item)");

    free_pager_range(&html_body_range);
    
    return ret;
}

#define TAGFILTER_STACK_HTML_ITEM_magic	0xFD1A

static struct tagfilter_stack_html_item {
    unsigned short               magic;     /* TAGFILTER_STACK_HTML_ITEM_magic */

    int                          refcount;  /* Referenced from 
					       struct tagfilter_html_state */
    
      /* text/html  list ul, ol  */
    unsigned int        list_depth;         /* count of nested lists such as
					       <ul> <ol>
					    */
    
    int        next_item_number;            /* item number  for next <li> on <ol> */
    enum ol_list_type type_item_number;     /* print format for next <li> on <ol> */
    
    unsigned int    reversed_item_number:1; /* boolean -- not supported 
					       without start value   */
    
    unsigned int    item_number_set:1;      /*  next_item_number is not default */
    
} * tagfilter_give_stack_html_item P_((struct tagfilter_stack_item *item)),
  * tagfilter_new_stack_html_item P_((void));


static void tagfilter_inc_stack_html_item_refcount P_((struct tagfilter_stack_html_item *ptr));

static void tagfilter_add_html_stack_item P_((struct tagfilter_tag_state *ret,
					      struct tagfilter_stack_item *stack_item));

static void tagfilter_inc_stack_html_item_refcount(ptr)
     struct tagfilter_stack_html_item *ptr;

{
    if (TAGFILTER_STACK_HTML_ITEM_magic != ptr->magic)
	mime_panic(__FILE__,__LINE__,
		   "tagfilter_inc_stack_html_item_refcount",
		   "Bad magic number (tagfilter_stack_html_item)");

    ptr->refcount++;
}
    

static void tagfilter_add_html_stack_item(ret,stack_item)
     struct tagfilter_tag_state *ret;
     struct tagfilter_stack_item *stack_item;
{
    
    if (ret) {
	if (TAGFILTER_TAG_STATE_magic != ret->magic)
	    mime_panic(__FILE__,__LINE__,"tagfilter_add_html_stack_item",
		       "Bad magic number (tagfilter_tag_state)");
	
	if (!ret->u.html)
	    ret->u.html = tagfilter_new_html_state();

	if (TAGFILTER_HTML_STATE_magic != ret->u.html->magic) 
	    mime_panic(__FILE__,__LINE__,"tagfilter_add_html_stack_item",
		       "Bad magic number (tagfilter_html_state)");
	
	if (ret->u.html->current_stack)
	    tagfilter_free_stack_html_item(& (ret->u.html->current_stack));
	
	if (stack_item) {
	    struct tagfilter_stack_html_item * html_item =
		tagfilter_give_stack_html_item(stack_item);
	    
	    ret->u.html->current_stack =  html_item;
	    tagfilter_inc_stack_html_item_refcount(ret->u.html->current_stack);
	}
    }    
}

static struct pager_range *new_html_item_hlp1
   P_((struct tagfilter_cmd       * current_tag,
       struct pager_range         * prev_range,
       struct out_state           * state_out,
       int                        * text_visible_p,
       int                          inherit_pg_flags,
       struct tagfilter_stack_item **stack,
       size_t                        stack_len,
       int                         * unsupport));
static struct pager_range *
    new_html_item_hlp1(current_tag,
		       prev_range,
		       state_out,
		       text_visible_p,
		       inherit_pg_flags,
		       stack,stack_len,
		       unsupport)
     struct tagfilter_cmd         * current_tag;
     struct pager_range           * prev_range;
     struct out_state             * state_out;
     int                          * text_visible_p;
     int                            inherit_pg_flags;
     struct tagfilter_stack_item ** stack;
     size_t                         stack_len;
     int                          * unsupport;
{
    struct pager_range * new_range = NULL;
    int  text_visible  = * text_visible_p;

    if (TAGFILTER_CMD_magic != current_tag->magic)
	mime_panic(__FILE__,__LINE__,
		   "new_html_item_hlp1",
		   "Bad magic number (tagfilter_cmd)");

    if (text_visible && ison(current_tag->cmd_mode,
			     TAGFLT_CMD_hide_text))
	text_visible = 0;
    else if (ison(current_tag->cmd_mode,
		  TAGFLT_CMD_show_text))
	text_visible = 1;

    
    switch (current_tag->code.html) {
	int i;
    case tfc_html_unsupported:

	if (unsupport)
	    *unsupport = 1;
	break;
	
    case tfc_html_none:     break;	
    case tfc_html_br: {
	int prev_flags = 0;
		
	if (prev_range &&
	    (prev_flags = get_pager_range_flags(prev_range)) &&
	    ison(prev_flags,PR_JOIN_LINES)) {

	    /* New range without flags */

	    new_range =
		state_add_simple_pager_range(state_out,
					     prev_range,
					     PR_FLAGS_ABSOLUTE,
					     0,0);


	}
	
	/* Re-set pg_flags */
	set_out_state_line_mode(state_out,inherit_pg_flags,
				new_range ? new_range : prev_range,
				1 /* Newline */);

	state_putc('\n',state_out);

    }
	break;
	
    case tfc_html_hr:
	state_putc('\n',state_out); 

	new_range =
	    state_add_simple_pager_range(
			state_out,
			prev_range,					     /* Not very good */
		        PR_CENTER_THIS|PR_FLAGS_ABSOLUTE, 
		        0 /* quote level */,
		        0 /* indent */);
	
	/* Re-set pg_flags */
	set_out_state_line_mode(state_out,inherit_pg_flags,
				new_range,0 );


	for (i = 0; i < 10; i++)
	    state_putc('-',state_out); 

	break;

    case tfc_html_p:
	state_putc('\n',state_out);

	new_range =
	    state_add_simple_pager_range(state_out,
					 prev_range,
					 PR_WORD_WRAP|PR_JOIN_LINES|
					 PR_ADD_SPACE,
					 0 /* quote level */,
					 0 /* indent */);
	
	/* Re-set pg_flags */
	set_out_state_line_mode(state_out,inherit_pg_flags,
				new_range,0 );
	break;

    case tfc_html_pre: {
	/* Calculate absolute range flags */
	int range_flags = PR_FLAGS_ABSOLUTE;
		
	if (prev_range)
	    setit(range_flags,get_pager_range_flags(prev_range));
	clearit(range_flags,PR_JOIN_LINES);
	clearit(range_flags,PR_ADD_SPACE);
	clearit(range_flags,PR_WORD_WRAP);
	clearit(range_flags,PR_SAME_LINE);

	new_range = state_add_simple_pager_range(state_out,
						 prev_range,
						 range_flags,
						 0,0);
	/* Re-set pg_flags */
	set_out_state_line_mode(state_out,inherit_pg_flags,
				new_range,0 );

    }
	break;
    case tfc_html_q:
	state_putc('"',state_out);
	break;
    case tfc_html_li: 
	
	new_range = state_add_simple_pager_range(state_out,
						 prev_range,
						 0,
						 0,0);
	set_out_state_line_mode(state_out,inherit_pg_flags,
				new_range,0 );


    
	break;

    case tfc_html_ul:
    case tfc_html_ol:
	
	new_range = state_add_simple_pager_range(state_out,
						 prev_range,
						 /* Try avoid indent given on source */
						 PR_COLLAPSE_NOINHERIT,
						 0,
						 /* Indent content 1 character */
						 1);

	set_out_state_line_mode(state_out,inherit_pg_flags,
				new_range,0 );

	break;
    case tfc_html_img:
	break;
    }
   
    * text_visible_p = text_visible;

    if (new_range) {
	DPRINT(Debug,21,(&Debug,
			 "new_html_item_hlp1=%p (new_range)"));
    }
    
    return new_range;
}

/* Parameters need check when used */
static void new_html_item_hlp2 P_((struct tagfilter_cmd       * current_tag,
				   struct pager_range         * new_range,
				   struct tagfilter_tag_state * tag_state,
				   struct out_state           * state_out,
				   int                          text_visible,
				   struct tagfilter_selection * tagfilter,
				   struct tagfilter_stack_item * stack_item,
				   struct tagfilter_stack_item **stack,
				   size_t                        stack_len,
				   int                         * unsupport
				   ));
static void new_html_item_hlp2(current_tag,new_range,
			       tag_state,state_out,
			       text_visible,tagfilter,
			       stack_item,stack,stack_len,
			       unsupport)
     struct tagfilter_cmd       * current_tag;
     struct pager_range         * new_range;
     struct tagfilter_tag_state * tag_state;
     struct out_state           * state_out;
     int                          text_visible;
     struct tagfilter_selection * tagfilter;
     struct tagfilter_stack_item * stack_item;
     struct tagfilter_stack_item **stack;
     size_t                        stack_len;
     int                         * unsupport;

{
    if (TAGFILTER_CMD_magic != current_tag->magic)
	mime_panic(__FILE__,__LINE__,
		   "new_html_item_hlp2",
		   "Bad magic number (tagfilter_cmd)");

    switch (current_tag->code.html) {
	int               next_item_number;
	enum ol_list_type type_item_number;
	
    case tfc_html_unsupported:
    case tfc_html_none:         break;      
    case tfc_html_br:           break;
    case tfc_html_hr:           break;
	
    case tfc_html_p:            break;
    case tfc_html_pre:          break;
    case tfc_html_q:            break;
    case tfc_html_li:

      	if (stack && stack_len > 0) {
	    size_t                 idx = stack_len -1;
	    unsigned int           list_depth = 0;

	    enum  tagfilter_cmd_html list_code = 0;

	    next_item_number = 0;  /* for <ol> <li> */
	    type_item_number = atr_type_none;
	    
	    DPRINT(Debug,21,(&Debug,
			     "new_html_item_hlp2: tfc_html_li: stack_len = %zu\n",
			     stack_len));

	    /* Only check last on stack */

	    if (TAGFILTER_STACK_ITEM_magic !=  stack[idx]->magic)
		mime_panic(__FILE__,__LINE__,"new_html_item_hlp2",
			   "Bad magic number (tagfilter_stack_item,stack[idx])");
	    
	    if (stack[idx]->u.html) {

		struct tagfilter_stack_html_item * html_item;
		char * cmd = NULL;

		int bad = 0;
		
		union tagfilter_cmd_code  cmd_code;


		
		cmd_code.code = 0;
		
		if (TAGFILTER_STACK_HTML_ITEM_magic != stack[idx]->u.html->magic)
		    mime_panic(__FILE__,__LINE__,
			       "new_html_item_hlp1",
			       "Bad magic number (tagfilter_stack_html_item)");

		html_item = stack[idx]->u.html;

		if (stack[idx]->current_tag) {
		    if (TAGFILTER_CMD_magic != stack[idx]->current_tag->magic)
			mime_panic(__FILE__,__LINE__,"new_html_item_hlp2",
				   "Bad magic number (tagfilter_cmd,stack[idx]->current_tag)");
		    cmd       = stack[idx]->current_tag->ascii_command;
		    cmd_code  = stack[idx]->current_tag->code;
		}
		
		if (html_item->list_depth > 0) {

		    list_depth       = html_item->list_depth;
		    next_item_number = html_item->next_item_number;
		    type_item_number = html_item->type_item_number;
		    list_code        = cmd_code.html;
				    
		    DPRINT(Debug,21,(&Debug,
				     "new_html_item_hlp2: tfc_html_li: #%zu %s list_depth=%u",
				     idx,
				     cmd ? cmd : "",
				     list_depth));
		    if (next_item_number) {
			
			if (html_item->reversed_item_number) {
			    if (html_item->item_number_set)
				html_item->next_item_number--;  /* Decrement for <ol> <li> */
			    else
				bad = 1;
			} else
			    html_item->next_item_number++;  /* Increment for <ol> <li> */
			
			DPRINT(Debug,21,(&Debug," net_item_number=%d",
					 next_item_number));	 
		    }
		    if (list_code) {
			DPRINT(Debug,21,(&Debug," list_code=%d",
					 list_code));
			
			switch (list_code) {
			case tfc_html_unsupported:
			    DPRINT(Debug,21,(&Debug," tfc_html_unsupported")); break;
			case tfc_html_none:
			    DPRINT(Debug,21,(&Debug," tfc_html_none"));        break;
			case tfc_html_br: DPRINT(Debug,21,(&Debug," tfc_html_br")); break;
			case tfc_html_hr: DPRINT(Debug,21,(&Debug," tfc_html_hr")); break;
			case tfc_html_p: DPRINT(Debug,21,(&Debug," tfc_html_p")); break;
			case tfc_html_pre:
			    DPRINT(Debug,21,(&Debug," tfc_html_pre")); break;
			case tfc_html_q: DPRINT(Debug,21,(&Debug," tfc_html_q")); break;
			case tfc_html_li: DPRINT(Debug,21,(&Debug," tfc_html_li")); break;
			case tfc_html_ul: DPRINT(Debug,21,(&Debug," tfc_html_ul")); break;
			case tfc_html_ol: DPRINT(Debug,21,(&Debug," tfc_html_ol")); break; 
			case tfc_html_img:
			    DPRINT(Debug,21,(&Debug," tfc_html_img")); break;
			}
		    }		
		    DPRINT(Debug,21,(&Debug,"\n"));
		
		    switch (list_code) {
			
		    case tfc_html_ul:
			
			switch (list_depth) {
			case 0:
			    break;
			case 1:
			    state_puts("* ",state_out);
			    break;
			case 2:
			    state_puts("+ ",state_out);
			    break;
			default:
			    state_puts("- ",state_out);
			    break;
			}
						
			break;

		    case tfc_html_ol:
			/* need formatting */

			if (tag_state) {			    
			    if (TAGFILTER_TAG_STATE_magic != tag_state->magic)
				mime_panic(__FILE__,__LINE__,"new_html_item_hlp2",
					   "Bad magic number (tagfilter_tag_state)");

			    
			    if (!tag_state->u.html)
				tag_state->u.html = tagfilter_new_html_state();

			    if (TAGFILTER_HTML_STATE_magic != tag_state->u.html->magic) 
				mime_panic(__FILE__,__LINE__,"new_html_item_hlp2",
					   "Bad magic number (tagfilter_html_state)");

			    if (tag_state->u.html->ol_stack)
				tagfilter_free_stack_html_item(&  (tag_state->u.html->ol_stack));

			    
			    tag_state->u.html->ol_stack =  html_item;
			    tagfilter_inc_stack_html_item_refcount(tag_state->u.html->ol_stack);
			    if (! bad) {
				tag_state->u.html->li_value      = next_item_number;
				tag_state->u.html->ol_type_value = type_item_number;
				tag_state->u.html->default_li_value = 1;
			    } else {
				tag_state->u.html->bad_li_value     = 1;
			    }
			    
			} else {
			    goto html_li_fail;
			}
			    
			break;
			
		    default:
		    case tfc_html_unsupported:
			/* Not supported list type */
		    goto html_li_fail;
		    }
		} else 
		    goto html_li_fail;	   		    
	    } else 
		goto html_li_fail;	    	    
	} else {
	html_li_fail:
	    if (unsupport) {
		DPRINT(Debug,21,(&Debug,
				 "new_html_item_hlp2: tfc_html_li: mark unsupported\n"));
	    

		*unsupport = 1;
	    }

	}

      break;
    case tfc_html_ul:
	next_item_number = 0;
	type_item_number = atr_type_none;
	
	if (next_item_number) {
	case tfc_html_ol:
	    next_item_number = 1;
	    type_item_number = atr_type_decimal;
	}
	
	if (stack_item) {

	    unsigned int        list_depth = 0;
	    struct tagfilter_stack_html_item * html_item2;
	    
	    if (TAGFILTER_STACK_ITEM_magic != stack_item->magic)
		mime_panic(__FILE__,__LINE__,"new_html_item_hlp2",
			   "Bad magic number (tagfilter_stack_item,stack_item)");

	    /* stack_item  is not yet added to stack */

	    if (stack && stack_len) {
		size_t               idx = stack_len;

		DPRINT(Debug,21,(&Debug,
				 "new_html_item_hlp2: tfc_html_ul/ol: stack_len = %zu\n",
				 stack_len));
		
		while (idx > 0) {
		    char * cmd = NULL;

		    
		    idx --;
		    
		    if (TAGFILTER_STACK_ITEM_magic !=  stack[idx]->magic)
			mime_panic(__FILE__,__LINE__,"new_html_item_hlp2",
				   "Bad magic number (tagfilter_stack_item,stack[idx])");

		    if (stack[idx]->current_tag) {
			if (TAGFILTER_CMD_magic != stack[idx]->current_tag->magic)
			    mime_panic(__FILE__,__LINE__,"new_html_item_hlp2",
				       "Bad magic number (tagfilter_cmd,stack[idx]->current_tag)");
			cmd = stack[idx]->current_tag->ascii_command;
		    }

		    if (stack[idx]->u.html) {
			struct tagfilter_stack_html_item * html_item;
			
			if (TAGFILTER_STACK_HTML_ITEM_magic != stack[idx]->u.html->magic)
			    mime_panic(__FILE__,__LINE__,
				       "new_html_item_hlp2",
				       "Bad magic number (tagfilter_stack_html_item)");

			html_item = stack[idx]->u.html;

			if (html_item->list_depth > 0) {
			    
			    list_depth = html_item->list_depth;
			    
			    DPRINT(Debug,21,(&Debug,
					     "new_html_item_hlp2: tfc_html_ul/ol: #%zu %s list_depth=%u\n",
					     idx,
					     cmd ? cmd : "",
					     list_depth));
			    break;
			}		    
		    }
		}
	    }
	    
	    list_depth++;

	    html_item2 =  tagfilter_give_stack_html_item(stack_item);

	    if (TAGFILTER_STACK_HTML_ITEM_magic != html_item2->magic)
		mime_panic(__FILE__,__LINE__,
			   "new_html_item_hlp2",
			   "Bad magic number (tagfilter_stack_html_item)");

	    
	    DPRINT(Debug,21,(&Debug,
			     "new_html_item_hlp2: tfc_html_ul/ol: Setting list_depth %u\n",
			     list_depth));
	    
	    html_item2->list_depth = list_depth;

	    if (next_item_number) {

		DPRINT(Debug,21,(&Debug,
				 "new_html_item_hlp2: tfc_html_ul/ol: Setting next_item_number %u\n",
				 next_item_number));
		
		html_item2->next_item_number = next_item_number;
		/* Default value -- with reversed should be number of
		   list items -- that is not supported
		 */
	    }

	    if (type_item_number) {
		html_item2->type_item_number = type_item_number;
	    }
		    
	} else {
	     DPRINT(Debug,21,(&Debug,
			      "new_html_item_hlp2: tfc_html_ul/ol: no current stack_item\n"));
	}
	
	break;
	
    case tfc_html_img:
	break;	
    }

}



static struct tagfilter_stack_item * process_new_html_item2
  P_((struct tagfilter_cmd       * current_tag,
      struct pager_range         * prev_range,
      struct tagfilter_tag_state * current_state,
      struct out_state           * state_out,
      struct tagfilter_selection * tagfilter,
      int                          is_last,
      struct tagfilter_tags      * used_tags,
      int                          inherit_pg_flags,
      int                          text_visible,
      const struct decode_opts   * decode_opt,
      struct tagfilter_stack_item **stack,
      size_t                        stack_len
      ));
static struct tagfilter_stack_item *
        process_new_html_item2(current_tag,prev_range,
			       current_state,
			       state_out,
			       tagfilter,is_last,
			       used_tags,
			       inherit_pg_flags,
			       text_visible,
			       decode_opt,
			       stack,stack_len
			       )
     struct tagfilter_cmd       * current_tag;
     struct pager_range         * prev_range;
     struct tagfilter_tag_state * current_state;
     struct out_state           * state_out;
     struct tagfilter_selection * tagfilter;
     int                          is_last;
     struct tagfilter_tags      * used_tags;
     int                          inherit_pg_flags;
     int                          text_visible;
     const struct decode_opts   * decode_opt;
     struct tagfilter_stack_item **stack;
     size_t                        stack_len;
{

    struct tagfilter_stack_item * ret = NULL;
    struct pager_range * new_range = NULL;

    union tagfilter_stack_uitem uitem;
    
    if (TAGFILTER_CMD_magic != current_tag->magic)
	mime_panic(__FILE__,__LINE__,"process_new_html_item2",
		   "Bad magic number (tagfilter_cmd)");

    if (current_tag->ascii_command) {
	DPRINT(Debug,21,(&Debug,
			 "process_new_html_item2: command %s%s\n",
			 current_tag->ascii_command,
			 is_last ? ", is last" : ""));
    }
    
    uitem.html = tagfilter_new_stack_html_item();

    if (TAGFILTER_STACK_HTML_ITEM_magic != uitem.html->magic)
	mime_panic(__FILE__,__LINE__,
		   "process_new_html_item2",
		   "Bad magic number (tagfilter_stack_html_item)");

    
    new_range = new_html_item_hlp1(current_tag,
				   prev_range,
				   state_out,
				   &text_visible,
				   inherit_pg_flags,
				   stack,stack_len,
				   NULL);
   
     if (!new_range &&
	 ison(current_tag->cmd_mode,TAGFLT_CMD_force_newline)) {
	 
	 DPRINT(Debug,21,(&Debug,
			  "process_new_html_item2: TAGFLT_CMD_force_newline set, forcing new range\n"));
	 
	 new_range = state_add_simple_pager_range(state_out,
						  prev_range,
						  0,0,0);
	 
     }


     DPRINT(Debug,21,(&Debug,
		      "process_new_html_item2: pg_flags %d %s\n",
		      current_tag->pg_flags,
		      give_pg_flags(current_tag->pg_flags)));

     
    ret = tagfilter_new_stack_item(new_range,
				   current_tag->pg_flags,
				   ison(current_tag->cmd_mode,
					TAGFLT_CMD_inherit_nested_tags) ?
				   used_tags  /* Transparent content model */ :
				   NULL /* Use current_tag */,
				   current_tag,uitem);
    uitem.html = NULL;
	
    if (TAGFILTER_STACK_ITEM_magic != ret->magic)
	mime_panic(__FILE__,__LINE__,"process_new_html_item2",
		   "Bad magic number (tagfilter_stack_item)");

    ret->text_visible = text_visible;


    new_html_item_hlp2(current_tag,new_range,
		       current_state,state_out,text_visible,
		       tagfilter,ret,stack,stack_len,
		       NULL);
        
    if (new_range)
	free_pager_range(& new_range);

    return ret;
}

				   
static struct tagfilter_tag_state * process_new_html_item
    P_((struct tagfilter_cmd       * current_tag,
	struct pager_range         * prev_range,
	struct tagfilter_stack_item **new_item,
	struct out_state           * state_out,
	struct tagfilter_global    * counter,
	struct tagfilter_selection * tagfilter,
	struct tagfilter_tags      * tags,
	int                         push_current,
	struct tagfilter_tags      * used_tags,
	int                          inherited_pg_flags,
	int text_visible,
	const struct decode_opts   * decode_opt	,
	struct tagfilter_stack_item **stack,
	size_t                        stack_len));
static struct tagfilter_tag_state *
                process_new_html_item(current_tag,
				      prev_range,
				      new_item,
				      state_out,
				      counter,
				      tagfilter,tags,
				      push_current,
				      used_tags,
				      inherit_pg_flags,
				      text_visible,
				      decode_opt,
				      stack,stack_len
				      )
     struct tagfilter_cmd       * current_tag;
     struct pager_range         * prev_range;
     struct tagfilter_stack_item **new_item;
     struct out_state           * state_out;
     struct tagfilter_global    * counter;
     struct tagfilter_selection * tagfilter;
     struct tagfilter_tags      * tags;
     int                          push_current /* Do not generate new_item now */;
     struct tagfilter_tags      * used_tags;
     int                          inherit_pg_flags;
     int                          text_visible;
     const struct decode_opts   * decode_opt;
     struct tagfilter_stack_item **stack;
     size_t                        stack_len;
{

    struct pager_range * new_range = NULL;
    struct tagfilter_tag_state *ret = NULL;

    int unsupport = 0;

    union tagfilter_stack_uitem uitem;
    
    if (TAGFILTER_CMD_magic != current_tag->magic)
	mime_panic(__FILE__,__LINE__,
		   "process_new_html_item",
		   "Bad magic number (tagfilter_cmd)");

    if (current_tag->ascii_command) {
	DPRINT(Debug,21,(&Debug,
			 "process_new_html_item: command %s%s\n",
			 current_tag->ascii_command,
			 push_current ? ", push current" : ""));
    }

    uitem.html = tagfilter_new_stack_html_item();

    if (TAGFILTER_STACK_HTML_ITEM_magic != uitem.html->magic)
	mime_panic(__FILE__,__LINE__,
		   "process_new_html_item",
		   "Bad magic number (tagfilter_stack_html_item)");


    
    new_range = new_html_item_hlp1(current_tag,
				   prev_range,
				   state_out,
				   &text_visible,
				   inherit_pg_flags,
				   stack,stack_len,
				   &unsupport
				   );
    
    if (current_tag->code.html > tfc_html_none)
	ret = tagfilter_new_tag_state(current_tag);

     if (!new_range &&
	 ison(current_tag->cmd_mode,TAGFLT_CMD_force_newline)) {
	 
	 DPRINT(Debug,21,(&Debug,
			  "process_new_html_item: TAGFLT_CMD_force_newline set, forcing new range\n"));
	 
	 new_range = state_add_simple_pager_range(state_out,
						  prev_range,
						  0,0,0);
	 
     }
    
    if (new_item && !push_current) {

	DPRINT(Debug,21,(&Debug,
			 "process_new_html_item: pg_flags %d %s\n",
			 current_tag->pg_flags,
			 give_pg_flags(current_tag->pg_flags)));
	
	*new_item = tagfilter_new_stack_item(new_range,
					     current_tag->pg_flags,
					     ison(current_tag->cmd_mode,
						  TAGFLT_CMD_inherit_nested_tags) ?
					     used_tags  /* Transparent content model */ :
					     NULL /* Use current_tag */,
					     current_tag,uitem);

	uitem.html = NULL;
	
	if (TAGFILTER_STACK_ITEM_magic != (*new_item)->magic)
	    mime_panic(__FILE__,__LINE__,"process_new_html_item",
		       "Bad magic number (tagfilter_stack_item)");
	
	(*new_item)->text_visible = text_visible;

    } else if (push_current && ret) {
	if (TAGFILTER_TAG_STATE_magic != ret->magic)
	   	mime_panic(__FILE__,__LINE__,"process_new_html_item",
		   "Bad magic number (tagfilter_tag_state)");
 

	
	if (!ret->u.html)
	    ret->u.html = tagfilter_new_html_state();

	tagfilter_html_state_push_command(ret->u.html,
					  current_tag,
					  used_tags);
	
    }

    if (uitem.html) {
	tagfilter_free_stack_html_item(& uitem.html);
    }
    
    new_html_item_hlp2(current_tag,new_range,
		       ret,state_out,
		       text_visible,tagfilter,
		       new_item ? *new_item: NULL,
		       stack,stack_len,
		       &unsupport);

    if (new_range)
	free_pager_range(& new_range);
    

    if (tfc_html_unsupported == current_tag->code.html) {

	DPRINT(Debug,21,(&Debug,
			 "process_new_html_item: command %s unsupported\n",
			 current_tag->ascii_command));
	
	tagfilter_add_unknown_tag(counter,current_tag,
				  tagfilter);
    } else if (unsupport) {
	/* XXX <li> still unsupported */
	
	DPRINT(Debug,21,(&Debug,
			 "process_new_html_item: command %s unsupported\n",
			 current_tag->ascii_command));

	
	tagfilter_add_unknown_tag(counter,current_tag,
				  tagfilter);

    }

    
    return ret;
}

struct tag_search_item {

    struct tagfilter_push_tag  * cmd_list;
    int                          cmd_list_len;
    
    struct tagfilter_tags      * tags;
};

/* Returns found item -- not refcounted */

static struct tag_search_item * generate_search_list
P_((struct tagfilter_tags      * tags,
    struct tag_search_item    ** search_xxx,
    int                        * search_xxx_len,
    struct string              * tag_name,
    struct tagfilter_selection * sel_backlink
    ));
static struct tag_search_item * generate_search_list(tags,
						     search_xxx,
						     search_xxx_len,
						     tag_name,
						     sel_backlink)
     struct tagfilter_tags      * tags;
     struct tag_search_item    ** search_xxx;
     int                        * search_xxx_len;
     struct string              * tag_name;
     struct tagfilter_selection * sel_backlink;
{
    struct tag_search_item    * ret = NULL; 
    struct tag_search_item    * search_list = * search_xxx;
    int                         search_list_len = * search_xxx_len;
    int i;
    
    for (i = -1; i < search_list_len; i++) {
	struct tagfilter_tags * t = NULL;
	struct tagfilter_cmd  * c = NULL;
		
	struct tagfilter_push_tag     * c_list      = NULL;
	int                             c_list_len  = 0;
	
	int k;
	
	if (i < 0) {
	    t = tags;

	    c_list      = NULL;
	    c_list_len  = 0;	
	    
	} else {
	    t           = search_list[i].tags;

	    c_list      = search_list[i].cmd_list;
	    c_list_len  = search_list[i].cmd_list_len;	    
	}

	if (!t)
	    continue;
	
	if (TAGFILTER_TAGS_magic != t->magic)
	    mime_panic(__FILE__,__LINE__,"generate_search_list",
		       "Bad magic number (tagfilter_tags)");

	DPRINT(Debug,21,(&Debug,
			 "generate_search_list: %d: Checking %s for %S\n",
			 i,t->context,tag_name));
			 
	
	c = locate_command(t,tag_name,sel_backlink);

	if (c) {
	    int j;
	    
	    if (TAGFILTER_CMD_magic != c->magic)
		mime_panic(__FILE__,__LINE__,"generate_search_list",
			   "Bad magic number (tagfilter_cmd)");
	    
	    search_list = safe_array_realloc(search_list,
					     search_list_len+1,
					     sizeof (search_list[0]));

	    /* Not really needed */
	    search_list[search_list_len].tags = c->nesting;
	    
	    search_list[search_list_len].cmd_list =
		safe_calloc(c_list_len+1,
			    sizeof (search_list[search_list_len].cmd_list[0]));

	    for (j = 0; j < c_list_len; j++) {
		search_list[search_list_len].cmd_list[j] = c_list[j];

		tagfilter_inc_cmd_refcount(search_list[search_list_len].
					   cmd_list[j].cmd);	
	    }

	    search_list[search_list_len].cmd_list[j].cmd       = c;
	    search_list[search_list_len].cmd_list[j].used_tags = t;
	    tagfilter_inc_cmd_refcount(search_list[search_list_len].
				       cmd_list[j].cmd);
	    search_list[search_list_len].cmd_list_len = j+1;

	    ret = & (search_list[search_list_len] );
	    search_list_len++;
	    goto done;	    
	}

	for (k = 0; t->tags[k]; k++) {
	    int a;
	    int j;
	    struct tagfilter_cmd  * C;
	    
	    if (TAGFILTER_CMD_magic != t->tags[k]->magic)
		mime_panic(__FILE__,__LINE__,"generate_search_list",
			   "Bad magic number (tagfilter_cmd)");

	    C = t->tags[k];
	    
	    if (! C->nesting) 
		continue;

	    

	    /* Only search nested if opening tag is optional */
	    if (isoff(C-> cmd_mode,TAGFLT_CMD_nested_search)) {

		if (C->ascii_command) {

		    DPRINT(Debug,22,(&Debug,
				     "generate_search_list: %d: %d: %s on %s does not support nested search\n",
				     i,k,C->ascii_command,
				     t->context));

		    
		}
		
		continue;
	    }

	    
	    for (a = 0; a < search_list_len; a++) {

		if (C->nesting == search_list[a].tags) {
		    goto found;
		}		    
	    }

	    if (C->ascii_command) {		
		DPRINT(Debug,21,(&Debug,
				 "generate_search_list: %d: %d: %s on %s -> added to search list %d\n",
				 i,k,C->ascii_command,
				 t->context,search_list_len));
	    }

	    search_list = safe_array_realloc(search_list,
					     search_list_len+1,
					     sizeof (search_list[0]));
	    
	    search_list[search_list_len].tags = C->nesting;
	    
	    search_list[search_list_len].cmd_list =
		safe_calloc(c_list_len+1,
			    sizeof (search_list[search_list_len].
				    cmd_list[0]));

	    for (j = 0; j < c_list_len; j++) {
		search_list[search_list_len].cmd_list[j] = c_list[j];

		tagfilter_inc_cmd_refcount(search_list[search_list_len].
					   cmd_list[j].cmd);	
	    }

	    search_list[search_list_len].cmd_list[j].cmd = C;
	    search_list[search_list_len].cmd_list[j].used_tags = t;
	    tagfilter_inc_cmd_refcount(search_list[search_list_len].
				       cmd_list[j].cmd);
	    search_list[search_list_len].cmd_list_len = j+1;

	    search_list_len++;
	    
	found:
	    ;	    
	}	
    }
    
 done:
    
    * search_xxx     = search_list;
    * search_xxx_len = search_list_len;
    
    return ret;
}

/* Invalidates also found item */

static void free_search_list P_((struct tag_search_item ** search_xxx,
				 int                  * search_xxx_len));
static void free_search_list(search_xxx,
			     search_xxx_len)
     struct tag_search_item   ** search_xxx;
     int                       * search_xxx_len;
{
    struct tag_search_item    * search_list     = * search_xxx;
    int                         search_list_len = * search_xxx_len;
    
    if (search_list) {
	int i;

	for (i = 0; i < search_list_len; i++) {

	    if (search_list[i].cmd_list) {
		int j;


		for (j = 0; j < search_list[i].cmd_list_len; j++) {
		    tagfilter_release_cmd(&
					  search_list[i].cmd_list[j].cmd);
		    search_list[i].cmd_list[j].used_tags = NULL;
		}

		free (search_list[i].cmd_list);
		search_list[i].cmd_list = NULL;
	    }
	    search_list[i].cmd_list_len = 0;

	}

	free(search_list);
	search_list = NULL;
    }
    search_list_len = 0;

    * search_xxx     = search_list;
    * search_xxx_len = search_list_len;    
}



E_( give_stack_item_f give_item_text_html)
struct tagfilter_tag_state *
  give_item_text_html P_((struct tagfilter_stack_item **new_item,
			  struct tagfilter_tags      * tags,
			  struct string              * tag_name,
			  struct tagfilter_stack_item **stack,
			  size_t                        stack_len,
			  size_t                     * stack_pos,
			  mime_t                     * body,
			  struct in_state            * state_in,
			  struct out_state           * state_out,
			  const struct decode_opts   * decode_opt,
			  charset_t                    text_charset,
			  struct tagfilter_selection * tagfilter,
			  struct tagfilter_global    * counter
			  ));
struct tagfilter_tag_state *give_item_text_html(new_item,tags,tag_name,
						stack,stack_len,stack_pos,
						body,state_in,state_out,
						decode_opt,
						text_charset, tagfilter,
						counter)
     struct tagfilter_stack_item **new_item;
     struct tagfilter_tags      * tags;
     struct string              * tag_name;
     struct tagfilter_stack_item **stack;
     size_t                        stack_len;
     size_t                     * stack_pos;
     mime_t                     * body;
     struct in_state            * state_in;
     struct out_state           * state_out;
     const struct decode_opts   * decode_opt;
     charset_t                    text_charset;
     struct tagfilter_selection * tagfilter;
     struct tagfilter_global    * counter;
{
    int text_visible = 1;
    struct tagfilter_tag_state * ret = NULL;
    struct tagfilter_cmd       * current_tag = NULL;

    size_t  STACK_pos = stack_len;

    struct tagfilter_tags      * used_tags = NULL;

    int push_current   = 0;
    
    
    /* Allocates new  tagfilter_tag_state 
       (also NULL return possible) -- 
       gives struct tagfilter_stack_item if
       push_stack_item() needs not be used */
    
    if (TAGFILTER_SELECTION_magic != tagfilter->magic) 
	mime_panic(__FILE__,__LINE__,"give_item_text_html",
		   "Bad magic number (tagfilter_selection)");

    if (new_item && *new_item)
	tagfilter_free_stack_item(new_item,tagfilter);

    * stack_pos = stack_len;

    if (!tags && tag_name) {
	DPRINT(Debug,20,(&Debug,
			 "give_item_text_html: %S - no tags, stack_len=%zu\n",
			 tag_name,stack_len));
    } else if (tags && tag_name) {
	if (TAGFILTER_TAGS_magic != tags->magic)
	    mime_panic(__FILE__,__LINE__,"give_item_text_html",
		       "Bad magic number (tagfilter_tags)");
	
	
	DPRINT(Debug,20,(&Debug,
			 "give_item_text_html: %S - for %s, stack_len=%zu\n",
			 tag_name,tags->context,stack_len));
    }
	
    if (tag_name && stack && stack_len > 0) {
	size_t i = stack_len;
       	
	/* Check for auto close */
	
	while (i > 0) {
	    i--;
	    
	    if (stack[i]) {
		if (TAGFILTER_STACK_ITEM_magic != stack[i]->magic)
		    mime_panic(__FILE__,__LINE__,
			       "give_item_text_html",
			       "Bad magic number (tagfilter_stack_item)");
		
		if (! stack[i]->current_tag)
		    continue;
		
		if (TAGFILTER_CMD_magic != stack[i]->current_tag->magic)
		    mime_panic(__FILE__,__LINE__,"give_item_text_html",
			       "Bad magic number (tagfilter_cmd)");
		
		if ( !  stack[i]->current_tag->auto_close)
		    break;
		
		current_tag = locate_command(stack[i]->current_tag->
					     auto_close,
					     tag_name,tagfilter);
		
		if (current_tag) {
		    used_tags = tags;
		    STACK_pos = i;

		    /* Do not generate new_item now */
		    if (stack[i]->current_tag->code.base >= tfc_none)
			push_current = 1;
		    
		}
				
		break;  /* End search anyway */
	    }
	}
    }
	        
    if (tag_name && tags && !current_tag) {

	if (TAGFILTER_TAGS_magic != tags->magic)
	    mime_panic(__FILE__,__LINE__,"give_item_text_html",
		       "Bad magic number (tagfilter_tags)");

	/* locate_command() does not check nested */
	
	current_tag = locate_command(tags,tag_name,tagfilter);
	
	if (current_tag) {
	    used_tags = tags;
	    STACK_pos = stack_len;
	    
	}    
    } 	

    if (tag_name && !current_tag) {

	/* Check upper on on tag */
	
	if (stack && stack_len > 0) {
	    size_t i = stack_len;

	    int maybe_push_current = 0;
	    
	    while (i > 0) {
		i--;
		
		if (stack[i]) {
		    if (TAGFILTER_STACK_ITEM_magic != stack[i]->magic)
			mime_panic(__FILE__,__LINE__,
				   "give_item_text_html",
				   "Bad magic number (tagfilter_stack_item)");
		    
		    if (! stack[i]->current_tag)
			continue;
		    
		    if (TAGFILTER_CMD_magic !=
			stack[i]->current_tag->magic)
			mime_panic(__FILE__,__LINE__,
				   "give_item_text_html",
				   "Bad magic number (tagfilter_cmd)");

		    /* Do not generate new_item now */
		    if (stack[i]->current_tag->code.base >= tfc_none)
			maybe_push_current = 1;
		    
		    
		    if (stack[i]->nested_tags) {
			current_tag = locate_command(stack[i]->
						     nested_tags,
						     tag_name,tagfilter);
			
			if (current_tag) {
			    /* Potentially replaces next tag on stack */
			    
			    STACK_pos = i+1;
			    used_tags = stack[i]->nested_tags;

			    /* Do not generate new_item now */
			    if (maybe_push_current)
				push_current = 1;

			    
			    break;
			}
		    }
		    
		    /* Stop searching if
		       TAGFLT_CMD_close_search
		       in not set */
		    
		    if (isoff(stack[i]->current_tag->cmd_mode,
			      TAGFLT_CMD_close_search))
			break;
		}			    
	    }
	}		
    }

    if (tag_name && !current_tag) {
	struct tag_search_item * search_list = NULL;
	int                      search_list_len = 0;
		
	struct tag_search_item * found_xx
	    = generate_search_list(tags,
				   &search_list,&search_list_len,
				   tag_name,
				   tagfilter);
	
	if (found_xx &&
	    found_xx->cmd_list &&
	    found_xx->cmd_list_len > 0) {
	    int i;
	    
	    ret =
		tagfilter_new_tag_state(found_xx->
					cmd_list[found_xx->
						 cmd_list_len-1].cmd);

	    if (TAGFILTER_TAG_STATE_magic != ret->magic)
	   	mime_panic(__FILE__,__LINE__,"give_item_text_html",
			   "Bad magic number (tagfilter_tag_state)");
	    

	    ret->u.html = tagfilter_new_html_state();

	    if (TAGFILTER_HTML_STATE_magic != ret->u.html->magic) 
		mime_panic(__FILE__,__LINE__,"give_item_text_html",
			   "Bad magic number (tagfilter_selection)");
		
	    ret->u.html->push_tag =
		safe_calloc(found_xx->cmd_list_len,
			    sizeof (ret->u.html->push_tag[0]));
	    
	    for (i = 0; i < found_xx->cmd_list_len; i++) {
		ret->u.html->push_tag[i] = found_xx->cmd_list[i];

		tagfilter_inc_cmd_refcount(ret->u.html->push_tag[i].cmd);
	    }
	    ret->u.html->push_tag_len = found_xx->cmd_list_len;
	} else {
	    DPRINT(Debug,21,(&Debug,
			     "give_item_text_html: %S - not found with nested search - search list len %d\n",
			     tag_name,search_list_len));
	}
	
	free_search_list(&search_list,&search_list_len);		
    }
		         
    if (current_tag && !ret) {
	int                  inherit_pg_flags = 0;
	struct pager_range * prev_range =
	    give_prev_range(stack,
			    STACK_pos,
			    & inherit_pg_flags,
			    & text_visible);
       		
	ret = process_new_html_item(current_tag,prev_range,new_item,
				    state_out,counter,
				    tagfilter,used_tags,push_current,
				    used_tags,
				    inherit_pg_flags,
				    text_visible,
				    decode_opt,
				    stack,stack_len);	        		
    }
    
    if (!current_tag && tag_name && !ret) {
	struct pager_range * prev_range UNUSED_VAROK =
	    give_prev_range(stack,
			    STACK_pos,
			    NULL,
			    & text_visible);
	
	if (counter) 	   
	    current_tag = tagfilter_give_unknown_tag(counter,tag_name,
						     tagfilter,
						     tfc_unsupported,
						     TAGFLT_CMD_inherit_nested_tags);
						     
	if (!current_tag)
	    current_tag = tagfilter_cmd_from_string(tag_name,
						    tfc_unsupported,
						    TAGFLT_CMD_inherit_nested_tags);

	if (TAGFILTER_CMD_magic != current_tag->magic)
	    mime_panic(__FILE__,__LINE__,"give_item_text_html",
		       "Bad magic number (tagfilter_cmd)");

	if (new_item) {

	    union tagfilter_stack_uitem uitem;

	    uitem.html = NULL;
	    
	    DPRINT(Debug,21,(&Debug,
			     "give_item_text_html: %S pg_flags %d %s\n",
			     tag_name,current_tag->pg_flags,
			     give_pg_flags(current_tag->pg_flags)));
	    
	    *new_item = tagfilter_new_stack_item(NULL,
						 current_tag->pg_flags,
						 ison(current_tag->cmd_mode,
						      TAGFLT_CMD_inherit_nested_tags) ?
						 tags  /* Transparent content model */ :
						 NULL /* Use current_tag */,
						 current_tag,uitem);

	    uitem.html = NULL;
	    
	}
	
	if (TAGFILTER_STACK_ITEM_magic != (*new_item)->magic)
	    mime_panic(__FILE__,__LINE__,"give_item_text_html",
		       "Bad magic number (tagfilter_stack_item)");
	
	
	(*new_item)->text_visible = text_visible;	
    }
    
       
    * stack_pos = STACK_pos;

    if (current_tag && !ret)
	ret = tagfilter_new_tag_state(current_tag);
    
    if (new_item && *new_item && current_tag) {
	if (TAGFILTER_STACK_ITEM_magic != (*new_item)->magic)
	    mime_panic(__FILE__,__LINE__,"give_item_text_html",
		       "Bad magic number ( tagfilter_stack_item)");
	if (TAGFILTER_CMD_magic != current_tag->magic)
	    mime_panic(__FILE__,__LINE__,"give_item_text_html",
		       "Bad magic number (tagfilter_cmd)");

	(*new_item)->force_eoln =
	    ison(current_tag->cmd_mode,TAGFLT_CMD_force_newline);
    }

    if (ret && new_item && *new_item)
	tagfilter_add_html_stack_item(ret,*new_item);
    

    
    
    if (current_tag)
	tagfilter_release_cmd(& current_tag);
    
    return ret;

}

				    
/* Return start position */
static char * html_ol_alpha_value P_((int li_value,enum ol_list_type t,
				    char number[], size_t number_len));
static char * html_ol_alpha_value(li_value,t,number,number_len)
     int li_value;
     enum ol_list_type t;
     char number[];
     size_t number_len;
{
    char * V = NULL;
    const int base = 26;
    
    
    if (atr_type_lalpha == t)
	V = "abcdefghijklmnopqrstuvwxyz";
    else if (atr_type_ualpha == t)
	V = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
    else
	mime_panic(__FILE__,__LINE__,"html_ol_alpha_value",
		   "Bad type");

    if (number_len < 2)
	mime_panic(__FILE__,__LINE__,"html_ol_alpha_value",
		   "Too short buffer");

    if (strlen(V) != base)
	mime_panic(__FILE__,__LINE__,"html_ol_alpha_value",
		   "Bad number of characters");

    number[--number_len] = '\0';

    while (number_len > 0 && li_value > 0) {
	int t;

	li_value--;
	
	t = li_value % base;
	
	li_value /= base;
	
	number[--number_len] = V[t];
    }

    if (0 == li_value)
	return &number[number_len];
    
    return NULL;
}


/* Like on https://stackoverflow.com/questions/23269143/c-program-that-converts-numbers-to-roman-numerals */

static struct roman_code {
    int  number;
    char *letters;
} uroman_number[] = {
    { 1000, "M"  }, 
    {  900, "CM" }, 
    {  500, "D"  }, 
    {  400, "CD" }, 
    {  100, "C"  }, 
    {   90, "XC" }, 
    {   50, "L"  },
    {   40, "XL" }, 
    {   10, "X"  }, 
    {    9, "IX" }, 
    {    5, "V"  }, 
    {    4, "IV" }, 
    {    1, "I"  },
    {    0, NULL }, 
};

static struct roman_code lroman_number[] = {
    { 1000, "m"  }, 
    {  900, "cm" }, 
    {  500, "d"  }, 
    {  400, "cd" }, 
    {  100, "c"  }, 
    {   90, "xc" }, 
    {   50, "l"  },
    {   40, "xl" }, 
    {   10, "x"  }, 
    {    9, "ix" }, 
    {    5, "v"  }, 
    {    4, "iv" }, 
    {    1, "i"  },
    {    0, NULL }, 
};

/* Return start position */
static char * html_ol_roman_value P_((int li_value,enum ol_list_type t,
				    char number[], size_t number_len));
static char * html_ol_roman_value(li_value,t,number,number_len)
     int li_value;
     enum ol_list_type t;
     char number[];
     size_t number_len;
{
    struct roman_code * V = NULL;
    char * pos = &(number[0]);
    int i;
    
    if (atr_type_lroman == t)
	V = &(lroman_number[0]);
    else if (atr_type_uroman == t)
	V = &(uroman_number[0]);
    else
	mime_panic(__FILE__,__LINE__,"html_ol_roman_value",
		    "Bad type");

    if (number_len < 2)
	mime_panic(__FILE__,__LINE__,"html_ol_roman_value",
		   "Too short buffer");

    if (li_value > 3999)
	return NULL;  /* Not supported */

    /* Numbers >= 4000 require bar over letters and are not supported here */
    
    for (i = 0; V[i].letters && pos < &number[number_len -1] && li_value > 0; i++) {
	int len = strlen( V[i].letters);
	
	while (pos < &number[number_len -1] && li_value >=  V[i].number) {
	    int j;
	    
	    if (pos + len >= number + number_len -1)
		return NULL;  /* Overflow? */

	    for (j = 0; j < len; j++)
		*pos++ = V[i].letters[j];

	    li_value -= V[i].number;
	}	
    }

    *pos = '\0'; 
    
    if (0 ==  li_value)
	return &number[0];

    return NULL;
}
    
E_(locate_stack_item_f locate_item_text_html)
struct tagfilter_tag_state *
        locate_item_text_html P_((struct string               * tag_name,
				  struct tagfilter_stack_item **stack,
				  size_t                        stack_len,
				  size_t                      * index
				  /* or stack_len if not found */,
				  mime_t                     * body,
				  struct in_state            * state_in,
				  struct out_state           * state_out,
				  charset_t                    text_charset,
				  struct tagfilter_selection * tagfilter));
struct tagfilter_tag_state * locate_item_text_html(tag_name,
						   stack,stack_len,
						   index,body,
						   state_in,state_out,
						   text_charset,tagfilter)
     struct string               * tag_name;
     struct tagfilter_stack_item **stack;
     size_t                        stack_len;
     size_t                      * index
     /* or stack_len if not found */;
     mime_t                     * body;
     struct in_state            * state_in;
     struct out_state           * state_out;
     charset_t                    text_charset;
     struct tagfilter_selection * tagfilter;
{
    /* Allocates new  tagfilter_tag_state 
       (also NULL return possible) */
    
    struct tagfilter_tag_state * ret = NULL;

   struct string_sort  * tag_name_X = NULL;  /* Possible lowercased
						already 
					     */
   
   if (TAGFILTER_SELECTION_magic != tagfilter->magic) 
	mime_panic(__FILE__,__LINE__,"locate_item_text_html",
		   "Bad magic number (tagfilter_selection)");
   
   ret = tagfilter_locate_stack_item(tagfilter,
				     &tag_name_X,
				     tag_name,
				     stack,stack_len,index);
   if (ret) {
       if (TAGFILTER_TAG_STATE_magic != ret->magic)
	   mime_panic(__FILE__,__LINE__,"locate_item_text_html",
		      "Bad magic number (tagfilter_tag_state)");

       
       if (! ret->u.html)
	   ret->u.html = tagfilter_new_html_state();
       
       if (TAGFILTER_HTML_STATE_magic != ret->u.html->magic) 
	   mime_panic(__FILE__,__LINE__,"locate_item_text_html",
		      "Bad magic number (tagfilter_html_state)");
   }
   
   
   if (tag_name_X)
       free_string_sort(& tag_name_X);
    
   return ret;
}

E_(end_tag_state_f end_tag_text_html)
void end_tag_text_html P_((struct tagfilter_tag_state **tag_state,
			   mime_t                     * body,
			   struct in_state            * state_in,
			   struct out_state           * state_out,
			   charset_t                    text_charset,
			   struct tagfilter_selection * tagfilter,
			   struct tagfilter_global    * counter));
void end_tag_text_html(tag_state,body,state_in,state_out,text_charset,tagfilter,
		       counter)
     struct tagfilter_tag_state **tag_state;
     mime_t                     * body;
     struct in_state            * state_in;
     struct out_state           * state_out;
     charset_t                    text_charset;
     struct tagfilter_selection * tagfilter;
     struct tagfilter_global    * counter;
{

    
    if (TAGFILTER_SELECTION_magic != tagfilter->magic) 
	mime_panic(__FILE__,__LINE__,"end_tag_text_html",
		   "Bad magic number (tagfilter_selection)");
        
    if (TAGFILTER_TAG_STATE_magic != (*tag_state)->magic)
	mime_panic(__FILE__,__LINE__,"end_tag_text_html",
		   "Bad magic number (tagfilter_tag_state)");

    
    if ((*tag_state)->u.html) {
	struct tagfilter_html_state * tag_html_state;
	int bad = 0;
		
	if (TAGFILTER_HTML_STATE_magic != (*tag_state)->u.html->magic) 
	    mime_panic(__FILE__,__LINE__,"end_tag_text_html",
			   "Bad magic number (tagfilter_html_state)");

	tag_html_state = (*tag_state)->u.html;
	
	if (tag_html_state->current_stack) {
	    struct tagfilter_stack_html_item * st;
	    
	    if (TAGFILTER_STACK_HTML_ITEM_magic !=
		tag_html_state->current_stack->magic)
		mime_panic(__FILE__,__LINE__,"end_tag_text_html",
			   "Bad magic number (tagfilter_stack_html_item)");
	    
	    st = tag_html_state->current_stack;
	    
	    if (tag_html_state->have_ol_start &&
		! tag_html_state->duplicate_ol_start) {
		st->next_item_number =
		    tag_html_state->ol_start_value;
		st->item_number_set  = 1;
	    }

	    if (tag_html_state->ol_type_value &&
		! tag_html_state->duplicate_ol_type) {
		st->type_item_number =
		    tag_html_state->ol_type_value;
	    }

	    if (st->reversed_item_number &&
		! st->item_number_set) {

		if ((*tag_state)->current_tag) {
		    struct tagfilter_cmd       * current_tag;

		    if (TAGFILTER_CMD_magic != (*tag_state)->current_tag->magic)
			mime_panic(__FILE__,__LINE__,
				   "end_tag_text_html",
				   "Bad magic number (tagfilter_cmd)");
		    
		    current_tag = (*tag_state)->current_tag;
			
		    DPRINT(Debug,21,(&Debug,
				     "end_tag_text_html: command %s unsupported\n",
				     current_tag->ascii_command));
		    
		    tagfilter_add_unknown_tag(counter,current_tag,
					      tagfilter);		    			
		}
	    }	    
	}

	
	if (tag_html_state->bad_li_value  ||
	    tag_html_state->have_li_value ||
	    tag_html_state->default_li_value) {
	    
	    int li_value              = tag_html_state->li_value;
	    enum ol_list_type li_type = tag_html_state->ol_type_value;
	    
	    if (tag_html_state->ol_stack) {		
		struct tagfilter_stack_html_item * ol_stack;
		
		if (TAGFILTER_STACK_HTML_ITEM_magic != tag_html_state->ol_stack->magic)
		    mime_panic(__FILE__,__LINE__,
			       "end_tag_text_html",
			       "Bad magic number (tagfilter_stack_html_item)");

		ol_stack = tag_html_state->ol_stack;

		if (tag_html_state->bad_li_value) {
		    bad = 1;
		} else if (tag_html_state->have_li_value) {
		    
		    if (ol_stack->reversed_item_number)
			ol_stack->next_item_number =
			    li_value-1;
		    else
			ol_stack->next_item_number =
			    li_value+1;
		    
		    ol_stack->item_number_set  = 1;
		    
		    bad = 0;		    
		}				
	    }
		    
	    if (tag_html_state->duplicate_li_value || bad) {
		state_printf(state_out,FRM("?. "),li_value);

		
	    } else {

		if (li_value > 0) {
		    char number[SLEN];
		    char * V;
	       		    
		    switch (li_type) {

		    case atr_type_lalpha:
		    case atr_type_ualpha:
			
			if (NULL != (V = html_ol_alpha_value(li_value,li_type,
							     number,sizeof number))) {

			    state_puts(V,state_out);
			    state_puts(". ",state_out);
			} else
			    goto  decimal_number;
			break;

		    case atr_type_lroman:  /* lower-roman */
		    case atr_type_uroman:   /* upper-roman */

			if (NULL != (V = html_ol_roman_value(li_value,li_type,
							     number,sizeof number))) {
			    
			    state_puts(V,state_out);
			    state_puts(". ",state_out);
			} else
			    goto  decimal_number;
			break;
			
		    case atr_type_none:
		    case atr_type_decimal:
			goto decimal_number;
		    }
		} else {
		decimal_number:
		    state_printf(state_out,FRM("%d. "),li_value);
		}
	    }	    	    
	}

       	
	tagfilter_free_html_state(& ((*tag_state)->u.html));
    }
    
    tagfilter_free_tag_state_common (tag_state);
}





E_(push_stack_item_f push_tag_text_html)
struct tagfilter_stack_item *
   push_tag_text_html P_((struct tagfilter_tag_state *tag_state,
			  struct tagfilter_stack_item **stack,
			  size_t                        stack_len,
			  mime_t                     * body,
			  struct in_state            * state_in,
			  struct out_state           * state_out,
			  const struct decode_opts   * decode_opt,
			  charset_t                    text_charset,
			  struct tagfilter_selection * tagfilter));
struct tagfilter_stack_item *push_tag_text_html(tag_state,stack,
						stack_len,body,state_in,
						state_out,decode_opt,text_charset,
						tagfilter)
     struct tagfilter_tag_state *tag_state;
     struct tagfilter_stack_item **stack;
     size_t                        stack_len;
     mime_t                     * body;
     struct in_state            * state_in;
     struct out_state           * state_out;
     const struct decode_opts   * decode_opt;
     charset_t                    text_charset;
     struct tagfilter_selection * tagfilter;
{
    struct tagfilter_stack_item *ret = NULL;
    
    if (TAGFILTER_SELECTION_magic != tagfilter->magic) 
	mime_panic(__FILE__,__LINE__,"push_tag_text_html",
		   "Bad magic number (tagfilter_selection)");
        
    if (TAGFILTER_TAG_STATE_magic != tag_state->magic)
	mime_panic(__FILE__,__LINE__,"push_tag_text_html",
		   "Bad magic number (tagfilter_tag_state)");

    if (tag_state->u.html) {
	struct tagfilter_cmd    *cmd = NULL;
	struct tagfilter_tags  * used_tags = NULL;
	int is_last = 0;
	
	if (TAGFILTER_HTML_STATE_magic != tag_state->u.html->magic) 
	    mime_panic(__FILE__,__LINE__,"push_tag_text_html",
		      "Bad magic number (tagfilter_html_state)");

	if (tag_state->u.html->push_tag) {
	    int i,t = 0;

	    /* This should select first comamnd, and 
	       move others
	    */
	    
	    for (i = 0; i < tag_state->u.html->push_tag_len; i++) {

		if (tag_state->u.html->push_tag[i].cmd) {

		    if (cmd) {

			if (t < i) {
			    if (tag_state->u.html->push_tag[t].cmd ||
				tag_state->u.html->push_tag[t].used_tags)
				mime_panic(__FILE__,__LINE__,
					   "push_tag_text_html",
					   "Target not empty");
			    else {
				tag_state->u.html->push_tag[t++] =
				    tag_state->u.html->push_tag[i];

				tag_state->u.html->push_tag[i].cmd = NULL;
				tag_state->u.html->push_tag[i].used_tags = NULL;
			    }			    
			}

			is_last = 0;

		    } else {
			cmd       = tag_state->u.html->push_tag[i].cmd;
			used_tags = tag_state->u.html->push_tag[i].used_tags;
			
			tag_state->u.html->push_tag[i].cmd = NULL;
			tag_state->u.html->push_tag[i].used_tags = NULL;
			
			is_last = 1;
		    }		    
		}
	    }
	    tag_state->u.html->push_tag_len = t;

	    if (0 == tag_state->u.html->push_tag_len) {
		free(tag_state->u.html->push_tag);
		tag_state->u.html->push_tag = NULL;
	    }	    
	}
	
	if (cmd) {
	    int  inherit_pg_flags = 0;
	    int text_visible = 1;
	    struct pager_range * prev_range =
		give_prev_range (stack,stack_len,
				 & inherit_pg_flags,
				 & text_visible
				 );

	    ret = process_new_html_item2(cmd,prev_range,
					 tag_state,
					 state_out,
					 tagfilter,
					 is_last,
					 used_tags,
					 inherit_pg_flags,
					 text_visible,
					 decode_opt,
					 stack,stack_len);
	    
	    tagfilter_release_cmd(& cmd);
	}	    
    }	

    return ret;
}

E_(close_stack_item_f close_tag_text_html)
int close_tag_text_html P_((struct tagfilter_tag_state *tag_state,
			    struct tagfilter_stack_item **stack,
			    size_t                        stack_len,
			    size_t                        idx,
			    enum stack_close_mode         close_pos,
			    mime_t                     * body,
			    struct in_state            * state_in,
			    struct out_state           * state_out,
			    charset_t                    text_charset,
			    struct tagfilter_selection * tagfilter,
			    struct tagfilter_global    * counter));
int close_tag_text_html(tag_state,stack,stack_len,idx,close_pos,
			    body,state_in,state_out,text_charset,
			tagfilter,counter)
     struct tagfilter_tag_state *tag_state;
     struct tagfilter_stack_item **stack;
     size_t                        stack_len;
     size_t                        idx;
     enum stack_close_mode         close_pos;
     mime_t                     * body;
     struct in_state            * state_in;
     struct out_state           * state_out;
     charset_t                    text_charset;
     struct tagfilter_selection * tagfilter;
     struct tagfilter_global    * counter;

{
    int ret = 1;   /* Return 1 when item can be emoved from stack */
    struct tagfilter_cmd       * current_tag = NULL;

    if (TAGFILTER_SELECTION_magic != tagfilter->magic) 
	mime_panic(__FILE__,__LINE__,"close_tag_text_html",
		   "Bad magic number (tagfilter_selection)");
    
    if (TAGFILTER_TAG_STATE_magic != tag_state->magic)
	mime_panic(__FILE__,__LINE__,"close_tag_text_html",
		   "Bad magic number (tagfilter_tag_state)");

    if (! stack || idx >= stack_len || ! stack[idx]) 
	mime_panic(__FILE__,__LINE__,"close_tag_text_html",
		   "Bad stack idx or stack item or no stack");
    
    if (TAGFILTER_STACK_ITEM_magic != stack[idx]->magic)
	mime_panic(__FILE__,__LINE__,
		   "close_tag_text_html",
		   "Bad magic number (tagfilter_stack_item)");

    if (! stack[idx]->current_tag)
	goto out;
    
    if (TAGFILTER_CMD_magic != stack[idx]->current_tag->magic)
	mime_panic(__FILE__,__LINE__,"close_tag_text_html",
		   "Bad magic number (tagfilter_cmd)");
    
    current_tag = stack[idx]->current_tag;

    if (stack[idx]->range) {
	DPRINT(Debug,21,(&Debug,"close_tag_text_html: stack #%zu: current range %p\n",
			 idx,stack[idx]->range));
    }
    
    switch (current_tag->code.html) {
    case tfc_html_unsupported:    /* Should not happen */
    case tfc_html_none:     break;	
	
    case tfc_html_br: /* Nothing to do */ break;
	
    case tfc_html_hr:
    case tfc_html_p:
	state_putc('\n',state_out); 
	
	/* tagfilter_handle_tag() will call set_out_state_line_mode() */
	break;
    case tfc_html_pre:              break;
    case tfc_html_q:
	state_putc('"',state_out);  
	break;
    case tfc_html_li:
	state_putc('\n',state_out);
	
	/* tagfilter_handle_tag() will call set_out_state_line_mode() */
	break;
    case tfc_html_ul:               break;
    case tfc_html_ol:               break;
    case tfc_html_img:

	if (tag_state->u.html) {
	    struct tagfilter_html_state * tag_html_state;

	    if (TAGFILTER_STACK_HTML_ITEM_magic !=
		tag_state->u.html->current_stack->magic)
		mime_panic(__FILE__,__LINE__,"close_tag_text_html",
			   "Bad magic number (tagfilter_stack_html_item)");

	    tag_html_state = tag_state->u.html;
	    
	    if (! tag_html_state->seen_img_alt) {

		DPRINT(Debug,21,(&Debug,
				 "close_tag_text_html: command %s unsupported\n",
				 current_tag->ascii_command));
				
		tagfilter_add_unknown_tag(counter,current_tag,
					  tagfilter);		
	    }
	}

	break;
    }
    
 out:
	
    return ret;
}

/* Only used with html */

#define TAGFILTER_ATR_STATE_magic	0xFD1B

static struct tagfilter_atr_state {

    unsigned short               magic;  /* TAGFILTER_ATR_STATE_magic */

    enum atr_code {
	atr_none = 0,
	atr_ol_start        /* <ol start=xxxx> */,
	atr_li_value        /* <li value=xxxxx> */,
	atr_ol_type         /* <ol type=X */,
	atr_img_alt         /* <img alt=xxxx> */,
	
    }   code;

    enum atr_pass {
	atr_pass_init  = -1,
	atr_start_seen = 0,
	atr_end_seen
    }   pass;

    enum atr_sign {
	atr_minus_sign  = -1,    /* used as multiplier */
	atr_sign_init   = 0,
	atr_sign_plus   = 1      /* used as multiplier */
    }  sign_seen;

    union atr_value {
	int               numeric;
	enum ol_list_type type_value;

    } v;
    
    unsigned  int fail:1;        /* attribute failed */
    unsigned  int have_value:1;  /* value seen */
    
    
} * tagfilter_alloc_atr_state P_((enum atr_code code));

static struct tagfilter_atr_state * tagfilter_alloc_atr_state(code)
     enum atr_code code;
{
    struct tagfilter_atr_state * ret =
	safe_zero_alloc(sizeof (*ret));

    ret->magic = TAGFILTER_ATR_STATE_magic;
    ret->code = code;

    ret->pass = atr_pass_init;
    
    ret->v.numeric = 0;

    ret->fail = 0;
    ret->have_value = 0;
    ret->sign_seen = atr_sign_init;

    return ret;
}

static void  tagfilter_free_atr_state P_(( struct tagfilter_atr_state **atr_state));
static void  tagfilter_free_atr_state(atr_state)
     struct tagfilter_atr_state **atr_state;
{
    if (TAGFILTER_ATR_STATE_magic !=  (*atr_state)->magic) 
	mime_panic(__FILE__,__LINE__,"tagfilter_free_atr_state",
		   "Bad magic number (tagfilter_atr_state)");

    (*atr_state)->magic = 0; /* Invalidate */
    free(*atr_state);
    *atr_state = NULL;
}


E_(start_attribute_f start_atr_text_html)
struct  tagfilter_atr_state *
   start_atr_text_html P_((struct string              * atr_name,
			 struct tagfilter_tag_state * tag_state,
			 mime_t                     * body,
			 struct in_state            * state_in,
			 struct out_state           * state_out,
			 charset_t                    text_charset,
			 struct tagfilter_selection * tagfilter));
struct  tagfilter_atr_state *start_atr_text_html(atr_name,tag_state,
						     body,state_in,
						     state_out,
						     text_charset,
						     tagfilter)
     struct string              * atr_name;
     struct tagfilter_tag_state * tag_state;
     mime_t                     * body;
     struct in_state            * state_in;
     struct out_state           * state_out;
     charset_t                    text_charset;
     struct tagfilter_selection * tagfilter;
{
    struct  tagfilter_atr_state * ret = NULL;
    
    if (TAGFILTER_SELECTION_magic != tagfilter->magic) 
	mime_panic(__FILE__,__LINE__,"start_atr_text_html",
	    "Bad magic number (tagfilter_selection)");
    
    if (TAGFILTER_TAG_STATE_magic !=  tag_state->magic)
	mime_panic(__FILE__,__LINE__,"start_atr_text_html",
	    "Bad magic number (tagfilter_tag_state)");
    
    /* Need  handle text/html attributes */
    
    switch (tag_state->current_tag->code.html) {
    case tfc_html_unsupported:    /* Should not happen */
    case tfc_html_none:     break;	
    case tfc_html_br:       break;
    case tfc_html_hr:       break;
    case tfc_html_p:        break;
    case tfc_html_pre:      break;
    case tfc_html_q:        break;
    case tfc_html_li:   
       if (atr_name) {
	   /* XXXXX Attribute names are converted to lower case */
	
	  if (string_matches_ascii(atr_name,cs2us("value"),
	                         SMA_bit_ignore_case,   /* XXXX */
	                         SMA_op_normal)) {
	  
	     ret = tagfilter_alloc_atr_state(atr_li_value);
	  
	  } else {
	  
	      DPRINT(Debug,22,(&Debug,
			   "start_atr_text_html: <li %S> not supported\n",
			   atr_name));				      
	  }
       }
      	
       break;
    case tfc_html_ul:       break;
    case tfc_html_ol:
	
      if (atr_name) {
	/* XXXXX Attribute names are converted to lower case */		
	if (string_matches_ascii(atr_name,cs2us("start"),
				 SMA_bit_ignore_case,   /* XXXX */
				 SMA_op_normal)) {
	  
	  ret = tagfilter_alloc_atr_state(atr_ol_start);

	} else if (string_matches_ascii(atr_name,cs2us("reversed"),
				 SMA_bit_ignore_case,   /* XXXX */
				 SMA_op_normal)) {

	    if (tag_state->u.html) {
		struct tagfilter_html_state * tag_html_state;
		
		if (TAGFILTER_HTML_STATE_magic != tag_state->u.html->magic) 
		    mime_panic(__FILE__,__LINE__,"start_atr_text_html",
			       "Bad magic number (tagfilter_html_state)");
		
		tag_html_state = tag_state->u.html;
		
		if (tag_html_state->current_stack) {
		    struct tagfilter_stack_html_item * st;
		    
		    if (TAGFILTER_STACK_HTML_ITEM_magic !=
			tag_html_state->current_stack->magic)
			mime_panic(__FILE__,__LINE__,"start_atr_text_html",
				   "Bad magic number (tagfilter_stack_html_item)");
	    
		    st = tag_html_state->current_stack;

		    st->reversed_item_number = 1;		    
		}
	    }

	} else if (string_matches_ascii(atr_name,cs2us("type"),
				 SMA_bit_ignore_case,   /* XXXX */
				 SMA_op_normal)) {
	    ret = tagfilter_alloc_atr_state(atr_ol_type);
	    
	} else {
	  
	  DPRINT(Debug,22,(&Debug,
			   "start_atr_text_html: <ol %S> not supported\n",
			   atr_name));				     
	}
      }
      break;
    case tfc_html_img:
	if (atr_name) {
	    /* XXXXX Attribute names are converted to lower case */
	    
	    if (string_matches_ascii(atr_name,cs2us("alt"),
				     SMA_bit_ignore_case,   /* XXXX */
				     SMA_op_normal)) {
		
		ret = tagfilter_alloc_atr_state(atr_img_alt);

		if (tag_state->u.html) {
		    struct tagfilter_html_state * tag_html_state;
		    
		    if (TAGFILTER_HTML_STATE_magic != tag_state->u.html->magic) 
			mime_panic(__FILE__,__LINE__,"start_atr_text_html",
				   "Bad magic number (tagfilter_html_state)");
		    
		    tag_html_state = tag_state->u.html;

		    /* Does not check if there is text on "alt" */
		    
		    tag_html_state->seen_img_alt = 1;
		}
		    		
	    } else {
		
		DPRINT(Debug,22,(&Debug,
				 "start_atr_text_html: <img %S> not supported\n",
				 atr_name));				      
	    }
	}
	
	break;
    }
    
    DPRINT(Debug,22,(&Debug, "start_atr_text_html="));
    if (ret) {
      DPRINT(Debug,22,(&Debug, "%p",ret));
    } else {
      DPRINT(Debug,22,(&Debug, "NULL"));
    }
    if (atr_name) {
      DPRINT(Debug,22,(&Debug, "; atr_name=%S",atr_name));
    }
    DPRINT(Debug,22,(&Debug, "\n"));
    
    return ret;
}

E_(end_attribute_f end_atr_text_html)
void end_atr_text_html P_((struct tagfilter_atr_state **atr_state,
			       struct tagfilter_tag_state * tag_state,
			       mime_t                     * body,
			       struct in_state            * state_in,
			       struct out_state           * state_out,
			       charset_t                    text_charset,
			       struct tagfilter_selection * tagfilter));
void end_atr_text_html(atr_state,tag_state,body,state_in,state_out,
			   text_charset,tagfilter)
     struct tagfilter_atr_state **atr_state;
     struct tagfilter_tag_state * tag_state;
     mime_t                     * body;
     struct in_state            * state_in;
     struct out_state           * state_out;
     charset_t                    text_charset;
     struct tagfilter_selection * tagfilter;
{

    if (TAGFILTER_SELECTION_magic != tagfilter->magic) 
	mime_panic(__FILE__,__LINE__,"end_atr_text_html",
		   "Bad magic number (tagfilter_selection)");

    if (TAGFILTER_TAG_STATE_magic !=  tag_state->magic)
	mime_panic(__FILE__,__LINE__,"end_atr_text_html",
		   "Bad magic number (tagfilter_tag_state)");

    if (TAGFILTER_ATR_STATE_magic !=  (*atr_state)->magic) 
	mime_panic(__FILE__,__LINE__,"end_atr_text_html",
		   "Bad magic number (tagfilter_atr_state)");

    if (! (*atr_state)->fail &&
	atr_end_seen == (*atr_state)->pass) {
		    
	switch ((*atr_state)->code) {
	case atr_none:       break;
	case atr_ol_start :
	 
	    DPRINT(Debug,22,(&Debug,
			     "end_atr_text_html: <ol start> numeric=%d sign_seen=%d",
			     (*atr_state)->v.numeric,
			     (*atr_state)->sign_seen));

	    switch ((*atr_state)->sign_seen) {
	    case atr_minus_sign: DPRINT(Debug,22,(&Debug," atr_minus_sign"));  break;
	    case atr_sign_init:  DPRINT(Debug,22,(&Debug," atr_sign_init"));   break;
	    case atr_sign_plus:  DPRINT(Debug,22,(&Debug," atr_sign_plus"));   break;
	    }
	    
	    if ((*atr_state)->have_value) {
		DPRINT(Debug,22,(&Debug," have value"));
	    } else {
		DPRINT(Debug,22,(&Debug," no value"));
	    }
	    DPRINT(Debug,22,(&Debug,"\n"));
	    
	    
	    if ((*atr_state)->have_value) {
		int numeric =  (*atr_state)->v.numeric;
		
		if ((*atr_state)->sign_seen) {
		    numeric =  (*atr_state)->v.numeric *
			(*atr_state)->sign_seen;
		}
		
		if (tag_state->u.html) {
		    if (TAGFILTER_HTML_STATE_magic != tag_state->u.html->magic) 
			mime_panic(__FILE__,__LINE__,"end_atr_text_html",
				   "Bad magic number (tagfilter_html_state)");
		    
		    if (tag_state->u.html->have_ol_start) {
			DPRINT(Debug,22,(&Debug,
					 "end_atr_text_html: <ol start> duplicate value %d\n",
					 numeric));
			
			tag_state->u.html->duplicate_ol_start = 1;
		    } else {
			tag_state->u.html->ol_start_value = numeric;
			tag_state->u.html->have_ol_start  = 1;

			DPRINT(Debug,22,(&Debug,
					 "end_seg_str_text_html: <ol start> value %d\n",
					 numeric));
		    }		    
		} else {
		    DPRINT(Debug,22,(&Debug,
				     "end_atr_text_html: <ol start> no html state, value %d lost\n",
				     numeric));
		    
		}
	    }
	    break;

	case atr_li_value:

	  DPRINT(Debug,22,(&Debug,
			   "end_atr_text_html: <li value> numeric=%d sign_seen=%d",
			   (*atr_state)->v.numeric,
			   (*atr_state)->sign_seen));
	  
	  switch ((*atr_state)->sign_seen) {
	  case atr_minus_sign: DPRINT(Debug,22,(&Debug," atr_minus_sign"));  break;
	  case atr_sign_init:  DPRINT(Debug,22,(&Debug," atr_sign_init"));   break;
	  case atr_sign_plus:  DPRINT(Debug,22,(&Debug," atr_sign_plus"));   break;
	  }
	  
	  if ((*atr_state)->have_value) {
	    DPRINT(Debug,22,(&Debug," have value"));
	  } else {
	    DPRINT(Debug,22,(&Debug," no value"));
	  }
	  DPRINT(Debug,22,(&Debug,"\n"));
	  
	  
	  if ((*atr_state)->have_value) {
	    int numeric =  (*atr_state)->v.numeric;
	    
	    if ((*atr_state)->sign_seen) {
	      numeric =  (*atr_state)->v.numeric *
		(*atr_state)->sign_seen;
	    }

	    if (tag_state->u.html) {
	      if (TAGFILTER_HTML_STATE_magic != tag_state->u.html->magic) 
		mime_panic(__FILE__,__LINE__,"end_atr_text_html",
			   "Bad magic number (tagfilter_html_state)");
	      
	      if (tag_state->u.html->have_li_value) {
		DPRINT(Debug,22,(&Debug,
				 "end_atr_text_html: <li value> duplicate value %d\n",
				 numeric));

		tag_state->u.html->duplicate_li_value = 1;
	      } else {
		tag_state->u.html->li_value = numeric;
		tag_state->u.html->have_li_value = 1;
		tag_state->u.html->bad_li_value = 0;
		
		DPRINT(Debug,22,(&Debug,
				 "end_seg_str_text_html: <li value> value %d\n",
				 numeric));
	      }
	    } else {
	      DPRINT(Debug,22,(&Debug,
			       "end_atr_text_html: <li value> no html state, value %d lost\n",
			       numeric));
	    }
	  }
	  break;
	case atr_ol_type:

	    DPRINT(Debug,22,(&Debug,
			     "end_atr_text_html: <ol type> type_value=%d",
			     (*atr_state)->v.type_value));

	    if (isascii((*atr_state)->v.type_value) &&
		isprint((*atr_state)->v.type_value)) {
		DPRINT(Debug,22,(&Debug," '%c'",(*atr_state)->v.type_value));
	    }

	    switch ((*atr_state)->v.type_value) {
	    case atr_type_none:    DPRINT(Debug,22,(&Debug," atr_type_none"));    break;
	    case atr_type_decimal: DPRINT(Debug,22,(&Debug," atr_type_decimal")); break;
	    case atr_type_lalpha:  DPRINT(Debug,22,(&Debug," atr_type_lalpha"));  break;
	    case atr_type_ualpha:  DPRINT(Debug,22,(&Debug," atr_type_ualpha"));  break;
	    case atr_type_lroman:  DPRINT(Debug,22,(&Debug," atr_type_lroman"));  break;
	    case atr_type_uroman:  DPRINT(Debug,22,(&Debug," atr_type_uroman"));  break;
	    }

	    if ((*atr_state)->have_value) {
		DPRINT(Debug,22,(&Debug," have value"));
	    } else {
		DPRINT(Debug,22,(&Debug," no value"));
	    }
	    DPRINT(Debug,22,(&Debug,"\n"));
	    
	    
	    if ((*atr_state)->have_value) {
		enum ol_list_type type_value = (*atr_state)->v.type_value;

		if (tag_state->u.html) {
		    if (TAGFILTER_HTML_STATE_magic != tag_state->u.html->magic) 
			mime_panic(__FILE__,__LINE__,"end_atr_text_html",
				   "Bad magic number (tagfilter_html_state)");

		    if (atr_type_none != tag_state->u.html->ol_type_value) {
			DPRINT(Debug,22,(&Debug,
					 "end_atr_text_html: <ol type> duplicate value %d\n",
					 type_value));

			tag_state->u.html->duplicate_ol_type = 1;
		    } else {
			tag_state->u.html->ol_type_value = type_value;

			DPRINT(Debug,22,(&Debug,
					 "end_seg_str_text_html: <ol type> value %d\n",
					 type_value));
		    }

		} else {
		    DPRINT(Debug,22,(&Debug,
				     "end_atr_text_html: <ol type> no html state, value %d lost\n",
			       type_value));
		}
	    }
	    
	    break;
	case atr_img_alt:	    
	    break;
	}
	
    } else {
	DPRINT(Debug,22,(&Debug,
			 "end_atr_text_html:"));
	switch ((*atr_state)->code) {
	case atr_none:       break;
	case atr_ol_start:   DPRINT(Debug,22,(&Debug," <ol start>")); break;
	case atr_li_value:   DPRINT(Debug,22,(&Debug," <li value>")); break;
	case atr_ol_type:    DPRINT(Debug,22,(&Debug," <ol type>")); break;
	case atr_img_alt:    DPRINT(Debug,22,(&Debug," <img alt>")); break;
	}
	if ((*atr_state)->fail) {
	    DPRINT(Debug,22,(&Debug," failed"));
	}
	DPRINT(Debug,22,(&Debug," pass=%d", (*atr_state)->pass));
	switch ((*atr_state)->pass) {
	case atr_pass_init:   DPRINT(Debug,22,(&Debug," atr_pass_init"));   break;
	case atr_start_seen:  DPRINT(Debug,22,(&Debug," atr_start_seen"));  break;
	case atr_end_seen:    DPRINT(Debug,22,(&Debug," atr_end_seen"));    break;
	}
	DPRINT(Debug,22,(&Debug,"\n"));
    }
    
    tagfilter_free_atr_state(atr_state);    
}

E_(start_end_atrvalue_f start_end_atrval_text_html)
void start_end_atrval_text_html
         P_((struct tagfilter_atr_state * atr_state,
	     enum atrvalue_start_end      start_end,
	     struct tagfilter_tag_state * tag_state,
	     mime_t                     * body,
	     struct in_state            * state_in,
	     struct out_state           * state_out,
	     charset_t                    text_charset,
	     struct tagfilter_selection * tagfilter));
void start_end_atrval_text_html(atr_state,start_end,tag_state,
				    body,state_in,state_out,
				    text_charset,tagfilter)
     struct tagfilter_atr_state * atr_state;
     enum atrvalue_start_end      start_end;
     struct tagfilter_tag_state * tag_state;
     mime_t                     * body;
     struct in_state            * state_in;
     struct out_state           * state_out;
     charset_t                    text_charset;
     struct tagfilter_selection * tagfilter;
{

    if (TAGFILTER_SELECTION_magic != tagfilter->magic) 
	mime_panic(__FILE__,__LINE__,"start_end_atrval_text_html",
		   "Bad magic number (tagfilter_selection)");

    if (TAGFILTER_TAG_STATE_magic !=  tag_state->magic)
	mime_panic(__FILE__,__LINE__,"start_end_atrval_text_html",
		   "Bad magic number (tagfilter_tag_state)");

    if (TAGFILTER_ATR_STATE_magic !=  atr_state->magic) 
	mime_panic(__FILE__,__LINE__,"start_end_atrval_text_html",
		   "Bad magic number (tagfilter_atr_state)");
    
    switch (start_end) {
    case atrvalue_start:
	if (atr_pass_init == atr_state->pass) {
	    atr_state->pass = atr_start_seen;
	    atr_state->v.numeric = 0;
	    /* XXXX reset all value types */
	}
	else
	    atr_state->fail = 1;
	break;
    case atrvalue_end:
	if (atr_start_seen == atr_state->pass)
	    atr_state->pass = atr_end_seen;
	else
	    atr_state->fail = 1;
	break;	
    }
}

E_(atrvalue_seg_string_f atrvalue_seg_str_text_html)
void atrvalue_seg_str_text_html
   P_((struct tagfilter_atr_state * atr_state,
       struct string              * atr_value_segment,
       struct tagfilter_tag_state * tag_state,
       mime_t                     * body,
       struct in_state            * state_in,
       struct out_state           * state_out,
       charset_t                    text_charset,
       struct tagfilter_selection * tagfilter));
void atrvalue_seg_str_text_html(atr_state,atr_value_segment,
				    tag_state,body,state_in,
				    state_out,text_charset,tagfilter)
     struct tagfilter_atr_state * atr_state;
     struct string              * atr_value_segment;
     struct tagfilter_tag_state * tag_state;
     mime_t                     * body;
     struct in_state            * state_in;
     struct out_state           * state_out;
     charset_t                    text_charset;
     struct tagfilter_selection * tagfilter;
{

    if (TAGFILTER_SELECTION_magic != tagfilter->magic) 
	mime_panic(__FILE__,__LINE__,"atrvalue_seg_str_text_html",
		   "Bad magic number (tagfilter_selection)");

    if (TAGFILTER_TAG_STATE_magic !=  tag_state->magic)
	mime_panic(__FILE__,__LINE__,"atrvalue_seg_str_text_html",
		   "Bad magic number (tagfilter_tag_state)");

    if (TAGFILTER_ATR_STATE_magic !=  atr_state->magic) 
	mime_panic(__FILE__,__LINE__,"start_seg_str_text_html",
		   "Bad magic number (tagfilter_atr_state)");
    
    if (! atr_state->fail &&
	atr_start_seen == atr_state->pass) {
	
	switch (atr_state->code) {
	    int len,i;
	    char * dstr UNUSED_VAROK;
	case atr_none:       break;
	case atr_ol_start:
	    dstr = "<ol start>";
	    if (0) {
	    case atr_li_value :
	      dstr = "<li value>";
	    }
	    len = string_len(atr_value_segment);
	    
	    DPRINT(Debug,22,(&Debug,
			     "start_seg_str_text_html: %s segment %S\n",
			     dstr,atr_value_segment));
	    
	    for (i = 0; i < len; i++) {
	      uint16 u =
		give_unicode_from_string(atr_value_segment,
					 i);
	      /* UNICODE_BAD_CHAR is returned if unicode
		 value not found
	      */
	      
	      if (! atr_state->have_value &&
		  atr_sign_init == atr_state->sign_seen) {
		
		switch (u) {
		case 0x0009 /* TAB */:
		case 0x000A /* LF  */:
		case 0x000C /* FF  */:
		case 0x000D /* CR  */:
		case 0x0020 /* SPACE */:
		  
		  /* skip whitespace and take next character */
		  continue;
		  
		case 0x002D /* HYPHEN-MINUS */:			
		  atr_state->sign_seen = atr_minus_sign;
		  /* handle sign and go to value */
		  continue;
		  
		case 0x002B /* PLUS SIGN */:
		  atr_state->sign_seen = atr_sign_plus;
		  /* handle sign and go to value */
		  continue;		       			
		}		    
	      }
	    
	      if (0x0030 /* 0 */ <= u && u <= 0x0039 /* 9 */) {
		  int v = u - 0x0030 /* 0 */;
		
		  int val;
		  
		  atr_state->have_value = 1;
		  
		  if (atr_state->v.numeric > INT_MAX/10) {
		      
		      /* overflow */
		      
		      atr_state->fail = 1;
		      break;
		  }
		  
		  val = atr_state->v.numeric * 10;
		  
		  if (INT_MAX - val < v) {
		      /* overflow */
		      
		      atr_state->fail = 1;
		      break;
		  }
		  
		  val += v;
		  atr_state->v.numeric = val;
		  
	      } else {
		  /* fail value */
		  /* XXX should parsing just terminate
		     on bad character without error?
		  */
		  
		  atr_state->fail = 1;
		  break;
	      }		    		
	    }
	    	    
	    DPRINT(Debug,22,(&Debug,
			     "start_seg_str_text_html: numeric (after segment) %d%s sign_seen=%d",
			     atr_state->v.numeric,
			     atr_state->fail ? " (failed)" : "",
			     atr_state->sign_seen));
	    switch (atr_state->sign_seen) {
	    case atr_minus_sign: DPRINT(Debug,22,(&Debug," atr_minus_sign"));  break;
	    case atr_sign_init:  DPRINT(Debug,22,(&Debug," atr_sign_init"));   break;
	    case atr_sign_plus:  DPRINT(Debug,22,(&Debug," atr_sign_plus"));   break;
	    }
	    
	    if (atr_state->have_value) {
	      DPRINT(Debug,22,(&Debug," have value"));
	    } else {
	      DPRINT(Debug,22,(&Debug," no value"));
	    }
	    DPRINT(Debug,22,(&Debug,"\n"));
	    
	    break;
	case atr_ol_type:

	    len = string_len(atr_value_segment);
	    if (1 == len) {
		
		if (atr_type_none == atr_state->v.type_value) {
		    uint16 u =
			give_unicode_from_string(atr_value_segment,0);
		    
		    switch(u) {
		    case /* 1 */ 0x0031: atr_state->v.type_value = atr_type_decimal; break;
		    case /* a */ 0x0061: atr_state->v.type_value = atr_type_lalpha;  break; 
		    case /* A */ 0x0041: atr_state->v.type_value = atr_type_ualpha;  break;
		    case /* i */ 0x0069: atr_state->v.type_value = atr_type_lroman;  break;
		    case /* I */ 0x0049: atr_state->v.type_value = atr_type_uroman;  break;
		    }

		    if (atr_type_none == atr_state->v.type_value)
			atr_state->fail = 1;
		    else 
			atr_state->have_value = 1;

		} else
		    atr_state->fail = 1;
	    } else
		atr_state->fail = 1;
	    
	    break;
	case atr_img_alt:
	    state_putstring(atr_value_segment,state_out);

	    break;
	    
	}
    } else 
      atr_state->fail = 1;
}

E_(atrvalue_seg_ent_f atrvalue_seg_ent_text_html)
void atrvalue_seg_ent_text_html
       P_((struct tagfilter_atr_state * atr_state,
	   struct out_entity          * atr_value_segment,
	   struct tagfilter_tag_state * tag_state,
	   mime_t                     * body,
	   struct in_state            * state_in,
	   struct out_state           * state_out,
	   charset_t                    text_charset,
	   struct tagfilter_selection * tagfilter));
void atrvalue_seg_ent_text_html(atr_state,atr_value_segment,
				    tag_state,body,state_in,state_out,
				    text_charset,tagfilter)
     struct tagfilter_atr_state * atr_state;
     struct out_entity          * atr_value_segment;
     struct tagfilter_tag_state * tag_state;
     mime_t                     * body;
     struct in_state            * state_in;
     struct out_state           * state_out;
     charset_t                    text_charset;
     struct tagfilter_selection * tagfilter;
{

    if (TAGFILTER_SELECTION_magic != tagfilter->magic) 
	mime_panic(__FILE__,__LINE__,"atrvalue_seg_ent_text_html",
		   "Bad magic number (tagfilter_selection)");

    if (TAGFILTER_TAG_STATE_magic !=  tag_state->magic)
	mime_panic(__FILE__,__LINE__,"atrvalue_seg_ent_text_html",
		   "Bad magic number (tagfilter_tag_state)");

    if (TAGFILTER_ATR_STATE_magic !=  atr_state->magic) 
	mime_panic(__FILE__,__LINE__,"start_seg_str_text_html",
		   "Bad magic number (tagfilter_atr_state)");
    
    if (! atr_state->fail &&
	atr_start_seen == atr_state->pass) {

	switch (atr_state->code) {
	case atr_none:       break;
	case atr_ol_start:
	    DPRINT(Debug,22,(&Debug,
			     "atrvalue_seg_ent_text_html: <ol start> entity segment not supported\n"));
	    
	    /* Entities are not supported for <ol start> attribute */
	    atr_state->fail = 1;
	    break;
	case atr_li_value:
	  DPRINT(Debug,22,(&Debug,
			   "atrvalue_seg_ent_text_html: <li value> entity segment not supported\n"));
	    
	    /* Entities are not supported for <ol start> attribute */
	    atr_state->fail = 1;
	    break;
	case atr_ol_type:
	  DPRINT(Debug,22,(&Debug,
			   "atrvalue_seg_ent_text_html: <ol type> entity segment not supported\n"));
	    
	    /* Entities are not supported for <ol start> attribute */
	    atr_state->fail = 1;
	    break;
	case atr_img_alt:
	    state_putentity(atr_value_segment,state_out);
	    break;
	}
		
    } else 
	atr_state->fail = 1;

}

static struct tagfilter_stack_html_item * tagfilter_new_stack_html_item()
{
    struct tagfilter_stack_html_item * ret = safe_zero_alloc(sizeof (*ret));

    ret->magic      = TAGFILTER_STACK_HTML_ITEM_magic;
    ret->refcount   = 1;
    ret->list_depth = 0;

    ret->next_item_number = 0;
    ret->reversed_item_number = 0;
    ret->item_number_set = 0;
    
    return ret;
}

static struct tagfilter_stack_html_item *  tagfilter_give_stack_html_item(item)
     struct tagfilter_stack_item *item;
{
    struct tagfilter_stack_html_item * ret = NULL;

    if (item->u.html) {
	if (TAGFILTER_STACK_HTML_ITEM_magic != item->u.html->magic)
	    mime_panic(__FILE__,__LINE__,
		       "stack_item_clear_text_html",
		       "Bad magic number ( tagfilter_give_stack_html_item)");

	ret = item->u.html;
    } else {
	item->u.html = tagfilter_new_stack_html_item();
	
	ret = item->u.html;
    }

    return ret;
}


static void tagfilter_free_stack_html_item(ptr)
    struct tagfilter_stack_html_item **ptr;
{

    if (TAGFILTER_STACK_HTML_ITEM_magic != (*ptr)->magic)
	mime_panic(__FILE__,__LINE__,
		   "tagfilter_free_stack_html_item",
		   "Bad magic number (tagfilter_stack_html_item)");

    if ((*ptr)->refcount < 1)
	mime_panic(__FILE__,__LINE__,
		   "tagfilter_free_stack_html_item",
		   "Bad refcount");

    (*ptr)->refcount--;
    if ((*ptr)->refcount > 0) {
	/* Just reset pointer */
	
	*ptr = NULL;
	return;
    }
        
    (*ptr)->magic = 0;  /* Invalidate */
    free(*ptr);
    *ptr = NULL;
}


E_(stack_item_clear_f stack_item_clear_text_html)
void stack_item_clear_text_html
  P_((struct tagfilter_stack_item * item,
      struct tagfilter_selection  * tagfilter));
void stack_item_clear_text_html(item,tagfilter)
     struct tagfilter_stack_item * item;
     struct tagfilter_selection  * tagfilter;
{
    if (TAGFILTER_STACK_ITEM_magic != item->magic)
	mime_panic(__FILE__,__LINE__,
		   "stack_item_clear_text_html",
		   "Bad magic number (tagfilter_stack_item)");
    
    if (TAGFILTER_SELECTION_magic != tagfilter->magic) 
	mime_panic(__FILE__,__LINE__,"stack_item_clear_text_html",
		   "Bad magic number (tagfilter_selection)");

    if (item->u.html) 
	tagfilter_free_stack_html_item(&(item->u.html));				           
}

/* tagfilter_cmd --------------------------------------- */

/* Derement refcount -- free if not static */
void tagfilter_release_cmd(cmd)
     struct tagfilter_cmd **cmd;
{
    if (TAGFILTER_CMD_magic != (*cmd)->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_release_cmd",
		   "Bad magic number (tagfilter_cmd)");
    
    if ((*cmd)->refcount < 1)
	mime_panic(__FILE__,__LINE__,"tagfilter_release_cmd",
		   "Bad refcount");

    (*cmd)->refcount--;

    if ((*cmd)->refcount > 0) {
	*cmd = NULL;
	return;
    }

    /* string_command is always dynamically allocated */

    if ((*cmd)->string_command)
	free_string_sort(& ((*cmd)->string_command));

    /* Rest may be static */
    
    if (ison((*cmd)->cmd_mode,TAGFLT_CMD_is_static)) {
	*cmd = NULL;
	return;
    }

    if ((*cmd)->ascii_command) {
	free((*cmd)->ascii_command);
	(*cmd)->ascii_command = NULL;
    }

    (*cmd)->nesting = NULL;   /* static */

    (*cmd)->magic = 0;
    free(*cmd);
    *cmd = NULL;
}

void tagfilter_inc_cmd_refcount(cmd)
     struct tagfilter_cmd *cmd;
{
    if (TAGFILTER_CMD_magic != cmd->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_inc_cmd_refcount",
		   "Bad magic number (tagfilter_cmd)");
    cmd->refcount++;
}

void tagfilter_free_sorted_tags()
{
    static struct tagfilter_tags * check_tags [] = {

	& text_enriched_tags,
	& text_html_top,
	& text_html_head_body,
	& text_html_metadata,
	& text_html_flow,
	& text_html_phrasing,
	
	& text_html_close_head,
	& text_html_noscript_head,
	& text_html_datalist,
	& text_html_details,
	& text_html_dl,
	& text_html_fieldset,
	& text_html_figure,
	& text_html_footer,
	& text_html_form,
	& text_html_hgroup,
	& text_html_label,
	& text_html_list_items,
	& text_html_close_li,
	& text_html_close_p,
	& text_html_address,
	& text_html_meter,
	& text_html_picture,
	& text_html_progress,
	& text_html_ruby,
	& text_html_table,
	& text_html_div_dl,
	& text_html_caption,
	& text_html_colgroup,
	& text_html_tr_list,
	& text_html_tr,
	& text_html_close_thead,
	& text_html_close_tbody,
	& text_html_close_tr,
	& text_html_close_td,
	& text_html_th,
	& text_html_close_th,
	& text_html_close_dtdd,
	& text_html_close_rtrp
    };
    static size_t check_tags_count = (sizeof check_tags) / sizeof (check_tags [0]);
    
    size_t x;

    for (x = 0; x < check_tags_count; x++) {
	if (check_tags[x]) {
	    if (TAGFILTER_TAGS_magic != check_tags[x]->magic) 
		mime_panic(__FILE__,__LINE__,"tagfilter_free_sorted_tags",
			   "Bad magic number (tagfilter_tags)");
	    
	    if (check_tags[x]->sorted_tags)
		free_sort_list( &(check_tags[x]->sorted_tags));
		
	}
    }

}



/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */

