static char rcsid[] = "@(#)$Id: resolv.c,v 1.38 2022/11/02 16:26:29 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.38 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "elmresolv.h"
#include "save_opts.h"
#include "rc_imp.h"

#include <sys/time.h>

DEBUG_VAR(Debug,__FILE__,"resolv");

#ifndef h_errno
extern int h_errno;
#endif
#include <errno.h> 

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
extern int errno;
#endif

const size_t MAX_resolv_bufsize = 
#ifdef NS_MAXMSG
  NS_MAXMSG
#else
  65535
#endif
;

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

#define RESOLV_CACHE_magic	0xF90E

struct resolv_cache {
    
    unsigned short magic;     /* RESOLV_CACHE_magic */

    char * lowercase_name;    /* ASCII lowercased */

    enum cache_type {
	cache_none,
	cache_dummy,
	cache_real
    }  cache_type;

    union {
	void  * none;

#define DUMMY_RESOLV_CACHE_magic	0xF914

	struct dummy_resolv_cache {
	    unsigned short magic;    /* DUMMY_RESOLV_CACHE_magic */

	    /* if dummy cache */
	    
	    char                     * record_name;
	    struct resolv_cache      * record;

	} * dummy;

#define REAL_RESOLV_CACHE_magic		0xF915

	struct real_resolv_cache {
	    unsigned short magic;  /* REAL_RESOLV_CACHE_magic */

	    /* real cache */

#define RESOLV_RESULT_magic	0xF90F

	    struct resolv_result {
		unsigned short magic;     /* RESOLV_RESULT_magic */
		
		struct schedule_timelimit valid_until;            /* absolute time: now + ttl */
		char                    * name;                   /* case preserving name */
		
		ns_type                   rectype;
		
		union resolv_record {
		    struct in_addr    ip4addr;    /* ns_t_a, not a pointer */
#ifdef HAVE_IN6
		    struct in6_addr * ip6addr;    /* ns_t_aaaa */
#endif
		    
#define NAME_RESULT_magic	0xF910
		    
		    struct name_result {    /* ns_t_ns, ns_t_cname, ns_t_ptr */
			unsigned short magic;     /* NAME_RESULT_magic */
			
			char         * name;      /* case preserving name */
			
			struct resolv_cache * name_it;
		    }               * domain_name;
		    
#define TEXT_RESULT_magic	0xF911
		    
		    struct text_result {          /* ns_t_txt */
			unsigned short magic;     /* TEXT_RESULT_magic */
			
			char **text_item;
			int    text_item_count;
			
		    }               * text;
		    
#define MX_RESULT_magic		0xF912
		    
		    struct mx_result {            /* ns_t_mx */
			unsigned short magic;     /* MX_RESULT_magic */
			
			char         * name;      /* case preserving name */
			struct resolv_cache * name_it;
			uint16        precedence;
			
		    }               * mail_exchanger;
		    
#define SRV_RESULT_magic	0xF913

		    struct srv_result {           /* ns_t_srv */
			unsigned short magic;     /* SRV_RESULT_magic */
			
			char                * name;      /* case preserving name */
			struct resolv_cache * name_it;
			
			uint16      priority;
			uint16      weight;
			uint16      port;
			
		    }             * server;		    
		}  data;

		int refcount;
	    }  ** result_items;
	    int   result_items_count;

	} * real;

    } u;

    int refcount;

};

static struct sortlist * cache_items = NULL;

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

static const unsigned char *cs2us P_((const char *str));
static const unsigned char *cs2us(str) 
     const char *str;
{
    return (const unsigned char *)str;
}


static struct resolv_cache * malloc_resolv_cache P_((const char *name));

S_(alloc_sort_item_f alloc_cache_item)
static void alloc_cache_item P_((union sort_item             * res,
				 const union sort_key          key,
				 const union sort_item_default def
				 ));
static void alloc_cache_item(res,key,def)
     union sort_item      * res;
     const union sort_key   key;
     const union sort_item_default def;
{
    res->resolv = malloc_resolv_cache(key.str);
}

S_(free_sort_item_f free_cache_item)
static void free_cache_item P_((union sort_item      * ptr)); /* Decrements refcount */
static void free_cache_item(ptr)
     union sort_item      * ptr; /* Decrements refcount */
{
    free_resolv_cache(& (ptr->resolv));
}

S_(inc_sort_item_refcount_f inc_cache_item_refcount)
static void inc_cache_item_refcount P_((union sort_item item));
static void inc_cache_item_refcount(item)
     union sort_item item;
{
    inc_resolv_cache_refcount(item.resolv);
}

S_(sort_item_debug_name_f cache_item_name)
static struct string * cache_item_name   P_((const union sort_item item));
static struct string * cache_item_name (item)
     const union sort_item item;
{
    struct resolv_cache *cache = item.resolv;

     if (RESOLV_CACHE_magic != cache->magic)
	 panic("RESOLV PANIC",__FILE__,__LINE__,"cache_item_name",
	      "Bad magic number (resolv_cache)",0);

     return new_string2(ASCII_SET,s2us(cache->lowercase_name));
}

S_(compare_sort_key_to_item_f cache_key_cmp_item)
static int cache_key_cmp_item P_((const union sort_key key,
				  const union sort_item item));
static int cache_key_cmp_item(key,item)
     const union sort_key key;
     const union sort_item item;
{
    struct resolv_cache *cache = item.resolv;
    int r;
    
    if (RESOLV_CACHE_magic != cache->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"cache_key_cmp_item",
	      "Bad magic number (resolv_cache)",0);
    
    r = istrcmp(key.str,cache->lowercase_name);
    
    if (r < 0) {
	DPRINT(Debug,16,(&Debug,
			 "cache_key_cmp_item: name=\"%s\" < name=\"%s\"\n",
			 key.str,cache->lowercase_name));
    } else if (r > 0) {
	DPRINT(Debug,16,(&Debug,
			 "cache_key_cmp_item: name=\"%s\" > name=\"%s\"\n",
			 key.str,cache->lowercase_name));
    } else {
	DPRINT(Debug,14,(&Debug,
			 "cache_key_cmp_item: name=\"%s\" = name=\"%s\"\n",
			 key.str,cache->lowercase_name));
    }

    return r;
}

S_(sort_key_debug_name_f cache_key_name)
static struct string * cache_key_name P_((const union sort_key key));
static struct string * cache_key_name(key)
     const union sort_key key;
{
    return new_string2(ASCII_SET,cs2us(key.str));
}

static struct sort_operation resolv_cache_operation = {
    SORT_OPERATION_magic,
    alloc_cache_item,
    free_cache_item,
    inc_cache_item_refcount,
    cache_item_name,
    cache_key_cmp_item,

    /* KEY operations */
    cache_key_name
    
};


enum interface_change_v {
    intch_clear_cache_flag,
    
    NUM_interface_change_v
};


static char * INTERFACE_change[] = { "clear-cache",
				     NULL
};

static FLAGS interface_change = {
    FLAGVAL(intch_clear_cache_flag),
    0L,0L,
    NUM_interface_change_v,&(INTERFACE_change[0]),
    
    FLAGVAL(intch_clear_cache_flag),
    1, NULL
};   

static int pending_interface_change = 0;

static void free_name_result P_((struct name_result **ptr));
static void free_name_result(ptr)
     struct name_result **ptr;
{
    if ( NAME_RESULT_magic != (*ptr)->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"free_name_result",
	      "Bad magic number",0);

    if ((*ptr)->name) {
	free((*ptr)->name);
	(*ptr)->name = NULL;
    }

    if ((*ptr)->name_it)
	free_resolv_cache(& ((*ptr)->name_it));
    
    (*ptr)->magic = 0;   /* Invalidate */
    free(*ptr);
    *ptr = NULL;
}

static void free_text_result P_((struct text_result **ptr));
static void free_text_result(ptr)
     struct text_result **ptr;
{
    if (TEXT_RESULT_magic != (*ptr)->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"free_text_result",
	      "Bad magic number",0);

    if ((*ptr)->text_item) {
	int i;

	for (i = 0; i < (*ptr)->text_item_count; i++) {
	    if ((*ptr)->text_item[i]) {
		free((*ptr)->text_item[i]);
		(*ptr)->text_item[i] = NULL;
	    }
	}

	free((*ptr)->text_item);
	(*ptr)->text_item = NULL;
    }
    (*ptr)->text_item_count = 0;

    (*ptr)->magic = 0;   /* Invalidate */
    free(*ptr);
    *ptr = NULL;    
}

static void free_mx_result P_((struct mx_result **ptr));
static void free_mx_result(ptr) 
     struct mx_result **ptr;
{
    if (MX_RESULT_magic != (*ptr)->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"free_mx_result",
	      "Bad magic number",0);

    if ((*ptr)->name) {
	free((*ptr)->name);
	(*ptr)->name = NULL;
    }

    if ((*ptr)->name_it)
	free_resolv_cache(& ((*ptr)->name_it));
    
    (*ptr)->magic = 0;   /* Invalidate */
    free(*ptr);
    *ptr = NULL;
}

static void free_srv_result P_((struct srv_result **ptr));
static void free_srv_result(ptr)
     struct srv_result **ptr;
{
    if (SRV_RESULT_magic != (*ptr)->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"free_srv_result",
	      "Bad magic number",0);
    
    if ((*ptr)->name) {
	free((*ptr)->name);
	(*ptr)->name = NULL;
    }

    if ((*ptr)->name_it)
	free_resolv_cache(& ((*ptr)->name_it));
    
    (*ptr)->magic = 0;   /* Invalidate */
    free(*ptr);
    *ptr = NULL;
}

static void free_resolv_result P_((struct resolv_result **ptr));
static void free_resolv_result(ptr)
     struct resolv_result **ptr;
{    
    if (RESOLV_RESULT_magic != (*ptr)->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"free_resolv_result",
	      "Bad magic number",0);

    if ((*ptr)->refcount < 1)
	panic("RESOLV PANIC",__FILE__,__LINE__,"free_resolv_result",
	      "Bad refcount",0);

    (*ptr)->refcount--;
    if ((*ptr)->refcount > 0) {
	(*ptr) = NULL;
	return;
    }
    
    if ((*ptr)->name) {
	free((*ptr)->name);
	(*ptr)->name = NULL;
    }

    switch ((*ptr)->rectype) {
    case ns_t_a:   break;
#ifdef HAVE_IN6
    case ns_t_aaaa:	
	if ((*ptr)->data.ip6addr) {
	    free((*ptr)->data.ip6addr);
	    (*ptr)->data.ip6addr = NULL;
	}
	break;
#endif
    case ns_t_ns:
    case ns_t_cname:
    case ns_t_ptr:
	if ((*ptr)->data.domain_name)
	    free_name_result(& ((*ptr)->data.domain_name));
	break;
    case ns_t_txt:
	if ((*ptr)->data.text)
	    free_text_result( & ((*ptr)->data.text));
	break;
    case ns_t_mx:
	if ((*ptr)->data.mail_exchanger)
	    free_mx_result(& ((*ptr)->data.mail_exchanger));
	break;
    case ns_t_srv:
	if ((*ptr)->data.server)
	    free_srv_result(& ((*ptr)->data.server));
	break;
    default:
	panic("RESOLV PANIC",__FILE__,__LINE__,"free_resolv_result",
	      "Bad rectype",0);
    }

    (*ptr)->magic = 0;   /* Invalidate */
    free(*ptr);
    *ptr = NULL;
}



/* Increments refcount */
static void add_result_item P_((struct real_resolv_cache *ptr,
				struct resolv_result * record));
/* Increments refcount */
static void add_result_item(ptr,record)
     struct real_resolv_cache *ptr;
     struct resolv_result * record;
{
    if (REAL_RESOLV_CACHE_magic	 != ptr->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"add_result_item",
	      "Bad magic number (real_resolv_cache)",0);
    
    if (RESOLV_RESULT_magic != record->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"add_result_item",
	      "Bad magic number (resolv_result)",0);

    if (1 == ptr->result_items_count &&
	ptr->result_items &&
	ptr->result_items[0] &&
	ns_t_cname == record->rectype &&
	record->name &&
	record->data.domain_name) {

	if (RESOLV_RESULT_magic !=  ptr->result_items[0]->magic)
	    panic("RESOLV PANIC",__FILE__,__LINE__,"add_result_item",
		  "Bad magic number (resolv_result, result_items array)",0);

	if (NAME_RESULT_magic != record->data.domain_name->magic)
	    panic("RESOLV PANIC",__FILE__,__LINE__,"add_result_item",
		  "Bad magic number (name_result, record)", 0);

	if (ns_t_cname == ptr->result_items[0]->rectype &&
	    ptr->result_items[0]->name &&
	    0 == strcmp(record->name,
			ptr->result_items[0]->name) &&
	     ptr->result_items[0]->data.domain_name) {

	    if (NAME_RESULT_magic != ptr->result_items[0]->data.domain_name->magic)
		panic("RESOLV PANIC",__FILE__,__LINE__,"add_result_item",
		      "Bad magic number (name_result, result_items array)", 0);
	    
	    if (record->data.domain_name->name &&
		! record->data.domain_name->name_it &&
		ptr->result_items[0]->data.domain_name->name &&
		0 == strcmp(record->data.domain_name->name,
			    ptr->result_items[0]->data.domain_name->name)) {

		DPRINT(Debug,14,(&Debug,
				 "add_result_item: %s: merging CNAME %s record\n",
				 record->name,record->data.domain_name->name));

		if (schedule_extend_valid_until(& (ptr->result_items[0]->valid_until),
						& (record->valid_until))) {
		    char * A =  schedule_timeout_string(& (ptr->result_items[0]->valid_until));

		    if (A) {
			DPRINT(Debug,14,(&Debug,
					 "add_result_item: %s: extended valid until to %s\n",
					 record->name,A));
			free(A);
		    }							
		}

		return;
	    }
	}	    			
    }
	
    ptr->result_items = 
	safe_array_realloc(ptr->result_items,
			   (ptr->result_items_count +1),
			   sizeof (ptr->result_items[0]));

    ptr->result_items[ptr->result_items_count] = record;
    ptr->result_items[ptr->result_items_count]->refcount++;

    ptr->result_items_count++;
}

static void copy_real_resolv P_((struct real_resolv_cache *target,
				 struct real_resolv_cache *source,
				 const struct schedule_timelimit * now));
static void copy_real_resolv(target,source,now)
     struct real_resolv_cache *target;
     struct real_resolv_cache *source;
     const struct schedule_timelimit * now;
{
    if (REAL_RESOLV_CACHE_magic != target->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"copy_real_resolv",
	      "Bad magic number (target)",0);
    if (REAL_RESOLV_CACHE_magic != target->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"copy_real_resolv",
	      "Bad magic number (source)",0);

    if (source->result_items &&
	source->result_items_count > 0) {
	int len = 
	    source->result_items_count +
	    target->result_items_count;
	int i;
	int skipped = 0;

	target->result_items = 
	    safe_array_realloc(target->result_items,
			       len, sizeof (target->result_items[0]));

	for (i = 0; 
	     i < source->result_items_count &&
		 target->result_items_count < len; 
	     i++) {
	    if (RESOLV_RESULT_magic != source->result_items[i]->magic) 
		panic("RESOLV PANIC",__FILE__,__LINE__,"copy_real_resolv",
		      "Bad magic number (resolv_result)",0);

	    if (schedule_valid_until_ok(& (source->result_items[i]->valid_until),
					now)) {

		/* can just share reference */
	
		target->result_items[target->result_items_count]
		    = source->result_items[i];
		target->result_items[target->result_items_count]
		    ->refcount++;
		target->result_items_count++;
	    } else
		skipped++;
	}

	DPRINT(Debug,14,(&Debug,
			 "copy_real_resolv: %d skipped, total len %d, source len %d\n",
			 skipped,
			 target->result_items_count,
			 source->result_items_count));
    }
}

static struct real_resolv_cache * malloc_real_resolv_cache P_((void));

static void  unshare_resolv_cache P_((struct resolv_cache **ptr,
				       const struct schedule_timelimit * now));
static void  unshare_resolv_cache(ptr,now)
     struct resolv_cache **ptr;
     const struct schedule_timelimit * now;
{
    struct resolv_cache * unshared = NULL;

    if (RESOLV_CACHE_magic != (*ptr)->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"unshare_resolv_cache",
	      "Bad magic number",0);

    unshared = malloc_resolv_cache((*ptr)->lowercase_name);
    unshared->cache_type = cache_real;
    unshared->u.real = malloc_real_resolv_cache();

    switch((*ptr)->cache_type) {
    case cache_none:
	DPRINT(Debug,14,(&Debug,
			 "unshare_resolv_cache: %s -- cache type none\n",
			 (*ptr)->lowercase_name));
	break;
    case cache_dummy:
	DPRINT(Debug,14,(&Debug,
			 "unshare_resolv_cache: %s -- cache type dummy\n",
			 (*ptr)->lowercase_name));

	break;

    case cache_real:
	DPRINT(Debug,14,(&Debug,
			 "unshare_resolv_cache: %s -- cache type real\n",
			 (*ptr)->lowercase_name));
	
	copy_real_resolv(unshared->u.real,(*ptr)->u.real,now);

	break;
    default:
	panic("RESOLV PANIC",__FILE__,__LINE__,"unshare_resolv_cache",
	      "Bad cache type",0);
    }

    free_resolv_cache(ptr);
    *ptr = unshared;
}


/* Returns > 0 if item used */
static int add_related_to_name_result P_((struct name_result              * name,
					  struct resolv_result            * record,
					  const struct schedule_timelimit * now));
static int add_related_to_name_result(name,record,now)
     struct name_result              * name;
     struct resolv_result            * record;
     const struct schedule_timelimit * now;
{
    int count = 0;

    if (NAME_RESULT_magic != name->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"add_related_to_name_result",
	      "Bad magic number (name_result)",0);

    if (RESOLV_RESULT_magic != record->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"add_related_to_name_result",
	      "Bad magic number (resolv_result)",0);
   
    if (0 == istrcmp(name->name,record->name)) {
	
	if (!name->name_it) {
	    name->name_it =  malloc_resolv_cache(name->name);
	    name->name_it->cache_type = cache_real;
	    name->name_it->u.real = malloc_real_resolv_cache();
	} else if (RESOLV_CACHE_magic  != name->name_it->magic)
	    	panic("RESOLV PANIC",__FILE__,__LINE__,
		      "add_related_to_name_result",
		      "Bad magic number (resolv_cache)",0);
		
	/* changes name->name_it */
	if (name->name_it->refcount > 1 ||
	    cache_real != name->name_it->cache_type)
	    unshare_resolv_cache(& (name->name_it),now);

	if (name->name_it->refcount != 1)
	    	panic("RESOLV PANIC",__FILE__,__LINE__,
		      "add_related_to_name_result",
		      "Bad refcount",0);
		
	if (cache_real != name->name_it->cache_type)
	    	panic("RESOLV PANIC",__FILE__,__LINE__,
		      "add_related_to_name_result",
		      "Bad cache type",0);
	
	add_result_item(name->name_it->u.real,
			   record);

	count = 1;
    }

    return count;
}
    
/* Returns > 0 if item used */
static int add_related_to_mx_result P_((struct mx_result                * mx,
					struct resolv_result            * record,
					const struct schedule_timelimit * now));
static int add_related_to_mx_result(mx,record,now)
     struct mx_result                * mx;
     struct resolv_result            * record;
     const struct schedule_timelimit * now;
{
    int count = 0;

    if (MX_RESULT_magic != mx->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"add_related_to_mx_result",
	      "Bad magic number (mx_result)",0);

    if (RESOLV_RESULT_magic != record->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"add_related_to_mx_result",
	      "Bad magic number (resolv_result)",0);
    
    if (0 == istrcmp(mx->name,record->name)) {
	
	if (!mx->name_it) {
	    mx->name_it =  malloc_resolv_cache(mx->name);
	    mx->name_it->cache_type = cache_real;
	    mx->name_it->u.real = malloc_real_resolv_cache();

	} else if (RESOLV_CACHE_magic  != mx->name_it->magic)
	    	panic("RESOLV PANIC",__FILE__,__LINE__,
		      "add_related_to_mx_result",
		      "Bad magic number (resolv_cache)",0);
		
	/* changes mx->name_it */
	if (mx->name_it->refcount > 1 || 
	    cache_real != mx->name_it->cache_type)	   
	    unshare_resolv_cache(& (mx->name_it),now);

	if (mx->name_it->refcount != 1)
	    	panic("RESOLV PANIC",__FILE__,__LINE__,
		      "add_related_to_mx_result",
		      "Bad refcount",0);
	
	if (cache_real != mx->name_it->cache_type)
	    panic("RESOLV PANIC",__FILE__,__LINE__,
		  "add_related_to_name_result",
		  "Bad cache type",0);
	
	add_result_item(mx->name_it->u.real,
			   record);
	
	count = 1;
    }

    return count;
}

/* Returns > 0 if item used */
static int add_related_to_srv_result P_((struct srv_result               * srv,
					 struct resolv_result            * record,
					 const struct schedule_timelimit * now));
static int add_related_to_srv_result(srv,record,now)
     struct srv_result                * srv;
     struct resolv_result             * record;
      const struct schedule_timelimit * now;
{
    int count = 0;

    if (SRV_RESULT_magic != srv->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"add_related_to_srv_result",
	      "Bad magic number (srv_result)",0);

    if (RESOLV_RESULT_magic != record->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"add_related_to_srv_result",
	      "Bad magic number (resolv_result)",0);
   
    if (0 == istrcmp(srv->name,record->name)) {
	
	if (!srv->name_it) {
	    srv->name_it =  malloc_resolv_cache(srv->name);
	    srv->name_it->cache_type = cache_real;
	    srv->name_it->u.real = malloc_real_resolv_cache();
	} else if (RESOLV_CACHE_magic  != srv->name_it->magic)
	    	panic("RESOLV PANIC",__FILE__,__LINE__,
		      "add_related_to_srv_result",
		      "Bad magic number (resolv_cache)",0);
		
	/* changes srv->name_it */
	if (srv->name_it->refcount > 1 ||
	    cache_real != srv->name_it->cache_type)
	    unshare_resolv_cache(& (srv->name_it),now);

	if (srv->name_it->refcount != 1)
	    	panic("RESOLV PANIC",__FILE__,__LINE__,
		      "add_related_to_srv_result",
		      "Bad refcount",0);
	
	if (cache_real != srv->name_it->cache_type)
	    panic("RESOLV PANIC",__FILE__,__LINE__,
		  "add_related_to_name_result",
		  "Bad cache type",0);

	add_result_item(srv->name_it->u.real,
			record);

	count = 1;
    }

    return count;
}

/* Returns > 0 if item used */
static int add_related_data P_((struct real_resolv_cache         * ptr,
				struct resolv_result             * record,
				 const struct schedule_timelimit * now));

static int add_related_data(ptr,record,now) 
     struct real_resolv_cache         * ptr;
     struct resolv_result             * record;
      const struct schedule_timelimit * now;
{
    int count = 0;

    if (REAL_RESOLV_CACHE_magic	 != ptr->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"add_related_data",
	      "Bad magic number (real_resolv_cache)",0);
    
    if (RESOLV_RESULT_magic != record->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"add_related_data",
	      "Bad magic number (resolv_result)",0);

    if (ptr->result_items) {
	int i;

	for (i = 0; i < ptr->result_items_count; i++) {
	    if (RESOLV_RESULT_magic != ptr->result_items[i]->magic) 
		panic("RESOLV PANIC",__FILE__,__LINE__,"add_related_data",
		      "Bad magic number (resolv_result)",0);
	    
	    switch(ptr->result_items[i]->rectype) {
	    case ns_t_a:
	    case ns_t_aaaa:
	    case ns_t_txt:
		break;

	    case ns_t_ns:
	    case ns_t_cname: 
	    case ns_t_ptr:
		count +=
		    add_related_to_name_result(ptr->result_items[i]->
					       data.domain_name,
					       record,now);
		break;
	    case ns_t_mx:
		count +=
		    add_related_to_mx_result(ptr->result_items[i]->
					     data.mail_exchanger,
					     record,now);
		break;
	    case ns_t_srv:
		count +=
		    add_related_to_srv_result(ptr->result_items[i]->
					      data.server,
					      record,now);
		break;
	    default:
		panic("RESOLV PANIC",__FILE__,__LINE__,"add_related_data",
		      "Bad rectype",0);		    
	    }
	}
    }

    return count;
}


static void free_result_items P_((struct real_resolv_cache *ptr));
static void free_result_items(ptr) 
    struct real_resolv_cache *ptr;
{
    if (REAL_RESOLV_CACHE_magic	 != ptr->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"free_result_items",
	      "Bad magic number",0);
    
    if (ptr->result_items) {
	int i;

	for (i = 0; i < ptr->result_items_count; i++) 
	    if (ptr->result_items[i])
		free_resolv_result(& (ptr->result_items[i]));
	free(ptr->result_items);
	ptr->result_items = NULL;
    }
    ptr->result_items_count = 0;
}


static void free_dummy_resolv_cache P_((struct dummy_resolv_cache **ptr));
static void free_dummy_resolv_cache(ptr)
     struct dummy_resolv_cache **ptr;
{
    if (DUMMY_RESOLV_CACHE_magic != (*ptr)->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"free_dummy_resolv_cache",
	      "Bad magic number",0);
    
    if ((*ptr)->record_name) {
	free((*ptr)->record_name);
	(*ptr)->record_name = NULL;
    }

    if ((*ptr)->record)
	free_resolv_cache(& ((*ptr)->record));

    (*ptr)->magic =  0;  /* Invalidate */
    free(*ptr);
    *ptr = NULL;
}

static void free_real_resolv_cache P_((struct real_resolv_cache **ptr));
static void free_real_resolv_cache(ptr)
     struct real_resolv_cache **ptr;
{
    if (REAL_RESOLV_CACHE_magic != (*ptr)->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"free_real_resolv_cache",
	      "Bad magic number",0);
 
    if ((*ptr)->result_items) 
	free_result_items(*ptr);

    (*ptr)->magic = 0;   /* Invalidate */
    free(*ptr);
    *ptr = NULL;
}

/* Decrements refcount */
void free_resolv_cache(ptr)
     struct resolv_cache **ptr;
{
    if (RESOLV_CACHE_magic != (*ptr)->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"free_resolv_cache",
	      "Bad magic number",0);

    if ((*ptr)->refcount < 1)
	panic("RESOLV PANIC",__FILE__,__LINE__,"free_resolv_cache",
	      "Bad refcount",0);

    (*ptr)->refcount--;
    if ((*ptr)->refcount > 0) {
	(*ptr) = NULL;
	return;
    }

    if ((*ptr)->lowercase_name) {
	free((*ptr)->lowercase_name);
	(*ptr)->lowercase_name = NULL;
    }

    switch((*ptr)->cache_type) {
    case cache_none:
	if ((*ptr)->u.none) 
	    panic("RESOLV PANIC",__FILE__,__LINE__,"free_resolv_cache",
		  "cache type none have data",0);
	break;
    case cache_dummy:
	if ((*ptr)->u.dummy)
	    free_dummy_resolv_cache(& ((*ptr)->u.dummy));
	break;
    case cache_real:
	if ((*ptr)->u.real)
	    free_real_resolv_cache(& ((*ptr)->u.real));
	break;
    default:
	 panic("RESOLV PANIC",__FILE__,__LINE__,"free_resolv_cache",
		  "Bad cache type",0);
    }

    (*ptr)->magic = 0;   /* Invalidate */
    free(*ptr);
    *ptr = NULL;
}

static struct dummy_resolv_cache * 
  malloc_dummy_resolv_cache P_((const char               * record_name,
				struct resolv_cache      * record));
static struct dummy_resolv_cache * 
  malloc_dummy_resolv_cache(record_name,record)
     const char               * record_name;
     struct resolv_cache      * record;
{
    struct dummy_resolv_cache *ret = safe_zero_alloc(sizeof (* ret));
    
    if (RESOLV_CACHE_magic != record->magic)
	 panic("RESOLV PANIC",__FILE__,__LINE__,
	       "malloc_dummy_resolv_cache",
	       "Bad magic number",0);

    ret->record_name = safe_strdup(record_name);
    ret->record      = record;
    ret->record->refcount++;

    ret->magic = DUMMY_RESOLV_CACHE_magic;

    return ret;
}

static struct real_resolv_cache * malloc_real_resolv_cache() 
{
    struct real_resolv_cache *ret = safe_zero_alloc(sizeof (* ret));

    ret->result_items       = NULL;
    ret->result_items_count = 0;

    ret->magic = REAL_RESOLV_CACHE_magic;

    return ret;
}

static struct resolv_cache * malloc_resolv_cache(name)
     const char *name;
{
    struct resolv_cache *ret = safe_zero_alloc(sizeof (* ret));

    char *c ;

    ret->lowercase_name  = safe_strdup(name);

    /* lowercase ascii */    
    for (c = ret->lowercase_name; *c; c++) {
	if (isascii(*c))
	    *c = tolower(*c);
    }

    ret->cache_type = cache_none;
    ret->u.none = NULL;

    ret->refcount = 1;
    ret->magic = RESOLV_CACHE_magic;
    return ret;
}

void inc_resolv_cache_refcount(cache)
     struct resolv_cache *cache;
{
    if (RESOLV_CACHE_magic != cache->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"inc_resolv_cache_refcount",
	      "Bad magic number (resolv_cache)",0);
    cache->refcount++;
}

static struct resolv_cache * give_resolv_cache P_((const char *name));
static struct resolv_cache * give_resolv_cache(name)
     const char *name;
{
    
    union sort_key     key;
    union sort_item_default def;
    union sort_item    res;
    size_t             res_idx = 0;

    struct resolv_cache * ret = NULL;

    DPRINT(Debug,14,(&Debug,
		    "give_resolv_cache: name=\"%s\"\n",name));

    if (!cache_items) {
	DPRINT(Debug,14,(&Debug,
			"give_resolv_cache: Creating cache_items\n"));
	
	cache_items = alloc_sort_list(&resolv_cache_operation,
				      1 /* prealloc one item */);
    }

    key.str  = name;
    def.dummy = NULL;
    res.dummy = NULL;

    /* Increments refcount, returns 1 if found */
    if (search_sort_list_item(cache_items,sort_list_search_create,
			      key,def,&res,&res_idx,
			      NULL /* append mode rewrite hint */)) {
	
	DPRINT(Debug,14,(&Debug,
			"give_resolv_cache: search_sort_list_item found or created item #%zu\n",
			res_idx));

	ret = res.resolv;
	
	DPRINT(Debug,14,(&Debug,
			 "give_resolv_cache: name=\"%s\" = [%zu] name=\"%s\"\n",
			 name,res_idx,ret->lowercase_name));
	
    } else {
	DPRINT(Debug,14,(&Debug,
			 "give_resolv_cache: search_sort_list_item failed to create entry\n"));

	ret = malloc_resolv_cache(name);
	
	DPRINT(Debug,14,(&Debug,
			 "give_resolv_cache: name=\"%s\" = [no index] name=\"%s\"\n",
			 name,ret->lowercase_name));	
    }
    
    DPRINT(Debug,14,(&Debug,
		     "give_resolv_cache=%p (refcount=%d)\n",
		     ret,ret->refcount));
    return ret;
}

static struct name_result * malloc_name_result P_((char *domain_name));
static struct name_result * malloc_name_result(domain_name)
     char *domain_name;
{
    struct name_result * ret = safe_zero_alloc(sizeof (*ret));

    ret->name = safe_strdup(domain_name);

    ret->magic = NAME_RESULT_magic;

    return ret;
}

static struct mx_result * malloc_mx_result P_((char *domain_name,
					       int precedence));
static struct mx_result * malloc_mx_result(domain_name,precedence)
     char *domain_name;
     int precedence;
{
    struct mx_result * ret = safe_zero_alloc(sizeof (*ret));

    ret->name = safe_strdup(domain_name);
    ret->name_it = NULL;  /* Possibly filled later */
    ret->precedence = precedence;

    ret->magic =  MX_RESULT_magic;
    
    return ret;
}

/* Destroys vector */
static struct text_result * malloc_text_result P_((char **vector,
						   int vector_len));
static struct text_result * malloc_text_result(vector,vector_len)
     char **vector;
     int vector_len;
{
    struct text_result * ret = safe_zero_alloc(sizeof (*ret));

    ret->text_item = NULL;
    ret->text_item_count = 0;

    if (vector_len > 0) {
	int i;

	ret->text_item = safe_calloc(vector_len, 
				     sizeof(ret->text_item[0]));

	for (i = 0; i < vector_len; i++) {
	    ret->text_item[i] = vector[i];
	    vector[i] = NULL;
	}
	ret->text_item_count = vector_len;
    }

    ret->magic = TEXT_RESULT_magic;
    return ret;
}

static struct srv_result * malloc_srv_result P_((char *domain_name,
						 int priority,
						 int weight,int port));
static struct srv_result * malloc_srv_result(domain_name,priority,
					     weight,port)
     char *domain_name;
     int priority;
     int weight;
     int port;
{
    struct srv_result *ret = safe_zero_alloc(sizeof (*ret));

    ret->name = safe_strdup(domain_name);
    ret->name_it = NULL;  /* Possibly filled later */
    ret->priority = priority;
    ret->weight   = weight;
    ret->port     = port;

    ret->magic = SRV_RESULT_magic;

    return ret;
}

static void update_next_resolv_cleanup P_((const struct schedule_timelimit * valid_until)) ;

static struct resolv_result * malloc_resolv_result P_((ns_msg *parsed_answer,
						       ns_rr *answer_record,
						       const struct schedule_timelimit * now));
static struct resolv_result * malloc_resolv_result(parsed_answer,
						   answer_record,
						   now)
     ns_msg *parsed_answer;
     ns_rr *answer_record;
     const struct schedule_timelimit * now;
{
    struct resolv_result *ret = NULL;

    char   * recname  = ns_rr_name(*answer_record);
    ns_class recclass = ns_rr_class(*answer_record);
    ns_type  rectype  = ns_rr_type(*answer_record);
    unsigned int ttl  = ns_rr_ttl(*answer_record);
    const unsigned char * walk_ptr = ns_rr_rdata(*answer_record);
    int      size_left = ns_rr_rdlen(*answer_record);
    union resolv_record record;

    /* bzero is defined on hdrs/defs.h */
    bzero((void *)&record,sizeof (record));


    if ( ns_c_in != recclass) {
	DPRINT(Debug,14,(&Debug,
			 "malloc_resolv_result=NULL; recclass=%d recname=%s\n",
			 recclass,
			 recname ? recname : "<NULL>"));
	return NULL;
    }

    if (!recname) {
	DPRINT(Debug,14,(&Debug,
			 "malloc_resolv_result=NULL; no name\n",
			 recname));
	return NULL;
    }

    switch(rectype) {
	union {
	    struct in_addr  ip4addr;
#ifdef HAVE_IN6
	    struct in6_addr ip6addr;
#endif	    

	    unsigned char buffer[16];
	} X;

    case ns_t_a:
	if (size_left >= sizeof (X.ip4addr)) {
	    memcpy(X.buffer,walk_ptr,sizeof (X.ip4addr));

	    walk_ptr  += sizeof (X.ip4addr);
	    size_left -= sizeof (X.ip4addr);

	    record.ip4addr = X.ip4addr;
	} else {
	    DPRINT(Debug,14,(&Debug,
			     "malloc_resolv_result=NULL; rectype=ns_t_a (%d), size_left=%d recname=%s\n",
			     rectype,size_left,recname));
	    return NULL;
	}
	break;

    case ns_t_ns:
    case ns_t_cname:
    case ns_t_ptr:
	if (size_left > 0) {
	  char domain_name[RESOLV_BUFSIZE];
	  int compressed_len;

	  compressed_len = 
	      ns_name_uncompress(ns_msg_base((*parsed_answer)),
				 ns_msg_end((*parsed_answer)),
			       walk_ptr,
			       domain_name,
			       sizeof domain_name);

	  if (compressed_len < 0) {
	      DPRINT(Debug,14,(&Debug,
			       "malloc_resolv_result=NULL; rectype=%d, failed to expand name, recname=%s\n",
			       rectype,recname));
	      return NULL;
	  }

	  walk_ptr  += compressed_len;
	  size_left -= compressed_len;

	  record.domain_name = malloc_name_result(domain_name);
	} else {
	    DPRINT(Debug,14,(&Debug,
			     "malloc_resolv_result=NULL; rectype=%d, size_left=%d, recname=%s\n",
			     rectype,size_left,recname));
	    return NULL;
	}
	break;
    case ns_t_mx:
	if (size_left > 2) {
	    int precedence;
	    char domain_name[RESOLV_BUFSIZE];
	    int compressed_len;

	    precedence = ns_get16(walk_ptr);
	    walk_ptr  += 2;
	    size_left -= 2;

	    compressed_len = 
		ns_name_uncompress(ns_msg_base((*parsed_answer)),
				   ns_msg_end((*parsed_answer)),
				   walk_ptr,
			       domain_name,
			       sizeof domain_name);

	    if (compressed_len < 0) {
		DPRINT(Debug,14,(&Debug,
				 "malloc_resolv_result=NULL; rectype=ns_t_mx (%d), failed to expand name, recname=%s\n",
				 rectype,recname));
		return NULL;
	    }

	    record.mail_exchanger = malloc_mx_result(domain_name,
						     precedence);

	} else {
	    DPRINT(Debug,14,(&Debug,
			     "malloc_resolv_result=NULL; rectype=ns_t_mx (%d), size_left=%d, recname=%s\n",
			     rectype,size_left,recname));
	    return NULL;
	}
	break;

#ifdef HAVE_IN6
    case ns_t_aaaa:
	if (size_left >= sizeof (X.ip6addr)) {
	    memcpy(X.buffer,walk_ptr,sizeof (X.ip6addr));
	  
	    walk_ptr  += sizeof (X.ip6addr);
	    size_left -= sizeof (X.ip6addr);

	    record.ip6addr = safe_malloc(sizeof (* record.ip6addr));

	    *record.ip6addr = X.ip6addr;
	} else {
	    DPRINT(Debug,14,(&Debug,
			     "malloc_resolv_result=NULL; rectype=ns_t_aaaa (%d), size_left=%d, recname=%s\n",
			     rectype,size_left,recname));
	    return NULL;
	}
	break;
#endif

    case ns_t_txt: 
	if (size_left < 0) {
	    DPRINT(Debug,14,(&Debug,
			     "malloc_resolv_result=NULL; rectype=ns_t_txt (%d), size_left=%d, recname=%s\n",
			     rectype,size_left,recname));
	    return NULL;
	} else {
	    char **vector = NULL;
	    int vector_len = 0;
	    int ok = 0;
	    	    
	    while (size_left > 0) {
		
		int len =  *walk_ptr;
		
		if (len > size_left + 1) {
		    DPRINT(Debug,14,(&Debug,
				     "malloc_resolv_result: rectype=ns_t_aaaa (%d), size_left=%d, failed to read text record, recname=%s\n",
				     rectype,size_left,recname));
		    
		    ok = 0;
		    break;		
		}
		ok = 1;

		walk_ptr ++;
		size_left --;

		vector = safe_array_realloc(vector,(vector_len+1), sizeof(vector[0]));
		vector[vector_len] = safe_malloc(len+1);
		
		memcpy(vector[vector_len],walk_ptr,len);
		vector[vector_len][len] = '\0';
		vector_len++;
		
		walk_ptr   += len;
		size_left -= len;

	    }

	    if (ok)                       /* Destroys vector */
		record.text = malloc_text_result(vector,vector_len);

	    if (vector) {
		int i;

		for (i = 0; i < vector_len; i++) {
		    if (vector[i]) {
			free(vector[i]);
			vector[i] = NULL;
		    }
		}

		free(vector);
		vector = NULL;
	    }

	    if (!ok) {
		DPRINT(Debug,14,(&Debug,
				 "malloc_resolv_result=NULL; rectype=ns_t_txt (%d),  recname=%s\n",
				 rectype,recname));
		return NULL;
	    }
	}
	break;
    case ns_t_srv:

	if (size_left > 6) {
	    int priority, weight, port;
	    char domain_name[RESOLV_BUFSIZE];
	    int compressed_len;

	    priority = ns_get16(walk_ptr);
	    walk_ptr  += 2;
	    size_left -= 2;
	  
	    weight = ns_get16(walk_ptr);
	    walk_ptr  += 2;
	    size_left -= 2;
	    
	    port = ns_get16(walk_ptr);
	    walk_ptr  += 2;
	    size_left -= 2;
	    
	    compressed_len = 
		ns_name_uncompress(ns_msg_base((*parsed_answer)),
				   ns_msg_end((*parsed_answer)),
				   walk_ptr,
				   domain_name,
				   sizeof domain_name);
	    
	    if (compressed_len < 0) {
		DPRINT(Debug,14,(&Debug,
				 "malloc_resolv_result=NULL; rectype=ns_t_srv (%d), size_left=%d, failed to expand name, recname=%s\n",
				 rectype,size_left,recname));
		return NULL;
	    }
	    
	    walk_ptr  += compressed_len;
	    size_left -= compressed_len;
	    
	    record.server = malloc_srv_result(domain_name,
					      priority,weight,port);
	} else {
	    DPRINT(Debug,14,(&Debug,
			     "malloc_resolv_result=NULL; rectype=ns_t_srv (%d), size_left=%d, recname=%s\n",
			     rectype,size_left,recname));
	    return NULL;
	    
	}
	break;
    default:
	DPRINT(Debug,14,(&Debug,
			 "malloc_resolv_result=NULL; rectype=%d, size_left=%d, recname=%s\n",
			 rectype,size_left,recname));
	return NULL;
    }

    if (0 != size_left) {
	DPRINT(Debug,14,(&Debug,
			 "malloc_resolv_result=NULL; rectype=%d, size_left=%d trailing bytes, recname=%s\n",
			 rectype,size_left,recname));
    }

    ret = safe_zero_alloc(sizeof (*ret));

    if (! schedule_have_timelimit(now)) {
	DPRINT(Debug,14,(&Debug,
			 "malloc_resolv_result: no time\n"));
	ret->valid_until = NO_schedule_timelimit;
	
    } else if (/* May fail at Tue Jan 19 03:14:07 2038 */
	       schedule_set_next_timeout0(& (ret->valid_until),now,
					  ttl,
					  /* Use time_MAX -1 on failure
					     May fail at Tue Jan 19 03:14:06 2038 */
					  se_use_before_time_MAX)) {

	char * valid_until_s = schedule_timeout_string(& (ret->valid_until));

	if (valid_until_s) {
	    DPRINT(Debug,14,(&Debug,
			     "malloc_resolv_result: setting valid_until %s\n",
			     valid_until_s));
	    free(valid_until_s);
	}

	update_next_resolv_cleanup(& (ret->valid_until));
    } else {
	DPRINT(Debug,14,(&Debug,
			 "malloc_resolv_result: time overflow\n"));
    }
    
    ret->name     = safe_strdup(recname);
    ret->rectype  = rectype;
    ret->data     = record;
    ret->refcount = 1;
    ret->magic    = RESOLV_RESULT_magic;

    DPRINT(Debug,14,(&Debug,
		     "malloc_resolv_result=%p; rectype=%d, name=%s\n",
		     ret,ret->rectype,ret->name));

    return ret;
}



struct resolv_cache * fill_resolv_cache(answer,parsed_answer,now,
					       search_name,looking_up,
					       have_error)
     struct resolv_cache             * answer;
     ns_msg                          * parsed_answer;
     const struct schedule_timelimit * now;
     char                           ** search_name;
     enum looking_up                 * looking_up;
     int                             * have_error;
{
    struct resolv_cache * ret = NULL;

    ns_sect sect;

    if (RESOLV_CACHE_magic != answer->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"fill_resolv_cache",
	      "Bad magic number",0);

    if (search_name && *search_name) {
	DPRINT(Debug,14,(&Debug,
			 "fill_resolv_cache: %s: clearing search name %s\n",
			 answer->lowercase_name,*search_name));
	free(*search_name);
	*search_name = NULL;
    }

    DPRINT(Debug,14,(&Debug,
		     "fill_resolv_cache: %s: cache type %d",
		     answer->lowercase_name,
		     answer->cache_type));
    switch(answer->cache_type) {
    case cache_real:   DPRINT(Debug,14,(&Debug," (real)")); break;
    case cache_dummy:  DPRINT(Debug,14,(&Debug," (dummy)")); break;
    case cache_none:   DPRINT(Debug,14,(&Debug," (none)")); break;
    }
    DPRINT(Debug,14,(&Debug,"\n"));

    
    for (sect = 0; sect < ns_s_max; sect++) {
	int count = ns_msg_count((*parsed_answer),sect);
	int recnum;

	for (recnum = 0; recnum < count; recnum++) {
	
	    ns_rr   answer_record;
	    int r;
    
	    char   * recname = NULL;
	    ns_class recclass;

	    errno = 0;
	    r = ns_parserr(parsed_answer,sect,recnum,&answer_record);
	    if (r < 0) {
		int err = errno;
		
		DPRINT(Debug,14,(&Debug,
				 "fill_resolv_cache: %s: section %d, record %d: failed to parse answer\n",
				 answer->lowercase_name,sect,recnum));

		if (have_error)
		    (*have_error)++;

		if (err)
		    lib_error(CATGETS(elm_msg_cat, 
				      ResolvSet,
				      ResolvLookFParseAnswerE,
				      "Looking up %s: Failed to parse answer: %s"),
			      answer->lowercase_name,
			      strerror(err));
		else
		    lib_error(CATGETS(elm_msg_cat, 
				      ResolvSet,
				      ResolvLookFParseAnswer,
				      "Looking up %s: Failed to parse answer"),
			      answer->lowercase_name);

		if (looking_up)
		    *looking_up = l_up_error;
		
		if (ret)
		    free_resolv_cache(&ret);
		goto out;
	    }
	    
	    recname  = ns_rr_name(answer_record);	   
	    recclass = ns_rr_class(answer_record);

	    if (!recname || ns_c_in != recclass) {
		DPRINT(Debug,14,(&Debug,
				 "fill_resolv_cache: %s: section %d, record %d: bad data\n",
				 answer->lowercase_name,sect,recnum)); 

		if (have_error)
		    (*have_error)++;

		lib_error(CATGETS(elm_msg_cat, 
				  ResolvSet,
				  ResolvLookBadAnswer,
				  "Looking up %s: bad answer"),
			  answer->lowercase_name);
		
		if (looking_up)
		    *looking_up = l_up_error;

		if (ret)
		    free_resolv_cache(&ret);

		goto out;		
	    }

	    switch (sect) {

	    case ns_s_qd:   /* look entry name on question */

		if (ret) {
		    DPRINT(Debug,14,(&Debug,
				     "fill_resolv_cache: %s: section %d, record %d: duplicate question\n",
				     answer->lowercase_name,sect,recnum)); 

		    if (have_error)
			(*have_error)++;

		    lib_error(CATGETS(elm_msg_cat, 
				      ResolvSet,
				      ResolvLookBadAnswer,
				      "Looking up %s: bad answer"),
			      answer->lowercase_name);

		    if (looking_up)
			*looking_up = l_up_error;

		    free_resolv_cache(&ret);
		    goto out;
		}

		if (search_name && !*search_name)
		    *search_name = safe_strdup(recname);

		switch(answer->cache_type) {
		case cache_real:
		    
		    if (0 != istrcmp(answer->lowercase_name,
				     recname)) {

			DPRINT(Debug,14,(&Debug,
					 "fill_resolv_cache: %s: section %d, record %d: record for %s, but answer is for %s\n",
					 answer->lowercase_name,sect,recnum,
					 answer->lowercase_name,
					 recname));
			
			if (have_error)
			    (*have_error)++;

			lib_error(CATGETS(elm_msg_cat, 
					  ResolvSet,
					  ResolvLookParsedAnswer,
					  "Looking up %s: Got %s (unexpected)"),
				  answer->lowercase_name,recname);

			if (looking_up)
			    *looking_up = l_up_error;			

			if (answer->u.real)
			    free_real_resolv_cache(& (answer->u.real));
			answer->cache_type = cache_none;

			DPRINT(Debug,14,(&Debug,
					 "fill_resolv_cache: %s: section %d, record %d: reseting cache",
					 answer->lowercase_name,sect,recnum));

			goto reset_cache;
		    }

		    ret = answer;
		    inc_resolv_cache_refcount(ret);
		    break;
		case cache_dummy:
		    if (0 == istrcmp(answer->lowercase_name,
				     recname)) {

			if (answer->u.dummy)
			    free_dummy_resolv_cache(& (answer->u.dummy));
			answer->cache_type = cache_none;
			
			DPRINT(Debug,14,(&Debug,
					 "fill_resolv_cache: %s: section %d, record %d: reseting cache\n",
					 answer->lowercase_name,sect,recnum));
			
			goto reset_cache;
		    }

		    if (!answer->u.dummy) {
			answer->cache_type = cache_none;
			
			DPRINT(Debug,14,(&Debug,
					 "fill_resolv_cache: %s: section %d, record %d: reseting cache\n",
					 answer->lowercase_name,sect,recnum));
			
			goto reset_cache;
		    }

		    if (DUMMY_RESOLV_CACHE_magic != answer->u.dummy->magic)
			panic("RESOLV PANIC",__FILE__,__LINE__,
			      "fill_resolv_cache",
			      "Bad dummy cache magic number",0);

		    if (0 != strcmp(answer->u.dummy->record_name,
				    recname)) {

			if (answer->u.dummy)
			    free_dummy_resolv_cache(& (answer->u.dummy));
			answer->cache_type = cache_none;
			
			DPRINT(Debug,14,(&Debug,
					 "fill_resolv_cache: %s: section %d, record %d: reseting cache\n",
					 answer->lowercase_name,sect,recnum));
			
			goto reset_cache;			
		    }

		    if (RESOLV_CACHE_magic != answer->u.dummy->record->magic)
			panic("RESOLV PANIC",__FILE__,__LINE__,
			      "fill_resolv_cache",
			      "Bad cache magic number",0);
		    
		    ret = answer->u.dummy->record;
		    inc_resolv_cache_refcount(ret);

		    if (0 != istrcmp(ret->lowercase_name,
				     recname)) 
			panic("RESOLV PANIC",__FILE__,__LINE__,
			      "fill_resolv_cache",
			      "Bad real cache record name",0);

		    DPRINT(Debug,14,(&Debug,
				     "fill_resolv_cache: %s: section %d, record %d: recname %s -- using cache record %s\n",
				     answer->lowercase_name,sect,recnum,
				     recname,
				     ret->lowercase_name));

		    break;

		case cache_none:
		reset_cache:
		    
		    if (answer->u.none)
			panic("RESOLV PANIC",__FILE__,__LINE__,
			      "fill_resolv_cache",
			      "Data with cache type none",0);

		    if (0 == istrcmp(answer->lowercase_name,
				     recname)) {
			/* This is real cache */
			
			ret = answer;
			inc_resolv_cache_refcount(ret);

			DPRINT(Debug,14,(&Debug,
					 "fill_resolv_cache: %s: section %d, record %d: recname %s -- using cache record %s\n",
					 answer->lowercase_name,sect,recnum,
					 recname,
					 ret->lowercase_name));


		    } else {
			ret = malloc_resolv_cache(recname);
		     
			/* Build dummy */

			answer->cache_type = cache_dummy;
			answer->u.dummy  = 
			    malloc_dummy_resolv_cache(recname,
						      ret);
			
			DPRINT(Debug,14,(&Debug,
					 "fill_resolv_cache: %s: section %d, record %d: recname %s -- using cache record %s\n",
					 answer->lowercase_name,sect,recnum,
					 recname,
					 ret->lowercase_name));


		    }

		    if (cache_none != ret->cache_type)
			panic("RESOLV PANIC",__FILE__,__LINE__,
			      "fill_resolv_cache",
			      "Bad cache type",0);
		    if (ret->u.none)
			panic("RESOLV PANIC",__FILE__,__LINE__,
			      "fill_resolv_cache",
			      "Data with cache type none",0);

		    ret->cache_type = cache_real;
		    ret->u.real = malloc_real_resolv_cache();
		}

		break;
		
	    case ns_s_an:  /* Look answer */ 
	    case ns_s_ar:  /* additional records */
		if (ret) {
		    struct resolv_result * 
			record = malloc_resolv_result(parsed_answer,
						      &answer_record,now);

		    if (!record) {
			DPRINT(Debug,14,(&Debug,
					 "fill_resolv_cache: %s: section %d, record %d: no record parsed\n",
					 answer->lowercase_name,sect,recnum));
			break;
		    }
		
		    if (cache_real != ret->cache_type)
			panic("RESOLV PANIC",__FILE__,__LINE__,
			      "fill_resolv_cache",
			      "Bad cache type",0);

		    if (RESOLV_RESULT_magic != record->magic)
			panic("RESOLV PANIC",__FILE__,__LINE__,
			      "fill_resolv_cache",
			      "Bad magic number",0);

		    if (ns_s_an == sect  &&
			0 == istrcmp(record->name,ret->lowercase_name)) {
			DPRINT(Debug,14,(&Debug,
					 "fill_resolv_cache: %s: section %d, record %d: inserting record %s as answer\n",
					 answer->lowercase_name,sect,recnum,
					 record->name));

			add_result_item(ret->u.real,record);

		    } else {
			if (add_related_data(ret->u.real,record,now)) {
			    DPRINT(Debug,14,(&Debug,
					     "fill_resolv_cache: %s: section %d, record %d: inserting record %s to related data\n",
					      answer->lowercase_name,sect,recnum,
					     record->name));

			} else {
			    DPRINT(Debug,14,(&Debug,
					     "fill_resolv_cache: %s: section %d, record %d: ignored record %s\n",
					     answer->lowercase_name,sect,recnum,
					     record->name));
			}

		    }

		    free_resolv_result(&record);
		} else {
		    DPRINT(Debug,14,(&Debug,
				     "fill_resolv_cache: %s: section %d, record %d: question not seen",
				     answer->lowercase_name,sect,recnum));

		    lib_error(CATGETS(elm_msg_cat, 
				      ResolvSet,
				      ResolvLookBadAnswer,
				      "Looking up %s: bad answer"),
			      answer->lowercase_name);

		    if (looking_up)
			*looking_up = l_up_error;

		    goto out;			
		}
		    	    
		break;

	    case ns_s_ns:   /* name servers */
		break;


	    default:
		DPRINT(Debug,14,(&Debug,
				 "fill_resolv_cache: %s: section %d, record %d: recname %s -- unexpected section\n",
				 answer->lowercase_name,sect,recnum));
				
		panic("RESOLV PANIC",__FILE__,__LINE__,
			      "fill_resolv_cache",
			      "Unexpected section",0);
	    }	
	}
    }

 out:

    if (ret) {
	DPRINT(Debug,14,(&Debug,
			 "fill_resolv_cache=%p (refcount=%d) name=\"%s\"\n",
			 ret,ret->refcount,ret->lowercase_name));
    }

    return ret;
}
	 
/* Returns number of cleared entries */

static int cleanup_resolv_cache P_((struct resolv_cache             * cache,
				    const struct schedule_timelimit * now,
				    struct schedule_timelimit       * next_cleanup));

/* Returns number of cleared entries */
static int cleanup_cache_items P_((const struct schedule_timelimit * now,
				   struct schedule_timelimit       * next_cleanup));
static int cleanup_cache_items(now,next_cleanup)
     const struct schedule_timelimit * now;
     struct schedule_timelimit       * next_cleanup;
{
    int count = 0;
    int entries UNUSED_VAROK = 0;



    if (cache_items) {
	size_t num = sort_list_len(cache_items);
	size_t i;
    

	for (i = 0; i < num; i++) {
	    union sort_item item;

	    item.resolv = 0;

	    /* increments refcount */
	    get_sort_list_item(cache_items,
			       sort_list_get_normal,
			       i,
			       &item);
	    
	    
	    if (item.resolv) {
		count += cleanup_resolv_cache(item.resolv,now,next_cleanup);
		entries++;
		free_resolv_cache(& item.resolv);
	    }
	}
    }

    DPRINT(Debug,14,(&Debug,"cleanup_cache_items=%d, %d entries",
		     count,entries));
    if (next_cleanup) {
	char * X = schedule_timeout_string(next_cleanup);
	
	if (X) {
	    DPRINT(Debug,14,(&Debug,
			    ", next cleanup %s",
			    X));			
	    free(X);
	}	
    }        
    DPRINT(Debug,14,(&Debug,"\n"));
    
    return count;
}

static struct timed_task_handle  * cleanup_resolv_handle = NULL;
struct schedule_timelimit          next_resolv_cleanup;

S_(timed_task_action_f cleanup_resolv_action)

static enum timed_task_r cleanup_resolv_action P_((struct timed_task *task,
						   const struct schedule_timelimit *now));
static enum timed_task_r cleanup_resolv_action(task,now)
     struct timed_task *task;
     const struct schedule_timelimit *now;
{
    int count UNUSED_VAROK;
    char * A;
    
    next_resolv_cleanup = NO_schedule_timelimit;

    count = cleanup_cache_items(now,&next_resolv_cleanup);

    DPRINT(Debug,9,(&Debug, "cleanup_resolv_action: %d cache items cleaned",
		    count));

    A = schedule_timeout_string(&next_resolv_cleanup);
    if (A) {
	DPRINT(Debug,9,(&Debug, ", next resolv cleanup %s",
			A));
	free(A);
    }
    
    DPRINT(Debug,9,(&Debug, "\n"));

    if (schedule_have_timelimit(&next_resolv_cleanup)) {
    
	if (alter_timed_task(&cleanup_resolv_handle,NULL,
			     NO_timed_task_free,
			     NO_inc_timed_task_refcount,
			     cleanup_resolv_action,
			     &next_resolv_cleanup)) {
	    
	    DPRINT(Debug,9,(&Debug, "cleanup_resolv_action: Updated resolv cleanup, handle %p\n",
			    cleanup_resolv_handle));
	}

    } else {
	DPRINT(Debug,9,(&Debug, "cleanup_resolv_action: freeing resolv cleanup handle %p\n",
			cleanup_resolv_handle));

	alter_timed_task(&cleanup_resolv_handle,NULL,
			 NO_timed_task_free,
			 NO_inc_timed_task_refcount,
			 NO_timed_task_action,
			 &next_resolv_cleanup);
    }


    return timed_task_default;
}

static void update_next_resolv_cleanup(valid_until)   
     const struct schedule_timelimit * valid_until;
{

    if (schedule_shorten_next_timeout(&next_resolv_cleanup,
				      valid_until)) {

	char * X = schedule_timeout_string(&next_resolv_cleanup);
			
	if (X) {
	    DPRINT(Debug,9,(&Debug,
			    "update_next_resolv_cleanup: setting or shortened next resolv cleanup to %s\n",
			    X));			
	    free(X);
	}	

	if (alter_timed_task(&cleanup_resolv_handle,NULL,
			     NO_timed_task_free,
			     NO_inc_timed_task_refcount,
			     cleanup_resolv_action,
			     &next_resolv_cleanup)) {
	    
	    DPRINT(Debug,9,(&Debug,
			    "update_next_resolv_cleanup: Updated resolv cleanup, timed task handle %p\n",
			    cleanup_resolv_handle));
	}
    }
}

/* Returns number of cleared entries */
static int cleanup_dummy_resolv_cache P_((struct dummy_resolv_cache       * target,
					  const struct schedule_timelimit * now,
					  struct schedule_timelimit       * next_cleanup));
static int cleanup_dummy_resolv_cache(target,now,next_cleanup) 
    struct dummy_resolv_cache       * target;
    const struct schedule_timelimit * now;
    struct schedule_timelimit       * next_cleanup;
{
    int count = 0;

    if (DUMMY_RESOLV_CACHE_magic != target->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"free_dummy_resolv_cache",
	      "Bad magic number",0);

    if (target->record)
	count = cleanup_resolv_cache(target->record,now,next_cleanup);

    return count;
}

/* Returns number of cleared entries */
static int cleanup_real_resolv P_((struct real_resolv_cache        * target,
				   const struct schedule_timelimit * now,
				   struct schedule_timelimit       * next_cleanup));
static int cleanup_real_resolv(target,now,next_cleanup)
     struct real_resolv_cache        * target;
     const struct schedule_timelimit * now;
     struct schedule_timelimit       * next_cleanup;
{
    int count = 0;

    if (REAL_RESOLV_CACHE_magic != target->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"cleanup_real_resolv",
	      "Bad magic number (target)",0);
    
    if (target->result_items) {
	int i;
	int t = 0;

	for (i = 0; i < target->result_items_count; i++) {

	    if (RESOLV_RESULT_magic != target->result_items[i]->magic) 
		panic("RESOLV PANIC",__FILE__,__LINE__,"cleanup_real_resolv",
		      "Bad magic number (resolv_result)",0);

	    if (schedule_valid_until_ok(& (target->result_items[i]->valid_until),
					now)) {

		if (next_cleanup) {
		    if (schedule_shorten_next_timeout(next_cleanup,
						      & (target->result_items[i]->valid_until))) {

			char * X = schedule_timeout_string(next_cleanup);
			
			if (X) {
			    DPRINT(Debug,14,(&Debug,
					     "cleanup_real_resolv: setting or shortened next_cleanup to %s\n",
					     X));			
			    free(X);
			}	
		    }
		}

		if (t == i) {
		    t++;
		    continue;
		}

		if (target->result_items[t]) 
		    panic("RESOLV PANIC",__FILE__,__LINE__,"cleanup_real_resolv",
		      "Target not cleared",0);

		target->result_items[t] =
		    target->result_items[i];
		target->result_items[i] = NULL;
		t++;

		
	    } else {

		free_resolv_result(& (target->result_items[i]));
		count++;
	    }
	}

	DPRINT(Debug,14,(&Debug,
			 "cleanup_real_resolv: %d free'ed, result len %d, original len %d",
			 count,
			 t,
			 target->result_items_count));
	if (next_cleanup) {
	    char * X = schedule_timeout_string(next_cleanup);

	    if (X) {
		DPRINT(Debug,14,(&Debug,
				 ", next cleanup %s",
				 X));
		 free(X);
	    }				     
	}
	DPRINT(Debug,14,(&Debug,"\n"));
	
	target->result_items_count = t;
    }

    return count;
}


/* Returns number of cleared entries */
static int cleanup_resolv_cache(cache,now,next_cleanup)
     struct resolv_cache             * cache; 
     const struct schedule_timelimit * now;
     struct schedule_timelimit       * next_cleanup;
{
    int count = 0;

    if (RESOLV_CACHE_magic != cache->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"cleanup_resolv_cache",
	      "Bad magic",0);

    switch(cache->cache_type) {
    case cache_none:
	DPRINT(Debug,14,(&Debug,
			 "cleanup_resolv_cache: %s -- cache type none\n",
			 cache->lowercase_name));
	break;
	
    case cache_dummy:
	DPRINT(Debug,14,(&Debug,
			 "cleanup_resolv_cache: %s -- cache type dummy\n",
			 cache->lowercase_name));

	if (cache->u.dummy)
	    count = cleanup_dummy_resolv_cache(cache->u.dummy,now,next_cleanup);
	break;
    case cache_real:
	DPRINT(Debug,14,(&Debug,
			 "cleanup_resolv_cache: %s -- cache type real\n",
			 cache->lowercase_name));
	if (cache->u.real)
	    count = cleanup_real_resolv(cache->u.real,now,next_cleanup);
	break;
    default:
	panic("RESOLV PANIC",__FILE__,__LINE__,"cleanup_resolv_cache",
	      "Bad cache type",0);	
    }

    return count;
}

/* Returns number of found items */
static int search_qtype_resolv_cache P_((struct resolv_cache             * walk,
					 const char                      * walk_name,
					 ns_type                           q_type,
					 struct resolv_result          *** result_vector,
					 int                             * result_vector_alloced,
					 int                             * result_vector_count,
					 const struct schedule_timelimit * now,
					 int                               depth,
					 struct name_result             ** fill_cname));

/* Returns number of found items */
static int search_qtype_real_resolv P_((struct real_resolv_cache        * r,
					const char                      * name,
					ns_type                           q_type,
					struct resolv_result          *** result_vector,
					int                             * result_vector_alloced,
					int                             * result_vector_count,
					const struct schedule_timelimit * now,
					int                               depth,
					struct name_result             ** fill_cname));
static int search_qtype_real_resolv(r,name,q_type,result_vector,result_vector_alloced,
				    result_vector_count,now,depth,fill_cname)
     struct real_resolv_cache        * r;
     const char                      * name;
     ns_type                           q_type;
     struct resolv_result          *** result_vector;
     int                             * result_vector_alloced;
     int                             * result_vector_count;
     const struct schedule_timelimit * now;
     int                               depth;
     struct name_result             ** fill_cname;
{
    int count = 0;
    struct resolv_result ** VECTOR = *result_vector;
    int VECTOR_alloced             = *result_vector_alloced;
    int VECTOR_count               = *result_vector_count;

    if (REAL_RESOLV_CACHE_magic != r->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"search_qtype_real_resolv",
		      "Bad magic number",0);

    if (r->result_items) {
	int skipped = 0;
	int duplicate_count = 0;
	int i;

	for (i = 0; i < r->result_items_count; i++) {
	    if (RESOLV_RESULT_magic != r->result_items[i]->magic) 
		panic("RESOLV PANIC",__FILE__,__LINE__,"search_qtype_real_resolv",
		      "Bad magic number (resolv_result)",0);

	    if (0 != istrcmp(name,r->result_items[i]->name))
		panic("RESOLV PANIC",__FILE__,__LINE__,"search_qtype_real_resolv",
		      "Bad name",0);
					    
	    if (schedule_valid_until_ok(& (r->result_items[i]->valid_until),
					now)) {
		int add = 0;

		if (VECTOR) {
		    int z;

		    for (z = 0; z < VECTOR_count; z++) {
			if (VECTOR[z] == r->result_items[i]) {
			    
			    DPRINT(Debug,14,(&Debug,
					     "search_qtype_real_resolv: %s, qtype=%d, duplicate item found\n",
					     name,q_type));
			    duplicate_count++;
			    goto duplicate;
			}
		    }
		}

		if (r->result_items[i]->rectype == q_type ||
		    ns_t_any == q_type) {
		    add = 1;
		    count++;
		} else if (ns_t_cname == r->result_items[i]->rectype)
		    add = 1;
			
		if (add) {
		    if (VECTOR_count >= VECTOR_alloced) {
			int x;

			VECTOR_alloced = VECTOR_count + 10;

			VECTOR = safe_array_realloc(VECTOR,
						    VECTOR_alloced,
						    sizeof(VECTOR[0]));
			
			for (x = VECTOR_count; x < VECTOR_alloced; x++)
			    VECTOR[x] = NULL; 
		    }

		    VECTOR[VECTOR_count] = 
			r->result_items[i];
		    VECTOR[VECTOR_count]->refcount++;
		    VECTOR_count++;
		}

	    duplicate:
		if (ns_t_cname == r->result_items[i]->rectype) {
		    
		    if (r->result_items[i]->data.domain_name) {
			if (NAME_RESULT_magic !=
			    r->result_items[i]->data.domain_name->magic)
			    panic("RESOLV PANIC",__FILE__,__LINE__,"search_qtype_real_resolv",
				  "Bad magic number (name_result)",0);
	
			if (depth > 15) {    /* Avoid too long recursion */
			    DPRINT(Debug,14,(&Debug,
					     "search_qtype_real_resolv: %s, too many CNAMEs, depth %d\n",
					     name,depth));
			} else if (r->result_items[i]->data.domain_name->name_it &&
				   r->result_items[i]->data.domain_name->name) {
			    int f;
			       
			    f = 
				search_qtype_resolv_cache(r->result_items[i]->data.domain_name->name_it,
							  r->result_items[i]->data.domain_name->name,
							  q_type,
							  &VECTOR,&VECTOR_alloced,&VECTOR_count,
							  now,
							  depth+1,
							  fill_cname);

			    count += f;

			    if (!f && fill_cname) {
				*fill_cname = r->result_items[i]->data.domain_name;
				
				DPRINT(Debug,14,(&Debug,
						 "search_qtype_real_resolv: %s, need resolve cname %s\n",
						 name,
						 r->result_items[i]->data.domain_name->name));
			    }
			    
			} else if (fill_cname &&
				   r->result_items[i]->data.domain_name->name) {
			    *fill_cname = r->result_items[i]->data.domain_name;

			    DPRINT(Debug,14,(&Debug,
					     "search_qtype_real_resolv: %s, need resolve cname %s\n",
					     name,
					     r->result_items[i]->data.domain_name->name));
			    
			}
		    }
		}
		
	    } else 
		skipped++;
	}

	DPRINT(Debug,14,(&Debug,
			 "search_qtype_real_resolv: %s, qtype=%d, %d skipped, %d duplicate, %d found, %d total; depth %d\n",
			 name,q_type,skipped,duplicate_count,count,VECTOR_count,depth));	
    }

    /* NULL terminated */

    if (VECTOR_alloced < VECTOR_count+1) {
	VECTOR_alloced = VECTOR_count+1;

	VECTOR = safe_array_realloc(VECTOR,
				    VECTOR_alloced,
				    sizeof(VECTOR[0]));
    }
    VECTOR[VECTOR_count] = NULL;

    *result_vector         = VECTOR;
    *result_vector_alloced = VECTOR_alloced;
    *result_vector_count   = VECTOR_count;
    
    return count;
}
					
static int search_qtype_resolv_cache(walk,walk_name,q_type,
				     result_vector,result_vector_alloced,
				     result_vector_count,now,depth,
				     fill_cname)
     struct resolv_cache             * walk;
     const char                      * walk_name;
     ns_type                           q_type;
     struct resolv_result          *** result_vector;
     int                             * result_vector_alloced;
     int                             * result_vector_count;
     const struct schedule_timelimit * now;
     int                               depth;
     struct name_result             ** fill_cname;
{
    int count = 0;
    struct resolv_result ** VECTOR = *result_vector;
    int VECTOR_alloced             = *result_vector_alloced;
    int VECTOR_count               = *result_vector_count;


    if (RESOLV_CACHE_magic != walk->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"search_qtype_resolv_cache",
	      "Bad cache type (resolv_cache)",0);

    switch (walk->cache_type) {
    case cache_real:

	if (walk->u.real) {
	    if (REAL_RESOLV_CACHE_magic != walk->u.real->magic)
		panic("RESOLV PANIC",__FILE__,__LINE__,"search_qtype_resolv_cache",
		      "Bad cache type (real_resolv_cache)",0);
	    
	    count = search_qtype_real_resolv(walk->u.real,walk_name,q_type,
					     &VECTOR,&VECTOR_alloced,&VECTOR_count,
					     now,depth,fill_cname);
	}

	break;
    case cache_none: 
    case cache_dummy:
	break;
    }

    *result_vector         = VECTOR;
    *result_vector_alloced = VECTOR_alloced;
    *result_vector_count   = VECTOR_count;

    return count;
}


/* Top half of fill_query_resolv_cache */
static int fill_check_resolv_cache P_((struct resolv_cache             * record,
				       const char                      * name,
				       const struct schedule_timelimit * now,
				       ns_type                           q_type,
				       /* use search -- real search result -- 
					   caller must free */
				       struct resolv_cache            ** search,
				       char                           ** search_name,
				       struct resolv_result          *** result_vector,
				       int                             * result_vector_alloced,
				       int                             * result_vector_count,	
				       int                             * need_query_p,
				       struct name_result             ** fill_cname_p));

static int fill_check_resolv_cache(record,name,now,q_type,search,search_name,
				   result_vector,result_vector_alloced,
				   result_vector_count,need_query_p,
				   fill_cname_p)
     struct resolv_cache             * record;
     const char                      * name;
     const struct schedule_timelimit * now;
     ns_type                           q_type;
     /* use search -- real search result -- 
	caller must free */
     struct resolv_cache            ** search;
     char                           ** search_name;
     struct resolv_result          *** result_vector;
     int                             * result_vector_alloced;
     int                             * result_vector_count;	
     int                             * need_query_p;
     struct name_result             ** fill_cname_p;
{
    int found = 0;
    struct resolv_result ** VECTOR = *result_vector;
    int VECTOR_alloced             = *result_vector_alloced;
    int VECTOR_count               = *result_vector_count;
    int NEED_query                 = *need_query_p;
    struct name_result * FILL_cname = *fill_cname_p;
    
    int have_expired;
    
    if (RESOLV_CACHE_magic != record->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"fill_check_resolv_cache",
	      "Bad magic number (resolv_cache)",0);

    if (0 != istrcmp(name,record->lowercase_name))
	panic("RESOLV PANIC",__FILE__,__LINE__,"fill_check_resolv_cache",
	      "Name not match",0);   
    
    /* Top half of fill_query_resolv_cache */

    have_expired = cleanup_resolv_cache(record,now,NULL);
    
    if (have_expired) 
	NEED_query = 1;
    else {
	
	struct resolv_cache * walk  = NULL;
	const char          * walk_name = NULL;
	
	switch (record->cache_type) {
	case cache_none: NEED_query = 1;
	    break;
	case cache_dummy:   /* have search reasult on cache */
	    if (search) {
		if (record->u.dummy) {
		    if (DUMMY_RESOLV_CACHE_magic != record->u.dummy->magic)
			panic("RESOLV PANIC",__FILE__,__LINE__,
			      "fill_check_resolv_cache",
			      "Bad magic (dummy_resolv_cache)",0);
		    
		    if (record->u.dummy->record) {
			if (RESOLV_CACHE_magic != record->u.dummy->record->magic)
			    panic("RESOLV PANIC",__FILE__,__LINE__,
				  "fill_check_resolv_cache",
				  "Bad magic",0);
			
			walk      = record->u.dummy->record;
			walk_name = record->u.dummy->record_name;
			
		    } else
			NEED_query = 1;
		} else
		    NEED_query = 1;
	    } else
		NEED_query = 1;;
	    break;
	case cache_real:
	    walk = record;
	    walk_name = name;
	    break;
	default:
	    panic("RESOLV PANIC",__FILE__,__LINE__,
		  "fill_check_resolv_cache",
		  "Bad cache type",0);
	}
	
	if (walk) {
	    if (RESOLV_CACHE_magic != walk->magic)
		panic("RESOLV PANIC",__FILE__,__LINE__,
		      "fill_check_resolv_cache",
		      "Bad magic",0);
	    
	    have_expired = cleanup_resolv_cache(walk,now,NULL);
	}
	if (have_expired) 
	    NEED_query = 1;
	else if (walk) {
	    
	    if (cache_real != walk->cache_type)
		panic("RESOLV PANIC",__FILE__,__LINE__,
		      "fill_check_resolv_cache",
		      "Bad cache type",0);
	    
	    if ((found = search_qtype_resolv_cache(walk,walk_name,q_type,
						   &VECTOR,&VECTOR_alloced,
						   &VECTOR_count,
						   now,0,
						   &FILL_cname))) {
		
		/* found answer */
		
		if (search) {
		    *search = walk;
		    (*search)->refcount++;
		}
		
		if (search_name)
		    *search_name = safe_strdup(walk_name);
		
	    } else if (FILL_cname) {
		if (NAME_RESULT_magic != FILL_cname->magic)
		    panic("RESOLV PANIC",__FILE__,__LINE__,
			  "fill_check_resolv_cache",
			  "Bad magic number (name_result)",0);
		if (!FILL_cname->name)
		    panic("RESOLV PANIC",__FILE__,__LINE__,
			  "fill_check_resolv_cache",
			  "CNAME missing name",0);
		
		DPRINT(Debug,14,(&Debug,
				 "fill_check_resolv_cache: %s: Need follow CNAME %s\n",
				 name,FILL_cname->name));
		
	    } else
		NEED_query = 1;
	    
	} else
	    NEED_query = 1;	    
    }


    *result_vector         = VECTOR;
    *result_vector_alloced = VECTOR_alloced;
    *result_vector_count   = VECTOR_count;
    *need_query_p          = NEED_query;
    *fill_cname_p          = FILL_cname;
    
    DPRINT(Debug,14,(&Debug,
		     "fill_check_resolv_cache=%d found: %s: %d total; need query = %d\n",
		     found,name,VECTOR_count,NEED_query));
    
    return found;
}

/* Return 1 if succeed */
static int lookup_query_blocked P_((const char                      * name,
				    ns_type                           q_type,
				    int                               is_search,
				    enum looking_up                 * looking_up,
				    unsigned char                     answer[],
				    size_t                            answer_size,
				    int                             * anslen_p,
				    ns_msg                          * parsed_answer,
				    int                             * have_error,
				    int                             * other,
				    int                             * err_p,
				    struct cancel_data             ** cancel_p
				    /* DNS lookup canceled, not really supported for blocked
				       queries
				    */
				    ));
static int lookup_query_blocked(name,q_type,is_search,looking_up,
				answer,answer_size,anslen_p,parsed_answer,
				have_error,other,err_p,cancel_p)
     const char                      * name;
     ns_type                           q_type;
     int                               is_search;
     enum looking_up                 * looking_up;
     unsigned char                     answer[];
     size_t                            answer_size;
     int                             * anslen_p;
     ns_msg                          * parsed_answer;    
     int                             * have_error;
     int                             * other;
     int                             * err_p;
     struct cancel_data             ** cancel_p 
     /* DNS lookup canceled, not really supported for blocked
	queries
     */;
{
    int ret = 0;
    int ANSlen = *anslen_p;

    if (!looking_up ||
	(l_up_printed != *looking_up &&
	 l_up_truncated_retry != *looking_up)
	) {

	if (cancel_p && *cancel_p) {
	    if (is_schedule_cancel(*cancel_p,NULL))
		set_cancel_message(*cancel_p,
				   -1 /* Show immediately */,
				   CATGETS(elm_msg_cat, 
					   ResolvSet,
					   ResolvLookingUp,
					   "Looking up %s ..."),
				   name);
	    else {
		DPRINT(Debug,14,(&Debug,
				 "lookup_query_blocked: 'Looking up' already printed.\n"));
	    }
	} else
	    lib_transient(CATGETS(elm_msg_cat, 
				  ResolvSet,
				  ResolvLookingUp,
				  "Looking up %s ..."),
			  name);
	
	if (looking_up)
	    *looking_up = l_up_printed;	

    }

    errno = 0;
    *err_p = 0;
    
    if (is_search)
	ANSlen = res_search(name,ns_c_in,q_type,
			    answer,answer_size);
    else
	ANSlen = res_query(name,ns_c_in,q_type,
			   answer,answer_size);

    if (ANSlen < 0) {
	int err = errno;

	if (HOST_NOT_FOUND != h_errno &&
	    NO_DATA        != h_errno &&
	    NO_RECOVERY    != h_errno &&
	    EINTR == err &&  cancel_p && *cancel_p &&
	    is_canceled(*cancel_p)) {
	    
	    DPRINT(Debug,14,(&Debug, 
			     "lookup_query_blocked: %s: query canceled\n",
			     name));

	    *err_p = err;

	    /* Flag this as error, so query is marked as failed */
	    if (have_error && !*have_error) {
		DPRINT(Debug,14,(&Debug, "lookup_query_blocked: %s: marking as failed\n",
				 name));
		*have_error = 1;
	    }
	    
	    goto fail;
	    
	} else {
	
	    switch(h_errno) {
	    case HOST_NOT_FOUND:
		lib_error(CATGETS(elm_msg_cat, 
				  ResolvSet,
				  ResolvLookNotFound,
				  "Looking up %s: not found"),
			  name);
		
		if (looking_up)
		    *looking_up = l_up_error;	
	    
		break;
	    
	    case NO_DATA:
		if (other)
		    *other = 1;
		else {
		    lib_transient(CATGETS(elm_msg_cat, 
					  ResolvSet,
					  ResolvLookNoData,
					  "Looking up %s: no data"),
				  name);
		    
		    if (looking_up)
			*looking_up = l_up_error;	
		}
		
		break;
	    case TRY_AGAIN:
		
		if (err)
		    lib_error(CATGETS(elm_msg_cat, 
				      ResolvSet,
				      ResolvLookErrno,
				      "Looking up %s: %s"),
			      name,strerror(err));
		else				      
		    lib_transient(CATGETS(elm_msg_cat, 
					  ResolvSet,
					  ResolvLookTryAgain,
					  "Looking up %s: not found yet"),
				  name);
		
		if (looking_up)
		    *looking_up = l_up_error;	
		
		if (have_error)
		    (*have_error)++;
		
		break;
	    case NO_RECOVERY:
		lib_error(CATGETS(elm_msg_cat, 
				  ResolvSet,
				  ResolvLookBadName,
				  "Looking up %s: bad name or refused query"),
			  name);
		
		if (looking_up)
		    *looking_up = l_up_error;	
		
		if (have_error)
		    (*have_error)++;
		
		break;
	    case NETDB_INTERNAL:
	    default:
		if (err) {
		    lib_error(CATGETS(elm_msg_cat, 
				      ResolvSet,
				      ResolvLookErrno,
				      "Looking up %s: %s"),
			      name,strerror(err));
		    *err_p = err;
		} else
		    lib_error(CATGETS(elm_msg_cat, 
				      ResolvSet,
				      ResolvLookFailed,
				      "Looking up %s: failed"),
			      name);
		
		if (looking_up)
		    *looking_up = l_up_error;	
		
		if (have_error)
		(*have_error)++;
		
		break;
	    }
	}
	
    } else {

	int r;
	
	errno = 0;
	r = ns_initparse(answer,ANSlen,parsed_answer);
	
	if (r < 0) {
	    int err = errno;
	    
	    if (have_error)
		(*have_error)++;
	    
	    if (err) {
		lib_error(CATGETS(elm_msg_cat, 
				  ResolvSet,
				  ResolvLookFParseAnswerE,
				  "Looking up %s: Failed to parse answer: %s"),
			  name,
			  strerror(err));
		*err_p = err;
	    } else
		lib_error(CATGETS(elm_msg_cat, 
				  ResolvSet,
				  ResolvLookFParseAnswer,
				  "Looking up %s: Failed to parse answer"),
			      name);

	    if (looking_up)
		*looking_up = l_up_error;	
	    
	} else {
	    
	    ret = 1;

	    DPRINT(Debug,14,(&Debug,"lookup_query_blocked: %s: succeed\n",
			     name));
	}

    }

 fail:
    
    *anslen_p = ANSlen;

    DPRINT(Debug,14,(&Debug,"lookup_query_blocked=%d; %s; answer len = %d",		     
		     ret,name,ANSlen));

    if (*err_p) {
	DPRINT(Debug,14,(&Debug,"; *err_p=%d (%s)",
			 *err_p,strerror(*err_p)));
    }

    DPRINT(Debug,14,(&Debug,"\n"));
    
    return ret;
}

S_(have_qm_f have_blocking_qm)
static int have_blocking_qm P_((void));
static int have_blocking_qm()
{
    static int ret = -1;

    if (-1 == ret) {	
	int r = res_init();

	ret = 0;
	
	if (0 == r) {
		    
	    DPRINT(Debug,12,(&Debug, "have_blocking_qm: Started\n"));
	    ret = 1;
	} else if (r < 0) {
	    int err = errno;

	    DPRINT(Debug,12,(&Debug,
			     "have_blocking_qm: res_init failed: errno = %d (%s)\n",
			     err,strerror(err)));
	    
	    lib_error(CATGETS(elm_msg_cat, 
			      ResolvSet,
			      ResolvResInitFailed,
			      "Failed to initialize with res_init: %s"),
		      strerror(err));
	}
    }
    
    DPRINT(Debug,12,(&Debug,
		     "have_blocking_qm=%d\n",ret));
    
    return ret;
}



S_(lookup_resolv_cache_qm_f lookup_resolv_cache_blocked)
static struct resolv_cache * lookup_resolv_cache_blocked
    P_((struct resolv_cache             * record,
	const char                      * name,
	const struct schedule_timelimit * now,
	ns_type                           q_type,
	int                               is_search,
	enum looking_up                 * looking_up,
	char                           ** search_name,
	int                             * have_error,
	int                             * other,
	struct cancel_data             ** cancel_p));
static struct resolv_cache * lookup_resolv_cache_blocked(record,name,now,
							 q_type,is_search,
							 looking_up,search_name,
							 have_error,other,
							 cancel_p)
     struct resolv_cache             * record;
     const char                      * name;
     const struct schedule_timelimit * now;
     ns_type                           q_type;
     int                               is_search;
     enum looking_up                 * looking_up;
     char                           ** search_name;
     int                             * have_error;
     int                             * other;
     struct cancel_data             ** cancel_p
     /* DNS lookup canceled, not really supported for blocked
	queries
     */;
{
    struct resolv_cache *X = NULL;
    unsigned char answer1[RESOLV_BUFSIZE]; 
    int anslen;
    ns_msg parsed_answer;

    int retry = 0;
    unsigned char * answer = answer1;
    size_t anssize = sizeof answer1;
               
    if (RESOLV_CACHE_magic != record->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"lookup_resolv_cache_blocked",
	      "Bad magic",0);
    if (0 != istrcmp(name,record->lowercase_name))
	panic("RESOLV PANIC",__FILE__,__LINE__,"lookup_resolv_cache_blocked",
	      "Name not match",0);

    do {
	int err = 0;
	
	retry = 0;
	
	if (lookup_query_blocked(name,q_type,is_search,
				 looking_up,answer,anssize,
				 &anslen,&parsed_answer,
				 have_error,other,&err,cancel_p)) { 
	    
	    X = fill_resolv_cache(record,&parsed_answer,now,search_name,
				  looking_up,have_error);
	    
	} else if (EMSGSIZE == err && anslen == anssize && anslen < MAX_resolv_bufsize) {
		    
	    anssize <<= 1;
	    if (anssize > MAX_resolv_bufsize)
		anssize = MAX_resolv_bufsize;
	    
	    if (answer == answer1)
		answer = safe_malloc(anssize);
	    else
		answer = safe_realloc(answer,anssize);
	    
	    lib_transient(CATGETS(elm_msg_cat, 
				  ResolvSet,
				  ResolvLookTruncatedRetry,
				  "Looking up %s: Answer truncated (%d bytes), retrying..."),
			  name,anslen);
	    
	    if (looking_up)
		*looking_up = l_up_truncated_retry;
	    retry = 1;

	    DPRINT(Debug,14,(&Debug,"lookup_resolv_cache_blocked: %s: retrying with size %lu\n",
			     name,(unsigned long)anssize));

	} else if (EINTR == err && cancel_p && *cancel_p &&
		   is_canceled(*cancel_p)) {
	    
	    DPRINT(Debug,14,(&Debug, "lookup_resolv_cache_blocked: %s: DNS lookup canceled\n",
			     name));

	    /* Flag this as error, so query is marked as failed */
	    if (have_error && !*have_error) {
		DPRINT(Debug,14,(&Debug, "lookup_resolv_cache_blocked: %s: marking as failed\n",
				 name));
		*have_error = 1;
	    }
	    
	    goto fail;
	}
    	
    } while (retry);

 fail:
    
    if (answer != answer1)
	free(answer);

    DPRINT(Debug,14,(&Debug,"lookup_resolv_cache_blocked: %s result\n",
		     X ? "have" : "no"));
    
    return X;
}

static struct resolv_cache * call_lookup_resolv_cache
   P_((struct resolv_cache             * record,
       const char                      * name,
       const struct schedule_timelimit * now,
       const ns_type                     q_type,
       const int                         is_search,
       enum looking_up                 * looking_up,
       char                           ** search_name,
       int                             * have_error,
       int                             * other,
       enum query_mode_val               qm,
       struct cancel_data             ** cancel_p 
       /* May be NULL, Used if dns lookup was cancelable */
       ));
						      
static int fill_query_resolv_cache P_((struct resolv_cache             * record,
				       const char                      * name,
				       const struct schedule_timelimit * now,
				       ns_type                           q_type,
					/* use search -- real search result -- 
					   caller must free */
				       struct resolv_cache            ** search,
				       char                           ** search_name,
				       struct resolv_result          *** result_vector,
				       int                             * result_vector_alloced,
				       int                             * result_vector_count,	    
				       enum looking_up                 * looking_up,
				       int                             * have_error,
				       int                             * other,
				       int                               recurse,
				       enum query_mode_val               qm,
				       struct cancel_data             ** cancel_p));
static int fill_query_resolv_cache(record,name,now,q_type,search,search_name,
				   result_vector,result_vector_alloced,
				   result_vector_count,looking_up,
				   have_error,other,recurse,qm,cancel_p)
     struct resolv_cache             * record;
     const char                      * name;
     const struct schedule_timelimit * now;
     ns_type                           q_type;
     /* use search -- real search result -- 
	caller must free */
     struct resolv_cache           ** search;
     char                          ** search_name;
     struct resolv_result         *** result_vector;
     int                            * result_vector_alloced;
     int                            * result_vector_count;	    
     enum looking_up                * looking_up;
     int                            * have_error;
     int                            * other;
     int                              recurse;
     enum query_mode_val              qm;
     struct cancel_data            ** cancel_p
     /* May be NULL, Used if dns lookup was cancelable */;
{
    int found = 0;
    struct resolv_result ** VECTOR = *result_vector;
    int VECTOR_alloced             = *result_vector_alloced;
    int VECTOR_count               = *result_vector_count;
    struct name_result  * fill_cname   = NULL;
    int need_query = 0;

	     
    DPRINT(Debug,14,(&Debug,
		     "fill_query_resolv_cache: %s: recurse level %d, %d answers\n",
		     name,recurse,VECTOR_count));

    if (RESOLV_CACHE_magic != record->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"fill_query_resolv_cache",
	      "Bad magic",0);

    if (0 != istrcmp(name,record->lowercase_name))
	panic("RESOLV PANIC",__FILE__,__LINE__,"fill_query_resolv_cache",
	      "Name not match",0);

    /* ------------------------------------------------------------  */

    found = fill_check_resolv_cache(record,name,now,q_type,
				    search,search_name,
				    &VECTOR,&VECTOR_alloced,&VECTOR_count,
				    &need_query,&fill_cname);
            
    if (other && !need_query)
	*other = 1;
    
    if (need_query) {
	char * walk_name = NULL;
	struct resolv_cache *X = NULL;

	DPRINT(Debug,14,(&Debug,
			 "fill_query_resolv_cache: %s: using query-mode = %d\n",
			 name,qm));

	X =  call_lookup_resolv_cache(record,name,now,q_type,
				      search != NULL,looking_up,
				      &walk_name,
				      have_error,other,
				      qm,cancel_p);
				      	
	if (X) {
	    
	    if (X != record && !search) {
		free_resolv_cache(&X);
		if (walk_name)
		    free(walk_name);
		goto failed;
	    }
	    
	    if ((found = search_qtype_resolv_cache(X,walk_name,q_type,
						   &VECTOR,&VECTOR_alloced,
						   &VECTOR_count,
						   now,0,
						   &fill_cname))) {
		
		if (!looking_up)
		    lib_transient(CATGETS(elm_msg_cat, 
					  ResolvSet,
					  ResolvLookOK,
					  "Looking up %s: OK"),
				  name);
		
		if (search_name) {
		    if (*search_name) 
			free(*search_name);
		    *search_name = walk_name;
		    walk_name = NULL;
		}
		
		if (search) {
		    *search = X;
		    X = NULL;
		}

	    } else if (fill_cname) {
		if (NAME_RESULT_magic != fill_cname->magic)
		    panic("RESOLV PANIC",__FILE__,__LINE__,
			  "fill_query_resolv_cache",
			  "Bad magic number (name_result)",0);
		if (!fill_cname->name)
		    panic("RESOLV PANIC",__FILE__,__LINE__,
			  "fill_query_resolv_cache",
			  "CNAME missing name",0);
		
		DPRINT(Debug,14,(&Debug,
				 "fill_query_resolv_cache: %s: Need follow CNAME %s\n",
				 name,fill_cname->name));
	    }

	    if (X)
		free_resolv_cache(&X);
	    if (walk_name)
		free(walk_name);
	    
	} else if (have_error && *have_error && cancel_p && *cancel_p &&
		   is_canceled(*cancel_p)) {
	    
	    DPRINT(Debug,14,(&Debug, "fill_query_resolv_cache: %s: DNS lookup canceled\n",
			     name));
	    
	    goto failed;
	}
    }

    if (fill_cname) {
	if (NAME_RESULT_magic != fill_cname->magic)
	    panic("RESOLV PANIC",__FILE__,__LINE__,
		  "fill_query_resolv_cache",
		  "Bad magic number (name_result)",0);
	if (!fill_cname->name)
	    panic("RESOLV PANIC",__FILE__,__LINE__,
		  "fill_query_resolv_cache",
		  "CNAME missing name",0);

	DPRINT(Debug,14,(&Debug,
			 "fill_query_resolv_cache: %s: Need follow CNAME %s\n",
			 name,fill_cname->name));
	
	if (recurse > 10) {
	    DPRINT(Debug,14,(&Debug,
			     "fill_query_resolv_cache: %s: Too made CNAMEs, depth %d\n",
			     name,recurse));
	    
	} else {

	    if (! fill_cname->name_it) {
		fill_cname->name_it = give_resolv_cache(fill_cname->name);

		DPRINT(Debug,14,(&Debug,
				 "fill_query_resolv_cache: %s: Using cache for %s\n",
				 name,fill_cname->name));
	    }

	    found += 
		fill_query_resolv_cache(fill_cname->name_it,
					fill_cname->name,
					now,
					q_type,NULL,NULL,
					&VECTOR,
					&VECTOR_alloced,
					&VECTOR_count,
					looking_up,
					have_error,
					other,
					recurse,qm,cancel_p);

	}
    }
    
 failed:
    *result_vector         = VECTOR;
    *result_vector_alloced = VECTOR_alloced;
    *result_vector_count   = VECTOR_count;

    DPRINT(Debug,14,(&Debug,
		     "fill_query_resolv_cache=%d found: %s: %d total, depth %d\n",
		     found,name,VECTOR_count,recurse));

    return found;
}

static void free_cache_items P_((void));

static struct resolv_cache * query_resolv_cache P_((const char                      * name,
						    ns_type                           q_type,
						    /* use search -- real search result -- 
						       caller must free */
						    struct resolv_cache            ** search,
						    char                           ** search_name,
						    /* Returns NULL terminated vector, caller
						       must free elems */
						    struct resolv_result          *** result_vector,
						    enum looking_up                 * looking_up,
						    int                             * found,
						    int                             * have_error,
						    int                             * other,
						    const struct schedule_timelimit * now,
						    enum query_mode_val               qm,
						    struct cancel_data             ** cancel_p 
						    /* May be NULL, Used if dns lookup 
						       was cancelable */
						    ));
static struct resolv_cache * query_resolv_cache(name,q_type,search,search_name,
						result_vector,looking_up,
						found,
						have_error,other,now,qm,
						cancel_p)
     const char                      * name;
     ns_type                           q_type;
     struct resolv_cache            ** search;
     char                           ** search_name;
     struct resolv_result          *** result_vector;
     enum looking_up                 * looking_up;
     int                             * found;
     int                             * have_error;
     int                             * other;
     const struct schedule_timelimit * now;
     enum query_mode_val               qm;
     struct cancel_data             ** cancel_p    
     /* May be NULL, Used if dns lookup was cancelable */;
{

    /*  NOTE:

        timed cleanup_resolv_cache() / cleanup_cache_items() 
        can clear found answers (struct resolv_result) from

        struct resolv_cache * ret

        when lib_error() or lib_transient() are called
        
        but these answers are still on 

        struct resolv_result          *** result_vector

        and this is refcounted and therefore not free'ed
        too early

    */
    
    struct resolv_cache   * ret = NULL;
    struct resolv_result ** VECTOR = NULL;
    int VECTOR_alloced = 0;
    int VECTOR_count   = 0;
    int f;
    const char * c;
    int          dots = 0;
    const char * alias = NULL;
    
    if (result_vector) 
	VECTOR = *result_vector;
    
    if (VECTOR) {

	int i;
	
	for (i = 0; VECTOR[i]; i++)
	    free_resolv_result(&(VECTOR[i]));			       
	VECTOR_alloced = i+1;
    }
    
    if (found)
	*found = 0;

    if (search && *search)
	free_resolv_cache(search);

    if (search_name && *search_name) {
	free(*search_name);
	*search_name = NULL;
    }

    if (pending_interface_change) {
	int need_clear = dt_flag_is_set(&interface_change,intch_clear_cache_flag);
	
	DPRINT(Debug,7,(&Debug,
			 "query_resolv_cache: Have pending_interface_change%s\n",
			 need_clear ? ", need clear cache" : ""));

	if (need_clear) {	    
	    free_cache_items();
	}
	
	pending_interface_change = 0;
    }

    for (c = name;*c; c++) {
	if ('.' == *c) {
	    dots++;
	}
    }

    DPRINT(Debug,14,(&Debug,"query_resolv_cache: %s: %d dots\n",
		     name,dots));
    

#ifdef RESOLV_HOSTALIAS
    if (!dots && (alias = hostalias(name))) {   
	DPRINT(Debug,7,(&Debug,
			"query_resolv_cache: %s: No dots, searching alias: %s\n",
			name,alias));

	ret    = give_resolv_cache(alias);
	
	f = fill_query_resolv_cache(ret,alias,now,
				    q_type,
				    NULL /* NO SEARCH */,
				    search_name,
				    &VECTOR,
				    &VECTOR_alloced,
				    &VECTOR_count,
				    looking_up,
				    have_error,
				    other,0,qm,
				    cancel_p);	
    }
#endif
	
    if (!ret) {
	DPRINT(Debug,14,(&Debug,
			 "query_resolv_cache: %s: No alias\n",
			 name));
	
	ret    = give_resolv_cache(name);
	
	f = fill_query_resolv_cache(ret,name,now,
				    q_type,
				    search,
				    search_name,
				    &VECTOR,
				    &VECTOR_alloced,
				    &VECTOR_count,
				    looking_up,
				    have_error,
				    other,0,qm,
				    cancel_p);
    }
	
    if (found)
	*found = f;
				
    if (result_vector) 
	*result_vector = VECTOR;
    else if (VECTOR) {
	int i;
	
	for (i = 0; VECTOR[i]; i++)
	    free_resolv_result(&(VECTOR[i]));			       
	free(VECTOR);
	VECTOR = NULL;
    }

    if (alias) {
	DPRINT(Debug,14,(&Debug,
			 "query_resolv_cache=%p %s: %s (alias %s): found %d\n",
			 ret,
			 ret->lowercase_name,
			 name,alias,f));
    } else {
	DPRINT(Debug,14,(&Debug,
			 "query_resolv_cache=%p %s: %s: found %d\n",
			 ret,
			 ret->lowercase_name,
			 name,f));
    }

    return ret;
}


S_(free_dt_enum_shared_f free_dt_verify_domain)
static void free_dt_verify_domain P_((struct dt_enum_shared **ptr));

enum resolv_verify_domain {
    rvd_mx_address,
    rvd_mx,
    rvd_any,
    NUM_resolv_verify_domain };

static char * RESOLV_VERIFY_DOMAIN[] = {
    "mx-address",
    "mx",
    "any",
    NULL
};

static int static_resolv_verify_domain;
static struct dt_enum_shared resolv_verify_domain = {
    DT_ENUM_SHARED_magic,
    "resolv",
    0,
    NUM_resolv_verify_domain,&(RESOLV_VERIFY_DOMAIN[0]),
    free_dt_verify_domain,
    NULL
};

static void free_dt_verify_domain(ptr)
     struct dt_enum_shared **ptr;
{
    /* static data is not free'ed */
    
    DPRINT(Debug,7,(&Debug, "free_dt_verify_domain\n"));

    if ((*ptr)->tag &&
	resolv_verify_domain.tag != (*ptr)->tag) {
	free((*ptr)->tag);

	(*ptr)->tag = NULL;
    }

    if (&resolv_verify_domain == *ptr) 
	static_resolv_verify_domain = 0;
    else
	free(*ptr);
    
    *ptr = NULL;
}

S_(free_dt_enum_shared_f free_dt_mserv_lookup)
static void free_dt_mserv_lookup P_((struct dt_enum_shared **ptr));

enum resolv_mserv_lookup {
    mserv_addresses,
    mserv_all_addresses,
    NUM_resolv_mserv_lookup };

static char * RESOLV_MSERV_LOOKUP[] = {
    "addresses",
    "all-addresses",
    NULL
};

static int static_resolv_mserv_lookup;
static struct dt_enum_shared resolv_mserv_lookup = {
    DT_ENUM_SHARED_magic,
    "resolv",
    0,
    NUM_resolv_mserv_lookup,&(RESOLV_MSERV_LOOKUP[0]),
    free_dt_mserv_lookup,
    NULL
};

static void free_dt_mserv_lookup(ptr)
     struct dt_enum_shared **ptr;
{
    /* static data is not free'ed */
    
    DPRINT(Debug,7,(&Debug, "free_dt_mserv_lookup\n"));

    if ((*ptr)->tag &&
	resolv_mserv_lookup.tag != (*ptr)->tag) {
	free((*ptr)->tag);

	(*ptr)->tag = NULL;
    }

    if (&resolv_mserv_lookup == *ptr) 
	static_resolv_mserv_lookup = 0;
    else
	free(*ptr);
    
    *ptr = NULL;
}

E_(RC_fill_enum_values_f RC_fill_enum_values)
struct dt_enum_shared * RC_fill_enum_values P_((struct dt_enumerate_info *ptr,
						const char *tag,
						int used_start_value,
						size_t dt_enumerate_info_size,
						size_t dt_enum_shared_size));
struct dt_enum_shared * RC_fill_enum_values(ptr,tag,used_start_value,
					    dt_enumerate_info_size,
					    dt_enum_shared_size)
     struct dt_enumerate_info *ptr;
     const char *tag;
     int used_start_value;
     size_t dt_enumerate_info_size;
     size_t dt_enum_shared_size;
{
    struct dt_enum_shared *ret = NULL;

    if (dt_enumerate_info_size != sizeof (*ptr)) {
	DPRINT(Debug,1,(&Debug,
			"RC_fill_enum_values: Bad dt_enumerate_info_size %d (should be %d)\n",
			dt_enumerate_info_size,
			sizeof (*ptr)));
	return NULL;
    }

    if (dt_enum_shared_size != sizeof (*ret)) {
	DPRINT(Debug,1,(&Debug,
			"RC_fill_enum_values: Bad dt_enum_shared_size %d (should be %d)\n",
			dt_enum_shared_size,
			sizeof (*ret)));
	return NULL;
    }
    

    DPRINT(Debug,7,(&Debug, "RC_fill_enum_values\n"));
   
    if (ptr == &verify_domain) {
	DPRINT(Debug,7,(&Debug, "   verify_domain\n"));
	
	if (static_resolv_verify_domain || 
	    0 != strcmp(tag,resolv_verify_domain.tag)) {
	    ret = safe_malloc(sizeof (*ret));
	    
	    *ret = resolv_verify_domain;
	} else {
	    ret = &resolv_verify_domain;
	    static_resolv_verify_domain = 1;
	}
	
	if (0 != strcmp(tag,ret->tag) || resolv_verify_domain.tag != ret->tag)
	    ret->tag = safe_strdup(tag);	
	
    } else if (ptr ==  &mail_services_lookup) {
	DPRINT(Debug,7,(&Debug, "   mail_services_lookup\n"));


	if (static_resolv_mserv_lookup || 
	    0 != strcmp(tag,resolv_mserv_lookup.tag)) {
	    ret = safe_malloc(sizeof (*ret));
	    
	    *ret = resolv_mserv_lookup;
	} else {
	    ret = &resolv_mserv_lookup;
	    static_resolv_mserv_lookup = 1;
	}

	if (0 != strcmp(tag,ret->tag) || resolv_mserv_lookup.tag != ret->tag)
	    ret->tag = safe_strdup(tag);

    } else {
	DPRINT(Debug,7,(&Debug, "   %p not found\n",ptr));
    }

    if (ret) {
	ret->start_value = 
	    used_start_value - ret->nlen;	
	ret->next_shared = ptr->first_shared;
	ptr->first_shared = ret;
	DPRINT(Debug,7,(&Debug, 
			"RC_fill_enum_values=%p; start_value=%d nlen=%d\n",
			ret,ret->start_value,ret->nlen));
    } else {
	DPRINT(Debug,7,(&Debug, "RC_fill_enum_values=NULL\n"));
    }

    return ret;
}

static struct rc_save_info_rec * set_enum_option P_((struct rc_save_info_rec   * rc_options,
						     const size_t                rc_option_count,
						     const char                * option_name,
						     const char                * shared_tag,
						     struct dt_enumerate_info ** enumerate_p,
						     struct dt_enum_shared    ** mytag_p));
struct rc_save_info_rec * set_enum_option(rc_options,rc_option_count,option_name,
					  shared_tag,enumerate_p,mytag_p)
     struct rc_save_info_rec   * rc_options;
     const size_t                rc_option_count;
     const char                * option_name;
     const char                * shared_tag;
     struct dt_enumerate_info ** enumerate_p;
     struct dt_enum_shared    ** mytag_p;
{
    struct rc_save_info_rec * rec =
	rc_locate_option(rc_options,rc_option_count,
			 option_name,NULL);

    if (enumerate_p)
	* enumerate_p = NULL;
    if (mytag_p)
	*mytag_p = NULL;

    if (rec) {
	if (ison(rec->flags,FL_CHANGED)) {
	    DPRINT(Debug,7,(&Debug,
			    "set_enum_option: %s value is changed, not a default value\n",
			    option_name));
		
	} else if (&rc_DT_ENUM == rec->dt_type) {
	    
	    if (rec->val.enumerate->val_as_string) {  /* Should not happen */
		DPRINT(Debug,7,(&Debug,
				"set_enum_option: %s: string value: %s\n",
				option_name,
				rec->val.enumerate->val_as_string));
		
	    } else {
		struct dt_enum_shared * mytag;
		
		for (mytag = rec->val.enumerate->first_shared;
		     mytag;
		     mytag = mytag -> next_shared) {
		    
		    if (mytag->magic !=   DT_ENUM_SHARED_magic)
			panic("SHARED PANIC",__FILE__,__LINE__,
			      "set_enum_option",
			      "Bad magic bumber",0);
		    
		    if (0 == strcmp(mytag->tag,shared_tag))
			break;
		}

		if (!mytag) 
		    mytag =
			shared_fill_dt_enumerate_values(rec->val.enumerate,
							shared_tag,
							NULL,
							rec->name);

		if (enumerate_p)
		    * enumerate_p = rec->val.enumerate;
		if (mytag_p)
		    *mytag_p      = mytag;

		DPRINT(Debug,7,(&Debug,
				"set_enum_option: %s found",
				rec->name));

		if (mytag) {
		    DPRINT(Debug,7,(&Debug,", mytag %s start value %d nlen %d",
				    mytag->tag,
				    mytag->start_value,
				    mytag->nlen));
		}

		DPRINT(Debug,7,(&Debug,"\n"));
		
		return rec;
	    }
	} else {
	    DPRINT(Debug,7,(&Debug,
			    "set_enum_option: %s have unexpected type\n",
			    option_name));
	}


    } else {
	DPRINT(Debug,7,(&Debug,
			"set_enum_option: %s not found",
			option_name));
    }

    return NULL;
}

static int change_enum_option_value  P_((struct rc_save_info_rec   * rc_options,
					 const size_t                rc_option_count,
					 const char                * shared_tag,
					 const char                * option_name,
					 unsigned                    option_value));
static int change_enum_option_value(rc_options,rc_option_count,shared_tag,option_name,
				    option_value)
     struct rc_save_info_rec   * rc_options;
     const size_t                rc_option_count;
     const char                * shared_tag;
     const char                * option_name;
     unsigned                    option_value;
{
    struct dt_enumerate_info  * enumerate_option = NULL;
    struct dt_enum_shared     * mytag = NULL;

    struct rc_save_info_rec * rec =
	set_enum_option(rc_options,rc_option_count,
			option_name,
			shared_tag,
			&enumerate_option,
			&mytag);

    if (rec) {
	
	if (mytag) {
	    int x;
	    
	    if (mytag->magic !=   DT_ENUM_SHARED_magic)
		panic("SHARED PANIC",__FILE__,__LINE__,
		      "change_enum_option_value",
		      "Bad magic bumber",0);

	    if (mytag->nlen < 0)
		if (mytag->magic != DT_ENUM_SHARED_magic)
		    panic("SHARED PANIC",__FILE__,__LINE__,
			  "change_enum_option_value",
			  "Bad value count",0);
	    
	    if (option_value >= (unsigned)(mytag->nlen))
		panic("SHARED PANIC",__FILE__,__LINE__,
		      "change_enum_option_value",
		      "Option value out of range",0);
	    x = option_value;
	    if (! mytag->list ||
		! mytag->list[x])
		panic("SHARED PANIC",__FILE__,__LINE__,
		      "change_enum_option_value",
		      "No option value",0);
	    
	    if (enumerate_option) {
		enumerate_option->val = 
		    mytag->start_value + x;
		
		DPRINT(Debug,7,(&Debug,
				"change_enum_option_value: %s: set to %s:%s (%d)\n",
				option_name,
				shared_tag,
				mytag->list[x],
				enumerate_option->val));
		
		return 1;
	    } else {
		DPRINT(Debug,7,(&Debug,
				"change_enum_option_value: %s: no enumerate option\n",
				option_name));
	    }
	} else {
	    
	    DPRINT(Debug,7,(&Debug,
			    "change_enum_option_value: %s: my (%s) values are not filled\n",
			    option_name,
			    shared_tag));
	}
    } else {
	DPRINT(Debug,7,(&Debug,
			"change_enum_option_value: %s: No change\n",
			option_name));

    }

    return 0;
}

static void change_verify_domain P_((struct rc_save_info_rec   * rc_options,
				     const size_t                rc_option_count,
				     const char                * shared_tag));
static void change_verify_domain(rc_options,rc_option_count,shared_tag)
     struct rc_save_info_rec   * rc_options;
     const size_t                rc_option_count;
     const char                * shared_tag;
{
    if (change_enum_option_value(rc_options,rc_option_count,shared_tag,
				 "verify-domain",rvd_mx_address)) {
	
	DPRINT(Debug,7,(&Debug,
			"change_verify_domain: Default value updated\n"));
    } else {
	DPRINT(Debug,7,(&Debug,
			"change_verify_domain: No change\n"));

    }
}

static void change_mserv_lookup P_((struct rc_save_info_rec   * rc_options,
				     const size_t                rc_option_count,
				     const char                * shared_tag));
static void change_mserv_lookup(rc_options,rc_option_count,shared_tag)
     struct rc_save_info_rec   * rc_options;
     const size_t                rc_option_count;
     const char                * shared_tag;
{
    switch (query_mode) {
    case qm_blocking:
	DPRINT(Debug,7,(&Debug,
			"change_mserv_lookup: query-mode=%s, no changes to default value of mail-services-lookup\n",
			QUERY_MODE[query_mode]));
    case NUM_query_mode:  /* Bad value */
	break;
	
    case qm_non_blocking:
	if (change_enum_option_value(rc_options,rc_option_count,shared_tag,
				     "mail-services-lookup",
				     mserv_all_addresses)) {

	    DPRINT(Debug,7,(&Debug,
			    "change_mserv_lookup: query-mode=%s, default value of mail-services-lookup updated\n",
			    QUERY_MODE[query_mode]));

	} else {
	    DPRINT(Debug,7,(&Debug,
			    "change_mserv_lookup: query-mode=%s, mail-services-lookup not changed\n",
			    QUERY_MODE[query_mode]));
	}
	break;
    }
}


E_(RC_change_config_f RC_change_config)
void RC_change_config P_((int *errors, int flag,
			  const char * tag,
			  struct rc_save_info_rec * rc_options,
			  size_t  rc_option_count,
			  size_t rec_size,
			  const char * shared_tag));

void RC_change_config(errors,flag,tag,rc_options,rc_option_count,
		      rc_size,shared_tag)
     int *errors; 
     int flag;
     const char * tag;
     struct rc_save_info_rec * rc_options;
     size_t  rc_option_count;
     size_t rc_size;
     const char * shared_tag;
{
    
    if (rc_size != sizeof(rc_options[0])) {
	DPRINT(Debug,1,(&Debug,
			"RC_change_config: Bad rc_size %d (should be %d)\n",
			rc_size,sizeof(rc_options[0])));
	
	return;
    }

    DPRINT(Debug,7,(&Debug, "RC_change_config\n"));

    if (!tag) {
	change_verify_domain(rc_options,rc_option_count,shared_tag);
	change_mserv_lookup(rc_options,rc_option_count,shared_tag);

    } else {
	DPRINT(Debug,7,(&Debug,
			"RC_change_config: %s config\n",tag));
    }
}

/* Returns domain name if matches to query-domain-blacklist */
const char * is_query_fallback_domain P_((const char * host_domain_name));

enum query_address { qa_default, qa_IPv4, qa_IPv6, 
		     qa_IPv6_IPv4, qa_IPv4_IPv6, NUM_query_address };

static char * QUERY_ADDRESS[NUM_query_address+1] = {  "default","IPv4", "IPv6", 
						      "IPv6-IPv4", "IPv4-IPv6", NULL };

static ENUMERATE query_address = {
    qa_default,
    NUM_query_address,&(QUERY_ADDRESS[0]),
    NULL,
    0   	/* no boolean */,
    NULL	/* not delayed */,
    NULL 
};

struct resolv_query_mode blocked_query_mode = {
    RESOLV_QUERY_MODE_magic,	
    have_blocking_qm,
    lookup_resolv_cache_blocked
};


static struct resolv_cache * call_lookup_resolv_cache(record,name,now,
						      q_type,is_search,
						      looking_up,search_name,
						      have_error,other,qm,cancel_p)
     struct resolv_cache             * record;
     const char                      * name;
     const struct schedule_timelimit * now;
     const ns_type                     q_type;
     const int                         is_search;
     enum looking_up                 * looking_up;
     char                           ** search_name;
     int                             * have_error;
     int                             * other;
     enum query_mode_val                   qm;
     struct cancel_data ** cancel_p 
     /* May be NULL, Used if dns lookup was cancelable */;
{
    if (qm < 0 || qm >= NUM_query_mode || !query_modes[qm])
	panic("RESOLV PANIC",__FILE__,__LINE__," call_lookup_resolv_cache",
	      "Bad query name",0);
    
    if (RESOLV_QUERY_MODE_magic != query_modes[qm]->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,
	      " call_lookup_resolv_cache",
	      "Bad magic number (resolv_query_mode)",0);
    
    return query_modes[qm]->
	lookup_resolv_cache(record,name,now,
			    q_type,is_search,
			    looking_up,search_name,
			    have_error,other,cancel_p);
}

enum search_mdomain { smd_query_exact, smd_search_name, smd_search_dotless,
		      NUM_search_mdomain };

static char * SEARCH_MDOMAIN[NUM_search_mdomain+1] = { "query-exact", "search-name", 
						       "search-dotless", 
						       NULL };

static ENUMERATE search_mdomain = {
    smd_search_name,
    NUM_search_mdomain, &(SEARCH_MDOMAIN[0]),
    NULL,
    1   	/* allow boolean ON, OFF, TRUE, FALSE, YES and NO */,
    NULL   	/* not delayed */,
    NULL
};

E_(verify_mail_domain_f verify_mail_domain2)
enum verify_domain_status verify_mail_domain2 
   P_((const char  *                     domain,
       char                           ** rewrite,
       int                               translated_value  /* -1 if not share specific */,
       const struct dt_enum_shared     * is_shared_value,
       size_t                            dt_enum_shared_size,
       const struct schedule_timelimit * now,
       size_t                            schedule_timelimit_size,
       struct schedule_timelimit       * valid_until,
       struct cancel_data             ** cancel_p    				   
       ));
enum verify_domain_status verify_mail_domain2(domain,rewrite,
					     translated_value,
					     is_shared_value,
					     dt_enum_shared_size,
					     now,
					     schedule_timelimit_size,
					     valid_until,cancel_p)
     const char                      * domain;
     char                           ** rewrite;
     int                               translated_value  /* -1 if not share specific */;
     const struct dt_enum_shared     * is_shared_value;
     size_t                            dt_enum_shared_size;
     const struct schedule_timelimit * now;
     size_t                            schedule_timelimit_size;
     struct schedule_timelimit       * valid_until;
     struct cancel_data             ** cancel_p;				   
{
    enum resolv_verify_domain mode = rvd_mx_address;
    enum query_address qa =
	give_dt_enumerate_as_int(&query_address);
    enum search_mdomain smd =
	give_dt_enumerate_as_int(&search_mdomain);
    const char * use_fallback;

    ns_type query_list[3];
    int query_list_len = 0;
    int a;
    int have_search = 0;
    const char *dot;
    enum looking_up looking_up = l_up_none;
    enum verify_domain_status ret = vd_status_no_routine;
    int have_error = 0;
    int other = 0;
    struct schedule_timelimit max_valid_until;
    int set_valid_until = 1;

    int                        need_interface_change =
	interface_change.val && !pending_interface_change;
    
    if (dt_enum_shared_size != sizeof (* is_shared_value)) {
	DPRINT(Debug,1,(&Debug,
			"verify_mail_domain2: Bad dt_enum_shared_size %d (should be %d)\n",
			dt_enum_shared_size,
			sizeof (* is_shared_value)));

	return vd_status_no_routine;
    }

    if (schedule_timelimit_size != sizeof (* now)) {
	DPRINT(Debug,1,(&Debug,
			"verify_mail_domain2: Bad (schedule_timelimit_size %d (should be %d)\n",
			schedule_timelimit_size,
			sizeof (* now)));
	return vd_status_no_routine;
    }
    set_MAX_schedule_timelimit(&max_valid_until);
    
    if (&resolv_verify_domain == is_shared_value) {
	/* translated_value is enum resolv_verify_domain */

	if (translated_value >= 0 && 
	    translated_value < NUM_resolv_verify_domain) {
	    mode = translated_value;

	    DPRINT(Debug,7,(&Debug, "verify_mail_domain2: mode=%s\n",
			    RESOLV_VERIFY_DOMAIN[mode]));
	}
    }
       
    dot = strchr(domain,'.');
    /* First char can not be '.' */
    if (&(domain[0]) == dot) {
	DPRINT(Debug,7,(&Debug, "verify_mail_domain2: %s: bad domain\n",
			domain));
	return vd_domain_bad_syntax;
    }


    use_fallback = is_query_fallback_domain(domain);
    if (use_fallback) {
	DPRINT(Debug,7,(&Debug, 
			"verify_mail_domain2: %s: Use fallback for %s\n",
			domain,use_fallback));
	return vd_status_fallback;
    }

    if (query_mode >= 0 && query_mode < NUM_query_mode && query_modes[query_mode]) {
	if (RESOLV_QUERY_MODE_magic != query_modes[query_mode]->magic)
	    panic("RESOLV PANIC",__FILE__,__LINE__,
		  "verify_mail_domain2",
		  "Bad magic number (resolv_query_mode)",0);

	if (! query_modes[query_mode]->have_query_mode()) {
	    lib_error(CATGETS(elm_msg_cat, 
			      ResolvSet,
			      ResolvQMNotAvailable,
			      "Resolv query-mode = %s is not available"),
		      QUERY_MODE[query_mode]);
	    return vd_status_no_routine;
	}
    } else {
	DPRINT(Debug,7,(&Debug, 
			"verify_mail_domain2: %s: query-mode = %d\n",
			domain,query_mode));
	return vd_status_no_routine;
    }
                   
    switch (smd) {
    case smd_query_exact:   
	have_search = 0;  
	DPRINT(Debug,7,(&Debug, 
			"verify_mail_domain2: search mail domain=%s; no search\n",
			 SEARCH_MDOMAIN[smd]));
	break;
    case smd_search_name:  
	have_search = 1;
	DPRINT(Debug,7,(&Debug, 
			"verify_mail_domain2: search mail domain=%s; have search\n",
			 SEARCH_MDOMAIN[smd]));
	break;
    case smd_search_dotless:
	have_search = ( NULL == dot);
	DPRINT(Debug,7,(&Debug, 
			"verify_mail_domain2: search mail domain=%s; domain=%s; %s\n",
			SEARCH_MDOMAIN[smd],domain,
			have_search ? "have search" : "no search"));
	break;
    case NUM_search_mdomain:  /* Bad value */
	break;
    }

    switch (mode) {
    case rvd_mx_address:
	query_list[query_list_len++] = ns_t_mx;

	switch (qa) {
	case qa_IPv4:
	    query_list[query_list_len++] = ns_t_a;
	    break;
	case qa_IPv6:
	    query_list[query_list_len++] = ns_t_aaaa;
	    break;
	case qa_IPv6_IPv4:
	    query_list[query_list_len++] = ns_t_aaaa;
	    query_list[query_list_len++] = ns_t_a;
	    break;
	case qa_default:   /* do not check interface addresses */
	case qa_IPv4_IPv6:
	    query_list[query_list_len++] = ns_t_a;
	    query_list[query_list_len++] = ns_t_aaaa;
	    break;
	case NUM_query_address:  /* bad value */
	    break;
	}
	break;
	
    case rvd_mx:
	query_list[query_list_len++] = ns_t_mx;
	break;

    case rvd_any:
	query_list[query_list_len++] =  ns_t_any;
	break;

    case NUM_resolv_verify_domain:   /* bad value */
	break;
    }

    
    if (query_list_len > sizeof (query_list) / sizeof (query_list[0]))
	panic("RESOLV PANIC",__FILE__,__LINE__,"verify_mail_domain2",
	      "overflow",0);

    if (query_list_len < 0) {
	DPRINT(Debug,7,(&Debug, "verify_mail_domain2: %s: no query\n",
			domain));
	return vd_status_no_routine;
    }

    if (need_interface_change) {
	int query_flags = INTERFACE_ADDRLIST | INTERFACE_CHANGE;
	
	int interfaces_count = 0;
	struct interface_info ** interfaces UNUSED_VAROK = 
	    get_interfaces(sizeof (struct interface_info),
			   sizeof (struct interface_addr),
			   &interfaces_count,&query_flags);


	if (0 == (query_flags & INTERFACE_CHANGE)) {
	    DPRINT(Debug,7,(&Debug, "verify_mail_domain2: %s: Interfaces changed\n",
			    domain));
	}
	    
	need_interface_change = 0;
	    
	if (0 == (query_flags & INTERFACE_ADDRLIST)) {
	    DPRINT(Debug,7,(&Debug, "verify_mail_domain2: %s: failed to get interface address list\n",
			    domain));
	}	
    }

    for (a = 0; a < query_list_len;  a++) {
	ns_type q_type = query_list[a];
	
	struct resolv_cache  * search = NULL;
	char                 * search_name = NULL;
	struct resolv_cache  * cache  = NULL;
	struct resolv_result ** result = NULL;
	int found = 0;
	
	other = 0;

	DPRINT(Debug,7,(&Debug, "verify_mail_domain2: %s: %s #%d: ",
			domain,have_search ? "search" : "query",a));
	switch(q_type) {
	case ns_t_mx:   DPRINT(Debug,7,(&Debug, "MX  : ")); break;
	case ns_t_a:    DPRINT(Debug,7,(&Debug, "A   : ")); break;
	case ns_t_aaaa: DPRINT(Debug,7,(&Debug, "AAAA: ")); break;
	case ns_t_any:  DPRINT(Debug,7,(&Debug, "ANY : ")); break;
	default:
	case ns_t_invalid:  /* bad value */                 break;
	}
	DPRINT(Debug,7,(&Debug, "\n"));
	    
	/* If answer expires during query, it may be removed from
	   struct resolv_cache  * cache (on some cases) but it is still on
	   struct resolv_result ** result
	*/
	
	cache = query_resolv_cache(domain,q_type,
				   have_search ? &search : NULL,
				   &search_name,
				   &result,&looking_up,&found,
				   &have_error,
				   &other,
				   now,query_mode,
				   cancel_p);

	if (cache) {

	    if (found && result) {
		int i;
		
		DPRINT(Debug,7,(&Debug, 
				"verify_mail_domain2: %s: found %d records\n",
				domain,found));

		if (rewrite && search_name) {
		    if (*rewrite)
			free(*rewrite);
		    *rewrite = safe_strdup(search_name);
		}

		for (i = 0; result[i]; i++) {
		    if (RESOLV_RESULT_magic != result[i]->magic)
				panic("RESOLV PANIC",__FILE__,__LINE__,
				      "verify_mail_domain2",
				      "Bad magic number (resolv_result)",0);

		    switch(result[i]->rectype) {
		    case ns_t_mx:
			if (result[i]->data.mail_exchanger) {
			    
			    if (MX_RESULT_magic != 
				result[i]->data.mail_exchanger->magic)
				panic("RESOLV PANIC",__FILE__,__LINE__,
				      "verify_mail_domain2",
				      "Bad magic number (mx_result)",0);
			    
			    if ('\0' == result[i]->
				data.mail_exchanger->name[0] ||
				0 == strcmp(result[i]->data.
					    mail_exchanger->name,
					    ".")) {
				
				DPRINT(Debug,7,(&Debug, 
						"verify_mail_domain2: %s does not accept mail (MX %d %s)\n",
						result[i]->name,
						result[i]->data.
						mail_exchanger->precedence,
						result[i]->data.
						mail_exchanger->name));
				
				
				if (vd_status_no_routine == ret)
				    ret =  vd_domain_no_mail;
				
			    } else {
				DPRINT(Debug,7,(&Debug, 
						"verify_mail_domain2: %s have MX %d %s\n",
						result[i]->name,
						result[i]->data.
						mail_exchanger->precedence,
						result[i]->data.
						mail_exchanger->name));
				
				ret = vd_status_verified;
			    }
			}		    
			break;
		    case ns_t_cname:
			if (result[i]->data.domain_name) {
			    if (NAME_RESULT_magic !=
				result[i]->data.domain_name->magic)
				panic("RESOLV PANIC",__FILE__,__LINE__,
				      "verify_mail_domain2",
				      "Bad magic number (name_result)",0);

			    DPRINT(Debug,7,(&Debug, 
					    "verify_mail_domain2: %s is alias of %s\n",
					    result[i]->name,
					    result[i]->data.
					    domain_name->name));

			    if (schedule_shorten_next_timeout(&max_valid_until,
							      & (result[i]->valid_until))) {

				char * A = schedule_timeout_string(&max_valid_until);
				if (A) {
				    DPRINT(Debug,7,(&Debug, 
						    "verify_mail_domain2: %s: => max valid until %s\n",
						    domain,A));
				    free(A);
				}				   				   
			    }
			}
			break;
		    case ns_t_a: 
			DPRINT(Debug,7,(&Debug, 
					"verify_mail_domain2: %s have IPv4 address",
					result[i]->name));
			
#ifdef USE_INET_XTOX_PN  
			{
			    char address[256];
			    const char *addr = 
				inet_ntop(AF_INET,& (result[i]->data.ip4addr),
					  address,sizeof address);
			    if (addr) {
				DPRINT(Debug,7,(&Debug," %s",addr));
			    }
			}			
#endif			
			DPRINT(Debug,7,(&Debug, "\n"));
			break;

#ifdef HAVE_IN6
		    case ns_t_aaaa:			
			if (result[i]->data.ip6addr) {
			    DPRINT(Debug,7,(&Debug, 
					    "verify_mail_domain2: %s have IPv6 address",
					    result[i]->name));
			    
#ifdef USE_INET_XTOX_PN  
			    {
				char address[256];
				const char *addr = 
				    addr = inet_ntop(AF_INET6,result[i]->data.ip6addr,
						     address,sizeof address);
				if (addr) {
				    DPRINT(Debug,7,(&Debug," %s",addr));
				}
			    }
#endif
			    DPRINT(Debug,7,(&Debug, "\n"));
			    
			}
			break;
#endif

		    default:   /* ? */
			break;
		    }

		    if (valid_until) {
			char * A = NULL;
			
			if (set_valid_until) {
			    * valid_until = result[i]->valid_until;
			    A = schedule_timeout_string(valid_until);
			    set_valid_until = 0;
			    
			} else if (schedule_extend_valid_until(valid_until,
							       &(result[i]->valid_until))) {
			    A = schedule_timeout_string(valid_until);
			}
			
			if (A) {
			    DPRINT(Debug,7,(&Debug, 
					    "verify_mail_domain2: %s: => valid until %s\n",
					    domain,A));
			    free(A);
			}	
		    }
		}

		if (vd_status_no_routine == ret)
		    ret =  vd_status_verified;
	    }
	    	    
	    free_resolv_cache(&cache);
	}

	if (result) {
	    int i;

	    for (i = 0; result[i]; i++)
		free_resolv_result(& result[i]);
	    free(result);
	    result = NULL;
	}

	if (search)
	    free_resolv_cache(&search); 
	if (search_name) {
	    free(search_name);
	    search_name = NULL;
	}

	if (have_error && cancel_p && *cancel_p &&
	    is_canceled(*cancel_p)) {
	    
	    DPRINT(Debug,7,(&Debug, "verify_mail_domain2: %s: DNS lookup canceled\n",
			    domain));
	    
	    ret = vd_status_failure;
	    goto fail;
	}
	
	if (vd_status_no_routine != ret)
	    break;
	if (!other) {
	    DPRINT(Debug,7,(&Debug, 
			    "verify_mail_domain2: %s - no other data\n",
			    domain));

	    break;
	}

    }

    if (valid_until) {

	if (set_valid_until) {
	    char * A = NULL;
	    
	    * valid_until = NO_schedule_timelimit;
	    A  = schedule_timeout_string(valid_until);
	    
	    if (A) {
		DPRINT(Debug,7,(&Debug, 
				"verify_mail_domain2: %s: => valid until %s\n",
				domain,A));
		free(A);
	    }
	    
	} else if (schedule_shorten_next_timeout(valid_until,
					  &  max_valid_until)) {
	    char * A  = schedule_timeout_string(valid_until);
	    if (A) {
		DPRINT(Debug,7,(&Debug, 
				"verify_mail_domain2: %s: => shortening valid until %s\n",
				domain,A));
		free(A);
	    }
	}
    }

    if (vd_status_no_routine == ret) {

	DPRINT(Debug,7,(&Debug, 
			"verify_mail_domain2: %s no result",
			domain));

	if (have_error) {
	    ret = vd_status_failure;
	    DPRINT(Debug,7,(&Debug, ", failure (%d errors)\n",
			    have_error));
	} else {
	    ret = vd_status_not_found;
	    DPRINT(Debug,7,(&Debug, ", not found\n"));
	}
    }

    
    switch (looking_up) {
    case  l_up_none: break;
    case  l_up_printed:
    case  l_up_truncated_retry:

	switch(ret) {
	case vd_status_no_routine:
	case vd_status_fallback:
	case vd_domain_bad_syntax:	    
	case vd_status_failure:
	    lib_error(CATGETS(elm_msg_cat, 
				  ResolvSet,
				  ResolvLookingUpSomeFailure,
				  "Looking up %s: some failure?"),
			  domain);
	    break;
	case vd_status_not_found:
	    if (other)
		lib_error(CATGETS(elm_msg_cat, 
				  ResolvSet,
				  ResolvLookNoData,
				  "Looking up %s: no data"),
			  domain);
	    else
		lib_error(CATGETS(elm_msg_cat, 
				  ResolvSet,
				  ResolvLookNotFound,
				  "Looking up %s: not found"),
		      domain);
	    break;
	case vd_status_verified:
	    lib_transient(CATGETS(elm_msg_cat, 
				  ResolvSet,
				  ResolvLookOK,
				  "Looking up %s: OK"),
			  domain);
	    break;
	case vd_domain_no_mail:
	    lib_error(CATGETS(elm_msg_cat, 
			      ResolvSet,
			      ResolvLookingNoMail,
			      "Looking up %s: Mail not accepted"),
		      domain);
	    break;
	}
	break;
	
    case l_up_error:
    case l_up_stalled:
	/* Error message was printed */
	break;	

    }

 fail:
    
    DPRINT(Debug,7,(&Debug, "verify_mail_domain2=%d",
		    ret));

    switch(ret) {
    case vd_status_failure:    DPRINT(Debug,7,(&Debug, " failure")); break;
    case vd_status_not_found:  DPRINT(Debug,7,(&Debug, " not found")); break;
    case vd_status_verified:   DPRINT(Debug,7,(&Debug, " verified")); break;
    case vd_domain_no_mail:    DPRINT(Debug,7,(&Debug, " no mail")); break;
    case vd_status_no_routine: DPRINT(Debug,7,(&Debug, " no routine")); break;
    case vd_status_fallback:   DPRINT(Debug,7,(&Debug, " fallback")); break;
    case vd_domain_bad_syntax: DPRINT(Debug,7,(&Debug, " bad syntax")); break;
    }
    
    DPRINT(Debug,7,(&Debug, "\n"));

    return ret;
}

enum search_mservice { sms_query_exact, sms_search_name, sms_search_dotless,
		       sms_fallback_dotless, sms_fallback_dotless_search_name,
		       sms_static_dotless, sms_static_dotless_search_name,
		       NUM_search_mservice };

static char * SEARCH_MSERVICE[NUM_search_mservice+1] = { "query-exact", "search-name", 
							 "search-dotless", "fallback-dotless", 
							 "fallback-dotless-search-name",
							 "static-hosts-dotless",
							 "static-hosts-dotless-search-name",
							 NULL};


static ENUMERATE search_mservice = {
    NUM_search_mservice  /* RC_post_init2 sets value */,
    NUM_search_mservice, &(SEARCH_MSERVICE[0]), 
    NULL,
    1   	/* allow boolean ON, OFF, TRUE, FALSE, YES and NO */,
    NULL   	/* not delayed */,
    NULL
};

E_(notify_if_change_f notify_if_change)
void notify_if_change P_((void));
void notify_if_change()
{
    int need_clear = dt_flag_is_set(&interface_change,intch_clear_cache_flag);
    char *s = " ";
    
    DPRINT(Debug,7,(&Debug, "notify_if_change:"));

    if (pending_interface_change) {
	DPRINT(Debug,7,(&Debug, "%spending_interface_change set",
			s));
	s = ", ";
    }
    
    if (need_clear) {
	DPRINT(Debug,7,(&Debug, "%sinterface-change=clear-cache",
			s));
	s = ", ";
    }
    
    if (! pending_interface_change) {
	DPRINT(Debug,7,(&Debug, "%ssetting pending_interface_change",
			s));

	s = ", ";
	pending_interface_change = 1;				
    }

    DPRINT(Debug,7,(&Debug, "\n"));
}


#ifdef REMOTE_MBX

E_(lookup_service_addresses_f lookup_service_addresses2)
enum address_lookup_status lookup_service_addresses2
   P_((const char                      * hostname,
       char                           ** rewrite,
       int                               translated_value  
       /* -1 if not share specific */,
       const struct dt_enum_shared     * is_shared_value,
       size_t                            dt_enum_shared_size,
       struct service_entry_addr      ** addr_list,
       int                             * addr_count,
       size_t                            service_entry_addr_size,
       struct service_entry_name      ** aliases_list,
       int                             * aliases_count,
       size_t                            service_entry_name_size,
       const struct service_type       * service,
       const struct schedule_timelimit * now,
       size_t                            schedule_timelimit_size,
       const struct schedule_timelimit * default_valid_until /* for static hosts lookup */,
       int                             * was_error_p,
       struct cancel_data             ** cancel_p));
enum address_lookup_status lookup_service_addresses2(hostname,
						     rewrite,translated_value,
						     is_shared_value,
						     dt_enum_shared_size,
						     addr_list,addr_count,
						     service_entry_addr_size,
						     aliases_list,aliases_count,
						     service_entry_name_size,
						     service,now,
						     schedule_timelimit_size,
						     default_valid_until,
						     was_error_p,
						     cancel_p)
     const char                      * hostname;
     char                           ** rewrite;
     int                               translated_value  /* -1 if not share specific */;
     const struct dt_enum_shared     * is_shared_value;
     size_t                            dt_enum_shared_size;
     struct service_entry_addr      ** addr_list;
     int                             * addr_count;
     size_t                            service_entry_addr_size;
     struct service_entry_name      ** aliases_list;
     int                             * aliases_count;
     size_t                            service_entry_name_size;
     const struct service_type       * service;
     const struct schedule_timelimit * now;
     size_t                            schedule_timelimit_size;
     const struct schedule_timelimit * default_valid_until /* for static hosts lookup */;
     int                             * was_error_p;
     struct cancel_data             ** cancel_p;   
{
    enum resolv_mserv_lookup   mode                = mserv_all_addresses;
    enum query_address         qa                  =
	give_dt_enumerate_as_int(&query_address);
    enum search_mservice       sms                 =
	give_dt_enumerate_as_int(&search_mservice);
    const char               * dot;
    const char               * use_fallback;
    int                        have_search         = 0;
    int                        try_static_hosts    = 0;
    const char               * static_hosts_tried  = NULL;
    ns_type                    query_list[3];
    int                        query_list_len      = 0;
    int                        a;
    enum looking_up            looking_up          = l_up_none;
    enum address_lookup_status ret                 = al_status_no_routine;
    enum address_lookup_status saved_ret           = al_status_no_routine;
    int                        have_error          = 0;
    int                        other               = 0;
    struct schedule_timelimit  max_valid_until;

    int                        old_addr_count      = *addr_count;
    int                        old_aliases_count   = *aliases_count;

    int                        need_interface_change =
	interface_change.val && !pending_interface_change;
    
    if (dt_enum_shared_size != sizeof (* is_shared_value)) {
	DPRINT(Debug,1,(&Debug,
			"lookup_service_addresses2: Bad dt_enum_shared_size %d (should be %d)\n",
			dt_enum_shared_size,
			sizeof (* is_shared_value)));
	
	return al_status_no_routine;
    }
    
    if (service_entry_addr_size != sizeof (** addr_list)) {
	DPRINT(Debug,1,(&Debug,
			"lookup_service_addresses2: Bad service_entry_addr_size %d (should be %d)\n",
			service_entry_addr_size,
			sizeof (** addr_list)));
	
	return al_status_no_routine;
    }

    if (service_entry_name_size != sizeof (** aliases_list)) {
	DPRINT(Debug,1,(&Debug,
		"lookup_service_addresses2: Bad (service_entry_name_size %d (should be %d)\n",
			service_entry_name_size,
			sizeof (** aliases_list)));
	return al_status_no_routine;
    }

    if (schedule_timelimit_size != sizeof (* now)) {
	DPRINT(Debug,1,(&Debug,
		"lookup_service_addresses2: Bad (schedule_timelimit_size %d (should be %d)\n",
			schedule_timelimit_size,
			sizeof (* now)));
	return al_status_no_routine;
    }

    set_MAX_schedule_timelimit(&max_valid_until);
    
    if (&resolv_mserv_lookup  == is_shared_value) {
	/* translated_value is enum resolv_mserv_lookup */
	
	if (translated_value >= 0 && 
	    translated_value < NUM_resolv_mserv_lookup) {
	    mode = translated_value;
	    
	    DPRINT(Debug,7,(&Debug, "lookup_service_addresses2: mode=%s\n",
			    RESOLV_MSERV_LOOKUP[mode]));
	}
    }
    
     dot = strchr(hostname,'.');
     /* first char can not be '.' */

     if (&(hostname[0]) == dot) {
	 DPRINT(Debug,7,(&Debug, "lookup_service_addresses2: %s: bad hostname\n",
			 hostname));

	 return al_status_bad_syntax;
     }

    use_fallback = is_query_fallback_domain(hostname);
    if (use_fallback) {
	DPRINT(Debug,7,(&Debug, "lookup_service_addresses2: %s: Use fallback for %s\n",
			hostname,use_fallback));
	
	return al_status_fallback;
    }

    if (query_mode >= 0 && query_mode < NUM_query_mode && query_modes[query_mode]) {
	if (RESOLV_QUERY_MODE_magic != query_modes[query_mode]->magic)
	    panic("RESOLV PANIC",__FILE__,__LINE__,
		  "lookup_service_addresses2",
		  "Bad magic number (resolv_query_mode)",0);

	if (!  query_modes[query_mode]->have_query_mode()) {
	    lib_error(CATGETS(elm_msg_cat, 
			      ResolvSet,
			      ResolvQMNotAvailable,
			      "Resolv query-mode = %s is not available"),
		      QUERY_MODE[query_mode]);
	    return vd_status_no_routine;
	}
    } else {
	DPRINT(Debug,7,(&Debug, 
			"lookup_service_addresses2: %s: query-mode = %d\n",
			hostname,query_mode));
	return vd_status_no_routine;
    }
    
    DPRINT(Debug,10,(&Debug, 
		    "lookup_service_addresses2: %s: search-mail-service = %d",
		    hostname,sms));
    if (sms >= 0 && sms <  NUM_search_mservice && SEARCH_MSERVICE[sms]) {
	DPRINT(Debug,10,(&Debug, " %s",
			SEARCH_MSERVICE[sms]));
    }
    DPRINT(Debug,10,(&Debug, "\n"));
    
    switch (sms) {
    case  sms_query_exact:
	have_search = 0;  
	DPRINT(Debug,7,(&Debug, 
			"lookup_service_addresses2: search mail service=%s; no search\n",
			SEARCH_MSERVICE[sms]));
	break;
    case sms_search_name:
	have_search = 1;
	DPRINT(Debug,7,(&Debug, 
			"lookup_service_addresses2: search mail service=%s; have search\n",
			SEARCH_MSERVICE[sms]));
	break;
    case sms_search_dotless:
	have_search = ( NULL == dot);
	DPRINT(Debug,7,(&Debug, 
			"lookup_service_addresses2: search mail service=%s; %s\n",
			SEARCH_MSERVICE[sms],
			have_search ? "have search" : "no search"));
	break;
    case sms_fallback_dotless:
	DPRINT(Debug,7,(&Debug, 
			"lookup_service_addresses2: search mail service=%s; ",
			SEARCH_MSERVICE[sms]));
	if (!dot) {
	    DPRINT(Debug,7,(&Debug, 
			    "%s: Use fallback\n",
			    hostname));
	    return al_status_fallback;
	}
	have_search = 0;
	DPRINT(Debug,7,(&Debug, "no search\n"));
	break;
    case sms_fallback_dotless_search_name:
	DPRINT(Debug,7,(&Debug, 
			"lookup_service_addresses2: search mail service=%s; ",
			SEARCH_MSERVICE[sms]));
	if (!dot) {
	    DPRINT(Debug,7,(&Debug, 
			    "%s: Use fallback\n",
			    hostname));
	    return al_status_fallback;
	}
	have_search = 1;
	DPRINT(Debug,7,(&Debug, "have search\n"));
	break;
    case sms_static_dotless:
	DPRINT(Debug,7,(&Debug, 
			"lookup_service_addresses2: search mail service=%s; try static hosts\n",
			SEARCH_MSERVICE[sms]));
	try_static_hosts = 1;
	break;
    case sms_static_dotless_search_name:
	DPRINT(Debug,7,(&Debug, 
			"lookup_service_addresses2: search mail service=%s; try static hosts, have_search\n",
			SEARCH_MSERVICE[sms]));
	try_static_hosts = 1;
	have_search = 1;
	break;
    case  NUM_search_mservice: /* Bad value */
	break;
    }

    if (try_static_hosts && NULL == dot) {
	 enum etc_hosts_status st ;

	DPRINT(Debug,7,(&Debug, 
			"lookup_service_addresses2: Trying %s from static hosts first\n",
			hostname));

	static_hosts_tried = hostname;
	st = lookup_static_host(hostname,now, default_valid_until,
				lc_dotless, rewrite,
				addr_list,addr_count,
				aliases_list,aliases_count,
				was_error_p,cancel_p);

	DPRINT(Debug,7,(&Debug, "lookup_service_addresses2: %s static hosts result %d",
			hostname,st));
	switch (st) {
	case etc_hosts_maybe:   DPRINT(Debug,7,(&Debug, " etc_hosts_maybe")); break;
	case etc_hosts_failed:  DPRINT(Debug,7,(&Debug, " etc_hosts_failed\n"));
	    if (cancel_p && *cancel_p &&
		is_canceled(*cancel_p)) {
		
		DPRINT(Debug,7,(&Debug, "lookup_service_addresses2: %s: lookup canceled\n",
				hostname));
		
		ret = al_status_failed;
		goto fail;
	    }
	    DPRINT(Debug,7,(&Debug, "lookup_service_addresses2: "));
	    break;
	case etc_hosts_not_found: DPRINT(Debug,7,(&Debug, " etc_hosts_not_found")); break;
	case etc_hosts_found:     DPRINT(Debug,7,(&Debug, " etc_hosts_found\n"));
	    ret = al_status_done;
	    goto done;
	case etc_hosts_parsed:    DPRINT(Debug,7,(&Debug, " etc_hosts_parsed")); break;
	case etc_hosts_found_cached: DPRINT(Debug,7,(&Debug, " etc_hosts_found_cached\n"));
	    ret = al_status_done;
	    goto done;
	}
	DPRINT(Debug,7,(&Debug, " -- continuing with DNS lookup\n"));
    }
    
    switch (mode) {
	const enum ipv_option * ipv_option_copy;
	size_t                  ipv_option_array_len;
	
    case mserv_addresses:
    case mserv_all_addresses:

	ipv_option_array_len = get_ipv_option(NULL,
					      &ipv_option_copy);

	
	if (ipv_option_copy && ipv_option_array_len > 0) {
	    int i;
	    
	    DPRINT(Debug,7,(&Debug, 
			    "lookup_service_addresses2: %s: Address type set by command line option\n",
			    hostname));
	    
	    for (i = 0; i < ipv_option_array_len &&
		     query_list_len < sizeof (query_list) / sizeof (query_list[0]);
		 i++) {

		switch (ipv_option_copy[i]) {
		case ipv4_option :
		    	DPRINT(Debug,7,(&Debug, 
					"lookup_service_addresses2: %s: querying IPv4 addresses\n",
					hostname));
			query_list[query_list_len++] = ns_t_a;
		    break;
		case ipv6_option :
		    DPRINT(Debug,7,(&Debug, 
				    "lookup_service_addresses2: %s: querying IPv6 addresses\n",
				    hostname));
		    query_list[query_list_len++] = ns_t_aaaa;
		    break;
		}
	    }

	} else {
	    switch (qa) {
	    case qa_IPv4:
		query_list[query_list_len++] = ns_t_a;
		break;
	    case qa_IPv6:
		query_list[query_list_len++] = ns_t_aaaa;
		break;
	    case qa_IPv6_IPv4:
		query_list[query_list_len++] = ns_t_aaaa;
		query_list[query_list_len++] = ns_t_a;
		break;
	    case qa_IPv4_IPv6:
		query_list[query_list_len++] = ns_t_a;
		query_list[query_list_len++] = ns_t_aaaa;
		break;
	    case qa_default: {
		int query_flags = INTERFACE_ADDRLIST |
		    ( need_interface_change ? INTERFACE_CHANGE : 0 );
		int interfaces_count = 0;
		struct interface_info ** interfaces = 
		    get_interfaces(sizeof (struct interface_info),
				   sizeof (struct interface_addr),
				   &interfaces_count,&query_flags);
		
		if (0 == (query_flags & INTERFACE_CHANGE)) {
		    DPRINT(Debug,7,(&Debug, "lookup_service_addresses2: %s: Interfaces changed\n",
				    hostname));
		}
		
		need_interface_change = 0;
		
		if (0 == (query_flags & INTERFACE_ADDRLIST)) {
		    DPRINT(Debug,7,(&Debug, "lookup_service_addresses2: %s: failed to get interface address list\n",
				    hostname));
		    query_list[query_list_len++] = ns_t_a;
		    query_list[query_list_len++] = ns_t_aaaa;
		    
		} else if (interfaces) {
		    int seen_ipv4 = 0, seen_ipv6 = 0;
		    
		    int i_idx;
		    
		    for (i_idx = 0; i_idx < interfaces_count; i_idx++) {
			
			if (interfaces[i_idx]) {
			    if (INTERFACE_INFO_magic != interfaces[i_idx]->magic)
				panic("RC PANIC",__FILE__,__LINE__,
				      "lookup_service_addresses2",
				      "Bad magic number (interface_info)",0);
			    
			    if (! (interfaces[i_idx]->valid_addrlist)) {
				DPRINT(Debug,7,(&Debug,"lookup_service_addresses2: %s: interface \"%s\" skipped, not on address list.\n",
						hostname, interfaces[i_idx]->name));
				continue;			    
			    }
			    
			    if (0 == strcmp(interfaces[i_idx]->name,"lo")) {
				DPRINT(Debug,7,(&Debug, 
						"lookup_service_addresses2: %s: interface \"%s\" is ignored, when looking interface addresses\n",
						hostname,interfaces[i_idx]->name));
				continue;
			    }
			    
			    if (interfaces[i_idx]->addr_list) {
				int a_idx;
				
				DPRINT(Debug,12,(&Debug, 
						 "lookup_service_addresses2: %s: interface \"%s\": %d addresses\n",
						 hostname,interfaces[i_idx]->name,interfaces[i_idx]->addr_count));
				
				for (a_idx = 0; a_idx < interfaces[i_idx]->addr_count;
				     a_idx++) {
				    
				    if (interfaces[i_idx]->addr_list[a_idx]) {
					
					if (INTERFACE_ADDR_magic !=
					    interfaces[i_idx]->addr_list[a_idx]->magic)
					    panic("RC PANIC",__FILE__,__LINE__,
						  "lookup_service_addresses2",
						  "Bad magic number (interface_addr)",
						  0);
					
					if (!  (interfaces[i_idx]->addr_list[a_idx]->valid_address)) {
					    DPRINT(Debug,12,(&Debug,
							     "lookup_service_addresses2: %s: interface \"%s\", address #%d skipped, not on address list.\n",
							     hostname,interfaces[i_idx]->name,a_idx));
					    continue;
					}
					
					if (interfaces[i_idx]->addr_list[a_idx]->interface_loopback) {
					    DPRINT(Debug,12,(&Debug,
							     "lookup_service_addresses2: %s: interface \"%s\", address #%d skipped, is on loopback interface.\n",
							     hostname,interfaces[i_idx]->name,a_idx));
					    
					    continue;
					}
				    
					if (interfaces[i_idx]->addr_list[a_idx]->address.sa) {
					    char mybuffer[256];		    
					    const char * s UNUSED_VAROK
						= give_SOCKADDR_ptr_as_string(interfaces[i_idx]->
									      addr_list[a_idx]->address,
									      interfaces[i_idx]->
									      addr_list[a_idx]->addrsize,
									      mybuffer,sizeof mybuffer);
					    
					    switch (interfaces[i_idx]->addr_list[a_idx]->address.sa->sa_family) {
					    case AF_INET:
						DPRINT(Debug,7,(&Debug, 
								"lookup_service_addresses2: %s: interface \"%s\": have IPv4 address: %s\n",
								hostname,interfaces[i_idx]->name, s ? s : "???"));
						seen_ipv4++;
						break;
#ifdef HAVE_IN6
					    case AF_INET6:
#ifdef HAVE_SCOPE
						if (interfaces[i_idx]->addr_list[a_idx]->address.sin6->sin6_scope_id) {
						    DPRINT(Debug,7,(&Debug, 
								    "lookup_service_addresses2: %s: interface \"%s\": IPv6 address with non-zero scope id ignored: %s\n",
								    hostname,interfaces[i_idx]->name, s ? s : "???"));
						    break;
						}
#endif
						DPRINT(Debug,7,(&Debug, 
								"lookup_service_addresses2: %s: interface \"%s\": have IPv6 address: %s\n",
								hostname,interfaces[i_idx]->name, s ? s : "???"));
						seen_ipv6++;
						break;
#endif
					    default:
						DPRINT(Debug,12,(&Debug,
								 "lookup_service_addresses2: %s: interface \"%s\", address #%d have family %d\n",
								 hostname,interfaces[i_idx]->name,a_idx,
								 interfaces[i_idx]->addr_list[a_idx]->address.sa->sa_family));
						break;
					    }			
					} else {
					    DPRINT(Debug,12,(&Debug,
							     "lookup_service_addresses2: %s: interface \"%s\", no address #%d set\n",
							     hostname,interfaces[i_idx]->name,a_idx));
					    
					}
				    } else {
					DPRINT(Debug,12,(&Debug,
							 "lookup_service_addresses2: %s: interface \"%s\", no address #%d info\n",
							 hostname,interfaces[i_idx]->name,a_idx));
				    }
				}
			    } else {
				DPRINT(Debug,12,(&Debug, 
						 "lookup_service_addresses2: %s: interface \"%s\": no addresses\n",
						 hostname,interfaces[i_idx]->name));
			    }
			}
		    }
		    
		    if (seen_ipv4) {
			DPRINT(Debug,7,(&Debug, 
					"lookup_service_addresses2: %s: seen IPv4, querying IPv4 addresses\n",
					hostname));
			query_list[query_list_len++] = ns_t_a;
		    }
		    
		    if (seen_ipv6) {
			DPRINT(Debug,7,(&Debug, 
					"lookup_service_addresses2: %s: seen IPv6, querying IPv6 addresses\n",
					hostname));
			query_list[query_list_len++] = ns_t_aaaa;
		    }
		    
		    if (0 == query_list_len) {
			DPRINT(Debug,7,(&Debug, 
					"lookup_service_addresses2: %s: no addresses, querying both IPv4 and IPv6 addresses\n",
					hostname));
			
			query_list[query_list_len++] = ns_t_a;
			query_list[query_list_len++] = ns_t_aaaa;
		    }
		    
		} else {
		    DPRINT(Debug,7,(&Debug, 
				    "lookup_service_addresses2: %s: no interfaces, querying both IPv4 and IPv6 addresses\n",
				    hostname));
		    
		    query_list[query_list_len++] = ns_t_a;
		    query_list[query_list_len++] = ns_t_aaaa;		
		}
	    }
		break;
	    case NUM_query_address:  /* bad value */
		break;
	    }
	}

	break;
    case NUM_resolv_mserv_lookup:  /* bad value */
	break;
    }
    
    if (query_list_len > sizeof (query_list) / sizeof (query_list[0]))
	panic("RESOLV PANIC",__FILE__,__LINE__,"lookup_service_addresses2",
	      "overflow",0);
    if (query_list_len <= 0) {
	DPRINT(Debug,7,(&Debug, "lookup_service_addresses2: %s: no query\n",
			hostname));
	return al_status_no_routine;
    }


    if (need_interface_change) {
	int query_flags = INTERFACE_ADDRLIST | INTERFACE_CHANGE;
	
	int interfaces_count = 0;
	struct interface_info ** interfaces UNUSED_VAROK = 
	    get_interfaces(sizeof (struct interface_info),
			   sizeof (struct interface_addr),
			   &interfaces_count,&query_flags);


	if (0 == (query_flags & INTERFACE_CHANGE)) {
	    DPRINT(Debug,7,(&Debug, "lookup_service_addresses2: %s: Interfaces changed\n",
			    hostname));
	}
	    
	need_interface_change = 0;
	    
	if (0 == (query_flags & INTERFACE_ADDRLIST)) {
	    DPRINT(Debug,7,(&Debug, "lookup_service_addresses2: %s: failed to get interface address list\n",
			    hostname));
	}	
    }
    
    for (a = 0; a < query_list_len;  a++) {
	ns_type q_type = query_list[a];
	
	struct resolv_cache  * search = NULL;
	char                 * search_name = NULL;
	struct resolv_cache  * cache  = NULL;
	struct resolv_result ** result = NULL;
	int found = 0;
	int ask_next = 0;
	
	other = 0;

	DPRINT(Debug,7,(&Debug, "lookup_service_addresses2: %s: %s #%d: ",
			hostname,have_search ? "search" : "query",a));

	switch(q_type) {
	case ns_t_a:    DPRINT(Debug,7,(&Debug, "A   : ")); break;
	case ns_t_aaaa: DPRINT(Debug,7,(&Debug, "AAAA: ")); break;
	default:
	case ns_t_invalid:  /* bad value */                 break;
	}
	DPRINT(Debug,7,(&Debug, "\n"));

	/* If answer expires during query, it may be removed from
	   struct resolv_cache  * cache (on some cases) but it is still on
	   struct resolv_result ** result
	*/
	
	cache = query_resolv_cache(hostname,q_type,
				   have_search ? &search : NULL,
				   &search_name,
				   &result,&looking_up,&found,
				   &have_error,
				   &other,
				   now,query_mode,
				   cancel_p);
	if (cache) {

	    if (found && result) {
		int i;
		
		DPRINT(Debug,7,(&Debug, 
				"lookup_service_addresses2: %s: found %d records\n",
				hostname,found));

		if (rewrite && search_name) {
		    if (*rewrite)
			free(*rewrite);
		    *rewrite = safe_strdup(search_name);
		}

		for (i = 0; result[i]; i++) {
		    if (RESOLV_RESULT_magic != result[i]->magic)
			panic("RESOLV PANIC",__FILE__,__LINE__,
			      "lookup_service_addresses2",
			      "Bad magic number (resolv_result)",0);
		    
		    switch(result[i]->rectype) {
		    case ns_t_cname:
			if (result[i]->data.domain_name) {
			    struct service_entry_name N;

			    if (NAME_RESULT_magic !=
				result[i]->data.domain_name->magic)
				panic("RESOLV PANIC",__FILE__,__LINE__,
				      "lookup_service_addresses2",
				      "Bad magic number (name_result)",0);
			    
			    DPRINT(Debug,7,(&Debug, 
					    "lookup_service_addresses2: %s is alias of %s\n",
					    result[i]->name,
					    result[i]->data.
					    domain_name->name));

			    /* bzero is defined on hdrs/defs.h */
			    bzero((void *)&N,sizeof N);
			    N.aliasname   = result[i]->data.domain_name->name;
			    N.valid_until = result[i]->valid_until;

			    l_append_name_list(aliases_list,
					       aliases_count,
					       &N,
					       1 /* malloc copy */);


			    if (schedule_shorten_next_timeout(&max_valid_until,
							      & (result[i]->valid_until))) {

				char * A = schedule_timeout_string(&max_valid_until);
				if (A) {
				    DPRINT(Debug,7,(&Debug, 
						    "lookup_service_addresses2: %s: => max valid until %s\n",
						    hostname,A));
				    free(A);
				}				   				   
			    }
			}
			break;
			
		    case ns_t_a: {
			struct service_entry_addr A;
			struct sockaddr_in        sin;

			DPRINT(Debug,7,(&Debug, 
					"lookup_service_addresses2: %s have IPv4 address",
					result[i]->name));
#ifdef USE_INET_XTOX_PN  
			{
			    char address[256];
			    const char *addr = 
				inet_ntop(AF_INET,& (result[i]->data.ip4addr),
					  address,sizeof address);
			    if (addr) {
				DPRINT(Debug,7,(&Debug," %s",addr));
			    }
			}			
#endif			
			DPRINT(Debug,7,(&Debug, "\n"));
			
			/* bzero is defined on hdrs/defs.h */
			bzero((void *)&A,sizeof A);
			bzero((void *)&sin,sizeof sin);

			A.addrsize    = sizeof sin;
			A.address.sin = &sin;
			A.valid_until = result[i]->valid_until;

			if (schedule_shorten_next_timeout(& (A.valid_until),
							  &  max_valid_until)) {
			    char * X  UNUSED_VAROK = schedule_timeout_string(&  (A.valid_until));
			    if (X) {
				DPRINT(Debug,7,(&Debug, 
						"lookup_service_addresses2: %s: => shortening valid until %s\n",
						hostname,X));
				free(X);
			    }
			} else {
			    char * X  UNUSED_VAROK = schedule_timeout_string(&  (A.valid_until));
			    if (X) {
				DPRINT(Debug,7,(&Debug, 
						"lookup_service_addresses2: %s: => valid until %s\n",
						hostname,X));
				free(X);
			    }
			}
			
			A.hostname    = result[i]->name;

			sin.sin_addr  = result[i]->data.ip4addr;
			sin.sin_family = AF_INET;
			sin.sin_port = htons(PORT_end);    /* No port given */

			l_append_addr_list(addr_list,addr_count,
					   &A,
					   1 /* malloc copy */);
		    }
			break;
			
#ifdef HAVE_IN6
		    case ns_t_aaaa: 			
			if (result[i]->data.ip6addr) {
			    struct service_entry_addr A;
			    struct sockaddr_in6       sin6;

			    DPRINT(Debug,7,(&Debug, 
					    "lookup_service_addresses2: %s have IPv6 address",
					    result[i]->name));

#ifdef USE_INET_XTOX_PN  
			    {
				char address[256];
				const char *addr = 
				    addr = inet_ntop(AF_INET6,result[i]->data.ip6addr,
						     address,sizeof address);
				if (addr) {
				    DPRINT(Debug,7,(&Debug," %s",addr));
				}
			    }
#endif
			    DPRINT(Debug,7,(&Debug, "\n"));
			    
			    /* bzero is defined on hdrs/defs.h */
			    bzero((void *)&A,sizeof A);
			    bzero((void *)&sin6,sizeof sin6);

			    A.addrsize    = sizeof sin6;
			    A.address.sin6 = &sin6;
			    A.valid_until = result[i]->valid_until;

			    if (schedule_shorten_next_timeout(& (A.valid_until),
							      &  max_valid_until)) {
				char * X  UNUSED_VAROK = schedule_timeout_string(&  (A.valid_until));
				if (X) {
				    DPRINT(Debug,7,(&Debug, 
						    "lookup_service_addresses2: %s: => shortening valid until %s\n",
						    hostname,X));
				    free(X);
				}
			    } else {
				char * X  UNUSED_VAROK = schedule_timeout_string(&  (A.valid_until));
				if (X) {
				    DPRINT(Debug,7,(&Debug, 
						    "lookup_service_addresses2: %s: => valid until %s\n",
						    hostname,X));
				    free(X);
				}
			    }
			    
			    A.hostname    = result[i]->name;

			    sin6.sin6_addr   = *(result[i]->data.ip6addr);
			    sin6.sin6_family = AF_INET6;
			    sin6.sin6_port = htons(PORT_end);  /* No port given */
			    /* sin6.sin6_flowinfo  left to zero
			       sin6.sin6_scope_id  left to zero
			    */

			    l_append_addr_list(addr_list,addr_count,
					       &A,
					       1 /* malloc copy */);
			}
			break;
#endif
			
		    default:   /* ? */
			break;
		    }		    
		}

		if (al_status_no_routine == saved_ret)
		    saved_ret =  al_status_done;

		if (mserv_all_addresses == mode)
		    ask_next = 1;
		else if (al_status_no_routine == ret)
		    ret = al_status_done;
	    }
	    
	    free_resolv_cache(&cache);
	}
	
	if (result) {
	    int i;
	    
	    for (i = 0; result[i]; i++)
		free_resolv_result(& result[i]);
	    free(result);
	    result = NULL;
	}
	
	if (search)
	    free_resolv_cache(&search); 
	if (search_name) {
	    free(search_name);
	    search_name = NULL;
	}
	
	if (have_error && cancel_p && *cancel_p &&
	    is_canceled(*cancel_p)) {
	    
	    DPRINT(Debug,7,(&Debug, "lookup_service_addresses2: %s: DNS lookup canceled\n",
			    hostname));
	    
	    ret = al_status_failed;
	    goto fail;
	}
	
	if (al_status_no_routine != ret)
	    break;
	
	if (!other && !ask_next) {
	    DPRINT(Debug,7,(&Debug, 
			    "lookup_service_addresses2: %s - no other data\n",
			    hostname));
	    
	    break;
	}	
    }
    
    if (al_status_no_routine != saved_ret) {
	ret = saved_ret;
    }

    if (al_status_no_routine == ret) {
	DPRINT(Debug,7,(&Debug, "lookup_service_addresses2: %s - no result",
			hostname));
	
	if (try_static_hosts && ! static_hosts_tried && dot) {
	    enum etc_hosts_status st ;
	    
	    DPRINT(Debug,7,(&Debug, 
			    "; trying %s from static hosts last.\n",
			    hostname));

	    static_hosts_tried = hostname;
	    st = lookup_static_host(hostname,now, default_valid_until,
				    lc_qualified, rewrite,
				    addr_list,addr_count,
				    aliases_list,aliases_count,
				    was_error_p,cancel_p);
	    
	    DPRINT(Debug,7,(&Debug, "lookup_service_addresses2: %s static hosts result %d",
			    hostname,st));
	    
	    switch (st) {
	    case etc_hosts_maybe:   DPRINT(Debug,7,(&Debug, " etc_hosts_maybe")); break;
	    case etc_hosts_failed:  DPRINT(Debug,7,(&Debug, " etc_hosts_failed\n"));
		if (cancel_p && *cancel_p &&
		    is_canceled(*cancel_p)) {
		    
		    DPRINT(Debug,7,(&Debug, "lookup_service_addresses2: %s: lookup canceled\n",
				    hostname));
		    
		    ret = al_status_failed;
		    goto fail;
		}
		DPRINT(Debug,7,(&Debug, "lookup_service_addresses2: "));
		break;
	    case etc_hosts_not_found: DPRINT(Debug,7,(&Debug, " etc_hosts_not_found")); break;
	    case etc_hosts_found:     DPRINT(Debug,7,(&Debug, " etc_hosts_found\n"));
		ret = al_status_done;
		goto done;
	    case etc_hosts_parsed:    DPRINT(Debug,7,(&Debug, " etc_hosts_parsed")); break;
	    case etc_hosts_found_cached: DPRINT(Debug,7,(&Debug, " etc_hosts_found_cached\n"));
		ret = al_status_done;
		goto done;
	    }
	    DPRINT(Debug,7,(&Debug, " -- returning original error"));       	      
	}

	if (have_error) {
	    ret = al_status_failed;

	    DPRINT(Debug,7,(&Debug, ", failure (%d errors)\n",
			    have_error));
	} else {
	    ret = al_status_not_found;
	    DPRINT(Debug,7,(&Debug, ", not found\n"));
	}
    }
    
    switch (looking_up) {
    case  l_up_none: break;
    case  l_up_printed:
    case  l_up_truncated_retry:

	if (was_error_p) {
	    switch(ret) {
	    case al_status_no_routine:
	    case al_status_fallback:
	    case al_status_bad_syntax:
	    case al_status_failed:
		lib_error(CATGETS(elm_msg_cat, 
				  ResolvSet,
				  ResolvLookingUpSomeFailure,
				  "Looking up %s: some failure?"),
			  hostname);
		*was_error_p = 1;
		break;
	    case al_status_not_found:
		if (other && query_list_len == 1 &&
		    ns_t_a == query_list[0]) {
		    lib_error(CATGETS(elm_msg_cat, 
				      ResolvSet,
				      ResolvLookingNoIPv4,
				      "Looking up %s: no IPv4 address"),
			      hostname);
		} else if (other && query_list_len == 1 &&
			   ns_t_aaaa == query_list[0]) {
		    lib_error(CATGETS(elm_msg_cat, 
				      ResolvSet,
				      ResolvLookingNoIPv6,
				      "Looking up %s: no IPv6 address"),
			      hostname);
		} else if (other) 
		    lib_error(CATGETS(elm_msg_cat, 
				      ResolvSet,
				      ResolvLookNoData,
				      "Looking up %s: no data"),
			      hostname);
		else
		    lib_error(CATGETS(elm_msg_cat, 
				      ResolvSet,
				      ResolvLookNotFound,
				      "Looking up %s: not found"),
			      hostname);
		*was_error_p = 1;
		break;
	    case al_status_done:
		lib_transient(CATGETS(elm_msg_cat, 
				      ResolvSet,
				      ResolvLookOK,
				      "Looking up %s: OK"),
			      hostname);
		break;
	    }
	} else {
	    lib_transient(FRM(""));
	}
	break;
    case l_up_error:
    case l_up_stalled:
	if (was_error_p)
	    *was_error_p = 1;	
	break;	
    }

 done:
 fail:
    
    DPRINT(Debug,7,(&Debug, "lookup_service_addresses2=%d",
		    ret));

    switch (ret) {
    case al_status_no_routine: DPRINT(Debug,7,(&Debug, " no routine")); break;
    case al_status_fallback:   DPRINT(Debug,7,(&Debug, " fallback")); break;
    case al_status_failed:     DPRINT(Debug,7,(&Debug, " failed")); break;
    case al_status_not_found:  DPRINT(Debug,7,(&Debug, " not found")); break;
    case al_status_done:       DPRINT(Debug,7,(&Debug, " done")); break;
    case al_status_bad_syntax: DPRINT(Debug,7,(&Debug, " bad syntax")); break;
    }

    if (rewrite && *rewrite) {
	DPRINT(Debug,7,(&Debug, ", rewrite=%s",
			*rewrite));
    }
    
    if (*addr_count > old_addr_count) {
	DPRINT(Debug,7,(&Debug, ", found %d address(es)",
			*addr_count - old_addr_count));
    }

    if (*aliases_count > old_aliases_count) {
	DPRINT(Debug,7,(&Debug, ", found %d alias(es)",
			*aliases_count - old_aliases_count));
    }

    DPRINT(Debug,7,(&Debug, ", *addr_count=%d, *aliases_count=%d\n",
		    *addr_count,
		    *aliases_count));
    
    return ret;
}



#endif

/* Check only existence of domain name */

enum domain_check_query { dlq_any, dlq_txt, dlq_mx, dlq_a, dlq_aaaa,
			  NUM_domain_check_query };

static char * DOMAIN_CHECK_QUERY[NUM_domain_check_query+1] = { "ANY", "TXT","MX","A","AAAA",
							       NULL };

static ENUMERATE domain_check_query = {
    dlq_any,
    NUM_domain_check_query, &(DOMAIN_CHECK_QUERY[0]),
    NULL,
    0		/* Not boolean */,
    NULL	/* not delayed */,
    NULL
};

E_(check_domain_name_f check_domain_name)
enum domname_check_status check_domain_name
   P_((const char                      * domainname,
       char                           ** rewrite,
       const struct schedule_timelimit * now,
       size_t                            schedule_timelimit_size,
       const struct schedule_timelimit * default_valid_until /* for other data */,
       struct schedule_timelimit       * valid_until,
       int                             * was_error_p,
       struct cancel_data             ** cancel_p));   
enum domname_check_status check_domain_name (domainname,rewrite,now,schedule_timelimit_size,
					     default_valid_until,valid_until,
					     was_error_p,cancel_p)
     const char                      * domainname;
     char                           ** rewrite;
     const struct schedule_timelimit * now;
     size_t                            schedule_timelimit_size;
     const struct schedule_timelimit * default_valid_until /* for other data */;
     struct schedule_timelimit       * valid_until;
     int                             * was_error_p;
     struct cancel_data             ** cancel_p;
{

    enum looking_up            looking_up          = l_up_none;
    enum domain_check_query    dcq                 =
	give_dt_enumerate_as_int(&domain_check_query);
    enum domname_check_status ret = dc_status_no_routine;
    int have_error = 0;
    int other = 0;
    int set_valid_until = 1;
    
    ns_type q_type = ns_t_any;
    char                 * search_name = NULL;
    struct resolv_cache  * cache  = NULL;
    struct resolv_result ** result = NULL;
    int found = 0;
    const char *dot;
    const char *is_query_blacklist;

    int                        need_interface_change =
	interface_change.val && !pending_interface_change;
    
    if (schedule_timelimit_size != sizeof (* now)) {
	DPRINT(Debug,1,(&Debug,
			"check_domain_name: Bad (schedule_timelimit_size %d (should be %d)\n",
			schedule_timelimit_size,
			sizeof (* now)));
	return dc_status_no_routine;
    }

    dot = strchr(domainname,'.');
    /* First char can not be '.' */
    if (&(domainname[0]) == dot) {
	DPRINT(Debug,7,(&Debug, "check_domain_name: %s: bad domain name\n",
			domainname));
	return dc_status_bad_syntax;
    }

    is_query_blacklist =  is_query_fallback_domain(domainname);
    if (is_query_blacklist) {
	DPRINT(Debug,7,(&Debug,
			"check_domain_name: %s: Query blacklisted for %s\n",
			domainname,is_query_blacklist));
	return dc_status_query_blacklisted;
    }

    if (query_mode >= 0 && query_mode < NUM_query_mode && query_modes[query_mode]) {
	if (RESOLV_QUERY_MODE_magic != query_modes[query_mode]->magic)
	    panic("RESOLV PANIC",__FILE__,__LINE__,
		  "check_domain_name",
		  "Bad magic number (resolv_query_mode)",0);

	if (! query_modes[query_mode]->have_query_mode()) {
	    lib_error(CATGETS(elm_msg_cat, 
			      ResolvSet,
			      ResolvQMNotAvailable,
			      "Resolv query-mode = %s is not available"),
		      QUERY_MODE[query_mode]);
	    return dc_status_no_routine;
	}
    } else {
	DPRINT(Debug,7,(&Debug, 
			"check_domain_name: %s: query-mode = %d\n",
			domainname,query_mode));
	return dc_status_no_routine;
    }

    switch (dcq) {
    case dlq_any:	q_type = ns_t_any;	break;
    case dlq_txt:	q_type = ns_t_txt;      break;
    case dlq_mx:	q_type = ns_t_mx;	break;
    case dlq_a:		q_type = ns_t_a;	break;	
    case dlq_aaaa:
#ifdef HAVE_IN6
	q_type = ns_t_aaaa;
#endif
	break;
    case NUM_domain_check_query:  /* not used */ break;
    }

    if (need_interface_change) {
	int query_flags = INTERFACE_ADDRLIST | INTERFACE_CHANGE;
	
	int interfaces_count = 0;
	struct interface_info ** interfaces UNUSED_VAROK = 
	    get_interfaces(sizeof (struct interface_info),
			   sizeof (struct interface_addr),
			   &interfaces_count,&query_flags);


	if (0 == (query_flags & INTERFACE_CHANGE)) {
	    DPRINT(Debug,7,(&Debug, "check_domain_name: %s: Interfaces changed\n",
			    domainname));
	}
	    
	need_interface_change = 0;
	    
	if (0 == (query_flags & INTERFACE_ADDRLIST)) {
	    DPRINT(Debug,7,(&Debug, "check_domain_name: %s: failed to get interface address list\n",
			    domainname));
	}	
    }
    
    DPRINT(Debug,7,(&Debug, "check_domain_name: %s: query ",
		    domainname));
    switch(q_type) {
    case ns_t_a:    DPRINT(Debug,7,(&Debug, "A   : ")); break;
    case ns_t_aaaa: DPRINT(Debug,7,(&Debug, "AAAA: ")); break;
    case ns_t_txt:  DPRINT(Debug,7,(&Debug, "TXT : ")); break;
    case ns_t_mx:   DPRINT(Debug,7,(&Debug, "MX  : ")); break;
    case ns_t_any:  DPRINT(Debug,7,(&Debug, "ANY : ")); break;
    default:
    case ns_t_invalid:  /* bad value */                 break;
    }
    DPRINT(Debug,7,(&Debug, "\n"));

    /* If answer expires during query, it may be removed from
       struct resolv_cache  * cache (on some cases) but it is still on
       struct resolv_result ** result
    */
    
    cache = query_resolv_cache(domainname,q_type,
			       NULL,
			       &search_name,
			       &result,&looking_up,&found,
			       &have_error,
			       &other,
			       now,query_mode,
			       cancel_p);

    if (cache) {
	
	if (found && result) {
	    int i;
	    
	    DPRINT(Debug,7,(&Debug, 
			    "check_domain_name: %s: found %d records\n",
			    domainname,found));
	    
	    if (rewrite && search_name) {
		if (*rewrite)
		    free(*rewrite);
		*rewrite = safe_strdup(search_name);
	    }

	    ret = dc_status_have_name;

	    for (i = 0; result[i]; i++) {
		if (RESOLV_RESULT_magic != result[i]->magic)
		    panic("RESOLV PANIC",__FILE__,__LINE__,
			  "check_domain_name",
			  "Bad magic number (resolv_result)",0);

	       
		if (valid_until) {
		    char * A = NULL;
		    
		    if (set_valid_until) {
			* valid_until = result[i]->valid_until;
			A = schedule_timeout_string(valid_until);
			set_valid_until = 0;

		    } else if (schedule_extend_valid_until(valid_until,
							   &(result[i]->valid_until))) {
			A = schedule_timeout_string(valid_until);
		    }
		    
		    if (A) {
			DPRINT(Debug,7,(&Debug, 
					"check_domain_name: %s: => valid until %s\n",
					domainname,A));
		    free(A);
		    }	
		}
	    }

	}
	
	free_resolv_cache(&cache);
    }

    if (result) {
	int i;
	
	for (i = 0; result[i]; i++)
	    free_resolv_result(& result[i]);
	free(result);
	result = NULL;
    }
    
    if (search_name) {
	free(search_name);
	search_name = NULL;
    }
    
    if (have_error && cancel_p && *cancel_p &&
	is_canceled(*cancel_p)) {
	
	DPRINT(Debug,7,(&Debug, "check_domain_name: %s: DNS lookup canceled\n",
			domainname));
	
	ret = dc_status_failure;
	goto fail;
    }

    if (other) {
	DPRINT(Debug,7,(&Debug, 
			"check_domain_name: %s - have other data\n",
			domainname));
	ret = dc_status_have_name;

	if (default_valid_until) {
	    
	    if (set_valid_until) {
		char * A = NULL;
		
		* valid_until =  * default_valid_until;
		set_valid_until = 0;
		
		A  = schedule_timeout_string(valid_until);
		if (A) {
		    DPRINT(Debug,7,(&Debug, 
				    "check_domain_name: %s: => valid until %s\n",
				    domainname,A));
		    free(A);
		}
	    }
	}	
    }
    
    if (dc_status_no_routine == ret) {

	DPRINT(Debug,7,(&Debug, 
			"check_domain_name: %s no result",
			domainname));

	if (have_error) {
	    ret = dc_status_failure;
	    DPRINT(Debug,7,(&Debug, ", failure (%d errors)",
			    have_error));
	} else {
	    ret = dc_status_not_found;
	    DPRINT(Debug,7,(&Debug, ", not found"));
	}

	if (default_valid_until) {
	    
	    if (set_valid_until) {
		char * A = NULL;
		
		* valid_until =  * default_valid_until;
		set_valid_until = 0;
		
		A  = schedule_timeout_string(valid_until);
		if (A) {
		    DPRINT(Debug,7,(&Debug, 
				    ", (default) valid until %s",
				    A));
		    free(A);
		}
	    }
	}	
	    
	DPRINT(Debug,7,(&Debug, "\n"));	
    }

    if (valid_until) {

	if (set_valid_until) {
	    char * A = NULL;
	    
	    * valid_until = NO_schedule_timelimit;
	    A  = schedule_timeout_string(valid_until);
	    
	    if (A) {
		DPRINT(Debug,7,(&Debug, 
				"check_domain_name: %s: => valid until %s\n",
				domainname,A));
		free(A);
	    }
	}
    }

    
    switch (looking_up) {
    case  l_up_none: break;
    case  l_up_printed:
    case  l_up_truncated_retry:
	
	if (was_error_p) {
	    switch (ret) {
	    case dc_status_query_blacklisted:
	    case dc_status_failure:
		
		lib_error(CATGETS(elm_msg_cat, 
				  ResolvSet,
				  ResolvLookingUpSomeFailure,
				  "Looking up %s: some failure?"),
			  hostname);
		*was_error_p = 1;
		break;
	    case dc_status_not_found:
		lib_error(CATGETS(elm_msg_cat, 
				  ResolvSet,
				  ResolvLookNotFound,
				  "Looking up %s: not found"),
			  hostname);
		*was_error_p = 1;
	    case dc_status_have_name:
		lib_transient(CATGETS(elm_msg_cat, 
				      ResolvSet,
				      ResolvLookOK,
				      "Looking up %s: OK"),
			      hostname);
		break;
	    case dc_status_no_routine:
	    case dc_status_bad_syntax:
		break;
	    }

	} else {
	    lib_transient(FRM(""));
	}
	break;
    case l_up_error:
    case l_up_stalled:
	if (was_error_p)
	    *was_error_p = 1;	
	break;	
    }

 fail:

    DPRINT(Debug,7,(&Debug, 
		    "check_domain_name=%d",
		    ret));
    switch(ret) {
    case dc_status_no_routine:        DPRINT(Debug,7,(&Debug," dc_status_no_routine")); break;
    case dc_status_query_blacklisted: DPRINT(Debug,7,(&Debug," dc_status_query_blacklisted"));
	break;
    case dc_status_failure:           DPRINT(Debug,7,(&Debug," dc_status_failure"));    break;
    case dc_status_not_found:         DPRINT(Debug,7,(&Debug," dc_status_not_found"));  break;
    case dc_status_have_name:         DPRINT(Debug,7,(&Debug," dc_status_have_name"));  break;
    case dc_status_bad_syntax:        DPRINT(Debug,7,(&Debug," dc_status_bad_syntax")); break;
    }
    DPRINT(Debug,7,(&Debug, "\n"));
    
    return ret;
}

       

/* Do not use resolver these names, but use fallback instead 

   If name is also on "special-use-domains-blacklist", these
   routines are not called
*/

static char * FALLBACK_DOMAINS[] = {

/* https://www.iana.org/assignments/special-use-domain-names/special-use-domain-names.xhtml
   Special-Use Domain Names

   List domains which should not passed to resolver (use local api instead)

   If domain name is given as ".domain" then it blacklists names under that
   domain. If name is given as "domain" then also "domain" itself (and 
   names under it) are blacklisted. 

*/

    /* "example"            [RFC6761] ; Name resolution APIs and 
       libraries SHOULD NOT recognize example names as special and 
       SHOULD NOT treat them differently.  Name resolution APIs SHOULD 
       send queries for example names to their
       configured caching DNS server(s). */
    /* "example.com"         [RFC6761] ;  -"- */
    /* "example.net"         [RFC6761] ;  -"- */
    /* "example.org"         [RFC6761] ;  -"- */

    "invalid"            /*  [RFC6761] ; Name resolution APIs and 
	libraries SHOULD recognize "invalid" names as special and 
	SHOULD always return immediate negative responses.  Name 
	resolution APIs SHOULD NOT send queries for "invalid" names 
	to their configured caching DNS server(s). */,
    ".local"             /*  [RFC6762] ; Name resolution APIs and 
	libraries SHOULD recognize these names as special and SHOULD 
	NOT send queries for these names to their configured (unicast) 
	caching DNS server(s).  */,
    "localhost"           /* [RFC6761] ; Name resolution APIs and 
       libraries SHOULD recognize localhost names as special and SHOULD 
       always return the IP loopback address for address queries and 
       negative responses for all other query types.  Name resolution 
       APIs SHOULD NOT send queries for localhost names to their 
       configured caching DNS server(s). */,
    ".onion"            /*   [RFC7686] ; Resolvers MUST either respond
       to requests for .onion names by resolving them according to
       [tor-rendezvous] or by responding with NXDOMAIN [RFC1035]. */,
     /* "test"                [RFC6761] ; Name resolution APIs and 
       libraries SHOULD NOT recognize test names as special and SHOULD 
       NOT treat them differently. */
    NULL
};

#define NUM_fallback_domains (sizeof (FALLBACK_DOMAINS) / \
			      sizeof(FALLBACK_DOMAINS[0]))-1

PATH query_bl_domains = {  /* "query-domain-blacklist" */
    PATH_ascii_only|PATH_initial,NULL,
    NUM_fallback_domains,
    &(FALLBACK_DOMAINS[0])
};


static ZZZ_SAVE_TYPE save_info_data[] = {
    { "interface-change", ZZZ_DT_FLAGS(&interface_change),     ZZZ_TAIL },
    { "query-address-type", ZZZ_DT_ENUM(&query_address),       ZZZ_TAIL },
    { "query-domain-blacklist",ZZZ_DT_PATH(&query_bl_domains), ZZZ_TAIL },
    { "query-mode", ZZZ_DT_FUNC(query_mode_func),              ZZZ_TAIL },
    { "search-mail-domain",ZZZ_DT_ENUM(&search_mdomain),       ZZZ_TAIL },
    { "search-mail-service", ZZZ_DT_ENUM(&search_mservice),    ZZZ_TAIL },
};

E_(provides_RC_options_f provides_RC_options2)
struct rc_save_info_rec * provides_RC_options2 
  P_((size_t *count, size_t *s_size));
struct rc_save_info_rec * provides_RC_options2(count,s_size)
     size_t *count; 
     size_t *s_size;
{
    DPRINT(Debug,7,(&Debug, "provides_RC_options2\n"));

    *s_size = sizeof (save_info_data[0]);

    *count = (sizeof save_info_data) / *s_size;
    return (struct rc_save_info_rec *) save_info_data;
}

static void free_cache_items()
{
    if (cache_items) {
	free_sort_list(& cache_items);
	
    }
}


E_(free_shared_cache_f free_shared_cache)
void free_shared_cache P_((void));
void free_shared_cache() 
{

    DPRINT(Debug,7,(&Debug, "free_shared_cache\n"));

    free_non_blocking_qm();
    free_resolv_config();

    free_cache_items();

    if (cleanup_resolv_handle) {
	DPRINT(Debug,9,(&Debug,
			"free_shared_cache: freeing resolv cleanup handle %p\n",
			cleanup_resolv_handle));

	alter_timed_task(&cleanup_resolv_handle,NULL,
			 NO_timed_task_free,
			 NO_inc_timed_task_refcount,
			 NO_timed_task_action,
			 &next_resolv_cleanup);
    }
}


/* Returns domain name if matches to query-domain-blacklist */
const char * is_query_fallback_domain(host_domain_name)
     const char * host_domain_name;
{
    return on_domain_blacklist(&query_bl_domains,
			       "query-domain-blacklist",
			       host_domain_name);
}

E_(RC_post_init_f RC_post_init2)
void RC_post_init2  P_((int *errors, int flag, const char * tag));
void RC_post_init2(errors, flag,tag)
     int *errors;
     int flag;
     const char * tag;
{
    DPRINT(Debug,4,(&Debug,"RC_post_init2 ... (resolv)\n"));

    if (NUM_search_mservice == search_mservice.val) {
	int ok = 0;
	
	switch (query_mode) {
	case qm_blocking:
	    search_mservice.val = sms_fallback_dotless_search_name;
	    ok = 1;
	    break;
	case qm_non_blocking:
	    search_mservice.val = sms_static_dotless_search_name;
	    ok = 1;
	    break;
	case NUM_query_mode:  /* Bad value */

	    DPRINT(Debug,8,(&Debug,
			    "RC_post_init2: query-mode=%d unexpected\n",
			    query_mode));
	    
	    break;
	}

	if (ok) {
	    DPRINT(Debug,8,(&Debug,
			    "RC_post_init2: query-mode=%s, setting search-mail-service=%s\n",
			    QUERY_MODE[query_mode],SEARCH_MSERVICE[search_mservice.val]));
	}
    }
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
