static char rcsid[] = "@(#)$Id: tls.c,v 2.34 2024/06/26 16:54:27 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.34 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "elmtls.h"

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

DEBUG_VAR(Debug,__FILE__,"tls");

#ifdef REMOTE_MBX

static unsigned char *s2us P_((char *str)) UNUSED_FUNCOK;
static unsigned char *s2us(str)
     char *str;
{
    return (unsigned char *)str;
}


/* RAND_ functions require only 20 bytes of randomnes ... 
 * See ENTROPY_NEEDED on openssl library
 *
 * RNG state size is 1023 bytes so more need not read ...
 * See STATE_SIZE on openssl library
 */

#define RAND_BYTES  1024
#define RAND_EGD_B   10

static int init_done       = 0;    /* -1 == failure
                                       1 == OK
				   */

/* Manual page: That's the global context structure which is created
                 by a server or client once per program life-time.
*/
static SSL_CTX *tls_xx_pCtx[NUM_tls_version];

char * TLS_VERSION[NUM_tls_version+1] = {
    "SSL",
    "TLS",
    "TLS 1",
    "TLS 1.1",
    "TLS 1.2",
    NULL
};


#include "rc_imp.h"
#include "save_opts.h"

/* "none" == tls_none = -1, so not in array */
static char * tls_version_values[NUM_tls_version+1] = {
    "ssl",    
    "tls",
    "tls-1",
    "tls-1.1",
    "tls-1.2",
    NULL
};


ENUMERATE default_tls_version = {
    v_tls            /* Default */,
    NUM_tls_version, &(tls_version_values[0]),
    NULL,
    0   	/* no boolean */,
    NULL   	/* not delayed */,
    NULL
};

static ESTR rand_file_e = {                /* Default set by RAND_file_name */
    ESTR_meta|ESTR_none,  NULL,  NULL
};
static ESTR rand_egd_e = {                /* No default value */
    ESTR_meta|ESTR_none,  NULL,  NULL
};
  
  
/* These are parameters of   SSL_CTX_load_verify_locations()               */
  
static ESTR CAfile_e = {                  /* Default set by X509_get_default_cert_file() */
    ESTR_meta|ESTR_none,  NULL,  NULL
};

static ESTR CApath_e = {                  /* Default set by X509_get_default_cert_dir() */
    ESTR_meta|ESTR_none,  NULL,  NULL
};

E_(wants_rand_bits_f wants_rand_bits)
void wants_rand_bits P_((const char *buf, int size, 
			 int entropy_bits));
void wants_rand_bits(buf,size,entropy_bits)
     const char *buf;
     int size;
     int entropy_bits;
{
    double entropy_bytes = entropy_bits / 8.0;

    RAND_add(buf,size,entropy_bytes);

    DPRINT(Debug,10,(&Debug,"RAND_add()ed %d bytes, entropy %.2f bytes\n",
		     size,entropy_bytes));
}

int tls_version_supported(config,tls)
     const char *config;
     enum tls_version tls;
{
    int ret = 0;

    if (tls >= 0 && tls < NUM_tls_version) {
	if (tls_xx_pCtx[tls])
	    ret = 1;
	else {
	    if (config)
		lib_error(CATGETS(elm_msg_cat, TlsSet,
				  TlsXXTlsVersionUnsupported,
				  "%s: %s unsupported"),
			  config,
			  tls_version_values[tls]);
	    else
		lib_error(CATGETS(elm_msg_cat, TlsSet,
				  TlsATlsVersionUnsupported,
				  "A TLS version %s unsupported"),
			  tls_version_values[tls]);
	    
	}
    }
    
    DPRINT(Debug,10,(&Debug,"tls_version_supported=%d",ret));
    if (config) {
	DPRINT(Debug,10,(&Debug," config=%s",config));
    }
    DPRINT(Debug,10,(&Debug," tls=%d",tls));
    switch(tls) {
    case tls_none: DPRINT(Debug,10,(&Debug, " tls_none")); break;
    case v_ssl:    DPRINT(Debug,10,(&Debug, " v_ssl"));    break;
    case v_tls:    DPRINT(Debug,10,(&Debug, " v_tls"));    break;
    case v_tls1:   DPRINT(Debug,10,(&Debug, " v_tls1"));   break;
    case v_tls1_1: DPRINT(Debug,10,(&Debug, " v_tls1_1")); break;
    case v_tls1_2: DPRINT(Debug,10,(&Debug, " v_tls1_2")); break;
    case NUM_tls_version: DPRINT(Debug,10,(&Debug, " NUM_tls_version")); break;
    }

    if (tls >= 0 && tls < NUM_tls_version) {
	DPRINT(Debug,10,(&Debug, " (%s)",
			 TLS_VERSION[tls]));
    }
    DPRINT(Debug,4,(&Debug,"\n"));


    return ret;
}

int tls_init(errors)
     int *errors;
{
    long ssl_error1 = 0;
    int junk = 0;
    int p = 0;

    int bytes = 0;
    const char * rand_file = NULL;
    const char * rand_egd = NULL;

    const char * CAfile = NULL;
    const char * CApath = NULL;
    enum tls_version v;

    /* Default cert file may not be exist, but it is still default value */
    const char * used_default_cert_file = NULL;
    const char * used_default_cert_dir  = NULL;

    int check_rand_file_writable = 1;
    int pCtx_found = 0;
    
    if (!errors)
	errors = &junk;

    if (init_done < 0)
	return 0;            /* Can't initialize */
    if (init_done > 0)
	return 1;            /* Already initialized */

    for (v = 0; v < NUM_tls_version; v++)
	tls_xx_pCtx[v] = NULL;
    
    SSL_load_error_strings(); 
    SSL_library_init();

    /* Seeding with random ? */

    for (v = 0; v < NUM_tls_version; v++) {
	int min_max_failed = 0;
	
	switch (v) {
	case v_ssl:
#ifdef  TLS_HAVE_SSLv23_METHOD
	    tls_xx_pCtx[v] = SSL_CTX_new(SSLv23_client_method());
	    if (tls_xx_pCtx[v]) {
		DPRINT(Debug,2,(&Debug,			    
				"tls: tls_init: %s SSL_CTX_new succeed %p -- SLv23_client_method\n",
				TLS_VERSION[v],tls_xx_pCtx[v]));
		pCtx_found++;
	    } else {
		DPRINT(Debug,2,(&Debug,			    
				"tls: tls_init: %s SSL_CTX_new failed -- SLv23_client_method\n",
				TLS_VERSION[v]));
	    }
#else
#ifdef TLS_HAVE_TLS_METHOD
	    tls_xx_pCtx[v] = SSL_CTX_new(TLS_client_method());
	    if (tls_xx_pCtx[v]) {
		int r;
		DPRINT(Debug,2,(&Debug,			    
				"tls: tls_init: %s SSL_CTX_new succeed %p -- TLS_client_method\n"),
		   TLS_VERSION[v],tls_xx_pCtx[v]);	    
		pCtx_found++;
		
		r = SSL_CTX_set_min_proto_version(tls_xx_pCtx[v],SSL3_VERSION);
		DPRINT(Debug,2,(&Debug,			    
				"tls: tls_init: %s SSL_CTX_set_min_proto_version=%d",
				TLS_VERSION[v],r));
		switch (r) {
		case 1: DPRINT(Debug,2,(&Debug," succeed")); break;
		case 0:
		    DPRINT(Debug,2,(&Debug," failed"));
		    min_max_failed = 1;
		    break;
		}
		DPRINT(Debug,2,(&Debug,"\n"));			    
		
	    } else {
		DPRINT(Debug,2,(&Debug,			    
				"tls: tls_init: %s SSL_CTX_new failed -- TLS_client_method\n",
				TLS_VERSION[v]));
	    }
#else		
	    DPRINT(Debug,2,(&Debug,			    
			    "tls: tls_init: No SLv23_client_method or TLS_client_method\n"));
#endif /* TLS_HAVE_TLS_METHOD */
#endif /* TLS_HAVE_SSLv23_METHOD */
	    break;
	    
	case v_tls:  /* This was just   ---  SSL_CTX_new(TLSv1_client_method()); -- now fall thru v_tls1 */
	    
#ifdef TLS_HAVE_TLS_METHOD
	    tls_xx_pCtx[v] = SSL_CTX_new(TLS_client_method());
	    if (tls_xx_pCtx[v]) {
		int r;
		DPRINT(Debug,2,(&Debug,			    
				"tls: tls_init: %s SSL_CTX_new succeed %p -- TLS_client_method\n",
				TLS_VERSION[v],tls_xx_pCtx[v]));	    
		pCtx_found++;
		
		r = SSL_CTX_set_min_proto_version(tls_xx_pCtx[v],TLS1_VERSION);
		DPRINT(Debug,2,(&Debug,			    
				"tls: tls_init: %s SSL_CTX_set_min_proto_version=%d",
				TLS_VERSION[v],r));
		switch (r) {
		case 1: DPRINT(Debug,2,(&Debug," succeed")); break;
		case 0:
		    DPRINT(Debug,2,(&Debug," failed"));
		    min_max_failed = 1;
		    break;
		}
		DPRINT(Debug,2,(&Debug,"\n"));			    
		
	    } else {
		DPRINT(Debug,2,(&Debug,			    
				"tls: tls_init: %s SSL_CTX_new failed -- TLS_client_method\n",
				TLS_VERSION[v]));		  
	    }
	    break;
#else		
	    DPRINT(Debug,2,(&Debug,			    
			    "tls: tls_init: %s No TLS_client_method -- fallthru v_tls1 aka (TLSv1_client_method)\n",
			    TLS_VERSION[v]));
	    /* FALLTHRU */
#endif
	    
	case v_tls1:
#ifdef TLS_HAVE_TLSv1_METHOD
	    tls_xx_pCtx[v] = SSL_CTX_new(TLSv1_client_method());
	    if (tls_xx_pCtx[v]) {
		DPRINT(Debug,2,(&Debug,			    
				"tls: tls_init: %s SSL_CTX_new succeed %p -- TLSv1_client_method\n",
				TLS_VERSION[v],tls_xx_pCtx[v]));	    
		pCtx_found++;
	    } else {
		DPRINT(Debug,2,(&Debug,			    
				"tls: tls_init: %s SSL_CTX_new failed -- TLSv1_client_method\n",
				TLS_VERSION[v]));
		
	    }
	    break;
#else
	    
#ifdef TLS_HAVE_TLS_METHOD
	    tls_xx_pCtx[v] = SSL_CTX_new(TLS_client_method());
	    if (tls_xx_pCtx[v]) {
		int r;
		DPRINT(Debug,2,(&Debug,			    
				"tls: tls_init: %s SSL_CTX_new succeed %p -- TLS_client_method\n",
				TLS_VERSION[v],tls_xx_pCtx[v]));	    
		pCtx_found++;
		
		r = SSL_CTX_set_min_proto_version(tls_xx_pCtx[v],TLS1_VERSION);
		DPRINT(Debug,2,(&Debug,			    
				"tls: tls_init: %s SSL_CTX_set_min_proto_version=%d",
				TLS_VERSION[v],r));
		switch (r) {
		case 1: DPRINT(Debug,2,(&Debug," succeed")); break;
		case 0:
		    DPRINT(Debug,2,(&Debug," failed"));
		    min_max_failed = 1;
		    break;
		}
		DPRINT(Debug,2,(&Debug,"\n"));			    
		
		r = SSL_CTX_set_max_proto_version(tls_xx_pCtx[v],TLS1_VERSION);
		DPRINT(Debug,2,(&Debug,			    
				"tls: tls_init: %s SSL_CTX_set_max_proto_version=%d",
				TLS_VERSION[v],r));
		switch (r) {
		case 1: DPRINT(Debug,2,(&Debug," succeed")); break;
		case 0:
		    DPRINT(Debug,2,(&Debug," failed"));
		    min_max_failed = 1;
		    break;
		}
		DPRINT(Debug,2,(&Debug,"\n"));			    
		
	    } else {
		DPRINT(Debug,2,(&Debug,			    
			    "tls: tls_init: %s SSL_CTX_new failed -- TLS_client_method\n",
				TLS_VERSION[v]));
	    }
	    break;
#else
	    DPRINT(Debug,2,(&Debug,			    
			    "tls: tls_init: %s No TLSv1_client_method or TLS_client_method",
			    TLS_VERSION[v]));
	    
	    if (v_tls == v) {
		DPRINT(Debug,2,(&Debug, " -- fallthru v_tls1_1 aka (TLSv1_1_client_method)\n"));
	    } else {
		DPRINT(Debug,2,(&Debug, "\n"));
		break;
	    }
	    /* FALLTHRU */
	    
#endif /* TLS_HAVE_TLS_METHOD */
#endif /* TLS_HAVE_TLSv1_METHOD */
	    
	case v_tls1_1:
#ifdef TLS_HAVE_TLSv1_1_METHOD
	    tls_xx_pCtx[v] = SSL_CTX_new(TLSv1_1_client_method());
	    if (tls_xx_pCtx[v]) {
		DPRINT(Debug,2,(&Debug,			    
				"tls: tls_init: %s SSL_CTX_new succeed %p -- TLSv1_1_client_method\n",
				TLS_VERSION[v],tls_xx_pCtx[v]));	    
		pCtx_found++;
	    } else {
		DPRINT(Debug,2,(&Debug,			    
				"tls: tls_init: %s SSL_CTX_new failed -- TLSv1_1_client_method\n",
				TLS_VERSION[v]));
		
	    }
	    break;
#else
#ifdef TLS_HAVE_TLS_METHOD
	    tls_xx_pCtx[v] = SSL_CTX_new(TLS_client_method());
	    if (tls_xx_pCtx[v]) {
		int r;
		DPRINT(Debug,2,(&Debug,			    
				"tls: tls_init: %s SSL_CTX_new succeed %p -- TLS_client_method\n",
				TLS_VERSION[v],tls_xx_pCtx[v]));	    
		pCtx_found++;
		
		r = SSL_CTX_set_min_proto_version(tls_xx_pCtx[v],TLS1_1_VERSION);
		DPRINT(Debug,2,(&Debug,			    
				"tls: tls_init: %s SSL_CTX_set_min_proto_version=%d",
				TLS_VERSION[v],r));
		switch (r) {
		case 1: DPRINT(Debug,2,(&Debug," succeed")); break;
		case 0:
		    DPRINT(Debug,2,(&Debug," failed"));
		    min_max_failed = 1;
		    break;
		}
		DPRINT(Debug,2,(&Debug,"\n"));			    
		
		r = SSL_CTX_set_max_proto_version(tls_xx_pCtx[v],TLS1_1_VERSION);
		DPRINT(Debug,2,(&Debug,			    
				"tls: tls_init: %s SSL_CTX_set_max_proto_version=%d",
				TLS_VERSION[v],r));
		switch (r) {
		case 1: DPRINT(Debug,2,(&Debug," succeed")); break;
		case 0:
		    DPRINT(Debug,2,(&Debug," failed"));
		    min_max_failed = 1;
		    break;
		}
		DPRINT(Debug,2,(&Debug,"\n"));			    
		
	    } else {
		DPRINT(Debug,2,(&Debug,			    
				"tls: tls_init: %s SSL_CTX_new failed -- TLS_client_method\n",
				TLS_VERSION[v]));
		
	    }
	    break;
#else
	    DPRINT(Debug,2,(&Debug,			    
			    "tls: tls_init: %s No TLSv1_1_client_method or TLS_client_method",
			    TLS_VERSION[v]));
	    
	    if (v_tls == v) {
		DPRINT(Debug,2,(&Debug, " -- fallthru v_tls1_2 aka (TLSv1_2_client_method)\n"));
	    } else {
		DPRINT(Debug,2,(&Debug, "\n"));
		break;
	    }
#endif /* TLS_HAVE_TLS_METHOD */
#endif /* TLS_HAVE_TLSv1_1_METHOD */
	    
	    /* FALLTHRU */
	    
	case v_tls1_2:
#ifdef TLS_HAVE_TLSv1_2_METHOD
	    tls_xx_pCtx[v] = SSL_CTX_new(TLSv1_2_client_method());
	    if (tls_xx_pCtx[v]) {
		DPRINT(Debug,2,(&Debug,			    
				"tls: tls_init: %s SSL_CTX_new succeed %p -- TLSv1_2_client_method\n",
				TLS_VERSION[v],tls_xx_pCtx[v]));	    
		pCtx_found++;
	    } else {
		DPRINT(Debug,2,(&Debug,			    
				"tls: tls_init: %s SSL_CTX_new failed -- TLSv1_2_client_method\n",
				TLS_VERSION[v]));
		
	    }
#else
#ifdef TLS_HAVE_TLS_METHOD
	    tls_xx_pCtx[v] = SSL_CTX_new(TLS_client_method());
	    if (tls_xx_pCtx[v]) {
		int r;
		DPRINT(Debug,2,(&Debug,			    
				"tls: tls_init: %s SSL_CTX_new succeed %p -- TLS_client_method\n",
				TLS_VERSION[v],tls_xx_pCtx[v]));	    
		pCtx_found++;
		
		r = SSL_CTX_set_min_proto_version(tls_xx_pCtx[v],TLS1_2_VERSION);
		DPRINT(Debug,2,(&Debug,			    
				"tls: tls_init: %s SSL_CTX_set_min_proto_version=%d",
				TLS_VERSION[v],r));
		switch (r) {
		case 1: DPRINT(Debug,2,(&Debug," succeed")); break;
		case 0:
		    DPRINT(Debug,2,(&Debug," failed"));
		    min_max_failed = 1;
		    break;
		}
		DPRINT(Debug,2,(&Debug,"\n"));			    
		
		r = SSL_CTX_set_max_proto_version(tls_xx_pCtx[v],TLS1_2_VERSION);
		DPRINT(Debug,2,(&Debug,			    
				"tls: tls_init: %s SSL_CTX_set_max_proto_version=%d",
				TLS_VERSION[v],r));
		switch (r) {
		case 1: DPRINT(Debug,2,(&Debug," succeed")); break;
		case 0:
		    DPRINT(Debug,2,(&Debug," failed"));
		    min_max_failed = 1;
		    break;
		}
		DPRINT(Debug,2,(&Debug,"\n"));			    
		
	    } else {
		DPRINT(Debug,2,(&Debug,			    
				"tls: tls_init: %s SSL_CTX_new failed -- TLS_client_method\n",
				TLS_VERSION[v]));
		
	    }
	    
#else
	    DPRINT(Debug,2,(&Debug,			    
			    "tls: tls_init: %s No TLSv1_2_client_method or TLS_client_method\n",
			    TLS_VERSION[v]));
#endif /* TLS_HAVE_TLS_METHOD */ 
#endif /* TLS_HAVE_TLSv1_2_METHOD */
	    
	    break;
	    
	case tls_none:
	case NUM_tls_version:
	    /* Not used */
	    break;
	}

	if (min_max_failed) {
	    DPRINT(Debug,2,(&Debug,			    
			    "tls: tls_init: %s SSL_CTX_set_min_proto_version or SSL_CTX_set_max_proto_version failed",
			    TLS_VERSION[v]));

	    if (tls_xx_pCtx[v]) {
		DPRINT(Debug,2,(&Debug,			    
				", discarding SSL_CTX %p",
				tls_xx_pCtx[v]));
		SSL_CTX_free(tls_xx_pCtx[v]);
		tls_xx_pCtx[v] = NULL;
	    }
	    DPRINT(Debug,2,(&Debug,"\n"));
	}	
    }
    
    if (!pCtx_found) {
	DPRINT(Debug,2,(&Debug,	   
			"tls: tls_init: all SSL_CTX_new failed\n"));
	init_done = -1;
	goto fail;
    } else {
	DPRINT(Debug,10,(&Debug,	   
			"tls: tls_init: %d/%d SSL_CTX_new succeed\n",
			pCtx_found,NUM_tls_version));

    }

#if 0
    n = SSL_CTX_use_certificate_file(pCtx,filename,SSL_FILETYPE_PEM);
    if (n > 0) ok;


    n = SSL_CTX_use_PrivateKey_file(pCtx,filename,SSL_FILETYPE_PEM);
    if (n > 0) ok;

#endif

    if (default_tls_version.val >= 0 && default_tls_version.val < NUM_tls_version &&
	/* Accept default even when it is not valid */
	v_tls != default_tls_version.val
	) {
	
	if (! tls_xx_pCtx[default_tls_version.val]) {
	    enum tls_version s = default_tls_version.val;

	    
	    lib_error(CATGETS(elm_msg_cat, TlsSet,
			      TlsDefTlsVersionUnsupported,
			      "default-tls-version: %s unsupported"),
		      tls_version_values[default_tls_version.val]);
	    
	    errors++;
	    default_tls_version.val = v_tls;
	    
	    for (v = s; v < NUM_tls_version; v++) {
		if (tls_xx_pCtx[v]) {
		    default_tls_version.val = v;
		    break;
		}
	    }
	    
	    /* replace saved value on .elmrc */
	    mark_local_changed(&default_tls_version,0);
	    	    
	}

    }
	
    
    /* If not set, use openSSL default ... */
    if (! rand_file_e.unexpanded) {
	char temp[SLEN];

	if (RAND_file_name(temp,sizeof temp)) {
	    set_dt_estr(&rand_file_e,temp,NULL,0,0);

	    /* Try avoid checking writability of default file 
	     * during installation 
	     *
	     * Also checking writability for root (uid == 0)
	     * is not very useful
	     */

	    check_rand_file_writable = getuid() != 0;
	} else 
	    set_dt_estr(&rand_file_e,"none",NULL,0,0);
    
    } else if (! expand_dt_estr( &rand_file_e,
				 "rand-file")) {
	set_dt_estr(&rand_file_e,"none",NULL,0,0);
	
	/* replace saved value on .elmrc with "none" */
	mark_local_changed(&rand_file_e,0);

	(*errors)++;
    } 
       
    if (!rand_egd_e.unexpanded) {
	set_dt_estr(&rand_egd_e,"none",NULL,0,0);

    } else if (! expand_dt_estr( &rand_egd_e,
				 "rand-egd")) {
	set_dt_estr(&rand_egd_e,"none",NULL,0,0);

	/* replace saved value on .elmrc with "none" */
	mark_local_changed(&rand_egd_e,0);

	(*errors)++;
    } else {

#ifndef TLS_HAVE_EGD
	if (rand_egd_e.unexpanded &&
	    0 != strcmp(rand_egd_e.unexpanded,"none")) {
	    
	    lib_error(CATGETS(elm_msg_cat, TlsSet,
			      TlsRndEgdNotsupported,
			      "rand-egd: %s is not supported"),
		      rand_egd_e.unexpanded);
	    
	    set_dt_estr(&rand_egd_e,"none",NULL,0,0);
	    
	    /* replace saved value on .elmrc with "none" */
	    mark_local_changed(&rand_egd_e,0);

	    (*errors)++;	    
	}	
#endif	
	
    }

    if (! CAfile_e.unexpanded) {
	used_default_cert_file = X509_get_default_cert_file();

	if (used_default_cert_file)
	    set_dt_estr(&CAfile_e,used_default_cert_file ,NULL,0,0);
	else
	    set_dt_estr(&CAfile_e,"none",NULL,0,0);

    } else if (! expand_dt_estr( &CAfile_e,
				 "trusted-ca-certificates-file")) {
	set_dt_estr(&CAfile_e,"none",NULL,0,0);

	/* replace saved value on .elmrc with "none" */
	mark_local_changed(&CAfile_e,0);
	
	(*errors)++;
    } 


    if (! CApath_e.unexpanded) {
	used_default_cert_dir = X509_get_default_cert_dir(); 
	if (used_default_cert_dir)
	    set_dt_estr(&CApath_e,used_default_cert_dir,NULL,0,0);
	else
	    set_dt_estr(&CApath_e,"none",NULL,0,0);

    } else if (! expand_dt_estr( &CApath_e,
				 "trusted-ca-certificates-dir")) {
	set_dt_estr(&CApath_e,"none",NULL,0,0);

	/* replace saved value on .elmrc with "none" */
	mark_local_changed(&CApath_e,0);
	
	(*errors)++;
    } 


    /* ------------------- */
    
    rand_file = give_dt_estr_as_str(&rand_file_e,"rand-file",NULL,NULL);
    if (rand_file) {
	int err,z;
	
	DPRINT(Debug,2,(&Debug,			    
			"tls: rand-file is %s, check_rand_file_writable=%d\n",
			rand_file,check_rand_file_writable));

	err = can_open(rand_file,"r");
	if (0 != err) {

	    if (err != ENOENT) {
		lib_error(CATGETS(elm_msg_cat, TlsSet,
				  TlsRandomFileNotReadable,
				  "TLS random file %s not readable: %s"),
			  rand_file,strerror(err));

		goto fail_rand_file;
	    }

	} else if (0 < (z = RAND_load_file(rand_file,RAND_BYTES)))
	    bytes += z;
		
	/* Check if we can write random bytes back ... */
	if (check_rand_file_writable)
	    err = can_open(rand_file,"w");
	else
	    err = 0;

	if (0 != err) {
	    
	    /* Error message? */
	    lib_error(CATGETS(elm_msg_cat, TlsSet,
			      TlsRandomFileNotWritable,
			      "TLS random file %s not writable: %s"),
		      rand_file,strerror(err));

	fail_rand_file:	    
	    (*errors)++;
	    
	    set_dt_estr(&rand_file_e,"none",NULL,0,0);

	    /* replace saved value on .elmrc with "none" */
	    mark_local_changed(&rand_file_e,0);

	}
    }


    rand_egd = give_dt_estr_as_str(&rand_egd_e,"rand-egd",NULL,NULL);
    if (rand_egd) {
	int z UNUSED_VAROK;
	int l;
	
	DPRINT(Debug,2,(&Debug,			    
			"tls: rand-egd is %s\n",rand_egd));

	l = RAND_BYTES - bytes;
	if (l < RAND_EGD_B)
	    l = RAND_EGD_B;

#ifdef TLS_HAVE_EGD
	if (0 < (z = RAND_egd_bytes(rand_egd,l)))
	    bytes += z;
	else
#endif
	    {
		lib_error(CATGETS(elm_msg_cat, TlsSet,
			      TlsEgdSocketDoesNotWork,
				  "EGD socket %s does not work"),
			  rand_egd);
	    }
    }

    DPRINT(Debug,2,(&Debug,			    
		    "tls: Seeded PRNG with %d bytes\n",bytes));

    CAfile = give_dt_estr_as_str(&CAfile_e,"trusted-ca-certificates-file",
				 NULL,NULL);
    if (CAfile) {
	int err;

	int is_default = 
	    used_default_cert_file &&
	    0 == strcmp(used_default_cert_file,CAfile);

	DPRINT(Debug,2,(&Debug,			    
			"tls: trusted-ca-certificates-file is %s%s\n",
			CAfile,
			is_default ? " (is openssl default)" : ""));
	
	err = can_open(CAfile,"r");
	if (0 != err) {

	    DPRINT(Debug,2,(&Debug,			    
			    "tls: trusted-ca-certificates-file: %s: %s\n",
			    CAfile,strerror(err)));

	    /* It is OK if default file does not exists */

	    if (err != ENOENT || !is_default) {
		lib_error(CATGETS(elm_msg_cat, TlsSet,
				  TlsTrustedCertNotReadable,
				  "TLS trusted-ca-certificates-file %s not readable: %s"),
			  CAfile,strerror(err));
		
		set_dt_estr(&CAfile_e,"none",NULL,0,0);
	
		/* replace saved value on .elmrc with "none" */
		mark_local_changed(&CAfile_e,0);

		(*errors)++;
	    }
		
	    CAfile = NULL;
	}	   
    }

    CApath = give_dt_estr_as_str(&CApath_e,"trusted-ca-certificates-dir",
				 NULL,NULL);
    if (CApath) {
	struct stat buffer;

	int is_default = 
	    used_default_cert_dir &&
	    0 == strcmp(used_default_cert_dir,CApath);

	DPRINT(Debug,2,(&Debug,			    
			"tls: trusted-ca-certificates-dir is %s%s\n",
			CApath,
			is_default ? " (is openssl default)" : ""));

	if (stat(CApath,&buffer) == 0) {
	    if (
#ifdef S_ISDIR
		!S_ISDIR(buffer.st_mode)
#else
		S_IFDIR != (buffer.st_mode & S_IFMT)
#endif
		) {
 
		lib_error(CATGETS(elm_msg_cat, TlsSet,
				  TlsTrustedCaCertDirNotDirectory,
				  "TLS trusted-ca-certificates-dir %s is not directory"),
			  CApath);
		
		set_dt_estr(&CApath_e,"none",NULL,0,0);
	
		/* replace saved value on .elmrc with "none" */
		mark_local_changed(&CApath_e,0);

		(*errors)++;

		CApath = NULL;
	    }

	} else {
	    int err = errno;

	    DPRINT(Debug,2,(&Debug,			    
			    "tls: trusted-ca-certificates-dir: %s: %s\n",
			    CApath,strerror(err)));

	    /* It is OK if default dir does not exists */

	    if (err != ENOENT || !is_default) {
		lib_error(CATGETS(elm_msg_cat, TlsSet,
				  TlsTrustedCaCertDirNotAvailable,
				  "TLS trusted-ca-certificates-dir %s not available: %s"),
			  CApath,strerror(err));
		
		set_dt_estr(&CApath_e,"none",NULL,0,0);

		/* replace saved value on .elmrc with "none" */
		mark_local_changed(&CApath_e,0);
	
		(*errors)++;
	    }
		
	    CApath = NULL;
	}
    }

    if (CApath || CAfile) {
	int failed = 0;

	for (v = 0; v < NUM_tls_version; v++) {

	    if (tls_xx_pCtx[v]) {
		if (! SSL_CTX_load_verify_locations(tls_xx_pCtx[v],CAfile,CApath)) {
		    DPRINT(Debug,2,(&Debug,
				    "Failed to set CAfile=%s / CApath=%s for %s\n",
				    CAfile ? CAfile : "<none>",
				    CApath ? CApath : "<none>",
				    TLS_VERSION[v]));
		    
		    failed = 1;
		}
	    }
	}
	
	if (failed)
	    lib_error(CATGETS(elm_msg_cat, TlsSet,
			      TlsFailedVerTrustedCertLoc,
			      "Failed to set verify location for trusted CA certificates."));

    }

    init_done = 1;

 fail:

    /* Clear error queue */
    while (0 != (ssl_error1 = ERR_get_error())) {
	char *s UNUSED_VAROK = ERR_error_string(ssl_error1,NULL);

	if (!p) {
	    DPRINT(Debug,13,(&Debug,			    
			     "   ... SSL error queue follows:\n"));
	    p = 1;
	}
	DPRINT(Debug,13,(&Debug,			    
			 "   ... SSL error %s (code %ld)\n",
			 s,ssl_error1));

    }



    return (init_done > 0);
}

struct shared_type_1 {      /* Type for TLS */
    struct stream_type    *type;

    BIO *pbioRead;
    BIO *pbioWrite;
    SSL *pSSL;

    X509 *peer_certificate;
    long verify_result;
    
    int wants;
    enum tls_connection_state state;

    int noread;   /* do not signal read */

    struct Read_Buffer read_buffer;
    struct Write_Buffer write_buffer;
}; 

#define READ_BLOCK   1024

static int ReadFromSSL P_((SSL *ssl, struct Read_Buffer *buffer,
			   int wanted));
static int ReadFromSSL(ssl,buffer,wanted)
     SSL *ssl; 
     struct Read_Buffer *buffer;
     int wanted;
{
    int n;

    if (wanted > 0) {
	buffer -> read_buffer = safe_realloc(buffer -> read_buffer,
					     buffer -> read_len + wanted);
	n = SSL_read(ssl, buffer -> read_buffer + buffer -> read_len, 
		     wanted);
    } else {
	buffer -> read_buffer = safe_realloc(buffer -> read_buffer,
					     buffer -> read_len + READ_BLOCK);
	n = SSL_read(ssl, buffer -> read_buffer + buffer -> read_len, 
		     READ_BLOCK);
    }

    return n;
}

static int WriteToSSL P_((SSL *ssl, struct Write_Buffer *buffer));
static int WriteToSSL(ssl,buffer)
     SSL *ssl;
     struct Write_Buffer *buffer;
{
    int n = SSL_write(ssl,buffer->write_buffer, buffer->write_len);
    
    return n;
}

S_(ss_ReadFromStream  ss_ReadFromTLS)
static int ss_ReadFromTLS P_((struct streamsched *ss,
			      int stack_idx,
			      struct Read_Buffer *buffer,
			      int wanted));
static int ss_ReadFromTLS (ss,stack_idx,buffer,wanted)
     struct streamsched *ss;
     int stack_idx;
     struct Read_Buffer *buffer;
     int wanted;
{
    int n;
    int err_x;
    int p = 0;
    long ssl_error1 = 0;

    DPRINT(Debug,15,(&Debug,			    
		"tls: ss_ReadFromTLS: ss=%p, stack_idx=%d\n",
		ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack || !ss->this_stack[stack_idx].generic_1 ||
	ss->this_stack[stack_idx].generic_1->type != &TLS_STREAM)
	panic("TLS PANIC",__FILE__,__LINE__,"ss_ReadFromTLS",
	      "Bad stack index or stack",0);

    n = ReadFromSSL(ss->this_stack[stack_idx].generic_1->pSSL,
		    buffer,wanted);

    if (n > 0) {
	ss->this_stack[stack_idx].generic_1->noread = 0;
	return n;
    }

    if (n == 0) {
	err_x = SSL_get_error(ss->this_stack[stack_idx].generic_1->pSSL,n);
	
	if (SSL_ERROR_ZERO_RETURN == err_x) {	    
	    ss->this_stack[stack_idx].generic_1->noread = 0;
	    /* Indicate EOF */
	    return 0;
	}
	
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_ReadFromTLS: SSL read returned %d, err_x=%d (not SSL_ERROR_ZERO_RETURN)\n",
			 n, err_x));
	
    } else {
	err_x = SSL_get_error(ss->this_stack[stack_idx].generic_1->pSSL,n);

	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_ReadFromTLS: SSL read returned %d (error), err_x=%d\n",
		    n, err_x));
	
    }
    
    switch(err_x) {
	long ssl_error;
    case SSL_ERROR_WANT_READ:
	ss->this_stack[stack_idx].generic_1->wants |= SS_read_act;
	ss->this_stack[stack_idx].generic_1->noread++;
	DPRINT(Debug,13,(&Debug,			    
			 "    ... adding want read -> noread=%d\n",
			 ss->this_stack[stack_idx].generic_1->noread));
	break;
    case SSL_ERROR_WANT_WRITE:
	ss->this_stack[stack_idx].generic_1->wants |= SS_write_act;
	ss->this_stack[stack_idx].generic_1->noread++;
	DPRINT(Debug,13,(&Debug,			    
			 "    ... adding want write -> noread=%d\n",
		    ss->this_stack[stack_idx].generic_1->noread));
	break;
    case SSL_ERROR_WANT_X509_LOOKUP:
	DPRINT(Debug,13,(&Debug,			    
			 "    ... WANTS X509 lOOKUP (SSL_CTX_set_client_cert_cb())\n"));
	break;
    case SSL_ERROR_SYSCALL:
	ssl_error = ERR_get_error();
	DPRINT(Debug,13,(&Debug,			    
			 "    ... SYSCALL ERROR (ssl_error=%ld)\n",
		    ssl_error));
	if (0 == ssl_error) {
	    if (-1 == n) {
		if (ss->error_state) {
		    DPRINT(Debug,13,(&Debug,			    
				     "   ... error state: %s\n",
				     ss->error_state));
		} else {
		    int err1 = errno;
		    DPRINT(Debug,13,(&Debug,			    
				     "   ... read error %s (errno %d)\n",
				     strerror(err1),err1));
		    
		    if (err1 != EINTR && err1 != EAGAIN && 
			err1 != EWOULDBLOCK) {
			ss->error_state = strmcpy(ss->error_state,
						  strerror(err1));
		    }
		}
	    } else if (0 == n) {
		/* FIXME: Messages */
		ss->error_state = strmcpy(ss->error_state,
					  "Unexpected EOF when reading");
	    }
	} else {
	    char *s = ERR_error_string(ssl_error,NULL);
	    DPRINT(Debug,13,(&Debug,			    
			     "   ... SSL error %s (code %ld)\n",
			     s,ssl_error));
	    ss->error_state = strmcpy(ss->error_state,
				      s);
	}
	break;
    case SSL_ERROR_SSL:
	ssl_error = ERR_get_error();
	DPRINT(Debug,13,(&Debug,			    
			 "    ... SSL ERROR (ssl_error=%ld)\n",
			 ssl_error));
	
	if (ssl_error != 0) {
	    char * s = ERR_error_string(ssl_error,NULL);
	    DPRINT(Debug,13,(&Debug,			    
			     "   ... SSL error %s (code %ld)\n",
			     s,ssl_error));
	    ss->error_state = strmcpy(ss->error_state,
				      s);
	}
    }

    /* Clear error queue */
    while (0 != (ssl_error1 = ERR_get_error())) {
	char *s UNUSED_VAROK = ERR_error_string(ssl_error1,NULL);

	if (!p) {
	    DPRINT(Debug,13,(&Debug,			    
			     "   ... Rest of SSL error queue follows:\n"));
	    p = 1;
	}
	DPRINT(Debug,13,(&Debug,			    
			 "   ... SSL error %s (code %ld)\n",
			 s,ssl_error1));

    }
    return -1;
}

S_(ss_WriteToStream ss_WriteToTLS)
static int ss_WriteToTLS P_((struct streamsched *ss, 
			     int stack_idx,
			     struct Write_Buffer *buffer));
static int ss_WriteToTLS(ss,stack_idx,buffer)
     struct streamsched *ss; 
     int stack_idx;
     struct Write_Buffer *buffer;
{
    int n;
    int err_x;
    int p = 0;
    long ssl_error1;

    DPRINT(Debug,15,(&Debug,			    
		     "tls: ss_WriteToTLS: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack || !ss->this_stack[stack_idx].generic_1 ||
	ss->this_stack[stack_idx].generic_1->type != &TLS_STREAM)
	panic("TLS PANIC",__FILE__,__LINE__,"ss_WriteToTLS",
	      "Bad stack index or stack",0);

    n = WriteToSSL(ss->this_stack[stack_idx].generic_1->pSSL,
		  buffer);

    if (n > 0)
	return n;

    err_x = SSL_get_error(ss->this_stack[stack_idx].generic_1->pSSL,n);

    DPRINT(Debug,13,(&Debug,			    
		     "tls: ss_WriteToTLS: SSL write returned %d, err_x=%d\n",
		     n, err_x));

    switch(err_x) {
	long ssl_error;
    case SSL_ERROR_ZERO_RETURN:
	DPRINT(Debug,13,(&Debug,			    
		    "    ... zero return????????\n"));	
	break;
    case SSL_ERROR_WANT_READ:
	ss->this_stack[stack_idx].generic_1->wants |= SS_read_act;
	DPRINT(Debug,13,(&Debug,			    
			 "    ... adding want read\n"));
	break;
    case SSL_ERROR_WANT_WRITE:
	ss->this_stack[stack_idx].generic_1->wants |= SS_write_act;
	DPRINT(Debug,13,(&Debug,			    
			 "    ... adding want write\n"));
	break;
    case SSL_ERROR_WANT_X509_LOOKUP:
	DPRINT(Debug,13,(&Debug,			    
			 "    ... WANTS X509 lOOKUP (SSL_CTX_set_client_cert_cb())\n"));
	break;
    case SSL_ERROR_SYSCALL:
	ssl_error = ERR_get_error();
	DPRINT(Debug,13,(&Debug,			    
			 "    ... SYSCALL ERROR (ssl_error=%ld)\n",
			 ssl_error));
	if (0 == ssl_error) {
	    if (-1 == n) {
		if (ss->error_state) {
		    DPRINT(Debug,13,(&Debug,			    
				     "   ... error state: %s\n",
				     ss->error_state));
		} else {
		    int err1 = errno;
		    DPRINT(Debug,13,(&Debug,			    
				     "   ... write error %s (errno %d)\n",
				     strerror(err1),err1));
		    
		    if (err1 != EINTR && err1 != EAGAIN && 
			err1 != EWOULDBLOCK) {
			ss->error_state = strmcpy(ss->error_state,
						  strerror(err1));
		    }
		}
	    }
	} else {
	    char *s = ERR_error_string(ssl_error,NULL);
	    DPRINT(Debug,13,(&Debug,			    
			     "   ... SSL error %s (code %ld)\n",
			     s,ssl_error));
	    ss->error_state = strmcpy(ss->error_state,
				      s);
	}
	break;
    case SSL_ERROR_SSL:
	ssl_error = ERR_get_error();
	DPRINT(Debug,13,(&Debug,			    
			 "    ... SSL ERROR (ssl_error=%ld)\n",
			 ssl_error));
	
	if (ssl_error != 0) {
	    char * s = ERR_error_string(ssl_error,NULL);
	    DPRINT(Debug,13,(&Debug,			    
			     "   ... SSL error %s (code %ld)\n",
			     s,ssl_error));
	    ss->error_state = strmcpy(ss->error_state,
				      s);
	}
    }

    /* Clear error queue */
    while (0 != (ssl_error1 = ERR_get_error())) {
	char *s UNUSED_VAROK = ERR_error_string(ssl_error1,NULL);

	if (!p) {
	    DPRINT(Debug,13,(&Debug,			    
			     "   ... Rest of SSL error queue follows:\n"));
	    p = 1;
	}
	DPRINT(Debug,13,(&Debug,			    
			 "   ... SSL error %s (code %ld)\n",
			 s,ssl_error1));
	
    }

    return -1;
}

S_(ss_FreeThis ss_FreeTLS)
static void ss_FreeTLS P_((struct streamsched *ss,
			   int stack_idx,int badpid));
static void ss_FreeTLS(ss,stack_idx,badpid)
     struct streamsched *ss;
     int stack_idx;
     int badpid;
{
    DPRINT(Debug,13,(&Debug,			    
		     "tls: ss_FreeTLS: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack || !ss->this_stack[stack_idx].generic_1 ||
	ss->this_stack[stack_idx].generic_1->type != &TLS_STREAM)
	panic("TLS PANIC",__FILE__,__LINE__,"ss_FreeTLS",
	      "Bad stack index or stack",0);

    if (0 == badpid) {
	/* Closing of SSL stream... */
	
	if (tls_connected == ss->this_stack[stack_idx].generic_1->state) {
	    DPRINT(Debug,15,(&Debug,			    
			     "tls: ss_FreeTLS:  closing (shutdown)\n"));
	    
	    ss->this_stack[stack_idx].generic_1->state = tls_closing;
	    
	    while (tls_closing == ss->this_stack[stack_idx].generic_1->state) {
		char * M = NULL;
		
		WaitStreamFor(ss,SS_shutdown_act);
		
		M =  RemoveStreamError(ss);
		if (M) {
		    lib_error(FRM("%s"),
			      M);
		    free(M);
		    break;
		}
	    }
	}
    }

    DPRINT(Debug,15,(&Debug,			    
		     "tls: ss_FreeTLS: SSL_free (%p):\n",
		     ss->this_stack[stack_idx].generic_1->pSSL));

    if (ss->this_stack[stack_idx].generic_1->pSSL)
	SSL_free(ss->this_stack[stack_idx].generic_1->pSSL);
    ss->this_stack[stack_idx].generic_1->pSSL      = NULL;

    /* These are freed by SSL_free */
    ss->this_stack[stack_idx].generic_1->pbioRead  = NULL;
    ss->this_stack[stack_idx].generic_1->pbioWrite = NULL;

    if (ss->this_stack[stack_idx].generic_1->peer_certificate)
	X509_free(ss->this_stack[stack_idx].generic_1->peer_certificate);
    ss->this_stack[stack_idx].generic_1->peer_certificate = NULL;
    ss->this_stack[stack_idx].generic_1->verify_result = 0;

    DPRINT(Debug,15,(&Debug,			    
		     "tls: ss_FreeTLS: buffers\n"));

    free_Read_Buffer(&(ss->this_stack[stack_idx].generic_1->read_buffer));
    free_Write_Buffer(&(ss->this_stack[stack_idx].generic_1->write_buffer));

    ss->this_stack[stack_idx].generic_1->state = tls_closed;

    if (0 == badpid) {
	const char * rand_file = NULL;

	rand_file = give_dt_estr_as_str(&rand_file_e,"rand-file",NULL,NULL);
	
	/* Write some bytes back to random generator */
	if (rand_file) {
	    int err = can_open(rand_file,"w");
	    
	    if (0 != err) {
		lib_error(CATGETS(elm_msg_cat, TlsSet,
				  TlsRandomFileNotWritable,
				  "TLS random file %s not writable: %s"),
			  rand_file,strerror(err));
		
		set_dt_estr(&rand_file_e,"none",NULL,0,0);
		
		/* replace saved value on .elmrc with "none" */
		mark_local_changed(&rand_file_e,0);
		
	    } else {
		RAND_write_file(rand_file);
		elm_chown (rand_file, userid, groupid,NULL);
	    }
	}
    }

    free(ss->this_stack[stack_idx].generic_1);
    ss->this_stack[stack_idx].generic_1 = NULL;

}

S_(ss_StreamAction ss_TLSReadAction)
static enum action_status ss_TLSReadAction P_((struct streamsched *ss, int stack_idx,
					       const struct schedule_timelimit * now));
static enum action_status ss_TLSReadAction(ss,stack_idx,now)
     struct streamsched *ss;
     int stack_idx;
     const struct schedule_timelimit * now;
{
    enum action_status ret = action_continue;

    int n;

    DPRINT(Debug,15,(&Debug,			    
		     "tls: ss_TLSReadAction: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack || !ss->this_stack[stack_idx].generic_1 ||
	ss->this_stack[stack_idx].generic_1->type != &TLS_STREAM)
	panic("TLS PANIC",__FILE__,__LINE__,"ss_TLSReadAction",
	      "Bad stack index or stack",0);
    
    if (stack_idx+1 >= ss->stack_len) 
	panic("TLS PANIC",__FILE__,__LINE__,"ss_TLSReadAction",
	      "No provider on stack!!!",0);
	

    if (ss->this_stack[stack_idx].generic_1->noread) {
	/* Reset for reconsideration */
	ss->this_stack[stack_idx].generic_1->noread = 0;
	DPRINT(Debug,13,(&Debug,			    
			 "    ... noread=%d\n",
			 ss->this_stack[stack_idx].generic_1->noread));
    }

    n = (*(ss -> this_stack[stack_idx+1].TYPE))->
	read_from_it(ss,stack_idx+1,
		     &(ss->this_stack[stack_idx].generic_1->read_buffer),
		     -1);

    if (n < 0) {      
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_TLSReadAction: read error %s\n",
			 ss->error_state ? ss->error_state : "(none)"));
    }

    if (n > 0) {
	int x;
	ss->this_stack[stack_idx].generic_1->read_buffer.read_len += n;
	
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_TLSReadAction: Read %d bytes (%d total)\n",
			 n,ss->this_stack[stack_idx].generic_1->
			 read_buffer.read_len));
	
	x = BIO_write(ss->this_stack[stack_idx].generic_1->pbioRead,
		      ss->this_stack[stack_idx].generic_1->
		      read_buffer.read_buffer,
		      ss->this_stack[stack_idx].generic_1->
		      read_buffer.read_len);
	
	if (x > 0) {
	    cut_line(& (ss->this_stack[stack_idx].generic_1->read_buffer),
		     x);
	} else {
	    DPRINT(Debug,13,(&Debug,			    
			     "... writing it to read BIO failed, ret=%d\n",
			     x));
	}
	
	if (ss->this_stack[stack_idx].generic_1->read_buffer.read_len >0) {
	    DPRINT(Debug,13,(&Debug,			    
			     "... %d chars LEFT, not written to read BIO\n",
			     ss->this_stack[stack_idx].generic_1->
			     read_buffer.read_len));
	}	
    }

    if (0 == n) {
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_TLSReadAction: EOF on read -- setting read BIOs EOF flag\n"));
	BIO_set_mem_eof_return(ss->this_stack[stack_idx].generic_1->pbioRead,
			       0);

	/* No LONGER wants read ... */
	ret = action_disable;
	ss->this_stack[stack_idx].generic_1->wants &= ~SS_read_act;
    }

    if (n < 0 &&
	ss->this_stack[stack_idx].generic_1->read_buffer.read_len == 0 &&
	ss->error_state) {

	/* No LONGER wants read ... */
	ret = action_disable;
	ss->this_stack[stack_idx].generic_1->wants &= ~SS_read_act;
    }

    DPRINT(Debug,15,(&Debug,			    
		     "tls: ss_TLSReadAction=%d %s\n",
		     ret, ret == action_disable ? "(disabling)" : ""));
    return ret;
}

S_(ss_StreamAction ss_TLSWriteAction)
static enum action_status ss_TLSWriteAction P_((struct streamsched *ss, int stack_idx,
						const struct schedule_timelimit * now));
static enum action_status ss_TLSWriteAction(ss,stack_idx,now)
     struct streamsched *ss;
     int stack_idx;
     const struct schedule_timelimit * now;
{
    char * s;
    int n,y;
    enum action_status ret = action_disable;
    int blocksize = 0;

    DPRINT(Debug,15,(&Debug,			    
		     "tls: ss_TLSWriteAction: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack || !ss->this_stack[stack_idx].generic_1 ||
	ss->this_stack[stack_idx].generic_1->type != &TLS_STREAM)
	panic("TLS PANIC",__FILE__,__LINE__,"ss_TLSWriteAction",
	      "Bad stack index or stack",0);
    
    if (stack_idx+1 >= ss->stack_len) 
	panic("TLS PANIC",__FILE__,__LINE__,"ss_TLSWriteAction",
	      "No provider on stack!!!",0);

    if (ss->this_stack[stack_idx].generic_1->noread) {
	/* Reset for reconsideration */
	ss->this_stack[stack_idx].generic_1->noread = 0;
	DPRINT(Debug,13,(&Debug,			    
			 "    ... noread=%d\n",
			 ss->this_stack[stack_idx].generic_1->noread));
    }

    blocksize = BIO_pending(ss->this_stack[stack_idx].generic_1->pbioWrite);
    if (blocksize < 1)
	blocksize = READ_BLOCK;
    else {
	DPRINT(Debug,13,(&Debug,			    
			 "    (%d bytes available on write BIO for read)\n",
			 blocksize));
    }

    s = safe_malloc(blocksize);
 
    n = BIO_read(ss->this_stack[stack_idx].generic_1->pbioWrite,
		 s,blocksize);

    if (n < 0) {
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_TLSWriteAction: Failed to read from WRITE BIO, ret=%d\n",
			 n));

	free(s);
	s = NULL;

    } else {
	/* add_to_Write_Buffer will free s */

	add_to_Write_Buffer( &(ss->this_stack[stack_idx].generic_1->
			       write_buffer),
			     &s,n);
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_TLSWriteAction: Readed %d bytes from write BIO (%d total)\n",
			 n,ss->this_stack[stack_idx].generic_1->
			 write_buffer.write_len));
    }
		
    if (ss->this_stack[stack_idx].generic_1->write_buffer.write_len > 0) {
	int x = (*(ss -> this_stack[stack_idx+1].TYPE))->
	    write_to_it(ss,stack_idx+1,
			&(ss->this_stack[stack_idx].generic_1->write_buffer));

	if (x < 0) {      
	    DPRINT(Debug,13,(&Debug,			    
			     "tls: ss_TLSWriteAction: write error %s\n",
			     ss->error_state ? ss->error_state : "(none)"));
	}

	if (x > 0) {
	    cut_Write_Buffer(&(ss->this_stack[stack_idx].generic_1->
			       write_buffer),x);
	    DPRINT(Debug,13,(&Debug,			    
			     " ... written %d bytes (%d left)\n",
			     x,ss->this_stack[stack_idx].generic_1->
			     write_buffer.write_len));
	    ret = action_continue;
	}	
    }

    /* If there is nothing read from BIO and write buffer flushed,
       then we not want longer write ... 
    */

    y = BIO_pending(ss->this_stack[stack_idx].generic_1->pbioWrite);
    if (y > 0) {
	DPRINT(Debug,13,(&Debug,			    
			 " ... %d bytes available on write BIO for read\n",
			 y));
    } else if (0 == ss->this_stack[stack_idx].generic_1->
	       write_buffer.write_len) {
	ss->this_stack[stack_idx].generic_1->wants &= ~SS_write_act;
	ret = action_disable;
    }

    DPRINT(Debug,15,(&Debug,			    
		     "tls: ss_TLSWriteAction=%d %s\n",
		     ret, ret == action_disable ? "(disabling)" : ""));
    return ret;
}

S_(ss_StreamSchedule ss_TLSSchedule)
static void ss_TLSSchedule  P_((struct streamsched *ss, int stack_idx,
				int previous_needs, 
				int *needs, int *provides));
static void ss_TLSSchedule(ss,stack_idx,previous_needs,
			   needs,provides)
     struct streamsched *ss;
     int stack_idx;
     int previous_needs;
     int *needs;
     int *provides;
{
    
    DPRINT(Debug,15,(&Debug,			    
		     "tls: ss_TLSSchedule: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack)
	panic("TLS PANIC",__FILE__,__LINE__,"ss_TLSSchedule",
	      "Bad stack index or stack",0);

    *provides = 0;
    *needs    = 0;

    if (!ss->this_stack[stack_idx].generic_1) {
	DPRINT(Debug,15,(&Debug,			    
			 "tls: ss_TLSSchedule:  closed (no data)\n"));
	return;
    }

    if (ss->this_stack[stack_idx].generic_1->type != &TLS_STREAM)
	panic("TLS PANIC",__FILE__,__LINE__,"ss_TLSSchedule",
	      "Bad stack index or stack",0);

    if (tls_closed == ss->this_stack[stack_idx].generic_1->state) {
	DPRINT(Debug,15,(&Debug,			    
			 "tls: ss_TLSSchedule:  state = closed\n"));
	return;
    }

    if (ss->this_stack[stack_idx].generic_1->pbioWrite) {
	int y;
	y = BIO_pending(ss->this_stack[stack_idx].generic_1->pbioWrite);
	if (y > 0) {
	    DPRINT(Debug,13,(&Debug,			    
			"tls: ss_TLSSchedule: %d bytes available on write BIO for read\n",
			y));
	    ss->this_stack[stack_idx].generic_1->wants |= SS_write_act;
	}
    } else {
	DPRINT(Debug,13,(&Debug,			    
		    "tls: ss_TLSSchedule: NO Write BIO\n"));
    }

    if (ss->this_stack[stack_idx].generic_1->pbioRead) {
	if (BIO_should_read(ss->this_stack[stack_idx].generic_1->pbioRead)) {
	    DPRINT(Debug,13,(&Debug,			    
			     "tls: ss_TLSSchedule: read BIO needs read data\n"));
	    ss->this_stack[stack_idx].generic_1->wants |= SS_read_act;
	}
    } else {	
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_TLSSchedule: NO Read BIO\n"));	
    }

    *needs    = ss->this_stack[stack_idx].generic_1->wants;

    switch (ss->this_stack[stack_idx].generic_1->state) {
    case tls_connected:	
	if (ss->this_stack[stack_idx].generic_1->pSSL) {
	    int n;
	    if ((n = SSL_pending(ss->this_stack[stack_idx].generic_1->pSSL)) > 0) {
		DPRINT(Debug,13,(&Debug,			    
				 "tls: ss_TLSSchedule: %d bytes available for read\n",
				 n));
		*provides |= SS_read_act;
		/* Reset noread */
		ss->this_stack[stack_idx].generic_1->noread = 0;
	    }
	} else {
	    DPRINT(Debug,15,(&Debug,			    
			     "tls: ss_TLSSchedule: No SSL!\n"));
	}

	
	if (ss->this_stack[stack_idx].generic_1->pbioRead) {
	    int n;
	    n = BIO_pending(ss->this_stack[stack_idx].generic_1->pbioRead);
	    if (n > 0) {
		DPRINT(Debug,15,(&Debug,			    
				 "tls: ss_TLSSchedule: %d bytes available on read BIO for read\n",
				 n));
		/* Because SSL_read reads from read BIO, we can now try read from it */
		*provides |= SS_read_act;
		ss->this_stack[stack_idx].generic_1->noread = 0;
	    }
	} else {
	    DPRINT(Debug,15,(&Debug,			    
			     "tls: ss_TLSSchedule: No read BIO!\n"));
	}
	
	/* We always 'provide' what upstream wants (if connected)
	   .. after all there is buffering BIOs on between
	   .. but do not provide read after failure because
	      it may cause loop
	*/
	if (ss->this_stack[stack_idx].generic_1->noread > 0 &&
	    0 != (previous_needs & SS_read_act)) {
	    DPRINT(Debug,13,(&Debug,			    
			     "tls: ss_TLSSchedule: will not provide read, noread=%d\n",
			     ss->this_stack[stack_idx].generic_1->noread
			));
	    *provides |= previous_needs &
		(SS_write_act|SS_timeout_act);

	} else {
	    *provides |= previous_needs &
		(SS_read_act|SS_write_act|SS_timeout_act);
	}

	break;
    case tls_not_connected:
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_TLSSchedule: Hanshake not completed yet...\n"));
	*needs |= SS_setup_act;
	break;
    case tls_closing:
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_TLSSchedule: Stream is closing...\n"));
	*needs |= SS_shutdown_act;
	break;
    default:
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_TLSSchedule: Have TLS error\n"));
	*provides |= previous_needs &
	    (SS_read_act|SS_write_act|SS_timeout_act);
	break;
    }
}


#if DEBUG

static char * peer_verify_result P_((long error));
static char * peer_verify_result(error)
     long error;
{
    switch (error) {
    case X509_V_OK: return "X509_V_OK";
    case X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT: 
	return "X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT"; break;
    case X509_V_ERR_UNABLE_TO_GET_CRL: return "X509_V_ERR_UNABLE_TO_GET_CRL";
    case X509_V_ERR_UNABLE_TO_DECRYPT_CERT_SIGNATURE:
	return "X509_V_ERR_UNABLE_TO_DECRYPT_CERT_SIGNATURE";
    case X509_V_ERR_UNABLE_TO_DECRYPT_CRL_SIGNATURE:
	return "X509_V_ERR_UNABLE_TO_DECRYPT_CRL_SIGNATURE";
    case X509_V_ERR_UNABLE_TO_DECODE_ISSUER_PUBLIC_KEY:
	return "X509_V_ERR_UNABLE_TO_DECODE_ISSUER_PUBLIC_KEY";
    case X509_V_ERR_CERT_SIGNATURE_FAILURE:
	return "X509_V_ERR_CERT_SIGNATURE_FAILURE";
    case X509_V_ERR_CRL_SIGNATURE_FAILURE:
	return "X509_V_ERR_CRL_SIGNATURE_FAILURE";
    case X509_V_ERR_CERT_NOT_YET_VALID:
	return "X509_V_ERR_CERT_NOT_YET_VALID";
    case X509_V_ERR_CERT_HAS_EXPIRED:
	return "X509_V_ERR_CERT_HAS_EXPIRED";
    case X509_V_ERR_ERROR_IN_CERT_NOT_AFTER_FIELD:
	return "X509_V_ERR_ERROR_IN_CERT_NOT_AFTER_FIELD";
    case X509_V_ERR_ERROR_IN_CRL_LAST_UPDATE_FIELD:
	return "X509_V_ERR_ERROR_IN_CRL_LAST_UPDATE_FIELD";
    case X509_V_ERR_ERROR_IN_CRL_NEXT_UPDATE_FIELD:
	return "X509_V_ERR_ERROR_IN_CRL_NEXT_UPDATE_FIELD";
    case X509_V_ERR_OUT_OF_MEM:
	return "X509_V_ERR_OUT_OF_MEM";
    case X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT:
	return "X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT";
    case X509_V_ERR_SELF_SIGNED_CERT_IN_CHAIN:
	return "X509_V_ERR_SELF_SIGNED_CERT_IN_CHAIN";
    case X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY:
	return "X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY";
    case X509_V_ERR_UNABLE_TO_VERIFY_LEAF_SIGNATURE:
	return "X509_V_ERR_UNABLE_TO_VERIFY_LEAF_SIGNATURE";
    case X509_V_ERR_CERT_CHAIN_TOO_LONG:
	return "X509_V_ERR_CERT_CHAIN_TOO_LONG";
    case X509_V_ERR_CERT_REVOKED:
	return "X509_V_ERR_CERT_REVOKED";
    case X509_V_ERR_INVALID_CA:
	return "X509_V_ERR_INVALID_CA";
    case X509_V_ERR_PATH_LENGTH_EXCEEDED:
	return "X509_V_ERR_PATH_LENGTH_EXCEEDED";
    case X509_V_ERR_INVALID_PURPOSE:
	return "X509_V_ERR_INVALID_PURPOSE";
    case X509_V_ERR_CERT_UNTRUSTED:
	return "X509_V_ERR_CERT_UNTRUSTED";
    case X509_V_ERR_CERT_REJECTED:
	return "X509_V_ERR_CERT_REJECTED";
#ifdef X509_V_ERR_SUBJECT_ISSUER_MISMATCH
    case X509_V_ERR_SUBJECT_ISSUER_MISMATCH:
	return "X509_V_ERR_SUBJECT_ISSUER_MISMATCH";
#endif
#ifdef X509_V_ERR_AKID_SKID_MISMATCH
    case X509_V_ERR_AKID_SKID_MISMATCH:
	return "X509_V_ERR_AKID_SKID_MISMATCH";
#endif
#ifdef X509_V_ERR_AKID_ISSUER_SERIAL_MISMATCH
    case X509_V_ERR_AKID_ISSUER_SERIAL_MISMATCH:
	return "X509_V_ERR_AKID_ISSUER_SERIAL_MISMATCH";
#endif
#ifdef X509_V_ERR_KEYUSAGE_NO_CERTSIGN
    case X509_V_ERR_KEYUSAGE_NO_CERTSIGN:
	return "X509_V_ERR_KEYUSAGE_NO_CERTSIGN";
#endif
    case X509_V_ERR_APPLICATION_VERIFICATION:
	return "X509_V_ERR_APPLICATION_VERIFICATION";
    }

    return "";  /* Unknown error code */
}

#endif


S_(ss_StreamAction ss_SetupTLSAction)
static enum action_status ss_SetupTLSAction P_((struct streamsched *ss,
						int stack_idx,
						const struct schedule_timelimit * now));
static enum action_status ss_SetupTLSAction(ss,stack_idx,now)
     struct streamsched *ss;
     int stack_idx;
     const struct schedule_timelimit * now;
{
    int n;
    int err_x;
    long ssl_error1;
    int p = 0;
    enum action_status ret = action_continue;

    DPRINT(Debug,13,(&Debug,			    
		     "tls: ss_SetupTLSAction: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack || !ss->this_stack[stack_idx].generic_1 ||
	ss->this_stack[stack_idx].generic_1->type != &TLS_STREAM)
	panic("TLS PANIC",__FILE__,__LINE__,"ss_SetupTLSAction",
	      "Bad stack index or stack",0);

    if (ss->this_stack[stack_idx].generic_1->noread) {
	/* Reset for reconsideration */
	ss->this_stack[stack_idx].generic_1->noread = 0;
	DPRINT(Debug,13,(&Debug,			    
			 "    ... noread=%d\n",
			 ss->this_stack[stack_idx].generic_1->noread));
    }


    n = SSL_connect(ss->this_stack[stack_idx].generic_1->pSSL);
    
    if (n > 0) {

	ss->this_stack[stack_idx].generic_1->state = tls_connected;

	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_SetupTLSAction: Setting connected -- SSL connect returned %d\n",
			 n));

	if (ss->this_stack[stack_idx].generic_1->peer_certificate)
	    X509_free(ss->this_stack[stack_idx].generic_1->peer_certificate);

	ss->this_stack[stack_idx].generic_1->peer_certificate =
	    SSL_get_peer_certificate(ss->this_stack[stack_idx].generic_1->pSSL);
	
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_SetupTLSAction: %s peer certificate\n",
			 (ss->this_stack[stack_idx].generic_1->peer_certificate ? 
			  "got" : "no")));

	ss->this_stack[stack_idx].generic_1->verify_result
	    = SSL_get_verify_result(ss->this_stack[stack_idx].generic_1->pSSL);
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_SetupTLSAction: verify result %ld %s (%s%s)\n",
			 ss->this_stack[stack_idx].generic_1->verify_result,
			 peer_verify_result(ss->this_stack[stack_idx].
					    generic_1->verify_result),
			 ss->this_stack[stack_idx].generic_1->verify_result
			 == X509_V_OK ? "no errors" : "failed",
			 ss->this_stack[stack_idx].generic_1->peer_certificate 
			 ? "" : ", not verified"));
       
	ret = action_disable;
	goto out;
    }
    err_x = SSL_get_error(ss->this_stack[stack_idx].generic_1->pSSL,n);
    
    DPRINT(Debug,13,(&Debug,			    
		     "tls: ss_SetupTLSAction: SSL connect returned %d, err_x=%d\n",
		     n, err_x));

    switch(err_x) {
	long ssl_error;
	int A UNUSED_VAROK;
	int B UNUSED_VAROK;

    case SSL_ERROR_ZERO_RETURN:
	DPRINT(Debug,13,(&Debug,			    
			 "    ... zero return????????\n"));	
	break;
    case SSL_ERROR_WANT_READ:
	ss->this_stack[stack_idx].generic_1->wants |= SS_read_act;
	DPRINT(Debug,13,(&Debug,			    
			 "    ... adding want read\n"));
	break;
    case SSL_ERROR_WANT_WRITE:
	ss->this_stack[stack_idx].generic_1->wants |= SS_write_act;
	DPRINT(Debug,13,(&Debug,			    
			 "    ... adding want write\n"));
	break;
    case SSL_ERROR_WANT_X509_LOOKUP:
	DPRINT(Debug,13,(&Debug,			    
			 "    ... WANTS X509 lOOKUP (SSL_CTX_set_client_cert_cb())\n"));
	break;
    case SSL_ERROR_SYSCALL:
	ssl_error = ERR_get_error();
	A = ERR_GET_LIB(ssl_error);
	B = ERR_GET_REASON(ssl_error);
	DPRINT(Debug,13,(&Debug,			    
			 "    ... SYSCALL ERROR (ssl_error=%ld: lib=%d, reason=%d)\n",
			 ssl_error,A,B));
	if (0 == ssl_error) {
	    if (-1 == n) {
		if (ss->error_state) {
		    DPRINT(Debug,13,(&Debug,			    
				     "   ... error state: %s\n",
				     ss->error_state));
		} else {
		    int err1 = errno;
		    DPRINT(Debug,13,(&Debug,			    
				     "   ... write error %s (errno %d)\n",
				     strerror(err1),err1));
		    
		    if (err1 != EINTR && err1 != EAGAIN && 
			err1 != EWOULDBLOCK) {
			ss->error_state = strmcpy(ss->error_state,
						  strerror(err1));
		    }
		}
	    }
	} else {
	    char *s = ERR_error_string(ssl_error,NULL);
	    DPRINT(Debug,13,(&Debug,			    
			     "   ... SSL error %s (code %ld)\n",
			     s,ssl_error));
	    ss->error_state = strmcpy(ss->error_state,
				      s);
	}
	ss->this_stack[stack_idx].generic_1->state = tls_error;
	DPRINT(Debug,13,(&Debug,			    
		    "tls: ss_SetupTLSAction: Setting error state\n"));
	break;
    case SSL_ERROR_SSL:
	ssl_error = ERR_get_error();
	A = ERR_GET_LIB(ssl_error);
	B = ERR_GET_REASON(ssl_error);
	DPRINT(Debug,13,(&Debug,			    
			 "    ... SSL ERROR (ssl_error=%ld: lib=%d, reason=%d)\n",
			 ssl_error,A,B));
	
	if (ssl_error != 0) {
	    char * s = ERR_error_string(ssl_error,NULL);
	    DPRINT(Debug,13,(&Debug,			    
			     "   ... SSL error %s (code %ld)\n",
			     s,ssl_error));
	    ss->error_state = strmcpy(ss->error_state,
				      s);
	}

	ss->this_stack[stack_idx].generic_1->state = tls_error;
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_SetupTLSAction: Setting error state\n"));
    }

 out:

    /* Clear error queue -- can verify result also put something to queue? */

    while (0 != (ssl_error1 = ERR_get_error())) {
	char *s UNUSED_VAROK = ERR_error_string(ssl_error1,NULL);

	if (!p) {
	    DPRINT(Debug,13,(&Debug,			    
			     "   ... Rest of SSL error queue follows:\n"));
	    p = 1;
	}
	DPRINT(Debug,13,(&Debug,			    
			 "   ... SSL error %s (code %ld)\n",
			 s,ssl_error1));
	
    }

    DPRINT(Debug,15,(&Debug,			    
		     "tls: ss_SetupTLSAction=%d %s\n",
		     ret, ret == action_disable ? "(disabling?)" : ""));

    return ret;
}

S_(ss_StreamAction ss_ShutdownTLSAction)
static enum action_status ss_ShutdownTLSAction P_((struct streamsched *ss,
						   int stack_idx,
						   const struct schedule_timelimit * now));
static enum action_status ss_ShutdownTLSAction(ss,stack_idx,now)
     struct streamsched *ss;
     int stack_idx;
     const struct schedule_timelimit * now;
{
    int n;
    int err_x;
    long ssl_error1;
    int p = 0;
    enum action_status ret = action_continue;

    DPRINT(Debug,15,(&Debug,			    
		     "tls: ss_ShutdownTLSAction: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack || !ss->this_stack[stack_idx].generic_1 ||
	ss->this_stack[stack_idx].generic_1->type != &TLS_STREAM)
	panic("TLS PANIC",__FILE__,__LINE__,"ss_shutdownTLSAction",
	      "Bad stack index or stack",0);


    n = SSL_shutdown(ss->this_stack[stack_idx].generic_1->pSSL);
    
    if (n > 0) {
	ss->this_stack[stack_idx].generic_1->state = tls_closed;
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_ShutdownTLSAction: Setting closed -- SSL shutdown returned %d\n",
			 n));
	ret = action_disable;
	goto out;
    }
    err_x = SSL_get_error(ss->this_stack[stack_idx].generic_1->pSSL,n);
    
    DPRINT(Debug,13,(&Debug,			    
		     "tls: ss_ShutdownTLSAction: SSL shutdown returned %d, err_x=%d\n",
		     n, err_x));

    switch(err_x) {
	long ssl_error;
	int UNUSED_VAROK A;
	int UNUSED_VAROK B;

    case SSL_ERROR_ZERO_RETURN:
	DPRINT(Debug,13,(&Debug,			    
			 "    ... zero return????????\n"));	
	break;
    case SSL_ERROR_WANT_READ:
	ss->this_stack[stack_idx].generic_1->wants |= SS_read_act;
	DPRINT(Debug,13,(&Debug,			    
			 "    ... adding want read\n"));
	break;
    case SSL_ERROR_WANT_WRITE:
	ss->this_stack[stack_idx].generic_1->wants |= SS_write_act;
	DPRINT(Debug,13,(&Debug,			    
			 "    ... adding want write\n"));
	break;
    case SSL_ERROR_WANT_X509_LOOKUP:
	DPRINT(Debug,13,(&Debug,			    
			 "    ... WANTS X509 lOOKUP (SSL_CTX_set_client_cert_cb())\n"));
	break;
    case SSL_ERROR_SYSCALL:
	ssl_error = ERR_get_error();
	A = ERR_GET_LIB(ssl_error);
	B = ERR_GET_REASON(ssl_error);
	DPRINT(Debug,13,(&Debug,			    
			 "    ... SYSCALL ERROR (ssl_error=%ld: lib=%d, reason=%d)\n",
			 ssl_error,A,B));
	if (0 == ssl_error) {
	    if (-1 == n) {
		if (ss->error_state) {
		    DPRINT(Debug,13,(&Debug,			    
				     "   ... error state: %s\n",
				     ss->error_state));
		} else {
		    int err1 = errno;
		    DPRINT(Debug,13,(&Debug,			    
				     "   ... write error %s (errno %d)\n",
				     strerror(err1),err1));
		    
		    if (err1 != EINTR && err1 != EAGAIN && 
			err1 != EWOULDBLOCK) {
			ss->error_state = strmcpy(ss->error_state,
						  strerror(err1));
		    }
		}
	    }
	} else {
	    char *s = ERR_error_string(ssl_error,NULL);
	    DPRINT(Debug,13,(&Debug,			    
			     "   ... SSL error %s (code %ld)\n",
			     s,ssl_error));
	    ss->error_state = strmcpy(ss->error_state,
				      s);
	}
	ss->this_stack[stack_idx].generic_1->state = tls_error;
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_ShutdownTLSAction: Setting error state\n"));
	break;
    case SSL_ERROR_SSL:
	ssl_error = ERR_get_error();
	A = ERR_GET_LIB(ssl_error);
	B = ERR_GET_REASON(ssl_error);
	DPRINT(Debug,13,(&Debug,			    
			 "    ... SSL ERROR (ssl_error=%ld: lib=%d, reason=%d)\n",
			 ssl_error,A,B));
	
	if (ssl_error != 0) {
	    char * s = ERR_error_string(ssl_error,NULL);
	    DPRINT(Debug,13,(&Debug,			    
			     "   ... SSL error %s (code %ld)\n",
			     s,ssl_error));
	    ss->error_state = strmcpy(ss->error_state,
				      s);
	}

	ss->this_stack[stack_idx].generic_1->state = tls_error;
	DPRINT(Debug,13,(&Debug,			    
			 "tls: ss_ShutdownTLSAction: Setting error state\n"));
    }

    /* Clear error queue */
    while (0 != (ssl_error1 = ERR_get_error())) {
	char *s UNUSED_VAROK = ERR_error_string(ssl_error1,NULL);

	if (!p) {
	    DPRINT(Debug,13,(&Debug,			    
			"   ... Rest of SSL error queue follows:\n"));
	    p = 1;
	}
	DPRINT(Debug,13,(&Debug,			    
		    "   ... SSL error %s (code %ld)\n",
		    s,ssl_error1));
	
    }

 out:
    DPRINT(Debug,13,(&Debug,			    
		     "tls: ss_ShutdownTLSAction=%d %s\n",
		     ret, ret == action_disable ? "(disabling?)" : ""));

    return ret;
}


S_(ss_StreamInfo ss_TLSInfo)
static void ss_TLSInfo P_((struct streamsched *ss, int stack_idx,
			   enum SS_info function,
			   int *int_val,  char **str_val,
			   struct string **string_val));
static void ss_TLSInfo(ss,stack_idx,function,int_val,str_val,
		       string_val)
     struct streamsched *ss; 
     int stack_idx;
     enum SS_info function;
     int *int_val; 
     char **str_val;
     struct string **string_val;
{    
    int x;

    DPRINT(Debug,15,(&Debug,			    
		     "tls: ss_TLSInfo: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack || !ss->this_stack[stack_idx].generic_1 ||
	ss->this_stack[stack_idx].generic_1->type != &TLS_STREAM)
	panic("TLS PANIC",__FILE__,__LINE__,"ss_TLSInfo",
	      "Bad stack index or stack",0);

    switch (function) {
    case SS_ssf:
	*int_val = SSL_get_cipher_bits(ss->this_stack[stack_idx].generic_1->
				       pSSL,&x);
	
	DPRINT(Debug,10,(&Debug,			    
			 "tls: ss_TLSInfo: cipher_bits=%d (aka ssf), alg_bits=%d\n",
			 *int_val,x));
	break;

    case SS_verify:
	*int_val =  ss_TLS_verify(ss,stack_idx,NULL,string_val);
	break;
   }    

}

#ifdef TLS_CHECK_HOST
#include <openssl/x509v3.h>
#include <openssl/x509.h>
#endif

S_(ss_StreamVerifyName ss_TLSVerifyName)
static int ss_TLSVerifyName P_((struct streamsched *ss, int stack_idx,
				enum SS_name_type name_type,
				const struct string *required_name,
				struct string      **returned_name));
static int ss_TLSVerifyName(ss,stack_idx,name_type,required_name,
			    returned_name)
     struct streamsched *ss; 
     int stack_idx;
     enum SS_name_type name_type;
     const struct string *required_name;
     struct string      **returned_name;
{
    int found = 0;

    charset_t utf8_set = MIME_name_to_charset("UTF-8",0);

    DPRINT(Debug,15,(&Debug,			    
		     "tls: ss_TLSVerifyName: ss=%p, stack_idx=%d, name_type=%d, required_name=%S\n",
		     ss,stack_idx,name_type,required_name));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack || !ss->this_stack[stack_idx].generic_1 ||
	ss->this_stack[stack_idx].generic_1->type != &TLS_STREAM)
	panic("TLS PANIC",__FILE__,__LINE__,"ss_TLSVerifyName",
	      "Bad stack index or stack",0);

    if (ss->this_stack[stack_idx].generic_1->peer_certificate && utf8_set) {

	switch(name_type) {
	    
	case SS_peer_cn: {

	    X509_NAME *peer_subject = 
		X509_get_subject_name(ss->this_stack[stack_idx].generic_1->peer_certificate);
	    
	    if (peer_subject) {
		int cn_idx;
		
		/* Loop through common names */
		for (cn_idx = X509_NAME_get_index_by_NID(peer_subject, 
							 NID_commonName, -1);
		     cn_idx != -1;
		     cn_idx = X509_NAME_get_index_by_NID(peer_subject, 
							 NID_commonName, 
							 cn_idx)) {
		    X509_NAME_ENTRY * peer_subject_cn =
			X509_NAME_get_entry(peer_subject,cn_idx);
		    ASN1_STRING     * peer_subject_cn_asn1 = 
			X509_NAME_ENTRY_get_data(peer_subject_cn);

		    unsigned char   * peer_subject_cn_utf8 = NULL;
		    int               peer_subject_cn_len  = 
			ASN1_STRING_to_UTF8(&peer_subject_cn_utf8,
					    peer_subject_cn_asn1);

		    struct string * res = new_string(utf8_set);
		    
		    add_streambytes_to_string(res,
					      peer_subject_cn_len,
					      peer_subject_cn_utf8,NULL);
		    
		    DPRINT(Debug,10,(&Debug,			    
				     "tls: ss_TLSVerifyName: %d: found name (CN): %S\n",
				     cn_idx,res));
		    
		    if (0 == string_cmp(required_name,res,-99)) {
			DPRINT(Debug,10,(&Debug,
					 "tls: ss_TLSVerifyName: %d: name matches: %S\n",
					 cn_idx,required_name));
			found = 1;
		    }
		    
		    if (returned_name) {
			if (*returned_name)
			    free_string(returned_name);
			*returned_name = res; res = NULL;
		    } else
			free_string(&res);
		    
		    OPENSSL_free(peer_subject_cn_utf8);

		    if (found)
			break;
		}		
	    }
	}
	    break;

	case SS_check_host: {

#ifdef TLS_CHECK_HOST

	    int failcount = 0;

	    /* XXX utf-8 is not correct */
	    struct string * req_name = convert_string2(utf8_set,required_name,&failcount);
	    /* man X509_check_host:

	       Per section 6.4.2 of RFC 6125, name values representing international domain names must be given in A-label
	       form. 
	    
	       this is not now supported -- but utf-8 is used
	    */

	    if (failcount) {
		DPRINT(Debug,10,(&Debug,			    
				 "tls: ss_TLSVerifyName: Failed to convert %S to utf-8, failcount %d\n",
				 required_name,failcount));
	    }
	    

	    if (req_name) {
		char *          req_name_res = NULL;
		int             req_name_len = 0;
	    

		bytestream_from_string(req_name,& req_name_res,& req_name_len);

		if (req_name_res) {
		    char * peername = NULL;

		    int r =  X509_check_host(ss->this_stack[stack_idx].generic_1->peer_certificate,
					     req_name_res,req_name_len,0,
					     &peername);

		    DPRINT(Debug,10,(&Debug,			    
				     "tls: ss_TLSVerifyName: X509_check_host returned %d",
				     r));

		    switch (r) {
		    case 1:
			DPRINT(Debug,10,(&Debug," succesful match"));
			found = 1;
			break;
		    case 0:
			DPRINT(Debug,10,(&Debug," failed match"));
			break;
		    case -1:
			DPRINT(Debug,10,(&Debug," internal error"));
			break;
		    case -2:
			DPRINT(Debug,10,(&Debug," malformed input"));
			break;
		    }
		    DPRINT(Debug,10,(&Debug,"\n"));
		    
		     if (returned_name) {
			if (*returned_name)
			    free_string(returned_name);

			if (peername) {
			    /* XXX utf-8 is not correct */
			    
			    *returned_name = new_string2(utf8_set,s2us(peername));
			}
			
		     }

		     if (peername)
			 OPENSSL_free(peername);

		    free(req_name_res);
		    req_name_res = NULL;
		}
	    }

	    if (req_name)
		free_string(&  req_name );
	    
#endif

	}
	    break;
	    
	}
    }

    DPRINT(Debug,10,(&Debug,			    
		     "tls: ss_TLSVerifyName=%d, required_name=%S",
		     found,
		     required_name));

    if (returned_name && *returned_name) {
	DPRINT(Debug,10,(&Debug,", *returned_name=%S",
			 *returned_name));	 
    }
    DPRINT(Debug,10,(&Debug,"\n"));

    return found;
}

struct stream_type TLS_STREAM = {
    SS_type_magic,
    "TLS",
    ss_ReadFromTLS,
    ss_WriteToTLS,
    ss_FreeTLS,
    ss_TLSReadAction,
    ss_TLSWriteAction,
    ss_TLSSchedule,
    ss_SetupTLSAction,
    ss_ShutdownTLSAction,
    ss_TLSInfo,
    ss_TLSVerifyName
};

union stream_types create_TLS_stream(tls)
     enum tls_version tls;
{
    union stream_types ret;
    struct shared_type_1 * S = safe_malloc(sizeof (struct shared_type_1));

    /* bzero is defined on hdrs/defs.h */
    bzero((void *)S,sizeof (struct shared_type_1));

    S->type = &TLS_STREAM;

    S->pbioRead  = NULL;
    S->pbioWrite = NULL;
    S->pSSL      = NULL;
    S->peer_certificate = NULL;
    S->verify_result = 0;
    S->wants     = 0;
    S->noread    = 1;
    S->state     = tls_not_connected;

    DPRINT(Debug,10,(&Debug,			    
		    "create_TLS_stream: version: %d",tls));
    switch(tls) {
    case tls_none: DPRINT(Debug,10,(&Debug, " tls_none")); break;
    case v_ssl:    DPRINT(Debug,10,(&Debug, " v_ssl"));    break;
    case v_tls:    DPRINT(Debug,10,(&Debug, " v_tls"));    break;
    case v_tls1:   DPRINT(Debug,10,(&Debug, " v_tls1"));   break;
    case v_tls1_1: DPRINT(Debug,10,(&Debug, " v_tls1_1")); break;
    case v_tls1_2: DPRINT(Debug,10,(&Debug, " v_tls1_2")); break;
    case NUM_tls_version: DPRINT(Debug,10,(&Debug, " NUM_tls_version")); break;
    }
    DPRINT(Debug,10,(&Debug,"\n"));

    
    zero_Read_Buffer( &(S->read_buffer));
    zero_Write_Buffer( &(S->write_buffer));

    if (tls >= 0 && tls < NUM_tls_version) {

	if (!tls_xx_pCtx[tls]) {
	    DPRINT(Debug,4,(&Debug,			    
			    "tls: create_TLS_stream: %s not available\n",
			    TLS_VERSION[tls]));
	    goto failure;
	}
	S->pSSL = SSL_new(tls_xx_pCtx[tls]);

	if (!S->pSSL) {
	    DPRINT(Debug,3,(&Debug,			    
			    "tls: create_TLS_stream: SSL_new failed (%s)\n",
			    TLS_VERSION[tls]));	
	    goto failure;
	}
    } else {
	DPRINT(Debug,3,(&Debug,			    
			"tls: create_TLS_stream: tls=%d bad\n",tls));
	goto failure;
    }

	  
    DPRINT(Debug,15,(&Debug,			    
		     "tls: create_TLS_stream: SSL_new=%p:\n",
		     S->pSSL));


    S->pbioRead = BIO_new(BIO_s_mem());

    if (!S->pbioRead) {
	DPRINT(Debug,3,(&Debug,			    
			"tls: create_TLS_stream: BIO_new failed\n"));
	       
	goto failure;
    }
    BIO_set_mem_eof_return(S->pbioRead,-1);   /* empty not EOF */

    S->pbioWrite = BIO_new(BIO_s_mem());

    if (!S->pbioWrite) {
	DPRINT(Debug,3,(&Debug,			    
			"tls: create_TLS_stream: BIO_new failed\n"));
	
	BIO_free(S->pbioRead);

	goto failure;
    }

    /* After this SSL_free() also frees BIOs */
    SSL_set_bio(S->pSSL,S->pbioRead,S->pbioWrite);


    ret.generic_1 = S;

    return ret;

 failure:

    if (S->pSSL)
	SSL_free(S->pSSL);
    S->pSSL = NULL;
    
    S->pbioRead  = NULL;
    S->pbioWrite = NULL;

    free(S);
    
    ret.TYPE = NULL;
    return ret;
}

enum tls_connection_state ss_TLS_state(ss,stack_idx)
     struct streamsched *ss;
     int stack_idx;
{
    DPRINT(Debug,15,(&Debug,			    
		     "tls: ss_TLS_state: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack)
	panic("TLS PANIC",__FILE__,__LINE__,"ss_TLS_state",
	      "Bad stack index or stack",0);

    if (!ss->this_stack[stack_idx].generic_1)
	return tls_closed;

    if (ss->this_stack[stack_idx].generic_1->type != &TLS_STREAM)
	panic("TLS PANIC",__FILE__,__LINE__,"ss_TLS_state",
	      "Bad stack index or stack",0);
    
    return ss->this_stack[stack_idx].generic_1->state;
}

enum tls_peer_verify ss_TLS_verify(ss,stack_idx,errormsg,peer_name) 
     struct streamsched *ss;
     int stack_idx;
     struct string **errormsg;
     struct string **peer_name;
{
    enum tls_peer_verify ret = tls_not_verified;

    DPRINT(Debug,15,(&Debug,			    
		     "tls: ss_TLS_verify: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    if (errormsg)
	*errormsg = NULL;
    if (peer_name)
	*peer_name = NULL;
    
    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack)
	panic("TLS PANIC",__FILE__,__LINE__,"ss_TLS_verify",
	      "Bad stack index or stack",0);

    if (!ss->this_stack[stack_idx].generic_1)
	return tls_not_verified;

    if (ss->this_stack[stack_idx].generic_1->type != &TLS_STREAM)
	panic("TLS PANIC",__FILE__,__LINE__,"ss_TLS_verify",
	      "Bad stack index or stack",0);
  
    if (ss->this_stack[stack_idx].generic_1->peer_certificate) {

	X509_NAME *peer_subject = 
	    X509_get_subject_name(ss->this_stack[stack_idx].generic_1->peer_certificate);

	/* XXXX */

	switch(ss->this_stack[stack_idx].generic_1->verify_result) {
	default:         ret = 	tls_verify_failed;

	    break;
	case X509_V_OK:  ret = 	tls_peer_verified; 

	    DPRINT(Debug,6,(&Debug,			    
			    "tls: ss_TLS_verify: verify OK\n"));
	    
	    break;	    

	case X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT:
	    ret = tls_not_verified;
	    if (errormsg)
		*errormsg = format_string(CATGETS(elm_msg_cat, TlsSet,
						  TlsSelfSignedCert,
						  "Self signed certificate"));
	    break;

	case X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT:
	case X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY:
	case X509_V_ERR_SELF_SIGNED_CERT_IN_CHAIN:
	case X509_V_ERR_CERT_UNTRUSTED:
	    ret = tls_not_verified;
	    if (errormsg)
		*errormsg = format_string(CATGETS(elm_msg_cat, TlsSet,
						  TlsCertIssuerNotTrusted,
						  "Certificate issuer not trusted"));
	    break;

	case X509_V_ERR_CERT_REVOKED:
	    ret = 	tls_verify_failed;
	    if (errormsg)
		*errormsg = format_string(CATGETS(elm_msg_cat, TlsSet,
						  TlsCertRevoked,
						  "Certificate revoked!"));
	    break;	    	    
	}

	if (peer_subject && peer_name) {
	    /* Pick first common name */

	    int cn_idx = X509_NAME_get_index_by_NID(peer_subject, NID_commonName, -1);
	    X509_NAME_ENTRY * peer_subject_cn = NULL;
	    ASN1_STRING     * peer_subject_cn_asn1 = NULL;

	    int               peer_subject_cn_len  = -1;
	    unsigned char   * peer_subject_cn_utf8 = NULL;
	    
	    charset_t utf8_set = MIME_name_to_charset("UTF-8",0);

	    DPRINT(Debug,12,(&Debug,			    
			    "tls: ss_TLS_verify: peer certificate subject CN index = %d\n",
			    cn_idx));

	    if (cn_idx >= 0)
		peer_subject_cn = X509_NAME_get_entry(peer_subject,cn_idx);
	    
	    if (peer_subject_cn)
		peer_subject_cn_asn1 = X509_NAME_ENTRY_get_data(peer_subject_cn);

	    if (peer_subject_cn_asn1)
		peer_subject_cn_len = ASN1_STRING_to_UTF8(&peer_subject_cn_utf8,
							  peer_subject_cn_asn1);
	   
	    if (peer_subject_cn_utf8 && peer_subject_cn_len > 0 && utf8_set) {
		*peer_name = new_string(utf8_set);

		add_streambytes_to_string(*peer_name,peer_subject_cn_len,
					  peer_subject_cn_utf8,NULL);
	    }
	    
	    if (peer_subject_cn_utf8) 
		OPENSSL_free(peer_subject_cn_utf8);

	    

	}



    } else {

	DPRINT(Debug,6,(&Debug,			    
			"tls: ss_TLS_verify: No peer certificate\n"));

	ret = tls_not_verified;
	if (errormsg)
	    *errormsg = format_string(CATGETS(elm_msg_cat, TlsSet,
					      TlsNoCert,
					      "No certificate"));

    }


    if (errormsg && *errormsg) {
	DPRINT(Debug,6,(&Debug,			    
			"tls: ss_TLS_verify: error message: %S\n",
			*errormsg));
    }
	
    if (peer_name && *peer_name) {
	DPRINT(Debug,6,(&Debug,			    
			"tls: ss_TLS_verify: peer name: %S\n",
			*peer_name));

    }

    return ret;
}

S_(SE_option_parse   tls_parse_on_option)
static int tls_parse_on_option P_((struct     SE_option *X,
				   const char       *name, /* without prefix */
				   const char       * value));
static int tls_parse_on_option(X,name,value)
     struct SE_option *X;
     const char       *name;   /* without prefix */
     const char       * value;
{
    int ret = 1;

    if (X->type != &tls_options ||
	!X->value) {
	panic("TLS PANIC",__FILE__,__LINE__,"tls_parse_on_option",
	      "Bad option storage",0);
    }

    if (0 == istrcmp(name,"starttls-version")) {
	if (!value) {
	    lib_error(CATGETS(elm_msg_cat, TlsSet,
			      TlsOptionRequiresValue,
			      "TLS option %s:%s requires value"),
		      X->prefix,name);
	    ret = 0;
	} else if (0 == istrcmp("none",value)) {
	    X->value->v_starttls = tls_none;
	} else {
	    int found = 0;
	    
	    enum tls_version v;
	    
	    for (v = 0; v < NUM_tls_version; v++) {
		if (0 == istrcmp(tls_version_values[v],value)) {
		    X->value->v_starttls = v;
		    found = 1;
		}
	    }
	    
	    if (!found) {
		/* Error message ? */
		lib_error(CATGETS(elm_msg_cat, TlsSet,
				  TlsBadOptionValue,
				  "Bad TLS option %s:%s value %s"),
			  X->prefix,name,value);
		    ret = 0;
	    }	    
	}
	
	if (ret)
	    X->value->flags |= TLS_starttls;
    } else if (0 == istrcmp(name,"tls-version")) {
	if (!value) {
	    lib_error(CATGETS(elm_msg_cat, TlsSet,
			      TlsOptionRequiresValue,
			      "TLS option %s:%s requires value"),
		      X->prefix,name);
	    ret = 0;
	} else if (0 == istrcmp("none",value)) {
	    X->value->v_tls = tls_none;
	} else {
	    int found = 0;
	    
	    enum tls_version v;
	    
	    for (v = 0; v < NUM_tls_version; v++) {
		if (0 == istrcmp(tls_version_values[v],value)) {
		    X->value->v_tls = v;
		    found = 1;
		}
	    }	

	    if (!found) {
		/* Error message ? */
		lib_error(CATGETS(elm_msg_cat, TlsSet,
				  TlsBadOptionValue,
				  "Bad TLS option %s:%s value %s"),
			  X->prefix,name,value);
		ret = 0;
	    }
	}
	if (ret)
	    X->value->flags |= TLS_tls;


    } else {
	/* Error message ? */
	lib_error(CATGETS(elm_msg_cat, TlsSet,
			  TlsBadOption,
			  "Bad TLS option %s:%s"),
		  X->prefix,name);
	ret = 0;
    }
	
    return ret;
}

S_(SE_option_values  tls_give_values);
static char * tls_give_values P_((struct     SE_option *X, 
				  const char *prefix));
/* return: malloced string -- caller must free */
static  char * tls_give_values(X,prefix)
     struct     SE_option *X;
     const char *prefix;
{
    char * ret = NULL;
    char * Y = "???";

    if (X->type != &tls_options ||
	!X->value) {
	panic("TLS PANIC",__FILE__,__LINE__,"tls_give_values",
	      "Bad option storage",0);
    }

    if (0 != (X->value->flags & TLS_starttls)) {

	switch (X->value->v_starttls) {
	case tls_none: Y = "none"; break;
	case v_ssl:    
	case v_tls:
	case v_tls1:
	case v_tls1_1:
	case v_tls1_2:
	    if (X->value->v_starttls >= 0 &&
		X->value->v_starttls < NUM_tls_version) {
		Y = tls_version_values[X->value->v_starttls];
	    }
	    break;
	case NUM_tls_version: break;
	}
	
	ret = elm_message(FRM("%s:starttls-version=%s"),
			  prefix,Y);
    }
    
    if (0 != (X->value->flags & TLS_tls)) {
	char * temp;

	switch (X->value->v_tls) {
	case tls_none: Y = "none"; break;
	case v_ssl:    
	case v_tls:
	case v_tls1:
	case v_tls1_1:
	case v_tls1_2:
	    if (X->value->v_tls >= 0 &&
		X->value->v_tls < NUM_tls_version) {
		Y = tls_version_values[X->value->v_tls];
	    }
	    break;
	case NUM_tls_version: break;
	}

	temp = elm_message(FRM("%s:tls-version=%s"),
			   prefix,Y);

	if (!ret)
	    ret = temp;
	else {
	    ret = elm_message(FRM("%s; %s"),
			      ret,temp);
	    free(temp);
	}
    }

    return ret;
}

S_(SE_option_zero    tls_zero_options);
static  void tls_zero_options   P_((struct     SE_option *X));
static void tls_zero_options(X)
     struct     SE_option *X;
{
    X->value = safe_malloc(sizeof (struct SE_option_value));

    bzero((void *)X->value, sizeof (*(X->value)));

    X->value->v_starttls = v_tls;  /* Default: Use TLS */
    X->value->v_tls      = v_tls;  
    X->value->flags      = 0;      /* No options given */
}

S_(SE_option_free    tls_free_options);
static void  tls_free_options   P_((struct     SE_option *X));
static void tls_free_options(X)
     struct     SE_option *X;
{
    if (X->value) {
	free(X->value);
	X->value = NULL;
    }
}

S_(SE_option_merge   tls_merge_options)
static void tls_merge_options  P_((struct     SE_option *target, 
				   struct     SE_option *source));
static void tls_merge_options(target,source)
     struct     SE_option *target;
     struct     SE_option *source;
{
    if (source->value->flags & TLS_starttls &&
	/* Entries on mail.services are prosessed on reverse order,
	   so do not set v_starttls if is already set. That way
	   last setting on mail.services takes precedence
	*/
	! (target->value->flags & TLS_starttls)) {
	target->value->flags |= TLS_starttls;
	target->value->v_starttls = source->value->v_starttls;
    }

    if (source->value->flags & TLS_tls &&
	! (target->value->flags & TLS_tls)) {

	target->value->flags |= TLS_tls;	
	target->value->v_tls = source->value->v_tls;
    }
}


S_(SE_option_opdate tls_update_options)
/* precious options from target with same name are removed */
static void tls_update_options  P_((struct     SE_option *target, 
				    struct     SE_option *source));
static void tls_update_options(target,source)
     struct     SE_option *target;
     struct     SE_option *source;
{
    if (source->value->flags & TLS_starttls) {
	/* have starttls-version set */

	target->value->flags |= TLS_starttls;	
   
	target->value->v_starttls = source->value->v_starttls;
    }

    if (source->value->flags & TLS_tls) {
	/* have tls-version set */

	target->value->flags |= TLS_tls;	
   
	target->value->v_tls = source->value->v_tls;
    }

}

struct SE_option_type tls_options = {
    SE_option_t_magic,
    tls_parse_on_option,
    tls_give_values,
    tls_zero_options,
    tls_free_options,
    tls_merge_options,
    tls_update_options
};

static struct SE_option_type *  OPTION_LIST[] = {
    &tls_options
};

E_(provides_shared_SEOT_f provides_shared_SEOT)
struct SE_option_type ** provides_shared_SEOT P_((int *count, size_t *s_size));
struct SE_option_type ** provides_shared_SEOT(count,s_size)
     int *count; 
     size_t *s_size;
{
    *s_size = sizeof (*OPTION_LIST[0]);

    *count = (sizeof OPTION_LIST) / sizeof (OPTION_LIST[0]);

    return &(OPTION_LIST[0]);
}

static ZZZ_SAVE_TYPE save_info_data[] = {
    { "default-tls-version", ZZZ_DT_ENUM(&default_tls_version),  ZZZ_TAIL },
    { "rand-egd",    ZZZ_DT_ESTR(&rand_egd_e),                   ZZZ_TAIL },
    { "rand-file",   ZZZ_DT_ESTR(&rand_file_e),                  ZZZ_TAIL },

    { "trusted-ca-certificates-dir", ZZZ_DT_ESTR(&CApath_e),     ZZZ_TAIL },
    { "trusted-ca-certificates-file",ZZZ_DT_ESTR(&CAfile_e),     ZZZ_TAIL },
};

E_(provides_RC_options_f provides_RC_options2)
struct rc_save_info_rec * provides_RC_options2 
  P_((size_t *count, size_t *s_size));
struct rc_save_info_rec * provides_RC_options2(count,s_size)
     size_t *count; 
     size_t *s_size;
{
    *s_size = sizeof (save_info_data[0]);

    *count = (sizeof save_info_data) / *s_size;
    return (struct rc_save_info_rec *) save_info_data;
}

S_(free_rc_hook free_tls_module);

E_(RC_post_init_f RC_post_init2)
void RC_post_init2  P_((int *errors, int flag, const char * tag));
void RC_post_init2(errors, flag,tag)
     int *errors;
     int flag;
     const char * tag;
{
    DPRINT(Debug,4,(&Debug,"RC_post_init2 ... (tls)\n"));

    tls_init(errors);

    post_init_change_config(errors,flag,tag,
			    (struct rc_save_info_rec *) save_info_data,
			    (sizeof save_info_data) / 
			    sizeof (save_info_data[0]),
			    sizeof (save_info_data[0]));


    add_free_rc_hook(&free_tls_module);
}


static void change_use_tls P_((struct rc_save_info_rec   * rc_options,
			       const size_t                rc_option_count,
			       const char                * shared_tag));
static void change_use_tls(rc_options,rc_option_count,shared_tag)
     struct rc_save_info_rec  * rc_options;
     const size_t               rc_option_count;
     const char               * shared_tag;
{
    const char                * option_name = "use-tls";
    
    struct rc_save_info_rec * rec =
	rc_locate_option(rc_options,rc_option_count,
			 option_name,NULL);

    if (rec) {
		
	if (&rc_DT_FLAGS  == rec->dt_type) {

	    if (ison(rec->flags,FL_CHANGED)) {
		DPRINT(Debug,7,(&Debug,
				"change_use_tls: %s value is changed, not a default value\n",
				option_name));

	    }


	    if (isoff(rec->flags,FL_CHANGED) ||
		! rec->val.flags->absolute) {	   
		long v = FLAGVAL(use_tls_implicit);

#ifdef TLS_CHECK_HOST
		/* display-check-host make sense as default only
		   when X509_check_host() is supported
		*/
		v |= FLAGVAL(use_tls_display_host);
#endif		

		if (! rec->val.flags->absolute && ison(rec->flags,FL_CHANGED)) {
		    v &= ~(rec->val.flags->incremental_sub);

		    if (!v) {
			DPRINT(Debug,7,(&Debug,
					"change_use_tls: %s: Value was removed\n",
					rec->name));
			return;
		    }
		    
		}

		if (rec->val.flags->val_as_string) {  
		    DPRINT(Debug,7,(&Debug,
				    "change_use_tls: %s: string value: %s\n",
				    rec->name,
				    rec->val.flags->val_as_string));
		    
		    free(rec->val.flags->val_as_string);
		    rec->val.flags->val_as_string = NULL;
		}
		
		v &= rec->val.flags->allowed_flags;
		if (v) {
		    const char * s = give_dt_flag_as_str(rec->val.flags);
		    
		    DPRINT(Debug,7,(&Debug,
				    "change_use_tls: %s found: %s\n",
				    rec->name,s));
		    
		    rec->val.flags->val            |= v;  /* Preserve previous flags --
							     that is FLAGVAL(use_tls_starttls)
							  */
		    rec->val.flags->absolute        = 1;
		    rec->val.flags->incremental_add = 0;
		    rec->val.flags->incremental_sub = 0;

		    if (rec->val.flags->val_as_string) {
			/* give_dt_flag_as_str() sets this */
			
			free(rec->val.flags->val_as_string);
			rec->val.flags->val_as_string = NULL;
		    }

		    s = give_dt_flag_as_str(rec->val.flags);
		    DPRINT(Debug,7,(&Debug,
				    "change_use_tls: %s changed to: %s\n",
				    rec->name,s));
		    
		} else {
		    DPRINT(Debug,7,(&Debug,
				    "change_use_tls: %s: value not allowed\n",
				    option_name));
		}
	    }
	    
	} else {
	    DPRINT(Debug,7,(&Debug,
			    "change_use_tls: %s have unexpected type\n",
			    rec->name));
	}
    } else {
	    DPRINT(Debug,7,(&Debug,"change_use_tls: %s not found\n",
			    option_name));
    }
    

}

E_(RC_change_config_f RC_change_config)
void RC_change_config P_((int *errors, int flag,
			  const char * tag,
			  struct rc_save_info_rec * rc_options,
			  size_t  rc_option_count,
			  size_t rec_size,
			  const char * shared_tag));

void RC_change_config(errors,flag,tag,rc_options,rc_option_count,
		      rc_size,shared_tag)
     int *errors; 
     int flag;
     const char * tag;
     struct rc_save_info_rec * rc_options;
     size_t  rc_option_count;
     size_t rc_size;
     const char * shared_tag;
{
    
    if (rc_size != sizeof(rc_options[0])) {
	DPRINT(Debug,1,(&Debug,
			"RC_change_config: Bad rc_size %d (should be %d)\n",
			rc_size,sizeof(rc_options[0])));
	
	return;
    }

    DPRINT(Debug,7,(&Debug, "RC_change_config\n"));

    if (!tag) {
	change_use_tls(rc_options,rc_option_count,shared_tag);

    } else {
	DPRINT(Debug,7,(&Debug,
			"RC_change_config: %s config\n",tag));
    }
}


static void free_tls_module P_((void));
static void free_tls_module(void)
{
    enum tls_version v;
    
    DPRINT(Debug,10,(&Debug,"free_tls_module ... \n"));
    
    for (v = 0; v < NUM_tls_version; v++) {
	if (tls_xx_pCtx[v]) {
	    SSL_CTX_free(tls_xx_pCtx[v]);
	    tls_xx_pCtx[v] = NULL;
	}
    }
}


#endif

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
