static char rcsid[] = "@(#)$Id: fileblock.c,v 2.10 2021/07/07 16:28:00 hurtta Exp $";

/*****************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.10 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> (was hurtta+elm@posti.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************
 * Some (minor) code copied from src/mailmsg2.c. That file is following
 * copyright:
 *
 *  The Elm Mail System 
 *
 * 			Copyright (c) 1988-1992 USENET Community Trust
 * 			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

#include "def_block.h"
#include "s_elm.h"
#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"textblock");

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif

static unsigned char * s2us P_((char *str));
static unsigned char * s2us(str)
     char *str;
{
    return (unsigned char *)str;
}

#define BLOCK_file_magic	0xED02

struct fileblock {
    unsigned short        magic;     /* BLOCK_file_magic */


    FILE               * fh;
    char               * filename; 
    charset_t            cs;

};

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif


S_(tb_free_block tb_free_fileblock)
static void tb_free_fileblock P_((struct text_block *block));
static void tb_free_fileblock(block)
     struct text_block *block;
{
    if (BLOCK_file_magic != block->u.fileblock->magic)
	panic("TEXT BLOCK PANIC",__FILE__,__LINE__,"tb_free_fileblock",
              "Bad magic number",0);
	
    if (block->u.fileblock->fh) {
	fclose(block->u.fileblock->fh);
	block->u.fileblock->fh = NULL;
    }

    if (block->u.fileblock->filename) {
	free(block->u.fileblock->filename);
	block->u.fileblock->filename = NULL;
    }

    block->u.fileblock->cs       = NULL;

    block->u.fileblock->magic = 0;   /* Invalidate */
    free(block->u.fileblock);
    block->u.fileblock = NULL;
}


S_(tb_free_block tb_alloc_fileblock)
static void tb_alloc_fileblock P_((struct text_block *block));
static void tb_alloc_fileblock(block)
     struct text_block *block;
{
    block->u.fileblock = safe_malloc(sizeof (* block->u.fileblock));

    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)block->u.fileblock,sizeof (* block->u.fileblock));

    block->u.fileblock->magic = BLOCK_file_magic;

    block->u.fileblock->fh       = NULL;
    block->u.fileblock->filename = NULL;
    block->u.fileblock->cs       = NULL;

}

S_(tb_get_charset_from_block tb_get_charset_from_fileblock)
static charset_t tb_get_charset_from_fileblock P_((struct text_block *block));
static charset_t tb_get_charset_from_fileblock(block)
     struct text_block *block;
{
   if (BLOCK_file_magic != block->u.fileblock->magic)
	panic("TEXT BLOCK PANIC",__FILE__,__LINE__,"tb_get_charset_from_block",
              "Bad magic number",0);
	
   if (! block->u.fileblock->cs)
	panic("TEXT BLOCK PANIC",__FILE__,__LINE__,"tb_get_charset_from_block",
              "Charset not set",0);
       

   return block->u.fileblock->cs;
}

/* idx == 0 is first line 
   \n is not returned
*/
S_(get_line_from_block get_line_from_fileblock)
static struct string * get_line_from_fileblock P_((struct text_block *block, int *idx, int *errors));
static struct string * get_line_from_fileblock(block,idx,errors)
     struct text_block *block; 
     int *idx; 
     int *errors;
{
    char * buffer = NULL;
    int r;
    struct string *ret = NULL;

    if (BLOCK_file_magic != block->u.fileblock->magic)
	panic("TEXT BLOCK PANIC",__FILE__,__LINE__,"get_line_from_fileblock",
              "Bad magic number",0);
    
    if (0 == idx)
	rewind(block->u.fileblock->fh);

    r =  malloc_gets(&buffer, 10000, block->u.fileblock->fh);

    if (-1 == r) {
	lib_error(CATGETS(elm_msg_cat, MeSet,
			  MeTooLongLineAt,
			  "Too long line at %s"),
		  block->u.fileblock->filename);
	(*errors)++;
    } else if (-2 == r) {

	if (ferror(block->u.fileblock->fh)) {
	    int err = errno;

	    lib_error(CATGETS(elm_msg_cat, MeSet,
			      MeFailedtoReadFrom,
			      "Failed to read from %s: %s"),
		      block->u.fileblock->filename,
		      strerror(err));
	    (*errors)++;
	}

	/* Otherwise this is end of file */
    } else {
	int f = 0;
	int ERR = 0;
	
	ret = new_string(block->u.fileblock->cs);

	if (add_streambytes_to_string(ret,r,s2us(buffer),&ERR) != r) 
	    f = 1;
	else if (ERR > 0)
	    f = 1;

	if (f) {
	    lib_error(CATGETS(elm_msg_cat, MeSet,
			      MeBadCharReadFrom,
			      "Readed some bad characters from %s"),
		      block->u.fileblock->filename);

	    (*errors)++;
	}
	    
    }
    

    if (buffer)
	free(buffer);
    return ret;
}

S_(tb_get_filename_from_block tb_get_filename_from_fileblock)
static const char * tb_get_filename_from_fileblock
P_((struct text_block *block));
static const char * tb_get_filename_from_fileblock(block) 
     struct text_block *block;
{
    if (BLOCK_file_magic != block->u.fileblock->magic)
	panic("TEXT BLOCK PANIC",__FILE__,__LINE__,
	      "tb_get_filename_from_fileblock",
              "Bad magic number",0);

    return block->u.fileblock->filename;
}

static struct text_block_type FILEBLOCK = {
    BLOCK_type_magic,
    tb_free_fileblock,
    tb_alloc_fileblock,
    tb_get_charset_from_fileblock,
    get_line_from_fileblock,
    tb_get_filename_from_fileblock
};


struct text_block * block_from_filename(filename,cs,quote)
     const char * filename; 
     charset_t cs;
     int quote;
{
    struct text_block *ret = NULL;

    int err = can_open(filename,"r");
    FILE * fh;

    if (err) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCouldNotOpenFile,
			  "Could not open file %s."), 
		  filename);
	
	return NULL;
    }


    fh = fopen(filename,"r");
    if (!fh) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCouldNotOpenFile,
			  "Could not open file %s."), 
		  filename);
	
	return NULL;
    }

    ret = alloc_block(&FILEBLOCK);

    ret->u.fileblock->fh = fh;
    ret->u.fileblock->filename = safe_strdup(filename);

    ret->u.fileblock->cs = cs;

    ret->quote = quote;

    return ret;
}



struct text_block * block_from_stdin()
{
    charset_t cs = system_charset;

    struct text_block *ret = NULL;
    const char * tmp = give_dt_estr_as_str(&temp_dir_e,"tmpdir",NULL,NULL);
    char * fname;
    FILE * fh;
    int c;
    int err = 0;
    
    if (!tmp)
        return NULL;

    fname = elm_message(FRM("%stdblock-%d"),
                        tmp, getpid ());


    fh = safeopen_rdwr(fname,&err);
    if (!fh) {
	lib_error(CATGETS(elm_msg_cat, MeSet,
			  MeFailedCreateTmpFile,
			  "Failed to create temporary file %s: %s"),
		  fname,strerror(err));

	free(fname);
	return NULL;		  
    }

    unlink(fname);

    while (EOF != (c = getc(stdin)))
	putc(c,fh);

    if (ferror(stdin)) {
	int err = errno;
	lib_error(CATGETS(elm_msg_cat, MeSet,
			  MeFailedReadStdin,
			  "Failed to read from standard input: %s"),
		  strerror(err));

	free(fname);
	fclose(fh);

	return NULL;
    }

    if (EOF == fflush(fh) || ferror(fh)) {
	int err = errno;

	lib_error(CATGETS(elm_msg_cat, MeSet,
			  MeFailedWriteTmpFile,
			  "Failed to write to temporary file %s: %s"),
		  fname,strerror(err));
	free(fname);
	fclose(fh);
	return NULL;		  
    }

    rewind(fh);

    if (is_utf8_charset(cs)) {
	/* Check if it is really UTF-8 charset */

	int need_enc = needs_encoding(fh,NULL,1);

	if ((need_enc & HAVE_8BIT) &&
	    !(need_enc & HAVE_UTF8)) {
	    lib_error(CATGETS(elm_msg_cat, MeSet,
			      MeStdinNotUTF8,
			      "Standard input is not UTF-8 text, using UNKNOWN-8BIT instead."));
	    cs = MIME_name_to_charset("UNKNOWN-8BIT",1);
	}

    }
       
    ret = alloc_block(&FILEBLOCK);

    ret->u.fileblock->fh = fh;
    ret->u.fileblock->filename = fname;

    ret->u.fileblock->cs = cs;

    ret->quote = 0;

    return ret;
}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */

