static char rcsid[] = "@(#)$Id: builtin++.c,v 2.38 2023/12/13 16:55:32 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.38 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 *
 *  Initially written by: Michael Elkins <elkins@aero.org>, 1995
 *****************************************************************************/

#include "def_elm.h"
#include "pager.h"
#include "s_elm.h"

DEBUG_VAR(Debug,__FILE__,"ui");

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

static int search_it P_((struct stringbuffer * bout, int start_idx,
			 struct string * search_pattern));

static int search_it(bout,start_idx,search_pattern)
     struct stringbuffer * bout; 
     int start_idx;
     struct string * search_pattern;
{
    int pg_flags;
    int ret = -1;
    int idx = start_idx;

    DPRINT(Debug,10,(&Debug,"Builtin pager: Search pattern: %S\n",
		     search_pattern));

    while (idx < linecount_stringbuffer(bout) && ret < 0) {
	struct pager_lineext *lineext = NULL;
	struct string * buffer = 
	    get_line_from_stringbuffer(bout,idx,&pg_flags,NULL,NULL,
				       &lineext);

	DPRINT(Debug,10,(&Debug,"Builtin pager: Search line %d first%s part: %S\n",
			 idx,
			 lineext ? "" : " and last",
			 buffer));

	if (find_pattern_from_string(buffer,search_pattern,1))
	    ret = idx;



	while (lineext && ret < 0) {
	    struct string * buffer2 = 
		get_lineext_and_walk(&lineext,&pg_flags,NULL,NULL);

	    DPRINT(Debug,10,(&Debug,"Builtin pager: Search line %d next%s part: %S\n",
			     idx,
			     lineext ? "" : " and last",
			     buffer2));

	    if (find_pattern_from_string(buffer2,search_pattern,1))
		ret = idx;

	    free_string(&buffer2);
	}
	
	idx++;
	free_string(&buffer);
	if (lineext)
	    free_pager_lineext(&lineext);
    }
    DPRINT(Debug,3,
	   (&Debug,"Builtin pager: Search from line %d %s (%d)\n",
	    start_idx, 
	    ret >= 0 ? "SUCCEED" : "FAILED",
	    ret));
    return ret;
}

S_(helper_get_param_f builtin_get_param)
static struct pager_param_value * builtin_get_param P_((union pager_help_param   buffer,
							struct pager_range     * range,
							enum pager_param_opcode  opcode,
							int                * next_pos,
							int                * found_pos));
static struct pager_param_value * builtin_get_param(buffer,range,opcode,
						    next_pos,found_pos)
     union pager_help_param   buffer;
     struct pager_range     * range;
     enum pager_param_opcode  opcode;
     int                    * next_pos;
     int                    * found_pos;
{
    struct pager_param_value *ret = NULL;

    ret =  get_pager_param_from_stringbuffer(buffer.stringbuf,
					     range,opcode,
					     next_pos,found_pos);
    
    return ret;
}

S_(helper_get_line_f builtin_get_line)
static struct string * builtin_get_line P_((union pager_help_param buffer,
					    int                   idx,
					    int                 * pg_flags,
					    enum range_changed  * range_changed,
					    struct pager_range ** range,
					    struct pager_lineext  ** first_lineext));
static struct string * builtin_get_line(buffer,idx,pg_flags,range_changed,range,
					first_lineext)
     union pager_help_param   buffer;
     int                      idx;
     int                    * pg_flags;
     enum range_changed     * range_changed;
     struct pager_range    ** range;
     struct pager_lineext  ** first_lineext;
{
    struct string * ret = NULL;

    if (range_changed)
	*range_changed = rc_same_range;

    if (first_lineext && *first_lineext)
	free_pager_lineext(first_lineext);

    if (pg_flags)
	*pg_flags = 0;
    
    if (idx >= linecount_stringbuffer(buffer.stringbuf)) {

	if (range && *range) {
	    free_pager_range(range);
	    if (range_changed)
		*range_changed = rc_newline;
	}

	goto out;
    }

    ret = get_line_from_stringbuffer(buffer.stringbuf,
				     idx,pg_flags,
				     range_changed,range,
				     first_lineext);

    DPRINT(Debug,9,(&Debug,
		    "builtin_get_line: idx=%d got line%s\n",
		    idx,
		    (first_lineext  && *first_lineext) ?
		    ", have first_lineext" : ""));
    DEBUG_PRINT_STRING(Debug,9,
		       "builtin_get_line=",
		       "builtin_get_line>",
		       ret);
    
				     
 out:
    return ret;
}

int builtinplusplus (bout,pager_page, is_helper, pager_cmds,
		     have_index)
     struct stringbuffer * bout;
     struct pager_page * pager_page;
     int is_helper;
     struct elm_commands *pager_cmds;
     enum pager_prompt have_index;
{   
    int is_end = 0;

    /* Current line for outputting and position on it ... */
    struct string         * buffer    = NULL;
    struct string_alt     * alt_data  = NULL;
    struct span_helper_data    data        = NULL_span_helper_data;
    struct pager_helper_data   helper_data = NULL_pager_helper_data;
    int                ret = '\0';

    struct string * search_pattern = NULL;        /* Last seach string */
    struct pager_range  * pager_range = NULL;

    struct pg_wrap_info * info = new_pg_wrap_info();

    int     goto_idx = 0, goto_X = 0;

    union pager_help_param     bout_buffer;

    struct pager_lineext * goto_lineext = NULL;
    struct pager_lineext *lineext_top = NULL;  /* line extension on top of display */

    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)&data,        sizeof data);
    bzero((void *)&helper_data, sizeof helper_data);
    
    data        = NULL_span_helper_data;
    helper_data = NULL_pager_helper_data;

    
    data.idx                    = 0;
    data.lineext                = NULL;
    data.buffer_x               = 0;
    data.virtual_line           = 0;  /* Index for next pg_wrap_info */
    data.force_word_wrap        = 0;
    data.p_width                = pager_paragraph_width;
    data.wrap_indicator         = pager_indicate_wrapping ? 1 : 0;

    bout_buffer.stringbuf = bout;

    helper_data.pg_flags   = 0;

    clear_error();
    
    for (;;) {
	int virtual_line_top;	    
	int idx_top;             /* line number of top of display  */

	int ch;                  /* command key read */
	int LINES, COLUMNS;

	int buffer_len = -1;

	data.start_of_line = 1;
	data.cur_col = 0;  /* Number of characters on current line */
	data.mayclear = 1;

	helper_data.search_matches = -1;  /* First idx on page 
					    where search matches */
	helper_data.scroll_line = 0;

	if (goto_idx >= 0 && goto_X >= 0) {
	goto_page:

	    DPRINT(Debug,5,(&Debug, 
			    "builtin++ -- Going page idx = %d, X=%d\n",
			    goto_idx,goto_X));
	    if (buffer)
		free_string (&buffer);
	    if (alt_data)
		free_string_alt(&alt_data);

	    data.idx        = goto_idx;
	    data.buffer_x   = goto_X;
	    
	    if (data.lineext)
		free_pager_lineext(&data.lineext);

	    if (goto_lineext) {
		data.lineext = goto_lineext; goto_lineext = NULL;

		DPRINT(Debug,5,(&Debug, 
				"builtin++ -- buffer_x=%d, going to line extension on buffer line %d\n",
				data.buffer_x,data.idx));

		alt_data = get_lineext_alt_and_walk(&data.lineext,
						    &helper_data.pg_flags,
						    NULL,
						    &pager_range);

		buffer_len = -1;

	    } else {
		if (data.idx < linecount_stringbuffer(bout)) {

		    DPRINT(Debug,5,(&Debug, 
				    "builtin++ -- buffer_x=%d, going to buffer line %d\n",
				    data.buffer_x,data.idx));

		    buffer = get_line_from_stringbuffer(bout,data.idx,
							&helper_data.pg_flags,
							NULL,
							&pager_range,
							&data.lineext); 
		    buffer_len = string_len(buffer);
		} else {
		    DPRINT(Debug,5,(&Debug, 
				    "builtin++ -- No buffer line %d\n",data.idx));
		    buffer_len = -1;
		}
		
	    }

	    if (alt_data && !buffer) {
		enum string_alt_type alt_data_type = string_alt_none;
		const union string_alt_value alt_data_value
		    = get_string_alt_value(alt_data,&alt_data_type);

		switch (alt_data_type) {
		case string_alt_none:
		    DPRINT(Debug,5,(&Debug, 
				    "builtin++ -- alt data empty on buffer line %d%s\n",
				    data.idx,
				    data.lineext ? ", have line extension" : ""));

		    break;
		case string_alt_text:
		    buffer = dup_string(alt_data_value.text);
		    buffer_len = string_len(buffer);

		    DPRINT(Debug,5,(&Debug, 
				    "builtin++ -- alt data string (len = %d) on buffer line %d%s\n",
				    buffer_len,data.idx,
				    data.lineext ? ", have line extension" : ""));
		    
		    break;
		case string_alt_entity:
		    DPRINT(Debug,5,(&Debug, 
				    "builtin++ -- alt data entity on buffer line %d%s\n",
				    data.idx,
				    data.lineext ? ", have line extension" : ""));
		    
		    break;
		}
	    }

	    if (buffer) {
		if (data.buffer_x > buffer_len) {
		    DPRINT(Debug,5,(&Debug, 
				    "builtin++ -- No buffer_x=%d (len=%d) on buffer line %d%s\n",
				    data.buffer_x,buffer_len,data.idx,
				    data.lineext ? ", have line extension" : ""
				    ));
		} else if (data.buffer_x == buffer_len) {
		    DPRINT(Debug,5,(&Debug, 
				    "builtin++ -- buffer_x=%d at end of line %d%s\n",
				    data.buffer_x,data.idx,
				    data.lineext ? ", have line extension" : ""
				    ));
		}
	    }
		    	
	    goto_idx = -1;
	    goto_X   = -1;
	}

    new_page:
    resize_mark:

	/* Check these also when page is empty !!! */
	if (menu_resized(pager_page->root)) {

	    menu_get_sizes(pager_page->root, &LINES, &COLUMNS);

	    DPRINT(Debug,5,(&Debug, 
			    "builtin++ top -- root resized\n"));

	}

	if (menu_need_redraw(pager_page->root)) {
	    DPRINT(Debug,5,(&Debug, 
			    "builtin++ top -- root need redraw\n"));

	}

	idx_top = data.idx;       /* line number of top of display  */
	virtual_line_top = data.virtual_line;

	if (lineext_top)
	    free_pager_lineext(&lineext_top);
	lineext_top = data.lineext;  /* line extension on top of display */
	if (lineext_top)
	    inc_pager_lineext_refcount(lineext_top);

	DPRINT(Debug,10,(&Debug, 
			 "builtin++  new page:  top idx = %d, top virtual line = %d, buffer_x = %d\n",
			 idx_top,virtual_line_top,data.buffer_x));

	data.cur_line = 0;   /* Number of visible lines printed
				so far */
        helper_data.FF_seen = 0;
	helper_data.word_wrapped = 0;
	helper_data.preformatted = 0;
	helper_data.hard_wrapped = 0;

	data.joined_line = 0;
	data.p_start_COL = 0;
	data.cur_col     = 0;    /* On beginning of line */

	menu_ClearScreen(pager_page->root);

    cr_restart:                  /* so that we can handle CR specially --
				    ie. don't reset page variables 
				 */

	menu_get_sizes(pager_page->root, &LINES, &COLUMNS);

	DPRINT(Debug,5,(&Debug, 
			"builtin++ top  idx = %d, scroll_line =%d, cur_line = %d\n",
			data.idx, helper_data.scroll_line, data.cur_line));

	while ((data.idx < linecount_stringbuffer(bout)) &&
	       (data.cur_line < LINES-1 || 
		(helper_data.scroll_line && data.cur_line < LINES)
		) && !helper_data.FF_seen) {
	    
	    enum pager_helper_status r;

	    if (menu_resized(pager_page->root)) {
		menu_get_sizes(pager_page->root, &LINES, &COLUMNS);

		if (buffer)
		    free_string(&buffer);
		if (alt_data)
		    free_string_alt(&alt_data);
		
		data.idx          = idx_top;
		data.virtual_line = virtual_line_top;
		
		if (goto_pg_wrap_info(info,virtual_line_top,
				      &goto_idx,&goto_X,
				      &goto_lineext)) {
		    DPRINT(Debug,10,(&Debug, 
				     "builtin++  -- resized .. used wrap info, index=%d, X=%d\n",
				     goto_idx,goto_X));
		    
		    goto goto_page;
		} 	
		
		DPRINT(Debug,10,(&Debug, 
				 "builtin++  -- resized index=%d\n",data.idx));

		goto new_page;
	    }

	    if (menu_need_redraw(pager_page->root)) {

		if (buffer)
		    free_string(&buffer);
		if (alt_data)
		    free_string_alt(&alt_data);
		
		data.idx          = idx_top;
		data.virtual_line = virtual_line_top;

		if (goto_pg_wrap_info(info,virtual_line_top,
				      &goto_idx,&goto_X,
				      &goto_lineext)) {
		    DPRINT(Debug,10,(&Debug, 
				     "builtin++  -- redraw .. used wrap info, index=%d, X=%d\n",
				     goto_idx,goto_X));
		    
		    goto goto_page;
		} 

		DPRINT(Debug,10,(&Debug, 
				 "builtin++  -- redraw index=%d\n",data.idx));

		goto new_page;

	    }
    
	    DPRINT(Debug,9,(&Debug, 
			    "builtin++ before line  idx = %d (virtual %d) buffer_x=%d start_of_line=%d\n",
			    data.idx,data.virtual_line,data.buffer_x,
			    data.start_of_line));

	    r = pager_helper_process_line(pager_page->root,
					  builtin_get_line,
					  bout_buffer,
					  &data,info,&buffer,
					  &pager_range,
					  &helper_data,
					  &alt_data,
					  builtin_get_param);
	    
	    if (buffer)
		buffer_len = string_len(buffer);
	    else 
		buffer_len = -1;
	    
	    if (r < 0)  /* EOF */
		break;
	    if (!r)
		goto read_key;   /* FAILURE */
	    
	}

	pager_helper_end_of_page(pager_page->root,&data,
				 &helper_data,-1,1);

    reprompt:
	menu_StartXX(pager_page->root,pg_STANDOUT);
	
	if (data.idx >= linecount_stringbuffer(bout)) {
	    if (is_helper)
		menu_PutLineX (pager_page->root,LINES-1, 0, 
			       CATGETS(elm_msg_cat, ElmSet, ElmBuiltinHelpEnd,
				       "Press 'q' or <return> to return:"));

	    else {
		switch  (have_index) {
		case prompt_return_index:
		    menu_PutLineX (pager_page->root,LINES-1, 0, 
				   CATGETS(elm_msg_cat, ElmSet, ElmBuiltinCommandi,
					   "Command ('i' to return to index, '?' for help):"));
		    break;
		case prompt_quit_pager:
		    menu_PutLineX (pager_page->root,LINES-1, 0, 
				   CATGETS(elm_msg_cat, ElmSet, ElmBuiltinCommandq,
					   "Command ('q' to quit pager, '?' for help):"));
		    break;
		case prompt_quit_or_return_index:
		    menu_PutLineX (pager_page->root,LINES-1, 0, 
				   CATGETS(elm_msg_cat, ElmSet, ElmBuiltinCommandqi,
					   "Command ('q' to quit pager, 'i' to return to index, '?' for help):"));
		                            
		    break;
		}
	    }
	    is_end = 1;

	} else {

	    int relative;
	    enum user_level_v ul = 
		give_dt_enumerate_as_int(&user_level);

	    if (linecount_stringbuffer(bout) < 1)
		relative = 100;
	    else
		relative = 100 * data.idx / linecount_stringbuffer(bout);
	    
	    if (data.force_word_wrap && helper_data.word_wrapped) {
		menu_PutLineX(pager_page->root,LINES-1, 0, 
			      CATGETS(elm_msg_cat, ElmSet, ElmBuiltInWordWrapped,
				      "Word wrapped (line %d-%d/%d, you've seen %d%%). Press 'W' to disable:"),
			      idx_top+1,data.idx,linecount_stringbuffer(bout),
			      relative);	      
	    } else if (data.force_word_wrap && helper_data.preformatted) {
		menu_PutLineX(pager_page->root,LINES-1, 0, 
			      CATGETS(elm_msg_cat, ElmSet, ElmBuiltInPreformatted,
				      "Preformatted (line %d-%d/%d, you've seen %d%%). Press 'W' to mute:"),
			      idx_top+1,data.idx,linecount_stringbuffer(bout),
			      relative);	      		
	    } else if (0 == data.p_width && pager_paragraph_width && 
		       helper_data.word_wrapped) {
		menu_PutLineX(pager_page->root,LINES-1, 0, 
			      CATGETS(elm_msg_cat, ElmSet, ElmBuiltInFullWidth,
				      "Full width (line %d-%d/%d, you've seen %d%%). Press 'w' to mute:"),
			      idx_top+1,data.idx,linecount_stringbuffer(bout),
			      relative);
	    } else if (is_helper) {
		menu_PutLineX(pager_page->root,
			      LINES-1, 0,CATGETS(elm_msg_cat, ElmSet, ElmBuiltHelpMore,
						 "Press <space> to continue, 'q' to return:"));
	    } else if (ul == user_level_beginner) {
		if (helper_data.hard_wrapped) {
		    struct string * temp = NULL;

		    if (data.wrap_indicator)
			temp =
			    format_string(CATGETS(elm_msg_cat, ElmSet, ElmBuiltInWrap0Hide,
 "Wrapped (line %d-%d/%d, you've seen %d%%). Press '\\' to hide indicator or <space> for more:"),
					  idx_top+1,data.idx,linecount_stringbuffer(bout),
					  relative);	  
		    else
			temp =
			    format_string(CATGETS(elm_msg_cat, ElmSet, ElmBuiltInWrap0Show,
 "Wrapped (line %d-%d/%d, you've seen %d%%). Press '\\' to show indicator or <space> for more:"), 		     
					  idx_top+1,data.idx,linecount_stringbuffer(bout),
					  relative);		  

		    /* Does not take account double wide characters! */
		    if (string_len(temp) < COLUMNS -5) 
			menu_PutLineX(pager_page->root,LINES-1, 0,
				      FRM("%S"),temp);
		    else
			menu_PutLineX(pager_page->root,LINES-1, 0, 
				      CATGETS(elm_msg_cat, ElmSet, ElmBuiltInWrapped0,
					      "Wrapped (line %d-%d/%d, you've seen %d%%). Press <space> for more:"), 
				      idx_top+1,data.idx,linecount_stringbuffer(bout),
				      relative);

		    free_string(&temp);
		} else 
		    menu_PutLineX(pager_page->root,LINES-1, 0, 
				  CATGETS(elm_msg_cat, ElmSet, ElmBuiltInMore0,
					  "MORE (line %d-%d/%d, you've seen %d%%). Press <space> for more:"), 
				  idx_top+1,data.idx,linecount_stringbuffer(bout),
				  relative);
	    } else if (helper_data.hard_wrapped) {
		if (data.wrap_indicator) 
		    menu_PutLineX(pager_page->root,LINES-1, 0, 
				  CATGETS(elm_msg_cat, ElmSet, ElmBuiltInWrappedHide,
					  "Wrapped (line %d-%d/%d, you've seen %d%%). Press '\\' to hide indicator:"),
				  idx_top+1,data.idx,linecount_stringbuffer(bout),
				  relative);	      
		else
		    menu_PutLineX(pager_page->root,LINES-1, 0, 
				  CATGETS(elm_msg_cat, ElmSet, ElmBuiltInWrappedShow,
					  "Wrapped (line %d-%d/%d, you've seen %d%%). Press '\\' to show indicator:"),
				  idx_top+1,data.idx,linecount_stringbuffer(bout),
				  relative);	      		    
	    } else
		menu_PutLineX(pager_page->root,
			      LINES-1, 0, 
			      CATGETS(elm_msg_cat, ElmSet, ElmBuiltInMore,
				      "MORE (line %d-%d/%d, you've seen %d%%):"), 
			      idx_top+1,data.idx,linecount_stringbuffer(bout),
			      relative);
	    is_end = 0;
	}
	menu_EndXX(pager_page->root,pg_STANDOUT);
	menu_CleartoEOS(pager_page->root);

    read_key:
	ch = menu_ReadCh(pager_page->root, REDRAW_MARK|READCH_CURSOR|READCH_resize|
			 READCH_sig_char);

	if (ch != '/') {
	    if (search_pattern) {
		free_string(&search_pattern);
	    }
	}

	switch (ch) {
	case TERMCH_interrupt_char:
	    ret = 0;
	    goto finish;

	case RESIZE_MARK:
	    DPRINT(Debug,4, (&Debug, "    ... resizing\n"));
	    data.idx          = idx_top;
	    if (data.lineext)
		free_pager_lineext(&data.lineext);
	    data.lineext      = lineext_top;
	    if (data.lineext)
		inc_pager_lineext_refcount(data.lineext);
	    data.virtual_line = virtual_line_top;

	    if (goto_pg_wrap_info(info,virtual_line_top,
				  &goto_idx,&goto_X,
				  &goto_lineext)) {
		DPRINT(Debug,10,(&Debug, 
				 "builtin++  -- resized .. used wrap info, index=%d, X=%d\n",
				 goto_idx,goto_X));
		
		goto goto_page;
	    } 
	    DPRINT(Debug,10,(&Debug, 
			     "builtin++  -- resized index=%d\n",data.idx));

	    /* Avoid incrementing of idx: */
	    if (buffer)
		free_string(&buffer);
	    if (alt_data)
		free_string_alt(&alt_data);

	    goto resize_mark;

	case ' ':
	case '+':
	case PAGEDOWN_MARK:
	    if (is_end) {
		ret = ' ';
		goto finish;
	    }
	break;

	case HOME_MARK:
	case '^': {     

	    data.idx = 0;
	    if (data.lineext)
		free_pager_lineext(&data.lineext);

	    data.virtual_line = 0;

	    /* Avoid incrementing of idx: */
	    if (buffer)    
		free_string (&buffer);
	    if (alt_data)
		free_string_alt(&alt_data);
	}
	break;

	case 'G': {    /* Go to last page --
	                  Not completely correct if multiline lines -
			  or folded paragraphs / lines
			*/

	    int old_idx = data.idx;
	    int tmp UNUSED_VAROK = data.virtual_line;
   
	    if (data.lineext)
		free_pager_lineext(&data.lineext);

	    data.idx = linecount_stringbuffer(bout)-LINES+1;
	    if (data.idx < 0)
		data.idx = 0;

	    data.virtual_line +=  (data.idx - old_idx);
	    if (data.virtual_line < 0)
		data.virtual_line = 0;

	    DPRINT(Debug,10,(&Debug, 
			     "builtin++  -- %04x virtual_line=%d > %d\n",
			     ch,tmp,data.virtual_line));

	    if (goto_pg_wrap_info(info,data.virtual_line,&goto_idx,&goto_X,
				  &goto_lineext)) {
		DPRINT(Debug,10,(&Debug, 
				 "builtin++  -- used wrap info, index=%d, X=%d\n",
				 goto_idx,goto_X));

		goto goto_page;
	    } 

	    DPRINT(Debug,10,(&Debug, 
			     "builtin++  -- calculated index=%d, buffer_x=%d\n",
			     data.idx,data.buffer_x));
	   
	    /* Avoid incrementing of idx: */
	    if (buffer)    
		free_string (&buffer);
	    if (alt_data)
		free_string_alt(&alt_data);

	    menu_ClearScreen(pager_page->root);
	}
	    break;
	case FIND_MARK:
	case '/':
	    menu_ClearLine (pager_page->root,LINES-1);
	    {
		int code;
		
		code = optionally_enter2(pager_page->root,
					 &search_pattern,
					 LINES-1, 0, 
					 OE_REDRAW_MARK|
					 OE_SIG_CHAR /* Ctrl-C */, 
					 CATGETS(elm_msg_cat, ElmSet, 
						 ElmBuiltinSearch,
						 "Search: "));

		if (REDRAW_MARK == code)
		    goto redraw;
		if (-1 == code) /* Ctrl-C */
		    goto reprompt;
		if (code != 0)
		    goto quit;
		if (search_pattern && string_len(search_pattern)) {
		    int start_idx;
		    /* Continue search from previous match */
		    if (helper_data.search_matches <0)
			start_idx = idx_top;
		    else if (helper_data.search_matches < linecount_stringbuffer(bout))
			start_idx = helper_data.search_matches+1;
		    else
			start_idx = idx_top;
		    helper_data.search_matches = search_it(bout,start_idx,
							   search_pattern);

		    if (helper_data.search_matches >= 0) {
			int old_idx = data.idx;
			int tmp UNUSED_VAROK = data.virtual_line;

			if (helper_data.search_matches > idx_top && 
			    helper_data.search_matches < idx_top + LINES -4) {
			    data.idx          = idx_top;
			    data.virtual_line = virtual_line_top;
			} else {
			    data.idx = helper_data.search_matches - LINES / 2;

			    data.virtual_line +=  (data.idx - old_idx);
			}
			/* virtual_line not calculated correctly */

			DPRINT(Debug,10,(&Debug, 
					 "builtin++  -- search, virtual_line=%d > %d\n",
					 tmp,data.virtual_line));


			if (data.lineext)
			    free_pager_lineext(&data.lineext);

			if (data.idx < 0)
			    data.idx = 0;
			if (data.virtual_line < 0)
			    data.virtual_line = 0;
			
			/* Avoid incrementing of idx: */
			if (buffer)    
			    free_string (&buffer);
			if (alt_data)
			    free_string_alt(&alt_data);
			goto new_page; /* and draw match */
		    } else {
			int relative;
			enum user_level_v ul = 
			    give_dt_enumerate_as_int(&user_level);

			if (linecount_stringbuffer(bout) < 1)
			    relative = 100;
			else
			    relative = 100 * data.idx / linecount_stringbuffer(bout);
			
			menu_ClearLine (pager_page->root,LINES-1);
			menu_StartXX(pager_page->root,pg_STANDOUT);
			if (ul == user_level_beginner)
			    menu_PutLineX (pager_page->root,LINES-1, 0, 
					   CATGETS(elm_msg_cat, ElmSet, ElmBuiltInNotFound0,
						   "NOT FOUND! (line %d-%d/%d, you've seen %d%%). Press <space> for more:"), 
				      idx_top+1,data.idx,
					   linecount_stringbuffer(bout),
					   relative);
			else
			    menu_PutLineX (pager_page->root,LINES-1, 0, 
					   CATGETS(elm_msg_cat, ElmSet, 
						   ElmBuiltInNotFound,
						   "NOT FOUND! (line %d-%d/%d, you've seen %d%%):"), 
					   idx_top+1,data.idx,
					   linecount_stringbuffer(bout),
					   relative);
			
			menu_EndXX(pager_page->root,pg_STANDOUT);
			goto read_key;
		    }
		} 
		goto reprompt;
	    }	    

	case PAGEUP_MARK:
	case '-': {
	    int tmp UNUSED_VAROK = data.virtual_line;

	    data.virtual_line = virtual_line_top - LINES+1;
	    if (data.virtual_line < 0)
		data.virtual_line = 0;

	    DPRINT(Debug,10,(&Debug, 
			     "builtin++  -- %04x virtual_line=%d > %d\n",
			     ch,tmp,data.virtual_line));

	    if (goto_pg_wrap_info(info,data.virtual_line,&goto_idx,&goto_X,
				  &goto_lineext)) {
		DPRINT(Debug,10,(&Debug, 
				 "builtin++  -- used wrap info, index=%d, X=%d\n",
				 goto_idx,goto_X));

		goto goto_page;	       
	    } 

	    if (data.lineext)
		free_pager_lineext(&data.lineext);

	    data.idx = idx_top - LINES+1;
	    if (data.idx < 0)
		data.idx = 0;

	    DPRINT(Debug,10,(&Debug, 
			     "builtin++  -- calculated index=%d, buffer_x=%d\n",
			     data.idx,data.buffer_x));
	    
	    /* Avoid incrementing of idx: */
	    if (buffer)
		free_string(&buffer);
	    if (alt_data)
		free_string_alt(&alt_data);
	}
	break;

	case ctrl('P'): {
	    int tmp UNUSED_VAROK = data.virtual_line;

	    data.virtual_line = virtual_line_top - 1;
	    if (data.virtual_line < 0)
		data.virtual_line = 0;

	    DPRINT(Debug,10,(&Debug, 
			     "builtin++  -- %04x virtual_line=%d > %d\n",
			     ch,tmp,data.virtual_line));
	    
	    if (goto_pg_wrap_info(info,data.virtual_line,&goto_idx,&goto_X,
				  &goto_lineext)) {
		DPRINT(Debug,10,(&Debug, 
				 "builtin++  -- used wrap info, index=%d, X=%d\n",
				 goto_idx,goto_X));
		
		goto goto_page;	       
	    } 

	    if (data.lineext)
		free_pager_lineext(&data.lineext);

	    data.idx = idx_top -1;
	    if (data.idx < 0)
		data.idx = 0;

	    DPRINT(Debug,10,(&Debug, 
			     "builtin++  -- calculated index=%d, buffer_x=%d\n",
			     data.idx,data.buffer_x));

	    /* Avoid incrementing of idx: */
	    if (buffer)
		free_string(&buffer);
	    if (alt_data)
		free_string_alt(&alt_data);
	}
	    break;

	case ctrl('D'):
	    if (is_end) 
		goto reprompt; /* Can't go down */

	    menu_Writechar(pager_page->root,'\r');		    
	    menu_CleartoEOLN(pager_page->root);	    

	    helper_data.scroll_line = LINES / 2;

	    idx_top += LINES / 2;
	    virtual_line_top += LINES / 2;

	    goto cr_restart; /* Hanlde Ctrl-D specially */
	    	   	    
	case ctrl('U'): {
	    int tmp UNUSED_VAROK = data.virtual_line;

	    data.virtual_line = virtual_line_top - LINES / 2;
	    if (data.virtual_line < 0)
		data.virtual_line = 0;
	    
	    DPRINT(Debug,10,(&Debug, 
			     "builtin++  -- %04x virtual_line=%d > %d\n",
			     ch,tmp,data.virtual_line));
	    
	    if (goto_pg_wrap_info(info,data.virtual_line,&goto_idx,&goto_X,
				  &goto_lineext)) {
		DPRINT(Debug,10,(&Debug, 
				 "builtin++  -- used wrap info, index=%d, X=%d\n",
				 goto_idx,goto_X));
		
		goto goto_page;	       
	    } 
	    
	    if (data.lineext)
		free_pager_lineext(&data.lineext);

	    data.idx = idx_top - LINES / 2;
	    if (data.idx < 0)
		data.idx = 0;

	    DPRINT(Debug,10,(&Debug, 
			     "builtin++  -- calculated index=%d, buffer_x=%d\n",
			     data.idx,data.buffer_x));

	    /* Avoid incrementing of idx: */
	    if (buffer)
		free_string(&buffer);
	    if (alt_data)
		free_string_alt(&alt_data);
	}
	    break;
	case 'W':	    
	    data.force_word_wrap = !data.force_word_wrap;

	    DPRINT(Debug,4,(&Debug, " ... %s word wrapping\n",
			    data.force_word_wrap ? "enabled" : "disabled"));
	    
	    /* Avoid incrementing of idx: */
	    if (buffer)
		free_string(&buffer);
	    if (alt_data)
		free_string_alt(&alt_data);

	    if (data.lineext)
		free_pager_lineext(&data.lineext);
	    data.lineext      = lineext_top;
	    if (data.lineext)
		inc_pager_lineext_refcount(data.lineext);

	    data.idx          = idx_top;
	    data.virtual_line = virtual_line_top;
	    goto resize_mark;

	case 'w':
	    if (data.p_width) 
		data.p_width  = 0;
	    else
		data.p_width  = pager_paragraph_width;
	    
	    DPRINT(Debug,4,(&Debug, " ... %s full width paragaphs\n",
			    data.p_width ? "disabled" : "enabled"));
	    
	    /* Avoid incrementing of idx: */
	    if (buffer)
		free_string(&buffer);
	    if (alt_data)
		free_string_alt(&alt_data);

	    if (data.lineext)
		free_pager_lineext(&data.lineext);
	    data.lineext      = lineext_top;
	    if (data.lineext)
		inc_pager_lineext_refcount(data.lineext);

	    data.idx          = idx_top;
	    data.virtual_line = virtual_line_top;
	    goto resize_mark;

	case '\\':
	    data.wrap_indicator = !data.wrap_indicator;

	    DPRINT(Debug,4,(&Debug, " ... %s wrap indicator\n",
			    data.wrap_indicator ? "disabled" : "enabled"));

	    /* Avoid incrementing of idx: */
	    if (buffer)
		free_string(&buffer);
	    if (alt_data)
		free_string_alt(&alt_data);

	    if (data.lineext)
		free_pager_lineext(&data.lineext);
	    data.lineext      = lineext_top;
	    if (data.lineext)
		inc_pager_lineext_refcount(data.lineext);

	    data.idx          = idx_top;
	    data.virtual_line = virtual_line_top;
	    goto resize_mark;

	case HELP_MARK:
	case '?': {

	    if (is_helper < 1) {
		struct elm_commands *cmds = 
		    have_index >= prompt_return_index ? 
		    give_builtinpager_commands() : 
		    give_builtinpg_core_commands();

		int ch;

	        if (pager_cmds) {
		    struct elm_commands *all_cmds =
			give_merged_commands(cmds,pager_cmds,
					     pg_BOLD,0);
		    
		    struct stringbuffer * b =
			give_commands_help(all_cmds);

		    ch = builtinplusplus(b,pager_page,is_helper+1,NULL, 
					 have_index /* ? */);

		    free_stringbuffer(&b);
		    free_commands(&all_cmds);
		} else {
		    struct stringbuffer * b =
			give_commands_help(cmds);

		    ch = builtinplusplus(b,pager_page,is_helper+1,NULL, 
					 have_index /* ? */);

		    free_stringbuffer(&b);
		}
		free_commands(&cmds);

		if (EOF == ch) {
		    ret = EOF;
		    goto finish;
		}
	    }	   
	}
	    /* FALLTHRU */
	redraw:
	case REDRAW_MARK:
	case ctrl('L'): 

	    if (data.lineext)
		free_pager_lineext(&data.lineext);
	    data.lineext      = lineext_top;
	    if (data.lineext)
		inc_pager_lineext_refcount(data.lineext);

	    data.idx          = idx_top;
	    data.virtual_line = virtual_line_top;

	    if (goto_pg_wrap_info(info,virtual_line_top,
				  &goto_idx,&goto_X,
				  &goto_lineext)) {
		DPRINT(Debug,10,(&Debug, 
				 "builtin++  -- redraw .. used wrap info, index=%d, X=%d\n",
				 goto_idx,goto_X));
		
		goto goto_page;
	    } 

	    DPRINT(Debug,10,(&Debug, 
			     "builtin++  -- redraw index=%d\n",data.idx));

	    if (buffer)
		free_string(&buffer);
	    if (alt_data)
		free_string_alt(&alt_data);
	    break;
	case '\n':
	case '\r':
	    if (helper_data.FF_seen)
		break;   /* New page */
	
	    menu_Writechar(pager_page->root,'\r');		    
	    menu_CleartoEOLN(pager_page->root);	    

	    if (is_helper && is_end) {
		ret = 0;
		goto finish;
	    }

	    helper_data.scroll_line = 1;
	    idx_top++;
	    virtual_line_top++;
	    data.virtual_line++;
	    goto cr_restart; /* Hanlde CR specially */
	    
	case EOF:
	    DPRINT(Debug,4,(&Debug, "builtin++  -- EOF\n"));
	    ret = EOF;
	    goto finish;

	case 'q':
	case 'x':
	quit:
	    DPRINT(Debug,4,(&Debug, "builtin++  -- quit\n"));
	    ret = 0;
	    goto finish;
	
	case 'i':
	    if (have_index >= prompt_return_index) {
		DPRINT(Debug,4,(&Debug, "builtin++  -- return to index\n"));
		ret = ch;
		goto finish;
	    }

	    /* FALLTHRU */
	default:
	    if (pager_cmds) {
		const struct elm_command * pager_cmd =
		    lookup_elm_command(pager_cmds,ch);

		if (pager_cmd) {
		    const struct string * S =
			elm_command_key_desc(pager_cmd);

		    DPRINT(Debug,4,(&Debug, "builtin++  -- returning command %d",
				    ch));
		    
		    if (S) {
			DPRINT(Debug,4,(&Debug, ", pager command: %S",S));
		    }

		    DPRINT(Debug,4,(&Debug, "\n"));

		    ret = ch;
		    goto finish;
		}

		DPRINT(Debug,4,(&Debug, "builtin++  -- bad command %d\n",
				ch));

		menu_Write_to_screen(pager_page->root,
				     FRM("%c??"), 07);
		FlushBuffer();
		if (sleepmsg > 0)
		    error_sleep((sleepmsg + 1) / 2);

		goto reprompt;
	    } else {
		DPRINT(Debug,4,(&Debug, "builtin++  -- returning command %d\n",
				ch));
		ret = ch;
		goto finish;
	    }
	    break;
	}
    }

 finish:

    if (lineext_top)
	free_pager_lineext(&lineext_top);

    clear_span_helper_data(&data);
    if (goto_lineext)
	free_pager_lineext(&goto_lineext);

    if (pager_range)
	free_pager_range(&pager_range);
    if (buffer)
	free_string(&buffer);
    if (alt_data)
	free_string_alt(&alt_data);
    if (info)
	free_pg_wrap_info(&info);

    DPRINT(Debug,9,(&Debug, 
		    "builtin++ returning %d",ret));
    if (isascii(ret) && isprint(ret)) {
	DPRINT(Debug,9,(&Debug, " (%c)",ret));
    }
    DPRINT(Debug,9,(&Debug, "\n"));
    return ret;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
