static char rcsid[] = "@(#)$Id: enter_helper.c,v 2.11 2022/07/14 14:16:01 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.11 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> (was hurtta+elm@posti.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_elm.h"
DEBUG_VAR(Debug,__FILE__,"ui");

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

const char * enter_mode_debug_name(em) 
    enum enter_mode em;
{
    switch (em) {
    case em_redraw: return "em_redraw";
    case em_enter:  return "em_enter";
    case em_prev:   return "em_prev";
    case em_next:   return "em_next";
    case em_wrap:   return "em_wrap";
    case em_redraw_initial: return "em_redraw_initial";
    case em_editor_escape:  return "em_editor_escape";
    case em_tabaction:      return "em_tabaction";
	
    case em_bs_prev:   return "em_bs_prev";
    case em_left:      return "em_left";
    case em_right:     return "em_right";
		  
    case em_page_up:   return "em_page_up";
    case em_page_down: return "em_page_down";

    case em_F1_key:    return "em_F1_key";
    case em_F2_key:    return "em_F2_key";
    case em_F3_key:    return "em_F3_key";
    case em_F4_key:    return "em_F4_key";
    case em_F5_key:    return "em_F5_key";
    case em_F6_key:    return "em_F6_key";
    case em_F7_key:    return "em_F7_key";
    case em_F8_key:    return "em_F8_key";
    case em_F9_key:    return "em_F9_key";
    case em_F10_key:   return "em_F10_key";
    case em_F11_key:   return "em_F11_key";
    case em_F12_key:   return "em_F12_key";

    }

    return NULL;
}	

void default_full_page(I,action,base_page)
    struct enter_info *I;
    enum full_page_action action;
    struct menu_context *base_page;
{
    if (ENTER_INFO_magic != I->magic)
        panic("INPUT PANIC",__FILE__,__LINE__,
              "default_full_page",
              "Bad magic number (enter_info)",0);
    
    DPRINT(Debug,9,(&Debug,
		    "default_full_page: action=%d",
		    action));

   switch (action) {
   case fp_resized:
       DPRINT(Debug,9,(&Debug," fp_resized"));
       break;
   case fp_redraw:
       DPRINT(Debug,9,(&Debug," fp_redraw"));
       break;
   }
   DPRINT(Debug,9,(&Debug,"\n"));

   switch (action) {
   case fp_resized:
       break;
   case fp_redraw:
       menu_ClearScreen(I->current_page);   /* Reset redraw flag */
       break;
   }
}

void default_alter_buffer(I,buffer,action,base_page)
     struct enter_info *I;
     struct string **buffer;
     enum alter_buffer_action action;
     struct menu_context *base_page;
{
    if (ENTER_INFO_magic != I->magic)
        panic("INPUT PANIC",__FILE__,__LINE__,
              "default_alter_buffer",
              "Bad magic number (enter_info)",0);
    
    DPRINT(Debug,9,(&Debug,
		    "default_alter_buffer: action=%d",
		    action));

    switch (action) {
    case ab_free_buffer:
	DPRINT(Debug,9,(&Debug," ab_free_buffer"));
	break;
	
    case ab_erase:
	DPRINT(Debug,9,(&Debug," ab_erase"));
	break;

    case ab_truncated:
	DPRINT(Debug,9,(&Debug," ab_truncated"));
	break;
	
    case ab_autocomma:
	DPRINT(Debug,9,(&Debug," ab_autocomma"));
    }
    if (*buffer) {
	DPRINT(Debug,9,(&Debug,", buffer %S",
			*buffer));
    }    
    DPRINT(Debug,9,(&Debug,"\n"));
    
    switch (action) {
	struct string * str;

    case ab_truncated:
    case ab_autocomma:
	break;
	
    case ab_free_buffer:
	free_string(buffer);
	break;
	
    case ab_erase:	
	str = new_string(get_string_type(*buffer));
	free_string(buffer);
	*buffer = str;
	
	break;
    }

}


void default_prompt_hint(I,phm,base_page)
     struct enter_info *I;
     enum prompt_hint_mode phm;
     struct menu_context *base_page;
{

    if (ENTER_INFO_magic != I->magic)
        panic("INPUT PANIC",__FILE__,__LINE__,
              "default_prompt_hint",
              "Bad magic number (enter_info)",0);

}


int enter_helper(info,prompt_area,base_page) 
     struct enter_info *info;
     struct menu_context *prompt_area;
     struct menu_context *base_page;
{
    int append_current, passwd, allowmime,
	redraw, alt_solidus, allowtab, use_editor_escape,
	tabaction, sig_char;
    struct string **buffer;
    int ret = 0;
    enum prompt_hint_state {
	no_prompt_hint = 0,
	prompt_hint_displayed = 1,
	prompt_hint_wait
    } have_prompt_hint = no_prompt_hint;
    int len;
    
    /* Only add comma on beginning of editing ... */
    int add_comma;

    if (ENTER_INFO_magic != info->magic)
        panic("INPUT PANIC",__FILE__,__LINE__,
              "enter_helper",
              "Bad magic number (enter_helper)",0);
   
    add_comma      = 0 != (info -> flags & OE_ADD_COMMA);
    
    info->ch_count = 0;
    buffer = info->give_buffer(info,em_redraw_initial, base_page);

    if (buffer && (!*buffer || 0 == (len = string_len(*buffer)))) {

	DPRINT(Debug,9,(&Debug,
			"enter_helper: Need wait for prompt hint\n"));
	
	have_prompt_hint = prompt_hint_wait;
	
    }

    
 restart:    /* give_buffer may modify flags .... */

    append_current = ison(info -> flags,OE_APPEND_CURRENT);
    passwd         = ison(info -> flags,OE_PASSWD);
    allowmime      = ison(info -> flags,OE_ALLOW_MIMEENC);
    redraw         = ison(info -> flags,OE_REDRAW_MARK) ? REDRAW_MARK : 0;
    alt_solidus       = ison(info -> flags,OE_ALT_SOLIDUS);
    use_editor_escape = ison(info -> flags,OE_EDITOR_ESCAPE);
    tabaction         = ison(info -> flags,OE_TABACTION);
    allowtab          = !tabaction;
    sig_char          = ison(info -> flags,OE_SIG_CHAR) ? READCH_sig_char : 0;

    DPRINT(Debug,4,
	   (&Debug, 
	    "== enter_helper: ch_count=%d %s%s%s%s%s%s%s%s%s%s\n",
	    info->ch_count,
	    append_current    ? "append_current " : "",
	    add_comma         ? "add_comma " : "",
	    passwd            ? "passwd " : "",
	    allowmime         ? "allowmime " : "",
	    redraw            ? "redraw " : "",
	    alt_solidus       ? "alt_solidus " : "",
	    use_editor_escape ? "use_editor_escape " : "",
	    allowtab          ? "allowtab " : "",
	    tabaction         ? "tabaction " : "",
	    sig_char          ? "sig_char "  : ""));

    while(buffer) {
	int escaped = OFF;

	do {
	    int wrap_column;	    
	    struct charset_state *ch = NULL;
	    int LINES,COLUMNS;
	    int lin,col;
	    int resize;
	    
	resize_mark:
	    resize = 0;

	    if (!info->current_page) {
		DPRINT(Debug,9,(&Debug,
				"enter_helper: menu_context erased .. resetting\n"));
		info->current_page = prompt_area;
	    }

	    menu_get_sizes(info->current_page, &LINES, &COLUMNS);

	    if (info->current_page != base_page &&
		info->current_page != prompt_area) {

		if (menu_resized(info->current_page)) {
		    
		    menu_get_sizes(info->current_page, &LINES, &COLUMNS);

		    DPRINT(Debug,9,(&Debug,
				    "enter_helper: resized LINES=%d COLUMNS=%d\n",
				    LINES,COLUMNS));

		    
		    info->full_page(info,fp_resized,base_page);

		    if (info->ch_count == 0) {
			buffer = info->give_buffer(info,em_redraw_initial,
						   base_page);

			if (!*buffer || 0 == (len = string_len(*buffer))) {
			    menu_GetXYLocation(info->current_page,&lin,&col);
			    
			    info->prompt_hint(info,phm_empty_buffer,base_page);
			    
			    have_prompt_hint = prompt_hint_displayed;
			
			    menu_MoveCursor(info->current_page,lin,col);
			}
		    } else
			buffer = info->give_buffer(info,em_redraw, base_page);
		    goto restart;

		} else if (menu_need_redraw(info->current_page)) {

		internal_redraw:
		    
		    DPRINT(Debug,9,(&Debug,
				    "enter_helper: page redraw\n"));

		    /* info->full_page() calls  
		       menu_ClearScreen(info->current_page); 
		    */
		    
		    info->full_page(info,fp_redraw,base_page);
		    
		    if (info->ch_count == 0) {
			buffer = info->give_buffer(info,em_redraw_initial,
						   base_page);

			if (!*buffer || 0 == (len = string_len(*buffer))) {
			    menu_GetXYLocation(info->current_page,&lin,&col);
			    
			    info->prompt_hint(info,phm_empty_buffer,base_page);
			    
			    have_prompt_hint = prompt_hint_displayed;
			
			    menu_MoveCursor(info->current_page,lin,col);
			}

			
		    } else
			buffer = info->give_buffer(info,em_redraw, base_page);
		    goto restart;

		}
	    } 

	    menu_GetXYLocation(info->current_page,&lin,&col);
	    
#ifdef BACKGROUD_PROCESSES      
	    if (handle_sigchld)
		sigchld_handler();
#endif

	    if (info->current_page != base_page &&
		info->current_page != prompt_area) {

		DPRINT(Debug,10,(&Debug, "enter_helper: asking REDRAW_MARK  and READCH_resize\n"));
		
		redraw = REDRAW_MARK;
		resize = READCH_resize;
	    }
	    
	    wrap_column = COLUMNS -5;
	    menu_MoveCursor(info->current_page,lin,col);

	    switch (have_prompt_hint) {
	    case prompt_hint_wait:
		DPRINT(Debug,10,(&Debug,
				 "enter_helper: peeking if there is key entered before printing prompt_hint\n"));
		ch = menu_ReadCh2(info->current_page,
				  redraw|READCH_CURSOR|resize|
				  (escaped ? 0 : 
				   READCH_term_char|sig_char)|
				  (escaped ? READCH_quote : 0)
				  |READCH_poll
				  );
		break;

	    default:
		ch = menu_ReadCh2(info->current_page,
				  redraw|READCH_CURSOR|resize|
				  (escaped ? 0 : 
				   READCH_term_char|sig_char)|
				  (escaped ? READCH_quote : 0));
		break;
	    }
	    
	    redraw         = ison(info -> flags,OE_REDRAW_MARK) ? REDRAW_MARK : 0;

	    switch (have_prompt_hint) {
	    case prompt_hint_displayed:
		DPRINT(Debug,10,(&Debug,
				 "enter_helper: erasing prompt hint\n"));
		info->prompt_hint(info,phm_clear,base_page);
		have_prompt_hint = no_prompt_hint;
		
		menu_MoveCursor(info->current_page,lin,col);
		break;
	    case no_prompt_hint:
	    case prompt_hint_wait:
		break;
	    }
	    
	reprocess_key:
	    if (!ch) {
		DPRINT(Debug,4,(&Debug, "-- enter_helper: EOF?\n"));
		/* we've hit EOF */
		
		ret = 1;
		goto out;	    
	    }
	    
	    if (get_state_caller_flags(ch)) {
		DPRINT(Debug,4,(&Debug, "-- enter_helper: flags: %d\n",
				get_state_caller_flags(ch)));
		add_comma = 0;
		
		switch(get_state_caller_flags(ch)) {
		    int len;
		    int tab_seen;
		case TIMEOUT_MARK:
		    	DPRINT(Debug,10,(&Debug,
					 "enter_helper: timeout on key peeking\n"));		    
		    if (have_prompt_hint > no_prompt_hint &&
			(!*buffer || 0 == (len = string_len(*buffer)))) {
			int lin,col;
			
			
			DPRINT(Debug,10,(&Debug,
					 "enter_helper: Displaying prompt_hint\n"));
			
			menu_GetXYLocation(info->current_page,&lin,&col);
			
			info->prompt_hint(info,phm_empty_buffer,base_page);
			
			have_prompt_hint = prompt_hint_displayed;
			
			menu_MoveCursor(info->current_page,lin,col);
		    } else {
			DPRINT(Debug,10,(&Debug,
					 "enter_helper: Ignoring  prompt_hint\n"));
			have_prompt_hint = no_prompt_hint;
		    }
			
		    goto resize_mark;
		    
		case RESIZE_MARK:
		    DPRINT(Debug,4,  (&Debug," ... resizing\n"));
		    goto resize_mark;

		case REDRAW_MARK:
		    if (info->current_page != base_page &&
			info->current_page != prompt_area) {
			
			DPRINT(Debug,4,  (&Debug," ... redrawing\n"));
			
			goto internal_redraw;
		    }

		    ret = REDRAW_MARK;
		    goto out;
		case TERMCH_eof_char:
		    /* we've hit EOF */
		    
		    ret = 1;
		    goto out;
		case TERMCH_backspace:		
		    if (!append_current && info->ch_count == 0 && *buffer) {
			info->alter_buffer(info,buffer,ab_erase,base_page);
			
			info->ch_count++;
			buffer = info->give_buffer(info,em_redraw, base_page);
			goto restart;
		    }

		    /* This is tricky. Here we are dealing with all situations
		     * under which a backspace (really whatever erase char is
		     * set to, not necessarily \b) erases the previous 
		     * character. It will erase unless escaped, because if 
		     * it's escaped it is taken literally. 
		     */
		    
		    escaped = OFF;
		    tab_seen = 0;
		    
		    if (!*buffer || 0 == ((len = string_len(*buffer)))) {
			/* Wrap to previous line if possible ... */
			buffer = info->give_buffer(info,em_bs_prev,base_page);
			if (!buffer)
			    goto restart;
		    }

		    if (*buffer && 0 < (len = string_len(*buffer))) {
			int X = 0;
			struct string * str = clip_from_string(*buffer,&X,
							       len-1);
			uint16 c = give_unicode_from_string(*buffer,len-1);
			
			if (0x0009 /* HT */  == c ||
			    /* FIXME:  We may need redraw that also if there is double
			               width characters -- following is only rough estimation
			               and not necessary always correct ....
			    */
			    0 != (CS_printable_len & 
				  charset_properties(get_string_type(*buffer))))
			    tab_seen++;
			else if(!passwd)
			    menu_Writechar(info->current_page,BACKSPACE);
			
			free_string(buffer);
			*buffer = str;
			
			info->alter_buffer(info,buffer,ab_truncated,base_page);
		    }
		    if(!passwd) {
			if (tab_seen) {   /* Redraw if TABulators or double width characters*/
			    buffer = info->give_buffer(info,em_redraw,
						       base_page);
			    goto restart;
			} else {
			    menu_Writechar(info->current_page,' ');
			    menu_Writechar(info->current_page,BACKSPACE);
			}
			FlushBuffer();
		    }
		    break;
		case TERMCH_word_erase:
		    if (!append_current && info->ch_count == 0 && *buffer) {

			info->alter_buffer(info,buffer,ab_erase,base_page);
			
			info->ch_count++;
			buffer = info->give_buffer(info,em_redraw,base_page);
			goto restart;
		    }
		    
		    if (!passwd) {	/* back up a word! */
			uint16 c;
			int X = 0;
			struct string * str;
			tab_seen = 0;
			
			if (!*buffer || 0 == (len = string_len(*buffer)))
			    continue;		/* no point staying here.. */
			len--;
			
			if (0x002F /* / */
			    == give_unicode_from_string(*buffer,len)) {
			    len--;
			    
			    menu_Writechar(info->current_page,BACKSPACE);
			    menu_Writechar(info->current_page,' ');
			    menu_Writechar(info->current_page,BACKSPACE);
			} else {			
			    while (len >= 0 && 
				   ( c = give_unicode_from_string(*buffer,
								  len)) &&
				   ( 0x0020 /* SPACE */ == c ||
				     0x0009 /* HT */    == c)) {
				len--;
				if (0x0009 /* HT */  == c)
				    tab_seen++;
				if (!tab_seen) {
				  menu_Writechar(info->current_page,BACKSPACE);
				  menu_Writechar(info->current_page,' ');
				  menu_Writechar(info->current_page,BACKSPACE);	
				}
			    }
			    
			    while (len >= 0 && 
				   ( c = give_unicode_from_string(*buffer,len)) &&
				   0x0020 /* SPACE */  != c &&
				   0x0009 /* HT */     != c &&
				   0x002F /* '/' */    != c) {
				
				len--;
				if (0x0009 /* HT */  == c) 
				    tab_seen++;
				if (!tab_seen) {
				  menu_Writechar(info->current_page,BACKSPACE);
				  menu_Writechar(info->current_page,' ');
				  menu_Writechar(info->current_page,BACKSPACE);	 
				}
			    }
			}
			str = clip_from_string(*buffer,&X,len+1);
			free_string(buffer);
			*buffer = str;
			
			info->alter_buffer(info,buffer,ab_truncated,base_page);
			
			if (tab_seen) {   /* Redraw if TABulators */
			    buffer = info->give_buffer(info,em_redraw,
						       base_page);
			    goto restart;
			} 
		    }
		case TERMCH_reprint_char:
		    if (info->ch_count == 0) {
			buffer = info->give_buffer(info,em_redraw_initial,
						   base_page);

			if (!*buffer || 0 == (len = string_len(*buffer))) {
			    menu_GetXYLocation(info->current_page,&lin,&col);
			    
			    info->prompt_hint(info,phm_empty_buffer,base_page);
			    
			    have_prompt_hint = 	prompt_hint_displayed;
			    
			    menu_MoveCursor(info->current_page,lin,col);
			}
			
		    } else
			buffer = info->give_buffer(info,em_redraw, base_page);
		    goto restart;
		case TERMCH_kill_line:
		    if (*buffer) {
			info->alter_buffer(info,buffer,ab_erase,base_page);
		    }
		    info->ch_count++;
		    buffer = info->give_buffer(info,em_redraw,base_page);
		    goto restart;
		case TERMCH_interrupt_char:
		    if (*buffer) {
			/* clean up string, and... */
			info->alter_buffer(info,buffer,ab_free_buffer,base_page);
		    }
		    
		    ret = -1;
		    goto out;
		case LEFT_MARK:
		    buffer = info->give_buffer(info,em_left,base_page);
		    goto restart;
		case UP_MARK:
		case DOWN_MARK:
		case PAGEUP_MARK:
		case PAGEDOWN_MARK:
		case RIGHT_MARK:
		    if (!append_current && info->ch_count == 0 && *buffer) {
			/* Goto end of buffer */
			info->ch_count++;
			buffer = info->give_buffer(info,em_redraw,base_page);
			goto restart;
		    }
		    switch(get_state_caller_flags(ch)) {
		    case UP_MARK:
			buffer = info->give_buffer(info,em_prev,base_page);
			goto restart;		
		    case DOWN_MARK:
			buffer = info->give_buffer(info,em_next,base_page);
			goto restart;		
		    case RIGHT_MARK:
			buffer = info->give_buffer(info,em_right,base_page);
                        goto restart;
		    case PAGEUP_MARK:
			buffer = info->give_buffer(info,em_page_up,base_page);
                        goto restart;
		    case PAGEDOWN_MARK:
			buffer = info->give_buffer(info,em_page_down,base_page);
                        goto restart;
		    }
		    break;

		case F1_KEY_MARK:
		  buffer = info->give_buffer(info,em_F1_key,base_page);
		  goto restart;
		case F2_KEY_MARK:
		  buffer = info->give_buffer(info,em_F2_key,base_page);
		  goto restart;
		case F3_KEY_MARK:
		  buffer = info->give_buffer(info,em_F3_key,base_page);
		  goto restart;
		case F4_KEY_MARK:
		  buffer = info->give_buffer(info,em_F4_key,base_page);
		  goto restart;
		case F5_KEY_MARK:
		  buffer = info->give_buffer(info,em_F5_key,base_page);
		  goto restart;
		case F6_KEY_MARK:
		  buffer = info->give_buffer(info,em_F6_key,base_page);
		  goto restart;
		case F7_KEY_MARK:
		  buffer = info->give_buffer(info,em_F7_key,base_page);
		  goto restart;
		case F8_KEY_MARK:
		  buffer = info->give_buffer(info,em_F8_key,base_page);
		  goto restart;
		case F9_KEY_MARK:
		  buffer = info->give_buffer(info,em_F9_key,base_page);
		  goto restart;
		case F10_KEY_MARK:
		  buffer = info->give_buffer(info,em_F10_key,base_page);
		  goto restart;
		case F11_KEY_MARK:
		  buffer = info->give_buffer(info,em_F11_key,base_page);
		  goto restart;
		case F12_KEY_MARK:
		  buffer = info->give_buffer(info,em_F12_key,base_page);
		  goto restart;
		}
	    } else if (use_editor_escape && 
		       state_is_onebyte(ch) == escape_char &&
		       info->ch_count == 0 &&
		       (!append_current || 
			!*buffer ||
			0 == string_len(*buffer))) {
		DPRINT(Debug,4,(&Debug,
				"-- enter_helper:  editor escape (%c)\n",
				escape_char));
		/* NOTE: em_editor_escape must NOT set OE_editor_escape */
		info -> flags &= ~OE_EDITOR_ESCAPE;
		buffer = info->give_buffer(info,em_editor_escape,base_page);
		goto restart;		
	    } else {
		uint16 code = give_unicode_from_state(ch);
		DPRINT(Debug,(passwd ? 51 : 4),
		       (&Debug,	    
			"-- enter_helper:  unicode=%04X\n",
					   code));
		if (info->ch_count++ == 0) {
		    if (code == 0x000A /* LF */ || 
			code == 0x000D /* CR */ ) {
			
			buffer = info->give_buffer(info,em_enter,base_page);
			goto restart;
		    } else if (alt_solidus && code == 0x002F /* '/' */) {
			ret = OE_ALT_SOLIDUS;
			goto out;
		    } else if (! append_current || !*buffer) {
			menu_CleartoEOLN(info->current_page);
			if (*buffer)
			    info->alter_buffer(info,buffer,ab_free_buffer,base_page);
			*buffer = new_string(get_state_charset(ch));
		    }
		}
		
		switch (code) {
		case 0x000A: /* LF */ 
		case 0x000D: /* CR */
		    buffer = info->give_buffer(info,em_enter,base_page);
		    goto restart;
		case 0x000C: /* Ctrl-L */ 
		    if (redraw) {
			buffer = info->give_buffer(info,em_redraw,base_page);
			ret = REDRAW_MARK;
			goto out;
		    }
		    goto other;
		case 0x0000: 
		    if (*buffer) {
			/* clean up string, and... */
			info->alter_buffer(info,buffer,ab_free_buffer,base_page);
		    }
		    
		    ret = -1;
		    goto out;
		case 0x003D:  /* '=' */
		    if (allowmime && !escaped && !passwd) {
			/* Allow user to PASTE mime encoded words 
			 * (not editing) 
			 */
			int len = string_len(*buffer);
			int ok = 1;
			int raw = '=';
			int x1,y1;
			int idx = 0;
			struct string * temp = NULL;
						
			/* MIME encoded word can be totally 75 characters 
			   ... */
			char raw_data[80];
			uint16 c;
			
			if (len > 0 &&
			    ( c = give_unicode_from_string(*buffer,len-1)) &&
			    0x0020 /* SPACE */  != c &&
			    0x0009 /* HT */     != c)
			    goto other;
			
			menu_GetXYLocation(info->current_page,&x1, &y1);
			menu_StartXX(info->current_page,pg_BOLD);
			while (ok) {
			    
			    if (idx > 78) {
				ok = 0;
				break;
			    }
			    raw_data[idx++] = raw;
			    menu_Writechar(info->current_page,raw);
			    
			    ch = menu_ReadCh2(info->current_page,
					      redraw|READCH_CURSOR|
					      READCH_term_char);
			    
			    if (get_state_caller_flags(ch)) {
				ok = 0;
				break;
			    }
			
			    raw = state_is_onebyte(ch);
			    DPRINT(Debug,4,(&Debug,
					    "-- enter_helper:  raw=%d (%c)\n",
					    raw,raw));
			    if (!raw) {
				ok = 0;
				break;
			    } else if ('\\' == raw) { 
				/* No escape in mime encoded word ... */
				ok = 0;
				break;
			    } else if (1 == idx && '?' != raw) {
				/* Not start with =? */
				ok = 0;
				break;
			    } else if (whitespace(raw) ||
				       '\r' == raw ||
				       '\n' == raw) {
				/* Terminating characters */
				break;
			    } else if (raw < ' ' || raw > 126) {
				ok = 0;
				break;
			    }
			}
			menu_EndXX(info->current_page,pg_BOLD);
			menu_MoveCursor(info->current_page,x1,y1);
			menu_CleartoEOLN(info->current_page);
			raw_data[idx] = '\0';
			
			DPRINT(Debug,4,(&Debug, 
					"-- enter_helper: ok=%d, encoded word=%s\n",
					ok,raw_data));
			
			temp = hdr_to_string(HDR_TEXT,raw_data,
					     display_charset,ok);
			append_string(buffer,temp,0);
			free_string(&temp);
			
			/* We redraw buffer on here so that it is
			   visible if catenate succeed or not ...
			*/
			buffer = info->give_buffer(info,em_redraw,base_page);
			if (!buffer)
			    goto restart;
			
			goto reprocess_key;
		    }
		    goto other;
		case 0x0009: /* HT */
		    if (tabaction) {
			DPRINT(Debug,4,(&Debug, "-- tabaction...\n"));
			buffer = info->give_buffer(info,em_tabaction,
						   base_page);
			if (!buffer)
			    goto restart;
			break;
		    }
		    goto other;
		default:
		other:
		    if (!*buffer)
			*buffer = new_string(get_state_charset(ch));
		    if (escaped) {  /* We erase last \ */
			int len = string_len(*buffer);
			int X = 0;
			struct string * str = 
			    clip_from_string(*buffer,&X,len-1);
			
			menu_Writechar(info->current_page,BACKSPACE);
			menu_Writechar(info->current_page,' ');
			menu_Writechar(info->current_page,BACKSPACE);
		     
			free_string(buffer);
			*buffer = str;
			
			info->alter_buffer(info,buffer,ab_truncated,base_page);
			
			escaped = OFF; 
		    } else if (code == 0x005C /* \ */ &&
			       !passwd) {
			escaped = ON;
		    } else if (add_comma &&
			       !passwd) {
			int line, col;
			int len;

			/* Add comma (,) after addresses on header
			   editing screen ... */
			
			menu_GetXYLocation(info->current_page,
					   &line, &col);
			
			if (col +1 >= COLUMNS) {
			    buffer = info->give_buffer(info,em_next,base_page);
			    if (!buffer)
				goto restart;
			}
			len = string_len(*buffer);

			/* If user typed comma we not need auto-add
			   another one
			*/
			if (0x002C /* ',' */    == code)
			    add_comma = 0;
			else if (len > 0) {
			    uint16 c,c2;

			    c = give_unicode_from_string(*buffer,len-1);
			    
			    /* Add comma before TAB  or
			       after '>' and ')' */
			    if ((0x0009 /* HT */    == code &&
				 0x0009 /* HT */    != c &&
				 0x0020 /* SPACE */ != c &&
				 0x002C /* ',' */   != c) ||
				(0x0020 /* SPACE */  != code &&
				 0x003E /* '>' */   == c) ||
				0x0029 /* ')' */   == c) {
				
			        add_ascii_to_string(*buffer,s2us(","));
				menu_Writechar(info->current_page,',');

				info->alter_buffer(info,buffer,ab_autocomma,base_page);
				
				/* Only add comma on beginning
				   of editing */
				add_comma = 0;				
			    } else if (len > 1 &&
				       (c2 = give_unicode_from_string(*buffer,
								     len-2)) &&
				       /* Add comma before last character */
				       0x002C /* ',' */   != c2 &&
				       0x0020 /* SPACE */ == c  &&
				       0x0028 /* '('  */ != code &&
				       0x003C /* '<'  */ != code) {
				int X = 0;
				struct string * str = 
				    clip_from_string(*buffer,&X,len-1);
				
				add_ascii_to_string(str,s2us(", "));
				free_string(buffer);
				*buffer = str;
				
				menu_Writechar(info->current_page,BACKSPACE);
				menu_Writechar(info->current_page,',');	  
				menu_Writechar(info->current_page,' ');	  

				info->alter_buffer(info,buffer,ab_autocomma,base_page);

				/* Only add comma on beginning
				   of editing */
				add_comma = 0;
			    } else if (0x0020 /* SPACE */ != code)
				add_comma = 0;
			}  
		    }
		    
		    /* Note that HT (0x0009) is not printable character
		       but space (0x0032) is printable character
		    */
		    if (0x0009 /* HT */    == code &&
			allowtab) {		    
			int line,col;

			menu_GetXYLocation(info->current_page,
					   &line, &col);
			
			if (!passwd &&
			    ((col / get_tabspacing() ) +1) * get_tabspacing()
			    >= COLUMNS) {
			    buffer = info->give_buffer(info,em_next,base_page);
			    if (!buffer)
				goto restart;
			}
			
			add_state_to_string(*buffer,ch);
			if (!passwd) {
			    menu_Write_to_screen(info->current_page,
						 FRM("%C"),ch);
			    
			    /* Call possible word wrapping routine 
			     */
			    menu_GetXYLocation(info->current_page,&line, &col);
			    if (col >= wrap_column) {
				buffer = info->give_buffer(info,em_wrap,
							   base_page);
				if (!buffer)
				    goto restart;
			    }
			}
		    } else if (!state_printable(ch) && !passwd) {
			/* non-printing character - warn with bell */
			menu_Writechar(info->current_page,'\007');	  
			
		    } else {
			int line, col;
			
			menu_GetXYLocation(info->current_page,
					   &line, &col);
			
			if (!passwd && col +1 >= COLUMNS) {
			    buffer = info->give_buffer(info,em_next,
						       base_page);
			    if (!buffer)
				goto restart;
			}
		    
			if (!*buffer)
			    *buffer = new_string(get_state_charset(ch));
			add_state_to_string(*buffer,ch);
			if (!passwd) {
			    menu_Write_to_screen(info->current_page,
						 FRM("%C"),ch);
			    
			    /* Call possible word wrapping routine */
			    menu_GetXYLocation(info->current_page,
					       &line, &col);
			    if (col >= wrap_column) {
				buffer = info->give_buffer(info,em_wrap,
							   base_page);
				if (!buffer)
				    goto restart;
			    }
			}
		    }
		}
	    }
	} while(1);
    }

 out:

    if (base_page != info->current_page &&
	prompt_area != info->current_page
	) {

	erase_menu_context(&  info->current_page);
	
	DPRINT(Debug,9,(&Debug,
			"enter_helper: erasing menu_context created by give_buffer()\n"));
	
	info->current_page = prompt_area;

	menu_trigger_redraw(base_page);

	if (base_page != prompt_area)
	    menu_trigger_redraw(prompt_area);
	
    }

    if (prompt_area != info->current_page) {
	DPRINT(Debug,9,(&Debug,
			"enter_helper: prompt_area was not current_page\n"));
	
	info->current_page = prompt_area;
	menu_trigger_redraw(prompt_area);
    }

    

    DPRINT(Debug,9,(&Debug,
		    "enter_helper=%d (returns)\n",ret));
    return ret;
}

void zero_enter_info(info) 
    struct enter_info *info;
{

    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)info,sizeof (*info));

    info->magic        = ENTER_INFO_magic;
    info->pvector      = NULL;
    info->give_buffer  = NULL;
    info->alter_buffer = default_alter_buffer;
    info->full_page    = default_full_page;
    info->prompt_hint  = default_prompt_hint;
    info->flags        = 0;
    info->ch_count     = 0;
    info->in_utils     = NULL;
    info->builtin      = NULL;
    info->browser      = NULL;
    info->address      = NULL;

    info->current_page = NULL;

}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
