static char rcsid[] = "@(#)$Id: fbrowser.c,v 2.31 2020/03/21 08:28:58 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.31 $   $State: Exp $
 *
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 * Partially based on Elm 2.5 src/fbrowser.c
 *
 *****************************************************************************/

#include "def_elm.h"
#include "s_fbrowser.h"

#include "fbrowser.h"
#include "browser.h"

#include <pwd.h>    /* getpwuid */


DEBUG_VAR(Debug,__FILE__,"ui");




#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif


static unsigned char * s2us P_((char *str));
static unsigned char * s2us(str)
     char *str;
{
    return (unsigned char *)str;
}


enum { fbrowser_mp_params,
       fbrowser_mp_COUNT };

#define FBROWSER_param_magic	0xF528

struct menu_anon_param {
    unsigned short magic;    /* FBROWSER_param_magic */
    
    struct string *title_prompt;

    struct string * current_dir;
    struct string * current_pattern;

    struct folder_browser *dir;

    int cur_page;
    int headers_per_page;
    int options;
    int time_len;
    int comment_column;

    int title_time_len;   /* time_len when title_page is drawn */
    
    struct dir_item {
	int num;

	struct string *time_s;
	
    }   * dir_items;   
    int   dir_items_len;

};


static void fbrowser_zero_menu_anon_param P_((struct menu_anon_param *A));
static void fbrowser_zero_menu_anon_param(A)
     struct menu_anon_param *A;
{
    A->magic = FBROWSER_param_magic;

    A->title_prompt     = NULL;
    A->current_dir      = NULL;
    A->current_pattern  = NULL;

    A->dir              = NULL;

    A->cur_page         = 0;
    A->headers_per_page = 1;
    A->options          = 0;
    A->time_len         = 10;
    A->comment_column   = 70;
    A->title_time_len   = 10;
    A->dir_items        = NULL;
    A->dir_items_len    = 0;
}

static void free_dir_items P_((struct menu_anon_param *A));

static void fbrowser_clear_menu_anon_param P_((struct menu_anon_param *A));
static void fbrowser_clear_menu_anon_param(A)
     struct menu_anon_param *A;
{
    if (FBROWSER_param_magic != A->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"fbrowser_clear_menu_anon_param",
	      "Bad magic number (menu_anon_param)",0);

    if (A->title_prompt)
	free_string(& (A->title_prompt));
    if (A->current_dir)
	free_string(& (A->current_dir));
    if (A->current_pattern)
	free_string(& (A->current_pattern));
    if (A->dir)
	free_dir(& (A->dir));

    A->cur_page         = 0;
    A->headers_per_page = 1;
    A->options          = 0;
    A->time_len         = 10;
    A->comment_column   = 70;
    A->title_time_len   = 10;
    
    free_dir_items(A);
    
}


static void make_menu_time P_((struct dir_item *item,
			       time_t tval));
static void make_menu_time(item,tval)
     struct dir_item *item;
     time_t tval;
{
    char timebuf[SLEN] = "";

    struct tm * timeptr = localtime(&tval);

    if (timeptr) {

	char * fbstr_entry_date_fmt = catgets(elm_msg_cat, FbrowserSet,
					      FbrowserEntryDateFmt,
					      "%Y-%b-%d");

	int 	nlen = strftime(timebuf,sizeof timebuf,fbstr_entry_date_fmt,timeptr);

	if (0 == nlen) {
	    /* strftime  failed */
	    
	    strfcpy(timebuf,"<?>",	sizeof timebuf);
	}
	
    } else {
	strfcpy(timebuf,"<?>",	sizeof timebuf);
    }

    if (item->time_s)
	free_string(& (item->time_s));

    /* system_charset is locale charset */
    item->time_s = new_string2(system_charset,s2us(timebuf));   
}


static void print_header_line P_((struct menu_context  * ptr,
				  int line,
				  int pgattr,
				  int title,
				  int arrow,
				  const struct fbentry * entry,
				  const struct string  * name_line,
				  const struct string  * comment,
				  struct menu_param    * list,
				  struct dir_item      * item
				  ));
static void print_header_line(ptr,line,pgattr,title,arrow,entry, name_line,comment,list,
			      item)
     struct menu_context  *ptr;
     int line;
     int pgattr;
     int title;
     int arrow;
     const struct fbentry * entry;
     const struct string  * name_line;
     const struct string  * comment;
     struct menu_param    * list;
     struct dir_item      * item;
{

    struct string * s_perm       = NULL;
    struct string * s_type       = NULL;
    struct string * s_owner      = NULL;
    struct string * s_date       = NULL;
    struct string * s_size       = NULL;
    struct string * s_filename = NULL;
    struct string * s_comment = NULL;

    struct string * s_dummy = new_string(display_charset);
        
    struct menu_anon_param *A = mp_lookup_anon(list,fbrowser_mp_params);
    int w;

    int LINES, COLUMNS;
    int lin,col;

    if (FBROWSER_param_magic != A->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"print_header_line",
	      "Bad magic number (menu_anon_param)",0);
    
    menu_get_sizes(ptr, &LINES, &COLUMNS);   
        
    if (title) {

	s_perm  = format_string(CATGETS(elm_msg_cat, FbrowserSet,
					FbrowserEnthdrPermission, "permission"));	
	s_type  = format_string(CATGETS(elm_msg_cat, FbrowserSet,
					FbrowserEnthdrType, "type"));	
	s_owner = format_string(CATGETS(elm_msg_cat, FbrowserSet,
					FbrowserEnthdrOwner, "owner"));
	s_size  = format_string(CATGETS(elm_msg_cat, FbrowserSet,
					FbrowserEnthdrSize, "size"));
	s_date  = format_string(CATGETS(elm_msg_cat, FbrowserSet,
					FbrowserEnthdrDate, "date"));

	DPRINT(Debug,14,(&Debug,
			 " ... print_header_line title time_len %d => %d\n",
			 A->title_time_len,A->time_len));
	
	A->title_time_len = A->time_len;
	
	s_filename = format_string(CATGETS(elm_msg_cat, FbrowserSet,
					   FbrowserEnthdrFilename, "filename"));
	s_comment  = format_string(CATGETS(elm_msg_cat, FbrowserSet,
					   FbrowserEnthdrComment,  "comment"));

    } else {

	if (entry) {
	    char type = '?';
	    char *mode1 = mode_to_str(entry->mode);
	    
	    
	    if (item) {
		if (! item->time_s) {
		    
		    make_menu_time(item, entry->mtime);
		}
		
		if (item->time_s) {	
		    int nlen;
		    
		    s_date = dup_string(item->time_s);
		    
		    nlen = string_len(s_date);
		    if (A->time_len < nlen) {
			
			DPRINT(Debug,14,(&Debug,
					 " ... print_header_line [%d] time_len %d => %d\n",
					 item->num,
					 A->time_len,nlen));
			
			A->time_len = nlen;
		    }
		}
	    }
	
	    
	    if (
#ifdef S_ISDIR
		S_ISDIR(entry->mode)
#else
		S_IFDIR == (entry->mode & S_IFMT)
#endif
		) {
		
		type = 'd';
		s_type = format_string(CATGETS(elm_msg_cat, FbrowserSet,
					       FbrowserEntryDir,"dir"));
		
		
	    } else if (
#ifdef S_ISREG
		       S_ISREG(entry->mode)
#else
		       S_IFREG == (entry->mode & S_IFMT)
#endif
		       ) {
		
		type = '-';
		s_type = format_string(CATGETS(elm_msg_cat, FbrowserSet,
					       FbrowserEntryFile,"file"));
	    }
	    
	    if (fb_show_protection) {
		
		struct passwd *pw = getpwuid(entry->uid); /* XXX fast_getpwuid */
		
		
		s_perm  = format_string(FRM("%c%s"),type,mode1);
		
		if (pw && pw->pw_name)
		    s_owner = new_string2(system_charset,s2us(pw->pw_name));
		else
		    s_owner = format_string(FRM("#%d"),entry->uid);
		
	    }
	    
	    s_size  = format_string(FRM("%8lu"),(unsigned long)entry->size);
	    
	    if (name_line)
		s_filename = dup_string(name_line);
	    else
		s_filename = dup_string(entry->display_name);
	    
	} else if (name_line)
	    s_filename = dup_string(name_line);

	if (comment)
	    s_comment = dup_string(comment);

    }

    menu_ClearLine(ptr,line);	    
    menu_MoveCursor(ptr,line,0);


    if (pgattr)
	menu_StartXX(ptr,pgattr);
	
    if (arrow)
	menu_PutLine0(ptr,line, 0, "->");
    else
	menu_PutLine0(ptr,line, 0, "  ");
    
    if (fb_show_protection)
	menu_Write_to_screen(ptr,FRM(" %-10.10S"),
			     s_perm ? s_perm : s_dummy);
    
    menu_Write_to_screen(ptr,FRM(" %-4.4S"),
			 s_type ? s_type : s_dummy);
    
    if (fb_show_protection)
	menu_Write_to_screen(ptr,FRM(" %-8.8S"),
			     s_owner ? s_owner : s_dummy);

    menu_Write_to_screen(ptr,FRM(" %8.8S"),
			 s_size ? s_size : s_dummy);

    menu_Write_to_screen(ptr,FRM(" %-*.*S "),
			 A->time_len, A->time_len,
			 s_date ? s_date : s_dummy);
    
    menu_GetXYLocation(ptr,&lin,&col);
    w = COLUMNS - col;
    
    if (w > 0) {
	int X = 0;
	int V;
	struct string *A1 = curses_printable_clip(s_filename ? s_filename : s_dummy,
						 &X,w,&V,w);

	menu_Write_to_screen(ptr,FRM("%S"),A1);	
	free_string(&A1);

	col += V;

	if (s_comment) {	    
	    int clen = string_len(s_comment);

	    if (title) {
		static const int FILENAME_LEN = 15;

		A->comment_column = COLUMNS * 2 / 3;
		if (A->comment_column <= col)
		    A->comment_column = col+1;

		if (V < FILENAME_LEN && 
		    (A->comment_column) - col < FILENAME_LEN-V && 
		    COLUMNS - (A->comment_column) > 20)
		    A->comment_column += FILENAME_LEN-V;	     
	    }

	    if (A->comment_column + clen < COLUMNS &&
		col < A->comment_column) {

		while (col < A->comment_column) {
		    menu_Writechar(ptr,' ');
		    col++;
		}
		
	    } else {
		menu_Writechar(ptr,' ');
		col++;
	    }

	    w = COLUMNS - col;
	    X = 0;
	    
	    if (w > 0) {
		A1 = curses_printable_clip(s_comment, &X,clen,&V,w);
		
		menu_Write_to_screen(ptr,FRM("%S"),A1);	
		free_string(&A1);
		
		col += V;
	    }
	}

	if (pgattr) {
	    while (col < COLUMNS) {
		menu_Writechar(ptr,' ');
		col++;
	    }
	}
    }

    if (pgattr)
	menu_EndXX(ptr,pgattr);

    menu_Writechar(ptr,'\r');
    menu_Writechar(ptr,'\n');

    free_string(& s_dummy);     

    if (s_comment)
	free_string(& s_comment);
    if (s_filename)
	free_string(& s_filename);
    if (s_date)
	free_string(& s_date);
    if (s_size)
	free_string(& s_size);
    if (s_owner)
	free_string(& s_owner);
    if (s_type)
	free_string(& s_type);
    if (s_perm)
	free_string(& s_perm);

}

/* Returns 1 if header_setup_line() need to be called 
*/
S_(header_setup_init fbrowser_header_init)
static int fbrowser_header_init  P_((struct menu_context  *ptr,
				     struct menu_param *list));
static int fbrowser_header_init(ptr,list)
     struct menu_context * ptr;
     struct menu_param   * list;
{
    struct menu_anon_param *A = mp_lookup_anon(list,fbrowser_mp_params);

    if (FBROWSER_param_magic != A->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"fbrowser_header_init",
	      "Bad magic number (menu_anon_param)",0);
    
    A->time_len          = 10;

    DPRINT(Debug,14,(&Debug,
		     "fbrowser_header_init time_len %d\n",
		     A->time_len));
    
    return 1;
}

/* Prescan formatting */
S_(header_setup_line fbrowser_init_header)
static void fbrowser_init_header P_((struct menu_context  *ptr,
				     struct menu_param *list,
				     int index));
static void fbrowser_init_header(ptr,list,index)
     struct menu_context * ptr;
     struct menu_param   * list;
     int index;
{
    struct menu_anon_param *A = mp_lookup_anon(list,fbrowser_mp_params);

    if (FBROWSER_param_magic != A->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"fbrowser_init_header",
	      "Bad magic number (menu_anon_param)",0);
    
    if (index >= 0 && index < A->dir_items_len) {

	int idx1                     = A->dir_items[index].num;

	const struct fbentry * entry        = fbrowser_give_entry(A->dir,idx1);

	if (entry) {
	    if (! A->dir_items[index].time_s)
		make_menu_time(& ( A->dir_items[index]),
			       entry->mtime);

	    if (A->dir_items[index].time_s) {

		int nlen = string_len(A->dir_items[index].time_s);
		if (A->time_len < nlen) {

		    DPRINT(Debug,14,(&Debug,
				     "fbrowser_init_header index %d [%d] time_len %d => %d\n",
				     index,
				     A->dir_items[index].num,
				     A->time_len,nlen));

		    A->time_len = nlen;
		}		
	    }
	    
	}
	
    }
}

S_(header_line_redraw fbrowser_show_header)
static void fbrowser_show_header P_((struct menu_context  *ptr,
				   struct menu_param *list,
				   int line_number,
				   int index,
				   int is_current));
static void fbrowser_show_header(ptr,list,line_number,index,is_current)
     struct menu_context  *ptr;
     struct menu_param *list;	   
     int line_number;
     int index;
     int is_current;
{
    struct menu_anon_param *A = mp_lookup_anon(list,fbrowser_mp_params);

    if (FBROWSER_param_magic != A->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"fbrowser_show_header",
	      "Bad magic number (menu_anon_param)",0);
    
    menu_ClearLine(ptr,line_number);
    menu_MoveCursor(ptr,line_number,0);
 
    if (index >= 0 && index < A->dir_items_len) {

	int idx1                     = A->dir_items[index].num;

	const struct string  * comment = NULL;
	const struct string  * name_line = give_line_dir(A->dir,idx1,NULL,&comment);
	const struct fbentry * entry        = fbrowser_give_entry(A->dir,idx1);

	DPRINT(Debug,14,(&Debug,
			 " ... fbrowser_show_header index %d [%d] time_len %d\n",
			 index,
			 idx1,
			 A->time_len));
	
	
	if (entry) 
	    print_header_line(ptr,
			      line_number,
			      has_highlighting && ! arrow_cursor && is_current ? pg_STANDOUT : 0,
			      0,
			      (arrow_cursor || !has_highlighting) && is_current,
			      entry,name_line,comment,list,
			      
			      & ( A->dir_items[index])
			      );
	else
	    goto fail;
	 

    } else {
    fail:
	if (has_highlighting && ! arrow_cursor) {
	    draw_dummy_hl(ptr,line_number,is_current);
	} else {
	    if (is_current)
		menu_PutLine0(ptr,line_number,0,"->");
	}
    }    
}

S_(header_line_redraw fbrowser_show_current)
static void fbrowser_show_current P_((struct menu_context  *ptr,
				    struct menu_param *list,
				    int line_number,
				    int index,
				    int is_current));
static void fbrowser_show_current(ptr,list,line_number,index,is_current)
     struct menu_context  *ptr;
     struct menu_param *list;	   
     int line_number;
     int index;
     int is_current;
{

    if (has_highlighting && ! arrow_cursor) {
	fbrowser_show_header(ptr,list,line_number,index,is_current);
    } else {
	if (!is_current)
	    menu_PutLine0(ptr,line_number,0,"  ");  /* remove old pointer... */
	else
	    menu_PutLine0(ptr,line_number,0,"->");    
    }
}

S_(subpage_simple_redraw sb_update_fbrowser_title)
static int sb_update_fbrowser_title(ptr,list)
     struct menu_context  *ptr;
     struct menu_param *list;
{
    struct menu_anon_param *A = mp_lookup_anon(list,fbrowser_mp_params);

    struct string * buffer = NULL;
    int max_page = 1;

    int LINES, COLUMNS;
    int lin,col;

    if (FBROWSER_param_magic != A->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"sb_update_fbrowser_title",
	      "Bad magic number (menu_anon_param)",0);
    
    if (A->dir_items_len > 0 &&
	A->headers_per_page > 0)
	max_page = (A->dir_items_len -1) / A->headers_per_page + 1;

    buffer = format_string(CATGETS(elm_msg_cat, FbrowserSet,
				   FbrowserHeaderPageOf,
				   "[page %d/%d]"),
			   A->cur_page,
			   max_page);
   
    menu_ClearScreen(ptr);
    menu_get_sizes(ptr, &LINES, &COLUMNS);   


    menu_PutLineX(ptr,0,COLUMNS-string_len(buffer),
		  FRM("%S"),buffer);
	
    menu_StartXX(ptr,pg_BOLD);

    menu_print_format_center(ptr,1,
			     FRM("%S [ELM %s]"),
			     A->title_prompt,
			     version_buff);
    
    menu_EndXX(ptr,pg_BOLD);


    menu_PutLineX(ptr,3,0,CATGETS(elm_msg_cat, FbrowserSet,
				  FbrowserHeaderDirectory, 
				  "Directory: %S"),
		  A->current_dir);
				   
    menu_GetXYLocation(ptr,&lin,&col);

    if (A->current_pattern) {

	struct string * buf = format_string(CATGETS(elm_msg_cat, FbrowserSet,
						    FbrowserHeaderPattern, 
						    "Pattern: %S"),
					    A->current_pattern);

	int len = string_len(buf);
	

	menu_PutLineX(ptr,col+len < COLUMNS ? 3 : 4,
		      COLUMNS - len,
		      FRM("%S"),buf);
	

	free_string(&buf);
    }


    if (fb_show_title) 
	print_header_line(ptr,5,pg_UNDERLINE,1,0,NULL,NULL,NULL,list,NULL);

    free_string(&buffer);

    return 1;   /* title completed */
}


S_(subpage_simple_redraw sb_update_fbrowser_menu)
static int sb_update_fbrowser_menu P_((struct menu_context  *ptr,
				     struct menu_param *list));
static int sb_update_fbrowser_menu(ptr,list)
     struct menu_context  *ptr;
     struct menu_param *list;
{
    enum user_level_v ul = 
	give_dt_enumerate_as_int(&user_level);

    menu_ClearScreen(ptr);

    if (ul == user_level_beginner) {

	if (has_highlighting && ! arrow_cursor)
	    menu_print_format_center(ptr,0,
				     CATGETS(elm_msg_cat, FbrowserSet,
					     FbrowserInstrDummy1,
					     "Move the highlighted selection:  j/k = down/up, +/- = down/up page"));
	else
	    menu_print_format_center(ptr,0,
				     CATGETS(elm_msg_cat, FbrowserSet,
					     FbrowserInstrDummy1A,
					     "Move the arrow selection:  j/k = down/up, +/- = down/up page"));
    
	menu_print_format_center(ptr,1,
				 CATGETS(elm_msg_cat, FbrowserSet,
					 FbrowserInstrDummy2,
					     "Press ENTER to make selection, ? for help, press \"/=~.\" to change directory"));

    	menu_print_format_center(ptr,2,
				 CATGETS(elm_msg_cat, FbrowserSet,
					 FbrowserInstDummy3,
					 "and enter name, or enter f)ilename, or q)uit."));

    } else {
	struct menu_anon_param *A = mp_lookup_anon(list,fbrowser_mp_params);

	if (FBROWSER_param_magic != A->magic)
	    panic("BROWSER PANIC",__FILE__,__LINE__,"sb_update_fbrowser_menu",
		  "Bad magic number (menu_anon_param)",0);
	
	DPRINT(Debug,14,(&Debug, 
			 "... options=%d (%s%s%s%s%s )\n",
			 A->options,
			 A->options ?                       "" : " none",
			 (A->options & FB_EXIST) ? " FB_EXIST" : "",
			 (A->options & FB_READ)  ? " FB_READ"  : "",
			 (A->options & FB_WRITE) ? " FB_WRITE" : "",
			 (A->options & FB_NOMBOX_CHECK)  ? " FB_NOMBOX_CHECK"  : ""));


	menu_print_format_center(ptr,0,
				 CATGETS(elm_msg_cat, FbrowserSet,
					 FbrowserInstNorm1,
					 "Use \"jk+-\" to move, \"/=~.\" to enter name, ENTER to select, "));
    
	
	if (0 == (A->options & FB_EXIST) &&
	    0 != (A->options & FB_WRITE))

	    menu_print_format_center(ptr,1,
				     CATGETS(elm_msg_cat, FbrowserSet,
					     FbrowserInstNorm2Direc,
					     "enter f)ilename, m)ake directory, or q)uit."));

	else
	    
	    menu_print_format_center(ptr,1,
				     CATGETS(elm_msg_cat, FbrowserSet,
					     FbrowserInstNorm2,
					     "enter f)ilename, or q)uit."));
    }

    return 1;
}

S_(subpage_simple_redraw sb_update_fbrowser_prompt)
static int sb_update_fbrowser_prompt P_((struct menu_context  *ptr,
				       struct menu_param *list));
static int sb_update_fbrowser_prompt(ptr,list)
     struct menu_context  *ptr;
     struct menu_param *list;
{    
    menu_ClearScreen(ptr);

    show_last_error();

    menu_PutLineX (ptr,0, 0, 
		   CATGETS(elm_msg_cat, FbrowserSet,
			   FbrowserMainPrompt, "Command: "));


    return 1;
}


/* 
 *  line 0                                 [pagenun] title_page
 *  line 1             title   prompt
 *  line 2
 *  line 3   current directory
 *  line 4
 *  line 5   header           ( fb_show_title ) 
 *  ...                                              selection_part
 *  ...
 *  line 0   help                                    menu_part
 *  line 1
    line 2
 *  line 3                     ( ul == 0 )
 *  line 0   prompt                                  prompt_page
 *  line 1
 *  line 2
 *  line 3
 */


static void update_pagenum P_((struct screen_parts *LOC,
			       struct menu_param *list));
static void update_pagenum(LOC,list)
     struct screen_parts *LOC;
     struct menu_param *list;
{
    struct menu_anon_param *A = mp_lookup_anon(list,fbrowser_mp_params);

    int last_cur_page;
    
    int top_line = menu_header_get(LOC->header_page,header_top_line);

    if (FBROWSER_param_magic != A->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"update_pagenum",
	      "Bad magic number (menu_anon_param)",0);

    last_cur_page         = A->cur_page;
	
    if (A->headers_per_page > 0)
	A->cur_page = top_line / A->headers_per_page +1;

    if (A->cur_page != last_cur_page) 
	menu_trigger_redraw(LOC->title_page);
    
}


static void change_page P_((struct screen_parts *LOC,
			    struct menu_param *list));
static void change_page(LOC,list)
     struct screen_parts *LOC;
     struct menu_param *list;
{
    int current = menu_header_get(LOC->header_page,header_current);
    struct menu_anon_param *A = mp_lookup_anon(list,fbrowser_mp_params);
    
    int last_cur_page;
    
    int top_line = menu_header_get(LOC->header_page,header_top_line);
    int last_top_line = top_line;

    if (FBROWSER_param_magic != A->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"change_page",
	      "Bad magic number (menu_anon_param)",0);

    last_cur_page = A->cur_page;
    
    if (A->headers_per_page > 0) {
	
	A->cur_page = current / A->headers_per_page +1;

	top_line = (A->cur_page -1) *  (A->headers_per_page);
    }

    if (A->cur_page != last_cur_page) {
	menu_trigger_redraw(LOC->title_page);
    }

    if (last_top_line != top_line)
	menu_header_change(LOC->header_page,header_top_line,top_line);
}




static void set_fbrowser_screen P_((struct menu_context  *page, 
				    struct screen_parts *LOC,
				    struct menu_param  *LIST));
static void set_fbrowser_screen(page,LOC,LIST)
     struct menu_context  *page;
     struct screen_parts *LOC;
     struct menu_param  *LIST;
{
    int   LINES, COLUMNS;
    struct menu_anon_param *A = mp_lookup_anon(LIST,fbrowser_mp_params);

    enum user_level_v ul = 
	give_dt_enumerate_as_int(&user_level);

    int title_len   = fb_show_title ? 6 : 5;
    int menu_len    = ul == user_level_beginner ? 4 : 3;

    if (FBROWSER_param_magic != A->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"set_fbrowser_screen",
	      "Bad magic number (menu_anon_param)",0);
    
    menu_get_sizes(page,&LINES, &COLUMNS);


    /* 3) menu part */


    if (LOC->menu_page && LINES < 14)
	erase_menu_context (&(LOC->menu_page));
    else if (LOC->menu_page)
	menu_subpage_relocate(LOC->menu_page,page,LINES-4-menu_len,menu_len);
    else if (mini_menu && LINES > 14)
	LOC->menu_page = new_menu_subpage(page,LINES-4-menu_len,menu_len,
					  sb_update_fbrowser_menu,
					  LIST);

    if (! LOC->menu_page)
	menu_len = 0;

    /* 1)  Title part of screen */

    if (! LOC->title_page)
	LOC->title_page = new_menu_subpage(page,0,title_len,sb_update_fbrowser_title,LIST);
    else
	menu_subpage_relocate(LOC->title_page,page,0,title_len);
    
    /* 2) selection part */
    A->headers_per_page = LINES-4-title_len-menu_len-1;
    if (A->headers_per_page < 1) {
	A->headers_per_page = 1;
    }
 
    if (! LOC->header_page)
	LOC->header_page = new_menu_header(page,title_len,
					   A->headers_per_page,
					   fbrowser_show_header,
					   fbrowser_show_current,
					   null_header_param_changed,
					   fbrowser_show_current,
					   null_header_line_separator_index,
					   header_separator_noredraw,
					   null_header_separator_start,
					   fbrowser_header_init,
					   fbrowser_init_header,
					   LIST);
    else 
	menu_header_relocate(LOC->header_page,page,
			     title_len,A->headers_per_page);



    /* 4) prompt part  */

    if (LOC->prompt_page)
	menu_subpage_relocate(LOC->prompt_page,page,LINES-4,4);
    else 
	LOC->prompt_page = new_menu_subpage(page,LINES-4,4,
					    sb_update_fbrowser_prompt,LIST);


    update_pagenum(LOC,LIST);
}



static void check_fbrowser_screen P_((struct screen_parts *LOC,
				    struct menu_param *list));
static void check_fbrowser_screen(LOC,list)
     struct screen_parts *LOC;
     struct menu_param *list;
{
    struct menu_anon_param *A = mp_lookup_anon(list,fbrowser_mp_params);
    
    update_pagenum(LOC,list);

    if (FBROWSER_param_magic != A->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__," check_fbrowser_screen",
	      "Bad magic number (menu_anon_param)",0);
    
    if (fb_show_title &&
	A->time_len != A->title_time_len) {
	DPRINT(Debug,14,(&Debug,
			 "check_fbrowser_screen: time_len %d => %d changed since last title_page redrawn\n",
			 A->title_time_len,A->time_len));
	
	menu_trigger_redraw(LOC->title_page);
    }
    

    /* 1) title page */
    if (menu_resized(LOC->title_page)) {
	DPRINT(Debug,1, (&Debug, "title page resized\n"));

    }
    if (menu_need_redraw(LOC->title_page)) {
	DPRINT(Debug,1, (&Debug, "title page redraw???\n"));
	sb_update_fbrowser_title(LOC->title_page,list);
    }

    /* 2) headers part */
    if (menu_resized(LOC->header_page)) {
	DPRINT(Debug,1, (&Debug, "header page resized\n"));
    }
    if (menu_need_redraw(LOC->header_page)) {
	DPRINT(Debug,1, (&Debug, "header page redraw\n"));
	menu_ClearScreen(LOC->header_page);
    }

    if (LOC->menu_page) {
	/* 3) menu page */
	if (menu_resized(LOC->menu_page)) {
	    DPRINT(Debug,1, (&Debug, "menu page resized\n"));
	    
	}
	if (menu_need_redraw(LOC->menu_page)) {
	    DPRINT(Debug,1, (&Debug, "menu page redraw\n"));
	    sb_update_fbrowser_menu(LOC->menu_page,list);
	}
    }

    /* 4) prompt part */
    if (menu_resized(LOC->prompt_page)) {
	DPRINT(Debug,1, (&Debug, "prompt page resized\n"));
    }
    if (menu_need_redraw(LOC->prompt_page)) {
	DPRINT(Debug,1, (&Debug, "prompt page redraw\n"));

	sb_update_fbrowser_prompt(LOC->prompt_page,list);
    }

}


static void first_item P_((void));
static void first_item()
{
    lib_error(CATGETS(elm_msg_cat, FbrowserSet,
		      FbrowserFirstFile,			
		      "You are on the first file!"));
}

static void last_item P_((void));
static void last_item()
{
    lib_error(CATGETS(elm_msg_cat, FbrowserSet,
		      FbrowserLastFile,			
		      "You are on the last file!"));
}

struct fb_sort_data {
    int i;
    const struct fbentry * E;
};


static int sort_dot P_((const struct fb_sort_data * A1,
			const struct fb_sort_data * B1));
static int sort_dot(A1,B1)
     const struct fb_sort_data * A1;
     const struct fb_sort_data * B1;
{
    if (0 == strcmp(A1->E->name,B1->E->name))
	return 0;

    if (0 == strcmp(A1->E->name,"."))
	return -1;
    if (0 == strcmp(B1->E->name,"."))
	return 1;
    if (0 == strcmp(A1->E->name,".."))
	return -1;
    if (0 == strcmp(B1->E->name,".."))
	return 1;

    return 0;

}


static int sort_by_name P_((const void *A, const void *B));
static int sort_by_name(A,B)
     const void *A;
     const void *B;
{
    const struct fb_sort_data * A1 = A;
    const struct fb_sort_data * B1 = B;

    int r = sort_dot(A1,B1);

    if (r) 
	return r;

    r = string_cmp(A1->E->display_name,
		   B1->E->display_name,
		   0 /* Failure */);

    if (0 == r)
	r = strcmp(A1->E->name,
		   B1->E->name);

    return r;
}

static int sort_by_revname P_((const void *A, const void *B));
static int sort_by_revname(A,B)
    const void *A;
     const void *B;
{
    const struct fb_sort_data * A1 = A;
    const struct fb_sort_data * B1 = B;

    int r = sort_dot(A1,B1);

    if (r) 
	return r;

    return -sort_by_name(A,B);
}


static int sort_by_mtime P_((const void *A, const void *B));
static int sort_by_mtime(A,B)
     const void *A;
     const void *B;
{
    const struct fb_sort_data * A1 = A;
    const struct fb_sort_data * B1 = B;

    int r = sort_dot(A1,B1);

    if (r) 
	return r;

    r = A1->E->mtime - B1->E->mtime;

    if (0 == r)
	r = strcmp(A1->E->name,
		   B1->E->name);

    return r;
}


static void free_dir_items(A)
     struct menu_anon_param *A;
{
    if (FBROWSER_param_magic != A->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"free_dir_items",
	      "Bad magic number (menu_anon_param)",0);

    if (A->dir_items) {
	int i;

	for (i = 0; i < A->dir_items_len; i++) {
	    if (A->dir_items[i].time_s)
		free_string(& (A->dir_items[i].time_s));	    		
	}

	free(A->dir_items);
    }
    A->dir_items_len = 0;
}


static int sort_by_revmtime P_((const void *A, const void *B));
static int sort_by_revmtime(A,B)
     const void *A;
     const void *B;
{
    const struct fb_sort_data * A1 = A;
    const struct fb_sort_data * B1 = B;

    int r = sort_dot(A1,B1);

    if (r) 
	return r;

    return -sort_by_mtime(A,B);
}


static void set_dir_item P_((struct menu_anon_param *A,int i, int num));
static void set_dir_item(A,i,num)
     struct menu_anon_param *A;
     int i;
     int num;
{

    if (i < 0 || i >= A->dir_items_len)
	panic("BROWSER PANIC",__FILE__,__LINE__,"set_dir_item",
	      "bad_number",0);

    A->dir_items[i].num = num;
    if (A->dir_items[i].time_s)
	free_string(& (A->dir_items[i].time_s));
	
}

static void fbrowser_sort_directory P_((struct folder_browser *dir,
					struct menu_anon_param *A,
					struct fb_sort_data       *D));
static void fbrowser_sort_directory(dir,A,D)
     struct folder_browser *dir;     
     struct menu_anon_param *A;
     struct fb_sort_data       *D;
{
    int i;

    if (! give_dt_sort_as_int(&fb_sort_order))
	return;

    if (FBROWSER_param_magic != A->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"fbrowser_sort_directory",
	      "Bad magic number (menu_anon_param)",0);
    
    if (A->dir_items_len > 10)
	lib_transient(CATGETS(elm_msg_cat, FbrowserSet,
			      FbrowserEnterdirSorting, "Sorting %S ..."),
		      A->current_dir);

    switch(give_dt_sort_as_int(&fb_sort_order)) {
    case LD_NONE:                 return;
    case LD_NAME_SORT:

        qsort(D,A->dir_items_len,sizeof (D[0]),
              sort_by_name);
        break;

    case REVERSE LD_NAME_SORT:

        qsort(D,A->dir_items_len,sizeof (D[0]),
              sort_by_revname);
	break;

    case LD_MTIME_SORT:

        qsort(D,A->dir_items_len,sizeof (D[0]),
	                   sort_by_mtime);
        break;

    case REVERSE LD_MTIME_SORT:
	qsort(D,A->dir_items_len,sizeof (D[0]),
	      sort_by_revmtime);

	break;


    }
	
    for (i = 0; i < A->dir_items_len; i++) 
	set_dir_item(A,i,D[i].i);
    
}

static void fbrowser_scan_directory P_((struct folder_browser *dir,
					struct menu_anon_param *A));
static void fbrowser_scan_directory(dir,A)
     struct folder_browser *dir;     
     struct menu_anon_param *A;
{
    int count;
    int i,x;
    int pattern_is_dot = 0;

    struct fb_sort_data * D;

    if (FBROWSER_param_magic != A->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"fbrowser_scan_directory",
	      "Bad magic number (menu_anon_param)",0);
    
    lib_transient(CATGETS(elm_msg_cat, FbrowserSet,
			  FbrowserEnterdirScanning, "Scanning %S ..."),
		  A->current_dir);
    
    count = dir_give_count(dir);
    if (count < 1) {
	free_dir_items(A);
	
	DPRINT(Debug,10,(&Debug, "fbrowser_scan_directory: Empty directory !!\n"));
	
	return;
    }

    DPRINT(Debug,12,(&Debug, "fbrowser_scan_directory: %d items\n",count));

    if (A->current_pattern &&
	string_len(A->current_pattern) > 0) {

	pattern_is_dot =  0x002E /* . */  == 
	    give_unicode_from_string(A->current_pattern,0);

	DPRINT(Debug,12,(&Debug, "fbrowser_scan_directory: pattern_is_dot =%d\n",
			 pattern_is_dot));

    }

    if (count > A->dir_items_len) {
    
	A->dir_items = safe_array_realloc(A->dir_items,
					  count, sizeof(A->dir_items[0]));

	for (i = A->dir_items_len; i < count; i++) {
	    A->dir_items[i].time_s = NULL;
	    A->dir_items[i].num    = 0;
	}
	A->dir_items_len = i;
	
    }
    
    D            = safe_calloc(count, sizeof(D[0]));
    
    x = 0;
    for (i = 0; i < count; i++) {

	const struct fbentry * E = fbrowser_give_entry(dir,i);

	if (! E) 
	    continue;

	/* For help for sorting */
	D[x].i = i;
	D[x].E = E;

	if (
#ifdef S_ISDIR
	    S_ISDIR(E->mode)
#else
	    S_IFDIR == (E->mode & S_IFMT)
#endif
	    ) {

	    if ('.' != E->name[0] ||
		0 == strcmp(E->name,".") || 0 == strcmp(E->name,"..")) {

		set_dir_item(A,x,i);
		
		x++;
	    } else {
		DPRINT(Debug,12,(&Debug, 
				 "fbrowser_scan_directory: [%d] Ignoring directory %S (name=%s) -- dot\n",
				 i,E->display_name,E->name));
	    }

	} else if (
#ifdef S_ISREG
	    S_ISREG(E->mode)
#else
	    S_IFREG == (E->mode & S_IFMT)
#endif
	    ) {

	    if ('.' == E->name[0] && !fb_show_dotfiles  && 
		! pattern_is_dot) {
		DPRINT(Debug,13,(&Debug, 
				 "fbrowser_scan_directory: [%d] Ignoring file %S (name=%s) -- dot\n",
				 i,E->display_name,E->name));		
		continue;
	    }

	    if (! A->current_pattern ||
		string_match(E->display_name,
			     A->current_pattern,0)) {
		set_dir_item(A,x,i);
		
		x++;
	    } else {
		DPRINT(Debug,13,(&Debug, 
				 "fbrowser_scan_directory: [%d] Ignoring file %S (name=%s) -- pattern\n",
				 i,E->display_name,E->name));
	    }

	}       	
    }

    if (A->dir_items_len > x) {
	for (i = x; i < A->dir_items_len; i++) {
	    if (A->dir_items[i].time_s)
		free_string(& (A->dir_items[i].time_s));
	}
    }
    
    A->dir_items_len = x;


    DPRINT(Debug,10,(&Debug, "fbrowser_scan_directory: dir_items_len=%d\n",
		     A->dir_items_len));

    fbrowser_sort_directory(dir,A,D);

    free(D);

}

static void fbrowser_resync P_((struct folder_browser *dir,
				struct menu_anon_param *A,
				struct screen_parts *LOC,
				struct menu_param *list));
static void fbrowser_resync(dir,A,LOC,list)
     struct folder_browser *dir;
     struct menu_anon_param *A;
     struct screen_parts *LOC;
     struct menu_param *list;
{
    int current = menu_header_get(LOC->header_page,header_current);
    int pos = -1;

    int i;

    if (FBROWSER_param_magic != A->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"fbrowser_resync",
	      "Bad magic number (menu_anon_param)",0);

    if (current >= 0 && current < A->dir_items_len) {
	const struct fbentry * entry UNUSED_VAROK;

	pos = A->dir_items[current].num;
	entry = fbrowser_give_entry(dir,pos);

	DPRINT(Debug,11, (&Debug, "fbrowser_resync: current=%d pos=%d\n",
			  current,pos));
    }

    /* fbrowser_scan directory do not affect underling pos
       position 
    */


    if (reload_dir(dir,&pos))
	fbrowser_scan_directory(dir,A);

    
    for (i = 0; i < A->dir_items_len; i++) {	
	if (pos == A->dir_items[i].num) 
	    break;		
    }

    if (i < A->dir_items_len) {
	DPRINT(Debug,11, (&Debug, 
			  "fbrowser_resync: FOUND pos=%d current=%d => %d\n",
			  pos,current,i));
	
	menu_header_change(LOC->header_page,header_current,i);	    
    } else {
	DPRINT(Debug,11, (&Debug, 
			  "fbrowser_resync: NOT FOUND pos=%d current=%d => 0\n",
			  pos,current));

	menu_header_change(LOC->header_page,header_current,0);	    
    }

    change_page(LOC,list);

    menu_trigger_redraw(LOC->title_page);
    menu_trigger_redraw(LOC->header_page);

}


static int fb_mbox_check P_((struct folder_browser *dir,
			     struct string         *buffer,
			     struct screen_parts   *LOC,
			     struct fbrowser_call  *fc,
			     int                    idx));
static int fb_mbox_check(dir,buffer,LOC,fc,idx)
     struct folder_browser *dir;
     struct string         *buffer;
     struct screen_parts   *LOC;
     struct fbrowser_call  *fc;
     int idx;
{
    int options = fc->options;

    DPRINT(Debug,12,(&Debug, 
		     "fb_mbox_check: options=%d (%s%s%s%s%s ) buffer=%S\n",
		     options,
		     options ?                       "" : " none",
		     (options & FB_EXIST) ? " FB_EXIST" : "",
		     (options & FB_READ)  ? " FB_READ"  : "",
		     (options & FB_WRITE) ? " FB_WRITE" : "",
		     (options & FB_NOMBOX_CHECK)  ? " FB_NOMBOX_CHECK"  : "",
		     buffer));

    if (! fbrowser_selection_have_access(dir,ACCESS_EXISTS,0)) {

	DPRINT(Debug,12,(&Debug, 
			 "fb_mbox_check: selection do not exists\n"));


	if (0 != (options & FB_EXIST)) {
	    
	    lib_error(CATGETS(elm_msg_cat, FbrowserSet,
			      FbrowserMboxNotExists,
			      "Cannot access \"%S\" -- it does not exists."),
		      buffer);
	    

	    DPRINT(Debug,12,(&Debug, 
			     "fb_mbox_check=0\n"));

	    return 0;
	}

	DPRINT(Debug,12,(&Debug, 
			 "fb_mbox_check=1\n"));
	
	return 1;
    }
    
    if (0 != (options & FB_READ) ||
	0 != (options & FB_WRITE)) {

	int mode = 0;

	if (0 != (options & FB_READ))
	    mode |= READ_ACCESS;

	if (0 != (options & FB_WRITE))
	    mode |= WRITE_ACCESS;

	if (! fbrowser_selection_have_access(dir,mode,1)) {
	    DPRINT(Debug,12,(&Debug, 
			     "fb_mbox_check=0 (no access, mode=%d)\n",mode));

	    return 0;		
	}
    }

    if (get_browser_sel_type(dir) == selection_folder) {
	DPRINT(Debug,12,(&Debug, 
			 "fb_mbox_check: selection_folder: mailbox expected\n"));

	
	if (0 != (options & FB_NOMBOX_CHECK)) {

	    DPRINT(Debug,12,(&Debug, 
			     "fb_mbox_check: Skipping mailbox check\n"));

	} else if (! fbrowser_selection_is_mbox(dir,1)) {
	    
	    DPRINT(Debug,12,(&Debug, 
			     "fb_mbox_check=0 (is not mailbox)\n"));
	    
	    return 0;
	}

    }

    DPRINT(Debug,12,(&Debug, 
		     "fb_mbox_check=1\n"));

    return 1;
}


static struct move_messages M = {
    first_item,
    last_item
};



    
enum fbrowser_result fbrowser(parent_page,dir,mailbox,buffer,fc)
     struct menu_context   *parent_page;
     struct folder_browser *dir;
     struct MailboxView    *mailbox;
     struct string        **buffer;
     struct fbrowser_call  *fc;
{
    int update = 1;
    enum fbrowser_result ret = fbrowser_quit;
    
    struct menu_context  *page = new_menu_context();
    struct screen_parts LOC = { NULL, NULL, NULL };

    struct menu_anon_param A;
    struct menu_param  PARAM[fbrowser_mp_COUNT+1] = {
	{ mp_anon_param, { 0 } },
	{ mp_END, { 0 } }
    };


    struct elm_commands *fbroser_cmds = give_fbrowser_commands();

    int   LINES, COLUMNS;	
    menu_get_sizes(page,&LINES, &COLUMNS);

    if (FBROWSER_CALL_magic != fc->magic)	
	panic("BROWSER PANIC",__FILE__,__LINE__,"fbrowser",
	      "Bad magic number (fbrowser_call)",0);

    fbrowser_zero_menu_anon_param(&A);
    
    A.title_prompt      = dup_string(fc->prompt);   
    A.current_dir       = NULL;   
    A.current_pattern   = NULL;   
    A.dir               = dir;
    A.dir_items         = NULL;
    A.dir_items_len     = 0;
    A.options           = fc->options;
    A.time_len          = 10;
    A.title_time_len    = 0;
    A.cur_page = 0;
    A.headers_per_page = LINES-15;
    if (A.headers_per_page < 1)
	A.headers_per_page = 1;
    A.comment_column    = COLUMNS * 2 / 3;
    
    mp_list_set_anon(PARAM,fbrowser_mp_params,&A);
   
    if (fc->flags & FBROWSER_USE_SELECTION) {

	/* selection is correct directory -- need change to it */

	if (!change_dir_to_selection(dir,buffer)) {

	    lib_error(CATGETS(elm_msg_cat, FbrowserSet,
			      FbrowserFailedInitialize,
			      "Failed to initialize file browser."));

	    ret = fbrowser_error;
	    goto OUT_1;
	}	       
	
	/* Changing directory resets filter */

    } else {
	int r;
	
	/* Invalidates 'filter' -- r == -1  failure 
                                   r == 0   no changes
	*/

	r = dir_reset_filter(dir);
	if (r < 0) {
	    lib_error(CATGETS(elm_msg_cat, FbrowserSet,
			      FbrowserFailedInitialize,
			      "Failed to initialize file browser."));

	    ret = fbrowser_error;
	    goto OUT_1;
	}

    }

    if (fc->filter) 
	A.current_pattern = dup_string(fc->filter);


    /* Include .dotfiles  to listing */
    fbrowser_set_dotmode(dir,1);
   
    if (! give_edit_buffer(dir,0,& A.current_dir,0)) {
	    lib_error(CATGETS(elm_msg_cat, FbrowserSet,
			      FbrowserFailedInitialize,
			      "Failed to initialize file browser."));

	    ret = fbrowser_error;
	    goto OUT_1;
    }

    if (fc->last_fbrowser_dir) {

	if (0 != string_cmp(fc->last_fbrowser_dir, A.current_dir,
			    0 /* compare failure */)) {
	    DPRINT(Debug,10, (&Debug, 
			     "fbrowser: Directory is changed from %S to %S\n",
			     fc->last_fbrowser_dir,A.current_dir));

	    if (fc->save_dir)
		free_string(& (fc->save_dir));
	    
	    fc->save_dir = fc->last_fbrowser_dir;
	    fc->last_fbrowser_dir = NULL;

	} else 
	    free_string(& (fc->last_fbrowser_dir));
    }

    
    set_fbrowser_screen(page,&LOC,PARAM);

    clear_selection_dir(dir);

    if (0) {
    RELOAD3:
	
	if (! give_edit_buffer(dir,0,& A.current_dir,0)) {
	    lib_error(CATGETS(elm_msg_cat, FbrowserSet,
			      FbrowserFailedInitialize,
			      "Failed to initialize file browser."));
	    
	    ret = fbrowser_error;
	    goto OUT;
	}

	reload_dir(dir,NULL);
    }

 RELOAD2:


    
    fbrowser_scan_directory(dir,&A);


    A.cur_page = 0;

    menu_header_change(LOC.header_page,header_current,0);	    
    menu_header_change(LOC.header_page,header_top_line,0);	    
    menu_trigger_redraw(LOC.header_page);

    menu_trigger_redraw(LOC.title_page);
    
    for (;;) {
	int ch = '\0';

	if (menu_resized(page)) {
	    set_fbrowser_screen(page,&LOC,PARAM);
	    
	    update = 1;
	}

	if (update || menu_need_redraw(page)) {
	    menu_ClearScreen(page);
	    
	    /* Call refresh routines of children */
	    menu_redraw_children(page);
	    
	    update = 0;
	    show_last_error(); 
	} 

	check_fbrowser_screen(&LOC,PARAM);

#ifdef BACKGROUD_PROCESSES      
	if (handle_sigchld)
	    sigchld_handler();
#endif
	
	{
	    int lin,col;
	    
	    menu_ClearLine(LOC.prompt_page,0);
	    
 	    menu_PutLineX (LOC.prompt_page,0, 0, 
			   CATGETS(elm_msg_cat, FbrowserSet,
				   FbrowserMainPrompt, "Command: "));
	    menu_GetXYLocation(LOC.prompt_page,&lin,&col);
	    
	    menu_CleartoEOS(LOC.prompt_page);   
	
	    show_last_error();
	    menu_MoveCursor(LOC.prompt_page,lin,col);
	    
	    ch = menu_ReadCh(LOC.prompt_page, 
			     REDRAW_MARK|READCH_CURSOR|READCH_resize|
			     READCH_sig_char);

	    menu_CleartoEOS(LOC.prompt_page);

	    if (isascii(ch) && isprint(ch)) {
		DPRINT(Debug,4,
		       (&Debug, "-- fbrowser command: %c [%d]\n",ch,ch));
	    } else {
		DPRINT(Debug,4,
		       (&Debug, "-- fbrowser command: [%d]\n",ch));
	    }

	    set_error("");	/* clear error buffer */
	}


	/* Translate keys not handled on do_movement1()
	   to equivalent keys 
	*/
	switch (ch) {
	case '1': 	 ch = HOME_MARK;     break;
	case LEFT_MARK:  ch = PAGEUP_MARK;   break;
	case RIGHT_MARK: ch = PAGEDOWN_MARK; break;

	case '=':    /* On do_movement this is equivalent of HOME_MARK */
	    
	    menu_Write_to_screen(LOC.prompt_page,
				 CATGETS(elm_msg_cat, FbrowserSet,
					 FbrowserEnterName,
					 "Enter a filename (changes browser)"));
	    FlushBuffer();


	    /* enter value from "~/Mail/" */

	    if (*buffer)
		free_string(buffer);
	    *buffer = new_string2(display_charset,s2us("=/"));

	    if (change_dir(dir, buffer)) {		
		ret = fbrowser_return_browser;

		DPRINT(Debug,9, (&Debug, 
				 "fbrowser: = => %S\n",*buffer));



		goto OUT;
	    }

	    error_sleep(sleepmsg);
	    goto RELOAD3;

	}

	ch = do_movement(LOC.header_page,ch,A.dir_items_len,&M);


	switch (ch) {
	case 0:  /* Movement done */

	    update_pagenum(&LOC,PARAM);

	    break;

	case RESIZE_MARK:
	    DPRINT(Debug,4, (&Debug, " ... resizing\n"));
	    
	    continue;
	    
	case ctrl('L'):
	case REDRAW_MARK:
	    DPRINT(Debug,4, (&Debug, " ... redrawing\n"));
	    
	    menu_ClearScreen(page);   /* Clear possible redraw mark */
	    
	    /* Call refresh routines of children */
	    menu_redraw_children(page);
	    
	    if (menu_need_redraw(LOC.prompt_page))		    
		menu_ClearScreen(LOC.prompt_page);   /* Clear redraw mark from prompt_area*/
	    

	    update = 1;
	    continue;

	case ctrl('R'):	{		/* recall previous dir */
	    struct string * temp = NULL;


	    menu_Write_to_screen(LOC.prompt_page,
				 CATGETS(elm_msg_cat, FbrowserSet,
					 FbrowserRecallDir,
					 "Recall previous direcory"));
	    FlushBuffer();


	    if (! fc->save_dir) {
		lib_error(CATGETS(elm_msg_cat, FbrowserSet,
				  FbrowserNoDirectorySaved, 
				  "No directory saved."));
		break;
	    }
	    
	    if (0 == string_cmp(A.current_dir,fc->save_dir,
				999 /* compare failure */)) {
		lib_error(CATGETS(elm_msg_cat, FbrowserSet,
				  FbrowserAlreadyIn,
				  "Already in \"%S\"."),
			  A.current_dir);
		break;
	    }

	    temp = dup_string(fc->save_dir);

	    if (change_dir(dir,&temp)) {
		free_string(& fc->save_dir);

		fc->save_dir = A.current_dir;
		A.current_dir = temp;
		
		free_string(buffer);
		*buffer = dup_string(temp);

		goto RELOAD2;
	    }

	    free_string(&temp);

	    error_sleep(sleepmsg);
	    goto RELOAD3;
	}
	    break;

	case '\r':			/* accept entry */
	case '\n':
	case ' ': {
	    int current = menu_header_get(LOC.header_page,header_current);
	    
	    menu_Write_to_screen(LOC.prompt_page,
				 CATGETS(elm_msg_cat, FbrowserSet,
					 FbrowserSelectEntry,
					 "Select directory entry"));
	    FlushBuffer();


	    if (current >= 0 && current < A.dir_items_len) {

		int idx = A.dir_items[current].num;
		const struct fbentry * entry = fbrowser_give_entry(dir,idx);

		if (entry) {

		    if (
#ifdef S_ISDIR
			S_ISDIR(entry->mode)
#else
			S_IFDIR == (entry->mode & S_IFMT)
#endif
			) {

			if (change_dir_to_entry(dir,idx,buffer)) {

			    if (fc->save_dir)
				free_string(& fc->save_dir);
			   fc->save_dir = A.current_dir;

			   A.current_dir = dup_string(*buffer);		       

			   goto RELOAD2;
			}

			error_sleep(sleepmsg);
			goto RELOAD3;

		    } else {

			if (fc->default_extension) {
			    int ok = browser_check_extension(entry->display_name,
							     fc->default_extension);

			    if (ok) {
				DPRINT(Debug,4, (&Debug, 
						 "fbrowser: filename %S includes extension %S\n",
						 entry->display_name,
						 fc->default_extension));
			    } else {

				DPRINT(Debug,4, (&Debug, 
						 "fbrowser: filename %S do not include extension %S\n",
						 entry->display_name,
						 fc->default_extension));


				if (*buffer)
				    free_string(buffer);
				
				*buffer = dir_cat_filename(dir,
							   entry->display_name);
				if (!*buffer)
				    break;   /* Error ? ? */


				ret = fbrowser_return_browser;
				
				goto OUT;
			    }						      
			}


			if (select_dir_item_by_idx(dir,idx,buffer)) {
			    
			    if (fb_mbox_check(dir,*buffer,&LOC, fc, idx)) {
				ret = fbrowser_selected;
				goto OUT;
			    }
			}
		    }
		}
	    }
	}
	    break;

	case HELP_MARK:
	case '?': {			/* help */
	    int X =  help_generic(fbroser_cmds,FALSE, page, LOC.prompt_page);
	    
	    if (EOF == X) 
		ret = fbrowser_INTR;          /* NOT BEST */
	}
	    break;

	case '!':			/* subshell */

	    menu_Writechar(LOC.prompt_page,'!');
	    FlushBuffer();

	    if (ALLOW_subshell)
		subshell(mailbox,page,LOC.prompt_page);
	    else
		lib_error(CATGETS(elm_msg_cat, FbrowserSet,
				  FbrowserNoSubshellCommand,
				  "Subshell command isn't configured in this version of ELM."));
	    break;

	    break;


	case '/':			/* enter value from "/" */

	    menu_Write_to_screen(LOC.prompt_page,
				 CATGETS(elm_msg_cat, FbrowserSet,
					 FbrowserEnterName,
					 "Enter a filename (changes browser)"));
	    FlushBuffer();


	    if (*buffer)
		free_string(buffer);
	    *buffer = new_string2(display_charset,s2us("/"));


	    if (change_dir(dir, buffer)) {		
		ret = fbrowser_return_browser;

		DPRINT(Debug,9, (&Debug, 
				 "fbrowser: / => %S\n",*buffer));

		goto OUT;
	    }

	    error_sleep(sleepmsg);
	    goto RELOAD3;

	case '~':			/* enter value from "$HOME/" */

	    menu_Write_to_screen(LOC.prompt_page,
				 CATGETS(elm_msg_cat, FbrowserSet,
					 FbrowserEnterName,
					 "Enter a filename (changes browser)"));
	    FlushBuffer();


	    if (*buffer)
		free_string(buffer);
	    *buffer = new_string2(display_charset,s2us("~/"));

	    if (change_dir(dir, buffer)) {		
		ret = fbrowser_return_browser;

		DPRINT(Debug,9, (&Debug, 
				 "fbrowser: ~ => %S\n",*buffer));

		goto OUT;
	    }

	    error_sleep(sleepmsg);
	    goto RELOAD3;

	case '.':			/* enter value from "." */

	    menu_Write_to_screen(LOC.prompt_page,
				 CATGETS(elm_msg_cat, FbrowserSet,
					 FbrowserEnterName,
					 "Enter a filename (changes browser)"));
	    FlushBuffer();


	    if (*buffer)
		free_string(buffer);
	    *buffer = new_string2(display_charset,s2us("./"));

	    if (change_dir(dir, buffer)) {		
		ret = fbrowser_return_browser;

		DPRINT(Debug,9, (&Debug, 
				 "fbrowser: . => %S\n",*buffer));
		
		goto OUT;
	    }

	    error_sleep(sleepmsg);
	    goto RELOAD3;

	case 'c':			/* change dir */

	    menu_Write_to_screen(LOC.prompt_page,
				 CATGETS(elm_msg_cat, FbrowserSet,
					 FbrowserChangeDir,
					 "Change a directory (changes browser)"));
	    FlushBuffer();


	    ret = fbrowser_return_browser;

	    goto OUT;

	case 'f':                       /* Fill default filename */
	case  F6_KEY_MARK:

	    menu_Write_to_screen(LOC.prompt_page,
				 CATGETS(elm_msg_cat, FbrowserSet,
					 FbrowserFillFilename,
					 "Fill default filename (changes browser)"));
	    FlushBuffer();

	    if (*buffer)
		free_string(buffer);

	    /* if filename is NULL, dir_cat_filename returns just
	       current directory
	    */
	    *buffer = dir_cat_filename(dir,
				       fc->default_filename);
	    if (!*buffer)
		break;   /* Error ? ? */

	    ret = fbrowser_return_browser;
				
	    goto OUT;

	case 'l':			/* change pattern (limit) */
	case 'p':{

	    /* FIXME --optionally_enter*  should use prompt_area */
	    int line;
	    int code;

	    struct string * buffer = NULL;


	    if (A.current_pattern)
		buffer = dup_string(A.current_pattern);


	    menu_Write_to_screen(LOC.prompt_page,
				 CATGETS(elm_msg_cat, FbrowserSet,
					 FbrowserChangePattern,
					 "Change pattern"));

	redraw_prompt:
	    line = menu_GetAbsLine(LOC.prompt_page,1);

	    code = optionally_enter2(page,
				     & buffer,line,0,
				     OE_REDRAW_MARK
				     |OE_SIG_CHAR /* Ctrl-C */,
				     CATGETS(elm_msg_cat, FbrowserSet,
					     FbrowserPatternPrompt,
					     "Pattern: "));

	    if (REDRAW_MARK == code) {
		/* NOTICE: using menu_trigger_redraw(page) on here
		   may cause redraw loop!
		*/

		menu_ClearScreen(page);   /* Clear possible redraw mark */
	
		/* Call refresh routines of children */
		menu_redraw_children(page);

		update = 1;  /* delay redraw */

		goto redraw_prompt;
	    }
	    
	    if(code==-1){
		/** User hit  Ctrl-C key! **/

		if (buffer)
		    free_string(& buffer);

		MoveCursor(line,0); 	
		CleartoEOLN();
		break;
	    }

	    if (A.current_pattern)
		free_string(& A.current_pattern);

	    if (buffer) {
		if (string_len(buffer) > 0)
		    A.current_pattern = buffer;
		else
		    free_string(& buffer);
	    }
	    goto RELOAD2;

	}
	    break;

	case 'm': {                     /* Make directory         */

	    /* FIXME --optionally_enter*  should use prompt_area */
	    int line;
	    int code;

	    struct string * buffer = NULL;

	    menu_Write_to_screen(LOC.prompt_page,
				 CATGETS(elm_msg_cat, FbrowserSet,
					 FbrowserMakeDirectory,
					 "Make directory"));


	redraw_prompt_2:
	    line = menu_GetAbsLine(LOC.prompt_page,1);


	    code = optionally_enter2(page,
				     & buffer,line,0,
				     OE_REDRAW_MARK
				     |OE_SIG_CHAR /* Ctrl-C */,
				     CATGETS(elm_msg_cat, FbrowserSet,
					     FbrowserDirectoryPrompt,
					     "Directory: "));

	    if (REDRAW_MARK == code) {
		/* NOTICE: using menu_trigger_redraw(page) on here
		   may cause redraw loop!
		*/

		menu_ClearScreen(page);   /* Clear possible redraw mark */
	
		/* Call refresh routines of children */
		menu_redraw_children(page);

		update = 1;  /* delay redraw */

		goto redraw_prompt_2;
	    }

	    if(code==-1){
		/** User hit  Ctrl-C key! **/

		if (buffer)
		    free_string(& buffer);

		MoveCursor(line,0); 	
		CleartoEOLN();
		break;
	    }

	    if (buffer) {
		int current UNUSED_VAROK = 
		    menu_header_get(LOC.header_page,header_current);
		int pos = -1;

		if (fbrowser_make_directory(dir,buffer,&pos)) {
		    int i;

		    fbrowser_scan_directory(dir,&A);

		    for (i = 0; i < A.dir_items_len; i++) {	
			if (pos == A.dir_items[i].num) 
			    break;		
		    }

		    if (i < A.dir_items_len) {
			DPRINT(Debug,11, (&Debug, 
					  ".... FOUND pos=%d current=%d => %d\n",
					  pos,current,i));
			
			menu_header_change(LOC.header_page,header_current,i);	    

			change_page(&LOC,PARAM);
		    } 
		    
		    menu_trigger_redraw(LOC.title_page);
		    menu_trigger_redraw(LOC.header_page);

		    lib_transient(CATGETS(elm_msg_cat, FbrowserSet,
					  FbrowserDirectoryCreated,
					  "Directory %S created."),
				  buffer);
		    
		} else
		    error_sleep(sleepmsg);

	    }

	    if (buffer)
		free_string(& buffer);

	}
	    break;

	case 'o': {			/* change options submenu */
	    int r;
	    int resort = 0;
	    int setscreen = 0;

	    menu_Write_to_screen(LOC.prompt_page,
				 CATGETS(elm_msg_cat, FbrowserSet,
					 FbrowserOptions,
					 "Options"));
	    FlushBuffer();

	    r = fbrowser_option(page,&resort, &setscreen);
		
	    if (EOF == r) {
		ret = fbrowser_INTR;
		goto OUT;
	    }

	    if (setscreen) {
		set_fbrowser_screen(page,&LOC,PARAM);
		
		update = 1;
	    }

	    if (resort)
		goto RELOAD2;
	}

	    break;
	    
	case 'q':
	case 'x':
	    menu_Write_to_screen(LOC.prompt_page,
				 CATGETS(elm_msg_cat, FbrowserSet,
					 FbrowserQuit,
					 "Quit"));
	    FlushBuffer();

	    ret = fbrowser_quit;
	    goto OUT;

	case TERMCH_interrupt_char:
	case EOF:
	    ret = fbrowser_INTR;
	    goto OUT;

	case F3_KEY_MARK:
	    mini_menu = ! mini_menu;

	    if (LOC.menu_page && !mini_menu)
		erase_menu_context (&(LOC.menu_page));

	    DPRINT(Debug,10,(&Debug,"mini_menu = %d \n",mini_menu));
	    
	    set_fbrowser_screen(page,&LOC,PARAM);

	    update = 1;
	    
	    break;

	case  F4_KEY_MARK: {
	    int r;

	    DPRINT(Debug,11, (&Debug,  " ... F4\n"));

	    /* NULL filter == reset filter */
	    r = dir_change_filter(dir,A.current_pattern);
	
	    DPRINT(Debug,10, (&Debug, 
			      "dir_change_filter gives %d\n",r));
    
	    if (r < 0) {
		lib_error(CATGETS(elm_msg_cat, FbrowserSet,
				  FbrowserFailedReInitialize,
				  "Failed to re-initialize file browser."));
		
		ret = fbrowser_error;
		goto OUT;
	    }
	}
	    ret = fbrowser_return_browser;
	    goto OUT;

	case '$':
	case F5_KEY_MARK:
	    menu_Write_to_screen(LOC.prompt_page,
				 CATGETS(elm_msg_cat, FbrowserSet,
					 FbrowserResyncDir,
					 "Resynchronize directory"));
	    FlushBuffer();
      	    
	    fbrowser_resync(dir,&A,&LOC,PARAM);

	    break;

	default:

	    if (isascii(ch) && isprint(ch))
		lib_error(CATGETS(elm_msg_cat, FbrowserSet,
				  FbrowserUnknownCommand,		       
				  "Unknown command: %c"), 
			  ch);
	    else
		lib_error(CATGETS(elm_msg_cat, FbrowserSet,
				  FbrowserUnknownCommand2,		       
				  "Unknown command."));
		
	    continue;
	}
    }

 OUT:

    free_mailbox_screen(&LOC);

 OUT_1:
    /* Do not show .dotfiles  to listing */
    fbrowser_set_dotmode(dir,0);

    if (fc->last_fbrowser_dir)
	free_string(& (fc->last_fbrowser_dir));
    
    fc->last_fbrowser_dir = A.current_dir;
    A.current_dir = NULL;
    A.dir = NULL;  /* Do not free */

    error_wait();
    
    erase_menu_context(&page);

    /* Force default return to parent page ... */
    menu_set_default(parent_page); 

    menu_trigger_redraw(parent_page); 

    if (*buffer) {
	DPRINT(Debug, 9, (&Debug,  "fbrowser: buffer=%S\n",
			  *buffer));
    }
    
    free_commands(&fbroser_cmds);   /* Decrement refcount */

    fbrowser_clear_menu_anon_param(&A);
    
    DPRINT(Debug,9, (&Debug, "fbrowser=%d\n",
		     ret));

    return ret;
}



/* ------------------------------------------------------------- */

struct fbrowser_call * alloc_fbrowser_call(filter,prompt,options)  /* initialize structure  */
    struct string  * filter;
     struct string  * prompt;
     int options;
{
    struct fbrowser_call *fc = safe_malloc(sizeof (*fc));

    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)fc,sizeof (*fc));

    fc->magic = FBROWSER_CALL_magic;

    fc->filter = filter ? dup_string(filter) : NULL;
    fc->prompt = dup_string(prompt);

    fc->flags    = 0;
    fc->options  = options;

    DPRINT(Debug,12,(&Debug, 
		     "alloc_fbrowser_call: options=%d (%s%s%s%s%s )\n",
		     options,
		     options ?                       "" : " none",
		     (options & FB_EXIST) ? " FB_EXIST" : "",
		     (options & FB_READ)  ? " FB_READ"  : "",
		     (options & FB_WRITE) ? " FB_WRITE" : "",
		     (options & FB_NOMBOX_CHECK)  ? " FB_NOMBOX_CHECK"  : ""));


    fc->filter    = NULL;
    fc->save_dir  = NULL;
    fc->last_fbrowser_dir = NULL;
    fc->default_extension = NULL;
    fc->default_filename  = NULL;

    return fc;
}

struct fbrowser_call * alloc_fbrowser_call_f(
#if ANSI_C
					    int options,
					    const char *format, 
					    const char *msg, 
					    ...
#else
					    options,format,msg,va_alist
#endif
					    )
#if !ANSI_C
     int options;
     const char *format;
     const char *msg;
     va_dcl
#endif
{
    struct fbrowser_call *fc = NULL;
    struct string *text;
    va_list vl;

    Va_start(vl, msg);           /*defined in hdrs/elm_defs.h  */

    text = elm_smessage(0,format,msg,vl);

    va_end(vl);


    fc = alloc_fbrowser_call(NULL,text,options);


    free_string(&text);
    
    return fc;
}

void set_fbrowser_flags(fc,flags)
     struct fbrowser_call *fc; 
     int flags;
{
    if (FBROWSER_CALL_magic != fc->magic)	
	panic("BROWSER PANIC",__FILE__,__LINE__,"set_fbrowser_flags",
	      "Bad magic number",0);

    fc->flags = flags;
}

void set_fbrowser_default_filename(fc,fn)
     struct fbrowser_call *fc;
     const struct string *fn;
{
    if (FBROWSER_CALL_magic != fc->magic)	
	panic("BROWSER PANIC",__FILE__,__LINE__,"set_fbrowser_flags",
	      "Bad magic number",0);


    if (fc->default_filename)
	free_string(& (fc->default_filename));
    if (fn)
	fc->default_filename = dup_string(fn);

}


void free_fbrowser_call(fc)                /* free structure fields */
     struct fbrowser_call **fc;   
{
    if (FBROWSER_CALL_magic != (*fc)->magic)	
	panic("BROWSER PANIC",__FILE__,__LINE__,"clear_fbowser_call",
	      "Bad magic number",0);

    if ((*fc)->filter)
	free_string(& ((*fc)->filter));
    
    if ((*fc)->prompt)
	free_string(& ((*fc)->prompt));

    if ((*fc)->save_dir)
	free_string(& ((*fc)->save_dir));

    if ((*fc)->last_fbrowser_dir)
	free_string(& ((*fc)->last_fbrowser_dir));
	  
    if ((*fc)->default_extension)
	free_string(& ((*fc)->default_extension));

    if ((*fc)->default_filename)
	free_string(& ((*fc)->default_filename));

    (*fc)->magic = 0;   /* Invalidate */
    free(*fc);
}


enum browser_status fbrowser_hook_enter(page,dir,mailbox,buffer,fc)     
     struct menu_context *page;
     struct folder_browser *dir;
     struct MailboxView *mailbox;
     struct string **buffer;
     struct fbrowser_call *fc;
{

    if (fbrowser_supported_on_dir(dir)) {
	enum fbrowser_result r ;

	DPRINT(Debug,10, (&Debug, 
			  "fbrowser_hook_enter: Calling fbrowser\n"));
	
	r =  fbrowser(page,dir,mailbox,buffer,fc);
	
	switch (r) {
	default:
	    
	    panic("BROWSER PANIC",__FILE__,__LINE__,"fbrowser_hook_enter",
		  "Bad return value of fbrowser",0);
	    
	case fbrowser_INTR:
	case fbrowser_quit:
	case fbrowser_error:
	    
	    DPRINT(Debug,10, (&Debug, 
			      "fbrowser_hook_enter: error  r = %d\n",
			      r));
	    
	    return browser_error;
	    
	case fbrowser_selected:
	    
	    DPRINT(Debug,10, (&Debug, 
			      "fbrowser_hook_enter: selected\n"));

	    return browser_select;

	case fbrowser_return_browser:
	    
	    DPRINT(Debug,10, (&Debug, 
			      "fbrowser_hook_enter: rerun browser\n"));
	    
	    return browser_rerun;
	    
	}
    }

    if (0 != (fc->flags & FBROWSER_USE_SELECTION)) {
	DPRINT(Debug,10, (&Debug, 
			  "fbrowser_hook_enter: selected (not supported)\n"));

	return browser_select;	
    }
    
    DPRINT(Debug,10, (&Debug, 
		      "fbrowser_hook_enter: error (not supported)\n"));
    return browser_error;
}





/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
