static char rcsid[] = "@(#)$Id: init.c,v 2.40 2023/12/13 16:55:32 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.40 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *  Based on Elm 2.4 hdrs/init.c. That code was following copyright:
 *
 *  The Elm Mail System 
 *
 *			Copyright (c) 1988-1992 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************
 *  Incorparated Elm 2.5 code from hdrs/init.c. 
 *  That code was following copyright:
 *
 *  The Elm Mail System 
 *
 *                      Copyright (c) 1988-1995 USENET Community Trust
 *                      Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

/***** Initialize - read in all the defaults etc etc 
*****/

#include "def_elm.h"
#include "patchlevel.h"
#include "s_elm.h"
#include "s_me.h"

#include "fbrowser.h"

DEBUG_VAR(Debug,__FILE__,"ELM");

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

#ifdef PWDINSYS
#  include <sys/pwd.h>
#else
#  include <pwd.h>
#endif

#include <errno.h>
#ifndef ANSI_C
extern int errno;		/* system error number on failure */
#endif       

extern char version_buff[];

SIGHAND_TYPE
#ifdef SIGTSTP
sig_user_stop P_((int)), sig_return_from_user_stop P_((int)),
#endif
#ifdef SIGWINCH
    winch_signal P_((int)),
#endif
    quit_signal P_((int)), term_signal P_((int)), ill_signal P_((int)),
    fpe_signal P_((int)), segv_signal P_((int)),
#ifdef SIGBUS
    bus_signal P_((int)),
#endif
    alarm_signal P_((int)), pipe_signal P_((int)), hup_signal P_((int)),
    usr1_signal P_((int)), usr2_signal P_((int)) ;

static struct folder_info * select_folder P_((struct AliasView *aview,
					      struct menu_context *page));
static struct folder_info * select_folder(aview,page)
     struct AliasView *aview;
     struct menu_context *page;
{
    struct folder_info    * folder = NULL;
    struct folder_browser * XXX    = new_browser(selection_folder);
    struct string         * buffer = NULL;
    struct fbrowser_call  * Fcall  = 
	alloc_fbrowser_call_f(FB_EXIST|FB_READ,
			      CATGETS(elm_msg_cat, ElmSet, 
				      ElmSelectFolderFB,
				      "Folder opening selection"));

    int flags = 0;
    int r = gen_browser(page, XXX,&buffer,
			Fcall,
			word_open,NULL,
			aview,
			NULL,
			CATGETS(elm_msg_cat, ElmSet, 
				ElmSelectFolder,
				"Select folder: "));
    int err = 0;
    
    if (!r)
	goto cleanup;
    
    flags = give_dir_flags(XXX);
    
    if (!flags) {		    
	goto cleanup;	       	
    }
    
    folder = folder_from_dir_item(XXX, TreatAsSpooled);
    if (!folder)
	goto cleanup;

    switch (sessionlock_folder(folder,SESSIONLOCK_CHECK,&err,NULL)) {
    case sessionlock_fail:
	DPRINT(Debug,1,
	       (&Debug, 
		"Error: given file %s as folder - unreadable", 
		folder->cur_folder_sys));
	if (err) {
	    DPRINT(Debug,1, (&Debug, " (%s); errno=%d",
			     strerror(err),err));
	}
	DPRINT(Debug,1,(&Debug,"\n"));
	
	/* Free folder structure */
	{
	    int r;

	    struct cancel_data *cd = NULL;

	    setup_mbx_cancel_message(&cd,mbx_leaving_mailbox);

	    DPRINT(Debug,9, (&Debug, 
			     "select_folder: freeing folder\n"));
	    	    
	    r = leave_old_folder(&folder,CLOSE_NORMAL,cd);
	    if (!r) {
		DPRINT(Debug,9, (&Debug, 
				 "select_folder: leave_old_folder failed\n"));
	    }
	    if (is_canceled(cd)) {
		DPRINT(Debug,9, (&Debug, 
				 "select_folder: Cancel on leave_old_folder\n"));
	    }
	    free_cancel(&cd);	    
	}
	
	break;
    case sessionlock_open:
    case sessionlock_reconnect:
	break;
    }
    
 cleanup:
    if (Fcall)
	free_fbrowser_call(&Fcall);

    if (buffer)
	free_string(&buffer);

    free_dir(&XXX);
    
    return folder;
}

static struct folder_info *arg_to_folder P_((char * arg, int prompt,
					     struct AliasView *aview,
					     struct menu_context *page));
static struct folder_info *arg_to_folder(arg,prompt,aview, page)
     char * arg; 
     int prompt;
     struct AliasView *aview;
     struct menu_context *page;
{
    struct folder_info *folder = NULL;

    /* Assuming that file name given with -f is on 
       local-fs-charset and not system_charset
    */
    struct folder_browser * XXX    = new_browser(selection_folder);
    struct string         * buffer = new_string2(local_fs_charset,
						 s2us(arg));
		
    struct fbrowser_call *Fcall = NULL;


    int flags = 0;
    int err = 0;
    
    if (!browser_expand(XXX,&buffer,NULL,aview,NULL)) {
	int r;

	if (!prompt)
	    goto cleanup;

	if (with_title && !with_title[0] && !wanted_title_string) { 
	    
	    set_terminal_titles(catgets(elm_msg_cat, 
					ElmSet, ElmTitleFolderSel, 
					"Elm: Folder selection"),
				wanted_icon_string);
	}
	
	if (with_title)
	    with_title = "";
	
	Fcall = alloc_fbrowser_call_f(FB_EXIST|FB_READ,
				     CATGETS(elm_msg_cat, ElmSet, 
					     ElmSelectFolderFB,
					     "Folder opening selection"));

    return_browser:

	r = gen_browser(page, XXX,&buffer,
			Fcall,
			word_open,NULL,
			aview,
			NULL,
			CATGETS(elm_msg_cat, ElmSet, 
				ElmSelectFolder,
				"Select folder: "));
					     

	if (!r)
	    goto cleanup;

	
	flags = give_dir_flags(XXX);

    } else {
    
	/* Returns 0 if no selection */
	
	flags = give_dir_flags(XXX);
    
	DPRINT(Debug,4, (&Debug, 
			 "arg_to_folder: %S have flags:%s%s%s%s%s%s%s%s\n",
			 buffer,
			 flags & BROWSER_NODIR    ?   " NODIR":    "",
			 flags & BROWSER_NOFOLDER ?   " NOFOLDER": "",
			 flags & BROWSER_MARKED   ?   " MARKED":   "",
			 
			 flags & BROWSER_MAILFILE ?   " MAILFILE": "",
			 flags & BROWSER_SELECTED ?   " SELECTED": "",
			 flags & BROWSER_EXIST    ?   " EXIST"   : "",
			 flags & BROWSER_DIRPREFIX ?   " DIRPREFIX"   : "",
			 !flags                   ?   " none"    : ""));
	

	/* Call fbrowser ? */
	if (prompt &&
	    0 != (flags & BROWSER_DIRPREFIX) &&
	    /* Folder == mailbox (normally file)*/
	    0 != (flags & BROWSER_NOFOLDER) &&   
	    0 == (flags & BROWSER_NODIR) &&   
	    fbrowser_supported_on_dir(XXX)) {
	    
	    enum fbrowser_result fbres;

	    Fcall = alloc_fbrowser_call_f(FB_EXIST|FB_READ,
					  CATGETS(elm_msg_cat, ElmSet, 
						  ElmSelectFolderFB,
						  "Folder opening selection"));

	    set_fbrowser_flags(Fcall,FBROWSER_USE_SELECTION);


	    fbres = fbrowser(page,XXX,NULL,&buffer,Fcall);

	    
	    switch(fbres) {
	    case fbrowser_INTR:
	    case fbrowser_quit:
	    case fbrowser_error:

		DPRINT(Debug,4, (&Debug, 
				 "arg_to_folder: fbrowser=%d cleanup\n",
				 fbres));

		goto cleanup;

	    case fbrowser_return_browser:

		DPRINT(Debug,4, (&Debug, 
				 "arg_to_folder: fbrowser=%d return to browser\n",
				 fbres));


		set_fbrowser_flags(Fcall,FBROWSER_RERUN);

		goto return_browser;

	    case fbrowser_selected:
		DPRINT(Debug,4, (&Debug, 
				 "arg_to_folder: fbrowser=%d selected\n",
				 fbres));
		break;

	    default:
 		DPRINT(Debug,4, (&Debug, 
				 "arg_to_folder: fbrowser=%d bad value\n",
				 fbres));

		panic("BROWSER PANIC",__FILE__,__LINE__,"arg_to_folder",
		      "Bad return value of fbrowser",0);

	    }
	    	 
	    flags = give_dir_flags(XXX);

	}	

    }


    if (!flags) {		    
	goto cleanup;	       	
    }
       
    folder = folder_from_dir_item(XXX, TreatAsSpooled);
    if (!folder)
	goto cleanup;

    switch (sessionlock_folder(folder,SESSIONLOCK_CHECK,&err,NULL)) {
    case sessionlock_fail:
	DPRINT(Debug,1,
	       (&Debug, 
		"Error: given file %s as folder - unreadable", 
		folder->cur_folder_sys));
	if (err) {
	    DPRINT(Debug,1,(&Debug, " (%s); errno=%d",
			    strerror(err),err));
	}
	DPRINT(Debug,1,(&Debug, "\n"));
	
	/* Free folder structure */
	{
            int r;

	    struct cancel_data *cd = NULL;

	    setup_mbx_cancel_message(&cd,mbx_leaving_mailbox);
	    
	    DPRINT(Debug,9, (&Debug, 
			     "arg_to_folder: freeing folder\n"));
	    
	    r = leave_old_folder (&folder,CLOSE_NORMAL,cd);
	    if (!r) {
		DPRINT(Debug,9, (&Debug, 
				 "arg_to_folder: leave_old_folder failed\n"));
	    }
	    if (is_canceled(cd)) {
		DPRINT(Debug,9, (&Debug, 
				 "arg_to_folder: Cancel on leave_old_folder\n"));
	    }
	    free_cancel(&cd);       
        }        	
	break;

    case sessionlock_open:
    case sessionlock_reconnect:
	break;
    }

 cleanup:
    if (Fcall)
	free_fbrowser_call(&Fcall);

    if (buffer)
	free_string(&buffer);

    free_dir(&XXX);
    
    return folder;
}

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

#ifdef REMOTE_MBX
S_(dump_conf_map_f dump_user_mail_services)
static int dump_user_mail_services P_((FILE *f,const char *actor,
				       char *version_buff,
				       int * errno_res));
static int dump_user_mail_services(f,actor,version_buff,errno_res) 
     FILE *f;
     const char *actor;
     char *version_buff;
     int *errno_res;
{
    int r = 1;
    
    if (user_mail_services_conf)
	r = dump_service_entries(f,user_mail_services_conf,
				 NULL,actor,version_buff,
				 user_mail_services_cs,
				 user_mail_services_pl);

    return r; 
}

S_(dump_message_f user_mail_services_written)
static void user_mail_services_written P_((char *fname));
static void user_mail_services_written(fname)
     char *fname;
{
    lib_error(CATGETS(elm_msg_cat, ElmSet, 
		      ElmMailServInfoSavedIn,
		      "Mail service info saved in file %s."), 
	      user_mail_services);
}
#endif


void initialize(main_aliases, 
		page)
     struct AliasView **main_aliases;
     struct menu_context **page;
{

    int init_scr_return; 
    const int read_flags = 0;
    int configerror = 0;
    
    
    elm_sfprintf(version_buff, sizeof version_buff,
		 FRM("%s PL%s"), VERSION, PATCHLEVEL);

    utils_open_debug();
    
    def_ans_yes = catgets(elm_msg_cat, ElmSet, ElmYes, "y");
    def_ans_no = catgets(elm_msg_cat, ElmSet, ElmNo, "n");
    nls_deleted = catgets(elm_msg_cat, ElmSet, ElmTitleDeleted, "[deleted]");
    nls_message = catgets(elm_msg_cat, ElmSet, ElmTitleMessage, "Message");
    nls_to = catgets(elm_msg_cat, ElmSet, ElmTitleTo, "To");
    nls_from = catgets(elm_msg_cat, ElmSet, ElmTitleFrom, "From");
    nls_page = catgets(elm_msg_cat, ElmSet, ElmTitlePage, "  Page %d");
    save_word = catgets(elm_msg_cat, ElmSet, ElmSave, "save");
    copy_word = catgets(elm_msg_cat, ElmSet, ElmCopy, "copy");
    cap_save_word = catgets(elm_msg_cat, ElmSet, ElmCapSave, "Save");
    cap_copy_word = catgets(elm_msg_cat, ElmSet, ElmCapCopy, "Copy");
    saved_word = catgets(elm_msg_cat, ElmSet, ElmSaved, "saved");
    copied_word = catgets(elm_msg_cat, ElmSet, ElmCopied, "copied");
    
    Stopped_Text = catgets(elm_msg_cat, ElmSet, ElmStoppedUseFGToReturn,
			   "\n\nStopped.  Use \"fg\" to return to ELM\n\n");
    Back_Text    = catgets(elm_msg_cat, ElmSet, ElmBackInElmRedraw,
			   "\nBack in ELM. (You might need to explicitly request a redraw.)\n\n");
    Segv_Text = catgets(elm_msg_cat, ElmSet, ElmSegmentViolationSignal,
			"\n\nSegment Violation signal!\n\n");
    Bus_Text  = catgets(elm_msg_cat, ElmSet, ElmBusErrorSignal,
			"\n\nBus Error signal!\n\n");
    Fpe_Text  = catgets(elm_msg_cat, ElmSet, ElmFloatingPointSignal,
			"\n\nFloating Point Exception signal!\n\n");
    Ill_Text  = catgets(elm_msg_cat, ElmSet, ElmIllegalInstructionSignal,
			"\n\nIllegal Instruction signal!\n\n");
    /*
     * Install the error trap to take if xmalloc() or friends fail.
     */
    safe_malloc_fail_handler = malloc_failed_exit;

    /* Assumes that locale_init() is called */

    init_opts_menu();   
    init_scr_return = InitScreen(page);
    
    out_util_setup();

    /* make all newly created files private */
    original_umask = umask(077);

    /* get username and home directory 
       save original user and group ids 
    */
    user_init();

    /* user_init sets have_saved_ids, mailgroupid, userid and groupid */
    if (have_saved_ids) {
	DPRINT(Debug,4,(&Debug, 
			"*** setting gid=%d (mailgroupid=%d)\n",
			groupid,mailgroupid));
	
	if (setgid(groupid) < 0) {
	    int err = errno;
	    lib_error(FRM("init: setgid(%d) FAILED: %s"),
		      groupid,strerror(err));
	}
    }
    
    /* make all newly created files private */
    original_umask = umask(077);
        
#ifdef DEBUG
    { 
	int d = panic_dprint("\n\
======================================================\n\
Debug output of the ELM program (version %s).\n",
			     version_buff);
	
	if (d >= 50) {
	    panic_dprint("WARNING: Edit manually out sensitive information from that file!\n");
    
	    lower_prompt("WARNING: Debug file may include passwords -- edit it!");
	    error_sleep(5+sleepmsg);	    
	}
    }
#endif


    init_addrlib(read_flags);
    init_mailerlib(read_flags);
    init_mboxlib(read_flags);
    init_misclib(read_flags);
    init_aliaslib(read_flags);
    init_melib(read_flags);
    
#ifdef REMOTE_MBX
    register_conf_write(USER_MAIL_SERVICES,user_mail_services,
			dump_user_mail_services,
			user_mail_services_written,
			NULL /* No automatic rewrite */,
			NO_merge_conf_map
			);
#endif


    /* Determine options that might be set in the global elmrc */
    if (init_defaults(read_flags) != 0) {
	sleep(1);
    }


    /* Determine options that might be set in the .elm/elmrc */
    if (read_rc_file(read_flags) != 0) {
	sleep(1);
	if (OPMODE_IS_INTERACTIVE(opmode) && !write_elmrc) {
	    struct string * andor = 
		format_string(CATGETS(elm_msg_cat, ElmSet,
				      ElmAndOr,"and/or"));
	    struct string * tags = conf_taglist(andor);
	
						
	    lib_error(CATGETS(elm_msg_cat, ElmSet,
			      ElmFixConfFiles,
			      "Fix %s, %S or let elm rebuild them with option '-w'."),
		      ELMRCFILE, tags);

	    error_wait();   /* Be sure that we do not exit before error is shown */

	    free_string(&tags);
	    free_string(&andor);

	    exit(1);
	}

	configerror = 1;
    }

    curses_check_terminal();
    
    if (!check_mail_file_list()) {
	exit(1);
    }

    
    if (have_DW_characters && wanted_display_charset &&
	wanted_display_charset != system_charset) {
	
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmArgsBadDW2,
			  "Option --have-double-wide-characters do not work with displaycharset=%s option.\n"),
		  raw_display_charset);
	
	exit(1);
    }

    /* directory_check() requires terminal */

    if (OPMODE_IS_INTERACTIVE(opmode)) {
	int lines   = -1;
	int columns = -1;
	char *cp;

	if (init_scr_return < 0) {
	    if (init_scr_return == -1) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmNoTerm,
				  "Sorry, but you must specify what type of terminal you're on if you want to\n\
run the \"elm\" program. (You need your environment variable \"TERM\" set.)\n"));
		DPRINT(Debug,1,
		       (&Debug, 
			"No $TERM variable in environment!\n"));
	    }
	    else if (init_scr_return == -2) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadTerm,
				  "You need a cursor-addressable terminal to run \"elm\" and I can't find any\n\
kind of termcap entry for \"%s\" - check your \"TERM\" setting...\n"),
			  getenv("TERM"));
		DPRINT(Debug,1,
		       (&Debug, 
			"$TERM variable is an unknown terminal type!\n"));
	    } else {
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmTermInitFailed,
				  "Failed trying to initialize your terminal entry: unknown return code %d\n"), 
			  init_scr_return);
		DPRINT(Debug,1,
		       (&Debug, 
			"Initscreen returned unknown code: %d\n",
			init_scr_return));
	    }
	    error_wait();   /* Be sure that we do not exit before error is shown */
	    exit(1);	    /* all the errors share this exit statement */
	}

	/** check to see if the user has defined a LINES or COLUMNS
	    value different to that in the termcap entry (for
	    windowing systems, of course!) **/


       
	if ((cp = getenv("LINES")) != NULL && isdigit(*cp)) {
	    sscanf(cp, "%d", &lines);
	}

	if ((cp = getenv("COLUMNS")) != NULL && isdigit(*cp))
	    sscanf(cp, "%d", &columns);

	set_root_menu(lines,columns);


	/* And read changes what just implied */
	menu_resized(*page);
    }

    /* Check if .elm and Mail directories exists */
    directory_check();

    /*
     * Locking problems are common when Elm is misconfigured/misinstalled.
     * Sometimes, rather than fixing the problem, people will set
     * Elm setuid=root.  This is *extremely* dangerous.  Make sure
     * this isn't happening.  This test needed to be deferred until
     * after the system rcfile was read in.
     */
    if (getuid() != geteuid() && !allow_setuid) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmInstalledSetuid,
"\n\
This version of Elm has been installed setuid=%d.  This is dangerous!\n\
Elm is NOT designed to run in this mode, and to do so can introduce\n\
grave security hazards.  \n\n"),
		  geteuid());
	
	error_wait();/* Be sure that we do not exit before error is shown */
	exit(1);
    }


    /*
     * Now that we are about to muck with the tty and mail folder,
     * it's a good time to get the signal handlers installed.
     */


    /*
     * If debug level is fairly low, ignore keyboard signals
     */

#ifdef DEBUG
    /* FIXME -- Probably wrong variable ... */
    if (Debug.active < 5) {	/* otherwise let the system trap 'em! */
	init_signals();
    }
    else {
	DPRINT(Debug,3,
	       (&Debug, 
		"\n*** Elm-Internal Signal Handlers Disabled due to debug level %d ***\n\n",
		Debug.active));
    }
#else
    init_signals();
#endif

    init_signals1();

#ifdef BACKGROUD_PROCESSES
    init_backgroud_handling();
#endif

    
    *main_aliases = new_aliasview();

    add_files_to_aview(*main_aliases);

    /* Check if old aliases file exists */
    aliases_check(*main_aliases);

    if (conf_backup_suffix) {

	int ok = 1;

	if (! check_options_backup_suffix("-b",conf_backup_suffix))
	    ok = 0;
		
	if (!check_conf_backup_suffix("-b",conf_backup_suffix))
	    ok = 0;
	
	if (!ok) {
	    error_wait(); /* Be sure that we do not exit before error is shown */
	    exit(1);
	}
    }
    
    DPRINT(Debug,12,(&Debug, "initialize: mini_menu=%d mini_menu_rc=%d ",
		     mini_menu,mini_menu_rc));

    mini_menu  &=   mini_menu_rc;

    DPRINT(Debug,12,(&Debug, " => mini_menu=%d\n",mini_menu));


    if (write_elmrc) {

	save_options(conf_backup_suffix);

	write_conf("ELM",version_buff,write_conf_full_rewrite,
		   conf_backup_suffix);
	
	error_sleep(1);
    } 

    if (OPMODE_IS_CHECKMODE(opmode)) {
	    DPRINT(Debug,5,
		   (&Debug, 
		    "initialize: check_only (-c) -- skipping rest of initialize.\n"));
	    return;
    }

    if (!Check_attachments()) {
	error_wait();   /* Be sure that we do not exit before error is shown */
	exit(1);
    }

    if (OPMODE_IS_INTERACTIVE(opmode)) {
	/* Now that we've read the rc file we can enter RAW mode */
	Raw(ON);
	/** clear the screen **/
	menu_ClearScreen(*page);

    }

    /* Possible terminal initialization for system charset */
    init_system_charset();

    if (-1 != wanted_switchmode)
	allow_charset_switching = wanted_switchmode;

    if (init_scr_return >= 0) {

	if (wanted_charset) {
		charset_t X = MIME_name_to_charset(wanted_charset,0);
		if (!X) {
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmArgsCharsetUnknown,
				      "Charset %s given with -D flag is unknown."),
			      wanted_charset);
		} else if (set_display_charset(X,0,
					       0 /* not signal */)) {
		    CarriageReturn();
		    Write_to_screen(CATGETS(elm_msg_cat, ElmSet, 
					    ElmDisplayCharsetChanged,
					    "[display charset changed to %s]"),
				    wanted_charset);
		    CarriageReturn();
		} else {
		    DPRINT(Debug,1,
			   (&Debug, 
			    "No way to set terminal to charset %s (-D option)\n",
			    wanted_charset));
		}
	} else if (wanted_display_charset &&
		   wanted_display_charset != system_charset) {
	    if (set_display_charset(wanted_display_charset,1,
				    0 /* No signal */)) {
		    const char *MIME_name_d = 
			get_charset_MIME_name(display_charset);

		    CarriageReturn();
		    Write_to_screen(CATGETS(elm_msg_cat, ElmSet, 
					    ElmDisplayCharsetChanged,
					    "[display charset changed to %s]"),
				    MIME_name_d ? 
				    MIME_name_d : 
				    "<no MIME name>");
		    CarriageReturn();
		} 
	}
	    
	if (wanted_title_string || wanted_icon_string) {
		if (! set_terminal_titles(wanted_title_string,
					  wanted_icon_string)) {
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmArgsTitleNoSet,
				      "Title (or icon) name for terminal can not set"));
		}
	}

	if (with_title && with_title[0]) {
		if (! set_terminal_titles(with_title,NULL)) {
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmArgsTitleNoSet1,
				      "Title name for terminal can not set"));
		}
	}
    }	
    	
    post_process_charset_options();

#ifdef DEBUG
    if (Debug.active >= 2 && Debug.active < 10) {
	debug_action_call(&Debug,
			  "hostname = %-20s \tusername = %-20s ",
			  hostname, username); 
	if (full_username_s) {
	    debug_action_call(&Debug,"\tfullname = %-20S",full_username_s);
	}
	debug_action_call(&Debug,"\n");
	debug_action_call(&Debug,	  
			  "home     = %-20s \n",
			  home);
	
	if (quote_prefix)
	    debug_action_call(&Debug,
			      " \tprefix   = %-20S \t\n\n",
			      quote_prefix);
    }
#endif


    if (! configerror) {
	/* Do automatic rewrites */
	write_conf("ELM",version_buff,write_conf_auto_rewrite,
		   conf_backup_suffix);
    }
}

static int ok_to_open_mailbox P_((struct MailboxView *main_mailbox));
static int ok_to_open_mailbox(main_mailbox)
     struct MailboxView *main_mailbox;
{
    int tmp = opmode & OPMODE_MASK;

    if (tmp != OPMODE_READ_NONEMPTY &&
	tmp != OPMODE_READ_UNREAD  &&
	tmp != OPMODE_READ_SELECT  &&
	tmp != OPMODE_READ) {

	DPRINT(Debug,1,(&Debug, 
			"ok_to_open_mailbox: Bad opmode=%d (masked=%d)\n",
			opmode,tmp));

	return 0;
    }

    if (tmp == OPMODE_READ_NONEMPTY ||
	tmp == OPMODE_READ_UNREAD) {
	int mc = get_message_count(main_mailbox);
	
	if (0 == mc) {
	    
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmNoMail,
			   "You have no mail."));
	    
	    return 0;
	}
    }


    if (tmp == OPMODE_READ_UNREAD) {
	int i;
	int unread = 0;
	int mc = get_message_count(main_mailbox);
       
	for (i = 0; i < mc; i++) {
	    if (ison_status_message(main_mailbox,i,status_basic,UNREAD))
		unread++;
	}

	if (!unread) {

	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmNoUnreadMail,
			      "You have no unread mail."));

	    return 0;
	}
    }

    return 1;
}

struct MailboxView * initialize_mailbox(requestedmfiles,  
					parent_page, main_aliases,
					ret_letter,cmds
					)
     char               ** requestedmfiles;  /* first mail files to open, empty if the default */    
     struct menu_context * parent_page;
     struct AliasView    * main_aliases;
     int                 * ret_letter;
     struct elm_commands * cmds /* may be NULL */;
{
    struct MailboxView   * main_mailbox = NULL;
    struct quota_display * quotad = NULL;
    struct cancel_data   * cd = NULL;
    
    int tmp = opmode & OPMODE_MASK;

    enum show_mail_quota_v  smq =
	give_dt_enumerate_as_int(&show_mail_quota_e); 
    
    DPRINT(Debug,20, (&Debug, 
		      "initialize_mailbox: show-mail-quota=%d",smq));
    switch (smq) {
    case show_mquota_OFF:     DPRINT(Debug,20, (&Debug, " show_mquota_OFF"));     break;
    case show_mquota_ON_open: DPRINT(Debug,20, (&Debug, " show_mquota_ON_open")); break;
    case NUM_show_mail_quota: DPRINT(Debug,20, (&Debug, " NUM_show_mail_quota")); break;
    }
    DPRINT(Debug,20, (&Debug, "\n"));
    
    if (ret_letter)
	*ret_letter = '\0';
    
    if (! OPMODE_IS_READMODE(tmp)) {

	DPRINT(Debug,1,(&Debug, 
			"initialize_mailbox: Bad opmode=%d (masked=%d)\n",
			opmode,tmp));

	return NULL;
    }

    if (mail_files_added == mail_files_used()) {
	char * one_mail_file = NULL;
	char * title         = NULL;
	char * icon          = NULL;
	
	main_mailbox = mailbox_view_mail_files(&one_mail_file);

	resort_mailbox(main_mailbox,0);

	if (with_title && with_title[0])
	    title =  with_title;
	else if (with_title || set_window_title) {
	    if (wanted_title_string)
		title = wanted_title_string;
	    else if (one_mail_file)
		title =
		    elm_message(CATGETS(elm_msg_cat, ElmSet, 
					ElmWindowTileMalFileN,
					"Elm: mail-file %s"),
				one_mail_file);
	    else
		title =
		    elm_message(CATGETS(elm_msg_cat, ElmSet, 
					ElmWindowTileMalFileCol,
					"Elm: Mail files collection"));
	}

	if (with_title || set_window_title) {
	    if (wanted_icon_string)
		icon = wanted_icon_string;
	    else if (one_mail_file) {
		/* Keep it simple ... */

		char * x = strrchr(one_mail_file,'/');
		if (x)
		    x++;
		else 
		    x = one_mail_file;

		icon = elm_message(CATGETS(elm_msg_cat, ElmSet, 
					   ElmWindowIcon,
					   "Elm: %s"),
				   x);
	    } else
		icon = elm_message(CATGETS(elm_msg_cat, ElmSet, 
					   ElmWindowIconMailFile,
					   "Elm: Mail files"));
	}

	if (title || icon)
	    set_terminal_titles(title,icon);

	if (one_mail_file)
	    free(one_mail_file);
	if (title &&
	    title != wanted_title_string &&
	    title != with_title)
	    free(title);
	if (icon &&
	    icon != wanted_icon_string)
	    free(icon);
		
    } else {

	struct folder_info *folder = NULL;
	int several_folders = 0;
	int start_idx = 0;  
	
	if ((opmode & OPMODE_READ_SELECT) == OPMODE_READ_SELECT) {
	    
	    if (with_title && !with_title[0] && !wanted_title_string) { 
		
		set_terminal_titles(catgets(elm_msg_cat, 
					    ElmSet, ElmTitleFolderSel, 
					    "Elm: Folder selection"),
				    wanted_icon_string);
	    }
	    
	    if (with_title)
		with_title = "";
	    
	    folder = select_folder(main_aliases, parent_page);
	    
	    if (requestedmfiles && requestedmfiles[0]) {	    
		several_folders++;
		start_idx = 0;
	    }

	    
	    
	    /* Determine the mail file to read */
	} else if (!requestedmfiles || ! requestedmfiles[0]) {
	    unsigned int defaultfile_keyword = 0;
	    unsigned int defaultfile_cmask   = 0;
	    
	    const char * default_val = 
		give_dt_estr_as_str(&defaultfile_e,
				    "incoming-mailbox",
				    &defaultfile_keyword,
				    &defaultfile_cmask);
	    
	    /* enter_new_folder() allows NULL folder name */
	    folder = enter_new_folder(default_val, TreatAsSpooled,
				      defaultfile_keyword,
				      defaultfile_cmask);
	} else {
	    /* Assuming that file name given with -f is on 
	       local-fs-charset and not system_charset
	    */
	    
	    if (requestedmfiles[1]) {
		several_folders++;
		start_idx = 1;  /* Index 0 is already processed! */
	    }	
	    
	    folder = arg_to_folder(requestedmfiles[0], !several_folders,
				   main_aliases, parent_page);
	}
	
	if (!folder && !several_folders) {
	    return NULL;
	} 

	if (folder) {
	    struct mail_quota   * mquota = NULL;
	    
	    /* if "check_size" is set then we don't want to read empty mbox */
	    if (tmp == OPMODE_READ_NONEMPTY && !several_folders) {
		if(check_mailfile_size(folder) != 0) {
		    int r;

		    setup_mbx_cancel_message(&cd,mbx_leaving_mailbox);
		    
		    DPRINT(Debug,9, (&Debug, 
				     "initialize_mailbox: freeing folder\n"));		    
		    r = leave_old_folder(&folder,CLOSE_NORMAL, cd);

		    if (!r) {
			DPRINT(Debug,9, (&Debug, 
					 "initialize_mailbox: leave_old_folder failed\n"));
		    }
		    if (is_canceled(cd)) {
			DPRINT(Debug,9, (&Debug, 
					 "initialize_mailbox: Cancel on leave_old_folder\n"));
		    }
		    
		    free_cancel(&cd);	    
		}
	    }
	    
	    if (with_title) 
		set_folder_window_title(folder,
					with_title[0] ? with_title : wanted_title_string,
					wanted_icon_string);
	    else if (set_window_title)
		set_folder_window_title(folder,
					wanted_title_string,
					wanted_icon_string);

	    switch (smq) {
	    case show_mquota_OFF:
		DPRINT(Debug,20, (&Debug, 
				  "initialize_mailbox: mail quota display disabled\n"));
		break;
	    case show_mquota_ON_open: 
		mquota = have_folder_quota(folder);
		DPRINT(Debug,20, (&Debug, 
				  "initialize_mailbox: mail quota %s\n",
				  mquota ? "supported" : "not available"));
		break;
	    case NUM_show_mail_quota:  break;
	    }

	    if (mquota) {
		setup_mbx_cancel_message(&cd,mbx_checking_mailbox);
						
		quotad = new_quota_display(mquota,parent_page,cd);

		if (is_canceled(cd)) {
		    DPRINT(Debug,9, (&Debug, 
				     "initialize_mailbox: Cancel on new_quota_display\n"));

		    if (mquota)
			free_mail_quota(&mquota);	
		    		    
		    if (main_mailbox)
			goto close_main_mailbox;
		    goto cleanup;
		}
		
		free_cancel(&cd);
	    }
	    
	    /* read in the folder! 
	       initializes main_mailbox	       
	     */
	    newmbox_1(folder, &main_mailbox, FALSE, 
		      parent_page,0,NULL,quotad);

	    if (mquota)
		free_mail_quota(&mquota);
	}

	if (several_folders) {
	    int i;
	    
	    for (i = start_idx; requestedmfiles[i]; i++) {
		struct folder_info *folder;

		/* Need swap current page / menu context */
		if (quotad) 
		    menu_set_default(parent_page);

		folder = 
		    arg_to_folder(requestedmfiles[i],FALSE,main_aliases,
				  parent_page);
		
		if (folder) {
		    struct mail_quota   * mquota = NULL;


		    switch (smq) {
		    case show_mquota_OFF:
			DPRINT(Debug,20, (&Debug, 
					  "initialize_mailbox: mail quota display disabled\n"));
			break;
		    case show_mquota_ON_open: 
			mquota = have_folder_quota(folder);
	    		DPRINT(Debug,20, (&Debug, 
					  "initialize_mailbox: mail quota %s\n",
					  mquota ? "supported" : "not available"));
			break;
		    case NUM_show_mail_quota:  break;
		    }

		    if (mquota) {

			setup_mbx_cancel_message(&cd,mbx_checking_mailbox);
									
			if (quotad) {
			    
			    /* Need swap current page / menu context */
			    menu_set_default(quota_display_current_page(quotad));
			    
			    quota_display_add_quota(quotad,mquota,cd);
			} else
			    quotad = new_quota_display(mquota,parent_page,cd);

			if (is_canceled(cd)) {
			    DPRINT(Debug,9, (&Debug, 
					     "initialize_mailbox: Cancel on quota_display_add_quota or new_quota_display\n"));

			    free_mail_quota(&mquota);	
			    
			    if (main_mailbox)
				goto close_main_mailbox;
			    goto cleanup;
			}

			free_cancel(&cd);
			free_mail_quota(&mquota);		    
		    }
		    
		    newmbox_1(folder, &main_mailbox,
			      main_mailbox ? TRUE : FALSE,
			      parent_page,0,NULL,quotad);
		}
	    }
	    
	    if (main_mailbox && wanted_mailbox_title) {
		/* must not free -- shared/stored to structure
		   -- set_mailbox_title will free_string()
		   it later */
		set_mailbox_title(main_mailbox,
			      new_string2(system_charset,
					  s2us(wanted_mailbox_title)));
	    }
	
	}
    }

    if (ret_letter && quotad)
	*ret_letter = quota_display_show(quotad,cmds);
    
    if (!main_mailbox) 
	goto cleanup;
    	
    if (!ok_to_open_mailbox(main_mailbox)) {
	int r;

    close_main_mailbox:

	setup_mbx_cancel_message(&cd,mbx_leaving_mailbox);

	DPRINT(Debug,9, (&Debug, 
                         "initialize_mailbox: freeing main_mailbox\n"));
		
	/* This also free's folder */
	r = free_mailbox(& main_mailbox,cd);
	if (!r) {
            DPRINT(Debug,9, (&Debug, 
                             "initialize_mailbox: free_mailbox failed\n"));
        }       
        if (is_canceled(cd)) {
            DPRINT(Debug,9, (&Debug, 
                             "initialize_mailbox: Cancel on free_mailbox\n"));
        }		
    }

 cleanup:
    if (cd)
	free_cancel(&cd);
    
    if (quotad)
	free_quota_display(&quotad,parent_page);
    
    DPRINT(Debug,9, (&Debug, 
		     "initialize_mailbox="));
    if (main_mailbox) {
	DPRINT(Debug,9, (&Debug, "%p",main_mailbox));
    } else {
	DPRINT(Debug,9, (&Debug, "NULL"));
    }
    if (ret_letter) {
	DPRINT(Debug,9, (&Debug, "; *ret_letter=%d",
			 *ret_letter));
	if (isascii(*ret_letter)) {
	    DPRINT(Debug,9, (&Debug, " (%c)",
			     *ret_letter));
	} else if (EOF ==  *ret_letter) {
	    DPRINT(Debug,9, (&Debug, " EOF"));
	}
    }    
    DPRINT(Debug,9, (&Debug,"\n"));
    
    return main_mailbox;
}


/* May not be 
      uin16 port
   because this is promoted to int
*/


/* Return 1 for open */
static int no_verify_remote_url P_((const struct url *url,
				    const struct string  * user,
				    const struct string  * host,
				    struct string  ** anon_passwd,  /* prompt for 'anon password' */
				    int port,
				    int default_port,
				    int   * ret_letter /* returns '\0' or EOF */));
static int no_verify_remote_url(url,user,host,anon_passwd,port,default_port,ret_letter)
     const struct url *url;
     const struct string  * user;
     const struct string  * host;
     struct string  ** anon_passwd;  /* prompt for 'anon password' */
     int port;
     int default_port;
     int   * ret_letter /* returns '\0' or EOF */;
{
    return 1;
}

struct MailboxView * initialize_mailbox_from_url(url,parent_page,ask,ret_letter,cmds)
     struct url * url;
     struct menu_context *parent_page;
     int ask;
     int   * ret_letter;
     struct elm_commands        * cmds /* may be NULL */;
{
    struct MailboxView *main_mailbox = NULL;

    struct folder_info *folder = NULL;
    struct cancel_data *cd = NULL;
    struct mail_quota  *mquota = NULL;
    struct quota_display *quotad = NULL;
    
    int tmp = opmode & OPMODE_MASK;

    enum show_mail_quota_v  smq =
	give_dt_enumerate_as_int(&show_mail_quota_e); 
    
    DPRINT(Debug,20, (&Debug, 
		      "initialize_mailbox_from_url: show-mail-quota=%d",smq));
    switch (smq) {
    case show_mquota_OFF:     DPRINT(Debug,20, (&Debug, " show_mquota_OFF"));     break;
    case show_mquota_ON_open: DPRINT(Debug,20, (&Debug, " show_mquota_ON_open")); break;
    case NUM_show_mail_quota: DPRINT(Debug,20, (&Debug, " NUM_show_mail_quota")); break;
    }
    DPRINT(Debug,20, (&Debug, "\n"));

    
     if (ret_letter)
	*ret_letter = '\0';
        
    if (! OPMODE_IS_READMODE(tmp)) {

	DPRINT(Debug,1,(&Debug, 
			"initialize_mailbox_from_url: Bad opmode=%d (masked=%d)\n",
			opmode,tmp));

	return NULL;
    }



    folder = get_folder_from_url(url,
				 ask ? verify_remote_url :
				 no_verify_remote_url,
				 ret_letter);
    
    if (!folder || (ret_letter && *ret_letter)) {
	goto fail;
    }

    if (tmp == OPMODE_READ_NONEMPTY) {
	if(check_mailfile_size(folder) != 0) {
	    
	    goto fail;
	}
    }

    switch (smq) {
    case show_mquota_OFF:
	DPRINT(Debug,20, (&Debug, 
			  "initialize_mailbox_from_url: mail quota display disabled\n"));
	break;
    case show_mquota_ON_open: 
	mquota = have_folder_quota(folder);
	DPRINT(Debug,20, (&Debug, 
			  "initialize_mailbox_from_url: mail quota %s\n",
			  mquota ? "supported" : "not available"));
	break;
    case NUM_show_mail_quota:  break;
    }

    if (mquota) {

	setup_mbx_cancel_message(&cd,mbx_checking_mailbox);
	
	quotad = new_quota_display(mquota,parent_page,cd);

	if (is_canceled(cd)) {
	    DPRINT(Debug,9, (&Debug, 
			     "initialize_mailbox: Cancel on new_quota_display\n"));
	    
	    if (main_mailbox)
		goto close_main_mailbox;
	    goto fail;
	}
		
	free_cancel(&cd);
    }
	
    /* folder is set to NULL, if it is not needed
       to free

       read in the folder! 
    */
    
    main_mailbox = enter_mailbox(&folder,parent_page,quotad);

    if (ret_letter && quotad)
	*ret_letter = quota_display_show(quotad,cmds);
        
    if (!main_mailbox)
	goto fail;
    
    if (!ok_to_open_mailbox(main_mailbox)) {
	int r;

    close_main_mailbox:
	setup_mbx_cancel_message(&cd,mbx_leaving_mailbox);

	DPRINT(Debug,9, (&Debug, 
                         "initialize_mailbox_from_url: freeing main_mailbox\n"));
	
	/* This also free's folder */
	r = free_mailbox(& main_mailbox, cd);
	if (!r) {
            DPRINT(Debug,9, (&Debug, 
                             "initialize_mailbox_from_url: free_mailbox failed\n"));
        }       
        if (is_canceled(cd)) {
            DPRINT(Debug,9, (&Debug, 
                             "initialize_mailbox_from_url: Cancel on free_mailbox\n"));
        }
    }

 fail:
    if (folder) {
	int r;

	setup_mbx_cancel_message(&cd,mbx_leaving_mailbox);
	
	DPRINT(Debug,9, (&Debug, 
                         "initialize_mailbox_from_url: freeing folder\n"));
	
	r = leave_old_folder(&folder,CLOSE_NORMAL,cd);

	if (!r) {
            DPRINT(Debug,9, (&Debug, 
                             "initialize_mailbox_from_url: leave_old_folder failed\n"));
        }       
        if (is_canceled(cd)) {
            DPRINT(Debug,9, (&Debug, 
                             "initialize_mailbox_from_url: Cancel on leave_old_folder (or free_mailbox)\n"));
        }
    }
    
    if (cd)
	free_cancel(&cd);
    if (mquota)
	free_mail_quota(&mquota);
    if (quotad)
	free_quota_display(&quotad,parent_page);

    DPRINT(Debug,9, (&Debug, 
		     "initialize_mailbox_from_url="));
    if (main_mailbox) {
	DPRINT(Debug,9, (&Debug, "%p",main_mailbox));
    } else {
	DPRINT(Debug,9, (&Debug, "NULL"));
    }
    if (ret_letter) {
	DPRINT(Debug,9, (&Debug, "; *ret_letter=%d",
			 *ret_letter));
	if (isascii(*ret_letter)) {
	    DPRINT(Debug,9, (&Debug, " (%c)",
			     *ret_letter));
	} else if (EOF ==  *ret_letter) {
	    DPRINT(Debug,9, (&Debug, " EOF"));
	}
    }    
    DPRINT(Debug,9, (&Debug, "\n"));
    
    return main_mailbox;
}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
