static char rcsid[] = "@(#)$Id: leavembox.c,v 2.18 2021/05/29 05:42:33 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.18 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *  Based on Elm 2.4 src/leavembox.c. That code was following copyright:
 *
 *  The Elm Mail System 
 *
 *			Copyright (c) 1988-1992 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************
 *  Incorparated Elm 2.5 code from src/leavembox.c. 
 *  That code was following copyright:
 *
 *  The Elm Mail System 
 *
 *                      Copyright (c) 1988-1995 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

/** leave current folder, updating etc. as needed...
  
**/

#include "def_elm.h"
#include "s_elm.h"

DEBUG_VAR(Debug,__FILE__,"mbox");

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif       

static void block_signals P_((void));
static void unblock_signals P_((void));

static unsigned char * s2us P_((char *str));
static unsigned char * s2us(str)
     char *str;
{
    return (unsigned char *)str;
}

static const unsigned char * cs2us P_((const char *str));
static const unsigned char * cs2us(str)
     const char *str;
{
    return (const unsigned char *)str;
}


static void action_message P_((int to_keep,int to_store, int to_delete));
static void action_message(to_keep,to_store,to_delete)
     int to_keep;
     int to_store;
     int to_delete;
{
    /* Formulate message as to number of keeps, stores, and deletes.
     * This is only complex so that the message is good English.
     */
	
    if (to_keep > 0) {
	if (to_store > 0) {
	    if (to_delete > 0) {
		if (to_keep == 1)
		    lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmLeaveKeepStoreDelete,
				      "[Keeping 1 message, storing %d, and deleting %d.]"), 
			      to_store, to_delete);
		else
		    lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmLeaveKeepStoreDeletePlural,
				      "[Keeping %d messages, storing %d, and deleting %d.]"), 
			      to_keep, to_store, to_delete);
	    } else {
		if (to_keep == 1)
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveKeepStore,
				      "[Keeping 1 message and storing %d.]"), 
			      to_store);
		else
		    lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmLeaveKeepStorePlural,
				      "[Keeping %d messages and storing %d.]"), 
			      to_keep, to_store);
	    }
	} else {
	    if (to_delete > 0) {
		if (to_keep == 1)
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveKeepDelete,
				      "[Keeping 1 message and deleting %d.]"), 
			      to_delete);
		else
		    lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmLeaveKeepDeletePlural,
				      "[Keeping %d messages and deleting %d.]"), 
			      to_keep, to_delete);
	    } else {
		if (to_keep == 1)
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveKeep,
				      "[Keeping message.]"));
		else
		    lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmLeaveKeepPlural,
				      "[Keeping all messages.]"));
	    }
	}
    } else if (to_store > 0) {
	if (to_delete > 0) {
	    if (to_store == 1)
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveStoreDelete,
				  "[Storing 1 message and deleting %d.]"), 
			  to_delete);
	    else
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmLeaveStoreDeletePlural,
				  "[Storing %d messages and deleting %d.]"), 
			  to_store, to_delete);
	} else {
	    if (to_store == 1)
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveStore,
				  "[Storing message.]"));
	    else
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveStorePlural,
				  "[Storing all messages.]"));
	}
    } else {
	if (to_delete > 0)
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveDelete,
			      "[Deleting all messages.]"));
    }
}

enum sync_store_class {
    sync_store_received = 0,
    sync_store_dnsmail,
    
    sync_store_COUNT
};

static struct sync_store_conf {
    struct dt_estr_info   * estr;
    enum exit_disposition   store;
    const char            * tag;
    const char            * text;
    
    
} sync_store_config[sync_store_COUNT] = {
    /* sync_store_received */
    { &recvd_mail_e,
      STORE_RECV,
      ">"   /* Received folder */,
      "received"
    },

    /* sync_store_dnsmail */
    { &dsn_mail_e,
      STORE_DSN,
      dsn_mail_tag,
      "dsn"
    }

};

static int exit_disposition_is_store P_((enum exit_disposition disposition,
				     enum sync_store_class  *class));
static int exit_disposition_is_store(disposition,class)
     enum exit_disposition  disposition;
     enum sync_store_class  *class;
{
    static enum sync_store_class vector[exit_disposition_COUNT];

    static int vector_done = 0;

    if (disposition < 0 || disposition >= exit_disposition_COUNT)
	panic("MBOX PANIC",__FILE__,__LINE__,
	      "exit_disposition_is_store",
	      "Bad disposition",0);

    if (! vector_done) {
	enum exit_disposition d;
	enum sync_store_class c; 
	
	for (d = 0; d < exit_disposition_COUNT; d++)
	    vector[d] = sync_store_COUNT;

	for (c = 0; c < sync_store_COUNT; c++) {
	    if (sync_store_config[c].store >= 0 ||
		sync_store_config[c].store <  exit_disposition_COUNT) {

		d = sync_store_config[c].store;

		if (vector[d] != sync_store_COUNT &&
		    vector[d] != c)
		    panic("MBOX PANIC",__FILE__,__LINE__,
			  "exit_disposition_is_store",
			  "Duplicate disposition",0);

		vector[d] = c;
	    }
	}

	vector_done = 1;	
    }

    *class = vector[disposition];

    if (vector[disposition] >= 0 &&
	vector[disposition] < sync_store_COUNT)
	return 1;
    
    return 0;
}

struct sync_store_item {
    int                     is_disabled;
    struct folder_browser * browser;
    struct string         * name;
    int                     flags;
    int                     can_store;
    int                     current_is_xx;
    int                     new_is_xx;
    int                     have_xx;   /* current is only xx folder */

    int                     have_store;  /* this disposition is added */

    
    WRITE_STATE             writeptr;
    
} ;

static enum real_save_stat {
    real_save_reconnect = -3,
    real_save_failure   = -2,
    real_save_newmail   = -1,
    real_save_none      = 0,
    real_save_done      = 1


} real_save P_((struct current_storage *storage,
		struct sync_store_item store_vector[sync_store_COUNT],
		int to_store,
		int to_keep,
		int resyncing, 
		int to_delete,
		int to_hide));

static enum real_save_stat real_save(storage, store_vector, 
				     to_store, to_keep, resyncing, to_delete,
				     to_hide)
     struct current_storage *storage;
     struct sync_store_item store_vector[sync_store_COUNT];
     int to_store;
     int to_keep;
     int resyncing;
     int to_delete;
     int to_hide;
{
    enum real_save_stat ret = real_save_none;
    int keep_failed = 0;
    struct cancel_data *cd = NULL;
    
    long new_bytes = 0;
    int newmails = 0;
    int i;
    enum sessionlock_status r;
    enum new_mail_stat r1;
    int err = 0;
    int rflush = 0;
    
    struct keep_folder_state * keep_state_ptr = NULL;
    
    /** next, let's lock the file up and make one last size check **/

    DPRINT(Debug,2,(&Debug,  
		    "real_save: Handling folder %S (%s)\n",
		    storage->current_folder->cur_folder_disp,
		    storage->current_folder->cur_folder_sys));


    r = open_folder_lock(LOCK_OUTGOING,storage->current_folder);
    
    switch (r) {
    case sessionlock_fail:
	DPRINT(Debug,2,(&Debug,  			
		    "real_save:  Failed to lock %S (%s)\n",
		    storage->current_folder->cur_folder_disp,
		    storage->current_folder->cur_folder_sys));

	ret = real_save_failure  /* failure */;
	goto clean;
    case sessionlock_open:
	DPRINT(Debug,10,(&Debug,
			 "real_save: Folder  %S (%s) OK\n",
			 storage->current_folder->cur_folder_disp,
			 storage->current_folder->cur_folder_sys));
	break;
    case sessionlock_reconnect:
	DPRINT(Debug,10,(&Debug,
			 "real_save: Folder  %S (%s) reconnected\n",
			 storage->current_folder->cur_folder_disp,
			 storage->current_folder->cur_folder_sys));

	unlock(0,storage->current_folder);

	ret = real_save_reconnect;
	goto clean;
    }

    /* 
     * lock routine will call clear_error so we print our action
     * message after it.
     */
    action_message(to_keep,to_store, to_delete);

    setup_mbx_cancel_message(&cd,mbx_checking_mailbox);
    
    rflush = flush_folder(storage->current_folder,&err,cd);
    if (rflush) {
	r1 = new_mail_on_folder(storage->current_folder,&new_bytes,&newmails,&err,
				cd);
    } else {
	const char * ftype = folder_type(storage->current_folder);

	if (cd)
	    free_cancel(&cd);       
	
	unlock(0,storage->current_folder);

	DPRINT(Debug,10,(&Debug,
			 "real_save: flush_folder failed\n"));

	if (err)
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmFailedIqnS,
				      "%s failed: %s"),
			      ftype,strerror(err));
	else
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmFailedIgn,
			      "%s failed..."),
		      ftype);
	ret = real_save_failure  /* failure */;
	goto clean;
    }

    switch (r1) {
    case new_mail_check_failed: {
	const char * ftype = folder_type(storage->current_folder);

	if (cd)
	    free_cancel(&cd);       
	
	unlock(0,storage->current_folder);

	if (err)
	    lib_error(CATGETS(elm_msg_cat, ElmSet,
			      ElmLeaveCheckFailedS,
			      "%s check failed: %s"),
		      ftype,strerror(err));
	else
	    lib_error(CATGETS(elm_msg_cat, ElmSet,
			      ElmLeaveCheckFailed,
			      "%s check failed..."),
		      ftype);

	ret = real_save_failure  /* failure */;
	goto clean;

    }
	break;
    case failed_mbx_ign_new_mail:
	DPRINT(Debug,12, (&Debug,
			  "real_save: folder failed, but ignoring new mail check anyway (POP?)\n"));
	break;
    case no_new_mail:
	break;
    case have_new_mail:
	DPRINT(Debug,10,(&Debug,
			 "real_save: Folder  %S (%s) have new mail.\n",
			 storage->current_folder->cur_folder_disp,
			 storage->current_folder->cur_folder_sys));

	if (cd)
	    free_cancel(&cd);       
	
	unlock(0,storage->current_folder);

	if (newmails > 1) {  /* IMAP mailbox, count is available */
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveXNewMailsArrived,
			      "%d new mails have just arrived. Resynchronizing..."),
		      newmails);
	} else {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveNewMailArrived,
			      "New mail has just arrived. Resynchronizing..."));
	}
	ret = real_save_newmail;
	goto clean;
    case new_mail_reconnect:
	DPRINT(Debug,10,(&Debug,
			 "real_save: Folder  %S (%s) reconnected.\n",
			 storage->current_folder->cur_folder_disp,
			 storage->current_folder->cur_folder_sys));

	if (cd)
	    free_cancel(&cd);       
	
	unlock(0,storage->current_folder);

	ret = real_save_reconnect;
	goto clean;
    }

    if (cd)
	free_cancel(&cd);       
    
    block_signals();
	
    DPRINT(Debug,2,(&Debug,  "real_save: Action...\n"));

    /* Store messages slated for storage in received mail folder */
    if (to_store > 0) {
	int have_failure = 0;
	enum sync_store_class c;
	
	/* Permissons check for and opening of =received done earlier */

       	DPRINT(Debug,10,(&Debug, 
			 "real_save: Copying from %s to (store) folder\n",
			 storage->current_folder->cur_folder_sys));	    

	DPRINT(Debug,2,(&Debug,  "real_save: Storing message%s ", plural(to_store)));
	
	for (i = 0; i < storage->message_count; i++) {
	    enum sync_store_class class = sync_store_COUNT;
	    
	    if (storage->headers[i] &&
		exit_disposition_is_store(storage->headers[i]->exit_disposition,
					  &class)) {

		DPRINT(Debug,2,(&Debug,  "#%d ", i+1));

		if (class >= 0 && class <= sync_store_COUNT) {
		    DPRINT(Debug,2,(&Debug,"> %s",
				    sync_store_config[class].text));
		    
		    if (! store_vector[class].is_disabled &&
			store_vector[class].can_store) {
			
			DPRINT(Debug,2,(&Debug,", "));
		
			if (!copy_message_d(storage->current_folder,
					    storage->headers[i],
					    store_vector[class].browser,
					    store_vector[class].writeptr,
					    CM_UPDATE_STATUS,NULL,
					    NULL)) {

			    DPRINT(Debug,2,(&Debug,"\n"));
			    
			    lib_error(CATGETS(elm_msg_cat, ElmSet, 
					      ElmCopyMessageFailed,
					      "OOPS! Storing of mail failed!"));
		    
			    unlock(0,storage->current_folder);
		    
			    unblock_signals();

			    ret = real_save_failure;
			    goto clean;
			}

		    } else {
			DPRINT(Debug,2,(&Debug," KEEP,"));
			storage->headers[i]->exit_disposition = KEEP;
		    }

		} else {
		    DPRINT(Debug,2,(&Debug," KEEP,"));
		    storage->headers[i]->exit_disposition = KEEP;
		}
	    }
	}

	DPRINT(Debug,2,(&Debug,"\n"));
	
	/* Be sure that writing to =received succeed before we
	   rewrite folder
	*/

	for (c = 0; c < sync_store_COUNT; c++) {

	    
	    if (store_vector[c].have_store &&
		store_vector[c].can_store) {
	
		if (!sync_write_folder(store_vector[c].browser,
				       store_vector[c].writeptr)) {

		    have_failure = 1;
		}
	    }
	}

	if (have_failure) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmCopyMessageFailed,
			      "OOPS! Storing of mail failed!"));
	    
	    unlock(0,storage->current_folder);
	    
	    unblock_signals();

	    ret = real_save_failure;
	    goto clean;
	}

	
	if (ferror_folder(storage->current_folder,TRUE)) {
	    unlock(0,storage->current_folder);
	    
	    DPRINT(Debug,1,(&Debug,  "real_save: error when reading mailfile\n"));
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmErrorReadMailfile,
			      "Error when reading mailfile %S"),
		      storage->current_folder->cur_folder_disp);
	    sleep_message();
	    
	    unblock_signals();

	    ret = real_save_failure;
	    goto clean;
	}
    }

    if (!prepare_keep_folder(storage->current_folder,
			     &keep_state_ptr)) {
	unlock(0,storage->current_folder);
	
	unblock_signals();

	ret = real_save_failure;
	goto clean;
    }

    for (i = 0; i < storage->message_count; i++) {
	if (storage->headers[i]) {
	    
	    int r = mark_keep_folder(storage->current_folder,keep_state_ptr,
				     storage->headers[i],
				     storage->headers[i]->exit_disposition 
				     == KEEP ||
				     storage->headers[i]->exit_disposition 
				     == HIDE);

	    if (!r) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmCopyMessageFailed,
				  "OOPS! Storing of mail failed!"));

		keep_failed = 1;

	    }

	}
    }

    if (keep_failed) {
	end_keep_folder_failure(storage->current_folder,&keep_state_ptr);

	unlock(0,storage->current_folder);
	
	unblock_signals();
	
	ret = real_save_failure;
	goto clean;
    } else {    
	if (!end_keep_folder(storage->current_folder,&keep_state_ptr)) {

	     DPRINT(Debug,12, (&Debug,
                              "real_save: end_keep_folder failed!\n"));

	    
	    if (POLL_method)
		wait_for_timeout(5);
	    else
		sleep(5);

	    unlock(0,storage->current_folder);

            unblock_signals();
            
            ret = real_save_failure;
            goto clean;
	}
    }

    /* Just unlock ... folder closing is not needed on here ... */

    unlock(0,storage->current_folder);

    unblock_signals();

    ret = real_save_done;
    
 clean:
    if (cd)
	free_cancel(&cd);       

    DPRINT(Debug,10,(&Debug,
		     "real_save=%d",ret));
    switch (ret) {
    case real_save_reconnect: DPRINT(Debug,10,(&Debug, " real_save_reconnect")); break;
    case real_save_failure:   DPRINT(Debug,10,(&Debug, " real_save_failure")); break;
    case real_save_newmail:   DPRINT(Debug,10,(&Debug, " real_save_newmail")); break;
    case real_save_none:      DPRINT(Debug,10,(&Debug, " real_save_none")); break;
    case real_save_done:      DPRINT(Debug,10,(&Debug, " real_save_done")); break;
    }
    DPRINT(Debug,10,(&Debug,"\n"));
    
    return ret;
}


void close_cleanup_mbox(mailbox)
     struct MailboxView *mailbox;
{
    int i, sc;
    
    int to_delete = 0, to_store = 0, to_keep = 0, to_unset = 0,
	to_hide = 0, to_bad = 0;

    struct cancel_data *cd = NULL;

    DPRINT(Debug,1,
	   (&Debug, "close_cleanup_mbox: -- closing folder --\n"));

    /* Examine dispositions done by sync_mailbox() */

    sc = get_storage_count(mailbox);
    for (i = 0; i < sc; i++) {
	struct current_storage *storage = get_storage(mailbox,i);
	int j;

	if (!storage)
	    continue;

	if (storage->headers) {
	    for(j = 0; j < storage->message_count; j++) {

		if (storage->headers[j]) {
		    enum sync_store_class class = 0;
		    
		    if (HIDE == storage->headers[j]->exit_disposition)
			to_hide++;
		    
		    else if (KEEP == storage->headers[j]->exit_disposition)
			to_keep++;

		    else if (DELETE == storage->headers[j]->exit_disposition)
			to_delete++;

		    else if (exit_disposition_is_store(storage->headers[j]->exit_disposition,
						       &class))
			to_store++;

		    else if (UNSET == storage->headers[j]->exit_disposition)
			to_unset++;

		    else {

			DPRINT(Debug,1,(&Debug, 
					"close_cleanup_mbox: Bad message disposition %d, storage=%d, index=%d\n",
					storage->headers[j]->exit_disposition,i,j));
					
			to_bad++;

		    }
		}
	    }
	}
    }

    DPRINT(Debug,3,(&Debug,  "close_cleanup_mbox: On close_cleanup_mbox: \n"));

    DPRINT(Debug,3,(&Debug,  "                    Messages to keep:   %d\n",to_keep));
    DPRINT(Debug,3,(&Debug,  "                    Messages to hide:   %d (keep)\n", to_hide));
    DPRINT(Debug,3,(&Debug,  "                    Messages to delete: %d\n", to_delete));
    DPRINT(Debug,3,(&Debug,  "                    Messages to store:  %d\n", to_store));
     
    DPRINT(Debug,1,(&Debug,  "                    Messages    unset:  %d\n",to_unset));
    DPRINT(Debug,1,(&Debug,  "                    Messages      bad:  %d\n",to_bad));

    if (to_unset > 0 || to_bad > 0)
	panic("MBOX PANIC",__FILE__,__LINE__,"close_cleanup_mbox",
	      "Something wrong in message dispositions!",0);      

    for (i = 0; i < sc; i++) {
	struct current_storage *storage = get_storage(mailbox,i);
	int r;
	
	if (!storage)
	    continue;
	
	if (!storage->current_folder)
	    continue;

	if (to_keep == 0 && 
	    to_hide == 0) {

	    /* consider_remove_folder closes folder if removed */
	    if (consider_remove_folder(storage->current_folder)) {
		
		/* i.e. if no messages were to be kept and this is not a spool
		 * folder and we aren't keeping empty non-spool folders,
		 * simply remove the old original folder and that's it!
		 */
		
		DPRINT(Debug,3,(&Debug,  "Removing folder!\n"));
		
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFolderSRemoved,
				  "Folder %S removed."),
			  storage->current_folder->cur_folder_disp
			  
			  );
		
		if (POLL_method)
		    wait_for_timeout(5);
		else
		    sleep(5);
		
		continue;
	    } 	    
	}

	setup_mbx_cancel_message(&cd,mbx_leaving_mailbox);
	
	DPRINT(Debug,9, (&Debug, 
			 "close_cleanup_mbox: closing folder\n"));
	
	r = close_folder(storage->current_folder,CLOSE_NORMAL,cd);

	if (!r) {
	    DPRINT(Debug,9, (&Debug, 
			     "close_cleanup_mbox: close_folder failed\n"));
	}
	if (is_canceled(cd)) {
	    DPRINT(Debug,9, (&Debug, 
			     "close_cleanup_mbox: Cancel on close_folder\n"));
	}
	
	free_cancel(&cd);
    }

    if (cd)
	free_cancel(&cd);
}

/* Both resyncing, quitting are set if leaving visited mailbox */





static enum sync_store_class header_to_class P_((struct header_rec *hdr));
static enum sync_store_class header_to_class(hdr)
     struct header_rec *hdr;
{
    enum sync_store_class c = sync_store_received;

    if (mime_type_is_dsn(& (hdr->mime_rec)))
	c = sync_store_dnsmail;

    return c;
}

static struct string * read_message_text P_((enum sync_store_class c,
					int plural));
static struct string * read_message_text(c,plural)
     enum sync_store_class c;     
     int plural;
{
    struct string * text = NULL;

    switch (c) {
    case sync_store_received:
	/* Not Used */
	if (plural)
	    text =  format_string(CATGETS(elm_msg_cat, ElmSet, 
					  ElmLeaveReadMsgS,
					  "read messages"));
	else
	    text =  format_string(CATGETS(elm_msg_cat, ElmSet, 
					  ElmLeaveReadMsg,
					  "read message"));
	break;
	
    case sync_store_dnsmail:
	if (plural)
	    text =  format_string(CATGETS(elm_msg_cat, ElmSet, 
					  ElmLeaveReadNtfS,
					  "read notifications"));
	else
	    text =  format_string(CATGETS(elm_msg_cat, ElmSet, 
					  ElmLeaveReadNtf,
					  "read notification"));
	break;

	
    case sync_store_COUNT:
	break;
    }
    
    return text;
}


static struct string * unread_message_text P_((enum sync_store_class c,
					int plural));
static struct string * unread_message_text(c,plural)
     enum sync_store_class c;     
     int plural;
{
    struct string * text = NULL;

    switch (c) {
    case sync_store_received:
	if (plural)
	    text =  format_string(CATGETS(elm_msg_cat, ElmSet, 
					  ElmLeaveUnReadMsgS,
					  "unread messages"));
	else
	    text =  format_string(CATGETS(elm_msg_cat, ElmSet, 
					  ElmLeaveUnReadMsg,
					  "unread message"));
	break;
	
    case sync_store_dnsmail:
	if (plural)
	    text =  format_string(CATGETS(elm_msg_cat, ElmSet, 
					  ElmLeaveUnReadNtfS,
					  "unread notifications"));
	else
	    text =  format_string(CATGETS(elm_msg_cat, ElmSet, 
					  ElmLeaveUnReadNtf,
					  "unread notification"));
	break;

	
    case sync_store_COUNT:
	break;
    }
    
    return text;
}



enum sync_mbox_stat sync_mbox(resyncing, quitting, prompt, new_folder, mailbox,
	      page,prompt_area)
     int resyncing, quitting, prompt;
     struct folder_info **new_folder;
     struct MailboxView *mailbox;
     struct menu_context  *page;
     struct menu_context *prompt_area;
{
    /** Close folder, deleting some messages, storing others in mbox,
	and keeping others, as directed by user input and elmrc options.
	
	Return	1	Folder altered
	        0	Folder not altered
		-1	New mail arrived during the process and
		        syncing was aborted.
	        -2      Leaving of folder failed
		-3      Mailbox is reconnected
		-4      Leaving of folder canceled (Ctrl-C)
		-5	EOF seen on user input
			
        If "resyncing" we are just writing out folder to reopen it. We
	   therefore only consider deletes and keeps, not stores to mbox.
	   Also we don't remove NEW status so that it can be preserved
	   across the resync.

	If "quitting" and "prompt" is false, then no prompting is done.
	    Otherwise prompting is dependent upon the variables
	    ask_delete, ask_keep, and ask_store, as set by elmrc
	    elmrc options.  This behavior makes the 'q' command prompt
	    just like 'c' and '$', while retaining the 'Q' command for a
	    quick exit that never prompts.
    **/

    int to_delete = 0, to_store = 0, to_keep = 0, to_ignore = 0,
	to_hide = 0;
    int marked_deleted=0;
    int 	i,
	l_ask_delete, l_ask_keep, l_ask_store;

    enum sync_mbox_stat return_value = sync_mbox_failure;
    
    struct sync_store_item sync_store_list[sync_store_COUNT];

    enum sync_store_class c;
    
            
    struct folder_browser * incoming_browser = new_browser(selection_folder);
    struct string         * incoming_name    = new_string(system_charset);

    int num_chgd_status     = 0;

    int sc, need_handle;

    int delay_redraw = 0;

    int have_incoming       = 0;  /* Incoming available or selected */
    int mailbox_count       = 0;  /* Mailboxes, without other folders */

    int failure_count   = 0;
    int reconnect_count = 0;
    int newmail_count   = 0;
    int saved_count     = 0;
    int open_close_failure = 0;
    
    struct string * current_mailbox_name = NULL;      /* Current mailbox name (if only one) */

    if (resyncing) {
	DPRINT(Debug,1,
	       (&Debug, "\n\n-- syncing folder --\n\n"));
    } else {
	DPRINT(Debug,1,
	       (&Debug, "\n\n-- leaving folder --\n\n"));
    }

    add_ascii_to_string(incoming_name,s2us("!"));        /* Incoming folder */
    have_incoming = select_dir_item(incoming_browser,&incoming_name,NULL);


    if (!have_incoming) {
	DPRINT(Debug,4,(&Debug,
			"sync_mbox: Incoming folder is not available!\n"));
    }
    

    for (c = 0; c < sync_store_COUNT; c++) {
	sync_store_list[c].is_disabled = dt_estr_is_disabled(sync_store_config[c].estr);
	sync_store_list[c].browser     = new_browser(selection_folder);
	sync_store_list[c].name        = new_string(system_charset);
	sync_store_list[c].flags       = 0;
	sync_store_list[c].can_store   = 0;
	sync_store_list[c].writeptr    = NULL;
	sync_store_list[c].current_is_xx = 0;
	sync_store_list[c].new_is_xx     = 0;
	sync_store_list[c].have_xx      = 0;
	sync_store_list[c].have_store   = 0;

	add_ascii_to_string(sync_store_list[c].name,cs2us(sync_store_config[c].tag));        
	sync_store_list[c].can_store =
	    select_dir_item(sync_store_list[c].browser,&sync_store_list[c].name,NULL);
	sync_store_list[c].have_xx =
	    sync_store_list[c].can_store;
	
	if (sync_store_list[c].can_store) {

	    struct string *buffer2 = selection_name_dir(sync_store_list[c].browser);
	    
	    int sc = get_storage_count(mailbox);
	    int i;

	    /* Replace editing buffer with expanded version ... */
	    
	    if (buffer2) {
		DPRINT(Debug,12,(&Debug,
				 "sync_mbox: Folder %S (%s) expands to %S\n",
				 sync_store_list[c].name,
				 sync_store_config[c].text,
				 buffer2));
		
		free_string(& (sync_store_list[c].name));

		sync_store_list[c].name = buffer2;
		buffer2 = NULL;
	    }
	    
	    for (i = 0; i < sc; i++) {
		struct current_storage *storage = get_storage(mailbox,i);
		
		if (storage && storage->current_folder) {
		    if (selection_is_folder(sync_store_list[c].browser,storage->current_folder)) {

			DPRINT(Debug,12,(&Debug,
					 "sync_mbox: Folder %S is %S (%s) folder\n",
					 storage->current_folder->cur_folder_disp,
					 sync_store_list[c].name,
					 sync_store_config[c].text
					 ));

			
			sync_store_list[c].current_is_xx++;
		    } else if (sync_store_list[c].have_xx) {

			DPRINT(Debug,12,(&Debug,
					 "sync_mbox: Folder %S is not %S (%s) folder\n",
					 storage->current_folder->cur_folder_disp,
					 sync_store_list[c].name,
					 sync_store_config[c].text
					 ));
			
			sync_store_list[c].have_xx = 0;
		    }
		}
	    }

	    if (new_folder && *new_folder)
		sync_store_list[c].new_is_xx =
		    selection_is_folder(sync_store_list[c].browser,*new_folder);

	    sync_store_list[c].flags = give_dir_flags(sync_store_list[c].browser);

	    
	    /* Create it if not exists yet ... */
	    if (0 == (sync_store_list[c].flags & BROWSER_EXIST)) {
		if (!create_selection_dir(sync_store_list[c].browser))
		    sync_store_list[c].can_store = 0;
		else {
		    DPRINT(Debug,1,(&Debug,  
				    "sync_mbox: Creating %s folder: %S",
				    sync_store_config[c].text,
				    sync_store_list[c].name));
		}
	    }
		
	    if (sc > 0) {
				
		DPRINT(Debug,4,(&Debug,
				"sync_mbox: %S (%s) folder %s only current folder (storage count %d)\n",
				sync_store_list[c].name,
				sync_store_config[c].text,
				sync_store_list[c].have_xx ? "is" : "is not",
				sc));
	    } else
		sync_store_list[c].have_xx = 0;
	    
	}

	if (sync_store_list[c].current_is_xx) {
	    DPRINT(Debug,4,(&Debug,
			    "sync_mbox: Current folder is %S (%s) folder!\n",
			    sync_store_list[c].name,
			    sync_store_config[c].text
			    ));
	}

	if (sync_store_list[c].new_is_xx) {
	    DPRINT(Debug,4,(&Debug,
			    "sync_mbox: New folder is %S (%s) folder!\n",
			    sync_store_list[c].name,
			    sync_store_config[c].text
			    ));
	}		

	if (! sync_store_list[c].can_store) {
	    DPRINT(Debug,4,(&Debug,
			    "sync_mbox: %S (%s) folder is not available!\n",
			    sync_store_list[c].name,
			    sync_store_config[c].text
			    ));
	}

	
	if (sync_store_list[c].is_disabled) {
	    DPRINT(Debug,4,(&Debug,  "sync_mbox: %S (%s) folder is disabled!\n",
			    sync_store_list[c].name,
			    sync_store_config[c].text
			    ));
	}		
    }
    
    /* Be sure that current number of messages match to that what
       is on storages .... */

    if (update_view(mailbox)) {
	return_value = sync_mbox_failure;  /* failure */
	goto cleanup;
    }

    /* ================================================================== */

    /* Check if ALL folders are read_only and 
       check if current folder is incoming
     */
    need_handle = 0;    
    sc = get_storage_count(mailbox);
    for (i = 0; i < sc; i++) {
	struct current_storage *storage = get_storage(mailbox,i);
	int j;

	int mode = 0;

	if (!storage)
	    continue;

	/* Clear the exit dispositions of all messages (except HIDE), 
	 * just in case they were left set by a previous call to this function
	 * that was interrupted by the receipt of new mail.
	 */
	if (storage->headers) {
	    for(j = 0; j < storage->message_count; j++) {
		if (storage->headers[j]) {
		    if (HIDE == storage->headers[j]->exit_disposition)
			to_hide++;
		    else
			storage->headers[j]->exit_disposition = UNSET;
		}
	    }
	}
	    
	if (! storage->current_folder) {
	    if (storage->headers) {
		for (j = 0; j < storage->message_count; j++) {
		    if (storage->headers[j]) {
			storage->headers[j]->exit_disposition = KEEP;
			to_ignore ++;
		    }
		}
	    }
	    continue;
	}

	mode = get_folder_mode(storage->current_folder);

	if (0 != (mode & FOLDER_MBOX)) {
	    mailbox_count++;
	    
	    /* Check if current mailbox is not incoming */
	    if (have_incoming && 
		!selection_is_folder(incoming_browser,storage->current_folder)) {

		DPRINT(Debug,12,(&Debug,
				 "sync_mbox: Mailbox %S is not incoming folder\n",
				 storage->current_folder->cur_folder_disp));

		have_incoming = 0;
	    }

	    if (!current_mailbox_name) {
		current_mailbox_name =  dup_string(storage->current_folder->cur_folder_disp);
	    }

	}

	if (0 != ( mode & FOLDER_RDONLY)) {
	    if (storage->headers) {
		for (j = 0; j < storage->message_count; j++) {
		    if (storage->headers[j]) {
			storage->headers[j]->exit_disposition = KEEP;
			to_keep ++;
		    }
		}
	    }
	    continue;
	}
	need_handle++;
    }


    if (0 == mailbox_count) {
	have_incoming = 0;
	DPRINT(Debug,3,(&Debug,  "sync_mbox: No mailboxes (storage count %d)\n",sc));
    } else {
	DPRINT(Debug,3,(&Debug,  "sync_mbox: %d mailboxes (storage count %d), incoming folder %s only current mailbox\n",
			mailbox_count,sc,
			have_incoming ? "is" : "is not"));

    }

    if (current_mailbox_name) {

	if (mailbox_count > 1) 
	    free_string(&current_mailbox_name);
	else {
	    DPRINT(Debug,3,(&Debug,  "sync_mbox: Current mailbox is %S\n",current_mailbox_name));
	}

    }

    DPRINT(Debug,3,(&Debug,  "sync_mbox: Messages to ignore: %d (no folder)\n", 
		    to_ignore));
    DPRINT(Debug,3,(&Debug,  "sync_mbox: Messages to keep:   %d (readonly)\n", 
		    to_keep));
    DPRINT(Debug,3,(&Debug,  "sync_mbox: Messages to hide:   %d (keep)\n", 
		    to_hide));


    if (0 == need_handle) {
	DPRINT(Debug,4,(&Debug,  "*** ALL folders are read only!\n"));
	for (i = 0; i < sc; i++) {
	    struct current_storage *storage = get_storage(mailbox,i);

	    if (!storage ||
		! storage->current_folder)
		continue;

	    if (0 != (get_folder_mode(storage->current_folder) & 
		      FOLDER_RDONLY)) {

		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmFolderSReadOnlyUnchanged,
				  "Folder %S is read-only and unchanged."),
			  storage->current_folder->cur_folder_disp
			  );

	    }
	}

	return_value = sync_mbox_nochange;	/* nothing changed */
	goto cleanup;
    }

    /* ================================================================== */

    /* Check if all folders are empty */

    need_handle = 0;    
    sc = get_storage_count(mailbox);
    for (i = 0; i < sc; i++) {
	struct current_storage *storage = get_storage(mailbox,i);

	if (!storage ||
	    storage->message_count == 0)
	    continue;
	
	need_handle++;
    }

    if (0 == need_handle) {
	DPRINT(Debug,4,(&Debug,  "*** ALL folders are empty!\n"));
	
	return_value = sync_mbox_nochange;	/* nothing changed */	
	
	goto cleanup;
    }
    
    /* =================================================================== */

    l_ask_delete = ((quitting && !prompt) ? FALSE : ask_delete);
    l_ask_keep = ((quitting && !prompt) ? FALSE : ask_keep);
    l_ask_store = ((quitting && !prompt) ? FALSE : ask_store);

    /* YES or NO on softkeys */
    /*
      if (hp_softkeys && ask_questions) {
      define_softkeys(YESNO);
      softkeys_on();
      }
    */

	  
    /* Determine if deleted messages are really to be deleted */

    /* we need to know if there are none, or one, or more to delete */
    need_handle = 0;    
    marked_deleted = 0;

    sc = get_storage_count(mailbox);
    for (i = 0; i < sc; i++) {
	struct current_storage *storage = get_storage(mailbox,i);
	int j;

	if (!storage)
	    continue;

	if (storage->headers)	
	    for (j=0; 
		 j<storage->message_count && marked_deleted<2; j++) {
		
		if (! storage->headers[j])
		    continue;

		/* If message is already marked to keep, we do not
		   have intrested ....
		*/
		if (storage->headers[j]->exit_disposition != UNSET)
		    continue;
		
		if (ison(storage->headers[j]->status, DELETED))
		    marked_deleted++;
	    }
    }

    /* ============================================================== */

    DPRINT(Debug,4,(&Debug,
		    "sync_mbox: Messages marked deleted: %d (at least)\n", marked_deleted));

    if(marked_deleted) {
	int answer = (always_del ? 
		       *def_ans_yes : *def_ans_no);	/* default answer */

	if (l_ask_delete && prompt_area) {
	    int def = answer;	    

	againD:

	    /* NOTICE: prompt_letter may return EOF */
	    if (marked_deleted == 1)
		answer = prompt_letter(0,"",def,
				       PROMPT_yesno|PROMPT_cancel|
				       PROMPT_redraw_mark|PROMPT_ctrlL,
				       prompt_area,
				       CATGETS(elm_msg_cat, ElmSet, 
						    ElmLeaveDeleteMessage,
						    "Delete message? (%c/%c) "), 
				       *def_ans_yes, *def_ans_no);
	    else
		answer = prompt_letter(0,"",def,
				       PROMPT_yesno|PROMPT_cancel|
				       PROMPT_redraw_mark|PROMPT_ctrlL,	
				       prompt_area,			       
				       CATGETS(elm_msg_cat, ElmSet, 
					       ElmLeaveDeleteMessages,
					       "Delete messages? (%c/%c) "), 
				       *def_ans_yes, *def_ans_no);

	    if (TERMCH_interrupt_char == answer) {
		return_value = sync_mbox_cancel;  /* failure */
		goto cleanup;	    
	    }

	    if (answer == ('L'&31) || answer == REDRAW_MARK) {
		menu_ClearScreen(page);      /* Reset possible redraw flag */

		/* Call refresh routines of children */
		menu_redraw_children(page);

		if (menu_need_redraw(prompt_area))
		    menu_ClearScreen(prompt_area);   /* but clear redraw mark from prompt_area*/

		delay_redraw++;   /* Can't trigger redraw yet... */
		goto againD;
	    }

	    if (answer == EOF) {
		return_value = sync_mbox_EOF;
		goto cleanup;	    
	    }
	}

	DPRINT(Debug,4,(&Debug,  "sync_mbox:    delete? %c\n", answer));

	if (answer == *def_ans_yes) {

	    sc = get_storage_count(mailbox);
	    for (i = 0; i < sc; i++) {
		struct current_storage *storage = get_storage(mailbox,i);
		int j;

		if (!storage)
		    continue;

		if (storage->headers)
		    for (j = 0; j < storage->message_count; j++) {
			
			
			if (! storage->headers[j])
			    continue;
			
			
			/* If message is already marked to keep, we do not
			   have intrested ....
			*/
			if (storage->headers[j]->exit_disposition != UNSET)
			    continue;
			
			if (ison(storage->headers[j]->status, DELETED)) {
			    storage->headers[j]->exit_disposition = DELETE;
			    to_delete++;
			}
		    }
	    }

	    DPRINT(Debug,9,(&Debug,
			    "sync_mbox:  %d mailboxes examined, found %d messages to delete\n",
			    i,to_delete));
	}
    }
    DPRINT(Debug,3,(&Debug,  "sync_mbox: Messages to delete: %d\n", to_delete));

    /* ================================================================== */

    /* If this is a non spool file (or if received folder is treated
     * as mailbox because of protection mask of folder) , or if we are 
     * merely resyncing, all messages with an unset disposition (i.e. 
     * not slated for deletion) are to be kept.
     * Otherwise, we need to determine if read and unread messages
     * are to be stored or kept.
     */

    need_handle = 0;    
    sc = get_storage_count(mailbox);
    for (i = 0; i < sc; i++) {
	
	struct current_storage *storage = get_storage(mailbox,i);
	int this_is_some = 0;
	int keep_all = 0;

	int this_is_xx[sync_store_COUNT];

	for (c = 0; c < sync_store_COUNT; c++) {
	    this_is_xx[c] = -1;  /* Not determined */
	}
	
	if (!storage ||	    
	    ! storage->current_folder || !storage->headers) 
	    continue;

	if (0 == (get_folder_mode(storage->current_folder) & FOLDER_MBOX) ||
	    resyncing) {

	    DPRINT(Debug,9,(&Debug,
			    "sync_mbox: %S not mailbox, or resyncing -- keeping all messages\n",
			    storage->current_folder->cur_folder_disp));
	    
	    keep_all = 1;
	    
	} else {
	
	    for (c = 0; c < sync_store_COUNT; c++) {
		if (sync_store_list[c].can_store &&
		    selection_is_folder(sync_store_list[c].browser,storage->current_folder)) {

		    DPRINT(Debug,9,(&Debug,
				     "sync_mbox: Folder %S is %S (%s) folder -- keeping some messages\n",
				    storage->current_folder->cur_folder_disp,
				    sync_store_list[c].name,
				    sync_store_config[c].text
				    ));
				     
		    this_is_some++;
		    this_is_xx[c] = 1;
		} else {
		    this_is_xx[c] = 0;
		}
	    }
	}
	
	if (keep_all ||  this_is_some) {
	    int j;
	    int was_all = 1;

	    for (j = 0; j < storage->message_count; j++) {
		if( storage->headers[j] &&
		    storage->headers[j]->exit_disposition == UNSET) {

		    if (keep_all) {		    
			storage->headers[j]->exit_disposition = KEEP;
			to_keep++;
		    } else {
			c = header_to_class(storage->headers[j]);

			if (c >= 0 && c < sync_store_COUNT &&
			    this_is_xx[c]) {

			    /* Already on this folder */
			    
			    storage->headers[j]->exit_disposition = KEEP;
			    to_keep++;
			} else if (sync_store_list[c].current_is_xx) {

			    /* Do not allow storing to open mailbox or folder
			       -- this does not work when same folder / malbox
			       is on middle of closing / resyncing here !!!
			    */
			    
			    storage->headers[j]->exit_disposition = KEEP;
			    to_keep++;
			    
			} else {
			    was_all = 0;
			}			
		    }					    
		}
	    }

	    if (was_all) {
		DPRINT(Debug,12,(&Debug,
				 "sync_mbox: Folder %S -- all messages kept (message count %d).\n",
				 storage->current_folder->cur_folder_disp,
				 storage->message_count));
		
		continue;
	    }
	}
	
	need_handle++;
    }

    DPRINT(Debug,3,(&Debug,  
		    "sync_mbox: Messages to keep: %d (not mbox or syncing) -- %d mboxes need handling\n", 
		    to_keep,need_handle));    

    if (need_handle) {

	int keep_this[sync_store_COUNT];
	int keep_some   = 0;
	int handle_some = 0;
	
	for (c = 0; c < sync_store_COUNT; c++) {
	    keep_this[c] = 0;
	    
	    if (sync_store_list[c].is_disabled) {
		DPRINT(Debug,3,(&Debug,
				"sync_mbox: Disabling %s folder processing...\n",
				sync_store_config[c].text
				));
		
		if (c > sync_store_received &&
		    ! keep_this[sync_store_received]) {
		    
		    keep_this[c] = -1;  /* Use received folder instead */
		} else {
		    keep_this[c] = 1;
		}
		
		goto disabled_1;
	    }

	    if (! sync_store_list[c].can_store) {
		int answer = '\0';
	       	
		if (prompt_area) {
		    
		    switch (c) {
			
		    case sync_store_received:
				
		    again1:
			
			/* NOTICE: prompt_letter may return EOF */
			answer = prompt_letter(0,"P",*def_ans_yes,
					       PROMPT_center|PROMPT_yesno|PROMPT_cancel|
					       PROMPT_redraw_mark|PROMPT_ctrlL,
					       prompt_area,
					       CATGETS(elm_msg_cat, ElmSet, 
						       ElmNoReceived,
						       "\"received\" folder not available, continue leaving folder? ([P]anic/%c/%c) "),
					       *def_ans_yes, *def_ans_no);
			
			
			if (answer == ('L'&31) || answer == REDRAW_MARK) {
			    menu_ClearScreen(page);  /* Reset possible redraw flag */
			    
			    /* Call refresh routines of children */
			    menu_redraw_children(page);
			    
			    if (menu_need_redraw(prompt_area))
				menu_ClearScreen(prompt_area);   /* but clear redraw mark from prompt_area*/
			    
			    delay_redraw++;   /* Can't trigger redraw yet... */
			    goto again1;
			}
			
			if (answer == EOF) {
			    return_value = sync_mbox_EOF;
			    goto cleanup;	    
			}
			
			if (answer == 'P')
			    emergency_exit(0);
			
			if (TERMCH_interrupt_char == answer) {
			    return_value = sync_mbox_cancel;  /* failure */
			    goto cleanup;	    
			}

			DPRINT(Debug,4,(&Debug,
					"sync_mbox:    %s not available, continue? %c\n",
					sync_store_config[c].text,
					answer));
			
			if (answer != *def_ans_yes) {
			    return_value = sync_mbox_failure;  /* failure */
			    goto cleanup;
			}

			keep_this[c] = 1;
		    
			break;

		    case sync_store_dnsmail:
			
			if (keep_this[sync_store_received])
			    break;

		    again1a:
		    
			answer = prompt_letter(0,"P",*def_ans_yes,
					       PROMPT_center|PROMPT_yesno|PROMPT_cancel|
					       PROMPT_redraw_mark|PROMPT_ctrlL,
					       prompt_area,
					       CATGETS(elm_msg_cat, ElmSet, 
						       ElmNoThisUseReceived,
						       "\"%s\" folder not available, use \"received\" folder instead? (%c/%c) "),
					       sync_store_config[c].text,
					       *def_ans_yes, *def_ans_no);
			
			if (answer == ('L'&31) || answer == REDRAW_MARK) {
			    menu_ClearScreen(page);  /* Reset possible redraw flag */
			    
			    /* Call refresh routines of children */
			    menu_redraw_children(page);
			    
			    if (menu_need_redraw(prompt_area))
				menu_ClearScreen(prompt_area);   /* but clear redraw mark from prompt_area*/
			    
			    delay_redraw++;   /* Can't trigger redraw yet... */
			    goto again1a;
			}
			
			if (answer == EOF) {
			    return_value = sync_mbox_EOF;
			    goto cleanup;	    
			}
			
			if (TERMCH_interrupt_char == answer) {
			    return_value = sync_mbox_cancel;  /* failure */
			    goto cleanup;	    
			}

			DPRINT(Debug,4,(&Debug,
					"sync_mbox:    %s not available, use received? %c\n",
					sync_store_config[c].text,
					answer));
			
			if (answer == *def_ans_yes)
			    keep_this[c] = -1;  /* Use received folder instead */
			else if (answer == *def_ans_no)
			    keep_this[c] = 1;
			else {  /* Should not hapen */
			    return_value = sync_mbox_failure;  /* failure */
			    goto cleanup;
			}


			
		    break;
		    
		    case sync_store_COUNT:
			break;
		    }
		    		    	      
		} else {
		    keep_this[c] = 1;
		}
	    

	    disabled_1:

		;
		
	    } else
		handle_some = 1;
	    
	}
	    
	sc = get_storage_count(mailbox);
	for (i = 0; i < sc; i++) {
	    struct current_storage *storage = get_storage(mailbox,i);
	    int j;
	    
	    if (!storage)
		continue;
	    
	    if (storage->headers) {
		for (j = 0; j < storage->message_count; j++) {
		    if(storage->headers[j]  &&
		       storage->headers[j]->exit_disposition == UNSET) {
			
			c = header_to_class(storage->headers[j]);
			
			if (c >= 0 && c < sync_store_COUNT &&
			    keep_this[c] < 0) {
			    
			    c = sync_store_received;
			}
			
			if (c >= 0 && c < sync_store_COUNT &&
			    keep_this[c] > 0) {
			    
			    storage->headers[j]->exit_disposition = KEEP;
			    to_keep++;
			    keep_some = 1;
			}
			
		    }
		}
	    }
	}
	
	
	if (keep_some) {
	    DPRINT(Debug,3,(&Debug,  
			    "sync_mbox: Messages to keep: %d   (no folder)\n", 
			    to_keep));
	}
	
	if (handle_some) {
	    int answer = '\0';
	    
	    int marked_read_x[sync_store_COUNT];
	    int need_move_x[sync_store_COUNT];
	    int need_keep_x[sync_store_COUNT];  
		
	    int ask_unread = 0;

	    int messages1[sync_store_COUNT];
	    int messages2[sync_store_COUNT];
	    
	    for (c = 0; c < sync_store_COUNT; c++) {
		marked_read_x[c] = 0;
		need_move_x[c]   = 0;
				
		/* Let's first see if user wants to store read messages 
		 * that aren't slated for deletion */
		
		
		need_keep_x[c]   = 0;
		messages1[c]     = 0;
		messages2[c]     = 0;

	    }
	    
	    
	    /* we need to know if there are none, or one, or more marked read 
	     */
	    
	    sc = get_storage_count(mailbox);
	    for (i = 0; i < sc; i++) {
		struct current_storage *storage = get_storage(mailbox,i);
		int j;
		
		if (!storage)
		    continue;
		
		if (storage->headers) {
		    
		    for (j=0; j < storage->message_count; j++) {
			
			if (storage->headers[j]) {
			    c = header_to_class(storage->headers[j]);
			    
			    if (c >= 0 && c < sync_store_COUNT) {
				messages1[c]++;
				
				if (keep_this[c] < 0) {
				    c = sync_store_received;
				}
			    }

			    if (c >= 0 && c < sync_store_COUNT) {
				messages2[c]++;
				
				if (c >= 0 && c < sync_store_COUNT &&
				    ! keep_this[c]) {
				    
				    if (isoff(storage->headers[j]->status, UNREAD)
					&& (storage->headers[j]->exit_disposition == UNSET)) {
				    
					marked_read_x[c]++;
				    }
				}
			    }
			}
		    }		
		}
	    }
	    
	    for (c = 0; c < sync_store_COUNT; c++) {

		DPRINT(Debug,3,(&Debug,  
				"sync_mbox: class %s: %d messages (%d messages after class translation), %d marked read\n",
				sync_store_config[c].text,
				messages1[c],
				messages2[c],
				marked_read_x[c]));
		
		if (marked_read_x[c] &&
		    0 == keep_this[c]
		    ) {
		    
		    answer = (always_store ? 
			      *def_ans_yes : *def_ans_no);	/* default answer */
		    
		    if(l_ask_store && prompt_area) {
			int def = answer;		       
			
			struct string * message = NULL;
		    againR:
			switch (c) {
			    
			case sync_store_received:
			    
			    
			    /* NOTICE: prompt_letter may return EOF */		    
			    if (marked_read_x[c] == 1)
				answer = prompt_letter(0,"",def,
						       PROMPT_yesno|PROMPT_cancel|
						       PROMPT_redraw_mark|PROMPT_ctrlL,		      
						       prompt_area,
						       CATGETS(elm_msg_cat, ElmSet, 
							       ElmLeaveMoveMessage,
							       "Move read message to \"received\" folder? (%c/%c) "),
						       *def_ans_yes, *def_ans_no);
			    else
				answer = prompt_letter(0,"",def,
						       PROMPT_yesno|PROMPT_cancel|
						       PROMPT_redraw_mark|PROMPT_ctrlL,		      
						       prompt_area,					       
						       CATGETS(elm_msg_cat, ElmSet, 
							       ElmLeaveMoveMessages,
							       "Move read messages to \"received\" folder? (%c/%c) "),
						       *def_ans_yes, *def_ans_no);
			    break;
			case sync_store_dnsmail:
			    if (!message)
				message =  read_message_text(c,(marked_read_x[c] != 1));
			    if (!message)
				break;
			    
			    if (keep_this[sync_store_received]) {
				answer = prompt_letter(0,"",def,
						       PROMPT_yesno|PROMPT_cancel|
						       PROMPT_redraw_mark|PROMPT_ctrlL,	
						       prompt_area,	
						       CATGETS(elm_msg_cat, ElmSet, 
							       ElmLeaveMoveX,
							       "Move %S to \"%s\" folder? (%c/%c) "),
						       message,
						       sync_store_config[c].text,
						       *def_ans_yes, *def_ans_no);
			    } else {
				answer = prompt_letter(0,"r",def,
						       PROMPT_yesno|PROMPT_cancel|
						       PROMPT_redraw_mark|PROMPT_ctrlL,	
						       prompt_area,	
						       CATGETS(elm_msg_cat, ElmSet, 
							       ElmLeaveMoveXRec,
							       "Move %S to \"%s\" or r)eceived folder? (%c/%c/r) "),
						       message,
						       sync_store_config[c].text,
						       *def_ans_yes, *def_ans_no);
			    }
			    
			    break;
			    
			case sync_store_COUNT:
			    break;
			}
			
			if (answer == ('L'&31) || answer == REDRAW_MARK) {
			    menu_ClearScreen(page);  /* Reset possible redraw flag */
			    
			    /* Call refresh routines of children */
			    menu_redraw_children(page);
			    
			    if (menu_need_redraw(prompt_area))
				menu_ClearScreen(prompt_area);   /* but clear redraw mark from prompt_area*/
			    
			    delay_redraw++;   /* Can't trigger redraw yet... */
			    goto againR;
			}

			if (message)
			    free_string(&message);
			
			if (TERMCH_interrupt_char == answer) {
			    return_value = sync_mbox_cancel;  
			    goto cleanup;	    
			}
			
			
			if (answer == EOF) {
			    return_value = sync_mbox_EOF;  
			    goto cleanup;	    
			}
						
			if (answer == *def_ans_no)
			    need_keep_x[c] = 1;
			
		    }

		    DPRINT(Debug,4,(&Debug,
				    "sync_mbox:    move messages to %s folder? %c\n",
				    sync_store_config[c].text,
				    answer));
		    
		    if (answer == *def_ans_yes) {
			need_move_x[c] = 1;
			ask_unread = 1;
		    } else if (answer == 'r') {
			need_move_x[c] = -1;
			ask_unread = 1;
		    }
		    		    
		} else {
		    ask_unread = 1;
		    DPRINT(Debug,3,(&Debug,  
				    "sync_mbox: class %s: No messages marked to read (or disposition is set), or class keeps messages\n",
				    sync_store_config[c].text
				    ));
		}		
	    }
	    
	    sc = get_storage_count(mailbox);
	    for (i = 0; i < sc; i++) {
		struct current_storage *storage = get_storage(mailbox,i);
		int j;
		
		int c1 = 0;
		int skipped = 0;
		
		if (!storage)
		    continue;
		
		if (storage->headers) {
		    for (j = 0; j < storage->message_count; j++) {
			
			if ((storage->headers[j] &&			    
			     isoff(storage->headers[j]->status, UNREAD))			   
			    && (storage->headers[j]->exit_disposition == UNSET)) {
			    
			    c = header_to_class(storage->headers[j]);
			    
			    if (c >= 0 && c < sync_store_COUNT &&
				keep_this[c] < 0) {
				
				c = sync_store_received;
			    }
			    
			    if (c >= 0 && c < sync_store_COUNT) {
				if (need_move_x[c] > 0) {
				    storage->headers[j]->exit_disposition =
					sync_store_config[c].store;
				    
				    sync_store_list[c].have_store   = 1;
				    
				    to_store++;
				    c1++;
				} else if (need_move_x[c] < 0) {
				    storage->headers[j]->exit_disposition =
					sync_store_config[sync_store_received].store;
				    sync_store_list[sync_store_received].have_store   = 1;
				    to_store++;
				    c1++;				    
				} else
				    goto need_keep;
			    } else {
			    need_keep:
				storage->headers[j]->exit_disposition = KEEP;
				to_keep++;
				c1++;
			    }
			} else {
			    skipped ++;
			}
		    }
		} 	    
				
		DPRINT(Debug,3,(&Debug, 
				"sync_mbox: mailbox #%d: %d (read) messages marked to keep or store; %d other messages skipped\n",
				i,c1,skipped));
	    }
	    	    
	    /* If we asked the user if read messages should be stored,
	     * and if the user wanted them kept instead, then certainly the
	     * user would want the unread messages kept as well.
	     */


	    if (ask_unread) {
		for (c = 0; c < sync_store_COUNT; c++) {
		    DPRINT(Debug,3,(&Debug,  
				    "sync_mbox: class %s: need %s unread messages\n",
				    sync_store_config[c].text,
				    need_keep_x[c] ? "keep" : "ask about"));
		}
	    }
	    DPRINT(Debug,3,(&Debug,  
			    "sync_mbox: %s ask about unread messages\n",
			    ask_unread ? "will" : "will not"));
	    
	    sc = get_storage_count(mailbox);
	    for (i = 0; i < sc; i++) {
		struct current_storage *storage = get_storage(mailbox,i);
		int j;
		int c1 = 0;
		int skipped = 0;
				
		if (!storage)
		    continue;
		
		if (storage->headers) {
		    for (j = 0; j < storage->message_count; j++) {
			if ((storage->headers[j] &&
			     ison(storage->headers[j]->status, UNREAD))
			    && (storage->headers[j]->
				exit_disposition == UNSET)) {

			    c = header_to_class(storage->headers[j]);
			    
			    if (c >= 0 && c < sync_store_COUNT &&
				keep_this[c] < 0) {
				
				c = sync_store_received;
			    }
			    
			    
			    if (c >= 0 && c < sync_store_COUNT &&
				need_keep_x[c]) {
				storage->headers[j]->exit_disposition = KEEP;
				to_keep++;
				c1++;
			    } else if (!ask_unread) {

				/* If we not ask about unread messages,
				   we keep them
				*/
				
				storage->headers[j]->exit_disposition = KEEP;
				to_keep++;
				c1++;
				    
			    } else {
				skipped ++;
			    }
			} else {
			    skipped ++;
			}
		    }	 
		    
		}
		
		DPRINT(Debug,3,(&Debug, 
				"sync_mbox: mailbox #%d: %d (unread) messages marked to keep; %d other messages skipped\n",
				i,c1,skipped));
	    }
		
		
	    if (ask_unread) {
		int marked_unread = 0;

		int marked_unread_x[sync_store_COUNT];
		int move_unread_x[sync_store_COUNT];
		
		for (c = 0; c < sync_store_COUNT; c++) {
		    marked_unread_x[c] = 0;
		    move_unread_x[c] = 0;
		}
		/* Determine if unread messages are to be kept */
				
		sc = get_storage_count(mailbox);
		for (i = 0; i < sc; i++) {
		    struct current_storage *storage = get_storage(mailbox,i);
		    int j;

		    int skipped = 0;
		    int c1 = 0;
		    
		    if (!storage)
			continue;
		    
		    if (storage->headers) {
			
			for (j=0;  j<storage->message_count; j++) {
			    if((storage->headers[j] &&
				ison(storage->headers[j]->status, UNREAD))
			       && (storage->headers[j]->
				   exit_disposition == UNSET)) {

				c = header_to_class(storage->headers[j]);

				if (c >= 0 && c < sync_store_COUNT &&
				    keep_this[c] < 0) {
				    
				    c = sync_store_received;
				}
			    
			    
				if (c >= 0 && c < sync_store_COUNT) {
				    marked_unread_x[c] ++;
				}
				
				
				marked_unread++;
				c1++;
			    } else {
				skipped++;
			    }
			}
		    }

		    DPRINT(Debug,3,(&Debug, 
				    "sync_mbox: mailbox #%d: %d (unread) messages to consider;  %d other messages skipped\n",
				    i,c1,skipped));
		}
		
		if (marked_unread) {
		    int class_count = 0;
		    enum sync_store_class only_class = sync_store_COUNT;

		     DPRINT(Debug,3,(&Debug, 
				     "sync_mbox: %d unread messsages (and no disposition)\n",
				     marked_unread));
		     
		     for (c = 0; c < sync_store_COUNT; c++) {
			  DPRINT(Debug,3,(&Debug,  
					  "sync_mbox: class %s: %d unread messsages (and no disposition)\n",
					  sync_store_config[c].text,
					  marked_unread_x[c]));

			  if (marked_unread_x[c] > 0) {
			      class_count ++;
			      only_class = c;
			  }
		     }

		     DPRINT(Debug,3,(&Debug, 
				     "sync_mbox: %d classes with unread messsages (and no disposition)",
				     class_count));

		     if (1 ==  class_count &&  only_class >= 0 && only_class < sync_store_COUNT) {
			 DPRINT(Debug,3,(&Debug,", only class %s",
					 sync_store_config[only_class].text));
		     }
		     DPRINT(Debug,3,(&Debug, "\n"));
		     
		     
		    answer = (always_keep ? 
			      *def_ans_yes : *def_ans_no);	/* default answer */


		    
		    if (l_ask_keep && prompt_area) {
			int def = answer;
			
			struct string * message = NULL;

		    againK:
			/* NOTICE: prompt_letter may return EOF */

			if (1 ==  class_count &&  only_class >= 0 && only_class < sync_store_COUNT) {
			    if (!message)
				message =  unread_message_text(only_class,marked_unread_x[c] != 1);

			    if (have_incoming) {
				answer = prompt_letter(0,"",def,
						       PROMPT_yesno|PROMPT_cancel|
						       PROMPT_redraw_mark|PROMPT_ctrlL,		      
						       prompt_area,
						       CATGETS(elm_msg_cat, ElmSet, 
							       ElmLeaveKeepX,
							       "Keep %S in incoming mailbox? (%c/%c) "),
						       message,
						       *def_ans_yes, *def_ans_no);
				
			    } else if (current_mailbox_name) {
				answer = prompt_letter(0,"",def,
						       PROMPT_yesno|PROMPT_cancel|
						       PROMPT_redraw_mark|PROMPT_ctrlL,		      
						       prompt_area,
						       CATGETS(elm_msg_cat, ElmSet, 
							       ElmLeaveKeepXName,
							       "Keep %S in %S mailbox? (%c/%c) "),
						       message,
						       current_mailbox_name,
						       *def_ans_yes, *def_ans_no);

			    } else {
				answer = prompt_letter(0,"",def,
						       PROMPT_yesno|PROMPT_cancel|
						       PROMPT_redraw_mark|PROMPT_ctrlL,		      
						       prompt_area,
						       CATGETS(elm_msg_cat, ElmSet, 
							       ElmLeaveKeepXCount,
							       "Keep %S in mailbox (%d mailboxes)? (%c/%c) "),
						       message,
						       mailbox_count,
						       *def_ans_yes, *def_ans_no);

			    }

			    if (answer != ('L'&31) && answer != REDRAW_MARK &&
				TERMCH_interrupt_char != answer && answer != EOF) {
				
				DPRINT(Debug,4,(&Debug,
						"sync_mbox:    keep %s unread messages? %c\n",
						sync_store_config[only_class].text,
						answer));
			    }
			    
			    if (answer == *def_ans_no)
				move_unread_x[only_class] = 1;
			    else if (answer == *def_ans_yes)
				move_unread_x[only_class] = 0;
			    
			} else {
			
			    if (have_incoming) {
				if (marked_unread == 1) {
				    answer = prompt_letter(0,"",def,
							   PROMPT_yesno|PROMPT_cancel|
							   PROMPT_redraw_mark|PROMPT_ctrlL,		      
							   prompt_area,
							   CATGETS(elm_msg_cat, ElmSet, 
								   ElmLeaveKeepMessage,
								   "Keep unread message in incoming mailbox? (%c/%c) "),
							   *def_ans_yes, *def_ans_no);
				} else {
				    answer = prompt_letter(0,"",def,
							   PROMPT_yesno|PROMPT_cancel|
							   PROMPT_redraw_mark|PROMPT_ctrlL,		      
							   prompt_area,
							   CATGETS(elm_msg_cat, ElmSet, ElmLeaveKeepMessages,
								   "Keep unread messages in incoming mailbox? (%c/%c) "),
							   *def_ans_yes, *def_ans_no);
				}
			    } else if (current_mailbox_name) {
				if (marked_unread == 1) {
				    answer = prompt_letter(0,"",def,
							   PROMPT_yesno|PROMPT_cancel|
							   PROMPT_redraw_mark|PROMPT_ctrlL,		      
							   prompt_area,
							   CATGETS(elm_msg_cat, ElmSet, 
								   ElmLeaveKeepMessageName,
								   "Keep unread message in %S mailbox? (%c/%c) "),
							   current_mailbox_name,
							   *def_ans_yes, *def_ans_no);
				} else {
				    answer = prompt_letter(0,"",def,
							   PROMPT_yesno|PROMPT_cancel|
							   PROMPT_redraw_mark|PROMPT_ctrlL,		      
							   prompt_area,
							   CATGETS(elm_msg_cat, ElmSet, ElmLeaveKeepMessagesName,
								   "Keep unread messages in %S mailbox? (%c/%c) "),
							   current_mailbox_name,
							   *def_ans_yes, *def_ans_no);
				
				}
				
			    } else {
				if (marked_unread == 1) {
				    answer = prompt_letter(0,"",def,
							   PROMPT_yesno|PROMPT_cancel|
							   PROMPT_redraw_mark|PROMPT_ctrlL,		      
							   prompt_area,
							   CATGETS(elm_msg_cat, ElmSet, 
								   ElmLeaveKeepMessageCount,
								   "Keep unread message in mailbox (%d mailboxes)? (%c/%c) "),
							   mailbox_count,
							   *def_ans_yes, *def_ans_no);
				} else {
				    answer = prompt_letter(0,"",def,
							   PROMPT_yesno|PROMPT_cancel|
							   PROMPT_redraw_mark|PROMPT_ctrlL,		      
							   prompt_area,
							   CATGETS(elm_msg_cat, ElmSet, ElmLeaveKeepMessagesCount,
								   "Keep unread messages in mailbox (%d mailboxes)? (%c/%c) "),
							   mailbox_count,
							   *def_ans_yes, *def_ans_no);
				}				
			    }

			    if (answer != ('L'&31) && answer != REDRAW_MARK &&
				TERMCH_interrupt_char != answer && answer != EOF) {
				DPRINT(Debug,4,(&Debug,
						"sync_mbox:    keep unread messages? %c\n",
						answer));
			    }
			    
			    for (c = 0; c < sync_store_COUNT; c++) {
				if (answer == *def_ans_no)
				    move_unread_x[c] = 1;
				else if (answer == *def_ans_yes)
				    move_unread_x[c] = 0;				
			    }
			}
			
			if (answer == ('L'&31) || answer == REDRAW_MARK) {
			    menu_ClearScreen(page); /* Reset possible redraw flag */
			    
			    /* Call refresh routines of children */
			    menu_redraw_children(page);
			    
			    if (menu_need_redraw(prompt_area))
				menu_ClearScreen(prompt_area);   /* but clear redraw mark from prompt_area*/
			    
			    delay_redraw++;   /* Can't trigger redraw yet... */
			    
			    goto againK;
			}
			
			if (message)
			    free_string(&message);
			
			if (TERMCH_interrupt_char == answer) {
			    return_value = sync_mbox_cancel;  
			    goto cleanup;	    
			}
			    
			
			if (answer == EOF) {
			    return_value = sync_mbox_EOF;  
			    goto cleanup;	    
			}
			
		    }
		    		    
		    sc = get_storage_count(mailbox);
		    for (i = 0; i < sc; i++) {
			struct current_storage *storage = 
			    get_storage(mailbox,i);
			
			int c1 = 0;
			int skipped = 0;
			
			if (!storage)
			    continue;
			
			if (storage->headers) {
			    int j;
			    
			    for (j = 0; j < storage->message_count; j++) {
				if ((storage->headers[j] &&
				     ison(storage->headers[j]->status, UNREAD))
				    && (storage->headers[j]->
					exit_disposition == UNSET)) {
				    
				    c = header_to_class(storage->headers[j]);
				    
				    if (c >= 0 && c < sync_store_COUNT) {
					if (keep_this[c] < 0) {
					    c = sync_store_received;
					} 
				    }
				    
				    if (c >= 0 && c < sync_store_COUNT &&
					move_unread_x[only_class]) {

					if (need_move_x[c] < 0) {
					    storage->headers[j]->exit_disposition =
						sync_store_config[sync_store_received].store;
					} else {					
					    storage->headers[j]->exit_disposition =
						sync_store_config[c].store;
					}
						
					sync_store_list[c].have_store   = 1;
					to_store++;
					c1++;
				    } else {
					storage->headers[j]->
					    exit_disposition = KEEP;
					to_keep++;
					c1++;
				    }	      
				} else {
				    skipped ++;
				}
			    }
			    
			}
			
			DPRINT(Debug,3,(&Debug, 
					"sync_mbox: mailbox #%d: %d (unread) messages marked to keep or store; %d other messages skipped\n",
					i,c1,skipped));
			
		    }
		} else {
		    DPRINT(Debug,3,(&Debug,  
				    "sync_mbox: No messages marked to unread (or disposition is set)\n"));
		    
		}
	    }
	}
	
    }


    sc = get_storage_count(mailbox);
    for (i = 0; i < sc; i++) {
	int c1 = 0;
	int handled = 0;
	struct current_storage *storage = 
	    get_storage(mailbox,i);

	if (!storage)
	    continue;

	if (storage->headers) {
	    int j;
	    
	    for (j = 0; j < storage->message_count; j++) {
				
		if (UNSET == storage->headers[j]->exit_disposition) {
		    storage->headers[j]->exit_disposition = KEEP;
		    to_keep++;
		    c1++;
		} else {
		    handled++;
		}
	    }
	}
	
	DPRINT(Debug,3,(&Debug, 
			"sync_mbox: mailbox #%d: %d unhandled messages marked to keep; %d other messages handled already\n",
			i,c1,handled));
    }
    	    
    DPRINT(Debug,3,(&Debug,  "sync_mbox: Messages to store: %d\n", to_store));
    DPRINT(Debug,3,(&Debug,  "sync_mbox: Messages to keep: %d (+ %d hide)\n", 
		    to_keep, to_hide));

    /* ================================================================ */

    /* get_message_count do not calclulate hidden (to_hide) messages !! */
    
    if(to_delete + to_store + to_keep + to_ignore != 
       get_message_count(mailbox)) {
	int LINES, COLUMNS;

	menu_get_sizes(page,&LINES, &COLUMNS);


	MoveCursor(LINES-1, 0);
	Raw(OFF);
	DPRINT(Debug,1,(&Debug, 			
			"sync_mbox: Error: %d to delete + %d to store + %d to keep + %d to ignore != %d message cnt\n",
			to_delete, to_store, to_keep, to_ignore, 
			get_message_count(mailbox)));

	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSomethingWrongInCounts,
			  "Something wrong in message counts! Folder unchanged.\n"));
	panic("MBOX PANIC",__FILE__,__LINE__,"sync_mbox",
	      "Something wrong in message counts!",0);      
    }
	  
    /* If we are not resyncing, we are leaving the mailfile and
     * the new messages are new no longer. Note that this changes
     * their status.
     */

    sc = get_storage_count(mailbox);
    for (i = 0; i < sc; i++) {
	struct current_storage *storage = get_storage(mailbox,i);
	int j;
	
	if (!resyncing && storage && storage->headers) {
	    for (j = 0; j < storage->message_count; j++) {
		if (storage->headers[j] &&
		    ison(storage->headers[j]->status, NEW)) {
		    clearit(storage->headers[j]->status, NEW);
		    storage->headers[j]->status_chgd = TRUE;
		}
	    }
	}
    }

    /* If all messages are to be kept and none have changed status
     * we don't need to do anything because the current folder won't
     * be changed by our writing it out - unless we are resyncing, in
     * which case we force the writing out of the mailfile.
     */

    sc = get_storage_count(mailbox);
    for (i = 0; i < sc; i++) {
	struct current_storage *storage = get_storage(mailbox,i);
	int j;
	
	if (!storage)
	    continue;

	if (storage->headers) 	    
	    for (num_chgd_status = 0, j = 0; j < storage->message_count; j++)
		if(storage->headers[j] &&
		   storage->headers[j]->status_chgd == TRUE)
		    num_chgd_status++;
    }

    if (!to_delete && !to_store && !num_chgd_status && !resyncing) {
	DPRINT(Debug,3,(&Debug,
			"sync_mbox: Folder(s) keep as is (%d folders)!\n",
			sc));
	
	if (1 == sc)
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFolderUnchanged,
			      "Folder unchanged."));
	else
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFoldersUnchangedCount,
			      "Folders unchanged (%d folders)."),
		      sc);
	    
	return_value = sync_mbox_nochange;
	goto cleanup;
    }
    
    if (to_store > 0) {

	int some_opened = 0;
	
	for (c = 0; c < sync_store_COUNT; c++) {	
       	
	    if (sync_store_list[c].new_is_xx &&
		*new_folder) {
		int M = get_folder_mode(*new_folder);

		if (0 != (M & FOLDER_FILE)) {
		    int r;
		
		    struct cancel_data *cd = NULL;
		    
		    setup_mbx_cancel_message(&cd,mbx_leaving_mailbox);
		    
		    DPRINT(Debug,3,(&Debug,
				    "sync_mbox: --> Need close new folder...\n"));
		
		    /* new_folder is same than =received (or other), 
		       we need close new_folder so that locking does not
		       cause problems
		   
		       Two problems:
		       - Possible locking on prepare_write_folder()
		       may fail on some situations
		       
		       - Unlocking on end_write_folder() may cause
		       that lock of new_folder is lost!
		       
		       leave_old_folder frees data.
		    */
		    
		    DPRINT(Debug,9, (&Debug, 
				     "sync_mbox: freeing new_folder\n"));
		    
		    r = leave_old_folder(new_folder,CLOSE_NORMAL,cd);
		    
		    if (!r) {
			DPRINT(Debug,9, (&Debug, 
					 "sync_mbox: close_folder failed\n"));
		    }
		    if (is_canceled(cd)) {
			DPRINT(Debug,9, (&Debug, 
					 "sync_mbox: Cancel on close_folder\n"));
		    }
		    free_cancel(&cd);
		    
		    
		    /* change_file() on quit.c will reopen 
		       new_folder when needed
		    */

		}
	    }
	    
	    /* Permissons check for =received (and others), and open of it ... */

	    if (sync_store_list[c].have_store &&
		sync_store_list[c].can_store) {
		
		if (prepare_write_folder(sync_store_list[c].browser,
					  &sync_store_list[c].writeptr)) {

		    some_opened = 1;

		} else {
		    
		    if (POLL_method)
			wait_for_timeout(5);
		    else
			sleep(5);

		    sync_store_list[c].can_store = 0;
		    open_close_failure ++;
		}
	    }
	}	

	if (!some_opened) {
	    DPRINT(Debug,9, (&Debug, 
			     "sync_mbox: No folder any folder opened for writing\n"));
	    
	    return_value = sync_mbox_nochange;
            goto cleanup;	    
	}	
    }


    /* Formulate message as to number of keeps, stores, and deletes.
     * This is only complex so that the message is good English.
     */

    action_message(to_keep,to_store, to_delete);

    need_handle = 0;
    sc = get_storage_count(mailbox);
    for (i = 0; i < sc; i++) {
	struct current_storage *storage = get_storage(mailbox,i);
	enum real_save_stat r;
	
	DPRINT(Debug,11,(&Debug,  
			"sync_mbox: storage #%d/%d\n",i,sc));

	if (! storage ||
	    ! storage->current_folder ||
	    ! storage->headers) {

	    DPRINT(Debug,11,(&Debug,  
			     "sync_mbox: storage #%d -- empty or no folder\n",
			     i));

	    continue;    /* IGNORE */
	}

	if (0 != (get_folder_mode(storage->current_folder) & 
		  FOLDER_RDONLY)) {
	    
	    DPRINT(Debug,11,(&Debug,  
			     "sync_mbox: storage #%d -- read-only folder\n",
			     i));
	    
	    continue;    /* IGNORE */
	}
	
	r = real_save(storage,
		      sync_store_list,
		      to_store, to_keep, resyncing, to_delete,
		      to_hide);

	DPRINT(Debug,11,(&Debug,  
			 "sync_mbox: real_save returned %d", 
			 r));

	switch (r) {
	case real_save_reconnect: DPRINT(Debug,10,(&Debug, " real_save_reconnect")); break;
	case real_save_failure:   DPRINT(Debug,10,(&Debug, " real_save_failure")); break;
	case real_save_newmail:   DPRINT(Debug,10,(&Debug, " real_save_newmail")); break;
	case real_save_none:      DPRINT(Debug,10,(&Debug, " real_save_none")); break;
	case real_save_done:      DPRINT(Debug,10,(&Debug, " real_save_done")); break;
	}

	
	DPRINT(Debug,11,(&Debug,"  for %S (%s)\n",
			storage->current_folder->cur_folder_disp,
			storage->current_folder->cur_folder_sys));

	switch (r) {
	case real_save_reconnect:
	    reconnect_count++;
	    break;
	case  real_save_failure:
	    failure_count++;
	    break;
	case sync_mbox_newmail:
	    newmail_count++;
	    
	    if (!need_handle) {
		/* New mail is arrived */
		
		return_value = sync_mbox_newmail;  /* failure */
		goto cleanup;	    
	    }
	    break;

	case real_save_done:
	    saved_count++;
	    break;
	case  real_save_none:
	    break;
	}
	
	if (r < real_save_none)
	    need_handle++;
    }

    if (to_store > 0) {
		
	for (c = 0; c < sync_store_COUNT; c++) {
	    if (sync_store_list[c].have_store &&
		sync_store_list[c].can_store) {
	
		if (!end_write_folder(sync_store_list[c].browser,
				      &sync_store_list[c].writeptr)) {

		    if (sync_store_list[c].can_store) 
			open_close_failure = 1;
		}
	    }	    
	}

	if (open_close_failure) {
	    DPRINT(Debug,11,(&Debug,
			     "sync_mbox: open_close_failure=%d\n",
			     open_close_failure));
	    
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmCopyMessageFailed,
			      "OOPS! Storing of mail failed!"));
	    
	    return_value = sync_mbox_failure;  /* failure */
	    goto cleanup;		
	}
    }

    DPRINT(Debug,11,(&Debug,
		     "sync_mbox: need_handle=%d failure_count=%d reconnect_count=%d newmail_count=%d saved_count=%d\n",
		     need_handle,failure_count,reconnect_count,newmail_count,saved_count));
    
    if (need_handle) {
	if (failure_count)	
	    return_value = sync_mbox_failure;  /* failure */
	else if (reconnect_count)
	    return_value = sync_mbox_reconnect;
	else
	    return_value = sync_mbox_failure;
	goto cleanup;
    }

    if (saved_count)
	return_value = sync_mbox_changed;
    else
	return_value = sync_mbox_nochange;


    /* Clear action message */
    lib_error(FRM(""));
	
 cleanup:

    if (incoming_browser)
        free_dir(&incoming_browser);
    if (incoming_name)
        free_string(&incoming_name);
    
    for (c = 0; c < sync_store_COUNT; c++) {
	if (sync_store_list[c].browser)
	    free_dir(&(sync_store_list[c].browser));
	if (sync_store_list[c].name)
	    free_string(& (sync_store_list[c].name));
	if (sync_store_list[c].writeptr) /* If not done ealrier ...  */
	    end_write_folder(sync_store_list[c].browser,
			     &sync_store_list[c].writeptr);
    }
	
    if (current_mailbox_name)
	free_string(&current_mailbox_name);

    if (delay_redraw)
	menu_trigger_redraw(page); 

    DPRINT(Debug,11,(&Debug,  "sync_mbox=%d",return_value));
    switch (return_value) {
    case sync_mbox_EOF:        DPRINT(Debug,11,(&Debug," sync_mbox_EOF"));       break;
    case sync_mbox_cancel:     DPRINT(Debug,11,(&Debug," sync_mbox_cancel"));    break;
    case sync_mbox_reconnect:  DPRINT(Debug,11,(&Debug," sync_mbox_reconnect")); break;
    case sync_mbox_failure:    DPRINT(Debug,11,(&Debug," sync_mbox_failure"));   break;
    case sync_mbox_newmail:    DPRINT(Debug,11,(&Debug," sync_mbox_newmail"));   break;
    case sync_mbox_nochange:   DPRINT(Debug,11,(&Debug," sync_mbox_nochange"));  break;
    case sync_mbox_changed:    DPRINT(Debug,11,(&Debug," sync_mbox_changed"));   break;
    }

    DPRINT(Debug,11,(&Debug,"\n"));
    
    return return_value;
}

#ifdef HASSIGPROCMASK
	sigset_t	toblock, oldset;
#else  /* HASSIGPROCMASK */
#  ifdef HASSIGBLOCK
	int		toblock, oldset;
#  else /* HASSIGBLOCK */
#    ifdef HASSIGHOLD
	/* Nothing required */
#    else /* HASSIGHOLD */
	SIGHAND_TYPE	(*oldhup)();
	SIGHAND_TYPE	(*oldint)();
	SIGHAND_TYPE	(*oldquit)();
	SIGHAND_TYPE	(*oldstop)();
#    endif /* HASSIGHOLD */
#  endif /* HASSIGBLOCK */
#endif /* HASSIGPROCMASK */

/*
 * Block all keyboard generated signals.  Need to do this while
 * rewriting mailboxes to avoid inadvertant corruption.  In
 * particular, a SIGHUP (from logging out under /bin/sh), can
 * corrupt a spool mailbox during an elm autosync.
 */

static void block_signals()
{
    /* XXXX cancel handler not actually installed on that point now
    */
    
    int have_cancel =  is_cancel_installed();

    DPRINT(Debug,1, (&Debug,  "block_signals: have_cancel=%d\n",
		     have_cancel));
    
#ifdef HASSIGPROCMASK
    sigemptyset(&oldset);
    sigemptyset(&toblock);
    sigaddset(&toblock, SIGHUP);
	
    if (have_cancel) {   /* have_cancel is possible only of sigaction() is available */
	 DPRINT(Debug,10, (&Debug,
			   "block_signals: Have cancel handler installed for SIGINT - no blocking\n"));
    } else {
	sigaddset(&toblock, SIGINT);
    }
    
	sigaddset(&toblock, SIGQUIT);
#ifdef SIGTSTP
	sigaddset(&toblock, SIGTSTP);
#endif /* SIGTSTP */

	sigprocmask(SIG_BLOCK, &toblock, &oldset);

#else /* HASSIGPROCMASK */
#  ifdef HASSIGBLOCK
	toblock = sigmask(SIGHUP) | sigmask(SIGQUIT);

	if (have_cancel) {   /* Should not happen */
	    DPRINT(Debug,1, (&Debug,
			     "block_signals: ERROR: have_cancel set but HASSIGPROCMASK not defined\n"));	    
	    DPRINT(Debug,10, (&Debug,
			      "block_signals: Have cancel handler installed for SIGINT - no blocking\n"));
	} else {
	    toblock |= sigmask(SIGINT);
	}
	
#ifdef SIGTSTP
	toblock |= sigmask(SIGTSTP);
#endif /* SIGTSTP */

	oldset = sigblock(toblock);

#  else /* HASSIGBLOCK */
#    ifdef HASSIGHOLD
	sighold(SIGHUP);
	sighold(SIGINT);
	sighold(SIGQUIT);
#ifdef SIGTSTP
	sighold(SIGTSTP);
#endif /* SIGTSTP */

#    else /* HASSIGHOLD */
	oldhup  = signal(SIGHUP, SIG_IGN);
	oldint  = signal(SIGINT, SIG_IGN);
	oldquit = signal(SIGQUIT, SIG_IGN);
#ifdef SIGTSTP
	oldstop = signal(SIGTSTP, SIG_IGN);
#endif /* SIGTSTP */
#    endif /* HASSIGHOLD */
#  endif /* HASSIGBLOCK */
#endif /* HASSIGPROCMASK */
}

/*
 * Inverse of the previous function.  Restore keyboard generated
 * signals.
 */
static void unblock_signals()
{
    DPRINT(Debug,1, (&Debug, "unblock_signals\n"));
#ifdef HASSIGPROCMASK
    sigprocmask(SIG_SETMASK, &oldset, (sigset_t *)0);

#else  /* HASSIGPROCMASK */
#  ifdef HASSIGBLOCK
    sigsetmask(oldset);

#  else /* HASSIGBLOCK */
#    ifdef HASSIGHOLD
	sigrelse(SIGHUP);
	sigrelse(SIGINT);
	sigrelse(SIGQUIT);
#ifdef SIGTSTP
	sigrelse(SIGTSTP);
#endif /* SIGTSTP */

#    else /* HASSIGHOLD */
	signal(SIGHUP, oldhup);
	signal(SIGINT, oldint);
	signal(SIGQUIT, oldquit);
#ifdef SIGTSTP
	signal(SIGTSTP, oldstop);
#endif /* SIGTSTP */

#    endif /* HASSIGHOLD */
#  endif /* HASSIGBLOCK */
#endif /* HASSIGPROCMASK */
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
