static char rcsid[] = "@(#)$Id: thread.c,v 2.8 2021/07/15 17:30:00 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.8 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> (was hurtta+elm@posti.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_messages.h"

DEBUG_VAR(Debug,__FILE__,"messages");

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

/* SHRT_MAX */
#include <limits.h>



#define THREAD_NAME_magic	0xF50A

static struct thread_name {
    unsigned short            magic;      /* THREAD_NAME_magic */

    int                      refcount;  
    
    struct string_sort  * thread_title;

    int            * thread_list;
    int              thread_list_len;

} * give_thread_name P_((struct sortlist   ** tname_map,
			 struct string_sort * thread_title,
			 size_t             * idx));

static struct thread_name * malloc_thread_name P_((struct string_sort  * thread_title));
static struct thread_name * malloc_thread_name(thread_title)
     struct string_sort  * thread_title;
{
    struct thread_name *  ret = safe_zero_alloc(sizeof(*ret));

    ret->magic  = THREAD_NAME_magic;
    
    ret->thread_title =  thread_title;
    inc_string_sort_refcount(ret->thread_title);
    
    ret->thread_list = NULL;
    ret->thread_list_len = 0;

    ret->refcount = 1;

    return ret;
}

static void  inc_thread_name_refcount P_((struct thread_name *name));
static void  inc_thread_name_refcount(name)
     struct thread_name *name;
{
    
    if (THREAD_NAME_magic != name->magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"inc_thread_name_refcount",
	      "Bad magic number",0);

    name->refcount++;
}

static void free_thread_name P_((struct thread_name **name));
static void free_thread_name(name)
     struct thread_name **name;
{
    if (THREAD_NAME_magic != (*name)->magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"free_thread_name",
	      "Bad magic number",0);
    
    if ((*name)->refcount < 1)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"free_thread_name",
	      "Bad refcount",0);

    (*name)->refcount--;

    if ((*name)->refcount > 0) {
	 
	*name = NULL;
	return;
    }

    if ((*name)->thread_title)
	free_string_sort(& ((*name)->thread_title));

    if ((*name)->thread_list) {
	free((*name)->thread_list);
	(*name)->thread_list = NULL;
    }
    (*name)->thread_list_len = 0;

    (*name)->magic = 0;  /* Invalidate */
    free(*name);
    *name = NULL;
    
}

S_(alloc_sort_item_f alloc_tname_item)
static void alloc_tname_item P_((union sort_item      * res,
				const union sort_key   key,
				const union sort_item_default def
				));
static void alloc_tname_item(res,key,def)
     union sort_item      * res;
     const union sort_key   key;
     const union sort_item_default def;  
{
    res->thread_name = malloc_thread_name(key.string_sort);
}

S_(free_sort_item_f free_tname_item)
static void free_tname_item P_((union sort_item      * ptr)); /* Decrements refcount */
static void free_tname_item(ptr)
     union sort_item      * ptr; /* Decrements refcount */
{
    free_thread_name( & (ptr->thread_name));
}

S_(inc_sort_item_refcount_f inc_tname_item_refcount)
static void inc_tname_item_refcount P_((union sort_item item));
static void inc_tname_item_refcount(item)
     union sort_item item;
{
    inc_thread_name_refcount(item.thread_name);
}

S_(sort_item_debug_name_f tname_item_name)
static struct string * tname_item_name   P_((const union sort_item item));
static struct string * tname_item_name (item)
     const union sort_item item;
{
    struct thread_name      * name = item.thread_name;

    if (THREAD_NAME_magic != name->magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"tname_item_name",
	      "Bad magic number",0);

    /* give_string_from_string_sort() returns original string with incremented refcount */

    return give_string_from_string_sort(name->thread_title);
}

S_(compare_sort_key_to_item_f tname_key_cmp_item)
static int tname_key_cmp_item P_((const union sort_key key,
				  const union sort_item item));
static int tname_key_cmp_item(key,item)
     const union sort_key key;
     const union sort_item item;
{
    struct thread_name      * name = item.thread_name;

    if (THREAD_NAME_magic != name->magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"tname_key_cmp_item",
	      "Bad magic number",0);

    return string_sort_cmp(key.string_sort,name->thread_title);
}

S_(sort_key_debug_name_f tname_key_name)
static struct string * tname_key_name P_((const union sort_key key));
static struct string * tname_key_name(key)
     const union sort_key key;
{

    /* give_string_from_string_sort() returns original string with incremented refcount */
    
    return give_string_from_string_sort(key.string_sort);
}

static struct sort_operation tname_map_operation = {
    SORT_OPERATION_magic,
    alloc_tname_item,
    free_tname_item,
    inc_tname_item_refcount,
    tname_item_name,
    tname_key_cmp_item,

    /* KEY operations */
    tname_key_name    
};

static struct thread_name * give_thread_name(tname_map, thread_title, idx)
     struct sortlist   ** tname_map; 
     struct string_sort * thread_title;   
     size_t             * idx;
{
    struct thread_name    * ret = NULL;
    union sort_key          key;
    union sort_item         res;
    union sort_item_default def;
    size_t                  res_idx = 0;

    /* give_string_from_string_sort() returns original string 
       with incremented refcount 
    */
    struct string * S2 = give_string_from_string_sort(thread_title);
    
    DPRINT(Debug,14,(&Debug,
		     "give_thread_name: thread_title: %S%s\n",
		     S2,
		     0 == string_len(S2) ? " (empty)" : ""));

    if (!*tname_map) {

	DPRINT(Debug,14,(&Debug,
			 "give_thread_name: Creating tname_map\n"));

	*tname_map = alloc_sort_list(&tname_map_operation,
				     1 /* prealloc one item */);

    }

    if (idx)
	*idx  =  0;
        
    key.string_sort = thread_title;
    def.dummy       = NULL;
    res.dummy       = NULL;

    /* Increments refcount, returns 1 if found */
    if (search_sort_list_item(*tname_map,sort_list_search_create,
			      key,def,&res,&res_idx,
			      NULL /* append_need_rewrite */)) {

	DPRINT(Debug,14,(&Debug,
			 "give_thread_name: search_sort_list_item found or created item #%zu\n",
			 res_idx));
	if (idx)
	    *idx  = res_idx;

	ret = res.thread_name;
    } else {

	DPRINT(Debug,14,(&Debug,
			  "give_thread_name: search_sort_list_item  failed to create entry\n"));

	ret = malloc_thread_name(thread_title);
    }

    /* Decrement refcount */
    free_string(&S2);
    
    return ret;
}

struct thread {
    struct thread_info *t;

    int  * msg_list;
    int    msg_list_len;
};

#define THREADVIEW_magic        0xF509

struct ThreadView {
    unsigned short            magic;    /* THREADVIEW_magic */

    struct sortlist  * name_root;

    struct thread       * thread;
    int                   thread_len;
};


static void free_thread_info P_((struct thread_info **t));
static void free_thread_info(t) 
     struct thread_info **t;
{
    if ((*t)->thread_subject)
	free_string_sort(& ((*t)->thread_subject));

    free(*t);
    *t = NULL;
}

static struct thread_info *malloc_thread_info P_((void));
static struct thread_info *malloc_thread_info()
{
    struct thread_info * ret= safe_malloc(sizeof (*ret));

    bzero((void *)ret, sizeof(*ret));

    ret->thread_subject = NULL;

    return ret;
}


void free_thread_view(t)
     struct ThreadView **t;
{
    if (THREADVIEW_magic  != (*t)->magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"free_thread_view",
	      "Bad magic number",0);
        
    if ((*t)->thread) {
	int i;
	for (i = 0; i < (*t)->thread_len; i++) {
	    if ((*t)->thread[i].t)
		free_thread_info(& (*t)->thread[i].t);


	    if ((*t)->thread[i].msg_list) {		
		free((*t)->thread[i].msg_list);
		(*t)->thread[i].msg_list = NULL;
	    }
	    (*t)->thread[i].msg_list_len = 0;
	}
	
	free((*t)->thread);
	(*t)->thread = NULL;
    }
    (*t)->thread_len = 0;

    if ((*t)->name_root)
	free_sort_list(& ((*t)->name_root));

    (*t)->magic = 0;   /* Invalidate */
    free(*t);
    *t = NULL;
}

static struct ThreadView * malloc_mailbox_thread P_((void));
static struct ThreadView * malloc_mailbox_thread()
{
    struct ThreadView * ret;

    ret = safe_malloc(sizeof (*ret));

    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)ret,sizeof (*ret));

    ret->magic      = THREADVIEW_magic;

    ret->name_root  = NULL;

    ret->thread     = NULL;
    ret->thread_len = 0;

    return ret;
}

static struct string *normalize_subject P_((const struct string *subject));
static struct string *normalize_subject(subject)
     const struct string *subject;
{
    struct string * temp   = skip_ascii_head_from_string(subject, 
							 s2us("Re: "),1);
    struct string * result = collapse_spaces_from_string(temp);

    free_string (&temp);

    return result;
}


struct thread_sort_data {
    struct MailboxView *mailbox;
    int index;
};

static int compare_thread P_((struct thread_sort_data *X1,
			      struct thread_sort_data *X2));
static int compare_thread(X1,X2) 
     struct thread_sort_data *X1;
     struct thread_sort_data *X2;
{
    struct header_rec *hdr1 = 
	X1->mailbox->mailbox_type->
	mt_give_header_it(X1->mailbox,X1->index,
			  & X1->mailbox->view[X1->index]);
    struct header_rec *hdr2 = 
	X2->mailbox->mailbox_type->
	mt_give_header_it(X2->mailbox,X2->index,
			  & X2->mailbox->view[X2->index]);
    
    struct string * from1, * from2;

    int ret;

    if (!hdr1 || !hdr2)
	return 0;

    if (!hdr1->subject && hdr2->subject) 
	return -1;    
    if (hdr1->subject && !hdr2->subject) 
	return 1;
    if (!hdr1->subject && !hdr2->subject) 
	return 0;

    from1 =  normalize_subject(hdr1->subject);
    from2 =  normalize_subject(hdr2->subject);

    ret = string_cmp(from1,from2, 
		     0 /* == Values not comparable */ );

    free_string(&from1);
    free_string(&from2);

    if (0 == ret) 
	ret = time_sent_compare(hdr1,hdr2);
    
    return ret;
}

int time_OK(t)
     time_t t;
{
    if (time_MAX <= t || t <= 0)
	return 0;

    return 1;
}

static int time_clean P_((time_t t));
static int time_clean(t)
     time_t t;
{
    if (! time_OK(t))
	return 0;

    if (t > (long) sort_thread_max_time * 24 * 60 * 60 &&
	t < time_MAX - (long) sort_thread_max_time * 24 * 60 * 60)
	return 1;
    
    return 0;
}

void update_mailbox_threads(mailbox)
     struct MailboxView *mailbox;
{
    int i;
    int not_added = 0;
    struct ThreadView * TV;

    if (mailbox->magic         != MAILBOXVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"update_mailbox_threads",
	      "Bad magic number",0);

    if (! mailbox->thread_view)
	mailbox->thread_view = malloc_mailbox_thread();

    if (mailbox->thread_view->magic != THREADVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"update_mailbox_threads",
	      "Bad thread view magic number",0);

    TV = mailbox->thread_view;

    for (i = 0; i < TV->thread_len; i++) {
	/* Just reset counters -- msg_list_len is realloced later */

	TV->thread[i].msg_list_len = 0;
    }

    if (mailbox->mailbox_type->magic != MAILBOXTYPE_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"update_mailbox_threads",
	      "Bad type magic number",0);

    /* Read numbers */

    for (i = 0; i < mailbox->view_len; i++) {

	if (-1 == mailbox->view[i].thread_number)
	    not_added++;
	else {
	    int index = mailbox->view[i].thread_number;
	    struct string      * subj1      = NULL;
	    struct string_sort * subj1_sort = NULL;
   	    
	    if (index < 0 || index >= TV->thread_len) {
		DPRINT(Debug,1,(&Debug,
				"update_mailbox_threads: [%d] bad thread index %d\n",
				i,index));
		mailbox->view[i].thread_number = -1;
		not_added++;
	    } else {
		struct header_rec *hdr = mailbox->mailbox_type->
		    mt_give_header_it(mailbox,i,
				      & mailbox->view[i]);

		struct thread_info * T = TV->thread[index].t;

		if (!hdr) {
		    DPRINT(Debug,7,(&Debug,
				    "update_mailbox_threads: [%d] thread index %d -- no header available\n",
				    i,index));
		    mailbox->view[i].thread_number = -1;
		    not_added++;
		    continue;
		}

		if (!hdr->subject) {
		    DPRINT(Debug,7,(&Debug,
				    "update_mailbox_threads: [%d] thread index %d -- no subject\n",
				    i,index));

		    subj1 = new_string(system_charset);
		} else {
		    subj1 = normalize_subject(hdr->subject);
		}

		/* shares string and increments refcount */
		subj1_sort = new_string_sort(subj1);

		
		if (0 != string_sort_cmp(T->thread_subject,subj1_sort)) {
		    /* give_string_from_string_sort() returns original string 
		       with incremented refcount 
		    */
		    struct string * S2 = give_string_from_string_sort(T->thread_subject);
		    
		    
		    DPRINT(Debug,7,(&Debug,
				    "update_mailbox_threads: [%d] thread index %d -- subject not match\n",
				    i,index));
		    mailbox->view[i].thread_number = -1;
		    not_added++;

		    DPRINT(Debug,7,(&Debug,"  thread %d=%S\n",
				    index,S2));
		    DPRINT(Debug,7,(&Debug,"  mail %d=%S\n",
				    i,subj1));

		    free_string(&subj1);
		    free_string(&S2);
		    free_string_sort(&subj1_sort);
		    
		    continue;		    
		}

		/* Accept even invalid time here  */
		if (hdr->time_sent     == T->time_sent_first &&
		    T->time_sent_first == T->time_sent_last &&

		    hdr->time_menu_year == T->year_sent_first &&
		    T->year_sent_first == T->year_sent_last) {

		    DPRINT(Debug,7,(&Debug,
				    "update_mailbox_threads: [%d] thread index %d -- matches time exactly or both times are invalid\n",
				    i,index));
		    
		} else if (! time_OK(hdr->time_sent)) {

		    const char * s UNUSED_VAROK = 
			(time_MAX == hdr->time_sent) ? "MAX overflow" : "usupported value";	


		    if (0 == T->year_sent_first ||
			0 == T->year_sent_last ||
			hdr->time_menu_year <= 0 ||
			hdr->time_menu_year >= SHRT_MAX ||
			hdr->time_menu_year <
			T->year_sent_first - sort_thread_max_time / 365 ||
			hdr->time_menu_year >
			T->year_sent_last + sort_thread_max_time) {
			
			DPRINT(Debug,7,(&Debug,
					"update_mailbox_threads: [%d] thread index %d -- have %s -- year not match\n",
					i,index,s));

			mailbox->view[i].thread_number = -1;
			not_added++;

			
			DPRINT(Debug,7,(&Debug,"  thread %d= first year=%d last year=%d\n",
					index,
					T->year_sent_first,
					T->year_sent_last));
			
			DPRINT(Debug,7,(&Debug,"  mail %d= time sent year=%d\n",
					i,
					hdr->time_menu_year));
			
			free_string(&subj1);
			free_string_sort(&subj1_sort);

			continue;	
		    }
		   
		} else if (! time_clean(T->time_sent_first) ||
			   ! time_clean(T->time_sent_last) ||
			   hdr->time_sent < 
			   T->time_sent_first - 
			   (long) sort_thread_max_time * 24 * 60 * 60 ||
			   hdr->time_sent >
			   T->time_sent_last +
			   (long) sort_thread_max_time * 24 * 60 * 60) {
		    
		    DPRINT(Debug,7,(&Debug,
				    "update_mailbox_threads: [%d] thread index %d -- time not match\n",
				    i,index));
		    mailbox->view[i].thread_number = -1;
		    not_added++;


		    DPRINT(Debug,7,(&Debug,"  thread %d= first time=%ld last time=%ld\n",
				    index,
				    (long) T->time_sent_first,
				    (long) T->time_sent_last));
		    DPRINT(Debug,7,(&Debug,"  mail %d= time sent=%ld\n",
				    i,
				    (long)hdr->time_sent));
		    
		    free_string(&subj1);
		    free_string_sort(&subj1_sort);

		    continue;		    
		}
		    
		TV->thread[index].msg_list =
		    safe_array_realloc(TV->thread[index].msg_list,
				       (TV->thread[index].msg_list_len+1),
				       sizeof (TV->thread[index].msg_list[0]));

		TV->thread[index].msg_list[TV->thread[index].msg_list_len]
		    = i;
		TV->thread[index].msg_list_len++;

		free_string(&subj1);

		/* Decrement refcount */
		free_string_sort(&subj1_sort);
		
	    }
	}
    }

    for (i = 0; i < TV->thread_len; i++) {
	TV->thread[i].t->num_messages =
	    TV->thread[i].msg_list_len;
    }

    DPRINT(Debug,7,(&Debug,
		    "update_mailbox_threads: not_added=%d\n",not_added));

    if (not_added > 0) {
	struct thread_sort_data  * sort_data = 
	    safe_calloc(not_added, sizeof(sort_data[0]));
	int idx = 0;

	int outer, inner = 0;

	typedef int (*compar) P_((const void *, const void *));
	compar X = (compar) compare_thread;


	for (i = 0; i < mailbox->view_len; i++) {
	    
	    if (-1 == mailbox->view[i].thread_number) {

		if (idx >= not_added)
		    panic("MBX VIEW PANIC",__FILE__,__LINE__,"update_mailbox_threads",
			  "Overflow",0);
		
		sort_data[idx].mailbox = mailbox;
		sort_data[idx].index   = i;
		idx++;
	    }
	}

	/* Sort with subject and sent_time */

	qsort(sort_data,idx,sizeof (sort_data[0]), X);

	for (outer = 0; outer < idx; outer = inner) {
	    int index = sort_data[outer].index;
	    struct header_rec *hdr;
	    struct string * subj1 = NULL;
	    struct string_sort * subj1_sort = NULL;
	    
	    time_t thread_first;
	    time_t thread_last;
	    short  thread_first_year;
	    short  thread_last_year;

	    int thread_idx;
	    int scan_idx;
	    int count;
	    int j;
	    
	    struct thread_name   * thread_name = NULL;
	    struct string        * thread_name_string = NULL;
	    size_t  current_thread_idx =  0;
	    struct string        * thread_subject_string = NULL;
	    
	    if (index < 0 || index >= mailbox->view_len)
		panic("MBX VIEW PANIC",__FILE__,__LINE__,"update_mailbox_threads",
		      "Bad index",0);
		
	    hdr = mailbox->mailbox_type->
		mt_give_header_it(mailbox,index,
				  & mailbox->view[index]);
	    
	    if (!hdr) {
		DPRINT(Debug,7,(&Debug,
				"update_mailbox_threads: [%d] -- no header available\n",
				index));
		inner = outer+1;
		continue;
	    }

	    if (!hdr->subject) {
		DPRINT(Debug,7,(&Debug,
				"update_mailbox_threads: [%d] -- no subject\n",
				index));
		
		subj1 = new_string(system_charset);
	    } else {
		subj1 = normalize_subject(hdr->subject);
	    }

	    /* shares string and increments refcount */
	    subj1_sort = new_string_sort(subj1);
	    
	    thread_first      = hdr->time_sent ;
	    thread_last       = hdr->time_sent ;
	    thread_first_year = hdr->time_menu_year;
	    thread_last_year  = hdr->time_menu_year;


	    for (inner = outer+1; inner < idx; inner++) {
		struct header_rec *hdr2;
		int index2 = sort_data[inner].index;
		struct string * subj2 = NULL;

		if (index2 < 0 || index2 >= mailbox->view_len)
		    panic("MBX VIEW PANIC",__FILE__,__LINE__,"update_mailbox_threads",
			  "Bad index",0);

		hdr2 = mailbox->mailbox_type->
		    mt_give_header_it(mailbox,index2,
				      & mailbox->view[index2]);

		if (!hdr2) {
		    DPRINT(Debug,7,(&Debug,
				    "update_mailbox_threads: [%d] -- no header available\n",
				    index2));
		    break;
		}

		if (!hdr2->subject) {
		    DPRINT(Debug,7,(&Debug,
				    "update_mailbox_threads: [%d] -- no subject\n",
				    index2));

		    subj2 = new_string(system_charset);
		} else {
		    subj2 = normalize_subject(hdr2->subject);
		}
		
		if (0 != string_cmp(subj2,subj1,
				    999 /* == Values not comparable */)) {
		    
		    free_string(& subj2);    /* NO MATCH */
		    break;
		}

		if (! time_OK(hdr2->time_sent)) {

		    if (0 == thread_first_year ||
			0 == thread_last_year ||
			hdr2->time_menu_year <= 0 ||
			hdr2->time_menu_year >= SHRT_MAX ||
			hdr2->time_menu_year <
			thread_first_year - sort_thread_max_time / 365 ||
			hdr2->time_menu_year >
			thread_last_year + sort_thread_max_time) {
		    
			free_string(& subj2);    /* NO MATCH */
			break;
		    }

		    
		    /* OK */

		    if (hdr2->time_menu_year < thread_first_year)
			thread_first_year = hdr2->time_menu_year;
		    if (hdr2->time_menu_year > thread_last_year)
			thread_last_year = hdr2->time_menu_year;
		    
		} else {

		    if (! time_clean(thread_first) ||
			! time_clean(thread_last) ||
			hdr2->time_sent < 
			thread_first - 
			(long) sort_thread_max_time * 24 * 60 * 60 ||
			hdr2->time_sent >
			thread_last +
			(long) sort_thread_max_time * 24 * 60 * 60) {
			
			free_string(& subj2);    /* NO MATCH */
			break;
		    }

		    /* OK */
		    
		    if (hdr2->time_sent < thread_first) {
			thread_first = hdr2->time_sent;

			if (hdr2->time_menu_year > 0 && 
			    hdr2->time_menu_year < SHRT_MAX) 
			    thread_first_year = hdr2->time_menu_year;
		    }

		    if (hdr2->time_sent > thread_last) {
			thread_last = hdr2->time_sent;
			if (hdr2->time_menu_year > 0 && 
			    hdr2->time_menu_year < SHRT_MAX)
			    thread_last_year = hdr2->time_menu_year;
		    }
		
		}

		free_string(& subj2);  
	    }
	    

	    thread_name = give_thread_name(& (TV->name_root), subj1_sort, &current_thread_idx);

	    thread_name_string =  give_string_from_string_sort(thread_name->thread_title);

	    DPRINT(Debug,7,(&Debug,
			    "update_mailbox_threads: thread subject %S => current subject index %zu\n",
			    thread_name_string,current_thread_idx));
	    
	    for (scan_idx = 0; scan_idx < thread_name->thread_list_len; scan_idx++) {
		
		struct thread_info * T;

		thread_idx = thread_name->thread_list[scan_idx];
		
		if (thread_idx < 0 || thread_idx >= TV->thread_len)
		    panic("MBX VIEW PANIC",__FILE__,__LINE__,"update_mailbox_threads",
			  "Bad thread index",0);
		    
		T = TV->thread[thread_idx].t;

		if (0 != string_sort_cmp(T->thread_subject,subj1_sort))
		    panic("MBX VIEW PANIC",__FILE__,__LINE__,"update_mailbox_threads",
			  "Bad thread index",0);

		/* Accept even invalid time here  */
		if (thread_last == thread_first &&
		    T->time_sent_first == T->time_sent_last &&
		    thread_last == T->time_sent_last &&

		    thread_first_year ==  thread_last_year &&
		    T->year_sent_first == T->year_sent_last  &&
		    thread_first_year == T->year_sent_first
		    
		    ) {

		    goto found_thread;
		
		} else if (time_OK(thread_last) &&
		    time_OK(thread_first) &&
		    time_clean(T->time_sent_first) &&
		    time_clean(T->time_sent_last)) {

		    if (thread_last > T->time_sent_first - 
			(long) sort_thread_max_time * 24 * 60 * 60 &&
			thread_first < T->time_sent_last +
			(long) sort_thread_max_time * 24 * 60 * 60) {
			
			if (thread_first < T->time_sent_first) {
			    T->time_sent_first = thread_first;
			    if (thread_first_year > 0 &&
				thread_first_year < SHRT_MAX)
				T->year_sent_first = thread_first_year;
			}
			if (thread_last > T->time_sent_last) {
			    T->time_sent_last = thread_last;

			    if (thread_last_year > 0 &&
				thread_last_year < SHRT_MAX)
				T->year_sent_last = thread_last_year;
			}			
			
			goto found_thread;
		    }

		} else if ( 0 != thread_first_year &&
			    0 != thread_last_year &&
			    0 != T->year_sent_first &&
			    0 != T->year_sent_last ) {

		    if (thread_last_year > T->year_sent_first -
			sort_thread_max_time / 365 &&
			thread_first_year < T->year_sent_last +
			sort_thread_max_time / 365) {

			if (thread_first_year <  T->year_sent_first)
			    T->year_sent_first = thread_first_year;

			if (thread_last_year > T->year_sent_last)
			    T->year_sent_last = thread_last_year;

			goto found_thread;
		    }
		}
	    }

	    TV->thread = safe_array_realloc(TV->thread, 
					    ( TV->thread_len+1),
					    sizeof (TV->thread[0]));	    
	    thread_idx = TV->thread_len;
	    
	    bzero((void *)(& (TV->thread[thread_idx])), 
		  sizeof(TV->thread[thread_idx]));

	    TV->thread_len++;

	    TV->thread[thread_idx].t = malloc_thread_info();
	    TV->thread[thread_idx].t->time_sent_first = thread_first;
	    TV->thread[thread_idx].t->time_sent_last  = thread_last;
	    TV->thread[thread_idx].t->year_sent_first = thread_first_year;
	    TV->thread[thread_idx].t->year_sent_last  = thread_last_year;

	    TV->thread[thread_idx].t->thread_subject  = subj1_sort;
	    inc_string_sort_refcount(TV->thread[thread_idx].t->thread_subject);

	    TV->thread[thread_idx].msg_list     = NULL;
	    TV->thread[thread_idx].msg_list_len = 0;

	    /* give_string_from_string_sort() returns original string with incremented refcount */
	    
	    DPRINT(Debug,7,(&Debug,
			    "update_mailbox_threads: Adding thread #%d to thread subject: %S (current subject index #%zu)\n",
			    thread_idx,
			    thread_name_string,
			    current_thread_idx));

	    thread_name->thread_list = 
		safe_array_realloc(thread_name->thread_list,
				   (thread_name->thread_list_len +1),
				   sizeof(thread_name->thread_list[0]));
	    thread_name->thread_list[thread_name->thread_list_len] = thread_idx;
	    thread_name->thread_list_len++;

	found_thread:

	    count = inner-outer;

	    thread_subject_string = give_string_from_string_sort(TV->thread[thread_idx].t->thread_subject);
	    
	    DPRINT(Debug,7,(&Debug,
			    "update_mailbox_threads: Adding %d messages to thread %d: %S (current subject index #%zu)\n",
			    count,thread_idx,
			    thread_subject_string,
			    current_thread_idx));
	    DPRINT(Debug,7,(&Debug,
			    "    messages sent time %ld - %ld\n",
			    (long)thread_first,
			    (long)thread_last));
	    DPRINT(Debug,7,(&Debug,
			    "    thread sent time %ld - %ld\n",
			    (long)(TV->thread[thread_idx].t->time_sent_first),
			    (long)(TV->thread[thread_idx].t->time_sent_last)));
	    DPRINT(Debug,7,(&Debug,
			    "    messages sent year %d - %d\n",
			    thread_first_year,
			    thread_last_year));
	    DPRINT(Debug,7,(&Debug,
			    "    thread sent year %d - %d\n",
			    TV->thread[thread_idx].t->year_sent_first,
			    TV->thread[thread_idx].t->year_sent_last));
	    
	    if (!count) 
		panic("MBX VIEW PANIC",__FILE__,__LINE__,"update_mailbox_threads",
		      "Empty count",0);
	    
	    count += TV->thread[thread_idx].msg_list_len;

	    TV->thread[thread_idx].msg_list  = 
		safe_array_realloc(TV->thread[thread_idx].msg_list,
				   count, sizeof(TV->thread[thread_idx].msg_list[0]));

	    for (j = outer; j < inner; j++) {
		int index = sort_data[j].index;
		if (TV->thread[thread_idx].msg_list_len >= count)
		    panic("MBX VIEW PANIC",__FILE__,__LINE__,"update_mailbox_threads",
			  "Overflow",0);
		    
		TV->thread[thread_idx].msg_list[TV->thread[thread_idx].msg_list_len]
		    = index;
		TV->thread[thread_idx].msg_list_len++;

		if (index < 0 || index >= mailbox->view_len)
		    panic("MBX VIEW PANIC",__FILE__,__LINE__,"update_mailbox_threads",
			  "Bad index",0);
		mailbox->view[index].thread_number = thread_idx;		    
	    }


	    TV->thread[thread_idx].t->num_messages =
		TV->thread[thread_idx].msg_list_len;
    
	    free_string(&subj1);
	    free_string_sort(&subj1_sort);
	    free_string(&thread_subject_string);

	    free_thread_name(& thread_name);
	    free_string(& thread_name_string);
	}

	free(sort_data);
    }

}

const struct thread_info * give_thread_info_s(s)
     struct sort_data *s;
{
    int idx;

    if (s->sort_data_type->magic != SORTDATATYPE_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"give_thread_info_s",
	      "Bad magic number",0);
    
    if (THREADVIEW_magic != s->t->magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"give_thread_info_s",
	      "Bad threadview magic number",0);

    idx = s->w.thread_number;
    if (-1 == idx)
	return NULL;     /* No thread information */

    if (idx < 0 || idx >= s->t->thread_len)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"give_thread_info_s",
	      "Bad thread index",0);

    return s->t->thread[idx].t;
	
}

int get_thread_count(mailbox, create)
     struct MailboxView *mailbox;
     int create;
{
    if (mailbox->magic         != MAILBOXVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"get_thread_count",
	      "Bad magic number",0);

    if (! mailbox->thread_view) {
	if (!create)
	    return -1;

	update_mailbox_threads(mailbox);
    }

    if (mailbox->thread_view->magic != THREADVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"get_thread_count",
	      "Bad thread view magic number",0);
    
    return mailbox->thread_view->thread_len;
}

const struct thread_info * give_thread_info(mailbox,idx)
     struct MailboxView * mailbox; 
     int idx;
{
    if (mailbox->magic         != MAILBOXVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"give_thread_info",
	      "Bad magic number",0);

    if (! mailbox->thread_view)
	return NULL;

    if (mailbox->thread_view->magic != THREADVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"give_thread_info",
	      "Bad thread view magic number",0);

    if (idx < 0 || idx >= mailbox->thread_view->thread_len)
	return NULL;

    return mailbox->thread_view->thread[idx].t;
}

/* caller must free result */
int * give_thread_message_list (mailbox,idx,reslen)
     struct MailboxView * mailbox; 
     int idx;
     int *reslen;
{
    int * res = NULL;
    int i;

    if (mailbox->magic         != MAILBOXVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"give_thread_message_list",
	      "Bad magic number",0);
    
    *reslen = 0;

    if (! mailbox->thread_view)
	return NULL;

    if (mailbox->thread_view->magic != THREADVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"give_thread_info",
	      "Bad thread view magic number",0);

    if (idx < 0 || idx >= mailbox->thread_view->thread_len)
	return NULL;
    
    if (mailbox->thread_view->thread[idx].msg_list_len < 1)
	return NULL;

    *reslen = mailbox->thread_view->thread[idx].msg_list_len;
    res = safe_calloc((*reslen), sizeof(res[0]));

    for (i = 0; i < *reslen; i++)
	res[i] = mailbox->thread_view->thread[idx].msg_list[i];

    return res;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
