static char rcsid[] = "@(#)$Id: out_utils.c,v 2.17 2022/05/18 18:28:32 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.17 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *  Based on Elm 2.4 src/out_utils.c. That code was following copyright:
 *
 *  The Elm Mail System 
 *
 *			Copyright (c) 1988-1992 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

/** This file contains routines used for output in the ELM program.

**/

#include "def_elm.h"
#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"ui");

static char *us2s P_((unsigned char *str));
static char *us2s(str) 
     unsigned char *str;
{
    return (char *)str;
}

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

static const unsigned char *cs2us P_((const char *str));
static const unsigned char *cs2us(str) 
     const char *str;
{
    return (const unsigned char *)str;
}


static char err_buffer[SLEN];		/* store last error message */
static struct string * err_buffer_W = NULL;

static time_t LAST_ERROR = 0;

static void error_middle P_((const char *s));
static void error_middle_W P_((const struct string *s));

static FILE *     messages_log_file = NULL;
static charset_t  messages_log_cs   = NULL;
static int        messages_log_pid  = 0;

int open_messages_log(filename)
     const char *filename;
{
    int err;
    int fd;
    FILE *f;
    int r;
    
    if (userid == -1 || groupid == -1) {
	/* save original user and group ids */
	userid  = getuid();
	groupid = getgid();	

	DPRINT(Debug,1,(&Debug,"open_messages_log: %s: Saving userid/groupid=%d/%d\n",
			filename,userid,groupid));

    }

    err = can_open(filename,"a+");
    if (0 != err) { 

	lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
			  "File %.50s is not writeable: %s"),
		  filename, strerror(err));
	return 0;
    }

    fd = open(filename,O_CREAT|O_RDWR|O_APPEND,00600);
    if (-1 == fd) {
	int err = errno;
	lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
			  "File %.50s is not writeable: %s"),
		  filename, strerror(err));
	return 0;
    }
    elm_chown(filename, userid, groupid, NULL); /* file owned by user */

#ifdef FD_CLOEXEC
    r = fcntl(fd, F_SETFD, FD_CLOEXEC);
#else
    r = fcntl(fd, F_SETFD, 1);
#endif    
    
    if (-1 == r) {
	int err UNUSED_VAROK = errno;
        
	DPRINT(Debug,14,(&Debug, 
			 "open_messages_log: Failed to set close-on-exec flag for %s: %s\n",
			 filename,
			 strerror(err)));
    } else if (0 == r) {
	DPRINT(Debug,14,(&Debug, 
			 "open_messages_log: Set close-on-exec flag for %s\n",
			 filename));
    }
    
    f = fdopen(fd,"a+");
    if (!f) {
	close(fd);
	return 0;
    }
    
    if (messages_log_file) {

	if (messages_log_cs &&
	    messages_log_pid == getpid()) {
	    
	    fprintf(messages_log_file,
		    "\n===== Changing logging to file %s\n",
		    filename);
	}
	fclose(messages_log_file);
    }
    
    messages_log_file = f;
    messages_log_cs  = NULL;  /* charset not determined yet */
    messages_log_pid = getpid();
    
#ifdef SETLINEBUF
    setlinebuf(messages_log_file);
#endif
 
    return 1;
}

void close_messages_log()
{

    if (messages_log_file) {
	const int pid = getpid();
		
	if (messages_log_cs &&
	    messages_log_pid == pid) {
	    
	    time_t now = 0;
	    
	    if (((time_t) -1) != time(&now)) {
		struct tm * zz = localtime(&now);
		
		if (!zz)
		    goto no_time14;

		fprintf(messages_log_file,
			"%d %02d:%02d:%02d --- closing messages log\n",
			pid,
			zz->tm_hour,
			zz->tm_min,
			zz->tm_sec);
	    } else {
	    no_time14:
		
		fprintf(messages_log_file,
			"%d --- closing messages log\n",
			pid);
	    }	    
	}
	
	fclose(messages_log_file);
	messages_log_file = NULL;
    }
    
}

static charset_t messages_log_ok P_((void));
static charset_t messages_log_ok()
{
    charset_t ret = NULL;

    static const char CHARSET[] = "@charset=";
    
    if (messages_log_file) {
	const int pid = getpid();

	if (messages_log_pid == pid) {
	    int started = 0;
	    
	    if (!messages_log_cs) {
		long pos = ftell(messages_log_file);

		if (0 ==  pos) {   /* On beginning of file */
		    char charset_buffer[80+sizeof CHARSET];
		    int len = mail_gets(charset_buffer,sizeof charset_buffer,messages_log_file);
		    
		    if (len > 0) {
			DPRINT(Debug,10,(&Debug,
					 "messages_log_ok: read %d characters: ",len));
			DEBUG_PRINT_BUFFER(Debug,10,len,s2us(charset_buffer));

			if ('\n' == charset_buffer[len-1]) {
			    /* mail_gets actually adds missing \n */

			    int i;
			    char * ok = NULL;

			    charset_buffer[len-1] = '\0';
			    
			    for (i = 0; i < len && i < sizeof CHARSET; i++) {

				/* Fully scanned when \0 on CHARSET buffer seen */
				if ('\0' == CHARSET[i]) {
				    ok = charset_buffer+i;
				    break;
				} else if (charset_buffer[i] != CHARSET[i]) {
				    DPRINT(Debug,10,(&Debug,
						     "messages_log_ok: match fail on pos %d\n",i));
				    break;
				}
			    }

			    if (ok) {
				charset_t set    = NULL;
				
				DPRINT(Debug,10,(&Debug,
						 "messages_log_ok: Got charset %s\n",ok));
				
				set = MIME_name_to_charset(ok,0);

				if (set) {
				    messages_log_cs = set;
				    started = 1;
				}
			    }
			    
			} else {
			    DPRINT(Debug,10,(&Debug," --- strange?\n"));
			}
			

		    } else if (feof(messages_log_file)) {
			int p;
			charset_t set    = NULL;
			const char * cs  = NULL;
			
			DPRINT(Debug,10,(&Debug,
					 "messages_log_ok: Empty file -- adding charset tag\n"));

			if (display_charset == system_charset &&
			    (p = charset_properties(system_charset)) &&
			    0 != (p & CS_printable) &&
			    0 != (p & CS_mapping)) {

			    cs = get_charset_MIME_name(system_charset);
			    if (cs) {
				set = system_charset;
				DPRINT(Debug,10,(&Debug,
						 "messages_log_ok: system charset %s selected\n",
						 cs));
			    } else
				goto try_utf8;
			    
			} else {
			try_utf8:
			    cs = "utf-8";

			    set = MIME_name_to_charset(cs,0);

			    if (set) {
				DPRINT(Debug,10,(&Debug,
						 "messages_log_ok: charset %s selected\n",
						 cs));
			    } else
				cs = NULL;
			    
			}

			if (set && cs) {
			    int len1 = fprintf(messages_log_file,"%s%s\n",CHARSET,cs);

			    if (len1 > 0) {
				DPRINT(Debug,10,(&Debug,
						 "messages_log_ok: %d characters written\n",len1));
				messages_log_cs = set;
				started = 1;
			    } else if (len1 < 0) {
				int err UNUSED_VAROK = errno;
				
				DPRINT(Debug,10,(&Debug,
						 "messages_log_ok: write failed: %s\n",strerror(err)));

			    }
			}			
		    }
		    		    
		} else if (-1 == pos) {
		    int err UNUSED_VAROK = errno;
		    
		    DPRINT(Debug,10,(&Debug,
				     "messages_log_ok: Ftell failed: %s\n",strerror(err)));
		}		
	    }

	    if (messages_log_cs) {
		int r = fseek(messages_log_file,0L,SEEK_END);

		if (0 == r) {
		    ret = messages_log_cs;
		    DPRINT(Debug,10,(&Debug,
				     "messages_log_ok: Seek to end succeed\n"));

		    if (started) {
			time_t now = 0;
			
			if (((time_t) -1) != time(&now)) {
			    struct tm * zz = localtime(&now);
			    
			    if (!zz)
				goto no_time14;
			    
			    fprintf(messages_log_file,
				    "%d %02d:%02d:%02d --- started messages log\n",
				    pid,
				    zz->tm_hour,
				    zz->tm_min,
				    zz->tm_sec);
			} else {
			no_time14:
			    
			    fprintf(messages_log_file,
				    "%d --- started messages log\n",
				    pid);
			}	    			
		    }
		    
		} else if (-1 == r) {
		    int err UNUSED_VAROK = errno;

		    DPRINT(Debug,10,(&Debug,
				     "messages_log_ok: Seek to end failed: %s\n",strerror(err)));
		}		
		
	    } else {
		DPRINT(Debug,10,(&Debug,
				 "messages_log_ok: log charset not found\n"));
	    }

	} else {

	    DPRINT(Debug,10,(&Debug,
			     "messages_log_ok: messages_log_pid=%d != my pid=%d\n",
			     messages_log_pid,pid));

	    fclose(messages_log_file);
	    messages_log_file = NULL;
	}	
    }
    

    if (ret) {
	const char * cs UNUSED_VAROK = get_charset_MIME_name(ret);

	DPRINT(Debug,10,(&Debug,"messages_log_ok=%p",ret));
	if (cs) {
	    DPRINT(Debug,10,(&Debug," %s",cs));
	}
	DPRINT(Debug,10,(&Debug,"\n"));
	       
    } else {
	DPRINT(Debug,10,(&Debug,"messages_log_ok=NULL\n"));

    }
	   
    return ret;
}

static void log_messages_logcs P_((const size_t len, const char * msg, int ok));
static void log_messages_logcs(len,msg,ok)
     const size_t len;
     const char * msg;
     int ok;
{
    if (messages_log_file && len > 0) {
	const int pid = getpid();

	if (messages_log_pid == pid) {

	    time_t now = 0;
	    size_t i;
	    int wasnl = 0;
	    struct tm * zz = NULL;
	    
	    if (((time_t) -1) != time(&now)) {
		zz = localtime(&now);
		
		if (!zz)
		    goto no_time14;

		fprintf(messages_log_file,
			"%d %02d:%02d:%02d %c ",
			pid,
			zz->tm_hour,
			zz->tm_min,
			zz->tm_sec,
			ok ? '>' : '?');

	    } else {
	    no_time14:
		
		fprintf(messages_log_file,
			"%d %c ",
			pid,
			ok ? '>' : '?');
	    }	    

	    for (i = 0; i < len; i++) {
		/* Print NL for both CR and NL */
		
		if ('\r' == msg[i]) { 
		    putc('\n',messages_log_file);
		    wasnl = 1;

		    if (i+1 < len && '\n' == msg[i+1])
			i++;
		    
		} else {
		    putc(msg[i],messages_log_file);

		    wasnl = ('\n' == msg[i]);
		}
			
		if (i+1 < len && wasnl) {
		    if (zz) {
			fprintf(messages_log_file,
				"%d %02d:%02d:%02d + ",
				pid,
				zz->tm_hour,
				zz->tm_min,
				zz->tm_sec);
		    } else {
			fprintf(messages_log_file,
				"%d + ",
				pid);

		    }
		}
	    }

	    if (!wasnl) {
		putc('\n',messages_log_file);
	    }
	    
	}
    }
}

static void log_messages_dispcs P_((const char * msg));
static void log_messages_dispcs(msg)
     const char * msg;
{
    if (messages_log_file) {
	const int pid = getpid();
	charset_t cs = NULL;
	
	if (messages_log_pid == pid &&
	    (cs = messages_log_ok())) {
	    int ok = 1;

	    size_t in_len = strlen(msg);
	    const char * msg0     = msg;
	    size_t       msg0_len = in_len;

	    char * buffer = NULL;
	    int    buffer_len = 0;
	    	    
	    if (cs != display_charset) {
		struct string * S_in = new_string(display_charset);       
		int in_ERRORS;
		int out_ERRORS;
		struct string * S_out = NULL;

		int rs_in = add_streambytes_to_string(S_in,
						      in_len,cs2us(msg),&in_ERRORS);

		if (rs_in < in_len) {
		    DPRINT(Debug,10,(&Debug,
				     "log_messages_dispcs: rs_in=%d < in_len=%zu\n",
				     rs_in,in_len));
		    ok = 0;
		}
		if (in_ERRORS) {
		    DPRINT(Debug,10,(&Debug,
				     "log_messages_dispcs: in_ERRORS=%d\n",
				     in_ERRORS));
		    ok = 0;
		}
		
		S_out = convert_string2(cs,S_in,&out_ERRORS);

		if (out_ERRORS) {
		    DPRINT(Debug,10,(&Debug,
				     "log_messages_dispcs: out_ERRORS=%d\n",
				     out_ERRORS));
		    ok = 0;
		}

		/* result is malloced */
		bytestream_from_string(S_out,&buffer,&buffer_len);

		msg0     = buffer;
		msg0_len = buffer_len;
		
		if (S_out)
		    free_string(&S_out);
		if (S_in)
		    free_string(&S_in);
	    }
	    
	    log_messages_logcs(msg0_len,msg0,ok);

	    if (buffer) {
		free(buffer);
		buffer = NULL;
	    }
	}
	
    }
}

static void log_messages_string P_((const struct string * smsg));
static void log_messages_string(smsg)
     const struct string *  smsg;
{
    if (messages_log_file) {
	const int pid = getpid();
	charset_t cs = NULL;
	
	if (messages_log_pid == pid &&
	    (cs = messages_log_ok())) {

	    int ok = 1;
	    const struct string *  smsg0 = smsg;
	    
	    char * buffer = NULL;
	    int    buffer_len = 0;

	    struct string * S_out = NULL;

	    if (cs != get_string_type(smsg)) {
		int out_ERRORS;

		S_out = convert_string2(cs,smsg,&out_ERRORS);

		if (out_ERRORS) {
		    DPRINT(Debug,10,(&Debug,
				     "log_messages_string: out_ERRORS=%d\n",
				     out_ERRORS));
		    ok = 0;
		}

		smsg0 = S_out;
	    }
		    
	    /* result is malloced */
	    bytestream_from_string(smsg0,&buffer,&buffer_len);

	    log_messages_logcs(buffer_len,buffer,ok);

	    if (buffer) {
		free(buffer);
		buffer = NULL;
	    }

	    if (S_out)
		free_string(&S_out);		
	}
    }
}

/* Returns 1 if error was shown (was on RawState()) */

int show_last_error()
{
    int ret = 0;
    
    /** rewrite last error message! **/
    if (RawState()) {

	if (err_buffer[0]) {
	    DPRINT(Debug,8,(&Debug,
			    "show_last_error: err_buffer=%s\n",
			    err_buffer));
	    error_middle(err_buffer);
	    ret = 1;
	}
	
	if (err_buffer_W) {
	    DPRINT(Debug,8,(&Debug,
			    "show_last_error: err_buffer_W=%S\n",
			    err_buffer_W));
	    error_middle_W(err_buffer_W);
	    ret = 1;
	}
	
    } else {
	DPRINT(Debug,8,(&Debug,
			"show_last_error: Not on 'Raw' state\n"));
    }

    DPRINT(Debug,8,(&Debug,
		    "show_last_error=%d\n",
		    ret));
    return ret;
}

void error_wait()
{
    time_t M = time(NULL);

    if (LAST_ERROR + sleepmsg > M && RawState()) {
	int W = LAST_ERROR + sleepmsg  - M;
	DPRINT(Debug,7,(&Debug, 
			"sleeping for %d seconds for error message\n",
			W));
	FlushBuffer();
	error_sleep(W);
    }
}

void clear_error()
{      
    if(RawState()) {
	int LINES, COLUMNS;
	
	menu_get_sizes(default_context,&LINES, &COLUMNS);
	
	if ('\0' != err_buffer[0] || err_buffer_W) 
	    error_wait();
	MoveCursor(LINES-1,0);
	CleartoEOLN();
	
	refresh_cancel_message();
    } else {
	if ('\0' != err_buffer[0] || err_buffer_W) 
	    fprintf(stderr,"\n");
    }

    err_buffer[0] = '\0';
    if (err_buffer_W)
	free_string(& err_buffer_W);
}

void Raw_OFF(exiting) 
     int exiting;
{
    if(RawState()) {
	int LINES, COLUMNS;

	menu_get_sizes(default_context,&LINES, &COLUMNS);

	if ('\0' != err_buffer[0] || err_buffer_W) 
	    error_wait();
	MoveCursor(LINES-1,0);
	CleartoEOLN();
    }
    Raw(OFF);
    NewLine();

    if (exiting)
	switch_title(2);

    if ('\0' != err_buffer[0]) 
	error_middle(err_buffer);

    if (err_buffer_W) 
	error_middle_W(err_buffer_W);

    FlushBuffer();
}

/* Used by newmbox.c */
int need_refresh_low_line() {
    if ('\0' != err_buffer[0] || err_buffer_W) {
	clear_error();
	return 1;
    }
    return 0;
}

void set_error(s)
     char *s;
{
    strfcpy(err_buffer, s, sizeof err_buffer);
    if (err_buffer_W)
	free_string(& err_buffer_W);

}

static void message_W P_((struct menu_context *page, const struct string *X,
			  int reserve));
static void message_W(page,X,reserve)
     struct menu_context *page;
     const struct string *X;
     int reserve;
{
    int LINES, COLUMNS;
    int pos = 0;
    int s = 0;
    int len = string_len(X);

    menu_get_sizes(page,&LINES, &COLUMNS);

    if (len < 1) {
	menu_MoveCursor(page,LINES-1,0);
	menu_CleartoEOLN(page);
    }
    
    while (1) {
	int space = COLUMNS-1-reserve;
	int vlen;
	struct string *Y;

	uint16 ch;
	int span;

	if (space < 5)
	    space = COLUMNS-1;

	/* Scan for \n */

	if (pos >= len)
	    break;

	ch = give_unicode_from_string(X,pos);

	if (0x000A /* LF */ == ch)
	    pos++;       /* Skip initial \n */

	
	for (span = 0; pos + span < len; span++) {
	    ch = give_unicode_from_string(X,pos+span);

	    if (0x000A /* LF */ == ch)
		break;	    
	}

	Y = curses_printable_clip(X,&pos,span,&vlen,space);

	if (!Y)
	    break;


	if (vlen > 0) {
	    int A = (COLUMNS-vlen)/2;

	    if (A + vlen > space) {
		A = space - vlen;
		if (A < 0)
		    A  = 0;
	    }

	    if (s)
		error_sleep(1);
	    s = 1;
	    
	    menu_MoveCursor(page,LINES-1,0);
	    menu_CleartoEOLN(page);
	    menu_PutLineX(page,LINES-1,A,FRM("%S"),Y);

	    FlushBuffer();
	    
	}
	free_string(&Y);
	
	if (!vlen) {
	    /* Not printable !! */
  
	    pos++;
	}
    }
}

static void error_middle_W P_((const struct string *s));
static void error_middle_W(s)
     const struct string *s;
{
    if(!RawState()) {

	struct string * R1 = convert_string(display_charset,s,1);
	char * str1 = us2s(stream_from_string(R1,0,NULL));

	int l = strlen(str1);
	int ret = fprintf(stderr,"%s",str1);

	if (l && str1[l-1] == '\r')
	    l--;

	if (ret > 0 && l && str1[l-1] != '\n')
	    ret += fprintf(stderr,"\n");

	free(str1);
	free_string(&R1);

	/* sleep if stderr points to terminal 
	 */
	if (ret > 0 && isatty(2)) {

	    if (POLL_method)
		wait_for_timeout(1);
	    else
		sleep(1);

	}

    } else {

	message_W(default_context,s,0);

    }
}


static void error_help P_((const char *s));
static void error_help (s)
     const char *s;
{

    static int   nested_call = 0;

    int LINES, COLUMNS;

    menu_get_sizes(default_context,&LINES, &COLUMNS);

    /* Was using PutLine0, which uses cur_Writechar.
       cur_Writechar assumes one byte charset
       therefore it does not work well with UTF-8
       error messages

       This does not center correctly UTF-8, but at least
       it is displayed ...
    */

    MoveCursor(LINES-1,0);
    CleartoEOLN();
    
    if (nested_call) {
	PutLine0(LINES-1,(COLUMNS-strlen(s))/2,s);

    } else {
	nested_call++;

	if (check_8bit_str(s))
	    PutLineX(LINES-1,(COLUMNS-strlen(s))/2,
		     FRM("%s"),s);
	else
	    PutLine0(LINES-1,(COLUMNS-strlen(s))/2,s);

	nested_call--;
    }

    /* PutLineX may trigger error message
           Mapfile /usr/local/lib/elm.map.bin/... is corrupted 
       to be printed, therefore do not use
       it if it is not necessary 
    */

    FlushBuffer();    
}

static void error_middle (s)
     const char *s;
{
    if(!RawState()) {
	fprintf(stderr,"%s\n", s);
	/* sleep if stderr points to terminal 
	 */
	if (isatty(2)) {

	    if (POLL_method)
		wait_for_timeout(1);
	    else
		sleep(1);

	}
    } else {
	int LINES, COLUMNS;

	menu_get_sizes(default_context,&LINES, &COLUMNS);
	
	if (strlen(s) < COLUMNS && NULL == strchr(s,'\n'))
	    error_help(s);
	else {
	    char * s1 = safe_strdup(s);
	    char *s2, *s3;
	    
	    for (s2=s1; *s2 != '\0'; s2 = s3) {
		char c;
		for (s3 = s2; *s3 != '\0' && s3 - s2 < COLUMNS && *s3 != '\n'; s3++);
		c = *s3;
		*s3 = '\0';
		error_help(s2);
		*s3 = c;
		if ('\n' == c)
		    s3++;
		if (*s3 != '\0') {   /* Wait between lines ...*/
		    error_sleep(1);
		}
	    }
	    free(s1);
	}
    }

}

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif


S_(err_handler_W error_W)
static int error_W P_((const struct string *s));
static int error_W(s)
     const struct string *s;
{
    if (err_buffer_W && 
	0 == string_cmp(err_buffer_W,s,99)) {
	DPRINT(Debug,7,(&Debug, 
			"-- duplicate error message\n"));
    } else {
	error_wait();
	LAST_ERROR = time(NULL);
    }
    
    DPRINT(Debug,1,(&Debug,  
		    "ERROR message: %S\n", 
		    s));

    log_messages_string(s);
    
    error_middle_W(s);

    if (err_buffer_W)
	free_string(& err_buffer_W);
    err_buffer_W = dup_string(s);
    err_buffer[0] = '\0';

    return string_len(s);
}



int error(s)
     const char *s;
{
    /** outputs error 's' to screen at line 22, centered! **/
  if (0 == strcmp(err_buffer,s)) {
      DPRINT(Debug,7,(&Debug, 
		      "-- duplicate error message\n"));
  } else {
      error_wait();
      LAST_ERROR = time(NULL);
  }
  
  DPRINT(Debug,1,(&Debug,  
		  "ERROR message: %s\n", 
		  s));

  log_messages_dispcs(s);
  
  error_middle(s);

  strfcpy(err_buffer, s, sizeof err_buffer);	/* save it too! */
  if (err_buffer_W)
      free_string(& err_buffer_W);

  return 0;
}

S_(err_handler_W transient_W)
static int transient_W P_((const struct string *s));
static int transient_W(s)
     const struct string *s;
{
    DPRINT(Debug,1,(&Debug,  
		    "TRANSIENT message: %S\n", 
		    s));
    LAST_ERROR = 0;  /* NO SLEEP  */

    log_messages_string(s);
    
    error_middle_W(s);

    if (err_buffer_W)
	free_string(& err_buffer_W);
    err_buffer_W = dup_string(s);
    err_buffer[0] = '\0';

    return string_len(s);
}

static char * prompt_it P_((const char *str, int pass));
static char * prompt_it (str,pass) 
     const char *str;
     int pass;
{
    int len = strlen(str);
    int status;
    char buffer[200];
    int oldstate = RawState();
    int LINES, COLUMNS;

    struct menu_context *cpage = Raw(ON);
	
    buffer[0] = '\0';

    log_messages_dispcs(str);
    
 redraw:
    menu_get_sizes(cpage,&LINES, &COLUMNS);

    PutLine0(LINES-3, 0,str);
    CleartoEOS();
    show_last_error();
    status = optionally_enter(buffer,LINES-1-2, len+1,
			      (pass ? OE_PASSWD : 0) |OE_REDRAW_MARK|
			      OE_SIG_CHAR /* Ctrl-C */,
			      sizeof buffer,
			      cpage);
    if (REDRAW_MARK == status) {
	menu_ClearScreen(cpage);   /* Clear possible redraw mark */
		
	/* Call refresh routines of children */
	menu_redraw_children(cpage);

	goto redraw;
    }
    MoveCursor(LINES-3,0);   /* Ack RETURN */
    CleartoEOS();

    Raw(oldstate);

    show_last_error();
    
    if (status != 0)
	return NULL;    
    return safe_strdup(buffer);
}

static void cancel_transient_X   P_((struct string *x, int reserve));
static void cancel_transient_X(x,reserve)
     struct string *x;
     int reserve;
{
    DPRINT(Debug,1,(&Debug,  
		    "CANCEL TRANSIENT message: %S\n", 
		    x));

    log_messages_string(x);
    
    message_W(default_context,x,reserve);
}

static void cancel_clear_X  P_((void));
static void cancel_clear_X()
{ 
    int LINES, COLUMNS;
	
    menu_get_sizes(default_context,&LINES, &COLUMNS);

    MoveCursor(LINES-1,0);
    CleartoEOLN();

    show_last_error();
}

static void cancel_progress_X P_((struct string *x, int *reserve,
				  struct string *progress));
static void cancel_progress_X(x,reserve,progress) 
     struct string *x; 
     int *reserve;
     struct string *progress;
{
    int LINES, COLUMNS;
    int pos = 0;
    struct string *Y;
    int l = string_len(progress);
    int vlen = 0;

    DPRINT(Debug,1,(&Debug,  
		    "CANCEL PROGRESS message: %S\n", 
		    x));
    DPRINT(Debug,1,(&Debug,
		    "       progress message: %S\n",
		    progress));

    menu_get_sizes(default_context,&LINES, &COLUMNS);

    Y = curses_printable_clip(progress,&pos,l,&vlen,COLUMNS-10);

    if (*reserve < vlen) {

	/* Reprint */

	*reserve = vlen;

	if (*reserve > COLUMNS-10) {
	    DPRINT(Debug,8,(&Debug,"reserve %d too big, limit %d\n",
			    *reserve,COLUMNS-10));
	    *reserve = COLUMNS-10;
	}

	message_W(default_context,x,*reserve);
    }

    DPRINT(Debug,8,(&Debug,"reserve %d vlen %d message: %S\n",
		    *reserve,vlen,Y));
    
    if (Y && *reserve > 0) {
	int pos1 = COLUMNS-*reserve-1;	    
	int pos2 = COLUMNS-vlen-1;

	menu_MoveCursor(default_context,LINES-1,pos1);
	menu_CleartoEOLN(default_context);
		
	if (pos2 >= pos1) 
	    menu_PutLineX(default_context,LINES-1,pos2,FRM("%S"),Y);
	else {
	    DPRINT(Debug,8,(&Debug,"Can't print progress message: pos2=%d (pos1=%d): %S\n",
			    pos2,pos1,Y));
	}
	
	FlushBuffer();
    }

    if (Y)
	free_string(&Y);

}

void out_util_setup() {
    set_error_handler_W(error_W);
    set_transient_handler_W(transient_W);

    set_prompt_handler(prompt_it);

    setup_cancel_cb(cancel_set_ttysig_X,
		    cancel_reset_ttysig_X,
		    cancel_transient_X,
		    cancel_clear_X,
		    cancel_progress_X);
}

void lower_prompt(s)
     char *s;
{
  /** prompt user for input on LINES-1 line, left justified **/

    log_messages_dispcs(s);
    
    if(RawState()) {
	int LINES, COLUMNS;
	
	menu_get_sizes(default_context,&LINES, &COLUMNS);
	
	PutLine0(LINES-1-1,0,s);
	CleartoEOLN();
    } else {
	fprintf(stderr,"%s\n",s);
    }
}

void prompt(s)
     char *s;
{
  /** prompt user for input on LINES-3 line, left justified **/

    log_messages_dispcs(s);
    
    if(RawState()) {
	int LINES, COLUMNS;
	
	menu_get_sizes(default_context,&LINES, &COLUMNS);
	
	PutLine0(LINES-4,0,s);
	CleartoEOLN();
    } else {
	fprintf(stderr,"%s\n",s);
    }
}

void prompt_s(page,S)
     struct menu_context  *page;
     const struct string *S;
{
  /** prompt user for input on LINES-3 line, left justified **/

    log_messages_string(S);
    
    if(RawState()) {
	int LINES, COLUMNS;

	menu_get_sizes(page,&LINES, &COLUMNS);

	menu_PutLineX(page,LINES-4,0,FRM("%S"),S);
	menu_CleartoEOLN(page);
    } else {
	lib_transient(FRM("%S\n"),S);
    }
}

void sleep_message() {
    if (sleepmsg > 0) {
	error_sleep(sleepmsg);
    }
}

void print_center(line,buffer)
     int line; 
     struct string * buffer;
{
    int l1 = string_len(buffer);
    int LINES, COLUMNS;

    menu_get_sizes(default_context,&LINES, &COLUMNS);

    if (l1 <  COLUMNS)
	PutLineX(line,(COLUMNS - l1)/2,FRM("%S"),buffer);        
}

void menu_print_center(ptr,line,buffer)
     struct menu_context  *ptr;
     int line; 
     const struct string * buffer;
{
    int l1 = string_len(buffer);
    int POS = 0;
    int visible_len = 0;
    struct string * temp = NULL;

    int LINES, COLUMNS;

    menu_get_sizes(ptr,&LINES, &COLUMNS);

    temp = curses_printable_clip(buffer,&POS,l1,
				 &visible_len,COLUMNS);
	
    menu_PutLineX(ptr,line,(COLUMNS - visible_len)/2,FRM("%S"),temp);        
    free_string(&temp);
}


void menu_print_format_center(
#if ANSI_C
       struct menu_context  *ptr,
       int lineY,const char *format, const char *line, ...
#else
       ptr,lineY,format,line,va_alist
#endif
       )
#if !ANSI_C
     struct menu_context  *ptr;
     int lineY;
     const char *format; 
     const char *line;
     va_dcl
#endif
{
    va_list vl;

    struct string *text;

    Va_start(vl, line);           /* defined in hdrs/elm_defs.h */       
    text = elm_smessage(0,format,line,vl);
    va_end(vl);

    menu_print_center(ptr,lineY,text);

    free_string(&text);
}

void print_format_center(
#if ANSI_C
       int lineY,const char *format, const char *line, ...
#else
       lineY,format,line,va_alist
#endif
)
#if !ANSI_C
       int lineY;
       const char *format; 
       const char *line;
       va_dcl
#endif
{
    va_list vl;

    struct string *text;

    Va_start(vl, line);           /* defined in hdrs/elm_defs.h */       
    text = elm_smessage(0,format,line,vl);
    va_end(vl);

    print_center(lineY,text);

    free_string(&text);
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
