static char rcsid[] = "@(#)$Id: reply.c,v 2.15 2018/07/29 17:54:21 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.15 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>  
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *  Based on Elm 2.4 src/reply.c. That code was following copyright:
 *
 *  The Elm Mail System 
 *
 *			Copyright (c) 1988-1992 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

/*** routine allows replying to the sender of the current message 

***/

#include "def_elm.h"
#include "s_elm.h"

DEBUG_VAR(Debug,__FILE__,"mail");

static char *us2s P_((unsigned char *str));
static char *us2s(str) 
     unsigned char *str;
{
    return (char *)str;
}


/** Note that this routine generates automatic header information
    for the subject and (obviously) to lines, but that these can
    be altered while in the editor composing the reply message! 
**/


#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif       


/* Prototype */
static void get_reply_subj P_((char *, char *, char *, int));

/* Determine the subject to use for a reply.  */
static void get_reply_subj(out_subj,in_subj,dflt_subj, size)
     char *out_subj;	 /* store the resulting subject here		*/
     char *in_subj;	/* subject of the original message		*/
     char *dflt_subj;	/* default to use if "in_subj" is empty		*/
     int size;
{
	if ( *in_subj == '\0' ) {
	  strfcpy(out_subj,dflt_subj, size);
	  return;
	}
	if (
	  ( in_subj[0] == 'r' || in_subj[0] == 'R' ) &&
	  ( in_subj[1] == 'e' || in_subj[1] == 'E' ) &&
	  ( in_subj[2] == ':' )
	) {
	  for ( in_subj += 3 ; whitespace(*in_subj) ; ++in_subj ) ;
	}
	strfcat( strfcpy( out_subj, "Re: ", size ), in_subj, size);
}

/* Retuns1 if duplicate */
static int optimize_and_add P_((const struct address * new_address, 
				struct addr_list  * full_address,
				const struct string *  new_group));

static int optimize_and_add(new_address1, full_address, new_group)
     const struct address * new_address1;
     struct addr_list     * full_address;
     const struct string  * new_group;
{
    const char **mailname_list = give_dt_path_as_elems(&mailname,"mailname");

    const char          * new_addr     = address_get_ascii_addr(new_address1);
    const struct string * new_fullname = address_get_phrase(new_address1);
    const struct string * new_comment  = address_get_comment(new_address1);

    int idx, addr_item_count  = addr_list_item_count(full_address);
    int      addr_group_count = addr_list_group_count(full_address);
    int new_group_idx = -1;

    struct split_addr  splitted_new_addr;
    int is_splitted;
    int new_addr_on_mailname = 0;
    int ret = 0;

    if (!new_addr)
	return 1;   /* ? error */

    is_splitted = split_internet_address(&splitted_new_addr,new_addr);

    if (is_splitted) {

	DPRINT(Debug,15,(&Debug, 
			 "optimize_and_add: new addr=%s => user part=%s%s, host part=%s\n",
			 new_addr,
			 splitted_new_addr.local_part,
			 splitted_new_addr.local_part_dequote > 0 ? 
			 " (dequoted)" : "",
			 splitted_new_addr.host_part));

	if (mailname_list) {
	    int x;

	    for (x = 0; mailname_list[x]; x++) {
		if (0 == istrcmp(splitted_new_addr.host_part,mailname_list[x])) {
		    new_addr_on_mailname = 1;

		    DPRINT(Debug,15,(&Debug, 
				     "  ... host part matches to mailname[%d]=%s\n",
				     x,mailname_list[x]));
		    
		    break;
		}		
	    }
	}
    }
		   
    /* This routine will add the new address to the list of addresses
     * in the full address buffer IFF it doesn't already occur.  
     *
     */
    
    if (new_group) {
	for (idx = 0; idx < addr_group_count; idx++) {
	    const struct string * group = addr_list_get_group(full_address,idx);
	    
	    if (0 == string_cmp(group,new_group,-99))
		new_group_idx = idx;	
	}

	if (-1 == new_group_idx) {
	    new_group_idx = add_group_to_list(full_address,new_group);
	    DPRINT(Debug,15,(&Debug, "optimize_and_add: new group = %S\n",new_group));
	}
    }

    for (idx = 0; idx < addr_item_count; idx++) {
	int group = -1;
	const struct address * address = 
	    addr_list_get_item(full_address,idx,&group);
	
	const char          * addr     = address_get_ascii_addr(address);
	const struct string * fullname = address_get_phrase(address);
	const struct string * comment  = address_get_comment(address);
		
	if (!addr)
	    continue;
	
	if (group == new_group_idx) {

	    struct split_addr  splitted1;

	    if (0 == strcmp(new_addr,addr)) {
		
		/* same address */
		
	    SAME_ADDR:
		if (!fullname || !string_len(fullname)) {
		    struct address * address1 = new_address(addr,new_fullname,comment);
		    
		    addr_list_replace_item(full_address,idx,address1,group);   
		    
		    free_address(&address1);
		    
		    /* Replace invalidates pointers, so reload new pointers */
		    addr     = address_get_ascii_addr(address);
		    fullname = address_get_phrase(address);
		    comment  = address_get_comment(address);
		}
		
		DPRINT(Debug,15,(&Debug, "optimize_and_add = 1:\n"));
		if (fullname) {
		    DPRINT(Debug,15,(&Debug, "        .fullname=%S\n",fullname));
		}
		DPRINT(Debug,15,(&Debug, "        .addr=%s\n",addr));

		ret = 1;	/* duplicate address */
		goto clear;
	    }

	    if (is_splitted && 
		split_internet_address(&splitted1,addr)) {

		DPRINT(Debug,15,(&Debug, 
				 "optimize_and_add: addr=%s => user part=%s%s, host part=%s\n",
				 addr,
				 splitted1.local_part,
				 splitted1.local_part_dequote > 0 ? 
				 " (dequoted)" : "",
				 splitted1.host_part));

		if (0 == strcmp(splitted_new_addr.local_part,
				splitted1.local_part)) {
		    
		    if (0 == istrcmp(splitted_new_addr.host_part,
				     splitted1.host_part)) {

			clear_split_addr(&splitted1);
			goto SAME_ADDR;
		    }

		    if (new_addr_on_mailname) {
			int on_mailname = 0;

			int x;
			
			for (x = 0; mailname_list[x]; x++) {
			    if (0 == istrcmp(splitted1.host_part,mailname_list[x])) {
				on_mailname = 1;
				
				DPRINT(Debug,15,(&Debug, 
						 "  ... host part matches to mailname[%d]=%s\n",
						 x,mailname_list[x]));
				
				break;
			    }		
			}

			if (on_mailname) {
			    
			    if (!fullname || !string_len(fullname) ||
				/* Normalize also host part */				
				0 != strcmp(splitted1.host_part,mailname_list[0])) {

				const struct string * F  = 
				    fullname && string_len(fullname) ? 
				    fullname : 
				    new_fullname;
				char * A1 = NULL;
				const char * have_mailname = useful_mailname();
				struct address * address1 = NULL;

				if (have_mailname) {
				    A1 = elm_message(FRM("%s@%s"),
						     splitted1.local_part_raw,
						     have_mailname);

				    if (0 != strcmp(addr,A1)) {
					DPRINT(Debug,15,(&Debug, " ... normalizes addr=%s => %s\n",
							 addr,A1));
				    }
				    
				} else {
				    A1 = safe_strdup(addr);
				}
								
				address1 = new_address(A1,F,comment);
				
				addr_list_replace_item(full_address,idx,address1,group);   
				
				free_address(&address1);
				free(A1);
				
				/* Replace invalidates pointers, so reload new pointers */
				addr     = address_get_ascii_addr(address);
				fullname = address_get_phrase(address);
				comment  = address_get_comment(address);
			    }
			    
			    DPRINT(Debug,15,(&Debug, "optimize_and_add = 1:\n"));
			    if (fullname) {
				DPRINT(Debug,15,(&Debug, "        .fullname=%S\n",fullname));
			    }
			    DPRINT(Debug,15,(&Debug, "        .addr=%s\n",addr));

			    ret = 1;	/* duplicate address */
			    goto clear;
			}			
		    }
		}

		clear_split_addr(&splitted1);
	    }
	}
    }

    add_address_to_list(full_address,new_address1,new_group_idx);
   
    DPRINT(Debug,15,(&Debug, "optimize_and_add = 0:\n"));
    if (new_fullname) {
	DPRINT(Debug,15,(&Debug, "        .fullname=%S\n",new_fullname));
    }
    DPRINT(Debug,15,(&Debug, "        .addr=%s\n",new_addr));
    if (new_comment) {
	DPRINT(Debug,15,(&Debug, "        .comment=%S\n",new_comment));
    }
    ret = 0;

 clear:
    if (is_splitted)
	clear_split_addr(&splitted_new_addr);


    return(ret);
}

void append_addresses_to_addr_list (target,list)
     struct addr_list *target;
     const struct addr_list *list;
{
    int idx, addr_item_count  = addr_list_item_count(list);

    for (idx = 0; idx < addr_item_count; idx++) {
	int group = -1;

	const struct address * address = 
	    addr_list_get_item(list,idx,&group);

	const struct string * groupname = NULL;

	if (group >= 0)
	    groupname =  addr_list_get_group(list,group);

	optimize_and_add(address, target, groupname);
    }
}

struct addr_list * handle_reply_to(current_header, infile)
     struct header_rec * current_header;
     FILE *infile;
{
    struct addr_list  * result = new_empty_addr_list(1);

    int count=0;

    if (current_header->reply_to) {
	int idx, addr_item_count = addr_list_item_count(current_header->reply_to);
	
	for (idx = 0; idx < addr_item_count; idx++) {
	    int group = -1;
	    const struct address * address = 
		addr_list_get_item(current_header->reply_to,idx,&group);
	    
	    const char          * addr  UNUSED_VAROK = address_get_ascii_addr(address);
	    
	    const struct string * groupname = NULL;
	    
	    if (group >= 0)
		groupname =  addr_list_get_group(current_header->reply_to,group);
	    
	    /* do NOT add sender's domain in here ! It was MTA's task
	       and if it is not added then there is good reason.
	       (adding of senders domain to unqualified addresses
	       BREAKS my local mailing lists!!!)
	       That is: It is possible that Reply-to is written by some
	       other site than From !!!
	    */
	    
	    if (!optimize_and_add(address, result,groupname)) {
		count++;
		DPRINT(Debug,3,(&Debug, 
				"handle_reply_to: (Reply-To) #%d: addr=%s\n",
				count,addr ? addr : "<NULL>")); 
	    }
	}	
    }
    
    /* if there wasn't reply-to header */
    if (!count) {
	
	if (current_header->from) {
	    int idx;
	    int len   = addr_list_item_count(current_header->from);

	    for (idx = 0; idx < len; idx++) {
		int group = -1;
		const struct address * address = 
		    addr_list_get_item(current_header->from,idx,&group);
		
		const char          * addr UNUSED_VAROK = 
		    address_get_ascii_addr(address);
		
		const struct string * groupname = NULL;

		if (group >= 0)
		    groupname =  addr_list_get_group(current_header->from,group);
		
		if (!optimize_and_add(address, result,groupname)) {
		    count++;
		    DPRINT(Debug,3,(&Debug, 
				    "handle_reply_to: (From) %d: addr=%s\n",
				    count,addr ? addr : "<NULL>")); 
		}
	    }
	}
    }	
    
    if (!count) {
	struct string *xx = new_string(display_charset);

	struct address * address1 = new_address(current_header->env_from,
						xx,xx);

	add_address_to_list(result,address1,-1);
	
	DPRINT(Debug,3,(&Debug, 
			"handle_reply_to: ENV: addr=%s\n",
			current_header->env_from)); 

	free_address(&address1);
	free_string(&xx);
    }
    
    return result;
}

struct addr_list * get_and_expand_everyone(current_header,
					   return_address,
					   usraddr)
     struct header_rec * current_header;
     struct addr_list * return_address;
     const struct address   * usraddr;   /* from address from hashmark */
{

    enum { h_to = 0, h_cc = 1, h_done = 2 } count;

    struct addr_list  * result = new_empty_addr_list(1);

    char * env_host = qstrpbrk(current_header->env_from,":");

    /* skip route */
    if (env_host)
	env_host = qstrpbrk(env_host,"@");
    else
	env_host = qstrpbrk(current_header->env_from,"@");


    /** As each address is taken, ensure that it
	isn't to the author of the message NOR to us.  If neither,
	prepend with current return address and append to result
    **/
    
    for (count = h_to ; count < h_done; count++) {
	struct addr_list * address_list = NULL;
    
	switch (count) {
	case h_to: address_list = current_header->to; break;
	case h_cc: address_list = current_header->cc; break;
	default:                               break;
	}
    
	if (address_list) {

	    int idx, addr_item_count = addr_list_item_count(address_list);

	    /* go through all addresses in this line */	    
	    for (idx = 0; idx < addr_item_count; idx++) {
		int group = -1;
		const struct address * address = 
		    addr_list_get_item(address_list,idx,&group);


		const char          * addr     = address_get_ascii_addr(address);

		if (addr && 0 == strcmp(addr,current_header->env_from)) 
		    continue;

		if (okay_address_l(address, return_address,
				   usraddr)) {

		    const struct string * groupname = NULL;
		
		    if (group >= 0)
			groupname =  addr_list_get_group(address_list,group);

		    /**
		       Some mailers can emit unqualified addresses in the
		       headers, e.g. a Cc to a local user might appear as
		       just "user" and not "user@dom.ain".  We do a real
		       low-rent check here.  If it looks like a domain
		       address then we will pass it through.  Otherwise we
		       tag domain from envelope sender address.
		    **/
	
		    if (addr && qchloc(addr, '@') < 0 && env_host) {

			const struct string * fullname = address_get_phrase(address);
			const struct string * comment  = address_get_comment(address);

			char * addr1 = safe_strdup(addr);
			struct address * address1 = NULL;

			addr1 = strmcat(addr1,env_host);

			address1 = new_address(addr1,fullname,comment);
			
			if (!optimize_and_add(address1,result,groupname)) {
			    DPRINT(Debug,3,(&Debug, 
					    "get_and_expand_everyone: got: addr=%s\n",addr1));
			}

			free_address(&address1);
			free(addr1); addr1 = NULL;
			
		    } else {
			
			if (!optimize_and_add(address,result,groupname)) {
			    DPRINT(Debug,3,(&Debug, 
					    "get_and_expand_everyone: got: addr=%s\n",
					    addr ? addr : "<NONE>"));
			}
		    }		    
		}
	    }
	}
    }

    return result;
}

void reply(index, mailbox, aview, page, prompt_area, header_area)
     int index;
     struct MailboxView *mailbox;
     struct AliasView *aview;
     struct menu_context  *page;
     struct menu_context  *prompt_area;
     struct menu_context  *header_area;
{
    /** Reply to the current message.  

        If screen nedd to redraw, use menu_trigger_redraw(page)
    **/
    
    char subject[SLEN];
    struct addr_list * rt = NULL;
    struct string * temp1 = NULL;
    char * temp2 = NULL;
    FILE *F = NULL;
    struct header_rec *current_header = NULL;    
    int r UNUSED_VAROK;

    if (!give_message_data(mailbox,index,
			   &current_header,&F,NULL,
			   NO_mime_parse))
	return;

    if (! current_header || !F)	
	return;

    rt = handle_reply_to(current_header,F);


    if (current_header->subject)
	temp1 = convert_string(display_charset,current_header->subject,1);
    else
	temp1 = new_string(display_charset);
    temp2 = us2s(stream_from_string(temp1,1,NULL));
    
    /* TODO: Fix this mess... */
    get_reply_subj( subject, temp2,		    
		    catgets(elm_msg_cat, ElmSet, ElmReYourMail, 
			    "Re: your mail"),
		    sizeof subject);
    
    r = send_msg_l(index,rt, NULL, subject,
		   MAIL_EDIT_MSG | MAIL_REPLYING,
		   mailbox, aview,
		   page, prompt_area,
		   current_header);

    DPRINT(Debug,4,(&Debug,"send_msg_l returned %d\n",r));
    
    if (rt)
	free_addr_list(&rt);	  
    
    free_string(&temp1);
    free(temp2); 
    
    /* XXX   Is this needed? */
    if (header_area) {
	int vis;
	struct menu_common MENU;

	set_mcommon_from_mbxview(&MENU,mailbox);
	
	vis = compute_visible(index+1, &MENU);
	menu_header_status_update(header_area,vis-1);
    }

    return;
}

void reply_to_everyone(index,mailbox, aview, page, prompt_area, header_area)
     int index;
     struct MailboxView *mailbox /* for save_copy */;
     struct AliasView *aview;
     struct menu_context *page;
     struct menu_context *prompt_area;
     struct menu_context *header_area;
{
    /** Reply to everyone who received the current message.  
	This includes other people in the 'To:' line and people
	in the 'Cc:' line too.  Returns non-zero iff the screen 
	has to be rewritten. **/
    
    char subject[SLEN];
    struct addr_list * rt;
    struct addr_list * cc;
  
    struct string * temp1 = NULL;
    char * temp2 = NULL;
    
    FILE *F = NULL;
    struct header_rec *current_header = NULL;    
    int r UNUSED_VAROK;

    const struct remote_server  * remote_server = NULL;
    struct address              * usraddr = NULL;

    if (mailbox)
	remote_server = give_message_remote_server(mailbox,index);

    if (remote_server)
	usraddr = give_remote_server_useraddr(remote_server,
					      NULL);

    if (!give_message_data(mailbox,index,
			   &current_header,&F,NULL,
			   NO_mime_parse))
	goto cleanup;

    if (! current_header || !F)	
	goto cleanup;

    rt = handle_reply_to(current_header,F);
    cc = get_and_expand_everyone(current_header,rt,
				 usraddr);

    if (current_header->subject)
	temp1 = convert_string(display_charset,current_header->subject,1);
    else
	temp1 = new_string(display_charset);
    temp2 = us2s(stream_from_string(temp1,1,NULL));
    
    /* TODO: Fix this mess ... */
    get_reply_subj( subject, temp2,
		    catgets(elm_msg_cat, ElmSet, ElmReYourMail, 
			    "Re: your mail"),
		    sizeof subject);
  
    r = send_msg_l(index,rt, cc, subject,
		   MAIL_EDIT_MSG | MAIL_REPLYING,
		   mailbox, aview,
		   page, prompt_area,NULL);

    DPRINT(Debug,4,(&Debug,"send_msg_l returned %d\n",r));

    if (rt)
	free_addr_list(&rt);	  
    if (cc)
	free_addr_list(&cc);	

    free_string(&temp1);
    free(temp2); 

    /* XXX   Is this needed? */
    if (header_area) {
	int vis;
	struct menu_common MENU;

	set_mcommon_from_mbxview(&MENU,mailbox);
	
	vis = compute_visible(index+1, &MENU);
	menu_header_status_update(header_area,vis-1);
    }


 cleanup:
    if (usraddr)
	free_address(& usraddr);
	
    return;
}

void forward(index,mailbox,aview, page, prompt_area,header_area)
     int index;
     struct MailboxView *mailbox;
     struct AliasView *aview;
     struct menu_context  *page;
     struct menu_context  *prompt_area;
     struct menu_context  *header_area;
{
    /** Forward the current message.  What this actually does is
	to temporarily set forwarding to true, then call 'send' to
	get the address and route the mail.   Modified to also set
	'noheader' to FALSE also, so that the original headers
	of the message sent are included in the message body also.

	if the main part of the screen has been changed
	  use menu_trigger_redraw()
    **/
    
    char subject[SLEN];
    int  edit_msg = FALSE;
    int li, co;
    struct header_rec *current_header =  give_header(mailbox,index);
    int r UNUSED_VAROK;

    menu_get_sizes(prompt_area, &li, &co);   

    if (! current_header) 
	return;

    {
	/* NOTICE: prompt_letter may return EOF */
	
	int X1;
	int delay_redraw = 0;

    again:
	X1 = prompt_letter(0,"",*def_ans_yes,
			   PROMPT_yesno|PROMPT_redraw_mark|PROMPT_ctrlL|
			   PROMPT_cancel,prompt_area,
			   
			   CATGETS(elm_msg_cat, ElmSet, ElmEditOutgoingMessage,
				   "Edit outgoing message? (%c/%c) "), 
			   *def_ans_yes, *def_ans_no);

	if (('L'&31)    == X1 ||
	    REDRAW_MARK == X1) {
	    menu_ClearScreen(page);   /* Reset possible redraw flag */
	    
	    /* Call refresh routines of children */
	     menu_redraw_children(page);

	     if (menu_need_redraw(prompt_area))		    
		 menu_ClearScreen(prompt_area);  /* clear redraw flag */

	    delay_redraw++;   /* Can't trigger redraw yet... */
	    goto again;
	}

	if (delay_redraw)
	    menu_trigger_redraw(page);

	if (TERMCH_interrupt_char == X1) 
	    goto out;
	
	if (EOF == X1)
	    goto out;

	edit_msg = (X1 != *def_ans_no);
    }

    if (current_header-> subject &&
	string_len(current_header->subject) > 0) {
	
	struct string * temp1 = NULL;
	char * temp2 = NULL;
	
	
	temp1 = convert_string(display_charset,
			       current_header->subject,1);

	temp2 = us2s(stream_from_string(temp1,1,NULL));
	
	
	strfcpy(subject, temp2, sizeof subject); 
	
	/* this next strange compare is to see if the last few chars are
	   already '(fwd)' before we tack another on */
	
	if (strlen(subject) < 6 || (strcmp(subject+strlen(subject)-5,
					   "(fwd)") != 0))
	    strfcat(subject, " (fwd)", sizeof subject);
	
	r = send_msg_l(index,
		       NULL, NULL, subject, 
		       (edit_msg ? MAIL_EDIT_MSG : 0 ) | 
		       MAIL_FORWARDING,
		       mailbox, aview,
		       page, prompt_area,
		       current_header);
	
	free_string(&temp1);
	free(temp2); 
	
    } else
	r = send_msg_l(index,
		       NULL, NULL,
		       /* TODO: Fix this mess... */
		       catgets(elm_msg_cat, ElmSet, ElmForwardedMail, 
			       "Forwarded mail..."), 
		       (edit_msg ? MAIL_EDIT_MSG : 0 ) | 
		       MAIL_FORWARDING,
		       mailbox, aview,
		       page, prompt_area,
		       current_header);

    DPRINT(Debug,4,(&Debug,"send_msg_l returned %d\n",r));
	
 out:
    
    menu_trigger_redraw(prompt_area);		
   
    if (header_area) {
	int vis;
	struct menu_common MENU;

	set_mcommon_from_mbxview(&MENU,mailbox);

	vis = compute_visible(index+1, &MENU);
	menu_header_status_update(header_area,vis-1);
    }
     
    return;
}

void get_return_name(address, name, trans_to_lowercase, size)
     const char *address;
     char *name;
     int   trans_to_lowercase;
     int size;
{
	/** Given the address (either a single address or a combined list 
	    of addresses) extract the login name of the first person on
	    the list and return it as 'name'.  Modified to stop at
	    any non-alphanumeric character. **/

	/** An important note to remember is that it isn't vital that this
	    always returns just the login name, but rather that it always
	    returns the SAME name.  If the persons' login happens to be,
	    for example, joe.richards, then it's arguable if the name 
	    should be joe, or the full login.  It's really immaterial, as
	    indicated before, so long as we ALWAYS return the same name! **/

	/** Another note: modified to return the argument as all lowercase
	    always, unless trans_to_lowercase is FALSE... **/

	/**
	 *  Yet another note: Modified to return a reasonable name
	 *  even when double quoted addresses and DecNet addresses
	 *  are embedded in a domain style address.
	 **/

	char * single_address = safe_strdup(address);
	register int	i, loc, iindex = 0,
			end, first = 0;
	register char	*c;

       	DPRINT(Debug,6,(&Debug, 
			"get_return_name called with (%s, <>, shift=%s, size=%d)\n",
			address, onoff(trans_to_lowercase), size));

	/* FIXME -- this is mostly bogus */

	/* Now is it an Internet address?? */

	if ((loc = qchloc(single_address, '@')) != -1) {	  /* Yes */

	    /*
	     *	Is it a double quoted address?
	     */

	    if (single_address[0] == '"') {
		first = 1;
		/*
		 *  Notice `end' will really be the index of
		 *  the last char in a double quoted address.
		 */
		loc = ((end = chloc (&single_address[1], '"')) == -1)
		    ? loc
		    : end;
	    }
	    else {
		first = 0;
	    }

	    /*
	     *	Hope it is not one of those weird X.400
	     *	addresses formatted like
	     *	/G=Jukka/S=Ukkonen/O=CSC/@fumail.fi
	     */

	    if (single_address[first] == '/') {
		/* OK then, let's assume it is one of them. */

		iindex = 0;
		
		if ((c = strstr (&single_address[first], "/G"))
		    || (c = strstr (&single_address[first], "/g"))) {

		    for (c += 2; *c && (*c++ != '='); );
		    for ( ;*c && (*c != '/') && iindex < size-1; c++) {
		      name[iindex++] = trans_to_lowercase
#ifdef ASCII_CTYPE
			&& isascii(*c)
#endif
			? tolower ((unsigned char)*c) : *c;
		    }
		    if (iindex > 0 && iindex < size-1) {
			name[iindex++] = '.';
		    }
		}
		if ((c = strstr (&single_address[first], "/S"))
		    || (c = strstr (&single_address[first], "/s"))) {

		    for (c += 2; *c && (*c++ != '='); );
		    for ( ;*c && (*c != '/') && iindex < size-1; c++) {
			name[iindex++] = trans_to_lowercase
#ifdef ASCII_CTYPE
			  && isascii(*c)
#endif
			  ? tolower ((unsigned char)*c) : *c;
		    }
		}
		name[iindex] = '\0';

		for (c = name; *c; c++) {
		    *c = ((*c == '.') || (*c == '-') || isalnum (*c))
			? *c : '_';
		}

		if (iindex == 0) {
		    strfcpy (name, "X.400.John.Doe", size);
		}
		
		goto done;
	    }

	    /*
	     *	Is it an embedded DecNet address?
	     */

	    while (NULL != (c = 
			    strstr (&single_address[first], "::"))) {
		first = c - single_address + 2;
	    }
		    

	    /*
	     *	At this point the algorithm is to keep shifting our
	     *	copy window left until we hit a '!'.  The login name
	     *	is then located between the '!' and the first meta-
	     *	character to it's right (ie '%', ':', '/' or '@').
	     */

	    for (i=loc; i > first-1 && single_address[i] != '!'; i--)
		if (single_address[i] == '%' || 
		    single_address[i] == ':' ||
		    single_address[i] == '/' ||
		    single_address[i] == '@') loc = i-1;
	
	    if (i < first || single_address[i] == '!') i++;

	    for (iindex = 0; iindex < loc - i + 1 && iindex < size-1; iindex++)
		if (trans_to_lowercase
#ifdef ASCII_CTYPE
		    && isascii(single_address[iindex+i])
#endif		    
		    )
		  name[iindex] = tolower((unsigned char)
					   single_address[iindex+i]);
		else
		    name[iindex] = single_address[iindex+i];
	    name[iindex] = '\0';

	}
	else {	/* easier - standard USENET address */

	    /*
	     *	This really is easier - we just cruise left from
	     *	the end of the string until we hit either a '!'
	     *	or the beginning of the line.  No sweat.
	     */

	    loc = strlen(single_address)-1; 	/* last char */

	    for (i = loc; 
		 i > -1 && single_address[i] != '!'
		 && single_address[i] != '.' && iindex < size-1; 
		 i--) {
		if (trans_to_lowercase
#ifdef ASCII_CTYPE
		    && isascii(single_address[iindex+i])
#endif		    
		    )
		  name[iindex++] = tolower((unsigned char)single_address[i]);
		else
		  name[iindex++] = single_address[i];
	    }
	    name[iindex] = '\0';
	    reverse(name);
	}


 done:

	for (iindex = 0; name[iindex]; iindex++) {
	    if (whitespace(name[iindex]))
		name[iindex] = '_';
	}

	free(single_address);
	return;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
