static char rcsid[] = "@(#)$Id: curs_input.c,v 2.19 2022/07/14 14:16:01 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.19 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI, 
 *                            hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************
 *
 * Some code copied from Elm 2.4 src/curses.c. It have following copyright:
 *
 * 			Copyright (c) 1988-1992 USENET Community Trust
 * 			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************
 * cur_FlushInput() partially based on Elm 2.5 src/curses.c.
 * It have following copyright:
 *
 *  The Elm Mail System  
 *
 *                      Copyright (c) 1988-1995 USENET Community Trust
 * 			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

#include "def_screen.h"

#ifdef TERMIOS
# include <termios.h>
#endif

DEBUG_VAR(Debug,__FILE__,"screen");

#include    <errno.h>
#ifndef ANSI_C
extern int errno;
#endif

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

static unsigned char *read_buffer   = NULL;
static int read_buffer_len = 0;
static int last_read_error = 0;

S_(action_routine background_read)
static enum action_status background_read P_((int                               fd,
					      union action_routine_data         data,
					      const struct schedule_timelimit * now));
static enum action_status background_read(fd,data,now)
     int                               fd; 
     union action_routine_data         data;
     const struct schedule_timelimit * now;
{
    int    result;
    unsigned char   ch;
    
    result = read(fd, &ch, 1);

    if (result < 0) {
	int err = errno;

	SIGDPRINT(Debug,4,(&Debug, 
			   "background_read: errno = %d [%s]\n",err,
			   strerror(err)));

	if((err == EINTR)
#ifdef	EAGAIN
	   || (err == EAGAIN)
#endif
#ifdef	EWOULDBLOCK
	   || (err == EWOULDBLOCK)
#endif
	   ) {
	    return 1;  /* Retry */
	}
	
	last_read_error = err;
	return action_disable;
    }
    
    if (0 == result) {
	SIGDPRINT(Debug,4,(&Debug,
			   "background_read: Got zero bytes...\n"));
	last_read_error = -1;
	return action_disable;
    }

    read_buffer = safe_realloc(read_buffer,read_buffer_len+1);
    SIGDPRINT(Debug,10,(&Debug,
	       "background_read(fd=%d)=1 [%d]",fd,read_buffer_len));
    SIGDPRINT(Debug,50,(&Debug,"=%d",ch));
    SIGDPRINT(Debug,10,(&Debug,"\n"));
    
    read_buffer[read_buffer_len++] = ch; 
    last_read_error = 0;

    return action_continue;
}


void clear_input_buffer() 
{
    SIGDPRINT(Debug,4,(&Debug, 
		       "Resetting background_read buffer (%d bytes)...\n",
		       read_buffer_len));
    read_buffer_len = 0;
    last_read_error = 0;
}

void error_sleep(seconds) 
     int seconds;
{

    if (POLL_method) {
	enum wait_for_status r1;
	
	if (read_buffer_len > 0) {
	    SIGDPRINT(Debug,5,(&Debug, 
			       "error_sleep(%d)  -- sleep skipped\n",
			       seconds));
	    return;
	}
	FlushBuffer();
	
	SIGDPRINT(Debug,5,(&Debug, 
			   "error_sleep(%d) ...\n",seconds));
	
	r1 = wait_for_action_or_timeout(background_read,seconds);

	switch (r1) {
	    int err UNUSED_VAROK;

	case wait_for_done:
	    DPRINT(Debug,5,(&Debug, 
			    "error_sleep: done\n"));
	    break;
	case wait_for_none:
	    DPRINT(Debug,5,(&Debug, 
			    "error_sleep: unexpected result from wait_for_action_or_timeout\n"));
	    break;
	case wait_for_error:
	    err = errno;

	    DPRINT(Debug,5,(&Debug, 
			    "error_sleep: errno=%d, %s\n",
			    err,strerror(err)));
	}
	
    } else {
	int r;
	
	FlushBuffer();

	r = sleep(seconds);

	if (r < 0) {
	    int err UNUSED_VAROK = errno;

	    DPRINT(Debug,5,(&Debug, 
			    "error_sleep: errno=%d, %s\n",
			    err,strerror(err)));
	    
	} else {
	    DPRINT(Debug,5,(&Debug, 
			    "error_sleep: %d seconds left\n",r));
	}
    }
    
    return;
}

void cur_FlushInput() {

    if (POLL_method) 
	clear_input_buffer();

#ifdef TERMIOS
    SIGDPRINT(Debug,4,(&Debug, 
		       "Clearing terminal input from fd %d\n",
		       terminal_fd));
    tcflush(terminal_fd,TCIFLUSH);
#endif
}


#ifdef TERMIOS
#include <termios.h>
#endif

struct charset_state * last_state = NULL;

void set_last_state(cs)
     charset_t cs;
{
    if (last_state && get_state_charset(last_state) != cs) {
	const char * MIME_name UNUSED_VAROK = 
	    get_charset_MIME_name(get_state_charset(last_state));

	DPRINT(Debug,4,(&Debug, 
			"Keyboard input charset is changed -- was %s\n",
			MIME_name ? MIME_name :	"<no MIME name>"));
	free_state(&last_state);
    }

    if (!last_state) {
	const char * MIME_name UNUSED_VAROK;

	last_state = new_state(cs);
	MIME_name = get_charset_MIME_name(get_state_charset(last_state));

	DPRINT(Debug,4,(&Debug,
		       "Keyboard input charset is %s\n",
			MIME_name ? MIME_name : "<no MIME name>"));
    }

}

static int key_resize_or_redraw P_((int flags));
static int key_resize_or_redraw(flags)
    int flags;
{
    int redraw      = (flags & READCH_MASK);
    int resize_flag = (flags & READCH_resize)    != 0; 

    int key = 0;

    check_changes();

    if (resize_flag && default_context->changed) {
	DPRINT(Debug,4,(&Debug,
			"key_resize_or_redraw: Pending resize on %p (default context) ...\n",
			default_context));

	key = RESIZE_MARK;
	goto done_key;
    }
    

    if (redraw && default_context->redraw) {
	DPRINT(Debug,4,(&Debug,
			"key_resize_or_redraw: Pending redraw on %p (default context)...\n",
			default_context));
	
	if(!cur_RawState()) {  /* Check that we have in 'raw' mode */
	    DPRINT(Debug,4,(&Debug,
			    "ket_resize_or_redraw: Going to Raw mode\n"));
	    cur_enter_screen(0,0);
	}
		
	cur_ClearScreen();
	key = redraw;
	goto done_key;
    }

 done_key:
    DPRINT(Debug,10,(&Debug, "key_resize_or_redraw=%d",key));

    switch (key) {
    case RESIZE_MARK:
	DPRINT(Debug,10,(&Debug, " RESIZE_MARK"));
	/* FALLTHRU */
    default:
	if (redraw && redraw == key) {
	    DPRINT(Debug,10,(&Debug, " redraw"));
	}

	if (key && isprint(key) && isascii(key)) {
	    DPRINT(Debug,10,(&Debug, " '%c'",key));
	}
	
    }
    DPRINT(Debug,10,(&Debug, "\n"));
    
    return key;      
}


static int got_resize_or_redraw P_((int flags));
static int got_resize_or_redraw(flags)
     int flags;
{
    int r = 0;

    int found_key = key_resize_or_redraw(flags);
    if (found_key) {
	DPRINT(Debug,4,(&Debug,
			"got_resize_or_redraw: Pending resize or redraw (key %d) on %p (default context) ...\n",
			found_key,default_context));
	
	reset_state(last_state,1);
	set_state_caller_flags(last_state,found_key);
	r = 1;
    }

	
    DPRINT(Debug,10,(&Debug, "got_resize_or_redraw=%d\n",r));
    
    return r;
}



struct charset_state * cur_ReadCh2(flags,break_flag)
     int flags;
     int break_flag;
{
    int redraw      = (flags & READCH_MASK);
    int cursorkeys  = (flags & READCH_CURSOR)    != 0;
    int nocursor    = (flags & READCH_NOCURSOR)  != 0 && cursor_control;
    int term_char   = (flags & READCH_term_char) != 0;
    int resize_flag = (flags & READCH_resize)    != 0; 
    int sig_char    = (flags & READCH_sig_char)  != 0;
    int quote_char  = (flags & READCH_quote)     != 0;
    int poll_it     = (flags & READCH_poll)      != 0;
    int line,col;

    unsigned char input_buffer[20];

    /*
     *	read a character with Raw mode set!
     *
     *	EAGAIN & EWOULDBLOCK are recognized just in case
     *	O_NONBLOCK happens to be in effect.
     */

    
    int read_p,found_key;
    charset_t utf8 = MIME_name_to_charset("UTF-8",0);

#ifdef TERMIOS
    struct tf_state T;

    bzero((void *)&T,sizeof T);
#endif

    DPRINT(Debug,4,(&Debug, 
		    "cur_ReadCh2: flags=%d %s%s%s%s%s%s%s%s %s\n",
		    flags,
		    redraw      ? " redraw"     : "",
		    cursorkeys  ? " cursorkeys" : "",
		    nocursor    ? " nocurosr"   : "",
		    term_char   ? " term_char"  : "",
		    resize_flag ? " resize_flag" : "",
		    sig_char    ? " sig_char"    : "",
		    quote_char  ? " quote_char"  : "",
		    poll_it     ? " poll"        : "",		       
		    break_flag  ? "break_flag" : ""));


    if (POLL_method) 
	change_action2(terminal_fd,0,background_read,no_action_routine,
		       no_action_routine,
		       no_free_action_data,no_inc_action_data_refcount,
		       badpid_remove_action,
		       NO_action_routine_data);    

    cur_GetXYLocation(&line,&col); /* Background actions may print error 
				      messages .. 
				   */
    DPRINT(Debug,4,(&Debug,"cur_ReadCh2: line=%d col=%d\n",
		    line,col));

    if (utf8_input_mode && utf8) {
	DPRINT(Debug,4,(&Debug,"cur_ReadCh2: Using utf-8 input\n"));

	set_last_state(utf8);
    } else
	set_last_state(system_charset);

    if (state_ready(last_state)) {
	const char * MIME_name UNUSED_VAROK = 
	    get_charset_MIME_name(get_state_charset(last_state));

	DPRINT(Debug,4,(&Debug, 
			"cur_ReadCh2: starting reading new character (set %s)\n",
			MIME_name ? MIME_name :	"<no MIME name>"));
	reset_state(last_state,0);
    }
    set_state_caller_flags(last_state,0);

#ifdef TERMIOS
    if (sig_char || quote_char) {
	if (toggle_lflag(&T,0,ISIG)) {
	    DPRINT(Debug,4,(&Debug, 
			    "cur_ReadCh2: Disabled generation of signals\n"));

	}
    } else {
	toggle_lflag(&T,0,0);      /* Just read values */
    }
#endif

 reinit_ReadChar:
    read_p = 0;
    found_key = 0;

    if (redraw && !cur_RawState()) { /* Check that we have in 'raw' mode 
					-- forces redraw is not on raw state
				      */
	DPRINT(Debug,4,(&Debug,
			"cur_ReadCh2: Going to Raw mode\n"));
	cur_enter_screen(0,0);
	cur_ClearScreen();

	reset_state(last_state,1);
	set_state_caller_flags(last_state,redraw);

#ifdef TERMIOS
	reset_lfag(&T);
#endif
	return last_state;
    }

    cur_MoveCursor(line,col,NULL);      
    FlushBuffer();

    check_changes();

    if (got_resize_or_redraw(flags)) {
	DPRINT(Debug,4,(&Debug,
			"cur_ReadCh2: Pending resize or redraw on %p (default context) ...\n",
			default_context));

#ifdef TERMIOS
	reset_lfag(&T);
#endif
	return last_state;
    }
      
    if ((CUR_intransmit != ON || default_context->redraw) &&
	cursorkeys && cursor_control && CUR_transmit_on) {
	DPRINT(Debug,4,(&Debug,
			"cur_ReadCh2: Enabling cursor keys...\n"));
	transmit_functions(ON);
    }

    if ((CUR_intransmit != OFF || default_context->redraw) &&
	nocursor && CUR_transmit_off) {
	DPRINT(Debug,4,(&Debug,
			"cur_ReadCh2: Disabling cursor keys...\n"));
	transmit_functions(OFF);
    }
    

    knode_parse(0);  /* Initialize */

    while (found_key == 0) {
	unsigned char   ch;

	if (POLL_method) {

	    if (do_clear_input) {
		do_clear_input = 0;
		clear_input_buffer();
		
		knode_parse(0);  /* Initialize */
	    }
	    
	    enter_terminal_hook();
	    
	    if (read_buffer_len < 1 && last_read_error == 0) {
		enum wait_for_status wait_result;
		int err = 0;

		
		wait_result =
		    wait_for_action_or_timeout_f(background_read,
						 poll_it ? 1 : -1 /* no timeout */,
						 ( resize_flag || redraw ) ?
						 WAIT_FOR_intr  /* Try to detect resize */
						 : 0,
						 &err);
						 						 
		switch (wait_result) {
		case wait_for_none:
		    DPRINT(Debug,4,(&Debug,
				    "cur_ReadCh2: unexpected result from wait_for_action_or_timeout_f...\n"));
		    /* FALLTHRU */
		case wait_for_done:

		    found_key =  key_resize_or_redraw(flags);
		    if (found_key) {
			DPRINT(Debug,4,(&Debug,
					"cur_ReadCh2: resize or redraw %d\n",
					found_key));
			continue;
		    }

		    break;
		case  wait_for_error:

		    DPRINT(Debug,4,(&Debug,
				    "cur_ReadCh2: wait_for_action_or_timeout_f... errno=%d, %s\n",
				    err,strerror(err)));
		    
		    
		    found_key =  key_resize_or_redraw(flags);
		    if (found_key) {
			DPRINT(Debug,4,(&Debug,
					"cur_ReadCh2: resize or redraw %d\n",
					found_key));
			continue;
		    }
		    
		    if (err != 0) {
									
			/* Return error: */
			if (break_flag   /* GetPrompt wants to see errors! */
			    ) {
			    DPRINT(Debug,4,(&Debug,
					    "       ... return error to caller\n"));
			    found_key = -1;
			    continue;
			}
			if (err == EINTR
#ifdef	EAGAIN
			    || (err == EAGAIN)
#endif
#ifdef	EWOULDBLOCK
			    || (err == EWOULDBLOCK)
#endif
			    ) {
			    
			    if (poll_it) {
				DPRINT(Debug,4,(&Debug,
						"       ... retry ... TIMEOUT_MARK\n"));
				found_key = TIMEOUT_MARK;
			    } else {
				DPRINT(Debug,4,(&Debug,
						"       ... retry (ignored)\n"));			    
			    }
			} else
			    found_key = -1;
			continue;		    
		    }
		    
		    if (poll_it) {
			DPRINT(Debug,4,(&Debug,
					"cur_ReadCh2: interrupted ... TIMEOUT_MARK\n"));
			found_key = TIMEOUT_MARK;
		    } else {
			DPRINT(Debug,4,(&Debug,
					"cur_ReadCh2: interrupted (ignored)\n"));
		    }
		    continue;
		}		
	    } else {
		DPRINT(Debug,4,(&Debug,
				"cur_ReadCh2: %d characters buffered",read_buffer_len));
		if ( last_read_error) {
		    DPRINT(Debug,4,(&Debug,", last read error %d",last_read_error));
		}
		DPRINT(Debug,4,(&Debug,"\n"));
	    }

	    
	    if (read_buffer_len > 0) {
		ch = read_buffer[0];
		if (--read_buffer_len > 0)
		    memmove(read_buffer,read_buffer+1,read_buffer_len);
	    } else if (last_read_error != 0) {
		if (-1 == last_read_error) {
		    DPRINT(Debug,4,(&Debug,
				    "cur_ReadCh2: background_read got zero bytes...\n"));
		    found_key = -1;
		    continue;
		}

		found_key =  key_resize_or_redraw(flags);
		if (found_key) {
		    DPRINT(Debug,4,(&Debug,
				    "cur_ReadCh2: resize or redraw %d\n",
				    found_key));
		    continue;
		}
		
		/* Return error: */
		if (break_flag   /* GetPrompt wants to see errors! */
		    ) {
		    found_key = -1;
		    continue;
		}
		
		if (last_read_error != 0) {
		    errno = last_read_error;
		    DPRINT(Debug,4,(&Debug,
				    "cur_ReadCh2: background_read got error %d (%s)\n",
				    last_read_error,
				    strerror(last_read_error)));
		    found_key = -1;
		}
		continue;         /* continue or quit */	    
	    } else {
		if (poll_it) {
		    DPRINT(Debug,4,(&Debug,
				    "cur_ReadCh2: ... TIMEOUT_MARK\n"));
		    found_key = TIMEOUT_MARK;
		} else {
		    DPRINT(Debug,1,(&Debug,
				    "cur_ReadCh2: SOFTWARE ERROR !!\n"));
		}
		break;
	    }

	} else {  /* no POLL_method */

	    int    result;
	    
	    if (poll_it) {
		DPRINT(Debug,4,(&Debug,
				"cur_ReadCh2: POLL is unsupported, returning TIMEOUT_MARK\n"));
		found_key = TIMEOUT_MARK;
	    } else {
		
		wait_can_signal = 1;
		result = read(terminal_fd, &ch, 1);
		wait_can_signal = 0;
		
		if (result < 0) {
		    int err = errno;
		    
		    DPRINT(Debug,4,(&Debug,
				    "cur_ReadCh2: errno = %d [%s]\n",err,
				    strerror(err)));

		    found_key =  key_resize_or_redraw(flags);
		    if (found_key) {
			DPRINT(Debug,4,(&Debug,
					"cur_ReadCh2: resize or redraw %d\n",
					found_key));
			continue;
		    }
		    
		    /* Return error: */
		    if (break_flag   /* GetPrompt wants to see errors! */
			) {
			found_key = -1;
			continue;
		    }
		    if((err == EINTR)
#ifdef	EAGAIN
		       || (err == EAGAIN)
#endif
#ifdef	EWOULDBLOCK
		       || (err == EWOULDBLOCK)
#endif
		       ) {
			continue;  /* Retry */
		    }
		    break;
		}
		if (0 == result) {
		    DPRINT(Debug,4,(&Debug,
				    "cur_ReadCh2: Got zero bytes...\n"));
		    found_key = -1;
		    continue;
		}
	    }
	}

#ifdef USE_DLOPEN
	{
	    union xxx_rand {
		int ch;
		char bytes[sizeof (int)];
	    } A;
	    A.ch = ch;

	    seed_rand_bits(A.bytes, sizeof A, 
			   3 /* Assume 3 bits per character ... */);
	}
#endif
	
	DPRINT(Debug,50,(&Debug,
			 "cur_ReadCh2: Looking char %d (read_p=%d)\n",ch,read_p));

	if (quote_char) {
	    DPRINT(Debug,4,(&Debug,
			    "cur_ReadCh2: Quoting current character %0xd\n",ch));
	    found_key = ch;
	    continue;
	}

	if (term_char) {
	    if (backspace == ch) 
		found_key = TERMCH_backspace;
	    if (kill_line == ch)
		found_key = TERMCH_kill_line;
	    if (word_erase  == ch)
		found_key = TERMCH_word_erase;
	    if (reprint_char    == ch)
		found_key = TERMCH_reprint_char;
	    if (eof_char        == ch)
		found_key = TERMCH_eof_char;

	    if (found_key) {
		DPRINT(Debug,4,(&Debug,
				"cur_ReadCh2: found termchar = %d (char=%02X)\n",
				found_key,ch));
		/* Soft reset state */
		reset_state(last_state,0); 
		continue;
	    }
	}

	if (sig_char) {
	    if (interrupt_char  == ch)
		found_key = TERMCH_interrupt_char;
	    if (VQUIT_char      == ch)
		found_key = TERMCH_interrupt_char;
	    if (VSUSP_char      == ch) {
		SIGHAND_TYPE (*sig1) P_((int)), (*sig2) P_((int));

		DPRINT(Debug,1,(&Debug,
				"cur_ReadCh2: Found suspend character!\n"));
	       
#ifdef SIGTSTP
		menu_context_redraw();
		switch_title(0);
		CarriageReturn();

		Raw(OFF);
		sig1 = signal(SIGTSTP, SIG_DFL);
		sig2 = signal(SIGCONT, SIG_DFL);

		WriteRaw(Stopped_Text);

		kill(getpid(), SIGSTOP);

		CarriageReturn();  
    
		WriteRaw(Back_Text);

		Raw(ON);

		signal(SIGTSTP, sig1);
		signal(SIGCONT, sig2);

		reset_state(last_state,1); 
		if (redraw) {
		    DPRINT(Debug,1,(&Debug,
				    "cur_ReadCh2: returning redraw because of suspend\n"));
		    cur_ClearScreen();
		    found_key = redraw;
		} else
		    goto reinit_ReadChar;	    
#endif
	    }

	    if (found_key) {
		DPRINT(Debug,4,(&Debug,
				"cur_ReadCh2: found sigchar = %d (char=%02X)\n",
				found_key,ch));
		/* Soft reset state */
		reset_state(last_state,0); 
		continue;
	    }
	}

	if (cursorkeys) {
	    int match = 0;
	    int temp2 = knode_parse(ch);

	    if (temp2 > 0) {  /* Found */
		match++;

		DPRINT(Debug,4,(&Debug,
				"cur_ReadCh2: consumed %d",ch));
		if (isascii(ch) && isprint(ch)) {
		    DPRINT(Debug,4,(&Debug," '%c'",ch));
		}
		DPRINT(Debug,4,(&Debug,"\n"));		
	    		
		DPRINT(Debug,4,(&Debug,
				"cur_ReadCh2: Found function key = %d\n",
				temp2));

		/* Soft reset state */
		reset_state(last_state,0); 

		if (temp2 >= TERMCH_min_char && !term_char) {
		    
		    DPRINT(Debug,4,(&Debug,
				    "cur_ReadCh2: Found key is terminal char and term_char is not set. Ignoring key.\n"));
		    
		    goto reinit_ReadChar;
		} else
		    found_key = temp2;

	    } else if (0 == temp2) {
		match++;
		DPRINT(Debug,4,(&Debug,
				"cur_ReadCh2: need more chars, consumed %d",ch));
		if (isascii(ch) && isprint(ch)) {
		    DPRINT(Debug,4,(&Debug," '%c'",ch));
		}
		DPRINT(Debug,4,(&Debug,"\n"));		
	    }
	    
	    if (read_p < sizeof input_buffer -1) {
		input_buffer[read_p] = ch;
		input_buffer[read_p+1] = '\0';
	    }

	    if (match == 0) {    /* Not in knode */
		if (read_p == 0) 
		    found_key = ch;  /* Normal key */
		else {
		    int i;
		    
		    /* But maybe escape sequence is valid state change ? */
		    	
		    for (i = 0; 
			 i <= read_p && i < sizeof input_buffer -1; 
			 i++) {

			if (state_ready(last_state)) {
			    DPRINT(Debug,4,(&Debug,
					    "cur_ReadCh2: Unsupporting -- function keys and state change sequences overlap? (OR BAD SEQUENCE) \n"));
			    goto BAD_sequence;
			}
			
			if (!add_streambyte_to_state(last_state,
						     input_buffer[i]))
			    goto BAD_sequence;
			
		    }
		    
		    if (state_ready(last_state))
			goto got_key;
		    DPRINT(Debug,4,(&Debug,
				    "cur_ReadCh2: need more bytes for character...\n"));
		    goto reinit_ReadChar;	    


		BAD_sequence:

		    /* Soft reset state */
		    reset_state(last_state,0); 

		    DPRINT(Debug,4,(&Debug,
				    "cur_ReadCh2: Bad escape sequence; ch = %d, read_p = %d\n",
				    ch,read_p));
#ifdef DEBUG
		    DPRINT(Debug,4,(&Debug,
				    "cur_ReadCh2: Sequence was:"));
		    for (i = 0; i <= read_p && i < sizeof input_buffer -1; i++) {
			if (isascii(input_buffer[i]) && isprint(input_buffer[i])) {
			    DPRINT(Debug,4,(&Debug," %c (0x%02X)", 
					    input_buffer[i],input_buffer[i]));
			} else {
			    DPRINT(Debug,4,(&Debug," 0x%02X", input_buffer[i]));
			}
		    }
		    if (read_p > sizeof input_buffer -1)
			DPRINT(Debug,4,(&Debug," ..."));
		    DPRINT(Debug,4,(&Debug,"\n"));
#endif

		    /* Ring a bell */
		    cur_Writechar('\007',NULL);
		    goto reinit_ReadChar;
		}
	    } else
		read_p++;
	} else
	    found_key = ch;
	
    }
    
    if (found_key <= 0 && got_resize_or_redraw(flags)) {

	DPRINT(Debug,4,(&Debug,
			"cur_ReadCh2: Pending resize or redraw on %p (default context) ...\n",
			default_context));
		
#ifdef TERMIOS
	reset_lfag(&T);
#endif
	
	return last_state;
    }

    cur_MoveCursor(line,col,NULL);  
    DPRINT(Debug,50,(&Debug,
		     "cur_ReadCh2: found_key=%d (line=%d col=%d)\n",
		     found_key, line,col));
    
    if (found_key < 0) {
	reset_state(last_state,1);
	DPRINT(Debug,4,(&Debug,
			"cur_ReadCh2=NULL (state)\n")); 

#ifdef TERMIOS
	reset_lfag(&T);
#endif

	return NULL;
    } else {
	if (found_key >= 256) {
	    reset_state(last_state,1);
	    set_state_caller_flags(last_state,found_key);
	} else {
	    set_state_caller_flags(last_state,0);
	    if (!add_streambyte_to_state(last_state,found_key)) {
		/* Ring a bell */
		Writechar('\007');
		DPRINT(Debug,4,(&Debug,
				"cur_ReadCh2: bad sequence...\n"));
		reset_state(last_state,1);
	    }
	}
    }
    
    if (!state_ready(last_state) &&
	!get_state_caller_flags(last_state)) {
	DPRINT(Debug,4,(&Debug,
			"cur_ReadCh2: need more bytes for character...\n"));
	goto reinit_ReadChar;
    }

 got_key:
    DPRINT(Debug,4,(&Debug,
		    "cur_ReadCh2=%p (state): caller_flags=%d, ready=%d\n",
		    last_state,get_state_caller_flags(last_state),
		    state_ready(last_state))); 

#ifdef TERMIOS
    reset_lfag(&T);
#endif

    return last_state;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
