static char rcsid[] = "@(#)$Id: curses.c,v 2.24 2024/06/16 10:40:36 hurtta Exp $";

/************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.24 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI, 
 *                            hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ************************************************************************
 *
 * Some code copied from Elm 2.4 src/curses.c. It have following copyright:
 *
 * 			Copyright (c) 1988-1992 USENET Community Trust
 * 			Copyright (c) 1986,1987 Dave Taylor
 *
 ************************************************************************/

#include "def_screen.h"
#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"screen");

#ifdef TERMIOS
# ifndef sun
#  include <sys/ioctl.h>	/* for TIOCGWINSZ */
# endif
#endif

#include    <errno.h>

/* Lose 'const' */
static char * cs2s P_((const char *str));
static char * cs2s(str)
     const char *str;
{

    return (char *)str;
}

static struct pg_modes {
    int mode;
    char * on;
    char * off;

    const char *set_on;
    const char *set_off;

} set_modes[] = {

    { pg_BOLD, NULL, NULL,   
      "md"     /* Start bold mode */  ,
      NULL     /* End bold mode   */ },  

    { pg_UNDERLINE, NULL, NULL,
      "us"     /* Start underlining   */  ,
      "ue"     /* End underlining     */ },

    { pg_STANDOUT, NULL, NULL,
      "so"     /* Start standout mode */  ,
      "se"     /* End standout mode   */ },

    { pg_BLINKING, NULL, NULL,
      "mb"     /* Start blinking      */  ,
      NULL     /* End blinking      */ },

    { pg_REVERSE, NULL, NULL,
      "mr"     /* Start reverse mode  */ ,
      NULL     /* End reverse mode */ },

    { pg_ITALIC, NULL, NULL,
      "ZH"    /* Enter italic mode */,
      "ZR"    /* End italic mode */ },

    { pg_DIM, NULL, NULL,
      "mh"    /* Start half-bright mode */,
      NULL    /* End half-bright mode */ },

    { pg_bit_SUBSCRIPT , NULL, NULL,
      "ZN"   /* Enter subscript mode */,
      "ZV"   /* End subscript mode */ },

    { pg_bit_SUPERSCRIPT, NULL, NULL,
      "ZO"   /* Enter superscript mode */,
      "ZW"   /* End superscript mode   */ }
    
       
};

static const struct ansi_sgr_attribute * ansi_attr[NUM_pg_attr_bits+5];
static size_t                            ansi_attr_len = 0;
static int                               ansi_attr_pg_flags = 0;

static char *CUR_end_modes = NULL;
/*     me   End all mode like so, us, mb, md and mr  */

static char *CUR_clearscreen = NULL, *CUR_moveto = NULL;
static char *CUR_up = NULL, *CUR_down = NULL;
static char *CUR_right = NULL, *CUR_left = NULL;
static char *CUR_cleartoeoln = NULL, *CUR_cleartoeos = NULL;
static char *CUR_set_memlock = NULL, *CUR_clear_memlock = NULL; 
static char *CUR_start_termcap = NULL, *CUR_end_termcap = NULL;
static char *CUR_bell = NULL;


char *CUR_transmit_on = NULL,  *CUR_transmit_off = NULL; 

int  has_highlighting;	      /* highlighting available? */

/* Some additional capacities not used in ../src/curses.c */

static char *CUR_up_many = NULL, *CUR_down_many = NULL;
static char *CUR_left_many = NULL, *CUR_right_many = NULL;


#if 0  /* SCROLL */
static char *CUR_set_scroll_region = NULL, *CUR_scroll_down = NULL;

static char *CUR_scroll_up = NULL, *CUR_scroll_up_many = NULL;
#endif

/* SO that early writes do not scrach */
static int CUR_lines     = DEFAULT_LINES_ON_TERMINAL;
static int CUR_columns   = DEFAULT_COLUMNS_ON_TERMINAL;

static int CUR_automargin, CUR_eatnewlineglitch;

int CUR_intransmit = -1;	        /* are we transmitting keys? */

char terminal_type[40] = "";
static char CUR_terminal[1024];              /* Storage for terminal entry */
static char CUR_capabilities[1024];          /* String for cursor motion */

static volatile int CUR_line  = -1;       /* initialize to "trash" */
static volatile int CUR_col   = -1;

int cursor_control    = 0;
int cur_tabspacing    = 8;   /* Default? */

/* find tab stops preceding or following a given column position 'a', where
 * the column position starts counting from 1, NOT 0!
 */
#define prev_tab(a)	(((((a-1)/cur_tabspacing))*cur_tabspacing)+1)
#define next_tab(a)	(((((a-1)/cur_tabspacing)+1)*cur_tabspacing)+1)


static char *us2s P_((unsigned char *str));
static char *us2s(str) 
     unsigned char *str;
{
    return (char *)str;
}

/* On SunOS 5.10  there is prototype
      int tputs(char *, int, int (*)(char))
   on /usr/include/term.h if __STDC__ is defined.

   On other hand without prototype 
        int outchar(char)
   becomes 
        int outchar(int)

   This workaround seems not be required when compiling
   with gcc on sun

   On linux prototype is 
        int tputs(const char *str, int affcnt, int (*putc)(int));

*/

#if defined(__STDC__) && defined(sun) && !defined(__GNUC__)

int OUTCHAR(char c) 
{
    return outchar(c);
}

/* Also on SunOS 5.10 there is declaration for tgoto only
   if __STDC__ is NOT defined.

   When SunOS declares tgoto, following probably fails 
*/

extern char *tgoto(const char *cap, int col, int row);

#else

#define OUTCHAR  outchar
#endif

static volatile enum rawstate CUR_inraw = OFF;                  /* are we IN rawmode?    */

static int original_tty_set = 0;
static term_buff CUR_raw_tty,CUR_original_tty;


int cur_InitScreen()
{
    int i;
	/* Set up all this fun stuff: returns zero if all okay, or;
        -1 indicating no terminal name associated with this shell,
        -2..-n  No termcap for this terminal type known
   */

    char *ptr = CUR_capabilities;	/* for buffering         */

	int     err;
	char *termenv;
	int fd;

	if ((termenv = getenv("TERM")) == NULL) return(-1);

	strfcpy(terminal_type, termenv, sizeof terminal_type);

	if ((err = tgetent(CUR_terminal, terminal_type)) != 1)
		return(err-2);
	
	/* CUR_line, CUR_col are assumed to be trash -- do not
	   set them to zero now ... 
	*/

	fd = open("/dev/tty",O_RDWR);
	if (fd < 0) {
	    int err UNUSED_VAROK = errno;

	    DPRINT(Debug,4, (&Debug, "Opening of /dev/tty failed: %s (errno %d)\n",
			     strerror(err),err));
	} else {
	    int pid = getpid();
	    FlushBuffer();
	    terminal_fd = fd;
	    DPRINT(Debug,4,(&Debug,
			    "/dev/tty is fd %d  -- pid for panic tty prompt is %d\n",
			    terminal_fd,pid));

	    set_panic_tty_fd(terminal_fd, pid);  
	}
	    

	if (CUR_inraw) 
	    panic("CURSES PANIC",__FILE__,__LINE__,"cur_InitScreen",
		  "CUR_inraw is set",0);
	
	if (get_term_chars(terminal_fd,&CUR_original_tty))
	    original_tty_set = 1;
	else
	    return -1;

	/* load in all those pesky values */
	CUR_clearscreen       = tgetstr("cl", &ptr);     /*  Clear screen and cursor home */
	CUR_moveto            = tgetstr("cm", &ptr);     /* Cursor move to row %1 and column %2 
							 (on screen) */
	CUR_up                = tgetstr("up", &ptr);     /* Cursor up one line */
	CUR_down              = tgetstr("do", &ptr);     /* Cursor down one line */
	CUR_right             = tgetstr("nd", &ptr);     /* Cursor right one character */
	CUR_left              = tgetstr("bc", &ptr);     /* backspace, if not ^H */
	CUR_bell              = tgetstr("bl", &ptr);     /* Audio bell */

	
	for (i =0 ; i < (sizeof set_modes) / sizeof (set_modes[0]); i++) {
	    set_modes[i].on  = tgetstr(cs2s(set_modes[i].set_on),&ptr);

	    if (set_modes[i].set_off)
		set_modes[i].off = tgetstr(cs2s(set_modes[i].set_off),&ptr);

	    switch(set_modes[i].mode) {

	    case pg_STANDOUT:
		if (set_modes[i].on && set_modes[i].off) {
		    has_highlighting = TRUE;
		}
		break;
	    }
	}
		
	CUR_end_modes        =  tgetstr("me", &ptr);     /* End all mode like so, 
							    us, mb, md and mr */

	CUR_cleartoeoln       = tgetstr("ce", &ptr);     /* Clear to end of line */
	CUR_cleartoeos        = tgetstr("cd", &ptr);     /* Clear to end of screen */
	CUR_lines	      	   = tgetnum("li");           /* Number of lines */
	CUR_columns	   = tgetnum("co");           /* Number of columns */
	cur_tabspacing	   =
	    ((cur_tabspacing=tgetnum("it"))<= 0 ? 8 : cur_tabspacing);
	     /* Difference between tab positions */
	
	CUR_automargin	   = tgetflag("am");    /* Automatic margins which means 
						   automatic line wrap */
	CUR_eatnewlineglitch   = tgetflag("xn");         /* Newline/wraparound glitch */
	CUR_transmit_on	   = tgetstr("ks", &ptr);     /* Turn keypad on  */
	CUR_transmit_off      = tgetstr("ke", &ptr);     /* turn keypad off */
	
	CUR_set_memlock	   = tgetstr("ml", &ptr);     /* ... not listed ... */
	CUR_clear_memlock	   = tgetstr("mu", &ptr);     /* ... not listed ... */
	
	CUR_start_termcap	   = tgetstr("ti", &ptr);   /* Begin program that uses 
							       cursor motion */
	CUR_end_termcap	   = tgetstr("te", &ptr);     /* End program that uses cursor 
							 motion */


	get_terminal_keys(&ptr);


	/* Linux termcap lists following .... 

	   mb   Start blinking
	   md   Start bold mode
	   me   End all mode like so, us, mb, md and mr
	   mh   Start half bright mode
	   mr   Start reverse mode
	   us   Start underlining
	   so   Start standout mode

	*/


        /* following strings are not copied from ../src/curses.c */

	CUR_up_many           = tgetstr("UP", &ptr);    /* Cursor up %1 lines */
	CUR_down_many         = tgetstr("DO", &ptr);    /* Cursor down %1 lines */
	CUR_left_many         = tgetstr("LE", &ptr);    /* Cursor left %1 lines */
	CUR_right_many        = tgetstr("RI", &ptr);    /* Cursor right %1 lines */

#if 0  /* SCROLL */
	CUR_set_scroll_region = tgetstr("cs", &ptr);    /* Scroll region from 
						        line %1 to %2 */
	CUR_scroll_down       = tgetstr("sr", &ptr);    /* Reverse scroll */

	CUR_scroll_up         = tgetstr("sf", &ptr);    /* Normal scroll one line
							-- aka 'cursor down' */
	CUR_scroll_up_many    = tgetstr("SF", &ptr);    /* Normal scroll n line
							-- aka 'cursor down' */

#endif

	if (CUR_transmit_on && CUR_transmit_off) {
	    cursor_control = TRUE;
	}


	if (!CUR_left) {
		CUR_left = "\b";
	}

	if (CUR_lines == 0) CUR_lines = DEFAULT_LINES_ON_TERMINAL;
	if (CUR_columns == 0) CUR_columns = DEFAULT_COLUMNS_ON_TERMINAL;

	return 0;
}

/* Called after config is read */
void curses_check_terminal()
{
    if (terminal_type[0]) {
	ansi_attr_pg_flags
	    = ansi_sgr_attributes(terminal_type,
				  ansi_attr,
				  (sizeof ansi_attr) /
				  sizeof (ansi_attr[0]),
				  &ansi_attr_len);
	DPRINT(Debug,4, (&Debug,
			 "curses_check_terminal: ansi_attr_pg_flags=%x (%s), terminal=%s\n",
			 ansi_attr_pg_flags,
			 give_pg_flags(ansi_attr_pg_flags),
			 terminal_type));

	
    } else {
	DPRINT(Debug,4, (&Debug,
			 "curses_check_terminal: cur_InitScreen not called?\n"));

    }
}



/* DO NOT return lines-1 (as ScreenSize() did) */

void cur_ScreenSize(lines, columns)
     int *lines, *columns;
{
	/** returns the number of lines and columns on the display. **/

#ifdef TIOCGWINSZ
	struct winsize w;

	if (ioctl(terminal_fd,TIOCGWINSZ,&w) != -1) {
		if (w.ws_row > 0)
			CUR_lines = w.ws_row;
		if (w.ws_col > 0)
			CUR_columns = w.ws_col;
	} else {
	    int err UNUSED_VAROK = errno;
	    
	    DPRINT(Debug,4, (&Debug, 
			     "ioctl(%d,TIOCGWINSZ,..) failed: %s (errno %d)\n",
			     terminal_fd,
			     strerror(err),err));		    
	}
#endif

	if (CUR_lines == 0) CUR_lines = DEFAULT_LINES_ON_TERMINAL;
	if (CUR_columns == 0) CUR_columns = DEFAULT_COLUMNS_ON_TERMINAL;



	*lines = CUR_lines;	   /* This do not return lines-1 
				      as ScreenSize() did
				   */     
	*columns = CUR_columns;
}

void cur_GetXYLocation(row,col)
     int *row,*col;
{
	/* return the current cursor location on the screen */

	*row = CUR_line;
	*col = CUR_col;
}

static int  need_moveabsolute = 0;
static volatile int raw_off_indicator = 0;

void InvalidateLocation()
{
    need_moveabsolute = 1;
    CUR_line = 0;
    CUR_col  = 0;
}

void cur_ClearScreen()
{
    /* clear the screen: returns -1 if not capable */

    CUR_line = 0;	/* clear leaves us at top... */
    CUR_col  = 0;

    if (raw_off_indicator && CUR_inraw) {

	SIGDPRINT(Debug,7,(&Debug, 
			   "cur_ClearScreen: clearing raw_off_indicator \n"));
	raw_off_indicator = 0;
    }

    if (default_context->redraw) {
	SIGDPRINT(Debug,7,(&Debug, 
			   "cur_ClearScreen: clearing redraw context from default context\n"));
	default_context->redraw  = 0;
    }
    CUR_intransmit = -1;   /* Re-set state */

    if (!CUR_clearscreen) {
	SIGDPRINT(Debug,4,(&Debug, "cur_ClearScreen: NO CUR_clearscreen set!!\n"));

	return;
    }

    tputs(CUR_clearscreen, 1, OUTCHAR);
    FlushBuffer();      /* clear the output buffer */
 
    return;
}

static void moveabsolute P_((int col, int row));

static void cur_CursorUp P_((int n));
static void cur_CursorUp(n)
     int n;
{
  /** move the cursor up 'n' lines **/
  /** Calling function must check that CUR_up is not null before calling **/

  if (need_moveabsolute)
      moveabsolute(CUR_col, CUR_line);

  CUR_line = (CUR_line-n > 0? CUR_line - n: 0);	/* up 'n' lines... */

  if (n > 1 && CUR_up_many) {
      char *stuff;

      stuff = tgoto(CUR_up_many,0,n);
      tputs(stuff, 1, OUTCHAR);
      
  } else while (n-- > 0)
      tputs(CUR_up, 1, OUTCHAR);

}

static void cur_CursorDown P_((int n));
static void cur_CursorDown(n)
     int n;
{
  /** move the cursor down 'n' lines **/
  /** Caller must check that CUR_down is not null before calling **/

  if (need_moveabsolute)
      moveabsolute(CUR_col, CUR_line);

  CUR_line = (CUR_line+n < CUR_lines? CUR_line + n: CUR_lines-1);    /* down 'n' lines... */

  if (n > 1 &&	CUR_down_many) {
      char *stuff;

      stuff = tgoto(CUR_down_many,0,n);
      tputs(stuff, 1, OUTCHAR);

  } else while (n-- > 0)
      tputs(CUR_down, 1, OUTCHAR);  
}


static void cur_CursorLeft P_((int n));
static void cur_CursorLeft(n)
     int n;
{
    /** move the cursor 'n' characters to the left **/
    /** Caller must check that CUR_left is not null before calling **/
    
    if (need_moveabsolute)
	moveabsolute(CUR_col, CUR_line);
    
    CUR_col = (CUR_col - n> 0? CUR_col - n: 0);	/* left 'n' chars... */
  
    if (n > 1 &&  CUR_left_many) {
	char *stuff;
	
	stuff = tgoto(CUR_left_many,0,n);
	tputs(stuff, 1, OUTCHAR);
	
    } else while (n-- > 0)
	tputs(CUR_left, 1, OUTCHAR);
        
}

static void cur_CursorRight P_((int n));
static void cur_CursorRight(n)
     int n;
{
    /** move the cursor 'n' characters to the right (nondestructive) **/
    /** Caller must check that CUR_right is not null before calling **/
    
    if (need_moveabsolute)
	moveabsolute(CUR_col, CUR_line);
    
    /* NOTE: if CUR_col == CUR_columns then it is outside of columns */

    CUR_col = (CUR_col+n < CUR_columns? CUR_col + n: CUR_columns);	
    /* right 'n' chars... */
    
    if (n > 1 &&  CUR_right_many) {
	char *stuff;
	
	stuff = tgoto(CUR_right_many,0,n);
	tputs(stuff, 1, OUTCHAR);
	
    } else while (n-- > 0)
	tputs(CUR_right, 1, OUTCHAR);
    
}

static void moveabsolute(col, row)
     int col, row;
{

    char *stuff;

    if (need_moveabsolute) {
	SIGDPRINT(Debug,4,(&Debug, 			   
			   "Curses: moveabsolute: Syncronizing cursos position (col=%d,row=%d)\n",
			   col,row));
    }
    
    stuff = tgoto(CUR_moveto, col, row);
    tputs(stuff, 1, OUTCHAR);
     
    if(need_moveabsolute)
	FlushBuffer();
       
    need_moveabsolute = 0;
}

static int in_area P_((int line, struct cur_printarea *lim));
static int in_area(line,lim)
     int line; 
     struct cur_printarea *lim;
{
    int r = line >= lim->first_line && line <= lim->last_line;

    if (line >= CUR_lines) {
	DPRINT(Debug,4, (&Debug, "in_area: Not in screen (line=%d)\n",line));
	r = 0;
    }

    DPRINT(Debug,12, (&Debug, 
		      "in_area(line=%d,lim=%p)=%d: first_line=%d last_line=%d\n",
		      line,lim,r,lim->first_line,lim->last_line));

    return r;
}

void cur_MoveCursor(row, col, lim)
     int row, col;
     struct cur_printarea *lim;

{
    /** move cursor to the specified row column on the screen.
	0,0 is the top left! **/
    
    int scrollafter = 0;

    /* we don't want to change "rows" or we'll mangle scrolling... */
    
    if (need_moveabsolute)
	moveabsolute(CUR_col, CUR_line);;
    
    if (lim) {
	int in = in_area(row,lim);
	

	if (lim->gone_out && in) {
	    DPRINT(Debug,4,(&Debug,"cur_MoveCursor: Going to printing area, enabling printing\n"));
	    lim->gone_out = 0;
	}

	if (!in) {

	    if ( ! lim->gone_out) {
		DPRINT(Debug,4,(&Debug,"cur_MoveCursor: Going out of printing area, disabling printing\n"));
		lim->gone_out = 1;
	    }

	    return;
	}

    }


    if (col < 0)
	col = 0;
    if (col >= CUR_columns)
	col = CUR_columns - 1;
    if (row < 0)
	row = 0;
    if (row >= CUR_lines) {
	if (col == 0)
	    scrollafter = row - CUR_lines +1;
	row = CUR_lines -1;
    }

    if (!CUR_moveto) {
	DPRINT(Debug,4,(&Debug, "curses: NO CUR_moveto set!!\n"));

	return;
    }
    
    if (row == CUR_line) {
	if (col == CUR_col)
	    return;				/* already there! */
	
	else if (col == 0)
	    outchar('\r');

	else if (abs(col - CUR_col) < 5     /* within 5 spaces... */  
		 ||
		 (CUR_left_many && CUR_right_many)
		 ) {	
	    if (col > CUR_col && CUR_right)
		cur_CursorRight(col - CUR_col);
	    else if (col < CUR_col &&  CUR_left)
		cur_CursorLeft(CUR_col - col);
	    else
		moveabsolute(col, row);
	} else 		
	    /* move along to the new x,y loc */
	    moveabsolute(col, row);

    } else if (CUR_line == row-1 && col == 0) {
	if (CUR_col != 0)
	    outchar('\r');
	outchar('\n');      
    }
    else if (col == CUR_col && (abs(row - CUR_line) < 5 
				||
				(CUR_up_many && CUR_down_many)
				)) {
	if (row < CUR_line && CUR_up)
	    cur_CursorUp(CUR_line - row);
	else if (row > CUR_line && CUR_down)
	    cur_CursorDown(row - CUR_line);
	else
	    moveabsolute(col, row);
    } else 
	moveabsolute(col, row);
    
    CUR_line = row;	/* to ensure we're really there... */
    CUR_col  = col;
    
    if (scrollafter) {

	DPRINT(Debug,49, (&Debug, " scrollafter=%d\n",scrollafter));
	
	if (lim) {
	    DPRINT(Debug,4,(&Debug,"cur_MoveCursor: scroling not allowed, disabling printing\n"));
	    lim->gone_out = 0;

	    FlushBuffer();	
	    return;
	}

	outchar('\r');

	while (scrollafter--) {
	    outchar('\n');
	}
    }

    FlushBuffer();	
}

static void write_ansi_sgr_attr P_((short ANSI_sgr_settings[],
				    int setcount));
static void write_ansi_sgr_attr(ANSI_sgr_settings,setcount)
     short ANSI_sgr_settings[];
     int setcount;
{
    int i;
    
    if (setcount < 1)
	return;

    /* Write CSI as ESC [ */
    outchar(0x1B  /* ESC (escape)  */);
    outchar('[');

    for (i = 0; i < setcount; i++) {
        char buffer[8];
	int n,j; 
	
	if (i > 0)
	    outchar(';');
	
	n = elm_sfprintf(buffer,sizeof buffer,FRM("%d"),
			 ANSI_sgr_settings[i]);
	for (j = 0; j < n; j++)
	    outchar(buffer[j]);
    }
    
    outchar('m');
}

void cur_changemode(mode,set,reset)
     int *mode; 
     int set; 
     int reset;
{
    int i;
    int found = 0;
    int f;
    int unsupported = 0;

    /* To allow switching modes from 'reset' to 'set', run reset first */

    if ((*mode == set && 0 == reset)  
	||
	(0 == set     && 0 == reset)  
	||
	(0 == *mode   && 0 == set  && ~0 != reset)
	) {

	DPRINT(Debug,9, (&Debug, 
			 "cur_changemode: nothing to do: for %x set and %x reset: mode = %x %s\n",
			 set,reset,*mode,
			 give_pg_flags(*mode)));

	return;
    }
    
    DPRINT(Debug,7, (&Debug, 
		     "cur_changemode: for %x (%s) set and ",
		     set, give_pg_flags(set)));

    DPRINT(Debug,7, (&Debug, "%x (%s) reset: ",
		     reset, give_pg_flags(reset)));

    DPRINT(Debug,7, (&Debug, "mode = %x (%s)\n",
		     *mode,give_pg_flags(*mode)));

    f = pg_set_or_disable_flags(NULL,set);
    if (ison(*mode,f)) {
	int d = f & *mode ;

	if (d) {
	    DPRINT(Debug,7, (&Debug, 
			     "cur_changemode: disable set %x => reset incompatible %x (%s)\n",
			     f,d, give_pg_flags(d)));

	    setit(reset,d);
	}
    }

    /* pg_STANDOUT is probably  pg_BOLD and pg_REVERSE 
       or
       pg_STANDOUT is same than pg_REVERSE   

       Probable should re-enable other if cleared
    */

    
    if (reset) {
	DPRINT(Debug,7, (&Debug, 
			 "cur_changemode: need reset %x %s\n",
			 reset, give_pg_flags(reset)));	       
    }
	    
    if (ison(reset,ansi_attr_pg_flags)) {

	short ANSI_sgr_settings [ NUM_pg_attr_bits ];
	int setcount = 0;
	size_t j;
	
	DPRINT(Debug,7, (&Debug, 
			 "cur_changemode: ansi_attr_len=%zu, ansi_attr_pg_flags %zu (%s)\n",
			 ansi_attr_len,
			 ansi_attr_pg_flags,
			 give_pg_flags(ansi_attr_pg_flags)));


	/* Do disables if possoble */

	for (j = 0;
	     j < ansi_attr_len &&
		 setcount < (sizeof ANSI_sgr_settings) /
		 sizeof (ANSI_sgr_settings[0]);
	     j++) {
	    if (ansi_attr[j]) {		
		if (ison(reset,ansi_attr[j]->pg_flag)) {
		    
		    ANSI_sgr_settings[setcount++] =
			ansi_attr[j]->end_value;
		    
		    clearit(*mode,ansi_attr[j]->pg_flag);
		    setit(found,ansi_attr[j]->pg_flag);

		    DPRINT(Debug,7, (&Debug, 
				     "cur_changemode: %s %x %s reset %x ",
				     ansi_attr[j]->name ?
				     ansi_attr[j]->name : "",
				     ansi_attr[j]->pg_flag,
				     give_pg_flags(ansi_attr[j]->pg_flag),
				     reset));

		    DPRINT(Debug,7, (&Debug, 
				      "=> mode = %x (%s)\n",
				      *mode,
				      give_pg_flags(*mode)));
		}		
	    }
	}

	write_ansi_sgr_attr(ANSI_sgr_settings,setcount);	
    }

    
    for (i =0 ; i < (sizeof set_modes) / sizeof (set_modes[0]); i++) {
	if (ison(reset,set_modes[i].mode)   && set_modes[i].off) {
	    tputs(set_modes[i].off, 1, OUTCHAR);
	    
	    clearit(*mode,set_modes[i].mode);
	    setit(found,set_modes[i].mode);

	    DPRINT(Debug,7, (&Debug, 
			     "cur_changemode: %s %x %s reset %x ",
			     set_modes[i].set_off ?
			     set_modes[i].set_off : "",
			     set_modes[i].mode,
			     give_pg_flags(set_modes[i].mode),
			     reset));

	    DPRINT(Debug,7, (&Debug, "=> mode = %x (%s)\n",
			     *mode,
			     give_pg_flags(*mode)));
	   
	}
    }

    if (reset != found && CUR_end_modes) {
	int needmode = *mode & ~reset;
	
	tputs(CUR_end_modes, 1, OUTCHAR); 
	
	DPRINT(Debug,7, (&Debug, 
			 "cur_changemode: all reset (for mode %x reset), need set %x (%s)\n",
			 reset,needmode,
			 give_pg_flags(needmode)));
	
	*mode = 0;
	
	setit(set,needmode);
    }

    if (set) {
	DPRINT(Debug,7, (&Debug, 
			 "cur_changemode: need set %x %s\n",
			 set, give_pg_flags(set)));	       
    }

    for (i =0 ; i < (sizeof set_modes) / sizeof (set_modes[0]); i++) {

	if (ison(set,set_modes[i].mode)   && set_modes[i].on) {
	    
	    tputs(set_modes[i].on, 1, OUTCHAR);
	    setit(*mode,set_modes[i].mode);

	    DPRINT(Debug,7, (&Debug, 
			     "cur_changemode: %s %x %s set %x ",
			     set_modes[i].set_on ?
			     set_modes[i].set_on : "",
			     set_modes[i].mode,
			     give_pg_flags(set_modes[i].mode),
			     set));

	    DPRINT(Debug,7, (&Debug, 
			     "=> mode = %x (%s)\n",
			     *mode,
			     give_pg_flags(*mode)));

	}

    }

    if (ison(set,ansi_attr_pg_flags)) {
	short ANSI_sgr_settings [ NUM_pg_attr_bits ];
	int setcount = 0;
	size_t j;
	
	DPRINT(Debug,7, (&Debug, 
			 "cur_changemode: ansi_attr_len=%zu, ansi_attr_pg_flags %zu (%s)\n",
			 ansi_attr_len,
			 ansi_attr_pg_flags,
			 give_pg_flags(ansi_attr_pg_flags)));

	/* Do enables if possible */

	for (j = 0;
	     j < ansi_attr_len &&
		 setcount < (sizeof ANSI_sgr_settings) /
		 sizeof (ANSI_sgr_settings[0]);
	     j++) {
	    if (ansi_attr[j]) {

		if (ison(set,ansi_attr[j]->pg_flag)) {

		    ANSI_sgr_settings[setcount++] =
			ansi_attr[j]->start_value;

		    setit(*mode,ansi_attr[j]->pg_flag);

		    DPRINT(Debug,7, (&Debug, 
				     "cur_changemode: %s %x %s set %x ",
				     ansi_attr[j]->name ?
				     ansi_attr[j]->name : "",
				     ansi_attr[j]->pg_flag,
				     give_pg_flags(ansi_attr[j]->pg_flag),
				     set));

		    DPRINT(Debug,7, (&Debug, 
				     "=> mode = %x (%s)\n",
				     *mode,
				     give_pg_flags(*mode)));

		}
	    }
	}

	write_ansi_sgr_attr(ANSI_sgr_settings,setcount);	
	
    }


    unsupported = set;
    clearit(unsupported,*mode);
    if (unsupported) {
	DPRINT(Debug,7, (&Debug, 
			 "cur_changemode: unsupported %x (%s)",
			 unsupported,give_pg_flags(unsupported)));
	
	DPRINT(Debug,7, (&Debug, " -> result %x (%s)\n",
			 *mode,give_pg_flags(*mode)));
    }
        
}

/*  ----------------------------------------------------------- */

static volatile int wrappedlastchar = 0;
static int tabexpand = 0;          /* Is terminal driver expanding tabs? */


/* called with value of -1 if just preprosessing needed 
   (ie need_moveabsolute) 

   returns value indication is character printing needed
*/
static int write_control P_((int ch, struct cur_printarea *lim));
static int write_control(ch,lim)
     int ch;
     struct cur_printarea *lim;
{
    /* Setting lim disables scrooling !!
     */

    int justwrapped, nt;
    static int warn = 0;

    if (need_moveabsolute)
	moveabsolute(CUR_col, CUR_line);

    justwrapped = 0;

    /* We do nothing if cursor is not currently on area ... */
    if (lim && !in_area(CUR_line,lim)) {
	if (!warn) {
	    SIGDPRINT(Debug,4,(&Debug,
			       "write_control: Cursor not in print area (lines %d-%d): line %d\n",
			       lim->first_line,
			       lim->last_line,
			       CUR_line));
	}
	warn++;
	return 0;
    }

    if (lim && lim->gone_out) {
	if (!warn) {
	    SIGDPRINT(Debug,4,(&Debug,
			       "write_control: Cursor was going out of print area (lines %d-%d): line %d\n",
			       lim->first_line,
			       lim->last_line,
			       CUR_line));
	}
	warn++;
	return 0;
    }

    warn = 0;

    switch (ch) {
    /* if return, just go to left column. */
    case 0x0D: /* CR */

	if (wrappedlastchar)
	    justwrapped = 1;                /* preserve wrap flag */
	else {
	    outchar('\r');
	    CUR_col = 0;
	}
	break;
    
	/* if newline and terminal just did a newline without our asking,
	 * do nothing, else output a newline and increment the line count */
    case 0x0A: /* LF */
	if (!wrappedlastchar) {

	    if (lim && CUR_line == lim->last_line) {

		SIGDPRINT(Debug,4,(&Debug,
				   "write_control: going out of printing area (CR), disabling printing\n"));

		/* Disable printing */
		need_moveabsolute = 1;
		lim->gone_out = 1;
		return 0;
	    }

	    outchar('\n');
	    if (CUR_line < CUR_lines-1)
		++CUR_line;
	}
	break;
    
    /* if backspace, move back  one space  if not already in column 0 */
    case 0x08:  /* BS */
	if (CUR_col != 0) {
	    tputs(CUR_left, 1, OUTCHAR);
	    CUR_col--;
	}

	else if (CUR_line > 0) {

	    if (lim && CUR_line == lim->first_line) {

		/* BACKSPACE does nothing */
		break;
	    }

	    CUR_col = CUR_columns - 1;
	    CUR_line--;
	    moveabsolute (CUR_col, CUR_line);
	}
	/* else BACKSPACE does nothing */
	break;

	/* if bell, ring the bell but don't advance the column */
    case 0x07: /* BEL */
	if (CUR_bell)
	    tputs(CUR_bell,1,OUTCHAR);
	else
	    outchar(ch);
	break;

    /* if a tab, output it */
    case 0x09: /* HT */
	/* If terminal driver is expanding tabs, don't trust it ... */
	if (!tabexpand)
	    outchar(ch);
	if((nt=next_tab(CUR_col+1)) > prev_tab(CUR_columns))
	    CUR_col = CUR_columns-1;
	else
	    CUR_col = nt-1;
	if (tabexpand)
	    moveabsolute (CUR_col, CUR_line);
	break;
    default:    /* Is 'printable' character (or -1) */

	if (lim && CUR_col == CUR_columns-1 && CUR_automargin && 
	    CUR_line == CUR_lines-1) {

	    SIGDPRINT(Debug,4,(&Debug,
			       "write_control: going out of printing area and screen, disabling printing on last column\n"));

	    /* Disable printing */
	    need_moveabsolute = 1;
	    lim->gone_out = 1;

	    return 0;
	}


	return 1;
    }
    
    wrappedlastchar = justwrapped;
    return 0;   /* Need no printing */
}

static void increment_position P_((int len,struct cur_printarea *lim));
static void increment_position(len,lim)
     int len;
     struct cur_printarea *lim;
{
    int justwrapped;

    int CUR_col_len = CUR_col + len;

    justwrapped = 0;
    
    if ( CUR_col_len > CUR_columns) {

	SIGDPRINT(Debug,1,(&Debug,
			   "increment_position: CUR_col %d + len %d = %d > CUR_columns %d\n",
			   
			   CUR_col,len,CUR_col_len,CUR_columns));
	
	CUR_col = 0;
	CUR_line = 0;
	moveabsolute (CUR_col, CUR_line);
	panic("CURSES PANIC",__FILE__,__LINE__,"increment_position",
	      "column position overflow",0);
    }

    /* if we only had one column left, simulate automargins if
     * the terminal doesn't have them */
    if (CUR_col_len == CUR_columns) {

	if (lim && CUR_col == CUR_columns-1 && CUR_line == lim->last_line) {
	    
	    SIGDPRINT(Debug,4,(&Debug,
			       "increment_position: going out of printing area (CR), disabling printing\n"));

	    /* Disable printing */
	    need_moveabsolute = 1;
	    lim->gone_out = 1;

	} else {

	    if (!CUR_automargin || CUR_eatnewlineglitch) {
		SIGDPRINT(Debug,40,(&Debug,
				    "increment_position: EOLN -- printing CR LF\n"));
		outchar('\r');
		outchar('\n');
	    } else {
		SIGDPRINT(Debug,40,(&Debug,
				    "increment_position: EOLN -- screen wrapping assumed\n"));
	    }
	    if (CUR_line < CUR_lines-1)
		++CUR_line;
	    CUR_col = 0;
	    justwrapped = 1;
	}
    }
    /* if we are here this means we have no interference from the
     * right margin - just increment the column position. */
    else {
	CUR_col += len;
    }

    wrappedlastchar = justwrapped;
}

/* Called by signal handler ... */

void WriteRaw(str)
     const char *str;
{
     const char * p;


     /* If WriteRaw() is called, then screen must be redraw
	later ...
     */
     menu_context_redraw();

     for (p = str; *p; p++) {
	 int ch = *p;
	 
	 /** write a character to the current screen location. **/
	 int need_print = 1;

	 ch &= 0xFF;

	 need_print = write_control(ch, NULL);

	 if (need_print) {
	     outchar(ch);
	     increment_position(1, NULL);
	 }
     }

     FlushBuffer();
}



/* This assumes one byte charset */
void cur_Writechar(ch, lim)
     int ch;
     struct cur_printarea *lim;
{
    /** write a character to the current screen location. **/
    int need_print = 1;

    ch &= 0xFF;

    /* ASCII assumed */
    need_print = write_control(ch,lim);

    if (need_print) {
#ifndef ASCII_CTYPE
	if (display_charset == system_charset) {
	    /* if some kind of non-printable character change to a '?' */

	    if(!isprint((unsigned char)ch))
		ch = '?';
	} else 
#endif
	    {
		/* If system (= locale) charset is not display charset
		 * need ask from charset system is that character
		 * printable -- complex and expensive!
		 */
		struct charset_state *st = new_state(display_charset);
		add_streambyte_to_state(st,ch);
		
		if (!state_ready(st) || !state_printable(st))
		    ch = '?';		
		free_state(&st);
	    }
	outchar(ch);
	increment_position(1,lim);
    }
}

void cur_WriteUnicode(unicode, lim)
     int unicode;
     struct cur_printarea *lim;
{
    /** write a character to the current screen location. **/
    int need_print = 1;

    /* ASCII assumed */
    if (unicode <= 31) 
	    need_print = write_control(unicode, lim);
    else 
	need_print = write_control(-1, lim);  /* HACK for position of cursor */

    if (need_print) {

	charset_t stored_display = target_display_charset;
	charset_t cs_buffer      = target_display_charset;

	struct cs_printable_len PRINTABLE_LEN;
	struct cs_printable_len *printable_len = NULL;

	charset_t utf8 = MIME_name_to_charset("UTF-8",0);

	uint16 vector[1];

	struct string  * handled = NULL;

	int tmp_mode  = CUR_modes;
	int POS = 0;
	int len1;

	if (utf8 && cs_buffer != utf8 && allow_charset_switching && 
	    0 != (charset_properties(utf8)  & CS_printable) &&
	    set_display_charset(utf8,1,
				0 /* No signal */ )) {
	    DPRINT(Debug,40,(&Debug, 
			     "cur_WriteUnicode: Changed to UTF-8\n"));
	    cs_buffer = utf8;
	}

	if (last_display_state && 
	    0 != (CS_printable_len & charset_properties(cs_buffer))) {
	    
	    PRINTABLE_LEN.max_len = CUR_columns-CUR_col;
	    PRINTABLE_LEN.ret_len = 0;
	    
	    printable_len = &PRINTABLE_LEN;
	}
	
	handled = new_string(cs_buffer);

	switch (unicode) {	
	case UNICODE_NO_BREAK_SPACE:
	    vector[0]    = 0x0020   /* space */;
	    break;
	case UNICODE_SOFT_HYPHEN:
	    vector[0] = 0x002D /* HYPHEN-MINUS */;
	    break;
	case UNICODE_BAD_CHAR:         /*  REPLACEMENT CHARACTER */
	   
	    if (0 == (CS_universal_set & 
		      charset_properties(cs_buffer))) {
		
		/* Show bold question mark instead */
		cur_changemode(&tmp_mode,pg_BOLD,pg_STANDOUT);
		vector[0] = 0x003F /* ? */;
	    }
	    /* fallthru */
	default:
	    vector[0]    = unicode;
	    break;
	}

	add_unicode_to_string(handled,1,vector);

	
	POS = 0;
	len1 = string_len(handled);

	DPRINT(Debug,40,(&Debug,
			 "cur_WriteUnicode: unicode=0x%04x handled=%S (len=%d)\n",
			 unicode,handled,len1));
	
	if (len1) {
	    char * s = us2s(streamclip_from_string(handled,&POS,len1,
						   last_display_state,printable_len));
	    char * s1;
	    int q = 0;

	    DPRINT(Debug,10,(&Debug,
			     "cur_WriteUnicode: stream="));

	    for (s1 = s; *s1; s1++) {
#ifdef DEBUG
		if (isascii(*s1) && isprint(*s1) && '"' != *s1) {
		    if (!q) { DPRINT(Debug,10,(&Debug," \"")); q = 1; }
		    DPRINT(Debug,10,(&Debug,"%c",*s1));
		} else {
		    if (q) { DPRINT(Debug,10,(&Debug,"\"")); q = 0; }
		    DPRINT(Debug,10,(&Debug," %02X", (unsigned char)*s1));
		}
#endif
	    }
	    
	    if (q) { DPRINT(Debug,10,(&Debug,"\"")); q = 0; }
	    if (printable_len) {
		DPRINT(Debug,10,(&Debug,
				 " printable_len { max_len %d ret_len %d }",
				 printable_len->max_len,printable_len->ret_len));
	    }
	    DPRINT(Debug,10,(&Debug,"\n"));
	    
	    for (s1 = s; *s1; s1++) {
		outchar(*s1);
	    }
	    
	    /* Restore mode */
	    
	    cur_changemode(&tmp_mode,CUR_modes,pg_BOLD);
	    
	    
	    if (printable_len)
		increment_position(printable_len->ret_len, lim);
	    else
		increment_position(1, lim);
	    
	    free(s);
	}
	free_string(&handled);

	if (cs_buffer != stored_display) {
	    set_display_charset(stored_display,
				0,
				0 /* not signal */);	
	}
    }
}





static int preinit_output P_((const struct string *S, struct string **buffer,
			      charset_t stored_display));
static int preinit_output(S,buffer, stored_display)
     const struct string *S;
     struct string **buffer;
     charset_t stored_display; 
{
    int sw = -1;
    charset_t S_cs = get_string_type(S);

    if (S_cs == display_charset)
	*buffer = dup_string(S);
    else if (target_display_charset == display_charset &&
	     S_cs == system_charset) {
	sw = 1;  /* Arrange return to back to display charset */
	*buffer = dup_string(S);
	switch_display_charset(0,
			       0 /* Not signale */ ); /* Print using system charset */
    } else if (allow_charset_switching && 
	       0 != (charset_properties(S_cs) & CS_printable) &&
	       stored_display &&
	       set_display_charset(S_cs,1,
				   0 /* Not signal */)) {
	sw = 2;  /* Arrange return to correct display charset */
	*buffer = dup_string(S);
    } else
	*buffer = convert_string(display_charset,S,0);

    return sw;
}

static void postprocess_output P_((charset_t stored_display, int sw));

static void postprocess_output(stored_display,sw)
     charset_t stored_display; 
     int sw;
{
    if (sw == 2)
	set_display_charset(stored_display,0,
			    0 /* Not signal */);
    else if (sw != -1) {
	switch_display_charset(sw,
			       0 /* Not signal */);
    }    
}


int curses_available_charset(cs)
     charset_t cs;
{

    if (cs == display_charset)
	return 1;
    if (cs == system_charset)
	return 1;

    if (!allow_charset_switching)
	return 0;
    
    return terminal_can_switch_to(terminal_type,cs,1 /* silent */, 0);

}

/* mallocs result -- result need to be valid characters and usable 
   may return NULL
*/
struct string * curses_available_string(S)
     const struct string *S;
{
#define NUM_charsets 255
    
    charset_t  charset_vector[NUM_charsets];
    int candinate [NUM_charsets];
    int cs_count;
    
    int len = string_len(S);
    int i,j;
    
    charset_t * csv = give_display_charsets(charset_vector,
					    NUM_charsets,
					    0 /* not signal */
					    );

    struct string * S1 = NULL;
    
    
    for (cs_count = 0; csv[cs_count]; cs_count++)
	candinate[cs_count] = 1;

    for (i = 0; i < len; i++) {
	uint16 c = give_unicode_from_string(S,i);
	
	if (UNICODE_BAD_CHAR == c) {


	    return NULL;
	}

	for (j = 0; j < cs_count; j++) {

	    enum charset_unicode_stat x;

	    x = string_type_have_unicode(csv[j],c);

	    switch (x) {
	    case charset_unicode_bad:
	    case charset_unicode_unknown:
	    case charset_missing_unicode:
		candinate[j] = 0;
		break;
	    case charset_have_unicode:
		break;                  
	    }               
	}
    }

    for (j = 0; j < cs_count; j++) {
	if (candinate[j]) {
	    int failcount = 0;
	    
	    S1 =  convert_string2(csv[j],
				  S,&failcount);
	    
	    if (S1 && 0 == failcount)
		break;
	    
	    if (S1)
		free_string(&S1);
	}
    }
    
    return S1;
}

struct string *curses_printable_clip(S,pos,len,visible_len,max_visible)
     const struct string *S;
     int *pos;
     int len;
     int *visible_len;
     int max_visible;
{
    struct string *buffer = NULL;
    charset_t stored_display = target_display_charset;
    int sw = -1;
 
    struct string *ret1   = NULL;

    sw =  preinit_output(S,&buffer, stored_display);

    {
	const char * MIME_name_S UNUSED_VAROK = get_string_MIME_name(S);
	const char * MIME_name_b UNUSED_VAROK = get_string_MIME_name(buffer);

	DPRINT(Debug,35,(&Debug, 
			 "curses_printable_clip: S=%S cs=%s  -- using %s\n",
			 S,
			 MIME_name_S ? MIME_name_S : "<no MIME name>",
			 MIME_name_b ? MIME_name_b : "<no MIME name>"));
    }
    DPRINT(Debug,35,(&Debug, "curses_printable_clip: len=%d max_visible=%d pos=%d\n",
		     len,max_visible,*pos));

    if (last_display_state && 
	0 != (CS_printable_len & 
	      charset_properties(get_string_type(buffer)))) {

	int l1;
	struct cs_printable_len PRINTABLE_LEN;
	
	PRINTABLE_LEN.max_len = max_visible;
	PRINTABLE_LEN.ret_len = 0;

	l1 = estimate_clip_string(buffer,*pos,len,last_display_state,&PRINTABLE_LEN);

	DPRINT(Debug,35,(&Debug,
			 "curses_printable_clip: l1=%d printable_len { max_len %d ret_len %d }\n",
			 l1,PRINTABLE_LEN.max_len,PRINTABLE_LEN.ret_len));

	
	if (l1 < 0) {
	    ret1         = NULL;
	    *visible_len = -1;
	} else {
	    *visible_len = PRINTABLE_LEN.ret_len;
	    ret1 = clip_from_string(buffer,pos,l1);
	}
    } else {
	if (max_visible < len)
	    len = max_visible;

	ret1 = clip_from_string(buffer,pos,len);

	*visible_len = string_len(ret1);
    }

    /* Free if converted */
    if (buffer != S)
	free_string(&buffer);

    postprocess_output(stored_display,sw);

    if (ret1) {
	DPRINT(Debug,35,(&Debug, "curses_printable_clip=%p=%S visible_len=%d pos=%d\n",
			 ret1,ret1,*visible_len,*pos));
    }
    return ret1;
}

void cur_PutLineS(S, lim)
     struct string *S;
     struct cur_printarea *lim;
{
    int idx = 0;
    struct string *buffer = NULL;
    charset_t stored_display = target_display_charset;
    int sw = -1;
    int buffer_len;
    charset_t cs_buffer;

    sw =  preinit_output(S,&buffer, stored_display);

    buffer_len = string_len(buffer);
    cs_buffer  = get_string_type(buffer);

    {
	const char * MIME_name_S UNUSED_VAROK = get_string_MIME_name(S);
	const char * MIME_name_b UNUSED_VAROK = get_string_MIME_name(buffer);

	DPRINT(Debug,40,(&Debug, 
			 "cur_PutLineS: S=%S cs=%s  -- using %s\n",
			 S,
			 MIME_name_S ? MIME_name_S : "<no MIME name>",
			 MIME_name_b ? MIME_name_b : "<no MIME name>"));
    }

    while (idx < buffer_len) {
	char * s = NULL, *s1;
	uint16 ch;
	int old_idx = idx;
	int need_print;
	int len = 1;
	int q = 0;
	struct cs_printable_len PRINTABLE_LEN;
	struct cs_printable_len *printable_len = NULL;
	struct string  * handled = NULL;
	

	int tmp_mode  = CUR_modes;

	DPRINT(Debug,47,(&Debug,
			 "cur_PutLineS: [%d] CUR_col=%d - %d characters left\n",
			 idx,CUR_col, CUR_columns-CUR_col));
	
	ch = give_unicode_from_string(buffer,idx);
	
	/* ASCII assumed */
	if (ch <= 31) 
	    need_print = write_control(ch, lim);
	else 
	    need_print = write_control(-1, lim);  /* HACK for position of cursor */
	if (!need_print) {

	    if (lim && !in_area(CUR_line,lim)) {
		DPRINT(Debug,40,(&Debug,
				 "cur_PutLineS: [%d] not in area ... \n",
				 idx));
		break;
	    }

	    DPRINT(Debug,40,(&Debug,
		       "cur_PutLineS: [%d] unicode=0x%04x (control)\n",
		       idx,ch));
	    idx++;
	    continue;   /* character 'printed' */
	}

	if (last_display_state && 
	    0 != (CS_printable_len & charset_properties(cs_buffer))) {

	    PRINTABLE_LEN.max_len = CUR_columns-CUR_col;
	    PRINTABLE_LEN.ret_len = 0;

	    printable_len = &PRINTABLE_LEN;
	}



	switch (ch) {	
	    uint16 vector[1];
	    uint16 ch1;

	case UNICODE_NO_BREAK_SPACE:

	    handled = new_string(cs_buffer);

	    vector[0]    = 0x0020   /* space */;
	    add_unicode_to_string(handled,1,vector);

	    DPRINT(Debug,41,(&Debug,
			     "cur_PutLineS: [%d] unicode=0x%04x -> 0x%04x\n",
			     idx,ch,vector[0]));

	    break;

	case UNICODE_SOFT_HYPHEN:

	    handled = new_string(cs_buffer);
	    ch1 = 0;

	    if (idx+1 < buffer_len) 
		ch1 = give_unicode_from_string(buffer,idx+1);

	    if (ch1 <= 31 ||
		ch1 == 0x0020   /* space */ ||
		ch1 == UNICODE_NO_BREAK_SPACE) {

		vector[0] = 0x002D /* HYPHEN-MINUS */;
		add_unicode_to_string(handled,1,vector);

		DPRINT(Debug,41,(&Debug,
				 "cur_PutLineS: [%d] unicode=0x%04x -> 0x%04x\n",
				 idx,ch,vector[0]));

	    }
	    break;

	case UNICODE_BAD_CHAR:         /*  REPLACEMENT CHARACTER */
	   
	    if (0 == (CS_universal_set & 
		      charset_properties(cs_buffer))) {
		
		/* Show bold question mark instead */
		cur_changemode(&tmp_mode,pg_BOLD,pg_STANDOUT);
		
		handled = new_string(cs_buffer);
		vector[0] = 0x003F /* ? */;
		add_unicode_to_string(handled,1,vector);
		
		DPRINT(Debug,41,(&Debug,
				 "cur_PutLineS: [%d] unicode=0x%04x -> 0x%04x\n",
				 idx,ch,vector[0]));
	    }
	    
	    break;
	}

	if (handled) {
	    int POS = 0;
	    int len1 = string_len(handled);

	    DPRINT(Debug,40,(&Debug,
			     "cur_PutLineS: [%d] unicode=0x%04x (special) handled=%S (len=%d)\n",
			     idx,ch,handled,len1));

	    idx++;                 /* Special character handled */
	    
	    if (!len1) {

		free_string(&handled);
		continue;
	    }

	    s = us2s(streamclip_from_string(handled,&POS,len1,
					    last_display_state,printable_len));
	       
	    free_string(&handled);
	    
	    DPRINT(Debug,10,(&Debug,
			     "cur_PutLineS: stream="));

	   
	} else {
	    uint16 ch1;

	    DPRINT(Debug,47,(&Debug,
			     "cur_PutLineS: [%d] unicode=0x%04x \n",idx,ch));

	    while (len < CUR_columns-CUR_col &&
		   len + idx < buffer_len &&
		   (ch1 = give_unicode_from_string(buffer,idx + len)) > 31) {
				
		if (UNICODE_NO_BREAK_SPACE  == ch1 ||
		    UNICODE_SOFT_HYPHEN     == ch1 ||
		    UNICODE_BAD_CHAR        == ch1) {
	       		    
		    break;
		}

		DPRINT(Debug,47,(&Debug,
				 "cur_PutLineS: [%d] unicode=0x%04x \n",idx+len,ch1));

		
		len++;
		
	    }

	    DPRINT(Debug,47,(&Debug,
			     "cur_PutLineS: [%d] CUR_col=%d - len %d\n",
			     idx, CUR_col,len));
	    	    
	    s = us2s(streamclip_from_string(buffer,&idx,len,
					    last_display_state,printable_len));
	    
	    DPRINT(Debug,10,(&Debug,
			     "cur_PutLineS: [%d-%d] stream=",
			     old_idx,idx-1));

	}

	for (s1 = s; *s1; s1++) {
#ifdef DEBUG
	    if (isascii(*s1) && isprint(*s1) && '"' != *s1) {
		if (!q) { DPRINT(Debug,10,(&Debug," \"")); q = 1; }
		DPRINT(Debug,10,(&Debug,"%c",*s1));
	    } else {
		if (q) { DPRINT(Debug,10,(&Debug,"\"")); q = 0; }
		DPRINT(Debug,10,(&Debug," %02X", (unsigned char)*s1));
	    }
#endif
	}

	if (q) { DPRINT(Debug,10,(&Debug,"\"")); q = 0; }
	if (printable_len) {
	    DPRINT(Debug,10,(&Debug,
			     " printable_len { max_len %d ret_len %d }",
			     printable_len->max_len,printable_len->ret_len));
	}
	DPRINT(Debug,10,(&Debug,"\n"));

	for (s1 = s; *s1; s1++) {
	    outchar(*s1);
	}

	/* Restore mode */

	cur_changemode(&tmp_mode,CUR_modes,pg_BOLD);


	if (printable_len) {
	    DPRINT(Debug,40,(&Debug, 
			     "cur_PutLineS: increment_position %d (printable_len)\n",
			     printable_len->ret_len));

	    increment_position(printable_len->ret_len, lim);
	} else {
	    int l = idx-old_idx;

	    DPRINT(Debug,40,(&Debug, 
			     "cur_PutLineS: increment_position %d (idx %d old_ind %d)\n",
			     l,idx,old_idx));
	    increment_position(l, lim);

	}
	if (idx == old_idx) {
	    DPRINT(Debug,10,(&Debug,"No any characters printed : wrap line\n"));
	    CleartoEOLN();
	    cur_Writechar('\r',lim);
	    cur_Writechar('\n',lim);	    

	} 

	free(s);
    }

    /* Free if converted */
    if (buffer != S)
	free_string(&buffer);

    postprocess_output(stored_display,sw);
}

void cur_PutLine0(row, col, line, lim)
     int row,col;
     const char *line;
     struct cur_printarea *lim;
{
    /** Write a zero argument line at location x,y **/

    if (lim && ! in_area(row,lim)) {
	DPRINT(Debug,10,(&Debug,
			 "cur_Putline0: (row %d,col %d) not in area row %d-%d\n",
			 row,col,
			 lim->first_line,
			 lim->last_line));
	
	lim->gone_out = 1;
	return;
    }
    if (lim)
	lim->gone_out = 0;

    cur_MoveCursor(row,col, lim);
    while(*line) {
	cur_Writechar(*line++, lim);    
	
	if (lim && !in_area(CUR_line,lim)) {
		DPRINT(Debug,40,(&Debug,
				 "cur_Putline0: not in area ... \n"));
		break;
	}


    }
}

void cur_CleartoEOLN()
{
    /** clear to end of line **/
    
    if (need_moveabsolute)
	moveabsolute(CUR_col, CUR_line);
    
    if (!CUR_cleartoeoln) {
	SIGDPRINT(Debug,4,(&Debug, "curses: NO CUR_cleartoeoln set!!\n"));

	return;
    }

    
    tputs(CUR_cleartoeoln, 1, OUTCHAR);
    
    return;
}


void cur_CleartoEOS()
{
    /** clear to end of screen **/
    
    if (need_moveabsolute)
	moveabsolute(CUR_col, CUR_line);
    
    if (!CUR_cleartoeos) {
	SIGDPRINT(Debug,4,(&Debug, "curses: NO CUR_cleartoeos set!!\n"));

	return;
    }
    
    tputs(CUR_cleartoeos, 1, OUTCHAR);
    
    return;
}

enum rawstate cur_RawState()
{
    /** returns either 1 or 0, for ON or OFF **/
    
    FlushBuffer();

    return( CUR_inraw );
}

#ifdef PTEM
#  include <sys/stream.h>
#  include <sys/ptem.h>
#endif

#ifdef TERMIOS
#define	ttsetattr(fd,where)	tcsetattr((fd),TCSADRAIN,(where))
#else	/*TERMIOS*/
# ifdef TERMIO
#define	ttsetattr(fd,where)	ioctl((fd),TCSETAW,(where))
# else
static int ttsetattr P_((int,struct tty_modes *)); /* Prototype */
# endif	/*TERMIO*/
#endif	/*TERMIOS*/


volatile int do_clear_input    = 1;
int     utf8_input_mode = 0;
int     utf8_error = 0;     /* Locale was not utf-8 */

/* NOTE: Is called from signal handler! */
void cur_leave_screen_signal(do_tite,do_charset)
     int do_tite; 
     int do_charset;
{
    int on_wait = 0;
    
    SIGDPRINT(Debug,4,(&Debug, 
		       "cur_leave_screen_signal(do_tite=%d,do_charset=%d): CUR_inraw=%d wait_can_signal=%d\n",
		       do_tite,do_charset, CUR_inraw,wait_can_signal));

    if (wait_can_signal) {
	SIGDPRINT(Debug,4,(&Debug,"cur_leave_screen_signal: Resetting wait_can_signal\n"));
	wait_can_signal = 0;
	on_wait = 1;
    }
    

    raw_off_indicator = 1;

    CUR_inraw = OFF;

    if (cursor_control)
	transmit_functions(OFF);

    /* If wait_can_signal was set, is is safe to call allocation routines,
       because signal was during syscall wait
    */
    
    if (do_charset)
	switch_display_charset(0,
			       !on_wait /* have signal */);

    /* We do NOT call switch_title(0) here because 
       raw state off may be called because running of program
    */
    
    if (use_tite && CUR_end_termcap && do_tite) {
	tputs(CUR_end_termcap, 1, OUTCHAR);	    
    }
    FlushBuffer();
    
    if (original_tty_set)
	(void) ttsetattr(terminal_fd,&CUR_original_tty);
    else {
	SIGDPRINT(Debug,4,(&Debug, 
			   "cur_leave_screen_signal: CUR_original_tty NOT set\n"));
    }
	
    do_clear_input = 1;

    SIGDPRINT(Debug,4,(&Debug, 
		       "cur_leave_screen_signal: Done\n"));
}

void cur_leave_screen(do_tite,do_charset)
     int do_tite; 
     int do_charset;
{

    DPRINT(Debug,4,(&Debug, 
		    "cur_leave_screen(do_tite=%d,do_charset=%d)\n",
		    do_tite,do_charset));


    cur_leave_screen_signal(do_tite,do_charset);

    do_clear_input = 0;

    if (POLL_method) {
	clear_action(terminal_fd);
	clear_input_buffer();
    }
}

void cur_enter_screen(do_tite,do_charset)
     int do_tite; 
     int do_charset;     
{

    utf8_error = 0;

    DPRINT(Debug,4,(&Debug, 
		    "cur_enter_screen(tite=%d,charset=%d):  CUR_inraw=%d",
		    do_tite,do_charset,CUR_inraw));
    switch (CUR_inraw) {
    case OFF: DPRINT(Debug,4,(&Debug," (OFF)")); break;
    case ON:  DPRINT(Debug,4,(&Debug," (ON)"));  break;
    }

    DPRINT(Debug,4,(&Debug,"\n"));
    

    if (!original_tty_set) 
	panic("CURSES PANIC",__FILE__,__LINE__,"cur_enter_screen",
	      "original_tty is not set",0);
    
    CUR_inraw = ON;

    do_clear_input = 0;

    if (POLL_method) {
	if (do_clear_input) 
	    clear_input_buffer();
    }

    (void) tcgetattr(terminal_fd, &CUR_raw_tty);    /** again! **/
    
#if !defined(TERMIO) && !defined(TERMIOS)
    CUR_raw_tty.sg_flags &= ~(ECHO);	/* echo off */
    CUR_raw_tty.sg_flags |= CBREAK;	/* raw on    */

#else
    CUR_raw_tty.c_lflag &= ~(ICANON | ECHO);	/* noecho raw mode        */
    
    CUR_raw_tty.c_cc[VMIN] = '\01';	/* minimum # of chars to queue    */
    CUR_raw_tty.c_cc[VTIME] = '\0';	/* minimum time to wait for input */
    
    
#if defined(OXTABS)
    /* Some systems use OXTABS bit */
    if (CUR_raw_tty.c_oflag & OXTABS) {
	DPRINT(Debug,4,(&Debug,
			   "curses: [OXTABS] Terminal driver is expanding tabs...\n"));
	tabexpand = 1;
    } else {
	if (tabexpand) 	      
	    DPRINT(Debug,4,(&Debug,
			    "curses: [OXTABS] Terminal driver isn't expanding tabs...\n"));
	tabexpand = 0;
    }
#endif /* defined(OXTABS) */
    
#if defined(TABDLY) && !defined(OXTABS)
    /* Some systems expands tab when TABDLY is XTABS */
    if ((CUR_raw_tty.c_oflag & TABDLY) == 
#ifdef XTABS
	XTABS
#else
	TAB3
#endif
	) {
	DPRINT(Debug,4,(&Debug,
		       "curses: [TABDLY] Terminal driver is expanding tabs...\n"));
	tabexpand = 1;
    } else {
	if (tabexpand) 
	    DPRINT(Debug,4,(&Debug,
			    "curses: [TABDLY] Terminal driver isn't expanding tabs...\n"));
	tabexpand = 0;
    }
#endif /* defined(TABDLY) && !defined(OXTABS) */
    
#if !defined(TABDLY) && !defined(OXTABS)
    
    /* If _POSIX_SOURCE is defined then OXTABS or TABDLY are not 
     * defined -- so print warning anyway
     */
    DPRINT(Debug,4,(&Debug,
		    "curses: No information is terminal driver expanding tabs!\n"));	  
#endif /* !defined(TABDLY) && !defined(OXTABS) */
    
#endif
    (void) ttsetattr(terminal_fd, &CUR_raw_tty);
    if (use_tite && CUR_start_termcap && do_tite) 
	tputs(CUR_start_termcap, 1, OUTCHAR);
    CUR_intransmit = -1; /* state unclear */
    CUR_inraw = ON;


#if defined (IUTF8) && defined(TERMIOS)

    if ((CUR_raw_tty.c_iflag & IUTF8)) {

	DPRINT(Debug,4,(&Debug,
			"curses: [IUTF8]  Terminal is on UTF-8 mode...\n"));

	if (do_charset) {

	    if (!is_utf8_charset(system_charset)) {

		charset_t utf8 = MIME_name_to_charset("UTF-8",0);

		utf8_error = 1;

		DPRINT(Debug,4,(&Debug,
				"curses: [IUTF8]  System charset is not UTF8\n"));

		if ((target_display_charset == system_charset || 
		     ! target_display_charset) && utf8) {

		    DPRINT(Debug,4,(&Debug,
				    "curses: [IUTF8]  Assuming UTF-8 display charset\n"));

		    target_display_charset = utf8;

		}
	    }
	}

	utf8_input_mode = 1;

    } else {

	if (utf8_input_mode) {

	    DPRINT(Debug,4,(&Debug,
			    "curses: [IUTF8]  Terminal is not on UTF-8 mode...\n"));
	}

	utf8_input_mode = 0;

    }
#endif

    
    if (do_charset) {
	switch_display_charset(1,
			       0 /* not signal */);
	switch_title(1);
    }
    need_moveabsolute = 1;

    if (utf8_error)
	lib_error(CATGETS(elm_msg_cat, MeSet,MeNotUTF8Locale,
			  "Terminal (driver) is on UTF-8 mode, but system charset or locale is not UTF-8"));

}    
	

#ifdef TERMIOS

int toggle_lflag(T,enab,disab) 
     struct tf_state *T;
     int enab,disab;
{
    T->old_raw_ok = 0;

    if (! CUR_inraw) {
	DPRINT(Debug,1,(&Debug,"toggle_lflag: CUR_inraw not set!\n"));
	return 0;
    }

    if (tcgetattr(terminal_fd,&CUR_raw_tty) < 0) {
	DPRINT(Debug,1,(&Debug,"toggle_lflag: tcgetattr failed\n"));
	return 0;
    }
    T->old_raw = CUR_raw_tty;
    T->old_raw_ok = 1;

    DPRINT(Debug,10,(&Debug,
		     "toggle_lflag: enable 0x%04x disable 0x%04x   old: 0x%04x\n",
		     enab,disab,CUR_raw_tty.c_lflag));

    CUR_raw_tty.c_lflag |= enab;
    CUR_raw_tty.c_lflag &= ~disab;

    if (tcsetattr(terminal_fd,TCSANOW,&CUR_raw_tty) < 0) {
	DPRINT(Debug,1,(&Debug,"reset_lfalg: tcsetattr failed\n"));
	return 0;
    }
    
    DPRINT(Debug,10,(&Debug,"toggle_lflag: set: 0x%04x\n",CUR_raw_tty.c_lflag));

    return 1;
}

void reset_lfag(T)
     struct tf_state *T;
{
    if (!T->old_raw_ok) {
	DPRINT(Debug,1,(&Debug,"reset_lfalg: old_raw not ok\n"));
	return;
    }

    if (! CUR_inraw) {
	DPRINT(Debug,1,(&Debug,"reset_lflag: CUR_inraw not set!\n"));
	return;
    }

    if (tcsetattr(terminal_fd,TCSANOW,& (T->old_raw)) < 0) {
	DPRINT(Debug,1,(&Debug,"toggle_lflag: tcsetattr failed\n"));
	return;
    }

    DPRINT(Debug,10,(&Debug,"reset_lflag: set: 0x%04x\n",T->old_raw.c_lflag));
}
#endif

/* NOTE: Is called from signal handler! */
void transmit_functions(newstate)
     int newstate;
{
    /** turn function key transmission to ON | OFF **/
    
    if (newstate != CUR_intransmit) {
	CUR_intransmit = newstate;
	if (newstate == ON)
	    tputs(CUR_transmit_on, 1, OUTCHAR);
	else 
	    tputs(CUR_transmit_off, 1, OUTCHAR);
	FlushBuffer();      /* clear the output buffer */
    }
}

int raw_off_called() {

    if (raw_off_indicator) {
	SIGDPRINT(Debug,7,(&Debug, "raw_off_called=1: raw_off_indicator is set\n"));
	return 1;
    }

    if (need_moveabsolute) {
	SIGDPRINT(Debug,7,(&Debug, "raw_off_called=1: need_moveabsolute is set\n"));

	return 1;
    }

    return 0;
}

#if !defined(TERMIO) && !defined(TERMIOS)
static int ttsetattr P_((int fd,
			 struct tty_modes *where));
static int ttsetattr(fd, where)
     int fd;
     struct tty_modes *where;
{
    if (ioctl(fd, TIOCSETP, &where->sgttyb) < 0)
	return(-1);
    if (ioctl(fd, TIOCSETC, &where->tchars) < 0)
	return(-1);
    return(0);
}
#endif

/* ----- following are copied from ../src/curses.c as documentation
         only ... they was not used on ELM ... also these termcap entries 
         seems not documented at least on Linux ....
*/ 


#if 0

static int _memory_locked = 0;		/* are we IN memlock??   */

int
HasMemlock()
{
    /** returns TRUE iff memory locking is available (a terminal
	feature that allows a specified portion of the screen to
	be "locked" & not cleared/scrolled... **/
    
    return ( CUR_set_memlock && CUR_clear_memlock );
}

static int _old_LINES;

int
StartMemlock()
{
    /** mark the current line as the "last" line of the portion to 
	be memory locked (always relative to the top line of the
	screen) Note that this will alter LINES so that it knows
	the top is locked.  This means that (plus) the program 
	will scroll nicely but (minus) End memlock MUST be called
	whenever we leave the locked-memory part of the program! **/
    
    if (! CUR_set_memlock)
	return(-1);
    
    if (! _memory_locked) {
	
	_old_LINES = elm_LINES;
	elm_LINES -= CUR_line;		/* we can't use this for scrolling */
	
	tputs(CUR_set_memlock, 1, OUTCHAR);
	
	_memory_locked = TRUE;
    }

    return(0);
}

int
EndMemlock()
{
    /** Clear the locked memory condition...  **/
    
    if (! CUR_set_memlock)
	return(-1);
    
    if (_memory_locked) {
	elm_LINES = _old_LINES;		/* back to old setting */
	
	tputs(CUR_clear_memlock, 1, OUTCHAR);
       
	_memory_locked = FALSE;
    }
    return(0);
}

#endif /* ndef ELM */

/* ----------------------------------------------------------------------- */

#if 0  /* SCROLL */
static void set_scrolling_region P_((int line1, int line2));
static void set_scrolling_region(line1,line2) 
     int line1; 
     int line2;
{
      char *stuff, *tgoto();

      stuff = tgoto(CUR_set_scroll_region,line1,line2);
      tputs(stuff, 1, OUTCHAR);
}

void limit_position(row,col)
     int *row; 
     int *col;
{
    if (*row < 0)
	*row = 0;
    if (*col < 0)
	*col = 0;
    if (*col >= CUR_columns)
	*col = CUR_columns - 1;

    if (*row >= CUR_lines) {
	*row = CUR_lines -1;
	*col = CUR_columns - 1;
    }	   
}
#endif

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
