static char rcsid[] = "@(#)$Id: screen.c,v 2.13 2024/06/16 10:40:36 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.13 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 * Some code copied from Elm 2.4 src/curses.c. It have following copyright:
 *
 * 			Copyright (c) 1988-1992 USENET Community Trust
 * 			Copyright (c) 1986,1987 Dave Taylor
 *
 * However on that code there is no practically Elm 2.4 code (from where that
 * copyright is) left. Only function names are same.
 *****************************************************************************/

#include "def_screen.h"

#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"screen");

#include    <errno.h>

static struct menu_context * sync_ctx   = NULL;

#define SYNC(x)  while (x != sync_ctx) { Syncronize(x); }

static void StartSyncronize P_((struct menu_context *ctx));

static void PutLineS P_((struct string *S));

static void PutLineS(S)
     struct string *S;
{

    DPRINT(Debug,49, (&Debug, "PutLineS(\"%S\")\n",S));
       
    enter_terminal_hook();

    SYNC(default_context);
    cur_PutLineS(S,NULL);      
}


static void Raw0 P_((int state));
static void Raw0(state)
     int state;
{
    Raw(state);
}

static void ClearScreen0 P_((void));
static void ClearScreen0()
{
    ClearScreen(0);
}

int InitScreen(struct menu_context  **page)
{
    int r;
    int lines,columns;

    DPRINT(Debug,49, (&Debug, "InitScreen(...)\n"));

    r = cur_InitScreen();

    set_start_run_hooks(print_status,RawState,Raw0,
			start_run_tty_init,ClearScreen0,
			PutLineS, print_status_cooked);

    /* Real lines .. not lines-1 !!! */
    cur_ScreenSize(&lines,&columns);

    set_root_menu(lines,columns);

    *page = new_menu_context();
    
    return r;
}

/* This assumes one byte charset */
void Writechar(ch)
     int ch;
{
    DPRINT(Debug,49, (&Debug, "Writechar('%c')\n",ch));

    enter_terminal_hook();

    SYNC(default_context);
    cur_Writechar(ch, NULL);
}

void menu_Writechar(ctx,ch)
     struct menu_context * ctx;
     int ch;
{
    DPRINT(Debug,49, (&Debug, "menu_Writechar(%p,'%c')\n",ctx,ch));

    if (MENU_CONTEXT_magic != ctx->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_Writechar",
	      "Bad type magic number",0);
    if (MENU_DRAW_magic  != ctx->routine->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_Writechar",
	      "Bad draw magic number",0);

    enter_terminal_hook();

    SYNC(ctx);
    ctx->routine->wra_Writechar(ctx,ch);
}

extern void menu_WriteUnicode(ctx,unicode)
     struct menu_context * ctx;
     int unicode;
{
    DPRINT(Debug,49, (&Debug, "menu_Writechar(%p,%04x)\n",ctx,unicode));

    if (MENU_CONTEXT_magic != ctx->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_WriteUnicode",
	      "Bad type magic number",0);
    if (MENU_DRAW_magic  != ctx->routine->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_WriteUnicode",
	      "Bad draw magic number",0);
    
    enter_terminal_hook();

    SYNC(ctx);
    ctx->routine->wra_WriteUnicode(ctx,unicode);
}

/*VARARGS2*/

void Write_to_screen (
#if ANSI_C
		      const char *format, const char *msg, ...
#else
		      format, msg, va_alist
#endif
		      )
#if !ANSI_C
     const char *format; 
     const char *msg;
     va_dcl
#endif
{
    va_list vl;
    struct string *text;

    DPRINT(Debug,49, (&Debug, "Write_to_screen(format=\"%s\",...)\n",format));

    enter_terminal_hook();

    SYNC(default_context);
    
    /** This routine writes to the screen at the current location.
	when done, it increments lines & columns accordingly by
	looking for "\n" sequences... **/

    Va_start(vl, msg);           /* defined in hdrs/elm_defs.h */
    
    text = elm_smessage(0,format,msg,vl);

    va_end(vl);
    
    PutLineS(text);
    free_string(&text);
     
}


/*VARARGS2*/

void menu_Write_to_screen (
#if ANSI_C
		      struct menu_context * ctx,
		      const char *format, const char *msg, ...
#else
		      ctx, format, msg, va_alist
#endif
		      )
#if !ANSI_C
     struct menu_context * ctx; 
     const char *format; 
     const char *msg;
     va_dcl
#endif
{
    va_list vl;
    struct string *text;

    DPRINT(Debug,49, (&Debug, 
		      "menu_Write_to_screen(%p,format=\"%s\",...)\n",
		      ctx,format));

    if (MENU_CONTEXT_magic != ctx->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_Write_to_screen",
	      "Bad type magic number",0);
    if (MENU_DRAW_magic  != ctx->routine->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_Write_to_screen",
	      "Bad draw magic number",0);

    enter_terminal_hook();

    SYNC(ctx);


    
    /** This routine writes to the screen at the current location.
	when done, it increments lines & columns accordingly by
	looking for "\n" sequences... **/

    Va_start(vl, msg);           /* defined in hdrs/elm_defs.h */
    
    text = elm_smessage(0,format,msg,vl);

    va_end(vl);
    
    ctx->routine->wra_PutLineS(ctx,text);

    free_string(&text);
        
}

void PutLine0(x, y, text)
     int x,y;
     const char *text;
{
    DPRINT(Debug,49, (&Debug, "PutLine0(%d,%d,\"%s\")\n",x,y,text));

    enter_terminal_hook();

    SYNC(default_context);
    cur_PutLine0(x,y,text,NULL);
}

void menu_PutLine0(ctx, x, y, text)
     struct menu_context * ctx; 
     int x,y;
     const char *text;
{
    DPRINT(Debug,49, (&Debug, 
		      "menu_PutLine0(%p,%d,%d,\"%s\")\n",
		      ctx,x,y,text));

    if (MENU_CONTEXT_magic != ctx->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_PutLine0",
	      "Bad type magic number",0);
    if (MENU_DRAW_magic  != ctx->routine->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_PutLine0",
	      "Bad draw magic number",0);

    enter_terminal_hook();

    SYNC(ctx);
    ctx->routine->wra_PutLine0(ctx,x,y,text);
}


void PutLineX(
#if ANSI_C
     int x, int y, const char *format, const char *line, ...
#else
     x, y, format, line, va_alist
#endif
     )
#if !ANSI_C
     int x; 
     int y; 
     const char *format; 
     const char *line; 
     va_dcl
#endif
{
    va_list vl;
    struct string *text;

    DPRINT(Debug,49, (&Debug, "PutLineX(%d,%d,format=\"%s\",...)\n",x,y,format));

    enter_terminal_hook();

    SYNC(default_context);
    
    MoveCursor(x,y);

    Va_start(vl, line);           /* defined in hdrs/elm_defs.h */
    
    text = elm_smessage(0,format,line,vl);

    va_end(vl);
    
    PutLineS(text);
    free_string(&text);


    
}


void menu_PutLineX(
#if ANSI_C
		   struct menu_context * ctx,
		   int x, int y, const char *format, const char *line, ...
#else
		   ctx, x, y, format, line, va_alist
#endif
     )
#if !ANSI_C
     struct menu_context * ctx; 
     int x; 
     int y; 
     const char *format; 
     const char *line; 
     va_dcl
#endif
{
    va_list vl;
    struct string *text;

    DPRINT(Debug,49, (&Debug, 
		      "menu_PutLineX(%p,%d,%d,format=\"%s\",...)\n",
		      ctx, x,y,format));

    if (MENU_CONTEXT_magic != ctx->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_PutLineX",
	      "Bad type magic number",0);
    if (MENU_DRAW_magic  != ctx->routine->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_PutLineX",
	      "Bad draw magic number",0);

    enter_terminal_hook();

    SYNC(ctx);
    ctx->routine->wra_MoveCursor(ctx, x, y);

    Va_start(vl, line);           /*defined in hdrs/elm_defs.h  */
        
    text = elm_smessage(0,format,line,vl);

    va_end(vl);
    
    ctx->routine->wra_PutLineS(ctx,text);

    free_string(&text);
    
}

void CleartoEOLN()
{
    DPRINT(Debug,49, (&Debug, "CleartoEOLN()\n"));

    enter_terminal_hook();

    SYNC(default_context);
    cur_CleartoEOLN();
}

void menu_CleartoEOLN(ctx)
     struct menu_context * ctx; 
{
    DPRINT(Debug,49, (&Debug, "menu_CleartoEOLN(%p)\n",ctx));

    if (MENU_CONTEXT_magic != ctx->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_CleartoEOLN",
	      "Bad type magic number",0);
    if (MENU_DRAW_magic  != ctx->routine->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_CleartoEOLN",
	      "Bad draw magic number",0);
    
    enter_terminal_hook();

    SYNC(ctx);    
    ctx->routine->wra_CleartoEOLN(ctx);
}

void CleartoEOS()
{
    DPRINT(Debug,49, (&Debug, "CleartoEOS()\n"));
    
    enter_terminal_hook();

    SYNC(default_context);
    cur_CleartoEOS();
}

void menu_CleartoEOS(ctx)
     struct menu_context * ctx; 
{
    DPRINT(Debug,49, (&Debug, "menu_CleartoEOS(%p)\n",ctx));
    
    if (MENU_CONTEXT_magic != ctx->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_CleartoEOS",
	      "Bad type magic number",0);
    if (MENU_DRAW_magic  != ctx->routine->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_CleartoEOS",
	      "Bad draw magic number",0);
    
    enter_terminal_hook();

    SYNC(ctx);
    ctx->routine->wra_CleartoEOS(ctx);
}

enum rawstate RawState()
{
    enum rawstate r;

    DPRINT(Debug,49, (&Debug, "RawState()\n"));

    r = cur_RawState();

    return r;
}

volatile int was_on_cooked = 1;
volatile int do_enter_terminal = 0;

void signal_leave_screen()
{
     int on_cooked   =  default_context == &COOKED_MENU;

     
     SIGDPRINT(Debug,4, (&Debug, 
			 "signal_leave_screen: on_cooked=%d do_enter_terminal=%d\n",
			 on_cooked, do_enter_terminal));


     if (!do_enter_terminal)
	 was_on_cooked = on_cooked;

     cur_leave_screen_signal(! on_cooked, 1);

}

void signal_enter_screen()
{

    SIGDPRINT(Debug,4, (&Debug, 
			"signal_enter_screen: was_on_cooked=%d do_enter_terminal=%d\n",
			was_on_cooked, do_enter_terminal));

    do_enter_terminal = 1;

}

void enter_terminal_hook()
{
    if (do_enter_terminal) {
	do_enter_terminal = 0;

	DPRINT(Debug,4, (&Debug, "enter_terminal_hook: Pending restoring terminal: was_on_cooked = %d\n",
			 was_on_cooked));

	cur_enter_screen(! was_on_cooked, 1);		
    }
}

struct menu_context * Raw(state)
     int state;
{
    int do_tite        = !(state & NO_TITE); 
    int do_charset     = 1;
    int on_cooked   = default_context == &COOKED_MENU;
    enum rawstate old = cur_RawState();

    DPRINT(Debug,4, (&Debug, 
		     "Raw(%d): do_tite=%d on_cooked=%d do_charset=%d old_state=%d\n",
		     state,do_tite,on_cooked,do_charset,old));

    state = state & ~NO_TITE;

    
    if (state == old) {
	DPRINT(Debug,4, (&Debug, "Raw: No state change!\n"));
    } else if (state == OFF)
	cur_leave_screen(do_tite,do_charset);
    else if (state == ON)
	cur_enter_screen(do_tite,do_charset);
    else
	panic("CURSES PANIC",__FILE__,__LINE__,"Raw",
	      "bad mode",0);

    if (cur_RawState() && do_tite && on_cooked) {
	
	DPRINT(Debug,4, (&Debug, "Raw: Switching from COOKED MENU\n"));
	default_context = ROOT_MENU.next_menu;
	
	if (!default_context) {
	    default_context = &ROOT_MENU;
	    DPRINT(Debug,4, (&Debug, "   ... To ROOT MENU\n"));
	}
	
	DPRINT(Debug,9, (&Debug, 
			 "Raw: default context to %p\n",
			 default_context));
		
	StartSyncronize(default_context);
	
	
	if (utf8_error) {
	    lib_error(CATGETS(elm_msg_cat, MeSet,MeNotUTF8Locale,
			      "Terminal (driver) is on UTF-8 mode, but system charset or locale is not UTF-8"));
	    
	    error_wait();
	}
	
	return default_context;
    }
    
    
    if (! cur_RawState() && do_tite && !on_cooked) {
	
	EndSyncronize(default_context);
	
	DPRINT(Debug,4, (&Debug, "Raw: Switching TO COOKED MENU\n"));
	
	default_context = &COOKED_MENU;	 
	
	DPRINT(Debug,9, (&Debug, 
			 "Raw: default context to %p\n",
			 default_context));
	
	return default_context;
    }      
    
    
    if (!cur_RawState()) {
	
	SIGDPRINT(Debug,4, (&Debug, "Raw: returning COOKED MENU\n"));
	
	return &COOKED_MENU;	 
    }
    
    return default_context;
}

struct charset_state * ReadCh2(flags)
     int flags;
{
    struct charset_state *st;

    DPRINT(Debug,49, (&Debug, "ReadCh2(%d)\n",flags));

    enter_terminal_hook();

    SYNC(default_context);
    st = cur_ReadCh2(flags,InGetPrompt);

    return st;
}

struct charset_state * menu_ReadCh2(ptr,flags)
     struct menu_context *ptr;
     int flags;
{
    struct charset_state *st;

    DPRINT(Debug,49, (&Debug, "menu_ReadCh2(%p,...,%d)\n",ptr,flags));

    if (MENU_CONTEXT_magic != ptr->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_ReadCh2",
	      "Bad type magic number",0);
    if (MENU_DRAW_magic  != ptr->routine->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_ReadCh2",
	      "Bad draw magic number",0);

    enter_terminal_hook();

    SYNC(ptr);
    if (ptr->redraw && (flags & READCH_MASK)) {
	set_last_state(system_charset);
	
	reset_state(last_state,1);
	set_state_caller_flags(last_state,flags & READCH_MASK);

	return last_state;   /* cur_ReadCh2 checks now default_contect 
			      */
    }

    if (ptr->changed && (flags & READCH_resize)) {
	set_last_state(system_charset);
	
	reset_state(last_state,1);
	set_state_caller_flags(last_state,RESIZE_MARK);

	return last_state;   /* cur_ReadCh2 checks now default_contect 
			      */
    }

    st = ptr->routine->wra_ReadCh2(ptr,flags,InGetPrompt);

    return st;
}

void FlushInput() {

    DPRINT(Debug,49, (&Debug, "FlushInput()\n"));

    cur_FlushInput();
}

int menu_ReadCh(ptr,flags)
     struct menu_context *ptr; 
     int flags;
{
    int c;

    DPRINT(Debug,49, (&Debug, "menu_ReadCh(%p,...,%d)\n",ptr,flags));

    if (ptr->magic != MENU_CONTEXT_magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_ReadCh",
	      "Bad type magic number",0);
    if (MENU_DRAW_magic  != ptr->routine->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_ReadCh",
	      "Bad draw magic number",0);

    enter_terminal_hook();

    SYNC(ptr);

    if (ptr->redraw && (flags & READCH_MASK)) {
	c = (flags & READCH_MASK);
	DPRINT(Debug,50, (&Debug, "menu_ReadCh=0x%02x (REDRAW_MARK)\n",c));
	return c;   /* cur_ReadCh2 checks now 
		       default_contect 
		    */
    }

    if (ptr->changed && (flags & READCH_resize)) {
	c = RESIZE_MARK;
	DPRINT(Debug,50, (&Debug, "menu_ReadCh=0x%02x (RESIZE_MARK)\n",c));
	return c;
    }


    do {
	struct charset_state *ch = ptr->routine->wra_ReadCh2(ptr,flags,InGetPrompt);

	if (!ch) {
	    c = EOF;
	    DPRINT(Debug,4,(&Debug, 
			    "menu_ReadCh==EOF (errno=%d)\n",errno));
	} else if (get_state_caller_flags(ch)) {
	    c = get_state_caller_flags(ch);
	    DPRINT(Debug,4,(&Debug, 
			    "menu_ReadCh==%d (flags)\n",c));
	} else
	    c =  state_is_onebyte(ch);
	if (!c) {
	    DPRINT(Debug,4,(&Debug,
			    "menu_ReadCh: Not a one byte character or zero byte!"));
	    /* Ring a bell */
	    ptr->routine->wra_Writechar(ptr,'\007');
	} else {
	    DPRINT(Debug,4,(&Debug,
			    "menu_ReadCh==%d (one byte character)\n",
			    c));
	}

	if (!c && (flags & READCH_poll)) {
	    c = TIMEOUT_MARK;
	    DPRINT(Debug,4, (&Debug, "menu_ReadCh==TIMEOUT_MARK\n"));
	}

    } while (0 == c);

    DPRINT(Debug,50, (&Debug, "menu_ReadCh=0x%02x\n",c));

    return c;
}

void ScreenSize(lines, columns)
     int *lines, *columns;
{

    DPRINT(Debug,49, (&Debug, "ScreenSize(*,*)\n"));

    /* cur_ScreenSize   DO NOT return lines-1 (as ScreenSize() did) */

    cur_ScreenSize(lines,columns);

    ((*lines)--);
}

void menu_ScreenSize(ctx,lines, columns)
     struct menu_context *ctx; 
     int *lines, *columns;
{

    DPRINT(Debug,49, (&Debug, "ScreenSize(%p,*,*)\n",ctx));

    if (ctx->magic != MENU_CONTEXT_magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_ScreenSize",
	      "Bad type magic number",0);
    if (MENU_DRAW_magic  != ctx->routine->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_ScreenSize",
	      "Bad draw magic number",0);


    /* menu_ScreenSize   DO NOT return lines-1 (as ScreenSize() did) */

    ctx->routine->wra_ScreenSize(ctx,lines,columns);
}


void GetXYLocation(x,y)
     int *x,*y;
{
    DPRINT(Debug,49, (&Debug, "GetXYLocation(*,*)\n"));

    enter_terminal_hook();

    SYNC(default_context);
    cur_GetXYLocation(x,y);
}

void menu_GetXYLocation(ctx,row,col)
     struct menu_context *ctx;
     int *row,*col;
{
    DPRINT(Debug,49, (&Debug, "menu_GetXYLocation(%p,*,*)\n",ctx));

    if (MENU_CONTEXT_magic != ctx->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_GetXYLocation",
	      "Bad type magic number",0);

    if (MENU_DRAW_magic  != ctx->routine->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_GetXYLocation",
	      "Bad draw magic number",0);

    enter_terminal_hook();

    SYNC(ctx);
    ctx->routine->wra_GetXYLocation(ctx,row,col);

    /* WARNING:

        row,col  !=  ctx->row, ctx->col

       because row,col are relative and ctc->row, ctx->col
       are absolute

    */

}


/* NOTE: Is called fron interrupt! */
void ClearScreen(interrupt)
     int interrupt;
{
    if (interrupt) {

	SIGDPRINT(Debug,49, (&Debug, "ClearScreen() -- interrupt\n"));

    } else {
	DPRINT(Debug,49, (&Debug, "ClearScreen() -- no interrupt\n"));

	enter_terminal_hook();

	SYNC(default_context);
    }

    cur_ClearScreen();
}

void menu_ClearScreen(ptr)
     struct menu_context *ptr;
{
    DPRINT(Debug,49, (&Debug, "menu_ClearScreen(%p,...)\n",ptr));

    if (ptr->magic != MENU_CONTEXT_magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_ClearScreen",
	      "Bad type magic number",0);


    if (MENU_DRAW_magic  != ptr->routine->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_GetXYLocation",
	      "Bad draw magic number",0);

    enter_terminal_hook();

    SYNC(ptr);
    ptr->routine->wra_ClearScreen(ptr);
}

void MoveCursor(row, col)
     int row, col;
{
    DPRINT(Debug,49, (&Debug, "MoveCursor(%d,%d)\n",row,col));

    enter_terminal_hook();

    SYNC(default_context);
    cur_MoveCursor(row, col, NULL);
}

void menu_MoveCursor(ctx,row, col)
     struct menu_context *ctx;
     int row, col;
{
    DPRINT(Debug,49, (&Debug, "MoveCursor(%p,%d,%d)\n",ctx,row,col));

    if (MENU_CONTEXT_magic != ctx->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_GetXYLocation",
	      "Bad type magic number",0);

    if (MENU_DRAW_magic  != ctx->routine->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_GetXYLocation",
	      "Bad draw magic number",0);

    enter_terminal_hook();

    SYNC(ctx);
    ctx->routine->wra_MoveCursor(ctx, row, col);
}

int menu_GetAbsLine(ctx,row)
     struct menu_context *ctx;
     int row;
{
    int result;
    
    DPRINT(Debug,49, (&Debug, "menu_GetAbsLine(%p,%d)\n",ctx,row));

    if (MENU_CONTEXT_magic != ctx->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_GetAbsLine",
	      "Bad type magic number",0);

    if (MENU_DRAW_magic  != ctx->routine->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_GetAbsLine",
	      "Bad draw magic number",0);

    ctx->routine->wra_calculate_line(ctx,row,&result);

    DPRINT(Debug,49, (&Debug, "menu_GetAbsLine=%d\n",result));
    return result;
}

int menu_translate_pos(ctx,row, col, res_ctx, res_row,res_col)
    struct menu_context *ctx;
    int row;
    int col;
    struct menu_context *res_ctx;
    int * res_row;
    int * res_col;
{
    int absline,r;
    int newrel = row;
    
    DPRINT(Debug,49, (&Debug, "menu_translate_pos(%p,%d,%d,%p,...,...)\n",ctx,row,col,res_ctx));

    
    
    if (MENU_CONTEXT_magic != ctx->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_translate_pos",
	      "Bad type magic number (ctx)",0);

    if (MENU_DRAW_magic  != ctx->routine->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_translate_pos",
	      "Bad draw magic number (ctx->routine)",0);

    ctx->routine->wra_calculate_line(ctx,row,&absline);

    
    DPRINT(Debug,49, (&Debug, "menu_translate_pos: absline=%d\n",
		      absline));

    if (MENU_CONTEXT_magic != res_ctx->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_translate_pos",
	      "Bad type magic number (resctx)",0);

    if (MENU_DRAW_magic  != res_ctx->routine->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_translate_pos",
	      "Bad draw magic number (resctx->routine)",0);

    
    r = res_ctx->routine->wra_calculate_rline(res_ctx,absline,
					     &newrel);

    DPRINT(Debug,49, (&Debug, "menu_translate_pos: newrel=%d\n",
		      newrel));

    
    if (res_row)
	*res_row = newrel;
    if (res_col)
	*res_col = col;    /* NO change */

    DPRINT(Debug,49, (&Debug, "menu_translate_pos=%d\n",r));

    return r;
}
    

void CarriageReturn()
{

    DPRINT(Debug,49, (&Debug, "CarriageReturn()\n"));

    enter_terminal_hook();

    SYNC(default_context);

    /** move the cursor to the beginning of the current line **/
    Writechar('\r');
}

void NewLine()
{
    DPRINT(Debug,49, (&Debug, "NewLine()\n"));
    /** move the cursor to the beginning of the next line **/

    enter_terminal_hook();

    SYNC(default_context);

    Writechar('\r');
    Writechar('\n');
    FlushBuffer();	
}

int CUR_modes = 0;

void StartBold()
{
    int add = pg_BOLD;
    int disable = pg_set_or_disable_flags(NULL,add);
	
    DPRINT(Debug,49, (&Debug, "StartBold()\n"));

    enter_terminal_hook();

    SYNC(default_context);
    cur_changemode(&CUR_modes, add,disable);
}

void EndBold()
{
    DPRINT(Debug,49, (&Debug, "EndBold()\n"));
    /** compliment of startbold **/

    enter_terminal_hook();

    SYNC(default_context);
    cur_changemode(&CUR_modes, 0, pg_BOLD);
}

void StartStandout() 
{
    int add = pg_STANDOUT;
    int disable = pg_set_or_disable_flags(NULL,add);
    
    DPRINT(Debug,49, (&Debug, "StartStandout()\n"));

    enter_terminal_hook();

    SYNC(default_context);
    cur_changemode(&CUR_modes, add,disable);
}

void EndStandout() 
{
    DPRINT(Debug,49, (&Debug, "EndStandout()\n"));

    enter_terminal_hook();

    SYNC(default_context);
    cur_changemode(&CUR_modes, 0, pg_STANDOUT);
}


void StartUnderline() 
{
    int add = pg_UNDERLINE;
    int disable = pg_set_or_disable_flags(NULL,add);

    DPRINT(Debug,49, (&Debug, "StartUnderline()\n"));

    enter_terminal_hook();

    SYNC(default_context);
    cur_changemode(&CUR_modes, add,disable);
}

void EndUnderline() 
{
    DPRINT(Debug,49, (&Debug, "EndUnderline()\n"));

    enter_terminal_hook();

    SYNC(default_context);
    cur_changemode(&CUR_modes, 0, pg_UNDERLINE);
}

void menu_StartXX(ctx,f)
     struct menu_context *ctx; 
     int f;
{
    int disable = pg_set_or_disable_flags(NULL,f);

    if (MENU_CONTEXT_magic != ctx->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_StartXX",
	      "Bad type magic number",0);

    if (MENU_DRAW_magic  != ctx->routine->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_StartXX",
	      "Bad draw magic number",0);

    enter_terminal_hook();

    DPRINT(Debug,7, (&Debug,"menu_StartXX(ctx=%p,f=%x %s)",
		     ctx,f,give_pg_flags(f)));

    if (disable) {
	DPRINT(Debug,7, (&Debug," - disable=%x %s",
			 disable,give_pg_flags(disable)));
    }
    DPRINT(Debug,7, (&Debug,"\n"));

	   
		     
    SYNC(ctx);
    ctx->routine->wra_changemode(ctx, f, disable);
}

void menu_EndXX(ctx,f)
     struct menu_context *ctx; 
     int f;
{
    if (MENU_CONTEXT_magic != ctx->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_EndXX",
	      "Bad type magic number",0);

    if (MENU_DRAW_magic  != ctx->routine->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_EndXX",
	      "Bad draw magic number",0);

    enter_terminal_hook();

    DPRINT(Debug,7, (&Debug,"menu_EndXX(ctx=%p,f=%x %s)\n",
		     ctx,f,give_pg_flags(f)));
    
    SYNC(ctx);
    ctx->routine->wra_changemode(ctx, 0, f);
}

void EndSyncronize(ctx)
     struct menu_context *ctx; 
{
    if (MENU_CONTEXT_magic != ctx->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"EndSyncronize",
	      "Bad type magic number",0);

    if (MENU_DRAW_magic  != ctx->routine->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"EndSyncronize",
	      "Bad draw magic number",0);

    if (sync_ctx   != ctx)
	return;               /* Nothing need to do */
    
    if (ctx == &COOKED_MENU) {
	DPRINT(Debug,9, (&Debug, "EndSyncronize: Skippig cooked meny syncronize\n"));
	sync_ctx = NULL;
	return;
    }

    DPRINT(Debug,9, (&Debug, "EndSyncronize: Syncronize from %p\n",ctx));

    ctx->routine->wra_SyncLeave(ctx);
    sync_ctx = NULL;
}

static void StartSyncronize(ctx)
     struct menu_context *ctx; 
{
    if (MENU_CONTEXT_magic != ctx->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"EndSyncronize",
	      "Bad type magic number",0);

    if (MENU_DRAW_magic  != ctx->routine->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"EndSyncronize",
	      "Bad draw magic number",0);

    if (sync_ctx   == ctx)
	return;               /* Nothing need to do */

    if (ctx == &COOKED_MENU) {
	DPRINT(Debug,9, (&Debug, "StartSyncronize: Skippig cooked menu syncronize\n"));
	sync_ctx = ctx;
	return;
    }

    DPRINT(Debug,9, (&Debug, "StartSyncronize: Syncronize to %p\n",ctx));

    ctx->routine->wra_SyncEnter(ctx);
    sync_ctx = ctx;
}


void Syncronize(ctx)
     struct menu_context *ctx; 
{

    enter_terminal_hook();

    if (sync_ctx   == ctx)
	return;              /* Nothing need to do */

    if (MENU_CONTEXT_magic != ctx->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"Syncronize",
	      "Bad type magic number",0);

    if (MENU_DRAW_magic  != ctx->routine->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"Syncronize",
	      "Bad draw magic number",0);

    if (sync_ctx == &COOKED_MENU) {
	DPRINT(Debug,9, (&Debug, "Syncronize: Skippig cooked menu syncronize\n"));
	sync_ctx = NULL;	
    }


    if (sync_ctx) {
	if (MENU_CONTEXT_magic != sync_ctx->magic)
	    panic("SCREEN PANIC",__FILE__,__LINE__,"Syncronize",
	      "Bad type magic number",0);

	if (MENU_DRAW_magic  != sync_ctx->routine->magic)
	    panic("SCREEN PANIC",__FILE__,__LINE__,"Syncronize",
		  "Bad draw magic number",0);

	DPRINT(Debug,9, (&Debug, "Syncronize: Syncronize from %p\n",sync_ctx));

	sync_ctx->routine->wra_SyncLeave(sync_ctx);
    }

    if (ctx == &COOKED_MENU) {
	DPRINT(Debug,9, (&Debug, "Syncronize: Skippig cooked menu syncronize\n"));
	sync_ctx = ctx;
	return;
    }

    DPRINT(Debug,9, (&Debug, "Syncronize: Syncronize to %p\n",ctx));

    ctx->routine->wra_SyncEnter(ctx);
    sync_ctx  = ctx;
}

void menu_set_default(ctx)
     struct menu_context *ctx; 
{
    if (default_context == ctx)
	return;

    if (MENU_CONTEXT_magic != ctx->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_set_default",
	      "Bad type magic number",0);

    if (ctx->routine != &CURSES_ROUTINES)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_set_default",
	      "Not a main menu",0);

    enter_terminal_hook();

    SYNC(ctx);
    default_context = ctx;
    
    DPRINT(Debug,9, (&Debug, "menu_set_default: Clearing screen, and setting redraw flag\n"));
    cur_ClearScreen();
    ctx->redraw = 1;


    DPRINT(Debug,9, (&Debug, 
		     "menu_set_default: default context to %p\n",
		     default_context));
}

int get_tabspacing()
{
    DPRINT(Debug,49, (&Debug, "get_tabspacing()\n"));
    return cur_tabspacing;
}

#ifdef TERMIOS
static struct tf_state ttysig_state;

#endif

void cancel_set_ttysig_X()
{
    DPRINT(Debug,49, (&Debug, "cancel_set_ttysig_X()\n"));

    enter_terminal_hook();

#ifdef TERMIOS
    toggle_lflag(&ttysig_state,ISIG,0);
#endif
}

void cancel_reset_ttysig_X()
{
    DPRINT(Debug,49, (&Debug, "cancel_reset_ttysig_X()\n"));

    enter_terminal_hook();

#ifdef TERMIOS
    reset_lfag(&ttysig_state);
#endif
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
