static char rcsid[] = "@(#)$Id: snd_display.c,v 2.7 2018/06/27 13:55:50 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.7 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_sndmail.h"
#include "s_elm.h"

DEBUG_VAR(Debug,__FILE__,"mail");

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

/* Layout
 *               1         2         3         4         5
 *     012345678901234567890123456789012345678901234567890       
 * 0                      Mail Sending Screen
 * 1
 * 2   Server:   xxxx      (verified as: xxxxx)
 * 3
 * 4   Message:  xxxxx bytes    (limit: xxxxx bytes)     
 * 5
 * 6   Sender:   
 * 7        xxxxxxxxx@bbbbbb    xxxxxxxxxx     yyyyyyyyyyyyyy
 * 8
 * 9   Recipients:         (yy)
 * 10  #001 xxxxxxxxx@bbbbbb    xxxxxxxxxx     yyyyyyyyyyyyyy
 * *   ---

 * x   Mail Status:             xxxxxxxxxx     yyyyyyyyyyyyyy

 *
 */


static void sd_size_changed P_((struct sending_display *sd));
static void sd_do_redraw P_((struct sending_display *sd));


struct menu_anon_param {
    struct sending_display *sd;
};
enum { snd_disp_self,
       snd_COL_server,
       snd_COL_verified,
       snd_COL_address,
       snd_COL_status,
       snd_COL_message,
       snd_disp_COUNT };


#define SENDING_DISPLAY_magic	0xF207

struct sending_display {
    unsigned short magic;       /* SENDING_DISPLAY_magic */

    /* Pager information */

    struct menu_context *page;

    struct menu_context * top_area;
    struct menu_context * rcpt_area;
    struct menu_context * status_area;
    struct menu_context * prompt_area;

    /* self reference */
    struct menu_param  PARAM[snd_disp_COUNT+1];
    struct menu_anon_param self;

    enum snd_current {
	snd_current_none = 0,
	snd_current_size,
	snd_current_from,
	snd_current_rcpt,
	snd_current_status } current;

    /* Report information */

#define SD_ADDR_STATUS_magic	0xF208

    struct sd_addr_status {
	unsigned short magic;       /* SD_ADDR_STATUS_magic */

	struct string        * addr;
	enum msd_addr_status   status;
	struct string        * message;
    } * from_addr;

    enum snd_prompt_mode {
	snd_prompt_none      = 0,
	snd_prompt_sender,
	snd_prompt_recipient,
	snd_prompt_final, 
	snd_prompt_size  } prompt_mode;
    enum msd_can_continue  can_continue;
    int                    prompt_rcpt_index;
    int                    cur_line, cur_col;

    long                   size_info[NUM_msd_size_info];

    struct sd_addr_status **rcpt_addrs;
    int                     rcpt_addr_count;

    enum msd_data_status   data_status;
    struct string        * data_message;

    struct string        * con_info_text[ NUM_msd_connection_info ];
    
    unsigned int       need_redraw : 1;
    unsigned int       interactive : 1;

    int refcount;
} * new_sending_display(rcpt_count, interactive) 
    int rcpt_count;
    int interactive;
{
    int have_screen = RawState();


    struct sending_display * ret = safe_malloc(sizeof (*ret));
    enum msd_connection_info x;
    enum msd_size_info       xa;

    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)ret,sizeof (*ret));

    ret->page = new_menu_context();

    if (have_screen)
	menu_ClearScreen(ret->page);
    else {
	DPRINT(Debug,10, (&Debug,"new_sending_display: not on full screen mode\n"));
    }

    ret->from_addr    = NULL;
    ret->rcpt_addrs   = 0;
    ret->rcpt_addr_count = 0;
    
    if (rcpt_count > 0) {
	int i;
	ret->rcpt_addrs = safe_calloc(rcpt_count,
				      sizeof(ret->rcpt_addrs[0]));

	for (i = 0; i < rcpt_count; i++)
	    ret->rcpt_addrs[i] = NULL;

	ret->rcpt_addr_count = rcpt_count;
    }

    ret->data_status  = msd_datastat_none;
    ret->data_message = NULL;

    for (x = 0; x < NUM_msd_connection_info; x++)
	ret->con_info_text[x] = NULL;

    ret->top_area     = NULL;
    ret->rcpt_area    = NULL;
    ret->status_area  = NULL;
    ret->prompt_area  = NULL;

    /* just build self reference */

    ret->self.sd = ret;

    ret->PARAM[snd_disp_self].t    = mp_anon_param;
    ret->PARAM[snd_COL_server].t   = mp_integer;
    ret->PARAM[snd_COL_verified].t = mp_integer;
    ret->PARAM[snd_COL_address].t  = mp_integer;
    ret->PARAM[snd_COL_status].t   = mp_integer;
    ret->PARAM[snd_COL_message].t  = mp_integer;
    ret->PARAM[snd_disp_COUNT].t = mp_END;

    mp_list_set_anon(ret->PARAM,snd_disp_self,
		     & (ret->self ));
    mp_list_set_integer(ret->PARAM,snd_COL_server,10);
    mp_list_set_integer(ret->PARAM,snd_COL_verified,20);

    if (rcpt_count > 999)
	mp_list_set_integer(ret->PARAM,snd_COL_address,6);
    else if (rcpt_count > 99)
	mp_list_set_integer(ret->PARAM,snd_COL_address,5);
    else if (rcpt_count > 9)
	mp_list_set_integer(ret->PARAM,snd_COL_address,4);
    else
	mp_list_set_integer(ret->PARAM,snd_COL_address,3);

    mp_list_set_integer(ret->PARAM,snd_COL_status,25);
    mp_list_set_integer(ret->PARAM,snd_COL_message,40);

    ret->current = snd_current_none;

    ret->prompt_mode    = snd_prompt_none;
    ret->can_continue   =  msd_mail_failed;
    ret->cur_line       = 0;
    ret->cur_col        = 0;

    for (xa  = 0; xa < NUM_msd_size_info; xa++)
	ret->size_info[xa] = -1;
	
    ret->interactive = 0;
    if (interactive) 
	ret->interactive = 1;

    ret->need_redraw = 1;
    ret->refcount = 1;
    ret->magic    = SENDING_DISPLAY_magic;

    if (have_screen) {
	/* Sets ret->prompt_area and ret->screen_lines */
	sd_size_changed(ret);

	sd_do_redraw(ret);

	FlushBuffer();
    }

    DPRINT(Debug,25, (&Debug,
		      "new_sending_display=%p  rcpt_count=%d, interactive=%d\n",
		      ret,rcpt_count,interactive));

    return ret;
}

static void sd_set_current P_((struct sending_display *sd,
			       enum snd_current  snd_current,
			       int rcpt_idx));
static void sd_set_current(sd,snd_current,rcpt_idx)
     struct sending_display * sd;
     enum snd_current         snd_current;
     int rcpt_idx;
{
    if (SENDING_DISPLAY_magic != sd->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "sd_set_current",
	      "Bad magic number",0);

    sd->current = snd_current;

    if (sd->rcpt_area) {
	switch (snd_current) {
	case snd_current_rcpt:
	    menu_header_change(sd->rcpt_area,
			       header_current,
			       rcpt_idx);
	    break;
	default:
	    menu_header_change(sd->rcpt_area,
			       header_current,
			       HEADER_NO_CURRENT);
	    break;
	}
    }
}

static void center_headers P_((struct menu_context * P,int idx));
static void center_headers(P,idx) 
     struct menu_context * P;
     int idx;
{
    int LINES, COLUMNS;
    int top = menu_header_get(P,header_top_line);
    
    menu_get_sizes(P,&LINES, &COLUMNS);

    if (idx < top || idx >= top + LINES) {
	top = idx - LINES/2;
	if (top < 0)
	    top = 0;

	menu_header_change(P,header_top_line,top);
    } 
}


static int scale P_((int col,int columns));
static int scale(col,columns)
     int col;
     int columns;
{
    if (columns > 75)

	return col * columns / 75;
    return col;

}

static void relocate_columns  P_((struct sending_display *sd));
static void relocate_columns(sd)
     struct sending_display *sd;
{
    int LINES, COLUMNS;

    int COL_verified = 20;
    int	COL_status   = 25;
    int COL_message  = 40;

    if (SENDING_DISPLAY_magic != sd->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "relocate_prompt_area",
	      "Bad magic number",0);
    
    menu_get_sizes(sd->page,&LINES, &COLUMNS);

    COL_verified = scale(COL_verified,COLUMNS);
    COL_status   = scale(COL_status,COLUMNS);
    COL_message  = scale(COL_message,COLUMNS);
    
    mp_list_set_integer(sd->PARAM,
			snd_COL_verified,
			COL_verified);
    mp_list_set_integer(sd->PARAM,
			snd_COL_status,
			COL_status);
    mp_list_set_integer(sd->PARAM,
			snd_COL_message,
			COL_message);
}

S_(subpage_simple_redraw sb_update_top_area)
static int sb_update_top_area P_((struct menu_context  *ptr,
			struct menu_param *list));

S_(subpage_simple_redraw sb_update_status_area)
static int sb_update_status_area P_((struct menu_context  *ptr,
				     struct menu_param *list));

S_(header_line_redraw sb_update_rcpt_line)
static void sb_update_rcpt_line P_((struct menu_context  *ptr,
				    struct menu_param *list,	   
				    int line_number,
				    int index,
				    int is_current));


S_(subpage_simple_redraw sb_update_prompt_area)
static int sb_update_prompt_area P_((struct menu_context  *ptr,
				     struct menu_param *list));


static void sd_size_changed(sd)
     struct sending_display *sd;
{
    int LINES, COLUMNS;
    int prompt_start;
    int status_start;

    int rcpt_start;

    if (SENDING_DISPLAY_magic != sd->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "sd_size_changed",
	      "Bad magic number",0);

    relocate_columns(sd);

    menu_get_sizes(sd->page,&LINES, &COLUMNS);
    prompt_start = LINES;   /* no prompt area */

    if (LINES > 8 && sd->interactive) {
		
	prompt_start = LINES-4;

	if (! sd->prompt_area)

	    sd->prompt_area = 
		new_menu_subpage(sd->page,prompt_start,4,
				 sb_update_prompt_area,
				 sd->PARAM);	
	else
	    menu_subpage_relocate(sd->prompt_area,sd->page,
				  prompt_start,4);
    } else if (sd->prompt_area) 
	erase_menu_context(& (sd->prompt_area));

    status_start = prompt_start;

    if (LINES > 3) {
	int w = 1;

	if (LINES > 5)
	    w = 3;
	else if (LINES > 4)
	    w = 2;

	status_start = prompt_start-w;

	if (! sd->status_area)
	    sd->status_area =
		new_menu_subpage(sd->page,status_start,w,
				 sb_update_status_area,
				 sd->PARAM);
	else
	    menu_subpage_relocate(sd->status_area,sd->page,
				  status_start,w);
    } else if (sd->status_area)
	erase_menu_context(& (sd->status_area));


    /* calculate space used for top area */
    rcpt_start = 2;
    if (sd->con_info_text[msd_server] ||
	sd->con_info_text[msd_verified_as])
	rcpt_start += 2;

    if (sd->size_info[msd_message_size] >= 0 ||
	sd->size_info[msd_size_limit] >= 0)
	rcpt_start += 2;

    if (sd->from_addr)
	rcpt_start += 3;
    if (0 < sd->rcpt_addr_count)
	rcpt_start++;

    if (rcpt_start >= status_start) {
	if (status_start > 1)
	    rcpt_start = status_start-1;
	else
	    rcpt_start = status_start;
    }

    if (rcpt_start > 0) {
	if (! sd->top_area)
	    sd->top_area =
		new_menu_subpage(sd->page,0,rcpt_start,
				 sb_update_top_area,
				 sd->PARAM);
	else
	    menu_subpage_relocate(sd->top_area,sd->page,
				  0,rcpt_start);
    } else if (sd->top_area)
	erase_menu_context(& (sd->top_area));


    if (rcpt_start > 0 && rcpt_start < status_start) {
	if (! sd->rcpt_area) {
	    sd->rcpt_area = 
		new_menu_header(sd->page,rcpt_start,
				status_start - rcpt_start,
				sb_update_rcpt_line,
				sb_update_rcpt_line,
				null_header_param_changed,
				header_line_noredraw,
				null_header_line_separator_index,
				header_separator_noredraw,
				null_header_separator_start,
				header_setup_noinit,
				header_setup_noline,
				sd->PARAM);
	    menu_header_change(sd->rcpt_area,header_current,
			       HEADER_NO_CURRENT);

	} else
	    menu_header_relocate(sd->rcpt_area,
				 sd->page,
				 rcpt_start,
				 status_start - rcpt_start);	
    } else if (sd->rcpt_area)
	erase_menu_context(& (sd->rcpt_area));	    
}

static void handle_prompt_area P_((struct sending_display *sd));
static void handle_prompt_area(sd)
     struct sending_display *sd;
{
    if (SENDING_DISPLAY_magic != sd->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "handle_prompt_area",
	      "Bad magic number",0);

    if (sd->prompt_area) {
	if (sd->need_redraw)
	    menu_trigger_redraw(sd->prompt_area);

	if (menu_resized(sd->prompt_area)) {
	    DPRINT(Debug,1, (&Debug, 
			     "handle_prompt_area: prompt area resized\n"));
	}

	if (menu_need_redraw(sd->prompt_area)) {
	    DPRINT(Debug,1, (&Debug, "handle_prompt_area: prompt area redraw???\n"));
	    sb_update_prompt_area(sd->prompt_area,sd->PARAM);
	    
	}    
    }

    show_last_error();	
}



static struct sd_addr_status * malloc_sd_addr_status P_((struct string  *addr,
							 enum msd_addr_status   status,
							 struct string        * message));
static struct sd_addr_status * malloc_sd_addr_status(addr,status,message)
     struct string      * addr;
     enum msd_addr_status status;
     struct string      * message;
{
    struct sd_addr_status * ret =  safe_malloc(sizeof (*ret));

    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)ret,sizeof (*ret));

    ret->addr    = addr ? dup_string(addr) : NULL;
    ret->status  = status;
    ret->message = message ? dup_string(message) : NULL;

    ret->magic = SD_ADDR_STATUS_magic;
    return ret;
}

static void update_sd_addr_status P_((struct sd_addr_status *st,
				      struct string        * addr,
				      enum msd_addr_status   status,
				      struct string        * message));
static void update_sd_addr_status(st,addr,status,message)
     struct sd_addr_status *st;
     struct string  *addr;
     enum msd_addr_status   status;
     struct string        * message;
{
    if (SD_ADDR_STATUS_magic != st->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,"free_sd_addr_status",
	      "Bad magic number",0);

    if (st->addr != addr) {
	if (st->addr)
	    free_string(& (st->addr));
	if (addr)
	    st->addr = dup_string(addr);
    }

    st->status  = status;

    if (st->message)
	free_string(& (st->message));
    if (message)
	st->message = dup_string(message);
}

static void free_sd_addr_status P_((struct sd_addr_status **st));
static void free_sd_addr_status(st)
     struct sd_addr_status **st;
{
    if (SD_ADDR_STATUS_magic != (*st)->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,"free_sd_addr_status",
	      "Bad magic number",0);

    if ((*st)->addr)
	free_string(& ((*st)->addr));
    if ((*st)->message)
	free_string(& ((*st)->message));

    (*st)->magic = 0;  /* Invalidate */
    free(*st);
    *st = NULL;    
}

E_(msd_free_sending_display_cb free_sending_display)
void free_sending_display(sd)
   struct sending_display **sd;
{
    enum msd_connection_info x;
    
    if (SENDING_DISPLAY_magic != (*sd)->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,"free_sending_display",
	      "Bad magic number",0);

    DPRINT(Debug,25, (&Debug,
		      "free_sending_display: *sd=%p  refcount=%d\n",
		      *sd,(*sd)->refcount));

    if ((*sd)->refcount < 1)
        panic("COMMANDS PANIC",__FILE__,__LINE__,"free_sending_display",
              "Bad refcount",0);

    (*sd)->refcount--;

    if ((*sd)->refcount > 0) {   /* Do not free */

        (*sd) = NULL;            /* Refefence count for this pointer is */
				 /* decremented, so this must have be reset */
        return;
    }
    
    if ((*sd)->from_addr)
	free_sd_addr_status(& ((*sd)->from_addr));
    
    if ((*sd)->rcpt_addrs) {
	int i;

	for (i = 0; i < (*sd)->rcpt_addr_count; i++) 
	    if ((*sd)->rcpt_addrs[i])
		free_sd_addr_status(& ((*sd)->rcpt_addrs[i]));

	free((*sd)->rcpt_addrs);
	(*sd)->rcpt_addrs = NULL;
    }
    (*sd)->rcpt_addr_count = 0;

    if ((*sd)->data_message)
	free_string(& ((*sd)->data_message));

    for (x = 0; x < NUM_msd_connection_info; x++)
	if ((*sd)->con_info_text[x])
	    free_string(& ((*sd)->con_info_text[x]));

    if ((*sd)->top_area)
	erase_menu_context(& ((*sd)->top_area));
    if ((*sd)->rcpt_area)
	erase_menu_context(& ((*sd)->rcpt_area));
    if ((*sd)->status_area)
	erase_menu_context(& ((*sd)->status_area));
    if ((*sd)->prompt_area)
	erase_menu_context(& ((*sd)->prompt_area));
    if ((*sd)->page)
	erase_menu_context(& ((*sd)->page));
   
    (*sd)->magic = 0;   /* Invalidate */
    free(*sd);
    *sd = NULL;
}

E_(msd_inc_sending_display_cb inc_sending_display_refcount)
void inc_sending_display_refcount(sd)
   struct sending_display *sd;
{
    if (SENDING_DISPLAY_magic != sd->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "inc_sending_display_refcount",
	      "Bad magic number",0);

    DPRINT(Debug,25, (&Debug,
		      "inc_sending_display_refcount: sd=%p old refcount=%d\n",
		      sd,sd->refcount));

    sd->refcount++;
}


/* ---------------------------------------------------------------------- */

static void sb_update_column P_((struct menu_context    * ptr,
				 int line,
				 int start_column,
				 int next_column,
				 struct string * text));
static void sb_update_column(ptr,line,start_column,next_column,text)
     struct menu_context    * ptr;
     int line;
     int start_column;
     int next_column;
     struct string * text;
{
    int POS = 0;
    int visible_len = 0;
    int len = string_len(text);
    int add_dots = 0;
    int limit = next_column - start_column;
    struct string * temp = 
	curses_printable_clip(text,
			      &POS,len,&visible_len,
			      limit);

    if (POS < len) {
	POS = 0;
	free_string(&temp);
    
	temp = curses_printable_clip(text,
				     &POS,len,
				     &visible_len,
				     limit-3);	
	add_dots = 1;
    }

    menu_PutLineX(ptr,line,start_column,
		  FRM("%S"),
		  temp);

    if (add_dots) {

	menu_StartXX(ptr,pg_BOLD);
	menu_PutLine0(ptr,line,start_column+visible_len,"...");
	menu_EndXX(ptr,pg_BOLD);

    }


    free_string(&temp);
}

/* returns next line */

static int sb_update_server_line P_((struct menu_context    * ptr,
				     struct menu_param      * list,
				     struct sending_display * sd,
				     int line));
static int sb_update_server_line(ptr,list,sd,line) 
     struct menu_context    * ptr;
     struct menu_param      * list;
     struct sending_display * sd;
     int line;
{
    int LINES, COLUMNS;
    int COL_server   =
	mp_lookup_integer(list,snd_COL_server);
    int COL_verified = 
	mp_lookup_integer(list,snd_COL_verified);
    int add_line = 0;


    if (SENDING_DISPLAY_magic != sd->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "sb_update_server_line",
	      "Bad magic number",0);

    menu_get_sizes(ptr,&LINES, &COLUMNS);

    if (sd->con_info_text[msd_server]) {
	int next_column = COLUMNS;

	add_line = 2;

	menu_StartXX(ptr,pg_BOLD);
	menu_PutLineX(ptr,line,0,
		      CATGETS(elm_msg_cat, ElmSet,
			      ElmSndServer,
			      "Server: "));
	menu_EndXX(ptr,pg_BOLD);

	if (sd->con_info_text[msd_verified_as] &&
	    COL_verified < COLUMNS)
	    next_column = COL_verified -1;
	
	sb_update_column(ptr,line,COL_server,next_column,
			 sd->con_info_text[msd_server]);
	
	if (sd->con_info_text[msd_verified_as]) {

	    /* Optimize display */
	    
	    if (0 == string_cmp(sd->con_info_text[msd_server],
				sd->con_info_text[msd_verified_as],99))  {

		menu_StartXX(ptr,pg_BOLD);
		menu_PutLineX(ptr,line,COL_verified,
			      CATGETS(elm_msg_cat, ElmSet,
					     ElmSndVerified,
					     "(verified)"));
		menu_EndXX(ptr,pg_BOLD);
		
	    } else {
		int lin,col;

		menu_StartXX(ptr,pg_BOLD);
		menu_PutLineX(ptr,line,COL_verified,
			      CATGETS(elm_msg_cat, ElmSet,
				      ElmSndVerifiedAs,
				      "(verified as: "));
		menu_EndXX(ptr,pg_BOLD);

		menu_GetXYLocation(ptr,&lin,&col);

		if (lin != line)
		    goto fail_line;

		sb_update_column(ptr,line,col,COLUMNS-1,
				 sd->con_info_text[msd_verified_as]);

		menu_StartXX(ptr,pg_BOLD);
		menu_Writechar(ptr,')');
		menu_EndXX(ptr,pg_BOLD);

	    }
	}

    } else if (sd->con_info_text[msd_verified_as]) {

	add_line = 2;

	menu_StartXX(ptr,pg_BOLD);
	menu_PutLineX(ptr,line,0,
		      CATGETS(elm_msg_cat, ElmSet,
			      ElmSndVerifiedAs2,
			      "Verified as: "));	
	menu_EndXX(ptr,pg_BOLD);

	sb_update_column(ptr,line,COL_server,COLUMNS,
			 sd->con_info_text[msd_verified_as]);
    }

 fail_line:
    return line + add_line;
}

static struct string * format_bytes P_((struct sending_display * sd,
					enum msd_size_info idx));
static struct string * format_bytes(sd,idx)
     struct sending_display * sd;
     enum msd_size_info idx;
{
    struct string * text = NULL;
    long size;

    if (SENDING_DISPLAY_magic != sd->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "format_bytes",
	      "Bad magic number",0);

    if (idx < 0 || idx >= NUM_msd_size_info)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "format_bytes",
	      "Bad info",0);

    size = sd->size_info[idx];

    if (size > 10000000)
	text = format_string(CATGETS(elm_msg_cat, ElmSet,
				     ElmSndSizeMBytes,
				     "%ld megabytes"),
			     size / 1000000); 
    else if (size > 10000)
	text = format_string(CATGETS(elm_msg_cat, ElmSet,
				     ElmSndSizeKBytes,
				     "%ld kilobytes"),
			     size / 1000); 
    else	
	text = format_string(CATGETS(elm_msg_cat, ElmSet,
				     ElmSndSizeBytes,
				     "%ld bytes"),
			     size);
    
    return text;
}

static int sb_update_size_line P_((struct menu_context    * ptr,
				     struct menu_param      * list,
				     struct sending_display * sd,
				     int line));
static int sb_update_size_line(ptr,list,sd,line) 
     struct menu_context    * ptr;
     struct menu_param      * list;
     struct sending_display * sd;
     int line;
{
    int LINES, COLUMNS;
    int COL_server   =
	mp_lookup_integer(list,snd_COL_server);
    int COL_verified = 
	mp_lookup_integer(list,snd_COL_verified);
    int COL_message =
	mp_lookup_integer(list,snd_COL_message);
    int add_line = 0;
    
    if (SENDING_DISPLAY_magic != sd->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "sb_update_size_line",
	      "Bad magic number",0);

    menu_get_sizes(ptr,&LINES, &COLUMNS);

    if (sd->size_info[msd_message_size] >= 0) {
	int next_column = COLUMNS;
	struct string * text = NULL;

	add_line = 2;

	menu_StartXX(ptr,pg_BOLD);
	menu_PutLineX(ptr,line,0,
		      CATGETS(elm_msg_cat, ElmSet,
			      ElmSndMessage,
			      "Message: "));
	menu_EndXX(ptr,pg_BOLD);

	if (sd->size_info[msd_size_limit] >= 0 &&
	    COL_message < COLUMNS)
	    next_column = COL_message -1;

	text = format_bytes(sd,msd_message_size);
       
	sb_update_column(ptr,line,COL_server,next_column,
			 text);

	free_string(&text);

	if (sd->size_info[msd_size_limit] >= 0) {
	    int lin,col;

	    menu_StartXX(ptr,pg_BOLD);
	    menu_PutLineX(ptr,line,COL_message,
			  CATGETS(elm_msg_cat, ElmSet,
				  ElmSndLimit,
				  "(limit: "));
	    menu_EndXX(ptr,pg_BOLD);

	    menu_GetXYLocation(ptr,&lin,&col);

	    if (lin != line)
		goto fail_line;

	    text = format_bytes(sd,msd_size_limit);

	    if (snd_current_size == sd->current)
		    menu_StartXX(ptr,pg_BOLD);
		
	    sb_update_column(ptr,line,col,COLUMNS-1,
			     text);
	    free_string(&text);

	    if (snd_current_size == sd->current)
		menu_EndXX(ptr,pg_BOLD);

	    menu_StartXX(ptr,pg_BOLD);
	    menu_Writechar(ptr,')');
	    menu_EndXX(ptr,pg_BOLD);

	}

    } else if (sd->size_info[msd_size_limit] >= 0) {
	struct string * text = NULL;
	    
	add_line = 2;

	menu_StartXX(ptr,pg_BOLD);
	menu_PutLineX(ptr,line,0,
		      CATGETS(elm_msg_cat, ElmSet,
			      ElmSndMesSizeLimit,
			      "Message size limit: "));
	menu_EndXX(ptr,pg_BOLD);
	
	text = format_bytes(sd,msd_size_limit);

	if (snd_current_size == sd->current)
	    menu_StartXX(ptr,pg_BOLD);

	sb_update_column(ptr,line,COL_verified,COLUMNS,
			 text);

	free_string(&text);

	if (snd_current_size == sd->current)
	    menu_EndXX(ptr,pg_BOLD);
    }

 fail_line:
    return line + add_line;
}

static void sb_update_addr_line P_((struct menu_context    * ptr,
				   struct menu_param      * list,
				   int line,
				   struct sd_addr_status * addr,
				   int is_current));
static void sb_update_addr_line(ptr,list,line,addr,is_current)
     struct menu_context    * ptr;
     struct menu_param      * list;
     int line;
     struct sd_addr_status * addr;
     int is_current;
{
   int LINES, COLUMNS;
    int COL_address  =
	mp_lookup_integer(list,snd_COL_address);
    int COL_status  =
	mp_lookup_integer(list,snd_COL_status);
    int COL_message =
	mp_lookup_integer(list,snd_COL_message);


    if (SD_ADDR_STATUS_magic != addr->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "sb_update_addr_line",
	      "Bad magic number (sd_addr_status)",0);

    menu_get_sizes(ptr,&LINES, &COLUMNS);

    if (addr->addr) {
	int next_column = COLUMNS;
	
	if (addr->message &&
	    COL_message < COLUMNS)
	    next_column = COL_message - 1;
	
	if (msd_addrstat_none != addr->status &&
	    COL_status < COLUMNS && COL_status < COL_message)
	    next_column = COL_status - 1;

	sb_update_column(ptr,line,COL_address,next_column,
			 addr->addr);
    }
    
    if (is_current)
	menu_StartXX(ptr,pg_BOLD);

    switch (addr->status) {
	    
    case msd_addr_tmpfail: 
	menu_PutLineX(ptr,line,COL_status,
		      CATGETS(elm_msg_cat, ElmSet,
			      ElmSndTempFail,
			      "failure (temporary) "));
	break;
	
    case msd_addr_fail:
	menu_PutLineX(ptr,line,COL_status,
		      CATGETS(elm_msg_cat, ElmSet,
			      ElmSndFail,
			      "failure "));
	break;
	
    case msd_addrstat_none:
	break;
	
    case msd_sending_addr:
	menu_PutLineX(ptr,line,COL_status,
		      CATGETS(elm_msg_cat, ElmSet,
			      ElmSndSending,
			      "sending "));
	break;
	
    case msd_addr_ok:
	menu_PutLineX(ptr,line,COL_status,
		      CATGETS(elm_msg_cat, ElmSet,
			      ElmSndOK,
			      "OK "));
	break;	        	    
    }
    
    if (is_current)
	menu_EndXX(ptr,pg_BOLD);

    if (addr->message) 
	sb_update_column(ptr,line,COL_message,COLUMNS,
			 addr->message);
}


static int sb_update_sender_lines P_((struct menu_context    * ptr,
				      struct menu_param      * list,
				      struct sending_display * sd,
				      int line));
static int sb_update_sender_lines(ptr,list,sd,line)
     struct menu_context    * ptr;
     struct menu_param      * list;
     struct sending_display * sd;
     int line;
{
    int LINES, COLUMNS;
#if 0
    int COL_address  =
	mp_lookup_integer(list,snd_COL_address);
    int COL_status  =
	mp_lookup_integer(list,snd_COL_status);
    int COL_message =
	mp_lookup_integer(list,snd_COL_message);
#endif
    int add_line = 0;

    if (SENDING_DISPLAY_magic != sd->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "sb_update_sender_lines",
	      "Bad magic number",0);

    menu_get_sizes(ptr,&LINES, &COLUMNS);

    if (sd->from_addr) {
	add_line = 3;

	menu_StartXX(ptr,pg_BOLD);
	menu_PutLineX(ptr,line,0,CATGETS(elm_msg_cat, ElmSet,
					 ElmSndSender,
					 "Sender: "));
	menu_EndXX(ptr,pg_BOLD);

	sb_update_addr_line(ptr,list,line+1,
			    sd->from_addr,
			    snd_current_from == sd->current);
	
    }
    
    return line + add_line;
}


static int sb_update_recipients_header P_((struct menu_context    * ptr,
					   struct menu_param      * list,
					   struct sending_display * sd,
					   int line));
static int sb_update_recipients_header(ptr,list,sd,line)
     struct menu_context    * ptr;
     struct menu_param      * list;
     struct sending_display * sd;
     int line;
{
    int COL_verified = 
	mp_lookup_integer(list,snd_COL_verified);
    int add_line = 0;

    if (SENDING_DISPLAY_magic != sd->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "sb_update_recipients_header",
	      "Bad magic number",0);

    if (1 == sd->rcpt_addr_count) {
	add_line = 1;

	menu_StartXX(ptr,pg_BOLD);
	menu_PutLineX(ptr,line,0,
		      CATGETS(elm_msg_cat, ElmSet,
			      ElmSndRecipient,
			      "Recipient: "));
	menu_EndXX(ptr,pg_BOLD);

    } else if (1 < sd->rcpt_addr_count) {
	add_line = 1;

	menu_StartXX(ptr,pg_BOLD);
	menu_PutLineX(ptr,line,0,
		      CATGETS(elm_msg_cat, ElmSet,
			      ElmSndRecipients,
			      "Recipients: "));	
	menu_EndXX(ptr,pg_BOLD);

	menu_PutLineX(ptr,line,COL_verified,
		      FRM("(%d)"),
		      sd->rcpt_addr_count);
    }

    return line + add_line;
}

/* Returns 1 if redraw not need to be signaled */
static int sb_update_top_area(ptr,list)
     struct menu_context  *ptr;
     struct menu_param *list;
{
    struct menu_anon_param * anon_disp_self  = 
	mp_lookup_anon(list,snd_disp_self);
    struct sending_display * sd = anon_disp_self->sd;
#if 0
    int COL_verified = 
	mp_lookup_integer(list,snd_COL_verified);
    int	COL_status   = 
	mp_lookup_integer(list,snd_COL_status);
    int COL_message  = 
	mp_lookup_integer(list,snd_COL_message);
#endif
    int line;

    if (SENDING_DISPLAY_magic != sd->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "sb_update_top_area",
	      "Bad magic number",0);

    DPRINT(Debug,25, (&Debug,
		      "sb_update_top_area: ptr=%p\n",
		      ptr));

    menu_ClearScreen(ptr);
    
    menu_StartXX(ptr,pg_BOLD);
    menu_print_format_center(ptr,0,
			     CATGETS(elm_msg_cat, ElmSet,
				     ElmSndScreenTitle, 
				     "Mail Sending Screen"));

    menu_EndXX(ptr,pg_BOLD);

    line = 2;

    line = sb_update_server_line(ptr,list,sd,line);
    line = sb_update_size_line(ptr,list,sd,line);
    line = sb_update_sender_lines(ptr,list,sd,line);
    line = sb_update_recipients_header(ptr,list,sd,line);

    return 1;
}


static int sb_update_status_area(ptr,list)
     struct menu_context  *ptr;
     struct menu_param *list;
{
    int LINES, COLUMNS;
    struct menu_anon_param * anon_disp_self  = 
	mp_lookup_anon(list,snd_disp_self);
    struct sending_display * sd = anon_disp_self->sd;
#if 0
    int COL_verified = 
	mp_lookup_integer(list,snd_COL_verified);
#endif
    int	COL_status   = 
	mp_lookup_integer(list,snd_COL_status);
    int COL_message  = 
	mp_lookup_integer(list,snd_COL_message);
    int line = 0;


    if (SENDING_DISPLAY_magic != sd->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "sb_update_status_area",
	      "Bad magic number",0);

    DPRINT(Debug,25, (&Debug,
		      "sb_update_status_area: ptr=%p\n",
		      ptr));

    menu_ClearScreen(ptr);

    menu_get_sizes(ptr,&LINES, &COLUMNS);
    
    if (LINES > 2)
	line = 1;

    menu_StartXX(ptr,pg_BOLD);
    menu_PutLineX(ptr,line,0,CATGETS(elm_msg_cat, ElmSet,
				  ElmSndMailStatus,
				  "Mail Status: "));
    menu_EndXX(ptr,pg_BOLD);
    
    if (snd_current_status == sd->current)
	menu_StartXX(ptr,pg_BOLD);

    switch (sd->data_status) {

    case msd_mail_tmpfail: 
	menu_PutLineX(ptr,line,COL_status,
		      CATGETS(elm_msg_cat, ElmSet,
			      ElmSndTempFail,
			      "failure (temporary) "));
	break;

    case msd_mail_fail:
	menu_PutLineX(ptr,line,COL_status,
		      CATGETS(elm_msg_cat, ElmSet,
			      ElmSndFail,
			      "failure "));
	break;

    case msd_mail_backgrounded:
	menu_PutLineX(ptr,line,COL_status,
		      CATGETS(elm_msg_cat, ElmSet,
			      ElmSndBackgrounded,
			      "in background "));
	break;

    case msd_calling_mailer:
	menu_PutLineX(ptr,line,COL_status,
		      CATGETS(elm_msg_cat, ElmSet,
			      ElmSndCallingMailer,
			      "calling mailer "));
	break;

    case msd_datastat_none:
	break;

    case msd_sending_data:
	menu_PutLineX(ptr,line,COL_status,
		      CATGETS(elm_msg_cat, ElmSet,
			      ElmSndSendingData,
			      "sending DATA "));
	break;
	
    case  msd_data_ok:
	menu_PutLineX(ptr,line,COL_status,
		      CATGETS(elm_msg_cat, ElmSet,
			      ElmSndDataOK,
			      "DATA OK (sending body) "));
	break;
	
    case msd_mail_sent:
	menu_PutLineX(ptr,line,COL_status,
		      CATGETS(elm_msg_cat, ElmSet,
			      ElmSndMailSent,
			      "mail sent "));
	break;
    }

    if (snd_current_status == sd->current)
	menu_EndXX(ptr,pg_BOLD);

    if (sd->data_message) {
	sb_update_column(ptr,line,COL_message,COLUMNS,
			 sd->data_message);
    }

    return 1;  /* Completed */
}

static void sb_update_rcpt_line(ptr,list,line_number,index,is_current)
     struct menu_context  * ptr;
     struct menu_param    * list;	   
     int                    line_number;
     int                    index;
     int                    is_current;
{
    int LINES, COLUMNS;
    struct menu_anon_param * anon_disp_self  = 
	mp_lookup_anon(list,snd_disp_self);
    struct sending_display * sd = anon_disp_self->sd;
    int COL_address  =
	mp_lookup_integer(list,snd_COL_address);
#if 0
    int COL_status  =
	mp_lookup_integer(list,snd_COL_status);
    int COL_message =
	mp_lookup_integer(list,snd_COL_message);
#endif

    if (SENDING_DISPLAY_magic != sd->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "sb_update_rcpt_line",
	      "Bad magic number",0);
    
    DPRINT(Debug,25, (&Debug,
		      "sb_update_rcpt_line: ptr=%p line_number=%d index=%d%s\n",
		      ptr,line_number,index,
		      is_current ? " is current" : ""));


    menu_get_sizes(ptr,&LINES, &COLUMNS);

    menu_ClearLine(ptr,line_number);

    if (index >= 0 && index < sd->rcpt_addr_count &&
	sd->rcpt_addrs[index]) {

	menu_PutLineX(ptr,line_number,0,
		      FRM("#%0*d "),
		      COL_address-2,
		      index+1);
	
	sb_update_addr_line(ptr,list,line_number,
			    sd->rcpt_addrs[index],
			    is_current);
    }
}

/* ------------------------------------------------------------- */

static void sd_do_check P_((struct sending_display *sd));
static void sd_do_check(sd)
     struct sending_display *sd;
{
    int recheck = 0;

    if (SENDING_DISPLAY_magic != sd->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "sd_do_check",
	      "Bad magic number",0);

    do {
	recheck = 0;

	if (sd->top_area) {
	    if (menu_resized(sd->top_area)) {
		DPRINT(Debug,1, (&Debug, "top area resized\n"));
	    }
	    
	    if (menu_need_redraw(sd->top_area)) {
		DPRINT(Debug,1, (&Debug, "title page redraw???\n"));
		
		sb_update_top_area(sd->top_area,sd->PARAM);
	    }
	}
	
	if (sd->rcpt_area) {
	    if (menu_resized(sd->rcpt_area)) {
		DPRINT(Debug,1, (&Debug, "rcpt area resized\n"));
	    }
	    
	    if (menu_need_redraw(sd->rcpt_area)) {
		DPRINT(Debug,1, (&Debug, "rcpt area redraw???\n"));
		
		menu_ClearScreen(sd->rcpt_area);
	    }
	}
	
	if (sd->status_area) {
	    if (menu_resized(sd->status_area)) {
		DPRINT(Debug,1, (&Debug, "status area resized\n"));
		
	    }
	    
	    if (menu_need_redraw(sd->status_area)) {
		DPRINT(Debug,1, (&Debug, "status area redraw???\n"));
		
		sb_update_status_area(sd->status_area,sd->PARAM);				  
	    }
	}

	handle_prompt_area(sd);

	if (menu_resized(sd->page)) {
	    DPRINT(Debug,1, (&Debug, "page resized\n"));

	    sd_size_changed(sd);
	    recheck = 1;
	}

	if (menu_need_redraw(sd->page)) {
	    DPRINT(Debug,1, (&Debug, "page redraw???\n"));
	    sd->need_redraw = 1;
	}
	
	if (sd->need_redraw) {
	    sd->need_redraw = 0;

	    DPRINT(Debug,1, (&Debug, "redrawing children\n"));
	    menu_ClearScreen(sd->page);
	    menu_redraw_children(sd->page);
	    recheck = 1;
	}

    } while (recheck);

}


static void sd_do_redraw(sd)
     struct sending_display *sd;
{
    if (SENDING_DISPLAY_magic != sd->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "sd_do_redraw",
	      "Bad magic number",0);

    sd->need_redraw = 1;
    sd_do_check(sd);
}


E_(msd_update_addr_cb update_addr_sd)
extern void update_addr_sd(sd,idx,addr,status,message)
    struct sending_display *sd;
    int                    idx;
    struct string        * addr;
    enum msd_addr_status   status;
    struct string        * message /* server */;
{
    /* Is this needed */
    int have_screen = RawState();

    if (SENDING_DISPLAY_magic != sd->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "update_addr_sd",
	      "Bad magic number",0);

    DPRINT(Debug,10, (&Debug,
		      "update_addr_sd: sd=%p idx=%d %s",
		      sd,idx,
		      idx >= 0 ? "(rcpt)" : "(from)"));
    if (addr) {
	DPRINT(Debug,10, (&Debug,", addr=%S",addr));
    }

    DPRINT(Debug,10, (&Debug,", status=%d ",status));
    switch (status) {
    case msd_addr_tmpfail:  
	DPRINT(Debug,10, (&Debug,"msd_addr_tmpfail"));
	break;
    case msd_addr_fail:
	DPRINT(Debug,10, (&Debug,"msd_addr_fail"));
	break;
    case  msd_addrstat_none:
	DPRINT(Debug,10, (&Debug,"msd_addrstat_none"));
	break;
    case msd_sending_addr:
	DPRINT(Debug,10, (&Debug,"msd_sending_addr"));
	break;
    case msd_addr_ok:
	DPRINT(Debug,10, (&Debug,"msd_addr_ok"));
	break;
    }
    if (message) {
	DPRINT(Debug,10, (&Debug,", message=\"%S\"",message));
    }
    if (!have_screen) {
	DPRINT(Debug,10, (&Debug,"; not on full screen mode"));
	
    }
    DPRINT(Debug,10, (&Debug,"\n"));
	

    if (idx < 0) {
	int flag = 0;

	if (sd->from_addr) {
	    update_sd_addr_status(sd->from_addr,addr,status,message);
	    
	} else {
	    sd->from_addr = malloc_sd_addr_status(addr,status,message);

	    if (have_screen)  
		sd_size_changed(sd);

	    flag = 1;
	}

	
	if (have_screen) { 

	    if (sd->rcpt_area && flag)
		menu_trigger_redraw(sd->rcpt_area);

	    if (sd->top_area)
		menu_trigger_redraw(sd->top_area);

	}
	

    } else {
	int flag = 0;
    
	if (idx >= sd->rcpt_addr_count) {
	    sd->rcpt_addrs = safe_array_realloc(sd->rcpt_addrs,
						(idx+1),
						sizeof(sd->rcpt_addrs[0]));
	    
	    while (sd->rcpt_addr_count <= idx) 
		sd->rcpt_addrs[sd->rcpt_addr_count++] = NULL;

	    if (have_screen) 
		sd_size_changed(sd);

	    flag = 1;
	}
	
	if (sd->rcpt_addrs[idx])
	    update_sd_addr_status(sd->rcpt_addrs[idx],addr,status,message);
	else {
	    sd->rcpt_addrs[idx] = malloc_sd_addr_status(addr,status,message);	    

	    if (have_screen) {
		sd_size_changed(sd);
		if (sd->rcpt_area)
		    menu_trigger_redraw(sd->rcpt_area);
	    }
	}
	
	if (have_screen)  {
	    if (flag && sd->top_area)
		menu_trigger_redraw(sd->top_area);

	    if (sd->rcpt_area) {
		menu_header_change_item(sd->rcpt_area,idx);
		center_headers(sd->rcpt_area,idx);
	    }
	}
    }

    if (have_screen) {
	sd_do_check(sd);
	FlushBuffer();
    }
}

static int sb_update_prompt_area(ptr,list) 
     struct menu_context  *ptr;
     struct menu_param *list;
{
    struct menu_anon_param * anon_disp_self  = 
	mp_lookup_anon(list,snd_disp_self);
    struct sending_display * sd = anon_disp_self->sd;

    int LINES, COLUMNS;
    int line = 0;

    if (SENDING_DISPLAY_magic != sd->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "sb_update_prompt_area",
	      "Bad magic number",0);

    menu_get_sizes(ptr,&LINES, &COLUMNS);    
    menu_ClearScreen(ptr);

    sd->cur_line = 0;
    sd->cur_col  = 0;

    if (LINES > 1) {
	line = 1;

	switch (sd->prompt_mode) {
	    
	case snd_prompt_none:
	    break;
    
	case snd_prompt_sender:
	    
	    if (sd->from_addr) {
		if (SD_ADDR_STATUS_magic != sd->from_addr->magic)
		    panic("SND DISPLAY PANIC",__FILE__,__LINE__,
			  "sb_update_prompt_area",
			  "Bad magic number (sd_addr_status)",0);
		
		if (sd->from_addr->addr) {
		    menu_print_format_center(ptr,0,
					     CATGETS(elm_msg_cat, ElmSet,
						     ElmSndAddrFailed,
						     "Sender address %S failed."),
					     sd->from_addr->addr);
		}	    
	    }
	    break;
	    
	case snd_prompt_recipient:
	    if (sd->prompt_rcpt_index >= 0 &&
		sd->prompt_rcpt_index < sd->rcpt_addr_count &&
		sd->rcpt_addrs[sd->prompt_rcpt_index]) {
		
		if (SD_ADDR_STATUS_magic != sd->rcpt_addrs[sd->prompt_rcpt_index]->magic) 
		    panic("SND DISPLAY PANIC",__FILE__,__LINE__,
			  "sb_update_prompt_area",
			  "Bad magic number (sd_addr_status)",0);
		
		if (sd->rcpt_addrs[sd->prompt_rcpt_index]->addr) {
		    menu_print_format_center(ptr,0,
					     CATGETS(elm_msg_cat, ElmSet,
						     ElmSndRAddrFailed,
						     "Recipient address %S failed."),
					     sd->rcpt_addrs[sd->prompt_rcpt_index]->addr);
		}
	    }
	    break;
	    
	case snd_prompt_final:
	    switch (sd->can_continue) {
	    case msd_mail_failed:
		menu_print_format_center(ptr,0,
					 CATGETS(elm_msg_cat, ElmSet,
						 ElmSndMailFailed,
						 "Mail failed."));
		break;
	    case msd_address_problem:
		menu_print_format_center(ptr,0,
					 CATGETS(elm_msg_cat, ElmSet,
						 ElmSndMailFailAddress,
						 "Mail have failed address."));
		break;
	    }
	    break;

	case snd_prompt_size:

	    /* long max (32-bit)       2147483647 */
	    if (sd->size_info[msd_message_size]       > 10000000 &&
		sd->size_info[msd_size_limit]         >  1000000)
		
		menu_print_format_center(ptr,0,
					 CATGETS(elm_msg_cat, ElmSet,
						 ElmSndSizeMExceed,
						 "Message size %ld megabytes exceed maximum allowed (%ld megabytes)."),
					 sd->size_info[msd_message_size]       / 1000000,
					 sd->size_info[msd_size_limit]         / 1000000);
	    

	    else if (sd->size_info[msd_message_size]       > 10000 &&
		     sd->size_info[msd_size_limit] >  1000)
		
		menu_print_format_center(ptr,0,
					 CATGETS(elm_msg_cat, ElmSet,
						 ElmSndSizeKExceed,
						 "Message size %ld kilobytes exceed maximum allowed (%ld kilobytes)."),
					 sd->size_info[msd_message_size]       / 1000,
					 sd->size_info[msd_size_limit]         / 1000);
	    
	    else
		menu_print_format_center(ptr,0,
					 CATGETS(elm_msg_cat, ElmSet,
						 ElmSndSizeExceed,
						 "Message size %ld bytes exceed maximum allowed (%ld bytes)."),
					 sd->size_info[msd_message_size],
					 sd->size_info[msd_size_limit]);
	    break;

	}
    }

    if (snd_prompt_none != sd->prompt_mode) {
	switch (sd->can_continue) {
	case msd_mail_failed:
	    menu_print_format_center(ptr,line,
				     CATGETS(elm_msg_cat, ElmSet,
					     ElmSndMailPromptF,
					     "c)ancel or e)dit mail? "));
	    
	    menu_GetXYLocation(ptr,&(sd->cur_line),&(sd->cur_col));
	    break;
	case msd_address_problem:
	    menu_print_format_center(ptr,line,
				     CATGETS(elm_msg_cat, ElmSet,
					     ElmSndMailPromptContinue,
					     "c)ancel, e)dit mail or continue sending (%c/%c) ? "),
				     *def_ans_yes, *def_ans_no);
	    menu_GetXYLocation(ptr,&(sd->cur_line),&(sd->cur_col));
	    break;
	}		
    }

    show_last_error();

    return 1;
 }


static enum msd_action msd_read_action P_((struct sending_display *sd,
					   enum msd_can_continue can_continue,
					   int have_timeout));
static enum msd_action msd_read_action(sd,can_continue,have_timeout)
     struct sending_display *sd;
     enum msd_can_continue can_continue;
     int have_timeout;
{
    enum msd_action ret = 
	msd_act_default  /* default action or not defined or OK */;
    int ch = 0;
    int done = 0;

    if (SENDING_DISPLAY_magic != sd->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "msd_read_action",
	      "Bad magic number",0);

    DPRINT(Debug,10, (&Debug,
		      "msd_read_action: can_continue=%d have_timeout=%d\n",
		      can_continue,have_timeout));


    if (sd->prompt_area) 
	sb_update_prompt_area(sd->prompt_area,sd->PARAM);
    else {
	DPRINT(Debug,10, (&Debug,
			  "msd_read_action: No prompt area\n"));
    }

    while (sd->prompt_area && !done) {
	
	if (have_timeout) {
	    DPRINT(Debug,10, (&Debug,
			      "msd_read_action: waiting for timout\n"));
	}

	sd_do_check(sd);
	
	if (sd->prompt_area)
	    ch = menu_ReadCh(sd->prompt_area,
			     REDRAW_MARK|READCH_resize|
			     READCH_CURSOR|READCH_sig_char|
			     (have_timeout ?  READCH_poll : 0));
	else
	    break;
	
	DPRINT(Debug,10, (&Debug,
			  "msd_read_action: ch=%d",ch));
	if (isascii(ch) && isprint(ch)) {
	    DPRINT(Debug,10, (&Debug," '%c'",ch));
	}
	DPRINT(Debug,10, (&Debug,"\n"));

	have_timeout = 0;

	if (msd_address_problem == can_continue
	    && *def_ans_yes == ch) {

	    menu_PutLineX(sd->prompt_area,
			  sd->cur_line,
			  sd->cur_col,
			  CATGETS(elm_msg_cat, ElmSet, ElmYesWord, 
				  "Yes."));
	    FlushBuffer();
	    error_sleep(1);

	    ret =  msd_act_send;
	    done = 1;
	    break;
	}

	if (*def_ans_no == ch) {

	    menu_PutLineX(sd->prompt_area,
			  sd->cur_line,
			  sd->cur_col,
			  CATGETS(elm_msg_cat, ElmSet, ElmNoWord, 
				  "No."));
	    FlushBuffer();
	    error_sleep(1);

	    ret = msd_act_cancel;
	    done = 1;
	    break;
	}

	switch (ch) {
	    
	case EOF:
	case TERMCH_eof_char:
	    ret = msd_act_EOF;
	    done = 1;
	    break;
	    
	case TERMCH_interrupt_char:
	    ret = msd_act_SIG;
	    done = 1;
	    break;
	    
	case RESIZE_MARK:
	    sd_size_changed(sd);
	    break;
	    
	case ctrl('L'):
	case REDRAW_MARK: 
	    sb_update_prompt_area(sd->prompt_area,sd->PARAM);

	    sd_do_redraw(sd);
	    break;

	case 'c':

	    menu_PutLineX(sd->prompt_area,
			  sd->cur_line,
			  sd->cur_col,
			  CATGETS(elm_msg_cat, ElmSet, ElmSndCancelMail, 
				  "Cancel mail."));
	    FlushBuffer();
	    error_sleep(1);	    

	    ret = msd_act_cancel;
	    done = 1;
	    break;

	case 'e':
	    menu_PutLineX(sd->prompt_area,
			  sd->cur_line,
			  sd->cur_col,
			  CATGETS(elm_msg_cat, ElmSet, ElmSndEditMail, 
				  "Edit mail."));
	    FlushBuffer();
	    error_sleep(1);

	    ret = msd_act_edit;
	    done = 1;
	    break;

	case PAGEUP_MARK:
	    if (sd->rcpt_area) {
		int header_LINES, header_COLUMNS;
		int top = menu_header_get(sd->rcpt_area,header_top_line);

		menu_get_sizes(sd->rcpt_area,&header_LINES, &header_COLUMNS);

		/* Does not use menu_header_change_page */
		
		top -= header_LINES;
		if (top < 0)
		    top = 0;
		menu_header_change(sd->rcpt_area,header_top_line,top);

	    }
	    break;

	case PAGEDOWN_MARK:
	    if (sd->rcpt_area) {
		int header_LINES, header_COLUMNS;
		int top = menu_header_get(sd->rcpt_area,header_top_line);

		menu_get_sizes(sd->rcpt_area,&header_LINES, &header_COLUMNS);

		/* Does not use menu_header_change_page */
		
		top += header_LINES;

		if (top + header_LINES > sd->rcpt_addr_count)
		    top = sd->rcpt_addr_count - header_LINES +1;
		if (top < 0)
		    top = 0;
		menu_header_change(sd->rcpt_area,header_top_line,top);
	    }
	    break;
	    
	case UP_MARK:
	    if (sd->rcpt_area) {
		int header_LINES, header_COLUMNS;
		int top = menu_header_get(sd->rcpt_area,header_top_line);

		menu_get_sizes(sd->rcpt_area,&header_LINES, &header_COLUMNS);

		if (top > 0) {
		    top--;
		    menu_header_change(sd->rcpt_area,header_top_line,top);
		}
	    }
	    break;

	case DOWN_MARK:
	    if (sd->rcpt_area) {
		int header_LINES, header_COLUMNS;
		int top = menu_header_get(sd->rcpt_area,header_top_line);

		menu_get_sizes(sd->rcpt_area,&header_LINES, &header_COLUMNS);

		if (top + header_LINES -1 < sd->rcpt_addr_count) {
		    top++;
		    menu_header_change(sd->rcpt_area,header_top_line,top);
		}
	    }
	    break;
	    
	case TIMEOUT_MARK:
	    DPRINT(Debug,10, (&Debug,"msd_read_action: timeout\n"));

	    ret = 
		msd_act_default  /* default action or not defined or OK */;
	    done = 1;
	    break;

	case '\n':
	    ret = 
		msd_act_default  /* default action or not defined or OK */;
	    done = 1;

	    menu_PutLineX(sd->prompt_area,
			  sd->cur_line,
			  sd->cur_col,
			  CATGETS(elm_msg_cat, ElmSet, ElmSndDefault, 
				  "Default action."));
	    
	    FlushBuffer();
	    error_sleep(1);
	    break;
	}
    }

    DPRINT(Debug,10, (&Debug,"msd_read_action=%d\n",ret));
    
    return ret;
 }

E_(msd_check_addr_cb check_addr_sd)
enum msd_action check_addr_sd(sd,idx,addr,status,message,can_continue)
    struct sending_display *sd;
    int                    idx;
    struct string        * addr;
    enum msd_addr_status   status;
    struct string        * message /* server */;
    enum msd_can_continue can_continue;
{
    /* Is this needed */
    int have_screen = RawState();
    enum msd_action ret = 
	msd_act_default  /* default action or not defined or OK */;


    if (SENDING_DISPLAY_magic != sd->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "update_addr_sd",
	      "Bad magic number",0);

    DPRINT(Debug,10, (&Debug,
		      "check_addr_sd: sd=%p idx=%d\n", 
		      sd,idx));

    if (have_screen) {
	if (idx < 0) 
	    sd_set_current(sd,snd_current_from,idx);
	else
	    sd_set_current(sd,snd_current_rcpt,idx);
    }

    if (idx < 0) {
	sd->prompt_mode = snd_prompt_sender;
    } else {
	sd->prompt_mode = snd_prompt_recipient;
	sd->prompt_rcpt_index = idx;
    }
    sd->can_continue = can_continue;
    sd->cur_line = 0;
    sd->cur_col  = 0;


    update_addr_sd(sd,idx,addr,status,message);

    

    if (have_screen) 
	ret = msd_read_action(sd,can_continue,
			      can_continue == msd_address_problem);

    sd->prompt_mode = snd_prompt_none;
    if (have_screen) {
	sd_set_current(sd,snd_current_none,idx);
	if (sd->prompt_area)
	    sb_update_prompt_area(sd->prompt_area,sd->PARAM);
    }

    DPRINT(Debug,10, (&Debug,"check_addr_sd=%d\n",ret));

    return  ret;
}

E_(msd_update_data_cb update_data_sd)
void update_data_sd(sd,status,message) 
    struct sending_display * sd;
    enum msd_data_status     status;
    struct string          * message /* server */;
{
    /* Is this needed */
    int have_screen = RawState();

    if (SENDING_DISPLAY_magic != sd->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "update_data_sd",
	      "Bad magic number",0);

    
    DPRINT(Debug,10, (&Debug,
		      "update_data_sd: sd=%p status=%d ",
		      sd,status));
    switch (status) {
    case  msd_mail_tmpfail:
	DPRINT(Debug,10, (&Debug,
			  "msd_mail_tmpfail"));
	break;
    case msd_mail_fail:
	DPRINT(Debug,10, (&Debug,
			  "msd_mail_fail"));
	break;
    case msd_mail_backgrounded:
	DPRINT(Debug,10, (&Debug,
			  "msd_mail_backgrounded"));
	break;
    case msd_calling_mailer:
	DPRINT(Debug,10, (&Debug,
			  "msd_calling_mailer"));
	break;
    case msd_datastat_none:
	DPRINT(Debug,10, (&Debug,
			  "msd_datastat_none"));
	break;
    case msd_sending_data:
	DPRINT(Debug,10, (&Debug,
			  "msd_sending_data"));
	break;
    case msd_data_ok:
	DPRINT(Debug,10, (&Debug,
			  "msd_data_ok"));
	break;
    case msd_mail_sent:
	DPRINT(Debug,10, (&Debug,
			  "msd_mail_sent"));
	break;
    }
    if (message) {
	DPRINT(Debug,10, (&Debug,", message=\"%S\"",message));
    }
    if (!have_screen) {
	DPRINT(Debug,10, (&Debug,"; not on full screen mode"));
	
    }
    DPRINT(Debug,10, (&Debug,"\n"));

    sd->data_status = status;

    if (sd->data_message)
	free_string(& (sd->data_message));

    if (message)
	sd->data_message = dup_string(message);

    if (have_screen) {
	sd_size_changed(sd);

	if (sd->status_area)
	    menu_trigger_redraw(sd->status_area);

	sd_do_check(sd);
	FlushBuffer();
    }
}

E_(msd_check_fail_cb check_fail_sd)
enum msd_action check_fail_sd(sd,status,message,can_continue)
    struct sending_display *sd;
    enum msd_data_status    status;
    struct string         * message /* server */;
    enum msd_can_continue   can_continue;
{
    /* Is this needed */
    int have_screen = RawState();
    enum msd_action ret = 
	msd_act_default  /* default action or not defined or OK */;

    if (SENDING_DISPLAY_magic != sd->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "check_fail_sd",
	      "Bad magic number",0);
 
    DPRINT(Debug,10, (&Debug,
		      "check_fail_sd: sd=%p\n", 
		      sd));

    if (have_screen)
	sd_set_current(sd,snd_current_status,0);

    sd->prompt_mode = snd_prompt_final;
    sd->can_continue = can_continue;
    sd->cur_line = 0;
    sd->cur_col  = 0;

    update_data_sd(sd,status,message);

    if (have_screen) 
	ret = msd_read_action(sd,can_continue,0);
 
    sd->prompt_mode = snd_prompt_none;    
    if (have_screen) {
	sd_set_current(sd,snd_current_none,0);
	if (sd->prompt_area)
	    sb_update_prompt_area(sd->prompt_area,sd->PARAM);
    }

    DPRINT(Debug,10, (&Debug,"check_fail_sd=%d\n",ret));
    return ret;   
}

E_(msd_update_con_cb update_con_sd)
void update_con_sd(sd,idx,text)
    struct sending_display *sd;
    enum msd_connection_info idx;
    struct string *text;
{
    /* Is this needed */
    int have_screen = RawState();
    int flag = 0;

    if (SENDING_DISPLAY_magic != sd->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "update_con_sd",
	      "Bad magic number",0);
 
    if (idx < 0 || idx >= NUM_msd_connection_info)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "update_con_sd",
	      "Bad info",0);
    
    DPRINT(Debug,10, (&Debug,"update_con_sd: sd=%p idx=%d ",
		      sd,idx));
    switch (idx) {
    case msd_server:
	DPRINT(Debug,10, (&Debug,"msd_server"));
	break;
    case msd_verified_as:
	DPRINT(Debug,10, (&Debug,"msd_verified_as"));
	break;
    case NUM_msd_connection_info:
	break;
    }
    if (text) {
	DPRINT(Debug,10, (&Debug,", text=\"%S\"",text));
    }
    if (!have_screen) {
	DPRINT(Debug,10, (&Debug,"; not on full screen mode"));
	
    }
    DPRINT(Debug,10, (&Debug,"\n"));

    if (sd->con_info_text[idx])
	free_string(& (sd->con_info_text[idx]));
    else if (! sd->con_info_text[msd_server] &&
	     ! sd->con_info_text[msd_verified_as])
	flag = 1;

    if (text)
	sd->con_info_text[idx] = dup_string(text);

    if (have_screen) {
	sd_size_changed(sd);

	if (sd->top_area)
	    menu_trigger_redraw(sd->top_area);

	if (sd->rcpt_area && flag)
	    menu_trigger_redraw(sd->rcpt_area);

	sd_do_check(sd);
	FlushBuffer();
    }
}

void update_size_cb P_((struct sending_display *sd,
			enum msd_size_info idx,
			long size));


E_(msd_check_size_cb size_check_sd)
enum msd_action size_check_sd P_((struct sending_display *sd,
				  long message_size,
				  long size_limit));
enum msd_action size_check_sd(sd,message_size,size_limit)
     struct sending_display *sd;
     long message_size;
     long size_limit;
{
    int have_screen = RawState();
    enum msd_action ret = 
	msd_act_default  /* default action or not defined or OK */;

    if (SENDING_DISPLAY_magic != sd->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "size_check_sd",
	      "Bad magic number",0);

    DPRINT(Debug,10, (&Debug,
		      "size_check_sd: sd=%p message_size=%ld size_limit=%ld\n", 
		      sd,message_size,size_limit));

    if (have_screen)
	sd_set_current(sd,snd_current_size,0);

    sd->prompt_mode = snd_prompt_size;
    sd->can_continue = msd_address_problem;   /* not really */
    sd->cur_line = 0;
    sd->cur_col  = 0;

    sd->size_info[msd_message_size]       = message_size;
    update_size_cb(sd,msd_size_limit,size_limit);

    if (have_screen) 
	ret = msd_read_action(sd,msd_address_problem,0);
   
    sd->prompt_mode = snd_prompt_none;  

    if (have_screen) {
	sd_set_current(sd,snd_current_none,0);
	if (sd->prompt_area)
	    sb_update_prompt_area(sd->prompt_area,sd->PARAM);
    }
 
    DPRINT(Debug,10, (&Debug,"size_check_sd=%d\n",ret));

    return ret;
}

E_(msd_update_size_cb update_size_cb)
void update_size_cb(sd,idx,size)
     struct sending_display *sd;
     enum msd_size_info idx;
     long size;
{
    int have_screen = RawState();
    int flag = 0;

    if (SENDING_DISPLAY_magic != sd->magic)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "update_size_cb",
	      "Bad magic number",0);

    if (idx < 0 || idx >= NUM_msd_size_info)
	panic("SND DISPLAY PANIC",__FILE__,__LINE__,
	      "update_size_cb",
	      "Bad info",0);
    
    DPRINT(Debug,10, (&Debug,"update_size_cb: sd=%p idx=%d ",
		      sd,idx));
    switch (idx) {
    case msd_message_size:
	DPRINT(Debug,10, (&Debug,"msd_message_size"));
	break;
    case msd_size_limit:
	DPRINT(Debug,10, (&Debug,"msd_size_limit"));
	break;
    case NUM_msd_size_info:
	break;
    }

    DPRINT(Debug,10, (&Debug,", size=%ld",size));
    if (!have_screen) {
	DPRINT(Debug,10, (&Debug,"; not on full screen mode"));
	
    }
    DPRINT(Debug,10, (&Debug,"\n"));

    if (sd->size_info[msd_message_size] < 0 &&
	sd->size_info[msd_size_limit] < 0)
	flag = 1;

    sd->size_info[idx] = size;

    if (have_screen) {
	sd_size_changed(sd);

	if (sd->top_area)
	    menu_trigger_redraw(sd->top_area);

	if (sd->rcpt_area && flag)
	    menu_trigger_redraw(sd->rcpt_area);

	sd_do_check(sd);
	FlushBuffer();
    }
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */

