static char rcsid[] = "@(#)$Id: thread.c,v 2.19 2024/06/26 16:54:27 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.19 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> (was hurtta+elm@posti.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_elm.h"
#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"mail");

static unsigned char * s2us P_((char *str));
static unsigned char * s2us(str)
     char *str;
{
    return (unsigned char *)str;
}


static void first_item P_((void));
static void first_item()
{
    lib_error(CATGETS(elm_msg_cat, MeSet, 
		      MeNoMoreThreadsAbove,
		      "No more threads above."));    
}

static void last_item P_((void));
static void last_item()
{
    lib_error(CATGETS(elm_msg_cat, MeSet, 
		      MeNoMoreThreadsBelow,
		      "No more threads below.")); 
}

static struct move_messages M = {
    first_item,
    last_item
};

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

/* SHRT_MAX */
#include <limits.h>

struct sort_list {
    char  status_letter;
    int   idx;

    struct string * menu_date_s;
    /* What is used for calculate menu_date_s */
    time_t first, last, columns;
};


struct menu_anon_param {
    struct MailboxView * mailbox; 

    struct sort_list   * sort_list;
    int                  sort_list_len;

    int index_width;
    int menu_date_width;
};

struct sort_data1 {
    struct MailboxView *mailbox;
    int index;
};


int compare_threads_1_sentd(X1,X2)
     const struct thread_info *X1;
     const struct thread_info *X2;
{
    long diff = 0;
    
    /* Accpet invalid date */
    
    if (X1->time_sent_first == X2->time_sent_first &&
	X1->year_sent_first == X2->year_sent_first)
	return 0;
    
    if (time_OK(X1->time_sent_first) &&
	time_OK(X2->time_sent_first))
	diff = (long) X1->time_sent_first - (long) X2->time_sent_first;
    else if (X1->year_sent_first > 0 &&
	     X2->year_sent_first > 0)
	diff = (long)X1->year_sent_first - (long)X2->year_sent_first;
    
    if (0 == diff) {
	if ( time_OK(X1->time_sent_first) &&
	     !  time_OK(X2->time_sent_first))
	    return -1;
	if (! time_OK(X1->time_sent_first) &&
	    time_OK(X2->time_sent_first))
	    return 1;
    }
    
    if (diff < 0)
	return -1;
    if (diff > 0)
	return 1;

    return 0;
}

int compare_threads_1_revsentd(X1,X2)
     const struct thread_info *X1;
     const struct thread_info *X2;
{
    long diff = 0;
    
    if (unstable_reverse_thread) {
	/* Move thread new location when new mail arrives,
	   therfore use newest sent time
	*/
	
	/* Accpet invalid date */
	
	if (X1->time_sent_last == X2->time_sent_last &&
	    X1->year_sent_last == X2->year_sent_last)
	    return 0;
	
	if (time_OK(X1->time_sent_last) &&
	    time_OK(X2->time_sent_last))
	    diff = (long)X1->time_sent_last - (long)X2->time_sent_last;
	else if ( X1->year_sent_last > 0 &&
		  X2->year_sent_last > 0)
	    diff = (long)X1->year_sent_last - (long)X2->year_sent_last;
	
	if (0 == diff) {
	    
	    if (time_OK(X1->time_sent_last) &&
		! time_OK(X2->time_sent_last))
		return -1;
	    if (!time_OK(X1->time_sent_last) &&
		time_OK(X2->time_sent_last))
		return 1;
	}
	
    } else {
	
	/* Accpet invalid date */
	
	if (X1->time_sent_first == X2->time_sent_first &&
	    X1->year_sent_first == X2->year_sent_first)
	    return 0;
	
	if ( time_OK(X1->time_sent_first) &&
	     !  time_OK(X2->time_sent_first))
	    return -1;
	if (! time_OK(X1->time_sent_first) &&
	    time_OK(X2->time_sent_first))
	    return 1;
	
	if (time_OK(X1->time_sent_first) &&
	    time_OK(X2->time_sent_first))
	    diff = X1->time_sent_first - X2->time_sent_first;
	else if (X1->year_sent_first > 0 &&
		 X2->year_sent_first > 0)
	    diff = X1->year_sent_first - X2->year_sent_first;
    }
    
    if (diff < 0)
	return 1;
    if (diff > 0)
	return -1;
    
    return 0;
}


static int compare_threads_1 P_((const struct thread_info *X1,
				 const struct thread_info *X2));
static int compare_threads_1(X1,X2)
     const struct thread_info *X1;
     const struct thread_info *X2;
{
    int ret;

    switch (give_dt_sort_as_int(&thread_sortby)) {

    case SENT_DATE:   /* as THREAD sorting */

	ret = compare_threads_1_sentd(X1,X2);
	
	return ret;

    case REVERSE SENT_DATE:

	ret = compare_threads_1_revsentd(X1,X2);
	
	return ret;

    case SUBJECT:

	if (! X1->thread_subject && ! X2->thread_subject)
	    return 0;
	if (! X1->thread_subject)
	    return -1;
	if (! X2->thread_subject)
	    return 1;

	ret = string_sort_cmp(X1->thread_subject,X2->thread_subject);

	return ret;

    case REVERSE SUBJECT:

	if (! X1->thread_subject && ! X2->thread_subject)
	    return 0;
	if (! X1->thread_subject)
	    return -1;
	if (! X2->thread_subject)
	    return 1;

	ret = string_sort_cmp(X1->thread_subject,X2->thread_subject);

	return -ret;
    }

    return 0;
}

static int compare_threads P_((struct sort_data1 *p1,
			       struct sort_data1 *p2));
static int compare_threads(p1,p2)
     struct sort_data1 *p1; 
     struct sort_data1 *p2;
{
    int ret;

    /* give_thread_info checks that index is correct range */
    const struct thread_info *X1 = give_thread_info(p1->mailbox,
						    p1->index);
    const struct thread_info *X2 = give_thread_info(p2->mailbox,
						    p2->index);

    if (!X1 && !X2) {
	return p2->index - p1->index;
    }

    if (!X1) 
	return -1;
    if (!X2) 
	return 1;

    ret = compare_threads_1(X1,X2);

    if (ret != 0)
	return ret;

    return p2->index - p1->index;
}


static void sort_threads P_((struct MailboxView *mailbox,
			     struct menu_anon_param *A,
			     struct menu_context *page));

static void sort_threads(mailbox,A,page)
     struct MailboxView *mailbox;
     struct menu_anon_param *A;
     struct menu_context *page;
{
    /* Be sure that we have threads available */
    int num_threads = get_thread_count(mailbox,0);
    int old_len = A->sort_list_len;
    int cur = menu_header_get(page,header_current);
    int top     = menu_header_get(page,header_top_line);
    int real_cur = 0;
    int i;

    struct sort_data1 * array;

    /* Little dirty ... */
    typedef int (*compar) P_((const void *, const void *));
    compar X = (compar) compare_threads;


    int li,co;
    menu_get_sizes(page, &li, &co);   


    if (num_threads < 1) {
	A->sort_list_len = 0;

	menu_header_change(page, header_top_line,0);
	menu_header_change(page,header_current,0);

	return;
    }

    if (cur >= 0 && cur < A->sort_list_len)
	real_cur = A->sort_list[cur].idx;

    A->sort_list = safe_array_realloc(A->sort_list,
				      num_threads,sizeof (A->sort_list[0]));

    for (i = old_len; i < num_threads; i++) {
	A->sort_list[i].idx = i;
	A->sort_list[i].status_letter = '\0';
	A->sort_list[i].menu_date_s = NULL;
	A->sort_list[i].first       = 0;
	A->sort_list[i].last        = 0;
	A->sort_list[i].columns     = 0;
    }
    A->sort_list_len = num_threads;

    array = safe_calloc(num_threads, sizeof(array[0]));

    for (i = 0; i < num_threads; i++) {
	array[i].index   = A->sort_list[i].idx;
	array[i].mailbox = mailbox;
    }

    qsort(array,num_threads,sizeof (array[0]), X);

    for (i = 0; i < num_threads; i++) {
	if (array[i].index   != A->sort_list[i].idx) {
	    A->sort_list[i].idx = array[i].index;
	    A->sort_list[i].status_letter = '\0';
	    if (A->sort_list[i].menu_date_s)
		free_string(& (A->sort_list[i].menu_date_s));
	}
    }
    free(array);

    /* FIXME -- not very effective */
    for (i = 0; i < A->sort_list_len; i++) {
	if (real_cur == A->sort_list[i].idx) {
	    menu_header_change(page,header_current,i);
	    if (i < top || i >= top+li)
		menu_header_change(page,header_top_line,i);
	    break;
	}
    }
    menu_trigger_redraw(page);
}

static int calculate_status P_((struct menu_anon_param *A,
				int index));

/* return 1 if changed */
static int calculate_status(A,index)
     struct menu_anon_param *A;
     int index;
{
    int * vector = NULL;
    int len;
    char new_val;
    int ret = 0;

    if (index < 0 || index > A->sort_list_len)
	panic("THREAD PANIC",__FILE__,__LINE__,"calculate_status",
	      "Bad index",0);

    vector = give_thread_message_list(A->mailbox,A->sort_list[index].idx,
				      &len);
    if (vector) {
	int i;
	int all_deleted = 1;
	int all_expired = 1;

	new_val = ' ';

	for (i = 0; i < len; i++) {
	    
	    if (!ison_status_message(A->mailbox,vector[i],status_basic,
				     DELETED))
		all_deleted = 0;
	    if (!ison_status_message(A->mailbox,vector[i],status_basic,
				     EXPIRED))
		all_expired = 0;
	    
	    if (' ' == new_val &&
		ison_status_message(A->mailbox,vector[i],status_basic,UNREAD))
		new_val = 'O';
	    
	    if ((' ' == new_val || 'O' == new_val) &&
		ison_status_message(A->mailbox,vector[i],status_basic,NEW))
		new_val = 'N';
	    
	}

	if (all_expired)
	    new_val = 'E';
	if (all_deleted)
	    new_val = 'D';

	free(vector);
    } else
	new_val = 'Z';

    if (A->sort_list[index].status_letter != new_val) {
	A->sort_list[index].status_letter = new_val;
	ret = 1;
    }

    DPRINT(Debug,16, (&Debug, "calculate_status=%d: index=%d status letter=%c\n",
		      ret,index,A->sort_list[index].status_letter));
    
    return ret;
}

static int calculate_menu_date P_((struct menu_anon_param *A,
				   int index,
				   const struct thread_info *X,
				   int columns));
static int calculate_menu_date(A,index,X,columns)
     struct menu_anon_param *A;
     int index;
     const struct thread_info *X;
     int columns;
{

    int reserve = A->index_width + 7;
    int ret = 0;
        
    if (index < 0 || index > A->sort_list_len)
	panic("THREAD PANIC",__FILE__,__LINE__,"calculate_menu_date",
	      "Bad index",0);
    
    if (!X) {
	DPRINT(Debug,16, (&Debug, "calculate_menu_date: index=%d, thread info not given\n",
			  index));
	
	
	/* give_thread_info checks that index is correct range */
	X = give_thread_info(A->mailbox,A->sort_list[index].idx);
    }

    if (X) {
	/* Message menu is show time as sending timezone,
	   but on thread view we use localtime 
	*/
	
	struct string * S = NULL;
	int add_last = 0;
	int l = 0;
	
	A->sort_list[index].first =  X->time_sent_first;
	A->sort_list[index].last  =  X->time_sent_last;
	A->sort_list[index].columns = columns;

        if (time_OK(X->time_sent_first)) {
	    
	    struct tm * tmres = localtime(& X->time_sent_first);
	    
	    if (tmres) {
		
		char timebuf1[SLEN] = "";
		
		char * thread_first_MD_fmt
		    = catgets(elm_msg_cat,MeSet,
			      MeThreadFrstMD,
			      "%b %d ");
		
		size_t len1 =
		    strftime(timebuf1,sizeof timebuf1,
			     thread_first_MD_fmt,
			     tmres);
		
		if (len1 > 0 ) {		
		    /* system_charset is locale charset */		
		    S = new_string2(system_charset,
				    s2us(timebuf1));
		    
		
		    l =  string_len(S);
		    
		    if (columns > l+15 + reserve) {
			
			if (X->time_sent_first != X->time_sent_last) 
			    add_last = 1;			
			else {
			    struct string *S1 = NULL;

			    char timebuf3[SLEN] = "";
			    char * thread_last_Y_fmt
				= catgets(elm_msg_cat,MeSet,
					  MeThreadLastY,
					  "%Y");
			    
			    size_t len3 =
				strftime(timebuf3,sizeof timebuf3,
					 thread_last_Y_fmt,
					 tmres);
			    
			    if (len3 > 0) {
				S1 =  new_string2(system_charset,
						  s2us(timebuf3));
				
			    } else {
				DPRINT(Debug,16,(&Debug,
						 "calculate_menu_date: strftime failed\n"));
				goto fail3;
			    }			    
			
			    
			    append_string(&S,S1,1);
			    free_string(&S1);
			    
			    l = string_len(S);
			}
		    } else {

		    fail3:
			DPRINT(Debug,16, (&Debug, "calculate_menu_date: truncating\n"));
		    }
		} else {
		    DPRINT(Debug,16,(&Debug,
				     "calculate_menu_date: strftime failed\n"));
		    goto fail1;
		}
	    
	    } else {
		DPRINT(Debug,16,(&Debug,
				 "calculate_menu_date: localtime failed\n"));
		
		goto fail1;
	    }

	} else {

	    if (X->year_sent_first > 0 && X->year_sent_first < SHRT_MAX) {
		const char * s UNUSED_VAROK = 
		    (time_MAX == X->time_sent_first) ? "MAX overflow" : "usupported value";		
		
		DPRINT(Debug,16,(&Debug,
				 "calculate_menu_date: index=%d, %s, sent first year %d\n",
				 index, s,X->year_sent_first));
		
		S = format_string( FRM("%04d "),
				   X->year_sent_first);

		l =  string_len(S);

		if (columns > l+15 + reserve) {
		    if (X->year_sent_first != X->year_sent_last)
			add_last = 1;	
		}

	    } else {
		
	    fail1:		    
		S = format_string( FRM("<?> "));

		l =  string_len(S);
	    }
	    
	} 
	
	if (add_last) {
	    
	    struct string *S1 = NULL;

	    if (time_OK(X->time_sent_last)) {

		struct tm * tmres = localtime(& X->time_sent_last);
		
		if (tmres) {
		    char timebuf2[SLEN] = "";
		    
		    char * thread_last_MDY_fmt
			= catgets(elm_msg_cat,MeSet,
				  MeThreadLastMDY,
				  "- %b %d %Y");
		    
		    size_t len2 =
			strftime(timebuf2,sizeof timebuf2,
				 thread_last_MDY_fmt,
				 tmres);
		    
		    if (len2 > 0) {			    
			if (columns < l+len2 + reserve) {
			    DPRINT(Debug,16,(&Debug,
					     "calculate_menu_date: does not fit: %s\n",
					     timebuf2));
			    goto overflow;
			}
			
			/* system_charset is locale charset */
			
			S1 =  new_string2(system_charset,
					  s2us(timebuf2));
		    } else {
			DPRINT(Debug,16,(&Debug,
					 "calculate_menu_date: strftime failed\n"));
			goto fail2;
		    }
		    
		} else {
		    DPRINT(Debug,16,(&Debug,
				 "calculate_menu_date: localtime failed\n"));
		
		    goto fail2;
		}

	    } else {

		if (X->year_sent_last > 0 && X->year_sent_last < SHRT_MAX) {
		    const char * s UNUSED_VAROK = 
			(time_MAX == X->time_sent_last) ? "MAX overflow" : "usupported value";		
		
		    DPRINT(Debug,16,(&Debug,
				     "calculate_menu_date: index=%d, %s, sent last year %d\n",
				     index, s,X->year_sent_last));
		
		    S1 = format_string( FRM("- %04d"),
					X->year_sent_last);

		} else {
		    
		fail2:
		    S1 = format_string(FRM("- <?>"));
		}
	    }

	    if (columns < l+string_len(S1) + reserve) {
		DPRINT(Debug,16,(&Debug,
				 "calculate_menu_date: does not fit: %S\n",
				 S1));
	    } else {
		append_string(&S,S1,1);
		l = string_len(S);	    
	    }
	    free_string(&S1);
	    

	overflow:
	    ;

	}
       
	if (! A->sort_list[index].menu_date_s ||
	    0 != string_cmp(A->sort_list[index].menu_date_s,
			    S,-99 /* Can't compare */)) {

	    if (A->sort_list[index].menu_date_s)
		free_string(& (A->sort_list[index].menu_date_s));
	    A->sort_list[index].menu_date_s = S;
	    S = NULL;
	    ret = 1;
	} else
	    free_string(&S);
		 
    }

    DPRINT(Debug,16, (&Debug, "calculate_menu_date=%d: index=%d",
		      ret,index));
    if (A->sort_list[index].menu_date_s) {
	DPRINT(Debug,16, (&Debug, " menu date=%S",
			  A->sort_list[index].menu_date_s));
    }
    DPRINT(Debug,16, (&Debug, " columns=%d\n",
		      columns));
    
    return ret;
}
	
enum { thread_mp_mailbox,
       thread_mp_param,
       thread_mp_COUNT };

/* Returns 1 if header_setup_line() need to be called */
S_(header_setup_init thread_header_init)
static int thread_header_init P_((struct menu_context  *ptr,
				  struct menu_param *list));
static int thread_header_init(ptr,list)
     struct menu_context * ptr;
     struct menu_param   * list;
{
    struct menu_anon_param *A = mp_lookup_anon(list,thread_mp_param);

    A->index_width     = 3;
    A->menu_date_width = 21;
    
    DPRINT(Debug,14,(&Debug,
		     "thread_header_init: index_width %d menu_date_width %d\n",
		     A->index_width,A->menu_date_width));

    return 1;
}

/* Prescan formatting */
S_( header_setup_line thread_header_line)
static void thread_header_line P_((struct menu_context  *ptr,
				   struct menu_param *list,
				   int index));
static void thread_header_line(ptr,list,index)
     struct menu_context  *ptr;
     struct menu_param *list;
     int index;
{
    struct menu_anon_param *A = mp_lookup_anon(list,thread_mp_param);

    /* Printed index is idex+1 */

    if (index >= 9999 && A->index_width < 5) {
	A->index_width = 5;
	
	DPRINT(Debug,14,(&Debug," ... thread_header_line index %d index_width %d\n",
			 index,A->index_width));	
    } else if (index >= 999 && A->index_width < 4) {
	A->index_width = 4;
	
	DPRINT(Debug,14,(&Debug," ... thread_header_line index %d index_width %d\n",
			 index,A->index_width));	
    }

    if (index >= 0 && index < A->sort_list_len) {

	/* give_thread_info checks that index is correct range */
	const struct thread_info *X
	    = give_thread_info(A->mailbox,A->sort_list[index].idx);

	int LINES, COLUMNS;

	menu_get_sizes(ptr, &LINES, &COLUMNS);

	
	if (X) {
	    if (! A->sort_list[index].menu_date_s ||
		A->sort_list[index].first !=  X->time_sent_first ||
		A->sort_list[index].last !=  X->time_sent_last||
		A->sort_list[index].columns != COLUMNS
		)
		calculate_menu_date(A,index,X,COLUMNS);
	    
	    if (A->sort_list[index].menu_date_s){
		int l = string_len(A->sort_list[index].menu_date_s);
		
		if (A->menu_date_width < l) {
		    A->menu_date_width = l;
		    
		    DPRINT(Debug,14,(&Debug," ... thread_header_line index %d menu_date_width %d\n",
				     index,A->menu_date_width));
		    
		}
	    }
	}
    }
    
}

S_(header_line_redraw thread_show_header)
static void thread_show_header P_((struct menu_context  *ptr,
				   struct menu_param *list,
				   int line_number,
				   int index,
				   int is_current));
static void thread_show_header(ptr,list,line_number,index,is_current)
     struct menu_context  *ptr;
     struct menu_param *list;	   
     int line_number;
     int index;
     int is_current;
{
    struct menu_anon_param *A = mp_lookup_anon(list,thread_mp_param);

    int LINES, COLUMNS;

    menu_get_sizes(ptr, &LINES, &COLUMNS);

    menu_ClearLine(ptr,line_number);	    
    menu_MoveCursor(ptr,line_number,0);

    if (is_current) {

	if (has_highlighting && ! arrow_cursor) {

	    menu_StartXX(ptr,pg_STANDOUT);

	    menu_PutLine0(ptr,line_number,0,"   ");

	} else {
	    menu_PutLine0(ptr,line_number,0,"-> ");

	}

    } else 
	menu_PutLine0(ptr,line_number,0,"   ");

    if (index >= 0 && index < A->sort_list_len) {
	const struct thread_info *X;

	if (!A->sort_list[index].status_letter)
	    calculate_status(A,index);
       	
	/* give_thread_info checks that index is correct range */
	X = give_thread_info(A->mailbox,A->sort_list[index].idx);
	if (X) {
	    /* Message menu is show time as sending timezone,
	       but on thread view we use localtime 
	    */
	    
	    struct string * S = NULL;
	    int l;
	    
	    if (! A->sort_list[index].menu_date_s ||
		A->sort_list[index].first !=  X->time_sent_first ||
		A->sort_list[index].last !=  X->time_sent_last ||
		A->sort_list[index].columns != COLUMNS)
		calculate_menu_date(A,index,X,COLUMNS);
	    	 	    
	
	    if (A->sort_list[index].menu_date_s)
		S = format_string(FRM("%c %*d %-*S "),
				  A->sort_list[index].status_letter ?
				  A->sort_list[index].status_letter : '?',
				  A->index_width,
				  index+1,
				  A->menu_date_width,
				  A->sort_list[index].menu_date_s);
	    else
		S = format_string(FRM("%c %*d %-*s "),
				  A->sort_list[index].status_letter ?
				  A->sort_list[index].status_letter : '?',
				  A->index_width,
				  index+1,
				  A->menu_date_width,
				  "");
		
	    l = string_len(S);
			           	    
	    if (COLUMNS > l+9) {
		struct string *S1;
		if (X->num_messages != 1) {
		    S1 = format_string(FRM("[%3d] "),
				       X->num_messages);
		} else {
		    S1 = format_string(FRM("      "));		    
		}

		append_string(&S,S1,1);
		free_string(&S1);

		l = string_len(S);		
	    }

	    menu_PutLineX(ptr,line_number,3,FRM("%S"),S);
	    free_string(&S);
	    
	    if (X->thread_subject && COLUMNS > l+3) {
		struct string * buffer2 = NULL;
		int X1 = 0;
		int subj_width = COLUMNS-l-3;
		int was_len;
		
		/* give_string_from_string_sort() returns original string 
		   with incremented refcount 
		*/
		struct string * S2 = give_string_from_string_sort(X->thread_subject);

		

		/* clip_from_string updates X1 */
		buffer2 = curses_printable_clip(S2,&X1,
						subj_width,
						&was_len,subj_width);
						
		
		if (buffer2) {
		    menu_PutLineX(ptr,line_number,l+3,FRM("%S"),
				  buffer2);		    
		    free_string(&buffer2);
		}	    		


		if (is_current && !arrow_cursor) {
		    while (was_len < subj_width) {
			menu_Writechar(ptr,' ');
			was_len++;
		    }
		}

		free_string(&S2);
		
	    } 
	}

    }

    if (is_current) {
	if (has_highlighting && ! arrow_cursor) {
	    menu_EndXX(ptr,pg_STANDOUT);
	}
    }
    menu_Writechar(ptr,'\r');
    menu_Writechar(ptr,'\n');
}

S_(header_line_redraw thread_show_current)
static void thread_show_current P_((struct menu_context  *ptr,
				    struct menu_param *list,
				    int line_number,
				    int index,
				    int is_current));
static void thread_show_current(ptr,list,line_number,index,is_current)
     struct menu_context  *ptr;
     struct menu_param *list;	   
     int line_number;
     int index;
     int is_current;
{

    if (has_highlighting && ! arrow_cursor) {
	thread_show_header(ptr,list,line_number,index,is_current);
    } else {
	if (!is_current)
	    menu_PutLine0(ptr,line_number,0,"  ");  /* remove old pointer... */
	else
	    menu_PutLine0(ptr,line_number,0,"->");
    }
}

S_(header_line_redraw thread_show_status)
static void thread_show_status P_((struct menu_context  *ptr,
				    struct menu_param *list,
				    int line_number,
				    int index,
				    int is_current));
static void thread_show_status(ptr,list,line_number,index,is_current)
     struct menu_context  *ptr;
     struct menu_param *list;	   
     int line_number;
     int index;
     int is_current;
{
    struct menu_anon_param *A = mp_lookup_anon(list,thread_mp_param);


    if (has_highlighting && ! arrow_cursor) {
	thread_show_header(ptr,list,line_number,index,is_current);
    } else {
	if (!is_current)
	    menu_PutLine0(ptr,line_number,0,"  ");  /* remove old pointer... */
	else
	    menu_PutLine0(ptr,line_number,0,"->");

	if (index >= 0 && index < A->sort_list_len) {
	    const struct thread_info *X;

	    /* give_thread_info checks that index is correct range */
	    X = give_thread_info(A->mailbox,A->sort_list[index].idx);
	    if (X) {

	    menu_PutLineX(ptr,line_number,3,
			  FRM("%c"),
			  A->sort_list[index].status_letter ?
			  A->sort_list[index].status_letter : '?');
	    }
	}
    }
}


S_(subpage_simple_redraw sb_update_thread_menu)
static int sb_update_thread_menu P_((struct menu_context  *ptr,
				     struct menu_param *list));
static int sb_update_thread_menu(ptr,list)
     struct menu_context  *ptr;
     struct menu_param *list;
{
    menu_ClearScreen(ptr);

    menu_print_format_center(ptr,0,
			     CATGETS(elm_msg_cat, MeSet, MeThreadMenuLine1,
				     "To select thread, press <return>.  j = move down, k = move up, q = quit menu"));
    menu_print_format_center(ptr,1,
			     CATGETS(elm_msg_cat, MeSet, MeThreadMenuLine2,
				     "Or m)ail message,  d)elete or u)ndelete messages of thread"));

    return 1;
}



S_(subpage_simple_redraw sb_update_threadtitle)
static int sb_update_threadtitle P_((struct menu_context  *ptr,
				     struct menu_param *list));
static int sb_update_threadtitle(ptr,list)
     struct menu_context  *ptr;
     struct menu_param *list;
{
    struct menu_common *mptr = mp_lookup_mcommon(list,thread_mp_mailbox);
    struct string * f1 = mcommon_title(mptr);
    struct menu_anon_param *A = mp_lookup_anon(list,thread_mp_param);
    int num_threads = get_thread_count(A->mailbox,0);

    struct string * buffer = NULL;
    struct string * buffer2 = NULL;

    int LINES, COLUMNS;
    int l1,l2,l;

    menu_ClearScreen(ptr);
    menu_get_sizes(ptr, &LINES, &COLUMNS);   

    
    if (1 == num_threads)
	buffer = format_string(CATGETS(elm_msg_cat, MeSet, 
				       MeShownWithThread,
				       "%S with 1 thread"),
			       f1);
    else
	buffer = format_string(CATGETS(elm_msg_cat, MeSet, 
				       MeShownWithThreads,
				       "%S with %d threads"),
			   f1,num_threads);

    l1 = string_len(buffer);
    
    buffer2 = format_string(FRM("[ELM %s]"),
			    version_buff);
    l2 = string_len(buffer2);
    l = l1 + l2 + 1; /* Assumed */
    
    menu_StartXX(ptr,pg_BOLD);
    if (l > COLUMNS) {
	if (l2 < COLUMNS)
	    menu_PutLineX(ptr,2,(COLUMNS - l2)/2,FRM("%S"),buffer2);
	if (l1 > COLUMNS) 
	    menu_PutLineX(ptr,1,1,FRM("%S"),buffer);
	else
	    menu_PutLineX(ptr,1,(COLUMNS - l1)/2,FRM("%S"),buffer);
    } else {
	menu_PutLineX(ptr,1,(COLUMNS - l)/2,FRM("%S %S"),buffer,buffer2);
    }
    menu_EndXX(ptr,pg_BOLD);

    free_string(&buffer2);
    free_string(&buffer);
    free_string(&f1);


    return 1;   /* subpage area updated completely */

}

static void set_thread_screen P_((struct menu_context  *page, 
				 struct screen_parts *LOC,
				 struct menu_param  *LIST));
static void set_thread_screen(page,LOC,LIST)
     struct menu_context  *page;
     struct screen_parts *LOC;
     struct menu_param  *LIST;
{
    int   LINES, COLUMNS;	
    int  headers_per_page;

    menu_get_sizes(page,&LINES, &COLUMNS);

    /* 1)  Title part of screen */

    if (! LOC->title_page)
	LOC->title_page = new_menu_subpage(page,0,4,sb_update_threadtitle,
					   LIST);
    else
	menu_subpage_relocate(LOC->title_page,page,0,4);


    /* 3) menu part */


    if (LOC->menu_page && LINES < 14)
	erase_menu_context (&(LOC->menu_page));
    else if (LOC->menu_page)
	menu_subpage_relocate(LOC->menu_page,page,LINES-7,3);
    else if (mini_menu && LINES > 14)
	LOC->menu_page = new_menu_subpage(page,LINES-7,3,
					  sb_update_thread_menu,LIST);

    /* 4) prompt part  */

    if (LOC->prompt_page)
	menu_subpage_relocate(LOC->prompt_page,page,LINES-4,4);
    else 
	LOC->prompt_page = new_menu_subpage(page,LINES-4,4,
					    subpage_simple_noredraw,LIST);


    /* 2) thread part */
    headers_per_page = LOC->menu_page ? LINES-12 : LINES-9;
    if (headers_per_page < 1) {
	headers_per_page = 1;
    }

    if (! LOC->header_page)
	LOC->header_page = new_menu_header(page,4,
					   headers_per_page,
					   thread_show_header,
					   thread_show_current,
					   null_header_param_changed,
					   thread_show_status,
					   null_header_line_separator_index,
					   header_separator_noredraw,
					   null_header_separator_start,
					   thread_header_init,
					   thread_header_line,
					   LIST);
    else 
	menu_header_relocate(LOC->header_page,page,
			     4,headers_per_page);


}


static void check_thread_screen P_((struct screen_parts *LOC,
				    struct menu_param *list));
static void check_thread_screen(LOC,list)
     struct screen_parts *LOC;
     struct menu_param *list;
{
    /* 1) title page */
    if (menu_resized(LOC->title_page)) {
	DPRINT(Debug,1, (&Debug, "title page resized\n"));

    }
    if (menu_need_redraw(LOC->title_page)) {
	DPRINT(Debug,1, (&Debug, "title page redraw???\n"));
	sb_update_threadtitle(LOC->title_page,list);
    }

    /* 2) headers part */
    if (menu_resized(LOC->header_page)) {
	DPRINT(Debug,1, (&Debug, "header page resized\n"));
    }
    if (menu_need_redraw(LOC->header_page)) {
	DPRINT(Debug,1, (&Debug, "header page redraw\n"));
	menu_ClearScreen(LOC->header_page);
    }


    if (LOC->menu_page) {
	/* 3) menu page */
	if (menu_resized(LOC->menu_page)) {
	    DPRINT(Debug,1, (&Debug, "menu page resized\n"));
	    
	}
	if (menu_need_redraw(LOC->menu_page)) {
	    DPRINT(Debug,1, (&Debug, "menu page redraw\n"));
	    sb_update_thread_menu(LOC->menu_page,list);
	}
    }

    /* 4) prompt part */
    if (menu_resized(LOC->prompt_page)) {
	DPRINT(Debug,1, (&Debug, "prompt page resized\n"));
    }
    if (menu_need_redraw(LOC->prompt_page)) {
	DPRINT(Debug,7, (&Debug, "prompt page redraw\n"));

	menu_ClearScreen(LOC->prompt_page);

	show_last_error();	/* for those operations that have to
				 * clear the footer except for a message.
				 */
    }

}


/* May return EOF */
int ViewThreads(mailbox,aview, parent_page)
     struct MailboxView *mailbox;
     struct AliasView *aview;
     struct menu_context  *parent_page;

{
    int retch = '\0';

    struct menu_context  * page;
    struct screen_parts  LOC  = { NULL, NULL, NULL, NULL };
    struct menu_common MENU;
    int mailbox_sort_needed = 0;
    struct menu_anon_param A;


    struct menu_param  PARAM[thread_mp_COUNT+1] = {   
	{ mp_menu_common, { 0 } },
	{ mp_anon_param, { 0 } },
	{ mp_END, { 0 }  }
    };
    int ch;
    int update = 1;
    int c;


    /* we need update thread list, becuase it may be changed meanwhile */
    update_mailbox_threads(mailbox);
    
    
    set_mcommon_from_mbxview(&MENU,mailbox);    
    mp_list_set_mcommon(PARAM,thread_mp_mailbox,&MENU);    

    A.mailbox = mailbox;
    A.sort_list = NULL;
    A.sort_list_len = 0;
    A.index_width   = 3;
    A.menu_date_width = 21;
    
    mp_list_set_anon(PARAM,thread_mp_param,&A);





    page = new_menu_context();
    set_thread_screen(page,&LOC,PARAM);

    sort_threads(mailbox,&A,LOC.header_page);

    c = get_current(mailbox);
    if (c > 0) {
	int i;
	struct folder_view INDEX;
	int li,co;
	int top     = menu_header_get(LOC.header_page,
				      header_top_line);
    
	INDEX.thread_number = -1;

	give_index_number(mailbox,c-1,&INDEX);

	menu_get_sizes(LOC.header_page, &li, &co);   


	/* FIXME -- not very effective */
	for (i = 0; i < A.sort_list_len; i++) {
	    if (INDEX.thread_number == A.sort_list[i].idx) {
		menu_header_change(LOC.header_page,header_current,i);
		if (i < top || i >= top+li)
		    menu_header_change(LOC.header_page, header_top_line,i);
	    }
	}	
    }


    for (;;) {

	if (menu_resized(page)) {
	    set_thread_screen(page,&LOC,PARAM);
	    
	    update = 1;
	} 

	new_mail_check(mailbox, page, &LOC,NULL);      

	if (update_view(mailbox)) {
	    update = 1;
	    mailbox_sort_needed = 1;

	    /* We do not sort mailbox, because we do not show
	       message list */

	    update_mailbox_threads(mailbox);
	    sort_threads(mailbox,&A,LOC.header_page);
	}

	
	if (update || menu_need_redraw(page)) {
	    menu_ClearScreen(page);
	    
	    /* Call refresh routines of children */
	    menu_redraw_children(page);
	    
	    update = 0;
	    show_last_error(); 
	} 

	check_thread_screen(&LOC, PARAM);

#ifdef BACKGROUD_PROCESSES      
	if (handle_sigchld)
	    sigchld_handler();
#endif
	
	{
	    int lin,col;
	    
	    menu_ClearLine(LOC.prompt_page,0);

	    menu_PutLineX (LOC.prompt_page,0, 0, 
			   CATGETS(elm_msg_cat, MeSet, MeThreadMenuPrompt,
				   "Thread command: "));
	    menu_GetXYLocation(LOC.prompt_page,&lin,&col);
	    
	    menu_CleartoEOS(LOC.prompt_page);   
	    
	    show_last_error();
	    menu_MoveCursor(LOC.prompt_page,lin,col);
	    
	    ch = menu_ReadCh(LOC.prompt_page, 
			     REDRAW_MARK|READCH_CURSOR|READCH_resize|
			     READCH_sig_char);

	    menu_CleartoEOS(LOC.prompt_page);

	    if (isascii(ch) && isprint(ch)) {
		DPRINT(Debug,4,
		       (&Debug, "-- Thread command: %c [%d]\n",ch,ch));
	    } else {
		DPRINT(Debug,4,
		       (&Debug, "-- Thread command: [%d]\n",ch));
	    }

	    set_error("");	/* clear error buffer */
	}

	ch = do_movement1(LOC.header_page,ch,A.sort_list_len,&M);

	switch (ch) {
	    int cur;
	    int top;
	    int li,co;

	case RESIZE_MARK:
	    DPRINT(Debug,4, (&Debug, " ... resizing\n"));
	    continue;

	case ctrl('L'):
	case REDRAW_MARK:
	    update = 1;
	    break;


	case HELP_MARK:
        case '?'        :  {
	    struct elm_commands *cmds = give_view_threads_commands();

	    int X =  help_generic(cmds,FALSE, page, LOC.prompt_page);
	    
	    free_commands(&cmds);   /* Decrement refcount */

	    if (EOF == X) {
		goto handle_EOF;
		/* Read failed, control tty died? */		
	    }

	    break;
	}

	case F3_KEY_MARK:
	    mini_menu = ! mini_menu;

	    if (LOC.menu_page && !mini_menu)
		erase_menu_context (&(LOC.menu_page));

	    DPRINT(Debug,10,(&Debug,"mini_menu = %d \n",mini_menu));

	    set_thread_screen(page,&LOC,PARAM);
		
	    update = 1;
	    
	    break;

	case DOWN_MARK :
	case 'j'    :  
	    cur = menu_header_get(LOC.header_page,header_current);	    
	    cur++;
	    while (0 <= cur && cur < A.sort_list_len) {
		if (A.sort_list[cur].status_letter != 'D' &&
		    A.sort_list[cur].status_letter != 'Z') {
		    goto found1;
		}
		cur++;
	    }
	    
	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeNoMoreUndThreadsBelow,
			      "No more undeleted threads below.")); 

	    break;
	found1:
	    top     = menu_header_get(LOC.header_page,header_top_line);
	    menu_header_change(LOC.header_page, header_current,cur);

	    menu_get_sizes(LOC.header_page, &li, &co);   

	    if (top+li <= cur)
		menu_header_change(LOC.header_page, header_top_line,cur);
	    break;

	case UP_MARK :
	case 'k'     :  
	    cur = menu_header_get(LOC.header_page,header_current);	    
	    cur--;
	    while (0 <= cur && cur < A.sort_list_len) {
		if (A.sort_list[cur].status_letter != 'D' &&
		    A.sort_list[cur].status_letter != 'Z') {
		    goto found2;
		}
		cur--;
	    }

	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeNoMoreUndThreadsAbove,
			      "No more undeleted threads above."));    

	    break;
	found2:
	    top     = menu_header_get(LOC.header_page,header_top_line);
	    menu_header_change(LOC.header_page, header_current,cur);

	    menu_get_sizes(LOC.header_page, &li, &co);   
	    if (top > cur) {
		top = cur-li+1;
		if (top < 0)
		    top = 0;

		menu_header_change(LOC.header_page, header_top_line,top);
	    }
	    break;
	case 'd':
	    cur = menu_header_get(LOC.header_page,header_current);

	    if (0 <= cur && cur < A.sort_list_len) {
		int len;
		int * vector = give_thread_message_list(mailbox,A.sort_list[cur].idx,
							&len);
		if (vector) {
		    int i;

		    for (i = 0; i < len; i++) {
			setf_status_message(mailbox,vector[i],status_basic,
					    DELETED);
		    }

		    free(vector);
		}

		if (calculate_status(&A,cur)) {
		    menu_header_status_update(LOC.header_page,cur);
		}
	    }	    

	    if (resolve_mode) {
		cur = menu_header_get(LOC.header_page,header_current);	    
		cur++;
		while (0 <= cur && cur < A.sort_list_len) {
		    if (A.sort_list[cur].status_letter != 'D' &&
			A.sort_list[cur].status_letter != 'Z') {
			goto found1;
		    }
		    cur++;
		}	   
	    }

	    break;
	case 'u':
	    cur = menu_header_get(LOC.header_page,header_current);

	    if (0 <= cur && cur < A.sort_list_len) {
		int len;
		int * vector = give_thread_message_list(mailbox,A.sort_list[cur].idx,
							&len);
		if (vector) {
		    int i;

		    for (i = 0; i < len; i++) {
			clearf_status_message(mailbox,vector[i],status_basic,
					    DELETED);
		    }

		    free(vector);
		}

		if (calculate_status(&A,cur)) {
		    menu_header_status_update(LOC.header_page,cur);
		}
	    }	    

	    if (resolve_mode) {
		cur = menu_header_get(LOC.header_page,header_current);	    
		cur++;
		while (0 <= cur && cur < A.sort_list_len) {
		    if (A.sort_list[cur].status_letter != 'Z') {
			goto found1;
		    }
		    cur++;
		}	   
	    }

	    break;

	case '\r':
	case '\n':
	    cur = menu_header_get(LOC.header_page,header_current);

	    if (0 <= cur && cur < A.sort_list_len) {
		int need_resort = 0;

		int r =
		    ViewThread1(mailbox,aview,A.sort_list[cur].idx,page,
				&need_resort);

		if (calculate_status(&A,cur)) {
		    menu_header_status_update(LOC.header_page,cur);
		}

		/* New mails may be arrived meanwhile */

		update = 1;
		if (need_resort)
		    mailbox_sort_needed = 1;
		
		/* We do not sort mailbox, because we do not show
		   message list */
		
		update_mailbox_threads(mailbox);
		sort_threads(mailbox,&A,LOC.header_page);

		if (EOF == r)
		    goto handle_EOF;
	    }
	    break;

	case 'm':
	    menu_Write_to_screen(LOC.prompt_page,
				 CATGETS(elm_msg_cat, MeSet,
					 MeMail,
					 "Mail"));
	    FlushBuffer();
	    send_msg_l(-1, NULL, NULL, NULL,
		       MAIL_EDIT_MSG,
		       mailbox, aview,page, LOC.prompt_page,
		       NULL);
	    break;

	handle_EOF:
	case EOF:
	    retch = EOF;

	case 'i':
	case 'q':
	case 'x':
	case TERMCH_interrupt_char:
	    goto OUT;

	case 0:
	    break;

	default:
	    if (ch > '0' && ch <= '9') {
		int value;

		struct string * str = format_string(CATGETS(elm_msg_cat, MeSet,
							    MeThreadItem,
							    "thread"));

		menu_Write_to_screen(LOC.prompt_page,
				     CATGETS(elm_msg_cat, MeSet,
					     MeNewCurrentThread,
					     "New current thread"));

		cur = menu_header_get(LOC.header_page,header_current);
		value = read_number(ch, str, cur+1, page, LOC.prompt_page);

		free_string(&str);

		if (value > A.sort_list_len) {
		    lib_error(CATGETS(elm_msg_cat, MeSet,
				      MeNotThatManyThread,
				      "Not that many threads."));

		} else if (value > 0) {
		    cur = value-1;

		    top     = menu_header_get(LOC.header_page,header_top_line);
		    menu_header_change(LOC.header_page, header_current,cur);
		    
		    menu_get_sizes(LOC.header_page, &li, &co);   
		    
		    if (top+li <= cur)
			menu_header_change(LOC.header_page, header_top_line,cur);
		    else if (top > cur) {
			top = cur-li+1;
			if (top < 0)
			    top = 0;
			
			menu_header_change(LOC.header_page, header_top_line,top);
		    }
		    
		}

	    } else {
		if (isascii(ch) && isprint(ch))
		    lib_error(CATGETS(elm_msg_cat, MeSet,
				      MeThreadUnknownCommand,		       
				      "Unknown command: %c"), 
			      ch);
		else
		    lib_error(CATGETS(elm_msg_cat, MeSet,
				      MeUnknownCommand2,			
				      "Unknown command."));
	    }
	}
    }
    
 OUT:
    free_mailbox_screen(&LOC);
    
    erase_menu_context(&page);
    
    if (mailbox_sort_needed)
	resort_mailbox(mailbox,1);

    menu_trigger_redraw(parent_page);

    /* Force default return to parent page ... */
    menu_set_default(parent_page); 

    if (A.sort_list) {
	int i;
	
	for (i = 0; i < A.sort_list_len; i++) {
	    if (A.sort_list[i].menu_date_s)
		free_string(& (A.sort_list[i].menu_date_s));	    
	}

	free(A.sort_list);
	A.sort_list = NULL;
	
    }
    A.sort_list_len = 0;
    
    
    return retch;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
