static char rcsid[] = "@(#)$Id: elmcharset.c,v 2.11 2018/12/21 11:21:43 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.11 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_utils.h"
#include "s_me.h"
#include "s_elm.h"
#include "cs_imp.h"
#include "misclib.h"

#include "reghelper.h"

DEBUG_VAR(Debug,__FILE__,"util");

extern char *optarg;		
extern int   optind;		

const char * program_name = "elmcharset";
int    register_fd  = -1;
char * register_module = NULL;

#include "charmapcopy.h"

static char version_buff[NLEN];

const char OPTION_LETTERS[] = "Gd:w:I:S:Mcr:C:";

int main P_((int argc, char *argv[]));
int main(argc, argv)
     int argc;
     char *argv[];
{
    int err = 0;
    int c;
    int global = 0;
    struct charset_map_item * MAP = NULL;
    char *targetfile = NULL;
    FILE * commentfile = NULL;
    int MIB = 0;
    int config_merge = 0;

    int read_flags = 0;
    
#if DEBUG
    init_debugfile("ELMCHARSET");
#endif

    while ((c = ELM_GETOPT(argc, argv, OPTION_LETTERS)) != EOF) {
	switch(c) {
	case 'G':
	    global++;
	    break;
	case 'd':
#if DEBUG
	    set_debugging(optarg);
#endif
	    /* Error is printed later */
	    break;
	}
    }
    optind = 1;     /* Reset scanning */

    locale_init();

    REGHELPER_INIT(argv[0]);

    read_flags = global ? READ_FLAG_GLOBAL_ONLY : 0;

    user_init();
    init_defaults(read_flags);
    
    while ((c = ELM_GETOPT(argc, argv, OPTION_LETTERS)) != EOF) {
	FILE *X;
	char buf[LONG_STRING];
	
	switch(c) {
	case 'G':
	    global++;
	    break;
	case 'c':
	    config_merge++;
	    break;

	case 'C':
	    if (0 != access(optarg,READ_ACCESS)) {
		int errcode = errno;
		lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotReadable,
				  "File %.50s is not readable: %s"),
			  optarg, strerror(errcode));
		err = 1;
		goto fail;
	    }

	    commentfile = fopen(optarg,"r");
	    if (!commentfile) {
		int errcode = errno;
		lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotReadable,
				  "File %.50s is not readable: %s"),
			  optarg, strerror(errcode));
		err = 1;
		goto fail;
		
	    }
	    break;

	case 'M':
	    MIB++;
	    break;
	case 'S':
	    if (0 == access(optarg,ACCESS_EXISTS)) {
		copydirs = safe_array_realloc(copydirs,(copydir_count+1), sizeof (*copydirs));
		copydirs[copydir_count++] = optarg;
	    } else {
		int errcode = errno;
		
		lib_error(CATGETS(elm_msg_cat, MeSet,
				  MeDirNotExists,
				  "Directory %s does not exists: %s"),
			  optarg,strerror(errcode));		
		err = 1;
	    }
	    break;
	case 'd':
#if DEBUG
	    set_debugging(optarg);	 
#else
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmArgsIngoringDebug,
			      "Warning: system created without debugging enabled - request ignored\n"));
#endif
	    break;
	case 'r':
	    set_user_rc_location(optarg); 
	    break;
	case 'w' : 
	    targetfile = optarg;
	    if (0 != access(targetfile,WRITE_ACCESS)) {
		int errcode = errno;
		
		if (errcode != ENOENT) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
				      "File %.50s is not writeable: %s"),
			      targetfile, strerror(errcode));
		    err = 1;
		    goto fail;
		}
	    }
	    break;
	case 'I' :
	    if (0 != access(optarg,READ_ACCESS)) {
		int errcode = errno;
		lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotReadable,
				  "File %.50s is not readable: %s"),
			  optarg, strerror(errcode));
		err = 1;
		goto fail;
	    }
	    X = fopen(optarg,"r");
	    if (!X) {
		int errcode = errno;
		lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotReadable,
				  "File %.50s is not readable: %s"),
			  optarg, strerror(errcode));
		err = 1;
		goto fail;
	    }
	    while(fgets(buf,sizeof buf, X) != NULL) {
		char *c;
		int l1 = strlen(buf);
		charset_t N;
		
		if ('\n' == buf[l1 -1]) 
		    buf[l1 - 1] = '\0';
		else {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeTooLongLine,
				      "%30s: Too long line: %.30s..."),
			      optarg,buf);
		    err = 1;
		    break;
		}
		l1--;

		while (l1-- > 0 && whitespace(buf[l1]))
		    buf[l1] = '\0';
		c = buf;
		while (*c && whitespace (*c)) /* skip leading whitespace */
		    c++;
		if ('#' == *c)
		    continue;
		if (!*c)
		    continue;
		
		/* Inore aliases and codesets */
		N = MIME_name_to_charset(c,CHARSET_noalias);
		if (N && 
		    0 == istrcmp(N->MIME_name,c)) {
		    N->flags &= ~SET_valid;
		    
		    DPRINT(Debug,1, (&Debug, 
				     "Ignoring charset %s\n",
				     N->MIME_name));
		} else {
		    DPRINT(Debug,5, (&Debug, 
				     "No charset %s to ignore%s%s\n",
				     c,N ? " -- was " : "",N->MIME_name));
		}
	    }
	    fclose(X);
	    break;
	case '?':
	    err = 1;
	    goto fail;
	}
    }
    
    
    elm_sfprintf(version_buff, sizeof version_buff,
		 FRM("%s PL%s"), VERSION, PATCHLEVEL);
    
#ifdef DEBUG
    { 
	int d = panic_dprint("\n\
======================================================\n\
Debug output of the ELMCHARSET program (version %s).\n",
			     version_buff);

	if (d >= 50) {
#if 0	
	    panic_dprint("WARNING: Edit manually out sensitive information from that file!\n");
	    
	    lower_prompt("WARNING: Debug file may include passwords -- edit it!");
	    sleep(5+sleepmsg);	    
#endif
	}
    }
#endif
    
    if (copydir_count)
	set_charmap_copy_callback(charmap_copy);

    if (!global)
	read_rc_file(read_flags);
    else
	post_init_check(read_flags);
    
    if (optind < argc) {
	int errcount = 0;

	MAP = load_charset_map(argv[optind],&errcount);
	if (!MAP || errcount) {
	    err = 1;
	    goto fail;
	}
    }

    if (config_merge) {

	if (!MAP) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeConfigMergeMap,
			      "Config merge (-c) requires map as argument"));
	    err = 1;
	    goto fail;
	}

	if (targetfile) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeConfigMergeNoW,
			      "Config merge (-c) Can not used with -w"));
	    err = 1;
	    goto fail;
	}

	if (global) {
	    change_charset_map(& system_charset_map, MAP);
	    free_charset_map(&MAP);

	    MAP        = system_charset_map;
	    targetfile = system_mime_charsets;

	} else {

	    change_charset_map(& user_charset_map, MAP);
	    free_charset_map(&MAP);

	    MAP = user_charset_map;
	    targetfile = user_mime_charsets;
	}

	if (0 != access(targetfile,WRITE_ACCESS)) {
	    int errcode = errno;
	    
	    if (errcode != ENOENT) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
				  "File %.50s is not writeable: %s"),
			  targetfile, strerror(errcode));
		err = 1;
		goto fail;
	    }
	}
    }

	  
    if (!MAP && !MIB) {
	if (global)
	    MAP = system_charset_map;
	else
	    MAP = user_charset_map;
    }
    
    if (targetfile) {
	char * tmp = elm_message(FRM("%s.N"),targetfile);
	int errcode = can_open(tmp,"w");
	FILE *f;
	int have_data = 0;

	if (errcode) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
			      "File %.50s is not writeable: %s"),
		      tmp, strerror(errcode));
	  
	    err = 1;
	    free(tmp);
	    goto fail;
	}
	f = fopen(tmp,"w");
	if (!f) {
	    int errcode = errno;
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
			      "File %.50s is not writeable: %s"),
		      tmp, strerror(errcode));
	    
	    err = 1;
	    free(tmp);
	    goto fail;
	}

	if (MIB)
	    have_data += dump_by_MIBenum(f,commentfile,
					 "ELMCHARSET",version_buff,
					 have_data);
	if (MAP)
	    have_data += dump_charset_map(f,MAP,commentfile,
					  "ELMCHARSET",version_buff,
					  have_data);

	if (EOF == fclose(f)) {
	    int errcode = errno;
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
			      "File %.50s is not writeable: %s"),
		      tmp, strerror(errcode));
	    
	    err = 1;
	    free(tmp);
	    goto fail;
	}
	if (0 != rename(tmp,targetfile)) {
	    int errcode = errno;
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotRenamed,
			      "Failed to rename temporary file to %.50s: %.30s"),
		      targetfile, strerror(errcode));
	    
	    err = 1;
	    free(tmp);
	    goto fail;
	}
	log_config(targetfile);
	free(tmp);

    } else {
	int have_data = 0;

	if (MIB)
	    have_data += dump_by_MIBenum(stdout,commentfile,
					 "ELMCHARSET",version_buff,
					 have_data);
	if (MAP)
	    have_data += dump_charset_map(stdout,MAP,commentfile,
					  "ELMCHARSET",version_buff,
					  have_data);
    }

 fail:
    if (commentfile)
	fclose(commentfile);

    if (MAP &&
	MAP != system_charset_map &&
	MAP != user_charset_map)
	free_charset_map(&MAP);

    if (err)
	lib_error(CATGETS(elm_msg_cat, MeSet, MeProgFailed,
			  "%s failed; exit code=%d"),
		  argv[0],err);
    
    return err;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
