static char rcsid[] = "@(#)$Id: elmtagfilterents.c,v 2.1 2023/12/13 16:55:32 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.1 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_utils.h"
#include "misclib.h"
#include "reghelper.h"
#include "s_me.h"
#include "s_elm.h"
#include "mboxlib.h"
#include "addrlib.h"
#include "melib.h"
#include "rc_imp.h"

extern char *optarg;		
extern int   optind;		

DEBUG_VAR(Debug,__FILE__,"util");

const char * program_name = "elmtagfilterents";
int    register_fd  = -1;
char * register_module = NULL;

static char version_buff[NLEN];

static const char OPTION_LETTERS[] = "Gd:cC:r:w:"; 

int main P_((int argc, char *argv[]));
int main(argc, argv)
     int argc;
     char *argv[];
{
    int err = 0;
    int global = 0;

    char *targetfile = NULL;
    FILE * commentfile = NULL;
    int config_merge = 0;

    int read_flags = 0;
    int c;

    struct tagfilter_entities_conf * MAP = NULL;
    charset_t                        cs = NULL;
    struct editor_propline         * pl = NULL;

#if DEBUG
    init_debugfile("ELMTAGFILTERENTS");
#endif

        while ((c = ELM_GETOPT(argc, argv, OPTION_LETTERS)) != EOF) {
	switch(c) {
	case 'G':
	    global++;
	    break;
	case 'd':
#if DEBUG
	    set_debugging(optarg);
#endif
	    /* Error is printed later */
	    break;
	}
    }
    optind = 1;     /* Reset scanning */

    locale_init();

    REGHELPER_INIT(argv[0]);

    read_flags = global ? READ_FLAG_GLOBAL_ONLY : 0;
    
    user_init();
    init_addrlib(read_flags);
    init_mboxlib(read_flags);
    init_melib(read_flags);
    init_defaults(read_flags);

    while ((c = ELM_GETOPT(argc, argv, OPTION_LETTERS)) != EOF) {
	switch(c) {
	case 'G':
	    global++;
	    break;
	case 'c':
	    config_merge++;
	    break;

	case 'C':
	    if (0 != access(optarg,READ_ACCESS)) {
		int errcode = errno;
		lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotReadable,
				  "File %.50s is not readable: %s"),
			  optarg, strerror(errcode));
		err++;
		goto fail;
	    }

	    commentfile = fopen(optarg,"r");
	    if (!commentfile) {
		int errcode = errno;
		lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotReadable,
				  "File %.50s is not readable: %s"),
			  optarg, strerror(errcode));
		err++;
		goto fail;
		
	    }
	    break;

	    	case 'd' : 
#if DEBUG
	    set_debugging(optarg);	 
#else
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmArgsIngoringDebug,
			      "Warning: system created without debugging enabled - request ignored\n"));
#endif
	    break;

	case 'r':
	    set_user_rc_location(optarg); 
	    break;

	case 'w' : 
	    targetfile = optarg;
	    if (0 != access(targetfile,WRITE_ACCESS)) {
		int errcode = errno;
		
		if (errcode != ENOENT) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
				      "File %.50s is not writeable: %s"),
			      targetfile, strerror(errcode));
		    err++;
		    goto fail;
		}
	    }
	    break;
	case '?':
	    err = 1;
	    goto fail;
	}
    }

    elm_sfprintf(version_buff, sizeof version_buff,
		 FRM("%s PL%s"), VERSION, PATCHLEVEL);

#ifdef DEBUG
    { 
	int d UNUSED_VAROK = panic_dprint("\n\
======================================================\n\
Debug output of the ELMTAGFILTERENTS program (version %s).\n",
			     version_buff);

#if 0	
	if (d >= 50) {
	    panic_dprint("WARNING: Edit manually out sensitive information from that file!\n");
    
	    lower_prompt("WARNING: Debug file may include passwords -- edit it!");
	    sleep(5+sleepmsg);	    
	}
#endif
    }
#endif

    if (!global)
	read_rc_file(read_flags);
    else
	post_init_check(read_flags);

    if (optind < argc) {
	int errcount = 0;
	int need_rewrite = 0;

	MAP = parse_tagfilter_entities(argv[optind],&errcount,&cs,&pl,
				       global ? SYSTEM_RC : LOCAL_RC,
				       & need_rewrite);

	if (!MAP || errcount) {
	    err = 1;
	    goto fail;
	}
    }
    
    if (config_merge) {
	int errcount = 0;

	if (!MAP) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeConfigMergeMap,
			      "Config merge (-c) requires map as argument"));
	    err++;
	    goto fail;
	}

	if (targetfile) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeConfigMergeNoW,
			      "Config merge (-c) Can not used with -w"));
	    err++;
	    goto fail;
	}

	if (pl)
	    free_editor_propline(&pl);


	if (global) {
	    if (!change_tagfilter_entities(& system_tagfilter_entities_conf,
					   MAP,SYSTEM_RC,
					   system_tagfilter_entfile,&errcount)) {
		err++;
	    }
		
	    free_tagfilter_entities_conf(&MAP);

	    MAP        =  system_tagfilter_entities_conf;
	    targetfile =  system_tagfilter_entfile;
	    cs         =  system_tagfilter_entities_cs;
	    pl         =  system_tagfilter_entities_pl;

	} else {
	    if (!change_tagfilter_entities(&  user_tagfilter_entities_conf,
					   MAP, LOCAL_RC,
					   user_tagfilter_entfile,&errcount
					   )) {
		err++;
	    }
	    
	    free_tagfilter_entities_conf(&MAP);

	    MAP        = user_tagfilter_entities_conf;
	    targetfile = user_tagfilter_entfile;
	    cs         = user_tagfilter_entities_cs;
	    pl         = user_tagfilter_entities_pl;	    
	}

	if (errcount) {
	    err = 1;
	}

	if (0 != access(targetfile,WRITE_ACCESS)) {
	    int errcode = errno;
	    
	    if (errcode != ENOENT) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
				  "File %.50s is not writeable: %s"),
			  targetfile, strerror(errcode));
		err++;
		goto fail;
	    }
	}
    }


    if (!MAP) {
	if (global) {
	    MAP = system_tagfilter_entities_conf;
	    cs  = system_tagfilter_entities_cs;
	    pl  = system_tagfilter_entities_pl;
	} else {
	    MAP = user_tagfilter_entities_conf;
	    cs  = user_tagfilter_entities_cs;
	    pl  = user_tagfilter_entities_pl;
	}
    }

  
    if (targetfile) {
	char * tmp = elm_message(FRM("%s.N"),targetfile);
	int errcode = can_open(tmp,"w");
	FILE *f;
	
	if (errcode) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
			      "File %.50s is not writeable: %s"),
		      tmp, strerror(errcode));
	    
	    err++;
	    goto fail_dump;
	}
	
	f = fopen(tmp,"w");
	if (!f) {
	    errcode = errno;
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
			      "File %.50s is not writeable: %s"),
		      tmp, strerror(errcode));
	    
	    err++;
	    goto fail_dump;
	}

	if (! dump_tagfilter_entities(f,MAP,commentfile,
				      "ELMTAGFILTERENTS",version_buff,
				      cs,pl,
				      global ? SYSTEM_RC : LOCAL_RC,
				      &errcode)) {
	    if (errcode)
		lib_error(CATGETS(elm_msg_cat, MeSet, MeConfFileWriteFailedC,
				  "Failed to write config file %.50s: %s"),
			  tmp,strerror(errcode));
	    else
		lib_error(CATGETS(elm_msg_cat, MeSet, MeConfFileWriteFailed,
				  "Failed to write config file %.50s"),
			  tmp);
	    fclose(f);
	    
	    err++;
	    goto fail_dump;
	}
				      
	if (EOF == fclose(f)) {
	    errcode = errno;
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
			      "File %.50s is not writeable: %s"),
		      tmp, strerror(errcode));
	    
	    err++;
	    goto fail_dump;
	}
	if (0 != rename(tmp,targetfile)) {
	    errcode = errno;
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotRenamed,
			      "Failed to rename temporary file to %.50s: %.30s"),
		      targetfile, strerror(errcode));
	    
	    err++;
	    goto fail_dump;
	}
	log_config(targetfile);

    fail_dump:
	
	free(tmp);

    } else {
	int errcode = 0;
	
	if (! dump_tagfilter_entities(stdout,MAP,commentfile,
				      "ELMTAGFILTERENTS",version_buff,
				      cs,pl,
				      global ? SYSTEM_RC : LOCAL_RC,
				      &errcode)) {

	    if (errcode)
		lib_error(CATGETS(elm_msg_cat, MeSet, MeConfPrintFailedC,
				  "Failed to print config: %s"),
			  strerror(errcode));
	    else
		lib_error(CATGETS(elm_msg_cat, MeSet, MeConfPrintFailed,
				  "Failed to print config"));

	    err++;
	    goto fail;
	}			  	    
    }
    
 fail:
    if (commentfile)
	fclose(commentfile);

    if (MAP &&
	MAP != system_tagfilter_entities_conf &&
	MAP != user_tagfilter_entities_conf)
	free_tagfilter_entities_conf(&MAP);	

    if (err)
      lib_error(CATGETS(elm_msg_cat, MeSet, MeProgFailed,
			"%s failed; exit code=%d"),
		argv[0],err);
    
    return err;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */

